"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
require("@aws-cdk/assert/jest");
const acm = require("@aws-cdk/aws-certificatemanager");
const lambda = require("@aws-cdk/aws-lambda");
const s3 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
const test_origin_1 = require("./test-origin");
let app;
let stack;
beforeEach(() => {
    app = new core_1.App();
    stack = new core_1.Stack(app, 'Stack', {
        env: { account: '1234', region: 'testregion' },
    });
});
test('minimal example renders correctly', () => {
    const origin = test_origin_1.defaultOrigin();
    new lib_1.Distribution(stack, 'MyDist', { defaultBehavior: { origin } });
    expect(stack).toHaveResource('AWS::CloudFront::Distribution', {
        DistributionConfig: {
            DefaultCacheBehavior: {
                ForwardedValues: { QueryString: false },
                TargetOriginId: 'StackMyDistOrigin1D6D5E535',
                ViewerProtocolPolicy: 'allow-all',
            },
            Enabled: true,
            HttpVersion: 'http2',
            IPV6Enabled: true,
            Origins: [{
                    DomainName: 'www.example.com',
                    Id: 'StackMyDistOrigin1D6D5E535',
                    CustomOriginConfig: {
                        OriginProtocolPolicy: 'https-only',
                    },
                }],
        },
    });
});
test('exhaustive example of props renders correctly', () => {
    const origin = test_origin_1.defaultOrigin();
    const certificate = acm.Certificate.fromCertificateArn(stack, 'Cert', 'arn:aws:acm:us-east-1:123456789012:certificate/12345678-1234-1234-1234-123456789012');
    new lib_1.Distribution(stack, 'MyDist', {
        defaultBehavior: { origin },
        certificate,
        comment: 'a test',
        defaultRootObject: 'index.html',
        enabled: false,
        enableIpv6: false,
        enableLogging: true,
        geoRestriction: lib_1.GeoRestriction.blacklist('US', 'GB'),
        httpVersion: lib_1.HttpVersion.HTTP1_1,
        logFilePrefix: 'logs/',
        logIncludesCookies: true,
        priceClass: lib_1.PriceClass.PRICE_CLASS_100,
        webAclId: '473e64fd-f30b-4765-81a0-62ad96dd167a',
    });
    expect(stack).toHaveResource('AWS::CloudFront::Distribution', {
        DistributionConfig: {
            DefaultCacheBehavior: {
                ForwardedValues: { QueryString: false },
                TargetOriginId: 'StackMyDistOrigin1D6D5E535',
                ViewerProtocolPolicy: 'allow-all',
            },
            Comment: 'a test',
            DefaultRootObject: 'index.html',
            Enabled: false,
            HttpVersion: 'http1.1',
            IPV6Enabled: false,
            Logging: {
                Bucket: { 'Fn::GetAtt': ['MyDistLoggingBucket9B8976BC', 'RegionalDomainName'] },
                IncludeCookies: true,
                Prefix: 'logs/',
            },
            Origins: [{
                    DomainName: 'www.example.com',
                    Id: 'StackMyDistOrigin1D6D5E535',
                    CustomOriginConfig: {
                        OriginProtocolPolicy: 'https-only',
                    },
                }],
            PriceClass: 'PriceClass_100',
            Restrictions: {
                GeoRestriction: {
                    Locations: ['US', 'GB'],
                    RestrictionType: 'blacklist',
                },
            },
            ViewerCertificate: {
                AcmCertificateArn: 'arn:aws:acm:us-east-1:123456789012:certificate/12345678-1234-1234-1234-123456789012',
                SslSupportMethod: 'sni-only',
                MinimumProtocolVersion: 'TLSv1.2_2018',
            },
        },
    });
});
describe('multiple behaviors', () => {
    test('a second behavior can\'t be specified with the catch-all path pattern', () => {
        const origin = test_origin_1.defaultOrigin();
        expect(() => {
            new lib_1.Distribution(stack, 'MyDist', {
                defaultBehavior: { origin },
                additionalBehaviors: {
                    '*': { origin },
                },
            });
        }).toThrow(/Only the default behavior can have a path pattern of \'*\'/);
    });
    test('a second behavior can be added to the original origin', () => {
        const origin = test_origin_1.defaultOrigin();
        new lib_1.Distribution(stack, 'MyDist', {
            defaultBehavior: { origin },
            additionalBehaviors: {
                'api/*': { origin },
            },
        });
        expect(stack).toHaveResource('AWS::CloudFront::Distribution', {
            DistributionConfig: {
                DefaultCacheBehavior: {
                    ForwardedValues: { QueryString: false },
                    TargetOriginId: 'StackMyDistOrigin1D6D5E535',
                    ViewerProtocolPolicy: 'allow-all',
                },
                CacheBehaviors: [{
                        PathPattern: 'api/*',
                        ForwardedValues: { QueryString: false },
                        TargetOriginId: 'StackMyDistOrigin1D6D5E535',
                        ViewerProtocolPolicy: 'allow-all',
                    }],
                Enabled: true,
                HttpVersion: 'http2',
                IPV6Enabled: true,
                Origins: [{
                        DomainName: 'www.example.com',
                        Id: 'StackMyDistOrigin1D6D5E535',
                        CustomOriginConfig: {
                            OriginProtocolPolicy: 'https-only',
                        },
                    }],
            },
        });
    });
    test('a second behavior can be added to a secondary origin', () => {
        const origin = test_origin_1.defaultOrigin();
        const origin2 = test_origin_1.defaultOrigin('origin2.example.com');
        new lib_1.Distribution(stack, 'MyDist', {
            defaultBehavior: { origin },
            additionalBehaviors: {
                'api/*': { origin: origin2 },
            },
        });
        expect(stack).toHaveResource('AWS::CloudFront::Distribution', {
            DistributionConfig: {
                DefaultCacheBehavior: {
                    ForwardedValues: { QueryString: false },
                    TargetOriginId: 'StackMyDistOrigin1D6D5E535',
                    ViewerProtocolPolicy: 'allow-all',
                },
                CacheBehaviors: [{
                        PathPattern: 'api/*',
                        ForwardedValues: { QueryString: false },
                        TargetOriginId: 'StackMyDistOrigin20B96F3AD',
                        ViewerProtocolPolicy: 'allow-all',
                    }],
                Enabled: true,
                HttpVersion: 'http2',
                IPV6Enabled: true,
                Origins: [{
                        DomainName: 'www.example.com',
                        Id: 'StackMyDistOrigin1D6D5E535',
                        CustomOriginConfig: {
                            OriginProtocolPolicy: 'https-only',
                        },
                    },
                    {
                        DomainName: 'origin2.example.com',
                        Id: 'StackMyDistOrigin20B96F3AD',
                        CustomOriginConfig: {
                            OriginProtocolPolicy: 'https-only',
                        },
                    }],
            },
        });
    });
    test('behavior creation order is preserved', () => {
        const origin = test_origin_1.defaultOrigin();
        const origin2 = test_origin_1.defaultOrigin('origin2.example.com');
        const dist = new lib_1.Distribution(stack, 'MyDist', {
            defaultBehavior: { origin },
            additionalBehaviors: {
                'api/1*': { origin: origin2 },
            },
        });
        dist.addBehavior('api/2*', origin);
        expect(stack).toHaveResource('AWS::CloudFront::Distribution', {
            DistributionConfig: {
                DefaultCacheBehavior: {
                    ForwardedValues: { QueryString: false },
                    TargetOriginId: 'StackMyDistOrigin1D6D5E535',
                    ViewerProtocolPolicy: 'allow-all',
                },
                CacheBehaviors: [{
                        PathPattern: 'api/1*',
                        ForwardedValues: { QueryString: false },
                        TargetOriginId: 'StackMyDistOrigin20B96F3AD',
                        ViewerProtocolPolicy: 'allow-all',
                    },
                    {
                        PathPattern: 'api/2*',
                        ForwardedValues: { QueryString: false },
                        TargetOriginId: 'StackMyDistOrigin1D6D5E535',
                        ViewerProtocolPolicy: 'allow-all',
                    }],
                Enabled: true,
                HttpVersion: 'http2',
                IPV6Enabled: true,
                Origins: [{
                        DomainName: 'www.example.com',
                        Id: 'StackMyDistOrigin1D6D5E535',
                        CustomOriginConfig: {
                            OriginProtocolPolicy: 'https-only',
                        },
                    },
                    {
                        DomainName: 'origin2.example.com',
                        Id: 'StackMyDistOrigin20B96F3AD',
                        CustomOriginConfig: {
                            OriginProtocolPolicy: 'https-only',
                        },
                    }],
            },
        });
    });
});
describe('certificates', () => {
    test('should fail if using an imported certificate from outside of us-east-1', () => {
        const origin = test_origin_1.defaultOrigin();
        const certificate = acm.Certificate.fromCertificateArn(stack, 'Cert', 'arn:aws:acm:eu-west-1:123456789012:certificate/12345678-1234-1234-1234-123456789012');
        expect(() => {
            new lib_1.Distribution(stack, 'Dist', {
                defaultBehavior: { origin },
                certificate,
            });
        }).toThrow(/Distribution certificates must be in the us-east-1 region and the certificate you provided is in eu-west-1./);
    });
    test('adding a certificate renders the correct ViewerCertificate property', () => {
        const certificate = acm.Certificate.fromCertificateArn(stack, 'Cert', 'arn:aws:acm:us-east-1:123456789012:certificate/12345678-1234-1234-1234-123456789012');
        new lib_1.Distribution(stack, 'Dist', {
            defaultBehavior: { origin: test_origin_1.defaultOrigin() },
            certificate,
        });
        expect(stack).toHaveResourceLike('AWS::CloudFront::Distribution', {
            DistributionConfig: {
                ViewerCertificate: {
                    AcmCertificateArn: 'arn:aws:acm:us-east-1:123456789012:certificate/12345678-1234-1234-1234-123456789012',
                    SslSupportMethod: 'sni-only',
                    MinimumProtocolVersion: 'TLSv1.2_2018',
                },
            },
        });
    });
});
describe('custom error responses', () => {
    test('should fail if responsePagePath is defined but responseCode is not', () => {
        const origin = test_origin_1.defaultOrigin();
        expect(() => {
            new lib_1.Distribution(stack, 'Dist', {
                defaultBehavior: { origin },
                errorResponses: [{
                        httpStatus: 404,
                        responsePagePath: '/errors/404.html',
                    }],
            });
        }).toThrow(/\'responseCode\' must be provided if \'responsePagePath\' is defined/);
    });
    test('should fail if only the error code is provided', () => {
        const origin = test_origin_1.defaultOrigin();
        expect(() => {
            new lib_1.Distribution(stack, 'Dist', {
                defaultBehavior: { origin },
                errorResponses: [{ httpStatus: 404 }],
            });
        }).toThrow(/A custom error response without either a \'responseCode\' or \'errorCachingMinTtl\' is not valid./);
    });
    test('should render the array of error configs if provided', () => {
        const origin = test_origin_1.defaultOrigin();
        new lib_1.Distribution(stack, 'Dist', {
            defaultBehavior: { origin },
            errorResponses: [{
                    httpStatus: 404,
                    responseHttpStatus: 404,
                    responsePagePath: '/errors/404.html',
                },
                {
                    httpStatus: 500,
                    ttl: core_1.Duration.seconds(2),
                }],
        });
        expect(stack).toHaveResourceLike('AWS::CloudFront::Distribution', {
            DistributionConfig: {
                CustomErrorResponses: [
                    {
                        ErrorCode: 404,
                        ResponseCode: 404,
                        ResponsePagePath: '/errors/404.html',
                    },
                    {
                        ErrorCachingMinTTL: 2,
                        ErrorCode: 500,
                    },
                ],
            },
        });
    });
});
describe('logging', () => {
    test('does not include logging if disabled and no bucket provided', () => {
        const origin = test_origin_1.defaultOrigin();
        new lib_1.Distribution(stack, 'MyDist', { defaultBehavior: { origin } });
        expect(stack).toHaveResourceLike('AWS::CloudFront::Distribution', {
            DistributionConfig: {
                Logging: assert_1.ABSENT,
            },
        });
    });
    test('throws error if logging disabled but bucket provided', () => {
        const origin = test_origin_1.defaultOrigin();
        expect(() => {
            new lib_1.Distribution(stack, 'MyDist', {
                defaultBehavior: { origin },
                enableLogging: false,
                logBucket: new s3.Bucket(stack, 'Bucket'),
            });
        }).toThrow(/Explicitly disabled logging but provided a logging bucket./);
    });
    test('creates bucket if none is provided', () => {
        const origin = test_origin_1.defaultOrigin();
        new lib_1.Distribution(stack, 'MyDist', {
            defaultBehavior: { origin },
            enableLogging: true,
        });
        expect(stack).toHaveResourceLike('AWS::CloudFront::Distribution', {
            DistributionConfig: {
                Logging: {
                    Bucket: { 'Fn::GetAtt': ['MyDistLoggingBucket9B8976BC', 'RegionalDomainName'] },
                },
            },
        });
    });
    test('uses existing bucket if provided', () => {
        const origin = test_origin_1.defaultOrigin();
        const loggingBucket = new s3.Bucket(stack, 'MyLoggingBucket');
        new lib_1.Distribution(stack, 'MyDist', {
            defaultBehavior: { origin },
            logBucket: loggingBucket,
        });
        expect(stack).toHaveResourceLike('AWS::CloudFront::Distribution', {
            DistributionConfig: {
                Logging: {
                    Bucket: { 'Fn::GetAtt': ['MyLoggingBucket4382CD04', 'RegionalDomainName'] },
                },
            },
        });
    });
    test('can set prefix and cookies', () => {
        const origin = test_origin_1.defaultOrigin();
        new lib_1.Distribution(stack, 'MyDist', {
            defaultBehavior: { origin },
            enableLogging: true,
            logFilePrefix: 'logs/',
            logIncludesCookies: true,
        });
        expect(stack).toHaveResourceLike('AWS::CloudFront::Distribution', {
            DistributionConfig: {
                Logging: {
                    Bucket: { 'Fn::GetAtt': ['MyDistLoggingBucket9B8976BC', 'RegionalDomainName'] },
                    IncludeCookies: true,
                    Prefix: 'logs/',
                },
            },
        });
    });
});
describe('with Lambda@Edge functions', () => {
    let lambdaFunction;
    let origin;
    beforeEach(() => {
        lambdaFunction = new lambda.Function(stack, 'Function', {
            runtime: lambda.Runtime.NODEJS,
            code: lambda.Code.fromInline('whatever'),
            handler: 'index.handler',
        });
        origin = test_origin_1.defaultOrigin();
    });
    test('can add an edge lambdas to the default behavior', () => {
        new lib_1.Distribution(stack, 'MyDist', {
            defaultBehavior: {
                origin,
                edgeLambdas: [
                    {
                        functionVersion: lambdaFunction.currentVersion,
                        eventType: lib_1.LambdaEdgeEventType.ORIGIN_REQUEST,
                    },
                ],
            },
        });
        expect(stack).toHaveResourceLike('AWS::CloudFront::Distribution', {
            DistributionConfig: {
                DefaultCacheBehavior: {
                    LambdaFunctionAssociations: [
                        {
                            EventType: 'origin-request',
                            LambdaFunctionARN: {
                                Ref: 'FunctionCurrentVersion4E2B2261477a5ae8059bbaa7813f752292c0f65e',
                            },
                        },
                    ],
                },
            },
        });
    });
    test('can add an edge lambdas to additional behaviors', () => {
        new lib_1.Distribution(stack, 'MyDist', {
            defaultBehavior: { origin },
            additionalBehaviors: {
                'images/*': {
                    origin,
                    edgeLambdas: [
                        {
                            functionVersion: lambdaFunction.currentVersion,
                            eventType: lib_1.LambdaEdgeEventType.VIEWER_REQUEST,
                        },
                    ],
                },
            },
        });
        expect(stack).toHaveResourceLike('AWS::CloudFront::Distribution', {
            DistributionConfig: {
                CacheBehaviors: [
                    {
                        PathPattern: 'images/*',
                        LambdaFunctionAssociations: [
                            {
                                EventType: 'viewer-request',
                                LambdaFunctionARN: {
                                    Ref: 'FunctionCurrentVersion4E2B2261477a5ae8059bbaa7813f752292c0f65e',
                                },
                            },
                        ],
                    },
                ],
            },
        });
    });
    test('fails creation when attempting to add the $LATEST function version as an edge Lambda to the default behavior', () => {
        expect(() => {
            new lib_1.Distribution(stack, 'MyDist', {
                defaultBehavior: {
                    origin,
                    edgeLambdas: [
                        {
                            functionVersion: lambdaFunction.latestVersion,
                            eventType: lib_1.LambdaEdgeEventType.ORIGIN_RESPONSE,
                        },
                    ],
                },
            });
        }).toThrow(/\$LATEST function version cannot be used for Lambda@Edge/);
    });
});
test('price class is included if provided', () => {
    const origin = test_origin_1.defaultOrigin();
    new lib_1.Distribution(stack, 'Dist', {
        defaultBehavior: { origin },
        priceClass: lib_1.PriceClass.PRICE_CLASS_200,
    });
    expect(stack).toHaveResourceLike('AWS::CloudFront::Distribution', {
        DistributionConfig: {
            PriceClass: 'PriceClass_200',
        },
    });
});
test('escape hatches are supported', () => {
    const dist = new lib_1.Distribution(stack, 'Dist', {
        defaultBehavior: { origin: test_origin_1.defaultOrigin },
    });
    const cfnDist = dist.node.defaultChild;
    cfnDist.addPropertyOverride('DistributionConfig.DefaultCacheBehavior.ForwardedValues.Headers', ['*']);
    expect(stack).toHaveResourceLike('AWS::CloudFront::Distribution', {
        DistributionConfig: {
            DefaultCacheBehavior: {
                ForwardedValues: {
                    Headers: ['*'],
                },
            },
        },
    });
});
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZGlzdHJpYnV0aW9uLnRlc3QuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJkaXN0cmlidXRpb24udGVzdC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOztBQUFBLDRDQUF5QztBQUN6QyxnQ0FBOEI7QUFDOUIsdURBQXVEO0FBQ3ZELDhDQUE4QztBQUM5QyxzQ0FBc0M7QUFDdEMsd0NBQXFEO0FBQ3JELGdDQUE4SDtBQUM5SCwrQ0FBOEM7QUFFOUMsSUFBSSxHQUFRLENBQUM7QUFDYixJQUFJLEtBQVksQ0FBQztBQUVqQixVQUFVLENBQUMsR0FBRyxFQUFFO0lBQ2QsR0FBRyxHQUFHLElBQUksVUFBRyxFQUFFLENBQUM7SUFDaEIsS0FBSyxHQUFHLElBQUksWUFBSyxDQUFDLEdBQUcsRUFBRSxPQUFPLEVBQUU7UUFDOUIsR0FBRyxFQUFFLEVBQUUsT0FBTyxFQUFFLE1BQU0sRUFBRSxNQUFNLEVBQUUsWUFBWSxFQUFFO0tBQy9DLENBQUMsQ0FBQztBQUNMLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLG1DQUFtQyxFQUFFLEdBQUcsRUFBRTtJQUM3QyxNQUFNLE1BQU0sR0FBRywyQkFBYSxFQUFFLENBQUM7SUFDL0IsSUFBSSxrQkFBWSxDQUFDLEtBQUssRUFBRSxRQUFRLEVBQUUsRUFBRSxlQUFlLEVBQUUsRUFBRSxNQUFNLEVBQUUsRUFBRSxDQUFDLENBQUM7SUFFbkUsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLGNBQWMsQ0FBQywrQkFBK0IsRUFBRTtRQUM1RCxrQkFBa0IsRUFBRTtZQUNsQixvQkFBb0IsRUFBRTtnQkFDcEIsZUFBZSxFQUFFLEVBQUUsV0FBVyxFQUFFLEtBQUssRUFBRTtnQkFDdkMsY0FBYyxFQUFFLDRCQUE0QjtnQkFDNUMsb0JBQW9CLEVBQUUsV0FBVzthQUNsQztZQUNELE9BQU8sRUFBRSxJQUFJO1lBQ2IsV0FBVyxFQUFFLE9BQU87WUFDcEIsV0FBVyxFQUFFLElBQUk7WUFDakIsT0FBTyxFQUFFLENBQUM7b0JBQ1IsVUFBVSxFQUFFLGlCQUFpQjtvQkFDN0IsRUFBRSxFQUFFLDRCQUE0QjtvQkFDaEMsa0JBQWtCLEVBQUU7d0JBQ2xCLG9CQUFvQixFQUFFLFlBQVk7cUJBQ25DO2lCQUNGLENBQUM7U0FDSDtLQUNGLENBQUMsQ0FBQztBQUNMLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLCtDQUErQyxFQUFFLEdBQUcsRUFBRTtJQUN6RCxNQUFNLE1BQU0sR0FBRywyQkFBYSxFQUFFLENBQUM7SUFDL0IsTUFBTSxXQUFXLEdBQUcsR0FBRyxDQUFDLFdBQVcsQ0FBQyxrQkFBa0IsQ0FBQyxLQUFLLEVBQUUsTUFBTSxFQUFFLHFGQUFxRixDQUFDLENBQUM7SUFFN0osSUFBSSxrQkFBWSxDQUFDLEtBQUssRUFBRSxRQUFRLEVBQUU7UUFDaEMsZUFBZSxFQUFFLEVBQUUsTUFBTSxFQUFFO1FBQzNCLFdBQVc7UUFDWCxPQUFPLEVBQUUsUUFBUTtRQUNqQixpQkFBaUIsRUFBRSxZQUFZO1FBQy9CLE9BQU8sRUFBRSxLQUFLO1FBQ2QsVUFBVSxFQUFFLEtBQUs7UUFDakIsYUFBYSxFQUFFLElBQUk7UUFDbkIsY0FBYyxFQUFFLG9CQUFjLENBQUMsU0FBUyxDQUFDLElBQUksRUFBRSxJQUFJLENBQUM7UUFDcEQsV0FBVyxFQUFFLGlCQUFXLENBQUMsT0FBTztRQUNoQyxhQUFhLEVBQUUsT0FBTztRQUN0QixrQkFBa0IsRUFBRSxJQUFJO1FBQ3hCLFVBQVUsRUFBRSxnQkFBVSxDQUFDLGVBQWU7UUFDdEMsUUFBUSxFQUFFLHNDQUFzQztLQUNqRCxDQUFDLENBQUM7SUFFSCxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsY0FBYyxDQUFDLCtCQUErQixFQUFFO1FBQzVELGtCQUFrQixFQUFFO1lBQ2xCLG9CQUFvQixFQUFFO2dCQUNwQixlQUFlLEVBQUUsRUFBRSxXQUFXLEVBQUUsS0FBSyxFQUFFO2dCQUN2QyxjQUFjLEVBQUUsNEJBQTRCO2dCQUM1QyxvQkFBb0IsRUFBRSxXQUFXO2FBQ2xDO1lBQ0QsT0FBTyxFQUFFLFFBQVE7WUFDakIsaUJBQWlCLEVBQUUsWUFBWTtZQUMvQixPQUFPLEVBQUUsS0FBSztZQUNkLFdBQVcsRUFBRSxTQUFTO1lBQ3RCLFdBQVcsRUFBRSxLQUFLO1lBQ2xCLE9BQU8sRUFBRTtnQkFDUCxNQUFNLEVBQUUsRUFBQyxZQUFZLEVBQUUsQ0FBQyw2QkFBNkIsRUFBRSxvQkFBb0IsQ0FBQyxFQUFDO2dCQUM3RSxjQUFjLEVBQUUsSUFBSTtnQkFDcEIsTUFBTSxFQUFFLE9BQU87YUFDaEI7WUFDRCxPQUFPLEVBQUUsQ0FBQztvQkFDUixVQUFVLEVBQUUsaUJBQWlCO29CQUM3QixFQUFFLEVBQUUsNEJBQTRCO29CQUNoQyxrQkFBa0IsRUFBRTt3QkFDbEIsb0JBQW9CLEVBQUUsWUFBWTtxQkFDbkM7aUJBQ0YsQ0FBQztZQUNGLFVBQVUsRUFBRSxnQkFBZ0I7WUFDNUIsWUFBWSxFQUFFO2dCQUNaLGNBQWMsRUFBRTtvQkFDZCxTQUFTLEVBQUUsQ0FBQyxJQUFJLEVBQUUsSUFBSSxDQUFDO29CQUN2QixlQUFlLEVBQUUsV0FBVztpQkFDN0I7YUFDRjtZQUNELGlCQUFpQixFQUFFO2dCQUNqQixpQkFBaUIsRUFBRSxxRkFBcUY7Z0JBQ3hHLGdCQUFnQixFQUFFLFVBQVU7Z0JBQzVCLHNCQUFzQixFQUFFLGNBQWM7YUFDdkM7U0FDRjtLQUNGLENBQUMsQ0FBQztBQUNMLENBQUMsQ0FBQyxDQUFDO0FBRUgsUUFBUSxDQUFDLG9CQUFvQixFQUFFLEdBQUcsRUFBRTtJQUVsQyxJQUFJLENBQUMsdUVBQXVFLEVBQUUsR0FBRyxFQUFFO1FBQ2pGLE1BQU0sTUFBTSxHQUFHLDJCQUFhLEVBQUUsQ0FBQztRQUUvQixNQUFNLENBQUMsR0FBRyxFQUFFO1lBQ1YsSUFBSSxrQkFBWSxDQUFDLEtBQUssRUFBRSxRQUFRLEVBQUU7Z0JBQ2hDLGVBQWUsRUFBRSxFQUFFLE1BQU0sRUFBRTtnQkFDM0IsbUJBQW1CLEVBQUU7b0JBQ25CLEdBQUcsRUFBRSxFQUFFLE1BQU0sRUFBRTtpQkFDaEI7YUFDRixDQUFDLENBQUM7UUFDTCxDQUFDLENBQUMsQ0FBQyxPQUFPLENBQUMsNERBQTRELENBQUMsQ0FBQztJQUMzRSxDQUFDLENBQUMsQ0FBQztJQUVILElBQUksQ0FBQyx1REFBdUQsRUFBRSxHQUFHLEVBQUU7UUFDakUsTUFBTSxNQUFNLEdBQUcsMkJBQWEsRUFBRSxDQUFDO1FBQy9CLElBQUksa0JBQVksQ0FBQyxLQUFLLEVBQUUsUUFBUSxFQUFFO1lBQ2hDLGVBQWUsRUFBRSxFQUFFLE1BQU0sRUFBRTtZQUMzQixtQkFBbUIsRUFBRTtnQkFDbkIsT0FBTyxFQUFFLEVBQUUsTUFBTSxFQUFFO2FBQ3BCO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLGNBQWMsQ0FBQywrQkFBK0IsRUFBRTtZQUM1RCxrQkFBa0IsRUFBRTtnQkFDbEIsb0JBQW9CLEVBQUU7b0JBQ3BCLGVBQWUsRUFBRSxFQUFFLFdBQVcsRUFBRSxLQUFLLEVBQUU7b0JBQ3ZDLGNBQWMsRUFBRSw0QkFBNEI7b0JBQzVDLG9CQUFvQixFQUFFLFdBQVc7aUJBQ2xDO2dCQUNELGNBQWMsRUFBRSxDQUFDO3dCQUNmLFdBQVcsRUFBRSxPQUFPO3dCQUNwQixlQUFlLEVBQUUsRUFBRSxXQUFXLEVBQUUsS0FBSyxFQUFFO3dCQUN2QyxjQUFjLEVBQUUsNEJBQTRCO3dCQUM1QyxvQkFBb0IsRUFBRSxXQUFXO3FCQUNsQyxDQUFDO2dCQUNGLE9BQU8sRUFBRSxJQUFJO2dCQUNiLFdBQVcsRUFBRSxPQUFPO2dCQUNwQixXQUFXLEVBQUUsSUFBSTtnQkFDakIsT0FBTyxFQUFFLENBQUM7d0JBQ1IsVUFBVSxFQUFFLGlCQUFpQjt3QkFDN0IsRUFBRSxFQUFFLDRCQUE0Qjt3QkFDaEMsa0JBQWtCLEVBQUU7NEJBQ2xCLG9CQUFvQixFQUFFLFlBQVk7eUJBQ25DO3FCQUNGLENBQUM7YUFDSDtTQUNGLENBQUMsQ0FBQztJQUNMLENBQUMsQ0FBQyxDQUFDO0lBRUgsSUFBSSxDQUFDLHNEQUFzRCxFQUFFLEdBQUcsRUFBRTtRQUNoRSxNQUFNLE1BQU0sR0FBRywyQkFBYSxFQUFFLENBQUM7UUFDL0IsTUFBTSxPQUFPLEdBQUcsMkJBQWEsQ0FBQyxxQkFBcUIsQ0FBQyxDQUFDO1FBQ3JELElBQUksa0JBQVksQ0FBQyxLQUFLLEVBQUUsUUFBUSxFQUFFO1lBQ2hDLGVBQWUsRUFBRSxFQUFFLE1BQU0sRUFBRTtZQUMzQixtQkFBbUIsRUFBRTtnQkFDbkIsT0FBTyxFQUFFLEVBQUUsTUFBTSxFQUFFLE9BQU8sRUFBRTthQUM3QjtTQUNGLENBQUMsQ0FBQztRQUVILE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxjQUFjLENBQUMsK0JBQStCLEVBQUU7WUFDNUQsa0JBQWtCLEVBQUU7Z0JBQ2xCLG9CQUFvQixFQUFFO29CQUNwQixlQUFlLEVBQUUsRUFBRSxXQUFXLEVBQUUsS0FBSyxFQUFFO29CQUN2QyxjQUFjLEVBQUUsNEJBQTRCO29CQUM1QyxvQkFBb0IsRUFBRSxXQUFXO2lCQUNsQztnQkFDRCxjQUFjLEVBQUUsQ0FBQzt3QkFDZixXQUFXLEVBQUUsT0FBTzt3QkFDcEIsZUFBZSxFQUFFLEVBQUUsV0FBVyxFQUFFLEtBQUssRUFBRTt3QkFDdkMsY0FBYyxFQUFFLDRCQUE0Qjt3QkFDNUMsb0JBQW9CLEVBQUUsV0FBVztxQkFDbEMsQ0FBQztnQkFDRixPQUFPLEVBQUUsSUFBSTtnQkFDYixXQUFXLEVBQUUsT0FBTztnQkFDcEIsV0FBVyxFQUFFLElBQUk7Z0JBQ2pCLE9BQU8sRUFBRSxDQUFDO3dCQUNSLFVBQVUsRUFBRSxpQkFBaUI7d0JBQzdCLEVBQUUsRUFBRSw0QkFBNEI7d0JBQ2hDLGtCQUFrQixFQUFFOzRCQUNsQixvQkFBb0IsRUFBRSxZQUFZO3lCQUNuQztxQkFDRjtvQkFDRDt3QkFDRSxVQUFVLEVBQUUscUJBQXFCO3dCQUNqQyxFQUFFLEVBQUUsNEJBQTRCO3dCQUNoQyxrQkFBa0IsRUFBRTs0QkFDbEIsb0JBQW9CLEVBQUUsWUFBWTt5QkFDbkM7cUJBQ0YsQ0FBQzthQUNIO1NBQ0YsQ0FBQyxDQUFDO0lBQ0wsQ0FBQyxDQUFDLENBQUM7SUFFSCxJQUFJLENBQUMsc0NBQXNDLEVBQUUsR0FBRyxFQUFFO1FBQ2hELE1BQU0sTUFBTSxHQUFHLDJCQUFhLEVBQUUsQ0FBQztRQUMvQixNQUFNLE9BQU8sR0FBRywyQkFBYSxDQUFDLHFCQUFxQixDQUFDLENBQUM7UUFDckQsTUFBTSxJQUFJLEdBQUcsSUFBSSxrQkFBWSxDQUFDLEtBQUssRUFBRSxRQUFRLEVBQUU7WUFDN0MsZUFBZSxFQUFFLEVBQUUsTUFBTSxFQUFFO1lBQzNCLG1CQUFtQixFQUFFO2dCQUNuQixRQUFRLEVBQUUsRUFBRSxNQUFNLEVBQUUsT0FBTyxFQUFFO2FBQzlCO1NBQ0YsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLFdBQVcsQ0FBQyxRQUFRLEVBQUUsTUFBTSxDQUFDLENBQUM7UUFFbkMsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLGNBQWMsQ0FBQywrQkFBK0IsRUFBRTtZQUM1RCxrQkFBa0IsRUFBRTtnQkFDbEIsb0JBQW9CLEVBQUU7b0JBQ3BCLGVBQWUsRUFBRSxFQUFFLFdBQVcsRUFBRSxLQUFLLEVBQUU7b0JBQ3ZDLGNBQWMsRUFBRSw0QkFBNEI7b0JBQzVDLG9CQUFvQixFQUFFLFdBQVc7aUJBQ2xDO2dCQUNELGNBQWMsRUFBRSxDQUFDO3dCQUNmLFdBQVcsRUFBRSxRQUFRO3dCQUNyQixlQUFlLEVBQUUsRUFBRSxXQUFXLEVBQUUsS0FBSyxFQUFFO3dCQUN2QyxjQUFjLEVBQUUsNEJBQTRCO3dCQUM1QyxvQkFBb0IsRUFBRSxXQUFXO3FCQUNsQztvQkFDRDt3QkFDRSxXQUFXLEVBQUUsUUFBUTt3QkFDckIsZUFBZSxFQUFFLEVBQUUsV0FBVyxFQUFFLEtBQUssRUFBRTt3QkFDdkMsY0FBYyxFQUFFLDRCQUE0Qjt3QkFDNUMsb0JBQW9CLEVBQUUsV0FBVztxQkFDbEMsQ0FBQztnQkFDRixPQUFPLEVBQUUsSUFBSTtnQkFDYixXQUFXLEVBQUUsT0FBTztnQkFDcEIsV0FBVyxFQUFFLElBQUk7Z0JBQ2pCLE9BQU8sRUFBRSxDQUFDO3dCQUNSLFVBQVUsRUFBRSxpQkFBaUI7d0JBQzdCLEVBQUUsRUFBRSw0QkFBNEI7d0JBQ2hDLGtCQUFrQixFQUFFOzRCQUNsQixvQkFBb0IsRUFBRSxZQUFZO3lCQUNuQztxQkFDRjtvQkFDRDt3QkFDRSxVQUFVLEVBQUUscUJBQXFCO3dCQUNqQyxFQUFFLEVBQUUsNEJBQTRCO3dCQUNoQyxrQkFBa0IsRUFBRTs0QkFDbEIsb0JBQW9CLEVBQUUsWUFBWTt5QkFDbkM7cUJBQ0YsQ0FBQzthQUNIO1NBQ0YsQ0FBQyxDQUFDO0lBQ0wsQ0FBQyxDQUFDLENBQUM7QUFDTCxDQUFDLENBQUMsQ0FBQztBQUVILFFBQVEsQ0FBQyxjQUFjLEVBQUUsR0FBRyxFQUFFO0lBRTVCLElBQUksQ0FBQyx3RUFBd0UsRUFBRSxHQUFHLEVBQUU7UUFDbEYsTUFBTSxNQUFNLEdBQUcsMkJBQWEsRUFBRSxDQUFDO1FBQy9CLE1BQU0sV0FBVyxHQUFHLEdBQUcsQ0FBQyxXQUFXLENBQUMsa0JBQWtCLENBQUMsS0FBSyxFQUFFLE1BQU0sRUFBRSxxRkFBcUYsQ0FBQyxDQUFDO1FBRTdKLE1BQU0sQ0FBQyxHQUFHLEVBQUU7WUFDVixJQUFJLGtCQUFZLENBQUMsS0FBSyxFQUFFLE1BQU0sRUFBRTtnQkFDOUIsZUFBZSxFQUFFLEVBQUUsTUFBTSxFQUFFO2dCQUMzQixXQUFXO2FBQ1osQ0FBQyxDQUFDO1FBQ0wsQ0FBQyxDQUFDLENBQUMsT0FBTyxDQUFDLDZHQUE2RyxDQUFDLENBQUM7SUFDNUgsQ0FBQyxDQUFDLENBQUM7SUFFSCxJQUFJLENBQUMscUVBQXFFLEVBQUUsR0FBRyxFQUFFO1FBQy9FLE1BQU0sV0FBVyxHQUFHLEdBQUcsQ0FBQyxXQUFXLENBQUMsa0JBQWtCLENBQUMsS0FBSyxFQUFFLE1BQU0sRUFBRSxxRkFBcUYsQ0FBQyxDQUFDO1FBRTdKLElBQUksa0JBQVksQ0FBQyxLQUFLLEVBQUUsTUFBTSxFQUFFO1lBQzlCLGVBQWUsRUFBRSxFQUFFLE1BQU0sRUFBRSwyQkFBYSxFQUFFLEVBQUU7WUFDNUMsV0FBVztTQUNaLENBQUMsQ0FBQztRQUVILE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxrQkFBa0IsQ0FBQywrQkFBK0IsRUFBRTtZQUNoRSxrQkFBa0IsRUFBRTtnQkFDbEIsaUJBQWlCLEVBQUU7b0JBQ2pCLGlCQUFpQixFQUFFLHFGQUFxRjtvQkFDeEcsZ0JBQWdCLEVBQUUsVUFBVTtvQkFDNUIsc0JBQXNCLEVBQUUsY0FBYztpQkFDdkM7YUFDRjtTQUNGLENBQUMsQ0FBQztJQUNMLENBQUMsQ0FBQyxDQUFDO0FBQ0wsQ0FBQyxDQUFDLENBQUM7QUFFSCxRQUFRLENBQUMsd0JBQXdCLEVBQUUsR0FBRyxFQUFFO0lBRXRDLElBQUksQ0FBQyxvRUFBb0UsRUFBRSxHQUFHLEVBQUU7UUFDOUUsTUFBTSxNQUFNLEdBQUcsMkJBQWEsRUFBRSxDQUFDO1FBRS9CLE1BQU0sQ0FBQyxHQUFHLEVBQUU7WUFDVixJQUFJLGtCQUFZLENBQUMsS0FBSyxFQUFFLE1BQU0sRUFBRTtnQkFDOUIsZUFBZSxFQUFFLEVBQUUsTUFBTSxFQUFFO2dCQUMzQixjQUFjLEVBQUUsQ0FBQzt3QkFDZixVQUFVLEVBQUUsR0FBRzt3QkFDZixnQkFBZ0IsRUFBRSxrQkFBa0I7cUJBQ3JDLENBQUM7YUFDSCxDQUFDLENBQUM7UUFDTCxDQUFDLENBQUMsQ0FBQyxPQUFPLENBQUMsc0VBQXNFLENBQUMsQ0FBQztJQUNyRixDQUFDLENBQUMsQ0FBQztJQUVILElBQUksQ0FBQyxnREFBZ0QsRUFBRSxHQUFHLEVBQUU7UUFDMUQsTUFBTSxNQUFNLEdBQUcsMkJBQWEsRUFBRSxDQUFDO1FBRS9CLE1BQU0sQ0FBQyxHQUFHLEVBQUU7WUFDVixJQUFJLGtCQUFZLENBQUMsS0FBSyxFQUFFLE1BQU0sRUFBRTtnQkFDOUIsZUFBZSxFQUFFLEVBQUUsTUFBTSxFQUFFO2dCQUMzQixjQUFjLEVBQUUsQ0FBQyxFQUFFLFVBQVUsRUFBRSxHQUFHLEVBQUUsQ0FBQzthQUN0QyxDQUFDLENBQUM7UUFDTCxDQUFDLENBQUMsQ0FBQyxPQUFPLENBQUMsbUdBQW1HLENBQUMsQ0FBQztJQUNsSCxDQUFDLENBQUMsQ0FBQztJQUVILElBQUksQ0FBQyxzREFBc0QsRUFBRSxHQUFHLEVBQUU7UUFDaEUsTUFBTSxNQUFNLEdBQUcsMkJBQWEsRUFBRSxDQUFDO1FBQy9CLElBQUksa0JBQVksQ0FBQyxLQUFLLEVBQUUsTUFBTSxFQUFFO1lBQzlCLGVBQWUsRUFBRSxFQUFFLE1BQU0sRUFBRTtZQUMzQixjQUFjLEVBQUUsQ0FBQztvQkFDZixVQUFVLEVBQUUsR0FBRztvQkFDZixrQkFBa0IsRUFBRSxHQUFHO29CQUN2QixnQkFBZ0IsRUFBRSxrQkFBa0I7aUJBQ3JDO2dCQUNEO29CQUNFLFVBQVUsRUFBRSxHQUFHO29CQUNmLEdBQUcsRUFBRSxlQUFRLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQztpQkFDekIsQ0FBQztTQUNILENBQUMsQ0FBQztRQUVILE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxrQkFBa0IsQ0FBQywrQkFBK0IsRUFBRTtZQUNoRSxrQkFBa0IsRUFBRTtnQkFDbEIsb0JBQW9CLEVBQUU7b0JBQ3BCO3dCQUNFLFNBQVMsRUFBRSxHQUFHO3dCQUNkLFlBQVksRUFBRSxHQUFHO3dCQUNqQixnQkFBZ0IsRUFBRSxrQkFBa0I7cUJBQ3JDO29CQUNEO3dCQUNFLGtCQUFrQixFQUFFLENBQUM7d0JBQ3JCLFNBQVMsRUFBRSxHQUFHO3FCQUNmO2lCQUNGO2FBQ0Y7U0FDRixDQUFDLENBQUM7SUFDTCxDQUFDLENBQUMsQ0FBQztBQUVMLENBQUMsQ0FBQyxDQUFDO0FBRUgsUUFBUSxDQUFDLFNBQVMsRUFBRSxHQUFHLEVBQUU7SUFDdkIsSUFBSSxDQUFDLDZEQUE2RCxFQUFFLEdBQUcsRUFBRTtRQUN2RSxNQUFNLE1BQU0sR0FBRywyQkFBYSxFQUFFLENBQUM7UUFDL0IsSUFBSSxrQkFBWSxDQUFDLEtBQUssRUFBRSxRQUFRLEVBQUUsRUFBRSxlQUFlLEVBQUUsRUFBRSxNQUFNLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFFbkUsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLGtCQUFrQixDQUFDLCtCQUErQixFQUFFO1lBQ2hFLGtCQUFrQixFQUFFO2dCQUNsQixPQUFPLEVBQUUsZUFBTTthQUNoQjtTQUNGLENBQUMsQ0FBQztJQUNMLENBQUMsQ0FBQyxDQUFDO0lBRUgsSUFBSSxDQUFDLHNEQUFzRCxFQUFFLEdBQUcsRUFBRTtRQUNoRSxNQUFNLE1BQU0sR0FBRywyQkFBYSxFQUFFLENBQUM7UUFFL0IsTUFBTSxDQUFDLEdBQUcsRUFBRTtZQUNWLElBQUksa0JBQVksQ0FBQyxLQUFLLEVBQUUsUUFBUSxFQUFFO2dCQUNoQyxlQUFlLEVBQUUsRUFBRSxNQUFNLEVBQUU7Z0JBQzNCLGFBQWEsRUFBRSxLQUFLO2dCQUNwQixTQUFTLEVBQUUsSUFBSSxFQUFFLENBQUMsTUFBTSxDQUFDLEtBQUssRUFBRSxRQUFRLENBQUM7YUFDMUMsQ0FBQyxDQUFDO1FBQ0wsQ0FBQyxDQUFDLENBQUMsT0FBTyxDQUFDLDREQUE0RCxDQUFDLENBQUM7SUFDM0UsQ0FBQyxDQUFDLENBQUM7SUFFSCxJQUFJLENBQUMsb0NBQW9DLEVBQUUsR0FBRyxFQUFFO1FBQzlDLE1BQU0sTUFBTSxHQUFHLDJCQUFhLEVBQUUsQ0FBQztRQUMvQixJQUFJLGtCQUFZLENBQUMsS0FBSyxFQUFFLFFBQVEsRUFBRTtZQUNoQyxlQUFlLEVBQUUsRUFBRSxNQUFNLEVBQUU7WUFDM0IsYUFBYSxFQUFFLElBQUk7U0FDcEIsQ0FBQyxDQUFDO1FBRUgsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLGtCQUFrQixDQUFDLCtCQUErQixFQUFFO1lBQ2hFLGtCQUFrQixFQUFFO2dCQUNsQixPQUFPLEVBQUU7b0JBQ1AsTUFBTSxFQUFFLEVBQUMsWUFBWSxFQUFFLENBQUMsNkJBQTZCLEVBQUUsb0JBQW9CLENBQUMsRUFBQztpQkFDOUU7YUFDRjtTQUNGLENBQUMsQ0FBQztJQUNMLENBQUMsQ0FBQyxDQUFDO0lBRUgsSUFBSSxDQUFDLGtDQUFrQyxFQUFFLEdBQUcsRUFBRTtRQUM1QyxNQUFNLE1BQU0sR0FBRywyQkFBYSxFQUFFLENBQUM7UUFDL0IsTUFBTSxhQUFhLEdBQUcsSUFBSSxFQUFFLENBQUMsTUFBTSxDQUFDLEtBQUssRUFBRSxpQkFBaUIsQ0FBQyxDQUFDO1FBQzlELElBQUksa0JBQVksQ0FBQyxLQUFLLEVBQUUsUUFBUSxFQUFFO1lBQ2hDLGVBQWUsRUFBRSxFQUFFLE1BQU0sRUFBRTtZQUMzQixTQUFTLEVBQUUsYUFBYTtTQUN6QixDQUFDLENBQUM7UUFFSCxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsa0JBQWtCLENBQUMsK0JBQStCLEVBQUU7WUFDaEUsa0JBQWtCLEVBQUU7Z0JBQ2xCLE9BQU8sRUFBRTtvQkFDUCxNQUFNLEVBQUUsRUFBQyxZQUFZLEVBQUUsQ0FBQyx5QkFBeUIsRUFBRSxvQkFBb0IsQ0FBQyxFQUFDO2lCQUMxRTthQUNGO1NBQ0YsQ0FBQyxDQUFDO0lBQ0wsQ0FBQyxDQUFDLENBQUM7SUFFSCxJQUFJLENBQUMsNEJBQTRCLEVBQUUsR0FBRyxFQUFFO1FBQ3RDLE1BQU0sTUFBTSxHQUFHLDJCQUFhLEVBQUUsQ0FBQztRQUMvQixJQUFJLGtCQUFZLENBQUMsS0FBSyxFQUFFLFFBQVEsRUFBRTtZQUNoQyxlQUFlLEVBQUUsRUFBRSxNQUFNLEVBQUU7WUFDM0IsYUFBYSxFQUFFLElBQUk7WUFDbkIsYUFBYSxFQUFFLE9BQU87WUFDdEIsa0JBQWtCLEVBQUUsSUFBSTtTQUN6QixDQUFDLENBQUM7UUFFSCxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsa0JBQWtCLENBQUMsK0JBQStCLEVBQUU7WUFDaEUsa0JBQWtCLEVBQUU7Z0JBQ2xCLE9BQU8sRUFBRTtvQkFDUCxNQUFNLEVBQUUsRUFBQyxZQUFZLEVBQUUsQ0FBQyw2QkFBNkIsRUFBRSxvQkFBb0IsQ0FBQyxFQUFDO29CQUM3RSxjQUFjLEVBQUUsSUFBSTtvQkFDcEIsTUFBTSxFQUFFLE9BQU87aUJBQ2hCO2FBQ0Y7U0FDRixDQUFDLENBQUM7SUFDTCxDQUFDLENBQUMsQ0FBQztBQUNMLENBQUMsQ0FBQyxDQUFDO0FBRUgsUUFBUSxDQUFDLDRCQUE0QixFQUFFLEdBQUcsRUFBRTtJQUMxQyxJQUFJLGNBQStCLENBQUM7SUFDcEMsSUFBSSxNQUFlLENBQUM7SUFFcEIsVUFBVSxDQUFDLEdBQUcsRUFBRTtRQUNkLGNBQWMsR0FBRyxJQUFJLE1BQU0sQ0FBQyxRQUFRLENBQUMsS0FBSyxFQUFFLFVBQVUsRUFBRTtZQUN0RCxPQUFPLEVBQUUsTUFBTSxDQUFDLE9BQU8sQ0FBQyxNQUFNO1lBQzlCLElBQUksRUFBRSxNQUFNLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxVQUFVLENBQUM7WUFDeEMsT0FBTyxFQUFFLGVBQWU7U0FDekIsQ0FBQyxDQUFDO1FBRUgsTUFBTSxHQUFHLDJCQUFhLEVBQUUsQ0FBQztJQUMzQixDQUFDLENBQUMsQ0FBQztJQUVILElBQUksQ0FBQyxpREFBaUQsRUFBRSxHQUFHLEVBQUU7UUFDM0QsSUFBSSxrQkFBWSxDQUFDLEtBQUssRUFBRSxRQUFRLEVBQUU7WUFDaEMsZUFBZSxFQUFFO2dCQUNmLE1BQU07Z0JBQ04sV0FBVyxFQUFFO29CQUNYO3dCQUNFLGVBQWUsRUFBRSxjQUFjLENBQUMsY0FBYzt3QkFDOUMsU0FBUyxFQUFFLHlCQUFtQixDQUFDLGNBQWM7cUJBQzlDO2lCQUNGO2FBQ0Y7U0FDRixDQUFDLENBQUM7UUFFSCxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsa0JBQWtCLENBQUMsK0JBQStCLEVBQUU7WUFDaEUsa0JBQWtCLEVBQUU7Z0JBQ2xCLG9CQUFvQixFQUFFO29CQUNwQiwwQkFBMEIsRUFBRTt3QkFDMUI7NEJBQ0UsU0FBUyxFQUFFLGdCQUFnQjs0QkFDM0IsaUJBQWlCLEVBQUU7Z0NBQ2pCLEdBQUcsRUFBRSxnRUFBZ0U7NkJBQ3RFO3lCQUNGO3FCQUNGO2lCQUNGO2FBQ0Y7U0FDRixDQUFDLENBQUM7SUFDTCxDQUFDLENBQUMsQ0FBQztJQUVILElBQUksQ0FBQyxpREFBaUQsRUFBRSxHQUFHLEVBQUU7UUFDM0QsSUFBSSxrQkFBWSxDQUFDLEtBQUssRUFBRSxRQUFRLEVBQUU7WUFDaEMsZUFBZSxFQUFFLEVBQUUsTUFBTSxFQUFFO1lBQzNCLG1CQUFtQixFQUFFO2dCQUNuQixVQUFVLEVBQUU7b0JBQ1YsTUFBTTtvQkFDTixXQUFXLEVBQUU7d0JBQ1g7NEJBQ0UsZUFBZSxFQUFFLGNBQWMsQ0FBQyxjQUFjOzRCQUM5QyxTQUFTLEVBQUUseUJBQW1CLENBQUMsY0FBYzt5QkFDOUM7cUJBQ0Y7aUJBQ0Y7YUFDRjtTQUNGLENBQUMsQ0FBQztRQUVILE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxrQkFBa0IsQ0FBQywrQkFBK0IsRUFBRTtZQUNoRSxrQkFBa0IsRUFBRTtnQkFDbEIsY0FBYyxFQUFFO29CQUNkO3dCQUNFLFdBQVcsRUFBRSxVQUFVO3dCQUN2QiwwQkFBMEIsRUFBRTs0QkFDMUI7Z0NBQ0UsU0FBUyxFQUFFLGdCQUFnQjtnQ0FDM0IsaUJBQWlCLEVBQUU7b0NBQ2pCLEdBQUcsRUFBRSxnRUFBZ0U7aUNBQ3RFOzZCQUNGO3lCQUNGO3FCQUNGO2lCQUNGO2FBQ0Y7U0FDRixDQUFDLENBQUM7SUFDTCxDQUFDLENBQUMsQ0FBQztJQUVILElBQUksQ0FBQyw4R0FBOEcsRUFBRSxHQUFHLEVBQUU7UUFDeEgsTUFBTSxDQUFDLEdBQUcsRUFBRTtZQUNWLElBQUksa0JBQVksQ0FBQyxLQUFLLEVBQUUsUUFBUSxFQUFFO2dCQUNoQyxlQUFlLEVBQUU7b0JBQ2YsTUFBTTtvQkFDTixXQUFXLEVBQUU7d0JBQ1g7NEJBQ0UsZUFBZSxFQUFFLGNBQWMsQ0FBQyxhQUFhOzRCQUM3QyxTQUFTLEVBQUUseUJBQW1CLENBQUMsZUFBZTt5QkFDL0M7cUJBQ0Y7aUJBQ0Y7YUFDRixDQUFDLENBQUM7UUFDTCxDQUFDLENBQUMsQ0FBQyxPQUFPLENBQUMsMERBQTBELENBQUMsQ0FBQztJQUN6RSxDQUFDLENBQUMsQ0FBQztBQUNMLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLHFDQUFxQyxFQUFFLEdBQUcsRUFBRTtJQUMvQyxNQUFNLE1BQU0sR0FBRywyQkFBYSxFQUFFLENBQUM7SUFDL0IsSUFBSSxrQkFBWSxDQUFDLEtBQUssRUFBRSxNQUFNLEVBQUU7UUFDOUIsZUFBZSxFQUFFLEVBQUUsTUFBTSxFQUFFO1FBQzNCLFVBQVUsRUFBRSxnQkFBVSxDQUFDLGVBQWU7S0FDdkMsQ0FBQyxDQUFDO0lBRUgsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLGtCQUFrQixDQUFDLCtCQUErQixFQUFFO1FBQ2hFLGtCQUFrQixFQUFFO1lBQ2xCLFVBQVUsRUFBRSxnQkFBZ0I7U0FDN0I7S0FDRixDQUFDLENBQUM7QUFDTCxDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyw4QkFBOEIsRUFBRSxHQUFHLEVBQUU7SUFDeEMsTUFBTSxJQUFJLEdBQUcsSUFBSSxrQkFBWSxDQUFDLEtBQUssRUFBRSxNQUFNLEVBQUU7UUFDM0MsZUFBZSxFQUFFLEVBQUUsTUFBTSxFQUFFLDJCQUFhLEVBQUU7S0FDM0MsQ0FBQyxDQUFDO0lBQ0gsTUFBTSxPQUFPLEdBQUcsSUFBSSxDQUFDLElBQUksQ0FBQyxZQUErQixDQUFDO0lBQzFELE9BQU8sQ0FBQyxtQkFBbUIsQ0FBQyxpRUFBaUUsRUFBRSxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUM7SUFFdEcsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLGtCQUFrQixDQUFDLCtCQUErQixFQUFFO1FBQ2hFLGtCQUFrQixFQUFFO1lBQ2xCLG9CQUFvQixFQUFFO2dCQUNwQixlQUFlLEVBQUU7b0JBQ2YsT0FBTyxFQUFFLENBQUMsR0FBRyxDQUFDO2lCQUNmO2FBQ0Y7U0FDRjtLQUNGLENBQUMsQ0FBQztBQUNMLENBQUMsQ0FBQyxDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgQUJTRU5UIH0gZnJvbSAnQGF3cy1jZGsvYXNzZXJ0JztcbmltcG9ydCAnQGF3cy1jZGsvYXNzZXJ0L2plc3QnO1xuaW1wb3J0ICogYXMgYWNtIGZyb20gJ0Bhd3MtY2RrL2F3cy1jZXJ0aWZpY2F0ZW1hbmFnZXInO1xuaW1wb3J0ICogYXMgbGFtYmRhIGZyb20gJ0Bhd3MtY2RrL2F3cy1sYW1iZGEnO1xuaW1wb3J0ICogYXMgczMgZnJvbSAnQGF3cy1jZGsvYXdzLXMzJztcbmltcG9ydCB7IEFwcCwgRHVyYXRpb24sIFN0YWNrIH0gZnJvbSAnQGF3cy1jZGsvY29yZSc7XG5pbXBvcnQgeyBDZm5EaXN0cmlidXRpb24sIERpc3RyaWJ1dGlvbiwgR2VvUmVzdHJpY3Rpb24sIEh0dHBWZXJzaW9uLCBJT3JpZ2luLCBMYW1iZGFFZGdlRXZlbnRUeXBlLCBQcmljZUNsYXNzIH0gZnJvbSAnLi4vbGliJztcbmltcG9ydCB7IGRlZmF1bHRPcmlnaW4gfSBmcm9tICcuL3Rlc3Qtb3JpZ2luJztcblxubGV0IGFwcDogQXBwO1xubGV0IHN0YWNrOiBTdGFjaztcblxuYmVmb3JlRWFjaCgoKSA9PiB7XG4gIGFwcCA9IG5ldyBBcHAoKTtcbiAgc3RhY2sgPSBuZXcgU3RhY2soYXBwLCAnU3RhY2snLCB7XG4gICAgZW52OiB7IGFjY291bnQ6ICcxMjM0JywgcmVnaW9uOiAndGVzdHJlZ2lvbicgfSxcbiAgfSk7XG59KTtcblxudGVzdCgnbWluaW1hbCBleGFtcGxlIHJlbmRlcnMgY29ycmVjdGx5JywgKCkgPT4ge1xuICBjb25zdCBvcmlnaW4gPSBkZWZhdWx0T3JpZ2luKCk7XG4gIG5ldyBEaXN0cmlidXRpb24oc3RhY2ssICdNeURpc3QnLCB7IGRlZmF1bHRCZWhhdmlvcjogeyBvcmlnaW4gfSB9KTtcblxuICBleHBlY3Qoc3RhY2spLnRvSGF2ZVJlc291cmNlKCdBV1M6OkNsb3VkRnJvbnQ6OkRpc3RyaWJ1dGlvbicsIHtcbiAgICBEaXN0cmlidXRpb25Db25maWc6IHtcbiAgICAgIERlZmF1bHRDYWNoZUJlaGF2aW9yOiB7XG4gICAgICAgIEZvcndhcmRlZFZhbHVlczogeyBRdWVyeVN0cmluZzogZmFsc2UgfSxcbiAgICAgICAgVGFyZ2V0T3JpZ2luSWQ6ICdTdGFja015RGlzdE9yaWdpbjFENkQ1RTUzNScsXG4gICAgICAgIFZpZXdlclByb3RvY29sUG9saWN5OiAnYWxsb3ctYWxsJyxcbiAgICAgIH0sXG4gICAgICBFbmFibGVkOiB0cnVlLFxuICAgICAgSHR0cFZlcnNpb246ICdodHRwMicsXG4gICAgICBJUFY2RW5hYmxlZDogdHJ1ZSxcbiAgICAgIE9yaWdpbnM6IFt7XG4gICAgICAgIERvbWFpbk5hbWU6ICd3d3cuZXhhbXBsZS5jb20nLFxuICAgICAgICBJZDogJ1N0YWNrTXlEaXN0T3JpZ2luMUQ2RDVFNTM1JyxcbiAgICAgICAgQ3VzdG9tT3JpZ2luQ29uZmlnOiB7XG4gICAgICAgICAgT3JpZ2luUHJvdG9jb2xQb2xpY3k6ICdodHRwcy1vbmx5JyxcbiAgICAgICAgfSxcbiAgICAgIH1dLFxuICAgIH0sXG4gIH0pO1xufSk7XG5cbnRlc3QoJ2V4aGF1c3RpdmUgZXhhbXBsZSBvZiBwcm9wcyByZW5kZXJzIGNvcnJlY3RseScsICgpID0+IHtcbiAgY29uc3Qgb3JpZ2luID0gZGVmYXVsdE9yaWdpbigpO1xuICBjb25zdCBjZXJ0aWZpY2F0ZSA9IGFjbS5DZXJ0aWZpY2F0ZS5mcm9tQ2VydGlmaWNhdGVBcm4oc3RhY2ssICdDZXJ0JywgJ2Fybjphd3M6YWNtOnVzLWVhc3QtMToxMjM0NTY3ODkwMTI6Y2VydGlmaWNhdGUvMTIzNDU2NzgtMTIzNC0xMjM0LTEyMzQtMTIzNDU2Nzg5MDEyJyk7XG5cbiAgbmV3IERpc3RyaWJ1dGlvbihzdGFjaywgJ015RGlzdCcsIHtcbiAgICBkZWZhdWx0QmVoYXZpb3I6IHsgb3JpZ2luIH0sXG4gICAgY2VydGlmaWNhdGUsXG4gICAgY29tbWVudDogJ2EgdGVzdCcsXG4gICAgZGVmYXVsdFJvb3RPYmplY3Q6ICdpbmRleC5odG1sJyxcbiAgICBlbmFibGVkOiBmYWxzZSxcbiAgICBlbmFibGVJcHY2OiBmYWxzZSxcbiAgICBlbmFibGVMb2dnaW5nOiB0cnVlLFxuICAgIGdlb1Jlc3RyaWN0aW9uOiBHZW9SZXN0cmljdGlvbi5ibGFja2xpc3QoJ1VTJywgJ0dCJyksXG4gICAgaHR0cFZlcnNpb246IEh0dHBWZXJzaW9uLkhUVFAxXzEsXG4gICAgbG9nRmlsZVByZWZpeDogJ2xvZ3MvJyxcbiAgICBsb2dJbmNsdWRlc0Nvb2tpZXM6IHRydWUsXG4gICAgcHJpY2VDbGFzczogUHJpY2VDbGFzcy5QUklDRV9DTEFTU18xMDAsXG4gICAgd2ViQWNsSWQ6ICc0NzNlNjRmZC1mMzBiLTQ3NjUtODFhMC02MmFkOTZkZDE2N2EnLFxuICB9KTtcblxuICBleHBlY3Qoc3RhY2spLnRvSGF2ZVJlc291cmNlKCdBV1M6OkNsb3VkRnJvbnQ6OkRpc3RyaWJ1dGlvbicsIHtcbiAgICBEaXN0cmlidXRpb25Db25maWc6IHtcbiAgICAgIERlZmF1bHRDYWNoZUJlaGF2aW9yOiB7XG4gICAgICAgIEZvcndhcmRlZFZhbHVlczogeyBRdWVyeVN0cmluZzogZmFsc2UgfSxcbiAgICAgICAgVGFyZ2V0T3JpZ2luSWQ6ICdTdGFja015RGlzdE9yaWdpbjFENkQ1RTUzNScsXG4gICAgICAgIFZpZXdlclByb3RvY29sUG9saWN5OiAnYWxsb3ctYWxsJyxcbiAgICAgIH0sXG4gICAgICBDb21tZW50OiAnYSB0ZXN0JyxcbiAgICAgIERlZmF1bHRSb290T2JqZWN0OiAnaW5kZXguaHRtbCcsXG4gICAgICBFbmFibGVkOiBmYWxzZSxcbiAgICAgIEh0dHBWZXJzaW9uOiAnaHR0cDEuMScsXG4gICAgICBJUFY2RW5hYmxlZDogZmFsc2UsXG4gICAgICBMb2dnaW5nOiB7XG4gICAgICAgIEJ1Y2tldDogeydGbjo6R2V0QXR0JzogWydNeURpc3RMb2dnaW5nQnVja2V0OUI4OTc2QkMnLCAnUmVnaW9uYWxEb21haW5OYW1lJ119LFxuICAgICAgICBJbmNsdWRlQ29va2llczogdHJ1ZSxcbiAgICAgICAgUHJlZml4OiAnbG9ncy8nLFxuICAgICAgfSxcbiAgICAgIE9yaWdpbnM6IFt7XG4gICAgICAgIERvbWFpbk5hbWU6ICd3d3cuZXhhbXBsZS5jb20nLFxuICAgICAgICBJZDogJ1N0YWNrTXlEaXN0T3JpZ2luMUQ2RDVFNTM1JyxcbiAgICAgICAgQ3VzdG9tT3JpZ2luQ29uZmlnOiB7XG4gICAgICAgICAgT3JpZ2luUHJvdG9jb2xQb2xpY3k6ICdodHRwcy1vbmx5JyxcbiAgICAgICAgfSxcbiAgICAgIH1dLFxuICAgICAgUHJpY2VDbGFzczogJ1ByaWNlQ2xhc3NfMTAwJyxcbiAgICAgIFJlc3RyaWN0aW9uczoge1xuICAgICAgICBHZW9SZXN0cmljdGlvbjoge1xuICAgICAgICAgIExvY2F0aW9uczogWydVUycsICdHQiddLFxuICAgICAgICAgIFJlc3RyaWN0aW9uVHlwZTogJ2JsYWNrbGlzdCcsXG4gICAgICAgIH0sXG4gICAgICB9LFxuICAgICAgVmlld2VyQ2VydGlmaWNhdGU6IHtcbiAgICAgICAgQWNtQ2VydGlmaWNhdGVBcm46ICdhcm46YXdzOmFjbTp1cy1lYXN0LTE6MTIzNDU2Nzg5MDEyOmNlcnRpZmljYXRlLzEyMzQ1Njc4LTEyMzQtMTIzNC0xMjM0LTEyMzQ1Njc4OTAxMicsXG4gICAgICAgIFNzbFN1cHBvcnRNZXRob2Q6ICdzbmktb25seScsXG4gICAgICAgIE1pbmltdW1Qcm90b2NvbFZlcnNpb246ICdUTFN2MS4yXzIwMTgnLFxuICAgICAgfSxcbiAgICB9LFxuICB9KTtcbn0pO1xuXG5kZXNjcmliZSgnbXVsdGlwbGUgYmVoYXZpb3JzJywgKCkgPT4ge1xuXG4gIHRlc3QoJ2Egc2Vjb25kIGJlaGF2aW9yIGNhblxcJ3QgYmUgc3BlY2lmaWVkIHdpdGggdGhlIGNhdGNoLWFsbCBwYXRoIHBhdHRlcm4nLCAoKSA9PiB7XG4gICAgY29uc3Qgb3JpZ2luID0gZGVmYXVsdE9yaWdpbigpO1xuXG4gICAgZXhwZWN0KCgpID0+IHtcbiAgICAgIG5ldyBEaXN0cmlidXRpb24oc3RhY2ssICdNeURpc3QnLCB7XG4gICAgICAgIGRlZmF1bHRCZWhhdmlvcjogeyBvcmlnaW4gfSxcbiAgICAgICAgYWRkaXRpb25hbEJlaGF2aW9yczoge1xuICAgICAgICAgICcqJzogeyBvcmlnaW4gfSxcbiAgICAgICAgfSxcbiAgICAgIH0pO1xuICAgIH0pLnRvVGhyb3coL09ubHkgdGhlIGRlZmF1bHQgYmVoYXZpb3IgY2FuIGhhdmUgYSBwYXRoIHBhdHRlcm4gb2YgXFwnKlxcJy8pO1xuICB9KTtcblxuICB0ZXN0KCdhIHNlY29uZCBiZWhhdmlvciBjYW4gYmUgYWRkZWQgdG8gdGhlIG9yaWdpbmFsIG9yaWdpbicsICgpID0+IHtcbiAgICBjb25zdCBvcmlnaW4gPSBkZWZhdWx0T3JpZ2luKCk7XG4gICAgbmV3IERpc3RyaWJ1dGlvbihzdGFjaywgJ015RGlzdCcsIHtcbiAgICAgIGRlZmF1bHRCZWhhdmlvcjogeyBvcmlnaW4gfSxcbiAgICAgIGFkZGl0aW9uYWxCZWhhdmlvcnM6IHtcbiAgICAgICAgJ2FwaS8qJzogeyBvcmlnaW4gfSxcbiAgICAgIH0sXG4gICAgfSk7XG5cbiAgICBleHBlY3Qoc3RhY2spLnRvSGF2ZVJlc291cmNlKCdBV1M6OkNsb3VkRnJvbnQ6OkRpc3RyaWJ1dGlvbicsIHtcbiAgICAgIERpc3RyaWJ1dGlvbkNvbmZpZzoge1xuICAgICAgICBEZWZhdWx0Q2FjaGVCZWhhdmlvcjoge1xuICAgICAgICAgIEZvcndhcmRlZFZhbHVlczogeyBRdWVyeVN0cmluZzogZmFsc2UgfSxcbiAgICAgICAgICBUYXJnZXRPcmlnaW5JZDogJ1N0YWNrTXlEaXN0T3JpZ2luMUQ2RDVFNTM1JyxcbiAgICAgICAgICBWaWV3ZXJQcm90b2NvbFBvbGljeTogJ2FsbG93LWFsbCcsXG4gICAgICAgIH0sXG4gICAgICAgIENhY2hlQmVoYXZpb3JzOiBbe1xuICAgICAgICAgIFBhdGhQYXR0ZXJuOiAnYXBpLyonLFxuICAgICAgICAgIEZvcndhcmRlZFZhbHVlczogeyBRdWVyeVN0cmluZzogZmFsc2UgfSxcbiAgICAgICAgICBUYXJnZXRPcmlnaW5JZDogJ1N0YWNrTXlEaXN0T3JpZ2luMUQ2RDVFNTM1JyxcbiAgICAgICAgICBWaWV3ZXJQcm90b2NvbFBvbGljeTogJ2FsbG93LWFsbCcsXG4gICAgICAgIH1dLFxuICAgICAgICBFbmFibGVkOiB0cnVlLFxuICAgICAgICBIdHRwVmVyc2lvbjogJ2h0dHAyJyxcbiAgICAgICAgSVBWNkVuYWJsZWQ6IHRydWUsXG4gICAgICAgIE9yaWdpbnM6IFt7XG4gICAgICAgICAgRG9tYWluTmFtZTogJ3d3dy5leGFtcGxlLmNvbScsXG4gICAgICAgICAgSWQ6ICdTdGFja015RGlzdE9yaWdpbjFENkQ1RTUzNScsXG4gICAgICAgICAgQ3VzdG9tT3JpZ2luQ29uZmlnOiB7XG4gICAgICAgICAgICBPcmlnaW5Qcm90b2NvbFBvbGljeTogJ2h0dHBzLW9ubHknLFxuICAgICAgICAgIH0sXG4gICAgICAgIH1dLFxuICAgICAgfSxcbiAgICB9KTtcbiAgfSk7XG5cbiAgdGVzdCgnYSBzZWNvbmQgYmVoYXZpb3IgY2FuIGJlIGFkZGVkIHRvIGEgc2Vjb25kYXJ5IG9yaWdpbicsICgpID0+IHtcbiAgICBjb25zdCBvcmlnaW4gPSBkZWZhdWx0T3JpZ2luKCk7XG4gICAgY29uc3Qgb3JpZ2luMiA9IGRlZmF1bHRPcmlnaW4oJ29yaWdpbjIuZXhhbXBsZS5jb20nKTtcbiAgICBuZXcgRGlzdHJpYnV0aW9uKHN0YWNrLCAnTXlEaXN0Jywge1xuICAgICAgZGVmYXVsdEJlaGF2aW9yOiB7IG9yaWdpbiB9LFxuICAgICAgYWRkaXRpb25hbEJlaGF2aW9yczoge1xuICAgICAgICAnYXBpLyonOiB7IG9yaWdpbjogb3JpZ2luMiB9LFxuICAgICAgfSxcbiAgICB9KTtcblxuICAgIGV4cGVjdChzdGFjaykudG9IYXZlUmVzb3VyY2UoJ0FXUzo6Q2xvdWRGcm9udDo6RGlzdHJpYnV0aW9uJywge1xuICAgICAgRGlzdHJpYnV0aW9uQ29uZmlnOiB7XG4gICAgICAgIERlZmF1bHRDYWNoZUJlaGF2aW9yOiB7XG4gICAgICAgICAgRm9yd2FyZGVkVmFsdWVzOiB7IFF1ZXJ5U3RyaW5nOiBmYWxzZSB9LFxuICAgICAgICAgIFRhcmdldE9yaWdpbklkOiAnU3RhY2tNeURpc3RPcmlnaW4xRDZENUU1MzUnLFxuICAgICAgICAgIFZpZXdlclByb3RvY29sUG9saWN5OiAnYWxsb3ctYWxsJyxcbiAgICAgICAgfSxcbiAgICAgICAgQ2FjaGVCZWhhdmlvcnM6IFt7XG4gICAgICAgICAgUGF0aFBhdHRlcm46ICdhcGkvKicsXG4gICAgICAgICAgRm9yd2FyZGVkVmFsdWVzOiB7IFF1ZXJ5U3RyaW5nOiBmYWxzZSB9LFxuICAgICAgICAgIFRhcmdldE9yaWdpbklkOiAnU3RhY2tNeURpc3RPcmlnaW4yMEI5NkYzQUQnLFxuICAgICAgICAgIFZpZXdlclByb3RvY29sUG9saWN5OiAnYWxsb3ctYWxsJyxcbiAgICAgICAgfV0sXG4gICAgICAgIEVuYWJsZWQ6IHRydWUsXG4gICAgICAgIEh0dHBWZXJzaW9uOiAnaHR0cDInLFxuICAgICAgICBJUFY2RW5hYmxlZDogdHJ1ZSxcbiAgICAgICAgT3JpZ2luczogW3tcbiAgICAgICAgICBEb21haW5OYW1lOiAnd3d3LmV4YW1wbGUuY29tJyxcbiAgICAgICAgICBJZDogJ1N0YWNrTXlEaXN0T3JpZ2luMUQ2RDVFNTM1JyxcbiAgICAgICAgICBDdXN0b21PcmlnaW5Db25maWc6IHtcbiAgICAgICAgICAgIE9yaWdpblByb3RvY29sUG9saWN5OiAnaHR0cHMtb25seScsXG4gICAgICAgICAgfSxcbiAgICAgICAgfSxcbiAgICAgICAge1xuICAgICAgICAgIERvbWFpbk5hbWU6ICdvcmlnaW4yLmV4YW1wbGUuY29tJyxcbiAgICAgICAgICBJZDogJ1N0YWNrTXlEaXN0T3JpZ2luMjBCOTZGM0FEJyxcbiAgICAgICAgICBDdXN0b21PcmlnaW5Db25maWc6IHtcbiAgICAgICAgICAgIE9yaWdpblByb3RvY29sUG9saWN5OiAnaHR0cHMtb25seScsXG4gICAgICAgICAgfSxcbiAgICAgICAgfV0sXG4gICAgICB9LFxuICAgIH0pO1xuICB9KTtcblxuICB0ZXN0KCdiZWhhdmlvciBjcmVhdGlvbiBvcmRlciBpcyBwcmVzZXJ2ZWQnLCAoKSA9PiB7XG4gICAgY29uc3Qgb3JpZ2luID0gZGVmYXVsdE9yaWdpbigpO1xuICAgIGNvbnN0IG9yaWdpbjIgPSBkZWZhdWx0T3JpZ2luKCdvcmlnaW4yLmV4YW1wbGUuY29tJyk7XG4gICAgY29uc3QgZGlzdCA9IG5ldyBEaXN0cmlidXRpb24oc3RhY2ssICdNeURpc3QnLCB7XG4gICAgICBkZWZhdWx0QmVoYXZpb3I6IHsgb3JpZ2luIH0sXG4gICAgICBhZGRpdGlvbmFsQmVoYXZpb3JzOiB7XG4gICAgICAgICdhcGkvMSonOiB7IG9yaWdpbjogb3JpZ2luMiB9LFxuICAgICAgfSxcbiAgICB9KTtcbiAgICBkaXN0LmFkZEJlaGF2aW9yKCdhcGkvMionLCBvcmlnaW4pO1xuXG4gICAgZXhwZWN0KHN0YWNrKS50b0hhdmVSZXNvdXJjZSgnQVdTOjpDbG91ZEZyb250OjpEaXN0cmlidXRpb24nLCB7XG4gICAgICBEaXN0cmlidXRpb25Db25maWc6IHtcbiAgICAgICAgRGVmYXVsdENhY2hlQmVoYXZpb3I6IHtcbiAgICAgICAgICBGb3J3YXJkZWRWYWx1ZXM6IHsgUXVlcnlTdHJpbmc6IGZhbHNlIH0sXG4gICAgICAgICAgVGFyZ2V0T3JpZ2luSWQ6ICdTdGFja015RGlzdE9yaWdpbjFENkQ1RTUzNScsXG4gICAgICAgICAgVmlld2VyUHJvdG9jb2xQb2xpY3k6ICdhbGxvdy1hbGwnLFxuICAgICAgICB9LFxuICAgICAgICBDYWNoZUJlaGF2aW9yczogW3tcbiAgICAgICAgICBQYXRoUGF0dGVybjogJ2FwaS8xKicsXG4gICAgICAgICAgRm9yd2FyZGVkVmFsdWVzOiB7IFF1ZXJ5U3RyaW5nOiBmYWxzZSB9LFxuICAgICAgICAgIFRhcmdldE9yaWdpbklkOiAnU3RhY2tNeURpc3RPcmlnaW4yMEI5NkYzQUQnLFxuICAgICAgICAgIFZpZXdlclByb3RvY29sUG9saWN5OiAnYWxsb3ctYWxsJyxcbiAgICAgICAgfSxcbiAgICAgICAge1xuICAgICAgICAgIFBhdGhQYXR0ZXJuOiAnYXBpLzIqJyxcbiAgICAgICAgICBGb3J3YXJkZWRWYWx1ZXM6IHsgUXVlcnlTdHJpbmc6IGZhbHNlIH0sXG4gICAgICAgICAgVGFyZ2V0T3JpZ2luSWQ6ICdTdGFja015RGlzdE9yaWdpbjFENkQ1RTUzNScsXG4gICAgICAgICAgVmlld2VyUHJvdG9jb2xQb2xpY3k6ICdhbGxvdy1hbGwnLFxuICAgICAgICB9XSxcbiAgICAgICAgRW5hYmxlZDogdHJ1ZSxcbiAgICAgICAgSHR0cFZlcnNpb246ICdodHRwMicsXG4gICAgICAgIElQVjZFbmFibGVkOiB0cnVlLFxuICAgICAgICBPcmlnaW5zOiBbe1xuICAgICAgICAgIERvbWFpbk5hbWU6ICd3d3cuZXhhbXBsZS5jb20nLFxuICAgICAgICAgIElkOiAnU3RhY2tNeURpc3RPcmlnaW4xRDZENUU1MzUnLFxuICAgICAgICAgIEN1c3RvbU9yaWdpbkNvbmZpZzoge1xuICAgICAgICAgICAgT3JpZ2luUHJvdG9jb2xQb2xpY3k6ICdodHRwcy1vbmx5JyxcbiAgICAgICAgICB9LFxuICAgICAgICB9LFxuICAgICAgICB7XG4gICAgICAgICAgRG9tYWluTmFtZTogJ29yaWdpbjIuZXhhbXBsZS5jb20nLFxuICAgICAgICAgIElkOiAnU3RhY2tNeURpc3RPcmlnaW4yMEI5NkYzQUQnLFxuICAgICAgICAgIEN1c3RvbU9yaWdpbkNvbmZpZzoge1xuICAgICAgICAgICAgT3JpZ2luUHJvdG9jb2xQb2xpY3k6ICdodHRwcy1vbmx5JyxcbiAgICAgICAgICB9LFxuICAgICAgICB9XSxcbiAgICAgIH0sXG4gICAgfSk7XG4gIH0pO1xufSk7XG5cbmRlc2NyaWJlKCdjZXJ0aWZpY2F0ZXMnLCAoKSA9PiB7XG5cbiAgdGVzdCgnc2hvdWxkIGZhaWwgaWYgdXNpbmcgYW4gaW1wb3J0ZWQgY2VydGlmaWNhdGUgZnJvbSBvdXRzaWRlIG9mIHVzLWVhc3QtMScsICgpID0+IHtcbiAgICBjb25zdCBvcmlnaW4gPSBkZWZhdWx0T3JpZ2luKCk7XG4gICAgY29uc3QgY2VydGlmaWNhdGUgPSBhY20uQ2VydGlmaWNhdGUuZnJvbUNlcnRpZmljYXRlQXJuKHN0YWNrLCAnQ2VydCcsICdhcm46YXdzOmFjbTpldS13ZXN0LTE6MTIzNDU2Nzg5MDEyOmNlcnRpZmljYXRlLzEyMzQ1Njc4LTEyMzQtMTIzNC0xMjM0LTEyMzQ1Njc4OTAxMicpO1xuXG4gICAgZXhwZWN0KCgpID0+IHtcbiAgICAgIG5ldyBEaXN0cmlidXRpb24oc3RhY2ssICdEaXN0Jywge1xuICAgICAgICBkZWZhdWx0QmVoYXZpb3I6IHsgb3JpZ2luIH0sXG4gICAgICAgIGNlcnRpZmljYXRlLFxuICAgICAgfSk7XG4gICAgfSkudG9UaHJvdygvRGlzdHJpYnV0aW9uIGNlcnRpZmljYXRlcyBtdXN0IGJlIGluIHRoZSB1cy1lYXN0LTEgcmVnaW9uIGFuZCB0aGUgY2VydGlmaWNhdGUgeW91IHByb3ZpZGVkIGlzIGluIGV1LXdlc3QtMS4vKTtcbiAgfSk7XG5cbiAgdGVzdCgnYWRkaW5nIGEgY2VydGlmaWNhdGUgcmVuZGVycyB0aGUgY29ycmVjdCBWaWV3ZXJDZXJ0aWZpY2F0ZSBwcm9wZXJ0eScsICgpID0+IHtcbiAgICBjb25zdCBjZXJ0aWZpY2F0ZSA9IGFjbS5DZXJ0aWZpY2F0ZS5mcm9tQ2VydGlmaWNhdGVBcm4oc3RhY2ssICdDZXJ0JywgJ2Fybjphd3M6YWNtOnVzLWVhc3QtMToxMjM0NTY3ODkwMTI6Y2VydGlmaWNhdGUvMTIzNDU2NzgtMTIzNC0xMjM0LTEyMzQtMTIzNDU2Nzg5MDEyJyk7XG5cbiAgICBuZXcgRGlzdHJpYnV0aW9uKHN0YWNrLCAnRGlzdCcsIHtcbiAgICAgIGRlZmF1bHRCZWhhdmlvcjogeyBvcmlnaW46IGRlZmF1bHRPcmlnaW4oKSB9LFxuICAgICAgY2VydGlmaWNhdGUsXG4gICAgfSk7XG5cbiAgICBleHBlY3Qoc3RhY2spLnRvSGF2ZVJlc291cmNlTGlrZSgnQVdTOjpDbG91ZEZyb250OjpEaXN0cmlidXRpb24nLCB7XG4gICAgICBEaXN0cmlidXRpb25Db25maWc6IHtcbiAgICAgICAgVmlld2VyQ2VydGlmaWNhdGU6IHtcbiAgICAgICAgICBBY21DZXJ0aWZpY2F0ZUFybjogJ2Fybjphd3M6YWNtOnVzLWVhc3QtMToxMjM0NTY3ODkwMTI6Y2VydGlmaWNhdGUvMTIzNDU2NzgtMTIzNC0xMjM0LTEyMzQtMTIzNDU2Nzg5MDEyJyxcbiAgICAgICAgICBTc2xTdXBwb3J0TWV0aG9kOiAnc25pLW9ubHknLFxuICAgICAgICAgIE1pbmltdW1Qcm90b2NvbFZlcnNpb246ICdUTFN2MS4yXzIwMTgnLFxuICAgICAgICB9LFxuICAgICAgfSxcbiAgICB9KTtcbiAgfSk7XG59KTtcblxuZGVzY3JpYmUoJ2N1c3RvbSBlcnJvciByZXNwb25zZXMnLCAoKSA9PiB7XG5cbiAgdGVzdCgnc2hvdWxkIGZhaWwgaWYgcmVzcG9uc2VQYWdlUGF0aCBpcyBkZWZpbmVkIGJ1dCByZXNwb25zZUNvZGUgaXMgbm90JywgKCkgPT4ge1xuICAgIGNvbnN0IG9yaWdpbiA9IGRlZmF1bHRPcmlnaW4oKTtcblxuICAgIGV4cGVjdCgoKSA9PiB7XG4gICAgICBuZXcgRGlzdHJpYnV0aW9uKHN0YWNrLCAnRGlzdCcsIHtcbiAgICAgICAgZGVmYXVsdEJlaGF2aW9yOiB7IG9yaWdpbiB9LFxuICAgICAgICBlcnJvclJlc3BvbnNlczogW3tcbiAgICAgICAgICBodHRwU3RhdHVzOiA0MDQsXG4gICAgICAgICAgcmVzcG9uc2VQYWdlUGF0aDogJy9lcnJvcnMvNDA0Lmh0bWwnLFxuICAgICAgICB9XSxcbiAgICAgIH0pO1xuICAgIH0pLnRvVGhyb3coL1xcJ3Jlc3BvbnNlQ29kZVxcJyBtdXN0IGJlIHByb3ZpZGVkIGlmIFxcJ3Jlc3BvbnNlUGFnZVBhdGhcXCcgaXMgZGVmaW5lZC8pO1xuICB9KTtcblxuICB0ZXN0KCdzaG91bGQgZmFpbCBpZiBvbmx5IHRoZSBlcnJvciBjb2RlIGlzIHByb3ZpZGVkJywgKCkgPT4ge1xuICAgIGNvbnN0IG9yaWdpbiA9IGRlZmF1bHRPcmlnaW4oKTtcblxuICAgIGV4cGVjdCgoKSA9PiB7XG4gICAgICBuZXcgRGlzdHJpYnV0aW9uKHN0YWNrLCAnRGlzdCcsIHtcbiAgICAgICAgZGVmYXVsdEJlaGF2aW9yOiB7IG9yaWdpbiB9LFxuICAgICAgICBlcnJvclJlc3BvbnNlczogW3sgaHR0cFN0YXR1czogNDA0IH1dLFxuICAgICAgfSk7XG4gICAgfSkudG9UaHJvdygvQSBjdXN0b20gZXJyb3IgcmVzcG9uc2Ugd2l0aG91dCBlaXRoZXIgYSBcXCdyZXNwb25zZUNvZGVcXCcgb3IgXFwnZXJyb3JDYWNoaW5nTWluVHRsXFwnIGlzIG5vdCB2YWxpZC4vKTtcbiAgfSk7XG5cbiAgdGVzdCgnc2hvdWxkIHJlbmRlciB0aGUgYXJyYXkgb2YgZXJyb3IgY29uZmlncyBpZiBwcm92aWRlZCcsICgpID0+IHtcbiAgICBjb25zdCBvcmlnaW4gPSBkZWZhdWx0T3JpZ2luKCk7XG4gICAgbmV3IERpc3RyaWJ1dGlvbihzdGFjaywgJ0Rpc3QnLCB7XG4gICAgICBkZWZhdWx0QmVoYXZpb3I6IHsgb3JpZ2luIH0sXG4gICAgICBlcnJvclJlc3BvbnNlczogW3tcbiAgICAgICAgaHR0cFN0YXR1czogNDA0LFxuICAgICAgICByZXNwb25zZUh0dHBTdGF0dXM6IDQwNCxcbiAgICAgICAgcmVzcG9uc2VQYWdlUGF0aDogJy9lcnJvcnMvNDA0Lmh0bWwnLFxuICAgICAgfSxcbiAgICAgIHtcbiAgICAgICAgaHR0cFN0YXR1czogNTAwLFxuICAgICAgICB0dGw6IER1cmF0aW9uLnNlY29uZHMoMiksXG4gICAgICB9XSxcbiAgICB9KTtcblxuICAgIGV4cGVjdChzdGFjaykudG9IYXZlUmVzb3VyY2VMaWtlKCdBV1M6OkNsb3VkRnJvbnQ6OkRpc3RyaWJ1dGlvbicsIHtcbiAgICAgIERpc3RyaWJ1dGlvbkNvbmZpZzoge1xuICAgICAgICBDdXN0b21FcnJvclJlc3BvbnNlczogW1xuICAgICAgICAgIHtcbiAgICAgICAgICAgIEVycm9yQ29kZTogNDA0LFxuICAgICAgICAgICAgUmVzcG9uc2VDb2RlOiA0MDQsXG4gICAgICAgICAgICBSZXNwb25zZVBhZ2VQYXRoOiAnL2Vycm9ycy80MDQuaHRtbCcsXG4gICAgICAgICAgfSxcbiAgICAgICAgICB7XG4gICAgICAgICAgICBFcnJvckNhY2hpbmdNaW5UVEw6IDIsXG4gICAgICAgICAgICBFcnJvckNvZGU6IDUwMCxcbiAgICAgICAgICB9LFxuICAgICAgICBdLFxuICAgICAgfSxcbiAgICB9KTtcbiAgfSk7XG5cbn0pO1xuXG5kZXNjcmliZSgnbG9nZ2luZycsICgpID0+IHtcbiAgdGVzdCgnZG9lcyBub3QgaW5jbHVkZSBsb2dnaW5nIGlmIGRpc2FibGVkIGFuZCBubyBidWNrZXQgcHJvdmlkZWQnLCAoKSA9PiB7XG4gICAgY29uc3Qgb3JpZ2luID0gZGVmYXVsdE9yaWdpbigpO1xuICAgIG5ldyBEaXN0cmlidXRpb24oc3RhY2ssICdNeURpc3QnLCB7IGRlZmF1bHRCZWhhdmlvcjogeyBvcmlnaW4gfSB9KTtcblxuICAgIGV4cGVjdChzdGFjaykudG9IYXZlUmVzb3VyY2VMaWtlKCdBV1M6OkNsb3VkRnJvbnQ6OkRpc3RyaWJ1dGlvbicsIHtcbiAgICAgIERpc3RyaWJ1dGlvbkNvbmZpZzoge1xuICAgICAgICBMb2dnaW5nOiBBQlNFTlQsXG4gICAgICB9LFxuICAgIH0pO1xuICB9KTtcblxuICB0ZXN0KCd0aHJvd3MgZXJyb3IgaWYgbG9nZ2luZyBkaXNhYmxlZCBidXQgYnVja2V0IHByb3ZpZGVkJywgKCkgPT4ge1xuICAgIGNvbnN0IG9yaWdpbiA9IGRlZmF1bHRPcmlnaW4oKTtcblxuICAgIGV4cGVjdCgoKSA9PiB7XG4gICAgICBuZXcgRGlzdHJpYnV0aW9uKHN0YWNrLCAnTXlEaXN0Jywge1xuICAgICAgICBkZWZhdWx0QmVoYXZpb3I6IHsgb3JpZ2luIH0sXG4gICAgICAgIGVuYWJsZUxvZ2dpbmc6IGZhbHNlLFxuICAgICAgICBsb2dCdWNrZXQ6IG5ldyBzMy5CdWNrZXQoc3RhY2ssICdCdWNrZXQnKSxcbiAgICAgIH0pO1xuICAgIH0pLnRvVGhyb3coL0V4cGxpY2l0bHkgZGlzYWJsZWQgbG9nZ2luZyBidXQgcHJvdmlkZWQgYSBsb2dnaW5nIGJ1Y2tldC4vKTtcbiAgfSk7XG5cbiAgdGVzdCgnY3JlYXRlcyBidWNrZXQgaWYgbm9uZSBpcyBwcm92aWRlZCcsICgpID0+IHtcbiAgICBjb25zdCBvcmlnaW4gPSBkZWZhdWx0T3JpZ2luKCk7XG4gICAgbmV3IERpc3RyaWJ1dGlvbihzdGFjaywgJ015RGlzdCcsIHtcbiAgICAgIGRlZmF1bHRCZWhhdmlvcjogeyBvcmlnaW4gfSxcbiAgICAgIGVuYWJsZUxvZ2dpbmc6IHRydWUsXG4gICAgfSk7XG5cbiAgICBleHBlY3Qoc3RhY2spLnRvSGF2ZVJlc291cmNlTGlrZSgnQVdTOjpDbG91ZEZyb250OjpEaXN0cmlidXRpb24nLCB7XG4gICAgICBEaXN0cmlidXRpb25Db25maWc6IHtcbiAgICAgICAgTG9nZ2luZzoge1xuICAgICAgICAgIEJ1Y2tldDogeydGbjo6R2V0QXR0JzogWydNeURpc3RMb2dnaW5nQnVja2V0OUI4OTc2QkMnLCAnUmVnaW9uYWxEb21haW5OYW1lJ119LFxuICAgICAgICB9LFxuICAgICAgfSxcbiAgICB9KTtcbiAgfSk7XG5cbiAgdGVzdCgndXNlcyBleGlzdGluZyBidWNrZXQgaWYgcHJvdmlkZWQnLCAoKSA9PiB7XG4gICAgY29uc3Qgb3JpZ2luID0gZGVmYXVsdE9yaWdpbigpO1xuICAgIGNvbnN0IGxvZ2dpbmdCdWNrZXQgPSBuZXcgczMuQnVja2V0KHN0YWNrLCAnTXlMb2dnaW5nQnVja2V0Jyk7XG4gICAgbmV3IERpc3RyaWJ1dGlvbihzdGFjaywgJ015RGlzdCcsIHtcbiAgICAgIGRlZmF1bHRCZWhhdmlvcjogeyBvcmlnaW4gfSxcbiAgICAgIGxvZ0J1Y2tldDogbG9nZ2luZ0J1Y2tldCxcbiAgICB9KTtcblxuICAgIGV4cGVjdChzdGFjaykudG9IYXZlUmVzb3VyY2VMaWtlKCdBV1M6OkNsb3VkRnJvbnQ6OkRpc3RyaWJ1dGlvbicsIHtcbiAgICAgIERpc3RyaWJ1dGlvbkNvbmZpZzoge1xuICAgICAgICBMb2dnaW5nOiB7XG4gICAgICAgICAgQnVja2V0OiB7J0ZuOjpHZXRBdHQnOiBbJ015TG9nZ2luZ0J1Y2tldDQzODJDRDA0JywgJ1JlZ2lvbmFsRG9tYWluTmFtZSddfSxcbiAgICAgICAgfSxcbiAgICAgIH0sXG4gICAgfSk7XG4gIH0pO1xuXG4gIHRlc3QoJ2NhbiBzZXQgcHJlZml4IGFuZCBjb29raWVzJywgKCkgPT4ge1xuICAgIGNvbnN0IG9yaWdpbiA9IGRlZmF1bHRPcmlnaW4oKTtcbiAgICBuZXcgRGlzdHJpYnV0aW9uKHN0YWNrLCAnTXlEaXN0Jywge1xuICAgICAgZGVmYXVsdEJlaGF2aW9yOiB7IG9yaWdpbiB9LFxuICAgICAgZW5hYmxlTG9nZ2luZzogdHJ1ZSxcbiAgICAgIGxvZ0ZpbGVQcmVmaXg6ICdsb2dzLycsXG4gICAgICBsb2dJbmNsdWRlc0Nvb2tpZXM6IHRydWUsXG4gICAgfSk7XG5cbiAgICBleHBlY3Qoc3RhY2spLnRvSGF2ZVJlc291cmNlTGlrZSgnQVdTOjpDbG91ZEZyb250OjpEaXN0cmlidXRpb24nLCB7XG4gICAgICBEaXN0cmlidXRpb25Db25maWc6IHtcbiAgICAgICAgTG9nZ2luZzoge1xuICAgICAgICAgIEJ1Y2tldDogeydGbjo6R2V0QXR0JzogWydNeURpc3RMb2dnaW5nQnVja2V0OUI4OTc2QkMnLCAnUmVnaW9uYWxEb21haW5OYW1lJ119LFxuICAgICAgICAgIEluY2x1ZGVDb29raWVzOiB0cnVlLFxuICAgICAgICAgIFByZWZpeDogJ2xvZ3MvJyxcbiAgICAgICAgfSxcbiAgICAgIH0sXG4gICAgfSk7XG4gIH0pO1xufSk7XG5cbmRlc2NyaWJlKCd3aXRoIExhbWJkYUBFZGdlIGZ1bmN0aW9ucycsICgpID0+IHtcbiAgbGV0IGxhbWJkYUZ1bmN0aW9uOiBsYW1iZGEuRnVuY3Rpb247XG4gIGxldCBvcmlnaW46IElPcmlnaW47XG5cbiAgYmVmb3JlRWFjaCgoKSA9PiB7XG4gICAgbGFtYmRhRnVuY3Rpb24gPSBuZXcgbGFtYmRhLkZ1bmN0aW9uKHN0YWNrLCAnRnVuY3Rpb24nLCB7XG4gICAgICBydW50aW1lOiBsYW1iZGEuUnVudGltZS5OT0RFSlMsXG4gICAgICBjb2RlOiBsYW1iZGEuQ29kZS5mcm9tSW5saW5lKCd3aGF0ZXZlcicpLFxuICAgICAgaGFuZGxlcjogJ2luZGV4LmhhbmRsZXInLFxuICAgIH0pO1xuXG4gICAgb3JpZ2luID0gZGVmYXVsdE9yaWdpbigpO1xuICB9KTtcblxuICB0ZXN0KCdjYW4gYWRkIGFuIGVkZ2UgbGFtYmRhcyB0byB0aGUgZGVmYXVsdCBiZWhhdmlvcicsICgpID0+IHtcbiAgICBuZXcgRGlzdHJpYnV0aW9uKHN0YWNrLCAnTXlEaXN0Jywge1xuICAgICAgZGVmYXVsdEJlaGF2aW9yOiB7XG4gICAgICAgIG9yaWdpbixcbiAgICAgICAgZWRnZUxhbWJkYXM6IFtcbiAgICAgICAgICB7XG4gICAgICAgICAgICBmdW5jdGlvblZlcnNpb246IGxhbWJkYUZ1bmN0aW9uLmN1cnJlbnRWZXJzaW9uLFxuICAgICAgICAgICAgZXZlbnRUeXBlOiBMYW1iZGFFZGdlRXZlbnRUeXBlLk9SSUdJTl9SRVFVRVNULFxuICAgICAgICAgIH0sXG4gICAgICAgIF0sXG4gICAgICB9LFxuICAgIH0pO1xuXG4gICAgZXhwZWN0KHN0YWNrKS50b0hhdmVSZXNvdXJjZUxpa2UoJ0FXUzo6Q2xvdWRGcm9udDo6RGlzdHJpYnV0aW9uJywge1xuICAgICAgRGlzdHJpYnV0aW9uQ29uZmlnOiB7XG4gICAgICAgIERlZmF1bHRDYWNoZUJlaGF2aW9yOiB7XG4gICAgICAgICAgTGFtYmRhRnVuY3Rpb25Bc3NvY2lhdGlvbnM6IFtcbiAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgRXZlbnRUeXBlOiAnb3JpZ2luLXJlcXVlc3QnLFxuICAgICAgICAgICAgICBMYW1iZGFGdW5jdGlvbkFSTjoge1xuICAgICAgICAgICAgICAgIFJlZjogJ0Z1bmN0aW9uQ3VycmVudFZlcnNpb240RTJCMjI2MTQ3N2E1YWU4MDU5YmJhYTc4MTNmNzUyMjkyYzBmNjVlJyxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgXSxcbiAgICAgICAgfSxcbiAgICAgIH0sXG4gICAgfSk7XG4gIH0pO1xuXG4gIHRlc3QoJ2NhbiBhZGQgYW4gZWRnZSBsYW1iZGFzIHRvIGFkZGl0aW9uYWwgYmVoYXZpb3JzJywgKCkgPT4ge1xuICAgIG5ldyBEaXN0cmlidXRpb24oc3RhY2ssICdNeURpc3QnLCB7XG4gICAgICBkZWZhdWx0QmVoYXZpb3I6IHsgb3JpZ2luIH0sXG4gICAgICBhZGRpdGlvbmFsQmVoYXZpb3JzOiB7XG4gICAgICAgICdpbWFnZXMvKic6IHtcbiAgICAgICAgICBvcmlnaW4sXG4gICAgICAgICAgZWRnZUxhbWJkYXM6IFtcbiAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgZnVuY3Rpb25WZXJzaW9uOiBsYW1iZGFGdW5jdGlvbi5jdXJyZW50VmVyc2lvbixcbiAgICAgICAgICAgICAgZXZlbnRUeXBlOiBMYW1iZGFFZGdlRXZlbnRUeXBlLlZJRVdFUl9SRVFVRVNULFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICBdLFxuICAgICAgICB9LFxuICAgICAgfSxcbiAgICB9KTtcblxuICAgIGV4cGVjdChzdGFjaykudG9IYXZlUmVzb3VyY2VMaWtlKCdBV1M6OkNsb3VkRnJvbnQ6OkRpc3RyaWJ1dGlvbicsIHtcbiAgICAgIERpc3RyaWJ1dGlvbkNvbmZpZzoge1xuICAgICAgICBDYWNoZUJlaGF2aW9yczogW1xuICAgICAgICAgIHtcbiAgICAgICAgICAgIFBhdGhQYXR0ZXJuOiAnaW1hZ2VzLyonLFxuICAgICAgICAgICAgTGFtYmRhRnVuY3Rpb25Bc3NvY2lhdGlvbnM6IFtcbiAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgIEV2ZW50VHlwZTogJ3ZpZXdlci1yZXF1ZXN0JyxcbiAgICAgICAgICAgICAgICBMYW1iZGFGdW5jdGlvbkFSTjoge1xuICAgICAgICAgICAgICAgICAgUmVmOiAnRnVuY3Rpb25DdXJyZW50VmVyc2lvbjRFMkIyMjYxNDc3YTVhZTgwNTliYmFhNzgxM2Y3NTIyOTJjMGY2NWUnLFxuICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICBdLFxuICAgICAgICAgIH0sXG4gICAgICAgIF0sXG4gICAgICB9LFxuICAgIH0pO1xuICB9KTtcblxuICB0ZXN0KCdmYWlscyBjcmVhdGlvbiB3aGVuIGF0dGVtcHRpbmcgdG8gYWRkIHRoZSAkTEFURVNUIGZ1bmN0aW9uIHZlcnNpb24gYXMgYW4gZWRnZSBMYW1iZGEgdG8gdGhlIGRlZmF1bHQgYmVoYXZpb3InLCAoKSA9PiB7XG4gICAgZXhwZWN0KCgpID0+IHtcbiAgICAgIG5ldyBEaXN0cmlidXRpb24oc3RhY2ssICdNeURpc3QnLCB7XG4gICAgICAgIGRlZmF1bHRCZWhhdmlvcjoge1xuICAgICAgICAgIG9yaWdpbixcbiAgICAgICAgICBlZGdlTGFtYmRhczogW1xuICAgICAgICAgICAge1xuICAgICAgICAgICAgICBmdW5jdGlvblZlcnNpb246IGxhbWJkYUZ1bmN0aW9uLmxhdGVzdFZlcnNpb24sXG4gICAgICAgICAgICAgIGV2ZW50VHlwZTogTGFtYmRhRWRnZUV2ZW50VHlwZS5PUklHSU5fUkVTUE9OU0UsXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIF0sXG4gICAgICAgIH0sXG4gICAgICB9KTtcbiAgICB9KS50b1Rocm93KC9cXCRMQVRFU1QgZnVuY3Rpb24gdmVyc2lvbiBjYW5ub3QgYmUgdXNlZCBmb3IgTGFtYmRhQEVkZ2UvKTtcbiAgfSk7XG59KTtcblxudGVzdCgncHJpY2UgY2xhc3MgaXMgaW5jbHVkZWQgaWYgcHJvdmlkZWQnLCAoKSA9PiB7XG4gIGNvbnN0IG9yaWdpbiA9IGRlZmF1bHRPcmlnaW4oKTtcbiAgbmV3IERpc3RyaWJ1dGlvbihzdGFjaywgJ0Rpc3QnLCB7XG4gICAgZGVmYXVsdEJlaGF2aW9yOiB7IG9yaWdpbiB9LFxuICAgIHByaWNlQ2xhc3M6IFByaWNlQ2xhc3MuUFJJQ0VfQ0xBU1NfMjAwLFxuICB9KTtcblxuICBleHBlY3Qoc3RhY2spLnRvSGF2ZVJlc291cmNlTGlrZSgnQVdTOjpDbG91ZEZyb250OjpEaXN0cmlidXRpb24nLCB7XG4gICAgRGlzdHJpYnV0aW9uQ29uZmlnOiB7XG4gICAgICBQcmljZUNsYXNzOiAnUHJpY2VDbGFzc18yMDAnLFxuICAgIH0sXG4gIH0pO1xufSk7XG5cbnRlc3QoJ2VzY2FwZSBoYXRjaGVzIGFyZSBzdXBwb3J0ZWQnLCAoKSA9PiB7XG4gIGNvbnN0IGRpc3QgPSBuZXcgRGlzdHJpYnV0aW9uKHN0YWNrLCAnRGlzdCcsIHtcbiAgICBkZWZhdWx0QmVoYXZpb3I6IHsgb3JpZ2luOiBkZWZhdWx0T3JpZ2luIH0sXG4gIH0pO1xuICBjb25zdCBjZm5EaXN0ID0gZGlzdC5ub2RlLmRlZmF1bHRDaGlsZCBhcyBDZm5EaXN0cmlidXRpb247XG4gIGNmbkRpc3QuYWRkUHJvcGVydHlPdmVycmlkZSgnRGlzdHJpYnV0aW9uQ29uZmlnLkRlZmF1bHRDYWNoZUJlaGF2aW9yLkZvcndhcmRlZFZhbHVlcy5IZWFkZXJzJywgWycqJ10pO1xuXG4gIGV4cGVjdChzdGFjaykudG9IYXZlUmVzb3VyY2VMaWtlKCdBV1M6OkNsb3VkRnJvbnQ6OkRpc3RyaWJ1dGlvbicsIHtcbiAgICBEaXN0cmlidXRpb25Db25maWc6IHtcbiAgICAgIERlZmF1bHRDYWNoZUJlaGF2aW9yOiB7XG4gICAgICAgIEZvcndhcmRlZFZhbHVlczoge1xuICAgICAgICAgIEhlYWRlcnM6IFsnKiddLFxuICAgICAgICB9LFxuICAgICAgfSxcbiAgICB9LFxuICB9KTtcbn0pO1xuIl19