"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LambdaEdgeEventType = exports.CachedMethods = exports.AllowedMethods = exports.SecurityPolicyProtocol = exports.SSLMethod = exports.OriginProtocolPolicy = exports.ViewerProtocolPolicy = exports.PriceClass = exports.HttpVersion = exports.Distribution = void 0;
const s3 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const cloudfront_generated_1 = require("./cloudfront.generated");
const cache_behavior_1 = require("./private/cache-behavior");
/**
 * A CloudFront distribution with associated origin(s) and caching behavior(s).
 *
 * @experimental
 */
class Distribution extends core_1.Resource {
    constructor(scope, id, props) {
        var _a, _b, _c, _d, _e;
        super(scope, id);
        this.additionalBehaviors = [];
        this.boundOrigins = [];
        this.originGroups = [];
        if (props.certificate) {
            const certificateRegion = core_1.Stack.of(this).parseArn(props.certificate.certificateArn).region;
            if (!core_1.Token.isUnresolved(certificateRegion) && certificateRegion !== 'us-east-1') {
                throw new Error(`Distribution certificates must be in the us-east-1 region and the certificate you provided is in ${certificateRegion}.`);
            }
        }
        const originId = this.addOrigin(props.defaultBehavior.origin);
        this.defaultBehavior = new cache_behavior_1.CacheBehavior(originId, { pathPattern: '*', ...props.defaultBehavior });
        if (props.additionalBehaviors) {
            Object.entries(props.additionalBehaviors).forEach(([pathPattern, behaviorOptions]) => {
                this.addBehavior(pathPattern, behaviorOptions.origin, behaviorOptions);
            });
        }
        this.certificate = props.certificate;
        this.errorResponses = (_a = props.errorResponses) !== null && _a !== void 0 ? _a : [];
        const distribution = new cloudfront_generated_1.CfnDistribution(this, 'Resource', { distributionConfig: {
                enabled: (_b = props.enabled) !== null && _b !== void 0 ? _b : true,
                origins: core_1.Lazy.anyValue({ produce: () => this.renderOrigins() }),
                originGroups: core_1.Lazy.anyValue({ produce: () => this.renderOriginGroups() }),
                defaultCacheBehavior: this.defaultBehavior._renderBehavior(),
                cacheBehaviors: core_1.Lazy.anyValue({ produce: () => this.renderCacheBehaviors() }),
                comment: props.comment,
                customErrorResponses: this.renderErrorResponses(),
                defaultRootObject: props.defaultRootObject,
                httpVersion: (_c = props.httpVersion) !== null && _c !== void 0 ? _c : HttpVersion.HTTP2,
                ipv6Enabled: (_d = props.enableIpv6) !== null && _d !== void 0 ? _d : true,
                logging: this.renderLogging(props),
                priceClass: (_e = props.priceClass) !== null && _e !== void 0 ? _e : undefined,
                restrictions: this.renderRestrictions(props.geoRestriction),
                viewerCertificate: this.certificate ? this.renderViewerCertificate(this.certificate) : undefined,
            } });
        this.domainName = distribution.attrDomainName;
        this.distributionDomainName = distribution.attrDomainName;
        this.distributionId = distribution.ref;
    }
    /**
     * Creates a Distribution construct that represents an external (imported) distribution.
     */
    static fromDistributionAttributes(scope, id, attrs) {
        return new class extends core_1.Resource {
            constructor() {
                super(scope, id);
                this.domainName = attrs.domainName;
                this.distributionDomainName = attrs.domainName;
                this.distributionId = attrs.distributionId;
            }
        }();
    }
    /**
     * Adds a new behavior to this distribution for the given pathPattern.
     *
     * @param pathPattern the path pattern (e.g., 'images/*') that specifies which requests to apply the behavior to.
     * @param origin the origin to use for this behavior
     * @param behaviorOptions the options for the behavior at this path.
     */
    addBehavior(pathPattern, origin, behaviorOptions = {}) {
        if (pathPattern === '*') {
            throw new Error('Only the default behavior can have a path pattern of \'*\'');
        }
        const originId = this.addOrigin(origin);
        this.additionalBehaviors.push(new cache_behavior_1.CacheBehavior(originId, { pathPattern, ...behaviorOptions }));
    }
    addOrigin(origin, isFailoverOrigin = false) {
        var _a;
        const existingOrigin = this.boundOrigins.find(boundOrigin => boundOrigin.origin === origin);
        if (existingOrigin) {
            return (_a = existingOrigin.originGroupId) !== null && _a !== void 0 ? _a : existingOrigin.originId;
        }
        else {
            const originIndex = this.boundOrigins.length + 1;
            const scope = new core_1.Construct(this, `Origin${originIndex}`);
            const originId = scope.node.uniqueId;
            const originBindConfig = origin.bind(scope, { originId });
            if (!originBindConfig.failoverConfig) {
                this.boundOrigins.push({ origin, originId, ...originBindConfig });
            }
            else {
                if (isFailoverOrigin) {
                    throw new Error('An Origin cannot use an Origin with its own failover configuration as its fallback origin!');
                }
                const groupIndex = this.originGroups.length + 1;
                const originGroupId = new core_1.Construct(this, `OriginGroup${groupIndex}`).node.uniqueId;
                this.boundOrigins.push({ origin, originId, originGroupId, ...originBindConfig });
                const failoverOriginId = this.addOrigin(originBindConfig.failoverConfig.failoverOrigin, true);
                this.addOriginGroup(originGroupId, originBindConfig.failoverConfig.statusCodes, originId, failoverOriginId);
                return originGroupId;
            }
            return originId;
        }
    }
    addOriginGroup(originGroupId, statusCodes, originId, failoverOriginId) {
        statusCodes = statusCodes !== null && statusCodes !== void 0 ? statusCodes : [500, 502, 503, 504];
        if (statusCodes.length === 0) {
            throw new Error('fallbackStatusCodes cannot be empty');
        }
        this.originGroups.push({
            failoverCriteria: {
                statusCodes: {
                    items: statusCodes,
                    quantity: statusCodes.length,
                },
            },
            id: originGroupId,
            members: {
                items: [
                    { originId },
                    { originId: failoverOriginId },
                ],
                quantity: 2,
            },
        });
    }
    renderOrigins() {
        const renderedOrigins = [];
        this.boundOrigins.forEach(boundOrigin => {
            if (boundOrigin.originProperty) {
                renderedOrigins.push(boundOrigin.originProperty);
            }
        });
        return renderedOrigins;
    }
    renderOriginGroups() {
        return this.originGroups.length === 0
            ? undefined
            : {
                items: this.originGroups,
                quantity: this.originGroups.length,
            };
    }
    renderCacheBehaviors() {
        if (this.additionalBehaviors.length === 0) {
            return undefined;
        }
        return this.additionalBehaviors.map(behavior => behavior._renderBehavior());
    }
    renderErrorResponses() {
        if (this.errorResponses.length === 0) {
            return undefined;
        }
        function validateCustomErrorResponse(errorResponse) {
            if (errorResponse.responsePagePath && !errorResponse.responseHttpStatus) {
                throw new Error('\'responseCode\' must be provided if \'responsePagePath\' is defined');
            }
            if (!errorResponse.responseHttpStatus && !errorResponse.ttl) {
                throw new Error('A custom error response without either a \'responseCode\' or \'errorCachingMinTtl\' is not valid.');
            }
        }
        this.errorResponses.forEach(e => validateCustomErrorResponse(e));
        return this.errorResponses.map(errorConfig => {
            var _a;
            return {
                errorCachingMinTtl: (_a = errorConfig.ttl) === null || _a === void 0 ? void 0 : _a.toSeconds(),
                errorCode: errorConfig.httpStatus,
                responseCode: errorConfig.responseHttpStatus,
                responsePagePath: errorConfig.responsePagePath,
            };
        });
    }
    renderLogging(props) {
        var _a;
        if (!props.enableLogging && !props.logBucket) {
            return undefined;
        }
        if (props.enableLogging === false && props.logBucket) {
            throw new Error('Explicitly disabled logging but provided a logging bucket.');
        }
        const bucket = (_a = props.logBucket) !== null && _a !== void 0 ? _a : new s3.Bucket(this, 'LoggingBucket');
        return {
            bucket: bucket.bucketRegionalDomainName,
            includeCookies: props.logIncludesCookies,
            prefix: props.logFilePrefix,
        };
    }
    renderRestrictions(geoRestriction) {
        return geoRestriction ? {
            geoRestriction: {
                restrictionType: geoRestriction.restrictionType,
                locations: geoRestriction.locations,
            },
        } : undefined;
    }
    renderViewerCertificate(certificate) {
        return {
            acmCertificateArn: certificate.certificateArn,
            sslSupportMethod: SSLMethod.SNI,
            minimumProtocolVersion: SecurityPolicyProtocol.TLS_V1_2_2018,
        };
    }
}
exports.Distribution = Distribution;
/** Maximum HTTP version to support */
var HttpVersion;
(function (HttpVersion) {
    /** HTTP 1.1 */
    HttpVersion["HTTP1_1"] = "http1.1";
    /** HTTP 2 */
    HttpVersion["HTTP2"] = "http2";
})(HttpVersion = exports.HttpVersion || (exports.HttpVersion = {}));
/**
 * The price class determines how many edge locations CloudFront will use for your distribution.
 * See https://aws.amazon.com/cloudfront/pricing/ for full list of supported regions.
 */
var PriceClass;
(function (PriceClass) {
    /** USA, Canada, Europe, & Israel */
    PriceClass["PRICE_CLASS_100"] = "PriceClass_100";
    /** PRICE_CLASS_100 + South Africa, Kenya, Middle East, Japan, Singapore, South Korea, Taiwan, Hong Kong, & Philippines */
    PriceClass["PRICE_CLASS_200"] = "PriceClass_200";
    /** All locations */
    PriceClass["PRICE_CLASS_ALL"] = "PriceClass_All";
})(PriceClass = exports.PriceClass || (exports.PriceClass = {}));
/**
 * How HTTPs should be handled with your distribution.
 */
var ViewerProtocolPolicy;
(function (ViewerProtocolPolicy) {
    /** HTTPS only */
    ViewerProtocolPolicy["HTTPS_ONLY"] = "https-only";
    /** Will redirect HTTP requests to HTTPS */
    ViewerProtocolPolicy["REDIRECT_TO_HTTPS"] = "redirect-to-https";
    /** Both HTTP and HTTPS supported */
    ViewerProtocolPolicy["ALLOW_ALL"] = "allow-all";
})(ViewerProtocolPolicy = exports.ViewerProtocolPolicy || (exports.ViewerProtocolPolicy = {}));
/**
 * Defines what protocols CloudFront will use to connect to an origin.
 */
var OriginProtocolPolicy;
(function (OriginProtocolPolicy) {
    /** Connect on HTTP only */
    OriginProtocolPolicy["HTTP_ONLY"] = "http-only";
    /** Connect with the same protocol as the viewer */
    OriginProtocolPolicy["MATCH_VIEWER"] = "match-viewer";
    /** Connect on HTTPS only */
    OriginProtocolPolicy["HTTPS_ONLY"] = "https-only";
})(OriginProtocolPolicy = exports.OriginProtocolPolicy || (exports.OriginProtocolPolicy = {}));
/**
 * The SSL method CloudFront will use for your distribution.
 *
 * Server Name Indication (SNI) - is an extension to the TLS computer networking protocol by which a client indicates
 *  which hostname it is attempting to connect to at the start of the handshaking process. This allows a server to present
 *  multiple certificates on the same IP address and TCP port number and hence allows multiple secure (HTTPS) websites
 * (or any other service over TLS) to be served by the same IP address without requiring all those sites to use the same certificate.
 *
 * CloudFront can use SNI to host multiple distributions on the same IP - which a large majority of clients will support.
 *
 * If your clients cannot support SNI however - CloudFront can use dedicated IPs for your distribution - but there is a prorated monthly charge for
 * using this feature. By default, we use SNI - but you can optionally enable dedicated IPs (VIP).
 *
 * See the CloudFront SSL for more details about pricing : https://aws.amazon.com/cloudfront/custom-ssl-domains/
 *
 */
var SSLMethod;
(function (SSLMethod) {
    SSLMethod["SNI"] = "sni-only";
    SSLMethod["VIP"] = "vip";
})(SSLMethod = exports.SSLMethod || (exports.SSLMethod = {}));
/**
 * The minimum version of the SSL protocol that you want CloudFront to use for HTTPS connections.
 * CloudFront serves your objects only to browsers or devices that support at least the SSL version that you specify.
 */
var SecurityPolicyProtocol;
(function (SecurityPolicyProtocol) {
    SecurityPolicyProtocol["SSL_V3"] = "SSLv3";
    SecurityPolicyProtocol["TLS_V1"] = "TLSv1";
    SecurityPolicyProtocol["TLS_V1_2016"] = "TLSv1_2016";
    SecurityPolicyProtocol["TLS_V1_1_2016"] = "TLSv1.1_2016";
    SecurityPolicyProtocol["TLS_V1_2_2018"] = "TLSv1.2_2018";
})(SecurityPolicyProtocol = exports.SecurityPolicyProtocol || (exports.SecurityPolicyProtocol = {}));
/**
 * The HTTP methods that the Behavior will accept requests on.
 */
class AllowedMethods {
    constructor(methods) { this.methods = methods; }
}
exports.AllowedMethods = AllowedMethods;
/** HEAD and GET */
AllowedMethods.ALLOW_GET_HEAD = new AllowedMethods(['GET', 'HEAD']);
/** HEAD, GET, and OPTIONS */
AllowedMethods.ALLOW_GET_HEAD_OPTIONS = new AllowedMethods(['GET', 'HEAD', 'OPTIONS']);
/** All supported HTTP methods */
AllowedMethods.ALLOW_ALL = new AllowedMethods(['GET', 'HEAD', 'OPTIONS', 'PUT', 'PATCH', 'POST', 'DELETE']);
/**
 * The HTTP methods that the Behavior will cache requests on.
 */
class CachedMethods {
    constructor(methods) { this.methods = methods; }
}
exports.CachedMethods = CachedMethods;
/** HEAD and GET */
CachedMethods.CACHE_GET_HEAD = new CachedMethods(['GET', 'HEAD']);
/** HEAD, GET, and OPTIONS */
CachedMethods.CACHE_GET_HEAD_OPTIONS = new CachedMethods(['GET', 'HEAD', 'OPTIONS']);
/**
 * The type of events that a Lambda@Edge function can be invoked in response to.
 */
var LambdaEdgeEventType;
(function (LambdaEdgeEventType) {
    /**
     * The origin-request specifies the request to the
     * origin location (e.g. S3)
     */
    LambdaEdgeEventType["ORIGIN_REQUEST"] = "origin-request";
    /**
     * The origin-response specifies the response from the
     * origin location (e.g. S3)
     */
    LambdaEdgeEventType["ORIGIN_RESPONSE"] = "origin-response";
    /**
     * The viewer-request specifies the incoming request
     */
    LambdaEdgeEventType["VIEWER_REQUEST"] = "viewer-request";
    /**
     * The viewer-response specifies the outgoing reponse
     */
    LambdaEdgeEventType["VIEWER_RESPONSE"] = "viewer-response";
})(LambdaEdgeEventType = exports.LambdaEdgeEventType || (exports.LambdaEdgeEventType = {}));
//# sourceMappingURL=data:application/json;base64,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