import os
from . import dirs
from . import col
from . import env
from . import utils
import shlex

def _write_targets(fh,targets):
    cmdpat='\t${P}%s\n'
    for t in targets:
        tn=t.name
        if '/' not in t.name:
            tn='${TRG}/%s'%tn
        ll=['%s:'%tn]
        for dn in t.deps:
            ll+=[dn if '/' in dn else '${TRG}/%s'%dn]
        fh.write('%s\n'%' '.join(ll))
        for c in t.code:
            fh.write(cmdpat%c)
        if '/' not in t.name:
            fh.write(cmdpat%('touch %s'%tn))
        if False and not t.contains_message:
            fh.write('\t${P}echo Target %s done\n\n'%t.name)
        else:
            fh.write('\n')

def ldflags_to_rpath_dirs(ldflags):
    res=[]
    if not ldflags:
        return res
    #NB: CMAKE_PLATFORM_IMPLICIT_LINK_DIRECTORIES does not always include all
    #relevant dirs, so we hardcode a list here:
    ignore=set(['/lib','/lib32','/lib64','/usr/lib','/usr/lib32','/usr/lib64'])
    for e in utils.shlex_split(ldflags):
        #Get dirs from -L/some/dir:
        if e.startswith('-L'):
            e=e[2:]
        #Get dirs from /some/dir/with_a_file.so:
        elif ( e and os.path.exists(e) and
               ( e.endswith('.so')
                 or e.endswith('.dylib')
                 or '.so.' in os.path.basename(e) ) ):
            e = os.path.dirname(e)
        if e and os.path.isdir(e):
            e = os.path.abspath(os.path.realpath(e))
            if e not in res and e not in ignore:
                res += [e]
    return res

_makefile_global_vars_cache = [None]
def get_makefile_global_vars():
    if _makefile_global_vars_cache[0] is None:
        from . import error
        error.error('get_makefile_global_vars called before makefile generated')
    return _makefile_global_vars_cache[0]

def write_main(global_targets,enabled_pkgnames,fh=None):
    if fh is None:
        with (dirs.makefiledir / 'Makefile').open('wt') as thefh:
            return write_main(global_targets,enabled_pkgnames,thefh)

    #the main makefile
    mf_var_cache = []
    _makefile_global_vars_cache[0] = mf_var_cache
    def _add_var(k,v):
        mf_var_cache.append( (str(k),str(v)) )
        fh.write(f'{k} := {v}\n')

    #global variables:
    fh.write('#Autogenerated file -- changes will be lost\n')
    fh.write('VERBOSE = 0\n\n')#can be overridden from cmd line
    fh.write('ifeq ($(VERBOSE), 1)\n')
    fh.write('  P := \n')#dont silence command printing
    fh.write('else\n')
    fh.write('  P := @\n')#silence command printing
    fh.write('endif\n\n')
    _add_var('MDIR',dirs.makefiledir)
    _add_var('INST',dirs.installdir)
    _add_var('PKG',dirs.pkgdirbase)
    #_add_var('SRC',dirs.codedir)
    _add_var('BLD',dirs.blddir)
    _add_var('INC','${INST}/%s'%dirs.incdirname)
    _add_var('TRG','${BLD}/named_targets')
    _add_var('EXT','${BLD}/extdeps')
    _add_var('LANG','${BLD}/langs')

    #Ensure the named_targets dir is created before any targets are triggered
    #(do not use _add_var for this, it is not a variable to be probed):
    fh.write('\nCHEAT_RUN0 := $(shell mkdir -p ${BLD}/named_targets)\n\n')

    for cname in [c for c in dir(col) if c.startswith('bldmsg_')]:
        _add_var('COL_%s'%cname.upper(),'"%s"'%getattr(col,cname))
    _add_var('COL_END',col.end)

    extdeps=env.env['extdeps']
    for extdep,info in extdeps.items():
        if info['present']:
            _add_var('CFLAGS_CXX_%s'%extdep,info['cflags_cxx'])
            _add_var('CFLAGS_C_%s'%extdep,info['cflags_c'])
            ldflags=info['ldflags']
            _add_var('LDFLAGS_%s'%extdep,ldflags)
            rpathdirs=[]
            if ldflags:
                rpathdirs = ldflags_to_rpath_dirs(info['ldflags'])
            for lang,info in env.env['system']['langs'].items():
                if info:#info is only available for available languages:
                    #TODO: join->shlex.join??
                    _existing_flags = shlex.split(ldflags)
                    for _bt in ('exe','lib'):
                        _l = [ info[f'rpath_flag_{_bt}']%shlex.quote(e)
                               for e in rpathdirs ]
                        _add_var('LDFLAGS_%s_%s_%s'%(extdep,_bt.upper(),lang),
                                ' '.join([e for e in _l
                                           if e not in _existing_flags]))

    sysgen = env.env['system']['general']
    fh.write('\n')
    _add_var('PYBIND11_EMBED_CFLAGS',
             sysgen['pybind11_embed_cflags_list'].replace(';',' '))
    _add_var('PYBIND11_EMBED_LDFLAGS',
             sysgen['pybind11_embed_linkflags_list'].replace(';',' '))
    _add_var('PYBIND11_MODULE_CFLAGS',
             sysgen['pybind11_module_cflags_list'].replace(';',' '))
    _add_var('PYBIND11_MODULE_LDFLAGS',
             sysgen['pybind11_module_linkflags_list'].replace(';',' '))
    fh.write('\n')

    block_cflags_for_exe = []
    block_cflags_for_shlib = ['-pie','-fPIE']
    def remove_flags(orig,blocked_flags):
        return ' '.join(e for e in shlex.split(orig) if e not in blocked_flags)

    for lang,info in env.env['system']['langs'].items():
        if info:#info is only available for available languages:
            _c, _l = info['cflags'], info['ldflags']
            _c_exe = remove_flags( _c, block_cflags_for_exe )
            _c_shlib = remove_flags( _c, block_cflags_for_shlib )
            _add_var('CFLAGSLANG_SHLIB_%s'%lang,_c_shlib)
            _add_var('CFLAGSLANG_EXE_%s'%lang,_c_exe)
            _add_var('LDFLAGSLANG_%s'%lang,_l)
            _add_var('LDFLAGSPREPENDLANG_%s'%lang,info['ldflags_prepend'])

    fh.write('\n')

    fh.write('ifeq ($(VERBOSE), -1)\n')
    fh.write('else\n')
    fh.write('  ifeq ($(COL_END), "")\n')
    fh.write('    $(info Build started)\n')
    fh.write('  else\n')
    fh.write('    $(info [94mBuild started[0m)\n')
    fh.write('  endif\n')
    fh.write('endif\n')

    #disable implicit rules
    fh.write('.SUFFIXES:\n\n')

    #Default target gathers up all pkg targets:
    fh.write('.PHONY: all\n\n')

    _tmp = ' '.join( ['all:']
                     + ['${TRG}/%s'%e for e in enabled_pkgnames]
                     + ['${TRG}/%s'%t.name for t in global_targets] )
    fh.write('%s\n\t@if [ ${VERBOSE} -ge 0 ]; then echo "%sAll done%s"; fi\n\n'%(
        _tmp,
        col.bldcol('global'),
        col.bldend))

    #Include package makefiles:
    for p in enabled_pkgnames:
        fh.write('include ${MDIR}/Makefile_%s.make\n'%p)

    fh.write('\n')
    #all other targets:
    _write_targets(fh,global_targets)

def write_pkg(pkg):
    assert pkg.targets
    with dirs.makefiledir.joinpath('Makefile_%s.make'%pkg.name).open('wt') as fh:
        #disable implicit rules
        fh.write('.SUFFIXES:\n\n')
        _write_targets(fh,pkg.targets)

