# (C) Copyright 2005-2021 Enthought, Inc., Austin, TX
# All rights reserved.
#
# This software is provided without warranty under the terms of the BSD
# license included in LICENSE.txt and may be redistributed only under
# the conditions described in the aforementioned license. The license
# is also available online at http://www.enthought.com/licenses/BSD.txt
#
# Thanks for using Enthought open source!
#
# Portions of this software are:
# Copyright (c) 2002-2004 John D. Hunter

"""
A collection of pre-defined colormap generator functions.

Each of the functions can be called with *min_value* and *max_value* parameters.
In that case, they produce a Colormap which linearly maps over the specified
range and has the color palette indicated by the function name.
"""

from numpy import array

# Local imports.
from .color_mapper import ColorMapper
from .discrete_color_mapper import DiscreteColorMapper
from .colormap_generators import (
    generate_cubehelix_palette,
    generate_diverging_palette,
)

# The colormaps will be added to this at the end of the file.
__all__ = [
    "reverse",
    "center",
    "color_map_functions",
    "color_map_dict",
    "color_map_name_dict",
]


# Utility functions.


def reverse(func):
    """Modify a colormap factory to reverse the color sequence

    Parameters
    ----------
    func : callable
        A colormap factory function like those provided in this module.

    Returns
    -------
    cmap : callable
        A wrapper factory function that can be used in place of the original
        factory function.
    """

    def cmap(range, **traits):
        cm = func(range, **traits)
        cm.reverse_colormap()
        return cm

    # Look a little like the wrapped function.
    cmap.__name__ = "reversed_" + func.__name__
    if func.__doc__ is not None:
        cmap.__doc__ = "Reversed: " + func.__doc__
    else:
        cmap.__doc__ = "Reversed: " + func.__name__
    return cmap


def center(func, center=0.0):
    """Modify the range of a colormap to be centered around the given value.

    For example, when passed a DataRange1D(low=-0.5, high=1.0), a colormap would
    usually have its lowest color at -0.5 and its highest at 1.0. Some colormaps
    are designed such that the middle color is special. Using this modifier, the
    example range would be modified to -1.0 and 1.0 to make 0.0 correspond with
    the middle color.

    Parameters
    ----------
    func : callable
        A colormap factory function like those provided in this module.
    center : float, optional
        The center value in dataspace.

    Returns
    -------
    cmap : callable
        A wrapper factory function that can be used in place of the original
        factory function.
    """

    def cmap(range, **traits):
        maxdev = max(abs(range.low - center), abs(range.high - center))
        range = range.clone_traits()
        range.low = center - maxdev
        range.high = center + maxdev
        return func(range, **traits)

    # Look a little like the wrapped function.
    cmap.__name__ = "centered_" + func.__name__
    if func.__doc__ is not None:
        cmap.__doc__ = "Centered: " + func.__doc__
    else:
        cmap.__doc__ = "Centered: " + func.__name__
    return cmap


def fix(func, range):
    """Apply the given range to a colormap rather than accept the one coming
    from the data.

    This is useful for colormaps intrinsically tied to a given scale, like
    bathymetry/elevation colormaps for GIS or for working around Chaco to
    implement custom behavior.

    Paramaters
    ----------
    func : callable
        A colormap factory function like those provided in this module.
    range : DataRange1D or (low, high) tuple.
        The range to apply.

    Returns
    -------
    cmap : callable
        A wrapper factory function that can be used in place of the original
        factory function.
    """
    if isinstance(range, tuple):
        # Adapt tuples to DataRange1D for convenience.
        from chaco.data_range_1d import DataRange1D

        range = DataRange1D(low=range[0], high=range[1])

    def cmap(dummy_range, **traits):
        # Use the range given to the fix() function, not the cmap() function.
        return func(range, **traits)

    # Look a little like the wrapped function.
    cmap.__name__ = "fixed_" + func.__name__
    if func.__doc__ is not None:
        cmap.__doc__ = "Fixed: " + func.__doc__
    else:
        cmap.__doc__ = "Fixed: " + func.__name__
    return cmap


# Colormaps.


def autumn(range, **traits):
    """ Generator function for the 'autumn' colormap. """

    _data = {
        "red": ((0.0, 1.0, 1.0), (1.0, 1.0, 1.0)),
        "green": ((0.0, 0.0, 0.0), (1.0, 1.0, 1.0)),
        "blue": ((0.0, 0.0, 0.0), (1.0, 0.0, 0.0)),
    }

    return ColorMapper.from_segment_map(_data, range=range, **traits)


def binary(range, **traits):
    """ Generator function for the 'binary' colormap. """

    _data = {
        "red": [(0.0, 1.0, 1.0), (1.0, 0.0, 0.0)],
        "green": [(0.0, 1.0, 1.0), (1.0, 0.0, 0.0)],
        "blue": [(0.0, 1.0, 1.0), (1.0, 0.0, 0.0)],
    }

    return ColorMapper.from_segment_map(_data, range=range, **traits)


def bone(range, **traits):
    """ Generator function for the 'bone' colormap. """

    _data = {
        "red": (
            (0.0, 0.0, 0.0),
            (0.746032, 0.652778, 0.652778),
            (1.0, 1.0, 1.0),
        ),
        "green": (
            (0.0, 0.0, 0.0),
            (0.365079, 0.319444, 0.319444),
            (0.746032, 0.777778, 0.777778),
            (1.0, 1.0, 1.0),
        ),
        "blue": (
            (0.0, 0.0, 0.0),
            (0.365079, 0.444444, 0.444444),
            (1.0, 1.0, 1.0),
        ),
    }

    return ColorMapper.from_segment_map(_data, range=range, **traits)


def cool(range, **traits):
    """ Generator function for the 'cool' colormap. """

    _data = {
        "red": ((0.0, 0.0, 0.0), (1.0, 1.0, 1.0)),
        "green": ((0.0, 1.0, 1.0), (1.0, 0.0, 0.0)),
        "blue": ((0.0, 1.0, 1.0), (1.0, 1.0, 1.0)),
    }

    return ColorMapper.from_segment_map(_data, range=range, **traits)


def copper(range, **traits):
    """ Generator function for the 'copper' colormap. """

    _data = {
        "red": (
            (0.0, 0.0, 0.0),
            (0.809524, 1.000000, 1.000000),
            (1.0, 1.0, 1.0),
        ),
        "green": ((0.0, 0.0, 0.0), (1.0, 0.7812, 0.7812)),
        "blue": ((0.0, 0.0, 0.0), (1.0, 0.4975, 0.4975)),
    }

    return ColorMapper.from_segment_map(_data, range=range, **traits)


def flag(range, **traits):
    """ Generator function for the 'flag' colormap. """

    _data = {
        "red": (
            (0.0, 1.0, 1.0),
            (0.015873, 1.000000, 1.000000),
            (0.031746, 0.000000, 0.000000),
            (0.047619, 0.000000, 0.000000),
            (0.063492, 1.000000, 1.000000),
            (0.079365, 1.000000, 1.000000),
            (0.095238, 0.000000, 0.000000),
            (0.111111, 0.000000, 0.000000),
            (0.126984, 1.000000, 1.000000),
            (0.142857, 1.000000, 1.000000),
            (0.158730, 0.000000, 0.000000),
            (0.174603, 0.000000, 0.000000),
            (0.190476, 1.000000, 1.000000),
            (0.206349, 1.000000, 1.000000),
            (0.222222, 0.000000, 0.000000),
            (0.238095, 0.000000, 0.000000),
            (0.253968, 1.000000, 1.000000),
            (0.269841, 1.000000, 1.000000),
            (0.285714, 0.000000, 0.000000),
            (0.301587, 0.000000, 0.000000),
            (0.317460, 1.000000, 1.000000),
            (0.333333, 1.000000, 1.000000),
            (0.349206, 0.000000, 0.000000),
            (0.365079, 0.000000, 0.000000),
            (0.380952, 1.000000, 1.000000),
            (0.396825, 1.000000, 1.000000),
            (0.412698, 0.000000, 0.000000),
            (0.428571, 0.000000, 0.000000),
            (0.444444, 1.000000, 1.000000),
            (0.460317, 1.000000, 1.000000),
            (0.476190, 0.000000, 0.000000),
            (0.492063, 0.000000, 0.000000),
            (0.507937, 1.000000, 1.000000),
            (0.523810, 1.000000, 1.000000),
            (0.539683, 0.000000, 0.000000),
            (0.555556, 0.000000, 0.000000),
            (0.571429, 1.000000, 1.000000),
            (0.587302, 1.000000, 1.000000),
            (0.603175, 0.000000, 0.000000),
            (0.619048, 0.000000, 0.000000),
            (0.634921, 1.000000, 1.000000),
            (0.650794, 1.000000, 1.000000),
            (0.666667, 0.000000, 0.000000),
            (0.682540, 0.000000, 0.000000),
            (0.698413, 1.000000, 1.000000),
            (0.714286, 1.000000, 1.000000),
            (0.730159, 0.000000, 0.000000),
            (0.746032, 0.000000, 0.000000),
            (0.761905, 1.000000, 1.000000),
            (0.777778, 1.000000, 1.000000),
            (0.793651, 0.000000, 0.000000),
            (0.809524, 0.000000, 0.000000),
            (0.825397, 1.000000, 1.000000),
            (0.841270, 1.000000, 1.000000),
            (0.857143, 0.000000, 0.000000),
            (0.873016, 0.000000, 0.000000),
            (0.888889, 1.000000, 1.000000),
            (0.904762, 1.000000, 1.000000),
            (0.920635, 0.000000, 0.000000),
            (0.936508, 0.000000, 0.000000),
            (0.952381, 1.000000, 1.000000),
            (0.968254, 1.000000, 1.000000),
            (0.984127, 0.000000, 0.000000),
            (1.0, 0.0, 0.0),
        ),
        "green": (
            (0.0, 0.0, 0.0),
            (0.015873, 1.000000, 1.000000),
            (0.031746, 0.000000, 0.000000),
            (0.063492, 0.000000, 0.000000),
            (0.079365, 1.000000, 1.000000),
            (0.095238, 0.000000, 0.000000),
            (0.126984, 0.000000, 0.000000),
            (0.142857, 1.000000, 1.000000),
            (0.158730, 0.000000, 0.000000),
            (0.190476, 0.000000, 0.000000),
            (0.206349, 1.000000, 1.000000),
            (0.222222, 0.000000, 0.000000),
            (0.253968, 0.000000, 0.000000),
            (0.269841, 1.000000, 1.000000),
            (0.285714, 0.000000, 0.000000),
            (0.317460, 0.000000, 0.000000),
            (0.333333, 1.000000, 1.000000),
            (0.349206, 0.000000, 0.000000),
            (0.380952, 0.000000, 0.000000),
            (0.396825, 1.000000, 1.000000),
            (0.412698, 0.000000, 0.000000),
            (0.444444, 0.000000, 0.000000),
            (0.460317, 1.000000, 1.000000),
            (0.476190, 0.000000, 0.000000),
            (0.507937, 0.000000, 0.000000),
            (0.523810, 1.000000, 1.000000),
            (0.539683, 0.000000, 0.000000),
            (0.571429, 0.000000, 0.000000),
            (0.587302, 1.000000, 1.000000),
            (0.603175, 0.000000, 0.000000),
            (0.634921, 0.000000, 0.000000),
            (0.650794, 1.000000, 1.000000),
            (0.666667, 0.000000, 0.000000),
            (0.698413, 0.000000, 0.000000),
            (0.714286, 1.000000, 1.000000),
            (0.730159, 0.000000, 0.000000),
            (0.761905, 0.000000, 0.000000),
            (0.777778, 1.000000, 1.000000),
            (0.793651, 0.000000, 0.000000),
            (0.825397, 0.000000, 0.000000),
            (0.841270, 1.000000, 1.000000),
            (0.857143, 0.000000, 0.000000),
            (0.888889, 0.000000, 0.000000),
            (0.904762, 1.000000, 1.000000),
            (0.920635, 0.000000, 0.000000),
            (0.952381, 0.000000, 0.000000),
            (0.968254, 1.000000, 1.000000),
            (0.984127, 0.000000, 0.000000),
            (1.0, 0.0, 0.0),
        ),
        "blue": (
            (0.0, 0.0, 0.0),
            (0.015873, 1.000000, 1.000000),
            (0.031746, 1.000000, 1.000000),
            (0.047619, 0.000000, 0.000000),
            (0.063492, 0.000000, 0.000000),
            (0.079365, 1.000000, 1.000000),
            (0.095238, 1.000000, 1.000000),
            (0.111111, 0.000000, 0.000000),
            (0.126984, 0.000000, 0.000000),
            (0.142857, 1.000000, 1.000000),
            (0.158730, 1.000000, 1.000000),
            (0.174603, 0.000000, 0.000000),
            (0.190476, 0.000000, 0.000000),
            (0.206349, 1.000000, 1.000000),
            (0.222222, 1.000000, 1.000000),
            (0.238095, 0.000000, 0.000000),
            (0.253968, 0.000000, 0.000000),
            (0.269841, 1.000000, 1.000000),
            (0.285714, 1.000000, 1.000000),
            (0.301587, 0.000000, 0.000000),
            (0.317460, 0.000000, 0.000000),
            (0.333333, 1.000000, 1.000000),
            (0.349206, 1.000000, 1.000000),
            (0.365079, 0.000000, 0.000000),
            (0.380952, 0.000000, 0.000000),
            (0.396825, 1.000000, 1.000000),
            (0.412698, 1.000000, 1.000000),
            (0.428571, 0.000000, 0.000000),
            (0.444444, 0.000000, 0.000000),
            (0.460317, 1.000000, 1.000000),
            (0.476190, 1.000000, 1.000000),
            (0.492063, 0.000000, 0.000000),
            (0.507937, 0.000000, 0.000000),
            (0.523810, 1.000000, 1.000000),
            (0.539683, 1.000000, 1.000000),
            (0.555556, 0.000000, 0.000000),
            (0.571429, 0.000000, 0.000000),
            (0.587302, 1.000000, 1.000000),
            (0.603175, 1.000000, 1.000000),
            (0.619048, 0.000000, 0.000000),
            (0.634921, 0.000000, 0.000000),
            (0.650794, 1.000000, 1.000000),
            (0.666667, 1.000000, 1.000000),
            (0.682540, 0.000000, 0.000000),
            (0.698413, 0.000000, 0.000000),
            (0.714286, 1.000000, 1.000000),
            (0.730159, 1.000000, 1.000000),
            (0.746032, 0.000000, 0.000000),
            (0.761905, 0.000000, 0.000000),
            (0.777778, 1.000000, 1.000000),
            (0.793651, 1.000000, 1.000000),
            (0.809524, 0.000000, 0.000000),
            (0.825397, 0.000000, 0.000000),
            (0.841270, 1.000000, 1.000000),
            (0.857143, 1.000000, 1.000000),
            (0.873016, 0.000000, 0.000000),
            (0.888889, 0.000000, 0.000000),
            (0.904762, 1.000000, 1.000000),
            (0.920635, 1.000000, 1.000000),
            (0.936508, 0.000000, 0.000000),
            (0.952381, 0.000000, 0.000000),
            (0.968254, 1.000000, 1.000000),
            (0.984127, 1.000000, 1.000000),
            (1.0, 0.0, 0.0),
        ),
    }

    return ColorMapper.from_segment_map(_data, range=range, **traits)


def seismic(range, **traits):
    """ Generator for 'seismic' colormap from matplotlib """
    _data = [
        (0.0, 0.0, 0.3),
        (0.0, 0.0, 1.0),
        (1.0, 1.0, 1.0),
        (1.0, 0.0, 0.0),
        (0.5, 0.0, 0.0),
    ]
    return ColorMapper.from_palette_array(_data, range=range, **traits)


def terrain(range, **traits):
    """ Generator for 'terrain' colormap from matplotlib """
    _data = {
        "red": [
            (0.0, 0.2, 0.2),
            (0.15, 0.0, 0.0),
            (0.25, 0.0, 0.0),
            (0.50, 1.0, 1.0),
            (0.75, 0.5, 0.5),
            (1.0, 1.0, 1.0),
        ],
        "green": [
            (0.0, 0.2, 0.2),
            (0.15, 0.6, 0.6),
            (0.25, 0.8, 0.8),
            (0.50, 1.0, 1.0),
            (0.75, 0.36, 0.36),
            (1.0, 1.0, 1.0),
        ],
        "blue": [
            (0.0, 0.6, 0.6),
            (0.15, 1.0, 1.0),
            (0.25, 0.4, 0.4),
            (0.50, 0.6, 0.6),
            (0.75, 0.33, 0.33),
            (1.0, 1.0, 1.0),
        ],
    }
    return ColorMapper.from_segment_map(_data, range=range, **traits)


def gray(range, **traits):
    """ Generator function for the 'gray' colormap. """

    _data = {
        "red": ((0.0, 0, 0), (1.0, 1, 1)),
        "green": ((0.0, 0, 0), (1.0, 1, 1)),
        "blue": ((0.0, 0, 0), (1.0, 1, 1)),
    }

    return ColorMapper.from_segment_map(_data, range=range, **traits)


def yarg(range, **traits):
    """ Generator function for the 'yarg' colormap. """

    _data = {
        "red": ((0.0, 1, 1), (1.0, 0, 0)),
        "green": ((0.0, 1, 1), (1.0, 0, 0)),
        "blue": ((0.0, 1, 1), (1.0, 0, 0)),
    }

    return ColorMapper.from_segment_map(_data, range=range, **traits)


def hot(range, **traits):
    """ Generator function for the 'hot' colormap. """

    _data = {
        "red": (
            (0.0, 0.0416, 0.0416),
            (0.365079, 1.000000, 1.000000),
            (1.0, 1.0, 1.0),
        ),
        "green": (
            (0.0, 0.0, 0.0),
            (0.365079, 0.000000, 0.000000),
            (0.746032, 1.000000, 1.000000),
            (1.0, 1.0, 1.0),
        ),
        "blue": (
            (0.0, 0.0, 0.0),
            (0.746032, 0.000000, 0.000000),
            (1.0, 1.0, 1.0),
        ),
    }

    return ColorMapper.from_segment_map(_data, range=range, **traits)


def hsv(range, **traits):
    """ Generator function for the 'hsv' colormap. """

    _data = {
        "red": (
            (0.0, 1.0, 1.0),
            (0.158730, 1.000000, 1.000000),
            (0.174603, 0.968750, 0.968750),
            (0.333333, 0.031250, 0.031250),
            (0.349206, 0.000000, 0.000000),
            (0.666667, 0.000000, 0.000000),
            (0.682540, 0.031250, 0.031250),
            (0.841270, 0.968750, 0.968750),
            (0.857143, 1.000000, 1.000000),
            (1.0, 1.0, 1.0),
        ),
        "green": (
            (0.0, 0.0, 0.0),
            (0.158730, 0.937500, 0.937500),
            (0.174603, 1.000000, 1.000000),
            (0.507937, 1.000000, 1.000000),
            (0.666667, 0.062500, 0.062500),
            (0.682540, 0.000000, 0.000000),
            (1.0, 0.0, 0.0),
        ),
        "blue": (
            (0.0, 0.0, 0.0),
            (0.333333, 0.000000, 0.000000),
            (0.349206, 0.062500, 0.062500),
            (0.507937, 1.000000, 1.000000),
            (0.841270, 1.000000, 1.000000),
            (0.857143, 0.937500, 0.937500),
            (1.0, 0.09375, 0.09375),
        ),
    }

    return ColorMapper.from_segment_map(_data, range=range, **traits)


def jet(range, **traits):
    """ Generator function for the 'jet' colormap. """

    _data = {
        "red": (
            (0.0, 0, 0),
            (0.35, 0, 0),
            (0.66, 1, 1),
            (0.89, 1, 1),
            (1, 0.5, 0.5),
        ),
        "green": (
            (0.0, 0, 0),
            (0.125, 0, 0),
            (0.375, 1, 1),
            (0.64, 1, 1),
            (0.91, 0, 0),
            (1, 0, 0),
        ),
        "blue": (
            (0.0, 0.5, 0.5),
            (0.11, 1, 1),
            (0.34, 1, 1),
            (0.65, 0, 0),
            (1, 0, 0),
        ),
    }

    return ColorMapper.from_segment_map(_data, range=range, **traits)


def pink(range, **traits):
    """ Generator function for the 'pink' colormap. """

    _data = {
        "red": (
            (0.0, 0.1178, 0.1178),
            (0.015873, 0.195857, 0.195857),
            (0.031746, 0.250661, 0.250661),
            (0.047619, 0.295468, 0.295468),
            (0.063492, 0.334324, 0.334324),
            (0.079365, 0.369112, 0.369112),
            (0.095238, 0.400892, 0.400892),
            (0.111111, 0.430331, 0.430331),
            (0.126984, 0.457882, 0.457882),
            (0.142857, 0.483867, 0.483867),
            (0.158730, 0.508525, 0.508525),
            (0.174603, 0.532042, 0.532042),
            (0.190476, 0.554563, 0.554563),
            (0.206349, 0.576204, 0.576204),
            (0.222222, 0.597061, 0.597061),
            (0.238095, 0.617213, 0.617213),
            (0.253968, 0.636729, 0.636729),
            (0.269841, 0.655663, 0.655663),
            (0.285714, 0.674066, 0.674066),
            (0.301587, 0.691980, 0.691980),
            (0.317460, 0.709441, 0.709441),
            (0.333333, 0.726483, 0.726483),
            (0.349206, 0.743134, 0.743134),
            (0.365079, 0.759421, 0.759421),
            (0.380952, 0.766356, 0.766356),
            (0.396825, 0.773229, 0.773229),
            (0.412698, 0.780042, 0.780042),
            (0.428571, 0.786796, 0.786796),
            (0.444444, 0.793492, 0.793492),
            (0.460317, 0.800132, 0.800132),
            (0.476190, 0.806718, 0.806718),
            (0.492063, 0.813250, 0.813250),
            (0.507937, 0.819730, 0.819730),
            (0.523810, 0.826160, 0.826160),
            (0.539683, 0.832539, 0.832539),
            (0.555556, 0.838870, 0.838870),
            (0.571429, 0.845154, 0.845154),
            (0.587302, 0.851392, 0.851392),
            (0.603175, 0.857584, 0.857584),
            (0.619048, 0.863731, 0.863731),
            (0.634921, 0.869835, 0.869835),
            (0.650794, 0.875897, 0.875897),
            (0.666667, 0.881917, 0.881917),
            (0.682540, 0.887896, 0.887896),
            (0.698413, 0.893835, 0.893835),
            (0.714286, 0.899735, 0.899735),
            (0.730159, 0.905597, 0.905597),
            (0.746032, 0.911421, 0.911421),
            (0.761905, 0.917208, 0.917208),
            (0.777778, 0.922958, 0.922958),
            (0.793651, 0.928673, 0.928673),
            (0.809524, 0.934353, 0.934353),
            (0.825397, 0.939999, 0.939999),
            (0.841270, 0.945611, 0.945611),
            (0.857143, 0.951190, 0.951190),
            (0.873016, 0.956736, 0.956736),
            (0.888889, 0.962250, 0.962250),
            (0.904762, 0.967733, 0.967733),
            (0.920635, 0.973185, 0.973185),
            (0.936508, 0.978607, 0.978607),
            (0.952381, 0.983999, 0.983999),
            (0.968254, 0.989361, 0.989361),
            (0.984127, 0.994695, 0.994695),
            (1.0, 1.0, 1.0),
        ),
        "green": (
            (0.0, 0.0, 0.0),
            (0.015873, 0.102869, 0.102869),
            (0.031746, 0.145479, 0.145479),
            (0.047619, 0.178174, 0.178174),
            (0.063492, 0.205738, 0.205738),
            (0.079365, 0.230022, 0.230022),
            (0.095238, 0.251976, 0.251976),
            (0.111111, 0.272166, 0.272166),
            (0.126984, 0.290957, 0.290957),
            (0.142857, 0.308607, 0.308607),
            (0.158730, 0.325300, 0.325300),
            (0.174603, 0.341178, 0.341178),
            (0.190476, 0.356348, 0.356348),
            (0.206349, 0.370899, 0.370899),
            (0.222222, 0.384900, 0.384900),
            (0.238095, 0.398410, 0.398410),
            (0.253968, 0.411476, 0.411476),
            (0.269841, 0.424139, 0.424139),
            (0.285714, 0.436436, 0.436436),
            (0.301587, 0.448395, 0.448395),
            (0.317460, 0.460044, 0.460044),
            (0.333333, 0.471405, 0.471405),
            (0.349206, 0.482498, 0.482498),
            (0.365079, 0.493342, 0.493342),
            (0.380952, 0.517549, 0.517549),
            (0.396825, 0.540674, 0.540674),
            (0.412698, 0.562849, 0.562849),
            (0.428571, 0.584183, 0.584183),
            (0.444444, 0.604765, 0.604765),
            (0.460317, 0.624669, 0.624669),
            (0.476190, 0.643958, 0.643958),
            (0.492063, 0.662687, 0.662687),
            (0.507937, 0.680900, 0.680900),
            (0.523810, 0.698638, 0.698638),
            (0.539683, 0.715937, 0.715937),
            (0.555556, 0.732828, 0.732828),
            (0.571429, 0.749338, 0.749338),
            (0.587302, 0.765493, 0.765493),
            (0.603175, 0.781313, 0.781313),
            (0.619048, 0.796819, 0.796819),
            (0.634921, 0.812029, 0.812029),
            (0.650794, 0.826960, 0.826960),
            (0.666667, 0.841625, 0.841625),
            (0.682540, 0.856040, 0.856040),
            (0.698413, 0.870216, 0.870216),
            (0.714286, 0.884164, 0.884164),
            (0.730159, 0.897896, 0.897896),
            (0.746032, 0.911421, 0.911421),
            (0.761905, 0.917208, 0.917208),
            (0.777778, 0.922958, 0.922958),
            (0.793651, 0.928673, 0.928673),
            (0.809524, 0.934353, 0.934353),
            (0.825397, 0.939999, 0.939999),
            (0.841270, 0.945611, 0.945611),
            (0.857143, 0.951190, 0.951190),
            (0.873016, 0.956736, 0.956736),
            (0.888889, 0.962250, 0.962250),
            (0.904762, 0.967733, 0.967733),
            (0.920635, 0.973185, 0.973185),
            (0.936508, 0.978607, 0.978607),
            (0.952381, 0.983999, 0.983999),
            (0.968254, 0.989361, 0.989361),
            (0.984127, 0.994695, 0.994695),
            (1.0, 1.0, 1.0),
        ),
        "blue": (
            (0.0, 0.0, 0.0),
            (0.015873, 0.102869, 0.102869),
            (0.031746, 0.145479, 0.145479),
            (0.047619, 0.178174, 0.178174),
            (0.063492, 0.205738, 0.205738),
            (0.079365, 0.230022, 0.230022),
            (0.095238, 0.251976, 0.251976),
            (0.111111, 0.272166, 0.272166),
            (0.126984, 0.290957, 0.290957),
            (0.142857, 0.308607, 0.308607),
            (0.158730, 0.325300, 0.325300),
            (0.174603, 0.341178, 0.341178),
            (0.190476, 0.356348, 0.356348),
            (0.206349, 0.370899, 0.370899),
            (0.222222, 0.384900, 0.384900),
            (0.238095, 0.398410, 0.398410),
            (0.253968, 0.411476, 0.411476),
            (0.269841, 0.424139, 0.424139),
            (0.285714, 0.436436, 0.436436),
            (0.301587, 0.448395, 0.448395),
            (0.317460, 0.460044, 0.460044),
            (0.333333, 0.471405, 0.471405),
            (0.349206, 0.482498, 0.482498),
            (0.365079, 0.493342, 0.493342),
            (0.380952, 0.503953, 0.503953),
            (0.396825, 0.514344, 0.514344),
            (0.412698, 0.524531, 0.524531),
            (0.428571, 0.534522, 0.534522),
            (0.444444, 0.544331, 0.544331),
            (0.460317, 0.553966, 0.553966),
            (0.476190, 0.563436, 0.563436),
            (0.492063, 0.572750, 0.572750),
            (0.507937, 0.581914, 0.581914),
            (0.523810, 0.590937, 0.590937),
            (0.539683, 0.599824, 0.599824),
            (0.555556, 0.608581, 0.608581),
            (0.571429, 0.617213, 0.617213),
            (0.587302, 0.625727, 0.625727),
            (0.603175, 0.634126, 0.634126),
            (0.619048, 0.642416, 0.642416),
            (0.634921, 0.650600, 0.650600),
            (0.650794, 0.658682, 0.658682),
            (0.666667, 0.666667, 0.666667),
            (0.682540, 0.674556, 0.674556),
            (0.698413, 0.682355, 0.682355),
            (0.714286, 0.690066, 0.690066),
            (0.730159, 0.697691, 0.697691),
            (0.746032, 0.705234, 0.705234),
            (0.761905, 0.727166, 0.727166),
            (0.777778, 0.748455, 0.748455),
            (0.793651, 0.769156, 0.769156),
            (0.809524, 0.789314, 0.789314),
            (0.825397, 0.808969, 0.808969),
            (0.841270, 0.828159, 0.828159),
            (0.857143, 0.846913, 0.846913),
            (0.873016, 0.865261, 0.865261),
            (0.888889, 0.883229, 0.883229),
            (0.904762, 0.900837, 0.900837),
            (0.920635, 0.918109, 0.918109),
            (0.936508, 0.935061, 0.935061),
            (0.952381, 0.951711, 0.951711),
            (0.968254, 0.968075, 0.968075),
            (0.984127, 0.984167, 0.984167),
            (1.0, 1.0, 1.0),
        ),
    }

    return ColorMapper.from_segment_map(_data, range=range, **traits)


def prism(range, **traits):
    """ Generator function for the 'prism' colormap. """

    _data = {
        "red": (
            (0.0, 1.0, 1.0),
            (0.031746, 1.000000, 1.000000),
            (0.047619, 0.000000, 0.000000),
            (0.063492, 0.000000, 0.000000),
            (0.079365, 0.666667, 0.666667),
            (0.095238, 1.000000, 1.000000),
            (0.126984, 1.000000, 1.000000),
            (0.142857, 0.000000, 0.000000),
            (0.158730, 0.000000, 0.000000),
            (0.174603, 0.666667, 0.666667),
            (0.190476, 1.000000, 1.000000),
            (0.222222, 1.000000, 1.000000),
            (0.238095, 0.000000, 0.000000),
            (0.253968, 0.000000, 0.000000),
            (0.269841, 0.666667, 0.666667),
            (0.285714, 1.000000, 1.000000),
            (0.317460, 1.000000, 1.000000),
            (0.333333, 0.000000, 0.000000),
            (0.349206, 0.000000, 0.000000),
            (0.365079, 0.666667, 0.666667),
            (0.380952, 1.000000, 1.000000),
            (0.412698, 1.000000, 1.000000),
            (0.428571, 0.000000, 0.000000),
            (0.444444, 0.000000, 0.000000),
            (0.460317, 0.666667, 0.666667),
            (0.476190, 1.000000, 1.000000),
            (0.507937, 1.000000, 1.000000),
            (0.523810, 0.000000, 0.000000),
            (0.539683, 0.000000, 0.000000),
            (0.555556, 0.666667, 0.666667),
            (0.571429, 1.000000, 1.000000),
            (0.603175, 1.000000, 1.000000),
            (0.619048, 0.000000, 0.000000),
            (0.634921, 0.000000, 0.000000),
            (0.650794, 0.666667, 0.666667),
            (0.666667, 1.000000, 1.000000),
            (0.698413, 1.000000, 1.000000),
            (0.714286, 0.000000, 0.000000),
            (0.730159, 0.000000, 0.000000),
            (0.746032, 0.666667, 0.666667),
            (0.761905, 1.000000, 1.000000),
            (0.793651, 1.000000, 1.000000),
            (0.809524, 0.000000, 0.000000),
            (0.825397, 0.000000, 0.000000),
            (0.841270, 0.666667, 0.666667),
            (0.857143, 1.000000, 1.000000),
            (0.888889, 1.000000, 1.000000),
            (0.904762, 0.000000, 0.000000),
            (0.920635, 0.000000, 0.000000),
            (0.936508, 0.666667, 0.666667),
            (0.952381, 1.000000, 1.000000),
            (0.984127, 1.000000, 1.000000),
            (1.0, 0.0, 0.0),
        ),
        "green": (
            (0.0, 0.0, 0.0),
            (0.031746, 1.000000, 1.000000),
            (0.047619, 1.000000, 1.000000),
            (0.063492, 0.000000, 0.000000),
            (0.095238, 0.000000, 0.000000),
            (0.126984, 1.000000, 1.000000),
            (0.142857, 1.000000, 1.000000),
            (0.158730, 0.000000, 0.000000),
            (0.190476, 0.000000, 0.000000),
            (0.222222, 1.000000, 1.000000),
            (0.238095, 1.000000, 1.000000),
            (0.253968, 0.000000, 0.000000),
            (0.285714, 0.000000, 0.000000),
            (0.317460, 1.000000, 1.000000),
            (0.333333, 1.000000, 1.000000),
            (0.349206, 0.000000, 0.000000),
            (0.380952, 0.000000, 0.000000),
            (0.412698, 1.000000, 1.000000),
            (0.428571, 1.000000, 1.000000),
            (0.444444, 0.000000, 0.000000),
            (0.476190, 0.000000, 0.000000),
            (0.507937, 1.000000, 1.000000),
            (0.523810, 1.000000, 1.000000),
            (0.539683, 0.000000, 0.000000),
            (0.571429, 0.000000, 0.000000),
            (0.603175, 1.000000, 1.000000),
            (0.619048, 1.000000, 1.000000),
            (0.634921, 0.000000, 0.000000),
            (0.666667, 0.000000, 0.000000),
            (0.698413, 1.000000, 1.000000),
            (0.714286, 1.000000, 1.000000),
            (0.730159, 0.000000, 0.000000),
            (0.761905, 0.000000, 0.000000),
            (0.793651, 1.000000, 1.000000),
            (0.809524, 1.000000, 1.000000),
            (0.825397, 0.000000, 0.000000),
            (0.857143, 0.000000, 0.000000),
            (0.888889, 1.000000, 1.000000),
            (0.904762, 1.000000, 1.000000),
            (0.920635, 0.000000, 0.000000),
            (0.952381, 0.000000, 0.000000),
            (0.984127, 1.000000, 1.000000),
            (1.0, 1.0, 1.0),
        ),
        "blue": (
            (0.0, 0.0, 0.0),
            (0.047619, 0.000000, 0.000000),
            (0.063492, 1.000000, 1.000000),
            (0.079365, 1.000000, 1.000000),
            (0.095238, 0.000000, 0.000000),
            (0.142857, 0.000000, 0.000000),
            (0.158730, 1.000000, 1.000000),
            (0.174603, 1.000000, 1.000000),
            (0.190476, 0.000000, 0.000000),
            (0.238095, 0.000000, 0.000000),
            (0.253968, 1.000000, 1.000000),
            (0.269841, 1.000000, 1.000000),
            (0.285714, 0.000000, 0.000000),
            (0.333333, 0.000000, 0.000000),
            (0.349206, 1.000000, 1.000000),
            (0.365079, 1.000000, 1.000000),
            (0.380952, 0.000000, 0.000000),
            (0.428571, 0.000000, 0.000000),
            (0.444444, 1.000000, 1.000000),
            (0.460317, 1.000000, 1.000000),
            (0.476190, 0.000000, 0.000000),
            (0.523810, 0.000000, 0.000000),
            (0.539683, 1.000000, 1.000000),
            (0.555556, 1.000000, 1.000000),
            (0.571429, 0.000000, 0.000000),
            (0.619048, 0.000000, 0.000000),
            (0.634921, 1.000000, 1.000000),
            (0.650794, 1.000000, 1.000000),
            (0.666667, 0.000000, 0.000000),
            (0.714286, 0.000000, 0.000000),
            (0.730159, 1.000000, 1.000000),
            (0.746032, 1.000000, 1.000000),
            (0.761905, 0.000000, 0.000000),
            (0.809524, 0.000000, 0.000000),
            (0.825397, 1.000000, 1.000000),
            (0.841270, 1.000000, 1.000000),
            (0.857143, 0.000000, 0.000000),
            (0.904762, 0.000000, 0.000000),
            (0.920635, 1.000000, 1.000000),
            (0.936508, 1.000000, 1.000000),
            (0.952381, 0.000000, 0.000000),
            (1.0, 0.0, 0.0),
        ),
    }

    return ColorMapper.from_segment_map(_data, range=range, **traits)


def spring(range, **traits):
    """ Generator function for the 'spring' colormap. """

    _data = {
        "red": ((0.0, 1.0, 1.0), (1.0, 1.0, 1.0)),
        "green": ((0.0, 0.0, 0.0), (1.0, 1.0, 1.0)),
        "blue": ((0.0, 1.0, 1.0), (1.0, 0.0, 0.0)),
    }

    return ColorMapper.from_segment_map(_data, range=range, **traits)


def summer(range, **traits):
    """ Generator function for the 'summer' colormap. """

    _data = {
        "red": ((0.0, 0.0, 0.0), (1.0, 1.0, 1.0)),
        "green": ((0.0, 0.5, 0.5), (1.0, 1.0, 1.0)),
        "blue": ((0.0, 0.4, 0.4), (1.0, 0.4, 0.4)),
    }

    return ColorMapper.from_segment_map(_data, range=range, **traits)


def winter(range, **traits):
    """ Generator function for the 'winter' colormap. """

    _data = {
        "red": ((0.0, 0.0, 0.0), (1.0, 0.0, 0.0)),
        "green": ((0.0, 0.0, 0.0), (1.0, 1.0, 1.0)),
        "blue": ((0.0, 1.0, 1.0), (1.0, 0.5, 0.5)),
    }

    return ColorMapper.from_segment_map(_data, range=range, **traits)


def cw1_004(range, **traits):
    """ Generator function for the Crumblingwalls cw1-004 gradient """

    colors = array(
        [
            (0.7176, 0.6980, 0.6118),
            (0.8000, 0.5373, 0.7059),
            (0.2510, 0.4588, 0.4902),
            (0.0588, 0.3176, 0.5137),
        ]
    )

    return ColorMapper.from_palette_array(colors, range=range, **traits)


def cw1_005(range, **traits):
    """ Generator function for the Crumblingwalls cw1-005 gradient """
    colors = array(
        [
            (0.7059, 0.1686, 0.0980),
            (0.7961, 0.5176, 0.2039),
            (0.2863, 0.3255, 0.1294),
        ]
    )

    return ColorMapper.from_palette_array(colors, range=range, **traits)


def cw1_006(range, **traits):
    """ Generator function for the Crumblingwalls cw1-006 gradient """

    colors = array(
        [
            (0.4275, 0.2824, 0.4667),
            (0.2039, 0.1843, 0.4667),
            (0.0863, 0.4078, 0.2078),
        ]
    )

    return ColorMapper.from_palette_array(colors, range=range, **traits)


def cw1_028(range, **traits):
    """ Generator function for the Crumblingwalls cw1-058 gradient """

    colors = array(
        [
            (0.2275, 0.2275, 0.4784),
            (0.3294, 0.5137, 0.8588),
            (0.4078, 0.8471, 0.8510),
        ]
    )

    return ColorMapper.from_palette_array(colors, range=range, **traits)


def gmt_drywet(range, **traits):

    _data = {
        "red": (
            (0.00000, 0.5255, 0.5255),
            (0.16670, 0.9333, 0.9333),
            (0.33330, 0.7059, 0.7059),
            (0.50000, 0.1961, 0.1961),
            (0.66670, 0.0471, 0.0471),
            (0.83330, 0.1490, 0.1490),
            (1.00000, 0.0314, 0.0314),
        ),
        "green": (
            (0.00000, 0.3804, 0.3804),
            (0.16670, 0.7804, 0.7804),
            (0.33330, 0.9333, 0.9333),
            (0.50000, 0.9333, 0.9333),
            (0.66670, 0.4706, 0.4706),
            (0.83330, 0.0039, 0.0039),
            (1.00000, 0.2000, 0.2000),
        ),
        "blue": (
            (0.00000, 0.1647, 0.1647),
            (0.16670, 0.3922, 0.3922),
            (0.33330, 0.5294, 0.5294),
            (0.50000, 0.9216, 0.9216),
            (0.66670, 0.9333, 0.9333),
            (0.83330, 0.7176, 0.7176),
            (1.00000, 0.4431, 0.4431),
        ),
    }

    return ColorMapper.from_segment_map(_data, range=range, **traits)


# The following colormaps come from ColorBrewer.
# Although the ColorBrewer colormaps are defined as discrete colormaps, we
# create continuous colormaps from them by linear interpolation in RGB
# colorspace, in the same manner as matplotlib


def Blues(range, **traits):
    """Generator for the 'Blues' colormap from ColorBrewer.

    Although the ColorBrewer colormaps are defined as discrete colormaps, we
    create continuous colormaps from them by linear interpolation in RGB
    colorspace.
    """
    _data = dict(
        red=[
            (0.0, 0.96862745098, 0.96862745098),
            (0.125, 0.870588235294, 0.870588235294),
            (0.25, 0.776470588235, 0.776470588235),
            (0.375, 0.619607843137, 0.619607843137),
            (0.5, 0.419607843137, 0.419607843137),
            (0.625, 0.258823529412, 0.258823529412),
            (0.75, 0.129411764706, 0.129411764706),
            (0.875, 0.0313725490196, 0.0313725490196),
            (1.0, 0.0313725490196, 0.0313725490196),
        ],
        green=[
            (0.0, 0.98431372549, 0.98431372549),
            (0.125, 0.921568627451, 0.921568627451),
            (0.25, 0.858823529412, 0.858823529412),
            (0.375, 0.792156862745, 0.792156862745),
            (0.5, 0.682352941176, 0.682352941176),
            (0.625, 0.572549019608, 0.572549019608),
            (0.75, 0.443137254902, 0.443137254902),
            (0.875, 0.317647058824, 0.317647058824),
            (1.0, 0.188235294118, 0.188235294118),
        ],
        blue=[
            (0.0, 1.0, 1.0),
            (0.125, 0.96862745098, 0.96862745098),
            (0.25, 0.937254901961, 0.937254901961),
            (0.375, 0.882352941176, 0.882352941176),
            (0.5, 0.839215686275, 0.839215686275),
            (0.625, 0.776470588235, 0.776470588235),
            (0.75, 0.709803921569, 0.709803921569),
            (0.875, 0.611764705882, 0.611764705882),
            (1.0, 0.419607843137, 0.419607843137),
        ],
    )
    return ColorMapper.from_segment_map(_data, range=range, **traits)


def BrBG(range, **traits):
    """Generator for the 'BrBG' colormap from ColorBrewer.

    This is a diverging colormap. It is good for data which is centered around
    a "special" value, like 0.

    Although the ColorBrewer colormaps are defined as discrete colormaps, we
    create continuous colormaps from them by linear interpolation in RGB
    colorspace.
    """
    _data = dict(
        red=[
            (0.0, 0.329411764706, 0.329411764706),
            (0.1, 0.549019607843, 0.549019607843),
            (0.2, 0.749019607843, 0.749019607843),
            (0.3, 0.874509803922, 0.874509803922),
            (0.4, 0.964705882353, 0.964705882353),
            (0.5, 0.960784313725, 0.960784313725),
            (0.6, 0.780392156863, 0.780392156863),
            (0.7, 0.501960784314, 0.501960784314),
            (0.8, 0.207843137255, 0.207843137255),
            (0.9, 0.00392156862745, 0.00392156862745),
            (1.0, 0.0, 0.0),
        ],
        green=[
            (0.0, 0.188235294118, 0.188235294118),
            (0.1, 0.317647058824, 0.317647058824),
            (0.2, 0.505882352941, 0.505882352941),
            (0.3, 0.760784313725, 0.760784313725),
            (0.4, 0.909803921569, 0.909803921569),
            (0.5, 0.960784313725, 0.960784313725),
            (0.6, 0.917647058824, 0.917647058824),
            (0.7, 0.803921568627, 0.803921568627),
            (0.8, 0.592156862745, 0.592156862745),
            (0.9, 0.4, 0.4),
            (1.0, 0.235294117647, 0.235294117647),
        ],
        blue=[
            (0.0, 0.0196078431373, 0.0196078431373),
            (0.1, 0.0392156862745, 0.0392156862745),
            (0.2, 0.176470588235, 0.176470588235),
            (0.3, 0.490196078431, 0.490196078431),
            (0.4, 0.764705882353, 0.764705882353),
            (0.5, 0.960784313725, 0.960784313725),
            (0.6, 0.898039215686, 0.898039215686),
            (0.7, 0.756862745098, 0.756862745098),
            (0.8, 0.560784313725, 0.560784313725),
            (0.9, 0.36862745098, 0.36862745098),
            (1.0, 0.188235294118, 0.188235294118),
        ],
    )
    return ColorMapper.from_segment_map(_data, range=range, **traits)


def BuGn(range, **traits):
    """Generator for the 'BuGn' colormap from ColorBrewer.

    Although the ColorBrewer colormaps are defined as discrete colormaps, we
    create continuous colormaps from them by linear interpolation in RGB
    colorspace.
    """
    _data = dict(
        red=[
            (0.0, 0.96862745098, 0.96862745098),
            (0.125, 0.898039215686, 0.898039215686),
            (0.25, 0.8, 0.8),
            (0.375, 0.6, 0.6),
            (0.5, 0.4, 0.4),
            (0.625, 0.254901960784, 0.254901960784),
            (0.75, 0.137254901961, 0.137254901961),
            (0.875, 0.0, 0.0),
            (1.0, 0.0, 0.0),
        ],
        green=[
            (0.0, 0.988235294118, 0.988235294118),
            (0.125, 0.960784313725, 0.960784313725),
            (0.25, 0.925490196078, 0.925490196078),
            (0.375, 0.847058823529, 0.847058823529),
            (0.5, 0.760784313725, 0.760784313725),
            (0.625, 0.682352941176, 0.682352941176),
            (0.75, 0.545098039216, 0.545098039216),
            (0.875, 0.427450980392, 0.427450980392),
            (1.0, 0.266666666667, 0.266666666667),
        ],
        blue=[
            (0.0, 0.992156862745, 0.992156862745),
            (0.125, 0.976470588235, 0.976470588235),
            (0.25, 0.901960784314, 0.901960784314),
            (0.375, 0.788235294118, 0.788235294118),
            (0.5, 0.643137254902, 0.643137254902),
            (0.625, 0.462745098039, 0.462745098039),
            (0.75, 0.270588235294, 0.270588235294),
            (0.875, 0.172549019608, 0.172549019608),
            (1.0, 0.105882352941, 0.105882352941),
        ],
    )
    return ColorMapper.from_segment_map(_data, range=range, **traits)


def BuPu(range, **traits):
    """Generator for the 'BuPu' colormap from ColorBrewer.

    Although the ColorBrewer colormaps are defined as discrete colormaps, we
    create continuous colormaps from them by linear interpolation in RGB
    colorspace.
    """
    _data = dict(
        red=[
            (0.0, 0.96862745098, 0.96862745098),
            (0.125, 0.878431372549, 0.878431372549),
            (0.25, 0.749019607843, 0.749019607843),
            (0.375, 0.619607843137, 0.619607843137),
            (0.5, 0.549019607843, 0.549019607843),
            (0.625, 0.549019607843, 0.549019607843),
            (0.75, 0.533333333333, 0.533333333333),
            (0.875, 0.505882352941, 0.505882352941),
            (1.0, 0.301960784314, 0.301960784314),
        ],
        green=[
            (0.0, 0.988235294118, 0.988235294118),
            (0.125, 0.925490196078, 0.925490196078),
            (0.25, 0.827450980392, 0.827450980392),
            (0.375, 0.737254901961, 0.737254901961),
            (0.5, 0.588235294118, 0.588235294118),
            (0.625, 0.419607843137, 0.419607843137),
            (0.75, 0.254901960784, 0.254901960784),
            (0.875, 0.0588235294118, 0.0588235294118),
            (1.0, 0.0, 0.0),
        ],
        blue=[
            (0.0, 0.992156862745, 0.992156862745),
            (0.125, 0.956862745098, 0.956862745098),
            (0.25, 0.901960784314, 0.901960784314),
            (0.375, 0.854901960784, 0.854901960784),
            (0.5, 0.776470588235, 0.776470588235),
            (0.625, 0.694117647059, 0.694117647059),
            (0.75, 0.61568627451, 0.61568627451),
            (0.875, 0.486274509804, 0.486274509804),
            (1.0, 0.294117647059, 0.294117647059),
        ],
    )
    return ColorMapper.from_segment_map(_data, range=range, **traits)


def GnBu(range, **traits):
    """Generator for the 'GnBu' colormap from ColorBrewer.

    Although the ColorBrewer colormaps are defined as discrete colormaps, we
    create continuous colormaps from them by linear interpolation in RGB
    colorspace.
    """
    _data = dict(
        red=[
            (0.0, 0.96862745098, 0.96862745098),
            (0.125, 0.878431372549, 0.878431372549),
            (0.25, 0.8, 0.8),
            (0.375, 0.658823529412, 0.658823529412),
            (0.5, 0.482352941176, 0.482352941176),
            (0.625, 0.305882352941, 0.305882352941),
            (0.75, 0.16862745098, 0.16862745098),
            (0.875, 0.0313725490196, 0.0313725490196),
            (1.0, 0.0313725490196, 0.0313725490196),
        ],
        green=[
            (0.0, 0.988235294118, 0.988235294118),
            (0.125, 0.952941176471, 0.952941176471),
            (0.25, 0.921568627451, 0.921568627451),
            (0.375, 0.866666666667, 0.866666666667),
            (0.5, 0.8, 0.8),
            (0.625, 0.701960784314, 0.701960784314),
            (0.75, 0.549019607843, 0.549019607843),
            (0.875, 0.407843137255, 0.407843137255),
            (1.0, 0.250980392157, 0.250980392157),
        ],
        blue=[
            (0.0, 0.941176470588, 0.941176470588),
            (0.125, 0.858823529412, 0.858823529412),
            (0.25, 0.772549019608, 0.772549019608),
            (0.375, 0.709803921569, 0.709803921569),
            (0.5, 0.76862745098, 0.76862745098),
            (0.625, 0.827450980392, 0.827450980392),
            (0.75, 0.745098039216, 0.745098039216),
            (0.875, 0.674509803922, 0.674509803922),
            (1.0, 0.505882352941, 0.505882352941),
        ],
    )
    return ColorMapper.from_segment_map(_data, range=range, **traits)


def Greens(range, **traits):
    """Generator for the 'Greens' colormap from ColorBrewer.

    Although the ColorBrewer colormaps are defined as discrete colormaps, we
    create continuous colormaps from them by linear interpolation in RGB
    colorspace.
    """
    _data = dict(
        red=[
            (0.0, 0.96862745098, 0.96862745098),
            (0.125, 0.898039215686, 0.898039215686),
            (0.25, 0.780392156863, 0.780392156863),
            (0.375, 0.63137254902, 0.63137254902),
            (0.5, 0.454901960784, 0.454901960784),
            (0.625, 0.254901960784, 0.254901960784),
            (0.75, 0.137254901961, 0.137254901961),
            (0.875, 0.0, 0.0),
            (1.0, 0.0, 0.0),
        ],
        green=[
            (0.0, 0.988235294118, 0.988235294118),
            (0.125, 0.960784313725, 0.960784313725),
            (0.25, 0.913725490196, 0.913725490196),
            (0.375, 0.850980392157, 0.850980392157),
            (0.5, 0.76862745098, 0.76862745098),
            (0.625, 0.670588235294, 0.670588235294),
            (0.75, 0.545098039216, 0.545098039216),
            (0.875, 0.427450980392, 0.427450980392),
            (1.0, 0.266666666667, 0.266666666667),
        ],
        blue=[
            (0.0, 0.960784313725, 0.960784313725),
            (0.125, 0.878431372549, 0.878431372549),
            (0.25, 0.752941176471, 0.752941176471),
            (0.375, 0.607843137255, 0.607843137255),
            (0.5, 0.462745098039, 0.462745098039),
            (0.625, 0.364705882353, 0.364705882353),
            (0.75, 0.270588235294, 0.270588235294),
            (0.875, 0.172549019608, 0.172549019608),
            (1.0, 0.105882352941, 0.105882352941),
        ],
    )
    return ColorMapper.from_segment_map(_data, range=range, **traits)


def Greys(range, **traits):
    """Generator for the 'Greys' colormap from ColorBrewer.

    Although the ColorBrewer colormaps are defined as discrete colormaps, we
    create continuous colormaps from them by linear interpolation in RGB
    colorspace.
    """
    _data = dict(
        red=[
            (0.0, 1.0, 1.0),
            (0.125, 0.941176470588, 0.941176470588),
            (0.25, 0.850980392157, 0.850980392157),
            (0.375, 0.741176470588, 0.741176470588),
            (0.5, 0.588235294118, 0.588235294118),
            (0.625, 0.450980392157, 0.450980392157),
            (0.75, 0.321568627451, 0.321568627451),
            (0.875, 0.145098039216, 0.145098039216),
            (1.0, 0.0, 0.0),
        ],
        green=[
            (0.0, 1.0, 1.0),
            (0.125, 0.941176470588, 0.941176470588),
            (0.25, 0.850980392157, 0.850980392157),
            (0.375, 0.741176470588, 0.741176470588),
            (0.5, 0.588235294118, 0.588235294118),
            (0.625, 0.450980392157, 0.450980392157),
            (0.75, 0.321568627451, 0.321568627451),
            (0.875, 0.145098039216, 0.145098039216),
            (1.0, 0.0, 0.0),
        ],
        blue=[
            (0.0, 1.0, 1.0),
            (0.125, 0.941176470588, 0.941176470588),
            (0.25, 0.850980392157, 0.850980392157),
            (0.375, 0.741176470588, 0.741176470588),
            (0.5, 0.588235294118, 0.588235294118),
            (0.625, 0.450980392157, 0.450980392157),
            (0.75, 0.321568627451, 0.321568627451),
            (0.875, 0.145098039216, 0.145098039216),
            (1.0, 0.0, 0.0),
        ],
    )
    return ColorMapper.from_segment_map(_data, range=range, **traits)


def OrRd(range, **traits):
    """Generator for the 'OrRd' colormap from ColorBrewer.

    Although the ColorBrewer colormaps are defined as discrete colormaps, we
    create continuous colormaps from them by linear interpolation in RGB
    colorspace.
    """
    _data = dict(
        red=[
            (0.0, 1.0, 1.0),
            (0.125, 0.996078431373, 0.996078431373),
            (0.25, 0.992156862745, 0.992156862745),
            (0.375, 0.992156862745, 0.992156862745),
            (0.5, 0.988235294118, 0.988235294118),
            (0.625, 0.937254901961, 0.937254901961),
            (0.75, 0.843137254902, 0.843137254902),
            (0.875, 0.701960784314, 0.701960784314),
            (1.0, 0.498039215686, 0.498039215686),
        ],
        green=[
            (0.0, 0.96862745098, 0.96862745098),
            (0.125, 0.909803921569, 0.909803921569),
            (0.25, 0.83137254902, 0.83137254902),
            (0.375, 0.733333333333, 0.733333333333),
            (0.5, 0.552941176471, 0.552941176471),
            (0.625, 0.396078431373, 0.396078431373),
            (0.75, 0.188235294118, 0.188235294118),
            (0.875, 0.0, 0.0),
            (1.0, 0.0, 0.0),
        ],
        blue=[
            (0.0, 0.925490196078, 0.925490196078),
            (0.125, 0.78431372549, 0.78431372549),
            (0.25, 0.619607843137, 0.619607843137),
            (0.375, 0.517647058824, 0.517647058824),
            (0.5, 0.349019607843, 0.349019607843),
            (0.625, 0.282352941176, 0.282352941176),
            (0.75, 0.121568627451, 0.121568627451),
            (0.875, 0.0, 0.0),
            (1.0, 0.0, 0.0),
        ],
    )
    return ColorMapper.from_segment_map(_data, range=range, **traits)


def Oranges(range, **traits):
    """Generator for the 'Oranges' colormap from ColorBrewer.

    Although the ColorBrewer colormaps are defined as discrete colormaps, we
    create continuous colormaps from them by linear interpolation in RGB
    colorspace.
    """
    _data = dict(
        red=[
            (0.0, 1.0, 1.0),
            (0.125, 0.996078431373, 0.996078431373),
            (0.25, 0.992156862745, 0.992156862745),
            (0.375, 0.992156862745, 0.992156862745),
            (0.5, 0.992156862745, 0.992156862745),
            (0.625, 0.945098039216, 0.945098039216),
            (0.75, 0.850980392157, 0.850980392157),
            (0.875, 0.650980392157, 0.650980392157),
            (1.0, 0.498039215686, 0.498039215686),
        ],
        green=[
            (0.0, 0.960784313725, 0.960784313725),
            (0.125, 0.901960784314, 0.901960784314),
            (0.25, 0.81568627451, 0.81568627451),
            (0.375, 0.682352941176, 0.682352941176),
            (0.5, 0.552941176471, 0.552941176471),
            (0.625, 0.411764705882, 0.411764705882),
            (0.75, 0.282352941176, 0.282352941176),
            (0.875, 0.211764705882, 0.211764705882),
            (1.0, 0.152941176471, 0.152941176471),
        ],
        blue=[
            (0.0, 0.921568627451, 0.921568627451),
            (0.125, 0.807843137255, 0.807843137255),
            (0.25, 0.635294117647, 0.635294117647),
            (0.375, 0.419607843137, 0.419607843137),
            (0.5, 0.235294117647, 0.235294117647),
            (0.625, 0.0745098039216, 0.0745098039216),
            (0.75, 0.00392156862745, 0.00392156862745),
            (0.875, 0.0117647058824, 0.0117647058824),
            (1.0, 0.0156862745098, 0.0156862745098),
        ],
    )
    return ColorMapper.from_segment_map(_data, range=range, **traits)


def PRGn(range, **traits):
    """Generator for the 'PRGn' colormap from ColorBrewer.

    This is a diverging colormap. It is good for data which is centered around
    a "special" value, like 0.

    Although the ColorBrewer colormaps are defined as discrete colormaps, we
    create continuous colormaps from them by linear interpolation in RGB
    colorspace.
    """
    _data = dict(
        red=[
            (0.0, 0.250980392157, 0.250980392157),
            (0.1, 0.462745098039, 0.462745098039),
            (0.2, 0.6, 0.6),
            (0.3, 0.760784313725, 0.760784313725),
            (0.4, 0.905882352941, 0.905882352941),
            (0.5, 0.96862745098, 0.96862745098),
            (0.6, 0.850980392157, 0.850980392157),
            (0.7, 0.650980392157, 0.650980392157),
            (0.8, 0.352941176471, 0.352941176471),
            (0.9, 0.105882352941, 0.105882352941),
            (1.0, 0.0, 0.0),
        ],
        green=[
            (0.0, 0.0, 0.0),
            (0.1, 0.164705882353, 0.164705882353),
            (0.2, 0.439215686275, 0.439215686275),
            (0.3, 0.647058823529, 0.647058823529),
            (0.4, 0.83137254902, 0.83137254902),
            (0.5, 0.96862745098, 0.96862745098),
            (0.6, 0.941176470588, 0.941176470588),
            (0.7, 0.858823529412, 0.858823529412),
            (0.8, 0.682352941176, 0.682352941176),
            (0.9, 0.470588235294, 0.470588235294),
            (1.0, 0.266666666667, 0.266666666667),
        ],
        blue=[
            (0.0, 0.294117647059, 0.294117647059),
            (0.1, 0.513725490196, 0.513725490196),
            (0.2, 0.670588235294, 0.670588235294),
            (0.3, 0.811764705882, 0.811764705882),
            (0.4, 0.909803921569, 0.909803921569),
            (0.5, 0.96862745098, 0.96862745098),
            (0.6, 0.827450980392, 0.827450980392),
            (0.7, 0.627450980392, 0.627450980392),
            (0.8, 0.380392156863, 0.380392156863),
            (0.9, 0.21568627451, 0.21568627451),
            (1.0, 0.105882352941, 0.105882352941),
        ],
    )
    return ColorMapper.from_segment_map(_data, range=range, **traits)


def PiYG(range, **traits):
    """Generator for the 'PiYG' colormap from ColorBrewer.

    This is a diverging colormap. It is good for data which is centered around
    a "special" value, like 0.

    Although the ColorBrewer colormaps are defined as discrete colormaps, we
    create continuous colormaps from them by linear interpolation in RGB
    colorspace.
    """
    _data = dict(
        red=[
            (0.0, 0.556862745098, 0.556862745098),
            (0.1, 0.772549019608, 0.772549019608),
            (0.2, 0.870588235294, 0.870588235294),
            (0.3, 0.945098039216, 0.945098039216),
            (0.4, 0.992156862745, 0.992156862745),
            (0.5, 0.96862745098, 0.96862745098),
            (0.6, 0.901960784314, 0.901960784314),
            (0.7, 0.721568627451, 0.721568627451),
            (0.8, 0.498039215686, 0.498039215686),
            (0.9, 0.301960784314, 0.301960784314),
            (1.0, 0.152941176471, 0.152941176471),
        ],
        green=[
            (0.0, 0.00392156862745, 0.00392156862745),
            (0.1, 0.105882352941, 0.105882352941),
            (0.2, 0.466666666667, 0.466666666667),
            (0.3, 0.713725490196, 0.713725490196),
            (0.4, 0.878431372549, 0.878431372549),
            (0.5, 0.96862745098, 0.96862745098),
            (0.6, 0.960784313725, 0.960784313725),
            (0.7, 0.882352941176, 0.882352941176),
            (0.8, 0.737254901961, 0.737254901961),
            (0.9, 0.572549019608, 0.572549019608),
            (1.0, 0.392156862745, 0.392156862745),
        ],
        blue=[
            (0.0, 0.321568627451, 0.321568627451),
            (0.1, 0.490196078431, 0.490196078431),
            (0.2, 0.682352941176, 0.682352941176),
            (0.3, 0.854901960784, 0.854901960784),
            (0.4, 0.937254901961, 0.937254901961),
            (0.5, 0.96862745098, 0.96862745098),
            (0.6, 0.81568627451, 0.81568627451),
            (0.7, 0.525490196078, 0.525490196078),
            (0.8, 0.254901960784, 0.254901960784),
            (0.9, 0.129411764706, 0.129411764706),
            (1.0, 0.0980392156863, 0.0980392156863),
        ],
    )
    return ColorMapper.from_segment_map(_data, range=range, **traits)


def PuBu(range, **traits):
    """Generator for the 'PuBu' colormap from ColorBrewer.

    Although the ColorBrewer colormaps are defined as discrete colormaps, we
    create continuous colormaps from them by linear interpolation in RGB
    colorspace.
    """
    _data = dict(
        red=[
            (0.0, 1.0, 1.0),
            (0.125, 0.925490196078, 0.925490196078),
            (0.25, 0.81568627451, 0.81568627451),
            (0.375, 0.650980392157, 0.650980392157),
            (0.5, 0.454901960784, 0.454901960784),
            (0.625, 0.211764705882, 0.211764705882),
            (0.75, 0.0196078431373, 0.0196078431373),
            (0.875, 0.0156862745098, 0.0156862745098),
            (1.0, 0.0078431372549, 0.0078431372549),
        ],
        green=[
            (0.0, 0.96862745098, 0.96862745098),
            (0.125, 0.905882352941, 0.905882352941),
            (0.25, 0.819607843137, 0.819607843137),
            (0.375, 0.741176470588, 0.741176470588),
            (0.5, 0.662745098039, 0.662745098039),
            (0.625, 0.564705882353, 0.564705882353),
            (0.75, 0.439215686275, 0.439215686275),
            (0.875, 0.352941176471, 0.352941176471),
            (1.0, 0.219607843137, 0.219607843137),
        ],
        blue=[
            (0.0, 0.98431372549, 0.98431372549),
            (0.125, 0.949019607843, 0.949019607843),
            (0.25, 0.901960784314, 0.901960784314),
            (0.375, 0.858823529412, 0.858823529412),
            (0.5, 0.811764705882, 0.811764705882),
            (0.625, 0.752941176471, 0.752941176471),
            (0.75, 0.690196078431, 0.690196078431),
            (0.875, 0.552941176471, 0.552941176471),
            (1.0, 0.345098039216, 0.345098039216),
        ],
    )
    return ColorMapper.from_segment_map(_data, range=range, **traits)


def PuBuGn(range, **traits):
    """Generator for the 'PuBuGn' colormap from ColorBrewer.

    Although the ColorBrewer colormaps are defined as discrete colormaps, we
    create continuous colormaps from them by linear interpolation in RGB
    colorspace.
    """
    _data = dict(
        red=[
            (0.0, 1.0, 1.0),
            (0.125, 0.925490196078, 0.925490196078),
            (0.25, 0.81568627451, 0.81568627451),
            (0.375, 0.650980392157, 0.650980392157),
            (0.5, 0.403921568627, 0.403921568627),
            (0.625, 0.211764705882, 0.211764705882),
            (0.75, 0.0078431372549, 0.0078431372549),
            (0.875, 0.00392156862745, 0.00392156862745),
            (1.0, 0.00392156862745, 0.00392156862745),
        ],
        green=[
            (0.0, 0.96862745098, 0.96862745098),
            (0.125, 0.886274509804, 0.886274509804),
            (0.25, 0.819607843137, 0.819607843137),
            (0.375, 0.741176470588, 0.741176470588),
            (0.5, 0.662745098039, 0.662745098039),
            (0.625, 0.564705882353, 0.564705882353),
            (0.75, 0.505882352941, 0.505882352941),
            (0.875, 0.423529411765, 0.423529411765),
            (1.0, 0.274509803922, 0.274509803922),
        ],
        blue=[
            (0.0, 0.98431372549, 0.98431372549),
            (0.125, 0.941176470588, 0.941176470588),
            (0.25, 0.901960784314, 0.901960784314),
            (0.375, 0.858823529412, 0.858823529412),
            (0.5, 0.811764705882, 0.811764705882),
            (0.625, 0.752941176471, 0.752941176471),
            (0.75, 0.541176470588, 0.541176470588),
            (0.875, 0.349019607843, 0.349019607843),
            (1.0, 0.211764705882, 0.211764705882),
        ],
    )
    return ColorMapper.from_segment_map(_data, range=range, **traits)


def PuOr(range, **traits):
    """Generator for the 'PuOr' colormap from ColorBrewer.

    This is a diverging colormap. It is good for data which is centered around
    a "special" value, like 0.

    Although the ColorBrewer colormaps are defined as discrete colormaps, we
    create continuous colormaps from them by linear interpolation in RGB
    colorspace.
    """
    _data = dict(
        red=[
            (0.0, 0.498039215686, 0.498039215686),
            (0.1, 0.701960784314, 0.701960784314),
            (0.2, 0.878431372549, 0.878431372549),
            (0.3, 0.992156862745, 0.992156862745),
            (0.4, 0.996078431373, 0.996078431373),
            (0.5, 0.96862745098, 0.96862745098),
            (0.6, 0.847058823529, 0.847058823529),
            (0.7, 0.698039215686, 0.698039215686),
            (0.8, 0.501960784314, 0.501960784314),
            (0.9, 0.329411764706, 0.329411764706),
            (1.0, 0.176470588235, 0.176470588235),
        ],
        green=[
            (0.0, 0.23137254902, 0.23137254902),
            (0.1, 0.345098039216, 0.345098039216),
            (0.2, 0.509803921569, 0.509803921569),
            (0.3, 0.721568627451, 0.721568627451),
            (0.4, 0.878431372549, 0.878431372549),
            (0.5, 0.96862745098, 0.96862745098),
            (0.6, 0.854901960784, 0.854901960784),
            (0.7, 0.670588235294, 0.670588235294),
            (0.8, 0.450980392157, 0.450980392157),
            (0.9, 0.152941176471, 0.152941176471),
            (1.0, 0.0, 0.0),
        ],
        blue=[
            (0.0, 0.0313725490196, 0.0313725490196),
            (0.1, 0.0235294117647, 0.0235294117647),
            (0.2, 0.078431372549, 0.078431372549),
            (0.3, 0.388235294118, 0.388235294118),
            (0.4, 0.713725490196, 0.713725490196),
            (0.5, 0.96862745098, 0.96862745098),
            (0.6, 0.921568627451, 0.921568627451),
            (0.7, 0.823529411765, 0.823529411765),
            (0.8, 0.674509803922, 0.674509803922),
            (0.9, 0.533333333333, 0.533333333333),
            (1.0, 0.294117647059, 0.294117647059),
        ],
    )
    return ColorMapper.from_segment_map(_data, range=range, **traits)


def PuRd(range, **traits):
    """Generator for the 'PuRd' colormap from ColorBrewer.

    Although the ColorBrewer colormaps are defined as discrete colormaps, we
    create continuous colormaps from them by linear interpolation in RGB
    colorspace.
    """
    _data = dict(
        red=[
            (0.0, 0.96862745098, 0.96862745098),
            (0.125, 0.905882352941, 0.905882352941),
            (0.25, 0.83137254902, 0.83137254902),
            (0.375, 0.788235294118, 0.788235294118),
            (0.5, 0.874509803922, 0.874509803922),
            (0.625, 0.905882352941, 0.905882352941),
            (0.75, 0.807843137255, 0.807843137255),
            (0.875, 0.596078431373, 0.596078431373),
            (1.0, 0.403921568627, 0.403921568627),
        ],
        green=[
            (0.0, 0.956862745098, 0.956862745098),
            (0.125, 0.882352941176, 0.882352941176),
            (0.25, 0.725490196078, 0.725490196078),
            (0.375, 0.580392156863, 0.580392156863),
            (0.5, 0.396078431373, 0.396078431373),
            (0.625, 0.160784313725, 0.160784313725),
            (0.75, 0.0705882352941, 0.0705882352941),
            (0.875, 0.0, 0.0),
            (1.0, 0.0, 0.0),
        ],
        blue=[
            (0.0, 0.976470588235, 0.976470588235),
            (0.125, 0.937254901961, 0.937254901961),
            (0.25, 0.854901960784, 0.854901960784),
            (0.375, 0.780392156863, 0.780392156863),
            (0.5, 0.690196078431, 0.690196078431),
            (0.625, 0.541176470588, 0.541176470588),
            (0.75, 0.337254901961, 0.337254901961),
            (0.875, 0.262745098039, 0.262745098039),
            (1.0, 0.121568627451, 0.121568627451),
        ],
    )
    return ColorMapper.from_segment_map(_data, range=range, **traits)


def Purples(range, **traits):
    """Generator for the 'Purples' colormap from ColorBrewer.

    Although the ColorBrewer colormaps are defined as discrete colormaps, we
    create continuous colormaps from them by linear interpolation in RGB
    colorspace.
    """
    _data = dict(
        red=[
            (0.0, 0.988235294118, 0.988235294118),
            (0.125, 0.937254901961, 0.937254901961),
            (0.25, 0.854901960784, 0.854901960784),
            (0.375, 0.737254901961, 0.737254901961),
            (0.5, 0.619607843137, 0.619607843137),
            (0.625, 0.501960784314, 0.501960784314),
            (0.75, 0.41568627451, 0.41568627451),
            (0.875, 0.329411764706, 0.329411764706),
            (1.0, 0.247058823529, 0.247058823529),
        ],
        green=[
            (0.0, 0.98431372549, 0.98431372549),
            (0.125, 0.929411764706, 0.929411764706),
            (0.25, 0.854901960784, 0.854901960784),
            (0.375, 0.741176470588, 0.741176470588),
            (0.5, 0.603921568627, 0.603921568627),
            (0.625, 0.490196078431, 0.490196078431),
            (0.75, 0.317647058824, 0.317647058824),
            (0.875, 0.152941176471, 0.152941176471),
            (1.0, 0.0, 0.0),
        ],
        blue=[
            (0.0, 0.992156862745, 0.992156862745),
            (0.125, 0.960784313725, 0.960784313725),
            (0.25, 0.921568627451, 0.921568627451),
            (0.375, 0.862745098039, 0.862745098039),
            (0.5, 0.78431372549, 0.78431372549),
            (0.625, 0.729411764706, 0.729411764706),
            (0.75, 0.639215686275, 0.639215686275),
            (0.875, 0.560784313725, 0.560784313725),
            (1.0, 0.490196078431, 0.490196078431),
        ],
    )
    return ColorMapper.from_segment_map(_data, range=range, **traits)


def RdBu(range, **traits):
    """Generator for the 'RdBu' colormap from ColorBrewer.

    This is a diverging colormap. It is good for data which is centered around
    a "special" value, like 0.

    Although the ColorBrewer colormaps are defined as discrete colormaps, we
    create continuous colormaps from them by linear interpolation in RGB
    colorspace.
    """
    _data = dict(
        red=[
            (0.0, 0.403921568627, 0.403921568627),
            (0.1, 0.698039215686, 0.698039215686),
            (0.2, 0.839215686275, 0.839215686275),
            (0.3, 0.956862745098, 0.956862745098),
            (0.4, 0.992156862745, 0.992156862745),
            (0.5, 0.96862745098, 0.96862745098),
            (0.6, 0.819607843137, 0.819607843137),
            (0.7, 0.572549019608, 0.572549019608),
            (0.8, 0.262745098039, 0.262745098039),
            (0.9, 0.129411764706, 0.129411764706),
            (1.0, 0.0196078431373, 0.0196078431373),
        ],
        green=[
            (0.0, 0.0, 0.0),
            (0.1, 0.0941176470588, 0.0941176470588),
            (0.2, 0.376470588235, 0.376470588235),
            (0.3, 0.647058823529, 0.647058823529),
            (0.4, 0.858823529412, 0.858823529412),
            (0.5, 0.96862745098, 0.96862745098),
            (0.6, 0.898039215686, 0.898039215686),
            (0.7, 0.772549019608, 0.772549019608),
            (0.8, 0.576470588235, 0.576470588235),
            (0.9, 0.4, 0.4),
            (1.0, 0.188235294118, 0.188235294118),
        ],
        blue=[
            (0.0, 0.121568627451, 0.121568627451),
            (0.1, 0.16862745098, 0.16862745098),
            (0.2, 0.301960784314, 0.301960784314),
            (0.3, 0.509803921569, 0.509803921569),
            (0.4, 0.780392156863, 0.780392156863),
            (0.5, 0.96862745098, 0.96862745098),
            (0.6, 0.941176470588, 0.941176470588),
            (0.7, 0.870588235294, 0.870588235294),
            (0.8, 0.764705882353, 0.764705882353),
            (0.9, 0.674509803922, 0.674509803922),
            (1.0, 0.380392156863, 0.380392156863),
        ],
    )
    return ColorMapper.from_segment_map(_data, range=range, **traits)


def RdGy(range, **traits):
    """Generator for the 'RdGy' colormap from ColorBrewer.

    This is a diverging colormap. It is good for data which is centered around
    a "special" value, like 0.

    Although the ColorBrewer colormaps are defined as discrete colormaps, we
    create continuous colormaps from them by linear interpolation in RGB
    colorspace.
    """
    _data = dict(
        red=[
            (0.0, 0.403921568627, 0.403921568627),
            (0.1, 0.698039215686, 0.698039215686),
            (0.2, 0.839215686275, 0.839215686275),
            (0.3, 0.956862745098, 0.956862745098),
            (0.4, 0.992156862745, 0.992156862745),
            (0.5, 1.0, 1.0),
            (0.6, 0.878431372549, 0.878431372549),
            (0.7, 0.729411764706, 0.729411764706),
            (0.8, 0.529411764706, 0.529411764706),
            (0.9, 0.301960784314, 0.301960784314),
            (1.0, 0.101960784314, 0.101960784314),
        ],
        green=[
            (0.0, 0.0, 0.0),
            (0.1, 0.0941176470588, 0.0941176470588),
            (0.2, 0.376470588235, 0.376470588235),
            (0.3, 0.647058823529, 0.647058823529),
            (0.4, 0.858823529412, 0.858823529412),
            (0.5, 1.0, 1.0),
            (0.6, 0.878431372549, 0.878431372549),
            (0.7, 0.729411764706, 0.729411764706),
            (0.8, 0.529411764706, 0.529411764706),
            (0.9, 0.301960784314, 0.301960784314),
            (1.0, 0.101960784314, 0.101960784314),
        ],
        blue=[
            (0.0, 0.121568627451, 0.121568627451),
            (0.1, 0.16862745098, 0.16862745098),
            (0.2, 0.301960784314, 0.301960784314),
            (0.3, 0.509803921569, 0.509803921569),
            (0.4, 0.780392156863, 0.780392156863),
            (0.5, 1.0, 1.0),
            (0.6, 0.878431372549, 0.878431372549),
            (0.7, 0.729411764706, 0.729411764706),
            (0.8, 0.529411764706, 0.529411764706),
            (0.9, 0.301960784314, 0.301960784314),
            (1.0, 0.101960784314, 0.101960784314),
        ],
    )
    return ColorMapper.from_segment_map(_data, range=range, **traits)


def RdPu(range, **traits):
    """Generator for the 'RdPu' colormap from ColorBrewer.

    Although the ColorBrewer colormaps are defined as discrete colormaps, we
    create continuous colormaps from them by linear interpolation in RGB
    colorspace.
    """
    _data = dict(
        red=[
            (0.0, 1.0, 1.0),
            (0.125, 0.992156862745, 0.992156862745),
            (0.25, 0.988235294118, 0.988235294118),
            (0.375, 0.980392156863, 0.980392156863),
            (0.5, 0.96862745098, 0.96862745098),
            (0.625, 0.866666666667, 0.866666666667),
            (0.75, 0.682352941176, 0.682352941176),
            (0.875, 0.478431372549, 0.478431372549),
            (1.0, 0.286274509804, 0.286274509804),
        ],
        green=[
            (0.0, 0.96862745098, 0.96862745098),
            (0.125, 0.878431372549, 0.878431372549),
            (0.25, 0.772549019608, 0.772549019608),
            (0.375, 0.623529411765, 0.623529411765),
            (0.5, 0.407843137255, 0.407843137255),
            (0.625, 0.203921568627, 0.203921568627),
            (0.75, 0.00392156862745, 0.00392156862745),
            (0.875, 0.00392156862745, 0.00392156862745),
            (1.0, 0.0, 0.0),
        ],
        blue=[
            (0.0, 0.952941176471, 0.952941176471),
            (0.125, 0.866666666667, 0.866666666667),
            (0.25, 0.752941176471, 0.752941176471),
            (0.375, 0.709803921569, 0.709803921569),
            (0.5, 0.63137254902, 0.63137254902),
            (0.625, 0.592156862745, 0.592156862745),
            (0.75, 0.494117647059, 0.494117647059),
            (0.875, 0.466666666667, 0.466666666667),
            (1.0, 0.41568627451, 0.41568627451),
        ],
    )
    return ColorMapper.from_segment_map(_data, range=range, **traits)


def RdYlBu(range, **traits):
    """Generator for the 'RdYlBu' colormap from ColorBrewer.

    This is a diverging colormap. It is good for data which is centered around
    a "special" value, like 0.

    Although the ColorBrewer colormaps are defined as discrete colormaps, we
    create continuous colormaps from them by linear interpolation in RGB
    colorspace.
    """
    _data = dict(
        red=[
            (0.0, 0.647058823529, 0.647058823529),
            (0.1, 0.843137254902, 0.843137254902),
            (0.2, 0.956862745098, 0.956862745098),
            (0.3, 0.992156862745, 0.992156862745),
            (0.4, 0.996078431373, 0.996078431373),
            (0.5, 1.0, 1.0),
            (0.6, 0.878431372549, 0.878431372549),
            (0.7, 0.670588235294, 0.670588235294),
            (0.8, 0.454901960784, 0.454901960784),
            (0.9, 0.270588235294, 0.270588235294),
            (1.0, 0.192156862745, 0.192156862745),
        ],
        green=[
            (0.0, 0.0, 0.0),
            (0.1, 0.188235294118, 0.188235294118),
            (0.2, 0.427450980392, 0.427450980392),
            (0.3, 0.682352941176, 0.682352941176),
            (0.4, 0.878431372549, 0.878431372549),
            (0.5, 1.0, 1.0),
            (0.6, 0.952941176471, 0.952941176471),
            (0.7, 0.850980392157, 0.850980392157),
            (0.8, 0.678431372549, 0.678431372549),
            (0.9, 0.458823529412, 0.458823529412),
            (1.0, 0.211764705882, 0.211764705882),
        ],
        blue=[
            (0.0, 0.149019607843, 0.149019607843),
            (0.1, 0.152941176471, 0.152941176471),
            (0.2, 0.262745098039, 0.262745098039),
            (0.3, 0.380392156863, 0.380392156863),
            (0.4, 0.564705882353, 0.564705882353),
            (0.5, 0.749019607843, 0.749019607843),
            (0.6, 0.972549019608, 0.972549019608),
            (0.7, 0.913725490196, 0.913725490196),
            (0.8, 0.819607843137, 0.819607843137),
            (0.9, 0.705882352941, 0.705882352941),
            (1.0, 0.58431372549, 0.58431372549),
        ],
    )
    return ColorMapper.from_segment_map(_data, range=range, **traits)


def RdYlGn(range, **traits):
    """Generator for the 'RdYlGn' colormap from ColorBrewer.

    This is a diverging colormap. It is good for data which is centered around
    a "special" value, like 0.

    Although the ColorBrewer colormaps are defined as discrete colormaps, we
    create continuous colormaps from them by linear interpolation in RGB
    colorspace.
    """
    _data = dict(
        red=[
            (0.0, 0.647058823529, 0.647058823529),
            (0.1, 0.843137254902, 0.843137254902),
            (0.2, 0.956862745098, 0.956862745098),
            (0.3, 0.992156862745, 0.992156862745),
            (0.4, 0.996078431373, 0.996078431373),
            (0.5, 1.0, 1.0),
            (0.6, 0.850980392157, 0.850980392157),
            (0.7, 0.650980392157, 0.650980392157),
            (0.8, 0.4, 0.4),
            (0.9, 0.101960784314, 0.101960784314),
            (1.0, 0.0, 0.0),
        ],
        green=[
            (0.0, 0.0, 0.0),
            (0.1, 0.188235294118, 0.188235294118),
            (0.2, 0.427450980392, 0.427450980392),
            (0.3, 0.682352941176, 0.682352941176),
            (0.4, 0.878431372549, 0.878431372549),
            (0.5, 1.0, 1.0),
            (0.6, 0.937254901961, 0.937254901961),
            (0.7, 0.850980392157, 0.850980392157),
            (0.8, 0.741176470588, 0.741176470588),
            (0.9, 0.596078431373, 0.596078431373),
            (1.0, 0.407843137255, 0.407843137255),
        ],
        blue=[
            (0.0, 0.149019607843, 0.149019607843),
            (0.1, 0.152941176471, 0.152941176471),
            (0.2, 0.262745098039, 0.262745098039),
            (0.3, 0.380392156863, 0.380392156863),
            (0.4, 0.545098039216, 0.545098039216),
            (0.5, 0.749019607843, 0.749019607843),
            (0.6, 0.545098039216, 0.545098039216),
            (0.7, 0.41568627451, 0.41568627451),
            (0.8, 0.388235294118, 0.388235294118),
            (0.9, 0.313725490196, 0.313725490196),
            (1.0, 0.21568627451, 0.21568627451),
        ],
    )
    return ColorMapper.from_segment_map(_data, range=range, **traits)


def Reds(range, **traits):
    """Generator for the 'Reds' colormap from ColorBrewer.

    Although the ColorBrewer colormaps are defined as discrete colormaps, we
    create continuous colormaps from them by linear interpolation in RGB
    colorspace.
    """
    _data = dict(
        red=[
            (0.0, 1.0, 1.0),
            (0.125, 0.996078431373, 0.996078431373),
            (0.25, 0.988235294118, 0.988235294118),
            (0.375, 0.988235294118, 0.988235294118),
            (0.5, 0.98431372549, 0.98431372549),
            (0.625, 0.937254901961, 0.937254901961),
            (0.75, 0.796078431373, 0.796078431373),
            (0.875, 0.647058823529, 0.647058823529),
            (1.0, 0.403921568627, 0.403921568627),
        ],
        green=[
            (0.0, 0.960784313725, 0.960784313725),
            (0.125, 0.878431372549, 0.878431372549),
            (0.25, 0.733333333333, 0.733333333333),
            (0.375, 0.572549019608, 0.572549019608),
            (0.5, 0.41568627451, 0.41568627451),
            (0.625, 0.23137254902, 0.23137254902),
            (0.75, 0.0941176470588, 0.0941176470588),
            (0.875, 0.0588235294118, 0.0588235294118),
            (1.0, 0.0, 0.0),
        ],
        blue=[
            (0.0, 0.941176470588, 0.941176470588),
            (0.125, 0.823529411765, 0.823529411765),
            (0.25, 0.63137254902, 0.63137254902),
            (0.375, 0.447058823529, 0.447058823529),
            (0.5, 0.290196078431, 0.290196078431),
            (0.625, 0.172549019608, 0.172549019608),
            (0.75, 0.113725490196, 0.113725490196),
            (0.875, 0.0823529411765, 0.0823529411765),
            (1.0, 0.0509803921569, 0.0509803921569),
        ],
    )
    return ColorMapper.from_segment_map(_data, range=range, **traits)


def Spectral(range, **traits):
    """Generator for the 'Spectral' colormap from ColorBrewer.

    This is a diverging colormap. It is good for data which is centered around
    a "special" value, like 0.

    Although the ColorBrewer colormaps are defined as discrete colormaps, we
    create continuous colormaps from them by linear interpolation in RGB
    colorspace.
    """
    _data = dict(
        red=[
            (0.0, 0.619607843137, 0.619607843137),
            (0.1, 0.835294117647, 0.835294117647),
            (0.2, 0.956862745098, 0.956862745098),
            (0.3, 0.992156862745, 0.992156862745),
            (0.4, 0.996078431373, 0.996078431373),
            (0.5, 1.0, 1.0),
            (0.6, 0.901960784314, 0.901960784314),
            (0.7, 0.670588235294, 0.670588235294),
            (0.8, 0.4, 0.4),
            (0.9, 0.196078431373, 0.196078431373),
            (1.0, 0.36862745098, 0.36862745098),
        ],
        green=[
            (0.0, 0.00392156862745, 0.00392156862745),
            (0.1, 0.243137254902, 0.243137254902),
            (0.2, 0.427450980392, 0.427450980392),
            (0.3, 0.682352941176, 0.682352941176),
            (0.4, 0.878431372549, 0.878431372549),
            (0.5, 1.0, 1.0),
            (0.6, 0.960784313725, 0.960784313725),
            (0.7, 0.866666666667, 0.866666666667),
            (0.8, 0.760784313725, 0.760784313725),
            (0.9, 0.533333333333, 0.533333333333),
            (1.0, 0.309803921569, 0.309803921569),
        ],
        blue=[
            (0.0, 0.258823529412, 0.258823529412),
            (0.1, 0.309803921569, 0.309803921569),
            (0.2, 0.262745098039, 0.262745098039),
            (0.3, 0.380392156863, 0.380392156863),
            (0.4, 0.545098039216, 0.545098039216),
            (0.5, 0.749019607843, 0.749019607843),
            (0.6, 0.596078431373, 0.596078431373),
            (0.7, 0.643137254902, 0.643137254902),
            (0.8, 0.647058823529, 0.647058823529),
            (0.9, 0.741176470588, 0.741176470588),
            (1.0, 0.635294117647, 0.635294117647),
        ],
    )
    return ColorMapper.from_segment_map(_data, range=range, **traits)


def YlGn(range, **traits):
    """Generator for the 'YlGn' colormap from ColorBrewer.

    Although the ColorBrewer colormaps are defined as discrete colormaps, we
    create continuous colormaps from them by linear interpolation in RGB
    colorspace.
    """
    _data = dict(
        red=[
            (0.0, 1.0, 1.0),
            (0.125, 0.96862745098, 0.96862745098),
            (0.25, 0.850980392157, 0.850980392157),
            (0.375, 0.678431372549, 0.678431372549),
            (0.5, 0.470588235294, 0.470588235294),
            (0.625, 0.254901960784, 0.254901960784),
            (0.75, 0.137254901961, 0.137254901961),
            (0.875, 0.0, 0.0),
            (1.0, 0.0, 0.0),
        ],
        green=[
            (0.0, 1.0, 1.0),
            (0.125, 0.988235294118, 0.988235294118),
            (0.25, 0.941176470588, 0.941176470588),
            (0.375, 0.866666666667, 0.866666666667),
            (0.5, 0.776470588235, 0.776470588235),
            (0.625, 0.670588235294, 0.670588235294),
            (0.75, 0.517647058824, 0.517647058824),
            (0.875, 0.407843137255, 0.407843137255),
            (1.0, 0.270588235294, 0.270588235294),
        ],
        blue=[
            (0.0, 0.898039215686, 0.898039215686),
            (0.125, 0.725490196078, 0.725490196078),
            (0.25, 0.639215686275, 0.639215686275),
            (0.375, 0.556862745098, 0.556862745098),
            (0.5, 0.474509803922, 0.474509803922),
            (0.625, 0.364705882353, 0.364705882353),
            (0.75, 0.262745098039, 0.262745098039),
            (0.875, 0.21568627451, 0.21568627451),
            (1.0, 0.160784313725, 0.160784313725),
        ],
    )
    return ColorMapper.from_segment_map(_data, range=range, **traits)


def YlGnBu(range, **traits):
    """Generator for the 'YlGnBu' colormap from ColorBrewer.

    Although the ColorBrewer colormaps are defined as discrete colormaps, we
    create continuous colormaps from them by linear interpolation in RGB
    colorspace.
    """
    _data = dict(
        red=[
            (0.0, 1.0, 1.0),
            (0.125, 0.929411764706, 0.929411764706),
            (0.25, 0.780392156863, 0.780392156863),
            (0.375, 0.498039215686, 0.498039215686),
            (0.5, 0.254901960784, 0.254901960784),
            (0.625, 0.113725490196, 0.113725490196),
            (0.75, 0.133333333333, 0.133333333333),
            (0.875, 0.145098039216, 0.145098039216),
            (1.0, 0.0313725490196, 0.0313725490196),
        ],
        green=[
            (0.0, 1.0, 1.0),
            (0.125, 0.972549019608, 0.972549019608),
            (0.25, 0.913725490196, 0.913725490196),
            (0.375, 0.803921568627, 0.803921568627),
            (0.5, 0.713725490196, 0.713725490196),
            (0.625, 0.56862745098, 0.56862745098),
            (0.75, 0.36862745098, 0.36862745098),
            (0.875, 0.203921568627, 0.203921568627),
            (1.0, 0.113725490196, 0.113725490196),
        ],
        blue=[
            (0.0, 0.850980392157, 0.850980392157),
            (0.125, 0.694117647059, 0.694117647059),
            (0.25, 0.705882352941, 0.705882352941),
            (0.375, 0.733333333333, 0.733333333333),
            (0.5, 0.76862745098, 0.76862745098),
            (0.625, 0.752941176471, 0.752941176471),
            (0.75, 0.658823529412, 0.658823529412),
            (0.875, 0.580392156863, 0.580392156863),
            (1.0, 0.345098039216, 0.345098039216),
        ],
    )
    return ColorMapper.from_segment_map(_data, range=range, **traits)


def YlOrBr(range, **traits):
    """Generator for the 'YlOrBr' colormap from ColorBrewer.

    Although the ColorBrewer colormaps are defined as discrete colormaps, we
    create continuous colormaps from them by linear interpolation in RGB
    colorspace.
    """
    _data = dict(
        red=[
            (0.0, 1.0, 1.0),
            (0.125, 1.0, 1.0),
            (0.25, 0.996078431373, 0.996078431373),
            (0.375, 0.996078431373, 0.996078431373),
            (0.5, 0.996078431373, 0.996078431373),
            (0.625, 0.925490196078, 0.925490196078),
            (0.75, 0.8, 0.8),
            (0.875, 0.6, 0.6),
            (1.0, 0.4, 0.4),
        ],
        green=[
            (0.0, 1.0, 1.0),
            (0.125, 0.96862745098, 0.96862745098),
            (0.25, 0.890196078431, 0.890196078431),
            (0.375, 0.76862745098, 0.76862745098),
            (0.5, 0.6, 0.6),
            (0.625, 0.439215686275, 0.439215686275),
            (0.75, 0.298039215686, 0.298039215686),
            (0.875, 0.203921568627, 0.203921568627),
            (1.0, 0.145098039216, 0.145098039216),
        ],
        blue=[
            (0.0, 0.898039215686, 0.898039215686),
            (0.125, 0.737254901961, 0.737254901961),
            (0.25, 0.56862745098, 0.56862745098),
            (0.375, 0.309803921569, 0.309803921569),
            (0.5, 0.160784313725, 0.160784313725),
            (0.625, 0.078431372549, 0.078431372549),
            (0.75, 0.0078431372549, 0.0078431372549),
            (0.875, 0.0156862745098, 0.0156862745098),
            (1.0, 0.0235294117647, 0.0235294117647),
        ],
    )
    return ColorMapper.from_segment_map(_data, range=range, **traits)


def YlOrRd(range, **traits):
    """Generator for the 'YlOrRd' colormap from ColorBrewer.

    Although the ColorBrewer colormaps are defined as discrete colormaps, we
    create continuous colormaps from them by linear interpolation in RGB
    colorspace.
    """
    _data = dict(
        red=[
            (0.0, 1.0, 1.0),
            (0.125, 1.0, 1.0),
            (0.25, 0.996078431373, 0.996078431373),
            (0.375, 0.996078431373, 0.996078431373),
            (0.5, 0.992156862745, 0.992156862745),
            (0.625, 0.988235294118, 0.988235294118),
            (0.75, 0.890196078431, 0.890196078431),
            (0.875, 0.741176470588, 0.741176470588),
            (1.0, 0.501960784314, 0.501960784314),
        ],
        green=[
            (0.0, 1.0, 1.0),
            (0.125, 0.929411764706, 0.929411764706),
            (0.25, 0.850980392157, 0.850980392157),
            (0.375, 0.698039215686, 0.698039215686),
            (0.5, 0.552941176471, 0.552941176471),
            (0.625, 0.305882352941, 0.305882352941),
            (0.75, 0.101960784314, 0.101960784314),
            (0.875, 0.0, 0.0),
            (1.0, 0.0, 0.0),
        ],
        blue=[
            (0.0, 0.8, 0.8),
            (0.125, 0.627450980392, 0.627450980392),
            (0.25, 0.462745098039, 0.462745098039),
            (0.375, 0.298039215686, 0.298039215686),
            (0.5, 0.235294117647, 0.235294117647),
            (0.625, 0.164705882353, 0.164705882353),
            (0.75, 0.109803921569, 0.109803921569),
            (0.875, 0.149019607843, 0.149019607843),
            (1.0, 0.149019607843, 0.149019607843),
        ],
    )
    return ColorMapper.from_segment_map(_data, range=range, **traits)


# The following colormaps are from Yorick, a derivative of the GIST package.


def gist_earth(range, **traits):
    """Generator for the 'gist_earth' colormap from GIST."""
    _data = dict(
        red=[
            (0.0, 0.0, 0.0),
            (0.0042016808874905109, 0.0, 0.0),
            (0.0084033617749810219, 0.0, 0.0),
            (0.012605042196810246, 0.0, 0.0),
            (0.016806723549962044, 0.0, 0.0),
            (0.021008403971791267, 0.0, 0.0),
            (0.025210084393620491, 0.0, 0.0),
            (0.029411764815449715, 0.0, 0.0),
            (0.033613447099924088, 0.0, 0.0),
            (
                0.037815127521753311,
                0.0039215688593685627,
                0.0039215688593685627,
            ),
            (
                0.042016807943582535,
                0.0078431377187371254,
                0.0078431377187371254,
            ),
            (
                0.046218488365411758,
                0.0078431377187371254,
                0.0078431377187371254,
            ),
            (0.050420168787240982, 0.011764706112444401, 0.011764706112444401),
            (0.054621849209070206, 0.015686275437474251, 0.015686275437474251),
            (0.058823529630899429, 0.019607843831181526, 0.019607843831181526),
            (0.063025213778018951, 0.019607843831181526, 0.019607843831181526),
            (0.067226894199848175, 0.023529412224888802, 0.023529412224888802),
            (0.071428574621677399, 0.027450980618596077, 0.027450980618596077),
            (0.075630255043506622, 0.031372550874948502, 0.031372550874948502),
            (0.079831935465335846, 0.031372550874948502, 0.031372550874948502),
            (0.08403361588716507, 0.035294119268655777, 0.035294119268655777),
            (0.088235296308994293, 0.039215687662363052, 0.039215687662363052),
            (0.092436976730823517, 0.043137256056070328, 0.043137256056070328),
            (0.09663865715265274, 0.043137256056070328, 0.043137256056070328),
            (0.10084033757448196, 0.047058824449777603, 0.047058824449777603),
            (0.10504201799631119, 0.050980392843484879, 0.050980392843484879),
            (0.10924369841814041, 0.054901961237192154, 0.054901961237192154),
            (0.11344537883996964, 0.058823529630899429, 0.058823529630899429),
            (0.11764705926179886, 0.058823529630899429, 0.058823529630899429),
            (0.12184873968362808, 0.062745101749897003, 0.062745101749897003),
            (0.1260504275560379, 0.066666670143604279, 0.066666670143604279),
            (0.13025210797786713, 0.070588238537311554, 0.070588238537311554),
            (0.13445378839969635, 0.070588238537311554, 0.070588238537311554),
            (0.13865546882152557, 0.074509806931018829, 0.074509806931018829),
            (0.1428571492433548, 0.078431375324726105, 0.078431375324726105),
            (0.14705882966518402, 0.08235294371843338, 0.08235294371843338),
            (0.15126051008701324, 0.086274512112140656, 0.086274512112140656),
            (0.15546219050884247, 0.086274512112140656, 0.086274512112140656),
            (0.15966387093067169, 0.090196080505847931, 0.090196080505847931),
            (0.16386555135250092, 0.094117648899555206, 0.094117648899555206),
            (0.16806723177433014, 0.098039217293262482, 0.098039217293262482),
            (0.17226891219615936, 0.10196078568696976, 0.10196078568696976),
            (0.17647059261798859, 0.10196078568696976, 0.10196078568696976),
            (0.18067227303981781, 0.10588235408067703, 0.10588235408067703),
            (0.18487395346164703, 0.10980392247438431, 0.10980392247438431),
            (0.18907563388347626, 0.11372549086809158, 0.11372549086809158),
            (0.19327731430530548, 0.11764705926179886, 0.11764705926179886),
            (0.1974789947271347, 0.12156862765550613, 0.12156862765550613),
            (0.20168067514896393, 0.12156862765550613, 0.12156862765550613),
            (0.20588235557079315, 0.12549020349979401, 0.12549020349979401),
            (0.21008403599262238, 0.12941177189350128, 0.12941177189350128),
            (0.2142857164144516, 0.13333334028720856, 0.13333334028720856),
            (0.21848739683628082, 0.13725490868091583, 0.13725490868091583),
            (0.22268907725811005, 0.14117647707462311, 0.14117647707462311),
            (0.22689075767993927, 0.14117647707462311, 0.14117647707462311),
            (0.23109243810176849, 0.14509804546833038, 0.14509804546833038),
            (0.23529411852359772, 0.14901961386203766, 0.14901961386203766),
            (0.23949579894542694, 0.15294118225574493, 0.15294118225574493),
            (0.24369747936725616, 0.15686275064945221, 0.15686275064945221),
            (0.24789915978908539, 0.16078431904315948, 0.16078431904315948),
            (0.25210085511207581, 0.16078431904315948, 0.16078431904315948),
            (0.25630253553390503, 0.16470588743686676, 0.16470588743686676),
            (0.26050421595573425, 0.16862745583057404, 0.16862745583057404),
            (0.26470589637756348, 0.17254902422428131, 0.17254902422428131),
            (0.2689075767993927, 0.17647059261798859, 0.17647059261798859),
            (0.27310925722122192, 0.18039216101169586, 0.18039216101169586),
            (0.27731093764305115, 0.18431372940540314, 0.18431372940540314),
            (0.28151261806488037, 0.18823529779911041, 0.18823529779911041),
            (0.28571429848670959, 0.18823529779911041, 0.18823529779911041),
            (0.28991597890853882, 0.18823529779911041, 0.18823529779911041),
            (0.29411765933036804, 0.19215686619281769, 0.19215686619281769),
            (0.29831933975219727, 0.19215686619281769, 0.19215686619281769),
            (0.30252102017402649, 0.19607843458652496, 0.19607843458652496),
            (0.30672270059585571, 0.19607843458652496, 0.19607843458652496),
            (0.31092438101768494, 0.20000000298023224, 0.20000000298023224),
            (0.31512606143951416, 0.20000000298023224, 0.20000000298023224),
            (0.31932774186134338, 0.20392157137393951, 0.20392157137393951),
            (0.32352942228317261, 0.20392157137393951, 0.20392157137393951),
            (0.32773110270500183, 0.20784313976764679, 0.20784313976764679),
            (0.33193278312683105, 0.20784313976764679, 0.20784313976764679),
            (0.33613446354866028, 0.21176470816135406, 0.21176470816135406),
            (0.3403361439704895, 0.21176470816135406, 0.21176470816135406),
            (0.34453782439231873, 0.21568627655506134, 0.21568627655506134),
            (0.34873950481414795, 0.21568627655506134, 0.21568627655506134),
            (0.35294118523597717, 0.21960784494876862, 0.21960784494876862),
            (0.3571428656578064, 0.21960784494876862, 0.21960784494876862),
            (0.36134454607963562, 0.22352941334247589, 0.22352941334247589),
            (0.36554622650146484, 0.22352941334247589, 0.22352941334247589),
            (0.36974790692329407, 0.22745098173618317, 0.22745098173618317),
            (0.37394958734512329, 0.22745098173618317, 0.22745098173618317),
            (0.37815126776695251, 0.23137255012989044, 0.23137255012989044),
            (0.38235294818878174, 0.23137255012989044, 0.23137255012989044),
            (0.38655462861061096, 0.23529411852359772, 0.23529411852359772),
            (0.39075630903244019, 0.23921568691730499, 0.23921568691730499),
            (0.39495798945426941, 0.23921568691730499, 0.23921568691730499),
            (0.39915966987609863, 0.24313725531101227, 0.24313725531101227),
            (0.40336135029792786, 0.24313725531101227, 0.24313725531101227),
            (0.40756303071975708, 0.24705882370471954, 0.24705882370471954),
            (0.4117647111415863, 0.24705882370471954, 0.24705882370471954),
            (0.41596639156341553, 0.25098040699958801, 0.25098040699958801),
            (0.42016807198524475, 0.25098040699958801, 0.25098040699958801),
            (0.42436975240707397, 0.25490197539329529, 0.25490197539329529),
            (0.4285714328289032, 0.25490197539329529, 0.25490197539329529),
            (0.43277311325073242, 0.25882354378700256, 0.25882354378700256),
            (0.43697479367256165, 0.26274511218070984, 0.26274511218070984),
            (0.44117647409439087, 0.26274511218070984, 0.26274511218070984),
            (0.44537815451622009, 0.26666668057441711, 0.26666668057441711),
            (0.44957983493804932, 0.26666668057441711, 0.26666668057441711),
            (0.45378151535987854, 0.27058824896812439, 0.27058824896812439),
            (0.45798319578170776, 0.27058824896812439, 0.27058824896812439),
            (0.46218487620353699, 0.27450981736183167, 0.27450981736183167),
            (0.46638655662536621, 0.27843138575553894, 0.27843138575553894),
            (0.47058823704719543, 0.28627452254295349, 0.28627452254295349),
            (0.47478991746902466, 0.29803922772407532, 0.29803922772407532),
            (0.47899159789085388, 0.30588236451148987, 0.30588236451148987),
            (0.48319327831268311, 0.31764706969261169, 0.31764706969261169),
            (0.48739495873451233, 0.32549020648002625, 0.32549020648002625),
            (0.49159663915634155, 0.33725491166114807, 0.33725491166114807),
            (0.49579831957817078, 0.34509804844856262, 0.34509804844856262),
            (0.5, 0.35686275362968445, 0.35686275362968445),
            (0.50420171022415161, 0.36862745881080627, 0.36862745881080627),
            (0.50840336084365845, 0.37647059559822083, 0.37647059559822083),
            (0.51260507106781006, 0.38823530077934265, 0.38823530077934265),
            (0.51680672168731689, 0.3960784375667572, 0.3960784375667572),
            (0.52100843191146851, 0.40784314274787903, 0.40784314274787903),
            (0.52521008253097534, 0.41568627953529358, 0.41568627953529358),
            (0.52941179275512695, 0.42745098471641541, 0.42745098471641541),
            (0.53361344337463379, 0.43529412150382996, 0.43529412150382996),
            (0.5378151535987854, 0.44705882668495178, 0.44705882668495178),
            (0.54201680421829224, 0.45882353186607361, 0.45882353186607361),
            (0.54621851444244385, 0.46666666865348816, 0.46666666865348816),
            (0.55042016506195068, 0.47450980544090271, 0.47450980544090271),
            (0.55462187528610229, 0.47843137383460999, 0.47843137383460999),
            (0.55882352590560913, 0.48627451062202454, 0.48627451062202454),
            (0.56302523612976074, 0.49411764740943909, 0.49411764740943909),
            (0.56722688674926758, 0.50196081399917603, 0.50196081399917603),
            (0.57142859697341919, 0.5058823823928833, 0.5058823823928833),
            (0.57563024759292603, 0.51372551918029785, 0.51372551918029785),
            (0.57983195781707764, 0.5215686559677124, 0.5215686559677124),
            (0.58403360843658447, 0.52941179275512695, 0.52941179275512695),
            (0.58823531866073608, 0.53333336114883423, 0.53333336114883423),
            (0.59243696928024292, 0.54117649793624878, 0.54117649793624878),
            (0.59663867950439453, 0.54901963472366333, 0.54901963472366333),
            (0.60084033012390137, 0.55294120311737061, 0.55294120311737061),
            (0.60504204034805298, 0.56078433990478516, 0.56078433990478516),
            (0.60924369096755981, 0.56862747669219971, 0.56862747669219971),
            (0.61344540119171143, 0.57647061347961426, 0.57647061347961426),
            (0.61764705181121826, 0.58431375026702881, 0.58431375026702881),
            (0.62184876203536987, 0.58823531866073608, 0.58823531866073608),
            (0.62605041265487671, 0.59607845544815063, 0.59607845544815063),
            (0.63025212287902832, 0.60392159223556519, 0.60392159223556519),
            (0.63445377349853516, 0.61176472902297974, 0.61176472902297974),
            (0.63865548372268677, 0.61568629741668701, 0.61568629741668701),
            (0.6428571343421936, 0.62352943420410156, 0.62352943420410156),
            (0.64705884456634521, 0.63137257099151611, 0.63137257099151611),
            (0.65126049518585205, 0.63921570777893066, 0.63921570777893066),
            (0.65546220541000366, 0.64705884456634521, 0.64705884456634521),
            (0.6596638560295105, 0.65098041296005249, 0.65098041296005249),
            (0.66386556625366211, 0.65882354974746704, 0.65882354974746704),
            (0.66806721687316895, 0.66666668653488159, 0.66666668653488159),
            (0.67226892709732056, 0.67450982332229614, 0.67450982332229614),
            (0.67647057771682739, 0.68235296010971069, 0.68235296010971069),
            (0.680672287940979, 0.68627452850341797, 0.68627452850341797),
            (0.68487393856048584, 0.69411766529083252, 0.69411766529083252),
            (0.68907564878463745, 0.70196080207824707, 0.70196080207824707),
            (0.69327729940414429, 0.70980393886566162, 0.70980393886566162),
            (0.6974790096282959, 0.71764707565307617, 0.71764707565307617),
            (0.70168066024780273, 0.71764707565307617, 0.71764707565307617),
            (0.70588237047195435, 0.72156864404678345, 0.72156864404678345),
            (0.71008402109146118, 0.72156864404678345, 0.72156864404678345),
            (0.71428573131561279, 0.72549021244049072, 0.72549021244049072),
            (0.71848738193511963, 0.72549021244049072, 0.72549021244049072),
            (0.72268909215927124, 0.729411780834198, 0.729411780834198),
            (0.72689074277877808, 0.729411780834198, 0.729411780834198),
            (0.73109245300292969, 0.73333334922790527, 0.73333334922790527),
            (0.73529410362243652, 0.73333334922790527, 0.73333334922790527),
            (0.73949581384658813, 0.73333334922790527, 0.73333334922790527),
            (0.74369746446609497, 0.73725491762161255, 0.73725491762161255),
            (0.74789917469024658, 0.73725491762161255, 0.73725491762161255),
            (0.75210082530975342, 0.74117648601531982, 0.74117648601531982),
            (0.75630253553390503, 0.74117648601531982, 0.74117648601531982),
            (0.76050418615341187, 0.7450980544090271, 0.7450980544090271),
            (0.76470589637756348, 0.7450980544090271, 0.7450980544090271),
            (0.76890754699707031, 0.7450980544090271, 0.7450980544090271),
            (0.77310925722122192, 0.74901962280273438, 0.74901962280273438),
            (0.77731090784072876, 0.74901962280273438, 0.74901962280273438),
            (0.78151261806488037, 0.75294119119644165, 0.75294119119644165),
            (0.78571426868438721, 0.75294119119644165, 0.75294119119644165),
            (0.78991597890853882, 0.75686275959014893, 0.75686275959014893),
            (0.79411762952804565, 0.76470589637756348, 0.76470589637756348),
            (0.79831933975219727, 0.76862746477127075, 0.76862746477127075),
            (0.8025209903717041, 0.77254903316497803, 0.77254903316497803),
            (0.80672270059585571, 0.7764706015586853, 0.7764706015586853),
            (0.81092435121536255, 0.78039216995239258, 0.78039216995239258),
            (0.81512606143951416, 0.78823530673980713, 0.78823530673980713),
            (0.819327712059021, 0.7921568751335144, 0.7921568751335144),
            (0.82352942228317261, 0.79607844352722168, 0.79607844352722168),
            (0.82773107290267944, 0.80000001192092896, 0.80000001192092896),
            (0.83193278312683105, 0.80392158031463623, 0.80392158031463623),
            (0.83613443374633789, 0.81176471710205078, 0.81176471710205078),
            (0.8403361439704895, 0.81568628549575806, 0.81568628549575806),
            (0.84453779458999634, 0.81960785388946533, 0.81960785388946533),
            (0.84873950481414795, 0.82352942228317261, 0.82352942228317261),
            (0.85294115543365479, 0.82745099067687988, 0.82745099067687988),
            (0.8571428656578064, 0.83529412746429443, 0.83529412746429443),
            (0.86134451627731323, 0.83921569585800171, 0.83921569585800171),
            (0.86554622650146484, 0.84313726425170898, 0.84313726425170898),
            (0.86974787712097168, 0.84705883264541626, 0.84705883264541626),
            (0.87394958734512329, 0.85098040103912354, 0.85098040103912354),
            (0.87815123796463013, 0.85882353782653809, 0.85882353782653809),
            (0.88235294818878174, 0.86274510622024536, 0.86274510622024536),
            (0.88655459880828857, 0.86666667461395264, 0.86666667461395264),
            (0.89075630903244019, 0.87058824300765991, 0.87058824300765991),
            (0.89495795965194702, 0.87450981140136719, 0.87450981140136719),
            (0.89915966987609863, 0.88235294818878174, 0.88235294818878174),
            (0.90336132049560547, 0.88627451658248901, 0.88627451658248901),
            (0.90756303071975708, 0.89019608497619629, 0.89019608497619629),
            (0.91176468133926392, 0.89411765336990356, 0.89411765336990356),
            (0.91596639156341553, 0.89803922176361084, 0.89803922176361084),
            (0.92016804218292236, 0.90588235855102539, 0.90588235855102539),
            (0.92436975240707397, 0.90980392694473267, 0.90980392694473267),
            (0.92857140302658081, 0.91372549533843994, 0.91372549533843994),
            (0.93277311325073242, 0.91764706373214722, 0.91764706373214722),
            (0.93697476387023926, 0.92156863212585449, 0.92156863212585449),
            (0.94117647409439087, 0.92941176891326904, 0.92941176891326904),
            (0.94537812471389771, 0.93333333730697632, 0.93333333730697632),
            (0.94957983493804932, 0.93725490570068359, 0.93725490570068359),
            (0.95378148555755615, 0.94117647409439087, 0.94117647409439087),
            (0.95798319578170776, 0.94509804248809814, 0.94509804248809814),
            (0.9621848464012146, 0.9529411792755127, 0.9529411792755127),
            (0.96638655662536621, 0.95686274766921997, 0.95686274766921997),
            (0.97058820724487305, 0.96078431606292725, 0.96078431606292725),
            (0.97478991746902466, 0.96470588445663452, 0.96470588445663452),
            (0.97899156808853149, 0.9686274528503418, 0.9686274528503418),
            (0.98319327831268311, 0.97647058963775635, 0.97647058963775635),
            (0.98739492893218994, 0.98039215803146362, 0.98039215803146362),
            (0.99159663915634155, 0.9843137264251709, 0.9843137264251709),
            (0.99579828977584839, 0.98823529481887817, 0.98823529481887817),
            (1.0, 0.99215686321258545, 0.99215686321258545),
        ],
        green=[
            (0.0, 0.0, 0.0),
            (0.0042016808874905109, 0.0, 0.0),
            (0.0084033617749810219, 0.0, 0.0),
            (0.012605042196810246, 0.0, 0.0),
            (0.016806723549962044, 0.0, 0.0),
            (0.021008403971791267, 0.0, 0.0),
            (0.025210084393620491, 0.0, 0.0),
            (0.029411764815449715, 0.0, 0.0),
            (0.033613447099924088, 0.011764706112444401, 0.011764706112444401),
            (0.037815127521753311, 0.023529412224888802, 0.023529412224888802),
            (0.042016807943582535, 0.031372550874948502, 0.031372550874948502),
            (0.046218488365411758, 0.043137256056070328, 0.043137256056070328),
            (0.050420168787240982, 0.050980392843484879, 0.050980392843484879),
            (0.054621849209070206, 0.062745101749897003, 0.062745101749897003),
            (0.058823529630899429, 0.070588238537311554, 0.070588238537311554),
            (0.063025213778018951, 0.08235294371843338, 0.08235294371843338),
            (0.067226894199848175, 0.090196080505847931, 0.090196080505847931),
            (0.071428574621677399, 0.10196078568696976, 0.10196078568696976),
            (0.075630255043506622, 0.10980392247438431, 0.10980392247438431),
            (0.079831935465335846, 0.12156862765550613, 0.12156862765550613),
            (0.08403361588716507, 0.12941177189350128, 0.12941177189350128),
            (0.088235296308994293, 0.14117647707462311, 0.14117647707462311),
            (0.092436976730823517, 0.14901961386203766, 0.14901961386203766),
            (0.09663865715265274, 0.16078431904315948, 0.16078431904315948),
            (0.10084033757448196, 0.16862745583057404, 0.16862745583057404),
            (0.10504201799631119, 0.17647059261798859, 0.17647059261798859),
            (0.10924369841814041, 0.18823529779911041, 0.18823529779911041),
            (0.11344537883996964, 0.19607843458652496, 0.19607843458652496),
            (0.11764705926179886, 0.20392157137393951, 0.20392157137393951),
            (0.12184873968362808, 0.21568627655506134, 0.21568627655506134),
            (0.1260504275560379, 0.22352941334247589, 0.22352941334247589),
            (0.13025210797786713, 0.23137255012989044, 0.23137255012989044),
            (0.13445378839969635, 0.23921568691730499, 0.23921568691730499),
            (0.13865546882152557, 0.25098040699958801, 0.25098040699958801),
            (0.1428571492433548, 0.25882354378700256, 0.25882354378700256),
            (0.14705882966518402, 0.26666668057441711, 0.26666668057441711),
            (0.15126051008701324, 0.27450981736183167, 0.27450981736183167),
            (0.15546219050884247, 0.28235295414924622, 0.28235295414924622),
            (0.15966387093067169, 0.29019609093666077, 0.29019609093666077),
            (0.16386555135250092, 0.30196079611778259, 0.30196079611778259),
            (0.16806723177433014, 0.30980393290519714, 0.30980393290519714),
            (0.17226891219615936, 0.31764706969261169, 0.31764706969261169),
            (0.17647059261798859, 0.32549020648002625, 0.32549020648002625),
            (0.18067227303981781, 0.3333333432674408, 0.3333333432674408),
            (0.18487395346164703, 0.34117648005485535, 0.34117648005485535),
            (0.18907563388347626, 0.3490196168422699, 0.3490196168422699),
            (0.19327731430530548, 0.35686275362968445, 0.35686275362968445),
            (0.1974789947271347, 0.364705890417099, 0.364705890417099),
            (0.20168067514896393, 0.37254902720451355, 0.37254902720451355),
            (0.20588235557079315, 0.3803921639919281, 0.3803921639919281),
            (0.21008403599262238, 0.38823530077934265, 0.38823530077934265),
            (0.2142857164144516, 0.39215686917304993, 0.39215686917304993),
            (0.21848739683628082, 0.40000000596046448, 0.40000000596046448),
            (0.22268907725811005, 0.40784314274787903, 0.40784314274787903),
            (0.22689075767993927, 0.41568627953529358, 0.41568627953529358),
            (0.23109243810176849, 0.42352941632270813, 0.42352941632270813),
            (0.23529411852359772, 0.42745098471641541, 0.42745098471641541),
            (0.23949579894542694, 0.43529412150382996, 0.43529412150382996),
            (0.24369747936725616, 0.44313725829124451, 0.44313725829124451),
            (0.24789915978908539, 0.45098039507865906, 0.45098039507865906),
            (0.25210085511207581, 0.45490196347236633, 0.45490196347236633),
            (0.25630253553390503, 0.46274510025978088, 0.46274510025978088),
            (0.26050421595573425, 0.47058823704719543, 0.47058823704719543),
            (0.26470589637756348, 0.47450980544090271, 0.47450980544090271),
            (0.2689075767993927, 0.48235294222831726, 0.48235294222831726),
            (0.27310925722122192, 0.49019607901573181, 0.49019607901573181),
            (0.27731093764305115, 0.49411764740943909, 0.49411764740943909),
            (0.28151261806488037, 0.50196081399917603, 0.50196081399917603),
            (0.28571429848670959, 0.50196081399917603, 0.50196081399917603),
            (0.28991597890853882, 0.5058823823928833, 0.5058823823928833),
            (0.29411765933036804, 0.5058823823928833, 0.5058823823928833),
            (0.29831933975219727, 0.50980395078659058, 0.50980395078659058),
            (0.30252102017402649, 0.51372551918029785, 0.51372551918029785),
            (0.30672270059585571, 0.51372551918029785, 0.51372551918029785),
            (0.31092438101768494, 0.51764708757400513, 0.51764708757400513),
            (0.31512606143951416, 0.5215686559677124, 0.5215686559677124),
            (0.31932774186134338, 0.5215686559677124, 0.5215686559677124),
            (0.32352942228317261, 0.52549022436141968, 0.52549022436141968),
            (0.32773110270500183, 0.52549022436141968, 0.52549022436141968),
            (0.33193278312683105, 0.52941179275512695, 0.52941179275512695),
            (0.33613446354866028, 0.53333336114883423, 0.53333336114883423),
            (0.3403361439704895, 0.53333336114883423, 0.53333336114883423),
            (0.34453782439231873, 0.5372549295425415, 0.5372549295425415),
            (0.34873950481414795, 0.54117649793624878, 0.54117649793624878),
            (0.35294118523597717, 0.54117649793624878, 0.54117649793624878),
            (0.3571428656578064, 0.54509806632995605, 0.54509806632995605),
            (0.36134454607963562, 0.54901963472366333, 0.54901963472366333),
            (0.36554622650146484, 0.54901963472366333, 0.54901963472366333),
            (0.36974790692329407, 0.55294120311737061, 0.55294120311737061),
            (0.37394958734512329, 0.55294120311737061, 0.55294120311737061),
            (0.37815126776695251, 0.55686277151107788, 0.55686277151107788),
            (0.38235294818878174, 0.56078433990478516, 0.56078433990478516),
            (0.38655462861061096, 0.56078433990478516, 0.56078433990478516),
            (0.39075630903244019, 0.56470590829849243, 0.56470590829849243),
            (0.39495798945426941, 0.56862747669219971, 0.56862747669219971),
            (0.39915966987609863, 0.56862747669219971, 0.56862747669219971),
            (0.40336135029792786, 0.57254904508590698, 0.57254904508590698),
            (0.40756303071975708, 0.57254904508590698, 0.57254904508590698),
            (0.4117647111415863, 0.57647061347961426, 0.57647061347961426),
            (0.41596639156341553, 0.58039218187332153, 0.58039218187332153),
            (0.42016807198524475, 0.58039218187332153, 0.58039218187332153),
            (0.42436975240707397, 0.58431375026702881, 0.58431375026702881),
            (0.4285714328289032, 0.58823531866073608, 0.58823531866073608),
            (0.43277311325073242, 0.58823531866073608, 0.58823531866073608),
            (0.43697479367256165, 0.59215688705444336, 0.59215688705444336),
            (0.44117647409439087, 0.59215688705444336, 0.59215688705444336),
            (0.44537815451622009, 0.59607845544815063, 0.59607845544815063),
            (0.44957983493804932, 0.60000002384185791, 0.60000002384185791),
            (0.45378151535987854, 0.60000002384185791, 0.60000002384185791),
            (0.45798319578170776, 0.60392159223556519, 0.60392159223556519),
            (0.46218487620353699, 0.60784316062927246, 0.60784316062927246),
            (0.46638655662536621, 0.60784316062927246, 0.60784316062927246),
            (0.47058823704719543, 0.61176472902297974, 0.61176472902297974),
            (0.47478991746902466, 0.61176472902297974, 0.61176472902297974),
            (0.47899159789085388, 0.61568629741668701, 0.61568629741668701),
            (0.48319327831268311, 0.61960786581039429, 0.61960786581039429),
            (0.48739495873451233, 0.61960786581039429, 0.61960786581039429),
            (0.49159663915634155, 0.62352943420410156, 0.62352943420410156),
            (0.49579831957817078, 0.62745100259780884, 0.62745100259780884),
            (0.5, 0.62745100259780884, 0.62745100259780884),
            (0.50420171022415161, 0.63137257099151611, 0.63137257099151611),
            (0.50840336084365845, 0.63137257099151611, 0.63137257099151611),
            (0.51260507106781006, 0.63529413938522339, 0.63529413938522339),
            (0.51680672168731689, 0.63921570777893066, 0.63921570777893066),
            (0.52100843191146851, 0.63921570777893066, 0.63921570777893066),
            (0.52521008253097534, 0.64313727617263794, 0.64313727617263794),
            (0.52941179275512695, 0.64705884456634521, 0.64705884456634521),
            (0.53361344337463379, 0.64705884456634521, 0.64705884456634521),
            (0.5378151535987854, 0.65098041296005249, 0.65098041296005249),
            (0.54201680421829224, 0.65098041296005249, 0.65098041296005249),
            (0.54621851444244385, 0.65490198135375977, 0.65490198135375977),
            (0.55042016506195068, 0.65882354974746704, 0.65882354974746704),
            (0.55462187528610229, 0.65882354974746704, 0.65882354974746704),
            (0.55882352590560913, 0.65882354974746704, 0.65882354974746704),
            (0.56302523612976074, 0.66274511814117432, 0.66274511814117432),
            (0.56722688674926758, 0.66274511814117432, 0.66274511814117432),
            (0.57142859697341919, 0.66666668653488159, 0.66666668653488159),
            (0.57563024759292603, 0.66666668653488159, 0.66666668653488159),
            (0.57983195781707764, 0.67058825492858887, 0.67058825492858887),
            (0.58403360843658447, 0.67058825492858887, 0.67058825492858887),
            (0.58823531866073608, 0.67450982332229614, 0.67450982332229614),
            (0.59243696928024292, 0.67450982332229614, 0.67450982332229614),
            (0.59663867950439453, 0.67450982332229614, 0.67450982332229614),
            (0.60084033012390137, 0.67843139171600342, 0.67843139171600342),
            (0.60504204034805298, 0.67843139171600342, 0.67843139171600342),
            (0.60924369096755981, 0.68235296010971069, 0.68235296010971069),
            (0.61344540119171143, 0.68235296010971069, 0.68235296010971069),
            (0.61764705181121826, 0.68627452850341797, 0.68627452850341797),
            (0.62184876203536987, 0.68627452850341797, 0.68627452850341797),
            (0.62605041265487671, 0.68627452850341797, 0.68627452850341797),
            (0.63025212287902832, 0.69019609689712524, 0.69019609689712524),
            (0.63445377349853516, 0.69019609689712524, 0.69019609689712524),
            (0.63865548372268677, 0.69411766529083252, 0.69411766529083252),
            (0.6428571343421936, 0.69411766529083252, 0.69411766529083252),
            (0.64705884456634521, 0.69803923368453979, 0.69803923368453979),
            (0.65126049518585205, 0.69803923368453979, 0.69803923368453979),
            (0.65546220541000366, 0.70196080207824707, 0.70196080207824707),
            (0.6596638560295105, 0.70196080207824707, 0.70196080207824707),
            (0.66386556625366211, 0.70196080207824707, 0.70196080207824707),
            (0.66806721687316895, 0.70588237047195435, 0.70588237047195435),
            (0.67226892709732056, 0.70588237047195435, 0.70588237047195435),
            (0.67647057771682739, 0.70980393886566162, 0.70980393886566162),
            (0.680672287940979, 0.70980393886566162, 0.70980393886566162),
            (0.68487393856048584, 0.7137255072593689, 0.7137255072593689),
            (0.68907564878463745, 0.7137255072593689, 0.7137255072593689),
            (0.69327729940414429, 0.71764707565307617, 0.71764707565307617),
            (0.6974790096282959, 0.71764707565307617, 0.71764707565307617),
            (0.70168066024780273, 0.7137255072593689, 0.7137255072593689),
            (0.70588237047195435, 0.70980393886566162, 0.70980393886566162),
            (0.71008402109146118, 0.70980393886566162, 0.70980393886566162),
            (0.71428573131561279, 0.70588237047195435, 0.70588237047195435),
            (0.71848738193511963, 0.70196080207824707, 0.70196080207824707),
            (0.72268909215927124, 0.69803923368453979, 0.69803923368453979),
            (0.72689074277877808, 0.69411766529083252, 0.69411766529083252),
            (0.73109245300292969, 0.69019609689712524, 0.69019609689712524),
            (0.73529410362243652, 0.68627452850341797, 0.68627452850341797),
            (0.73949581384658813, 0.68235296010971069, 0.68235296010971069),
            (0.74369746446609497, 0.67843139171600342, 0.67843139171600342),
            (0.74789917469024658, 0.67450982332229614, 0.67450982332229614),
            (0.75210082530975342, 0.67058825492858887, 0.67058825492858887),
            (0.75630253553390503, 0.66666668653488159, 0.66666668653488159),
            (0.76050418615341187, 0.66274511814117432, 0.66274511814117432),
            (0.76470589637756348, 0.65882354974746704, 0.65882354974746704),
            (0.76890754699707031, 0.65490198135375977, 0.65490198135375977),
            (0.77310925722122192, 0.65098041296005249, 0.65098041296005249),
            (0.77731090784072876, 0.64705884456634521, 0.64705884456634521),
            (0.78151261806488037, 0.64313727617263794, 0.64313727617263794),
            (0.78571426868438721, 0.63921570777893066, 0.63921570777893066),
            (0.78991597890853882, 0.63921570777893066, 0.63921570777893066),
            (0.79411762952804565, 0.64313727617263794, 0.64313727617263794),
            (0.79831933975219727, 0.64313727617263794, 0.64313727617263794),
            (0.8025209903717041, 0.64705884456634521, 0.64705884456634521),
            (0.80672270059585571, 0.64705884456634521, 0.64705884456634521),
            (0.81092435121536255, 0.65098041296005249, 0.65098041296005249),
            (0.81512606143951416, 0.65490198135375977, 0.65490198135375977),
            (0.819327712059021, 0.65490198135375977, 0.65490198135375977),
            (0.82352942228317261, 0.65882354974746704, 0.65882354974746704),
            (0.82773107290267944, 0.66274511814117432, 0.66274511814117432),
            (0.83193278312683105, 0.66666668653488159, 0.66666668653488159),
            (0.83613443374633789, 0.67058825492858887, 0.67058825492858887),
            (0.8403361439704895, 0.67450982332229614, 0.67450982332229614),
            (0.84453779458999634, 0.67843139171600342, 0.67843139171600342),
            (0.84873950481414795, 0.68235296010971069, 0.68235296010971069),
            (0.85294115543365479, 0.68627452850341797, 0.68627452850341797),
            (0.8571428656578064, 0.69019609689712524, 0.69019609689712524),
            (0.86134451627731323, 0.69411766529083252, 0.69411766529083252),
            (0.86554622650146484, 0.69803923368453979, 0.69803923368453979),
            (0.86974787712097168, 0.70196080207824707, 0.70196080207824707),
            (0.87394958734512329, 0.70980393886566162, 0.70980393886566162),
            (0.87815123796463013, 0.7137255072593689, 0.7137255072593689),
            (0.88235294818878174, 0.72156864404678345, 0.72156864404678345),
            (0.88655459880828857, 0.72549021244049072, 0.72549021244049072),
            (0.89075630903244019, 0.73333334922790527, 0.73333334922790527),
            (0.89495795965194702, 0.73725491762161255, 0.73725491762161255),
            (0.89915966987609863, 0.7450980544090271, 0.7450980544090271),
            (0.90336132049560547, 0.75294119119644165, 0.75294119119644165),
            (0.90756303071975708, 0.7607843279838562, 0.7607843279838562),
            (0.91176468133926392, 0.76862746477127075, 0.76862746477127075),
            (0.91596639156341553, 0.7764706015586853, 0.7764706015586853),
            (0.92016804218292236, 0.78431373834609985, 0.78431373834609985),
            (0.92436975240707397, 0.7921568751335144, 0.7921568751335144),
            (0.92857140302658081, 0.80000001192092896, 0.80000001192092896),
            (0.93277311325073242, 0.80784314870834351, 0.80784314870834351),
            (0.93697476387023926, 0.81568628549575806, 0.81568628549575806),
            (0.94117647409439087, 0.82745099067687988, 0.82745099067687988),
            (0.94537812471389771, 0.83529412746429443, 0.83529412746429443),
            (0.94957983493804932, 0.84313726425170898, 0.84313726425170898),
            (0.95378148555755615, 0.85490196943283081, 0.85490196943283081),
            (0.95798319578170776, 0.86666667461395264, 0.86666667461395264),
            (0.9621848464012146, 0.87450981140136719, 0.87450981140136719),
            (0.96638655662536621, 0.88627451658248901, 0.88627451658248901),
            (0.97058820724487305, 0.89803922176361084, 0.89803922176361084),
            (0.97478991746902466, 0.90980392694473267, 0.90980392694473267),
            (0.97899156808853149, 0.92156863212585449, 0.92156863212585449),
            (0.98319327831268311, 0.93333333730697632, 0.93333333730697632),
            (0.98739492893218994, 0.94509804248809814, 0.94509804248809814),
            (0.99159663915634155, 0.95686274766921997, 0.95686274766921997),
            (0.99579828977584839, 0.97254902124404907, 0.97254902124404907),
            (1.0, 0.9843137264251709, 0.9843137264251709),
        ],
        blue=[
            (0.0, 0.0, 0.0),
            (0.0042016808874905109, 0.18039216101169586, 0.18039216101169586),
            (0.0084033617749810219, 0.22745098173618317, 0.22745098173618317),
            (0.012605042196810246, 0.27058824896812439, 0.27058824896812439),
            (0.016806723549962044, 0.31764706969261169, 0.31764706969261169),
            (0.021008403971791267, 0.36078432202339172, 0.36078432202339172),
            (0.025210084393620491, 0.40784314274787903, 0.40784314274787903),
            (0.029411764815449715, 0.45490196347236633, 0.45490196347236633),
            (0.033613447099924088, 0.45490196347236633, 0.45490196347236633),
            (0.037815127521753311, 0.45490196347236633, 0.45490196347236633),
            (0.042016807943582535, 0.45490196347236633, 0.45490196347236633),
            (0.046218488365411758, 0.45490196347236633, 0.45490196347236633),
            (0.050420168787240982, 0.45882353186607361, 0.45882353186607361),
            (0.054621849209070206, 0.45882353186607361, 0.45882353186607361),
            (0.058823529630899429, 0.45882353186607361, 0.45882353186607361),
            (0.063025213778018951, 0.45882353186607361, 0.45882353186607361),
            (0.067226894199848175, 0.45882353186607361, 0.45882353186607361),
            (0.071428574621677399, 0.46274510025978088, 0.46274510025978088),
            (0.075630255043506622, 0.46274510025978088, 0.46274510025978088),
            (0.079831935465335846, 0.46274510025978088, 0.46274510025978088),
            (0.08403361588716507, 0.46274510025978088, 0.46274510025978088),
            (0.088235296308994293, 0.46274510025978088, 0.46274510025978088),
            (0.092436976730823517, 0.46666666865348816, 0.46666666865348816),
            (0.09663865715265274, 0.46666666865348816, 0.46666666865348816),
            (0.10084033757448196, 0.46666666865348816, 0.46666666865348816),
            (0.10504201799631119, 0.46666666865348816, 0.46666666865348816),
            (0.10924369841814041, 0.46666666865348816, 0.46666666865348816),
            (0.11344537883996964, 0.47058823704719543, 0.47058823704719543),
            (0.11764705926179886, 0.47058823704719543, 0.47058823704719543),
            (0.12184873968362808, 0.47058823704719543, 0.47058823704719543),
            (0.1260504275560379, 0.47058823704719543, 0.47058823704719543),
            (0.13025210797786713, 0.47058823704719543, 0.47058823704719543),
            (0.13445378839969635, 0.47450980544090271, 0.47450980544090271),
            (0.13865546882152557, 0.47450980544090271, 0.47450980544090271),
            (0.1428571492433548, 0.47450980544090271, 0.47450980544090271),
            (0.14705882966518402, 0.47450980544090271, 0.47450980544090271),
            (0.15126051008701324, 0.47450980544090271, 0.47450980544090271),
            (0.15546219050884247, 0.47843137383460999, 0.47843137383460999),
            (0.15966387093067169, 0.47843137383460999, 0.47843137383460999),
            (0.16386555135250092, 0.47843137383460999, 0.47843137383460999),
            (0.16806723177433014, 0.47843137383460999, 0.47843137383460999),
            (0.17226891219615936, 0.47843137383460999, 0.47843137383460999),
            (0.17647059261798859, 0.48235294222831726, 0.48235294222831726),
            (0.18067227303981781, 0.48235294222831726, 0.48235294222831726),
            (0.18487395346164703, 0.48235294222831726, 0.48235294222831726),
            (0.18907563388347626, 0.48235294222831726, 0.48235294222831726),
            (0.19327731430530548, 0.48235294222831726, 0.48235294222831726),
            (0.1974789947271347, 0.48627451062202454, 0.48627451062202454),
            (0.20168067514896393, 0.48627451062202454, 0.48627451062202454),
            (0.20588235557079315, 0.48627451062202454, 0.48627451062202454),
            (0.21008403599262238, 0.48627451062202454, 0.48627451062202454),
            (0.2142857164144516, 0.48627451062202454, 0.48627451062202454),
            (0.21848739683628082, 0.49019607901573181, 0.49019607901573181),
            (0.22268907725811005, 0.49019607901573181, 0.49019607901573181),
            (0.22689075767993927, 0.49019607901573181, 0.49019607901573181),
            (0.23109243810176849, 0.49019607901573181, 0.49019607901573181),
            (0.23529411852359772, 0.49019607901573181, 0.49019607901573181),
            (0.23949579894542694, 0.49411764740943909, 0.49411764740943909),
            (0.24369747936725616, 0.49411764740943909, 0.49411764740943909),
            (0.24789915978908539, 0.49411764740943909, 0.49411764740943909),
            (0.25210085511207581, 0.49411764740943909, 0.49411764740943909),
            (0.25630253553390503, 0.49411764740943909, 0.49411764740943909),
            (0.26050421595573425, 0.49803921580314636, 0.49803921580314636),
            (0.26470589637756348, 0.49803921580314636, 0.49803921580314636),
            (0.2689075767993927, 0.49803921580314636, 0.49803921580314636),
            (0.27310925722122192, 0.49803921580314636, 0.49803921580314636),
            (0.27731093764305115, 0.49803921580314636, 0.49803921580314636),
            (0.28151261806488037, 0.50196081399917603, 0.50196081399917603),
            (0.28571429848670959, 0.49411764740943909, 0.49411764740943909),
            (0.28991597890853882, 0.49019607901573181, 0.49019607901573181),
            (0.29411765933036804, 0.48627451062202454, 0.48627451062202454),
            (0.29831933975219727, 0.48235294222831726, 0.48235294222831726),
            (0.30252102017402649, 0.47843137383460999, 0.47843137383460999),
            (0.30672270059585571, 0.47058823704719543, 0.47058823704719543),
            (0.31092438101768494, 0.46666666865348816, 0.46666666865348816),
            (0.31512606143951416, 0.46274510025978088, 0.46274510025978088),
            (0.31932774186134338, 0.45882353186607361, 0.45882353186607361),
            (0.32352942228317261, 0.45098039507865906, 0.45098039507865906),
            (0.32773110270500183, 0.44705882668495178, 0.44705882668495178),
            (0.33193278312683105, 0.44313725829124451, 0.44313725829124451),
            (0.33613446354866028, 0.43529412150382996, 0.43529412150382996),
            (0.3403361439704895, 0.43137255311012268, 0.43137255311012268),
            (0.34453782439231873, 0.42745098471641541, 0.42745098471641541),
            (0.34873950481414795, 0.42352941632270813, 0.42352941632270813),
            (0.35294118523597717, 0.41568627953529358, 0.41568627953529358),
            (0.3571428656578064, 0.4117647111415863, 0.4117647111415863),
            (0.36134454607963562, 0.40784314274787903, 0.40784314274787903),
            (0.36554622650146484, 0.40000000596046448, 0.40000000596046448),
            (0.36974790692329407, 0.3960784375667572, 0.3960784375667572),
            (0.37394958734512329, 0.39215686917304993, 0.39215686917304993),
            (0.37815126776695251, 0.38431373238563538, 0.38431373238563538),
            (0.38235294818878174, 0.3803921639919281, 0.3803921639919281),
            (0.38655462861061096, 0.37647059559822083, 0.37647059559822083),
            (0.39075630903244019, 0.36862745881080627, 0.36862745881080627),
            (0.39495798945426941, 0.364705890417099, 0.364705890417099),
            (0.39915966987609863, 0.36078432202339172, 0.36078432202339172),
            (0.40336135029792786, 0.35294118523597717, 0.35294118523597717),
            (0.40756303071975708, 0.3490196168422699, 0.3490196168422699),
            (0.4117647111415863, 0.34509804844856262, 0.34509804844856262),
            (0.41596639156341553, 0.33725491166114807, 0.33725491166114807),
            (0.42016807198524475, 0.3333333432674408, 0.3333333432674408),
            (0.42436975240707397, 0.32941177487373352, 0.32941177487373352),
            (0.4285714328289032, 0.32156863808631897, 0.32156863808631897),
            (0.43277311325073242, 0.31764706969261169, 0.31764706969261169),
            (0.43697479367256165, 0.31372550129890442, 0.31372550129890442),
            (0.44117647409439087, 0.30588236451148987, 0.30588236451148987),
            (0.44537815451622009, 0.30196079611778259, 0.30196079611778259),
            (0.44957983493804932, 0.29803922772407532, 0.29803922772407532),
            (0.45378151535987854, 0.29019609093666077, 0.29019609093666077),
            (0.45798319578170776, 0.28627452254295349, 0.28627452254295349),
            (0.46218487620353699, 0.27843138575553894, 0.27843138575553894),
            (0.46638655662536621, 0.27450981736183167, 0.27450981736183167),
            (0.47058823704719543, 0.27843138575553894, 0.27843138575553894),
            (0.47478991746902466, 0.28235295414924622, 0.28235295414924622),
            (0.47899159789085388, 0.28235295414924622, 0.28235295414924622),
            (0.48319327831268311, 0.28627452254295349, 0.28627452254295349),
            (0.48739495873451233, 0.28627452254295349, 0.28627452254295349),
            (0.49159663915634155, 0.29019609093666077, 0.29019609093666077),
            (0.49579831957817078, 0.29411765933036804, 0.29411765933036804),
            (0.5, 0.29411765933036804, 0.29411765933036804),
            (0.50420171022415161, 0.29803922772407532, 0.29803922772407532),
            (0.50840336084365845, 0.29803922772407532, 0.29803922772407532),
            (0.51260507106781006, 0.30196079611778259, 0.30196079611778259),
            (0.51680672168731689, 0.30196079611778259, 0.30196079611778259),
            (0.52100843191146851, 0.30588236451148987, 0.30588236451148987),
            (0.52521008253097534, 0.30980393290519714, 0.30980393290519714),
            (0.52941179275512695, 0.30980393290519714, 0.30980393290519714),
            (0.53361344337463379, 0.31372550129890442, 0.31372550129890442),
            (0.5378151535987854, 0.31372550129890442, 0.31372550129890442),
            (0.54201680421829224, 0.31764706969261169, 0.31764706969261169),
            (0.54621851444244385, 0.32156863808631897, 0.32156863808631897),
            (0.55042016506195068, 0.32156863808631897, 0.32156863808631897),
            (0.55462187528610229, 0.32156863808631897, 0.32156863808631897),
            (0.55882352590560913, 0.32549020648002625, 0.32549020648002625),
            (0.56302523612976074, 0.32549020648002625, 0.32549020648002625),
            (0.56722688674926758, 0.32549020648002625, 0.32549020648002625),
            (0.57142859697341919, 0.32941177487373352, 0.32941177487373352),
            (0.57563024759292603, 0.32941177487373352, 0.32941177487373352),
            (0.57983195781707764, 0.32941177487373352, 0.32941177487373352),
            (0.58403360843658447, 0.3333333432674408, 0.3333333432674408),
            (0.58823531866073608, 0.3333333432674408, 0.3333333432674408),
            (0.59243696928024292, 0.3333333432674408, 0.3333333432674408),
            (0.59663867950439453, 0.33725491166114807, 0.33725491166114807),
            (0.60084033012390137, 0.33725491166114807, 0.33725491166114807),
            (0.60504204034805298, 0.33725491166114807, 0.33725491166114807),
            (0.60924369096755981, 0.34117648005485535, 0.34117648005485535),
            (0.61344540119171143, 0.34117648005485535, 0.34117648005485535),
            (0.61764705181121826, 0.34117648005485535, 0.34117648005485535),
            (0.62184876203536987, 0.34509804844856262, 0.34509804844856262),
            (0.62605041265487671, 0.34509804844856262, 0.34509804844856262),
            (0.63025212287902832, 0.34509804844856262, 0.34509804844856262),
            (0.63445377349853516, 0.3490196168422699, 0.3490196168422699),
            (0.63865548372268677, 0.3490196168422699, 0.3490196168422699),
            (0.6428571343421936, 0.3490196168422699, 0.3490196168422699),
            (0.64705884456634521, 0.35294118523597717, 0.35294118523597717),
            (0.65126049518585205, 0.35294118523597717, 0.35294118523597717),
            (0.65546220541000366, 0.35294118523597717, 0.35294118523597717),
            (0.6596638560295105, 0.35686275362968445, 0.35686275362968445),
            (0.66386556625366211, 0.35686275362968445, 0.35686275362968445),
            (0.66806721687316895, 0.35686275362968445, 0.35686275362968445),
            (0.67226892709732056, 0.36078432202339172, 0.36078432202339172),
            (0.67647057771682739, 0.36078432202339172, 0.36078432202339172),
            (0.680672287940979, 0.36078432202339172, 0.36078432202339172),
            (0.68487393856048584, 0.364705890417099, 0.364705890417099),
            (0.68907564878463745, 0.364705890417099, 0.364705890417099),
            (0.69327729940414429, 0.364705890417099, 0.364705890417099),
            (0.6974790096282959, 0.36862745881080627, 0.36862745881080627),
            (0.70168066024780273, 0.36862745881080627, 0.36862745881080627),
            (0.70588237047195435, 0.36862745881080627, 0.36862745881080627),
            (0.71008402109146118, 0.37254902720451355, 0.37254902720451355),
            (0.71428573131561279, 0.37254902720451355, 0.37254902720451355),
            (0.71848738193511963, 0.37254902720451355, 0.37254902720451355),
            (0.72268909215927124, 0.37647059559822083, 0.37647059559822083),
            (0.72689074277877808, 0.37647059559822083, 0.37647059559822083),
            (0.73109245300292969, 0.3803921639919281, 0.3803921639919281),
            (0.73529410362243652, 0.3803921639919281, 0.3803921639919281),
            (0.73949581384658813, 0.3803921639919281, 0.3803921639919281),
            (0.74369746446609497, 0.38431373238563538, 0.38431373238563538),
            (0.74789917469024658, 0.38431373238563538, 0.38431373238563538),
            (0.75210082530975342, 0.38431373238563538, 0.38431373238563538),
            (0.75630253553390503, 0.38823530077934265, 0.38823530077934265),
            (0.76050418615341187, 0.38823530077934265, 0.38823530077934265),
            (0.76470589637756348, 0.38823530077934265, 0.38823530077934265),
            (0.76890754699707031, 0.39215686917304993, 0.39215686917304993),
            (0.77310925722122192, 0.39215686917304993, 0.39215686917304993),
            (0.77731090784072876, 0.39215686917304993, 0.39215686917304993),
            (0.78151261806488037, 0.3960784375667572, 0.3960784375667572),
            (0.78571426868438721, 0.3960784375667572, 0.3960784375667572),
            (0.78991597890853882, 0.40784314274787903, 0.40784314274787903),
            (0.79411762952804565, 0.41568627953529358, 0.41568627953529358),
            (0.79831933975219727, 0.42352941632270813, 0.42352941632270813),
            (0.8025209903717041, 0.43529412150382996, 0.43529412150382996),
            (0.80672270059585571, 0.44313725829124451, 0.44313725829124451),
            (0.81092435121536255, 0.45490196347236633, 0.45490196347236633),
            (0.81512606143951416, 0.46274510025978088, 0.46274510025978088),
            (0.819327712059021, 0.47450980544090271, 0.47450980544090271),
            (0.82352942228317261, 0.48235294222831726, 0.48235294222831726),
            (0.82773107290267944, 0.49411764740943909, 0.49411764740943909),
            (0.83193278312683105, 0.5058823823928833, 0.5058823823928833),
            (0.83613443374633789, 0.51372551918029785, 0.51372551918029785),
            (0.8403361439704895, 0.52549022436141968, 0.52549022436141968),
            (0.84453779458999634, 0.5372549295425415, 0.5372549295425415),
            (0.84873950481414795, 0.54509806632995605, 0.54509806632995605),
            (0.85294115543365479, 0.55686277151107788, 0.55686277151107788),
            (0.8571428656578064, 0.56862747669219971, 0.56862747669219971),
            (0.86134451627731323, 0.58039218187332153, 0.58039218187332153),
            (0.86554622650146484, 0.58823531866073608, 0.58823531866073608),
            (0.86974787712097168, 0.60000002384185791, 0.60000002384185791),
            (0.87394958734512329, 0.61176472902297974, 0.61176472902297974),
            (0.87815123796463013, 0.62352943420410156, 0.62352943420410156),
            (0.88235294818878174, 0.63529413938522339, 0.63529413938522339),
            (0.88655459880828857, 0.64705884456634521, 0.64705884456634521),
            (0.89075630903244019, 0.65882354974746704, 0.65882354974746704),
            (0.89495795965194702, 0.66666668653488159, 0.66666668653488159),
            (0.89915966987609863, 0.67843139171600342, 0.67843139171600342),
            (0.90336132049560547, 0.69019609689712524, 0.69019609689712524),
            (0.90756303071975708, 0.70196080207824707, 0.70196080207824707),
            (0.91176468133926392, 0.7137255072593689, 0.7137255072593689),
            (0.91596639156341553, 0.72549021244049072, 0.72549021244049072),
            (0.92016804218292236, 0.74117648601531982, 0.74117648601531982),
            (0.92436975240707397, 0.75294119119644165, 0.75294119119644165),
            (0.92857140302658081, 0.76470589637756348, 0.76470589637756348),
            (0.93277311325073242, 0.7764706015586853, 0.7764706015586853),
            (0.93697476387023926, 0.78823530673980713, 0.78823530673980713),
            (0.94117647409439087, 0.80000001192092896, 0.80000001192092896),
            (0.94537812471389771, 0.81176471710205078, 0.81176471710205078),
            (0.94957983493804932, 0.82745099067687988, 0.82745099067687988),
            (0.95378148555755615, 0.83921569585800171, 0.83921569585800171),
            (0.95798319578170776, 0.85098040103912354, 0.85098040103912354),
            (0.9621848464012146, 0.86274510622024536, 0.86274510622024536),
            (0.96638655662536621, 0.87843137979507446, 0.87843137979507446),
            (0.97058820724487305, 0.89019608497619629, 0.89019608497619629),
            (0.97478991746902466, 0.90196079015731812, 0.90196079015731812),
            (0.97899156808853149, 0.91764706373214722, 0.91764706373214722),
            (0.98319327831268311, 0.92941176891326904, 0.92941176891326904),
            (0.98739492893218994, 0.94509804248809814, 0.94509804248809814),
            (0.99159663915634155, 0.95686274766921997, 0.95686274766921997),
            (0.99579828977584839, 0.97254902124404907, 0.97254902124404907),
            (1.0, 0.9843137264251709, 0.9843137264251709),
        ],
    )
    return ColorMapper.from_segment_map(_data, range=range, **traits)


def gist_gray(range, **traits):
    """Generator for the 'gist_gray' colormap from GIST."""
    _data = dict(
        red=[
            (0.0, 0.0, 0.0),
            (
                0.0042016808874905109,
                0.0039215688593685627,
                0.0039215688593685627,
            ),
            (
                0.0084033617749810219,
                0.0078431377187371254,
                0.0078431377187371254,
            ),
            (0.012605042196810246, 0.011764706112444401, 0.011764706112444401),
            (0.016806723549962044, 0.015686275437474251, 0.015686275437474251),
            (0.021008403971791267, 0.019607843831181526, 0.019607843831181526),
            (0.025210084393620491, 0.023529412224888802, 0.023529412224888802),
            (0.029411764815449715, 0.027450980618596077, 0.027450980618596077),
            (0.033613447099924088, 0.035294119268655777, 0.035294119268655777),
            (0.037815127521753311, 0.039215687662363052, 0.039215687662363052),
            (0.042016807943582535, 0.043137256056070328, 0.043137256056070328),
            (0.046218488365411758, 0.047058824449777603, 0.047058824449777603),
            (0.050420168787240982, 0.050980392843484879, 0.050980392843484879),
            (0.054621849209070206, 0.054901961237192154, 0.054901961237192154),
            (0.058823529630899429, 0.058823529630899429, 0.058823529630899429),
            (0.063025213778018951, 0.062745101749897003, 0.062745101749897003),
            (0.067226894199848175, 0.066666670143604279, 0.066666670143604279),
            (0.071428574621677399, 0.070588238537311554, 0.070588238537311554),
            (0.075630255043506622, 0.074509806931018829, 0.074509806931018829),
            (0.079831935465335846, 0.078431375324726105, 0.078431375324726105),
            (0.08403361588716507, 0.08235294371843338, 0.08235294371843338),
            (0.088235296308994293, 0.086274512112140656, 0.086274512112140656),
            (0.092436976730823517, 0.090196080505847931, 0.090196080505847931),
            (0.09663865715265274, 0.098039217293262482, 0.098039217293262482),
            (0.10084033757448196, 0.10196078568696976, 0.10196078568696976),
            (0.10504201799631119, 0.10588235408067703, 0.10588235408067703),
            (0.10924369841814041, 0.10980392247438431, 0.10980392247438431),
            (0.11344537883996964, 0.11372549086809158, 0.11372549086809158),
            (0.11764705926179886, 0.11764705926179886, 0.11764705926179886),
            (0.12184873968362808, 0.12156862765550613, 0.12156862765550613),
            (0.1260504275560379, 0.12549020349979401, 0.12549020349979401),
            (0.13025210797786713, 0.12941177189350128, 0.12941177189350128),
            (0.13445378839969635, 0.13333334028720856, 0.13333334028720856),
            (0.13865546882152557, 0.13725490868091583, 0.13725490868091583),
            (0.1428571492433548, 0.14117647707462311, 0.14117647707462311),
            (0.14705882966518402, 0.14509804546833038, 0.14509804546833038),
            (0.15126051008701324, 0.14901961386203766, 0.14901961386203766),
            (0.15546219050884247, 0.15294118225574493, 0.15294118225574493),
            (0.15966387093067169, 0.16078431904315948, 0.16078431904315948),
            (0.16386555135250092, 0.16470588743686676, 0.16470588743686676),
            (0.16806723177433014, 0.16862745583057404, 0.16862745583057404),
            (0.17226891219615936, 0.17254902422428131, 0.17254902422428131),
            (0.17647059261798859, 0.17647059261798859, 0.17647059261798859),
            (0.18067227303981781, 0.18039216101169586, 0.18039216101169586),
            (0.18487395346164703, 0.18431372940540314, 0.18431372940540314),
            (0.18907563388347626, 0.18823529779911041, 0.18823529779911041),
            (0.19327731430530548, 0.19215686619281769, 0.19215686619281769),
            (0.1974789947271347, 0.19607843458652496, 0.19607843458652496),
            (0.20168067514896393, 0.20000000298023224, 0.20000000298023224),
            (0.20588235557079315, 0.20392157137393951, 0.20392157137393951),
            (0.21008403599262238, 0.20784313976764679, 0.20784313976764679),
            (0.2142857164144516, 0.21176470816135406, 0.21176470816135406),
            (0.21848739683628082, 0.21568627655506134, 0.21568627655506134),
            (0.22268907725811005, 0.22352941334247589, 0.22352941334247589),
            (0.22689075767993927, 0.22745098173618317, 0.22745098173618317),
            (0.23109243810176849, 0.23137255012989044, 0.23137255012989044),
            (0.23529411852359772, 0.23529411852359772, 0.23529411852359772),
            (0.23949579894542694, 0.23921568691730499, 0.23921568691730499),
            (0.24369747936725616, 0.24313725531101227, 0.24313725531101227),
            (0.24789915978908539, 0.24705882370471954, 0.24705882370471954),
            (0.25210085511207581, 0.25098040699958801, 0.25098040699958801),
            (0.25630253553390503, 0.25490197539329529, 0.25490197539329529),
            (0.26050421595573425, 0.25882354378700256, 0.25882354378700256),
            (0.26470589637756348, 0.26274511218070984, 0.26274511218070984),
            (0.2689075767993927, 0.26666668057441711, 0.26666668057441711),
            (0.27310925722122192, 0.27058824896812439, 0.27058824896812439),
            (0.27731093764305115, 0.27450981736183167, 0.27450981736183167),
            (0.28151261806488037, 0.27843138575553894, 0.27843138575553894),
            (0.28571429848670959, 0.28627452254295349, 0.28627452254295349),
            (0.28991597890853882, 0.29019609093666077, 0.29019609093666077),
            (0.29411765933036804, 0.29411765933036804, 0.29411765933036804),
            (0.29831933975219727, 0.29803922772407532, 0.29803922772407532),
            (0.30252102017402649, 0.30196079611778259, 0.30196079611778259),
            (0.30672270059585571, 0.30588236451148987, 0.30588236451148987),
            (0.31092438101768494, 0.30980393290519714, 0.30980393290519714),
            (0.31512606143951416, 0.31372550129890442, 0.31372550129890442),
            (0.31932774186134338, 0.31764706969261169, 0.31764706969261169),
            (0.32352942228317261, 0.32156863808631897, 0.32156863808631897),
            (0.32773110270500183, 0.32549020648002625, 0.32549020648002625),
            (0.33193278312683105, 0.32941177487373352, 0.32941177487373352),
            (0.33613446354866028, 0.3333333432674408, 0.3333333432674408),
            (0.3403361439704895, 0.33725491166114807, 0.33725491166114807),
            (0.34453782439231873, 0.34117648005485535, 0.34117648005485535),
            (0.34873950481414795, 0.3490196168422699, 0.3490196168422699),
            (0.35294118523597717, 0.35294118523597717, 0.35294118523597717),
            (0.3571428656578064, 0.35686275362968445, 0.35686275362968445),
            (0.36134454607963562, 0.36078432202339172, 0.36078432202339172),
            (0.36554622650146484, 0.364705890417099, 0.364705890417099),
            (0.36974790692329407, 0.36862745881080627, 0.36862745881080627),
            (0.37394958734512329, 0.37254902720451355, 0.37254902720451355),
            (0.37815126776695251, 0.37647059559822083, 0.37647059559822083),
            (0.38235294818878174, 0.3803921639919281, 0.3803921639919281),
            (0.38655462861061096, 0.38431373238563538, 0.38431373238563538),
            (0.39075630903244019, 0.38823530077934265, 0.38823530077934265),
            (0.39495798945426941, 0.39215686917304993, 0.39215686917304993),
            (0.39915966987609863, 0.3960784375667572, 0.3960784375667572),
            (0.40336135029792786, 0.40000000596046448, 0.40000000596046448),
            (0.40756303071975708, 0.40392157435417175, 0.40392157435417175),
            (0.4117647111415863, 0.4117647111415863, 0.4117647111415863),
            (0.41596639156341553, 0.41568627953529358, 0.41568627953529358),
            (0.42016807198524475, 0.41960784792900085, 0.41960784792900085),
            (0.42436975240707397, 0.42352941632270813, 0.42352941632270813),
            (0.4285714328289032, 0.42745098471641541, 0.42745098471641541),
            (0.43277311325073242, 0.43137255311012268, 0.43137255311012268),
            (0.43697479367256165, 0.43529412150382996, 0.43529412150382996),
            (0.44117647409439087, 0.43921568989753723, 0.43921568989753723),
            (0.44537815451622009, 0.44313725829124451, 0.44313725829124451),
            (0.44957983493804932, 0.44705882668495178, 0.44705882668495178),
            (0.45378151535987854, 0.45098039507865906, 0.45098039507865906),
            (0.45798319578170776, 0.45490196347236633, 0.45490196347236633),
            (0.46218487620353699, 0.45882353186607361, 0.45882353186607361),
            (0.46638655662536621, 0.46274510025978088, 0.46274510025978088),
            (0.47058823704719543, 0.46666666865348816, 0.46666666865348816),
            (0.47478991746902466, 0.47450980544090271, 0.47450980544090271),
            (0.47899159789085388, 0.47843137383460999, 0.47843137383460999),
            (0.48319327831268311, 0.48235294222831726, 0.48235294222831726),
            (0.48739495873451233, 0.48627451062202454, 0.48627451062202454),
            (0.49159663915634155, 0.49019607901573181, 0.49019607901573181),
            (0.49579831957817078, 0.49411764740943909, 0.49411764740943909),
            (0.5, 0.49803921580314636, 0.49803921580314636),
            (0.50420171022415161, 0.50196081399917603, 0.50196081399917603),
            (0.50840336084365845, 0.5058823823928833, 0.5058823823928833),
            (0.51260507106781006, 0.50980395078659058, 0.50980395078659058),
            (0.51680672168731689, 0.51372551918029785, 0.51372551918029785),
            (0.52100843191146851, 0.51764708757400513, 0.51764708757400513),
            (0.52521008253097534, 0.5215686559677124, 0.5215686559677124),
            (0.52941179275512695, 0.52549022436141968, 0.52549022436141968),
            (0.53361344337463379, 0.52941179275512695, 0.52941179275512695),
            (0.5378151535987854, 0.5372549295425415, 0.5372549295425415),
            (0.54201680421829224, 0.54117649793624878, 0.54117649793624878),
            (0.54621851444244385, 0.54509806632995605, 0.54509806632995605),
            (0.55042016506195068, 0.54901963472366333, 0.54901963472366333),
            (0.55462187528610229, 0.55294120311737061, 0.55294120311737061),
            (0.55882352590560913, 0.55686277151107788, 0.55686277151107788),
            (0.56302523612976074, 0.56078433990478516, 0.56078433990478516),
            (0.56722688674926758, 0.56470590829849243, 0.56470590829849243),
            (0.57142859697341919, 0.56862747669219971, 0.56862747669219971),
            (0.57563024759292603, 0.57254904508590698, 0.57254904508590698),
            (0.57983195781707764, 0.57647061347961426, 0.57647061347961426),
            (0.58403360843658447, 0.58039218187332153, 0.58039218187332153),
            (0.58823531866073608, 0.58431375026702881, 0.58431375026702881),
            (0.59243696928024292, 0.58823531866073608, 0.58823531866073608),
            (0.59663867950439453, 0.59215688705444336, 0.59215688705444336),
            (0.60084033012390137, 0.60000002384185791, 0.60000002384185791),
            (0.60504204034805298, 0.60392159223556519, 0.60392159223556519),
            (0.60924369096755981, 0.60784316062927246, 0.60784316062927246),
            (0.61344540119171143, 0.61176472902297974, 0.61176472902297974),
            (0.61764705181121826, 0.61568629741668701, 0.61568629741668701),
            (0.62184876203536987, 0.61960786581039429, 0.61960786581039429),
            (0.62605041265487671, 0.62352943420410156, 0.62352943420410156),
            (0.63025212287902832, 0.62745100259780884, 0.62745100259780884),
            (0.63445377349853516, 0.63137257099151611, 0.63137257099151611),
            (0.63865548372268677, 0.63529413938522339, 0.63529413938522339),
            (0.6428571343421936, 0.63921570777893066, 0.63921570777893066),
            (0.64705884456634521, 0.64313727617263794, 0.64313727617263794),
            (0.65126049518585205, 0.64705884456634521, 0.64705884456634521),
            (0.65546220541000366, 0.65098041296005249, 0.65098041296005249),
            (0.6596638560295105, 0.65490198135375977, 0.65490198135375977),
            (0.66386556625366211, 0.66274511814117432, 0.66274511814117432),
            (0.66806721687316895, 0.66666668653488159, 0.66666668653488159),
            (0.67226892709732056, 0.67058825492858887, 0.67058825492858887),
            (0.67647057771682739, 0.67450982332229614, 0.67450982332229614),
            (0.680672287940979, 0.67843139171600342, 0.67843139171600342),
            (0.68487393856048584, 0.68235296010971069, 0.68235296010971069),
            (0.68907564878463745, 0.68627452850341797, 0.68627452850341797),
            (0.69327729940414429, 0.69019609689712524, 0.69019609689712524),
            (0.6974790096282959, 0.69411766529083252, 0.69411766529083252),
            (0.70168066024780273, 0.69803923368453979, 0.69803923368453979),
            (0.70588237047195435, 0.70196080207824707, 0.70196080207824707),
            (0.71008402109146118, 0.70588237047195435, 0.70588237047195435),
            (0.71428573131561279, 0.70980393886566162, 0.70980393886566162),
            (0.71848738193511963, 0.7137255072593689, 0.7137255072593689),
            (0.72268909215927124, 0.71764707565307617, 0.71764707565307617),
            (0.72689074277877808, 0.72549021244049072, 0.72549021244049072),
            (0.73109245300292969, 0.729411780834198, 0.729411780834198),
            (0.73529410362243652, 0.73333334922790527, 0.73333334922790527),
            (0.73949581384658813, 0.73725491762161255, 0.73725491762161255),
            (0.74369746446609497, 0.74117648601531982, 0.74117648601531982),
            (0.74789917469024658, 0.7450980544090271, 0.7450980544090271),
            (0.75210082530975342, 0.74901962280273438, 0.74901962280273438),
            (0.75630253553390503, 0.75294119119644165, 0.75294119119644165),
            (0.76050418615341187, 0.75686275959014893, 0.75686275959014893),
            (0.76470589637756348, 0.7607843279838562, 0.7607843279838562),
            (0.76890754699707031, 0.76470589637756348, 0.76470589637756348),
            (0.77310925722122192, 0.76862746477127075, 0.76862746477127075),
            (0.77731090784072876, 0.77254903316497803, 0.77254903316497803),
            (0.78151261806488037, 0.7764706015586853, 0.7764706015586853),
            (0.78571426868438721, 0.78039216995239258, 0.78039216995239258),
            (0.78991597890853882, 0.78823530673980713, 0.78823530673980713),
            (0.79411762952804565, 0.7921568751335144, 0.7921568751335144),
            (0.79831933975219727, 0.79607844352722168, 0.79607844352722168),
            (0.8025209903717041, 0.80000001192092896, 0.80000001192092896),
            (0.80672270059585571, 0.80392158031463623, 0.80392158031463623),
            (0.81092435121536255, 0.80784314870834351, 0.80784314870834351),
            (0.81512606143951416, 0.81176471710205078, 0.81176471710205078),
            (0.819327712059021, 0.81568628549575806, 0.81568628549575806),
            (0.82352942228317261, 0.81960785388946533, 0.81960785388946533),
            (0.82773107290267944, 0.82352942228317261, 0.82352942228317261),
            (0.83193278312683105, 0.82745099067687988, 0.82745099067687988),
            (0.83613443374633789, 0.83137255907058716, 0.83137255907058716),
            (0.8403361439704895, 0.83529412746429443, 0.83529412746429443),
            (0.84453779458999634, 0.83921569585800171, 0.83921569585800171),
            (0.84873950481414795, 0.84313726425170898, 0.84313726425170898),
            (0.85294115543365479, 0.85098040103912354, 0.85098040103912354),
            (0.8571428656578064, 0.85490196943283081, 0.85490196943283081),
            (0.86134451627731323, 0.85882353782653809, 0.85882353782653809),
            (0.86554622650146484, 0.86274510622024536, 0.86274510622024536),
            (0.86974787712097168, 0.86666667461395264, 0.86666667461395264),
            (0.87394958734512329, 0.87058824300765991, 0.87058824300765991),
            (0.87815123796463013, 0.87450981140136719, 0.87450981140136719),
            (0.88235294818878174, 0.87843137979507446, 0.87843137979507446),
            (0.88655459880828857, 0.88235294818878174, 0.88235294818878174),
            (0.89075630903244019, 0.88627451658248901, 0.88627451658248901),
            (0.89495795965194702, 0.89019608497619629, 0.89019608497619629),
            (0.89915966987609863, 0.89411765336990356, 0.89411765336990356),
            (0.90336132049560547, 0.89803922176361084, 0.89803922176361084),
            (0.90756303071975708, 0.90196079015731812, 0.90196079015731812),
            (0.91176468133926392, 0.90588235855102539, 0.90588235855102539),
            (0.91596639156341553, 0.91372549533843994, 0.91372549533843994),
            (0.92016804218292236, 0.91764706373214722, 0.91764706373214722),
            (0.92436975240707397, 0.92156863212585449, 0.92156863212585449),
            (0.92857140302658081, 0.92549020051956177, 0.92549020051956177),
            (0.93277311325073242, 0.92941176891326904, 0.92941176891326904),
            (0.93697476387023926, 0.93333333730697632, 0.93333333730697632),
            (0.94117647409439087, 0.93725490570068359, 0.93725490570068359),
            (0.94537812471389771, 0.94117647409439087, 0.94117647409439087),
            (0.94957983493804932, 0.94509804248809814, 0.94509804248809814),
            (0.95378148555755615, 0.94901961088180542, 0.94901961088180542),
            (0.95798319578170776, 0.9529411792755127, 0.9529411792755127),
            (0.9621848464012146, 0.95686274766921997, 0.95686274766921997),
            (0.96638655662536621, 0.96078431606292725, 0.96078431606292725),
            (0.97058820724487305, 0.96470588445663452, 0.96470588445663452),
            (0.97478991746902466, 0.9686274528503418, 0.9686274528503418),
            (0.97899156808853149, 0.97647058963775635, 0.97647058963775635),
            (0.98319327831268311, 0.98039215803146362, 0.98039215803146362),
            (0.98739492893218994, 0.9843137264251709, 0.9843137264251709),
            (0.99159663915634155, 0.98823529481887817, 0.98823529481887817),
            (0.99579828977584839, 0.99215686321258545, 0.99215686321258545),
            (1.0, 0.99607843160629272, 0.99607843160629272),
        ],
        green=[
            (0.0, 0.0, 0.0),
            (
                0.0042016808874905109,
                0.0039215688593685627,
                0.0039215688593685627,
            ),
            (
                0.0084033617749810219,
                0.0078431377187371254,
                0.0078431377187371254,
            ),
            (0.012605042196810246, 0.011764706112444401, 0.011764706112444401),
            (0.016806723549962044, 0.015686275437474251, 0.015686275437474251),
            (0.021008403971791267, 0.019607843831181526, 0.019607843831181526),
            (0.025210084393620491, 0.023529412224888802, 0.023529412224888802),
            (0.029411764815449715, 0.027450980618596077, 0.027450980618596077),
            (0.033613447099924088, 0.035294119268655777, 0.035294119268655777),
            (0.037815127521753311, 0.039215687662363052, 0.039215687662363052),
            (0.042016807943582535, 0.043137256056070328, 0.043137256056070328),
            (0.046218488365411758, 0.047058824449777603, 0.047058824449777603),
            (0.050420168787240982, 0.050980392843484879, 0.050980392843484879),
            (0.054621849209070206, 0.054901961237192154, 0.054901961237192154),
            (0.058823529630899429, 0.058823529630899429, 0.058823529630899429),
            (0.063025213778018951, 0.062745101749897003, 0.062745101749897003),
            (0.067226894199848175, 0.066666670143604279, 0.066666670143604279),
            (0.071428574621677399, 0.070588238537311554, 0.070588238537311554),
            (0.075630255043506622, 0.074509806931018829, 0.074509806931018829),
            (0.079831935465335846, 0.078431375324726105, 0.078431375324726105),
            (0.08403361588716507, 0.08235294371843338, 0.08235294371843338),
            (0.088235296308994293, 0.086274512112140656, 0.086274512112140656),
            (0.092436976730823517, 0.090196080505847931, 0.090196080505847931),
            (0.09663865715265274, 0.098039217293262482, 0.098039217293262482),
            (0.10084033757448196, 0.10196078568696976, 0.10196078568696976),
            (0.10504201799631119, 0.10588235408067703, 0.10588235408067703),
            (0.10924369841814041, 0.10980392247438431, 0.10980392247438431),
            (0.11344537883996964, 0.11372549086809158, 0.11372549086809158),
            (0.11764705926179886, 0.11764705926179886, 0.11764705926179886),
            (0.12184873968362808, 0.12156862765550613, 0.12156862765550613),
            (0.1260504275560379, 0.12549020349979401, 0.12549020349979401),
            (0.13025210797786713, 0.12941177189350128, 0.12941177189350128),
            (0.13445378839969635, 0.13333334028720856, 0.13333334028720856),
            (0.13865546882152557, 0.13725490868091583, 0.13725490868091583),
            (0.1428571492433548, 0.14117647707462311, 0.14117647707462311),
            (0.14705882966518402, 0.14509804546833038, 0.14509804546833038),
            (0.15126051008701324, 0.14901961386203766, 0.14901961386203766),
            (0.15546219050884247, 0.15294118225574493, 0.15294118225574493),
            (0.15966387093067169, 0.16078431904315948, 0.16078431904315948),
            (0.16386555135250092, 0.16470588743686676, 0.16470588743686676),
            (0.16806723177433014, 0.16862745583057404, 0.16862745583057404),
            (0.17226891219615936, 0.17254902422428131, 0.17254902422428131),
            (0.17647059261798859, 0.17647059261798859, 0.17647059261798859),
            (0.18067227303981781, 0.18039216101169586, 0.18039216101169586),
            (0.18487395346164703, 0.18431372940540314, 0.18431372940540314),
            (0.18907563388347626, 0.18823529779911041, 0.18823529779911041),
            (0.19327731430530548, 0.19215686619281769, 0.19215686619281769),
            (0.1974789947271347, 0.19607843458652496, 0.19607843458652496),
            (0.20168067514896393, 0.20000000298023224, 0.20000000298023224),
            (0.20588235557079315, 0.20392157137393951, 0.20392157137393951),
            (0.21008403599262238, 0.20784313976764679, 0.20784313976764679),
            (0.2142857164144516, 0.21176470816135406, 0.21176470816135406),
            (0.21848739683628082, 0.21568627655506134, 0.21568627655506134),
            (0.22268907725811005, 0.22352941334247589, 0.22352941334247589),
            (0.22689075767993927, 0.22745098173618317, 0.22745098173618317),
            (0.23109243810176849, 0.23137255012989044, 0.23137255012989044),
            (0.23529411852359772, 0.23529411852359772, 0.23529411852359772),
            (0.23949579894542694, 0.23921568691730499, 0.23921568691730499),
            (0.24369747936725616, 0.24313725531101227, 0.24313725531101227),
            (0.24789915978908539, 0.24705882370471954, 0.24705882370471954),
            (0.25210085511207581, 0.25098040699958801, 0.25098040699958801),
            (0.25630253553390503, 0.25490197539329529, 0.25490197539329529),
            (0.26050421595573425, 0.25882354378700256, 0.25882354378700256),
            (0.26470589637756348, 0.26274511218070984, 0.26274511218070984),
            (0.2689075767993927, 0.26666668057441711, 0.26666668057441711),
            (0.27310925722122192, 0.27058824896812439, 0.27058824896812439),
            (0.27731093764305115, 0.27450981736183167, 0.27450981736183167),
            (0.28151261806488037, 0.27843138575553894, 0.27843138575553894),
            (0.28571429848670959, 0.28627452254295349, 0.28627452254295349),
            (0.28991597890853882, 0.29019609093666077, 0.29019609093666077),
            (0.29411765933036804, 0.29411765933036804, 0.29411765933036804),
            (0.29831933975219727, 0.29803922772407532, 0.29803922772407532),
            (0.30252102017402649, 0.30196079611778259, 0.30196079611778259),
            (0.30672270059585571, 0.30588236451148987, 0.30588236451148987),
            (0.31092438101768494, 0.30980393290519714, 0.30980393290519714),
            (0.31512606143951416, 0.31372550129890442, 0.31372550129890442),
            (0.31932774186134338, 0.31764706969261169, 0.31764706969261169),
            (0.32352942228317261, 0.32156863808631897, 0.32156863808631897),
            (0.32773110270500183, 0.32549020648002625, 0.32549020648002625),
            (0.33193278312683105, 0.32941177487373352, 0.32941177487373352),
            (0.33613446354866028, 0.3333333432674408, 0.3333333432674408),
            (0.3403361439704895, 0.33725491166114807, 0.33725491166114807),
            (0.34453782439231873, 0.34117648005485535, 0.34117648005485535),
            (0.34873950481414795, 0.3490196168422699, 0.3490196168422699),
            (0.35294118523597717, 0.35294118523597717, 0.35294118523597717),
            (0.3571428656578064, 0.35686275362968445, 0.35686275362968445),
            (0.36134454607963562, 0.36078432202339172, 0.36078432202339172),
            (0.36554622650146484, 0.364705890417099, 0.364705890417099),
            (0.36974790692329407, 0.36862745881080627, 0.36862745881080627),
            (0.37394958734512329, 0.37254902720451355, 0.37254902720451355),
            (0.37815126776695251, 0.37647059559822083, 0.37647059559822083),
            (0.38235294818878174, 0.3803921639919281, 0.3803921639919281),
            (0.38655462861061096, 0.38431373238563538, 0.38431373238563538),
            (0.39075630903244019, 0.38823530077934265, 0.38823530077934265),
            (0.39495798945426941, 0.39215686917304993, 0.39215686917304993),
            (0.39915966987609863, 0.3960784375667572, 0.3960784375667572),
            (0.40336135029792786, 0.40000000596046448, 0.40000000596046448),
            (0.40756303071975708, 0.40392157435417175, 0.40392157435417175),
            (0.4117647111415863, 0.4117647111415863, 0.4117647111415863),
            (0.41596639156341553, 0.41568627953529358, 0.41568627953529358),
            (0.42016807198524475, 0.41960784792900085, 0.41960784792900085),
            (0.42436975240707397, 0.42352941632270813, 0.42352941632270813),
            (0.4285714328289032, 0.42745098471641541, 0.42745098471641541),
            (0.43277311325073242, 0.43137255311012268, 0.43137255311012268),
            (0.43697479367256165, 0.43529412150382996, 0.43529412150382996),
            (0.44117647409439087, 0.43921568989753723, 0.43921568989753723),
            (0.44537815451622009, 0.44313725829124451, 0.44313725829124451),
            (0.44957983493804932, 0.44705882668495178, 0.44705882668495178),
            (0.45378151535987854, 0.45098039507865906, 0.45098039507865906),
            (0.45798319578170776, 0.45490196347236633, 0.45490196347236633),
            (0.46218487620353699, 0.45882353186607361, 0.45882353186607361),
            (0.46638655662536621, 0.46274510025978088, 0.46274510025978088),
            (0.47058823704719543, 0.46666666865348816, 0.46666666865348816),
            (0.47478991746902466, 0.47450980544090271, 0.47450980544090271),
            (0.47899159789085388, 0.47843137383460999, 0.47843137383460999),
            (0.48319327831268311, 0.48235294222831726, 0.48235294222831726),
            (0.48739495873451233, 0.48627451062202454, 0.48627451062202454),
            (0.49159663915634155, 0.49019607901573181, 0.49019607901573181),
            (0.49579831957817078, 0.49411764740943909, 0.49411764740943909),
            (0.5, 0.49803921580314636, 0.49803921580314636),
            (0.50420171022415161, 0.50196081399917603, 0.50196081399917603),
            (0.50840336084365845, 0.5058823823928833, 0.5058823823928833),
            (0.51260507106781006, 0.50980395078659058, 0.50980395078659058),
            (0.51680672168731689, 0.51372551918029785, 0.51372551918029785),
            (0.52100843191146851, 0.51764708757400513, 0.51764708757400513),
            (0.52521008253097534, 0.5215686559677124, 0.5215686559677124),
            (0.52941179275512695, 0.52549022436141968, 0.52549022436141968),
            (0.53361344337463379, 0.52941179275512695, 0.52941179275512695),
            (0.5378151535987854, 0.5372549295425415, 0.5372549295425415),
            (0.54201680421829224, 0.54117649793624878, 0.54117649793624878),
            (0.54621851444244385, 0.54509806632995605, 0.54509806632995605),
            (0.55042016506195068, 0.54901963472366333, 0.54901963472366333),
            (0.55462187528610229, 0.55294120311737061, 0.55294120311737061),
            (0.55882352590560913, 0.55686277151107788, 0.55686277151107788),
            (0.56302523612976074, 0.56078433990478516, 0.56078433990478516),
            (0.56722688674926758, 0.56470590829849243, 0.56470590829849243),
            (0.57142859697341919, 0.56862747669219971, 0.56862747669219971),
            (0.57563024759292603, 0.57254904508590698, 0.57254904508590698),
            (0.57983195781707764, 0.57647061347961426, 0.57647061347961426),
            (0.58403360843658447, 0.58039218187332153, 0.58039218187332153),
            (0.58823531866073608, 0.58431375026702881, 0.58431375026702881),
            (0.59243696928024292, 0.58823531866073608, 0.58823531866073608),
            (0.59663867950439453, 0.59215688705444336, 0.59215688705444336),
            (0.60084033012390137, 0.60000002384185791, 0.60000002384185791),
            (0.60504204034805298, 0.60392159223556519, 0.60392159223556519),
            (0.60924369096755981, 0.60784316062927246, 0.60784316062927246),
            (0.61344540119171143, 0.61176472902297974, 0.61176472902297974),
            (0.61764705181121826, 0.61568629741668701, 0.61568629741668701),
            (0.62184876203536987, 0.61960786581039429, 0.61960786581039429),
            (0.62605041265487671, 0.62352943420410156, 0.62352943420410156),
            (0.63025212287902832, 0.62745100259780884, 0.62745100259780884),
            (0.63445377349853516, 0.63137257099151611, 0.63137257099151611),
            (0.63865548372268677, 0.63529413938522339, 0.63529413938522339),
            (0.6428571343421936, 0.63921570777893066, 0.63921570777893066),
            (0.64705884456634521, 0.64313727617263794, 0.64313727617263794),
            (0.65126049518585205, 0.64705884456634521, 0.64705884456634521),
            (0.65546220541000366, 0.65098041296005249, 0.65098041296005249),
            (0.6596638560295105, 0.65490198135375977, 0.65490198135375977),
            (0.66386556625366211, 0.66274511814117432, 0.66274511814117432),
            (0.66806721687316895, 0.66666668653488159, 0.66666668653488159),
            (0.67226892709732056, 0.67058825492858887, 0.67058825492858887),
            (0.67647057771682739, 0.67450982332229614, 0.67450982332229614),
            (0.680672287940979, 0.67843139171600342, 0.67843139171600342),
            (0.68487393856048584, 0.68235296010971069, 0.68235296010971069),
            (0.68907564878463745, 0.68627452850341797, 0.68627452850341797),
            (0.69327729940414429, 0.69019609689712524, 0.69019609689712524),
            (0.6974790096282959, 0.69411766529083252, 0.69411766529083252),
            (0.70168066024780273, 0.69803923368453979, 0.69803923368453979),
            (0.70588237047195435, 0.70196080207824707, 0.70196080207824707),
            (0.71008402109146118, 0.70588237047195435, 0.70588237047195435),
            (0.71428573131561279, 0.70980393886566162, 0.70980393886566162),
            (0.71848738193511963, 0.7137255072593689, 0.7137255072593689),
            (0.72268909215927124, 0.71764707565307617, 0.71764707565307617),
            (0.72689074277877808, 0.72549021244049072, 0.72549021244049072),
            (0.73109245300292969, 0.729411780834198, 0.729411780834198),
            (0.73529410362243652, 0.73333334922790527, 0.73333334922790527),
            (0.73949581384658813, 0.73725491762161255, 0.73725491762161255),
            (0.74369746446609497, 0.74117648601531982, 0.74117648601531982),
            (0.74789917469024658, 0.7450980544090271, 0.7450980544090271),
            (0.75210082530975342, 0.74901962280273438, 0.74901962280273438),
            (0.75630253553390503, 0.75294119119644165, 0.75294119119644165),
            (0.76050418615341187, 0.75686275959014893, 0.75686275959014893),
            (0.76470589637756348, 0.7607843279838562, 0.7607843279838562),
            (0.76890754699707031, 0.76470589637756348, 0.76470589637756348),
            (0.77310925722122192, 0.76862746477127075, 0.76862746477127075),
            (0.77731090784072876, 0.77254903316497803, 0.77254903316497803),
            (0.78151261806488037, 0.7764706015586853, 0.7764706015586853),
            (0.78571426868438721, 0.78039216995239258, 0.78039216995239258),
            (0.78991597890853882, 0.78823530673980713, 0.78823530673980713),
            (0.79411762952804565, 0.7921568751335144, 0.7921568751335144),
            (0.79831933975219727, 0.79607844352722168, 0.79607844352722168),
            (0.8025209903717041, 0.80000001192092896, 0.80000001192092896),
            (0.80672270059585571, 0.80392158031463623, 0.80392158031463623),
            (0.81092435121536255, 0.80784314870834351, 0.80784314870834351),
            (0.81512606143951416, 0.81176471710205078, 0.81176471710205078),
            (0.819327712059021, 0.81568628549575806, 0.81568628549575806),
            (0.82352942228317261, 0.81960785388946533, 0.81960785388946533),
            (0.82773107290267944, 0.82352942228317261, 0.82352942228317261),
            (0.83193278312683105, 0.82745099067687988, 0.82745099067687988),
            (0.83613443374633789, 0.83137255907058716, 0.83137255907058716),
            (0.8403361439704895, 0.83529412746429443, 0.83529412746429443),
            (0.84453779458999634, 0.83921569585800171, 0.83921569585800171),
            (0.84873950481414795, 0.84313726425170898, 0.84313726425170898),
            (0.85294115543365479, 0.85098040103912354, 0.85098040103912354),
            (0.8571428656578064, 0.85490196943283081, 0.85490196943283081),
            (0.86134451627731323, 0.85882353782653809, 0.85882353782653809),
            (0.86554622650146484, 0.86274510622024536, 0.86274510622024536),
            (0.86974787712097168, 0.86666667461395264, 0.86666667461395264),
            (0.87394958734512329, 0.87058824300765991, 0.87058824300765991),
            (0.87815123796463013, 0.87450981140136719, 0.87450981140136719),
            (0.88235294818878174, 0.87843137979507446, 0.87843137979507446),
            (0.88655459880828857, 0.88235294818878174, 0.88235294818878174),
            (0.89075630903244019, 0.88627451658248901, 0.88627451658248901),
            (0.89495795965194702, 0.89019608497619629, 0.89019608497619629),
            (0.89915966987609863, 0.89411765336990356, 0.89411765336990356),
            (0.90336132049560547, 0.89803922176361084, 0.89803922176361084),
            (0.90756303071975708, 0.90196079015731812, 0.90196079015731812),
            (0.91176468133926392, 0.90588235855102539, 0.90588235855102539),
            (0.91596639156341553, 0.91372549533843994, 0.91372549533843994),
            (0.92016804218292236, 0.91764706373214722, 0.91764706373214722),
            (0.92436975240707397, 0.92156863212585449, 0.92156863212585449),
            (0.92857140302658081, 0.92549020051956177, 0.92549020051956177),
            (0.93277311325073242, 0.92941176891326904, 0.92941176891326904),
            (0.93697476387023926, 0.93333333730697632, 0.93333333730697632),
            (0.94117647409439087, 0.93725490570068359, 0.93725490570068359),
            (0.94537812471389771, 0.94117647409439087, 0.94117647409439087),
            (0.94957983493804932, 0.94509804248809814, 0.94509804248809814),
            (0.95378148555755615, 0.94901961088180542, 0.94901961088180542),
            (0.95798319578170776, 0.9529411792755127, 0.9529411792755127),
            (0.9621848464012146, 0.95686274766921997, 0.95686274766921997),
            (0.96638655662536621, 0.96078431606292725, 0.96078431606292725),
            (0.97058820724487305, 0.96470588445663452, 0.96470588445663452),
            (0.97478991746902466, 0.9686274528503418, 0.9686274528503418),
            (0.97899156808853149, 0.97647058963775635, 0.97647058963775635),
            (0.98319327831268311, 0.98039215803146362, 0.98039215803146362),
            (0.98739492893218994, 0.9843137264251709, 0.9843137264251709),
            (0.99159663915634155, 0.98823529481887817, 0.98823529481887817),
            (0.99579828977584839, 0.99215686321258545, 0.99215686321258545),
            (1.0, 0.99607843160629272, 0.99607843160629272),
        ],
        blue=[
            (0.0, 0.0, 0.0),
            (
                0.0042016808874905109,
                0.0039215688593685627,
                0.0039215688593685627,
            ),
            (
                0.0084033617749810219,
                0.0078431377187371254,
                0.0078431377187371254,
            ),
            (0.012605042196810246, 0.011764706112444401, 0.011764706112444401),
            (0.016806723549962044, 0.015686275437474251, 0.015686275437474251),
            (0.021008403971791267, 0.019607843831181526, 0.019607843831181526),
            (0.025210084393620491, 0.023529412224888802, 0.023529412224888802),
            (0.029411764815449715, 0.027450980618596077, 0.027450980618596077),
            (0.033613447099924088, 0.035294119268655777, 0.035294119268655777),
            (0.037815127521753311, 0.039215687662363052, 0.039215687662363052),
            (0.042016807943582535, 0.043137256056070328, 0.043137256056070328),
            (0.046218488365411758, 0.047058824449777603, 0.047058824449777603),
            (0.050420168787240982, 0.050980392843484879, 0.050980392843484879),
            (0.054621849209070206, 0.054901961237192154, 0.054901961237192154),
            (0.058823529630899429, 0.058823529630899429, 0.058823529630899429),
            (0.063025213778018951, 0.062745101749897003, 0.062745101749897003),
            (0.067226894199848175, 0.066666670143604279, 0.066666670143604279),
            (0.071428574621677399, 0.070588238537311554, 0.070588238537311554),
            (0.075630255043506622, 0.074509806931018829, 0.074509806931018829),
            (0.079831935465335846, 0.078431375324726105, 0.078431375324726105),
            (0.08403361588716507, 0.08235294371843338, 0.08235294371843338),
            (0.088235296308994293, 0.086274512112140656, 0.086274512112140656),
            (0.092436976730823517, 0.090196080505847931, 0.090196080505847931),
            (0.09663865715265274, 0.098039217293262482, 0.098039217293262482),
            (0.10084033757448196, 0.10196078568696976, 0.10196078568696976),
            (0.10504201799631119, 0.10588235408067703, 0.10588235408067703),
            (0.10924369841814041, 0.10980392247438431, 0.10980392247438431),
            (0.11344537883996964, 0.11372549086809158, 0.11372549086809158),
            (0.11764705926179886, 0.11764705926179886, 0.11764705926179886),
            (0.12184873968362808, 0.12156862765550613, 0.12156862765550613),
            (0.1260504275560379, 0.12549020349979401, 0.12549020349979401),
            (0.13025210797786713, 0.12941177189350128, 0.12941177189350128),
            (0.13445378839969635, 0.13333334028720856, 0.13333334028720856),
            (0.13865546882152557, 0.13725490868091583, 0.13725490868091583),
            (0.1428571492433548, 0.14117647707462311, 0.14117647707462311),
            (0.14705882966518402, 0.14509804546833038, 0.14509804546833038),
            (0.15126051008701324, 0.14901961386203766, 0.14901961386203766),
            (0.15546219050884247, 0.15294118225574493, 0.15294118225574493),
            (0.15966387093067169, 0.16078431904315948, 0.16078431904315948),
            (0.16386555135250092, 0.16470588743686676, 0.16470588743686676),
            (0.16806723177433014, 0.16862745583057404, 0.16862745583057404),
            (0.17226891219615936, 0.17254902422428131, 0.17254902422428131),
            (0.17647059261798859, 0.17647059261798859, 0.17647059261798859),
            (0.18067227303981781, 0.18039216101169586, 0.18039216101169586),
            (0.18487395346164703, 0.18431372940540314, 0.18431372940540314),
            (0.18907563388347626, 0.18823529779911041, 0.18823529779911041),
            (0.19327731430530548, 0.19215686619281769, 0.19215686619281769),
            (0.1974789947271347, 0.19607843458652496, 0.19607843458652496),
            (0.20168067514896393, 0.20000000298023224, 0.20000000298023224),
            (0.20588235557079315, 0.20392157137393951, 0.20392157137393951),
            (0.21008403599262238, 0.20784313976764679, 0.20784313976764679),
            (0.2142857164144516, 0.21176470816135406, 0.21176470816135406),
            (0.21848739683628082, 0.21568627655506134, 0.21568627655506134),
            (0.22268907725811005, 0.22352941334247589, 0.22352941334247589),
            (0.22689075767993927, 0.22745098173618317, 0.22745098173618317),
            (0.23109243810176849, 0.23137255012989044, 0.23137255012989044),
            (0.23529411852359772, 0.23529411852359772, 0.23529411852359772),
            (0.23949579894542694, 0.23921568691730499, 0.23921568691730499),
            (0.24369747936725616, 0.24313725531101227, 0.24313725531101227),
            (0.24789915978908539, 0.24705882370471954, 0.24705882370471954),
            (0.25210085511207581, 0.25098040699958801, 0.25098040699958801),
            (0.25630253553390503, 0.25490197539329529, 0.25490197539329529),
            (0.26050421595573425, 0.25882354378700256, 0.25882354378700256),
            (0.26470589637756348, 0.26274511218070984, 0.26274511218070984),
            (0.2689075767993927, 0.26666668057441711, 0.26666668057441711),
            (0.27310925722122192, 0.27058824896812439, 0.27058824896812439),
            (0.27731093764305115, 0.27450981736183167, 0.27450981736183167),
            (0.28151261806488037, 0.27843138575553894, 0.27843138575553894),
            (0.28571429848670959, 0.28627452254295349, 0.28627452254295349),
            (0.28991597890853882, 0.29019609093666077, 0.29019609093666077),
            (0.29411765933036804, 0.29411765933036804, 0.29411765933036804),
            (0.29831933975219727, 0.29803922772407532, 0.29803922772407532),
            (0.30252102017402649, 0.30196079611778259, 0.30196079611778259),
            (0.30672270059585571, 0.30588236451148987, 0.30588236451148987),
            (0.31092438101768494, 0.30980393290519714, 0.30980393290519714),
            (0.31512606143951416, 0.31372550129890442, 0.31372550129890442),
            (0.31932774186134338, 0.31764706969261169, 0.31764706969261169),
            (0.32352942228317261, 0.32156863808631897, 0.32156863808631897),
            (0.32773110270500183, 0.32549020648002625, 0.32549020648002625),
            (0.33193278312683105, 0.32941177487373352, 0.32941177487373352),
            (0.33613446354866028, 0.3333333432674408, 0.3333333432674408),
            (0.3403361439704895, 0.33725491166114807, 0.33725491166114807),
            (0.34453782439231873, 0.34117648005485535, 0.34117648005485535),
            (0.34873950481414795, 0.3490196168422699, 0.3490196168422699),
            (0.35294118523597717, 0.35294118523597717, 0.35294118523597717),
            (0.3571428656578064, 0.35686275362968445, 0.35686275362968445),
            (0.36134454607963562, 0.36078432202339172, 0.36078432202339172),
            (0.36554622650146484, 0.364705890417099, 0.364705890417099),
            (0.36974790692329407, 0.36862745881080627, 0.36862745881080627),
            (0.37394958734512329, 0.37254902720451355, 0.37254902720451355),
            (0.37815126776695251, 0.37647059559822083, 0.37647059559822083),
            (0.38235294818878174, 0.3803921639919281, 0.3803921639919281),
            (0.38655462861061096, 0.38431373238563538, 0.38431373238563538),
            (0.39075630903244019, 0.38823530077934265, 0.38823530077934265),
            (0.39495798945426941, 0.39215686917304993, 0.39215686917304993),
            (0.39915966987609863, 0.3960784375667572, 0.3960784375667572),
            (0.40336135029792786, 0.40000000596046448, 0.40000000596046448),
            (0.40756303071975708, 0.40392157435417175, 0.40392157435417175),
            (0.4117647111415863, 0.4117647111415863, 0.4117647111415863),
            (0.41596639156341553, 0.41568627953529358, 0.41568627953529358),
            (0.42016807198524475, 0.41960784792900085, 0.41960784792900085),
            (0.42436975240707397, 0.42352941632270813, 0.42352941632270813),
            (0.4285714328289032, 0.42745098471641541, 0.42745098471641541),
            (0.43277311325073242, 0.43137255311012268, 0.43137255311012268),
            (0.43697479367256165, 0.43529412150382996, 0.43529412150382996),
            (0.44117647409439087, 0.43921568989753723, 0.43921568989753723),
            (0.44537815451622009, 0.44313725829124451, 0.44313725829124451),
            (0.44957983493804932, 0.44705882668495178, 0.44705882668495178),
            (0.45378151535987854, 0.45098039507865906, 0.45098039507865906),
            (0.45798319578170776, 0.45490196347236633, 0.45490196347236633),
            (0.46218487620353699, 0.45882353186607361, 0.45882353186607361),
            (0.46638655662536621, 0.46274510025978088, 0.46274510025978088),
            (0.47058823704719543, 0.46666666865348816, 0.46666666865348816),
            (0.47478991746902466, 0.47450980544090271, 0.47450980544090271),
            (0.47899159789085388, 0.47843137383460999, 0.47843137383460999),
            (0.48319327831268311, 0.48235294222831726, 0.48235294222831726),
            (0.48739495873451233, 0.48627451062202454, 0.48627451062202454),
            (0.49159663915634155, 0.49019607901573181, 0.49019607901573181),
            (0.49579831957817078, 0.49411764740943909, 0.49411764740943909),
            (0.5, 0.49803921580314636, 0.49803921580314636),
            (0.50420171022415161, 0.50196081399917603, 0.50196081399917603),
            (0.50840336084365845, 0.5058823823928833, 0.5058823823928833),
            (0.51260507106781006, 0.50980395078659058, 0.50980395078659058),
            (0.51680672168731689, 0.51372551918029785, 0.51372551918029785),
            (0.52100843191146851, 0.51764708757400513, 0.51764708757400513),
            (0.52521008253097534, 0.5215686559677124, 0.5215686559677124),
            (0.52941179275512695, 0.52549022436141968, 0.52549022436141968),
            (0.53361344337463379, 0.52941179275512695, 0.52941179275512695),
            (0.5378151535987854, 0.5372549295425415, 0.5372549295425415),
            (0.54201680421829224, 0.54117649793624878, 0.54117649793624878),
            (0.54621851444244385, 0.54509806632995605, 0.54509806632995605),
            (0.55042016506195068, 0.54901963472366333, 0.54901963472366333),
            (0.55462187528610229, 0.55294120311737061, 0.55294120311737061),
            (0.55882352590560913, 0.55686277151107788, 0.55686277151107788),
            (0.56302523612976074, 0.56078433990478516, 0.56078433990478516),
            (0.56722688674926758, 0.56470590829849243, 0.56470590829849243),
            (0.57142859697341919, 0.56862747669219971, 0.56862747669219971),
            (0.57563024759292603, 0.57254904508590698, 0.57254904508590698),
            (0.57983195781707764, 0.57647061347961426, 0.57647061347961426),
            (0.58403360843658447, 0.58039218187332153, 0.58039218187332153),
            (0.58823531866073608, 0.58431375026702881, 0.58431375026702881),
            (0.59243696928024292, 0.58823531866073608, 0.58823531866073608),
            (0.59663867950439453, 0.59215688705444336, 0.59215688705444336),
            (0.60084033012390137, 0.60000002384185791, 0.60000002384185791),
            (0.60504204034805298, 0.60392159223556519, 0.60392159223556519),
            (0.60924369096755981, 0.60784316062927246, 0.60784316062927246),
            (0.61344540119171143, 0.61176472902297974, 0.61176472902297974),
            (0.61764705181121826, 0.61568629741668701, 0.61568629741668701),
            (0.62184876203536987, 0.61960786581039429, 0.61960786581039429),
            (0.62605041265487671, 0.62352943420410156, 0.62352943420410156),
            (0.63025212287902832, 0.62745100259780884, 0.62745100259780884),
            (0.63445377349853516, 0.63137257099151611, 0.63137257099151611),
            (0.63865548372268677, 0.63529413938522339, 0.63529413938522339),
            (0.6428571343421936, 0.63921570777893066, 0.63921570777893066),
            (0.64705884456634521, 0.64313727617263794, 0.64313727617263794),
            (0.65126049518585205, 0.64705884456634521, 0.64705884456634521),
            (0.65546220541000366, 0.65098041296005249, 0.65098041296005249),
            (0.6596638560295105, 0.65490198135375977, 0.65490198135375977),
            (0.66386556625366211, 0.66274511814117432, 0.66274511814117432),
            (0.66806721687316895, 0.66666668653488159, 0.66666668653488159),
            (0.67226892709732056, 0.67058825492858887, 0.67058825492858887),
            (0.67647057771682739, 0.67450982332229614, 0.67450982332229614),
            (0.680672287940979, 0.67843139171600342, 0.67843139171600342),
            (0.68487393856048584, 0.68235296010971069, 0.68235296010971069),
            (0.68907564878463745, 0.68627452850341797, 0.68627452850341797),
            (0.69327729940414429, 0.69019609689712524, 0.69019609689712524),
            (0.6974790096282959, 0.69411766529083252, 0.69411766529083252),
            (0.70168066024780273, 0.69803923368453979, 0.69803923368453979),
            (0.70588237047195435, 0.70196080207824707, 0.70196080207824707),
            (0.71008402109146118, 0.70588237047195435, 0.70588237047195435),
            (0.71428573131561279, 0.70980393886566162, 0.70980393886566162),
            (0.71848738193511963, 0.7137255072593689, 0.7137255072593689),
            (0.72268909215927124, 0.71764707565307617, 0.71764707565307617),
            (0.72689074277877808, 0.72549021244049072, 0.72549021244049072),
            (0.73109245300292969, 0.729411780834198, 0.729411780834198),
            (0.73529410362243652, 0.73333334922790527, 0.73333334922790527),
            (0.73949581384658813, 0.73725491762161255, 0.73725491762161255),
            (0.74369746446609497, 0.74117648601531982, 0.74117648601531982),
            (0.74789917469024658, 0.7450980544090271, 0.7450980544090271),
            (0.75210082530975342, 0.74901962280273438, 0.74901962280273438),
            (0.75630253553390503, 0.75294119119644165, 0.75294119119644165),
            (0.76050418615341187, 0.75686275959014893, 0.75686275959014893),
            (0.76470589637756348, 0.7607843279838562, 0.7607843279838562),
            (0.76890754699707031, 0.76470589637756348, 0.76470589637756348),
            (0.77310925722122192, 0.76862746477127075, 0.76862746477127075),
            (0.77731090784072876, 0.77254903316497803, 0.77254903316497803),
            (0.78151261806488037, 0.7764706015586853, 0.7764706015586853),
            (0.78571426868438721, 0.78039216995239258, 0.78039216995239258),
            (0.78991597890853882, 0.78823530673980713, 0.78823530673980713),
            (0.79411762952804565, 0.7921568751335144, 0.7921568751335144),
            (0.79831933975219727, 0.79607844352722168, 0.79607844352722168),
            (0.8025209903717041, 0.80000001192092896, 0.80000001192092896),
            (0.80672270059585571, 0.80392158031463623, 0.80392158031463623),
            (0.81092435121536255, 0.80784314870834351, 0.80784314870834351),
            (0.81512606143951416, 0.81176471710205078, 0.81176471710205078),
            (0.819327712059021, 0.81568628549575806, 0.81568628549575806),
            (0.82352942228317261, 0.81960785388946533, 0.81960785388946533),
            (0.82773107290267944, 0.82352942228317261, 0.82352942228317261),
            (0.83193278312683105, 0.82745099067687988, 0.82745099067687988),
            (0.83613443374633789, 0.83137255907058716, 0.83137255907058716),
            (0.8403361439704895, 0.83529412746429443, 0.83529412746429443),
            (0.84453779458999634, 0.83921569585800171, 0.83921569585800171),
            (0.84873950481414795, 0.84313726425170898, 0.84313726425170898),
            (0.85294115543365479, 0.85098040103912354, 0.85098040103912354),
            (0.8571428656578064, 0.85490196943283081, 0.85490196943283081),
            (0.86134451627731323, 0.85882353782653809, 0.85882353782653809),
            (0.86554622650146484, 0.86274510622024536, 0.86274510622024536),
            (0.86974787712097168, 0.86666667461395264, 0.86666667461395264),
            (0.87394958734512329, 0.87058824300765991, 0.87058824300765991),
            (0.87815123796463013, 0.87450981140136719, 0.87450981140136719),
            (0.88235294818878174, 0.87843137979507446, 0.87843137979507446),
            (0.88655459880828857, 0.88235294818878174, 0.88235294818878174),
            (0.89075630903244019, 0.88627451658248901, 0.88627451658248901),
            (0.89495795965194702, 0.89019608497619629, 0.89019608497619629),
            (0.89915966987609863, 0.89411765336990356, 0.89411765336990356),
            (0.90336132049560547, 0.89803922176361084, 0.89803922176361084),
            (0.90756303071975708, 0.90196079015731812, 0.90196079015731812),
            (0.91176468133926392, 0.90588235855102539, 0.90588235855102539),
            (0.91596639156341553, 0.91372549533843994, 0.91372549533843994),
            (0.92016804218292236, 0.91764706373214722, 0.91764706373214722),
            (0.92436975240707397, 0.92156863212585449, 0.92156863212585449),
            (0.92857140302658081, 0.92549020051956177, 0.92549020051956177),
            (0.93277311325073242, 0.92941176891326904, 0.92941176891326904),
            (0.93697476387023926, 0.93333333730697632, 0.93333333730697632),
            (0.94117647409439087, 0.93725490570068359, 0.93725490570068359),
            (0.94537812471389771, 0.94117647409439087, 0.94117647409439087),
            (0.94957983493804932, 0.94509804248809814, 0.94509804248809814),
            (0.95378148555755615, 0.94901961088180542, 0.94901961088180542),
            (0.95798319578170776, 0.9529411792755127, 0.9529411792755127),
            (0.9621848464012146, 0.95686274766921997, 0.95686274766921997),
            (0.96638655662536621, 0.96078431606292725, 0.96078431606292725),
            (0.97058820724487305, 0.96470588445663452, 0.96470588445663452),
            (0.97478991746902466, 0.9686274528503418, 0.9686274528503418),
            (0.97899156808853149, 0.97647058963775635, 0.97647058963775635),
            (0.98319327831268311, 0.98039215803146362, 0.98039215803146362),
            (0.98739492893218994, 0.9843137264251709, 0.9843137264251709),
            (0.99159663915634155, 0.98823529481887817, 0.98823529481887817),
            (0.99579828977584839, 0.99215686321258545, 0.99215686321258545),
            (1.0, 0.99607843160629272, 0.99607843160629272),
        ],
    )
    return ColorMapper.from_segment_map(_data, range=range, **traits)


def gist_heat(range, **traits):
    """Generator for the 'gist_heat' colormap from GIST."""
    _data = dict(
        red=[
            (0.0, 0.0, 0.0),
            (
                0.0042016808874905109,
                0.0039215688593685627,
                0.0039215688593685627,
            ),
            (
                0.0084033617749810219,
                0.0078431377187371254,
                0.0078431377187371254,
            ),
            (0.012605042196810246, 0.015686275437474251, 0.015686275437474251),
            (0.016806723549962044, 0.019607843831181526, 0.019607843831181526),
            (0.021008403971791267, 0.027450980618596077, 0.027450980618596077),
            (0.025210084393620491, 0.031372550874948502, 0.031372550874948502),
            (0.029411764815449715, 0.039215687662363052, 0.039215687662363052),
            (0.033613447099924088, 0.043137256056070328, 0.043137256056070328),
            (0.037815127521753311, 0.050980392843484879, 0.050980392843484879),
            (0.042016807943582535, 0.058823529630899429, 0.058823529630899429),
            (0.046218488365411758, 0.066666670143604279, 0.066666670143604279),
            (0.050420168787240982, 0.070588238537311554, 0.070588238537311554),
            (0.054621849209070206, 0.078431375324726105, 0.078431375324726105),
            (0.058823529630899429, 0.08235294371843338, 0.08235294371843338),
            (0.063025213778018951, 0.090196080505847931, 0.090196080505847931),
            (0.067226894199848175, 0.094117648899555206, 0.094117648899555206),
            (0.071428574621677399, 0.10196078568696976, 0.10196078568696976),
            (0.075630255043506622, 0.10588235408067703, 0.10588235408067703),
            (0.079831935465335846, 0.10980392247438431, 0.10980392247438431),
            (0.08403361588716507, 0.11764705926179886, 0.11764705926179886),
            (0.088235296308994293, 0.12156862765550613, 0.12156862765550613),
            (0.092436976730823517, 0.12941177189350128, 0.12941177189350128),
            (0.09663865715265274, 0.13333334028720856, 0.13333334028720856),
            (0.10084033757448196, 0.14117647707462311, 0.14117647707462311),
            (0.10504201799631119, 0.14509804546833038, 0.14509804546833038),
            (0.10924369841814041, 0.15294118225574493, 0.15294118225574493),
            (0.11344537883996964, 0.15686275064945221, 0.15686275064945221),
            (0.11764705926179886, 0.16470588743686676, 0.16470588743686676),
            (0.12184873968362808, 0.16862745583057404, 0.16862745583057404),
            (0.1260504275560379, 0.18039216101169586, 0.18039216101169586),
            (0.13025210797786713, 0.18431372940540314, 0.18431372940540314),
            (0.13445378839969635, 0.19215686619281769, 0.19215686619281769),
            (0.13865546882152557, 0.19607843458652496, 0.19607843458652496),
            (0.1428571492433548, 0.20392157137393951, 0.20392157137393951),
            (0.14705882966518402, 0.20784313976764679, 0.20784313976764679),
            (0.15126051008701324, 0.21568627655506134, 0.21568627655506134),
            (0.15546219050884247, 0.21960784494876862, 0.21960784494876862),
            (0.15966387093067169, 0.22352941334247589, 0.22352941334247589),
            (0.16386555135250092, 0.23137255012989044, 0.23137255012989044),
            (0.16806723177433014, 0.23529411852359772, 0.23529411852359772),
            (0.17226891219615936, 0.24313725531101227, 0.24313725531101227),
            (0.17647059261798859, 0.24705882370471954, 0.24705882370471954),
            (0.18067227303981781, 0.25490197539329529, 0.25490197539329529),
            (0.18487395346164703, 0.25882354378700256, 0.25882354378700256),
            (0.18907563388347626, 0.26666668057441711, 0.26666668057441711),
            (0.19327731430530548, 0.27058824896812439, 0.27058824896812439),
            (0.1974789947271347, 0.27450981736183167, 0.27450981736183167),
            (0.20168067514896393, 0.28235295414924622, 0.28235295414924622),
            (0.20588235557079315, 0.28627452254295349, 0.28627452254295349),
            (0.21008403599262238, 0.29803922772407532, 0.29803922772407532),
            (0.2142857164144516, 0.30588236451148987, 0.30588236451148987),
            (0.21848739683628082, 0.30980393290519714, 0.30980393290519714),
            (0.22268907725811005, 0.31764706969261169, 0.31764706969261169),
            (0.22689075767993927, 0.32156863808631897, 0.32156863808631897),
            (0.23109243810176849, 0.32941177487373352, 0.32941177487373352),
            (0.23529411852359772, 0.3333333432674408, 0.3333333432674408),
            (0.23949579894542694, 0.33725491166114807, 0.33725491166114807),
            (0.24369747936725616, 0.34509804844856262, 0.34509804844856262),
            (0.24789915978908539, 0.3490196168422699, 0.3490196168422699),
            (0.25210085511207581, 0.36078432202339172, 0.36078432202339172),
            (0.25630253553390503, 0.36862745881080627, 0.36862745881080627),
            (0.26050421595573425, 0.37254902720451355, 0.37254902720451355),
            (0.26470589637756348, 0.3803921639919281, 0.3803921639919281),
            (0.2689075767993927, 0.38431373238563538, 0.38431373238563538),
            (0.27310925722122192, 0.38823530077934265, 0.38823530077934265),
            (0.27731093764305115, 0.3960784375667572, 0.3960784375667572),
            (0.28151261806488037, 0.40000000596046448, 0.40000000596046448),
            (0.28571429848670959, 0.40784314274787903, 0.40784314274787903),
            (0.28991597890853882, 0.4117647111415863, 0.4117647111415863),
            (0.29411765933036804, 0.42352941632270813, 0.42352941632270813),
            (0.29831933975219727, 0.43137255311012268, 0.43137255311012268),
            (0.30252102017402649, 0.43529412150382996, 0.43529412150382996),
            (0.30672270059585571, 0.44313725829124451, 0.44313725829124451),
            (0.31092438101768494, 0.44705882668495178, 0.44705882668495178),
            (0.31512606143951416, 0.45098039507865906, 0.45098039507865906),
            (0.31932774186134338, 0.45882353186607361, 0.45882353186607361),
            (0.32352942228317261, 0.46274510025978088, 0.46274510025978088),
            (0.32773110270500183, 0.47058823704719543, 0.47058823704719543),
            (0.33193278312683105, 0.47450980544090271, 0.47450980544090271),
            (0.33613446354866028, 0.48235294222831726, 0.48235294222831726),
            (0.3403361439704895, 0.48627451062202454, 0.48627451062202454),
            (0.34453782439231873, 0.49411764740943909, 0.49411764740943909),
            (0.34873950481414795, 0.49803921580314636, 0.49803921580314636),
            (0.35294118523597717, 0.50196081399917603, 0.50196081399917603),
            (0.3571428656578064, 0.50980395078659058, 0.50980395078659058),
            (0.36134454607963562, 0.51372551918029785, 0.51372551918029785),
            (0.36554622650146484, 0.5215686559677124, 0.5215686559677124),
            (0.36974790692329407, 0.52549022436141968, 0.52549022436141968),
            (0.37394958734512329, 0.53333336114883423, 0.53333336114883423),
            (0.37815126776695251, 0.54509806632995605, 0.54509806632995605),
            (0.38235294818878174, 0.54901963472366333, 0.54901963472366333),
            (0.38655462861061096, 0.55294120311737061, 0.55294120311737061),
            (0.39075630903244019, 0.56078433990478516, 0.56078433990478516),
            (0.39495798945426941, 0.56470590829849243, 0.56470590829849243),
            (0.39915966987609863, 0.57254904508590698, 0.57254904508590698),
            (0.40336135029792786, 0.57647061347961426, 0.57647061347961426),
            (0.40756303071975708, 0.58431375026702881, 0.58431375026702881),
            (0.4117647111415863, 0.58823531866073608, 0.58823531866073608),
            (0.41596639156341553, 0.59607845544815063, 0.59607845544815063),
            (0.42016807198524475, 0.60000002384185791, 0.60000002384185791),
            (0.42436975240707397, 0.60784316062927246, 0.60784316062927246),
            (0.4285714328289032, 0.61176472902297974, 0.61176472902297974),
            (0.43277311325073242, 0.61568629741668701, 0.61568629741668701),
            (0.43697479367256165, 0.62352943420410156, 0.62352943420410156),
            (0.44117647409439087, 0.62745100259780884, 0.62745100259780884),
            (0.44537815451622009, 0.63529413938522339, 0.63529413938522339),
            (0.44957983493804932, 0.63921570777893066, 0.63921570777893066),
            (0.45378151535987854, 0.64705884456634521, 0.64705884456634521),
            (0.45798319578170776, 0.65098041296005249, 0.65098041296005249),
            (0.46218487620353699, 0.66274511814117432, 0.66274511814117432),
            (0.46638655662536621, 0.66666668653488159, 0.66666668653488159),
            (0.47058823704719543, 0.67450982332229614, 0.67450982332229614),
            (0.47478991746902466, 0.67843139171600342, 0.67843139171600342),
            (0.47899159789085388, 0.68627452850341797, 0.68627452850341797),
            (0.48319327831268311, 0.69019609689712524, 0.69019609689712524),
            (0.48739495873451233, 0.69803923368453979, 0.69803923368453979),
            (0.49159663915634155, 0.70196080207824707, 0.70196080207824707),
            (0.49579831957817078, 0.70980393886566162, 0.70980393886566162),
            (0.5, 0.7137255072593689, 0.7137255072593689),
            (0.50420171022415161, 0.72549021244049072, 0.72549021244049072),
            (0.50840336084365845, 0.729411780834198, 0.729411780834198),
            (0.51260507106781006, 0.73725491762161255, 0.73725491762161255),
            (0.51680672168731689, 0.74117648601531982, 0.74117648601531982),
            (0.52100843191146851, 0.74901962280273438, 0.74901962280273438),
            (0.52521008253097534, 0.75294119119644165, 0.75294119119644165),
            (0.52941179275512695, 0.7607843279838562, 0.7607843279838562),
            (0.53361344337463379, 0.76470589637756348, 0.76470589637756348),
            (0.5378151535987854, 0.77254903316497803, 0.77254903316497803),
            (0.54201680421829224, 0.7764706015586853, 0.7764706015586853),
            (0.54621851444244385, 0.78823530673980713, 0.78823530673980713),
            (0.55042016506195068, 0.7921568751335144, 0.7921568751335144),
            (0.55462187528610229, 0.80000001192092896, 0.80000001192092896),
            (0.55882352590560913, 0.80392158031463623, 0.80392158031463623),
            (0.56302523612976074, 0.81176471710205078, 0.81176471710205078),
            (0.56722688674926758, 0.81568628549575806, 0.81568628549575806),
            (0.57142859697341919, 0.82352942228317261, 0.82352942228317261),
            (0.57563024759292603, 0.82745099067687988, 0.82745099067687988),
            (0.57983195781707764, 0.83137255907058716, 0.83137255907058716),
            (0.58403360843658447, 0.83921569585800171, 0.83921569585800171),
            (0.58823531866073608, 0.84313726425170898, 0.84313726425170898),
            (0.59243696928024292, 0.85098040103912354, 0.85098040103912354),
            (0.59663867950439453, 0.85490196943283081, 0.85490196943283081),
            (0.60084033012390137, 0.86274510622024536, 0.86274510622024536),
            (0.60504204034805298, 0.86666667461395264, 0.86666667461395264),
            (0.60924369096755981, 0.87450981140136719, 0.87450981140136719),
            (0.61344540119171143, 0.87843137979507446, 0.87843137979507446),
            (0.61764705181121826, 0.88627451658248901, 0.88627451658248901),
            (0.62184876203536987, 0.89019608497619629, 0.89019608497619629),
            (0.62605041265487671, 0.89411765336990356, 0.89411765336990356),
            (0.63025212287902832, 0.90588235855102539, 0.90588235855102539),
            (0.63445377349853516, 0.91372549533843994, 0.91372549533843994),
            (0.63865548372268677, 0.91764706373214722, 0.91764706373214722),
            (0.6428571343421936, 0.92549020051956177, 0.92549020051956177),
            (0.64705884456634521, 0.92941176891326904, 0.92941176891326904),
            (0.65126049518585205, 0.93725490570068359, 0.93725490570068359),
            (0.65546220541000366, 0.94117647409439087, 0.94117647409439087),
            (0.6596638560295105, 0.94509804248809814, 0.94509804248809814),
            (0.66386556625366211, 0.9529411792755127, 0.9529411792755127),
            (0.66806721687316895, 0.95686274766921997, 0.95686274766921997),
            (0.67226892709732056, 0.96470588445663452, 0.96470588445663452),
            (0.67647057771682739, 0.9686274528503418, 0.9686274528503418),
            (0.680672287940979, 0.97647058963775635, 0.97647058963775635),
            (0.68487393856048584, 0.98039215803146362, 0.98039215803146362),
            (0.68907564878463745, 0.98823529481887817, 0.98823529481887817),
            (0.69327729940414429, 0.99215686321258545, 0.99215686321258545),
            (0.6974790096282959, 1.0, 1.0),
            (0.70168066024780273, 1.0, 1.0),
            (0.70588237047195435, 1.0, 1.0),
            (0.71008402109146118, 1.0, 1.0),
            (0.71428573131561279, 1.0, 1.0),
            (0.71848738193511963, 1.0, 1.0),
            (0.72268909215927124, 1.0, 1.0),
            (0.72689074277877808, 1.0, 1.0),
            (0.73109245300292969, 1.0, 1.0),
            (0.73529410362243652, 1.0, 1.0),
            (0.73949581384658813, 1.0, 1.0),
            (0.74369746446609497, 1.0, 1.0),
            (0.74789917469024658, 1.0, 1.0),
            (0.75210082530975342, 1.0, 1.0),
            (0.75630253553390503, 1.0, 1.0),
            (0.76050418615341187, 1.0, 1.0),
            (0.76470589637756348, 1.0, 1.0),
            (0.76890754699707031, 1.0, 1.0),
            (0.77310925722122192, 1.0, 1.0),
            (0.77731090784072876, 1.0, 1.0),
            (0.78151261806488037, 1.0, 1.0),
            (0.78571426868438721, 1.0, 1.0),
            (0.78991597890853882, 1.0, 1.0),
            (0.79411762952804565, 1.0, 1.0),
            (0.79831933975219727, 1.0, 1.0),
            (0.8025209903717041, 1.0, 1.0),
            (0.80672270059585571, 1.0, 1.0),
            (0.81092435121536255, 1.0, 1.0),
            (0.81512606143951416, 1.0, 1.0),
            (0.819327712059021, 1.0, 1.0),
            (0.82352942228317261, 1.0, 1.0),
            (0.82773107290267944, 1.0, 1.0),
            (0.83193278312683105, 1.0, 1.0),
            (0.83613443374633789, 1.0, 1.0),
            (0.8403361439704895, 1.0, 1.0),
            (0.84453779458999634, 1.0, 1.0),
            (0.84873950481414795, 1.0, 1.0),
            (0.85294115543365479, 1.0, 1.0),
            (0.8571428656578064, 1.0, 1.0),
            (0.86134451627731323, 1.0, 1.0),
            (0.86554622650146484, 1.0, 1.0),
            (0.86974787712097168, 1.0, 1.0),
            (0.87394958734512329, 1.0, 1.0),
            (0.87815123796463013, 1.0, 1.0),
            (0.88235294818878174, 1.0, 1.0),
            (0.88655459880828857, 1.0, 1.0),
            (0.89075630903244019, 1.0, 1.0),
            (0.89495795965194702, 1.0, 1.0),
            (0.89915966987609863, 1.0, 1.0),
            (0.90336132049560547, 1.0, 1.0),
            (0.90756303071975708, 1.0, 1.0),
            (0.91176468133926392, 1.0, 1.0),
            (0.91596639156341553, 1.0, 1.0),
            (0.92016804218292236, 1.0, 1.0),
            (0.92436975240707397, 1.0, 1.0),
            (0.92857140302658081, 1.0, 1.0),
            (0.93277311325073242, 1.0, 1.0),
            (0.93697476387023926, 1.0, 1.0),
            (0.94117647409439087, 1.0, 1.0),
            (0.94537812471389771, 1.0, 1.0),
            (0.94957983493804932, 1.0, 1.0),
            (0.95378148555755615, 1.0, 1.0),
            (0.95798319578170776, 1.0, 1.0),
            (0.9621848464012146, 1.0, 1.0),
            (0.96638655662536621, 1.0, 1.0),
            (0.97058820724487305, 1.0, 1.0),
            (0.97478991746902466, 1.0, 1.0),
            (0.97899156808853149, 1.0, 1.0),
            (0.98319327831268311, 1.0, 1.0),
            (0.98739492893218994, 1.0, 1.0),
            (0.99159663915634155, 1.0, 1.0),
            (0.99579828977584839, 1.0, 1.0),
            (1.0, 1.0, 1.0),
        ],
        green=[
            (0.0, 0.0, 0.0),
            (0.0042016808874905109, 0.0, 0.0),
            (0.0084033617749810219, 0.0, 0.0),
            (0.012605042196810246, 0.0, 0.0),
            (0.016806723549962044, 0.0, 0.0),
            (0.021008403971791267, 0.0, 0.0),
            (0.025210084393620491, 0.0, 0.0),
            (0.029411764815449715, 0.0, 0.0),
            (0.033613447099924088, 0.0, 0.0),
            (0.037815127521753311, 0.0, 0.0),
            (0.042016807943582535, 0.0, 0.0),
            (0.046218488365411758, 0.0, 0.0),
            (0.050420168787240982, 0.0, 0.0),
            (0.054621849209070206, 0.0, 0.0),
            (0.058823529630899429, 0.0, 0.0),
            (0.063025213778018951, 0.0, 0.0),
            (0.067226894199848175, 0.0, 0.0),
            (0.071428574621677399, 0.0, 0.0),
            (0.075630255043506622, 0.0, 0.0),
            (0.079831935465335846, 0.0, 0.0),
            (0.08403361588716507, 0.0, 0.0),
            (0.088235296308994293, 0.0, 0.0),
            (0.092436976730823517, 0.0, 0.0),
            (0.09663865715265274, 0.0, 0.0),
            (0.10084033757448196, 0.0, 0.0),
            (0.10504201799631119, 0.0, 0.0),
            (0.10924369841814041, 0.0, 0.0),
            (0.11344537883996964, 0.0, 0.0),
            (0.11764705926179886, 0.0, 0.0),
            (0.12184873968362808, 0.0, 0.0),
            (0.1260504275560379, 0.0, 0.0),
            (0.13025210797786713, 0.0, 0.0),
            (0.13445378839969635, 0.0, 0.0),
            (0.13865546882152557, 0.0, 0.0),
            (0.1428571492433548, 0.0, 0.0),
            (0.14705882966518402, 0.0, 0.0),
            (0.15126051008701324, 0.0, 0.0),
            (0.15546219050884247, 0.0, 0.0),
            (0.15966387093067169, 0.0, 0.0),
            (0.16386555135250092, 0.0, 0.0),
            (0.16806723177433014, 0.0, 0.0),
            (0.17226891219615936, 0.0, 0.0),
            (0.17647059261798859, 0.0, 0.0),
            (0.18067227303981781, 0.0, 0.0),
            (0.18487395346164703, 0.0, 0.0),
            (0.18907563388347626, 0.0, 0.0),
            (0.19327731430530548, 0.0, 0.0),
            (0.1974789947271347, 0.0, 0.0),
            (0.20168067514896393, 0.0, 0.0),
            (0.20588235557079315, 0.0, 0.0),
            (0.21008403599262238, 0.0, 0.0),
            (0.2142857164144516, 0.0, 0.0),
            (0.21848739683628082, 0.0, 0.0),
            (0.22268907725811005, 0.0, 0.0),
            (0.22689075767993927, 0.0, 0.0),
            (0.23109243810176849, 0.0, 0.0),
            (0.23529411852359772, 0.0, 0.0),
            (0.23949579894542694, 0.0, 0.0),
            (0.24369747936725616, 0.0, 0.0),
            (0.24789915978908539, 0.0, 0.0),
            (0.25210085511207581, 0.0, 0.0),
            (0.25630253553390503, 0.0, 0.0),
            (0.26050421595573425, 0.0, 0.0),
            (0.26470589637756348, 0.0, 0.0),
            (0.2689075767993927, 0.0, 0.0),
            (0.27310925722122192, 0.0, 0.0),
            (0.27731093764305115, 0.0, 0.0),
            (0.28151261806488037, 0.0, 0.0),
            (0.28571429848670959, 0.0, 0.0),
            (0.28991597890853882, 0.0, 0.0),
            (0.29411765933036804, 0.0, 0.0),
            (0.29831933975219727, 0.0, 0.0),
            (0.30252102017402649, 0.0, 0.0),
            (0.30672270059585571, 0.0, 0.0),
            (0.31092438101768494, 0.0, 0.0),
            (0.31512606143951416, 0.0, 0.0),
            (0.31932774186134338, 0.0, 0.0),
            (0.32352942228317261, 0.0, 0.0),
            (0.32773110270500183, 0.0, 0.0),
            (0.33193278312683105, 0.0, 0.0),
            (0.33613446354866028, 0.0, 0.0),
            (0.3403361439704895, 0.0, 0.0),
            (0.34453782439231873, 0.0, 0.0),
            (0.34873950481414795, 0.0, 0.0),
            (0.35294118523597717, 0.0, 0.0),
            (0.3571428656578064, 0.0, 0.0),
            (0.36134454607963562, 0.0, 0.0),
            (0.36554622650146484, 0.0, 0.0),
            (0.36974790692329407, 0.0, 0.0),
            (0.37394958734512329, 0.0, 0.0),
            (0.37815126776695251, 0.0, 0.0),
            (0.38235294818878174, 0.0, 0.0),
            (0.38655462861061096, 0.0, 0.0),
            (0.39075630903244019, 0.0, 0.0),
            (0.39495798945426941, 0.0, 0.0),
            (0.39915966987609863, 0.0, 0.0),
            (0.40336135029792786, 0.0, 0.0),
            (0.40756303071975708, 0.0, 0.0),
            (0.4117647111415863, 0.0, 0.0),
            (0.41596639156341553, 0.0, 0.0),
            (0.42016807198524475, 0.0, 0.0),
            (0.42436975240707397, 0.0, 0.0),
            (0.4285714328289032, 0.0, 0.0),
            (0.43277311325073242, 0.0, 0.0),
            (0.43697479367256165, 0.0, 0.0),
            (0.44117647409439087, 0.0, 0.0),
            (0.44537815451622009, 0.0, 0.0),
            (0.44957983493804932, 0.0, 0.0),
            (0.45378151535987854, 0.0, 0.0),
            (0.45798319578170776, 0.0, 0.0),
            (0.46218487620353699, 0.0, 0.0),
            (0.46638655662536621, 0.0, 0.0),
            (0.47058823704719543, 0.0, 0.0),
            (0.47478991746902466, 0.0, 0.0),
            (
                0.47899159789085388,
                0.0039215688593685627,
                0.0039215688593685627,
            ),
            (0.48319327831268311, 0.011764706112444401, 0.011764706112444401),
            (0.48739495873451233, 0.019607843831181526, 0.019607843831181526),
            (0.49159663915634155, 0.027450980618596077, 0.027450980618596077),
            (0.49579831957817078, 0.035294119268655777, 0.035294119268655777),
            (0.5, 0.043137256056070328, 0.043137256056070328),
            (0.50420171022415161, 0.058823529630899429, 0.058823529630899429),
            (0.50840336084365845, 0.066666670143604279, 0.066666670143604279),
            (0.51260507106781006, 0.070588238537311554, 0.070588238537311554),
            (0.51680672168731689, 0.078431375324726105, 0.078431375324726105),
            (0.52100843191146851, 0.086274512112140656, 0.086274512112140656),
            (0.52521008253097534, 0.094117648899555206, 0.094117648899555206),
            (0.52941179275512695, 0.10196078568696976, 0.10196078568696976),
            (0.53361344337463379, 0.10980392247438431, 0.10980392247438431),
            (0.5378151535987854, 0.11764705926179886, 0.11764705926179886),
            (0.54201680421829224, 0.12549020349979401, 0.12549020349979401),
            (0.54621851444244385, 0.13725490868091583, 0.13725490868091583),
            (0.55042016506195068, 0.14509804546833038, 0.14509804546833038),
            (0.55462187528610229, 0.15294118225574493, 0.15294118225574493),
            (0.55882352590560913, 0.16078431904315948, 0.16078431904315948),
            (0.56302523612976074, 0.16862745583057404, 0.16862745583057404),
            (0.56722688674926758, 0.17647059261798859, 0.17647059261798859),
            (0.57142859697341919, 0.18431372940540314, 0.18431372940540314),
            (0.57563024759292603, 0.19215686619281769, 0.19215686619281769),
            (0.57983195781707764, 0.20000000298023224, 0.20000000298023224),
            (0.58403360843658447, 0.20392157137393951, 0.20392157137393951),
            (0.58823531866073608, 0.21176470816135406, 0.21176470816135406),
            (0.59243696928024292, 0.21960784494876862, 0.21960784494876862),
            (0.59663867950439453, 0.22745098173618317, 0.22745098173618317),
            (0.60084033012390137, 0.23529411852359772, 0.23529411852359772),
            (0.60504204034805298, 0.24313725531101227, 0.24313725531101227),
            (0.60924369096755981, 0.25098040699958801, 0.25098040699958801),
            (0.61344540119171143, 0.25882354378700256, 0.25882354378700256),
            (0.61764705181121826, 0.26666668057441711, 0.26666668057441711),
            (0.62184876203536987, 0.27058824896812439, 0.27058824896812439),
            (0.62605041265487671, 0.27843138575553894, 0.27843138575553894),
            (0.63025212287902832, 0.29411765933036804, 0.29411765933036804),
            (0.63445377349853516, 0.30196079611778259, 0.30196079611778259),
            (0.63865548372268677, 0.30980393290519714, 0.30980393290519714),
            (0.6428571343421936, 0.31764706969261169, 0.31764706969261169),
            (0.64705884456634521, 0.32549020648002625, 0.32549020648002625),
            (0.65126049518585205, 0.3333333432674408, 0.3333333432674408),
            (0.65546220541000366, 0.33725491166114807, 0.33725491166114807),
            (0.6596638560295105, 0.34509804844856262, 0.34509804844856262),
            (0.66386556625366211, 0.35294118523597717, 0.35294118523597717),
            (0.66806721687316895, 0.36078432202339172, 0.36078432202339172),
            (0.67226892709732056, 0.36862745881080627, 0.36862745881080627),
            (0.67647057771682739, 0.37647059559822083, 0.37647059559822083),
            (0.680672287940979, 0.38431373238563538, 0.38431373238563538),
            (0.68487393856048584, 0.39215686917304993, 0.39215686917304993),
            (0.68907564878463745, 0.40000000596046448, 0.40000000596046448),
            (0.69327729940414429, 0.40392157435417175, 0.40392157435417175),
            (0.6974790096282959, 0.4117647111415863, 0.4117647111415863),
            (0.70168066024780273, 0.41960784792900085, 0.41960784792900085),
            (0.70588237047195435, 0.42745098471641541, 0.42745098471641541),
            (0.71008402109146118, 0.43529412150382996, 0.43529412150382996),
            (0.71428573131561279, 0.45098039507865906, 0.45098039507865906),
            (0.71848738193511963, 0.45882353186607361, 0.45882353186607361),
            (0.72268909215927124, 0.46666666865348816, 0.46666666865348816),
            (0.72689074277877808, 0.47058823704719543, 0.47058823704719543),
            (0.73109245300292969, 0.47843137383460999, 0.47843137383460999),
            (0.73529410362243652, 0.48627451062202454, 0.48627451062202454),
            (0.73949581384658813, 0.49411764740943909, 0.49411764740943909),
            (0.74369746446609497, 0.50196081399917603, 0.50196081399917603),
            (0.74789917469024658, 0.50980395078659058, 0.50980395078659058),
            (0.75210082530975342, 0.51764708757400513, 0.51764708757400513),
            (0.75630253553390503, 0.53333336114883423, 0.53333336114883423),
            (0.76050418615341187, 0.5372549295425415, 0.5372549295425415),
            (0.76470589637756348, 0.54509806632995605, 0.54509806632995605),
            (0.76890754699707031, 0.55294120311737061, 0.55294120311737061),
            (0.77310925722122192, 0.56078433990478516, 0.56078433990478516),
            (0.77731090784072876, 0.56862747669219971, 0.56862747669219971),
            (0.78151261806488037, 0.57647061347961426, 0.57647061347961426),
            (0.78571426868438721, 0.58431375026702881, 0.58431375026702881),
            (0.78991597890853882, 0.59215688705444336, 0.59215688705444336),
            (0.79411762952804565, 0.60000002384185791, 0.60000002384185791),
            (0.79831933975219727, 0.61176472902297974, 0.61176472902297974),
            (0.8025209903717041, 0.61960786581039429, 0.61960786581039429),
            (0.80672270059585571, 0.62745100259780884, 0.62745100259780884),
            (0.81092435121536255, 0.63529413938522339, 0.63529413938522339),
            (0.81512606143951416, 0.64313727617263794, 0.64313727617263794),
            (0.819327712059021, 0.65098041296005249, 0.65098041296005249),
            (0.82352942228317261, 0.65882354974746704, 0.65882354974746704),
            (0.82773107290267944, 0.66666668653488159, 0.66666668653488159),
            (0.83193278312683105, 0.67058825492858887, 0.67058825492858887),
            (0.83613443374633789, 0.67843139171600342, 0.67843139171600342),
            (0.8403361439704895, 0.68627452850341797, 0.68627452850341797),
            (0.84453779458999634, 0.69411766529083252, 0.69411766529083252),
            (0.84873950481414795, 0.70196080207824707, 0.70196080207824707),
            (0.85294115543365479, 0.70980393886566162, 0.70980393886566162),
            (0.8571428656578064, 0.71764707565307617, 0.71764707565307617),
            (0.86134451627731323, 0.72549021244049072, 0.72549021244049072),
            (0.86554622650146484, 0.73333334922790527, 0.73333334922790527),
            (0.86974787712097168, 0.73725491762161255, 0.73725491762161255),
            (0.87394958734512329, 0.7450980544090271, 0.7450980544090271),
            (0.87815123796463013, 0.75294119119644165, 0.75294119119644165),
            (0.88235294818878174, 0.76862746477127075, 0.76862746477127075),
            (0.88655459880828857, 0.7764706015586853, 0.7764706015586853),
            (0.89075630903244019, 0.78431373834609985, 0.78431373834609985),
            (0.89495795965194702, 0.7921568751335144, 0.7921568751335144),
            (0.89915966987609863, 0.80000001192092896, 0.80000001192092896),
            (0.90336132049560547, 0.80392158031463623, 0.80392158031463623),
            (0.90756303071975708, 0.81176471710205078, 0.81176471710205078),
            (0.91176468133926392, 0.81960785388946533, 0.81960785388946533),
            (0.91596639156341553, 0.82745099067687988, 0.82745099067687988),
            (0.92016804218292236, 0.83529412746429443, 0.83529412746429443),
            (0.92436975240707397, 0.84313726425170898, 0.84313726425170898),
            (0.92857140302658081, 0.85098040103912354, 0.85098040103912354),
            (0.93277311325073242, 0.85882353782653809, 0.85882353782653809),
            (0.93697476387023926, 0.86666667461395264, 0.86666667461395264),
            (0.94117647409439087, 0.87058824300765991, 0.87058824300765991),
            (0.94537812471389771, 0.87843137979507446, 0.87843137979507446),
            (0.94957983493804932, 0.88627451658248901, 0.88627451658248901),
            (0.95378148555755615, 0.89411765336990356, 0.89411765336990356),
            (0.95798319578170776, 0.90196079015731812, 0.90196079015731812),
            (0.9621848464012146, 0.90980392694473267, 0.90980392694473267),
            (0.96638655662536621, 0.92549020051956177, 0.92549020051956177),
            (0.97058820724487305, 0.93333333730697632, 0.93333333730697632),
            (0.97478991746902466, 0.93725490570068359, 0.93725490570068359),
            (0.97899156808853149, 0.94509804248809814, 0.94509804248809814),
            (0.98319327831268311, 0.9529411792755127, 0.9529411792755127),
            (0.98739492893218994, 0.96078431606292725, 0.96078431606292725),
            (0.99159663915634155, 0.9686274528503418, 0.9686274528503418),
            (0.99579828977584839, 0.97647058963775635, 0.97647058963775635),
            (1.0, 0.9843137264251709, 0.9843137264251709),
        ],
        blue=[
            (0.0, 0.0, 0.0),
            (0.0042016808874905109, 0.0, 0.0),
            (0.0084033617749810219, 0.0, 0.0),
            (0.012605042196810246, 0.0, 0.0),
            (0.016806723549962044, 0.0, 0.0),
            (0.021008403971791267, 0.0, 0.0),
            (0.025210084393620491, 0.0, 0.0),
            (0.029411764815449715, 0.0, 0.0),
            (0.033613447099924088, 0.0, 0.0),
            (0.037815127521753311, 0.0, 0.0),
            (0.042016807943582535, 0.0, 0.0),
            (0.046218488365411758, 0.0, 0.0),
            (0.050420168787240982, 0.0, 0.0),
            (0.054621849209070206, 0.0, 0.0),
            (0.058823529630899429, 0.0, 0.0),
            (0.063025213778018951, 0.0, 0.0),
            (0.067226894199848175, 0.0, 0.0),
            (0.071428574621677399, 0.0, 0.0),
            (0.075630255043506622, 0.0, 0.0),
            (0.079831935465335846, 0.0, 0.0),
            (0.08403361588716507, 0.0, 0.0),
            (0.088235296308994293, 0.0, 0.0),
            (0.092436976730823517, 0.0, 0.0),
            (0.09663865715265274, 0.0, 0.0),
            (0.10084033757448196, 0.0, 0.0),
            (0.10504201799631119, 0.0, 0.0),
            (0.10924369841814041, 0.0, 0.0),
            (0.11344537883996964, 0.0, 0.0),
            (0.11764705926179886, 0.0, 0.0),
            (0.12184873968362808, 0.0, 0.0),
            (0.1260504275560379, 0.0, 0.0),
            (0.13025210797786713, 0.0, 0.0),
            (0.13445378839969635, 0.0, 0.0),
            (0.13865546882152557, 0.0, 0.0),
            (0.1428571492433548, 0.0, 0.0),
            (0.14705882966518402, 0.0, 0.0),
            (0.15126051008701324, 0.0, 0.0),
            (0.15546219050884247, 0.0, 0.0),
            (0.15966387093067169, 0.0, 0.0),
            (0.16386555135250092, 0.0, 0.0),
            (0.16806723177433014, 0.0, 0.0),
            (0.17226891219615936, 0.0, 0.0),
            (0.17647059261798859, 0.0, 0.0),
            (0.18067227303981781, 0.0, 0.0),
            (0.18487395346164703, 0.0, 0.0),
            (0.18907563388347626, 0.0, 0.0),
            (0.19327731430530548, 0.0, 0.0),
            (0.1974789947271347, 0.0, 0.0),
            (0.20168067514896393, 0.0, 0.0),
            (0.20588235557079315, 0.0, 0.0),
            (0.21008403599262238, 0.0, 0.0),
            (0.2142857164144516, 0.0, 0.0),
            (0.21848739683628082, 0.0, 0.0),
            (0.22268907725811005, 0.0, 0.0),
            (0.22689075767993927, 0.0, 0.0),
            (0.23109243810176849, 0.0, 0.0),
            (0.23529411852359772, 0.0, 0.0),
            (0.23949579894542694, 0.0, 0.0),
            (0.24369747936725616, 0.0, 0.0),
            (0.24789915978908539, 0.0, 0.0),
            (0.25210085511207581, 0.0, 0.0),
            (0.25630253553390503, 0.0, 0.0),
            (0.26050421595573425, 0.0, 0.0),
            (0.26470589637756348, 0.0, 0.0),
            (0.2689075767993927, 0.0, 0.0),
            (0.27310925722122192, 0.0, 0.0),
            (0.27731093764305115, 0.0, 0.0),
            (0.28151261806488037, 0.0, 0.0),
            (0.28571429848670959, 0.0, 0.0),
            (0.28991597890853882, 0.0, 0.0),
            (0.29411765933036804, 0.0, 0.0),
            (0.29831933975219727, 0.0, 0.0),
            (0.30252102017402649, 0.0, 0.0),
            (0.30672270059585571, 0.0, 0.0),
            (0.31092438101768494, 0.0, 0.0),
            (0.31512606143951416, 0.0, 0.0),
            (0.31932774186134338, 0.0, 0.0),
            (0.32352942228317261, 0.0, 0.0),
            (0.32773110270500183, 0.0, 0.0),
            (0.33193278312683105, 0.0, 0.0),
            (0.33613446354866028, 0.0, 0.0),
            (0.3403361439704895, 0.0, 0.0),
            (0.34453782439231873, 0.0, 0.0),
            (0.34873950481414795, 0.0, 0.0),
            (0.35294118523597717, 0.0, 0.0),
            (0.3571428656578064, 0.0, 0.0),
            (0.36134454607963562, 0.0, 0.0),
            (0.36554622650146484, 0.0, 0.0),
            (0.36974790692329407, 0.0, 0.0),
            (0.37394958734512329, 0.0, 0.0),
            (0.37815126776695251, 0.0, 0.0),
            (0.38235294818878174, 0.0, 0.0),
            (0.38655462861061096, 0.0, 0.0),
            (0.39075630903244019, 0.0, 0.0),
            (0.39495798945426941, 0.0, 0.0),
            (0.39915966987609863, 0.0, 0.0),
            (0.40336135029792786, 0.0, 0.0),
            (0.40756303071975708, 0.0, 0.0),
            (0.4117647111415863, 0.0, 0.0),
            (0.41596639156341553, 0.0, 0.0),
            (0.42016807198524475, 0.0, 0.0),
            (0.42436975240707397, 0.0, 0.0),
            (0.4285714328289032, 0.0, 0.0),
            (0.43277311325073242, 0.0, 0.0),
            (0.43697479367256165, 0.0, 0.0),
            (0.44117647409439087, 0.0, 0.0),
            (0.44537815451622009, 0.0, 0.0),
            (0.44957983493804932, 0.0, 0.0),
            (0.45378151535987854, 0.0, 0.0),
            (0.45798319578170776, 0.0, 0.0),
            (0.46218487620353699, 0.0, 0.0),
            (0.46638655662536621, 0.0, 0.0),
            (0.47058823704719543, 0.0, 0.0),
            (0.47478991746902466, 0.0, 0.0),
            (0.47899159789085388, 0.0, 0.0),
            (0.48319327831268311, 0.0, 0.0),
            (0.48739495873451233, 0.0, 0.0),
            (0.49159663915634155, 0.0, 0.0),
            (0.49579831957817078, 0.0, 0.0),
            (0.5, 0.0, 0.0),
            (0.50420171022415161, 0.0, 0.0),
            (0.50840336084365845, 0.0, 0.0),
            (0.51260507106781006, 0.0, 0.0),
            (0.51680672168731689, 0.0, 0.0),
            (0.52100843191146851, 0.0, 0.0),
            (0.52521008253097534, 0.0, 0.0),
            (0.52941179275512695, 0.0, 0.0),
            (0.53361344337463379, 0.0, 0.0),
            (0.5378151535987854, 0.0, 0.0),
            (0.54201680421829224, 0.0, 0.0),
            (0.54621851444244385, 0.0, 0.0),
            (0.55042016506195068, 0.0, 0.0),
            (0.55462187528610229, 0.0, 0.0),
            (0.55882352590560913, 0.0, 0.0),
            (0.56302523612976074, 0.0, 0.0),
            (0.56722688674926758, 0.0, 0.0),
            (0.57142859697341919, 0.0, 0.0),
            (0.57563024759292603, 0.0, 0.0),
            (0.57983195781707764, 0.0, 0.0),
            (0.58403360843658447, 0.0, 0.0),
            (0.58823531866073608, 0.0, 0.0),
            (0.59243696928024292, 0.0, 0.0),
            (0.59663867950439453, 0.0, 0.0),
            (0.60084033012390137, 0.0, 0.0),
            (0.60504204034805298, 0.0, 0.0),
            (0.60924369096755981, 0.0, 0.0),
            (0.61344540119171143, 0.0, 0.0),
            (0.61764705181121826, 0.0, 0.0),
            (0.62184876203536987, 0.0, 0.0),
            (0.62605041265487671, 0.0, 0.0),
            (0.63025212287902832, 0.0, 0.0),
            (0.63445377349853516, 0.0, 0.0),
            (0.63865548372268677, 0.0, 0.0),
            (0.6428571343421936, 0.0, 0.0),
            (0.64705884456634521, 0.0, 0.0),
            (0.65126049518585205, 0.0, 0.0),
            (0.65546220541000366, 0.0, 0.0),
            (0.6596638560295105, 0.0, 0.0),
            (0.66386556625366211, 0.0, 0.0),
            (0.66806721687316895, 0.0, 0.0),
            (0.67226892709732056, 0.0, 0.0),
            (0.67647057771682739, 0.0, 0.0),
            (0.680672287940979, 0.0, 0.0),
            (0.68487393856048584, 0.0, 0.0),
            (0.68907564878463745, 0.0, 0.0),
            (0.69327729940414429, 0.0, 0.0),
            (0.6974790096282959, 0.0, 0.0),
            (0.70168066024780273, 0.0, 0.0),
            (0.70588237047195435, 0.0, 0.0),
            (0.71008402109146118, 0.0, 0.0),
            (0.71428573131561279, 0.0, 0.0),
            (0.71848738193511963, 0.0, 0.0),
            (0.72268909215927124, 0.0, 0.0),
            (0.72689074277877808, 0.0, 0.0),
            (0.73109245300292969, 0.0, 0.0),
            (0.73529410362243652, 0.0, 0.0),
            (0.73949581384658813, 0.0, 0.0),
            (0.74369746446609497, 0.0, 0.0),
            (0.74789917469024658, 0.0, 0.0),
            (0.75210082530975342, 0.0, 0.0),
            (0.75630253553390503, 0.027450980618596077, 0.027450980618596077),
            (0.76050418615341187, 0.043137256056070328, 0.043137256056070328),
            (0.76470589637756348, 0.058823529630899429, 0.058823529630899429),
            (0.76890754699707031, 0.074509806931018829, 0.074509806931018829),
            (0.77310925722122192, 0.090196080505847931, 0.090196080505847931),
            (0.77731090784072876, 0.10588235408067703, 0.10588235408067703),
            (0.78151261806488037, 0.12156862765550613, 0.12156862765550613),
            (0.78571426868438721, 0.13725490868091583, 0.13725490868091583),
            (0.78991597890853882, 0.15294118225574493, 0.15294118225574493),
            (0.79411762952804565, 0.16862745583057404, 0.16862745583057404),
            (0.79831933975219727, 0.20000000298023224, 0.20000000298023224),
            (0.8025209903717041, 0.21176470816135406, 0.21176470816135406),
            (0.80672270059585571, 0.22745098173618317, 0.22745098173618317),
            (0.81092435121536255, 0.24313725531101227, 0.24313725531101227),
            (0.81512606143951416, 0.25882354378700256, 0.25882354378700256),
            (0.819327712059021, 0.27450981736183167, 0.27450981736183167),
            (0.82352942228317261, 0.29019609093666077, 0.29019609093666077),
            (0.82773107290267944, 0.30588236451148987, 0.30588236451148987),
            (0.83193278312683105, 0.32156863808631897, 0.32156863808631897),
            (0.83613443374633789, 0.33725491166114807, 0.33725491166114807),
            (0.8403361439704895, 0.35294118523597717, 0.35294118523597717),
            (0.84453779458999634, 0.36862745881080627, 0.36862745881080627),
            (0.84873950481414795, 0.38431373238563538, 0.38431373238563538),
            (0.85294115543365479, 0.40000000596046448, 0.40000000596046448),
            (0.8571428656578064, 0.4117647111415863, 0.4117647111415863),
            (0.86134451627731323, 0.42745098471641541, 0.42745098471641541),
            (0.86554622650146484, 0.44313725829124451, 0.44313725829124451),
            (0.86974787712097168, 0.45882353186607361, 0.45882353186607361),
            (0.87394958734512329, 0.47450980544090271, 0.47450980544090271),
            (0.87815123796463013, 0.49019607901573181, 0.49019607901573181),
            (0.88235294818878174, 0.5215686559677124, 0.5215686559677124),
            (0.88655459880828857, 0.5372549295425415, 0.5372549295425415),
            (0.89075630903244019, 0.55294120311737061, 0.55294120311737061),
            (0.89495795965194702, 0.56862747669219971, 0.56862747669219971),
            (0.89915966987609863, 0.58431375026702881, 0.58431375026702881),
            (0.90336132049560547, 0.60000002384185791, 0.60000002384185791),
            (0.90756303071975708, 0.61176472902297974, 0.61176472902297974),
            (0.91176468133926392, 0.62745100259780884, 0.62745100259780884),
            (0.91596639156341553, 0.64313727617263794, 0.64313727617263794),
            (0.92016804218292236, 0.65882354974746704, 0.65882354974746704),
            (0.92436975240707397, 0.67450982332229614, 0.67450982332229614),
            (0.92857140302658081, 0.69019609689712524, 0.69019609689712524),
            (0.93277311325073242, 0.70588237047195435, 0.70588237047195435),
            (0.93697476387023926, 0.72156864404678345, 0.72156864404678345),
            (0.94117647409439087, 0.73725491762161255, 0.73725491762161255),
            (0.94537812471389771, 0.75294119119644165, 0.75294119119644165),
            (0.94957983493804932, 0.76862746477127075, 0.76862746477127075),
            (0.95378148555755615, 0.78431373834609985, 0.78431373834609985),
            (0.95798319578170776, 0.80000001192092896, 0.80000001192092896),
            (0.9621848464012146, 0.81176471710205078, 0.81176471710205078),
            (0.96638655662536621, 0.84313726425170898, 0.84313726425170898),
            (0.97058820724487305, 0.85882353782653809, 0.85882353782653809),
            (0.97478991746902466, 0.87450981140136719, 0.87450981140136719),
            (0.97899156808853149, 0.89019608497619629, 0.89019608497619629),
            (0.98319327831268311, 0.90588235855102539, 0.90588235855102539),
            (0.98739492893218994, 0.92156863212585449, 0.92156863212585449),
            (0.99159663915634155, 0.93725490570068359, 0.93725490570068359),
            (0.99579828977584839, 0.9529411792755127, 0.9529411792755127),
            (1.0, 0.9686274528503418, 0.9686274528503418),
        ],
    )
    return ColorMapper.from_segment_map(_data, range=range, **traits)


def gist_ncar(range, **traits):
    """Generator for the 'gist_ncar' colormap from GIST."""
    _data = dict(
        red=[
            (0.0, 0.0, 0.0),
            (0.0050505050458014011, 0.0, 0.0),
            (0.010101010091602802, 0.0, 0.0),
            (0.015151515603065491, 0.0, 0.0),
            (0.020202020183205605, 0.0, 0.0),
            (0.025252524763345718, 0.0, 0.0),
            (0.030303031206130981, 0.0, 0.0),
            (0.035353533923625946, 0.0, 0.0),
            (0.040404040366411209, 0.0, 0.0),
            (0.045454546809196472, 0.0, 0.0),
            (0.050505049526691437, 0.0, 0.0),
            (0.0555555559694767, 0.0, 0.0),
            (0.060606062412261963, 0.0, 0.0),
            (0.065656565129756927, 0.0, 0.0),
            (0.070707067847251892, 0.0, 0.0),
            (0.075757578015327454, 0.0, 0.0),
            (0.080808080732822418, 0.0, 0.0),
            (0.085858583450317383, 0.0, 0.0),
            (0.090909093618392944, 0.0, 0.0),
            (0.095959596335887909, 0.0, 0.0),
            (0.10101009905338287, 0.0, 0.0),
            (0.10606060922145844, 0.0, 0.0),
            (0.1111111119389534, 0.0, 0.0),
            (0.11616161465644836, 0.0, 0.0),
            (0.12121212482452393, 0.0, 0.0),
            (0.12626262009143829, 0.0, 0.0),
            (0.13131313025951385, 0.0, 0.0),
            (0.13636364042758942, 0.0, 0.0),
            (0.14141413569450378, 0.0, 0.0),
            (0.14646464586257935, 0.0, 0.0),
            (0.15151515603065491, 0.0, 0.0),
            (0.15656565129756927, 0.0, 0.0),
            (0.16161616146564484, 0.0, 0.0),
            (0.1666666716337204, 0.0, 0.0),
            (0.17171716690063477, 0.0, 0.0),
            (0.17676767706871033, 0.0, 0.0),
            (0.18181818723678589, 0.0, 0.0),
            (0.18686868250370026, 0.0, 0.0),
            (0.19191919267177582, 0.0, 0.0),
            (0.19696970283985138, 0.0, 0.0),
            (0.20202019810676575, 0.0, 0.0),
            (0.20707070827484131, 0.0, 0.0),
            (0.21212121844291687, 0.0, 0.0),
            (0.21717171370983124, 0.0, 0.0),
            (0.2222222238779068, 0.0, 0.0),
            (0.22727273404598236, 0.0, 0.0),
            (0.23232322931289673, 0.0, 0.0),
            (0.23737373948097229, 0.0, 0.0),
            (0.24242424964904785, 0.0, 0.0),
            (0.24747474491596222, 0.0, 0.0),
            (0.25252524018287659, 0.0, 0.0),
            (0.25757575035095215, 0.0, 0.0),
            (0.26262626051902771, 0.0, 0.0),
            (0.26767677068710327, 0.0, 0.0),
            (0.27272728085517883, 0.0, 0.0),
            (0.27777779102325439, 0.0, 0.0),
            (0.28282827138900757, 0.0, 0.0),
            (0.28787878155708313, 0.0, 0.0),
            (0.29292929172515869, 0.0, 0.0),
            (0.29797980189323425, 0.0, 0.0),
            (0.30303031206130981, 0.0, 0.0),
            (0.30808082222938538, 0.0, 0.0),
            (0.31313130259513855, 0.0, 0.0),
            (
                0.31818181276321411,
                0.0039215688593685627,
                0.0039215688593685627,
            ),
            (0.32323232293128967, 0.043137256056070328, 0.043137256056070328),
            (0.32828283309936523, 0.08235294371843338, 0.08235294371843338),
            (0.3333333432674408, 0.11764705926179886, 0.11764705926179886),
            (0.33838382363319397, 0.15686275064945221, 0.15686275064945221),
            (0.34343433380126953, 0.19607843458652496, 0.19607843458652496),
            (0.34848484396934509, 0.23137255012989044, 0.23137255012989044),
            (0.35353535413742065, 0.27058824896812439, 0.27058824896812439),
            (0.35858586430549622, 0.30980393290519714, 0.30980393290519714),
            (0.36363637447357178, 0.3490196168422699, 0.3490196168422699),
            (0.36868685483932495, 0.38431373238563538, 0.38431373238563538),
            (0.37373736500740051, 0.40392157435417175, 0.40392157435417175),
            (0.37878787517547607, 0.41568627953529358, 0.41568627953529358),
            (0.38383838534355164, 0.42352941632270813, 0.42352941632270813),
            (0.3888888955116272, 0.43137255311012268, 0.43137255311012268),
            (0.39393940567970276, 0.44313725829124451, 0.44313725829124451),
            (0.39898988604545593, 0.45098039507865906, 0.45098039507865906),
            (0.40404039621353149, 0.45882353186607361, 0.45882353186607361),
            (0.40909090638160706, 0.47058823704719543, 0.47058823704719543),
            (0.41414141654968262, 0.47843137383460999, 0.47843137383460999),
            (0.41919192671775818, 0.49019607901573181, 0.49019607901573181),
            (0.42424243688583374, 0.50196081399917603, 0.50196081399917603),
            (0.42929291725158691, 0.52549022436141968, 0.52549022436141968),
            (0.43434342741966248, 0.54901963472366333, 0.54901963472366333),
            (0.43939393758773804, 0.57254904508590698, 0.57254904508590698),
            (0.4444444477558136, 0.60000002384185791, 0.60000002384185791),
            (0.44949495792388916, 0.62352943420410156, 0.62352943420410156),
            (0.45454546809196472, 0.64705884456634521, 0.64705884456634521),
            (0.4595959484577179, 0.67058825492858887, 0.67058825492858887),
            (0.46464645862579346, 0.69411766529083252, 0.69411766529083252),
            (0.46969696879386902, 0.72156864404678345, 0.72156864404678345),
            (0.47474747896194458, 0.7450980544090271, 0.7450980544090271),
            (0.47979798913002014, 0.76862746477127075, 0.76862746477127075),
            (0.4848484992980957, 0.7921568751335144, 0.7921568751335144),
            (0.48989897966384888, 0.81568628549575806, 0.81568628549575806),
            (0.49494948983192444, 0.83921569585800171, 0.83921569585800171),
            (0.5, 0.86274510622024536, 0.86274510622024536),
            (0.50505048036575317, 0.88627451658248901, 0.88627451658248901),
            (0.51010102033615112, 0.90980392694473267, 0.90980392694473267),
            (0.5151515007019043, 0.93333333730697632, 0.93333333730697632),
            (0.52020204067230225, 0.95686274766921997, 0.95686274766921997),
            (0.52525252103805542, 0.98039215803146362, 0.98039215803146362),
            (0.53030300140380859, 1.0, 1.0),
            (0.53535354137420654, 1.0, 1.0),
            (0.54040402173995972, 1.0, 1.0),
            (0.54545456171035767, 1.0, 1.0),
            (0.55050504207611084, 1.0, 1.0),
            (0.55555558204650879, 1.0, 1.0),
            (0.56060606241226196, 1.0, 1.0),
            (0.56565654277801514, 1.0, 1.0),
            (0.57070708274841309, 1.0, 1.0),
            (0.57575756311416626, 1.0, 1.0),
            (0.58080810308456421, 1.0, 1.0),
            (0.58585858345031738, 1.0, 1.0),
            (0.59090906381607056, 1.0, 1.0),
            (0.59595960378646851, 1.0, 1.0),
            (0.60101008415222168, 1.0, 1.0),
            (0.60606062412261963, 1.0, 1.0),
            (0.6111111044883728, 1.0, 1.0),
            (0.61616164445877075, 1.0, 1.0),
            (0.62121212482452393, 1.0, 1.0),
            (0.6262626051902771, 1.0, 1.0),
            (0.63131314516067505, 1.0, 1.0),
            (0.63636362552642822, 1.0, 1.0),
            (0.64141416549682617, 1.0, 1.0),
            (0.64646464586257935, 1.0, 1.0),
            (0.65151512622833252, 1.0, 1.0),
            (0.65656566619873047, 1.0, 1.0),
            (0.66161614656448364, 1.0, 1.0),
            (0.66666668653488159, 1.0, 1.0),
            (0.67171716690063477, 1.0, 1.0),
            (0.67676764726638794, 1.0, 1.0),
            (0.68181818723678589, 1.0, 1.0),
            (0.68686866760253906, 1.0, 1.0),
            (0.69191920757293701, 1.0, 1.0),
            (0.69696968793869019, 1.0, 1.0),
            (0.70202022790908813, 1.0, 1.0),
            (0.70707070827484131, 1.0, 1.0),
            (0.71212118864059448, 1.0, 1.0),
            (0.71717172861099243, 1.0, 1.0),
            (0.72222220897674561, 1.0, 1.0),
            (0.72727274894714355, 1.0, 1.0),
            (0.73232322931289673, 1.0, 1.0),
            (0.7373737096786499, 1.0, 1.0),
            (0.74242424964904785, 1.0, 1.0),
            (0.74747473001480103, 1.0, 1.0),
            (0.75252526998519897, 1.0, 1.0),
            (0.75757575035095215, 1.0, 1.0),
            (0.7626262903213501, 1.0, 1.0),
            (0.76767677068710327, 1.0, 1.0),
            (0.77272725105285645, 1.0, 1.0),
            (0.77777779102325439, 1.0, 1.0),
            (0.78282827138900757, 1.0, 1.0),
            (0.78787881135940552, 1.0, 1.0),
            (0.79292929172515869, 1.0, 1.0),
            (0.79797977209091187, 0.96470588445663452, 0.96470588445663452),
            (0.80303031206130981, 0.92549020051956177, 0.92549020051956177),
            (0.80808079242706299, 0.89019608497619629, 0.89019608497619629),
            (0.81313133239746094, 0.85098040103912354, 0.85098040103912354),
            (0.81818181276321411, 0.81568628549575806, 0.81568628549575806),
            (0.82323235273361206, 0.7764706015586853, 0.7764706015586853),
            (0.82828283309936523, 0.74117648601531982, 0.74117648601531982),
            (0.83333331346511841, 0.70196080207824707, 0.70196080207824707),
            (0.83838385343551636, 0.66666668653488159, 0.66666668653488159),
            (0.84343433380126953, 0.62745100259780884, 0.62745100259780884),
            (0.84848487377166748, 0.61960786581039429, 0.61960786581039429),
            (0.85353535413742065, 0.65098041296005249, 0.65098041296005249),
            (0.85858583450317383, 0.68235296010971069, 0.68235296010971069),
            (0.86363637447357178, 0.7137255072593689, 0.7137255072593689),
            (0.86868685483932495, 0.7450980544090271, 0.7450980544090271),
            (0.8737373948097229, 0.77254903316497803, 0.77254903316497803),
            (0.87878787517547607, 0.80392158031463623, 0.80392158031463623),
            (0.88383835554122925, 0.83529412746429443, 0.83529412746429443),
            (0.8888888955116272, 0.86666667461395264, 0.86666667461395264),
            (0.89393937587738037, 0.89803922176361084, 0.89803922176361084),
            (0.89898991584777832, 0.92941176891326904, 0.92941176891326904),
            (0.90404039621353149, 0.93333333730697632, 0.93333333730697632),
            (0.90909093618392944, 0.93725490570068359, 0.93725490570068359),
            (0.91414141654968262, 0.93725490570068359, 0.93725490570068359),
            (0.91919189691543579, 0.94117647409439087, 0.94117647409439087),
            (0.92424243688583374, 0.94509804248809814, 0.94509804248809814),
            (0.92929291725158691, 0.94509804248809814, 0.94509804248809814),
            (0.93434345722198486, 0.94901961088180542, 0.94901961088180542),
            (0.93939393758773804, 0.9529411792755127, 0.9529411792755127),
            (0.94444441795349121, 0.9529411792755127, 0.9529411792755127),
            (0.94949495792388916, 0.95686274766921997, 0.95686274766921997),
            (0.95454543828964233, 0.96078431606292725, 0.96078431606292725),
            (0.95959597826004028, 0.96470588445663452, 0.96470588445663452),
            (0.96464645862579346, 0.9686274528503418, 0.9686274528503418),
            (0.96969699859619141, 0.97254902124404907, 0.97254902124404907),
            (0.97474747896194458, 0.97647058963775635, 0.97647058963775635),
            (0.97979795932769775, 0.98039215803146362, 0.98039215803146362),
            (0.9848484992980957, 0.9843137264251709, 0.9843137264251709),
            (0.98989897966384888, 0.98823529481887817, 0.98823529481887817),
            (0.99494951963424683, 0.99215686321258545, 0.99215686321258545),
            (1.0, 0.99607843160629272, 0.99607843160629272),
        ],
        green=[
            (0.0, 0.0, 0.0),
            (
                0.0050505050458014011,
                0.035294119268655777,
                0.035294119268655777,
            ),
            (0.010101010091602802, 0.074509806931018829, 0.074509806931018829),
            (0.015151515603065491, 0.10980392247438431, 0.10980392247438431),
            (0.020202020183205605, 0.14901961386203766, 0.14901961386203766),
            (0.025252524763345718, 0.18431372940540314, 0.18431372940540314),
            (0.030303031206130981, 0.22352941334247589, 0.22352941334247589),
            (0.035353533923625946, 0.25882354378700256, 0.25882354378700256),
            (0.040404040366411209, 0.29803922772407532, 0.29803922772407532),
            (0.045454546809196472, 0.3333333432674408, 0.3333333432674408),
            (0.050505049526691437, 0.37254902720451355, 0.37254902720451355),
            (0.0555555559694767, 0.36862745881080627, 0.36862745881080627),
            (0.060606062412261963, 0.3333333432674408, 0.3333333432674408),
            (0.065656565129756927, 0.29411765933036804, 0.29411765933036804),
            (0.070707067847251892, 0.25882354378700256, 0.25882354378700256),
            (0.075757578015327454, 0.21960784494876862, 0.21960784494876862),
            (0.080808080732822418, 0.18431372940540314, 0.18431372940540314),
            (0.085858583450317383, 0.14509804546833038, 0.14509804546833038),
            (0.090909093618392944, 0.10980392247438431, 0.10980392247438431),
            (0.095959596335887909, 0.070588238537311554, 0.070588238537311554),
            (0.10101009905338287, 0.035294119268655777, 0.035294119268655777),
            (0.10606060922145844, 0.0, 0.0),
            (0.1111111119389534, 0.074509806931018829, 0.074509806931018829),
            (0.11616161465644836, 0.14509804546833038, 0.14509804546833038),
            (0.12121212482452393, 0.21568627655506134, 0.21568627655506134),
            (0.12626262009143829, 0.28627452254295349, 0.28627452254295349),
            (0.13131313025951385, 0.36078432202339172, 0.36078432202339172),
            (0.13636364042758942, 0.43137255311012268, 0.43137255311012268),
            (0.14141413569450378, 0.50196081399917603, 0.50196081399917603),
            (0.14646464586257935, 0.57254904508590698, 0.57254904508590698),
            (0.15151515603065491, 0.64705884456634521, 0.64705884456634521),
            (0.15656565129756927, 0.71764707565307617, 0.71764707565307617),
            (0.16161616146564484, 0.7607843279838562, 0.7607843279838562),
            (0.1666666716337204, 0.78431373834609985, 0.78431373834609985),
            (0.17171716690063477, 0.80784314870834351, 0.80784314870834351),
            (0.17676767706871033, 0.83137255907058716, 0.83137255907058716),
            (0.18181818723678589, 0.85490196943283081, 0.85490196943283081),
            (0.18686868250370026, 0.88235294818878174, 0.88235294818878174),
            (0.19191919267177582, 0.90588235855102539, 0.90588235855102539),
            (0.19696970283985138, 0.92941176891326904, 0.92941176891326904),
            (0.20202019810676575, 0.9529411792755127, 0.9529411792755127),
            (0.20707070827484131, 0.97647058963775635, 0.97647058963775635),
            (0.21212121844291687, 0.99607843160629272, 0.99607843160629272),
            (0.21717171370983124, 0.99607843160629272, 0.99607843160629272),
            (0.2222222238779068, 0.99215686321258545, 0.99215686321258545),
            (0.22727273404598236, 0.99215686321258545, 0.99215686321258545),
            (0.23232322931289673, 0.99215686321258545, 0.99215686321258545),
            (0.23737373948097229, 0.98823529481887817, 0.98823529481887817),
            (0.24242424964904785, 0.98823529481887817, 0.98823529481887817),
            (0.24747474491596222, 0.9843137264251709, 0.9843137264251709),
            (0.25252524018287659, 0.9843137264251709, 0.9843137264251709),
            (0.25757575035095215, 0.98039215803146362, 0.98039215803146362),
            (0.26262626051902771, 0.98039215803146362, 0.98039215803146362),
            (0.26767677068710327, 0.98039215803146362, 0.98039215803146362),
            (0.27272728085517883, 0.98039215803146362, 0.98039215803146362),
            (0.27777779102325439, 0.9843137264251709, 0.9843137264251709),
            (0.28282827138900757, 0.9843137264251709, 0.9843137264251709),
            (0.28787878155708313, 0.98823529481887817, 0.98823529481887817),
            (0.29292929172515869, 0.98823529481887817, 0.98823529481887817),
            (0.29797980189323425, 0.99215686321258545, 0.99215686321258545),
            (0.30303031206130981, 0.99215686321258545, 0.99215686321258545),
            (0.30808082222938538, 0.99607843160629272, 0.99607843160629272),
            (0.31313130259513855, 0.99607843160629272, 0.99607843160629272),
            (0.31818181276321411, 0.99607843160629272, 0.99607843160629272),
            (0.32323232293128967, 0.97647058963775635, 0.97647058963775635),
            (0.32828283309936523, 0.95686274766921997, 0.95686274766921997),
            (0.3333333432674408, 0.93725490570068359, 0.93725490570068359),
            (0.33838382363319397, 0.92156863212585449, 0.92156863212585449),
            (0.34343433380126953, 0.90196079015731812, 0.90196079015731812),
            (0.34848484396934509, 0.88235294818878174, 0.88235294818878174),
            (0.35353535413742065, 0.86274510622024536, 0.86274510622024536),
            (0.35858586430549622, 0.84705883264541626, 0.84705883264541626),
            (0.36363637447357178, 0.82745099067687988, 0.82745099067687988),
            (0.36868685483932495, 0.80784314870834351, 0.80784314870834351),
            (0.37373736500740051, 0.81568628549575806, 0.81568628549575806),
            (0.37878787517547607, 0.83529412746429443, 0.83529412746429443),
            (0.38383838534355164, 0.85098040103912354, 0.85098040103912354),
            (0.3888888955116272, 0.87058824300765991, 0.87058824300765991),
            (0.39393940567970276, 0.89019608497619629, 0.89019608497619629),
            (0.39898988604545593, 0.90980392694473267, 0.90980392694473267),
            (0.40404039621353149, 0.92549020051956177, 0.92549020051956177),
            (0.40909090638160706, 0.94509804248809814, 0.94509804248809814),
            (0.41414141654968262, 0.96470588445663452, 0.96470588445663452),
            (0.41919192671775818, 0.9843137264251709, 0.9843137264251709),
            (0.42424243688583374, 1.0, 1.0),
            (0.42929291725158691, 1.0, 1.0),
            (0.43434342741966248, 1.0, 1.0),
            (0.43939393758773804, 1.0, 1.0),
            (0.4444444477558136, 1.0, 1.0),
            (0.44949495792388916, 1.0, 1.0),
            (0.45454546809196472, 1.0, 1.0),
            (0.4595959484577179, 1.0, 1.0),
            (0.46464645862579346, 1.0, 1.0),
            (0.46969696879386902, 1.0, 1.0),
            (0.47474747896194458, 1.0, 1.0),
            (0.47979798913002014, 1.0, 1.0),
            (0.4848484992980957, 1.0, 1.0),
            (0.48989897966384888, 1.0, 1.0),
            (0.49494948983192444, 1.0, 1.0),
            (0.5, 1.0, 1.0),
            (0.50505048036575317, 1.0, 1.0),
            (0.51010102033615112, 1.0, 1.0),
            (0.5151515007019043, 1.0, 1.0),
            (0.52020204067230225, 1.0, 1.0),
            (0.52525252103805542, 1.0, 1.0),
            (0.53030300140380859, 0.99215686321258545, 0.99215686321258545),
            (0.53535354137420654, 0.98039215803146362, 0.98039215803146362),
            (0.54040402173995972, 0.96470588445663452, 0.96470588445663452),
            (0.54545456171035767, 0.94901961088180542, 0.94901961088180542),
            (0.55050504207611084, 0.93333333730697632, 0.93333333730697632),
            (0.55555558204650879, 0.91764706373214722, 0.91764706373214722),
            (0.56060606241226196, 0.90588235855102539, 0.90588235855102539),
            (0.56565654277801514, 0.89019608497619629, 0.89019608497619629),
            (0.57070708274841309, 0.87450981140136719, 0.87450981140136719),
            (0.57575756311416626, 0.85882353782653809, 0.85882353782653809),
            (0.58080810308456421, 0.84313726425170898, 0.84313726425170898),
            (0.58585858345031738, 0.83137255907058716, 0.83137255907058716),
            (0.59090906381607056, 0.81960785388946533, 0.81960785388946533),
            (0.59595960378646851, 0.81176471710205078, 0.81176471710205078),
            (0.60101008415222168, 0.80000001192092896, 0.80000001192092896),
            (0.60606062412261963, 0.78823530673980713, 0.78823530673980713),
            (0.6111111044883728, 0.7764706015586853, 0.7764706015586853),
            (0.61616164445877075, 0.76470589637756348, 0.76470589637756348),
            (0.62121212482452393, 0.75294119119644165, 0.75294119119644165),
            (0.6262626051902771, 0.74117648601531982, 0.74117648601531982),
            (0.63131314516067505, 0.729411780834198, 0.729411780834198),
            (0.63636362552642822, 0.70980393886566162, 0.70980393886566162),
            (0.64141416549682617, 0.66666668653488159, 0.66666668653488159),
            (0.64646464586257935, 0.62352943420410156, 0.62352943420410156),
            (0.65151512622833252, 0.58039218187332153, 0.58039218187332153),
            (0.65656566619873047, 0.5372549295425415, 0.5372549295425415),
            (0.66161614656448364, 0.49411764740943909, 0.49411764740943909),
            (0.66666668653488159, 0.45098039507865906, 0.45098039507865906),
            (0.67171716690063477, 0.40392157435417175, 0.40392157435417175),
            (0.67676764726638794, 0.36078432202339172, 0.36078432202339172),
            (0.68181818723678589, 0.31764706969261169, 0.31764706969261169),
            (0.68686866760253906, 0.27450981736183167, 0.27450981736183167),
            (0.69191920757293701, 0.24705882370471954, 0.24705882370471954),
            (0.69696968793869019, 0.21960784494876862, 0.21960784494876862),
            (0.70202022790908813, 0.19607843458652496, 0.19607843458652496),
            (0.70707070827484131, 0.16862745583057404, 0.16862745583057404),
            (0.71212118864059448, 0.14509804546833038, 0.14509804546833038),
            (0.71717172861099243, 0.11764705926179886, 0.11764705926179886),
            (0.72222220897674561, 0.090196080505847931, 0.090196080505847931),
            (0.72727274894714355, 0.066666670143604279, 0.066666670143604279),
            (0.73232322931289673, 0.039215687662363052, 0.039215687662363052),
            (0.7373737096786499, 0.015686275437474251, 0.015686275437474251),
            (0.74242424964904785, 0.0, 0.0),
            (0.74747473001480103, 0.0, 0.0),
            (0.75252526998519897, 0.0, 0.0),
            (0.75757575035095215, 0.0, 0.0),
            (0.7626262903213501, 0.0, 0.0),
            (0.76767677068710327, 0.0, 0.0),
            (0.77272725105285645, 0.0, 0.0),
            (0.77777779102325439, 0.0, 0.0),
            (0.78282827138900757, 0.0, 0.0),
            (0.78787881135940552, 0.0, 0.0),
            (0.79292929172515869, 0.0, 0.0),
            (0.79797977209091187, 0.015686275437474251, 0.015686275437474251),
            (0.80303031206130981, 0.031372550874948502, 0.031372550874948502),
            (0.80808079242706299, 0.050980392843484879, 0.050980392843484879),
            (0.81313133239746094, 0.066666670143604279, 0.066666670143604279),
            (0.81818181276321411, 0.086274512112140656, 0.086274512112140656),
            (0.82323235273361206, 0.10588235408067703, 0.10588235408067703),
            (0.82828283309936523, 0.12156862765550613, 0.12156862765550613),
            (0.83333331346511841, 0.14117647707462311, 0.14117647707462311),
            (0.83838385343551636, 0.15686275064945221, 0.15686275064945221),
            (0.84343433380126953, 0.17647059261798859, 0.17647059261798859),
            (0.84848487377166748, 0.20000000298023224, 0.20000000298023224),
            (0.85353535413742065, 0.23137255012989044, 0.23137255012989044),
            (0.85858583450317383, 0.25882354378700256, 0.25882354378700256),
            (0.86363637447357178, 0.29019609093666077, 0.29019609093666077),
            (0.86868685483932495, 0.32156863808631897, 0.32156863808631897),
            (0.8737373948097229, 0.35294118523597717, 0.35294118523597717),
            (0.87878787517547607, 0.38431373238563538, 0.38431373238563538),
            (0.88383835554122925, 0.41568627953529358, 0.41568627953529358),
            (0.8888888955116272, 0.44313725829124451, 0.44313725829124451),
            (0.89393937587738037, 0.47450980544090271, 0.47450980544090271),
            (0.89898991584777832, 0.5058823823928833, 0.5058823823928833),
            (0.90404039621353149, 0.52941179275512695, 0.52941179275512695),
            (0.90909093618392944, 0.55294120311737061, 0.55294120311737061),
            (0.91414141654968262, 0.57254904508590698, 0.57254904508590698),
            (0.91919189691543579, 0.59607845544815063, 0.59607845544815063),
            (0.92424243688583374, 0.61960786581039429, 0.61960786581039429),
            (0.92929291725158691, 0.64313727617263794, 0.64313727617263794),
            (0.93434345722198486, 0.66274511814117432, 0.66274511814117432),
            (0.93939393758773804, 0.68627452850341797, 0.68627452850341797),
            (0.94444441795349121, 0.70980393886566162, 0.70980393886566162),
            (0.94949495792388916, 0.729411780834198, 0.729411780834198),
            (0.95454543828964233, 0.75294119119644165, 0.75294119119644165),
            (0.95959597826004028, 0.78039216995239258, 0.78039216995239258),
            (0.96464645862579346, 0.80392158031463623, 0.80392158031463623),
            (0.96969699859619141, 0.82745099067687988, 0.82745099067687988),
            (0.97474747896194458, 0.85098040103912354, 0.85098040103912354),
            (0.97979795932769775, 0.87450981140136719, 0.87450981140136719),
            (0.9848484992980957, 0.90196079015731812, 0.90196079015731812),
            (0.98989897966384888, 0.92549020051956177, 0.92549020051956177),
            (0.99494951963424683, 0.94901961088180542, 0.94901961088180542),
            (1.0, 0.97254902124404907, 0.97254902124404907),
        ],
        blue=[
            (0.0, 0.50196081399917603, 0.50196081399917603),
            (0.0050505050458014011, 0.45098039507865906, 0.45098039507865906),
            (0.010101010091602802, 0.40392157435417175, 0.40392157435417175),
            (0.015151515603065491, 0.35686275362968445, 0.35686275362968445),
            (0.020202020183205605, 0.30980393290519714, 0.30980393290519714),
            (0.025252524763345718, 0.25882354378700256, 0.25882354378700256),
            (0.030303031206130981, 0.21176470816135406, 0.21176470816135406),
            (0.035353533923625946, 0.16470588743686676, 0.16470588743686676),
            (0.040404040366411209, 0.11764705926179886, 0.11764705926179886),
            (0.045454546809196472, 0.070588238537311554, 0.070588238537311554),
            (0.050505049526691437, 0.019607843831181526, 0.019607843831181526),
            (0.0555555559694767, 0.047058824449777603, 0.047058824449777603),
            (0.060606062412261963, 0.14509804546833038, 0.14509804546833038),
            (0.065656565129756927, 0.23921568691730499, 0.23921568691730499),
            (0.070707067847251892, 0.3333333432674408, 0.3333333432674408),
            (0.075757578015327454, 0.43137255311012268, 0.43137255311012268),
            (0.080808080732822418, 0.52549022436141968, 0.52549022436141968),
            (0.085858583450317383, 0.61960786581039429, 0.61960786581039429),
            (0.090909093618392944, 0.71764707565307617, 0.71764707565307617),
            (0.095959596335887909, 0.81176471710205078, 0.81176471710205078),
            (0.10101009905338287, 0.90588235855102539, 0.90588235855102539),
            (0.10606060922145844, 1.0, 1.0),
            (0.1111111119389534, 1.0, 1.0),
            (0.11616161465644836, 1.0, 1.0),
            (0.12121212482452393, 1.0, 1.0),
            (0.12626262009143829, 1.0, 1.0),
            (0.13131313025951385, 1.0, 1.0),
            (0.13636364042758942, 1.0, 1.0),
            (0.14141413569450378, 1.0, 1.0),
            (0.14646464586257935, 1.0, 1.0),
            (0.15151515603065491, 1.0, 1.0),
            (0.15656565129756927, 1.0, 1.0),
            (0.16161616146564484, 1.0, 1.0),
            (0.1666666716337204, 1.0, 1.0),
            (0.17171716690063477, 1.0, 1.0),
            (0.17676767706871033, 1.0, 1.0),
            (0.18181818723678589, 1.0, 1.0),
            (0.18686868250370026, 1.0, 1.0),
            (0.19191919267177582, 1.0, 1.0),
            (0.19696970283985138, 1.0, 1.0),
            (0.20202019810676575, 1.0, 1.0),
            (0.20707070827484131, 1.0, 1.0),
            (0.21212121844291687, 0.99215686321258545, 0.99215686321258545),
            (0.21717171370983124, 0.95686274766921997, 0.95686274766921997),
            (0.2222222238779068, 0.91764706373214722, 0.91764706373214722),
            (0.22727273404598236, 0.88235294818878174, 0.88235294818878174),
            (0.23232322931289673, 0.84313726425170898, 0.84313726425170898),
            (0.23737373948097229, 0.80392158031463623, 0.80392158031463623),
            (0.24242424964904785, 0.76862746477127075, 0.76862746477127075),
            (0.24747474491596222, 0.729411780834198, 0.729411780834198),
            (0.25252524018287659, 0.69019609689712524, 0.69019609689712524),
            (0.25757575035095215, 0.65490198135375977, 0.65490198135375977),
            (0.26262626051902771, 0.61568629741668701, 0.61568629741668701),
            (0.26767677068710327, 0.56470590829849243, 0.56470590829849243),
            (0.27272728085517883, 0.50980395078659058, 0.50980395078659058),
            (0.27777779102325439, 0.45098039507865906, 0.45098039507865906),
            (0.28282827138900757, 0.39215686917304993, 0.39215686917304993),
            (0.28787878155708313, 0.3333333432674408, 0.3333333432674408),
            (0.29292929172515869, 0.27843138575553894, 0.27843138575553894),
            (0.29797980189323425, 0.21960784494876862, 0.21960784494876862),
            (0.30303031206130981, 0.16078431904315948, 0.16078431904315948),
            (0.30808082222938538, 0.10588235408067703, 0.10588235408067703),
            (0.31313130259513855, 0.047058824449777603, 0.047058824449777603),
            (0.31818181276321411, 0.0, 0.0),
            (0.32323232293128967, 0.0, 0.0),
            (0.32828283309936523, 0.0, 0.0),
            (0.3333333432674408, 0.0, 0.0),
            (0.33838382363319397, 0.0, 0.0),
            (0.34343433380126953, 0.0, 0.0),
            (0.34848484396934509, 0.0, 0.0),
            (0.35353535413742065, 0.0, 0.0),
            (0.35858586430549622, 0.0, 0.0),
            (0.36363637447357178, 0.0, 0.0),
            (0.36868685483932495, 0.0, 0.0),
            (0.37373736500740051, 0.0, 0.0),
            (0.37878787517547607, 0.0, 0.0),
            (0.38383838534355164, 0.0, 0.0),
            (0.3888888955116272, 0.0, 0.0),
            (0.39393940567970276, 0.0, 0.0),
            (0.39898988604545593, 0.0, 0.0),
            (0.40404039621353149, 0.0, 0.0),
            (0.40909090638160706, 0.0, 0.0),
            (0.41414141654968262, 0.0, 0.0),
            (0.41919192671775818, 0.0, 0.0),
            (
                0.42424243688583374,
                0.0039215688593685627,
                0.0039215688593685627,
            ),
            (0.42929291725158691, 0.027450980618596077, 0.027450980618596077),
            (0.43434342741966248, 0.050980392843484879, 0.050980392843484879),
            (0.43939393758773804, 0.074509806931018829, 0.074509806931018829),
            (0.4444444477558136, 0.094117648899555206, 0.094117648899555206),
            (0.44949495792388916, 0.11764705926179886, 0.11764705926179886),
            (0.45454546809196472, 0.14117647707462311, 0.14117647707462311),
            (0.4595959484577179, 0.16470588743686676, 0.16470588743686676),
            (0.46464645862579346, 0.18823529779911041, 0.18823529779911041),
            (0.46969696879386902, 0.21176470816135406, 0.21176470816135406),
            (0.47474747896194458, 0.23529411852359772, 0.23529411852359772),
            (0.47979798913002014, 0.22352941334247589, 0.22352941334247589),
            (0.4848484992980957, 0.20000000298023224, 0.20000000298023224),
            (0.48989897966384888, 0.17647059261798859, 0.17647059261798859),
            (0.49494948983192444, 0.15294118225574493, 0.15294118225574493),
            (0.5, 0.12941177189350128, 0.12941177189350128),
            (0.50505048036575317, 0.10980392247438431, 0.10980392247438431),
            (0.51010102033615112, 0.086274512112140656, 0.086274512112140656),
            (0.5151515007019043, 0.062745101749897003, 0.062745101749897003),
            (0.52020204067230225, 0.039215687662363052, 0.039215687662363052),
            (0.52525252103805542, 0.015686275437474251, 0.015686275437474251),
            (0.53030300140380859, 0.0, 0.0),
            (0.53535354137420654, 0.0, 0.0),
            (0.54040402173995972, 0.0, 0.0),
            (0.54545456171035767, 0.0, 0.0),
            (0.55050504207611084, 0.0, 0.0),
            (0.55555558204650879, 0.0, 0.0),
            (0.56060606241226196, 0.0, 0.0),
            (0.56565654277801514, 0.0, 0.0),
            (0.57070708274841309, 0.0, 0.0),
            (0.57575756311416626, 0.0, 0.0),
            (0.58080810308456421, 0.0, 0.0),
            (
                0.58585858345031738,
                0.0039215688593685627,
                0.0039215688593685627,
            ),
            (
                0.59090906381607056,
                0.0078431377187371254,
                0.0078431377187371254,
            ),
            (0.59595960378646851, 0.011764706112444401, 0.011764706112444401),
            (0.60101008415222168, 0.019607843831181526, 0.019607843831181526),
            (0.60606062412261963, 0.023529412224888802, 0.023529412224888802),
            (0.6111111044883728, 0.031372550874948502, 0.031372550874948502),
            (0.61616164445877075, 0.035294119268655777, 0.035294119268655777),
            (0.62121212482452393, 0.043137256056070328, 0.043137256056070328),
            (0.6262626051902771, 0.047058824449777603, 0.047058824449777603),
            (0.63131314516067505, 0.054901961237192154, 0.054901961237192154),
            (0.63636362552642822, 0.054901961237192154, 0.054901961237192154),
            (0.64141416549682617, 0.050980392843484879, 0.050980392843484879),
            (0.64646464586257935, 0.043137256056070328, 0.043137256056070328),
            (0.65151512622833252, 0.039215687662363052, 0.039215687662363052),
            (0.65656566619873047, 0.031372550874948502, 0.031372550874948502),
            (0.66161614656448364, 0.027450980618596077, 0.027450980618596077),
            (0.66666668653488159, 0.019607843831181526, 0.019607843831181526),
            (0.67171716690063477, 0.015686275437474251, 0.015686275437474251),
            (0.67676764726638794, 0.011764706112444401, 0.011764706112444401),
            (
                0.68181818723678589,
                0.0039215688593685627,
                0.0039215688593685627,
            ),
            (0.68686866760253906, 0.0, 0.0),
            (0.69191920757293701, 0.0, 0.0),
            (0.69696968793869019, 0.0, 0.0),
            (0.70202022790908813, 0.0, 0.0),
            (0.70707070827484131, 0.0, 0.0),
            (0.71212118864059448, 0.0, 0.0),
            (0.71717172861099243, 0.0, 0.0),
            (0.72222220897674561, 0.0, 0.0),
            (0.72727274894714355, 0.0, 0.0),
            (0.73232322931289673, 0.0, 0.0),
            (0.7373737096786499, 0.0, 0.0),
            (0.74242424964904785, 0.031372550874948502, 0.031372550874948502),
            (0.74747473001480103, 0.12941177189350128, 0.12941177189350128),
            (0.75252526998519897, 0.22352941334247589, 0.22352941334247589),
            (0.75757575035095215, 0.32156863808631897, 0.32156863808631897),
            (0.7626262903213501, 0.41568627953529358, 0.41568627953529358),
            (0.76767677068710327, 0.50980395078659058, 0.50980395078659058),
            (0.77272725105285645, 0.60784316062927246, 0.60784316062927246),
            (0.77777779102325439, 0.70196080207824707, 0.70196080207824707),
            (0.78282827138900757, 0.79607844352722168, 0.79607844352722168),
            (0.78787881135940552, 0.89411765336990356, 0.89411765336990356),
            (0.79292929172515869, 0.98823529481887817, 0.98823529481887817),
            (0.79797977209091187, 1.0, 1.0),
            (0.80303031206130981, 1.0, 1.0),
            (0.80808079242706299, 1.0, 1.0),
            (0.81313133239746094, 1.0, 1.0),
            (0.81818181276321411, 1.0, 1.0),
            (0.82323235273361206, 1.0, 1.0),
            (0.82828283309936523, 1.0, 1.0),
            (0.83333331346511841, 1.0, 1.0),
            (0.83838385343551636, 1.0, 1.0),
            (0.84343433380126953, 1.0, 1.0),
            (0.84848487377166748, 0.99607843160629272, 0.99607843160629272),
            (0.85353535413742065, 0.98823529481887817, 0.98823529481887817),
            (0.85858583450317383, 0.9843137264251709, 0.9843137264251709),
            (0.86363637447357178, 0.97647058963775635, 0.97647058963775635),
            (0.86868685483932495, 0.9686274528503418, 0.9686274528503418),
            (0.8737373948097229, 0.96470588445663452, 0.96470588445663452),
            (0.87878787517547607, 0.95686274766921997, 0.95686274766921997),
            (0.88383835554122925, 0.94901961088180542, 0.94901961088180542),
            (0.8888888955116272, 0.94509804248809814, 0.94509804248809814),
            (0.89393937587738037, 0.93725490570068359, 0.93725490570068359),
            (0.89898991584777832, 0.93333333730697632, 0.93333333730697632),
            (0.90404039621353149, 0.93333333730697632, 0.93333333730697632),
            (0.90909093618392944, 0.93725490570068359, 0.93725490570068359),
            (0.91414141654968262, 0.93725490570068359, 0.93725490570068359),
            (0.91919189691543579, 0.94117647409439087, 0.94117647409439087),
            (0.92424243688583374, 0.94509804248809814, 0.94509804248809814),
            (0.92929291725158691, 0.94509804248809814, 0.94509804248809814),
            (0.93434345722198486, 0.94901961088180542, 0.94901961088180542),
            (0.93939393758773804, 0.9529411792755127, 0.9529411792755127),
            (0.94444441795349121, 0.9529411792755127, 0.9529411792755127),
            (0.94949495792388916, 0.95686274766921997, 0.95686274766921997),
            (0.95454543828964233, 0.96078431606292725, 0.96078431606292725),
            (0.95959597826004028, 0.96470588445663452, 0.96470588445663452),
            (0.96464645862579346, 0.9686274528503418, 0.9686274528503418),
            (0.96969699859619141, 0.97254902124404907, 0.97254902124404907),
            (0.97474747896194458, 0.97647058963775635, 0.97647058963775635),
            (0.97979795932769775, 0.98039215803146362, 0.98039215803146362),
            (0.9848484992980957, 0.9843137264251709, 0.9843137264251709),
            (0.98989897966384888, 0.98823529481887817, 0.98823529481887817),
            (0.99494951963424683, 0.99215686321258545, 0.99215686321258545),
            (1.0, 0.99607843160629272, 0.99607843160629272),
        ],
    )
    return ColorMapper.from_segment_map(_data, range=range, **traits)


def gist_rainbow(range, **traits):
    """Generator for the 'gist_rainbow' colormap from GIST."""
    _data = dict(
        red=[
            (0.0, 1.0, 1.0),
            (0.0042016808874905109, 1.0, 1.0),
            (0.0084033617749810219, 1.0, 1.0),
            (0.012605042196810246, 1.0, 1.0),
            (0.016806723549962044, 1.0, 1.0),
            (0.021008403971791267, 1.0, 1.0),
            (0.025210084393620491, 1.0, 1.0),
            (0.029411764815449715, 1.0, 1.0),
            (0.033613447099924088, 1.0, 1.0),
            (0.037815127521753311, 1.0, 1.0),
            (0.042016807943582535, 1.0, 1.0),
            (0.046218488365411758, 1.0, 1.0),
            (0.050420168787240982, 1.0, 1.0),
            (0.054621849209070206, 1.0, 1.0),
            (0.058823529630899429, 1.0, 1.0),
            (0.063025213778018951, 1.0, 1.0),
            (0.067226894199848175, 1.0, 1.0),
            (0.071428574621677399, 1.0, 1.0),
            (0.075630255043506622, 1.0, 1.0),
            (0.079831935465335846, 1.0, 1.0),
            (0.08403361588716507, 1.0, 1.0),
            (0.088235296308994293, 1.0, 1.0),
            (0.092436976730823517, 1.0, 1.0),
            (0.09663865715265274, 1.0, 1.0),
            (0.10084033757448196, 1.0, 1.0),
            (0.10504201799631119, 1.0, 1.0),
            (0.10924369841814041, 1.0, 1.0),
            (0.11344537883996964, 1.0, 1.0),
            (0.11764705926179886, 1.0, 1.0),
            (0.12184873968362808, 1.0, 1.0),
            (0.1260504275560379, 1.0, 1.0),
            (0.13025210797786713, 1.0, 1.0),
            (0.13445378839969635, 1.0, 1.0),
            (0.13865546882152557, 1.0, 1.0),
            (0.1428571492433548, 1.0, 1.0),
            (0.14705882966518402, 1.0, 1.0),
            (0.15126051008701324, 1.0, 1.0),
            (0.15546219050884247, 1.0, 1.0),
            (0.15966387093067169, 1.0, 1.0),
            (0.16386555135250092, 1.0, 1.0),
            (0.16806723177433014, 1.0, 1.0),
            (0.17226891219615936, 1.0, 1.0),
            (0.17647059261798859, 1.0, 1.0),
            (0.18067227303981781, 1.0, 1.0),
            (0.18487395346164703, 1.0, 1.0),
            (0.18907563388347626, 1.0, 1.0),
            (0.19327731430530548, 1.0, 1.0),
            (0.1974789947271347, 1.0, 1.0),
            (0.20168067514896393, 1.0, 1.0),
            (0.20588235557079315, 1.0, 1.0),
            (0.21008403599262238, 1.0, 1.0),
            (0.2142857164144516, 1.0, 1.0),
            (0.21848739683628082, 1.0, 1.0),
            (0.22268907725811005, 0.96078431606292725, 0.96078431606292725),
            (0.22689075767993927, 0.94117647409439087, 0.94117647409439087),
            (0.23109243810176849, 0.92156863212585449, 0.92156863212585449),
            (0.23529411852359772, 0.89803922176361084, 0.89803922176361084),
            (0.23949579894542694, 0.87843137979507446, 0.87843137979507446),
            (0.24369747936725616, 0.85882353782653809, 0.85882353782653809),
            (0.24789915978908539, 0.83529412746429443, 0.83529412746429443),
            (0.25210085511207581, 0.81568628549575806, 0.81568628549575806),
            (0.25630253553390503, 0.7921568751335144, 0.7921568751335144),
            (0.26050421595573425, 0.77254903316497803, 0.77254903316497803),
            (0.26470589637756348, 0.75294119119644165, 0.75294119119644165),
            (0.2689075767993927, 0.729411780834198, 0.729411780834198),
            (0.27310925722122192, 0.70980393886566162, 0.70980393886566162),
            (0.27731093764305115, 0.68627452850341797, 0.68627452850341797),
            (0.28151261806488037, 0.66666668653488159, 0.66666668653488159),
            (0.28571429848670959, 0.62352943420410156, 0.62352943420410156),
            (0.28991597890853882, 0.60392159223556519, 0.60392159223556519),
            (0.29411765933036804, 0.58431375026702881, 0.58431375026702881),
            (0.29831933975219727, 0.56078433990478516, 0.56078433990478516),
            (0.30252102017402649, 0.54117649793624878, 0.54117649793624878),
            (0.30672270059585571, 0.51764708757400513, 0.51764708757400513),
            (0.31092438101768494, 0.49803921580314636, 0.49803921580314636),
            (0.31512606143951416, 0.47843137383460999, 0.47843137383460999),
            (0.31932774186134338, 0.45490196347236633, 0.45490196347236633),
            (0.32352942228317261, 0.43529412150382996, 0.43529412150382996),
            (0.32773110270500183, 0.41568627953529358, 0.41568627953529358),
            (0.33193278312683105, 0.39215686917304993, 0.39215686917304993),
            (0.33613446354866028, 0.37254902720451355, 0.37254902720451355),
            (0.3403361439704895, 0.3490196168422699, 0.3490196168422699),
            (0.34453782439231873, 0.32941177487373352, 0.32941177487373352),
            (0.34873950481414795, 0.28627452254295349, 0.28627452254295349),
            (0.35294118523597717, 0.26666668057441711, 0.26666668057441711),
            (0.3571428656578064, 0.24705882370471954, 0.24705882370471954),
            (0.36134454607963562, 0.22352941334247589, 0.22352941334247589),
            (0.36554622650146484, 0.20392157137393951, 0.20392157137393951),
            (0.36974790692329407, 0.18039216101169586, 0.18039216101169586),
            (0.37394958734512329, 0.16078431904315948, 0.16078431904315948),
            (0.37815126776695251, 0.14117647707462311, 0.14117647707462311),
            (0.38235294818878174, 0.11764705926179886, 0.11764705926179886),
            (0.38655462861061096, 0.098039217293262482, 0.098039217293262482),
            (0.39075630903244019, 0.074509806931018829, 0.074509806931018829),
            (0.39495798945426941, 0.054901961237192154, 0.054901961237192154),
            (0.39915966987609863, 0.035294119268655777, 0.035294119268655777),
            (0.40336135029792786, 0.011764706112444401, 0.011764706112444401),
            (0.40756303071975708, 0.0, 0.0),
            (0.4117647111415863, 0.0, 0.0),
            (0.41596639156341553, 0.0, 0.0),
            (0.42016807198524475, 0.0, 0.0),
            (0.42436975240707397, 0.0, 0.0),
            (0.4285714328289032, 0.0, 0.0),
            (0.43277311325073242, 0.0, 0.0),
            (0.43697479367256165, 0.0, 0.0),
            (0.44117647409439087, 0.0, 0.0),
            (0.44537815451622009, 0.0, 0.0),
            (0.44957983493804932, 0.0, 0.0),
            (0.45378151535987854, 0.0, 0.0),
            (0.45798319578170776, 0.0, 0.0),
            (0.46218487620353699, 0.0, 0.0),
            (0.46638655662536621, 0.0, 0.0),
            (0.47058823704719543, 0.0, 0.0),
            (0.47478991746902466, 0.0, 0.0),
            (0.47899159789085388, 0.0, 0.0),
            (0.48319327831268311, 0.0, 0.0),
            (0.48739495873451233, 0.0, 0.0),
            (0.49159663915634155, 0.0, 0.0),
            (0.49579831957817078, 0.0, 0.0),
            (0.5, 0.0, 0.0),
            (0.50420171022415161, 0.0, 0.0),
            (0.50840336084365845, 0.0, 0.0),
            (0.51260507106781006, 0.0, 0.0),
            (0.51680672168731689, 0.0, 0.0),
            (0.52100843191146851, 0.0, 0.0),
            (0.52521008253097534, 0.0, 0.0),
            (0.52941179275512695, 0.0, 0.0),
            (0.53361344337463379, 0.0, 0.0),
            (0.5378151535987854, 0.0, 0.0),
            (0.54201680421829224, 0.0, 0.0),
            (0.54621851444244385, 0.0, 0.0),
            (0.55042016506195068, 0.0, 0.0),
            (0.55462187528610229, 0.0, 0.0),
            (0.55882352590560913, 0.0, 0.0),
            (0.56302523612976074, 0.0, 0.0),
            (0.56722688674926758, 0.0, 0.0),
            (0.57142859697341919, 0.0, 0.0),
            (0.57563024759292603, 0.0, 0.0),
            (0.57983195781707764, 0.0, 0.0),
            (0.58403360843658447, 0.0, 0.0),
            (0.58823531866073608, 0.0, 0.0),
            (0.59243696928024292, 0.0, 0.0),
            (0.59663867950439453, 0.0, 0.0),
            (0.60084033012390137, 0.0, 0.0),
            (0.60504204034805298, 0.0, 0.0),
            (0.60924369096755981, 0.0, 0.0),
            (0.61344540119171143, 0.0, 0.0),
            (0.61764705181121826, 0.0, 0.0),
            (0.62184876203536987, 0.0, 0.0),
            (0.62605041265487671, 0.0, 0.0),
            (0.63025212287902832, 0.0, 0.0),
            (0.63445377349853516, 0.0, 0.0),
            (0.63865548372268677, 0.0, 0.0),
            (0.6428571343421936, 0.0, 0.0),
            (0.64705884456634521, 0.0, 0.0),
            (0.65126049518585205, 0.0, 0.0),
            (0.65546220541000366, 0.0, 0.0),
            (0.6596638560295105, 0.0, 0.0),
            (0.66386556625366211, 0.0, 0.0),
            (0.66806721687316895, 0.0, 0.0),
            (0.67226892709732056, 0.0, 0.0),
            (0.67647057771682739, 0.0, 0.0),
            (0.680672287940979, 0.0, 0.0),
            (0.68487393856048584, 0.0, 0.0),
            (0.68907564878463745, 0.0, 0.0),
            (0.69327729940414429, 0.0, 0.0),
            (0.6974790096282959, 0.0, 0.0),
            (0.70168066024780273, 0.0, 0.0),
            (0.70588237047195435, 0.0, 0.0),
            (0.71008402109146118, 0.0, 0.0),
            (0.71428573131561279, 0.0, 0.0),
            (0.71848738193511963, 0.0, 0.0),
            (0.72268909215927124, 0.0, 0.0),
            (0.72689074277877808, 0.0, 0.0),
            (0.73109245300292969, 0.0, 0.0),
            (0.73529410362243652, 0.0, 0.0),
            (0.73949581384658813, 0.0, 0.0),
            (0.74369746446609497, 0.0, 0.0),
            (0.74789917469024658, 0.0, 0.0),
            (0.75210082530975342, 0.0, 0.0),
            (0.75630253553390503, 0.0, 0.0),
            (0.76050418615341187, 0.0, 0.0),
            (0.76470589637756348, 0.0, 0.0),
            (0.76890754699707031, 0.0, 0.0),
            (0.77310925722122192, 0.0, 0.0),
            (0.77731090784072876, 0.0, 0.0),
            (
                0.78151261806488037,
                0.0078431377187371254,
                0.0078431377187371254,
            ),
            (0.78571426868438721, 0.027450980618596077, 0.027450980618596077),
            (0.78991597890853882, 0.070588238537311554, 0.070588238537311554),
            (0.79411762952804565, 0.094117648899555206, 0.094117648899555206),
            (0.79831933975219727, 0.11372549086809158, 0.11372549086809158),
            (0.8025209903717041, 0.13333334028720856, 0.13333334028720856),
            (0.80672270059585571, 0.15686275064945221, 0.15686275064945221),
            (0.81092435121536255, 0.17647059261798859, 0.17647059261798859),
            (0.81512606143951416, 0.19607843458652496, 0.19607843458652496),
            (0.819327712059021, 0.21960784494876862, 0.21960784494876862),
            (0.82352942228317261, 0.23921568691730499, 0.23921568691730499),
            (0.82773107290267944, 0.26274511218070984, 0.26274511218070984),
            (0.83193278312683105, 0.28235295414924622, 0.28235295414924622),
            (0.83613443374633789, 0.30196079611778259, 0.30196079611778259),
            (0.8403361439704895, 0.32549020648002625, 0.32549020648002625),
            (0.84453779458999634, 0.34509804844856262, 0.34509804844856262),
            (0.84873950481414795, 0.364705890417099, 0.364705890417099),
            (0.85294115543365479, 0.40784314274787903, 0.40784314274787903),
            (0.8571428656578064, 0.43137255311012268, 0.43137255311012268),
            (0.86134451627731323, 0.45098039507865906, 0.45098039507865906),
            (0.86554622650146484, 0.47058823704719543, 0.47058823704719543),
            (0.86974787712097168, 0.49411764740943909, 0.49411764740943909),
            (0.87394958734512329, 0.51372551918029785, 0.51372551918029785),
            (0.87815123796463013, 0.53333336114883423, 0.53333336114883423),
            (0.88235294818878174, 0.55686277151107788, 0.55686277151107788),
            (0.88655459880828857, 0.57647061347961426, 0.57647061347961426),
            (0.89075630903244019, 0.60000002384185791, 0.60000002384185791),
            (0.89495795965194702, 0.61960786581039429, 0.61960786581039429),
            (0.89915966987609863, 0.63921570777893066, 0.63921570777893066),
            (0.90336132049560547, 0.66274511814117432, 0.66274511814117432),
            (0.90756303071975708, 0.68235296010971069, 0.68235296010971069),
            (0.91176468133926392, 0.70588237047195435, 0.70588237047195435),
            (0.91596639156341553, 0.7450980544090271, 0.7450980544090271),
            (0.92016804218292236, 0.76862746477127075, 0.76862746477127075),
            (0.92436975240707397, 0.78823530673980713, 0.78823530673980713),
            (0.92857140302658081, 0.80784314870834351, 0.80784314870834351),
            (0.93277311325073242, 0.83137255907058716, 0.83137255907058716),
            (0.93697476387023926, 0.85098040103912354, 0.85098040103912354),
            (0.94117647409439087, 0.87450981140136719, 0.87450981140136719),
            (0.94537812471389771, 0.89411765336990356, 0.89411765336990356),
            (0.94957983493804932, 0.91372549533843994, 0.91372549533843994),
            (0.95378148555755615, 0.93725490570068359, 0.93725490570068359),
            (0.95798319578170776, 0.95686274766921997, 0.95686274766921997),
            (0.9621848464012146, 0.97647058963775635, 0.97647058963775635),
            (0.96638655662536621, 1.0, 1.0),
            (0.97058820724487305, 1.0, 1.0),
            (0.97478991746902466, 1.0, 1.0),
            (0.97899156808853149, 1.0, 1.0),
            (0.98319327831268311, 1.0, 1.0),
            (0.98739492893218994, 1.0, 1.0),
            (0.99159663915634155, 1.0, 1.0),
            (0.99579828977584839, 1.0, 1.0),
            (1.0, 1.0, 1.0),
        ],
        green=[
            (0.0, 0.0, 0.0),
            (0.0042016808874905109, 0.0, 0.0),
            (0.0084033617749810219, 0.0, 0.0),
            (0.012605042196810246, 0.0, 0.0),
            (0.016806723549962044, 0.0, 0.0),
            (0.021008403971791267, 0.0, 0.0),
            (0.025210084393620491, 0.0, 0.0),
            (0.029411764815449715, 0.0, 0.0),
            (0.033613447099924088, 0.019607843831181526, 0.019607843831181526),
            (0.037815127521753311, 0.043137256056070328, 0.043137256056070328),
            (0.042016807943582535, 0.062745101749897003, 0.062745101749897003),
            (0.046218488365411758, 0.086274512112140656, 0.086274512112140656),
            (0.050420168787240982, 0.10588235408067703, 0.10588235408067703),
            (0.054621849209070206, 0.12549020349979401, 0.12549020349979401),
            (0.058823529630899429, 0.14901961386203766, 0.14901961386203766),
            (0.063025213778018951, 0.16862745583057404, 0.16862745583057404),
            (0.067226894199848175, 0.18823529779911041, 0.18823529779911041),
            (0.071428574621677399, 0.21176470816135406, 0.21176470816135406),
            (0.075630255043506622, 0.23137255012989044, 0.23137255012989044),
            (0.079831935465335846, 0.25490197539329529, 0.25490197539329529),
            (0.08403361588716507, 0.27450981736183167, 0.27450981736183167),
            (0.088235296308994293, 0.29411765933036804, 0.29411765933036804),
            (0.092436976730823517, 0.31764706969261169, 0.31764706969261169),
            (0.09663865715265274, 0.35686275362968445, 0.35686275362968445),
            (0.10084033757448196, 0.3803921639919281, 0.3803921639919281),
            (0.10504201799631119, 0.40000000596046448, 0.40000000596046448),
            (0.10924369841814041, 0.42352941632270813, 0.42352941632270813),
            (0.11344537883996964, 0.44313725829124451, 0.44313725829124451),
            (0.11764705926179886, 0.46274510025978088, 0.46274510025978088),
            (0.12184873968362808, 0.48627451062202454, 0.48627451062202454),
            (0.1260504275560379, 0.5058823823928833, 0.5058823823928833),
            (0.13025210797786713, 0.52941179275512695, 0.52941179275512695),
            (0.13445378839969635, 0.54901963472366333, 0.54901963472366333),
            (0.13865546882152557, 0.56862747669219971, 0.56862747669219971),
            (0.1428571492433548, 0.59215688705444336, 0.59215688705444336),
            (0.14705882966518402, 0.61176472902297974, 0.61176472902297974),
            (0.15126051008701324, 0.63137257099151611, 0.63137257099151611),
            (0.15546219050884247, 0.65490198135375977, 0.65490198135375977),
            (0.15966387093067169, 0.69803923368453979, 0.69803923368453979),
            (0.16386555135250092, 0.71764707565307617, 0.71764707565307617),
            (0.16806723177433014, 0.73725491762161255, 0.73725491762161255),
            (0.17226891219615936, 0.7607843279838562, 0.7607843279838562),
            (0.17647059261798859, 0.78039216995239258, 0.78039216995239258),
            (0.18067227303981781, 0.80000001192092896, 0.80000001192092896),
            (0.18487395346164703, 0.82352942228317261, 0.82352942228317261),
            (0.18907563388347626, 0.84313726425170898, 0.84313726425170898),
            (0.19327731430530548, 0.86666667461395264, 0.86666667461395264),
            (0.1974789947271347, 0.88627451658248901, 0.88627451658248901),
            (0.20168067514896393, 0.90588235855102539, 0.90588235855102539),
            (0.20588235557079315, 0.92941176891326904, 0.92941176891326904),
            (0.21008403599262238, 0.94901961088180542, 0.94901961088180542),
            (0.2142857164144516, 0.9686274528503418, 0.9686274528503418),
            (0.21848739683628082, 0.99215686321258545, 0.99215686321258545),
            (0.22268907725811005, 1.0, 1.0),
            (0.22689075767993927, 1.0, 1.0),
            (0.23109243810176849, 1.0, 1.0),
            (0.23529411852359772, 1.0, 1.0),
            (0.23949579894542694, 1.0, 1.0),
            (0.24369747936725616, 1.0, 1.0),
            (0.24789915978908539, 1.0, 1.0),
            (0.25210085511207581, 1.0, 1.0),
            (0.25630253553390503, 1.0, 1.0),
            (0.26050421595573425, 1.0, 1.0),
            (0.26470589637756348, 1.0, 1.0),
            (0.2689075767993927, 1.0, 1.0),
            (0.27310925722122192, 1.0, 1.0),
            (0.27731093764305115, 1.0, 1.0),
            (0.28151261806488037, 1.0, 1.0),
            (0.28571429848670959, 1.0, 1.0),
            (0.28991597890853882, 1.0, 1.0),
            (0.29411765933036804, 1.0, 1.0),
            (0.29831933975219727, 1.0, 1.0),
            (0.30252102017402649, 1.0, 1.0),
            (0.30672270059585571, 1.0, 1.0),
            (0.31092438101768494, 1.0, 1.0),
            (0.31512606143951416, 1.0, 1.0),
            (0.31932774186134338, 1.0, 1.0),
            (0.32352942228317261, 1.0, 1.0),
            (0.32773110270500183, 1.0, 1.0),
            (0.33193278312683105, 1.0, 1.0),
            (0.33613446354866028, 1.0, 1.0),
            (0.3403361439704895, 1.0, 1.0),
            (0.34453782439231873, 1.0, 1.0),
            (0.34873950481414795, 1.0, 1.0),
            (0.35294118523597717, 1.0, 1.0),
            (0.3571428656578064, 1.0, 1.0),
            (0.36134454607963562, 1.0, 1.0),
            (0.36554622650146484, 1.0, 1.0),
            (0.36974790692329407, 1.0, 1.0),
            (0.37394958734512329, 1.0, 1.0),
            (0.37815126776695251, 1.0, 1.0),
            (0.38235294818878174, 1.0, 1.0),
            (0.38655462861061096, 1.0, 1.0),
            (0.39075630903244019, 1.0, 1.0),
            (0.39495798945426941, 1.0, 1.0),
            (0.39915966987609863, 1.0, 1.0),
            (0.40336135029792786, 1.0, 1.0),
            (0.40756303071975708, 1.0, 1.0),
            (0.4117647111415863, 1.0, 1.0),
            (0.41596639156341553, 1.0, 1.0),
            (0.42016807198524475, 1.0, 1.0),
            (0.42436975240707397, 1.0, 1.0),
            (0.4285714328289032, 1.0, 1.0),
            (0.43277311325073242, 1.0, 1.0),
            (0.43697479367256165, 1.0, 1.0),
            (0.44117647409439087, 1.0, 1.0),
            (0.44537815451622009, 1.0, 1.0),
            (0.44957983493804932, 1.0, 1.0),
            (0.45378151535987854, 1.0, 1.0),
            (0.45798319578170776, 1.0, 1.0),
            (0.46218487620353699, 1.0, 1.0),
            (0.46638655662536621, 1.0, 1.0),
            (0.47058823704719543, 1.0, 1.0),
            (0.47478991746902466, 1.0, 1.0),
            (0.47899159789085388, 1.0, 1.0),
            (0.48319327831268311, 1.0, 1.0),
            (0.48739495873451233, 1.0, 1.0),
            (0.49159663915634155, 1.0, 1.0),
            (0.49579831957817078, 1.0, 1.0),
            (0.5, 1.0, 1.0),
            (0.50420171022415161, 1.0, 1.0),
            (0.50840336084365845, 1.0, 1.0),
            (0.51260507106781006, 1.0, 1.0),
            (0.51680672168731689, 1.0, 1.0),
            (0.52100843191146851, 1.0, 1.0),
            (0.52521008253097534, 1.0, 1.0),
            (0.52941179275512695, 1.0, 1.0),
            (0.53361344337463379, 1.0, 1.0),
            (0.5378151535987854, 1.0, 1.0),
            (0.54201680421829224, 1.0, 1.0),
            (0.54621851444244385, 1.0, 1.0),
            (0.55042016506195068, 1.0, 1.0),
            (0.55462187528610229, 1.0, 1.0),
            (0.55882352590560913, 1.0, 1.0),
            (0.56302523612976074, 1.0, 1.0),
            (0.56722688674926758, 1.0, 1.0),
            (0.57142859697341919, 1.0, 1.0),
            (0.57563024759292603, 1.0, 1.0),
            (0.57983195781707764, 1.0, 1.0),
            (0.58403360843658447, 1.0, 1.0),
            (0.58823531866073608, 1.0, 1.0),
            (0.59243696928024292, 1.0, 1.0),
            (0.59663867950439453, 0.98039215803146362, 0.98039215803146362),
            (0.60084033012390137, 0.93725490570068359, 0.93725490570068359),
            (0.60504204034805298, 0.91764706373214722, 0.91764706373214722),
            (0.60924369096755981, 0.89411765336990356, 0.89411765336990356),
            (0.61344540119171143, 0.87450981140136719, 0.87450981140136719),
            (0.61764705181121826, 0.85490196943283081, 0.85490196943283081),
            (0.62184876203536987, 0.83137255907058716, 0.83137255907058716),
            (0.62605041265487671, 0.81176471710205078, 0.81176471710205078),
            (0.63025212287902832, 0.78823530673980713, 0.78823530673980713),
            (0.63445377349853516, 0.76862746477127075, 0.76862746477127075),
            (0.63865548372268677, 0.74901962280273438, 0.74901962280273438),
            (0.6428571343421936, 0.72549021244049072, 0.72549021244049072),
            (0.64705884456634521, 0.70588237047195435, 0.70588237047195435),
            (0.65126049518585205, 0.68235296010971069, 0.68235296010971069),
            (0.65546220541000366, 0.66274511814117432, 0.66274511814117432),
            (0.6596638560295105, 0.64313727617263794, 0.64313727617263794),
            (0.66386556625366211, 0.60000002384185791, 0.60000002384185791),
            (0.66806721687316895, 0.58039218187332153, 0.58039218187332153),
            (0.67226892709732056, 0.55686277151107788, 0.55686277151107788),
            (0.67647057771682739, 0.5372549295425415, 0.5372549295425415),
            (0.680672287940979, 0.51372551918029785, 0.51372551918029785),
            (0.68487393856048584, 0.49411764740943909, 0.49411764740943909),
            (0.68907564878463745, 0.47450980544090271, 0.47450980544090271),
            (0.69327729940414429, 0.45098039507865906, 0.45098039507865906),
            (0.6974790096282959, 0.43137255311012268, 0.43137255311012268),
            (0.70168066024780273, 0.4117647111415863, 0.4117647111415863),
            (0.70588237047195435, 0.38823530077934265, 0.38823530077934265),
            (0.71008402109146118, 0.36862745881080627, 0.36862745881080627),
            (0.71428573131561279, 0.34509804844856262, 0.34509804844856262),
            (0.71848738193511963, 0.32549020648002625, 0.32549020648002625),
            (0.72268909215927124, 0.30588236451148987, 0.30588236451148987),
            (0.72689074277877808, 0.26274511218070984, 0.26274511218070984),
            (0.73109245300292969, 0.24313725531101227, 0.24313725531101227),
            (0.73529410362243652, 0.21960784494876862, 0.21960784494876862),
            (0.73949581384658813, 0.20000000298023224, 0.20000000298023224),
            (0.74369746446609497, 0.17647059261798859, 0.17647059261798859),
            (0.74789917469024658, 0.15686275064945221, 0.15686275064945221),
            (0.75210082530975342, 0.13725490868091583, 0.13725490868091583),
            (0.75630253553390503, 0.11372549086809158, 0.11372549086809158),
            (0.76050418615341187, 0.094117648899555206, 0.094117648899555206),
            (0.76470589637756348, 0.070588238537311554, 0.070588238537311554),
            (0.76890754699707031, 0.050980392843484879, 0.050980392843484879),
            (0.77310925722122192, 0.031372550874948502, 0.031372550874948502),
            (
                0.77731090784072876,
                0.0078431377187371254,
                0.0078431377187371254,
            ),
            (0.78151261806488037, 0.0, 0.0),
            (0.78571426868438721, 0.0, 0.0),
            (0.78991597890853882, 0.0, 0.0),
            (0.79411762952804565, 0.0, 0.0),
            (0.79831933975219727, 0.0, 0.0),
            (0.8025209903717041, 0.0, 0.0),
            (0.80672270059585571, 0.0, 0.0),
            (0.81092435121536255, 0.0, 0.0),
            (0.81512606143951416, 0.0, 0.0),
            (0.819327712059021, 0.0, 0.0),
            (0.82352942228317261, 0.0, 0.0),
            (0.82773107290267944, 0.0, 0.0),
            (0.83193278312683105, 0.0, 0.0),
            (0.83613443374633789, 0.0, 0.0),
            (0.8403361439704895, 0.0, 0.0),
            (0.84453779458999634, 0.0, 0.0),
            (0.84873950481414795, 0.0, 0.0),
            (0.85294115543365479, 0.0, 0.0),
            (0.8571428656578064, 0.0, 0.0),
            (0.86134451627731323, 0.0, 0.0),
            (0.86554622650146484, 0.0, 0.0),
            (0.86974787712097168, 0.0, 0.0),
            (0.87394958734512329, 0.0, 0.0),
            (0.87815123796463013, 0.0, 0.0),
            (0.88235294818878174, 0.0, 0.0),
            (0.88655459880828857, 0.0, 0.0),
            (0.89075630903244019, 0.0, 0.0),
            (0.89495795965194702, 0.0, 0.0),
            (0.89915966987609863, 0.0, 0.0),
            (0.90336132049560547, 0.0, 0.0),
            (0.90756303071975708, 0.0, 0.0),
            (0.91176468133926392, 0.0, 0.0),
            (0.91596639156341553, 0.0, 0.0),
            (0.92016804218292236, 0.0, 0.0),
            (0.92436975240707397, 0.0, 0.0),
            (0.92857140302658081, 0.0, 0.0),
            (0.93277311325073242, 0.0, 0.0),
            (0.93697476387023926, 0.0, 0.0),
            (0.94117647409439087, 0.0, 0.0),
            (0.94537812471389771, 0.0, 0.0),
            (0.94957983493804932, 0.0, 0.0),
            (0.95378148555755615, 0.0, 0.0),
            (0.95798319578170776, 0.0, 0.0),
            (0.9621848464012146, 0.0, 0.0),
            (0.96638655662536621, 0.0, 0.0),
            (0.97058820724487305, 0.0, 0.0),
            (0.97478991746902466, 0.0, 0.0),
            (0.97899156808853149, 0.0, 0.0),
            (0.98319327831268311, 0.0, 0.0),
            (0.98739492893218994, 0.0, 0.0),
            (0.99159663915634155, 0.0, 0.0),
            (0.99579828977584839, 0.0, 0.0),
            (1.0, 0.0, 0.0),
        ],
        blue=[
            (0.0, 0.16470588743686676, 0.16470588743686676),
            (0.0042016808874905109, 0.14117647707462311, 0.14117647707462311),
            (0.0084033617749810219, 0.12156862765550613, 0.12156862765550613),
            (0.012605042196810246, 0.10196078568696976, 0.10196078568696976),
            (0.016806723549962044, 0.078431375324726105, 0.078431375324726105),
            (0.021008403971791267, 0.058823529630899429, 0.058823529630899429),
            (0.025210084393620491, 0.039215687662363052, 0.039215687662363052),
            (0.029411764815449715, 0.015686275437474251, 0.015686275437474251),
            (0.033613447099924088, 0.0, 0.0),
            (0.037815127521753311, 0.0, 0.0),
            (0.042016807943582535, 0.0, 0.0),
            (0.046218488365411758, 0.0, 0.0),
            (0.050420168787240982, 0.0, 0.0),
            (0.054621849209070206, 0.0, 0.0),
            (0.058823529630899429, 0.0, 0.0),
            (0.063025213778018951, 0.0, 0.0),
            (0.067226894199848175, 0.0, 0.0),
            (0.071428574621677399, 0.0, 0.0),
            (0.075630255043506622, 0.0, 0.0),
            (0.079831935465335846, 0.0, 0.0),
            (0.08403361588716507, 0.0, 0.0),
            (0.088235296308994293, 0.0, 0.0),
            (0.092436976730823517, 0.0, 0.0),
            (0.09663865715265274, 0.0, 0.0),
            (0.10084033757448196, 0.0, 0.0),
            (0.10504201799631119, 0.0, 0.0),
            (0.10924369841814041, 0.0, 0.0),
            (0.11344537883996964, 0.0, 0.0),
            (0.11764705926179886, 0.0, 0.0),
            (0.12184873968362808, 0.0, 0.0),
            (0.1260504275560379, 0.0, 0.0),
            (0.13025210797786713, 0.0, 0.0),
            (0.13445378839969635, 0.0, 0.0),
            (0.13865546882152557, 0.0, 0.0),
            (0.1428571492433548, 0.0, 0.0),
            (0.14705882966518402, 0.0, 0.0),
            (0.15126051008701324, 0.0, 0.0),
            (0.15546219050884247, 0.0, 0.0),
            (0.15966387093067169, 0.0, 0.0),
            (0.16386555135250092, 0.0, 0.0),
            (0.16806723177433014, 0.0, 0.0),
            (0.17226891219615936, 0.0, 0.0),
            (0.17647059261798859, 0.0, 0.0),
            (0.18067227303981781, 0.0, 0.0),
            (0.18487395346164703, 0.0, 0.0),
            (0.18907563388347626, 0.0, 0.0),
            (0.19327731430530548, 0.0, 0.0),
            (0.1974789947271347, 0.0, 0.0),
            (0.20168067514896393, 0.0, 0.0),
            (0.20588235557079315, 0.0, 0.0),
            (0.21008403599262238, 0.0, 0.0),
            (0.2142857164144516, 0.0, 0.0),
            (0.21848739683628082, 0.0, 0.0),
            (0.22268907725811005, 0.0, 0.0),
            (0.22689075767993927, 0.0, 0.0),
            (0.23109243810176849, 0.0, 0.0),
            (0.23529411852359772, 0.0, 0.0),
            (0.23949579894542694, 0.0, 0.0),
            (0.24369747936725616, 0.0, 0.0),
            (0.24789915978908539, 0.0, 0.0),
            (0.25210085511207581, 0.0, 0.0),
            (0.25630253553390503, 0.0, 0.0),
            (0.26050421595573425, 0.0, 0.0),
            (0.26470589637756348, 0.0, 0.0),
            (0.2689075767993927, 0.0, 0.0),
            (0.27310925722122192, 0.0, 0.0),
            (0.27731093764305115, 0.0, 0.0),
            (0.28151261806488037, 0.0, 0.0),
            (0.28571429848670959, 0.0, 0.0),
            (0.28991597890853882, 0.0, 0.0),
            (0.29411765933036804, 0.0, 0.0),
            (0.29831933975219727, 0.0, 0.0),
            (0.30252102017402649, 0.0, 0.0),
            (0.30672270059585571, 0.0, 0.0),
            (0.31092438101768494, 0.0, 0.0),
            (0.31512606143951416, 0.0, 0.0),
            (0.31932774186134338, 0.0, 0.0),
            (0.32352942228317261, 0.0, 0.0),
            (0.32773110270500183, 0.0, 0.0),
            (0.33193278312683105, 0.0, 0.0),
            (0.33613446354866028, 0.0, 0.0),
            (0.3403361439704895, 0.0, 0.0),
            (0.34453782439231873, 0.0, 0.0),
            (0.34873950481414795, 0.0, 0.0),
            (0.35294118523597717, 0.0, 0.0),
            (0.3571428656578064, 0.0, 0.0),
            (0.36134454607963562, 0.0, 0.0),
            (0.36554622650146484, 0.0, 0.0),
            (0.36974790692329407, 0.0, 0.0),
            (0.37394958734512329, 0.0, 0.0),
            (0.37815126776695251, 0.0, 0.0),
            (0.38235294818878174, 0.0, 0.0),
            (0.38655462861061096, 0.0, 0.0),
            (0.39075630903244019, 0.0, 0.0),
            (0.39495798945426941, 0.0, 0.0),
            (0.39915966987609863, 0.0, 0.0),
            (0.40336135029792786, 0.0, 0.0),
            (
                0.40756303071975708,
                0.0039215688593685627,
                0.0039215688593685627,
            ),
            (0.4117647111415863, 0.047058824449777603, 0.047058824449777603),
            (0.41596639156341553, 0.066666670143604279, 0.066666670143604279),
            (0.42016807198524475, 0.090196080505847931, 0.090196080505847931),
            (0.42436975240707397, 0.10980392247438431, 0.10980392247438431),
            (0.4285714328289032, 0.12941177189350128, 0.12941177189350128),
            (0.43277311325073242, 0.15294118225574493, 0.15294118225574493),
            (0.43697479367256165, 0.17254902422428131, 0.17254902422428131),
            (0.44117647409439087, 0.19215686619281769, 0.19215686619281769),
            (0.44537815451622009, 0.21568627655506134, 0.21568627655506134),
            (0.44957983493804932, 0.23529411852359772, 0.23529411852359772),
            (0.45378151535987854, 0.25882354378700256, 0.25882354378700256),
            (0.45798319578170776, 0.27843138575553894, 0.27843138575553894),
            (0.46218487620353699, 0.29803922772407532, 0.29803922772407532),
            (0.46638655662536621, 0.32156863808631897, 0.32156863808631897),
            (0.47058823704719543, 0.34117648005485535, 0.34117648005485535),
            (0.47478991746902466, 0.38431373238563538, 0.38431373238563538),
            (0.47899159789085388, 0.40392157435417175, 0.40392157435417175),
            (0.48319327831268311, 0.42745098471641541, 0.42745098471641541),
            (0.48739495873451233, 0.44705882668495178, 0.44705882668495178),
            (0.49159663915634155, 0.46666666865348816, 0.46666666865348816),
            (0.49579831957817078, 0.49019607901573181, 0.49019607901573181),
            (0.5, 0.50980395078659058, 0.50980395078659058),
            (0.50420171022415161, 0.52941179275512695, 0.52941179275512695),
            (0.50840336084365845, 0.55294120311737061, 0.55294120311737061),
            (0.51260507106781006, 0.57254904508590698, 0.57254904508590698),
            (0.51680672168731689, 0.59607845544815063, 0.59607845544815063),
            (0.52100843191146851, 0.61568629741668701, 0.61568629741668701),
            (0.52521008253097534, 0.63529413938522339, 0.63529413938522339),
            (0.52941179275512695, 0.65882354974746704, 0.65882354974746704),
            (0.53361344337463379, 0.67843139171600342, 0.67843139171600342),
            (0.5378151535987854, 0.72156864404678345, 0.72156864404678345),
            (0.54201680421829224, 0.74117648601531982, 0.74117648601531982),
            (0.54621851444244385, 0.76470589637756348, 0.76470589637756348),
            (0.55042016506195068, 0.78431373834609985, 0.78431373834609985),
            (0.55462187528610229, 0.80392158031463623, 0.80392158031463623),
            (0.55882352590560913, 0.82745099067687988, 0.82745099067687988),
            (0.56302523612976074, 0.84705883264541626, 0.84705883264541626),
            (0.56722688674926758, 0.87058824300765991, 0.87058824300765991),
            (0.57142859697341919, 0.89019608497619629, 0.89019608497619629),
            (0.57563024759292603, 0.90980392694473267, 0.90980392694473267),
            (0.57983195781707764, 0.93333333730697632, 0.93333333730697632),
            (0.58403360843658447, 0.9529411792755127, 0.9529411792755127),
            (0.58823531866073608, 0.97254902124404907, 0.97254902124404907),
            (0.59243696928024292, 0.99607843160629272, 0.99607843160629272),
            (0.59663867950439453, 1.0, 1.0),
            (0.60084033012390137, 1.0, 1.0),
            (0.60504204034805298, 1.0, 1.0),
            (0.60924369096755981, 1.0, 1.0),
            (0.61344540119171143, 1.0, 1.0),
            (0.61764705181121826, 1.0, 1.0),
            (0.62184876203536987, 1.0, 1.0),
            (0.62605041265487671, 1.0, 1.0),
            (0.63025212287902832, 1.0, 1.0),
            (0.63445377349853516, 1.0, 1.0),
            (0.63865548372268677, 1.0, 1.0),
            (0.6428571343421936, 1.0, 1.0),
            (0.64705884456634521, 1.0, 1.0),
            (0.65126049518585205, 1.0, 1.0),
            (0.65546220541000366, 1.0, 1.0),
            (0.6596638560295105, 1.0, 1.0),
            (0.66386556625366211, 1.0, 1.0),
            (0.66806721687316895, 1.0, 1.0),
            (0.67226892709732056, 1.0, 1.0),
            (0.67647057771682739, 1.0, 1.0),
            (0.680672287940979, 1.0, 1.0),
            (0.68487393856048584, 1.0, 1.0),
            (0.68907564878463745, 1.0, 1.0),
            (0.69327729940414429, 1.0, 1.0),
            (0.6974790096282959, 1.0, 1.0),
            (0.70168066024780273, 1.0, 1.0),
            (0.70588237047195435, 1.0, 1.0),
            (0.71008402109146118, 1.0, 1.0),
            (0.71428573131561279, 1.0, 1.0),
            (0.71848738193511963, 1.0, 1.0),
            (0.72268909215927124, 1.0, 1.0),
            (0.72689074277877808, 1.0, 1.0),
            (0.73109245300292969, 1.0, 1.0),
            (0.73529410362243652, 1.0, 1.0),
            (0.73949581384658813, 1.0, 1.0),
            (0.74369746446609497, 1.0, 1.0),
            (0.74789917469024658, 1.0, 1.0),
            (0.75210082530975342, 1.0, 1.0),
            (0.75630253553390503, 1.0, 1.0),
            (0.76050418615341187, 1.0, 1.0),
            (0.76470589637756348, 1.0, 1.0),
            (0.76890754699707031, 1.0, 1.0),
            (0.77310925722122192, 1.0, 1.0),
            (0.77731090784072876, 1.0, 1.0),
            (0.78151261806488037, 1.0, 1.0),
            (0.78571426868438721, 1.0, 1.0),
            (0.78991597890853882, 1.0, 1.0),
            (0.79411762952804565, 1.0, 1.0),
            (0.79831933975219727, 1.0, 1.0),
            (0.8025209903717041, 1.0, 1.0),
            (0.80672270059585571, 1.0, 1.0),
            (0.81092435121536255, 1.0, 1.0),
            (0.81512606143951416, 1.0, 1.0),
            (0.819327712059021, 1.0, 1.0),
            (0.82352942228317261, 1.0, 1.0),
            (0.82773107290267944, 1.0, 1.0),
            (0.83193278312683105, 1.0, 1.0),
            (0.83613443374633789, 1.0, 1.0),
            (0.8403361439704895, 1.0, 1.0),
            (0.84453779458999634, 1.0, 1.0),
            (0.84873950481414795, 1.0, 1.0),
            (0.85294115543365479, 1.0, 1.0),
            (0.8571428656578064, 1.0, 1.0),
            (0.86134451627731323, 1.0, 1.0),
            (0.86554622650146484, 1.0, 1.0),
            (0.86974787712097168, 1.0, 1.0),
            (0.87394958734512329, 1.0, 1.0),
            (0.87815123796463013, 1.0, 1.0),
            (0.88235294818878174, 1.0, 1.0),
            (0.88655459880828857, 1.0, 1.0),
            (0.89075630903244019, 1.0, 1.0),
            (0.89495795965194702, 1.0, 1.0),
            (0.89915966987609863, 1.0, 1.0),
            (0.90336132049560547, 1.0, 1.0),
            (0.90756303071975708, 1.0, 1.0),
            (0.91176468133926392, 1.0, 1.0),
            (0.91596639156341553, 1.0, 1.0),
            (0.92016804218292236, 1.0, 1.0),
            (0.92436975240707397, 1.0, 1.0),
            (0.92857140302658081, 1.0, 1.0),
            (0.93277311325073242, 1.0, 1.0),
            (0.93697476387023926, 1.0, 1.0),
            (0.94117647409439087, 1.0, 1.0),
            (0.94537812471389771, 1.0, 1.0),
            (0.94957983493804932, 1.0, 1.0),
            (0.95378148555755615, 1.0, 1.0),
            (0.95798319578170776, 1.0, 1.0),
            (0.9621848464012146, 1.0, 1.0),
            (0.96638655662536621, 0.99607843160629272, 0.99607843160629272),
            (0.97058820724487305, 0.97647058963775635, 0.97647058963775635),
            (0.97478991746902466, 0.9529411792755127, 0.9529411792755127),
            (0.97899156808853149, 0.91372549533843994, 0.91372549533843994),
            (0.98319327831268311, 0.89019608497619629, 0.89019608497619629),
            (0.98739492893218994, 0.87058824300765991, 0.87058824300765991),
            (0.99159663915634155, 0.85098040103912354, 0.85098040103912354),
            (0.99579828977584839, 0.82745099067687988, 0.82745099067687988),
            (1.0, 0.80784314870834351, 0.80784314870834351),
        ],
    )
    return ColorMapper.from_segment_map(_data, range=range, **traits)


def gist_stern(range, **traits):
    """Generator for the 'gist_stern' colormap from GIST."""
    _data = dict(
        red=[
            (0.0, 0.0, 0.0),
            (
                0.0042016808874905109,
                0.070588238537311554,
                0.070588238537311554,
            ),
            (0.0084033617749810219, 0.14117647707462311, 0.14117647707462311),
            (0.012605042196810246, 0.21176470816135406, 0.21176470816135406),
            (0.016806723549962044, 0.28235295414924622, 0.28235295414924622),
            (0.021008403971791267, 0.35294118523597717, 0.35294118523597717),
            (0.025210084393620491, 0.42352941632270813, 0.42352941632270813),
            (0.029411764815449715, 0.49803921580314636, 0.49803921580314636),
            (0.033613447099924088, 0.56862747669219971, 0.56862747669219971),
            (0.037815127521753311, 0.63921570777893066, 0.63921570777893066),
            (0.042016807943582535, 0.78039216995239258, 0.78039216995239258),
            (0.046218488365411758, 0.85098040103912354, 0.85098040103912354),
            (0.050420168787240982, 0.92156863212585449, 0.92156863212585449),
            (0.054621849209070206, 0.99607843160629272, 0.99607843160629272),
            (0.058823529630899429, 0.97647058963775635, 0.97647058963775635),
            (0.063025213778018951, 0.95686274766921997, 0.95686274766921997),
            (0.067226894199848175, 0.93725490570068359, 0.93725490570068359),
            (0.071428574621677399, 0.91764706373214722, 0.91764706373214722),
            (0.075630255043506622, 0.89803922176361084, 0.89803922176361084),
            (0.079831935465335846, 0.87450981140136719, 0.87450981140136719),
            (0.08403361588716507, 0.85490196943283081, 0.85490196943283081),
            (0.088235296308994293, 0.83529412746429443, 0.83529412746429443),
            (0.092436976730823517, 0.81568628549575806, 0.81568628549575806),
            (0.09663865715265274, 0.79607844352722168, 0.79607844352722168),
            (0.10084033757448196, 0.77254903316497803, 0.77254903316497803),
            (0.10504201799631119, 0.75294119119644165, 0.75294119119644165),
            (0.10924369841814041, 0.73333334922790527, 0.73333334922790527),
            (0.11344537883996964, 0.7137255072593689, 0.7137255072593689),
            (0.11764705926179886, 0.69411766529083252, 0.69411766529083252),
            (0.12184873968362808, 0.67450982332229614, 0.67450982332229614),
            (0.1260504275560379, 0.63137257099151611, 0.63137257099151611),
            (0.13025210797786713, 0.61176472902297974, 0.61176472902297974),
            (0.13445378839969635, 0.59215688705444336, 0.59215688705444336),
            (0.13865546882152557, 0.57254904508590698, 0.57254904508590698),
            (0.1428571492433548, 0.54901963472366333, 0.54901963472366333),
            (0.14705882966518402, 0.52941179275512695, 0.52941179275512695),
            (0.15126051008701324, 0.50980395078659058, 0.50980395078659058),
            (0.15546219050884247, 0.49019607901573181, 0.49019607901573181),
            (0.15966387093067169, 0.47058823704719543, 0.47058823704719543),
            (0.16386555135250092, 0.45098039507865906, 0.45098039507865906),
            (0.16806723177433014, 0.42745098471641541, 0.42745098471641541),
            (0.17226891219615936, 0.40784314274787903, 0.40784314274787903),
            (0.17647059261798859, 0.38823530077934265, 0.38823530077934265),
            (0.18067227303981781, 0.36862745881080627, 0.36862745881080627),
            (0.18487395346164703, 0.3490196168422699, 0.3490196168422699),
            (0.18907563388347626, 0.32549020648002625, 0.32549020648002625),
            (0.19327731430530548, 0.30588236451148987, 0.30588236451148987),
            (0.1974789947271347, 0.28627452254295349, 0.28627452254295349),
            (0.20168067514896393, 0.26666668057441711, 0.26666668057441711),
            (0.20588235557079315, 0.24705882370471954, 0.24705882370471954),
            (0.21008403599262238, 0.20392157137393951, 0.20392157137393951),
            (0.2142857164144516, 0.18431372940540314, 0.18431372940540314),
            (0.21848739683628082, 0.16470588743686676, 0.16470588743686676),
            (0.22268907725811005, 0.14509804546833038, 0.14509804546833038),
            (0.22689075767993927, 0.12549020349979401, 0.12549020349979401),
            (0.23109243810176849, 0.10196078568696976, 0.10196078568696976),
            (0.23529411852359772, 0.08235294371843338, 0.08235294371843338),
            (0.23949579894542694, 0.062745101749897003, 0.062745101749897003),
            (0.24369747936725616, 0.043137256056070328, 0.043137256056070328),
            (0.24789915978908539, 0.023529412224888802, 0.023529412224888802),
            (0.25210085511207581, 0.25098040699958801, 0.25098040699958801),
            (0.25630253553390503, 0.25490197539329529, 0.25490197539329529),
            (0.26050421595573425, 0.25882354378700256, 0.25882354378700256),
            (0.26470589637756348, 0.26274511218070984, 0.26274511218070984),
            (0.2689075767993927, 0.26666668057441711, 0.26666668057441711),
            (0.27310925722122192, 0.27058824896812439, 0.27058824896812439),
            (0.27731093764305115, 0.27450981736183167, 0.27450981736183167),
            (0.28151261806488037, 0.27843138575553894, 0.27843138575553894),
            (0.28571429848670959, 0.28235295414924622, 0.28235295414924622),
            (0.28991597890853882, 0.28627452254295349, 0.28627452254295349),
            (0.29411765933036804, 0.29411765933036804, 0.29411765933036804),
            (0.29831933975219727, 0.29803922772407532, 0.29803922772407532),
            (0.30252102017402649, 0.30196079611778259, 0.30196079611778259),
            (0.30672270059585571, 0.30588236451148987, 0.30588236451148987),
            (0.31092438101768494, 0.30980393290519714, 0.30980393290519714),
            (0.31512606143951416, 0.31372550129890442, 0.31372550129890442),
            (0.31932774186134338, 0.31764706969261169, 0.31764706969261169),
            (0.32352942228317261, 0.32156863808631897, 0.32156863808631897),
            (0.32773110270500183, 0.32549020648002625, 0.32549020648002625),
            (0.33193278312683105, 0.32941177487373352, 0.32941177487373352),
            (0.33613446354866028, 0.3333333432674408, 0.3333333432674408),
            (0.3403361439704895, 0.33725491166114807, 0.33725491166114807),
            (0.34453782439231873, 0.34117648005485535, 0.34117648005485535),
            (0.34873950481414795, 0.34509804844856262, 0.34509804844856262),
            (0.35294118523597717, 0.3490196168422699, 0.3490196168422699),
            (0.3571428656578064, 0.35294118523597717, 0.35294118523597717),
            (0.36134454607963562, 0.35686275362968445, 0.35686275362968445),
            (0.36554622650146484, 0.36078432202339172, 0.36078432202339172),
            (0.36974790692329407, 0.364705890417099, 0.364705890417099),
            (0.37394958734512329, 0.36862745881080627, 0.36862745881080627),
            (0.37815126776695251, 0.37647059559822083, 0.37647059559822083),
            (0.38235294818878174, 0.3803921639919281, 0.3803921639919281),
            (0.38655462861061096, 0.38431373238563538, 0.38431373238563538),
            (0.39075630903244019, 0.38823530077934265, 0.38823530077934265),
            (0.39495798945426941, 0.39215686917304993, 0.39215686917304993),
            (0.39915966987609863, 0.3960784375667572, 0.3960784375667572),
            (0.40336135029792786, 0.40000000596046448, 0.40000000596046448),
            (0.40756303071975708, 0.40392157435417175, 0.40392157435417175),
            (0.4117647111415863, 0.40784314274787903, 0.40784314274787903),
            (0.41596639156341553, 0.4117647111415863, 0.4117647111415863),
            (0.42016807198524475, 0.41568627953529358, 0.41568627953529358),
            (0.42436975240707397, 0.41960784792900085, 0.41960784792900085),
            (0.4285714328289032, 0.42352941632270813, 0.42352941632270813),
            (0.43277311325073242, 0.42745098471641541, 0.42745098471641541),
            (0.43697479367256165, 0.43137255311012268, 0.43137255311012268),
            (0.44117647409439087, 0.43529412150382996, 0.43529412150382996),
            (0.44537815451622009, 0.43921568989753723, 0.43921568989753723),
            (0.44957983493804932, 0.44313725829124451, 0.44313725829124451),
            (0.45378151535987854, 0.44705882668495178, 0.44705882668495178),
            (0.45798319578170776, 0.45098039507865906, 0.45098039507865906),
            (0.46218487620353699, 0.45882353186607361, 0.45882353186607361),
            (0.46638655662536621, 0.46274510025978088, 0.46274510025978088),
            (0.47058823704719543, 0.46666666865348816, 0.46666666865348816),
            (0.47478991746902466, 0.47058823704719543, 0.47058823704719543),
            (0.47899159789085388, 0.47450980544090271, 0.47450980544090271),
            (0.48319327831268311, 0.47843137383460999, 0.47843137383460999),
            (0.48739495873451233, 0.48235294222831726, 0.48235294222831726),
            (0.49159663915634155, 0.48627451062202454, 0.48627451062202454),
            (0.49579831957817078, 0.49019607901573181, 0.49019607901573181),
            (0.5, 0.49411764740943909, 0.49411764740943909),
            (0.50420171022415161, 0.50196081399917603, 0.50196081399917603),
            (0.50840336084365845, 0.5058823823928833, 0.5058823823928833),
            (0.51260507106781006, 0.50980395078659058, 0.50980395078659058),
            (0.51680672168731689, 0.51372551918029785, 0.51372551918029785),
            (0.52100843191146851, 0.51764708757400513, 0.51764708757400513),
            (0.52521008253097534, 0.5215686559677124, 0.5215686559677124),
            (0.52941179275512695, 0.52549022436141968, 0.52549022436141968),
            (0.53361344337463379, 0.52941179275512695, 0.52941179275512695),
            (0.5378151535987854, 0.53333336114883423, 0.53333336114883423),
            (0.54201680421829224, 0.5372549295425415, 0.5372549295425415),
            (0.54621851444244385, 0.54509806632995605, 0.54509806632995605),
            (0.55042016506195068, 0.54901963472366333, 0.54901963472366333),
            (0.55462187528610229, 0.55294120311737061, 0.55294120311737061),
            (0.55882352590560913, 0.55686277151107788, 0.55686277151107788),
            (0.56302523612976074, 0.56078433990478516, 0.56078433990478516),
            (0.56722688674926758, 0.56470590829849243, 0.56470590829849243),
            (0.57142859697341919, 0.56862747669219971, 0.56862747669219971),
            (0.57563024759292603, 0.57254904508590698, 0.57254904508590698),
            (0.57983195781707764, 0.57647061347961426, 0.57647061347961426),
            (0.58403360843658447, 0.58039218187332153, 0.58039218187332153),
            (0.58823531866073608, 0.58431375026702881, 0.58431375026702881),
            (0.59243696928024292, 0.58823531866073608, 0.58823531866073608),
            (0.59663867950439453, 0.59215688705444336, 0.59215688705444336),
            (0.60084033012390137, 0.59607845544815063, 0.59607845544815063),
            (0.60504204034805298, 0.60000002384185791, 0.60000002384185791),
            (0.60924369096755981, 0.60392159223556519, 0.60392159223556519),
            (0.61344540119171143, 0.60784316062927246, 0.60784316062927246),
            (0.61764705181121826, 0.61176472902297974, 0.61176472902297974),
            (0.62184876203536987, 0.61568629741668701, 0.61568629741668701),
            (0.62605041265487671, 0.61960786581039429, 0.61960786581039429),
            (0.63025212287902832, 0.62745100259780884, 0.62745100259780884),
            (0.63445377349853516, 0.63137257099151611, 0.63137257099151611),
            (0.63865548372268677, 0.63529413938522339, 0.63529413938522339),
            (0.6428571343421936, 0.63921570777893066, 0.63921570777893066),
            (0.64705884456634521, 0.64313727617263794, 0.64313727617263794),
            (0.65126049518585205, 0.64705884456634521, 0.64705884456634521),
            (0.65546220541000366, 0.65098041296005249, 0.65098041296005249),
            (0.6596638560295105, 0.65490198135375977, 0.65490198135375977),
            (0.66386556625366211, 0.65882354974746704, 0.65882354974746704),
            (0.66806721687316895, 0.66274511814117432, 0.66274511814117432),
            (0.67226892709732056, 0.66666668653488159, 0.66666668653488159),
            (0.67647057771682739, 0.67058825492858887, 0.67058825492858887),
            (0.680672287940979, 0.67450982332229614, 0.67450982332229614),
            (0.68487393856048584, 0.67843139171600342, 0.67843139171600342),
            (0.68907564878463745, 0.68235296010971069, 0.68235296010971069),
            (0.69327729940414429, 0.68627452850341797, 0.68627452850341797),
            (0.6974790096282959, 0.69019609689712524, 0.69019609689712524),
            (0.70168066024780273, 0.69411766529083252, 0.69411766529083252),
            (0.70588237047195435, 0.69803923368453979, 0.69803923368453979),
            (0.71008402109146118, 0.70196080207824707, 0.70196080207824707),
            (0.71428573131561279, 0.70980393886566162, 0.70980393886566162),
            (0.71848738193511963, 0.7137255072593689, 0.7137255072593689),
            (0.72268909215927124, 0.71764707565307617, 0.71764707565307617),
            (0.72689074277877808, 0.72156864404678345, 0.72156864404678345),
            (0.73109245300292969, 0.72549021244049072, 0.72549021244049072),
            (0.73529410362243652, 0.729411780834198, 0.729411780834198),
            (0.73949581384658813, 0.73333334922790527, 0.73333334922790527),
            (0.74369746446609497, 0.73725491762161255, 0.73725491762161255),
            (0.74789917469024658, 0.74117648601531982, 0.74117648601531982),
            (0.75210082530975342, 0.7450980544090271, 0.7450980544090271),
            (0.75630253553390503, 0.75294119119644165, 0.75294119119644165),
            (0.76050418615341187, 0.75686275959014893, 0.75686275959014893),
            (0.76470589637756348, 0.7607843279838562, 0.7607843279838562),
            (0.76890754699707031, 0.76470589637756348, 0.76470589637756348),
            (0.77310925722122192, 0.76862746477127075, 0.76862746477127075),
            (0.77731090784072876, 0.77254903316497803, 0.77254903316497803),
            (0.78151261806488037, 0.7764706015586853, 0.7764706015586853),
            (0.78571426868438721, 0.78039216995239258, 0.78039216995239258),
            (0.78991597890853882, 0.78431373834609985, 0.78431373834609985),
            (0.79411762952804565, 0.78823530673980713, 0.78823530673980713),
            (0.79831933975219727, 0.79607844352722168, 0.79607844352722168),
            (0.8025209903717041, 0.80000001192092896, 0.80000001192092896),
            (0.80672270059585571, 0.80392158031463623, 0.80392158031463623),
            (0.81092435121536255, 0.80784314870834351, 0.80784314870834351),
            (0.81512606143951416, 0.81176471710205078, 0.81176471710205078),
            (0.819327712059021, 0.81568628549575806, 0.81568628549575806),
            (0.82352942228317261, 0.81960785388946533, 0.81960785388946533),
            (0.82773107290267944, 0.82352942228317261, 0.82352942228317261),
            (0.83193278312683105, 0.82745099067687988, 0.82745099067687988),
            (0.83613443374633789, 0.83137255907058716, 0.83137255907058716),
            (0.8403361439704895, 0.83529412746429443, 0.83529412746429443),
            (0.84453779458999634, 0.83921569585800171, 0.83921569585800171),
            (0.84873950481414795, 0.84313726425170898, 0.84313726425170898),
            (0.85294115543365479, 0.84705883264541626, 0.84705883264541626),
            (0.8571428656578064, 0.85098040103912354, 0.85098040103912354),
            (0.86134451627731323, 0.85490196943283081, 0.85490196943283081),
            (0.86554622650146484, 0.85882353782653809, 0.85882353782653809),
            (0.86974787712097168, 0.86274510622024536, 0.86274510622024536),
            (0.87394958734512329, 0.86666667461395264, 0.86666667461395264),
            (0.87815123796463013, 0.87058824300765991, 0.87058824300765991),
            (0.88235294818878174, 0.87843137979507446, 0.87843137979507446),
            (0.88655459880828857, 0.88235294818878174, 0.88235294818878174),
            (0.89075630903244019, 0.88627451658248901, 0.88627451658248901),
            (0.89495795965194702, 0.89019608497619629, 0.89019608497619629),
            (0.89915966987609863, 0.89411765336990356, 0.89411765336990356),
            (0.90336132049560547, 0.89803922176361084, 0.89803922176361084),
            (0.90756303071975708, 0.90196079015731812, 0.90196079015731812),
            (0.91176468133926392, 0.90588235855102539, 0.90588235855102539),
            (0.91596639156341553, 0.90980392694473267, 0.90980392694473267),
            (0.92016804218292236, 0.91372549533843994, 0.91372549533843994),
            (0.92436975240707397, 0.91764706373214722, 0.91764706373214722),
            (0.92857140302658081, 0.92156863212585449, 0.92156863212585449),
            (0.93277311325073242, 0.92549020051956177, 0.92549020051956177),
            (0.93697476387023926, 0.92941176891326904, 0.92941176891326904),
            (0.94117647409439087, 0.93333333730697632, 0.93333333730697632),
            (0.94537812471389771, 0.93725490570068359, 0.93725490570068359),
            (0.94957983493804932, 0.94117647409439087, 0.94117647409439087),
            (0.95378148555755615, 0.94509804248809814, 0.94509804248809814),
            (0.95798319578170776, 0.94901961088180542, 0.94901961088180542),
            (0.9621848464012146, 0.9529411792755127, 0.9529411792755127),
            (0.96638655662536621, 0.96078431606292725, 0.96078431606292725),
            (0.97058820724487305, 0.96470588445663452, 0.96470588445663452),
            (0.97478991746902466, 0.9686274528503418, 0.9686274528503418),
            (0.97899156808853149, 0.97254902124404907, 0.97254902124404907),
            (0.98319327831268311, 0.97647058963775635, 0.97647058963775635),
            (0.98739492893218994, 0.98039215803146362, 0.98039215803146362),
            (0.99159663915634155, 0.9843137264251709, 0.9843137264251709),
            (0.99579828977584839, 0.98823529481887817, 0.98823529481887817),
            (1.0, 0.99215686321258545, 0.99215686321258545),
        ],
        green=[
            (0.0, 0.0, 0.0),
            (
                0.0042016808874905109,
                0.0039215688593685627,
                0.0039215688593685627,
            ),
            (
                0.0084033617749810219,
                0.0078431377187371254,
                0.0078431377187371254,
            ),
            (0.012605042196810246, 0.011764706112444401, 0.011764706112444401),
            (0.016806723549962044, 0.015686275437474251, 0.015686275437474251),
            (0.021008403971791267, 0.019607843831181526, 0.019607843831181526),
            (0.025210084393620491, 0.023529412224888802, 0.023529412224888802),
            (0.029411764815449715, 0.027450980618596077, 0.027450980618596077),
            (0.033613447099924088, 0.031372550874948502, 0.031372550874948502),
            (0.037815127521753311, 0.035294119268655777, 0.035294119268655777),
            (0.042016807943582535, 0.043137256056070328, 0.043137256056070328),
            (0.046218488365411758, 0.047058824449777603, 0.047058824449777603),
            (0.050420168787240982, 0.050980392843484879, 0.050980392843484879),
            (0.054621849209070206, 0.054901961237192154, 0.054901961237192154),
            (0.058823529630899429, 0.058823529630899429, 0.058823529630899429),
            (0.063025213778018951, 0.062745101749897003, 0.062745101749897003),
            (0.067226894199848175, 0.066666670143604279, 0.066666670143604279),
            (0.071428574621677399, 0.070588238537311554, 0.070588238537311554),
            (0.075630255043506622, 0.074509806931018829, 0.074509806931018829),
            (0.079831935465335846, 0.078431375324726105, 0.078431375324726105),
            (0.08403361588716507, 0.08235294371843338, 0.08235294371843338),
            (0.088235296308994293, 0.086274512112140656, 0.086274512112140656),
            (0.092436976730823517, 0.090196080505847931, 0.090196080505847931),
            (0.09663865715265274, 0.094117648899555206, 0.094117648899555206),
            (0.10084033757448196, 0.098039217293262482, 0.098039217293262482),
            (0.10504201799631119, 0.10196078568696976, 0.10196078568696976),
            (0.10924369841814041, 0.10588235408067703, 0.10588235408067703),
            (0.11344537883996964, 0.10980392247438431, 0.10980392247438431),
            (0.11764705926179886, 0.11372549086809158, 0.11372549086809158),
            (0.12184873968362808, 0.11764705926179886, 0.11764705926179886),
            (0.1260504275560379, 0.12549020349979401, 0.12549020349979401),
            (0.13025210797786713, 0.12941177189350128, 0.12941177189350128),
            (0.13445378839969635, 0.13333334028720856, 0.13333334028720856),
            (0.13865546882152557, 0.13725490868091583, 0.13725490868091583),
            (0.1428571492433548, 0.14117647707462311, 0.14117647707462311),
            (0.14705882966518402, 0.14509804546833038, 0.14509804546833038),
            (0.15126051008701324, 0.14901961386203766, 0.14901961386203766),
            (0.15546219050884247, 0.15294118225574493, 0.15294118225574493),
            (0.15966387093067169, 0.15686275064945221, 0.15686275064945221),
            (0.16386555135250092, 0.16078431904315948, 0.16078431904315948),
            (0.16806723177433014, 0.16470588743686676, 0.16470588743686676),
            (0.17226891219615936, 0.16862745583057404, 0.16862745583057404),
            (0.17647059261798859, 0.17254902422428131, 0.17254902422428131),
            (0.18067227303981781, 0.17647059261798859, 0.17647059261798859),
            (0.18487395346164703, 0.18039216101169586, 0.18039216101169586),
            (0.18907563388347626, 0.18431372940540314, 0.18431372940540314),
            (0.19327731430530548, 0.18823529779911041, 0.18823529779911041),
            (0.1974789947271347, 0.19215686619281769, 0.19215686619281769),
            (0.20168067514896393, 0.19607843458652496, 0.19607843458652496),
            (0.20588235557079315, 0.20000000298023224, 0.20000000298023224),
            (0.21008403599262238, 0.20784313976764679, 0.20784313976764679),
            (0.2142857164144516, 0.21176470816135406, 0.21176470816135406),
            (0.21848739683628082, 0.21568627655506134, 0.21568627655506134),
            (0.22268907725811005, 0.21960784494876862, 0.21960784494876862),
            (0.22689075767993927, 0.22352941334247589, 0.22352941334247589),
            (0.23109243810176849, 0.22745098173618317, 0.22745098173618317),
            (0.23529411852359772, 0.23137255012989044, 0.23137255012989044),
            (0.23949579894542694, 0.23529411852359772, 0.23529411852359772),
            (0.24369747936725616, 0.23921568691730499, 0.23921568691730499),
            (0.24789915978908539, 0.24313725531101227, 0.24313725531101227),
            (0.25210085511207581, 0.25098040699958801, 0.25098040699958801),
            (0.25630253553390503, 0.25490197539329529, 0.25490197539329529),
            (0.26050421595573425, 0.25882354378700256, 0.25882354378700256),
            (0.26470589637756348, 0.26274511218070984, 0.26274511218070984),
            (0.2689075767993927, 0.26666668057441711, 0.26666668057441711),
            (0.27310925722122192, 0.27058824896812439, 0.27058824896812439),
            (0.27731093764305115, 0.27450981736183167, 0.27450981736183167),
            (0.28151261806488037, 0.27843138575553894, 0.27843138575553894),
            (0.28571429848670959, 0.28235295414924622, 0.28235295414924622),
            (0.28991597890853882, 0.28627452254295349, 0.28627452254295349),
            (0.29411765933036804, 0.29411765933036804, 0.29411765933036804),
            (0.29831933975219727, 0.29803922772407532, 0.29803922772407532),
            (0.30252102017402649, 0.30196079611778259, 0.30196079611778259),
            (0.30672270059585571, 0.30588236451148987, 0.30588236451148987),
            (0.31092438101768494, 0.30980393290519714, 0.30980393290519714),
            (0.31512606143951416, 0.31372550129890442, 0.31372550129890442),
            (0.31932774186134338, 0.31764706969261169, 0.31764706969261169),
            (0.32352942228317261, 0.32156863808631897, 0.32156863808631897),
            (0.32773110270500183, 0.32549020648002625, 0.32549020648002625),
            (0.33193278312683105, 0.32941177487373352, 0.32941177487373352),
            (0.33613446354866028, 0.3333333432674408, 0.3333333432674408),
            (0.3403361439704895, 0.33725491166114807, 0.33725491166114807),
            (0.34453782439231873, 0.34117648005485535, 0.34117648005485535),
            (0.34873950481414795, 0.34509804844856262, 0.34509804844856262),
            (0.35294118523597717, 0.3490196168422699, 0.3490196168422699),
            (0.3571428656578064, 0.35294118523597717, 0.35294118523597717),
            (0.36134454607963562, 0.35686275362968445, 0.35686275362968445),
            (0.36554622650146484, 0.36078432202339172, 0.36078432202339172),
            (0.36974790692329407, 0.364705890417099, 0.364705890417099),
            (0.37394958734512329, 0.36862745881080627, 0.36862745881080627),
            (0.37815126776695251, 0.37647059559822083, 0.37647059559822083),
            (0.38235294818878174, 0.3803921639919281, 0.3803921639919281),
            (0.38655462861061096, 0.38431373238563538, 0.38431373238563538),
            (0.39075630903244019, 0.38823530077934265, 0.38823530077934265),
            (0.39495798945426941, 0.39215686917304993, 0.39215686917304993),
            (0.39915966987609863, 0.3960784375667572, 0.3960784375667572),
            (0.40336135029792786, 0.40000000596046448, 0.40000000596046448),
            (0.40756303071975708, 0.40392157435417175, 0.40392157435417175),
            (0.4117647111415863, 0.40784314274787903, 0.40784314274787903),
            (0.41596639156341553, 0.4117647111415863, 0.4117647111415863),
            (0.42016807198524475, 0.41568627953529358, 0.41568627953529358),
            (0.42436975240707397, 0.41960784792900085, 0.41960784792900085),
            (0.4285714328289032, 0.42352941632270813, 0.42352941632270813),
            (0.43277311325073242, 0.42745098471641541, 0.42745098471641541),
            (0.43697479367256165, 0.43137255311012268, 0.43137255311012268),
            (0.44117647409439087, 0.43529412150382996, 0.43529412150382996),
            (0.44537815451622009, 0.43921568989753723, 0.43921568989753723),
            (0.44957983493804932, 0.44313725829124451, 0.44313725829124451),
            (0.45378151535987854, 0.44705882668495178, 0.44705882668495178),
            (0.45798319578170776, 0.45098039507865906, 0.45098039507865906),
            (0.46218487620353699, 0.45882353186607361, 0.45882353186607361),
            (0.46638655662536621, 0.46274510025978088, 0.46274510025978088),
            (0.47058823704719543, 0.46666666865348816, 0.46666666865348816),
            (0.47478991746902466, 0.47058823704719543, 0.47058823704719543),
            (0.47899159789085388, 0.47450980544090271, 0.47450980544090271),
            (0.48319327831268311, 0.47843137383460999, 0.47843137383460999),
            (0.48739495873451233, 0.48235294222831726, 0.48235294222831726),
            (0.49159663915634155, 0.48627451062202454, 0.48627451062202454),
            (0.49579831957817078, 0.49019607901573181, 0.49019607901573181),
            (0.5, 0.49411764740943909, 0.49411764740943909),
            (0.50420171022415161, 0.50196081399917603, 0.50196081399917603),
            (0.50840336084365845, 0.5058823823928833, 0.5058823823928833),
            (0.51260507106781006, 0.50980395078659058, 0.50980395078659058),
            (0.51680672168731689, 0.51372551918029785, 0.51372551918029785),
            (0.52100843191146851, 0.51764708757400513, 0.51764708757400513),
            (0.52521008253097534, 0.5215686559677124, 0.5215686559677124),
            (0.52941179275512695, 0.52549022436141968, 0.52549022436141968),
            (0.53361344337463379, 0.52941179275512695, 0.52941179275512695),
            (0.5378151535987854, 0.53333336114883423, 0.53333336114883423),
            (0.54201680421829224, 0.5372549295425415, 0.5372549295425415),
            (0.54621851444244385, 0.54509806632995605, 0.54509806632995605),
            (0.55042016506195068, 0.54901963472366333, 0.54901963472366333),
            (0.55462187528610229, 0.55294120311737061, 0.55294120311737061),
            (0.55882352590560913, 0.55686277151107788, 0.55686277151107788),
            (0.56302523612976074, 0.56078433990478516, 0.56078433990478516),
            (0.56722688674926758, 0.56470590829849243, 0.56470590829849243),
            (0.57142859697341919, 0.56862747669219971, 0.56862747669219971),
            (0.57563024759292603, 0.57254904508590698, 0.57254904508590698),
            (0.57983195781707764, 0.57647061347961426, 0.57647061347961426),
            (0.58403360843658447, 0.58039218187332153, 0.58039218187332153),
            (0.58823531866073608, 0.58431375026702881, 0.58431375026702881),
            (0.59243696928024292, 0.58823531866073608, 0.58823531866073608),
            (0.59663867950439453, 0.59215688705444336, 0.59215688705444336),
            (0.60084033012390137, 0.59607845544815063, 0.59607845544815063),
            (0.60504204034805298, 0.60000002384185791, 0.60000002384185791),
            (0.60924369096755981, 0.60392159223556519, 0.60392159223556519),
            (0.61344540119171143, 0.60784316062927246, 0.60784316062927246),
            (0.61764705181121826, 0.61176472902297974, 0.61176472902297974),
            (0.62184876203536987, 0.61568629741668701, 0.61568629741668701),
            (0.62605041265487671, 0.61960786581039429, 0.61960786581039429),
            (0.63025212287902832, 0.62745100259780884, 0.62745100259780884),
            (0.63445377349853516, 0.63137257099151611, 0.63137257099151611),
            (0.63865548372268677, 0.63529413938522339, 0.63529413938522339),
            (0.6428571343421936, 0.63921570777893066, 0.63921570777893066),
            (0.64705884456634521, 0.64313727617263794, 0.64313727617263794),
            (0.65126049518585205, 0.64705884456634521, 0.64705884456634521),
            (0.65546220541000366, 0.65098041296005249, 0.65098041296005249),
            (0.6596638560295105, 0.65490198135375977, 0.65490198135375977),
            (0.66386556625366211, 0.65882354974746704, 0.65882354974746704),
            (0.66806721687316895, 0.66274511814117432, 0.66274511814117432),
            (0.67226892709732056, 0.66666668653488159, 0.66666668653488159),
            (0.67647057771682739, 0.67058825492858887, 0.67058825492858887),
            (0.680672287940979, 0.67450982332229614, 0.67450982332229614),
            (0.68487393856048584, 0.67843139171600342, 0.67843139171600342),
            (0.68907564878463745, 0.68235296010971069, 0.68235296010971069),
            (0.69327729940414429, 0.68627452850341797, 0.68627452850341797),
            (0.6974790096282959, 0.69019609689712524, 0.69019609689712524),
            (0.70168066024780273, 0.69411766529083252, 0.69411766529083252),
            (0.70588237047195435, 0.69803923368453979, 0.69803923368453979),
            (0.71008402109146118, 0.70196080207824707, 0.70196080207824707),
            (0.71428573131561279, 0.70980393886566162, 0.70980393886566162),
            (0.71848738193511963, 0.7137255072593689, 0.7137255072593689),
            (0.72268909215927124, 0.71764707565307617, 0.71764707565307617),
            (0.72689074277877808, 0.72156864404678345, 0.72156864404678345),
            (0.73109245300292969, 0.72549021244049072, 0.72549021244049072),
            (0.73529410362243652, 0.729411780834198, 0.729411780834198),
            (0.73949581384658813, 0.73333334922790527, 0.73333334922790527),
            (0.74369746446609497, 0.73725491762161255, 0.73725491762161255),
            (0.74789917469024658, 0.74117648601531982, 0.74117648601531982),
            (0.75210082530975342, 0.7450980544090271, 0.7450980544090271),
            (0.75630253553390503, 0.75294119119644165, 0.75294119119644165),
            (0.76050418615341187, 0.75686275959014893, 0.75686275959014893),
            (0.76470589637756348, 0.7607843279838562, 0.7607843279838562),
            (0.76890754699707031, 0.76470589637756348, 0.76470589637756348),
            (0.77310925722122192, 0.76862746477127075, 0.76862746477127075),
            (0.77731090784072876, 0.77254903316497803, 0.77254903316497803),
            (0.78151261806488037, 0.7764706015586853, 0.7764706015586853),
            (0.78571426868438721, 0.78039216995239258, 0.78039216995239258),
            (0.78991597890853882, 0.78431373834609985, 0.78431373834609985),
            (0.79411762952804565, 0.78823530673980713, 0.78823530673980713),
            (0.79831933975219727, 0.79607844352722168, 0.79607844352722168),
            (0.8025209903717041, 0.80000001192092896, 0.80000001192092896),
            (0.80672270059585571, 0.80392158031463623, 0.80392158031463623),
            (0.81092435121536255, 0.80784314870834351, 0.80784314870834351),
            (0.81512606143951416, 0.81176471710205078, 0.81176471710205078),
            (0.819327712059021, 0.81568628549575806, 0.81568628549575806),
            (0.82352942228317261, 0.81960785388946533, 0.81960785388946533),
            (0.82773107290267944, 0.82352942228317261, 0.82352942228317261),
            (0.83193278312683105, 0.82745099067687988, 0.82745099067687988),
            (0.83613443374633789, 0.83137255907058716, 0.83137255907058716),
            (0.8403361439704895, 0.83529412746429443, 0.83529412746429443),
            (0.84453779458999634, 0.83921569585800171, 0.83921569585800171),
            (0.84873950481414795, 0.84313726425170898, 0.84313726425170898),
            (0.85294115543365479, 0.84705883264541626, 0.84705883264541626),
            (0.8571428656578064, 0.85098040103912354, 0.85098040103912354),
            (0.86134451627731323, 0.85490196943283081, 0.85490196943283081),
            (0.86554622650146484, 0.85882353782653809, 0.85882353782653809),
            (0.86974787712097168, 0.86274510622024536, 0.86274510622024536),
            (0.87394958734512329, 0.86666667461395264, 0.86666667461395264),
            (0.87815123796463013, 0.87058824300765991, 0.87058824300765991),
            (0.88235294818878174, 0.87843137979507446, 0.87843137979507446),
            (0.88655459880828857, 0.88235294818878174, 0.88235294818878174),
            (0.89075630903244019, 0.88627451658248901, 0.88627451658248901),
            (0.89495795965194702, 0.89019608497619629, 0.89019608497619629),
            (0.89915966987609863, 0.89411765336990356, 0.89411765336990356),
            (0.90336132049560547, 0.89803922176361084, 0.89803922176361084),
            (0.90756303071975708, 0.90196079015731812, 0.90196079015731812),
            (0.91176468133926392, 0.90588235855102539, 0.90588235855102539),
            (0.91596639156341553, 0.90980392694473267, 0.90980392694473267),
            (0.92016804218292236, 0.91372549533843994, 0.91372549533843994),
            (0.92436975240707397, 0.91764706373214722, 0.91764706373214722),
            (0.92857140302658081, 0.92156863212585449, 0.92156863212585449),
            (0.93277311325073242, 0.92549020051956177, 0.92549020051956177),
            (0.93697476387023926, 0.92941176891326904, 0.92941176891326904),
            (0.94117647409439087, 0.93333333730697632, 0.93333333730697632),
            (0.94537812471389771, 0.93725490570068359, 0.93725490570068359),
            (0.94957983493804932, 0.94117647409439087, 0.94117647409439087),
            (0.95378148555755615, 0.94509804248809814, 0.94509804248809814),
            (0.95798319578170776, 0.94901961088180542, 0.94901961088180542),
            (0.9621848464012146, 0.9529411792755127, 0.9529411792755127),
            (0.96638655662536621, 0.96078431606292725, 0.96078431606292725),
            (0.97058820724487305, 0.96470588445663452, 0.96470588445663452),
            (0.97478991746902466, 0.9686274528503418, 0.9686274528503418),
            (0.97899156808853149, 0.97254902124404907, 0.97254902124404907),
            (0.98319327831268311, 0.97647058963775635, 0.97647058963775635),
            (0.98739492893218994, 0.98039215803146362, 0.98039215803146362),
            (0.99159663915634155, 0.9843137264251709, 0.9843137264251709),
            (0.99579828977584839, 0.98823529481887817, 0.98823529481887817),
            (1.0, 0.99215686321258545, 0.99215686321258545),
        ],
        blue=[
            (0.0, 0.0, 0.0),
            (
                0.0042016808874905109,
                0.0039215688593685627,
                0.0039215688593685627,
            ),
            (
                0.0084033617749810219,
                0.011764706112444401,
                0.011764706112444401,
            ),
            (0.012605042196810246, 0.019607843831181526, 0.019607843831181526),
            (0.016806723549962044, 0.027450980618596077, 0.027450980618596077),
            (0.021008403971791267, 0.035294119268655777, 0.035294119268655777),
            (0.025210084393620491, 0.043137256056070328, 0.043137256056070328),
            (0.029411764815449715, 0.050980392843484879, 0.050980392843484879),
            (0.033613447099924088, 0.058823529630899429, 0.058823529630899429),
            (0.037815127521753311, 0.066666670143604279, 0.066666670143604279),
            (0.042016807943582535, 0.08235294371843338, 0.08235294371843338),
            (0.046218488365411758, 0.090196080505847931, 0.090196080505847931),
            (0.050420168787240982, 0.098039217293262482, 0.098039217293262482),
            (0.054621849209070206, 0.10588235408067703, 0.10588235408067703),
            (0.058823529630899429, 0.11372549086809158, 0.11372549086809158),
            (0.063025213778018951, 0.12156862765550613, 0.12156862765550613),
            (0.067226894199848175, 0.12941177189350128, 0.12941177189350128),
            (0.071428574621677399, 0.13725490868091583, 0.13725490868091583),
            (0.075630255043506622, 0.14509804546833038, 0.14509804546833038),
            (0.079831935465335846, 0.15294118225574493, 0.15294118225574493),
            (0.08403361588716507, 0.16078431904315948, 0.16078431904315948),
            (0.088235296308994293, 0.16862745583057404, 0.16862745583057404),
            (0.092436976730823517, 0.17647059261798859, 0.17647059261798859),
            (0.09663865715265274, 0.18431372940540314, 0.18431372940540314),
            (0.10084033757448196, 0.19215686619281769, 0.19215686619281769),
            (0.10504201799631119, 0.20000000298023224, 0.20000000298023224),
            (0.10924369841814041, 0.20784313976764679, 0.20784313976764679),
            (0.11344537883996964, 0.21568627655506134, 0.21568627655506134),
            (0.11764705926179886, 0.22352941334247589, 0.22352941334247589),
            (0.12184873968362808, 0.23137255012989044, 0.23137255012989044),
            (0.1260504275560379, 0.24705882370471954, 0.24705882370471954),
            (0.13025210797786713, 0.25490197539329529, 0.25490197539329529),
            (0.13445378839969635, 0.26274511218070984, 0.26274511218070984),
            (0.13865546882152557, 0.27058824896812439, 0.27058824896812439),
            (0.1428571492433548, 0.27843138575553894, 0.27843138575553894),
            (0.14705882966518402, 0.28627452254295349, 0.28627452254295349),
            (0.15126051008701324, 0.29411765933036804, 0.29411765933036804),
            (0.15546219050884247, 0.30196079611778259, 0.30196079611778259),
            (0.15966387093067169, 0.30980393290519714, 0.30980393290519714),
            (0.16386555135250092, 0.31764706969261169, 0.31764706969261169),
            (0.16806723177433014, 0.32549020648002625, 0.32549020648002625),
            (0.17226891219615936, 0.3333333432674408, 0.3333333432674408),
            (0.17647059261798859, 0.34117648005485535, 0.34117648005485535),
            (0.18067227303981781, 0.3490196168422699, 0.3490196168422699),
            (0.18487395346164703, 0.35686275362968445, 0.35686275362968445),
            (0.18907563388347626, 0.364705890417099, 0.364705890417099),
            (0.19327731430530548, 0.37254902720451355, 0.37254902720451355),
            (0.1974789947271347, 0.3803921639919281, 0.3803921639919281),
            (0.20168067514896393, 0.38823530077934265, 0.38823530077934265),
            (0.20588235557079315, 0.3960784375667572, 0.3960784375667572),
            (0.21008403599262238, 0.4117647111415863, 0.4117647111415863),
            (0.2142857164144516, 0.41960784792900085, 0.41960784792900085),
            (0.21848739683628082, 0.42745098471641541, 0.42745098471641541),
            (0.22268907725811005, 0.43529412150382996, 0.43529412150382996),
            (0.22689075767993927, 0.44313725829124451, 0.44313725829124451),
            (0.23109243810176849, 0.45098039507865906, 0.45098039507865906),
            (0.23529411852359772, 0.45882353186607361, 0.45882353186607361),
            (0.23949579894542694, 0.46666666865348816, 0.46666666865348816),
            (0.24369747936725616, 0.47450980544090271, 0.47450980544090271),
            (0.24789915978908539, 0.48235294222831726, 0.48235294222831726),
            (0.25210085511207581, 0.49803921580314636, 0.49803921580314636),
            (0.25630253553390503, 0.5058823823928833, 0.5058823823928833),
            (0.26050421595573425, 0.51372551918029785, 0.51372551918029785),
            (0.26470589637756348, 0.5215686559677124, 0.5215686559677124),
            (0.2689075767993927, 0.52941179275512695, 0.52941179275512695),
            (0.27310925722122192, 0.5372549295425415, 0.5372549295425415),
            (0.27731093764305115, 0.54509806632995605, 0.54509806632995605),
            (0.28151261806488037, 0.55294120311737061, 0.55294120311737061),
            (0.28571429848670959, 0.56078433990478516, 0.56078433990478516),
            (0.28991597890853882, 0.56862747669219971, 0.56862747669219971),
            (0.29411765933036804, 0.58431375026702881, 0.58431375026702881),
            (0.29831933975219727, 0.59215688705444336, 0.59215688705444336),
            (0.30252102017402649, 0.60000002384185791, 0.60000002384185791),
            (0.30672270059585571, 0.60784316062927246, 0.60784316062927246),
            (0.31092438101768494, 0.61568629741668701, 0.61568629741668701),
            (0.31512606143951416, 0.62352943420410156, 0.62352943420410156),
            (0.31932774186134338, 0.63137257099151611, 0.63137257099151611),
            (0.32352942228317261, 0.63921570777893066, 0.63921570777893066),
            (0.32773110270500183, 0.64705884456634521, 0.64705884456634521),
            (0.33193278312683105, 0.65490198135375977, 0.65490198135375977),
            (0.33613446354866028, 0.66274511814117432, 0.66274511814117432),
            (0.3403361439704895, 0.67058825492858887, 0.67058825492858887),
            (0.34453782439231873, 0.67843139171600342, 0.67843139171600342),
            (0.34873950481414795, 0.68627452850341797, 0.68627452850341797),
            (0.35294118523597717, 0.69411766529083252, 0.69411766529083252),
            (0.3571428656578064, 0.70196080207824707, 0.70196080207824707),
            (0.36134454607963562, 0.70980393886566162, 0.70980393886566162),
            (0.36554622650146484, 0.71764707565307617, 0.71764707565307617),
            (0.36974790692329407, 0.72549021244049072, 0.72549021244049072),
            (0.37394958734512329, 0.73333334922790527, 0.73333334922790527),
            (0.37815126776695251, 0.74901962280273438, 0.74901962280273438),
            (0.38235294818878174, 0.75686275959014893, 0.75686275959014893),
            (0.38655462861061096, 0.76470589637756348, 0.76470589637756348),
            (0.39075630903244019, 0.77254903316497803, 0.77254903316497803),
            (0.39495798945426941, 0.78039216995239258, 0.78039216995239258),
            (0.39915966987609863, 0.78823530673980713, 0.78823530673980713),
            (0.40336135029792786, 0.79607844352722168, 0.79607844352722168),
            (0.40756303071975708, 0.80392158031463623, 0.80392158031463623),
            (0.4117647111415863, 0.81176471710205078, 0.81176471710205078),
            (0.41596639156341553, 0.81960785388946533, 0.81960785388946533),
            (0.42016807198524475, 0.82745099067687988, 0.82745099067687988),
            (0.42436975240707397, 0.83529412746429443, 0.83529412746429443),
            (0.4285714328289032, 0.84313726425170898, 0.84313726425170898),
            (0.43277311325073242, 0.85098040103912354, 0.85098040103912354),
            (0.43697479367256165, 0.85882353782653809, 0.85882353782653809),
            (0.44117647409439087, 0.86666667461395264, 0.86666667461395264),
            (0.44537815451622009, 0.87450981140136719, 0.87450981140136719),
            (0.44957983493804932, 0.88235294818878174, 0.88235294818878174),
            (0.45378151535987854, 0.89019608497619629, 0.89019608497619629),
            (0.45798319578170776, 0.89803922176361084, 0.89803922176361084),
            (0.46218487620353699, 0.91372549533843994, 0.91372549533843994),
            (0.46638655662536621, 0.92156863212585449, 0.92156863212585449),
            (0.47058823704719543, 0.92941176891326904, 0.92941176891326904),
            (0.47478991746902466, 0.93725490570068359, 0.93725490570068359),
            (0.47899159789085388, 0.94509804248809814, 0.94509804248809814),
            (0.48319327831268311, 0.9529411792755127, 0.9529411792755127),
            (0.48739495873451233, 0.96078431606292725, 0.96078431606292725),
            (0.49159663915634155, 0.9686274528503418, 0.9686274528503418),
            (0.49579831957817078, 0.97647058963775635, 0.97647058963775635),
            (0.5, 0.9843137264251709, 0.9843137264251709),
            (0.50420171022415161, 1.0, 1.0),
            (0.50840336084365845, 0.9843137264251709, 0.9843137264251709),
            (0.51260507106781006, 0.9686274528503418, 0.9686274528503418),
            (0.51680672168731689, 0.9529411792755127, 0.9529411792755127),
            (0.52100843191146851, 0.93333333730697632, 0.93333333730697632),
            (0.52521008253097534, 0.91764706373214722, 0.91764706373214722),
            (0.52941179275512695, 0.90196079015731812, 0.90196079015731812),
            (0.53361344337463379, 0.88627451658248901, 0.88627451658248901),
            (0.5378151535987854, 0.86666667461395264, 0.86666667461395264),
            (0.54201680421829224, 0.85098040103912354, 0.85098040103912354),
            (0.54621851444244385, 0.81960785388946533, 0.81960785388946533),
            (0.55042016506195068, 0.80000001192092896, 0.80000001192092896),
            (0.55462187528610229, 0.78431373834609985, 0.78431373834609985),
            (0.55882352590560913, 0.76862746477127075, 0.76862746477127075),
            (0.56302523612976074, 0.75294119119644165, 0.75294119119644165),
            (0.56722688674926758, 0.73333334922790527, 0.73333334922790527),
            (0.57142859697341919, 0.71764707565307617, 0.71764707565307617),
            (0.57563024759292603, 0.70196080207824707, 0.70196080207824707),
            (0.57983195781707764, 0.68627452850341797, 0.68627452850341797),
            (0.58403360843658447, 0.66666668653488159, 0.66666668653488159),
            (0.58823531866073608, 0.65098041296005249, 0.65098041296005249),
            (0.59243696928024292, 0.63529413938522339, 0.63529413938522339),
            (0.59663867950439453, 0.61960786581039429, 0.61960786581039429),
            (0.60084033012390137, 0.60000002384185791, 0.60000002384185791),
            (0.60504204034805298, 0.58431375026702881, 0.58431375026702881),
            (0.60924369096755981, 0.56862747669219971, 0.56862747669219971),
            (0.61344540119171143, 0.55294120311737061, 0.55294120311737061),
            (0.61764705181121826, 0.53333336114883423, 0.53333336114883423),
            (0.62184876203536987, 0.51764708757400513, 0.51764708757400513),
            (0.62605041265487671, 0.50196081399917603, 0.50196081399917603),
            (0.63025212287902832, 0.46666666865348816, 0.46666666865348816),
            (0.63445377349853516, 0.45098039507865906, 0.45098039507865906),
            (0.63865548372268677, 0.43529412150382996, 0.43529412150382996),
            (0.6428571343421936, 0.41960784792900085, 0.41960784792900085),
            (0.64705884456634521, 0.40000000596046448, 0.40000000596046448),
            (0.65126049518585205, 0.38431373238563538, 0.38431373238563538),
            (0.65546220541000366, 0.36862745881080627, 0.36862745881080627),
            (0.6596638560295105, 0.35294118523597717, 0.35294118523597717),
            (0.66386556625366211, 0.3333333432674408, 0.3333333432674408),
            (0.66806721687316895, 0.31764706969261169, 0.31764706969261169),
            (0.67226892709732056, 0.30196079611778259, 0.30196079611778259),
            (0.67647057771682739, 0.28627452254295349, 0.28627452254295349),
            (0.680672287940979, 0.26666668057441711, 0.26666668057441711),
            (0.68487393856048584, 0.25098040699958801, 0.25098040699958801),
            (0.68907564878463745, 0.23529411852359772, 0.23529411852359772),
            (0.69327729940414429, 0.21960784494876862, 0.21960784494876862),
            (0.6974790096282959, 0.20000000298023224, 0.20000000298023224),
            (0.70168066024780273, 0.18431372940540314, 0.18431372940540314),
            (0.70588237047195435, 0.16862745583057404, 0.16862745583057404),
            (0.71008402109146118, 0.15294118225574493, 0.15294118225574493),
            (0.71428573131561279, 0.11764705926179886, 0.11764705926179886),
            (0.71848738193511963, 0.10196078568696976, 0.10196078568696976),
            (0.72268909215927124, 0.086274512112140656, 0.086274512112140656),
            (0.72689074277877808, 0.066666670143604279, 0.066666670143604279),
            (0.73109245300292969, 0.050980392843484879, 0.050980392843484879),
            (0.73529410362243652, 0.035294119268655777, 0.035294119268655777),
            (0.73949581384658813, 0.019607843831181526, 0.019607843831181526),
            (0.74369746446609497, 0.0, 0.0),
            (0.74789917469024658, 0.011764706112444401, 0.011764706112444401),
            (0.75210082530975342, 0.027450980618596077, 0.027450980618596077),
            (0.75630253553390503, 0.058823529630899429, 0.058823529630899429),
            (0.76050418615341187, 0.074509806931018829, 0.074509806931018829),
            (0.76470589637756348, 0.086274512112140656, 0.086274512112140656),
            (0.76890754699707031, 0.10196078568696976, 0.10196078568696976),
            (0.77310925722122192, 0.11764705926179886, 0.11764705926179886),
            (0.77731090784072876, 0.13333334028720856, 0.13333334028720856),
            (0.78151261806488037, 0.14901961386203766, 0.14901961386203766),
            (0.78571426868438721, 0.16078431904315948, 0.16078431904315948),
            (0.78991597890853882, 0.17647059261798859, 0.17647059261798859),
            (0.79411762952804565, 0.19215686619281769, 0.19215686619281769),
            (0.79831933975219727, 0.22352941334247589, 0.22352941334247589),
            (0.8025209903717041, 0.23529411852359772, 0.23529411852359772),
            (0.80672270059585571, 0.25098040699958801, 0.25098040699958801),
            (0.81092435121536255, 0.26666668057441711, 0.26666668057441711),
            (0.81512606143951416, 0.28235295414924622, 0.28235295414924622),
            (0.819327712059021, 0.29803922772407532, 0.29803922772407532),
            (0.82352942228317261, 0.30980393290519714, 0.30980393290519714),
            (0.82773107290267944, 0.32549020648002625, 0.32549020648002625),
            (0.83193278312683105, 0.34117648005485535, 0.34117648005485535),
            (0.83613443374633789, 0.35686275362968445, 0.35686275362968445),
            (0.8403361439704895, 0.37254902720451355, 0.37254902720451355),
            (0.84453779458999634, 0.38431373238563538, 0.38431373238563538),
            (0.84873950481414795, 0.40000000596046448, 0.40000000596046448),
            (0.85294115543365479, 0.41568627953529358, 0.41568627953529358),
            (0.8571428656578064, 0.43137255311012268, 0.43137255311012268),
            (0.86134451627731323, 0.44705882668495178, 0.44705882668495178),
            (0.86554622650146484, 0.45882353186607361, 0.45882353186607361),
            (0.86974787712097168, 0.47450980544090271, 0.47450980544090271),
            (0.87394958734512329, 0.49019607901573181, 0.49019607901573181),
            (0.87815123796463013, 0.5058823823928833, 0.5058823823928833),
            (0.88235294818878174, 0.5372549295425415, 0.5372549295425415),
            (0.88655459880828857, 0.54901963472366333, 0.54901963472366333),
            (0.89075630903244019, 0.56470590829849243, 0.56470590829849243),
            (0.89495795965194702, 0.58039218187332153, 0.58039218187332153),
            (0.89915966987609863, 0.59607845544815063, 0.59607845544815063),
            (0.90336132049560547, 0.61176472902297974, 0.61176472902297974),
            (0.90756303071975708, 0.62352943420410156, 0.62352943420410156),
            (0.91176468133926392, 0.63921570777893066, 0.63921570777893066),
            (0.91596639156341553, 0.65490198135375977, 0.65490198135375977),
            (0.92016804218292236, 0.67058825492858887, 0.67058825492858887),
            (0.92436975240707397, 0.68627452850341797, 0.68627452850341797),
            (0.92857140302658081, 0.69803923368453979, 0.69803923368453979),
            (0.93277311325073242, 0.7137255072593689, 0.7137255072593689),
            (0.93697476387023926, 0.729411780834198, 0.729411780834198),
            (0.94117647409439087, 0.7450980544090271, 0.7450980544090271),
            (0.94537812471389771, 0.7607843279838562, 0.7607843279838562),
            (0.94957983493804932, 0.77254903316497803, 0.77254903316497803),
            (0.95378148555755615, 0.78823530673980713, 0.78823530673980713),
            (0.95798319578170776, 0.80392158031463623, 0.80392158031463623),
            (0.9621848464012146, 0.81960785388946533, 0.81960785388946533),
            (0.96638655662536621, 0.84705883264541626, 0.84705883264541626),
            (0.97058820724487305, 0.86274510622024536, 0.86274510622024536),
            (0.97478991746902466, 0.87843137979507446, 0.87843137979507446),
            (0.97899156808853149, 0.89411765336990356, 0.89411765336990356),
            (0.98319327831268311, 0.90980392694473267, 0.90980392694473267),
            (0.98739492893218994, 0.92156863212585449, 0.92156863212585449),
            (0.99159663915634155, 0.93725490570068359, 0.93725490570068359),
            (0.99579828977584839, 0.9529411792755127, 0.9529411792755127),
            (1.0, 0.9686274528503418, 0.9686274528503418),
        ],
    )
    return ColorMapper.from_segment_map(_data, range=range, **traits)


def gist_yarg(range, **traits):
    """Generator for the 'gist_yarg' colormap from GIST."""
    _data = dict(
        red=[
            (0.0, 1.0, 1.0),
            (0.0042016808874905109, 0.99607843160629272, 0.99607843160629272),
            (0.0084033617749810219, 0.99215686321258545, 0.99215686321258545),
            (0.012605042196810246, 0.98823529481887817, 0.98823529481887817),
            (0.016806723549962044, 0.9843137264251709, 0.9843137264251709),
            (0.021008403971791267, 0.98039215803146362, 0.98039215803146362),
            (0.025210084393620491, 0.97647058963775635, 0.97647058963775635),
            (0.029411764815449715, 0.97254902124404907, 0.97254902124404907),
            (0.033613447099924088, 0.96470588445663452, 0.96470588445663452),
            (0.037815127521753311, 0.96078431606292725, 0.96078431606292725),
            (0.042016807943582535, 0.95686274766921997, 0.95686274766921997),
            (0.046218488365411758, 0.9529411792755127, 0.9529411792755127),
            (0.050420168787240982, 0.94901961088180542, 0.94901961088180542),
            (0.054621849209070206, 0.94509804248809814, 0.94509804248809814),
            (0.058823529630899429, 0.94117647409439087, 0.94117647409439087),
            (0.063025213778018951, 0.93725490570068359, 0.93725490570068359),
            (0.067226894199848175, 0.93333333730697632, 0.93333333730697632),
            (0.071428574621677399, 0.92941176891326904, 0.92941176891326904),
            (0.075630255043506622, 0.92549020051956177, 0.92549020051956177),
            (0.079831935465335846, 0.92156863212585449, 0.92156863212585449),
            (0.08403361588716507, 0.91764706373214722, 0.91764706373214722),
            (0.088235296308994293, 0.91372549533843994, 0.91372549533843994),
            (0.092436976730823517, 0.90980392694473267, 0.90980392694473267),
            (0.09663865715265274, 0.90196079015731812, 0.90196079015731812),
            (0.10084033757448196, 0.89803922176361084, 0.89803922176361084),
            (0.10504201799631119, 0.89411765336990356, 0.89411765336990356),
            (0.10924369841814041, 0.89019608497619629, 0.89019608497619629),
            (0.11344537883996964, 0.88627451658248901, 0.88627451658248901),
            (0.11764705926179886, 0.88235294818878174, 0.88235294818878174),
            (0.12184873968362808, 0.87843137979507446, 0.87843137979507446),
            (0.1260504275560379, 0.87450981140136719, 0.87450981140136719),
            (0.13025210797786713, 0.87058824300765991, 0.87058824300765991),
            (0.13445378839969635, 0.86666667461395264, 0.86666667461395264),
            (0.13865546882152557, 0.86274510622024536, 0.86274510622024536),
            (0.1428571492433548, 0.85882353782653809, 0.85882353782653809),
            (0.14705882966518402, 0.85490196943283081, 0.85490196943283081),
            (0.15126051008701324, 0.85098040103912354, 0.85098040103912354),
            (0.15546219050884247, 0.84705883264541626, 0.84705883264541626),
            (0.15966387093067169, 0.83921569585800171, 0.83921569585800171),
            (0.16386555135250092, 0.83529412746429443, 0.83529412746429443),
            (0.16806723177433014, 0.83137255907058716, 0.83137255907058716),
            (0.17226891219615936, 0.82745099067687988, 0.82745099067687988),
            (0.17647059261798859, 0.82352942228317261, 0.82352942228317261),
            (0.18067227303981781, 0.81960785388946533, 0.81960785388946533),
            (0.18487395346164703, 0.81568628549575806, 0.81568628549575806),
            (0.18907563388347626, 0.81176471710205078, 0.81176471710205078),
            (0.19327731430530548, 0.80784314870834351, 0.80784314870834351),
            (0.1974789947271347, 0.80392158031463623, 0.80392158031463623),
            (0.20168067514896393, 0.80000001192092896, 0.80000001192092896),
            (0.20588235557079315, 0.79607844352722168, 0.79607844352722168),
            (0.21008403599262238, 0.7921568751335144, 0.7921568751335144),
            (0.2142857164144516, 0.78823530673980713, 0.78823530673980713),
            (0.21848739683628082, 0.78431373834609985, 0.78431373834609985),
            (0.22268907725811005, 0.7764706015586853, 0.7764706015586853),
            (0.22689075767993927, 0.77254903316497803, 0.77254903316497803),
            (0.23109243810176849, 0.76862746477127075, 0.76862746477127075),
            (0.23529411852359772, 0.76470589637756348, 0.76470589637756348),
            (0.23949579894542694, 0.7607843279838562, 0.7607843279838562),
            (0.24369747936725616, 0.75686275959014893, 0.75686275959014893),
            (0.24789915978908539, 0.75294119119644165, 0.75294119119644165),
            (0.25210085511207581, 0.74901962280273438, 0.74901962280273438),
            (0.25630253553390503, 0.7450980544090271, 0.7450980544090271),
            (0.26050421595573425, 0.74117648601531982, 0.74117648601531982),
            (0.26470589637756348, 0.73725491762161255, 0.73725491762161255),
            (0.2689075767993927, 0.73333334922790527, 0.73333334922790527),
            (0.27310925722122192, 0.729411780834198, 0.729411780834198),
            (0.27731093764305115, 0.72549021244049072, 0.72549021244049072),
            (0.28151261806488037, 0.72156864404678345, 0.72156864404678345),
            (0.28571429848670959, 0.7137255072593689, 0.7137255072593689),
            (0.28991597890853882, 0.70980393886566162, 0.70980393886566162),
            (0.29411765933036804, 0.70588237047195435, 0.70588237047195435),
            (0.29831933975219727, 0.70196080207824707, 0.70196080207824707),
            (0.30252102017402649, 0.69803923368453979, 0.69803923368453979),
            (0.30672270059585571, 0.69411766529083252, 0.69411766529083252),
            (0.31092438101768494, 0.69019609689712524, 0.69019609689712524),
            (0.31512606143951416, 0.68627452850341797, 0.68627452850341797),
            (0.31932774186134338, 0.68235296010971069, 0.68235296010971069),
            (0.32352942228317261, 0.67843139171600342, 0.67843139171600342),
            (0.32773110270500183, 0.67450982332229614, 0.67450982332229614),
            (0.33193278312683105, 0.67058825492858887, 0.67058825492858887),
            (0.33613446354866028, 0.66666668653488159, 0.66666668653488159),
            (0.3403361439704895, 0.66274511814117432, 0.66274511814117432),
            (0.34453782439231873, 0.65882354974746704, 0.65882354974746704),
            (0.34873950481414795, 0.65098041296005249, 0.65098041296005249),
            (0.35294118523597717, 0.64705884456634521, 0.64705884456634521),
            (0.3571428656578064, 0.64313727617263794, 0.64313727617263794),
            (0.36134454607963562, 0.63921570777893066, 0.63921570777893066),
            (0.36554622650146484, 0.63529413938522339, 0.63529413938522339),
            (0.36974790692329407, 0.63137257099151611, 0.63137257099151611),
            (0.37394958734512329, 0.62745100259780884, 0.62745100259780884),
            (0.37815126776695251, 0.62352943420410156, 0.62352943420410156),
            (0.38235294818878174, 0.61960786581039429, 0.61960786581039429),
            (0.38655462861061096, 0.61568629741668701, 0.61568629741668701),
            (0.39075630903244019, 0.61176472902297974, 0.61176472902297974),
            (0.39495798945426941, 0.60784316062927246, 0.60784316062927246),
            (0.39915966987609863, 0.60392159223556519, 0.60392159223556519),
            (0.40336135029792786, 0.60000002384185791, 0.60000002384185791),
            (0.40756303071975708, 0.59607845544815063, 0.59607845544815063),
            (0.4117647111415863, 0.58823531866073608, 0.58823531866073608),
            (0.41596639156341553, 0.58431375026702881, 0.58431375026702881),
            (0.42016807198524475, 0.58039218187332153, 0.58039218187332153),
            (0.42436975240707397, 0.57647061347961426, 0.57647061347961426),
            (0.4285714328289032, 0.57254904508590698, 0.57254904508590698),
            (0.43277311325073242, 0.56862747669219971, 0.56862747669219971),
            (0.43697479367256165, 0.56470590829849243, 0.56470590829849243),
            (0.44117647409439087, 0.56078433990478516, 0.56078433990478516),
            (0.44537815451622009, 0.55686277151107788, 0.55686277151107788),
            (0.44957983493804932, 0.55294120311737061, 0.55294120311737061),
            (0.45378151535987854, 0.54901963472366333, 0.54901963472366333),
            (0.45798319578170776, 0.54509806632995605, 0.54509806632995605),
            (0.46218487620353699, 0.54117649793624878, 0.54117649793624878),
            (0.46638655662536621, 0.5372549295425415, 0.5372549295425415),
            (0.47058823704719543, 0.53333336114883423, 0.53333336114883423),
            (0.47478991746902466, 0.52549022436141968, 0.52549022436141968),
            (0.47899159789085388, 0.5215686559677124, 0.5215686559677124),
            (0.48319327831268311, 0.51764708757400513, 0.51764708757400513),
            (0.48739495873451233, 0.51372551918029785, 0.51372551918029785),
            (0.49159663915634155, 0.50980395078659058, 0.50980395078659058),
            (0.49579831957817078, 0.5058823823928833, 0.5058823823928833),
            (0.5, 0.50196081399917603, 0.50196081399917603),
            (0.50420171022415161, 0.49803921580314636, 0.49803921580314636),
            (0.50840336084365845, 0.49411764740943909, 0.49411764740943909),
            (0.51260507106781006, 0.49019607901573181, 0.49019607901573181),
            (0.51680672168731689, 0.48627451062202454, 0.48627451062202454),
            (0.52100843191146851, 0.48235294222831726, 0.48235294222831726),
            (0.52521008253097534, 0.47843137383460999, 0.47843137383460999),
            (0.52941179275512695, 0.47450980544090271, 0.47450980544090271),
            (0.53361344337463379, 0.47058823704719543, 0.47058823704719543),
            (0.5378151535987854, 0.46274510025978088, 0.46274510025978088),
            (0.54201680421829224, 0.45882353186607361, 0.45882353186607361),
            (0.54621851444244385, 0.45490196347236633, 0.45490196347236633),
            (0.55042016506195068, 0.45098039507865906, 0.45098039507865906),
            (0.55462187528610229, 0.44705882668495178, 0.44705882668495178),
            (0.55882352590560913, 0.44313725829124451, 0.44313725829124451),
            (0.56302523612976074, 0.43921568989753723, 0.43921568989753723),
            (0.56722688674926758, 0.43529412150382996, 0.43529412150382996),
            (0.57142859697341919, 0.43137255311012268, 0.43137255311012268),
            (0.57563024759292603, 0.42745098471641541, 0.42745098471641541),
            (0.57983195781707764, 0.42352941632270813, 0.42352941632270813),
            (0.58403360843658447, 0.41960784792900085, 0.41960784792900085),
            (0.58823531866073608, 0.41568627953529358, 0.41568627953529358),
            (0.59243696928024292, 0.4117647111415863, 0.4117647111415863),
            (0.59663867950439453, 0.40784314274787903, 0.40784314274787903),
            (0.60084033012390137, 0.40000000596046448, 0.40000000596046448),
            (0.60504204034805298, 0.3960784375667572, 0.3960784375667572),
            (0.60924369096755981, 0.39215686917304993, 0.39215686917304993),
            (0.61344540119171143, 0.38823530077934265, 0.38823530077934265),
            (0.61764705181121826, 0.38431373238563538, 0.38431373238563538),
            (0.62184876203536987, 0.3803921639919281, 0.3803921639919281),
            (0.62605041265487671, 0.37647059559822083, 0.37647059559822083),
            (0.63025212287902832, 0.37254902720451355, 0.37254902720451355),
            (0.63445377349853516, 0.36862745881080627, 0.36862745881080627),
            (0.63865548372268677, 0.364705890417099, 0.364705890417099),
            (0.6428571343421936, 0.36078432202339172, 0.36078432202339172),
            (0.64705884456634521, 0.35686275362968445, 0.35686275362968445),
            (0.65126049518585205, 0.35294118523597717, 0.35294118523597717),
            (0.65546220541000366, 0.3490196168422699, 0.3490196168422699),
            (0.6596638560295105, 0.34509804844856262, 0.34509804844856262),
            (0.66386556625366211, 0.33725491166114807, 0.33725491166114807),
            (0.66806721687316895, 0.3333333432674408, 0.3333333432674408),
            (0.67226892709732056, 0.32941177487373352, 0.32941177487373352),
            (0.67647057771682739, 0.32549020648002625, 0.32549020648002625),
            (0.680672287940979, 0.32156863808631897, 0.32156863808631897),
            (0.68487393856048584, 0.31764706969261169, 0.31764706969261169),
            (0.68907564878463745, 0.31372550129890442, 0.31372550129890442),
            (0.69327729940414429, 0.30980393290519714, 0.30980393290519714),
            (0.6974790096282959, 0.30588236451148987, 0.30588236451148987),
            (0.70168066024780273, 0.30196079611778259, 0.30196079611778259),
            (0.70588237047195435, 0.29803922772407532, 0.29803922772407532),
            (0.71008402109146118, 0.29411765933036804, 0.29411765933036804),
            (0.71428573131561279, 0.29019609093666077, 0.29019609093666077),
            (0.71848738193511963, 0.28627452254295349, 0.28627452254295349),
            (0.72268909215927124, 0.28235295414924622, 0.28235295414924622),
            (0.72689074277877808, 0.27450981736183167, 0.27450981736183167),
            (0.73109245300292969, 0.27058824896812439, 0.27058824896812439),
            (0.73529410362243652, 0.26666668057441711, 0.26666668057441711),
            (0.73949581384658813, 0.26274511218070984, 0.26274511218070984),
            (0.74369746446609497, 0.25882354378700256, 0.25882354378700256),
            (0.74789917469024658, 0.25490197539329529, 0.25490197539329529),
            (0.75210082530975342, 0.25098040699958801, 0.25098040699958801),
            (0.75630253553390503, 0.24705882370471954, 0.24705882370471954),
            (0.76050418615341187, 0.24313725531101227, 0.24313725531101227),
            (0.76470589637756348, 0.23921568691730499, 0.23921568691730499),
            (0.76890754699707031, 0.23529411852359772, 0.23529411852359772),
            (0.77310925722122192, 0.23137255012989044, 0.23137255012989044),
            (0.77731090784072876, 0.22745098173618317, 0.22745098173618317),
            (0.78151261806488037, 0.22352941334247589, 0.22352941334247589),
            (0.78571426868438721, 0.21960784494876862, 0.21960784494876862),
            (0.78991597890853882, 0.21176470816135406, 0.21176470816135406),
            (0.79411762952804565, 0.20784313976764679, 0.20784313976764679),
            (0.79831933975219727, 0.20392157137393951, 0.20392157137393951),
            (0.8025209903717041, 0.20000000298023224, 0.20000000298023224),
            (0.80672270059585571, 0.19607843458652496, 0.19607843458652496),
            (0.81092435121536255, 0.19215686619281769, 0.19215686619281769),
            (0.81512606143951416, 0.18823529779911041, 0.18823529779911041),
            (0.819327712059021, 0.18431372940540314, 0.18431372940540314),
            (0.82352942228317261, 0.18039216101169586, 0.18039216101169586),
            (0.82773107290267944, 0.17647059261798859, 0.17647059261798859),
            (0.83193278312683105, 0.17254902422428131, 0.17254902422428131),
            (0.83613443374633789, 0.16862745583057404, 0.16862745583057404),
            (0.8403361439704895, 0.16470588743686676, 0.16470588743686676),
            (0.84453779458999634, 0.16078431904315948, 0.16078431904315948),
            (0.84873950481414795, 0.15686275064945221, 0.15686275064945221),
            (0.85294115543365479, 0.14901961386203766, 0.14901961386203766),
            (0.8571428656578064, 0.14509804546833038, 0.14509804546833038),
            (0.86134451627731323, 0.14117647707462311, 0.14117647707462311),
            (0.86554622650146484, 0.13725490868091583, 0.13725490868091583),
            (0.86974787712097168, 0.13333334028720856, 0.13333334028720856),
            (0.87394958734512329, 0.12941177189350128, 0.12941177189350128),
            (0.87815123796463013, 0.12549020349979401, 0.12549020349979401),
            (0.88235294818878174, 0.12156862765550613, 0.12156862765550613),
            (0.88655459880828857, 0.11764705926179886, 0.11764705926179886),
            (0.89075630903244019, 0.11372549086809158, 0.11372549086809158),
            (0.89495795965194702, 0.10980392247438431, 0.10980392247438431),
            (0.89915966987609863, 0.10588235408067703, 0.10588235408067703),
            (0.90336132049560547, 0.10196078568696976, 0.10196078568696976),
            (0.90756303071975708, 0.098039217293262482, 0.098039217293262482),
            (0.91176468133926392, 0.094117648899555206, 0.094117648899555206),
            (0.91596639156341553, 0.086274512112140656, 0.086274512112140656),
            (0.92016804218292236, 0.08235294371843338, 0.08235294371843338),
            (0.92436975240707397, 0.078431375324726105, 0.078431375324726105),
            (0.92857140302658081, 0.074509806931018829, 0.074509806931018829),
            (0.93277311325073242, 0.070588238537311554, 0.070588238537311554),
            (0.93697476387023926, 0.066666670143604279, 0.066666670143604279),
            (0.94117647409439087, 0.062745101749897003, 0.062745101749897003),
            (0.94537812471389771, 0.058823529630899429, 0.058823529630899429),
            (0.94957983493804932, 0.054901961237192154, 0.054901961237192154),
            (0.95378148555755615, 0.050980392843484879, 0.050980392843484879),
            (0.95798319578170776, 0.047058824449777603, 0.047058824449777603),
            (0.9621848464012146, 0.043137256056070328, 0.043137256056070328),
            (0.96638655662536621, 0.039215687662363052, 0.039215687662363052),
            (0.97058820724487305, 0.035294119268655777, 0.035294119268655777),
            (0.97478991746902466, 0.031372550874948502, 0.031372550874948502),
            (0.97899156808853149, 0.023529412224888802, 0.023529412224888802),
            (0.98319327831268311, 0.019607843831181526, 0.019607843831181526),
            (0.98739492893218994, 0.015686275437474251, 0.015686275437474251),
            (0.99159663915634155, 0.011764706112444401, 0.011764706112444401),
            (
                0.99579828977584839,
                0.0078431377187371254,
                0.0078431377187371254,
            ),
            (1.0, 0.0039215688593685627, 0.0039215688593685627),
        ],
        green=[
            (0.0, 1.0, 1.0),
            (0.0042016808874905109, 0.99607843160629272, 0.99607843160629272),
            (0.0084033617749810219, 0.99215686321258545, 0.99215686321258545),
            (0.012605042196810246, 0.98823529481887817, 0.98823529481887817),
            (0.016806723549962044, 0.9843137264251709, 0.9843137264251709),
            (0.021008403971791267, 0.98039215803146362, 0.98039215803146362),
            (0.025210084393620491, 0.97647058963775635, 0.97647058963775635),
            (0.029411764815449715, 0.97254902124404907, 0.97254902124404907),
            (0.033613447099924088, 0.96470588445663452, 0.96470588445663452),
            (0.037815127521753311, 0.96078431606292725, 0.96078431606292725),
            (0.042016807943582535, 0.95686274766921997, 0.95686274766921997),
            (0.046218488365411758, 0.9529411792755127, 0.9529411792755127),
            (0.050420168787240982, 0.94901961088180542, 0.94901961088180542),
            (0.054621849209070206, 0.94509804248809814, 0.94509804248809814),
            (0.058823529630899429, 0.94117647409439087, 0.94117647409439087),
            (0.063025213778018951, 0.93725490570068359, 0.93725490570068359),
            (0.067226894199848175, 0.93333333730697632, 0.93333333730697632),
            (0.071428574621677399, 0.92941176891326904, 0.92941176891326904),
            (0.075630255043506622, 0.92549020051956177, 0.92549020051956177),
            (0.079831935465335846, 0.92156863212585449, 0.92156863212585449),
            (0.08403361588716507, 0.91764706373214722, 0.91764706373214722),
            (0.088235296308994293, 0.91372549533843994, 0.91372549533843994),
            (0.092436976730823517, 0.90980392694473267, 0.90980392694473267),
            (0.09663865715265274, 0.90196079015731812, 0.90196079015731812),
            (0.10084033757448196, 0.89803922176361084, 0.89803922176361084),
            (0.10504201799631119, 0.89411765336990356, 0.89411765336990356),
            (0.10924369841814041, 0.89019608497619629, 0.89019608497619629),
            (0.11344537883996964, 0.88627451658248901, 0.88627451658248901),
            (0.11764705926179886, 0.88235294818878174, 0.88235294818878174),
            (0.12184873968362808, 0.87843137979507446, 0.87843137979507446),
            (0.1260504275560379, 0.87450981140136719, 0.87450981140136719),
            (0.13025210797786713, 0.87058824300765991, 0.87058824300765991),
            (0.13445378839969635, 0.86666667461395264, 0.86666667461395264),
            (0.13865546882152557, 0.86274510622024536, 0.86274510622024536),
            (0.1428571492433548, 0.85882353782653809, 0.85882353782653809),
            (0.14705882966518402, 0.85490196943283081, 0.85490196943283081),
            (0.15126051008701324, 0.85098040103912354, 0.85098040103912354),
            (0.15546219050884247, 0.84705883264541626, 0.84705883264541626),
            (0.15966387093067169, 0.83921569585800171, 0.83921569585800171),
            (0.16386555135250092, 0.83529412746429443, 0.83529412746429443),
            (0.16806723177433014, 0.83137255907058716, 0.83137255907058716),
            (0.17226891219615936, 0.82745099067687988, 0.82745099067687988),
            (0.17647059261798859, 0.82352942228317261, 0.82352942228317261),
            (0.18067227303981781, 0.81960785388946533, 0.81960785388946533),
            (0.18487395346164703, 0.81568628549575806, 0.81568628549575806),
            (0.18907563388347626, 0.81176471710205078, 0.81176471710205078),
            (0.19327731430530548, 0.80784314870834351, 0.80784314870834351),
            (0.1974789947271347, 0.80392158031463623, 0.80392158031463623),
            (0.20168067514896393, 0.80000001192092896, 0.80000001192092896),
            (0.20588235557079315, 0.79607844352722168, 0.79607844352722168),
            (0.21008403599262238, 0.7921568751335144, 0.7921568751335144),
            (0.2142857164144516, 0.78823530673980713, 0.78823530673980713),
            (0.21848739683628082, 0.78431373834609985, 0.78431373834609985),
            (0.22268907725811005, 0.7764706015586853, 0.7764706015586853),
            (0.22689075767993927, 0.77254903316497803, 0.77254903316497803),
            (0.23109243810176849, 0.76862746477127075, 0.76862746477127075),
            (0.23529411852359772, 0.76470589637756348, 0.76470589637756348),
            (0.23949579894542694, 0.7607843279838562, 0.7607843279838562),
            (0.24369747936725616, 0.75686275959014893, 0.75686275959014893),
            (0.24789915978908539, 0.75294119119644165, 0.75294119119644165),
            (0.25210085511207581, 0.74901962280273438, 0.74901962280273438),
            (0.25630253553390503, 0.7450980544090271, 0.7450980544090271),
            (0.26050421595573425, 0.74117648601531982, 0.74117648601531982),
            (0.26470589637756348, 0.73725491762161255, 0.73725491762161255),
            (0.2689075767993927, 0.73333334922790527, 0.73333334922790527),
            (0.27310925722122192, 0.729411780834198, 0.729411780834198),
            (0.27731093764305115, 0.72549021244049072, 0.72549021244049072),
            (0.28151261806488037, 0.72156864404678345, 0.72156864404678345),
            (0.28571429848670959, 0.7137255072593689, 0.7137255072593689),
            (0.28991597890853882, 0.70980393886566162, 0.70980393886566162),
            (0.29411765933036804, 0.70588237047195435, 0.70588237047195435),
            (0.29831933975219727, 0.70196080207824707, 0.70196080207824707),
            (0.30252102017402649, 0.69803923368453979, 0.69803923368453979),
            (0.30672270059585571, 0.69411766529083252, 0.69411766529083252),
            (0.31092438101768494, 0.69019609689712524, 0.69019609689712524),
            (0.31512606143951416, 0.68627452850341797, 0.68627452850341797),
            (0.31932774186134338, 0.68235296010971069, 0.68235296010971069),
            (0.32352942228317261, 0.67843139171600342, 0.67843139171600342),
            (0.32773110270500183, 0.67450982332229614, 0.67450982332229614),
            (0.33193278312683105, 0.67058825492858887, 0.67058825492858887),
            (0.33613446354866028, 0.66666668653488159, 0.66666668653488159),
            (0.3403361439704895, 0.66274511814117432, 0.66274511814117432),
            (0.34453782439231873, 0.65882354974746704, 0.65882354974746704),
            (0.34873950481414795, 0.65098041296005249, 0.65098041296005249),
            (0.35294118523597717, 0.64705884456634521, 0.64705884456634521),
            (0.3571428656578064, 0.64313727617263794, 0.64313727617263794),
            (0.36134454607963562, 0.63921570777893066, 0.63921570777893066),
            (0.36554622650146484, 0.63529413938522339, 0.63529413938522339),
            (0.36974790692329407, 0.63137257099151611, 0.63137257099151611),
            (0.37394958734512329, 0.62745100259780884, 0.62745100259780884),
            (0.37815126776695251, 0.62352943420410156, 0.62352943420410156),
            (0.38235294818878174, 0.61960786581039429, 0.61960786581039429),
            (0.38655462861061096, 0.61568629741668701, 0.61568629741668701),
            (0.39075630903244019, 0.61176472902297974, 0.61176472902297974),
            (0.39495798945426941, 0.60784316062927246, 0.60784316062927246),
            (0.39915966987609863, 0.60392159223556519, 0.60392159223556519),
            (0.40336135029792786, 0.60000002384185791, 0.60000002384185791),
            (0.40756303071975708, 0.59607845544815063, 0.59607845544815063),
            (0.4117647111415863, 0.58823531866073608, 0.58823531866073608),
            (0.41596639156341553, 0.58431375026702881, 0.58431375026702881),
            (0.42016807198524475, 0.58039218187332153, 0.58039218187332153),
            (0.42436975240707397, 0.57647061347961426, 0.57647061347961426),
            (0.4285714328289032, 0.57254904508590698, 0.57254904508590698),
            (0.43277311325073242, 0.56862747669219971, 0.56862747669219971),
            (0.43697479367256165, 0.56470590829849243, 0.56470590829849243),
            (0.44117647409439087, 0.56078433990478516, 0.56078433990478516),
            (0.44537815451622009, 0.55686277151107788, 0.55686277151107788),
            (0.44957983493804932, 0.55294120311737061, 0.55294120311737061),
            (0.45378151535987854, 0.54901963472366333, 0.54901963472366333),
            (0.45798319578170776, 0.54509806632995605, 0.54509806632995605),
            (0.46218487620353699, 0.54117649793624878, 0.54117649793624878),
            (0.46638655662536621, 0.5372549295425415, 0.5372549295425415),
            (0.47058823704719543, 0.53333336114883423, 0.53333336114883423),
            (0.47478991746902466, 0.52549022436141968, 0.52549022436141968),
            (0.47899159789085388, 0.5215686559677124, 0.5215686559677124),
            (0.48319327831268311, 0.51764708757400513, 0.51764708757400513),
            (0.48739495873451233, 0.51372551918029785, 0.51372551918029785),
            (0.49159663915634155, 0.50980395078659058, 0.50980395078659058),
            (0.49579831957817078, 0.5058823823928833, 0.5058823823928833),
            (0.5, 0.50196081399917603, 0.50196081399917603),
            (0.50420171022415161, 0.49803921580314636, 0.49803921580314636),
            (0.50840336084365845, 0.49411764740943909, 0.49411764740943909),
            (0.51260507106781006, 0.49019607901573181, 0.49019607901573181),
            (0.51680672168731689, 0.48627451062202454, 0.48627451062202454),
            (0.52100843191146851, 0.48235294222831726, 0.48235294222831726),
            (0.52521008253097534, 0.47843137383460999, 0.47843137383460999),
            (0.52941179275512695, 0.47450980544090271, 0.47450980544090271),
            (0.53361344337463379, 0.47058823704719543, 0.47058823704719543),
            (0.5378151535987854, 0.46274510025978088, 0.46274510025978088),
            (0.54201680421829224, 0.45882353186607361, 0.45882353186607361),
            (0.54621851444244385, 0.45490196347236633, 0.45490196347236633),
            (0.55042016506195068, 0.45098039507865906, 0.45098039507865906),
            (0.55462187528610229, 0.44705882668495178, 0.44705882668495178),
            (0.55882352590560913, 0.44313725829124451, 0.44313725829124451),
            (0.56302523612976074, 0.43921568989753723, 0.43921568989753723),
            (0.56722688674926758, 0.43529412150382996, 0.43529412150382996),
            (0.57142859697341919, 0.43137255311012268, 0.43137255311012268),
            (0.57563024759292603, 0.42745098471641541, 0.42745098471641541),
            (0.57983195781707764, 0.42352941632270813, 0.42352941632270813),
            (0.58403360843658447, 0.41960784792900085, 0.41960784792900085),
            (0.58823531866073608, 0.41568627953529358, 0.41568627953529358),
            (0.59243696928024292, 0.4117647111415863, 0.4117647111415863),
            (0.59663867950439453, 0.40784314274787903, 0.40784314274787903),
            (0.60084033012390137, 0.40000000596046448, 0.40000000596046448),
            (0.60504204034805298, 0.3960784375667572, 0.3960784375667572),
            (0.60924369096755981, 0.39215686917304993, 0.39215686917304993),
            (0.61344540119171143, 0.38823530077934265, 0.38823530077934265),
            (0.61764705181121826, 0.38431373238563538, 0.38431373238563538),
            (0.62184876203536987, 0.3803921639919281, 0.3803921639919281),
            (0.62605041265487671, 0.37647059559822083, 0.37647059559822083),
            (0.63025212287902832, 0.37254902720451355, 0.37254902720451355),
            (0.63445377349853516, 0.36862745881080627, 0.36862745881080627),
            (0.63865548372268677, 0.364705890417099, 0.364705890417099),
            (0.6428571343421936, 0.36078432202339172, 0.36078432202339172),
            (0.64705884456634521, 0.35686275362968445, 0.35686275362968445),
            (0.65126049518585205, 0.35294118523597717, 0.35294118523597717),
            (0.65546220541000366, 0.3490196168422699, 0.3490196168422699),
            (0.6596638560295105, 0.34509804844856262, 0.34509804844856262),
            (0.66386556625366211, 0.33725491166114807, 0.33725491166114807),
            (0.66806721687316895, 0.3333333432674408, 0.3333333432674408),
            (0.67226892709732056, 0.32941177487373352, 0.32941177487373352),
            (0.67647057771682739, 0.32549020648002625, 0.32549020648002625),
            (0.680672287940979, 0.32156863808631897, 0.32156863808631897),
            (0.68487393856048584, 0.31764706969261169, 0.31764706969261169),
            (0.68907564878463745, 0.31372550129890442, 0.31372550129890442),
            (0.69327729940414429, 0.30980393290519714, 0.30980393290519714),
            (0.6974790096282959, 0.30588236451148987, 0.30588236451148987),
            (0.70168066024780273, 0.30196079611778259, 0.30196079611778259),
            (0.70588237047195435, 0.29803922772407532, 0.29803922772407532),
            (0.71008402109146118, 0.29411765933036804, 0.29411765933036804),
            (0.71428573131561279, 0.29019609093666077, 0.29019609093666077),
            (0.71848738193511963, 0.28627452254295349, 0.28627452254295349),
            (0.72268909215927124, 0.28235295414924622, 0.28235295414924622),
            (0.72689074277877808, 0.27450981736183167, 0.27450981736183167),
            (0.73109245300292969, 0.27058824896812439, 0.27058824896812439),
            (0.73529410362243652, 0.26666668057441711, 0.26666668057441711),
            (0.73949581384658813, 0.26274511218070984, 0.26274511218070984),
            (0.74369746446609497, 0.25882354378700256, 0.25882354378700256),
            (0.74789917469024658, 0.25490197539329529, 0.25490197539329529),
            (0.75210082530975342, 0.25098040699958801, 0.25098040699958801),
            (0.75630253553390503, 0.24705882370471954, 0.24705882370471954),
            (0.76050418615341187, 0.24313725531101227, 0.24313725531101227),
            (0.76470589637756348, 0.23921568691730499, 0.23921568691730499),
            (0.76890754699707031, 0.23529411852359772, 0.23529411852359772),
            (0.77310925722122192, 0.23137255012989044, 0.23137255012989044),
            (0.77731090784072876, 0.22745098173618317, 0.22745098173618317),
            (0.78151261806488037, 0.22352941334247589, 0.22352941334247589),
            (0.78571426868438721, 0.21960784494876862, 0.21960784494876862),
            (0.78991597890853882, 0.21176470816135406, 0.21176470816135406),
            (0.79411762952804565, 0.20784313976764679, 0.20784313976764679),
            (0.79831933975219727, 0.20392157137393951, 0.20392157137393951),
            (0.8025209903717041, 0.20000000298023224, 0.20000000298023224),
            (0.80672270059585571, 0.19607843458652496, 0.19607843458652496),
            (0.81092435121536255, 0.19215686619281769, 0.19215686619281769),
            (0.81512606143951416, 0.18823529779911041, 0.18823529779911041),
            (0.819327712059021, 0.18431372940540314, 0.18431372940540314),
            (0.82352942228317261, 0.18039216101169586, 0.18039216101169586),
            (0.82773107290267944, 0.17647059261798859, 0.17647059261798859),
            (0.83193278312683105, 0.17254902422428131, 0.17254902422428131),
            (0.83613443374633789, 0.16862745583057404, 0.16862745583057404),
            (0.8403361439704895, 0.16470588743686676, 0.16470588743686676),
            (0.84453779458999634, 0.16078431904315948, 0.16078431904315948),
            (0.84873950481414795, 0.15686275064945221, 0.15686275064945221),
            (0.85294115543365479, 0.14901961386203766, 0.14901961386203766),
            (0.8571428656578064, 0.14509804546833038, 0.14509804546833038),
            (0.86134451627731323, 0.14117647707462311, 0.14117647707462311),
            (0.86554622650146484, 0.13725490868091583, 0.13725490868091583),
            (0.86974787712097168, 0.13333334028720856, 0.13333334028720856),
            (0.87394958734512329, 0.12941177189350128, 0.12941177189350128),
            (0.87815123796463013, 0.12549020349979401, 0.12549020349979401),
            (0.88235294818878174, 0.12156862765550613, 0.12156862765550613),
            (0.88655459880828857, 0.11764705926179886, 0.11764705926179886),
            (0.89075630903244019, 0.11372549086809158, 0.11372549086809158),
            (0.89495795965194702, 0.10980392247438431, 0.10980392247438431),
            (0.89915966987609863, 0.10588235408067703, 0.10588235408067703),
            (0.90336132049560547, 0.10196078568696976, 0.10196078568696976),
            (0.90756303071975708, 0.098039217293262482, 0.098039217293262482),
            (0.91176468133926392, 0.094117648899555206, 0.094117648899555206),
            (0.91596639156341553, 0.086274512112140656, 0.086274512112140656),
            (0.92016804218292236, 0.08235294371843338, 0.08235294371843338),
            (0.92436975240707397, 0.078431375324726105, 0.078431375324726105),
            (0.92857140302658081, 0.074509806931018829, 0.074509806931018829),
            (0.93277311325073242, 0.070588238537311554, 0.070588238537311554),
            (0.93697476387023926, 0.066666670143604279, 0.066666670143604279),
            (0.94117647409439087, 0.062745101749897003, 0.062745101749897003),
            (0.94537812471389771, 0.058823529630899429, 0.058823529630899429),
            (0.94957983493804932, 0.054901961237192154, 0.054901961237192154),
            (0.95378148555755615, 0.050980392843484879, 0.050980392843484879),
            (0.95798319578170776, 0.047058824449777603, 0.047058824449777603),
            (0.9621848464012146, 0.043137256056070328, 0.043137256056070328),
            (0.96638655662536621, 0.039215687662363052, 0.039215687662363052),
            (0.97058820724487305, 0.035294119268655777, 0.035294119268655777),
            (0.97478991746902466, 0.031372550874948502, 0.031372550874948502),
            (0.97899156808853149, 0.023529412224888802, 0.023529412224888802),
            (0.98319327831268311, 0.019607843831181526, 0.019607843831181526),
            (0.98739492893218994, 0.015686275437474251, 0.015686275437474251),
            (0.99159663915634155, 0.011764706112444401, 0.011764706112444401),
            (
                0.99579828977584839,
                0.0078431377187371254,
                0.0078431377187371254,
            ),
            (1.0, 0.0039215688593685627, 0.0039215688593685627),
        ],
        blue=[
            (0.0, 1.0, 1.0),
            (0.0042016808874905109, 0.99607843160629272, 0.99607843160629272),
            (0.0084033617749810219, 0.99215686321258545, 0.99215686321258545),
            (0.012605042196810246, 0.98823529481887817, 0.98823529481887817),
            (0.016806723549962044, 0.9843137264251709, 0.9843137264251709),
            (0.021008403971791267, 0.98039215803146362, 0.98039215803146362),
            (0.025210084393620491, 0.97647058963775635, 0.97647058963775635),
            (0.029411764815449715, 0.97254902124404907, 0.97254902124404907),
            (0.033613447099924088, 0.96470588445663452, 0.96470588445663452),
            (0.037815127521753311, 0.96078431606292725, 0.96078431606292725),
            (0.042016807943582535, 0.95686274766921997, 0.95686274766921997),
            (0.046218488365411758, 0.9529411792755127, 0.9529411792755127),
            (0.050420168787240982, 0.94901961088180542, 0.94901961088180542),
            (0.054621849209070206, 0.94509804248809814, 0.94509804248809814),
            (0.058823529630899429, 0.94117647409439087, 0.94117647409439087),
            (0.063025213778018951, 0.93725490570068359, 0.93725490570068359),
            (0.067226894199848175, 0.93333333730697632, 0.93333333730697632),
            (0.071428574621677399, 0.92941176891326904, 0.92941176891326904),
            (0.075630255043506622, 0.92549020051956177, 0.92549020051956177),
            (0.079831935465335846, 0.92156863212585449, 0.92156863212585449),
            (0.08403361588716507, 0.91764706373214722, 0.91764706373214722),
            (0.088235296308994293, 0.91372549533843994, 0.91372549533843994),
            (0.092436976730823517, 0.90980392694473267, 0.90980392694473267),
            (0.09663865715265274, 0.90196079015731812, 0.90196079015731812),
            (0.10084033757448196, 0.89803922176361084, 0.89803922176361084),
            (0.10504201799631119, 0.89411765336990356, 0.89411765336990356),
            (0.10924369841814041, 0.89019608497619629, 0.89019608497619629),
            (0.11344537883996964, 0.88627451658248901, 0.88627451658248901),
            (0.11764705926179886, 0.88235294818878174, 0.88235294818878174),
            (0.12184873968362808, 0.87843137979507446, 0.87843137979507446),
            (0.1260504275560379, 0.87450981140136719, 0.87450981140136719),
            (0.13025210797786713, 0.87058824300765991, 0.87058824300765991),
            (0.13445378839969635, 0.86666667461395264, 0.86666667461395264),
            (0.13865546882152557, 0.86274510622024536, 0.86274510622024536),
            (0.1428571492433548, 0.85882353782653809, 0.85882353782653809),
            (0.14705882966518402, 0.85490196943283081, 0.85490196943283081),
            (0.15126051008701324, 0.85098040103912354, 0.85098040103912354),
            (0.15546219050884247, 0.84705883264541626, 0.84705883264541626),
            (0.15966387093067169, 0.83921569585800171, 0.83921569585800171),
            (0.16386555135250092, 0.83529412746429443, 0.83529412746429443),
            (0.16806723177433014, 0.83137255907058716, 0.83137255907058716),
            (0.17226891219615936, 0.82745099067687988, 0.82745099067687988),
            (0.17647059261798859, 0.82352942228317261, 0.82352942228317261),
            (0.18067227303981781, 0.81960785388946533, 0.81960785388946533),
            (0.18487395346164703, 0.81568628549575806, 0.81568628549575806),
            (0.18907563388347626, 0.81176471710205078, 0.81176471710205078),
            (0.19327731430530548, 0.80784314870834351, 0.80784314870834351),
            (0.1974789947271347, 0.80392158031463623, 0.80392158031463623),
            (0.20168067514896393, 0.80000001192092896, 0.80000001192092896),
            (0.20588235557079315, 0.79607844352722168, 0.79607844352722168),
            (0.21008403599262238, 0.7921568751335144, 0.7921568751335144),
            (0.2142857164144516, 0.78823530673980713, 0.78823530673980713),
            (0.21848739683628082, 0.78431373834609985, 0.78431373834609985),
            (0.22268907725811005, 0.7764706015586853, 0.7764706015586853),
            (0.22689075767993927, 0.77254903316497803, 0.77254903316497803),
            (0.23109243810176849, 0.76862746477127075, 0.76862746477127075),
            (0.23529411852359772, 0.76470589637756348, 0.76470589637756348),
            (0.23949579894542694, 0.7607843279838562, 0.7607843279838562),
            (0.24369747936725616, 0.75686275959014893, 0.75686275959014893),
            (0.24789915978908539, 0.75294119119644165, 0.75294119119644165),
            (0.25210085511207581, 0.74901962280273438, 0.74901962280273438),
            (0.25630253553390503, 0.7450980544090271, 0.7450980544090271),
            (0.26050421595573425, 0.74117648601531982, 0.74117648601531982),
            (0.26470589637756348, 0.73725491762161255, 0.73725491762161255),
            (0.2689075767993927, 0.73333334922790527, 0.73333334922790527),
            (0.27310925722122192, 0.729411780834198, 0.729411780834198),
            (0.27731093764305115, 0.72549021244049072, 0.72549021244049072),
            (0.28151261806488037, 0.72156864404678345, 0.72156864404678345),
            (0.28571429848670959, 0.7137255072593689, 0.7137255072593689),
            (0.28991597890853882, 0.70980393886566162, 0.70980393886566162),
            (0.29411765933036804, 0.70588237047195435, 0.70588237047195435),
            (0.29831933975219727, 0.70196080207824707, 0.70196080207824707),
            (0.30252102017402649, 0.69803923368453979, 0.69803923368453979),
            (0.30672270059585571, 0.69411766529083252, 0.69411766529083252),
            (0.31092438101768494, 0.69019609689712524, 0.69019609689712524),
            (0.31512606143951416, 0.68627452850341797, 0.68627452850341797),
            (0.31932774186134338, 0.68235296010971069, 0.68235296010971069),
            (0.32352942228317261, 0.67843139171600342, 0.67843139171600342),
            (0.32773110270500183, 0.67450982332229614, 0.67450982332229614),
            (0.33193278312683105, 0.67058825492858887, 0.67058825492858887),
            (0.33613446354866028, 0.66666668653488159, 0.66666668653488159),
            (0.3403361439704895, 0.66274511814117432, 0.66274511814117432),
            (0.34453782439231873, 0.65882354974746704, 0.65882354974746704),
            (0.34873950481414795, 0.65098041296005249, 0.65098041296005249),
            (0.35294118523597717, 0.64705884456634521, 0.64705884456634521),
            (0.3571428656578064, 0.64313727617263794, 0.64313727617263794),
            (0.36134454607963562, 0.63921570777893066, 0.63921570777893066),
            (0.36554622650146484, 0.63529413938522339, 0.63529413938522339),
            (0.36974790692329407, 0.63137257099151611, 0.63137257099151611),
            (0.37394958734512329, 0.62745100259780884, 0.62745100259780884),
            (0.37815126776695251, 0.62352943420410156, 0.62352943420410156),
            (0.38235294818878174, 0.61960786581039429, 0.61960786581039429),
            (0.38655462861061096, 0.61568629741668701, 0.61568629741668701),
            (0.39075630903244019, 0.61176472902297974, 0.61176472902297974),
            (0.39495798945426941, 0.60784316062927246, 0.60784316062927246),
            (0.39915966987609863, 0.60392159223556519, 0.60392159223556519),
            (0.40336135029792786, 0.60000002384185791, 0.60000002384185791),
            (0.40756303071975708, 0.59607845544815063, 0.59607845544815063),
            (0.4117647111415863, 0.58823531866073608, 0.58823531866073608),
            (0.41596639156341553, 0.58431375026702881, 0.58431375026702881),
            (0.42016807198524475, 0.58039218187332153, 0.58039218187332153),
            (0.42436975240707397, 0.57647061347961426, 0.57647061347961426),
            (0.4285714328289032, 0.57254904508590698, 0.57254904508590698),
            (0.43277311325073242, 0.56862747669219971, 0.56862747669219971),
            (0.43697479367256165, 0.56470590829849243, 0.56470590829849243),
            (0.44117647409439087, 0.56078433990478516, 0.56078433990478516),
            (0.44537815451622009, 0.55686277151107788, 0.55686277151107788),
            (0.44957983493804932, 0.55294120311737061, 0.55294120311737061),
            (0.45378151535987854, 0.54901963472366333, 0.54901963472366333),
            (0.45798319578170776, 0.54509806632995605, 0.54509806632995605),
            (0.46218487620353699, 0.54117649793624878, 0.54117649793624878),
            (0.46638655662536621, 0.5372549295425415, 0.5372549295425415),
            (0.47058823704719543, 0.53333336114883423, 0.53333336114883423),
            (0.47478991746902466, 0.52549022436141968, 0.52549022436141968),
            (0.47899159789085388, 0.5215686559677124, 0.5215686559677124),
            (0.48319327831268311, 0.51764708757400513, 0.51764708757400513),
            (0.48739495873451233, 0.51372551918029785, 0.51372551918029785),
            (0.49159663915634155, 0.50980395078659058, 0.50980395078659058),
            (0.49579831957817078, 0.5058823823928833, 0.5058823823928833),
            (0.5, 0.50196081399917603, 0.50196081399917603),
            (0.50420171022415161, 0.49803921580314636, 0.49803921580314636),
            (0.50840336084365845, 0.49411764740943909, 0.49411764740943909),
            (0.51260507106781006, 0.49019607901573181, 0.49019607901573181),
            (0.51680672168731689, 0.48627451062202454, 0.48627451062202454),
            (0.52100843191146851, 0.48235294222831726, 0.48235294222831726),
            (0.52521008253097534, 0.47843137383460999, 0.47843137383460999),
            (0.52941179275512695, 0.47450980544090271, 0.47450980544090271),
            (0.53361344337463379, 0.47058823704719543, 0.47058823704719543),
            (0.5378151535987854, 0.46274510025978088, 0.46274510025978088),
            (0.54201680421829224, 0.45882353186607361, 0.45882353186607361),
            (0.54621851444244385, 0.45490196347236633, 0.45490196347236633),
            (0.55042016506195068, 0.45098039507865906, 0.45098039507865906),
            (0.55462187528610229, 0.44705882668495178, 0.44705882668495178),
            (0.55882352590560913, 0.44313725829124451, 0.44313725829124451),
            (0.56302523612976074, 0.43921568989753723, 0.43921568989753723),
            (0.56722688674926758, 0.43529412150382996, 0.43529412150382996),
            (0.57142859697341919, 0.43137255311012268, 0.43137255311012268),
            (0.57563024759292603, 0.42745098471641541, 0.42745098471641541),
            (0.57983195781707764, 0.42352941632270813, 0.42352941632270813),
            (0.58403360843658447, 0.41960784792900085, 0.41960784792900085),
            (0.58823531866073608, 0.41568627953529358, 0.41568627953529358),
            (0.59243696928024292, 0.4117647111415863, 0.4117647111415863),
            (0.59663867950439453, 0.40784314274787903, 0.40784314274787903),
            (0.60084033012390137, 0.40000000596046448, 0.40000000596046448),
            (0.60504204034805298, 0.3960784375667572, 0.3960784375667572),
            (0.60924369096755981, 0.39215686917304993, 0.39215686917304993),
            (0.61344540119171143, 0.38823530077934265, 0.38823530077934265),
            (0.61764705181121826, 0.38431373238563538, 0.38431373238563538),
            (0.62184876203536987, 0.3803921639919281, 0.3803921639919281),
            (0.62605041265487671, 0.37647059559822083, 0.37647059559822083),
            (0.63025212287902832, 0.37254902720451355, 0.37254902720451355),
            (0.63445377349853516, 0.36862745881080627, 0.36862745881080627),
            (0.63865548372268677, 0.364705890417099, 0.364705890417099),
            (0.6428571343421936, 0.36078432202339172, 0.36078432202339172),
            (0.64705884456634521, 0.35686275362968445, 0.35686275362968445),
            (0.65126049518585205, 0.35294118523597717, 0.35294118523597717),
            (0.65546220541000366, 0.3490196168422699, 0.3490196168422699),
            (0.6596638560295105, 0.34509804844856262, 0.34509804844856262),
            (0.66386556625366211, 0.33725491166114807, 0.33725491166114807),
            (0.66806721687316895, 0.3333333432674408, 0.3333333432674408),
            (0.67226892709732056, 0.32941177487373352, 0.32941177487373352),
            (0.67647057771682739, 0.32549020648002625, 0.32549020648002625),
            (0.680672287940979, 0.32156863808631897, 0.32156863808631897),
            (0.68487393856048584, 0.31764706969261169, 0.31764706969261169),
            (0.68907564878463745, 0.31372550129890442, 0.31372550129890442),
            (0.69327729940414429, 0.30980393290519714, 0.30980393290519714),
            (0.6974790096282959, 0.30588236451148987, 0.30588236451148987),
            (0.70168066024780273, 0.30196079611778259, 0.30196079611778259),
            (0.70588237047195435, 0.29803922772407532, 0.29803922772407532),
            (0.71008402109146118, 0.29411765933036804, 0.29411765933036804),
            (0.71428573131561279, 0.29019609093666077, 0.29019609093666077),
            (0.71848738193511963, 0.28627452254295349, 0.28627452254295349),
            (0.72268909215927124, 0.28235295414924622, 0.28235295414924622),
            (0.72689074277877808, 0.27450981736183167, 0.27450981736183167),
            (0.73109245300292969, 0.27058824896812439, 0.27058824896812439),
            (0.73529410362243652, 0.26666668057441711, 0.26666668057441711),
            (0.73949581384658813, 0.26274511218070984, 0.26274511218070984),
            (0.74369746446609497, 0.25882354378700256, 0.25882354378700256),
            (0.74789917469024658, 0.25490197539329529, 0.25490197539329529),
            (0.75210082530975342, 0.25098040699958801, 0.25098040699958801),
            (0.75630253553390503, 0.24705882370471954, 0.24705882370471954),
            (0.76050418615341187, 0.24313725531101227, 0.24313725531101227),
            (0.76470589637756348, 0.23921568691730499, 0.23921568691730499),
            (0.76890754699707031, 0.23529411852359772, 0.23529411852359772),
            (0.77310925722122192, 0.23137255012989044, 0.23137255012989044),
            (0.77731090784072876, 0.22745098173618317, 0.22745098173618317),
            (0.78151261806488037, 0.22352941334247589, 0.22352941334247589),
            (0.78571426868438721, 0.21960784494876862, 0.21960784494876862),
            (0.78991597890853882, 0.21176470816135406, 0.21176470816135406),
            (0.79411762952804565, 0.20784313976764679, 0.20784313976764679),
            (0.79831933975219727, 0.20392157137393951, 0.20392157137393951),
            (0.8025209903717041, 0.20000000298023224, 0.20000000298023224),
            (0.80672270059585571, 0.19607843458652496, 0.19607843458652496),
            (0.81092435121536255, 0.19215686619281769, 0.19215686619281769),
            (0.81512606143951416, 0.18823529779911041, 0.18823529779911041),
            (0.819327712059021, 0.18431372940540314, 0.18431372940540314),
            (0.82352942228317261, 0.18039216101169586, 0.18039216101169586),
            (0.82773107290267944, 0.17647059261798859, 0.17647059261798859),
            (0.83193278312683105, 0.17254902422428131, 0.17254902422428131),
            (0.83613443374633789, 0.16862745583057404, 0.16862745583057404),
            (0.8403361439704895, 0.16470588743686676, 0.16470588743686676),
            (0.84453779458999634, 0.16078431904315948, 0.16078431904315948),
            (0.84873950481414795, 0.15686275064945221, 0.15686275064945221),
            (0.85294115543365479, 0.14901961386203766, 0.14901961386203766),
            (0.8571428656578064, 0.14509804546833038, 0.14509804546833038),
            (0.86134451627731323, 0.14117647707462311, 0.14117647707462311),
            (0.86554622650146484, 0.13725490868091583, 0.13725490868091583),
            (0.86974787712097168, 0.13333334028720856, 0.13333334028720856),
            (0.87394958734512329, 0.12941177189350128, 0.12941177189350128),
            (0.87815123796463013, 0.12549020349979401, 0.12549020349979401),
            (0.88235294818878174, 0.12156862765550613, 0.12156862765550613),
            (0.88655459880828857, 0.11764705926179886, 0.11764705926179886),
            (0.89075630903244019, 0.11372549086809158, 0.11372549086809158),
            (0.89495795965194702, 0.10980392247438431, 0.10980392247438431),
            (0.89915966987609863, 0.10588235408067703, 0.10588235408067703),
            (0.90336132049560547, 0.10196078568696976, 0.10196078568696976),
            (0.90756303071975708, 0.098039217293262482, 0.098039217293262482),
            (0.91176468133926392, 0.094117648899555206, 0.094117648899555206),
            (0.91596639156341553, 0.086274512112140656, 0.086274512112140656),
            (0.92016804218292236, 0.08235294371843338, 0.08235294371843338),
            (0.92436975240707397, 0.078431375324726105, 0.078431375324726105),
            (0.92857140302658081, 0.074509806931018829, 0.074509806931018829),
            (0.93277311325073242, 0.070588238537311554, 0.070588238537311554),
            (0.93697476387023926, 0.066666670143604279, 0.066666670143604279),
            (0.94117647409439087, 0.062745101749897003, 0.062745101749897003),
            (0.94537812471389771, 0.058823529630899429, 0.058823529630899429),
            (0.94957983493804932, 0.054901961237192154, 0.054901961237192154),
            (0.95378148555755615, 0.050980392843484879, 0.050980392843484879),
            (0.95798319578170776, 0.047058824449777603, 0.047058824449777603),
            (0.9621848464012146, 0.043137256056070328, 0.043137256056070328),
            (0.96638655662536621, 0.039215687662363052, 0.039215687662363052),
            (0.97058820724487305, 0.035294119268655777, 0.035294119268655777),
            (0.97478991746902466, 0.031372550874948502, 0.031372550874948502),
            (0.97899156808853149, 0.023529412224888802, 0.023529412224888802),
            (0.98319327831268311, 0.019607843831181526, 0.019607843831181526),
            (0.98739492893218994, 0.015686275437474251, 0.015686275437474251),
            (0.99159663915634155, 0.011764706112444401, 0.011764706112444401),
            (
                0.99579828977584839,
                0.0078431377187371254,
                0.0078431377187371254,
            ),
            (1.0, 0.0039215688593685627, 0.0039215688593685627),
        ],
    )
    return ColorMapper.from_segment_map(_data, range=range, **traits)


def CubicYF(range, **traits):
    """Generator of the 'CubicYF' colormap from Matteo Niccoli.

    Lab-based rainbow scheme with cubic-law luminance.

    http://mycarta.wordpress.com/color-palettes/
    """
    palette = array(
        [
            [0.5151, 0.0482, 0.6697],
            [0.5199, 0.1762, 0.8083],
            [0.4884, 0.2912, 0.9234],
            [0.4297, 0.3855, 0.9921],
            [0.3893, 0.4792, 0.9775],
            [0.3337, 0.5650, 0.9056],
            [0.2795, 0.6419, 0.8287],
            [0.2210, 0.7123, 0.7258],
            [0.2468, 0.7612, 0.6248],
            [0.2833, 0.8125, 0.5069],
            [0.3198, 0.8492, 0.3956],
            [0.3602, 0.8896, 0.2919],
            [0.4568, 0.9136, 0.3018],
            [0.6033, 0.9255, 0.3295],
            [0.7066, 0.9255, 0.3414],
            [0.8000, 0.9255, 0.3529],
        ]
    )
    return ColorMapper.from_palette_array(palette, range=range, **traits)


def CubicL(range, **traits):
    """Generator of the 'CubicL' colormap from Matteo Niccoli.

    Lab-based rainbow scheme with cubic-law luminance, like `CubicYF`
    but with red at the high end, a modest deviation from being
    completely perceptual.

    http://mycarta.wordpress.com/color-palettes/
    """
    palette = array(
        [
            [0.4706, 0.0000, 0.5216],
            [0.5137, 0.0527, 0.7096],
            [0.4942, 0.2507, 0.8781],
            [0.4296, 0.3858, 0.9922],
            [0.3691, 0.5172, 0.9495],
            [0.2963, 0.6191, 0.8515],
            [0.2199, 0.7134, 0.7225],
            [0.2643, 0.7836, 0.5756],
            [0.3094, 0.8388, 0.4248],
            [0.3623, 0.8917, 0.2858],
            [0.5200, 0.9210, 0.3137],
            [0.6800, 0.9255, 0.3386],
            [0.8000, 0.9255, 0.3529],
            [0.8706, 0.8549, 0.3608],
            [0.9514, 0.7466, 0.3686],
            [0.9765, 0.5887, 0.3569],
        ]
    )
    return ColorMapper.from_palette_array(palette, range=range, **traits)


def LinearL(range, **traits):
    """Generator of the 'LinearL' colormap from Matteo Niccoli.

    Lab-based linear lightness rainbow.

    http://mycarta.wordpress.com/color-palettes/
    """
    palette = array(
        [
            [0.0143, 0.0143, 0.0143],
            [0.1413, 0.0555, 0.1256],
            [0.1761, 0.0911, 0.2782],
            [0.1710, 0.1314, 0.4540],
            [0.1074, 0.2234, 0.4984],
            [0.0686, 0.3044, 0.5068],
            [0.0008, 0.3927, 0.4267],
            [0.0000, 0.4763, 0.3464],
            [0.0000, 0.5565, 0.2469],
            [0.0000, 0.6381, 0.1638],
            [0.2167, 0.6966, 0.0000],
            [0.3898, 0.7563, 0.0000],
            [0.6912, 0.7795, 0.0000],
            [0.8548, 0.8041, 0.4555],
            [0.9712, 0.8429, 0.7287],
            [0.9692, 0.9273, 0.8961],
        ]
    )
    return ColorMapper.from_palette_array(palette, range=range, **traits)


def LinearLHot(range, **traits):
    """Generator of the 'LinearLHot' colormap from Matteo Niccoli.

    Linear lightness modification of the `hot` colormap.

    http://mycarta.wordpress.com/color-palettes/
    """
    palette = array(
        [
            [0.0225, 0.0121, 0.0121],
            [0.1927, 0.0225, 0.0311],
            [0.3243, 0.0106, 0.0000],
            [0.4463, 0.0000, 0.0091],
            [0.5706, 0.0000, 0.0737],
            [0.6969, 0.0000, 0.1337],
            [0.8213, 0.0000, 0.1792],
            [0.8636, 0.0000, 0.0565],
            [0.8821, 0.2555, 0.0000],
            [0.8720, 0.4182, 0.0000],
            [0.8424, 0.5552, 0.0000],
            [0.8031, 0.6776, 0.0000],
            [0.7659, 0.7870, 0.0000],
            [0.8170, 0.8296, 0.0000],
            [0.8853, 0.8896, 0.4113],
            [0.9481, 0.9486, 0.7165],
        ]
    )
    return ColorMapper.from_palette_array(palette, range=range, **traits)


def CoolWarm(range, **traits):
    """Generator of Kenneth Moreland's CoolWarm colormap.

    Blue-White-Red with smooth lightness transitions. Good for applying to 3D
    surfaces or otherwise have extra shading applied.

    http://www.sandia.gov/~kmorel/documents/ColorMaps/ColorMapsExpanded.pdf
    """
    cool = array([59, 76, 192]) / 255.0
    warm = array([180, 4, 38]) / 255.0
    palette = generate_diverging_palette(cool, warm, 256)
    return ColorMapper.from_palette_array(palette, range=range, **traits)


def CubeHelix(range, **traits):
    """Generator of Dave Green's CubeHelix colormap.

    Sequential colormap with a linear lightness increasing from black to white
    deviating away from gray in a tapered helix.

    https://www.mrao.cam.ac.uk/~dag/CUBEHELIX/
    """
    palette = generate_cubehelix_palette()
    return ColorMapper.from_palette_array(palette, range=range, **traits)


# An MIT licensed, colorblind-friendly heatmap from Wistia:
#   https://github.com/wistia/heatmap-palette
#   http://wistia.com/blog/heatmaps-for-colorblindness
#
def wistia(range, **traits):
    """An MIT licensed, colorblind-friendly heatmap from Wistia

    See:
        https://github.com/wistia/heatmap-palette
        http://wistia.com/blog/heatmaps-for-colorblindness.
    """
    _data = {
        "red": [
            (0.0, 0.8941176470588236, 0.8941176470588236),
            (0.25, 1.0, 1.0),
            (0.5, 1.0, 1.0),
            (0.75, 1.0, 1.0),
            (1.0, 0.9882352941176471, 0.9882352941176471),
        ],
        "green": [
            (0.0, 1.0, 1.0),
            (0.25, 0.9098039215686274, 0.9098039215686274),
            (0.5, 0.7411764705882353, 0.7411764705882353),
            (0.75, 0.6274509803921569, 0.6274509803921569),
            (1.0, 0.4980392156862745, 0.4980392156862745),
        ],
        "blue": [
            (0.0, 0.47843137254901963, 0.47843137254901963),
            (0.25, 0.10196078431372549, 0.10196078431372549),
            (0.5, 0.0, 0.0),
            (0.75, 0.0, 0.0),
            (1.0, 0.0, 0.0),
        ],
    }
    return ColorMapper.from_segment_map(_data, range=range, **traits)


# New colormaps from matplotlib's 2015 update


def magma(range, **traits):
    """ Generator for the 'magma' colormap from matplolib """
    _data = [
        [0.001462, 0.000466, 0.013866],
        [0.002258, 0.001295, 0.018331],
        [0.003279, 0.002305, 0.023708],
        [0.004512, 0.003490, 0.029965],
        [0.005950, 0.004843, 0.037130],
        [0.007588, 0.006356, 0.044973],
        [0.009426, 0.008022, 0.052844],
        [0.011465, 0.009828, 0.060750],
        [0.013708, 0.011771, 0.068667],
        [0.016156, 0.013840, 0.076603],
        [0.018815, 0.016026, 0.084584],
        [0.021692, 0.018320, 0.092610],
        [0.024792, 0.020715, 0.100676],
        [0.028123, 0.023201, 0.108787],
        [0.031696, 0.025765, 0.116965],
        [0.035520, 0.028397, 0.125209],
        [0.039608, 0.031090, 0.133515],
        [0.043830, 0.033830, 0.141886],
        [0.048062, 0.036607, 0.150327],
        [0.052320, 0.039407, 0.158841],
        [0.056615, 0.042160, 0.167446],
        [0.060949, 0.044794, 0.176129],
        [0.065330, 0.047318, 0.184892],
        [0.069764, 0.049726, 0.193735],
        [0.074257, 0.052017, 0.202660],
        [0.078815, 0.054184, 0.211667],
        [0.083446, 0.056225, 0.220755],
        [0.088155, 0.058133, 0.229922],
        [0.092949, 0.059904, 0.239164],
        [0.097833, 0.061531, 0.248477],
        [0.102815, 0.063010, 0.257854],
        [0.107899, 0.064335, 0.267289],
        [0.113094, 0.065492, 0.276784],
        [0.118405, 0.066479, 0.286321],
        [0.123833, 0.067295, 0.295879],
        [0.129380, 0.067935, 0.305443],
        [0.135053, 0.068391, 0.315000],
        [0.140858, 0.068654, 0.324538],
        [0.146785, 0.068738, 0.334011],
        [0.152839, 0.068637, 0.343404],
        [0.159018, 0.068354, 0.352688],
        [0.165308, 0.067911, 0.361816],
        [0.171713, 0.067305, 0.370771],
        [0.178212, 0.066576, 0.379497],
        [0.184801, 0.065732, 0.387973],
        [0.191460, 0.064818, 0.396152],
        [0.198177, 0.063862, 0.404009],
        [0.204935, 0.062907, 0.411514],
        [0.211718, 0.061992, 0.418647],
        [0.218512, 0.061158, 0.425392],
        [0.225302, 0.060445, 0.431742],
        [0.232077, 0.059889, 0.437695],
        [0.238826, 0.059517, 0.443256],
        [0.245543, 0.059352, 0.448436],
        [0.252220, 0.059415, 0.453248],
        [0.258857, 0.059706, 0.457710],
        [0.265447, 0.060237, 0.461840],
        [0.271994, 0.060994, 0.465660],
        [0.278493, 0.061978, 0.469190],
        [0.284951, 0.063168, 0.472451],
        [0.291366, 0.064553, 0.475462],
        [0.297740, 0.066117, 0.478243],
        [0.304081, 0.067835, 0.480812],
        [0.310382, 0.069702, 0.483186],
        [0.316654, 0.071690, 0.485380],
        [0.322899, 0.073782, 0.487408],
        [0.329114, 0.075972, 0.489287],
        [0.335308, 0.078236, 0.491024],
        [0.341482, 0.080564, 0.492631],
        [0.347636, 0.082946, 0.494121],
        [0.353773, 0.085373, 0.495501],
        [0.359898, 0.087831, 0.496778],
        [0.366012, 0.090314, 0.497960],
        [0.372116, 0.092816, 0.499053],
        [0.378211, 0.095332, 0.500067],
        [0.384299, 0.097855, 0.501002],
        [0.390384, 0.100379, 0.501864],
        [0.396467, 0.102902, 0.502658],
        [0.402548, 0.105420, 0.503386],
        [0.408629, 0.107930, 0.504052],
        [0.414709, 0.110431, 0.504662],
        [0.420791, 0.112920, 0.505215],
        [0.426877, 0.115395, 0.505714],
        [0.432967, 0.117855, 0.506160],
        [0.439062, 0.120298, 0.506555],
        [0.445163, 0.122724, 0.506901],
        [0.451271, 0.125132, 0.507198],
        [0.457386, 0.127522, 0.507448],
        [0.463508, 0.129893, 0.507652],
        [0.469640, 0.132245, 0.507809],
        [0.475780, 0.134577, 0.507921],
        [0.481929, 0.136891, 0.507989],
        [0.488088, 0.139186, 0.508011],
        [0.494258, 0.141462, 0.507988],
        [0.500438, 0.143719, 0.507920],
        [0.506629, 0.145958, 0.507806],
        [0.512831, 0.148179, 0.507648],
        [0.519045, 0.150383, 0.507443],
        [0.525270, 0.152569, 0.507192],
        [0.531507, 0.154739, 0.506895],
        [0.537755, 0.156894, 0.506551],
        [0.544015, 0.159033, 0.506159],
        [0.550287, 0.161158, 0.505719],
        [0.556571, 0.163269, 0.505230],
        [0.562866, 0.165368, 0.504692],
        [0.569172, 0.167454, 0.504105],
        [0.575490, 0.169530, 0.503466],
        [0.581819, 0.171596, 0.502777],
        [0.588158, 0.173652, 0.502035],
        [0.594508, 0.175701, 0.501241],
        [0.600868, 0.177743, 0.500394],
        [0.607238, 0.179779, 0.499492],
        [0.613617, 0.181811, 0.498536],
        [0.620005, 0.183840, 0.497524],
        [0.626401, 0.185867, 0.496456],
        [0.632805, 0.187893, 0.495332],
        [0.639216, 0.189921, 0.494150],
        [0.645633, 0.191952, 0.492910],
        [0.652056, 0.193986, 0.491611],
        [0.658483, 0.196027, 0.490253],
        [0.664915, 0.198075, 0.488836],
        [0.671349, 0.200133, 0.487358],
        [0.677786, 0.202203, 0.485819],
        [0.684224, 0.204286, 0.484219],
        [0.690661, 0.206384, 0.482558],
        [0.697098, 0.208501, 0.480835],
        [0.703532, 0.210638, 0.479049],
        [0.709962, 0.212797, 0.477201],
        [0.716387, 0.214982, 0.475290],
        [0.722805, 0.217194, 0.473316],
        [0.729216, 0.219437, 0.471279],
        [0.735616, 0.221713, 0.469180],
        [0.742004, 0.224025, 0.467018],
        [0.748378, 0.226377, 0.464794],
        [0.754737, 0.228772, 0.462509],
        [0.761077, 0.231214, 0.460162],
        [0.767398, 0.233705, 0.457755],
        [0.773695, 0.236249, 0.455289],
        [0.779968, 0.238851, 0.452765],
        [0.786212, 0.241514, 0.450184],
        [0.792427, 0.244242, 0.447543],
        [0.798608, 0.247040, 0.444848],
        [0.804752, 0.249911, 0.442102],
        [0.810855, 0.252861, 0.439305],
        [0.816914, 0.255895, 0.436461],
        [0.822926, 0.259016, 0.433573],
        [0.828886, 0.262229, 0.430644],
        [0.834791, 0.265540, 0.427671],
        [0.840636, 0.268953, 0.424666],
        [0.846416, 0.272473, 0.421631],
        [0.852126, 0.276106, 0.418573],
        [0.857763, 0.279857, 0.415496],
        [0.863320, 0.283729, 0.412403],
        [0.868793, 0.287728, 0.409303],
        [0.874176, 0.291859, 0.406205],
        [0.879464, 0.296125, 0.403118],
        [0.884651, 0.300530, 0.400047],
        [0.889731, 0.305079, 0.397002],
        [0.894700, 0.309773, 0.393995],
        [0.899552, 0.314616, 0.391037],
        [0.904281, 0.319610, 0.388137],
        [0.908884, 0.324755, 0.385308],
        [0.913354, 0.330052, 0.382563],
        [0.917689, 0.335500, 0.379915],
        [0.921884, 0.341098, 0.377376],
        [0.925937, 0.346844, 0.374959],
        [0.929845, 0.352734, 0.372677],
        [0.933606, 0.358764, 0.370541],
        [0.937221, 0.364929, 0.368567],
        [0.940687, 0.371224, 0.366762],
        [0.944006, 0.377643, 0.365136],
        [0.947180, 0.384178, 0.363701],
        [0.950210, 0.390820, 0.362468],
        [0.953099, 0.397563, 0.361438],
        [0.955849, 0.404400, 0.360619],
        [0.958464, 0.411324, 0.360014],
        [0.960949, 0.418323, 0.359630],
        [0.963310, 0.425390, 0.359469],
        [0.965549, 0.432519, 0.359529],
        [0.967671, 0.439703, 0.359810],
        [0.969680, 0.446936, 0.360311],
        [0.971582, 0.454210, 0.361030],
        [0.973381, 0.461520, 0.361965],
        [0.975082, 0.468861, 0.363111],
        [0.976690, 0.476226, 0.364466],
        [0.978210, 0.483612, 0.366025],
        [0.979645, 0.491014, 0.367783],
        [0.981000, 0.498428, 0.369734],
        [0.982279, 0.505851, 0.371874],
        [0.983485, 0.513280, 0.374198],
        [0.984622, 0.520713, 0.376698],
        [0.985693, 0.528148, 0.379371],
        [0.986700, 0.535582, 0.382210],
        [0.987646, 0.543015, 0.385210],
        [0.988533, 0.550446, 0.388365],
        [0.989363, 0.557873, 0.391671],
        [0.990138, 0.565296, 0.395122],
        [0.990871, 0.572706, 0.398714],
        [0.991558, 0.580107, 0.402441],
        [0.992196, 0.587502, 0.406299],
        [0.992785, 0.594891, 0.410283],
        [0.993326, 0.602275, 0.414390],
        [0.993834, 0.609644, 0.418613],
        [0.994309, 0.616999, 0.422950],
        [0.994738, 0.624350, 0.427397],
        [0.995122, 0.631696, 0.431951],
        [0.995480, 0.639027, 0.436607],
        [0.995810, 0.646344, 0.441361],
        [0.996096, 0.653659, 0.446213],
        [0.996341, 0.660969, 0.451160],
        [0.996580, 0.668256, 0.456192],
        [0.996775, 0.675541, 0.461314],
        [0.996925, 0.682828, 0.466526],
        [0.997077, 0.690088, 0.471811],
        [0.997186, 0.697349, 0.477182],
        [0.997254, 0.704611, 0.482635],
        [0.997325, 0.711848, 0.488154],
        [0.997351, 0.719089, 0.493755],
        [0.997351, 0.726324, 0.499428],
        [0.997341, 0.733545, 0.505167],
        [0.997285, 0.740772, 0.510983],
        [0.997228, 0.747981, 0.516859],
        [0.997138, 0.755190, 0.522806],
        [0.997019, 0.762398, 0.528821],
        [0.996898, 0.769591, 0.534892],
        [0.996727, 0.776795, 0.541039],
        [0.996571, 0.783977, 0.547233],
        [0.996369, 0.791167, 0.553499],
        [0.996162, 0.798348, 0.559820],
        [0.995932, 0.805527, 0.566202],
        [0.995680, 0.812706, 0.572645],
        [0.995424, 0.819875, 0.579140],
        [0.995131, 0.827052, 0.585701],
        [0.994851, 0.834213, 0.592307],
        [0.994524, 0.841387, 0.598983],
        [0.994222, 0.848540, 0.605696],
        [0.993866, 0.855711, 0.612482],
        [0.993545, 0.862859, 0.619299],
        [0.993170, 0.870024, 0.626189],
        [0.992831, 0.877168, 0.633109],
        [0.992440, 0.884330, 0.640099],
        [0.992089, 0.891470, 0.647116],
        [0.991688, 0.898627, 0.654202],
        [0.991332, 0.905763, 0.661309],
        [0.990930, 0.912915, 0.668481],
        [0.990570, 0.920049, 0.675675],
        [0.990175, 0.927196, 0.682926],
        [0.989815, 0.934329, 0.690198],
        [0.989434, 0.941470, 0.697519],
        [0.989077, 0.948604, 0.704863],
        [0.988717, 0.955742, 0.712242],
        [0.988367, 0.962878, 0.719649],
        [0.988033, 0.970012, 0.727077],
        [0.987691, 0.977154, 0.734536],
        [0.987387, 0.984288, 0.742002],
        [0.987053, 0.991438, 0.749504],
    ]
    return ColorMapper.from_palette_array(_data, range=range, **traits)


def inferno(range, **traits):
    """ Generator for the 'inferno' colormap from matplolib """
    _data = [
        [0.001462, 0.000466, 0.013866],
        [0.002267, 0.001270, 0.018570],
        [0.003299, 0.002249, 0.024239],
        [0.004547, 0.003392, 0.030909],
        [0.006006, 0.004692, 0.038558],
        [0.007676, 0.006136, 0.046836],
        [0.009561, 0.007713, 0.055143],
        [0.011663, 0.009417, 0.063460],
        [0.013995, 0.011225, 0.071862],
        [0.016561, 0.013136, 0.080282],
        [0.019373, 0.015133, 0.088767],
        [0.022447, 0.017199, 0.097327],
        [0.025793, 0.019331, 0.105930],
        [0.029432, 0.021503, 0.114621],
        [0.033385, 0.023702, 0.123397],
        [0.037668, 0.025921, 0.132232],
        [0.042253, 0.028139, 0.141141],
        [0.046915, 0.030324, 0.150164],
        [0.051644, 0.032474, 0.159254],
        [0.056449, 0.034569, 0.168414],
        [0.061340, 0.036590, 0.177642],
        [0.066331, 0.038504, 0.186962],
        [0.071429, 0.040294, 0.196354],
        [0.076637, 0.041905, 0.205799],
        [0.081962, 0.043328, 0.215289],
        [0.087411, 0.044556, 0.224813],
        [0.092990, 0.045583, 0.234358],
        [0.098702, 0.046402, 0.243904],
        [0.104551, 0.047008, 0.253430],
        [0.110536, 0.047399, 0.262912],
        [0.116656, 0.047574, 0.272321],
        [0.122908, 0.047536, 0.281624],
        [0.129285, 0.047293, 0.290788],
        [0.135778, 0.046856, 0.299776],
        [0.142378, 0.046242, 0.308553],
        [0.149073, 0.045468, 0.317085],
        [0.155850, 0.044559, 0.325338],
        [0.162689, 0.043554, 0.333277],
        [0.169575, 0.042489, 0.340874],
        [0.176493, 0.041402, 0.348111],
        [0.183429, 0.040329, 0.354971],
        [0.190367, 0.039309, 0.361447],
        [0.197297, 0.038400, 0.367535],
        [0.204209, 0.037632, 0.373238],
        [0.211095, 0.037030, 0.378563],
        [0.217949, 0.036615, 0.383522],
        [0.224763, 0.036405, 0.388129],
        [0.231538, 0.036405, 0.392400],
        [0.238273, 0.036621, 0.396353],
        [0.244967, 0.037055, 0.400007],
        [0.251620, 0.037705, 0.403378],
        [0.258234, 0.038571, 0.406485],
        [0.264810, 0.039647, 0.409345],
        [0.271347, 0.040922, 0.411976],
        [0.277850, 0.042353, 0.414392],
        [0.284321, 0.043933, 0.416608],
        [0.290763, 0.045644, 0.418637],
        [0.297178, 0.047470, 0.420491],
        [0.303568, 0.049396, 0.422182],
        [0.309935, 0.051407, 0.423721],
        [0.316282, 0.053490, 0.425116],
        [0.322610, 0.055634, 0.426377],
        [0.328921, 0.057827, 0.427511],
        [0.335217, 0.060060, 0.428524],
        [0.341500, 0.062325, 0.429425],
        [0.347771, 0.064616, 0.430217],
        [0.354032, 0.066925, 0.430906],
        [0.360284, 0.069247, 0.431497],
        [0.366529, 0.071579, 0.431994],
        [0.372768, 0.073915, 0.432400],
        [0.379001, 0.076253, 0.432719],
        [0.385228, 0.078591, 0.432955],
        [0.391453, 0.080927, 0.433109],
        [0.397674, 0.083257, 0.433183],
        [0.403894, 0.085580, 0.433179],
        [0.410113, 0.087896, 0.433098],
        [0.416331, 0.090203, 0.432943],
        [0.422549, 0.092501, 0.432714],
        [0.428768, 0.094790, 0.432412],
        [0.434987, 0.097069, 0.432039],
        [0.441207, 0.099338, 0.431594],
        [0.447428, 0.101597, 0.431080],
        [0.453651, 0.103848, 0.430498],
        [0.459875, 0.106089, 0.429846],
        [0.466100, 0.108322, 0.429125],
        [0.472328, 0.110547, 0.428334],
        [0.478558, 0.112764, 0.427475],
        [0.484789, 0.114974, 0.426548],
        [0.491022, 0.117179, 0.425552],
        [0.497257, 0.119379, 0.424488],
        [0.503493, 0.121575, 0.423356],
        [0.509730, 0.123769, 0.422156],
        [0.515967, 0.125960, 0.420887],
        [0.522206, 0.128150, 0.419549],
        [0.528444, 0.130341, 0.418142],
        [0.534683, 0.132534, 0.416667],
        [0.540920, 0.134729, 0.415123],
        [0.547157, 0.136929, 0.413511],
        [0.553392, 0.139134, 0.411829],
        [0.559624, 0.141346, 0.410078],
        [0.565854, 0.143567, 0.408258],
        [0.572081, 0.145797, 0.406369],
        [0.578304, 0.148039, 0.404411],
        [0.584521, 0.150294, 0.402385],
        [0.590734, 0.152563, 0.400290],
        [0.596940, 0.154848, 0.398125],
        [0.603139, 0.157151, 0.395891],
        [0.609330, 0.159474, 0.393589],
        [0.615513, 0.161817, 0.391219],
        [0.621685, 0.164184, 0.388781],
        [0.627847, 0.166575, 0.386276],
        [0.633998, 0.168992, 0.383704],
        [0.640135, 0.171438, 0.381065],
        [0.646260, 0.173914, 0.378359],
        [0.652369, 0.176421, 0.375586],
        [0.658463, 0.178962, 0.372748],
        [0.664540, 0.181539, 0.369846],
        [0.670599, 0.184153, 0.366879],
        [0.676638, 0.186807, 0.363849],
        [0.682656, 0.189501, 0.360757],
        [0.688653, 0.192239, 0.357603],
        [0.694627, 0.195021, 0.354388],
        [0.700576, 0.197851, 0.351113],
        [0.706500, 0.200728, 0.347777],
        [0.712396, 0.203656, 0.344383],
        [0.718264, 0.206636, 0.340931],
        [0.724103, 0.209670, 0.337424],
        [0.729909, 0.212759, 0.333861],
        [0.735683, 0.215906, 0.330245],
        [0.741423, 0.219112, 0.326576],
        [0.747127, 0.222378, 0.322856],
        [0.752794, 0.225706, 0.319085],
        [0.758422, 0.229097, 0.315266],
        [0.764010, 0.232554, 0.311399],
        [0.769556, 0.236077, 0.307485],
        [0.775059, 0.239667, 0.303526],
        [0.780517, 0.243327, 0.299523],
        [0.785929, 0.247056, 0.295477],
        [0.791293, 0.250856, 0.291390],
        [0.796607, 0.254728, 0.287264],
        [0.801871, 0.258674, 0.283099],
        [0.807082, 0.262692, 0.278898],
        [0.812239, 0.266786, 0.274661],
        [0.817341, 0.270954, 0.270390],
        [0.822386, 0.275197, 0.266085],
        [0.827372, 0.279517, 0.261750],
        [0.832299, 0.283913, 0.257383],
        [0.837165, 0.288385, 0.252988],
        [0.841969, 0.292933, 0.248564],
        [0.846709, 0.297559, 0.244113],
        [0.851384, 0.302260, 0.239636],
        [0.855992, 0.307038, 0.235133],
        [0.860533, 0.311892, 0.230606],
        [0.865006, 0.316822, 0.226055],
        [0.869409, 0.321827, 0.221482],
        [0.873741, 0.326906, 0.216886],
        [0.878001, 0.332060, 0.212268],
        [0.882188, 0.337287, 0.207628],
        [0.886302, 0.342586, 0.202968],
        [0.890341, 0.347957, 0.198286],
        [0.894305, 0.353399, 0.193584],
        [0.898192, 0.358911, 0.188860],
        [0.902003, 0.364492, 0.184116],
        [0.905735, 0.370140, 0.179350],
        [0.909390, 0.375856, 0.174563],
        [0.912966, 0.381636, 0.169755],
        [0.916462, 0.387481, 0.164924],
        [0.919879, 0.393389, 0.160070],
        [0.923215, 0.399359, 0.155193],
        [0.926470, 0.405389, 0.150292],
        [0.929644, 0.411479, 0.145367],
        [0.932737, 0.417627, 0.140417],
        [0.935747, 0.423831, 0.135440],
        [0.938675, 0.430091, 0.130438],
        [0.941521, 0.436405, 0.125409],
        [0.944285, 0.442772, 0.120354],
        [0.946965, 0.449191, 0.115272],
        [0.949562, 0.455660, 0.110164],
        [0.952075, 0.462178, 0.105031],
        [0.954506, 0.468744, 0.099874],
        [0.956852, 0.475356, 0.094695],
        [0.959114, 0.482014, 0.089499],
        [0.961293, 0.488716, 0.084289],
        [0.963387, 0.495462, 0.079073],
        [0.965397, 0.502249, 0.073859],
        [0.967322, 0.509078, 0.068659],
        [0.969163, 0.515946, 0.063488],
        [0.970919, 0.522853, 0.058367],
        [0.972590, 0.529798, 0.053324],
        [0.974176, 0.536780, 0.048392],
        [0.975677, 0.543798, 0.043618],
        [0.977092, 0.550850, 0.039050],
        [0.978422, 0.557937, 0.034931],
        [0.979666, 0.565057, 0.031409],
        [0.980824, 0.572209, 0.028508],
        [0.981895, 0.579392, 0.026250],
        [0.982881, 0.586606, 0.024661],
        [0.983779, 0.593849, 0.023770],
        [0.984591, 0.601122, 0.023606],
        [0.985315, 0.608422, 0.024202],
        [0.985952, 0.615750, 0.025592],
        [0.986502, 0.623105, 0.027814],
        [0.986964, 0.630485, 0.030908],
        [0.987337, 0.637890, 0.034916],
        [0.987622, 0.645320, 0.039886],
        [0.987819, 0.652773, 0.045581],
        [0.987926, 0.660250, 0.051750],
        [0.987945, 0.667748, 0.058329],
        [0.987874, 0.675267, 0.065257],
        [0.987714, 0.682807, 0.072489],
        [0.987464, 0.690366, 0.079990],
        [0.987124, 0.697944, 0.087731],
        [0.986694, 0.705540, 0.095694],
        [0.986175, 0.713153, 0.103863],
        [0.985566, 0.720782, 0.112229],
        [0.984865, 0.728427, 0.120785],
        [0.984075, 0.736087, 0.129527],
        [0.983196, 0.743758, 0.138453],
        [0.982228, 0.751442, 0.147565],
        [0.981173, 0.759135, 0.156863],
        [0.980032, 0.766837, 0.166353],
        [0.978806, 0.774545, 0.176037],
        [0.977497, 0.782258, 0.185923],
        [0.976108, 0.789974, 0.196018],
        [0.974638, 0.797692, 0.206332],
        [0.973088, 0.805409, 0.216877],
        [0.971468, 0.813122, 0.227658],
        [0.969783, 0.820825, 0.238686],
        [0.968041, 0.828515, 0.249972],
        [0.966243, 0.836191, 0.261534],
        [0.964394, 0.843848, 0.273391],
        [0.962517, 0.851476, 0.285546],
        [0.960626, 0.859069, 0.298010],
        [0.958720, 0.866624, 0.310820],
        [0.956834, 0.874129, 0.323974],
        [0.954997, 0.881569, 0.337475],
        [0.953215, 0.888942, 0.351369],
        [0.951546, 0.896226, 0.365627],
        [0.950018, 0.903409, 0.380271],
        [0.948683, 0.910473, 0.395289],
        [0.947594, 0.917399, 0.410665],
        [0.946809, 0.924168, 0.426373],
        [0.946392, 0.930761, 0.442367],
        [0.946403, 0.937159, 0.458592],
        [0.946903, 0.943348, 0.474970],
        [0.947937, 0.949318, 0.491426],
        [0.949545, 0.955063, 0.507860],
        [0.951740, 0.960587, 0.524203],
        [0.954529, 0.965896, 0.540361],
        [0.957896, 0.971003, 0.556275],
        [0.961812, 0.975924, 0.571925],
        [0.966249, 0.980678, 0.587206],
        [0.971162, 0.985282, 0.602154],
        [0.976511, 0.989753, 0.616760],
        [0.982257, 0.994109, 0.631017],
        [0.988362, 0.998364, 0.644924],
    ]
    return ColorMapper.from_palette_array(_data, range=range, **traits)


def plasma(range, **traits):
    """ Generator for the 'plasma' colormap from matplolib """
    _data = [
        [0.050383, 0.029803, 0.527975],
        [0.063536, 0.028426, 0.533124],
        [0.075353, 0.027206, 0.538007],
        [0.086222, 0.026125, 0.542658],
        [0.096379, 0.025165, 0.547103],
        [0.105980, 0.024309, 0.551368],
        [0.115124, 0.023556, 0.555468],
        [0.123903, 0.022878, 0.559423],
        [0.132381, 0.022258, 0.563250],
        [0.140603, 0.021687, 0.566959],
        [0.148607, 0.021154, 0.570562],
        [0.156421, 0.020651, 0.574065],
        [0.164070, 0.020171, 0.577478],
        [0.171574, 0.019706, 0.580806],
        [0.178950, 0.019252, 0.584054],
        [0.186213, 0.018803, 0.587228],
        [0.193374, 0.018354, 0.590330],
        [0.200445, 0.017902, 0.593364],
        [0.207435, 0.017442, 0.596333],
        [0.214350, 0.016973, 0.599239],
        [0.221197, 0.016497, 0.602083],
        [0.227983, 0.016007, 0.604867],
        [0.234715, 0.015502, 0.607592],
        [0.241396, 0.014979, 0.610259],
        [0.248032, 0.014439, 0.612868],
        [0.254627, 0.013882, 0.615419],
        [0.261183, 0.013308, 0.617911],
        [0.267703, 0.012716, 0.620346],
        [0.274191, 0.012109, 0.622722],
        [0.280648, 0.011488, 0.625038],
        [0.287076, 0.010855, 0.627295],
        [0.293478, 0.010213, 0.629490],
        [0.299855, 0.009561, 0.631624],
        [0.306210, 0.008902, 0.633694],
        [0.312543, 0.008239, 0.635700],
        [0.318856, 0.007576, 0.637640],
        [0.325150, 0.006915, 0.639512],
        [0.331426, 0.006261, 0.641316],
        [0.337683, 0.005618, 0.643049],
        [0.343925, 0.004991, 0.644710],
        [0.350150, 0.004382, 0.646298],
        [0.356359, 0.003798, 0.647810],
        [0.362553, 0.003243, 0.649245],
        [0.368733, 0.002724, 0.650601],
        [0.374897, 0.002245, 0.651876],
        [0.381047, 0.001814, 0.653068],
        [0.387183, 0.001434, 0.654177],
        [0.393304, 0.001114, 0.655199],
        [0.399411, 0.000859, 0.656133],
        [0.405503, 0.000678, 0.656977],
        [0.411580, 0.000577, 0.657730],
        [0.417642, 0.000564, 0.658390],
        [0.423689, 0.000646, 0.658956],
        [0.429719, 0.000831, 0.659425],
        [0.435734, 0.001127, 0.659797],
        [0.441732, 0.001540, 0.660069],
        [0.447714, 0.002080, 0.660240],
        [0.453677, 0.002755, 0.660310],
        [0.459623, 0.003574, 0.660277],
        [0.465550, 0.004545, 0.660139],
        [0.471457, 0.005678, 0.659897],
        [0.477344, 0.006980, 0.659549],
        [0.483210, 0.008460, 0.659095],
        [0.489055, 0.010127, 0.658534],
        [0.494877, 0.011990, 0.657865],
        [0.500678, 0.014055, 0.657088],
        [0.506454, 0.016333, 0.656202],
        [0.512206, 0.018833, 0.655209],
        [0.517933, 0.021563, 0.654109],
        [0.523633, 0.024532, 0.652901],
        [0.529306, 0.027747, 0.651586],
        [0.534952, 0.031217, 0.650165],
        [0.540570, 0.034950, 0.648640],
        [0.546157, 0.038954, 0.647010],
        [0.551715, 0.043136, 0.645277],
        [0.557243, 0.047331, 0.643443],
        [0.562738, 0.051545, 0.641509],
        [0.568201, 0.055778, 0.639477],
        [0.573632, 0.060028, 0.637349],
        [0.579029, 0.064296, 0.635126],
        [0.584391, 0.068579, 0.632812],
        [0.589719, 0.072878, 0.630408],
        [0.595011, 0.077190, 0.627917],
        [0.600266, 0.081516, 0.625342],
        [0.605485, 0.085854, 0.622686],
        [0.610667, 0.090204, 0.619951],
        [0.615812, 0.094564, 0.617140],
        [0.620919, 0.098934, 0.614257],
        [0.625987, 0.103312, 0.611305],
        [0.631017, 0.107699, 0.608287],
        [0.636008, 0.112092, 0.605205],
        [0.640959, 0.116492, 0.602065],
        [0.645872, 0.120898, 0.598867],
        [0.650746, 0.125309, 0.595617],
        [0.655580, 0.129725, 0.592317],
        [0.660374, 0.134144, 0.588971],
        [0.665129, 0.138566, 0.585582],
        [0.669845, 0.142992, 0.582154],
        [0.674522, 0.147419, 0.578688],
        [0.679160, 0.151848, 0.575189],
        [0.683758, 0.156278, 0.571660],
        [0.688318, 0.160709, 0.568103],
        [0.692840, 0.165141, 0.564522],
        [0.697324, 0.169573, 0.560919],
        [0.701769, 0.174005, 0.557296],
        [0.706178, 0.178437, 0.553657],
        [0.710549, 0.182868, 0.550004],
        [0.714883, 0.187299, 0.546338],
        [0.719181, 0.191729, 0.542663],
        [0.723444, 0.196158, 0.538981],
        [0.727670, 0.200586, 0.535293],
        [0.731862, 0.205013, 0.531601],
        [0.736019, 0.209439, 0.527908],
        [0.740143, 0.213864, 0.524216],
        [0.744232, 0.218288, 0.520524],
        [0.748289, 0.222711, 0.516834],
        [0.752312, 0.227133, 0.513149],
        [0.756304, 0.231555, 0.509468],
        [0.760264, 0.235976, 0.505794],
        [0.764193, 0.240396, 0.502126],
        [0.768090, 0.244817, 0.498465],
        [0.771958, 0.249237, 0.494813],
        [0.775796, 0.253658, 0.491171],
        [0.779604, 0.258078, 0.487539],
        [0.783383, 0.262500, 0.483918],
        [0.787133, 0.266922, 0.480307],
        [0.790855, 0.271345, 0.476706],
        [0.794549, 0.275770, 0.473117],
        [0.798216, 0.280197, 0.469538],
        [0.801855, 0.284626, 0.465971],
        [0.805467, 0.289057, 0.462415],
        [0.809052, 0.293491, 0.458870],
        [0.812612, 0.297928, 0.455338],
        [0.816144, 0.302368, 0.451816],
        [0.819651, 0.306812, 0.448306],
        [0.823132, 0.311261, 0.444806],
        [0.826588, 0.315714, 0.441316],
        [0.830018, 0.320172, 0.437836],
        [0.833422, 0.324635, 0.434366],
        [0.836801, 0.329105, 0.430905],
        [0.840155, 0.333580, 0.427455],
        [0.843484, 0.338062, 0.424013],
        [0.846788, 0.342551, 0.420579],
        [0.850066, 0.347048, 0.417153],
        [0.853319, 0.351553, 0.413734],
        [0.856547, 0.356066, 0.410322],
        [0.859750, 0.360588, 0.406917],
        [0.862927, 0.365119, 0.403519],
        [0.866078, 0.369660, 0.400126],
        [0.869203, 0.374212, 0.396738],
        [0.872303, 0.378774, 0.393355],
        [0.875376, 0.383347, 0.389976],
        [0.878423, 0.387932, 0.386600],
        [0.881443, 0.392529, 0.383229],
        [0.884436, 0.397139, 0.379860],
        [0.887402, 0.401762, 0.376494],
        [0.890340, 0.406398, 0.373130],
        [0.893250, 0.411048, 0.369768],
        [0.896131, 0.415712, 0.366407],
        [0.898984, 0.420392, 0.363047],
        [0.901807, 0.425087, 0.359688],
        [0.904601, 0.429797, 0.356329],
        [0.907365, 0.434524, 0.352970],
        [0.910098, 0.439268, 0.349610],
        [0.912800, 0.444029, 0.346251],
        [0.915471, 0.448807, 0.342890],
        [0.918109, 0.453603, 0.339529],
        [0.920714, 0.458417, 0.336166],
        [0.923287, 0.463251, 0.332801],
        [0.925825, 0.468103, 0.329435],
        [0.928329, 0.472975, 0.326067],
        [0.930798, 0.477867, 0.322697],
        [0.933232, 0.482780, 0.319325],
        [0.935630, 0.487712, 0.315952],
        [0.937990, 0.492667, 0.312575],
        [0.940313, 0.497642, 0.309197],
        [0.942598, 0.502639, 0.305816],
        [0.944844, 0.507658, 0.302433],
        [0.947051, 0.512699, 0.299049],
        [0.949217, 0.517763, 0.295662],
        [0.951344, 0.522850, 0.292275],
        [0.953428, 0.527960, 0.288883],
        [0.955470, 0.533093, 0.285490],
        [0.957469, 0.538250, 0.282096],
        [0.959424, 0.543431, 0.278701],
        [0.961336, 0.548636, 0.275305],
        [0.963203, 0.553865, 0.271909],
        [0.965024, 0.559118, 0.268513],
        [0.966798, 0.564396, 0.265118],
        [0.968526, 0.569700, 0.261721],
        [0.970205, 0.575028, 0.258325],
        [0.971835, 0.580382, 0.254931],
        [0.973416, 0.585761, 0.251540],
        [0.974947, 0.591165, 0.248151],
        [0.976428, 0.596595, 0.244767],
        [0.977856, 0.602051, 0.241387],
        [0.979233, 0.607532, 0.238013],
        [0.980556, 0.613039, 0.234646],
        [0.981826, 0.618572, 0.231287],
        [0.983041, 0.624131, 0.227937],
        [0.984199, 0.629718, 0.224595],
        [0.985301, 0.635330, 0.221265],
        [0.986345, 0.640969, 0.217948],
        [0.987332, 0.646633, 0.214648],
        [0.988260, 0.652325, 0.211364],
        [0.989128, 0.658043, 0.208100],
        [0.989935, 0.663787, 0.204859],
        [0.990681, 0.669558, 0.201642],
        [0.991365, 0.675355, 0.198453],
        [0.991985, 0.681179, 0.195295],
        [0.992541, 0.687030, 0.192170],
        [0.993032, 0.692907, 0.189084],
        [0.993456, 0.698810, 0.186041],
        [0.993814, 0.704741, 0.183043],
        [0.994103, 0.710698, 0.180097],
        [0.994324, 0.716681, 0.177208],
        [0.994474, 0.722691, 0.174381],
        [0.994553, 0.728728, 0.171622],
        [0.994561, 0.734791, 0.168938],
        [0.994495, 0.740880, 0.166335],
        [0.994355, 0.746995, 0.163821],
        [0.994141, 0.753137, 0.161404],
        [0.993851, 0.759304, 0.159092],
        [0.993482, 0.765499, 0.156891],
        [0.993033, 0.771720, 0.154808],
        [0.992505, 0.777967, 0.152855],
        [0.991897, 0.784239, 0.151042],
        [0.991209, 0.790537, 0.149377],
        [0.990439, 0.796859, 0.147870],
        [0.989587, 0.803205, 0.146529],
        [0.988648, 0.809579, 0.145357],
        [0.987621, 0.815978, 0.144363],
        [0.986509, 0.822401, 0.143557],
        [0.985314, 0.828846, 0.142945],
        [0.984031, 0.835315, 0.142528],
        [0.982653, 0.841812, 0.142303],
        [0.981190, 0.848329, 0.142279],
        [0.979644, 0.854866, 0.142453],
        [0.977995, 0.861432, 0.142808],
        [0.976265, 0.868016, 0.143351],
        [0.974443, 0.874622, 0.144061],
        [0.972530, 0.881250, 0.144923],
        [0.970533, 0.887896, 0.145919],
        [0.968443, 0.894564, 0.147014],
        [0.966271, 0.901249, 0.148180],
        [0.964021, 0.907950, 0.149370],
        [0.961681, 0.914672, 0.150520],
        [0.959276, 0.921407, 0.151566],
        [0.956808, 0.928152, 0.152409],
        [0.954287, 0.934908, 0.152921],
        [0.951726, 0.941671, 0.152925],
        [0.949151, 0.948435, 0.152178],
        [0.946602, 0.955190, 0.150328],
        [0.944152, 0.961916, 0.146861],
        [0.941896, 0.968590, 0.140956],
        [0.940015, 0.975158, 0.131326],
    ]
    return ColorMapper.from_palette_array(_data, range=range, **traits)


def viridis(range, **traits):
    """ Generator for the 'viridis' colormap from matplolib """
    _data = [
        [0.267004, 0.004874, 0.329415],
        [0.268510, 0.009605, 0.335427],
        [0.269944, 0.014625, 0.341379],
        [0.271305, 0.019942, 0.347269],
        [0.272594, 0.025563, 0.353093],
        [0.273809, 0.031497, 0.358853],
        [0.274952, 0.037752, 0.364543],
        [0.276022, 0.044167, 0.370164],
        [0.277018, 0.050344, 0.375715],
        [0.277941, 0.056324, 0.381191],
        [0.278791, 0.062145, 0.386592],
        [0.279566, 0.067836, 0.391917],
        [0.280267, 0.073417, 0.397163],
        [0.280894, 0.078907, 0.402329],
        [0.281446, 0.084320, 0.407414],
        [0.281924, 0.089666, 0.412415],
        [0.282327, 0.094955, 0.417331],
        [0.282656, 0.100196, 0.422160],
        [0.282910, 0.105393, 0.426902],
        [0.283091, 0.110553, 0.431554],
        [0.283197, 0.115680, 0.436115],
        [0.283229, 0.120777, 0.440584],
        [0.283187, 0.125848, 0.444960],
        [0.283072, 0.130895, 0.449241],
        [0.282884, 0.135920, 0.453427],
        [0.282623, 0.140926, 0.457517],
        [0.282290, 0.145912, 0.461510],
        [0.281887, 0.150881, 0.465405],
        [0.281412, 0.155834, 0.469201],
        [0.280868, 0.160771, 0.472899],
        [0.280255, 0.165693, 0.476498],
        [0.279574, 0.170599, 0.479997],
        [0.278826, 0.175490, 0.483397],
        [0.278012, 0.180367, 0.486697],
        [0.277134, 0.185228, 0.489898],
        [0.276194, 0.190074, 0.493001],
        [0.275191, 0.194905, 0.496005],
        [0.274128, 0.199721, 0.498911],
        [0.273006, 0.204520, 0.501721],
        [0.271828, 0.209303, 0.504434],
        [0.270595, 0.214069, 0.507052],
        [0.269308, 0.218818, 0.509577],
        [0.267968, 0.223549, 0.512008],
        [0.266580, 0.228262, 0.514349],
        [0.265145, 0.232956, 0.516599],
        [0.263663, 0.237631, 0.518762],
        [0.262138, 0.242286, 0.520837],
        [0.260571, 0.246922, 0.522828],
        [0.258965, 0.251537, 0.524736],
        [0.257322, 0.256130, 0.526563],
        [0.255645, 0.260703, 0.528312],
        [0.253935, 0.265254, 0.529983],
        [0.252194, 0.269783, 0.531579],
        [0.250425, 0.274290, 0.533103],
        [0.248629, 0.278775, 0.534556],
        [0.246811, 0.283237, 0.535941],
        [0.244972, 0.287675, 0.537260],
        [0.243113, 0.292092, 0.538516],
        [0.241237, 0.296485, 0.539709],
        [0.239346, 0.300855, 0.540844],
        [0.237441, 0.305202, 0.541921],
        [0.235526, 0.309527, 0.542944],
        [0.233603, 0.313828, 0.543914],
        [0.231674, 0.318106, 0.544834],
        [0.229739, 0.322361, 0.545706],
        [0.227802, 0.326594, 0.546532],
        [0.225863, 0.330805, 0.547314],
        [0.223925, 0.334994, 0.548053],
        [0.221989, 0.339161, 0.548752],
        [0.220057, 0.343307, 0.549413],
        [0.218130, 0.347432, 0.550038],
        [0.216210, 0.351535, 0.550627],
        [0.214298, 0.355619, 0.551184],
        [0.212395, 0.359683, 0.551710],
        [0.210503, 0.363727, 0.552206],
        [0.208623, 0.367752, 0.552675],
        [0.206756, 0.371758, 0.553117],
        [0.204903, 0.375746, 0.553533],
        [0.203063, 0.379716, 0.553925],
        [0.201239, 0.383670, 0.554294],
        [0.199430, 0.387607, 0.554642],
        [0.197636, 0.391528, 0.554969],
        [0.195860, 0.395433, 0.555276],
        [0.194100, 0.399323, 0.555565],
        [0.192357, 0.403199, 0.555836],
        [0.190631, 0.407061, 0.556089],
        [0.188923, 0.410910, 0.556326],
        [0.187231, 0.414746, 0.556547],
        [0.185556, 0.418570, 0.556753],
        [0.183898, 0.422383, 0.556944],
        [0.182256, 0.426184, 0.557120],
        [0.180629, 0.429975, 0.557282],
        [0.179019, 0.433756, 0.557430],
        [0.177423, 0.437527, 0.557565],
        [0.175841, 0.441290, 0.557685],
        [0.174274, 0.445044, 0.557792],
        [0.172719, 0.448791, 0.557885],
        [0.171176, 0.452530, 0.557965],
        [0.169646, 0.456262, 0.558030],
        [0.168126, 0.459988, 0.558082],
        [0.166617, 0.463708, 0.558119],
        [0.165117, 0.467423, 0.558141],
        [0.163625, 0.471133, 0.558148],
        [0.162142, 0.474838, 0.558140],
        [0.160665, 0.478540, 0.558115],
        [0.159194, 0.482237, 0.558073],
        [0.157729, 0.485932, 0.558013],
        [0.156270, 0.489624, 0.557936],
        [0.154815, 0.493313, 0.557840],
        [0.153364, 0.497000, 0.557724],
        [0.151918, 0.500685, 0.557587],
        [0.150476, 0.504369, 0.557430],
        [0.149039, 0.508051, 0.557250],
        [0.147607, 0.511733, 0.557049],
        [0.146180, 0.515413, 0.556823],
        [0.144759, 0.519093, 0.556572],
        [0.143343, 0.522773, 0.556295],
        [0.141935, 0.526453, 0.555991],
        [0.140536, 0.530132, 0.555659],
        [0.139147, 0.533812, 0.555298],
        [0.137770, 0.537492, 0.554906],
        [0.136408, 0.541173, 0.554483],
        [0.135066, 0.544853, 0.554029],
        [0.133743, 0.548535, 0.553541],
        [0.132444, 0.552216, 0.553018],
        [0.131172, 0.555899, 0.552459],
        [0.129933, 0.559582, 0.551864],
        [0.128729, 0.563265, 0.551229],
        [0.127568, 0.566949, 0.550556],
        [0.126453, 0.570633, 0.549841],
        [0.125394, 0.574318, 0.549086],
        [0.124395, 0.578002, 0.548287],
        [0.123463, 0.581687, 0.547445],
        [0.122606, 0.585371, 0.546557],
        [0.121831, 0.589055, 0.545623],
        [0.121148, 0.592739, 0.544641],
        [0.120565, 0.596422, 0.543611],
        [0.120092, 0.600104, 0.542530],
        [0.119738, 0.603785, 0.541400],
        [0.119512, 0.607464, 0.540218],
        [0.119423, 0.611141, 0.538982],
        [0.119483, 0.614817, 0.537692],
        [0.119699, 0.618490, 0.536347],
        [0.120081, 0.622161, 0.534946],
        [0.120638, 0.625828, 0.533488],
        [0.121380, 0.629492, 0.531973],
        [0.122312, 0.633153, 0.530398],
        [0.123444, 0.636809, 0.528763],
        [0.124780, 0.640461, 0.527068],
        [0.126326, 0.644107, 0.525311],
        [0.128087, 0.647749, 0.523491],
        [0.130067, 0.651384, 0.521608],
        [0.132268, 0.655014, 0.519661],
        [0.134692, 0.658636, 0.517649],
        [0.137339, 0.662252, 0.515571],
        [0.140210, 0.665859, 0.513427],
        [0.143303, 0.669459, 0.511215],
        [0.146616, 0.673050, 0.508936],
        [0.150148, 0.676631, 0.506589],
        [0.153894, 0.680203, 0.504172],
        [0.157851, 0.683765, 0.501686],
        [0.162016, 0.687316, 0.499129],
        [0.166383, 0.690856, 0.496502],
        [0.170948, 0.694384, 0.493803],
        [0.175707, 0.697900, 0.491033],
        [0.180653, 0.701402, 0.488189],
        [0.185783, 0.704891, 0.485273],
        [0.191090, 0.708366, 0.482284],
        [0.196571, 0.711827, 0.479221],
        [0.202219, 0.715272, 0.476084],
        [0.208030, 0.718701, 0.472873],
        [0.214000, 0.722114, 0.469588],
        [0.220124, 0.725509, 0.466226],
        [0.226397, 0.728888, 0.462789],
        [0.232815, 0.732247, 0.459277],
        [0.239374, 0.735588, 0.455688],
        [0.246070, 0.738910, 0.452024],
        [0.252899, 0.742211, 0.448284],
        [0.259857, 0.745492, 0.444467],
        [0.266941, 0.748751, 0.440573],
        [0.274149, 0.751988, 0.436601],
        [0.281477, 0.755203, 0.432552],
        [0.288921, 0.758394, 0.428426],
        [0.296479, 0.761561, 0.424223],
        [0.304148, 0.764704, 0.419943],
        [0.311925, 0.767822, 0.415586],
        [0.319809, 0.770914, 0.411152],
        [0.327796, 0.773980, 0.406640],
        [0.335885, 0.777018, 0.402049],
        [0.344074, 0.780029, 0.397381],
        [0.352360, 0.783011, 0.392636],
        [0.360741, 0.785964, 0.387814],
        [0.369214, 0.788888, 0.382914],
        [0.377779, 0.791781, 0.377939],
        [0.386433, 0.794644, 0.372886],
        [0.395174, 0.797475, 0.367757],
        [0.404001, 0.800275, 0.362552],
        [0.412913, 0.803041, 0.357269],
        [0.421908, 0.805774, 0.351910],
        [0.430983, 0.808473, 0.346476],
        [0.440137, 0.811138, 0.340967],
        [0.449368, 0.813768, 0.335384],
        [0.458674, 0.816363, 0.329727],
        [0.468053, 0.818921, 0.323998],
        [0.477504, 0.821444, 0.318195],
        [0.487026, 0.823929, 0.312321],
        [0.496615, 0.826376, 0.306377],
        [0.506271, 0.828786, 0.300362],
        [0.515992, 0.831158, 0.294279],
        [0.525776, 0.833491, 0.288127],
        [0.535621, 0.835785, 0.281908],
        [0.545524, 0.838039, 0.275626],
        [0.555484, 0.840254, 0.269281],
        [0.565498, 0.842430, 0.262877],
        [0.575563, 0.844566, 0.256415],
        [0.585678, 0.846661, 0.249897],
        [0.595839, 0.848717, 0.243329],
        [0.606045, 0.850733, 0.236712],
        [0.616293, 0.852709, 0.230052],
        [0.626579, 0.854645, 0.223353],
        [0.636902, 0.856542, 0.216620],
        [0.647257, 0.858400, 0.209861],
        [0.657642, 0.860219, 0.203082],
        [0.668054, 0.861999, 0.196293],
        [0.678489, 0.863742, 0.189503],
        [0.688944, 0.865448, 0.182725],
        [0.699415, 0.867117, 0.175971],
        [0.709898, 0.868751, 0.169257],
        [0.720391, 0.870350, 0.162603],
        [0.730889, 0.871916, 0.156029],
        [0.741388, 0.873449, 0.149561],
        [0.751884, 0.874951, 0.143228],
        [0.762373, 0.876424, 0.137064],
        [0.772852, 0.877868, 0.131109],
        [0.783315, 0.879285, 0.125405],
        [0.793760, 0.880678, 0.120005],
        [0.804182, 0.882046, 0.114965],
        [0.814576, 0.883393, 0.110347],
        [0.824940, 0.884720, 0.106217],
        [0.835270, 0.886029, 0.102646],
        [0.845561, 0.887322, 0.099702],
        [0.855810, 0.888601, 0.097452],
        [0.866013, 0.889868, 0.095953],
        [0.876168, 0.891125, 0.095250],
        [0.886271, 0.892374, 0.095374],
        [0.896320, 0.893616, 0.096335],
        [0.906311, 0.894855, 0.098125],
        [0.916242, 0.896091, 0.100717],
        [0.926106, 0.897330, 0.104071],
        [0.935904, 0.898570, 0.108131],
        [0.945636, 0.899815, 0.112838],
        [0.955300, 0.901065, 0.118128],
        [0.964894, 0.902323, 0.123941],
        [0.974417, 0.903590, 0.130215],
        [0.983868, 0.904867, 0.136897],
        [0.993248, 0.906157, 0.143936],
    ]
    return ColorMapper.from_palette_array(_data, range=range, **traits)


# Colorbrewer quantitative colormaps


def accent(range, **traits):
    """ Generator for the 'accent' colormap from ColorBrewer. """
    data = [
        [0.49803921580314636, 0.7882353067398071, 0.49803921580314636],
        [0.7450980544090271, 0.6823529601097107, 0.8313725590705872],
        [0.9921568632125854, 0.7529411911964417, 0.5254902243614197],
        [1.0, 1.0, 0.6000000238418579],
        [0.21960784494876862, 0.42352941632270813, 0.6901960968971252],
        [0.9411764740943909, 0.007843137718737125, 0.49803921580314636],
        [0.7490196228027344, 0.35686275362968445, 0.09019608050584793],
        [0.4000000059604645, 0.4000000059604645, 0.4000000059604645],
    ]
    return DiscreteColorMapper.from_palette_array(data, range=range, **traits)


def Dark2(range, **traits):
    """ Generator for the 'Dark2' colormap from ColorBrewer. """
    data = [
        [0.10588235408067703, 0.6196078658103943, 0.46666666865348816],
        [0.8509804010391235, 0.37254902720451355, 0.007843137718737125],
        [0.4588235318660736, 0.43921568989753723, 0.7019608020782471],
        [0.9058823585510254, 0.16078431904315948, 0.5411764979362488],
        [0.4000000059604645, 0.6509804129600525, 0.11764705926179886],
        [0.9019607901573181, 0.6705882549285889, 0.007843137718737125],
        [0.6509804129600525, 0.4627451002597809, 0.11372549086809158],
        [0.4000000059604645, 0.4000000059604645, 0.4000000059604645],
    ]
    return DiscreteColorMapper.from_palette_array(data, range=range, **traits)


def Paired(range, **traits):
    """ Generator for the 'Paired' colormap from ColorBrewer. """
    data = [
        [0.6509804129600525, 0.8078431487083435, 0.8901960849761963],
        [0.12156862765550613, 0.47058823704719543, 0.7058823704719543],
        [0.6980392336845398, 0.8745098114013672, 0.5411764979362488],
        [0.20000000298023224, 0.6274510025978088, 0.1725490242242813],
        [0.9843137264251709, 0.6039215922355652, 0.6000000238418579],
        [0.8901960849761963, 0.10196078568696976, 0.10980392247438431],
        [0.9921568632125854, 0.7490196228027344, 0.43529412150382996],
        [1.0, 0.49803921580314636, 0.0],
        [0.7921568751335144, 0.6980392336845398, 0.8392156958580017],
        [0.4156862795352936, 0.239215686917305, 0.6039215922355652],
        [1.0, 1.0, 0.6000000238418579],
        [0.6941176652908325, 0.3490196168422699, 0.1568627506494522],
    ]
    return DiscreteColorMapper.from_palette_array(data, range=range, **traits)


def Pastel1(range, **traits):
    """ Generator for the 'Pastel1' colormap from ColorBrewer. """
    data = [
        [0.9843137264251709, 0.7058823704719543, 0.6823529601097107],
        [0.7019608020782471, 0.8039215803146362, 0.8901960849761963],
        [0.800000011920929, 0.9215686321258545, 0.772549033164978],
        [0.8705882430076599, 0.7960784435272217, 0.8941176533699036],
        [0.9960784316062927, 0.8509804010391235, 0.6509804129600525],
        [1.0, 1.0, 0.800000011920929],
        [0.8980392217636108, 0.8470588326454163, 0.7411764860153198],
        [0.9921568632125854, 0.8549019694328308, 0.9254902005195618],
        [0.9490196108818054, 0.9490196108818054, 0.9490196108818054],
    ]
    return DiscreteColorMapper.from_palette_array(data, range=range, **traits)


def Pastel2(range, **traits):
    """Generator for the 'Pastel2' colormap from ColorBrewer.

    Although the ColorBrewer colormaps are defined as discrete colormaps, we
    create continuous colormaps from them by linear interpolation in RGB
    colorspace.
    """

    data = [
        [0.7019608020782471, 0.886274516582489, 0.8039215803146362],
        [0.9921568632125854, 0.8039215803146362, 0.6745098233222961],
        [0.7960784435272217, 0.8352941274642944, 0.9098039269447327],
        [0.95686274766922, 0.7921568751335144, 0.8941176533699036],
        [0.9019607901573181, 0.9607843160629272, 0.7882353067398071],
        [1.0, 0.9490196108818054, 0.6823529601097107],
        [0.9450980424880981, 0.886274516582489, 0.800000011920929],
        [0.800000011920929, 0.800000011920929, 0.800000011920929],
    ]
    return DiscreteColorMapper.from_palette_array(data, range=range, **traits)


def Set1(range, **traits):
    """Generator for the 'Set1' colormap from ColorBrewer."""
    _data = [
        [0.8941176533699036, 0.10196078568696976, 0.10980392247438431],
        [0.21568627655506134, 0.4941176474094391, 0.7215686440467834],
        [0.3019607961177826, 0.686274528503418, 0.29019609093666077],
        [0.5960784554481506, 0.30588236451148987, 0.6392157077789307],
        [1.0, 0.49803921580314636, 0.0],
        [1.0, 1.0, 0.20000000298023224],
        [0.6509804129600525, 0.33725491166114807, 0.1568627506494522],
        [0.9686274528503418, 0.5058823823928833, 0.7490196228027344],
        [0.6000000238418579, 0.6000000238418579, 0.6000000238418579],
    ]
    return DiscreteColorMapper.from_palette_array(_data, range=range, **traits)


def Set2(range, **traits):
    """Generator for the 'Set1' colormap from ColorBrewer.

    Although the ColorBrewer colormaps are defined as discrete colormaps, we
    create continuous colormaps from them by linear interpolation in RGB
    colorspace.
    """
    data = [
        [0.4000000059604645, 0.7607843279838562, 0.6470588445663452],
        [0.9882352948188782, 0.5529412031173706, 0.3843137323856354],
        [0.5529412031173706, 0.6274510025978088, 0.7960784435272217],
        [0.9058823585510254, 0.5411764979362488, 0.7647058963775635],
        [0.6509804129600525, 0.8470588326454163, 0.3294117748737335],
        [1.0, 0.8509804010391235, 0.18431372940540314],
        [0.8980392217636108, 0.7686274647712708, 0.5803921818733215],
        [0.7019608020782471, 0.7019608020782471, 0.7019608020782471],
    ]
    return DiscreteColorMapper.from_palette_array(data, range=range, **traits)


def Set3(range, **traits):
    """Generator for the 'Set1' colormap from ColorBrewer.

    Although the ColorBrewer colormaps are defined as discrete colormaps, we
    create continuous colormaps from them by linear interpolation in RGB
    colorspace.
    """
    data = [
        [0.5529412031173706, 0.8274509906768799, 0.7803921699523926],
        [1.0, 1.0, 0.7019608020782471],
        [0.7450980544090271, 0.729411780834198, 0.8549019694328308],
        [0.9843137264251709, 0.501960813999176, 0.4470588266849518],
        [0.501960813999176, 0.6941176652908325, 0.8274509906768799],
        [0.9921568632125854, 0.7058823704719543, 0.3843137323856354],
        [0.7019608020782471, 0.8705882430076599, 0.4117647111415863],
        [0.9882352948188782, 0.8039215803146362, 0.8980392217636108],
        [0.8509804010391235, 0.8509804010391235, 0.8509804010391235],
        [0.7372549176216125, 0.501960813999176, 0.7411764860153198],
        [0.800000011920929, 0.9215686321258545, 0.772549033164978],
        [1.0, 0.929411768913269, 0.43529412150382996],
    ]
    return DiscreteColorMapper.from_palette_array(data, range=range, **traits)


# Make the convenient list of all the function names as well as a dictionary
# of name->function mappings.  These are useful for UI editors.

color_map_functions = [
    autumn,
    binary,
    bone,
    cool,
    copper,
    flag,
    seismic,
    terrain,
    gray,
    yarg,
    hot,
    hsv,
    jet,
    pink,
    prism,
    spring,
    summer,
    winter,
    cw1_004,
    cw1_005,
    cw1_006,
    cw1_028,
    gmt_drywet,
    Blues,
    BrBG,
    BuGn,
    BuPu,
    GnBu,
    Greens,
    Greys,
    OrRd,
    Oranges,
    PRGn,
    PiYG,
    PuBu,
    PuBuGn,
    PuOr,
    PuRd,
    Purples,
    RdBu,
    RdGy,
    RdPu,
    RdYlBu,
    RdYlGn,
    Reds,
    Spectral,
    YlGn,
    YlGnBu,
    YlOrBr,
    YlOrRd,  
    gist_earth,
    gist_gray,
    gist_heat,
    gist_ncar,
    gist_rainbow,
    gist_stern,
    gist_yarg,
    CubicYF,
    CubicL,
    LinearL,
    LinearLHot,
    CoolWarm,
    CubeHelix,
    wistia,
    magma,
    inferno,
    plasma,
    viridis,
]

color_map_dict = {}
for func in color_map_functions:
    color_map_dict[func] = func.__name__

color_map_name_dict = {}
for func in color_map_functions:
    color_map_name_dict[func.__name__] = func
    __all__.append(func.__name__)

discrete_color_map_functions = [
    accent,
    Dark2,
    Paired,
    Pastel1,
    Pastel2,
    Set1,
    Set2,
    Set3,
]

discrete_color_map_dict = {}
for func in discrete_color_map_functions:
    discrete_color_map_dict[func] = func.__name__

discrete_color_map_name_dict = {}
for func in discrete_color_map_functions:
    discrete_color_map_name_dict[func.__name__] = func
    __all__.append(func.__name__)
