"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DataFormat = exports.SerializationLibrary = exports.OutputFormat = exports.InputFormat = void 0;
/**
 * Absolute class name of the Hadoop `InputFormat` to use when reading table files.
 */
class InputFormat {
    constructor(className) {
        this.className = className;
    }
}
exports.InputFormat = InputFormat;
/**
 * InputFormat for Avro files.
 *
 * @see https://hive.apache.org/javadocs/r1.2.2/api/org/apache/hadoop/hive/ql/io/avro/AvroContainerInputFormat.html
 */
InputFormat.AVRO = new InputFormat('org.apache.hadoop.hive.ql.io.avro.AvroContainerInputFormat');
/**
 * InputFormat for Cloudtrail Logs.
 *
 * @see https://docs.aws.amazon.com/athena/latest/ug/cloudtrail.html
 */
InputFormat.CLOUDTRAIL = new InputFormat('com.amazon.emr.cloudtrail.CloudTrailInputFormat');
/**
 * InputFormat for Orc files.
 *
 * @see https://hive.apache.org/javadocs/r1.2.2/api/org/apache/hadoop/hive/ql/io/orc/OrcInputFormat.html
 */
InputFormat.ORC = new InputFormat('org.apache.hadoop.hive.ql.io.orc.OrcInputFormat');
/**
 * InputFormat for Parquet files.
 *
 * @see https://hive.apache.org/javadocs/r1.2.2/api/org/apache/hadoop/hive/ql/io/parquet/MapredParquetInputFormat.html
 */
InputFormat.PARQUET = new InputFormat('org.apache.hadoop.hive.ql.io.parquet.MapredParquetInputFormat');
/**
 * An InputFormat for plain text files. Files are broken into lines. Either linefeed or
 * carriage-return are used to signal end of line. Keys are the position in the file, and
 * values are the line of text.
 * JSON & CSV files are examples of this InputFormat
 *
 * @see https://hadoop.apache.org/docs/stable/api/org/apache/hadoop/mapred/TextInputFormat.html
 */
InputFormat.TEXT = new InputFormat('org.apache.hadoop.mapred.TextInputFormat');
/**
 * Absolute class name of the Hadoop `OutputFormat` to use when writing table files.
 */
class OutputFormat {
    constructor(className) {
        this.className = className;
    }
}
exports.OutputFormat = OutputFormat;
/**
 * Writes text data with a null key (value only).
 *
 * @see https://hive.apache.org/javadocs/r2.2.0/api/org/apache/hadoop/hive/ql/io/HiveIgnoreKeyTextOutputFormat.html
 */
OutputFormat.HIVE_IGNORE_KEY_TEXT = new OutputFormat('org.apache.hadoop.hive.ql.io.HiveIgnoreKeyTextOutputFormat');
/**
 * OutputFormat for Avro files.
 *
 * @see https://hive.apache.org/javadocs/r1.2.2/api/org/apache/hadoop/hive/ql/io/avro/AvroContainerOutputFormat.html
 */
OutputFormat.AVRO = new InputFormat('org.apache.hadoop.hive.ql.io.avro.AvroContainerOutputFormat');
/**
 * OutputFormat for Orc files.
 *
 * @see https://hive.apache.org/javadocs/r1.2.2/api/org/apache/hadoop/hive/ql/io/orc/OrcOutputFormat.html
 */
OutputFormat.ORC = new InputFormat('org.apache.hadoop.hive.ql.io.orc.OrcOutputFormat');
/**
 * OutputFormat for Parquet files.
 *
 * @see https://hive.apache.org/javadocs/r1.2.2/api/org/apache/hadoop/hive/ql/io/parquet/MapredParquetOutputFormat.html
 */
OutputFormat.PARQUET = new OutputFormat('org.apache.hadoop.hive.ql.io.parquet.MapredParquetOutputFormat');
/**
 * Serialization library to use when serializing/deserializing (SerDe) table records.
 *
 * @see https://cwiki.apache.org/confluence/display/Hive/SerDe
 */
class SerializationLibrary {
    constructor(className) {
        this.className = className;
    }
}
exports.SerializationLibrary = SerializationLibrary;
/**
 * @see https://hive.apache.org/javadocs/r1.2.2/api/org/apache/hadoop/hive/serde2/avro/AvroSerDe.html
 */
SerializationLibrary.AVRO = new SerializationLibrary('org.apache.hadoop.hive.serde2.avro.AvroSerDe');
/**
 * @see https://docs.aws.amazon.com/athena/latest/ug/cloudtrail.html
 */
SerializationLibrary.CLOUDTRAIL = new SerializationLibrary('com.amazon.emr.hive.serde.CloudTrailSerde');
/**
 * @see https://docs.aws.amazon.com/athena/latest/ug/grok.html
 */
SerializationLibrary.GROK = new SerializationLibrary('com.amazonaws.glue.serde.GrokSerDe');
/**
 * @see https://hive.apache.org/javadocs/r1.2.2/api/org/apache/hive/hcatalog/data/JsonSerDe.html
 */
SerializationLibrary.HIVE_JSON = new SerializationLibrary('org.apache.hive.hcatalog.data.JsonSerDe');
/**
 * @see https://hive.apache.org/javadocs/r1.2.2/api/org/apache/hadoop/hive/serde2/lazy/LazySimpleSerDe.html
 */
SerializationLibrary.LAZY_SIMPLE = new SerializationLibrary('org.apache.hadoop.hive.serde2.lazy.LazySimpleSerDe');
/**
 * @see https://hive.apache.org/javadocs/r1.2.2/api/org/apache/hadoop/hive/serde2/OpenCSVSerde.html
 */
SerializationLibrary.OPEN_CSV = new SerializationLibrary('org.apache.hadoop.hive.serde2.OpenCSVSerde');
/**
 * @see https://github.com/rcongiu/Hive-JSON-Serde
 */
SerializationLibrary.OPENX_JSON = new SerializationLibrary('org.openx.data.jsonserde.JsonSerDe');
/**
 * @see https://hive.apache.org/javadocs/r1.2.2/api/org/apache/hadoop/hive/ql/io/orc/OrcSerde.html
 */
SerializationLibrary.ORC = new SerializationLibrary('org.apache.hadoop.hive.ql.io.orc.OrcSerde');
/**
 * @see https://hive.apache.org/javadocs/r1.2.2/api/org/apache/hadoop/hive/ql/io/parquet/serde/ParquetHiveSerDe.html
 */
SerializationLibrary.PARQUET = new SerializationLibrary('org.apache.hadoop.hive.ql.io.parquet.serde.ParquetHiveSerDe');
/**
 * @see https://hive.apache.org/javadocs/r1.2.2/api/org/apache/hadoop/hive/serde2/RegexSerDe.html
 */
SerializationLibrary.REGEXP = new SerializationLibrary('org.apache.hadoop.hive.serde2.RegexSerDe');
/**
 * Defines the input/output formats and ser/de for a single DataFormat.
 */
class DataFormat {
    constructor(props) {
        this.inputFormat = props.inputFormat;
        this.outputFormat = props.outputFormat;
        this.serializationLibrary = props.serializationLibrary;
    }
}
exports.DataFormat = DataFormat;
/**
 * DataFormat for Apache Web Server Logs. Also works for CloudFront logs
 *
 * @see https://docs.aws.amazon.com/athena/latest/ug/apache.html
 */
DataFormat.APACHE_LOGS = new DataFormat({
    inputFormat: InputFormat.TEXT,
    outputFormat: OutputFormat.HIVE_IGNORE_KEY_TEXT,
    serializationLibrary: SerializationLibrary.REGEXP,
});
/**
 * DataFormat for Apache Avro
 *
 * @see https://docs.aws.amazon.com/athena/latest/ug/avro.html
 */
DataFormat.AVRO = new DataFormat({
    inputFormat: InputFormat.AVRO,
    outputFormat: OutputFormat.AVRO,
    serializationLibrary: SerializationLibrary.AVRO,
});
/**
 * DataFormat for CloudTrail logs stored on S3
 *
 * @see https://docs.aws.amazon.com/athena/latest/ug/cloudtrail.html
 */
DataFormat.CLOUDTRAIL_LOGS = new DataFormat({
    inputFormat: InputFormat.CLOUDTRAIL,
    outputFormat: OutputFormat.HIVE_IGNORE_KEY_TEXT,
    serializationLibrary: SerializationLibrary.CLOUDTRAIL,
});
/**
 * DataFormat for CSV Files
 *
 * @see https://docs.aws.amazon.com/athena/latest/ug/csv.html
 */
DataFormat.CSV = new DataFormat({
    inputFormat: InputFormat.TEXT,
    outputFormat: OutputFormat.HIVE_IGNORE_KEY_TEXT,
    serializationLibrary: SerializationLibrary.OPEN_CSV,
});
/**
 * Stored as plain text files in JSON format.
 * Uses OpenX Json SerDe for serialization and deseralization.
 *
 * @see https://docs.aws.amazon.com/athena/latest/ug/json.html
 */
DataFormat.JSON = new DataFormat({
    inputFormat: InputFormat.TEXT,
    outputFormat: OutputFormat.HIVE_IGNORE_KEY_TEXT,
    serializationLibrary: SerializationLibrary.OPENX_JSON,
});
/**
 * DataFormat for Logstash Logs, using the GROK SerDe
 *
 * @see https://docs.aws.amazon.com/athena/latest/ug/grok.html
 */
DataFormat.LOGSTASH = new DataFormat({
    inputFormat: InputFormat.TEXT,
    outputFormat: OutputFormat.HIVE_IGNORE_KEY_TEXT,
    serializationLibrary: SerializationLibrary.GROK,
});
/**
 * DataFormat for Apache ORC (Optimized Row Columnar)
 *
 * @see https://docs.aws.amazon.com/athena/latest/ug/orc.html
 */
DataFormat.ORC = new DataFormat({
    inputFormat: InputFormat.ORC,
    outputFormat: OutputFormat.ORC,
    serializationLibrary: SerializationLibrary.ORC,
});
/**
 * DataFormat for Apache Parquet
 *
 * @see https://docs.aws.amazon.com/athena/latest/ug/parquet.html
 */
DataFormat.PARQUET = new DataFormat({
    inputFormat: InputFormat.PARQUET,
    outputFormat: OutputFormat.PARQUET,
    serializationLibrary: SerializationLibrary.PARQUET,
});
/**
 * DataFormat for TSV (Tab-Separated Values)
 *
 * @see https://docs.aws.amazon.com/athena/latest/ug/lazy-simple-serde.html
 */
DataFormat.TSV = new DataFormat({
    inputFormat: InputFormat.TEXT,
    outputFormat: OutputFormat.HIVE_IGNORE_KEY_TEXT,
    serializationLibrary: SerializationLibrary.LAZY_SIMPLE,
});
//# sourceMappingURL=data:application/json;base64,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