# Copyright 2021 The FastEstimator Authors. All Rights Reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
# ==============================================================================
from typing import Iterable, Tuple, Union

from albumentations.imgaug.transforms import IAAAdditiveGaussianNoise as IAAAdditiveGaussianNoiseAlb

from fastestimator.op.numpyop.univariate.univariate import ImageOnlyAlbumentation
from fastestimator.util.traceability_util import traceable


@traceable()
class IAAAdditiveGaussianNoise(ImageOnlyAlbumentation):
    """Add gaussian noise to the input image.

    Args:
        inputs: Key(s) of images to be modified.
        outputs: Key(s) into which to write the modified images.
        mode: What mode(s) to execute this Op in. For example, "train", "eval", "test", or "infer". To execute
            regardless of mode, pass None. To execute in all modes except for a particular one, you can pass an argument
            like "!infer" or "!train".
        ds_id: What dataset id(s) to execute this Op in. To execute regardless of ds_id, pass None. To execute in all
            ds_ids except for a particular one, you can pass an argument like "!ds1".
        loc: mean of the normal distribution that generates the noise. Default: 0.
        scale: standard deviation of the normal distribution that generates the noise. Default: (0.01 * 255, 0.05 * 255)

    Image types:
        uint8, float32
    """
    def __init__(self,
                 inputs: Union[str, Iterable[str]],
                 outputs: Union[str, Iterable[str]],
                 mode: Union[None, str, Iterable[str]] = None,
                 ds_id: Union[None, str, Iterable[str]] = None,
                 loc: int = 0,
                 scale:Tuple[float, float] =(2.5500000000000003, 12.75)):
        super().__init__(
            IAAAdditiveGaussianNoiseAlb(loc=loc, scale=scale, always_apply=True),
            inputs=inputs,
            outputs=outputs,
            mode=mode,
            ds_id=ds_id)
