from basinmaker.func.qgis import *
from basinmaker.func.pdtable import *
from basinmaker.func.rarray import *
from basinmaker.utilities.utilities import *
import pandas as pd
import numpy as np
import tempfile
import os
import importlib 
import sqlite3

def GenerateHRUS_qgis(
    Path_Subbasin_Ply,
    Landuse_info,
    Soil_info,
    Veg_info,
    Sub_Lake_ID="HyLakeId",
    Sub_ID="SubId",
    Path_Connect_Lake_ply="#",
    Path_Non_Connect_Lake_ply="#",
    Lake_Id="Hylak_id",
    Path_Landuse_Ply="#",
    Landuse_ID="Landuse_ID",
    Path_Soil_Ply="#",
    Soil_ID="Soil_ID",
    Path_Veg_Ply="#",
    Veg_ID="Veg_ID",
    Path_Other_Ply_1="#",
    Other_Ply_ID_1="O_ID_1",
    Path_Other_Ply_2="#",
    Other_Ply_ID_2="O_ID_2",
    DEM="#",
    Project_crs="EPSG:3573",
    OutputFolder="#",
    qgis_prefix_path="#",
    min_hru_pct_sub_area = 0,
    importance_order = ['Landuse_ID'],        
):

    """Generate HRU polygons and their attributes needed by hydrological model

    Function that be used to overlay: subbasin polygon, lake polygon (optional)
    , Land use polygon (optional), soil type polygon(optional),
    vegetation polygon (optional), and two other user defined polygons
    (optional).
    Non-lake HRU polygons in a subbasin is defined by an unique
    combination of all user provided datasets.
    A lake HRU polygon is defined the same as the provided lake polygon.
    All value of landuse and Veg polygon covered by lake will
    be changed to 1, indicating it is a covered by lake.
    All value of the soil polygon covered by the lake will be change to
    the soil id of the polygon covered by the lake with largest area.

    Parameters
    ----------
    Path_Subbasin_Ply                 : string
        It is the path of the subbasin polygon, which is generated by
        toolbox. if not generated by toolbox, the attribute table should
        including following attribute.
        ##############Subbasin related attributes###########################
        SubID           - integer, The subbasin Id
        DowSubId        - integer, The downstream subbasin ID of this
                                   subbasin
        IsLake          - integer, If the subbasin is a lake / reservior
                                   subbasin. 1 yes, <0, no
        IsObs           - integer, If the subbasin contains a observation
                                   gauge. 1 yes, < 0 no.
        RivLength       - float,   The length of the river in current
                                   subbasin in m
        RivSlope        - float,   The slope of the river path in
                                   current subbasin, in m/m
        FloodP_n        - float,   Flood plain manning's coefficient, in -
        Ch_n            - float,   main channel manning's coefficient, in -
        BkfWidth        - float,   the bankfull width of the main channel
                                   in m
        BkfDepth        - float,   the bankfull depth of the main channel
                                   in m
        HyLakeId        - integer, the lake id
        LakeVol         - float,   the Volume of the lake in km3
        LakeDepth       - float,   the average depth of the lake m
        LakeArea        - float,   the area of the lake in m2
    Landuse_info                      : string
        Path to a csv file that contains landuse information, including
        following attributes:
        Landuse_ID (can be any string)  - integer, the landuse ID in the
                                                   landuse polygon
        LAND_USE_C                      - string,  the landuse class name
                                                   for each landuse Type
    Soil_info                        : string
        Path to a csv file that contains soil information, including
        following attributes:
        Soil_ID (can be any string)     - integer, the Soil ID in the
                                                   soil polygon
        SOIL_PROF                       - string,  the Soil profile name
                                                   for each soil type
    Veg_info                         : string
        Path to a csv file that contains vegetation information, including
        following attributes:
        Veg_ID (can be any string)      - integer, the vegetation ID in the
                                                   vegetation polygon
        VEG_C                           - string,  the vegetation class name
                                                   for each vegetation Type
    Sub_Lake_ID                      : string (optional)
        The column name of the lake id in the subbasin polygon
    Sub_ID                           : string (optional)
        The column name of the subbasin id in the subbasin polygon
    Path_Connect_Lake_ply            : string (Optional)
        Path to the connected lake's polygon
    Path_Non_Connect_Lake_ply        : string (Optional)
        Path to the non connected lake's polygon
    Lake_Id                          : string (Optional)
        The the column name in lake polygon indicate the lake ID.
    Path_Landuse_Ply                 : string (Optional)
        Path to the landuse polygon. when Path_Landuse_Ply is not
        provided. The Landuse ID in Landuse_info should be
        1: land, -1: lake
    Landuse_ID                       : string (Optional)
        the the column name in landuse polygon and Landuse_info csv
        indicate the landuse ID. when Path_Landuse_Ply is not
        provided. The Landuse ID should be
        1: land, -1: lake.
    Path_Soil_Ply                    : string (Optional)
        Path to the soil polygon. when soil polygon is not
        provided. The Soil ID in Soil_info should be the same
        as Landuse ID.
    Soil_ID                          : string (Optional)
        the the column name in soil polygon and soil_info csv
        indicate the soil ID. when soil polygon is not
        provided. The Soil ID in Soil_info should be the same
        as Landuse ID.
    Path_Veg_Ply                     : string (Optional)
        Path to the vegetation polygon. when Veg polygon is not
        provided. The Veg ID in Veg_info should be the same
        as Landuse ID.
    Veg_ID                           : string (Optional)
        the the column name in vegetation polygon and veg_info csv
        indicate the vegetation ID. when Veg polygon is not
        provided. The Veg ID in Veg_info should be the same
        as Landuse ID.
    Path_Other_Ply_1                 : string (Optional)
        Path to the other polygon that will be used to define HRU,
        such as elevation band, or aspect.
    Other_Ply_ID_1                   : string (Optional)
        the the column name in Other_Ply_1 polygon
        indicate the landuse ID.
    Path_Other_Ply_2                 : string (Optional)
        Path to the other polygon that will be used to define HRU,
        such as elevation band, or aspect.
    Other_Ply_ID_2                   : string (Optional)
        the the column name in Other_Ply_2 polygon
        indicate the landuse ID.
    DEM                              : string (optional)
        the path to a raster elevation dataset, that will be used to
        calcuate average apspect, elevation and slope within each HRU.
        if no data is provided, basin average value will be used for
        each HRU.
    Project_crs                      : string
        the EPSG code of a projected coodinate system that will be used to
        calcuate HRU area and slope.
    OutputFolder                     : string
        The path to the folder that will save output HRU polygon.

    Notes
    -------
    Following ouput files will be generated in "<OutputFolder>/"
    'finalcat_hru_info.shp'              - HRU polygon and it's attributes


    Returns:
    -------
       None

    Examples
    -------
    >>> from ToolboxClass import LRRT
    >>> import pandas as pd
    >>> DataFolder = "C:/Path_to_foldr_of_example_dataset_provided_in_Github_wiki/"
    >>> RTtool=LRRT()
    >>> RTtool.GenerateHRUS(OutputFolder = DataFolder,
                           Path_Subbasin_Ply = os.path.join(DataFolder,"finalcat_info.shp"),
                           Path_Connect_Lake_ply = os.path.join(DataFolder,'Con_Lake_Ply.shp'),
                           Path_Non_Connect_Lake_ply = os.path.join(DataFolder,'Non_Con_Lake_Ply.shp'),
                           Path_Landuse_Ply = os.path.join(DataFolder,'modislanduse_exp_lg_pre.shp'),
                           Landuse_ID = 'gridcode',
                           Path_Soil_Ply = os.path.join(DataFolder,'ca_all_slc_v3r2_exp_lg.shp'),
                           Soil_ID = 'POLY_ID',
                           Landuse_info=os.path.join(DataFolder,'landuse_info.csv'),
                           Soil_info=os.path.join(DataFolder,'soil_info.csv'),
                           Veg_info=os.path.join(DataFolder,'veg_info.csv'),
                           DEM = os.path.join(DataFolder,'na_dem_15s_1.tif')
                           )

    """
    QgsApplication.setPrefixPath(qgis_prefix_path, True)
    Qgs = QgsApplication([], False)
    Qgs.initQgis()
    from processing.core.Processing import Processing
    from processing.tools import dataobjects
    from qgis import processing

    feedback = QgsProcessingFeedback()
    Processing.initialize()
    QgsApplication.processingRegistry().addProvider(QgsNativeAlgorithms())
    context = dataobjects.createContext()
    context.setInvalidGeometryCheck(QgsFeatureRequest.GeometryNoCheck)

    if not os.path.exists(OutputFolder):
        os.makedirs(OutputFolder)

    tempfolder = os.path.join(
        tempfile.gettempdir(), "basinmaker_hru" + str(np.random.randint(1, 10000 + 1))
    )

    if not os.path.exists(tempfolder):
        os.makedirs(tempfolder)
        
    Merge_layer_list = []
    output_hru_shp = os.path.join(OutputFolder, "finalcat_hru_info.shp")
    ### First overlay the subbasin layer with lake polygon, the new unique
    # id will be 'HRULake_ID'

    Sub_Lake_HRU_Layer, trg_crs, fieldnames_list = GeneratelandandlakeHRUS(
        processing,
        context,
        OutputFolder,
        Path_Subbasin_ply=Path_Subbasin_Ply,
        Path_Connect_Lake_ply=Path_Connect_Lake_ply,
        Path_Non_Connect_Lake_ply=Path_Non_Connect_Lake_ply,
        Sub_ID=Sub_ID,
        Sub_Lake_ID=Sub_Lake_ID,
        Lake_Id=Lake_Id,
    )
        
    crs_info = Sub_Lake_HRU_Layer.crs()
    if crs_info.isGeographic():
        union_snap_distance = 0.000133333 # 90 m 
    else:
        union_snap_distance = 20
    
    
    All_HRUS = processing.run("native:extractbyexpression", {
                     'INPUT':Sub_Lake_HRU_Layer,
                     'EXPRESSION':'\"HRU_IsLake\"  =  1',
                     'OUTPUT':"memory:",
                     'FAIL_OUTPUT':"memory:"})
    Lake_HRUs = All_HRUS['OUTPUT']
    Land_HRUs = All_HRUS['FAIL_OUTPUT']
    
    fieldnames_list.extend(
        [
            Landuse_ID,
            Soil_ID,
            Veg_ID,
            "LAND_USE_C",
            "VEG_C",
            "SOIL_PROF",
            "HRU_Slope",
            "HRU_Area",
            "HRU_Aspect",
        ]
    )
    dissolve_filedname_list = ["HRULake_ID"]
    Merge_layer_list.append(Land_HRUs)


    if Path_Soil_Ply != "#":
        layer_soil_dis = Reproj_Clip_Dissolve_Simplify_Polygon(
            processing,
            context,
            Path_Soil_Ply,
            Project_crs,
            trg_crs,
            Soil_ID,
            Land_HRUs,
        )
        Merge_layer_list.append(layer_soil_dis)
        dissolve_filedname_list.append(Soil_ID)
    
    print(" prepare soil polygon done")  
    
    
    
    #### check which data will be inlucded to determine HRU
    if Path_Landuse_Ply != "#":
        layer_landuse_dis = Reproj_Clip_Dissolve_Simplify_Polygon(
            processing,
            context,
            Path_Landuse_Ply,
            Project_crs,
            trg_crs,
            Landuse_ID,
            Land_HRUs,
        )
        Merge_layer_list.append(layer_landuse_dis)
        dissolve_filedname_list.append(Landuse_ID)
    
    print(" prepare landuse polygon done")    
    # qgis_vector_field_calculator(
    #     processing=processing,
    #     context=context,
    #     INPUT=layer_landuse_dis,
    #     FIELD_NAME='asdfd',
    #     FORMULA='1',
    #     FIELD_PRECISION=3,
    #     OUTPUT=os.path.join(OutputFolder,'landuse_proj.shp'),
    # )
    
    if Path_Veg_Ply != "#":
        layer_veg_dis = Reproj_Clip_Dissolve_Simplify_Polygon(
            processing,
            context,
            Path_Veg_Ply,
            Project_crs,
            trg_crs,
            Veg_ID,
            Land_HRUs,
        )
        Merge_layer_list.append(layer_veg_dis)
        dissolve_filedname_list.append(Veg_ID)
    
    print(" prepare Veg polygon done") 
    if Path_Other_Ply_1 != "#":
        layer_other_1_dis = Reproj_Clip_Dissolve_Simplify_Polygon(
            processing,
            context,
            Path_Other_Ply_1,
            Project_crs,
            trg_crs,
            Other_Ply_ID_1,
            Land_HRUs,
        )
        Merge_layer_list.append(layer_other_1_dis)
        fieldnames_list.append(Other_Ply_ID_1)
        dissolve_filedname_list.append(Other_Ply_ID_1)
     
    print(" prepare Other polygon 1 done") 
    if Path_Other_Ply_2 != "#":
        layer_other_2_dis = Reproj_Clip_Dissolve_Simplify_Polygon(
            processing,
            context,
            Path_Other_Ply_2,
            Project_crs,
            trg_crs,
            Other_Ply_ID_2,
            Land_HRUs,
        )
        Merge_layer_list.append(layer_other_2_dis)
        fieldnames_list.append(Other_Ply_ID_2)
        dissolve_filedname_list.append(Other_Ply_ID_2)

    fieldnames = set(fieldnames_list)
    print(" prepare Other polygon 2 done") 
    #### uniion polygons in the Merge_layer_list
    mem_union = Union_Ply_Layers_And_Simplify(
        processing,
        context,
        Merge_layer_list,
        dissolve_filedname_list,
        fieldnames,
        OutputFolder,
        union_snap_distance,
        tempfolder,
        trg_crs,
    )
    
    #####
    print(" End union") 
    Landuse_info_data = pd.read_csv(Landuse_info)
    Soil_info_data = pd.read_csv(Soil_info)
    Veg_info_data = pd.read_csv(Veg_info)

    # qgis_vector_field_calculator(
    #     processing=processing,
    #     context=context,
    #     INPUT=mem_union,
    #     FIELD_NAME='asdfd',
    #     FORMULA='1',
    #     FIELD_PRECISION=3,
    #     OUTPUT=os.path.join(OutputFolder,'test3.shp'),
    # )
    
    # landuse polygon is not provided,landused id the same as IS_lake
    if Path_Landuse_Ply == "#":
        formula = "1"
        mem_union_landuse = qgis_vector_field_calculator(
            processing=processing,
            context=context,
            INPUT=mem_union,
            FIELD_NAME=Landuse_ID,
            FORMULA=formula,
            FIELD_PRECISION=3,
            OUTPUT="memory:",
        )["OUTPUT"]
    else:
        mem_union_landuse = mem_union

    Lake_HRUs = qgis_vector_field_calculator(
        processing=processing,
        context=context,
        INPUT=Lake_HRUs,
        FIELD_NAME=Landuse_ID,
        FORMULA="-1",
        FIELD_PRECISION=3,
        OUTPUT="memory:",
    )["OUTPUT"]
        
    # if soil is not provied, it the value,will be the same as land use
    if Path_Soil_Ply == "#":
        formula = "1"
        mem_union_soil = qgis_vector_field_calculator(
            processing=processing,
            context=context,
            INPUT=mem_union_landuse,
            FIELD_NAME=Soil_ID,
            FORMULA=formula,
            FIELD_PRECISION=3,
            OUTPUT="memory:",
        )["OUTPUT"]
    else:
        mem_union_soil = mem_union_landuse
        
    Lake_HRUs = qgis_vector_field_calculator(
        processing=processing,
        context=context,
        INPUT=Lake_HRUs,
        FIELD_NAME=Soil_ID,
        FORMULA="-1",
        FIELD_PRECISION=3,
        OUTPUT="memory:",
    )["OUTPUT"]
    
    # if no vegetation polygon is provide vegetation, will be the same as landuse
    if Path_Veg_Ply == "#":

        mem_union_veg = qgis_vector_field_calculator(
            processing=processing,
            context=context,
            INPUT=mem_union_soil,
            FIELD_NAME=Veg_ID,
            FORMULA=' \"Landuse_ID\" ',
            FIELD_PRECISION=3,
            OUTPUT="memory:",
        )["OUTPUT"]
    else:
        mem_union_veg = mem_union_soil
        
    Lake_HRUs = qgis_vector_field_calculator(
        processing=processing,
        context=context,
        INPUT=Lake_HRUs,
        FIELD_NAME=Veg_ID,
        FORMULA="-1",
        FIELD_PRECISION=3,
        OUTPUT="memory:",
    )["OUTPUT"]
    
            

    if Path_Other_Ply_1 == "#":
        formula = "1"
        mem_union_o1 = qgis_vector_field_calculator(
            processing=processing,
            context=context,
            INPUT=mem_union_veg,
            FIELD_NAME=Other_Ply_ID_1,
            FORMULA=formula,
            FIELD_PRECISION=3,
            OUTPUT="memory:",
        )["OUTPUT"]
    else:
        mem_union_o1 = mem_union_veg
        
    Lake_HRUs = qgis_vector_field_calculator(
        processing=processing,
        context=context,
        INPUT=Lake_HRUs,
        FIELD_NAME=Other_Ply_ID_1,
        FORMULA="-1",
        FIELD_PRECISION=3,
        OUTPUT="memory:",
    )["OUTPUT"]
            
        
    if Path_Other_Ply_2 == "#":
        formula = "1"
        mem_union_o2 = qgis_vector_field_calculator(
            processing=processing,
            context=context,
            INPUT=mem_union_o1,
            FIELD_NAME=Other_Ply_ID_2,
            FORMULA=formula,
            FIELD_PRECISION=3,
            OUTPUT="memory:",
        )["OUTPUT"]
    else:
        mem_union_o2 = mem_union_o1

    Lake_HRUs = qgis_vector_field_calculator(
        processing=processing,
        context=context,
        INPUT=Lake_HRUs,
        FIELD_NAME=Other_Ply_ID_2,
        FORMULA="-1",
        FIELD_PRECISION=3,
        OUTPUT="memory:",
    )["OUTPUT"]
    
    print("merge land and lake HRUs")
    
    hru_layer_draft = processing.run("native:mergevectorlayers", {
                   'LAYERS':[Lake_HRUs,mem_union_o2],
                   'CRS':None,'OUTPUT':"memory:"})["OUTPUT"]
                   
    print("Update HRU attributes ")
    hru_layer_draft = processing.run("native:refactorfields", {'INPUT':hru_layer_draft,'FIELDS_MAPPING':[
    {'expression': '\"SubId\"','length': 12,'name': 'SubId','precision': 0,'type': 4},
    {'expression': '\"HyLakeId\"','length': 12,'name': 'HyLakeId','precision': 0,'type': 4},
    {'expression': '\"Hylak_id\"','length': 12,'name': 'Hylak_id','precision': 0,'type': 4},
    {'expression': '\"HRULake_ID\"','length': 12,'name': 'HRULake_ID','precision': 0,'type': 4},
    {'expression': '\"HRU_IsLake\"','length': 12,'name': 'HRU_IsLake','precision': 0,'type': 4},
    {'expression': '\"Soil_ID\"','length': 12,'name': 'Soil_ID','precision': 0,'type': 4},
    {'expression': '\"Veg_ID\"','length': 12,'name': 'Veg_ID','precision': 0,'type': 4},
    {'expression': '\"O_ID_1\"','length': 12,'name': 'O_ID_1','precision': 0,'type': 4},
    {'expression': '\"O_ID_2\"','length': 12,'name': 'O_ID_2','precision': 0,'type': 4},
    {'expression': '\"Landuse_ID\"','length': 12,'name': 'Landuse_ID','precision': 0,'type': 4},
    ],
    'OUTPUT':"memory:"})["OUTPUT"]



    # qgis_vector_field_calculator(
    #     processing=processing,
    #     context=context,
    #     INPUT=hru_layer_draft,
    #     FIELD_NAME='asdfd23',
    #     FORMULA='1',
    #     FIELD_PRECISION=3,
    #     OUTPUT=os.path.join(OutputFolder,'test4.shp'),
    # )
    
 
#    hru_layer_draft = mem_union_o2

    HRU_draf_final = Define_HRU_Attributes(
        processing,
        context,
        Project_crs,
        trg_crs,
        hru_layer_draft,
        dissolve_filedname_list,
        Sub_ID,
        Landuse_ID,
        Soil_ID,
        Veg_ID,
        Other_Ply_ID_1,
        Other_Ply_ID_2,
        Landuse_info_data,
        Soil_info_data,
        Veg_info_data,
        DEM,
        Path_Subbasin_Ply,
        OutputFolder,
        min_hru_pct_sub_area,
        importance_order,        
    )

    qgis_vector_field_calculator(
        processing=processing,
        context=context,
        INPUT=HRU_draf_final,
        FIELD_NAME="HRU_ID",
        FORMULA="HRU_ID_New",
        FIELD_PRECISION=0,
        NEW_FIELD=False,
        OUTPUT=output_hru_shp,
    )

    Clean_Attribute_Name(output_hru_shp, COLUMN_NAMES_CONSTANT_HRU)
    del Land_HRUs, mem_union
    Qgs.exit()


def GeneratelandandlakeHRUS(
    processing,
    context,
    OutputFolder,
    Path_Subbasin_ply,
    Path_Connect_Lake_ply="#",
    Path_Non_Connect_Lake_ply="#",
    Sub_ID="SubId",
    Sub_Lake_ID="HyLakeId",
    Lake_Id="Hylak_id",
):

    """Overlay subbasin polygon and lake polygons

    Function that will overlay subbasin polygon and lake polygon

    Parameters
    ----------
    processing                        : qgis object
    context                           : qgis object
    OutputFolder                     : string
        The path to the folder that will save output HRU polygon.

    Path_Subbasin_Ply                 : string
        It is the path of the subbasin polygon, which is generated by
        toolbox. if not generated by toolbox, the attribute table should
        including following attribute.
        ##############Subbasin related attributes###########################
        SubID           - integer, The subbasin Id
        DowSubId        - integer, The downstream subbasin ID of this
                                   subbasin
        IsLake          - integer, If the subbasin is a lake / reservior
                                   subbasin. 1 yes, <0, no
        IsObs           - integer, If the subbasin contains a observation
                                   gauge. 1 yes, < 0 no.
        RivLength       - float,   The length of the river in current
                                   subbasin in m
        RivSlope        - float,   The slope of the river path in
                                   current subbasin, in m/m
        FloodP_n        - float,   Flood plain manning's coefficient, in -
        Ch_n            - float,   main channel manning's coefficient, in -
        BkfWidth        - float,   the bankfull width of the main channel
                                   in m
        BkfDepth        - float,   the bankfull depth of the main channel
                                   in m
        HyLakeId        - integer, the lake id
        LakeVol         - float,   the Volume of the lake in km3
        LakeDepth       - float,   the average depth of the lake m
        LakeArea        - float,   the area of the lake in m2
    Path_Connect_Lake_ply            : string (Optional)
        Path to the connected lake's polygon
    Path_Non_Connect_Lake_ply        : string (Optional)
        Path to the non connected lake's polygon
    Sub_ID                           : string (optional)
        The column name of the subbasin id in the subbasin polygon
    Sub_Lake_ID                      : string (optional)
        The column name of the lake id in the subbasin polygon
    Lake_Id                          : string (Optional)
        The the column name in lake polygon indicate the lake ID.


    Notes
    -------
        None

    Returns:
    -------
        Sub_Lake_HRU['OUTPUT']                   : qgis object
            it is a polygon after overlay between subbasin polygon and
            lake polygon
        Sub_Lake_HRU['OUTPUT'].crs().authid()    : string
            it is a string indicate the geospatial reference used by SubBasin
            polygon
        ['HRULake_ID','HRU_IsLake',Sub_ID]       : list
            it is a string list
    """

    # Define output path
    Path_finalcat_hru_out = os.path.join(OutputFolder, "finalcat_hru_lake_info.shp")

    # Fix geometry errors in subbasin polygon
    Subfixgeo = qgis_vector_fix_geometries(
        processing, context, INPUT=Path_Subbasin_ply, OUTPUT="memory:"
    )

    # Create a file name list that will be strored in output attribute table
    fieldnames_list = [
        "HRULake_ID",
        "HRU_IsLake",
        Lake_Id,
        Sub_ID,
        Sub_Lake_ID,
    ]  ### attribubte name in the need to be saved
    fieldnames = set(fieldnames_list)

    # if no lake polygon is provided, use subId as HRULake_ID.
    if Path_Connect_Lake_ply == "#" and Path_Non_Connect_Lake_ply == "#":
        memresult_addlakeid = qgis_vector_field_calculator(
            processing=processing,
            context=context,
            FORMULA="-1",
            FIELD_NAME="Hylak_id",
            INPUT=Subfixgeo["OUTPUT"],
            OUTPUT="memory:",
        )
        memresult_addhruid = qgis_vector_field_calculator(
            processing=processing,
            context=context,
            FORMULA=' "SubId" ',
            FIELD_NAME="HRULake_ID",
            INPUT=memresult_addlakeid["OUTPUT"],
            OUTPUT="memory:",
        )
        Sub_Lake_HRU = qgis_vector_field_calculator(
            processing=processing,
            context=context,
            FORMULA="-1",
            FIELD_NAME="HRU_IsLake",
            INPUT=memresult_addhruid["OUTPUT"],
            OUTPUT="memory:",
        )
        # remove column not in fieldnames
        Sub_Lake_HRU, temp_out = Clean_Attribute_Name(
            Sub_Lake_HRU["OUTPUT"], fieldnames, Input_Is_Feature_In_Mem=True
        )
        crs_id = qgis_vector_return_crs_id(
            processing, context, Sub_Lake_HRU, Input_Is_Feature_In_Mem=True
        )
        return Sub_Lake_HRU, crs_id, ["HRULake_ID", "HRU_IsLake", Sub_ID]

    # fix lake polygon  geometry
    if Path_Connect_Lake_ply != "#":
        ConLakefixgeo = qgis_vector_fix_geometries(
            processing, context, INPUT=Path_Connect_Lake_ply, OUTPUT="memory:"
        )
    # fix lake polygon geometry
    if Path_Non_Connect_Lake_ply != "#":
        NonConLakefixgeo = qgis_vector_fix_geometries(
            processing, context, INPUT=Path_Non_Connect_Lake_ply, OUTPUT="memory:"
        )

    # Merge connected and non connected lake polygons first
    if Path_Connect_Lake_ply != "#" and Path_Non_Connect_Lake_ply != "#":
        meme_Alllakeply = qgis_vector_merge_vector_layers(
            processing,
            context,
            INPUT_Layer_List=[ConLakefixgeo["OUTPUT"], NonConLakefixgeo["OUTPUT"]],
            OUTPUT="memory:",
        )
    elif Path_Connect_Lake_ply != "#" and Path_Non_Connect_Lake_ply == "#":
        meme_Alllakeply = ConLakefixgeo
    elif Path_Connect_Lake_ply == "#" and Path_Non_Connect_Lake_ply != "#":
        meme_Alllakeply = NonConLakefixgeo
    else:
        print("should never happened......")

    # union merged polygon and subbasin polygon
    mem_sub_lake_union_temp = qgis_vector_union_two_layers(
        processing=processing,
        context=context,
        INPUT=Subfixgeo["OUTPUT"],
        OVERLAY=meme_Alllakeply["OUTPUT"],
        OUTPUT="memory:",
    )["OUTPUT"]

    # fix union geometry
    mem_sub_lake_union = qgis_vector_fix_geometries(
        processing, context, INPUT=mem_sub_lake_union_temp, OUTPUT="memory:"
    )["OUTPUT"]

    # add attribute
    layer_cat = mem_sub_lake_union
    # obtain projection crs id
    SpRef_in = qgis_vector_return_crs_id(
        processing, context, layer_cat, Input_Is_Feature_In_Mem=True
    )  ### get Raster spatialReference id

    # add attribute to layer
    layer_cat = qgis_vector_add_attributes(
        processing,
        context,
        INPUT_Layer=layer_cat,
        attribute_list=[
            QgsField("HRULake_ID", QVariant.Int),
            QgsField("HRU_IsLake", QVariant.Int),
        ],
    )

    # remove column not in fieldnames
    layer_cat = qgis_vector_field_calculator(
        processing=processing,
        context=context,
        FORMULA=" @row_number",
        FIELD_NAME="HRU_ID_Tp",
        INPUT=layer_cat,
        OUTPUT="memory:",
    )["OUTPUT"]

    # create HRU_lAKE_ID
    
    # obtain attribute table in vector
    Attri_table = Obtain_Attribute_Table(processing, context, layer_cat)
    # determine lake hru id
    Attri_table = Attri_table.fillna(-1)
    Attri_table = Attri_table.replace(to_replace="NULL", value=-1)
    Attri_table = Determine_Lake_HRU_Id(Attri_table)
    # copy determined lake hru id to vector
    layer_cat = Copy_Pddataframe_to_shpfile(
        Path_shpfile=layer_cat,
        Pddataframe=Attri_table,
        link_col_nm_shp="HRU_ID_Tp",
        link_col_nm_df="HRU_ID_Tp",
        UpdateColNM=["HRU_IsLake", "HRULake_ID", "SubId", "HyLakeId"],
        Input_Is_Feature_In_Mem=True,
    )
    # clean attribute table
    layer_cat, temp_not_used = Clean_Attribute_Name(
        layer_cat, fieldnames, Input_Is_Feature_In_Mem=True, Col_NM_Max="SubId"
    )

    # dissolve and fix geometry export output
    mem_union_fix = qgis_vector_fix_geometries(
        processing, context, INPUT=layer_cat, OUTPUT="memory:"
    )["OUTPUT"]
    Sub_Lake_HRU1 = qgis_vector_dissolve(
        processing,
        context,
        INPUT=mem_union_fix,
        FIELD=["HRULake_ID"],
        OUTPUT=os.path.join(
            tempfile.gettempdir(), str(np.random.randint(1, 10000 + 1)) + "tempfile.shp"
        ),
    )["OUTPUT"]
    Sub_Lake_HRU2 = qgis_vector_dissolve(
        processing,
        context,
        INPUT=Sub_Lake_HRU1,
        FIELD=["HRULake_ID"],
        OUTPUT=Path_finalcat_hru_out,
    )
    Sub_Lake_HRU = qgis_vector_dissolve(
        processing, context, INPUT=Sub_Lake_HRU1, FIELD=["HRULake_ID"], OUTPUT="memory:"
    )

    del layer_cat
    crs_id = qgis_vector_return_crs_id(
        processing, context, Sub_Lake_HRU["OUTPUT"], Input_Is_Feature_In_Mem=True
    )
    return Sub_Lake_HRU["OUTPUT"], crs_id, ["HRULake_ID", "HRU_IsLake", Sub_ID]


############


def Reproj_Clip_Dissolve_Simplify_Polygon(
    processing, context, layer_path, Project_crs, trg_crs, Class_Col, Layer_clip
):
    """Preprocess user provided polygons

    Function that will reproject clip input polygon with subbasin polygon
    and will dissolve the input polygon based on their ID, such as landuse id
    or soil id.

    Parameters
    ----------
    processing                        : qgis object
    context                           : qgis object
    layer_path                        : string
        The path to a specific polygon, for example path to landuse layer
    Project_crs                       : string
        the EPSG code of a projected coodinate system that will be used to
        calcuate HRU area and slope.
    trg_crs                           : string
        the EPSG code of a  coodinate system that will be used to
        calcuate reproject input polygon
    Class_Col                         : string
        the column name in the input polygon (layer_path) that contains
        their ID, for example land use ID or soil ID.
    Layer_clip                        : qgis object
        A shpfile with extent of the watershed, will be used to clip input
        input polygon
    Notes
    -------
        # TODO: May be add some function to simplify the input polygons
                for example, remove the landuse type with small areas
                or merge small landuse polygon into the surrounding polygon

    Returns:
    -------
        layer_dis                  : qgis object
            it is a polygon after preprocess
    """
    layer_proj = qgis_vector_reproject_layers(
        processing, context, layer_path, trg_crs, "memory:"
    )["OUTPUT"]
    layer_fix = qgis_vector_fix_geometries(
        processing, context, INPUT=layer_proj, OUTPUT="memory:"
    )["OUTPUT"]
    layer_clip = qgis_vector_clip(
        processing, context, layer_fix, Layer_clip, "memory:"
    )["OUTPUT"]
    #    layer_clip = processing.run("native:clip", {'INPUT':layer_fix,'OVERLAY':Layer_clip,'OUTPUT':'memory:'})['OUTPUT']
    # layer_dis = qgis_vector_dissolve(
    #     processing, context, INPUT=layer_clip, FIELD=[Class_Col], OUTPUT="memory:"
    # )["OUTPUT"]
    qgis_vector_create_spatial_index(processing, context, layer_clip)
    return layer_clip


def Union_Ply_Layers_And_Simplify(
    processing,
    context,
    Merge_layer_list,
    dissolve_filedname_list,
    fieldnames,
    OutputFolder,
    union_snap_distance,
    tempfolder,
    trg_crs,
):
    """Union input QGIS polygon layers

    Function will union polygon layers in Merge_layer_list
    dissove the union result based on field name in
    dissolve_filedname_list
    and remove all field names not in fieldnames

    Parameters
    ----------
    processing                        : qgis object
    context                           : qgis object
    Merge_layer_list                  : list
        a list contain polygons layers needs to be unioned
    dissolve_filedname_list           : list
        a list contain column name of ID in each polygon layer
        in Merge_layer_list
    fieldnames                        : list
        a list contain column names that are needed in the return
        QGIS polygon layers
    OutputFolder                      : String
        The path to a folder to save result during the processing

    Returns:
    -------
    mem_union_dis                  : qgis object
        it is a polygon object that generated by overlay all input
        layers
    """
    num = str(np.random.randint(1, 10000 + 1))

    has_grass = 1
    has_grass_sec = 1
    try:
        import grass.script as grass
    except ImportError or ModuleNotFoundError:
        has_grass = 0
    try:
        from grass_session import Session
    except ImportError or ModuleNotFoundError:
        has_grass_sec = 0    
    
    if has_grass and has_grass_sec:
        use_grass = 1
    else:
        use_grass = 0
    
    if use_grass:
        import grass.script as grass
        import grass.script.setup as gsetup
        from grass.pygrass.modules import Module
        from grass.pygrass.modules.shortcuts import general as g
        from grass.pygrass.modules.shortcuts import raster as r
        from grass.script import array as garray
        from grass.script import core as gcore
        from grass_session import Session
        import subprocess

        os.environ.update(
            dict(GRASS_COMPRESS_NULLS="1", GRASS_COMPRESSOR="ZSTD", GRASS_VERBOSE="1")
        )

        grassdb = os.getenv("GISDBASE")

        if not os.path.exists(grassdb):
            os.makedirs(grassdb)

        grass_location = 'location_hru'
                    
        PERMANENT = Session()
        PERMANENT.open(
            gisdb=grassdb,
            location=grass_location,
            create_opts=trg_crs,
        )                                              

        sqlpath = os.path.join(
            grassdb, grass_location, "PERMANENT", "sqlite", "sqlite.db"
        )
        
    ##union polygons
    if len(Merge_layer_list) == 1:
                
        mem_union = Merge_layer_list[0]
        mem_union_fix_ext = qgis_vector_fix_geometries(
            processing, context, INPUT=mem_union, OUTPUT="memory:"
        )["OUTPUT"]
        
    elif len(Merge_layer_list) > 1:
        
        for i in range(0, len(Merge_layer_list)):
            
            if i == 0:
                
                mem_union = Merge_layer_list[i]
                mem_union_fix_ext = qgis_vector_fix_geometries(
                    processing, context, INPUT=mem_union, OUTPUT="memory:"
                )["OUTPUT"]
                qgis_vector_create_spatial_index(processing, context, mem_union_fix_ext)
                
            else:
                
                mem_union_fix_temp = mem_union_fix_ext
                del mem_union_fix_ext
                print("union ith  layer ", i)
                input_layer_i = qgis_vector_fix_geometries(
                    processing, context, INPUT=Merge_layer_list[i], OUTPUT="memory:"
                )["OUTPUT"]
                
                adjusted_i = input_layer_i              
                adjusted_i_fix = qgis_vector_fix_geometries(
                    processing, context, INPUT = adjusted_i, OUTPUT="memory:"
                )["OUTPUT"]
                
                
                if use_grass:
                    con = sqlite3.connect(
                        os.path.join(grassdb, grass_location, "PERMANENT", "sqlite", "sqlite.db")
                    )
                    
                    grass_layer_1 = qgis_vector_fix_geometries(
                        processing, context, INPUT=mem_union_fix_temp, OUTPUT=os.path.join(grassdb,'union_input_1_'+str(i)+'_.shp')
                    )["OUTPUT"]
                    
                    grass_layer_2 = qgis_vector_fix_geometries(
                        processing, context, INPUT=adjusted_i_fix, OUTPUT=os.path.join(grassdb,'union_input_2_'+str(i)+'_.shp')
                    )["OUTPUT"]
                                                       
                    grass.run_command(
                        "v.import",
                        input=grass_layer_1,
                        output='layer_1_'+str(i),
                        overwrite=True,
                    )

                    grass.run_command(
                        "v.import",
                        input=grass_layer_2,
                        output='layer_2_'+str(i),
                        overwrite=True,
                    )

                    grass.run_command(
                        "v.overlay",
                        ainput='layer_1_'+str(i),
                        atype = 'area',
                        binput = 'layer_2_'+str(i),
                        btype = 'area',
                        operator = 'or',
                        output = 'union_'+str(i),
                        overwrite=True,
                        olayer = '1,0,0',
                    )
                    # try to rename column names 
                    
                    #obtain current column names
                    # p = grass.start_command("db.columns",table = 'union_'+str(i),stdout=grass.PIPE)
                    # col_names = p.communicate()[0]
                    # col_names = str(col_names).rstrip().split('\\r\\n')
                    sqlstat="SELECT * FROM %s" %('union_'+str(i))
                    union_pd = pd.read_sql_query(sqlstat, con)
                    col_names = union_pd.columns
                    # rename or delete the column
                    exist_names = []     
                    for icolnm in col_names:
                        if 'cat' in icolnm or icolnm == "'":
                            continue 
                                                
                        # avoid two same column
                        if icolnm[2:].lower() not in exist_names:
                            
                            grass.run_command(
                            'v.db.renamecolumn',
                             map = 'union_'+str(i),
                             column ="%s,%s"% (icolnm,icolnm[2:]),
                            )     
                                                       
                        exist_names.append(icolnm[2:].lower())    
                        # print(exist_names,icolnm)

                    grass.run_command(
                        "v.out.ogr",
                        input='union_'+str(i),
                        output = os.path.join(grassdb,'union_'+str(i)+'_.shp'),
                        format = 'ESRI_Shapefile',
                        overwrite=True,
                    )
                    
                    mem_union =  os.path.join(grassdb,'union_'+str(i)+'_.shp')
                    con.close()                                           
                else:                

                     mem_union = qgis_vector_union_two_layers(
                        processing,
                        context,
                        mem_union_fix_temp,
                        adjusted_i_fix,
                        "memory:",
                        OVERLAY_FIELDS_PREFIX="",
                    )["OUTPUT"]    
                                    
                mem_union_fix = qgis_vector_fix_geometries(
                    processing, context, INPUT=mem_union, OUTPUT="memory:"
                )["OUTPUT"]
                
                mem_union_fix_ext = mem_union_fix
                qgis_vector_create_spatial_index(processing, context, mem_union_fix_ext)
    else:
        print("No polygon needs to be overlaied.........should not happen ")
    
    ## remove non interested filed
    mem_union_fix_ext, temp_out_notused = Clean_Attribute_Name(
        mem_union_fix_ext, fieldnames, Input_Is_Feature_In_Mem=True
    )
    mem_union_dis = mem_union_fix_ext
    # qgis_vector_dissolve(
    #     processing,
    #     context,
    #     INPUT=mem_union_fix_ext,
    #     FIELD=dissolve_filedname_list,
    #     OUTPUT=os.path.join(tempfile.gettempdir(),num + "_dissolve_union.shp")
    # )
    #
    # mem_union_dis = qgis_vector_dissolve(
    #     processing,
    #     context,
    #     INPUT=os.path.join(tempfile.gettempdir(),num + "_dissolve_union.shp"),
    #     FIELD=dissolve_filedname_list,
    #     OUTPUT="memory:",
    # )["OUTPUT"]

    if use_grass:
        PERMANENT.close() 
                    
    return mem_union_dis


def Define_HRU_Attributes(
    processing,
    context,
    Project_crs,
    trg_crs,
    hru_layer,
    dissolve_filedname_list,
    Sub_ID,
    Landuse_ID,
    Soil_ID,
    Veg_ID,
    Other_Ply_ID_1,
    Other_Ply_ID_2,
    Landuse_info_data,
    Soil_info_data,
    Veg_info_data,
    DEM,
    Path_Subbasin_Ply,
    OutputFolder,
    min_hru_pct_sub_area,
    importance_order,
):

    """Generate attributes of each HRU

    Function will generate attributes that are needed by Raven and
    other hydrological models for each HRU

    Parameters
    ----------
    processing                        : qgis object
    context                           : qgis object
    Project_crs                       : string
        the EPSG code of a projected coodinate system that will be used to
        calcuate HRU area and slope.
    hru_layer                         : qgis object
        a polygon layer generated by overlay all input polygons
    dissolve_filedname_list           : list
        a list contain column name of ID in each polygon layer
        in Merge_layer_list
    Sub_ID                            : string
        The column name of the subbasin id in the subbasin polygon
    Landuse_ID                        : string
        the the column name in landuse polygon and Landuse_info csv
        indicate the landuse ID. when Path_Landuse_Ply is not
        provided. The Landuse ID should be
        1: land, -1: lake.
    Soil_ID                           : string
        the the column name in soil polygon and soil_info csv
        indicate the soil ID. when soil polygon is not
        provided. The Soil ID in Soil_info should be the same
        as Landuse ID.
    Veg_ID                            : string
        the the column name in vegetation polygon and veg_info csv
        indicate the vegetation ID. when Veg polygon is not
        provided. The Veg ID in Veg_info should be the same
        as Landuse ID.

    Landuse_info                      : Dataframe
        a dataframe that contains landuse information, including
        following attributes:
        Landuse_ID (can be any string)  - integer, the landuse ID in the
                                                   landuse polygon
        LAND_USE_C                      - string,  the landuse class name
                                                   for each landuse Type
    Soil_info                         : Dataframe
        a dataframe that contains soil information, including
        following attributes:
        Soil_ID (can be any string)     - integer, the Soil ID in the
                                                   soil polygon
        SOIL_PROF                       - string,  the Soil profile name
                                                   for each soil type
    Veg_info                          : Dataframe
        a dataframe file that contains vegetation information, including
        following attributes:
        Veg_ID (can be any string)      - integer, the vegetation ID in the
                                                   vegetation polygon
        VEG_C                           - string,  the vegetation class name
                                                   for each vegetation Type
    DEM                               : string (optional)
        the path to a raster elevation dataset, that will be used to
        calcuate average apspect, elevation and slope within each HRU.
        if no data is provided, basin average value will be used for
        each HRU.
    Path_Subbasin_Ply                 : string
        It is the path of the subbasin polygon, which is generated by
        toolbox. if not generated by toolbox, the attribute table should
        including following attribute.
        ##############Subbasin related attributes###########################
        SubID           - integer, The subbasin Id
        DowSubId        - integer, The downstream subbasin ID of this
                                   subbasin
        IsLake          - integer, If the subbasin is a lake / reservior
                                   subbasin. 1 yes, <0, no
        IsObs           - integer, If the subbasin contains a observation
                                   gauge. 1 yes, < 0 no.
        RivLength       - float,   The length of the river in current
                                   subbasin in m
        RivSlope        - float,   The slope of the river path in
                                   current subbasin, in m/m
        FloodP_n        - float,   Flood plain manning's coefficient, in -
        Ch_n            - float,   main channel manning's coefficient, in -
        BkfWidth        - float,   the bankfull width of the main channel
                                   in m
        BkfDepth        - float,   the bankfull depth of the main channel
                                   in m
        HyLakeId        - integer, the lake id
        LakeVol         - float,   the Volume of the lake in km3
        LakeDepth       - float,   the average depth of the lake m
        LakeArea        - float,   the area of the lake in m2
    OutputFolder                      : String
        The path to a folder to save result during the processing

    Returns:
    -------
    HRU_draf_final                  : qgis object
        it is a polygon object that generated by overlay all input
        layers and inlcude all needed attribue for hydrological model
        like RAVEN
    """
    num = str(np.random.randint(1, 10000 + 1))
    ### calcuate area of each feature
    formular = "area(transform($geometry, '%s','%s'))" % (
        hru_layer.crs().authid(),
        Project_crs,
    )
    layer_area = qgis_vector_field_calculator(
        processing=processing,
        context=context,
        FORMULA=formular,
        FIELD_NAME="HRU_Area",
        INPUT=hru_layer,
        OUTPUT="memory:",
    )["OUTPUT"]
    layer_area_id = qgis_vector_field_calculator(
        processing=processing,
        context=context,
        FORMULA=" @row_number + 1",
        FIELD_NAME="HRU_ID",
        INPUT=layer_area,
        OUTPUT="memory:",
    )["OUTPUT"]

    ### add attributes columns
    attribute_list = [
        QgsField("LAND_USE_C", QVariant.String),
        QgsField("VEG_C", QVariant.String),
        QgsField("SOIL_PROF", QVariant.String),
        QgsField("HRU_CenX", QVariant.Double),
        QgsField("HRU_CenY", QVariant.Double),
        QgsField("HRU_ID_New", QVariant.Double),
    ]

    layer_area_id = qgis_vector_add_attributes(
        processing, context, INPUT_Layer=layer_area_id, attribute_list=attribute_list
    )

    # qgis_vector_field_calculator(
    #     processing=processing,
    #     context=context,
    #     FORMULA=" @row_number + 1",
    #     FIELD_NAME="HRU_ID6",
    #     INPUT=layer_area_id,
    #     OUTPUT=os.path.join(OutputFolder,'tesdfdsfd.shp'),
    # )
    
    ### Determine HRU attribute HruID, LAND_USE_C,VEG_C,SOIL_PROF
    Attri_table = Obtain_Attribute_Table(processing, context, layer_area_id)    
    Attri_table.to_csv(
        os.path.join(tempfile.gettempdir(), "attribute_pre.csv"), sep=","
    )
        
    Attri_table = Attri_table.fillna(0)
    
    Attri_table = Attri_table.sort_values(by=[Sub_ID,Soil_ID,Landuse_ID]).copy(deep=True).reset_index()
        
    Attri_table = Determine_HRU_Attributes(
        Attri_table,
        Sub_ID,
        Landuse_ID,
        Soil_ID,
        Veg_ID,
        Other_Ply_ID_1,
        Other_Ply_ID_2,
        Landuse_info_data,
        Soil_info_data,
        Veg_info_data,
    )
    Attri_table.to_csv(os.path.join(tempfile.gettempdir(), "attribute.csv"))
    
#    Attri_table = Attri_table.sort_values(by=[Sub_ID,Soil_ID]).copy(deep=True)
    
    layer_area_id = Copy_Pddataframe_to_shpfile(
        Path_shpfile=layer_area_id,
        Pddataframe=Attri_table,
        link_col_nm_shp="HRU_ID",
        link_col_nm_df="HRU_ID",
        UpdateColNM=[
            Sub_ID,
            Landuse_ID,
            Soil_ID,
            Veg_ID,
            Other_Ply_ID_1,
            Other_Ply_ID_2,
            "LAND_USE_C",
            "SOIL_PROF",
            "VEG_C",
            "HRU_ID_New",
        ],
        Input_Is_Feature_In_Mem=True,
    )
    ### merge lake hru.
    qgis_vector_dissolve(
        processing,
        context,
        INPUT=layer_area_id,
        FIELD="HRU_ID_New",
        OUTPUT=os.path.join(tempfile.gettempdir(), num + "_dissolve_union2.shp"),
    )

    qgis_vector_dissolve(
        processing,
        context,
        INPUT=os.path.join(tempfile.gettempdir(), num + "_dissolve_union2.shp"),
        FIELD="HRU_ID_New",
        OUTPUT=os.path.join(tempfile.gettempdir(), num + "_dissolve_union3.shp"),
    )

    HRU_draft = qgis_vector_dissolve(
        processing,
        context,
        os.path.join(tempfile.gettempdir(), num + "_dissolve_union3.shp"),
        "HRU_ID_New",
        "memory:",
    )["OUTPUT"]

    # HRU_draft = qgis_vector_fix_geometries(
    #     processing, context, INPUT=HRU_draft, OUTPUT="memory:"
    # )["OUTPUT"]

    # qgis_vector_field_calculator(
    #     processing=processing,
    #     context=context,
    #     FORMULA=" @row_number + 1",
    #     FIELD_NAME="HRU_ID3",
    #     INPUT=HRU_draft,
    #     OUTPUT=os.path.join(OutputFolder,'tesdfdsfd555555.shp'),
    # )
    
    formular = "area(transform($geometry, '%s','%s'))" % (
        HRU_draft.crs().authid(),
        Project_crs,
    )

    HRU_draft = qgis_vector_field_calculator(
        processing=processing,
        context=context,
        FORMULA=formular,
        FIELD_NAME="HRU_Area2",
        INPUT=HRU_draft,
        OUTPUT="memory:",
    )["OUTPUT"]

    HRU_draft = qgis_vector_extract_by_attribute(
        processing,
        context,
        INPUT_Layer=HRU_draft,
        FIELD="SubId",
        OPERATOR=1,
        VALUE=0,
        OUTPUT="memory:",
    )["OUTPUT"]

    HRU_draft = qgis_vector_extract_by_attribute(
        processing,
        context,
        INPUT_Layer=HRU_draft,
        FIELD="HRU_Area2",
        OPERATOR=1,
        VALUE=0,
        OUTPUT="memory:",
    )["OUTPUT"]



    ###### simplify HRU


    Attri_table = Obtain_Attribute_Table(processing, context, HRU_draft)    
    Attri_table['HRU_Area'] = Attri_table['HRU_Area2']     
    hruinfo = simplidfy_hrus(min_hru_pct_sub_area,Attri_table,importance_order)
    
#    hruinfo = hruinfo.sort_values(by=[Sub_ID,Soil_ID]).copy(deep=True)

    #######
    
    layer_area_id = Copy_Pddataframe_to_shpfile(
        Path_shpfile=HRU_draft,
        Pddataframe=hruinfo,
        link_col_nm_shp="HRU_ID",
        link_col_nm_df="HRU_ID",
        UpdateColNM=[
            Sub_ID,
            Landuse_ID,
            Soil_ID,
            Veg_ID,
            Other_Ply_ID_1,
            Other_Ply_ID_2,
            "LAND_USE_C",
            "SOIL_PROF",
            "VEG_C",
            "HRU_ID_New",
        ],
        Input_Is_Feature_In_Mem=True,
    )

    ### merge lake hru.
    qgis_vector_dissolve(
        processing,
        context,
        INPUT=layer_area_id,
        FIELD="HRU_ID_New",
        OUTPUT=os.path.join(tempfile.gettempdir(), num + "_dissolve_union2b.shp"),
    )

    qgis_vector_dissolve(
        processing,
        context,
        INPUT=os.path.join(tempfile.gettempdir(), num + "_dissolve_union2b.shp"),
        FIELD="HRU_ID_New",
        OUTPUT=os.path.join(tempfile.gettempdir(), num + "_dissolve_union3b.shp"),
    )

    HRU_draft = qgis_vector_dissolve(
        processing,
        context,
        os.path.join(tempfile.gettempdir(), num + "_dissolve_union3b.shp"),
        "HRU_ID_New",
        "memory:",
    )["OUTPUT"]

    HRU_draft = qgis_vector_fix_geometries(
        processing, context, INPUT=HRU_draft, OUTPUT="memory:"
    )["OUTPUT"]
     
    formular = "area(transform($geometry, '%s','%s'))" % (
        HRU_draft.crs().authid(),
        Project_crs,
    )

    HRU_draft = qgis_vector_field_calculator(
        processing=processing,
        context=context,
        FORMULA=formular,
        FIELD_NAME="HRU_Area2",
        INPUT=HRU_draft,
        OUTPUT="memory:",
    )["OUTPUT"]
    

    formular = "x(centroid(transform($geometry,'%s','%s')))" % (
        HRU_draft.crs().authid(),
        "EPSG:4326",
    )

    HRU_draft = qgis_vector_field_calculator(
        processing=processing,
        context=context,
        FORMULA=formular,
        FIELD_NAME="HRU_CenX",
        INPUT=HRU_draft,
        OUTPUT="memory:",
    )["OUTPUT"]
        

    formular = "y(centroid(transform($geometry,'%s','%s')))" % (
        HRU_draft.crs().authid(),
        "EPSG:4326",
    )

    HRU_draft = qgis_vector_field_calculator(
        processing=processing,
        context=context,
        FORMULA=formular,
        FIELD_NAME="HRU_CenY",
        INPUT=HRU_draft,
        OUTPUT="memory:",
    )["OUTPUT"]
        
    # HRU_draft = Add_centroid_to_feature(
    #     Path_feagure=HRU_draft,
    #     centroidx_nm="HRU_CenX",
    #     centroidy_nm="HRU_CenY",
    #     Input_Is_Feature_In_Mem=True,
    # )

    ### add subbasin attribute back to hru polygons

    HRU_draft_sub_info = qgis_vector_join_attribute_table(
        processing,
        context,
        INPUT1=HRU_draft,
        FIELD1=Sub_ID,
        INPUT2=Path_Subbasin_Ply,
        FIELD2=Sub_ID,
        OUTPUT="memory:",
    )["OUTPUT"]

    if DEM != "#":

        HRU_draft_proj = qgis_vector_reproject_layers(
            processing, context, HRU_draft_sub_info, Project_crs, "memory:"
        )["OUTPUT"]

        DEM_proj = qgis_raster_gdal_warpreproject(
            processing, Input=DEM, TARGET_CRS=Project_crs, Output="TEMPORARY_OUTPUT"
        )["OUTPUT"]

        ext = HRU_draft_proj.extent()
        projwin = "%s,%s,%s,%s [%s]" % (
            ext.xMinimum(),
            ext.xMaximum(),
            ext.yMinimum(),
            ext.yMaximum(),
            Project_crs,
        )
        DEM_clip = qgis_raster_clip_raster_by_extent(
            processing, Input=DEM_proj, PROJWIN=projwin, Output="TEMPORARY_OUTPUT"
        )["OUTPUT"]

        DEM_slope = qgis_raster_slope(
            processing, Input=DEM_clip, Output="TEMPORARY_OUTPUT"
        )["OUTPUT"]
        DEM_aspect = qgis_raster_aspect(
            processing, Input=DEM_clip, Output="TEMPORARY_OUTPUT"
        )["OUTPUT"]

        qgis_raster_zonal_statistics(
            processing,
            INPUT_RASTER=DEM_slope,
            INPUT_VECTOR=HRU_draft_proj,
            COLUMN_PREFIX="HRU_S_",
        )
        qgis_raster_zonal_statistics(
            processing,
            INPUT_RASTER=DEM_aspect,
            INPUT_VECTOR=HRU_draft_proj,
            COLUMN_PREFIX="HRU_A_",
        )
        qgis_raster_zonal_statistics(
            processing,
            INPUT_RASTER=DEM_clip,
            INPUT_VECTOR=HRU_draft_proj,
            COLUMN_PREFIX="HRU_E_",
        )

        HRU_draft_reproj = qgis_vector_reproject_layers(
            processing,
            context,
            INPUT=HRU_draft_proj,
            TARGET_CRS=trg_crs,
            OUTPUT="memory:",
        )["OUTPUT"]

    else:
        ## if no dem provided hru slope will use subbasin slope aspect
        ## and elevation
        formula = ' "%s" ' % "BasSlope"
        HRU_draft_sub_info_S = qgis_vector_field_calculator(
            processing=processing,
            context=context,
            FORMULA=formula,
            FIELD_NAME="HRU_S_mean",
            INPUT=HRU_draft_sub_info,
            OUTPUT="memory:",
            FIELD_PRECISION=3,
        )["OUTPUT"]

        formula = ' "%s" ' % "BasAspect"
        HRU_draft_sub_info_A = qgis_vector_field_calculator(
            processing=processing,
            context=context,
            FORMULA=formula,
            FIELD_NAME="HRU_A_mean",
            INPUT=HRU_draft_sub_info_S,
            OUTPUT="memory:",
            FIELD_PRECISION=3,
        )["OUTPUT"]

        formula = ' "%s" ' % "MeanElev"
        HRU_draft_sub_info_S = qgis_vector_field_calculator(
            processing=processing,
            context=context,
            FORMULA=formula,
            FIELD_NAME="HRU_E_mean",
            INPUT=HRU_draft_sub_info_A,
            OUTPUT="memory:",
            FIELD_PRECISION=3,
        )["OUTPUT"]

        HRU_draft_reproj = HRU_draft_sub_info_S

    ### update HRU area
    formular = "area(transform($geometry, '%s','%s'))" % (trg_crs, Project_crs)
    HRU_draf_final = qgis_vector_field_calculator(
        processing=processing,
        context=context,
        FORMULA=formular,
        FIELD_NAME="HRU_Area",
        INPUT=HRU_draft_reproj,
        OUTPUT="memory:",
        NEW_FIELD=False,
        FIELD_PRECISION=3,
    )["OUTPUT"]

    qgis_vector_extract_by_attribute(
        processing,
        context,
        INPUT_Layer=HRU_draf_final,
        FIELD="HRU_Area",
        OPERATOR=1,
        VALUE="0",
        OUTPUT=os.path.join(tempfile.gettempdir(), num + "_extract_non_zero_area1.shp"),
    )

    qgis_vector_extract_by_attribute(
        processing,
        context,
        INPUT_Layer=os.path.join(
            tempfile.gettempdir(), num + "_extract_non_zero_area1.shp"
        ),
        FIELD="HRU_Area",
        OPERATOR=1,
        VALUE="0",
        OUTPUT=os.path.join(tempfile.gettempdir(), num + "_extract_non_zero_area2.shp"),
    )

    HRU_draf_final = qgis_vector_extract_by_attribute(
        processing,
        context,
        INPUT_Layer=os.path.join(
            tempfile.gettempdir(), num + "_extract_non_zero_area2.shp"
        ),
        FIELD="HRU_Area",
        OPERATOR=1,
        VALUE="0",
        OUTPUT="memory:",
    )["OUTPUT"]
    return HRU_draf_final
