""" CsvPaths' intent is to help you manage and automate your use
    of the CsvPath library. it makes it easier to scale your CSV quality control. """

from abc import ABC, abstractmethod
from typing import List, Any
import traceback
from datetime import datetime, timezone
from .managers.errors.error import Error
from .managers.errors.error_comms import ErrorCommunications
from .managers.errors.error_manager import ErrorManager
from .managers.errors.error_collector import ErrorCollector
from .util.config import Config
from .util.log_utility import LogUtility
from .util.metadata_parser import MetadataParser
from .util.exceptions import InputException, CsvPathsException
from .managers.paths.paths_manager import PathsManager
from .managers.files.file_manager import FileManager
from .managers.results.results_manager import ResultsManager
from .managers.results.result import Result
from . import CsvPath


class CsvPathsPublic(ABC):
    """this abstract class is the public interface for CsvPaths.

    a CsvPaths instance manages applying any number of csvpaths
    to any number of files. CsvPaths applies sets of csvpaths
    to a given file, on demand. Think of CsvPaths as a session
    object. It gives you a way to manage files, csvpaths, and
    the results generated by applying paths to files. It is not
    intended for concurrent use. If you need multiple threads,
    create multiple CsvPaths instances.
    """

    @abstractmethod
    def csvpath(self) -> CsvPath:  # pragma: no cover
        """Gets a CsvPath object primed with a reference to this CsvPaths"""

    @abstractmethod
    def collect_paths(self, *, pathsname, filename) -> None:  # pragma: no cover
        """Sequentially does a CsvPath.collect() on filename for every named path. lines are collected into results."""

    @abstractmethod
    def fast_forward_paths(self, *, pathsname, filename) -> None:  # pragma: no cover
        """Sequentially does a CsvPath.fast_forward() on filename for every named path"""

    @abstractmethod
    def next_paths(self, *, pathsname, filename) -> None:  # pragma: no cover
        """Does a CsvPath.next() on filename for every line against every named path in sequence"""

    @abstractmethod
    def collect_by_line(
        self, *, pathsname, filename, if_all_agree=False, collect_when_not_matched=False
    ):  # pragma: no cover
        """Does a CsvPath.collect() on filename where each row is considered
        by every named path before the next row starts

        next_by_line for if_all_agree and collect_when_not_matched.
        """

    @abstractmethod
    def fast_forward_by_line(
        self, *, pathsname, filename, if_all_agree=False, collect_when_not_matched=False
    ):  # pragma: no cover
        """Does a CsvPath.fast_forward() on filename where each row is
        considered by every named path before the next row starts

        next_by_line for if_all_agree and collect_when_not_matched.
        """

    @abstractmethod
    def next_by_line(
        self,
        *,
        pathsname,
        filename,
        collect: bool = False,
        if_all_agree=False,
        collect_when_not_matched=False,
    ) -> List[Any]:  # pragma: no cover
        """Does a CsvPath.next() on filename where each row is considered
        by every named path before the next row starts.

        if_all_agree=True means all the CsvPath instances must match for
        the line to be kept. However, every CsvPath instance will keep its
        own matches in its results regardless of if every line kept was
        returned to the caller by CsvPaths.

        collect_when_not_matched=True inverts the match so that lines
        which did not match are returned, rather than the default behavior.
        """


class CsvPathsCoordinator(ABC):
    """This abstract class defines callbacks for CsvPath instances to
    broadcast state to their siblings through CsvPaths. A CsvPath
    instance might stop the entire run, rather than each CsvPath
    instance needing to contain the same logic that stops their
    participation in a run.
    """

    @abstractmethod
    def stop_all(self) -> None:  # pragma: no cover
        """Stops every CsvPath instance in a run"""

    @abstractmethod
    def fail_all(self) -> None:  # pragma: no cover
        """Fails every CsvPath instance in a run"""

    @abstractmethod
    def skip_all(self) -> None:  # pragma: no cover
        """skips the line for every CsvPath instance in a run"""

    @abstractmethod
    def advance_all(self, lines: int) -> None:  # pragma: no cover
        """advances every CsvPath instance in a run"""


class CsvPaths(CsvPathsPublic, CsvPathsCoordinator, ErrorCollector):
    """Manages the application of csvpaths to files. Csvpaths must be grouped and named.
    Files must be named. Results are held by the results_manager.
    """

    # pylint: disable=too-many-instance-attributes

    def __init__(
        self,
        *,
        delimiter=",",
        quotechar='"',
        skip_blank_lines=True,
        print_default=True,
        config: Config = None,
    ):
        self._config = Config() if config is None else config
        #
        # managers centralize activities, offer async potential, and
        # are where integrations hook in. ErrorManager functionality
        # must be available in CsvPath too. The others are CsvPaths
        # only.
        #
        self._set_managers()
        """
        self.paths_manager = PathsManager(csvpaths=self)
        self.file_manager = FileManager(csvpaths=self)
        self.results_manager = ResultsManager(csvpaths=self)
        self.ecoms = ErrorCommunications(csvpaths=self)
        self._error_manager = ErrorManager(csvpaths=self)
        """
        #
        # TODO:
        # self.print_manager = ... <<<=== should we do this?
        #
        #
        self.print_default = print_default
        self.delimiter = delimiter
        self.quotechar = quotechar
        self.skip_blank_lines = skip_blank_lines
        self.current_matcher: CsvPath = None
        self.logger = LogUtility.logger(self)
        self._errors = []
        # coordinator attributes
        self._stop_all = False
        self._fail_all = False
        self._skip_all = False
        self._advance_all = 0
        self._current_run_time = None
        self._run_time_str = None
        self.named_paths_name = None
        self.named_file_name = None

        #
        # metrics is for OTLP OpenTelemetry. it should only
        # be used by the OTLP listener. it is here because
        # the integration may need a long-lived presence. if
        # needed, the first OTLP listener will set it up
        # before spinning up a thread. any other OTLP
        # listener threads that need to use a long-lived metric
        # will work with this property.
        #
        self.metrics = None
        self.logger.info("initialized CsvPaths")

    def _set_managers(self) -> None:
        self.paths_manager = PathsManager(csvpaths=self)
        self.file_manager = FileManager(csvpaths=self)
        self.results_manager = ResultsManager(csvpaths=self)
        self.ecoms = ErrorCommunications(csvpaths=self)
        self._error_manager = ErrorManager(csvpaths=self)

    @property
    def error_manager(self) -> ErrorManager:
        return self._error_manager

    @error_manager.setter
    def error_manager(self, em: ErrorManager) -> None:
        if em.csvpaths is None:
            raise Exception("CsvPaths cannot be None")
        self._error_manager = em

    def run_time_str(self, pathsname=None) -> str:
        """adds the stringified current run time to the named-paths
        group home_dir to create the run_dir"""
        if self._run_time_str is None and pathsname is None:
            raise CsvPathsException(
                "Cannot have None in both run_time_str and pathsname"
            )
        if self._run_time_str is None:
            self._run_time_str = self.results_manager.get_run_time_str(
                pathsname, self.current_run_time
            )
        return self._run_time_str

    @property
    def current_run_time(self) -> datetime:
        """gets the time marking the start of the run. used to create the run home directory."""
        if self._current_run_time is None:
            self._current_run_time = datetime.now(timezone.utc)
        return self._current_run_time

    def clear_run_coordination(self) -> None:
        """run coordination is the set of signals that csvpaths send to affect
        one another through the CsvPaths instance"""
        self._stop_all = False
        self._fail_all = False
        self._skip_all = False
        self._advance_all = 0
        self._current_run_time = None
        self._run_time_str = None
        self.logger.debug("Cleared run coordination")

    def csvpath(self) -> CsvPath:
        path = CsvPath(
            csvpaths=self,
            delimiter=self.delimiter,
            quotechar=self.quotechar,
            skip_blank_lines=self.skip_blank_lines,
            #
            # in the usual case we don't want csvpaths and its csvpath children
            # to share the same config. sharing doesn't offer much. the flexibility
            # of having separate configs is valuable.
            #
            config=None,
            print_default=self.print_default,
            error_manager=self.error_manager,
        )
        return path

    def stop_all(self) -> None:  # pragma: no cover
        self._stop_all = True

    def fail_all(self) -> None:  # pragma: no cover
        self._fail_all = True

    def skip_all(self) -> None:  # pragma: no cover
        self._skip_all = True

    def advance_all(self, lines: int) -> None:  # pragma: no cover
        self._advance_all = lines

    @property
    def errors(self) -> List[Error]:  # pylint: disable=C0116
        return self._errors

    def collect_error(self, error: Error) -> None:  # pylint: disable=C0116
        self._errors.append(error)

    def has_errors(self) -> bool:  # pylint: disable=C0116
        return len(self._errors) > 0

    @property
    def config(self) -> Config:  # pylint: disable=C0116
        if not self._config:
            self._config = Config()  # pragma: no cover
        return self._config

    #
    # this is the preferred way to update config. it is preferred because
    # csvpath and csvpaths work off the same config file, even though they,
    # in some cases, have separate keys. if you update the config directly
    # before a run starts using the CsvPaths's Config you have to remember
    # to save and reload for it to effect both CsvPaths and CsvPath. this
    # method does the save and reload every time.
    #
    def add_to_config(self, section, key, value) -> None:
        self.config.add_to_config(section=section, key=key, value=value)
        self.config.save_config()
        self.config.reload()
        self._set_managers()

    def clean(self, *, paths) -> None:
        """at this time we do not recommend reusing CsvPaths, but it is doable
        you should clean before reuse unless you want to accumulate results."""
        self.results_manager.clean_named_results(paths)
        self.clear_run_coordination()
        # self.error_manager.reset()
        self._errors = []
        self.named_paths_name = None
        self.named_file_name = None

    def collect_paths(self, *, pathsname, filename) -> None:
        paths = self.paths_manager.get_named_paths(pathsname)
        if paths is None:
            raise InputException(f"No named-paths found for {pathsname}")
        if len(paths) == 0:
            raise InputException(f"Named-paths group {pathsname} is empty")
        if "" in paths:
            raise InputException(
                f"Named-paths group {pathsname} has one or more empty csvpaths"
            )
        file = self.file_manager.get_named_file(filename)
        if file is None:
            raise InputException(f"No named-file found for {filename}")
        self.logger.info("Prepping %s and %s", filename, pathsname)
        self.clean(paths=pathsname)
        self.logger.info(
            "Beginning collect_paths %s with %s paths", pathsname, len(paths)
        )
        crt = self.run_time_str(pathsname)
        results = []
        #
        # run starts here
        #
        self.results_manager.start_run(
            run_dir=crt, pathsname=pathsname, filename=filename
        )
        #
        #
        #
        for i, path in enumerate(paths):
            csvpath = self.csvpath()
            result = Result(
                csvpath=csvpath,
                file_name=filename,
                paths_name=pathsname,
                run_index=i,
                run_time=self.current_run_time,
                run_dir=crt,
            )
            # casting a broad net because if "raise" not in the error policy we
            # want to never fail during a run
            try:
                self._load_csvpath(
                    csvpath=csvpath,
                    path=path,
                    file=file,
                    pathsname=pathsname,
                    filename=filename,
                    crt=crt,
                )
                #
                # the add has to come after _load_csvpath because we need the identity or index
                # to be stable and the identity is found in load, if it exists.
                #
                self.results_manager.add_named_result(result)
                lines = result.lines
                self.logger.debug("Collecting lines using a %s", type(lines))
                csvpath.collect(lines=lines)
                if lines is None:
                    self.logger.error(  # pragma: no cover
                        "Unexpected None for lines after collect_paths: file: %s, match: %s",
                        file,
                        csvpath.match,
                    )
                #
                # this is obviously not a good idea for very large files!
                #
                result.unmatched = csvpath.unmatched
            except Exception as ex:  # pylint: disable=W0718
                # ex.trace = traceback.format_exc()
                # ex.source = self
                if self.error_manager.csvpaths is None:
                    raise Exception("ErrorManager's CsvPaths cannot be None")
                self.error_manager.handle_error(source=self, msg=f"{ex}")
                if self.ecoms.do_i_raise():
                    self.results_manager.save(result)
                    raise
            self.results_manager.save(result)
            results.append(result)
        #
        # run ends here
        #
        self.results_manager.complete_run(
            run_dir=crt, pathsname=pathsname, results=results
        )
        #
        # update/write run manifests here
        #  - validity (are all paths valid)
        #  - paths-completeness (did they all run and complete)
        #  - method (collect, fast_forward, next)
        #  - timestamp
        #
        self.clear_run_coordination()
        self.logger.info(
            "Completed collect_paths %s with %s paths", pathsname, len(paths)
        )

    def _load_csvpath(
        self,
        *,
        csvpath: CsvPath,
        path: str,
        file: str,
        pathsname: str = None,
        filename,
        by_line: bool = False,
        crt: str,
    ) -> None:
        # file is the physical file (+/- if preceding mode) filename is the named-file name
        self.logger.debug("Beginning to load csvpath %s with file %s", path, file)
        csvpath.named_paths_name = pathsname
        self.named_paths_name = pathsname
        csvpath.named_file_name = filename
        self.named_file_name = filename
        #
        # by_line==True means we are starting a run that is breadth-first ultimately using
        # next_by_line(). by_line runs cannot be source-mode preceding and have different
        # semantics around csvpaths influencing one another.
        #
        # we strip comments from above the path so we need to extract them first
        path = MetadataParser(self).extract_metadata(instance=csvpath, csvpath=path)
        identity = csvpath.identity
        self.logger.debug("Csvpath %s after metadata extract: %s", identity, path)
        # update the settings using the metadata fields we just collected
        csvpath.update_settings_from_metadata()
        #
        # if we have a reference, resolve it. we may not actually use the file
        # if we're in source-mode: preceding, but that doesn't matter from the
        # pov of the reference.
        #
        if filename.startswith("$"):
            self.logger.debug(
                "File name is a reference: %s. Replacing the path passed in with the reffed data file path.",
                filename,
            )
            file = self.results_manager.data_file_for_reference(filename, not_name=crt)
        #
        #
        #
        if csvpath.data_from_preceding is True:
            if by_line is True:
                raise CsvPathsException(
                    "Breadth-first runs do not support source-mode preceding because each line of data flows through each csvpath in order already"
                )
            #
            # we are in source-mode: preceding
            # that means we ignore the original data file path and
            # instead use the data.csv from the preceding csvpath. that is,
            # assuming there is a preceding csvpath and it created and
            # saved data.
            #
            # find the preceding csvpath in the named-paths
            #
            # we may be in a file reference like: $sourcemode.csvpaths.source2:from. if so
            # we just need the named-paths name.
            #
            if pathsname.startswith("$"):
                self.logger.debug(
                    "Named-paths name is a reference: %s. Stripping it down to just the actual named-paths name.",
                    pathsname,
                )
                pathsname = pathsname[1 : pathsname.find(".")]
            result = self.results_manager.get_last_named_result(
                name=pathsname, before=csvpath.identity
            )
            if result is not None:
                # get its data.csv path for this present run
                # swap in that path for the regular origin path
                file = result.data_file_path
                csvpath.metadata["source-mode-source"] = file
                self.logger.info(
                    "Csvpath identified as %s uses last csvpath's data.csv at %s as source",
                    csvpath.identity,
                    file,
                )
            else:
                self.logger.warning(
                    "Cannot find a preceding data file to use for csvpath identified as %s running in source-mode",
                    csvpath.identity,
                )
        f = path.find("[")
        self.logger.debug("Csvpath matching part starts at char # %s", f)
        apath = f"${file}{path[f:]}"
        self.logger.info("Parsing csvpath %s", apath)
        csvpath.parse(apath)
        #
        # ready to run. time to register the run. this is separate from
        # the run_register.py (ResultsRegister) event
        #
        # crt = self.run_time_str(pathsname)
        # fingerprint = self.file_manager.get_fingerprint_for_name(filename)
        #
        self.logger.debug("Done loading csvpath")

    def fast_forward_paths(self, *, pathsname, filename):
        """runs the named-paths group named without collecting matches."""
        paths = self.paths_manager.get_named_paths(pathsname)
        file = self.file_manager.get_named_file(filename)
        self.logger.info("Prepping %s and %s", filename, pathsname)
        self.clean(paths=pathsname)
        self.logger.info(
            "Beginning FF %s with %s paths against file %s. No match results will be held.",
            pathsname,
            len(paths),
            filename,
        )
        crt = self.run_time_str(pathsname)
        #
        # run starts here
        #
        self.results_manager.start_run(
            run_dir=crt, pathsname=pathsname, filename=filename
        )
        #
        #
        #
        results = []
        for i, path in enumerate(paths):
            csvpath = self.csvpath()
            self.logger.debug("Beginning to FF CsvPath instance: %s", csvpath)
            result = Result(
                csvpath=csvpath,
                file_name=filename,
                paths_name=pathsname,
                run_index=i,
                run_time=self.current_run_time,
                run_dir=crt,
            )
            try:
                self._load_csvpath(
                    csvpath=csvpath,
                    path=path,
                    file=file,
                    pathsname=pathsname,
                    filename=filename,
                    crt=crt,
                )
                #
                # the add has to come after _load_csvpath because we need the identity or index
                # to be stable and the identity is found in load, if it exists.
                #
                self.results_manager.add_named_result(result)
                self.logger.info(
                    "Parsed csvpath %s pointed at %s and starting to fast-forward",
                    i,
                    file,
                )
                csvpath.fast_forward()
                self.logger.info(
                    "Completed fast forward of csvpath %s against %s", i, file
                )
            except Exception as ex:  # pylint: disable=W0718
                # ex.trace = traceback.format_exc()
                # ex.source = self
                self.error_manager.handle_error(source=self, msg=f"{ex}")
                if self.ecoms.do_i_raise():
                    self.results_manager.save(result)
                    raise
            self.results_manager.save(result)
            results.append(result)
        #
        # run ends here
        #
        self.results_manager.complete_run(
            run_dir=crt, pathsname=pathsname, results=results
        )
        self.clear_run_coordination()
        self.logger.info(
            "Completed fast_forward_paths %s with %s paths", pathsname, len(paths)
        )

    def next_paths(
        self, *, pathsname, filename, collect: bool = False
    ):  # pylint: disable=R0914
        """appends the Result for each CsvPath to the end of
        each line it produces. this is so that the caller can easily
        interrogate the CsvPath for its path parts, file, etc."""
        paths = self.paths_manager.get_named_paths(pathsname)
        file = self.file_manager.get_named_file(filename)
        self.logger.info("Prepping %s and %s", filename, pathsname)
        self.clean(paths=pathsname)
        self.logger.info("Beginning next_paths with %s paths", len(paths))
        crt = self.run_time_str(pathsname)
        #
        # run starts here
        #
        self.results_manager.start_run(
            run_dir=crt, pathsname=pathsname, filename=filename
        )
        #
        #
        #
        results = []
        for i, path in enumerate(paths):
            if self._skip_all:
                skip_err = "Found the skip-all signal set. skip_all() is"
                skip_err = f"{skip_err} only for breadth-first runs using the"
                skip_err = f"{skip_err} '_by_line' methods. It has the same"
                skip_err = f"{skip_err} effect as skip() in a"
                skip_err = f"{skip_err} serial run like this one."
                self.logger.error(skip_err)
            if self._stop_all:
                self.logger.warning("Stop-all set. Shutting down run.")
                break
            if self._advance_all > 0:
                advance_err = "Found the advance-all signal set. advance_all() is"
                advance_err = f"{advance_err} only for breadth-first runs using the"
                advance_err = f"{advance_err} '_by_line' methods. It has the same"
                advance_err = f"{advance_err} effect as advance() in a"
                advance_err = f"{advance_err} serial run like this one."
                self.logger.error(advance_err)
            csvpath = self.csvpath()
            result = Result(
                csvpath=csvpath,
                file_name=filename,
                paths_name=pathsname,
                run_index=i,
                run_time=self.current_run_time,
                run_dir=crt,
            )
            if self._fail_all:
                self.logger.warning(
                    "Fail-all set. Failing all remaining CsvPath instances in the run."
                )
                csvpath.is_valid = False
            try:
                self._load_csvpath(
                    csvpath=csvpath,
                    path=path,
                    file=file,
                    pathsname=pathsname,
                    filename=filename,
                    crt=crt,
                )
                #
                # the add has to come after _load_csvpath because we need the identity or index
                # to be stable and the identity is found in load, if it exists.
                #
                self.results_manager.add_named_result(result)
                for line in csvpath.next():
                    #
                    # removed dec 1. why was this? it doesn't seem to make sense and
                    # removing it doesn't break any unit tests. was it a mistake?
                    #
                    # line.append(result)
                    if collect:
                        result.append(line)
                        result.unmatched = csvpath.unmatched
                    yield line
            except Exception as ex:  # pylint: disable=W0718
                self.error_manager.handle_error(source=self, msg=f"{ex}")
                if self.ecoms.do_i_raise():
                    self.results_manager.save(result)
                    raise
            self.results_manager.save(result)
            results.append(result)
        #
        # run ends here
        #
        self.results_manager.complete_run(
            run_dir=crt, pathsname=pathsname, results=results
        )
        self.clear_run_coordination()

    # =============== breadth first processing ================

    def collect_by_line(
        self, *, pathsname, filename, if_all_agree=False, collect_when_not_matched=False
    ):
        self.logger.info(
            "Starting collect_by_line for paths: %s and file: %s", pathsname, filename
        )
        lines = []
        for line in self.next_by_line(  # pylint: disable=W0612
            pathsname=pathsname,
            filename=filename,
            collect=True,
            if_all_agree=if_all_agree,
            collect_when_not_matched=collect_when_not_matched,
        ):
            # re: W0612: we need 'line' in order to do the iteration. we have to iterate.
            lines.append(line)
        self.logger.info(
            "Completed collect_by_line for paths: %s and file: %s", pathsname, filename
        )
        #
        # the results have all the lines according to what CsvPath captured them, but
        # since we're doing if_all_agree T/F we should return the union here. for some
        # files this obviously makes the data in memory problem even bigger, but it's
        # operator's responsibility to know if that will be a problem for their use
        # case.
        #
        return lines

    def fast_forward_by_line(
        self, *, pathsname, filename, if_all_agree=False, collect_when_not_matched=False
    ):
        self.logger.info(
            "Starting fast_forward_by_line for paths: %s and file: %s",
            pathsname,
            filename,
        )
        for line in self.next_by_line(  # pylint: disable=W0612
            pathsname=pathsname,
            filename=filename,
            collect=False,
            if_all_agree=if_all_agree,
            collect_when_not_matched=collect_when_not_matched,
        ):
            # re: W0612: we need 'line' in order to do the iteration. we have to iterate.
            pass
        self.logger.info(
            "Completed fast_forward_by_line for paths: %s and file: %s",
            pathsname,
            filename,
        )

    def next_by_line(  # pylint: disable=R0912,R0915,R0914
        self,
        *,
        pathsname,
        filename,
        collect: bool = False,
        if_all_agree=False,
        collect_when_not_matched=False,
    ) -> List[Any]:
        # re: R0912 -- absolutely. plan to refactor.
        self.logger.info("Prepping %s and %s", filename, pathsname)
        self.clean(paths=pathsname)
        fn = self.file_manager.get_named_file(filename)
        paths = self.paths_manager.get_named_paths(pathsname)
        if (
            paths is None or not isinstance(paths, list) or len(paths) == 0
        ):  # pragma: no cover
            raise InputException(
                f"Pathsname '{pathsname}' must name a list of csvpaths"
            )
        #
        # experiment!
        #
        crt = self.run_time_str(pathsname)
        #
        # also use of crt below
        #
        csvpath_objects = self._load_csvpath_objects(
            paths=paths,
            named_file=fn,
            collect_when_not_matched=collect_when_not_matched,
            filename=filename,
            pathsname=pathsname,
            crt=crt,
        )
        #
        # prep has to come after _load_csvpath_objects because we need the identity or
        # indexes to be stable and the identity is found in the load, if it exists.
        #
        self._prep_csvpath_results(
            csvpath_objects=csvpath_objects,
            filename=filename,
            pathsname=pathsname,
            crt=crt,
        )
        #
        # setting fn into the csvpath is less obviously useful at CsvPaths
        # but we'll do it for consistency.
        #
        self.logger.info("Beginning next_by_line with %s paths", len(csvpath_objects))
        reader = FileManager.get_reader(
            fn, delimiter=self.delimiter, quotechar=self.quotechar
        )
        stopped_count: List[int] = []
        for line in reader.next():
            # for line in reader:  # pylint: disable=R1702
            # question to self: should this default be in a central place
            # so that we can switch to OR, in part by changing the default?
            keep = if_all_agree
            self._skip_all = False
            self._advance_all = 0
            try:
                # p is a (CsvPath, List[List[str]]) where the second item is
                # the line-by-line results of the first item's matching
                for p in csvpath_objects:
                    self.current_matcher = p[0]
                    if self._fail_all:
                        self.logger.warning(
                            "Fail-all set. Setting CsvPath is_valid to False."
                        )
                        self.current_matcher.is_valid = False
                    if self._stop_all:
                        self.logger.warning("Stop-all set. Shutting down run.")
                        self.current_matcher.stopped = True
                        continue
                    if self._skip_all:
                        self.logger.warning("Skip-all set. Continuing to next.")
                        #
                        # all following CsvPaths must have their
                        # line_monitors incremented
                        #
                        self.current_matcher.track_line(line)
                        continue
                    if self._advance_all > 0:
                        logtxt = "Advance-all set. Setting advance. "
                        logtxt = f"{logtxt}CsvPath and its Matcher will handle the advancing."
                        self.logger.info(logtxt)
                        #
                        # CsvPath will handle advancing so we don't need to do
                        # anything, including track_line(line). we just need to
                        # see if we're setting advance or increasing it.
                        #
                        a = self.current_matcher.advance_count
                        if self._advance_all > a:
                            self.current_matcher.advance_count = self._advance_all
                        #
                        # all following CsvPaths must have their
                        # advance incremented -- with the advance not being simply
                        # additive, have to be mindful of any existing advance
                        # count!
                        #
                    if self.current_matcher.stopped:  # pylint: disable=R1724
                        continue

                    #
                    # allowing the match to happen regardless of keep
                    # because we may want side-effects or to have different
                    # results in different named-results, as well as the
                    # union
                    #
                    self.logger.debug(
                        "considering line with csvpath identified as: %s",
                        self.current_matcher.identity,
                    )
                    matched = False
                    self.current_matcher.track_line(line)
                    #
                    # re: W0212: treating _consider_line something like package private
                    #
                    matched = (
                        self.current_matcher._consider_line(  # pylint:disable=W0212
                            line
                        )
                    )
                    if self.current_matcher.stopped:
                        stopped_count.append(1)
                    if if_all_agree:
                        keep = keep and matched
                    else:
                        keep = keep or matched
                    #
                    # not doing continue if we have if_all_agree and not keep as we
                    # used to do allows individual results to have lines that in
                    # aggregate we do not keep.
                    #
                    if matched and collect:
                        line = self.current_matcher.limit_collection(line)
                        p[1].append(line)
            except Exception as ex:  # pylint: disable=W0718
                # ex.trace = traceback.format_exc()
                # ex.source = self
                self.error_manager.handle_error(source=self, msg=f"{ex}")
                if self.ecoms.do_i_raise():
                    for r in csvpath_objects:
                        result = r[1]
                        result.unmatched = r[0].unmatched
                        self.results_manager.save(result)
                    raise
            # we yield even if we stopped in this iteration.
            # caller needs to see what we stopped on.
            #
            # ! we only yield if keep is True
            #
            if keep:
                yield line
            if sum(stopped_count) == len(csvpath_objects):
                break
        results = []
        for r in csvpath_objects:
            result = r[1]
            results.append(result)
            result.unmatched = r[0].unmatched
            self.results_manager.save(result)

        #
        # run ends here
        #
        self.results_manager.complete_run(
            run_dir=results[0].run_dir, pathsname=pathsname, results=results
        )
        self.clear_run_coordination()

    def _load_csvpath_objects(
        self,
        *,
        paths: List[str],
        named_file: str,
        collect_when_not_matched=False,
        filename,
        pathsname,
        crt: str,
    ):
        csvpath_objects = []
        for path in paths:
            csvpath = self.csvpath()
            csvpath.collect_when_not_matched = collect_when_not_matched
            try:
                self._load_csvpath(
                    csvpath=csvpath,
                    path=path,
                    file=named_file,
                    filename=filename,
                    pathsname=pathsname,
                    by_line=True,
                    crt=crt,
                )
                if csvpath.data_from_preceding is True:
                    # this exception raise may be redundant, but I'm leaving it for now for good measure.
                    raise CsvPathsException(
                        "Csvpath identified as {csvpath.identity} is set to use preceding data, but CsvPaths's by_line methods do not permit that"
                    )
                csvpath_objects.append([csvpath, []])
            except Exception as ex:  # pylint: disable=W0718
                ex.trace = traceback.format_exc()
                ex.source = self
                # the error collector is the Results. it registers itself with
                # the csvpath as the error collector. not as straightforward but
                # effectively same as we do above
                self.error_manager.handle_error(source=self, msg=f"{ex}")
        return csvpath_objects

    def _prep_csvpath_results(self, *, csvpath_objects, filename, pathsname, crt: str):
        #
        # run starts here
        #
        self.results_manager.start_run(
            run_dir=crt, pathsname=pathsname, filename=filename
        )
        #
        #
        #
        for i, csvpath in enumerate(csvpath_objects):
            try:
                #
                # Result will set itself into its CsvPath as error collector
                # printer, etc.
                #
                result = Result(
                    csvpath=csvpath[0],
                    file_name=filename,
                    paths_name=pathsname,
                    lines=csvpath[1],
                    run_index=i,
                    run_time=self.current_run_time,
                    run_dir=crt,
                    by_line=True,
                )
                csvpath[1] = result
                #
                # the add has to come after _load_csvpath because we need the identity or index
                # to be stable and the identity is found in load, if it exists.
                #
                self.results_manager.add_named_result(result)
            except Exception as ex:  # pylint: disable=W0718
                """
                ex.trace = traceback.format_exc()
                ex.source = self
                ErrorHandler(csvpaths=self, error_collector=csvpath).handle_error(ex)
                """
                self.error_manager.handle_error(source=self, msg=f"{ex}")
                #
                # keep this comment for modelines avoidance
                #
