"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.TagManager = exports.TagType = void 0;
var TagType;
(function (TagType) {
    TagType["STANDARD"] = "StandardTag";
    TagType["AUTOSCALING_GROUP"] = "AutoScalingGroupTag";
    TagType["MAP"] = "StringToStringMap";
    TagType["KEY_VALUE"] = "KeyValue";
    TagType["NOT_TAGGABLE"] = "NotTaggable";
})(TagType = exports.TagType || (exports.TagType = {}));
/**
 * Standard tags are a list of { key, value } objects
 */
class StandardFormatter {
    parseTags(rosPropertyTags, priority) {
        if (!Array.isArray(rosPropertyTags)) {
            throw new Error(`Invalid tag input expected array of {key, value} have ${JSON.stringify(rosPropertyTags)}`);
        }
        const tags = [];
        for (const tag of rosPropertyTags) {
            if (tag.key === undefined || tag.value === undefined) {
                throw new Error(`Invalid tag input expected {key, value} have ${JSON.stringify(tag)}`);
            }
            // using interp to ensure Token is now string
            tags.push({
                key: `${tag.key}`,
                value: `${tag.value}`,
                priority,
            });
        }
        return tags;
    }
    formatTags(tags) {
        const rosTags = [];
        for (const tag of tags) {
            rosTags.push({
                key: tag.key,
                value: tag.value,
            });
        }
        return rosTags.length === 0 ? undefined : rosTags;
    }
}
/**
 * ASG tags are a list of { key, value, propagateAtLaunch } objects
 */
class AsgFormatter {
    parseTags(rosPropertyTags, priority) {
        const tags = [];
        if (!Array.isArray(rosPropertyTags)) {
            throw new Error(`Invalid tag input expected array of {key, value, propagateAtLaunch} have ${JSON.stringify(rosPropertyTags)}`);
        }
        for (const tag of rosPropertyTags) {
            if (tag.key === undefined ||
                tag.value === undefined ||
                tag.propagateAtLaunch === undefined) {
                throw new Error(`Invalid tag input expected {key, value, propagateAtLaunch} have ${JSON.stringify(tag)}`);
            }
            // using interp to ensure Token is now string
            tags.push({
                key: `${tag.key}`,
                value: `${tag.value}`,
                priority,
                applyToLaunchedInstances: !!tag.propagateAtLaunch,
            });
        }
        return tags;
    }
    formatTags(tags) {
        const rosTags = [];
        for (const tag of tags) {
            rosTags.push({
                key: tag.key,
                value: tag.value,
                propagateAtLaunch: tag.applyToLaunchedInstances !== false,
            });
        }
        return rosTags.length === 0 ? undefined : rosTags;
    }
}
/**
 * Some ROS constructs use a { key: value } map for tags
 */
class MapFormatter {
    parseTags(rosPropertyTags, priority) {
        const tags = [];
        if (Array.isArray(rosPropertyTags) || typeof rosPropertyTags !== "object") {
            throw new Error(`Invalid tag input expected map of {key: value} have ${JSON.stringify(rosPropertyTags)}`);
        }
        for (const [key, value] of Object.entries(rosPropertyTags)) {
            tags.push({
                key,
                value: `${value}`,
                priority,
            });
        }
        return tags;
    }
    formatTags(tags) {
        const rosTags = {};
        for (const tag of tags) {
            rosTags[`${tag.key}`] = `${tag.value}`;
        }
        return Object.keys(rosTags).length === 0 ? undefined : rosTags;
    }
}
/**
 * StackTags are of the format { Key: key, Value: value }
 */
class KeyValueFormatter {
    parseTags(keyValueTags, priority) {
        const tags = [];
        for (const key in keyValueTags) {
            if (keyValueTags.hasOwnProperty(key)) {
                const value = keyValueTags[key];
                tags.push({
                    key,
                    value,
                    priority,
                });
            }
        }
        return tags;
    }
    formatTags(unformattedTags) {
        const tags = [];
        unformattedTags.forEach((tag) => {
            tags.push({
                Key: tag.key,
                Value: tag.value,
            });
        });
        return tags;
    }
}
class NoFormat {
    parseTags(_rosPropertyTags) {
        return [];
    }
    formatTags(_tags) {
        return undefined;
    }
}
const TAG_FORMATTERS = {
    [TagType.AUTOSCALING_GROUP]: new AsgFormatter(),
    [TagType.STANDARD]: new StandardFormatter(),
    [TagType.MAP]: new MapFormatter(),
    [TagType.KEY_VALUE]: new KeyValueFormatter(),
    [TagType.NOT_TAGGABLE]: new NoFormat(),
};
/**
 * TagManager facilitates a common implementation of tagging for Constructs.
 */
class TagManager {
    constructor(tagType, resourceTypeName, tagStructure, options = {}) {
        this.tags = new Map();
        this.priorities = new Map();
        this.initialTagPriority = 50;
        this.resourceTypeName = resourceTypeName;
        this.tagFormatter = TAG_FORMATTERS[tagType];
        if (tagStructure !== undefined) {
            this._setTag(...this.tagFormatter.parseTags(tagStructure, this.initialTagPriority));
        }
        this.tagPropertyName = options.tagPropertyName || "tags";
    }
    /**
     * Check whether the given construct is Taggable
     */
    static isTaggable(construct) {
        return construct.tags !== undefined;
    }
    /**
     * Adds the specified tag to the array of tags
     *
     */
    setTag(key, value, priority = 0, applyToLaunchedInstances = true) {
        // This method mostly exists because we don't want to expose the 'Tag' type used (it will be confusing
        // to users).
        this._setTag({ key, value, priority, applyToLaunchedInstances });
    }
    /**
     * Removes the specified tag from the array if it exists
     *
     * @param key The tag to remove
     * @param priority The priority of the remove operation
     */
    removeTag(key, priority) {
        if (priority >= (this.priorities.get(key) || 0)) {
            this.tags.delete(key);
            this.priorities.set(key, priority);
        }
    }
    /**
     * Renders tags into the proper format based on TagType
     */
    renderTags() {
        const sortedTags = Array.from(this.tags.values()).sort((a, b) => a.key.localeCompare(b.key));
        let temp = this.tagFormatter.formatTags(sortedTags);
        return temp;
    }
    /**
     * Determine if the aspect applies here
     *
     * Looks at the include and exclude resourceTypeName arrays to determine if
     * the aspect applies here
     */
    applyTagAspectHere(include, exclude) {
        if (exclude &&
            exclude.length > 0 &&
            exclude.indexOf(this.resourceTypeName) !== -1) {
            return false;
        }
        if (include &&
            include.length > 0 &&
            include.indexOf(this.resourceTypeName) === -1) {
            return false;
        }
        return true;
    }
    /**
     * Returns true if there are any tags defined
     */
    hasTags() {
        return this.tags.size > 0;
    }
    _setTag(...tags) {
        for (const tag of tags) {
            if (tag.priority >= (this.priorities.get(tag.key) || 0)) {
                this.tags.set(tag.key, tag);
                this.priorities.set(tag.key, tag.priority);
            }
        }
    }
}
exports.TagManager = TagManager;
//# sourceMappingURL=data:application/json;base64,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