"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.NestedStack = void 0;
const construct_compat_1 = require("./construct-compat");
const stack_1 = require("./stack");
const token_1 = require("./token");
const lazy_1 = require("./lazy");
const ros_fn_1 = require("./ros-fn");
const ros_pseudo_1 = require("./ros-pseudo");
const ros_stack_1 = require("./ros-stack");
const ros_element_1 = require("./ros-element");
const NESTED_STACK_SYMBOL = Symbol.for("ros-cdk-core.NestedStack");
class NestedStack extends stack_1.Stack {
    constructor(scope, id, props = {}) {
        super(scope, id);
        let stackProps = {
            parameters: lazy_1.Lazy.anyValue({
                produce: () => Object.keys(this.parameters).length > 0 ? this.parameters : undefined,
            }),
            timeoutMins: props.timeout,
        };
        if (props.templateUrl) {
            stackProps.templateUrl = props.templateUrl;
        }
        else {
            stackProps.templateBody = lazy_1.Lazy.stringValue({
                produce: () => this.templateBody,
            });
        }
        const parentScope = new construct_compat_1.Construct(scope, id + ".NestedStack");
        this.resource = new ros_stack_1.RosStack(parentScope, `${id}`, stackProps);
        this._contextualStackName = this.contextualAttribute(ros_pseudo_1.RosPseudo.stackName, token_1.Token.asString(ros_fn_1.Fn.select(1, ros_fn_1.Fn.split("/", this.resource.ref))));
        this._contextualStackId = this.contextualAttribute(ros_pseudo_1.RosPseudo.stackId, this.resource.ref);
        Object.defineProperty(this, NESTED_STACK_SYMBOL, { value: true });
        this.parameters = props.parameters || {};
    }
    static isNestedStack(x) {
        return x != null && typeof x === "object" && NESTED_STACK_SYMBOL in x;
    }
    /**
     * @internal
     */
    _toRosTemplate() {
        const template = {};
        const elements = rosElements(this);
        if (elements.length === 0)
            return;
        const fragments = elements.map((e) => this.resolve(e._toRosTemplate()));
        for (const fragment of fragments) {
            merge(template, fragment);
        }
        this.templateBody = this.resolve(template);
        return this.templateBody;
    }
    get stackName() {
        return this._contextualStackName;
    }
    get stackId() {
        return this._contextualStackId;
    }
    setParameter(name, value) {
        this.parameters[name] = value;
    }
    getAtt(attributeName) {
        return token_1.Token.asString(this.resource.getAtt("Outputs." + attributeName));
    }
    contextualAttribute(innerValue, outerValue) {
        return token_1.Token.asString({
            resolve: (context) => {
                if (stack_1.Stack.of(context.scope) === this) {
                    return innerValue;
                }
                else {
                    return outerValue;
                }
            },
        });
    }
}
exports.NestedStack = NestedStack;
function rosElements(node, into = []) {
    if (ros_element_1.RosElement.isRosElement(node)) {
        into.push(node);
    }
    for (const child of node.node.children) {
        // Don't recurse into a substack
        if (stack_1.Stack.isStack(child)) {
            continue;
        }
        rosElements(child, into);
    }
    return into;
}
function merge(template, fragment) {
    for (const section of Object.keys(fragment)) {
        const src = fragment[section];
        // create top-level section if it doesn't exist
        const dest = template[section];
        if (!dest) {
            template[section] = src;
        }
        else {
            template[section] = mergeSection(section, dest, src);
        }
    }
}
function mergeSection(section, val1, val2) {
    switch (section) {
        case "Description":
            return `${val1}\n${val2}`;
        case "Resources":
        case "Conditions":
        case "Parameters":
        case "Outputs":
        case "Mappings":
        case "Metadata":
            return mergeObjectsWithoutDuplicates(section, val1, val2);
        default:
            throw new Error(`CDK doesn't know how to merge two instances of the template section '${section}' - ` +
                "please remove one of them from your code");
    }
}
function mergeObjectsWithoutDuplicates(section, dest, src) {
    if (typeof dest !== "object") {
        throw new Error(`Expecting ${JSON.stringify(dest)} to be an object`);
    }
    if (typeof src !== "object") {
        throw new Error(`Expecting ${JSON.stringify(src)} to be an object`);
    }
    // add all entities from source section to destination section
    for (const id of Object.keys(src)) {
        if (id in dest) {
            throw new Error(`section '${section}' already contains '${id}'`);
        }
        dest[id] = src[id];
    }
    return dest;
}
//# sourceMappingURL=data:application/json;base64,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