import gym
import numpy as np
from typing import List, Union, Tuple, Callable

from lightRaven.policy import PolicyBase, FAPolicy, TabularPolicy
from lightRaven.sampling import SamplingBase


def safety_test(obs_shape: int, act_shape: int, p_type: str, theta: np.ndarray, sampler: SamplingBase, ref_size: int,
                ci_ub: Callable, g_funcs: List[Callable], delta=0.05) -> Union[Tuple[float, str], Tuple[float, np.ndarray]]:
    """ Run safety test over the candidate policies.

    Parameters
    ----------
    obs_shape : int
        The observation space size of the environment
    act_shape : int
        The observation space size of the environment
    p_type : str
        Indicating policy type from 'fa' (functional approximation), 'tb' (tabular method)
    theta : np.ndarray
        A collections of candidate policies generated by the learning agent
        [NOTE: use a significance level `delta/n` to prevent Multiple Comparison Problem]
    sampler : SamplingMethod
        The sampling method for performance estimation [IS/PDIS]
    ref_size : int
        The size of the dataset in the safety test.
        Used in the concentration bound calculation to prevent producing a bound that is overly conservative
    g_funcs : List[Callable]
        A :obj:`list` of user-defined constraint functions for safety test.
    delta : float
        [Default =0.05] The significance level for the safety test to get a high confidence performance lower bound
        of a candidate policy
    """
    assert isinstance(sampler, SamplingBase)
    policy = None
    if p_type == 'fa':
        policy = FAPolicy(obs_shape, act_shape, theta)
    elif p_type == 'tb':
        policy = TabularPolicy(obs_shape, act_shape, theta)

    assert isinstance(policy, PolicyBase)
    sampler.load_eval_policy(policy)

    rewards = sampler.get_est()

    for g in g_funcs:
        if ci_ub(g(rewards), ref_size=ref_size, c=1, delta=delta) > 0:
            return -np.infty, "NSF"
    return np.average(rewards), theta
