import numpy as np
from scipy import stats, optimize


def _t_interval(sample: np.ndarray, ref_size, c, delta, mode) -> float:
    """
    A generic function wrapper for calculating concentration lower bounds

    Parameters
    ----------
    sample : np.ndarray
        Sample data
    ref_size : (int, optional)
        Size of the safety dataset
    c : float
        A hyperparameter used to scale the concentration interval generated by `ci_ub`.
        For student t, use `2` in candidate selection; Otherwise use `1`.
    delta : float
        The significance level.
    mode : str
        "one" for one-sided test; "two" for two-sided test

    Returns
    ----------
    float: The concentration range using Student's t-test given the sample and significance level

    :param sample: np.ndarray
    :param ref_size: int
    :param c: float
    :param delta: float
    :param mode: str
    :return: float
    """
    data_size = sample.shape[0]
    sample_mean = np.mean(sample)
    sample_variance = np.sum((sample - sample_mean) ** 2) / (data_size - 1)
    sample_std = np.sqrt(sample_variance)

    if mode == "one":
        t = stats.t.ppf(1 - delta, ref_size - 1)
        interval = t * sample_std / np.sqrt(ref_size)
    elif mode == "two":
        t = stats.t.ppf(1 - delta / 2, ref_size - 1)
        interval = t * sample_std / np.sqrt(ref_size)
    else:
        raise ValueError(f"mode can only be \"one\" or \"two\".")

    return c * interval


def _hoeffding_interval(sample: np.ndarray, ref_size, c, delta, mode) -> float:
    """
    A helper function calculates the concentration interval using Hoeffding inequality.
    No assumptions over the sampled distribution.

    Parameters
    ----------
    sample : np.ndarray
        Sample data
    ref_size : (int, optional)
        Size of the safety dataset
    c : float
        A hyperparameter used to scale the concentration interval generated by `ci_ub`.
        For student t, use `2` in candidate selection; Otherwise use `1`.
    delta : float
        The significance level.
    mode : str
        "one" for one-sided test; "two" for two-sided test

    Returns
    ----------
    float: The concentration range using Hoeffding's inequality given the sample and significance level

    :param sample: np.ndarray
    :param ref_size: int
    :param c: float
    :param delta: float
    :param mode: str
    :return: float
    """
    sample_max, sample_min = np.max(sample), np.min(sample)
    b = sample_max - sample_min

    if mode == "one":
        interval = b * np.sqrt(np.log(1 / delta) / (2 * ref_size))
    elif mode == "two":
        interval = b * np.sqrt(np.log(2 / delta) / (2 * ref_size))
    else:
        raise ValueError(f"mode can only be \"one\" or \"two\".")

    return c * interval


def _mpeb_interval(sample: np.ndarray, ref_size, c, delta, mode) -> float:
    """
    A helper function calculates the concentration interval using Hoeffding inequality.
    No assumptions over the sampled distribution.

    Parameters
    ----------
    sample : np.ndarray
        Sample data
    ref_size : (int, optional)
        Size of the safety dataset
    c : float
        A hyperparameter used to scale the concentration interval generated by `ci_ub`.
        For student t, use `2` in candidate selection; Otherwise use `1`.
    delta : float
        The significance level.
    mode : str
        "one" for one-sided test; "two" for two-sided test

    Returns
    ----------
    float: The concentration range using Maurer-Pontil Empirical Bernstein Bound given the sample and significance level

    :param sample: np.ndarray
    :param ref_size: int
    :param c: float
    :param delta: float
    :param mode: str
    :return: float
    """
    data_size = sample.shape[0]
    sample_max, sample_min = np.max(sample), np.min(sample)
    b = sample_max - sample_min

    diff_mat = np.repeat(sample.reshape(-1, 1), data_size, axis=1) - sample
    diff_mat = diff_mat ** 2

    if mode == "one":
        interval_1 = (7 * b * np.log(2 / delta)) / (3 * ref_size - 3)
        interval_2 = np.sqrt(
            (np.log(2 / delta) * np.sum(diff_mat)) / (ref_size - 1)
        ) / ref_size
    elif mode == "two":
        interval_1 = (7 * b * np.log(2 / (delta / 2))) / (3 * ref_size - 3)
        interval_2 = np.sqrt(
            2.0 * (np.log(2 / delta) * np.sum(diff_mat)) / (ref_size - 1)
        ) / ref_size
    else:
        raise ValueError(f"mode can only be \"one\" or \"two\".")

    return c * (interval_1 + interval_2)


def _mcma(sample: np.ndarray, ref_size, c, delta, mode, mc_size) -> float:
    """
    A function calculates the 1-delta percentile of the sample using Monte-Carlo M_alpha method.
    No assumptions over the sampled distribution.

    For more details, please check the paper:
        "A New Confidence Interval for the Mean of a Bounded Random Variable"
        https://arxiv.org/pdf/1905.06208.pdf

    Parameters
    ----------
    sample : np.ndarray
        The sample data
    ref_size : (int, optional)
        [NOT USED] Size of the safety dataset
    c : float
        [NOT USED] A hyperparameter used to scale the concentration interval generated by `ci_ub`.
    delta : float
        The significance level.
    mode : str
        [NOT USED] "one" for one-sided test; "two" for two-sided test
    mc_size : int
        The sample size for Monte Carlo estimation

    Returns
    ----------
    float: The percentile using Monte-Carlo M_alpha given the sample and significance level

    :param sample: np.ndarray
    :param ref_size: int
    :param c: float
    :param delta: float
    :param mode: str
    :param mc_size: int
    :return: float
    """

    mc_size = int(mc_size)
    n = sample.shape[0]
    z = np.sort(sample)
    s_prime = np.ones(n)
    s_prime[:-1] = z[1:]
    s = s_prime - z

    u = np.sort(np.random.uniform(0, 1, (mc_size, n)), axis=1)
    ms = 1 - (u @ s)
    ms = np.sort(ms.ravel())

    return ms[int(np.ceil((1-delta) * mc_size))]


def jf_t_lb(sample: np.ndarray, ref_size=None, c=None, delta=0.05, mode='one') -> float:
    """
    A function calculates the lower bound using student-t distribution.

    Parameters
    ----------
    sample : np.ndarray
        Sample data
    ref_size : (int, optional)
        Size of the safety dataset
    c : (float, optional)
        A hyperparameter used to scale the concentration interval generated by `ci_ub`.
        For student t, use `2` in candidate selection; Otherwise use `1`.
    delta : (float, optional)
        The significance level.
    mode : (str, optional)
        "one" for one-sided test; "two" for two-sided test

    Returns
    ----------
    float: The concentration lower bound using Student's t-test given the sample and significance level

    :param sample: np.ndarray
    :param ref_size: int
    :param c: float
    :param delta: float
    :param mode: str
    :return: float
    """
    reference_size = sample.shape[0] if ref_size is None else ref_size
    interval_correction = 1 if c is None else c

    sample_mean = np.mean(sample)
    interval = _t_interval(sample, reference_size, interval_correction, delta, mode)

    return sample_mean - interval


def jf_t_ub(sample: np.ndarray, ref_size=None, c=None, delta=0.05, mode='one') -> float:
    """
    A function calculates the lower bound using student-t distribution.

    Parameters
    ----------
    sample : np.ndarray
        Sample data
    ref_size : (int, optional)
        Size of the safety dataset
    c : (float, optional)
        A hyperparameter used to scale the concentration interval generated by `ci_ub`.
        For student t, use `2` in candidate selection; Otherwise use `1`.
    delta : (float, optional)
        The significance level.
    mode : (str, optional)
        "one" for one-sided test; "two" for two-sided test

    Returns
    ----------
    float: The concentration upper bound using Student's t-test given the sample and significance level

    :param sample: np.ndarray
    :param ref_size: int
    :param c: float
    :param delta: float
    :param mode: str
    :return: float
    """
    reference_size = sample.shape[0] if ref_size is None else ref_size
    interval_correction = 1 if c is None else c

    sample_mean = np.mean(sample)
    interval = _t_interval(sample, reference_size, interval_correction, delta, mode)

    return sample_mean + interval


def jf_hoeffding_lb(sample: np.ndarray, ref_size=None, c=None, delta=0.05, mode="one"):
    """
    A function calculates the lower bound using Hoeffding inequality.

    Parameters
    ----------
    sample : np.ndarray
        Sample data
    ref_size : (int, optional)
        Size of the safety dataset
    c : (float, optional)
        A hyperparameter used to scale the concentration interval generated by `ci_ub`.
        For student t, use `2` in candidate selection; Otherwise use `1`.
    delta : (float, optional)
        The significance level.
    mode : (str, optional)
        "one" for one-sided test; "two" for two-sided test

    Returns
    ----------
    float: The concentration lower bound using Hoeffding inequality given the sample and significance level

    :param sample: np.ndarray
    :param ref_size: int
    :param c: float
    :param delta: float
    :param mode: str
    :return: float
    """
    ref_size = sample.shape[0] if ref_size is None else ref_size
    c = 1 if c is None else c

    sample_mean = np.mean(sample)
    interval = _hoeffding_interval(sample, ref_size, c, delta, mode)

    return sample_mean - interval


def jf_hoeffding_ub(sample: np.ndarray, ref_size=None, c=None, delta=0.05, mode="one"):
    """
    A function calculates the lower bound using Hoeffding inequality.

    Parameters
    ----------
    sample : np.ndarray
        Sample data
    ref_size : (int, optional)
        Size of the safety dataset
    c : (float, optional)
        A hyperparameter used to scale the concentration interval generated by `ci_ub`.
        For student t, use `2` in candidate selection; Otherwise use `1`.
    delta : (float, optional)
        The significance level.
    mode : (str, optional)
        "one" for one-sided test; "two" for two-sided test

    Returns
    ----------
    float: The concentration upper bound using Hoeffding inequality given the sample and significance level

    :param sample: np.ndarray
    :param ref_size: int
    :param c: float
    :param delta: float
    :param mode: str
    :return: float
    """
    ref_size = sample.shape[0] if ref_size is None else ref_size
    c = 1 if c is None else c

    sample_mean = np.mean(sample)
    interval = _hoeffding_interval(sample, ref_size, c, delta, mode)

    return sample_mean + interval


def jf_mpeb_lb(sample: np.ndarray, ref_size=None, c=None, delta=0.05, mode="one"):
    """
    A  function calculates the lower bound using Maurer-Pontil Empirical Bernstein Bounds.

    Parameters
    ----------
    sample : np.ndarray
        Sample data
    ref_size : (int, optional)
        Size of the safety dataset
    c : (float, optional)
        A hyperparameter used to scale the concentration interval generated by `ci_ub`.
        For student t, use `2` in candidate selection; Otherwise use `1`.
    delta : (float, optional)
        The significance level.
    mode : (str, optional)
        "one" for one-sided test; "two" for two-sided test

    Returns
    ----------
    float: The concentration lower bound using Maurer-Pontil Empirical Bernstein Bounds
           given the sample and significance level

    :param sample: np.ndarray
    :param ref_size: int
    :param c: float
    :param delta: float
    :param mode: str
    :return: float
    """
    ref_size = sample.shape[0] if ref_size is None else ref_size
    c = 1 if c is None else c

    sample_mean = np.mean(sample)
    interval = _mpeb_interval(sample, ref_size, c, delta, mode)

    return sample_mean - interval


def jf_mpeb_ub(sample: np.ndarray, ref_size=None, c=None, delta=0.05, mode="one"):
    """
    A  function calculates the upper bound using Maurer-Pontil Empirical Bernstein Bounds.

    Parameters
    ----------
    sample : np.ndarray
        Sample data
    ref_size : (int, optional)
        Size of the safety dataset
    c : (float, optional)
        A hyperparameter used to scale the concentration interval generated by `ci_ub`.
        For student t, use `2` in candidate selection; Otherwise use `1`.
    delta : (float, optional)
        The significance level.
    mode : (str, optional)
        "one" for one-sided test; "two" for two-sided test

    Returns
    ----------
    float: The concentration upper bound using Maurer-Pontil Empirical Bernstein Bounds
           given the sample and significance level

    :param sample: np.ndarray
    :param ref_size: int
    :param c: float
    :param delta: float
    :param mode: str
    :return: float
    """
    ref_size = sample.shape[0] if ref_size is None else ref_size
    c = 1 if c is None else c

    sample_mean = np.mean(sample)
    interval = _mpeb_interval(sample, ref_size, c, delta, mode)

    return sample_mean + interval


def jf_mcma_lb(sample: np.ndarray, ref_size=None, c=None, delta=0.05, mode=None, mc_size=10000) -> float:
    """
    A function calculates the lower bound using Monte-Carlo M_alpha method.

    For more details, please check the paper:
        "A New Confidence Interval for the Mean of a Bounded Random Variable"
        https://arxiv.org/pdf/1905.06208.pdf

    Parameters
    ----------
    sample : np.ndarray
        Sample data
    ref_size : (int, optional)
        Size of the safety dataset
    c : (float, optional)
        A hyperparameter used to scale the concentration interval generated by `ci_ub`.
        For student t, use `2` in candidate selection; Otherwise use `1`.
    mode : (str, optional)
        "one" for one-sided test; "two" for two-sided test
    delta : (float, optional)
        The significance level.
    mc_size : int
        The sample size for Monte Carlo estimation

    Returns
    ----------
    float: The concentration lower bound using Monte-Carlo M_alpha given the sample and significance level

    :param sample: np.ndarray
    :param ref_size: int
    :param c: float
    :param mode: str
    :param delta: float
    :param mc_size: int
    :return: float
    """
    return _mcma(sample, ref_size, c, 1-delta, mode, mc_size)


def jf_mcma_ub(sample: np.ndarray, ref_size=None, c=None, mode=None, delta=0.05, mc_size=10000) -> float:
    """
    A function calculates the upper bound using Monte-Carlo M_alpha method.

    For more details, please check the paper:
        "A New Confidence Interval for the Mean of a Bounded Random Variable"
        https://arxiv.org/pdf/1905.06208.pdf

    Parameters
    ----------
    sample : np.ndarray
        Sample data
    ref_size : (int, optional)
        Size of the safety dataset
    c : (float, optional)
        A hyperparameter used to scale the concentration interval generated by `ci_ub`.
        For student t, use `2` in candidate selection; Otherwise use `1`.
    mode : (str, optional)
        "one" for one-sided test; "two" for two-sided test
    delta : (float, optional)
        The significance level.
    mc_size : int
        The sample size for Monte Carlo estimation

    Returns
    ----------
    float: The concentration upper bound using Monte-Carlo M_alpha given the sample and significance level

    :param sample: np.ndarray
    :param ref_size: int
    :param c: float
    :param mode: str
    :param delta: float
    :param mc_size: int
    :return: float
    """
    return _mcma(sample, ref_size, c, delta, mode, mc_size)


if __name__ == "__main__":
    # pop = np.random.randn(100)
    pop = np.random.geometric(0.35, 100)
    sample = np.random.choice(pop, 20)
    sample_prime = (sample - sample.min()) / (sample.max() - sample.min())
    s_sum = sample_prime.sum()
    s_mean = sample_prime.mean()
    s_squared = 0
    for i in range(sample_prime.size):
        s_squared += sample_prime[i] * sample_prime[i]
    print(f"{s_sum} {s_mean} {s_squared}")
    print(f"mean: {sample.mean()}; size: {sample.size}, b: {sample.max()}, a: {sample.min()}")
    print(f"    MC Ma CI: [{jf_mcma_lb(sample)}, {jf_mcma_ub(sample)}]")
    print(f"Student t CI: [{jf_t_lb(sample, c=2)}, {jf_t_ub(sample, c=2)}]")
    print(f"Hoeffding CI: [{jf_hoeffding_lb(sample, c=1)}, {jf_hoeffding_ub(sample, c=1)}]")
    print(f"     MPeB CI: [{jf_mpeb_lb(sample, c=1)}, {jf_mpeb_ub(sample, c=1)}]")
    # print(f"  extMPeB CI: [{extMPeB_lb(sample, correction=1)}, {extMPeB_ub(sample, correction=1)}]")