"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.invokeAspects = exports.ModelSynthesizer = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const fs = require("fs-extra");
const annotations_1 = require("./annotations");
const aspect_1 = require("./aspect");
const yaml_1 = require("./private/yaml");
class ModelSynthesizer {
    constructor(model, continueOnErrorAnnotations = false) {
        this.model = model;
        this.continueOnErrorAnnotations = continueOnErrorAnnotations;
    }
    synthesize(session) {
        invokeAspects(this.model);
        if (!session.skipValidation) {
            const errors = this.model.node
                .findAll()
                .map((node) => node.node
                .validate()
                .map((error) => ({ message: error, source: node })))
                .reduce((prev, curr) => [...prev, ...curr], []);
            if (errors.length > 0) {
                const errorList = errors
                    .map((e) => `[${e.source.node.path}] ${e.message}`)
                    .join("\n  ");
                throw new Error(`Validation failed with the following errors:\n  ${errorList}`);
            }
            const manifest = session.manifest;
            const modelManifest = manifest.forModel(this.model);
            const workingDirectory = path.join(session.outdir, modelManifest.workingDirectory);
            if (!fs.existsSync(workingDirectory))
                fs.mkdirSync(workingDirectory);
            // collect Annotations into Manifest
            const annotations = this.model.node
                .findAll()
                .map((node) => ({
                node,
                metadatas: node.node.metadata.filter(isAnnotationMetadata),
            }))
                .map(({ node, metadatas }) => metadatas.map((metadata) => ({
                constructPath: node.node.path,
                level: metadata.type,
                message: metadata.data,
                stacktrace: metadata.trace,
            })))
                .reduce((list, metadatas) => [...list, ...metadatas], []); // Array.flat()
            // it is readonly but this is the place where we are allowed to write to it
            modelManifest.annotations = annotations;
            // abort if one or more error annotations have been encountered
            if (!this.continueOnErrorAnnotations &&
                annotations.some(isErrorAnnotation)) {
                throw new Error(`Encountered Annotations with level "ERROR":\n${annotations
                    .filter(isErrorAnnotation)
                    .map((a) => `[${a.constructPath}] ${a.message}`)
                    .join("\n")}`);
            }
            const threagile = this.model._toThreagile();
            yaml_1.Yaml.save(path.join(session.outdir, modelManifest.synthesizedModelPath), [
                threagile,
            ]);
        }
    }
}
exports.ModelSynthesizer = ModelSynthesizer;
_a = JSII_RTTI_SYMBOL_1;
ModelSynthesizer[_a] = { fqn: "cdktg.ModelSynthesizer", version: "0.0.23" };
const invokedByPath = {};
/**
 * Invoke aspects on the given construct tree.
 *
 * originally from https://github.com/aws/aws-cdk/blob/dcae3eead0dbf9acb1ed80ba95bb104c64cb1bd7/packages/@aws-cdk/core/lib/private/synthesis.ts#L99-L137
 */
function invokeAspects(root) {
    let nestedAspectWarning = false;
    recurse(root, []);
    function recurse(construct, inheritedAspects) {
        const node = construct.node;
        const aspects = aspect_1.Aspects.of(construct);
        const allAspectsHere = [...(inheritedAspects ?? []), ...aspects.all];
        const nodeAspectsCount = aspects.all.length;
        for (const aspect of allAspectsHere) {
            let invoked = invokedByPath[node.path];
            if (!invoked) {
                invoked = invokedByPath[node.path] = [];
            }
            if (invoked.includes(aspect)) {
                continue;
            }
            aspect.visit(construct);
            // if an aspect was added to the node while invoking another aspect it will not be invoked, emit a warning
            // the `nestedAspectWarning` flag is used to prevent the warning from being emitted for every child
            if (!nestedAspectWarning && nodeAspectsCount !== aspects.all.length) {
                annotations_1.Annotations.of(construct).addWarning("We detected an Aspect was added via another Aspect, and will not be applied");
                nestedAspectWarning = true;
            }
            // mark as invoked for this node
            invoked.push(aspect);
        }
        for (const child of construct.node.children) {
            recurse(child, allAspectsHere);
        }
    }
}
exports.invokeAspects = invokeAspects;
const annotationMetadataEntryTypes = [
    annotations_1.AnnotationMetadataEntryType.INFO,
    annotations_1.AnnotationMetadataEntryType.WARN,
    annotations_1.AnnotationMetadataEntryType.ERROR,
];
function isAnnotationMetadata(metadata) {
    return annotationMetadataEntryTypes.includes(metadata.type);
}
function isErrorAnnotation(annotation) {
    return annotation.level === annotations_1.AnnotationMetadataEntryType.ERROR;
}
//# sourceMappingURL=data:application/json;base64,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