from __future__ import annotations


import logging
import os
import pathlib
import random
import warnings
from email.header import Header
from typing import Iterable

from .elements import *
from .templates import *

logger = logging.getLogger(__name__)


class Document:
    """
    A document represents a markdown file. Documents store
    a collection of blocks which are appended with new lines
    between to generate the markdown document. Document methods
    are intended to provided convenience when generating a
    markdown file. However, the functionality is not exhaustive.
    To get the full range of markdown functionality, you can
    take advantage of the :func:`add_block` function to provide
    custom markdown block.

    :param str name: 
        the file name of the document without the extension
        
        .. deprecated:: 0.13.0
            parameter is now optional and will be removed in 1.0.0
    """

    def __init__(self, name: str = None) -> None:
        self._name: str = name
        self._ext: str = ".md"
        self._contents: list[Block] = list()
        if name:
            warnings.warn(
                "name parameter has been deprecated as of 0.13.0", 
                DeprecationWarning
            )

    def __str__(self):
        """
        Renders the markdown document from a list of blocks.

        :return: the document as a markdown string
        """
        return "\n\n".join(str(block) for block in self._contents)

    def render(self) -> str:
        """
        Renders the markdown document from a list of blocks.
        
        .. deprecated:: 0.14.0
            removed in favor of __str__

        :return: the document as a markdown string
        """
        return str(self)

    def check_for_errors(self) -> None:
        """
        A convenience method which can be used to verify the
        integrity of the document. Results will be printed to
        standard out.

        .. versionadded:: 0.2.0
        """
        verification = Verification()
        for block in self._contents:
            verification.absorb(block.verify())
        if verification.passes_inspection():
            print("No errors found!")
        else:
            print(verification)

    def add_element(self, element: Element) -> Element:
        """
        A generic function for appending elements to the document.
        Use this function when you want a little more control over
        what the output looks like.

        .. code-block:: Python

            doc.add_element(Heading(Inline("Python is Cool!"), 2))

        .. versionchanged:: 0.2.0
            Returns Element generated by this method instead of None.
        .. deprecated:: 0.14.0
            replaced in favor of :func:`add_block`

        :param Element element: a markdown object (e.g., Table, Heading, etc.)
        :return: the Element added to this Document
        """
        warnings.warn("use add_block instead", DeprecationWarning)
        assert isinstance(element, Element)
        self._contents.append(element)
        logger.debug(f"Added element to document\n{element}")
        return element
    
    def add_block(self, block: Block) -> Block:
        """
        A generic function for appending blocks to the document.
        Use this function when you want a little more control over
        what the output looks like.

        .. code-block:: Python

            doc.add_block(Heading("Python is Cool!"), 2))

        .. versionadded:: 0.14.0
           replaces :func:`add_element`

        :param Block block: a markdown block (e.g., Table, Heading, etc.)
        :return: the Block added to this Document
        """
        assert isinstance(block, Block)
        self._contents.append(block)
        logger.debug(f"Added block to document\n{block}")
        return block
    
    def add_raw(self, text: str) -> Raw:
        """
        A convenience method which adds text as-is to the document:
        
        .. code-block:: Python

            doc.add_raw("X: 5\\nY: 4\\nZ: 3")

        :param str text: some text 
        :return: the Raw block added to this Document
        """
        raw = Raw(text)
        self._contents.append(raw)
        return raw

    def add_heading(self, text: str, level: int = 1) -> Heading:
        """
        A convenience method which adds a simple heading to the document:

        .. code-block:: Python

            doc.add_heading("Welcome to SnakeMD!")

        .. versionadded:: 0.13.0
           replaces :func:`add_header`

        :param str text: the text for the heading
        :param int level: the level of the heading from 1 to 6
        :return: the Heading added to this Document
        """
        assert 1 <= level <= 6
        heading = Heading(Inline(text), level)
        self._contents.append(heading)
        logger.debug(f"Added heading to document\n{heading}")
        return heading
    
    def add_header(self, text: str, level: int = 1) -> Header:
        """
        A convenience method which adds a simple header to the document:

        .. code-block:: Python

            doc.add_header("Welcome to SnakeMD!")

        .. versionchanged:: 0.2.0
           returns Header generated by this method instead of None.
           
        .. deprecated:: 0.13.0
            use :func:`add_heading` instead

        :param str text: the text for the header
        :param int level: the level of the header from 1 to 6
        :return: the Header added to this Document
        """
        warnings.warn(
            "add_header has been deprecated as of 0.13.0 and replaced with add_heading", 
            DeprecationWarning
        )
        assert 1 <= level <= 6
        header = Header(Inline(text), level)
        self._contents.append(header)
        logger.debug(f"Added header to document\n{header}")
        return header

    def add_paragraph(self, text: str) -> Paragraph:
        """
        A convenience method which adds a simple paragraph of text to the document:

        .. code-block:: Python

            doc.add_paragraph("Mitochondria is the powerhouse of the cell.")

        .. versionchanged:: 0.2.0
           Returns Paragraph generated by this method instead of None.

        :param str text: any arbitrary text
        :return: the Paragraph added to this Document
        """
        paragraph = Paragraph([Inline(text)])
        self._contents.append(paragraph)
        logger.debug(f"Added paragraph to document\n{paragraph}")
        return paragraph

    def add_ordered_list(self, items: Iterable[str]) -> MDList:
        """
        A convenience method which adds a simple ordered list to the document:

        .. code-block:: Python

            doc.add_ordered_list(["Goku", "Piccolo", "Vegeta"])

        .. versionchanged:: 0.2.0
           Returns MDList generated by this method instead of None.

        :param Iterable[str] items: a "list" of strings
        :return: the MDList added to this Document
        """
        md_list = MDList([Inline(item) for item in items], ordered=True)
        self._contents.append(md_list)
        logger.debug(f"Added ordered list to document\n{md_list}")
        return md_list

    def add_unordered_list(self, items: Iterable[str]) -> MDList:
        """
        A convenience method which adds a simple unordered list to the document.

        .. code-block:: Python

            doc.add_unordered_list(["Deku", "Bakugo", "Kirishima"])

        .. versionchanged:: 0.2.0
           Returns MDList generated by this method instead of None.

        :param Iterable[str] items: a "list" of strings
        :return: the MDList added to this Document
        """
        md_list = MDList([Inline(item) for item in items])
        self._contents.append(md_list)
        logger.debug(f"Added unordered list to document\n{md_list}")
        return md_list

    def add_checklist(self, items: Iterable[str]) -> MDList:
        """
        A convenience method which adds a simple checklist to the document.

        .. code-block:: Python

            doc.add_checklist(["Okabe", "Mayuri", "Kurisu"])

        .. versionadded:: 0.10.0

        :param Iterable[str] items: a "list" of strings
        :return: the MDCheckList added to this Document
        """
        md_checklist = MDList([Inline(item) for item in items], checked=False)
        self._contents.append(md_checklist)
        logger.debug(f"Added checklist to document\n{md_checklist}")
        return md_checklist

    def add_table(
        self,
        header: Iterable[str],
        data: Iterable[Iterable[str]],
        align: Iterable[Table.Align] = None,
        indent: int = 0
    ) -> Table:
        """
        A convenience method which adds a simple table to the document:

        .. code-block:: Python

            doc.add_table(
                ["Place", "Name"],
                [
                    ["1st", "Robert"],
                    ["2nd", "Rae"]
                ],
                [snakemd.Table.Align.CENTER, snakemd.Table.Align.RIGHT],
                0
            )

        .. versionchanged:: 0.2.0
            Returns Table generated by this method instead of None.
        .. versionchanged:: 0.4.0
            Added optional alignment parameter
        .. versionchanged:: 0.11.0
            Added optional indentation parameter

        :param Iterable[str] header: a "list" of strings
        :param Iterable[Iterable[str]] data: a "list" of "lists" of strings
        :param Iterable[Table.Align] align: a "list" of column alignment values;
            defaults to None
        :param int indent: indent size for the whole table
        :return: the Table added to this Document
        """
        header = [Paragraph([text]) for text in header]
        data = [[Paragraph([item]) for item in row] for row in data]
        table = Table(header, data, align, indent)
        self._contents.append(table)
        logger.debug(f"Added table to document\n{table}")
        return table

    def add_code(self, code: str, lang: str = "generic") -> Code:
        """
        A convenience method which adds a code block to the document:

        .. code-block:: Python

            doc.add_code("x = 5")

        .. versionchanged:: 0.2.0
            returns Paragraph generated by this method instead of None
        .. versionchanged:: 0.15.0
            returns Code block generated by this method instead of Paragraph

        :param str code: a preformatted code string
        :param str lang: the language for syntax highlighting
        :return: the Code block added to this Document
        """
        code_block = Code(code, lang=lang)
        self._contents.append(code_block)
        logger.debug(f"Added code block to document\n{code_block}")
        return code_block

    def add_quote(self, text: str) -> Paragraph:
        """
        A convenience method which adds a blockquote to the document:

        .. code-block:: Python

            doc.add_quote("Welcome to the Internet!")

        .. versionchanged:: 0.2.0
           Returns Paragraph generated by this method instead of None.

        :param str text: the text to be quoted
        :return: the Paragraph added to this Document
        """
        paragraph = Paragraph([Inline(text)], quote=True)
        self._contents.append(paragraph)
        logger.debug(f"Added code block to document\n{paragraph}")
        return paragraph

    def add_horizontal_rule(self) -> HorizontalRule:
        """
        A convenience method which adds a horizontal rule to the document:

        .. code-block:: Python

            doc.add_horizontal_rule()

        .. versionadded:: 0.2.0

        :return: the HorizontalRule added to this Document
        """
        hr = HorizontalRule()
        self._contents.append(hr)
        logger.debug(f"Added code block to document\n{hr}")
        return hr

    def add_table_of_contents(self, levels: range = range(2, 3)) -> TableOfContents:
        """
        A convenience method which creates a table of contents. This function
        can be called where you want to add a table of contents to your
        document. The table itself is lazy loaded, so it always captures
        all of the heading blocks regardless of where the table of contents
        is added to the document.

        .. code-block:: Python

            doc.add_table_of_contents()

        .. versionchanged:: 0.2.0
           Fixed a bug where table of contents could only be rendered once.

        .. versionchanged:: 0.8.0
           Added optional levels parameter

        :param range levels: a range of heading levels to be included in the table of contents
        :return: the TableOfContents added to this Document
        """
        toc = TableOfContents(self, levels=levels)
        self._contents.append(toc)
        logger.debug(f"Added code block to document (unable to render until file is complete)")
        return toc

    def scramble(self) -> None:
        """
        A silly method which mixes all of the blocks in this document in
        a random order.
        """
        random.shuffle(self._contents)
        logger.debug(f"Scrambled document")

    def output_page(self, dump_dir: str = "", encoding: str = "utf-8") -> None:
        """
        Generates the markdown file. Assumes UTF-8 encoding.
        
        .. deprecated:: 0.13.0
            Use :func:`dump` instead

        :param str dump_dir: the path to where you want to dump the file
        :param str encoding: the encoding to use
        """
        warnings.warn(
            "output_page has been deprecated as of 0.13.0 and replaced with dump", 
            DeprecationWarning
        )
        pathlib.Path(dump_dir).mkdir(parents=True, exist_ok=True)
        output_file = open(
            os.path.join(dump_dir, self._get_file_name()), 
            "w+", 
            encoding=encoding
        )
        output_file.write(str(self))
        output_file.close()

    def _get_file_name(self) -> str:
        """
        A helper function for generating the file name.
        """
        file_name = f"{'-'.join(self._name.split())}{self._ext}"
        return file_name
    
    def dump(self, name: str, dir: str | os.PathLike = "", ext: str = "md", encoding: str = "utf-8") -> None:
        """
        Outputs the markdown document to a file. This method assumes the output directory
        is the current working directory. Any alternative directory provided will be
        made if it does not already exist. This method also assumes a file extension of md
        and a file encoding of utf-8, all of which are configurable through the method
        parameters.
        
        .. code-block:: Python
        
            doc.dump("README")
            
        .. versionadded:: 0.13.0
            Replaces the :func:`output_page` method
            
        :param str name: the name of the markdown file to output without the file extension
        :param str | os.PathLike dir: the output directory for the markdown file; defaults to ""
        :param str ext: the output file extension; defaults to "md"
        :param str encoding: the encoding to use; defaults to utf-8
        """
        pathlib.Path(dir).mkdir(parents=True, exist_ok=True)
        with open(
            os.path.join(dir, f"{name}.{ext}"),
            "w+",
            encoding=encoding
        ) as output_file:
            output_file.write(str(self))
