"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const cdk = require("@aws-cdk/core");
const secrets = require("@aws-cdk/aws-secretsmanager");
const s3 = require("@aws-cdk/aws-s3");
const index_1 = require("../lib/index");
describe("DatadogIntegrationStack", () => {
    let stack, secret;
    beforeEach(() => {
        stack = new cdk.Stack();
        secret = new secrets.Secret(stack, "Secret");
    });
    describe("with default parameters", () => {
        it("creates the macro policy stack", () => {
            const datadogStack = new index_1.DatadogIntegrationStack(stack, "DatadogStack", {
                apiKey: secret,
                externalId: "not-a-real-id",
            });
            assert_1.expect(datadogStack).to(assert_1.haveResource("AWS::CloudFormation::Stack", {
                TemplateURL: "https://datadog-cloudformation-template.s3.amazonaws.com/aws/datadog_policy_macro.yaml",
            }));
        });
        it("creates the integration role", () => {
            const datadogStack = new index_1.DatadogIntegrationStack(stack, "DatadogStack", {
                apiKey: secret,
                externalId: "not-a-real-id",
            });
            assert_1.expect(datadogStack).to(assert_1.haveResource("AWS::CloudFormation::Stack", {
                TemplateURL: "https://datadog-cloudformation-template.s3.amazonaws.com/aws/datadog_integration_role.yaml",
            }));
        });
        it("passes the expected default params to the integration role stack", () => {
            const datadogStack = new index_1.DatadogIntegrationStack(stack, "DatadogStack", {
                apiKey: secret,
                externalId: "not-a-real-id",
            });
            assert_1.expect(datadogStack).to(assert_1.haveResource("AWS::CloudFormation::Stack", {
                TemplateURL: "https://datadog-cloudformation-template.s3.amazonaws.com/aws/datadog_integration_role.yaml",
                Parameters: {
                    ExternalId: "not-a-real-id",
                    Permissions: "Full",
                    IAMRoleName: "DatadogIntegrationRole",
                    LogArchives: "",
                    CloudTrails: "",
                    DdAWSAccountId: "464622532012",
                },
            }));
        });
        it("creates the forwarder against the latest YML", () => {
            const datadogStack = new index_1.DatadogIntegrationStack(stack, "DatadogStack", {
                apiKey: secret,
                externalId: "not-a-real-id",
            });
            assert_1.expect(datadogStack).to(assert_1.haveResource("AWS::CloudFormation::Stack", {
                TemplateURL: "https://datadog-cloudformation-template.s3.amazonaws.com/aws/forwarder/latest.yaml",
            }));
        });
        it("passes the expected default params to the forwarder stack", () => {
            const datadogStack = new index_1.DatadogIntegrationStack(stack, "DatadogStack", {
                apiKey: secret,
                externalId: "not-a-real-id",
            });
            assert_1.expect(datadogStack).to(assert_1.haveResource("AWS::CloudFormation::Stack", {
                TemplateURL: "https://datadog-cloudformation-template.s3.amazonaws.com/aws/forwarder/latest.yaml",
                Parameters: {
                    DdApiKey: "USE_ARN",
                    DdApiKeySecretArn: {
                        "Fn::ImportValue": "Stack:OutputRefSecretA720EF05",
                    },
                    DdSite: "datadoghq.com",
                    FunctionName: "DatadogForwarder",
                },
            }));
        });
    });
    describe("integration role customization", () => {
        it("allows customizing the role name", () => {
            const datadogStack = new index_1.DatadogIntegrationStack(stack, "DatadogStack", {
                apiKey: secret,
                externalId: "not-a-real-id",
                iamRoleName: "MyRoleName",
            });
            assert_1.expect(datadogStack).to(assert_1.haveResource("AWS::CloudFormation::Stack", {
                TemplateURL: "https://datadog-cloudformation-template.s3.amazonaws.com/aws/datadog_integration_role.yaml",
                Parameters: {
                    ExternalId: "not-a-real-id",
                    Permissions: "Full",
                    IAMRoleName: "MyRoleName",
                    LogArchives: "",
                    CloudTrails: "",
                    DdAWSAccountId: "464622532012",
                },
            }));
        });
        it("allows customizing the permissions level", () => {
            const datadogStack = new index_1.DatadogIntegrationStack(stack, "DatadogStack", {
                apiKey: secret,
                externalId: "not-a-real-id",
                permissions: "Core",
            });
            assert_1.expect(datadogStack).to(assert_1.haveResource("AWS::CloudFormation::Stack", {
                TemplateURL: "https://datadog-cloudformation-template.s3.amazonaws.com/aws/datadog_integration_role.yaml",
                Parameters: {
                    ExternalId: "not-a-real-id",
                    Permissions: "Core",
                    IAMRoleName: "DatadogIntegrationRole",
                    LogArchives: "",
                    CloudTrails: "",
                    DdAWSAccountId: "464622532012",
                },
            }));
        });
        it("allows passing a single log archive bucket", () => {
            const bucket = new s3.Bucket(stack, "Bucket");
            const datadogStack = new index_1.DatadogIntegrationStack(stack, "DatadogStack", {
                apiKey: secret,
                externalId: "not-a-real-id",
                logArchives: [bucket],
            });
            assert_1.expect(datadogStack).to(assert_1.haveResource("AWS::CloudFormation::Stack", {
                TemplateURL: "https://datadog-cloudformation-template.s3.amazonaws.com/aws/datadog_integration_role.yaml",
                Parameters: {
                    ExternalId: "not-a-real-id",
                    Permissions: "Full",
                    IAMRoleName: "DatadogIntegrationRole",
                    LogArchives: {
                        "Fn::ImportValue": "Stack:OutputRefBucket83908E77",
                    },
                    CloudTrails: "",
                    DdAWSAccountId: "464622532012",
                },
            }));
        });
        it("allows passing multiple log archive buckets", () => {
            const bucket1 = new s3.Bucket(stack, "Bucket1");
            const bucket2 = new s3.Bucket(stack, "Bucket2");
            const datadogStack = new index_1.DatadogIntegrationStack(stack, "DatadogStack", {
                apiKey: secret,
                externalId: "not-a-real-id",
                logArchives: [bucket1, bucket2],
            });
            assert_1.expect(datadogStack).to(assert_1.haveResource("AWS::CloudFormation::Stack", {
                TemplateURL: "https://datadog-cloudformation-template.s3.amazonaws.com/aws/datadog_integration_role.yaml",
                Parameters: {
                    ExternalId: "not-a-real-id",
                    Permissions: "Full",
                    IAMRoleName: "DatadogIntegrationRole",
                    LogArchives: {
                        // the sub-assert
                        "Fn::Join": [
                            "",
                            [
                                {
                                    "Fn::ImportValue": "Stack:OutputRefBucket12520700A",
                                },
                                ",",
                                {
                                    "Fn::ImportValue": "Stack:OutputRefBucket25524B414",
                                },
                            ],
                        ],
                    },
                    CloudTrails: "",
                    DdAWSAccountId: "464622532012",
                },
            }));
        });
        it("allows passing a single cloudtrail bucket", () => {
            const bucket = new s3.Bucket(stack, "Bucket");
            const datadogStack = new index_1.DatadogIntegrationStack(stack, "DatadogStack", {
                apiKey: secret,
                externalId: "not-a-real-id",
                cloudTrails: [bucket],
            });
            assert_1.expect(datadogStack).to(assert_1.haveResource("AWS::CloudFormation::Stack", {
                TemplateURL: "https://datadog-cloudformation-template.s3.amazonaws.com/aws/datadog_integration_role.yaml",
                Parameters: {
                    ExternalId: "not-a-real-id",
                    Permissions: "Full",
                    IAMRoleName: "DatadogIntegrationRole",
                    LogArchives: "",
                    CloudTrails: {
                        "Fn::ImportValue": "Stack:OutputRefBucket83908E77",
                    },
                    DdAWSAccountId: "464622532012",
                },
            }));
        });
        it("allows passing multiple cloudtrail buckets", () => {
            const bucket1 = new s3.Bucket(stack, "Bucket1");
            const bucket2 = new s3.Bucket(stack, "Bucket2");
            const datadogStack = new index_1.DatadogIntegrationStack(stack, "DatadogStack", {
                apiKey: secret,
                externalId: "not-a-real-id",
                cloudTrails: [bucket1, bucket2],
            });
            assert_1.expect(datadogStack).to(assert_1.haveResource("AWS::CloudFormation::Stack", {
                TemplateURL: "https://datadog-cloudformation-template.s3.amazonaws.com/aws/datadog_integration_role.yaml",
                Parameters: {
                    ExternalId: "not-a-real-id",
                    Permissions: "Full",
                    IAMRoleName: "DatadogIntegrationRole",
                    LogArchives: "",
                    CloudTrails: {
                        // the sub-assert
                        "Fn::Join": [
                            "",
                            [
                                {
                                    "Fn::ImportValue": "Stack:OutputRefBucket12520700A",
                                },
                                ",",
                                {
                                    "Fn::ImportValue": "Stack:OutputRefBucket25524B414",
                                },
                            ],
                        ],
                    },
                    DdAWSAccountId: "464622532012",
                },
            }));
        });
        it("allows passing additional, arbitrary parameters", () => {
            const datadogStack = new index_1.DatadogIntegrationStack(stack, "DatadogStack", {
                apiKey: secret,
                externalId: "not-a-real-id",
                additionalIntegrationRoleParams: {
                    Foo: "bar",
                },
            });
            assert_1.expect(datadogStack).to(assert_1.haveResource("AWS::CloudFormation::Stack", {
                TemplateURL: "https://datadog-cloudformation-template.s3.amazonaws.com/aws/datadog_integration_role.yaml",
                Parameters: {
                    ExternalId: "not-a-real-id",
                    Permissions: "Full",
                    IAMRoleName: "DatadogIntegrationRole",
                    LogArchives: "",
                    CloudTrails: "",
                    DdAWSAccountId: "464622532012",
                    Foo: "bar",
                },
            }));
        });
        it("allows additional parameters to override parameters set other props", () => {
            const datadogStack = new index_1.DatadogIntegrationStack(stack, "DatadogStack", {
                apiKey: secret,
                externalId: "not-a-real-id",
                additionalIntegrationRoleParams: {
                    Permissions: "SomeNewThing",
                },
            });
            assert_1.expect(datadogStack).to(assert_1.haveResource("AWS::CloudFormation::Stack", {
                TemplateURL: "https://datadog-cloudformation-template.s3.amazonaws.com/aws/datadog_integration_role.yaml",
                Parameters: {
                    ExternalId: "not-a-real-id",
                    Permissions: "SomeNewThing",
                    IAMRoleName: "DatadogIntegrationRole",
                    LogArchives: "",
                    CloudTrails: "",
                    DdAWSAccountId: "464622532012",
                },
            }));
        });
    });
    describe("policy macro customization", () => {
        it("is not present when disabled", () => {
            const datadogStack = new index_1.DatadogIntegrationStack(stack, "DatadogStack", {
                apiKey: secret,
                externalId: "not-a-real-id",
                installDatadogPolicyMacro: false,
            });
            assert_1.expect(datadogStack).notTo(assert_1.haveResource("AWS::CloudFormation::Stack", {
                TemplateURL: "https://datadog-cloudformation-template.s3.amazonaws.com/aws/datadog_policy_macro.yaml",
            }));
        });
    });
    describe("forwarder stack customization", () => {
        it("allows passing a custom datadog collection site", () => {
            const datadogStack = new index_1.DatadogIntegrationStack(stack, "DatadogStack", {
                apiKey: secret,
                externalId: "not-a-real-id",
                site: "datadoghq.eu",
            });
            assert_1.expect(datadogStack).to(assert_1.haveResource("AWS::CloudFormation::Stack", {
                TemplateURL: "https://datadog-cloudformation-template.s3.amazonaws.com/aws/forwarder/latest.yaml",
                Parameters: {
                    DdApiKey: "USE_ARN",
                    DdApiKeySecretArn: {
                        "Fn::ImportValue": "Stack:OutputRefSecretA720EF05",
                    },
                    DdSite: "datadoghq.eu",
                    FunctionName: "DatadogForwarder",
                },
            }));
        });
        it("allows customizing the function name", () => {
            const datadogStack = new index_1.DatadogIntegrationStack(stack, "DatadogStack", {
                apiKey: secret,
                externalId: "not-a-real-id",
                forwarderName: "MyForwarder",
            });
            assert_1.expect(datadogStack).to(assert_1.haveResource("AWS::CloudFormation::Stack", {
                TemplateURL: "https://datadog-cloudformation-template.s3.amazonaws.com/aws/forwarder/latest.yaml",
                Parameters: {
                    DdApiKey: "USE_ARN",
                    DdApiKeySecretArn: {
                        "Fn::ImportValue": "Stack:OutputRefSecretA720EF05",
                    },
                    DdSite: "datadoghq.com",
                    FunctionName: "MyForwarder",
                },
            }));
        });
        it("allows locking the forwarder version", () => {
            const datadogStack = new index_1.DatadogIntegrationStack(stack, "DatadogStack", {
                apiKey: secret,
                externalId: "not-a-real-id",
                forwarderVersion: "3.9.0",
            });
            assert_1.expect(datadogStack).to(assert_1.haveResource("AWS::CloudFormation::Stack", {
                TemplateURL: "https://datadog-cloudformation-template.s3.amazonaws.com/aws/forwarder/3.9.0.yaml",
                Parameters: {
                    DdApiKey: "USE_ARN",
                    DdApiKeySecretArn: {
                        "Fn::ImportValue": "Stack:OutputRefSecretA720EF05",
                    },
                    DdSite: "datadoghq.com",
                    FunctionName: "DatadogForwarder",
                },
            }));
        });
        it("allows passing additional, arbitrary parameters", () => {
            const datadogStack = new index_1.DatadogIntegrationStack(stack, "DatadogStack", {
                apiKey: secret,
                externalId: "not-a-real-id",
                additionalForwarderParams: {
                    Foo: "bar",
                },
            });
            assert_1.expect(datadogStack).to(assert_1.haveResource("AWS::CloudFormation::Stack", {
                TemplateURL: "https://datadog-cloudformation-template.s3.amazonaws.com/aws/forwarder/latest.yaml",
                Parameters: {
                    DdApiKey: "USE_ARN",
                    DdApiKeySecretArn: {
                        "Fn::ImportValue": "Stack:OutputRefSecretA720EF05",
                    },
                    DdSite: "datadoghq.com",
                    FunctionName: "DatadogForwarder",
                    Foo: "bar",
                },
            }));
        });
        it("allows additional parameters to override parameters set other props", () => {
            const datadogStack = new index_1.DatadogIntegrationStack(stack, "DatadogStack", {
                apiKey: secret,
                externalId: "not-a-real-id",
                additionalForwarderParams: {
                    DdApiKeySecretArn: "some-hardcoded-thing",
                },
            });
            assert_1.expect(datadogStack).to(assert_1.haveResource("AWS::CloudFormation::Stack", {
                TemplateURL: "https://datadog-cloudformation-template.s3.amazonaws.com/aws/forwarder/latest.yaml",
                Parameters: {
                    DdApiKey: "USE_ARN",
                    DdApiKeySecretArn: "some-hardcoded-thing",
                    DdSite: "datadoghq.com",
                    FunctionName: "DatadogForwarder",
                },
            }));
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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