"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("@aws-cdk/core");
const secrets = require("@aws-cdk/aws-secretsmanager");
const config_1 = require("../lib/config");
describe("applyDefaultsToConfig", () => {
    let apiKey;
    beforeEach(() => {
        const app = new cdk.App();
        const construct = new cdk.Stack(app, "MyStack");
        apiKey = new secrets.Secret(construct, "DatadogApiKey");
    });
    describe("site", () => {
        it("defaults to datadoghq.com", () => {
            const config = {
                apiKey,
                externalId: "a-fake-external-id",
            };
            const configWithDefaults = config_1.applyDefaultsToConfig(config);
            expect(configWithDefaults.site).toEqual("datadoghq.com");
        });
        it("can be overridden", () => {
            const config = {
                apiKey,
                externalId: "a-fake-external-id",
                site: "datadoghq.eu",
            };
            const configWithDefaults = config_1.applyDefaultsToConfig(config);
            expect(configWithDefaults.site).toEqual("datadoghq.eu");
        });
    });
    describe("iamRoleName", () => {
        it("defaults to DatadogIntegrationRole", () => {
            const config = {
                apiKey,
                externalId: "a-fake-external-id",
            };
            const configWithDefaults = config_1.applyDefaultsToConfig(config);
            expect(configWithDefaults.iamRoleName).toEqual("DatadogIntegrationRole");
        });
        it("can be overridden", () => {
            const config = {
                apiKey,
                externalId: "a-fake-external-id",
                iamRoleName: "MySuperSpecialRoleName",
            };
            const configWithDefaults = config_1.applyDefaultsToConfig(config);
            expect(configWithDefaults.iamRoleName).toEqual("MySuperSpecialRoleName");
        });
    });
    describe("permissions", () => {
        it("defaults to Full", () => {
            const config = {
                apiKey,
                externalId: "a-fake-external-id",
            };
            const configWithDefaults = config_1.applyDefaultsToConfig(config);
            expect(configWithDefaults.permissions).toEqual("Full");
        });
        it("can be overridden", () => {
            const config = {
                apiKey,
                externalId: "a-fake-external-id",
                permissions: "Core",
            };
            const configWithDefaults = config_1.applyDefaultsToConfig(config);
            expect(configWithDefaults.permissions).toEqual("Core");
        });
    });
    describe("forwarderName", () => {
        it("defaults to DatadogForwarder", () => {
            const config = {
                apiKey,
                externalId: "a-fake-external-id",
            };
            const configWithDefaults = config_1.applyDefaultsToConfig(config);
            expect(configWithDefaults.forwarderName).toEqual("DatadogForwarder");
        });
        it("can be overridden", () => {
            const config = {
                apiKey,
                externalId: "a-fake-external-id",
                forwarderName: "MySuperSpecialForwarderName",
            };
            const configWithDefaults = config_1.applyDefaultsToConfig(config);
            expect(configWithDefaults.forwarderName).toEqual("MySuperSpecialForwarderName");
        });
    });
    describe("forwarderVersion", () => {
        it("defaults to latest", () => {
            const config = {
                apiKey,
                externalId: "a-fake-external-id",
            };
            const configWithDefaults = config_1.applyDefaultsToConfig(config);
            expect(configWithDefaults.forwarderVersion).toEqual("latest");
        });
        it("can be overridden", () => {
            const config = {
                apiKey,
                externalId: "a-fake-external-id",
                forwarderVersion: "3.8.0",
            };
            const configWithDefaults = config_1.applyDefaultsToConfig(config);
            expect(configWithDefaults.forwarderVersion).toEqual("3.8.0");
        });
    });
    describe("installDatadogPolicyMacro", () => {
        it("defaults to true", () => {
            const config = {
                apiKey,
                externalId: "a-fake-external-id",
            };
            const configWithDefaults = config_1.applyDefaultsToConfig(config);
            expect(configWithDefaults.installDatadogPolicyMacro).toEqual(true);
        });
        it("can be overridden", () => {
            const config = {
                apiKey,
                externalId: "a-fake-external-id",
                installDatadogPolicyMacro: false,
            };
            const configWithDefaults = config_1.applyDefaultsToConfig(config);
            expect(configWithDefaults.installDatadogPolicyMacro).toEqual(false);
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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