"""
SiCSi.py

Disilicon carbide, Si2C, ground state surface.
Calculated by Changala and Baraban (``P2'').

Input: Cartesians in Si--C--Si order (Angstroms)
Output: energy in cm^-1

"""


import nitrogen as n2
import nitrogen.autodiff.forward as adf
import numpy as np 


def Vfun(X, deriv = 0, out = None, var = None):
    """
    expected order Si--C--Si
    """
    x = n2.dfun.X2adf(X, deriv, var)
    
    Si1 = [x[0], x[1], x[2]]
    C =   [x[3], x[4], x[5]]
    Si2 = [x[6], x[7], x[8]]
    
    v1 = [ Si1[i]-C[i] for i in range(3)] # Si1 <-- C
    v2 = [ Si2[i]-C[i] for i in range(3)] # Si2 <-- C
    
    r1 = adf.sqrt(v1[0]*v1[0] + v1[1]*v1[1] + v1[2]*v1[2])
    r2 = adf.sqrt(v2[0]*v2[0] + v2[1]*v2[1] + v2[2]*v2[2])
    dot = v1[0]*v2[0] + v1[1]*v2[1] + v1[2]*v2[2]
    theta = adf.acos(dot/(r1*r2)) * 180.0 / n2.pi # Angle in degrees 

    Rlow, Rhigh = 1.1, 2.5 # Angstroms 
    thlow = 50.0 # degrees 
    
    nd = r1.d.shape[0] 
    
    I = r1.d[0:1] < Rlow 
    r1.d[0:1][I] = Rlow 
    for i in range(1,nd):
        r1.d[i:(i+1)][I] = 0.0 
        
    I = r2.d[0:1] < Rlow 
    r2.d[0:1][I] = Rlow 
    for i in range(1,nd):
        r2.d[i:(i+1)][I] = 0.0     
        
    I = r1.d[0:1] > Rhigh 
    r1.d[0:1][I] = Rhigh
    for i in range(1,nd):
        r1.d[i:(i+1)][I] = 0.0 
        
    I = r2.d[0:1] > Rhigh
    r2.d[0:1][I] = Rhigh
    for i in range(1,nd):
        r2.d[i:(i+1)][I] = 0.0    
        
    I = theta.d[0:1] < thlow
    theta.d[0:1][I] = thlow
    for i in range(1,nd):
        theta.d[i:(i+1)][I] = 0.0 
        
    v = 0.5 * (calcPolySurf(r1,r2,theta) + calcPolySurf(r2,r1,theta)) 
    
  
    return n2.dfun.adf2array([v], out)

######################################
# 
# Define module-scope PES DFun object
#
PES = n2.dfun.DFun(Vfun, nf = 1, nx = 9)
#
#
######################################

r1Pows = np.array([0,1,0,2,0,1,3,0,2,1,4,0,3,1,2,5,0,4,1,3,2,6,0,5,1,4,2,3,7,0,6,1,5,2,4,3,0,1,0,2,0,1,3,0,2,1,4,0,3,1,2,5,0,4,1,3,2,6,0,5,1,4,2,3,7,0,6,1,5,2,4,3,0,1,0,2,0,1,3,0,2,1,4,0,3,1,2,5,0,4,1,3,2,6,0,5,1,4,2,3,7,0,6,1,5,2,4,3,0,1,0,2,0,1,3,0,2,1,4,0,3,1,2,5,0,4,1,3,2,6,0,5,1,4,2,3,7,0,6,1,5,2,4,3,0,1,0,2,0,1,3,0,2,1,4,0,3,1,2,5,0,4,1,3,2,6,0,5,1,4,2,3,7,0,6,1,5,2,4,3,0,1,0,2,0,1,3,0,2,1,4,0,3,1,2,5,0,4,1,3,2,6,0,5,1,4,2,3,7,0,6,1,5,2,4,3,0,1,0,2,0,1,3,0,2,1,4,0,3,1,2,5,0,4,1,3,2,6,0,5,1,4,2,3,7,0,6,1,5,2,4,3,0,1,0,2,0,1,3,0,2,1,4,0,3,1,2,5,0,4,1,3,2,6,0,5,1,4,2,3,7,0,6,1,5,2,4,3])
r2Pows = np.array([0,0,1,0,2,1,0,3,1,2,0,4,1,3,2,0,5,1,4,2,3,0,6,1,5,2,4,3,0,7,1,6,2,5,3,4,0,0,1,0,2,1,0,3,1,2,0,4,1,3,2,0,5,1,4,2,3,0,6,1,5,2,4,3,0,7,1,6,2,5,3,4,0,0,1,0,2,1,0,3,1,2,0,4,1,3,2,0,5,1,4,2,3,0,6,1,5,2,4,3,0,7,1,6,2,5,3,4,0,0,1,0,2,1,0,3,1,2,0,4,1,3,2,0,5,1,4,2,3,0,6,1,5,2,4,3,0,7,1,6,2,5,3,4,0,0,1,0,2,1,0,3,1,2,0,4,1,3,2,0,5,1,4,2,3,0,6,1,5,2,4,3,0,7,1,6,2,5,3,4,0,0,1,0,2,1,0,3,1,2,0,4,1,3,2,0,5,1,4,2,3,0,6,1,5,2,4,3,0,7,1,6,2,5,3,4,0,0,1,0,2,1,0,3,1,2,0,4,1,3,2,0,5,1,4,2,3,0,6,1,5,2,4,3,0,7,1,6,2,5,3,4,0,0,1,0,2,1,0,3,1,2,0,4,1,3,2,0,5,1,4,2,3,0,6,1,5,2,4,3,0,7,1,6,2,5,3,4])
thPows = np.array([0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,14,14,14,14,14,14,14,14,14,14,14,14,14,14,14,14,14,14,14,14,14,14,14,14,14,14,14,14,14,14,14,14,14,14,14,14])
coeffs = np.array([857.524944436589750,
348.469821171883210,
348.469821171911690,
5443.563464453269300,
5443.563464453101000,
-185.736021844085000,
-1773.551296948723100,
-1773.551296948772900,
-39.424930249215194,
-39.424930249178068,
357.302753340487980,
357.302753340925050,
2.978103432803976,
2.978103432846262,
-10.271482409660670,
-57.481455813625132,
-57.481455813714277,
-1.810796248450347,
-1.810796248510299,
3.405997584100971,
3.405997584108084,
7.693750299290053,
7.693750299004279,
0.991306103855796,
0.991306103824207,
-0.245316934830057,
-0.245316934831653,
-0.161940340166463,
-0.677483209224872,
-0.677483209097739,
-0.200003806389216,
-0.200003806362585,
0.045153632066720,
0.045153632070150,
-0.063635702293336,
-0.063635702276371,
-566.495262046914260,
89.964777451444562,
89.964777450644448,
-27.096284960526972,
-27.096284956447917,
-230.932935556440670,
12.486590271658427,
12.486590273172705,
62.304904522863701,
62.304904521700053,
-4.127845408347543,
-4.127845419021619,
-6.751313098911380,
-6.751313100106451,
-12.603594690717873,
2.175405330159390,
2.175405332398710,
3.192251318064605,
3.192251319701453,
2.833645615075563,
2.833645614455758,
-0.649054240482784,
-0.649054233518797,
-0.860742110007082,
-0.860742109136492,
-1.506608366535071,
-1.506608367134738,
-0.768762272168247,
0.094378742605697,
0.094378739349885,
0.093689682805923,
0.093689682008723,
0.242960209895550,
0.242960210323787,
0.167464484941908,
0.167464484856969,
-185.844872400474910,
-144.114909562458420,
-144.114909559094260,
36.250199454761400,
36.250199438140555,
71.109949577048567,
-3.424215781222824,
-3.424215787755155,
-9.435671224622544,
-9.435671219621829,
2.915687948582403,
2.915687991362336,
3.135841418140202,
3.135841423267464,
-0.008505708985316,
-1.986156017526157,
-1.986156026073269,
-2.907243328205579,
-2.907243335064556,
-3.452847322296749,
-3.452847319318788,
0.450324264214388,
0.450324236808496,
0.685745646093821,
0.685745642331916,
1.630622126986429,
1.630622129120096,
1.539599023467446,
-0.050614504799787,
-0.050614492212930,
0.035569928110464,
0.035569931486543,
-0.227887724525032,
-0.227887726239519,
-0.131010099141859,
-0.131010098708201,
114.342168123665270,
-9.208909674986314,
-9.208909680254456,
-5.235070220453003,
-5.235070194189582,
0.197269078726563,
-2.859558874056298,
-2.859558863872173,
-5.356719504790620,
-5.356719512672485,
-0.812152195048845,
-0.812152261894224,
-1.502984503080665,
-1.502984511281582,
5.622951318315400,
1.072685504554364,
1.072685517848108,
1.940784084136423,
1.940784094869559,
1.562087350564884,
1.562087345658653,
-0.142566247721197,
-0.142566205080932,
-0.185100997575388,
-0.185100991459240,
-1.003236851003374,
-1.003236853773475,
-0.652893586787717,
-0.002734223788443,
-0.002734243286331,
-0.134306558524505,
-0.134306563913510,
0.070884367953079,
0.070884370476769,
-0.034060940931459,
-0.034060941726997,
0.643972062574285,
-6.119141657076965,
-6.119141653157724,
2.292874774473658,
2.292874754620535,
1.249117484825980,
2.431721586532534,
2.431721579053826,
1.136223454397856,
1.136223460275486,
0.008667250263555,
0.008667300488826,
1.433504054785482,
1.433504060998114,
-3.003913030788170,
-0.431929575300207,
-0.431929585391862,
-0.909615196101607,
-0.909615204083237,
-0.631883299523556,
-0.631883295787372,
0.013970497251278,
0.013970465142776,
-0.056450683827899,
-0.056450688539870,
0.454934080408079,
0.454934082164588,
0.054025939459424,
0.014259649593626,
0.014259664293740,
0.106384660480378,
0.106384664570111,
0.013660360089970,
0.013660358302853,
0.054762752556010,
0.054762753214185,
7.664109010852639,
0.816389502036582,
0.816389500544450,
-0.662836705791993,
-0.662836698123450,
0.988988031295440,
-0.962821995815533,
-0.962821993007310,
-0.310011979557733,
-0.310011981798325,
0.120929955700951,
0.120929936361873,
-0.616205852537852,
-0.616205854937544,
1.229553493196015,
0.108930553545810,
0.108930557481928,
0.242229612909543,
0.242229615946483,
0.114778158269556,
0.114778156830336,
0.005866628447440,
0.005866640866683,
0.057436795947634,
0.057436797797048,
-0.136858984911828,
-0.136858985503382,
0.081393580857975,
-0.006577944535367,
-0.006577950237564,
-0.040200034392770,
-0.040200035979143,
-0.016517921536019,
-0.016517920878567,
-0.022433443174491,
-0.022433443440922,
-1.834833487535643,
-0.260017462748362,
-0.260017462467663,
0.229356133206182,
0.229356131747105,
-0.109750661452689,
0.189435798455859,
0.189435797933974,
-0.033093467147403,
-0.033093466725418,
-0.044466083262276,
-0.044466079588752,
0.139844369116602,
0.139844369573163,
-0.248288892718547,
-0.015065549628503,
-0.015065550384948,
-0.032178910353485,
-0.032178910924440,
-0.005942978828766,
-0.005942978556291,
-0.001602663832454,
-0.001602666203547,
-0.015723853731769,
-0.015723854089089,
0.023626645960527,
0.023626646062725,
-0.029891899764101,
0.001179643562862,
0.001179644654839,
0.007694846706688,
0.007694847010474,
0.004181289161628,
0.004181289040409,
0.004188383222575,
0.004188383273964,
0.263300368953268,
-0.011203496834087,
-0.011203496854759,
-0.031460172037944,
-0.031460171929588,
0.028794972755268,
-0.012319779069096,
-0.012319779031044,
0.007822153067736,
0.007822153036620,
0.004856997564303,
0.004856997291750,
-0.014186375554142,
-0.014186375588065,
0.021513892787542,
0.000808637179770,
0.000808637236432,
0.001969847682185,
0.001969847724200,
-0.000473430376598,
-0.000473430396754,
0.000099195157436,
0.000099195334225,
0.001453362473860,
0.001453362500818,
-0.001848793242119,
-0.001848793249222,
0.003284129725790,
-0.000067383607051,
-0.000067383688693,
-0.000601445134069,
-0.000601445156832,
-0.000324613740563,
-0.000324613731726,
-0.000317768237857,
-0.000317768241642])

def calcPolySurf(r1,r2,theta):
    """
    r1,r2: Angstroms
    theta: degrees

    """
    
    Rref,Rscale = 1.7023, 0.2 
    thetaRef, thetaScale = 180.0, 50.0 
    
    thetaPowMax = 14 
    rPowMax = 7 
    Nterms = 288 
    
    # Calculate scaled coordinates and 
    # their powers
    #
    r1S = [adf.const_like(1.0, r1), (r1 - Rref)/Rscale] 
    for i in range(2,rPowMax+1):
        r1S.append(r1S[1] * r1S[i-1]) 
        
    r2S = [adf.const_like(1.0, r2), (r2 - Rref)/Rscale] 
    for i in range(2,rPowMax+1):
        r2S.append(r2S[1] * r2S[i-1]) 
        
    thS = [adf.const_like(1.0, theta), (theta - thetaRef)/thetaScale ] 
    for i in range(2,thetaPowMax+1):
        thS.append(thS[1] * thS[i-1]) 
        
    v = adf.const_like(0.0, r1) 
    for i in range(Nterms):
        v += coeffs[i] * r1S[r1Pows[i]] * r2S[r2Pows[i]] * thS[thPows[i]]
        
        
    return v 
    