"""
H2O2_MK2012_BO.py

Hydrogen peroxide, H2O2, ground state surface from
Ref [1]_. The coefficients are available from the references
supplementary information as the 'Born-Oppenheimer PES', which 
I understand to equal the "V+C+R+H" results.

The surface is implemented in internal coordinates.

X1 ... O1 -- H1 bond length (Angstroms)
X2 ... O2 -- H2 bond length ( "    "  )
X3 ... O1 -- O2 bond length ( "    "  )
X4 ... O2-O1-H1 bond angle (degrees)
X5 ... O1-O2-H2 bond angle ( "   " )
X6 ... dihedral angle ( "   " )

References
----------

.. [1] P. Malyszek and J. Koput. J. Comp. Chem. 34, 337-344 (2013).
       https://doi.org/10.1002/jcc.23137

"""


import nitrogen as n2
import nitrogen.autodiff.forward as adf
import numpy as np 


def Vfun(X, deriv = 0, out = None, var = None):
    """
    expected order : r1, r2, R, a1, a2, tau
    """
    x = n2.dfun.X2adf(X, deriv, var)
    
    r1 = x[0]
    r2 = x[1]
    R = x[2]
    a1 = x[3]
    a2 = x[4]
    tau = x[5]
    
    # Define reference values 
    Re = 1.45539378 # Angstroms
    re = 0.96252476 # Angstroms
    ae = 101.08194717 # degrees 
    
    q1 = (r1 - re) / r1  # Simons-Parr-Finlan coordinates
    q2 = (r2 - re) / r2 
    q3 = (R - Re) / R 
    q4 = (a1 - ae) * np.pi/180.0  # radians
    q5 = (a2 - ae) * np.pi/180.0  # radians
    q6 = tau * np.pi/180.0 # radians 
    
    # Calculate surface 
    
    v = calcsurf([q1,q2,q3,q4,q5,q6]) * n2.constants.Eh
  
    return n2.dfun.adf2array([v], out)

######################################
# 
# Define module-scope PES DFun object
#
PES = n2.dfun.DFun(Vfun, nf = 1, nx = 6)
#
#
######################################

def calcsurf(q):
    
    max_pow = [5,5,5,6,6,6] # max_pow[5] is really the max freq. of dihedral
    
    qpow = []
    for i in range(5):
        qi = [adf.const_like(1.0, q[i]), q[i]]
        for p in range(2,max_pow[i]+1):
            qi.append(qi[1] * qi[p-1]) # qi ** p 
        qpow.append(qi)
    
    # Calculate cos(n*q6)
    cosq = [ adf.cos(n * q[5]) for n in range(max_pow[5] + 1)]
    qpow.append(cosq) 

    v = 0.0
    
    nterms = powers.shape[0]
    
    for i in range(nterms):
        c = coeffs[i] 
        
        v += c * \
            qpow[0][powers[i,0]] * \
            qpow[1][powers[i,1]] * \
            qpow[2][powers[i,2]] * \
            qpow[3][powers[i,3]] * \
            qpow[4][powers[i,4]] * \
            qpow[5][powers[i,5]]
    
    return v



powers = np.array([
[ 0, 0, 0, 0, 0, 0],
[ 0, 0, 0, 0, 0, 1],
[ 0, 0, 0, 0, 0, 2],
[ 0, 0, 0, 0, 0, 3],
[ 0, 0, 0, 0, 0, 4],
[ 0, 0, 0, 0, 0, 5],
[ 0, 0, 0, 0, 0, 6],
[ 0, 0, 2, 0, 0, 0],
[ 2, 0, 0, 0, 0, 0],
[ 0, 2, 0, 0, 0, 0],
[ 0, 0, 0, 2, 0, 0],
[ 0, 0, 0, 0, 2, 0],
[ 1, 0, 1, 0, 0, 0],
[ 0, 1, 1, 0, 0, 0],
[ 0, 0, 1, 1, 0, 0],
[ 0, 0, 1, 0, 1, 0],
[ 1, 1, 0, 0, 0, 0],
[ 1, 0, 0, 1, 0, 0],
[ 0, 1, 0, 0, 1, 0],
[ 1, 0, 0, 0, 1, 0],
[ 0, 1, 0, 1, 0, 0],
[ 0, 0, 0, 1, 1, 0],
[ 0, 0, 3, 0, 0, 0],
[ 3, 0, 0, 0, 0, 0],
[ 0, 3, 0, 0, 0, 0],
[ 0, 0, 0, 3, 0, 0],
[ 0, 0, 0, 0, 3, 0],
[ 1, 0, 2, 0, 0, 0],
[ 0, 1, 2, 0, 0, 0],
[ 0, 0, 2, 1, 0, 0],
[ 0, 0, 2, 0, 1, 0],
[ 2, 0, 1, 0, 0, 0],
[ 0, 2, 1, 0, 0, 0],
[ 0, 0, 1, 2, 0, 0],
[ 0, 0, 1, 0, 2, 0],
[ 1, 2, 0, 0, 0, 0],
[ 2, 1, 0, 0, 0, 0],
[ 1, 0, 0, 2, 0, 0],
[ 0, 1, 0, 0, 2, 0],
[ 2, 0, 0, 1, 0, 0],
[ 0, 2, 0, 0, 1, 0],
[ 1, 0, 0, 0, 2, 0],
[ 0, 1, 0, 2, 0, 0],
[ 2, 0, 0, 0, 1, 0],
[ 0, 2, 0, 1, 0, 0],
[ 0, 0, 0, 1, 2, 0],
[ 0, 0, 0, 2, 1, 0],
[ 1, 1, 1, 0, 0, 0],
[ 1, 0, 1, 1, 0, 0],
[ 0, 1, 1, 0, 1, 0],
[ 1, 0, 1, 0, 1, 0],
[ 0, 1, 1, 1, 0, 0],
[ 0, 0, 1, 1, 1, 0],
[ 1, 1, 0, 1, 0, 0],
[ 1, 1, 0, 0, 1, 0],
[ 1, 0, 0, 1, 1, 0],
[ 0, 1, 0, 1, 1, 0],
[ 0, 0, 4, 0, 0, 0],
[ 4, 0, 0, 0, 0, 0],
[ 0, 4, 0, 0, 0, 0],
[ 0, 0, 0, 4, 0, 0],
[ 0, 0, 0, 0, 4, 0],
[ 2, 0, 2, 0, 0, 0],
[ 0, 2, 2, 0, 0, 0],
[ 0, 0, 2, 2, 0, 0],
[ 0, 0, 2, 0, 2, 0],
[ 2, 2, 0, 0, 0, 0],
[ 2, 0, 0, 2, 0, 0],
[ 0, 2, 0, 0, 2, 0],
[ 0, 0, 0, 2, 2, 0],
[ 1, 0, 3, 0, 0, 0],
[ 0, 1, 3, 0, 0, 0],
[ 0, 0, 3, 1, 0, 0],
[ 0, 0, 3, 0, 1, 0],
[ 3, 0, 0, 1, 0, 0],
[ 0, 3, 0, 0, 1, 0],
[ 3, 0, 1, 0, 0, 0],
[ 0, 3, 1, 0, 0, 0],
[ 0, 0, 1, 3, 0, 0],
[ 0, 0, 1, 0, 3, 0],
[ 1, 3, 0, 0, 0, 0],
[ 3, 1, 0, 0, 0, 0],
[ 1, 0, 0, 3, 0, 0],
[ 0, 1, 0, 0, 3, 0],
[ 1, 0, 0, 0, 3, 0],
[ 0, 1, 0, 3, 0, 0],
[ 0, 0, 0, 1, 3, 0],
[ 0, 0, 0, 3, 1, 0],
[ 1, 1, 2, 0, 0, 0],
[ 1, 0, 2, 1, 0, 0],
[ 0, 1, 2, 0, 1, 0],
[ 1, 0, 2, 0, 1, 0],
[ 0, 1, 2, 1, 0, 0],
[ 0, 0, 2, 1, 1, 0],
[ 2, 0, 0, 1, 1, 0],
[ 0, 2, 0, 1, 1, 0],
[ 1, 0, 1, 2, 0, 0],
[ 0, 1, 1, 0, 2, 0],
[ 1, 0, 0, 1, 2, 0],
[ 0, 1, 0, 2, 1, 0],
[ 1, 0, 0, 2, 1, 0],
[ 0, 1, 0, 1, 2, 0],
[ 0, 0, 5, 0, 0, 0],
[ 5, 0, 0, 0, 0, 0],
[ 0, 5, 0, 0, 0, 0],
[ 0, 0, 0, 5, 0, 0],
[ 0, 0, 0, 0, 5, 0],
[ 0, 0, 0, 6, 0, 0],
[ 0, 0, 0, 0, 6, 0],
[ 0, 0, 0, 4, 1, 0],
[ 0, 0, 0, 1, 4, 0],
[ 0, 0, 0, 3, 2, 0],
[ 0, 0, 0, 2, 3, 0],
[ 0, 0, 1, 4, 0, 0],
[ 0, 0, 1, 0, 4, 0],
[ 0, 0, 2, 3, 0, 0],
[ 0, 0, 2, 0, 3, 0],
[ 1, 0, 0, 4, 0, 0],
[ 0, 1, 0, 0, 4, 0],
[ 2, 0, 0, 3, 0, 0],
[ 0, 2, 0, 0, 3, 0],
[ 0, 0, 1, 0, 0, 1],
[ 1, 0, 0, 0, 0, 1],
[ 0, 1, 0, 0, 0, 1],
[ 0, 0, 0, 1, 0, 1],
[ 0, 0, 0, 0, 1, 1],
[ 0, 0, 2, 0, 0, 1],
[ 2, 0, 0, 0, 0, 1],
[ 0, 2, 0, 0, 0, 1],
[ 0, 0, 0, 2, 0, 1],
[ 0, 0, 0, 0, 2, 1],
[ 1, 0, 1, 0, 0, 1],
[ 0, 1, 1, 0, 0, 1],
[ 0, 0, 1, 1, 0, 1],
[ 0, 0, 1, 0, 1, 1],
[ 1, 1, 0, 0, 0, 1],
[ 1, 0, 0, 1, 0, 1],
[ 0, 1, 0, 0, 1, 1],
[ 1, 0, 0, 0, 1, 1],
[ 0, 1, 0, 1, 0, 1],
[ 0, 0, 0, 1, 1, 1],
[ 0, 0, 3, 0, 0, 1],
[ 3, 0, 0, 0, 0, 1],
[ 0, 3, 0, 0, 0, 1],
[ 0, 0, 0, 3, 0, 1],
[ 0, 0, 0, 0, 3, 1],
[ 1, 0, 2, 0, 0, 1],
[ 0, 1, 2, 0, 0, 1],
[ 0, 0, 2, 1, 0, 1],
[ 0, 0, 2, 0, 1, 1],
[ 0, 0, 1, 2, 0, 1],
[ 0, 0, 1, 0, 2, 1],
[ 1, 2, 0, 0, 0, 1],
[ 2, 1, 0, 0, 0, 1],
[ 1, 0, 0, 2, 0, 1],
[ 0, 1, 0, 0, 2, 1],
[ 1, 0, 0, 0, 2, 1],
[ 0, 1, 0, 2, 0, 1],
[ 0, 0, 0, 1, 2, 1],
[ 0, 0, 0, 2, 1, 1],
[ 1, 1, 1, 0, 0, 1],
[ 1, 0, 0, 1, 1, 1],
[ 0, 1, 0, 1, 1, 1],
[ 0, 0, 0, 4, 0, 1],
[ 0, 0, 0, 0, 4, 1],
[ 0, 0, 0, 5, 0, 1],
[ 0, 0, 0, 0, 5, 1],
[ 0, 0, 1, 3, 0, 1],
[ 0, 0, 1, 0, 3, 1],
[ 0, 0, 2, 2, 0, 1],
[ 0, 0, 2, 0, 2, 1],
[ 0, 0, 0, 1, 3, 1],
[ 0, 0, 0, 3, 1, 1],
[ 0, 0, 0, 2, 2, 1],
[ 1, 0, 0, 3, 0, 1],
[ 0, 1, 0, 0, 3, 1],
[ 1, 0, 0, 0, 3, 1],
[ 0, 1, 0, 3, 0, 1],
[ 2, 0, 0, 2, 0, 1],
[ 0, 2, 0, 0, 2, 1],
[ 2, 0, 0, 0, 2, 1],
[ 0, 2, 0, 2, 0, 1],
[ 1, 0, 2, 1, 0, 1],
[ 0, 1, 2, 0, 1, 1],
[ 2, 0, 1, 1, 0, 1],
[ 0, 2, 1, 0, 1, 1],
[ 1, 0, 1, 2, 0, 1],
[ 0, 1, 1, 0, 2, 1],
[ 0, 0, 1, 0, 0, 2],
[ 1, 0, 0, 0, 0, 2],
[ 0, 1, 0, 0, 0, 2],
[ 0, 0, 0, 1, 0, 2],
[ 0, 0, 0, 0, 1, 2],
[ 0, 0, 2, 0, 0, 2],
[ 2, 0, 0, 0, 0, 2],
[ 0, 2, 0, 0, 0, 2],
[ 0, 0, 0, 2, 0, 2],
[ 0, 0, 0, 0, 2, 2],
[ 1, 0, 1, 0, 0, 2],
[ 0, 1, 1, 0, 0, 2],
[ 0, 0, 1, 1, 0, 2],
[ 0, 0, 1, 0, 1, 2],
[ 1, 1, 0, 0, 0, 2],
[ 1, 0, 0, 1, 0, 2],
[ 0, 1, 0, 0, 1, 2],
[ 1, 0, 0, 0, 1, 2],
[ 0, 1, 0, 1, 0, 2],
[ 0, 0, 0, 1, 1, 2],
[ 0, 0, 3, 0, 0, 2],
[ 3, 0, 0, 0, 0, 2],
[ 0, 3, 0, 0, 0, 2],
[ 0, 0, 0, 3, 0, 2],
[ 0, 0, 0, 0, 3, 2],
[ 0, 0, 0, 2, 1, 2],
[ 0, 0, 0, 1, 2, 2],
[ 0, 0, 1, 2, 0, 2],
[ 0, 0, 1, 0, 2, 2],
[ 1, 0, 2, 0, 0, 2],
[ 0, 1, 2, 0, 0, 2],
[ 2, 0, 1, 0, 0, 2],
[ 0, 2, 1, 0, 0, 2],
[ 0, 0, 0, 4, 0, 2],
[ 0, 0, 0, 0, 4, 2],
[ 0, 0, 0, 1, 3, 2],
[ 0, 0, 0, 3, 1, 2],
[ 0, 0, 0, 2, 2, 2],
[ 2, 0, 0, 1, 0, 2],
[ 0, 2, 0, 0, 1, 2],
[ 1, 0, 0, 2, 0, 2],
[ 0, 1, 0, 0, 2, 2],
[ 1, 0, 0, 0, 2, 2],
[ 0, 1, 0, 2, 0, 2],
[ 1, 0, 1, 1, 0, 2],
[ 0, 1, 1, 0, 1, 2],
[ 1, 0, 1, 0, 1, 2],
[ 0, 1, 1, 1, 0, 2],
[ 0, 0, 1, 3, 0, 2],
[ 0, 0, 1, 0, 3, 2],
[ 0, 0, 1, 0, 0, 3],
[ 1, 0, 0, 0, 0, 3],
[ 0, 1, 0, 0, 0, 3],
[ 0, 0, 0, 1, 0, 3],
[ 0, 0, 0, 0, 1, 3],
[ 0, 0, 2, 0, 0, 3],
[ 2, 0, 0, 0, 0, 3],
[ 0, 2, 0, 0, 0, 3],
[ 0, 0, 0, 2, 0, 3],
[ 0, 0, 0, 0, 2, 3],
[ 0, 0, 0, 1, 1, 3],
[ 0, 0, 3, 0, 0, 3],
[ 0, 0, 0, 3, 0, 3],
[ 0, 0, 0, 0, 3, 3],
[ 0, 0, 0, 1, 2, 3],
[ 0, 0, 0, 2, 1, 3],
[ 0, 0, 1, 1, 0, 3],
[ 0, 0, 1, 0, 1, 3],
[ 1, 0, 0, 1, 0, 3],
[ 0, 1, 0, 0, 1, 3],
[ 1, 0, 0, 0, 1, 3],
[ 0, 1, 0, 1, 0, 3],
[ 0, 0, 2, 1, 0, 3],
[ 0, 0, 2, 0, 1, 3],
[ 0, 0, 1, 0, 0, 4],
[ 1, 0, 0, 0, 0, 4],
[ 0, 1, 0, 0, 0, 4],
[ 0, 0, 0, 1, 0, 4],
[ 0, 0, 0, 0, 1, 4],
[ 0, 0, 2, 0, 0, 4],
[ 0, 0, 0, 2, 0, 4],
[ 0, 0, 0, 0, 2, 4],
[ 0, 0, 0, 1, 1, 4],
[ 0, 0, 1, 1, 0, 4],
[ 0, 0, 1, 0, 1, 4],
[ 0, 0, 1, 0, 0, 5],
[ 1, 0, 0, 0, 0, 5],
[ 0, 1, 0, 0, 0, 5],
[ 0, 0, 0, 1, 0, 5],
[ 0, 0, 0, 0, 1, 5],

])
    

coeffs = np.array([
    0.00396113,
    0.00480467,
    0.00319464,
    0.00027000,
    0.00005320,
    0.00001046,
    0.00000198,
    1.07033434,
    0.85653683,
    0.85653683,
    0.11106789,
    0.11106789,
   -0.03845365,
   -0.03845365,
    0.18422191,
    0.18422191,
    0.00034113,
   -0.00668184,
   -0.00668184,
   -0.00196092,
   -0.00196092,
    0.01745600,
   -1.18345015,
   -0.23754920,
   -0.23754920,
   -0.02615109,
   -0.02615109,
   -0.15446390,
   -0.15446390,
   -0.35538073,
   -0.35538073,
    0.07925928,
    0.07925928,
   -0.26791829,
   -0.26791829,
   -0.00408369,
   -0.00408369,
   -0.01932873,
   -0.01932873,
   -0.01107772,
   -0.01107772,
   -0.00815584,
   -0.00815584,
    0.00335581,
    0.00335581,
   -0.01353685,
   -0.01353685,
    0.01297257,
   -0.07451442,
   -0.07451442,
   -0.03375914,
   -0.03375914,
   -0.01676549,
   -0.00495022,
   -0.00495022,
   -0.00455155,
   -0.00455155,
   -0.91114371,
   -0.38794237,
   -0.38794237,
   -0.00503667,
   -0.00503667,
   -0.46529063,
   -0.46529063,
    0.07499351,
    0.07499351,
   -0.00797550,
   -0.04057873,
   -0.04057873,
    0.00364187,
    0.47679811,
    0.47679811,
   -0.41694250,
   -0.41694250,
   -0.06417306,
   -0.06417306,
    0.26181548,
    0.26181548,
    0.10331249,
    0.10331249,
   -0.01676460,
   -0.01676460,
    0.04972082,
    0.04972082,
    0.00353493,
    0.00353493,
    0.00451312,
    0.00451312,
   -0.05443810,
    0.06639535,
    0.06639535,
    0.03811353,
    0.03811353,
   -0.09184951,
   -0.01594273,
   -0.01594273,
    0.01376699,
    0.01376699,
   -0.01343723,
   -0.01343723,
   -0.00670949,
   -0.00670949,
   -0.73372711,
   -0.30456872,
   -0.30456872,
    0.00586074,
    0.00586074,
   -0.00552073,
   -0.00552073,
    0.00280767,
    0.00280767,
    0.00672968,
    0.00672968,
    0.06842595,
    0.06842595,
    0.04147135,
    0.04147135,
   -0.05190823,
   -0.05190823,
    0.04144437,
    0.04144437,
   -0.01680658,
   -0.00099460,
   -0.00099460,
   -0.01175779,
   -0.01175779,
    0.01740404,
   -0.00533897,
   -0.00533897,
    0.00542184,
    0.00542184,
    0.00264603,
    0.00264603,
    0.01858930,
    0.01858930,
    0.00512310,
   -0.00971598,
   -0.00971598,
   -0.00509582,
   -0.00509582,
   -0.01358156,
    0.00367419,
   -0.00961479,
   -0.00961479,
   -0.00113508,
   -0.00113508,
    0.01029005,
    0.01029005,
   -0.03739058,
   -0.03739058,
   -0.00705804,
   -0.00705804,
    0.00741845,
    0.00741845,
    0.00418133,
    0.00418133,
   -0.00554620,
   -0.00554620,
   -0.02043781,
   -0.02043781,
   -0.02426849,
    0.00152690,
    0.00152690,
    0.00050652,
    0.00050652,
    0.00149716,
    0.00149716,
    0.02229684,
    0.02229684,
    0.07188395,
    0.07188395,
    0.01643349,
    0.01643349,
    0.01815678,
    0.01600059,
    0.01600059,
    0.00731700,
    0.00731700,
    0.02671159,
    0.02671159,
    0.01555363,
    0.01555363,
   -0.13026695,
   -0.13026695,
    0.07343892,
    0.07343892,
   -0.08867641,
   -0.08867641,
   -0.01179302,
    0.00172440,
    0.00172440,
   -0.00153101,
   -0.00153101,
    0.01975589,
    0.00411038,
    0.00411038,
    0.00300448,
    0.00300448,
   -0.00741363,
   -0.00741363,
    0.00347946,
    0.00347946,
   -0.00036795,
    0.00250419,
    0.00250419,
   -0.00168054,
   -0.00168054,
    0.00914122,
   -0.02025215,
    0.00928423,
    0.00928423,
   -0.00593406,
   -0.00593406,
   -0.01229921,
   -0.01229921,
    0.00834341,
    0.00834341,
    0.03750856,
    0.03750856,
   -0.04339958,
   -0.04339958,
    0.00459296,
    0.00459296,
    0.00363642,
    0.00363642,
    0.00987656,
    0.01434378,
    0.01434378,
    0.00072217,
    0.00072217,
   -0.00159170,
   -0.00159170,
   -0.03962068,
   -0.03962068,
   -0.01733743,
   -0.01733743,
    0.02663106,
    0.02663106,
   -0.00188350,
    0.00052211,
    0.00052211,
   -0.00089434,
   -0.00089434,
    0.00482125,
    0.00031285,
    0.00031285,
    0.00103126,
    0.00103126,
    0.00224530,
   -0.00366922,
   -0.00033319,
   -0.00033319,
   -0.00319144,
   -0.00319144,
    0.00447017,
    0.00447017,
   -0.00147806,
   -0.00147806,
   -0.00085493,
   -0.00085493,
   -0.01101426,
   -0.01101426,
   -0.00043148,
    0.00013592,
    0.00013592,
   -0.00019174,
   -0.00019174,
    0.00122345,
    0.00026701,
    0.00026701,
    0.00054541,
    0.00057269,
    0.00057269,
   -0.00009076,
    0.00002273,
    0.00002273,
   -0.00004075,
   -0.00004075
    ])