# iosdiff.py
#
# Copyright (C) Robert Franklin <rcf34@cam.ac.uk>



"""Cisco IOS configuration differences module.

This module compares two configurations for Cisco IOS devices and
produces a delta configuration.
"""



# --- imports ---



import re

import netaddr

from .configdiff import DiffConvert, DiffConfig, pathstr



# --- functions ---



def _print_list(l, prefix="", start=None, step=10):
    """This function 'prints' a list of strings 'l' by adding a prefix
    onto the start of each entry and numbering them.  It's used to
    render IP access-lists and prefix-lists.

    The 'prefix' is a fixed string and defaults to the empty string.
    This is correct for access-lists but could be set to 'ip prefix-list
    NAME' for a prefix-list.  If this contains text, it will most likely
    need to end with a space to separate it from the rest of the string.

    The 'start' and 'step' arguments control the numbering of the
    entries and are added next.  If a start value is not specified, it
    defaults to the step value; the step value defaults to 10.

    A space and the entry value is then added.

    The resulting 'printed' list is then returned to be added to the
    configuration.
    """

    if start is None:
        start = step

    # initialise the result list
    r = []

    n = start
    for i in l:
        r.append(prefix + str(n) + " " + i)
        n += step

    return r



def _isintphysical(i):
    """This function returns if a particular interface is physical or
    not.  The names it matches are those that the are used as the
    canonical ones in the parser.
    """

    return re.match("^(Mgmt|Eth|Fa|Gi|Te|Fo)\d", i)



# --- converter classes ---



# _cvts = []
#
# This is a list of converter classes to be added to the
# CiscoIOSConfigDiff object by the _add_converters() method.  Each
# class will be instantiated and added.

_cvts = []



# SYSTEM



class _Cvt_Hostname(DiffConvert):
    cmd = "hostname",

    def remove(self, old, rem, args):
        return "no hostname"

    def update(self, new, upd, args):
        return "hostname " + upd

_cvts.append(_Cvt_Hostname)



# INTERFACE ...



class _Cvt_Int(DiffConvert):
    cmd = "interface", None

    def remove(self, old, rem, args):
        name, = args

        # only remove the interface if it's not physical
        if _isintphysical(name):
            return

        # remove if interface no longer exists, or exists flag has gone
        if (not rem) or ("exists" in rem):
            return "no interface " + name

    def update(self, new, upd, args):
        name, = args
        if "exists" in upd:
            return "interface " + name

_cvts.append(_Cvt_Int)


class _CvtContext_Int(DiffConvert):
    "Abstract class for interface context converters to subclass."
    context = "interface", None


# we put the 'interface / shutdown' at the start to shut it down before
# we do any [re]configuration

class _Cvt_Int_Shutdown(_CvtContext_Int):
    cmd = "shutdown",
    name = "shutdown"

    def update(self, new, upd, args):
        name, = args
        return "interface " + name, " " + ("" if upd else "no " ) + "shutdown"

_cvts.append(_Cvt_Int_Shutdown)


class _Cvt_Int_CDPEna(_CvtContext_Int):
    cmd = "cdp-enable",

    def remove(self, old, rem, args):
        name, = args
        # if the 'cdp enable' option is not present, that doesn't mean
        # it's disabled but just that it's not specified, so we assume
        # the default is for it to be enabled
        return "interface " + name, " cdp enable"

    def update(self, new, upd, args):
        name, = args
        return ("interface " + name,
                " " + ("" if new else "no ") + "cdp enable")

_cvts.append(_Cvt_Int_CDPEna)


class _Cvt_Int_ChnGrp(_CvtContext_Int):
    cmd = "channel-group",

    def remove(self, old, rem, args):
        name, = args
        return "interface " + name, " no channel-group"

    def update(self, new, upd, args):
        name, = args
        id_, mode = new
        return ("interface " + name, " channel-group %d%s"
                    % (id_, mode if mode else ""))

_cvts.append(_Cvt_Int_ChnGrp)


class _Cvt_Int_Desc(_CvtContext_Int):
    cmd = "description",

    def remove(self, old, rem, args):
        name, = args
        return "interface " + name, " no description"

    def update(self, new, upd, args):
        name, = args
        return "interface " + name, " description " + upd

_cvts.append(_Cvt_Int_Desc)


class _Cvt_Int_Encap(_CvtContext_Int):
    cmd = "encapsulation",

    def remove(self, old, rem, args):
        name, = args
        return "interface " + name, " no encapsulation " + old

    def update(self, new, upd, args):
        name, = args
        return "interface " + name, " encapsulation " + upd

_cvts.append(_Cvt_Int_Encap)


class _Cvt_Int_IPAccGrp(_CvtContext_Int):
    cmd = "ip-access-group",

    def remove(self, old, rem, args):
        name, = args
        l = ["interface " + name]

        # if there are differences, just removed the specific group(s);
        # if the whole block is gone, remove all groups there before
        for dir in sorted(rem or old):
            l.append(" no ip access-group " + dir)

        return l

    def update(self, new, upd, args):
        name, = args
        l = ["interface " + name]
        for dir in sorted(upd):
            l.append(" ip access-group %s %s" % (upd[dir], dir))
        return l

_cvts.append(_Cvt_Int_IPAccGrp)


class _Cvt_Int_IPAddr(_CvtContext_Int):
    cmd = "ip-address",

    def remove(self, old, rem, args):
        name, = args
        return "interface " + name, " no ip address"

    def update(self, new, upd, args):
        name, = args
        return "interface " + name, " ip address " + upd

_cvts.append(_Cvt_Int_IPAddr)


class _Cvt_Int_IPAddrSec(_CvtContext_Int):
    cmd = "ip-address-secondary",

    def remove(self, old, rem, args):
        name, = args
        l = ["interface " + name]
        for addr in sorted(rem or old):
            l.append(" no ip address %s secondary" % addr)
        return l

    def update(self, new, upd, args):
        name, = args
        l = ["interface %s" % name]
        for addr in sorted(upd):
            l.append(" ip address %s secondary" % addr)
        return l

_cvts.append(_Cvt_Int_IPAddrSec)


class _Cvt_Int_IPHlp(_CvtContext_Int):
    cmd = "ip-helper-address",

    def remove(self, old, rem, args):
        name, = args
        l = ["interface %s" % name]
        for addr in sorted(rem or old):
            l.append(" no ip helper-address " + addr)
        return l

    def update(self, new, upd, args):
        name, = args
        l = ["interface %s" % name]
        l.extend([ (" ip helper-address " + a) for a in upd ])
        return l

_cvts.append(_Cvt_Int_IPHlp)


class _Cvt_Int_IP6Addr(_CvtContext_Int):
    cmd = "ipv6-address",

    def remove(self, old, rem, args):
        name, = args
        l = ["interface %s" % name]
        for addr in sorted(rem or old):
            l.append(" no ipv6 address " + addr)
        return l

    def update(self, new, upd, args):
        name, = args
        l = ["interface %s" % name]
        for addr in sorted(upd):
            l.append(" ipv6 address " + addr)
        return l

_cvts.append(_Cvt_Int_IP6Addr)


class _Cvt_Int_IPIGMPVer(_CvtContext_Int):
    cmd = "ip-igmp-version",

    def remove(self, old, rem, args):
        name, = args
        return "interface " + name, " no ip igmp version"

    def update(self, new, upd, args):
        name, = args
        return "interface " + name, " ip igmp version " + upd

_cvts.append(_Cvt_Int_IPIGMPVer)


class _Cvt_Int_IPMcastBdry(_CvtContext_Int):
    cmd = "ip-multicast-boundary",

    def remove(self, old, rem, args):
        name, = args
        return "interface " + name, " no ip multicast boundary"

    def update(self, new, upd, args):
        name, = args
        return "interface " + name, " ip multicast boundary " + upd

_cvts.append(_Cvt_Int_IPMcastBdry)


class _Cvt_Int_IPPIMMode(_CvtContext_Int):
    cmd = "ip-pim", "mode"

    def remove(self, old, rem, args):
        name, = args
        return "interface " + name, " no ip pim " + old

    def update(self, new, upd, args):
        name, = args
        return "interface " + name, " ip pim " + upd

_cvts.append(_Cvt_Int_IPPIMMode)


class _Cvt_Int_IPPIMBSRBdr(_CvtContext_Int):
    cmd = "ip-pim", "bsr-border"

    def remove(self, old, rem, args):
        name, = args
        return "interface " + name, " no ip pim bsr-border"

    def update(self, new, upd, args):
        name, = args
        return "interface " + name, " ip pim bsr-border"

_cvts.append(_Cvt_Int_IPPIMBSRBdr)


class _Cvt_Int_IPProxyARP(_CvtContext_Int):
    cmd = "ip-proxy-arp",

    def update(self, new, upd, args):
        name, = args
        return ("interface " + name,
                " " + ("" if new else "no ") + "ip proxy-arp")

_cvts.append(_Cvt_Int_IPProxyARP)


class _Cvt_Int_ServPol(_CvtContext_Int):
    cmd = "service-policy",

    def remove(self, old, rem, args):
        name, = args
        l = ["interface %s" % name]
        for policy in sorted(rem or old):
            l.append(" no service-policy " + policy)
        return l

    def update(self, new, upd, args):
        name, = args
        l = ["interface %s" % name]
        for policy in sorted(upd):
            l.append(" service-policy " + policy)
        return l

_cvts.append(_Cvt_Int_ServPol)


class _Cvt_Int_StandbyIP(_CvtContext_Int):
    cmd = "standby", "group", None, "ip"

    def remove(self, old, rem, args):
        name, grp = args
        return "interface " + name, " no standby %d ip" % grp

    def update(self, new, upd, args):
        name, grp = args
        return "interface " + name, " standby %d ip %s" % (grp, upd)

_cvts.append(_Cvt_Int_StandbyIP)


class _Cvt_Int_StandbyIPSec(_CvtContext_Int):
    cmd = "standby", "group", None, "ip-secondary"

    def remove(self, old, rem, args):
        name, grp = args
        l = ["interface " + name]
        for addr in sorted(rem or old):
            l.append(" no standby %d ip %s secondary" % (grp, addr))
        return l

    def update(self, new, upd, args):
        name, grp = args
        l = ["interface " + name]
        for addr in sorted(upd):
            l.append(" standby %d ip %s secondary" % (grp, addr))
        return l

_cvts.append(_Cvt_Int_StandbyIPSec)


class _Cvt_Int_StandbyIP6(_CvtContext_Int):
    cmd = "standby", "group", None, "ipv6"

    def remove(self, old, rem, args):
        name, grp = args
        l = ["interface " + name]
        for addr in sorted(rem or old):
            l.append(" no standby %d ipv6 %s" % (grp, addr))
        return l

    def update(self, new, upd, args):
        name, grp = args
        l = ["interface " + name]
        for addr in sorted(upd):
            l.append(" standby %d ipv6 %s" % (grp, addr))
        return l

_cvts.append(_Cvt_Int_StandbyIP6)


class _Cvt_Int_StandbyPreempt(_CvtContext_Int):
    cmd = "standby", "group", None, "preempt"

    def remove(self, old, rem, args):
        name, grp = args
        return "interface " + name, " no standby %d preempt" % grp

    def update(self, new, upd, args):
        name, grp = args
        return "interface " + name, " standby %d preempt" % grp

_cvts.append(_Cvt_Int_StandbyPreempt)


class _Cvt_Int_StandbyPri(_CvtContext_Int):
    cmd = "standby", "group", None, "priority"

    def remove(self, old, rem, args):
        name, grp = args
        return "interface " + name, " no standby %d priority" % grp

    def update(self, new, upd, args):
        name, grp = args
        return "interface " + name, " standby %d priority %d" % (grp, upd)

_cvts.append(_Cvt_Int_StandbyPri)


class _Cvt_Int_StandbyTimers(_CvtContext_Int):
    cmd = "standby", "group", None, "timers"

    def remove(self, old, rem, args):
        name, grp = args
        return "interface " + name, " no standby %d timers" % grp

    def update(self, new, upd, args):
        name, grp = args
        return "interface " + name, " standby %d timers %s" % (grp, upd)

_cvts.append(_Cvt_Int_StandbyTimers)


class _Cvt_Int_StandbyTrk(_CvtContext_Int):
    cmd = "standby", "group", None, "track"

    def remove(self, old, rem, args):
        name, grp = args
        l = ["interface " + name]
        for trk in sorted(rem or old):
            l.append(" no standby %d track %s" % (grp, trk))
        return l

    def update(self, new, upd, args):
        name, grp = args
        l = ["interface " + name]
        for trk in sorted(upd):
            l.append(" standby %d track %s" % (grp, trk)
                     + (" %s" % upd[trk] if upd[trk] else ""))
        return l

_cvts.append(_Cvt_Int_StandbyTrk)


class _Cvt_Int_StandbyVer(_CvtContext_Int):
    cmd = "standby", "version"

    def remove(self, old, rem, args):
        name, = args
        return "interface " + name, " no standby version"

    def update(self, new, upd, args):
        name, = args
        return "interface " + name, " standby version %d" % upd

_cvts.append(_Cvt_Int_StandbyVer)


class _Cvt_Int_SwPortTrkNtv(_CvtContext_Int):
    # we just match the interface as we need to look inside it to see if
    # the interface is part of a channel group
    cmd = tuple()
    ext = "switchport-trunk-native",

    def remove(self, old, rem, args):
        name, = args

        # if this interface is in a port-channel, we do all changes
        # there, so skip this
        if "channel-group" in old:
            return None

        return "interface " + name, " no switchport trunk native vlan"

    def update(self, new, upd, args):
        name, = args

        # if this interface is in a port-channel, we do all changes
        # there, so skip this
        if "channel-group" in new:
            return None

        return ("interface " + name,
                " switchport trunk native vlan %d" % self.get_ext(upd))

_cvts.append(_Cvt_Int_SwPortTrkNtv)


class _Cvt_Int_SwPortTrkAlw(_CvtContext_Int):
    # we just match the interface as we need to look inside it to see if
    # the interface is part of a channel group
    cmd = tuple()
    ext = "switchport-trunk-allow",

    def remove(self, old, rem, args):
        name, = args

        # if this interface is in a port-channel, we do all changes
        # there, so skip this
        if "channel-group" in old:
            return None

        l = ["interface " + name]
        for tag in sorted(self.get_ext(rem)):
            l.append(" switchport trunk allowed vlan remove %d" % tag)
        return l

    def update(self, new, upd, args):
        name, = args

        # if this interface is in a port-channel, we do all changes
        # there, so skip this
        if "channel-group" in new:
            return None

        l = ["interface " + name]
        for tag in sorted(self.get_ext(upd)):
            l.append(" switchport trunk allowed vlan add %d" % tag)
        return l

_cvts.append(_Cvt_Int_SwPortTrkAlw)


class _Cvt_Int_VRFFwd(_CvtContext_Int):
    cmd = "vrf-forwarding",

    def remove(self, old, rem, args):
        name, = args
        return "interface " + name, " no vrf forwarding"

    def update(self, new, upd, args):
        name, = args
        return "interface " + name, " vrf forwarding " + upd
        # TODO: need to find some way to trigger reapplication of IP
        # information (address, HSRP, etc.)

_cvts.append(_Cvt_Int_VRFFwd)


class _Cvt_Int_XConn(_CvtContext_Int):
    cmd = "xconnect",

    def remove(self, old, rem, args):
        name, = args
        return "interface " + name, " no xconnect"

    def update(self, new, upd, args):
        name, = args
        return "interface " + name, " xconnect " + upd

_cvts.append(_Cvt_Int_XConn)


# we put the 'interface / no shutdown' at the end to only enable the
# interface once it's been correctly [re]configured

class _Cvt_Int_NoShutdown(_CvtContext_Int):
    cmd = "shutdown",
    name = "no-shutdown"

    def remove(self, old, rem, args):
        name, = args
        return "interface " + name, " no shutdown"

_cvts.append(_Cvt_Int_NoShutdown)



# IP[V6] ACCESS-LIST ...



class _Cvt_IPACL_Std(DiffConvert):
    cmd = "ip-access-list-standard", None

    def remove(self, old, rem, args):
        name, = args
        return (["no ip access-list standard " + name]
                + _print_list(old, "!- "))

    def update(self, new, upd, args):
        name, = args
        return ["ip access-list standard " + name] + _print_list(new, " ")

_cvts.append(_Cvt_IPACL_Std)


class _Cvt_IPACL_Ext(DiffConvert):
    cmd = "ip-access-list-extended", None

    def remove(self, old, rem, args):
        # include the old list commented out, for comparison
        name, = args
        return (["no ip access-list extended " + name]
                + _print_list(old, "!- "))

    def update(self, new, upd, args):
        name, = args
        return ["ip access-list extended " + name] + _print_list(upd, " ")

_cvts.append(_Cvt_IPACL_Ext)


class _Cvt_IPV6ACL_Ext(DiffConvert):
    cmd = "ipv6-access-list", None

    def remove(self, old, rem, args):
        name, = args
        return (["no ipv6 access-list " + name]
                + _print_list(old, "!- "))

    def update(self, new, upd, args):
        name, = args
        return ["ipv6 access-list " + name] + _print_list(upd, " ")

_cvts.append(_Cvt_IPV6ACL_Ext)



# IP[V6] PREFIX-LIST ...



class _Cvt_IPPfxList(DiffConvert):
    cmd = "ip-prefix-list", None

    def remove(self, old, rem, args):
        # include the old list commented out, for comparison
        name, = args
        return (["no ip prefix-list " + name]
                + _print_list(old, "!- ", step=5))

    def update(self, new, upd, args):
        name, = args
        return _print_list(upd, "ip prefix-list %s seq " % name, step=5)

_cvts.append(_Cvt_IPPfxList)


class _Cvt_IPV6PfxList(DiffConvert):
    cmd = "ipv6-prefix-list", None

    def remove(self, old, rem, args):
        # include the old list commented out, for comparison
        name, = args
        return (["no ipv6 prefix-list " + name]
                + _print_list(old, "!- ", step=5))

    def update(self, new, upd, args):
        name, = args
        return _print_list(upd, "ipv6 prefix-list %s seq " % name, step=5)

_cvts.append(_Cvt_IPV6PfxList)



# IP[V6] ROUTE ...



class _Cvt_IPRoute(DiffConvert):
    cmd = "ip-route", None

    def remove(self, old, rem, args):
        route, = args
        return "no ip route " + route

    def update(self, new, upd, args):
        route, = args
        return "ip route " + route

_cvts.append(_Cvt_IPRoute)


class _Cvt_IP6Route(DiffConvert):
    cmd = "ipv6-route", None

    def remove(self, old, rem, args):
        route, = args
        return "no ipv6 route " + route

    def update(self, new, upd, args):
        route, = args
        return "ipv6 route " + route

_cvts.append(_Cvt_IP6Route)



# [NO] SPANNING-TREE ...



class _Cvt_NoSTP(DiffConvert):
    cmd = "no-spanning-tree-vlan", None

    def remove(self, old, rem, args):
        tag, = args
        # removing 'no spanning-tree' enables spanning-tree
        return "spanning-tree vlan " + str(tag)

    def update(self, new, upd, args):
        tag, = args
        # adding 'no spanning-tree' disables spanning-tree
        return "no spanning-tree vlan " + str(tag)

_cvts.append(_Cvt_NoSTP)


class _Cvt_STP_Pri(DiffConvert):
    cmd = "spanning-tree-vlan-priority", None

    def remove(self, old, rem, args):
        tag, = args
        return "no spanning-tree vlan %d priority" % tag

    def update(self, new, upd, args):
        tag, = args
        return "spanning-tree vlan %d priority %d" % (tag, upd)

_cvts.append(_Cvt_STP_Pri)



# TRACK ...



class _Cvt_Track(DiffConvert):
    cmd = "track", None
    ext = "criterion",

    def remove(self, old, rem, args):
        obj, = args
        return "no track %d" % obj

    def update(self, new, upd, args):
        obj, = args
        return "track %d %s" % (obj, upd["criterion"])

_cvts.append(_Cvt_Track)


class _CvtContext_Track(DiffConvert):
    context = "track", None


class _Cvt_Track_Delay(_CvtContext_Track):
    cmd = "delay",

    def remove(self, old, rem, args):
        obj, = args
        return "track %d" % obj, " no delay"

    def update(self, new, upd, args):
        obj, = args
        return "track %d" % obj, " delay " + upd

_cvts.append(_Cvt_Track_Delay)


class _Cvt_Track_IPVRF(_CvtContext_Track):
    cmd = "ip-vrf",

    def remove(self, old, rem, args):
        obj, = args
        return "track %d" % obj, " no ip vrf"

    def update(self, new, upd, args):
        obj, = args
        return "track %d" % obj, " ip vrf " + upd

_cvts.append(_Cvt_Track_IPVRF)


class _Cvt_Track_Obj(_CvtContext_Track):
    context = "track", None
    cmd = "object",

    def remove(self, old, rem, args):
        obj, = args
        return "track %d" % obj, " no object " + rem

    def update(self, new, upd, args):
        obj, = args
        return "track %d" % obj, " object " + upd

_cvts.append(_Cvt_Track_Obj)



# VLAN ...



class _Cvt_Vlan(DiffConvert):
    cmd = "vlan", None

    def remove(self, old, rem, args):
        tag, = args
        return "no vlan %d" % tag

    def update(self, new, upd, args):
        tag, = args
        if upd and ("exists" in upd):
            return "vlan %d" % tag

_cvts.append(_Cvt_Vlan)


class _Cvt_Vlan_Name(DiffConvert):
    context = "vlan", None
    cmd = "name",

    def remove(self, old, rem, args):
        tag, = args
        return "vlan %d" % tag, " no name"

    def update(self, new, upd, args):
        tag, = args
        return "vlan %d" % tag, " name " + upd

_cvts.append(_Cvt_Vlan_Name)



# --- context parser ---



class CiscoIOSDiffConfig(DiffConfig):
    """This class is used to compare two IOS configuration files and
    generate a configuration file to transform one into the other.
    """


    def _add_converters(self):
        "This method adds the converters for Cisco IOS."

        for cvt_class in _cvts:
            self._add_converter(cvt_class())


    def _explain_comment(self, path):
        """This method overrides the empty inherited one to return a
        Cisco IOS comment giving the matched path.
        """

        return "! => " + pathstr(path)


    def _changes_end(self):
        """This method overrides the empty inherited one to return a
        single line saying 'end'.
        """

        return ["end"]


    def init_rules_tree(self):
        """This method extends the inherited one to add some rules to
        the tree for the default CoPP (Control Plane Policing) IPv4
        extended and IPv6 ACLs.
        """

        super().init_rules_tree()

        self._rules_tree.update( {
            "ios-default": {
                "ip-access-list-extended": {
                    "acl-copp-match-igmp": {},
                    "acl-copp-match-pim-data": {},
                },
                "ipv6-access-list": {
                    "acl-copp-match-mld": {},
                    "acl-copp-match-ndv6": {},
                    "acl-copp-match-ndv6hl": {},
                    "acl-copp-match-pimv6-data": {},
                },
            },
        } )


    def init_rules_active(self):
        """This method extends the inherited one to add in some active
        rules to exclude the portions of the rules tree set up in
        init_rules_tree().
        """

        super().init_rules_active()

        self._rules_active.append( (False, ("ios-default", ) ) )
