"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LegendPosition = exports.Color = exports.Shading = exports.SingleValueWidget = exports.GraphWidget = exports.AlarmWidget = void 0;
const cdk = require("@aws-cdk/core");
const rendering_1 = require("./private/rendering");
const widget_1 = require("./widget");
/**
 * Display the metric associated with an alarm, including the alarm line
 */
class AlarmWidget extends widget_1.ConcreteWidget {
    constructor(props) {
        super(props.width || 6, props.height || 6);
        this.props = props;
    }
    toJson() {
        return [{
                type: 'metric',
                width: this.width,
                height: this.height,
                x: this.x,
                y: this.y,
                properties: {
                    view: 'timeSeries',
                    title: this.props.title,
                    region: this.props.region || cdk.Aws.REGION,
                    annotations: {
                        alarms: [this.props.alarm.alarmArn],
                    },
                    yAxis: {
                        left: this.props.leftYAxis !== undefined ? this.props.leftYAxis : undefined,
                    },
                },
            }];
    }
}
exports.AlarmWidget = AlarmWidget;
/**
 * A dashboard widget that displays metrics
 */
class GraphWidget extends widget_1.ConcreteWidget {
    constructor(props) {
        super(props.width || 6, props.height || 6);
        this.props = props;
    }
    toJson() {
        const horizontalAnnotations = [
            ...(this.props.leftAnnotations || []).map(mapAnnotation('left')),
            ...(this.props.rightAnnotations || []).map(mapAnnotation('right')),
        ];
        const metrics = rendering_1.allMetricsGraphJson(this.props.left || [], this.props.right || []);
        return [{
                type: 'metric',
                width: this.width,
                height: this.height,
                x: this.x,
                y: this.y,
                properties: {
                    view: 'timeSeries',
                    title: this.props.title,
                    region: this.props.region || cdk.Aws.REGION,
                    stacked: this.props.stacked,
                    metrics: metrics.length > 0 ? metrics : undefined,
                    annotations: horizontalAnnotations.length > 0 ? { horizontal: horizontalAnnotations } : undefined,
                    yAxis: {
                        left: this.props.leftYAxis !== undefined ? this.props.leftYAxis : undefined,
                        right: this.props.rightYAxis !== undefined ? this.props.rightYAxis : undefined,
                    },
                    legend: this.props.legendPosition !== undefined ? { position: this.props.legendPosition } : undefined,
                    liveData: this.props.liveData,
                },
            }];
    }
}
exports.GraphWidget = GraphWidget;
/**
 * A dashboard widget that displays the most recent value for every metric
 */
class SingleValueWidget extends widget_1.ConcreteWidget {
    constructor(props) {
        super(props.width || 6, props.height || 3);
        this.props = props;
    }
    toJson() {
        return [{
                type: 'metric',
                width: this.width,
                height: this.height,
                x: this.x,
                y: this.y,
                properties: {
                    view: 'singleValue',
                    title: this.props.title,
                    region: this.props.region || cdk.Aws.REGION,
                    metrics: rendering_1.allMetricsGraphJson(this.props.metrics, []),
                    setPeriodToTimeRange: this.props.setPeriodToTimeRange,
                },
            }];
    }
}
exports.SingleValueWidget = SingleValueWidget;
/**
 * Fill shading options that will be used with an annotation
 */
var Shading;
(function (Shading) {
    /**
     * Don't add shading
     */
    Shading["NONE"] = "none";
    /**
     * Add shading above the annotation
     */
    Shading["ABOVE"] = "above";
    /**
     * Add shading below the annotation
     */
    Shading["BELOW"] = "below";
})(Shading = exports.Shading || (exports.Shading = {}));
/**
 * A set of standard colours that can be used in annotations in a GraphWidget.
 */
class Color {
}
exports.Color = Color;
/** blue - hex #1f77b4 */
Color.BLUE = '#1f77b4';
/** brown - hex #8c564b */
Color.BROWN = '#8c564b';
/** green - hex #2ca02c */
Color.GREEN = '#2ca02c';
/** grey - hex #7f7f7f */
Color.GREY = '#7f7f7f';
/** orange - hex #ff7f0e */
Color.ORANGE = '#ff7f0e';
/** pink - hex #e377c2 */
Color.PINK = '#e377c2';
/** purple - hex #9467bd */
Color.PURPLE = '#9467bd';
/** red - hex #d62728 */
Color.RED = '#d62728';
/**
 * The position of the legend on a GraphWidget.
 */
var LegendPosition;
(function (LegendPosition) {
    /**
     * Legend appears below the graph (default).
     */
    LegendPosition["BOTTOM"] = "bottom";
    /**
     * Add shading above the annotation
     */
    LegendPosition["RIGHT"] = "right";
    /**
     * Add shading below the annotation
     */
    LegendPosition["HIDDEN"] = "hidden";
})(LegendPosition = exports.LegendPosition || (exports.LegendPosition = {}));
function mapAnnotation(yAxis) {
    return (a) => {
        return { ...a, yAxis };
    };
}
//# sourceMappingURL=data:application/json;base64,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