# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['rand',
 'rand.providers',
 'rand.providers.contribs',
 'rand.providers.ds',
 'rand.providers.en',
 'rand.providers.url']

package_data = \
{'': ['*']}

install_requires = \
['peewee>=3.13.3,<4.0.0', 'pytest-cov>=2.0,<3.0', 'requests>=2.23.0,<3.0.0']

setup_kwargs = {
    'name': 'rand',
    'version': '0.3.0',
    'description': 'Generate String from regex pattern',
    'long_description': 'rand\n====\n\n[![Travis (.org)](https://img.shields.io/travis/kororo/rand)](https://pypi.python.org/project/rand/)\n[![Coverage Status](https://coveralls.io/repos/github/kororo/rand/badge.svg?branch=master)](https://coveralls.io/github/kororo/rand?branch=master)\n[![MIT license](https://img.shields.io/badge/License-MIT-blue.svg)](https://lbesson.mit-license.org/)\n[![PyPI pyversions](https://img.shields.io/pypi/pyversions/rand.svg)](https://pypi.python.org/project/rand/)\n[![PyPI - Downloads](https://img.shields.io/pypi/dm/rand)](https://pypi.python.org/project/rand/)\n\n---\n\nRandom generated String from regex pattern\n\n# WARNING\n\nThe library **rand** is still in working-in-progress. It is subject to high possibility of API changes. Would appreciate feedback, suggestions or help.\n\n# Why?\n\nThere are lot of existing projects similar to **rand**, they are powerful and have similar goals and results. However most of them are old projects/non-maintained and non-MIT licenses.\n\nThis is a good opportunity for **rand** to be the library to help generate random data for any projects and gather all other existing libraries to be the main driver.\n\n\n# Install\n\nUse pip or clone this repository and execute the setup.py file.\n\n```shell script\n$ pip install rand\n```\n\n# Usages\n\nBasic usage **rand** examples\n\n```python\n# import module\nfrom rand import Rand\n\n# initialise object\nrnd = Rand()\n\n# generate pattern literal\nrnd.gen(\'koro\') # [\'koro\']\nrnd.gen(\'28\') # [\'28\']\nrnd.gen(\'a-z\') # [\'a-z\']\n\n# generate pattern any\nrnd.gen(\'.\') # any char in string.printable\n\n# generate pattern branch\nrnd.gen(\'ko|ro\') # either [\'ko\'] or [\'ro\']\nrnd.gen(\'ko|ro|ro\') # either [\'ko\'] or [\'ro\']\n\n# generate pattern in\nrnd.gen(\'[kororo]\') # either [\'k\'] or [\'o\'] or [\'r\']\nrnd.gen(\'k[o]r[o]r[o]\') # [\'kororo\']\n\n# generate pattern repeat\nrnd.gen(\'r{2,8}\') # char r in length between 2 to 8 times\n\n# generate pattern range\nrnd.gen(\'[a-z]\') # char between a to z\n\n# generate pattern subpattern\nrnd.gen(\'(ro)\') # [\'ro\']\n```\n\nProviders\n---------\n\nThe library **rand** at core only provide random generators based on regex. Providers are built to allow extensions for rand.\n\n## Built-in Providers\n\nThere are a few built-in providers inside **rand**\n\n### EN Provider\n\nThis library covers most usage around English requirements.\n\n```python\nfrom rand import Rand\n\n\nrnd = Rand()\nrnd.gen(\'(:en_vocal:)\') # char either a, i, u, e, o\n```\n\n### Dataset Provider\n\nThis library helps on getting data from dataset such as Python object or Database with [peewee](https://github.com/coleifer/peewee).\n\n```python\nfrom rand import Rand\nfrom rand.providers.ds import RandDatasetBaseProvider, ListDatasetTarget\n\n\n# example using dict of list\ndb = {\'names\': [\'test1\', \'test1\'], \'cities\': [\'test2\', \'test2\']}\nds = RandDatasetBaseProvider(prefix=\'ds\', target=ListDatasetTarget(db=db))\nrnd = Rand()\nrnd.register_provider(ds)\nrnd.gen(\'(:ds_get:)\', [\'names\'])  # [\'test1\']\nrnd.gen(\'(:ds_get:)\', [\'cities\']) # [\'test2\']\n# or, magic getattr\nrnd.gen(\'(:ds_get_names:)-(:ds_get_cities:)\') # [\'test1-test2\']\n\n# example of database using peewee\nfrom peewee import Proxy\nfrom playhouse.sqlite_ext import CSqliteExtDatabase\nfrom rand.providers.ds import RandDatasetBaseProvider, DBDatasetTarget\ndb = Proxy()\n# ensure to have table with name "names", contains column at least (id, name)\ndb.initialize(CSqliteExtDatabase(\':memory:\', bloomfilter=True))\nds = RandDatasetBaseProvider(prefix=\'ds\', target=DBDatasetTarget(db=db))\nrnd = Rand()\nrnd.register_provider(ds)\nrnd.gen(\'(:ds_get:)\', [\'names\']) # [\'test\']\ndb.close()\n```\n\n## Integration Providers\n\nThe library *rand* also has integration with existing projects such as Faker. Ensure you have faker library installed.\n\n### [Faker](https://github.com/joke2k/faker)\n\nThere is super basic integration with Faker for now, soon will be more implemented.\n\n```shell script\n# ensure you have Faker installed\npip install Faker\n```\n\n```python\nfrom rand import Rand\n\n\nrnd = Rand()\nrnd.gen(\'(:faker_hexify:)\') # abc\n```\n\n## Custom Parse\n\nExtending rand is simple, there are register_parse and register_provider, both of this has special level of customisation.\n\n### Simple Register Parse\n\nThis is example of creating custom parse in simple way.\n\n```python\nfrom rand import Rand\n\n# init rand class\nrnd = Rand()\n\n# the definition\ndef parse_test1(pattern, opts=None):\n    return \'test1\'\n\n# the registration\nrnd.register_parse(name=\'test1\', fn=parse_test1)\n# test it\nrnd.gen(\'(:test1:)\')\n```\n\n### Decorator Wrapper Register Parse\n\nThis is faster way with decorator pattern to register custom parse.\n\n```python\nfrom rand import Rand\n\n# init rand class\nrnd = Rand()\n\n# the definition\n@rnd.register_parse_wrapper(name=\'test1\')\ndef parse_test1(pattern, opts=None):\n    return \'test1\'\n\n# test it\nrnd.gen(\'(:test1:)\')\n```\n\n## Custom Providers\n\nCustom provider is upper level customisation in **rand**, it behaves quite differently than custom parse  \n\n### Simple Register Provider\n\nBelow is sample code on how to integrate an existing class definition (TestProxy) to Rand.\n\n```python\nfrom rand.providers.base import RandBaseProvider\nfrom rand import Rand\n\nclass TestProvider(RandBaseProvider):\n    def _parse_fn(self, pattern, opts=None):\n        return \'test\'\n\n    def parse(self, name: str, pattern: any, opts: dict):\n        # name always start with _parse_[PREFIX], normalise first\n        parsed_name = self.get_parse_name(name)\n        if parsed_name:\n            return self._parse_fn(pattern, opts)\n        return None\n\n# init rand class\nrnd = Rand()\nrnd.register_provider(TestProvider(prefix=\'test_fn\'))\nassert rnd.gen(\'(:test_fn:)\') == \'test\'\n```\n\n```python\nfrom rand import Rand\nfrom rand.providers.base import RandProxyBaseProvider\n\n# class definition\nclass TestProxy:\n    # simple function definition to return args values\n    def target(self, arg1=\'def1\', arg2=\'def2\'):\n        return \'%s-%s\' % (arg1, arg2)\n\n# init rand class\nrnd = Rand()\n\n# create proxy provider helper and register to rand\ntest_proxy = RandProxyBaseProvider(prefix=\'test\', target=TestProxy())\nrnd.register_provider(test_proxy)\n\n# test\nprint(rnd.gen(\'(:test_target:)\')) # [\'def1-def2\']\nprint(rnd.gen(\'(:test_target:)\', [\'ok1\'])) # [\'ok1-def2\']\nprint(rnd.gen(\'(:test_target:)\', [\'ok1\', \'ok2\'])) # [\'ok1-def2\']\nprint(rnd.gen(\'(:test_target:)\', [[\'ok1\', \'ok2\']])) # [\'ok1-ok2\']\nprint(rnd.gen(\'(:test_target:)\', [[\'ok1\', \'ok2\'], \'ok3\'])) # [\'ok1-ok2\']\nprint(rnd.gen(\'(:test_target:)\', [{\'arg1\': \'ok1\'}])) # [\'ok1-def2\']\nprint(rnd.gen(\'(:test_target:)\', [{\'arg1\': \'ok1\', \'arg2\': \'ok2\'}])) # [\'ok1-ok2\']\n```\n\n### Decorator Wrapper Register Provider\n\nThis is faster way with decorator pattern to register custom provider.\n\n```python\nfrom rand import Rand\n\n# init rand class\nrnd = Rand()\n\n@rnd.register_provider_fn_wrapper(prefix=\'test2\')\ndef parse_test2(pattern, opts=None):\n    return \'test2\'\n\nprint(rnd.gen(\'(:test2:)\'))  # \'test2\'\n```\n\n## Different Between Custom Parser and Provider\n\nThe way **rand** works, register_parse taps into the core of **rand**, following the token from sre_parse, when the definition is returned,\nit is possible to return sre_parse token with existing token name or custom token which points to the custom definition.\n\n```python\nfrom rand import Rand\n\n# init rand class\nrnd = Rand()\n\n@rnd.register_parse_wrapper(name=\'test1\')\ndef parse_test1(pattern, opts=None):\n    return \'test1\'\n\n\n@rnd.register_parse_wrapper(name=\'test2\')\ndef parse_test2(pattern, opts=None):\n    return rnd.sre_parse_compile_parse(\'(:test1:)\')\n\n\nprint(rnd.gen(\'(:test2:)\'))  # \'test1\'\n```\n\n\n\n\n\n\n# Test\n\nRun test by installing packages and run tox\n\n```shell script\n$ pip install poetry tox\n$ tox\n$ tox -e py36 -- tests/test_ds.py\n```\n\nFor hot-reload development coding\n```shell script\n$ npm i -g nodemon\n$ nodemon -w rand --exec python -c "from rand import Rand"\n```\n\n# Help?\n\nAny feedback, suggestions and integration with 3rd-party libraries can be added using PR or create issues if needed helps. \n\n# Similar Projects\n\nList of projects similar to **rand**:\n- [exrex](https://github.com/asciimoo/exrex): Irregular methods on regular expressions\n- [xeger](https://github.com/crdoconnor/xeger): Library to generate random strings from regular expressions\n- [strgen](https://github.com/paul-wolf/strgen): A Python module for a template language that generates randomized data\n\n# Acknowdlge Projects\n\nList of projects that **rand** depends on:\n- [peewee](https://github.com/coleifer/peewee): a small, expressive orm -- supports postgresql, mysql and sqlite\n- [pytest](https://github.com/pytest-dev/pytest/): The pytest framework makes it easy to write small tests, yet scales to support complex functional testing\n- [coverage](https://github.com/nedbat/coveragepy): Code coverage measurement for Python\n- [pytest-cov](https://github.com/pytest-dev/pytest-cov): Coverage plugin for pytest\n',
    'author': 'kororo',
    'author_email': None,
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/kororo/rand',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
