"""
This module exposes exceptions used by refinery.
"""
from __future__ import annotations

from refinery.lib.types import Collection, buf


class RefineryPartialResult(ValueError):
    """
    This exception indicates that a partial result is available.
    """
    def __init__(self, message: str, partial: buf, rest: buf | None = None):
        super().__init__(message)
        self.message = message
        self.partial = partial
        self.rest = rest

    def __str__(self):
        return self.message


class RefineryImportError(ImportError):
    """
    A special variant of the `ImportError` exception used for missing dependencies.
    """
    def __init__(self, info: str | None):
        super().__init__()
        self.info = info

    def __repr__(self):
        return F'{self.__class__.__name__}({self.info!r})'


class RefineryImportMissing(RefineryImportError):
    """
    A special variant of the `ImportError` exception which is raised when a dependency of a
    refinery unit is not installed in the current environment. The exception also provides hints
    about what package has to be installed in order to make that module available.
    """
    def __init__(self, missing: str, dependencies: Collection[str] = (), info: str | None = None):
        super().__init__(info)
        import shlex
        self.missing = missing
        self.install = shlex.join(dependencies)
        self.dependencies = dependencies or (missing,)

    def __str__(self):
        return self.missing

    def __repr__(self):
        return F'{self.__class__.__name__}({self.missing!r})'


class RefineryCriticalException(RuntimeError):
    """
    If this exception is thrown, processing of the entire input stream
    is aborted instead of just aborting the processing of the current
    chunk.
    """


class RefineryPotentialUserError(RuntimeError):
    """
    This exception can be raised by a unit to inform the user about a
    suspected input error.
    """


class RefineryException(RuntimeError):
    """
    This is an exception that was not generated by an external library.
    """
