"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.KeyPair = exports.KeyLength = void 0;
const cfn = require("@aws-cdk/aws-cloudformation");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/core");
const statement = require("cdk-iam-floyd");
const path = require("path");
const resourceType = 'Custom::EC2-Key-Pair';
const ID = `CFN::Resource::${resourceType}`;
const cleanID = ID.replace(/:+/g, '-');
const lambdaTimeout = 3; // minutes
var KeyLength;
(function (KeyLength) {
    KeyLength[KeyLength["L2048"] = 2048] = "L2048";
    KeyLength[KeyLength["L4096"] = 4096] = "L4096";
})(KeyLength = exports.KeyLength || (exports.KeyLength = {}));
/**
 * An EC2 Key Pair
 */
class KeyPair extends cdk.Construct {
    /**
     * Defines a new EC2 Key Pair. The private key will be stored in AWS Secrets Manager
     */
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        /**
         * ARN of the private key in AWS Secrets Manager
         */
        this.arn = '';
        /**
         * Name of the Key Pair
         */
        this.name = '';
        this.prefix = '';
        if (props.removePrivateKeyAfterDays &&
            (props.removePrivateKeyAfterDays < 0 ||
                (props.removePrivateKeyAfterDays > 0 &&
                    props.removePrivateKeyAfterDays < 7) ||
                props.removePrivateKeyAfterDays > 30)) {
            scope.node.addError(`Parameter removePrivateKeyAfterDays must be 0 or between 7 and 30. Got ${props.removePrivateKeyAfterDays}`);
        }
        const stack = cdk.Stack.of(this).stackName;
        this.prefix = props.resourcePrefix || stack;
        const fn = this.ensureLambda();
        this.tags = new cdk.TagManager(cdk.TagType.MAP, 'Custom::EC2-Key-Pair');
        this.tags.setTag('CreatedBy', ID);
        const key = new cfn.CustomResource(this, `EC2-Key-Pair-${props.name}`, {
            provider: cfn.CustomResourceProvider.fromLambda(fn),
            resourceType: resourceType,
            properties: {
                Name: props.name,
                Description: props.description || '',
                KeyLength: props.keyLength || KeyLength.L2048,
                Kms: ((_a = props.kms) === null || _a === void 0 ? void 0 : _a.keyArn) || 'alias/aws/secretsmanager',
                RemovePrivateKeyAfterDays: props.removePrivateKeyAfterDays || 0,
                SecretPrefix: props.secretPrefix || 'ec2-private-key/',
                StackName: stack,
                Tags: cdk.Lazy.anyValue({
                    produce: () => this.tags.renderTags(),
                }),
            },
        });
        if (typeof props.kms !== 'undefined') {
            props.kms.grantEncryptDecrypt(fn.role);
            key.node.addDependency(props.kms);
            key.node.addDependency(fn.role);
        }
        this.arn = key.getAttString('PrivateKeyARN');
        this.name = key.getAttString('KeyPairName');
    }
    ensureLambda() {
        const stack = cdk.Stack.of(this);
        const constructName = 'EC2-Key-Name-Manager-Lambda';
        const existing = stack.node.tryFindChild(constructName);
        if (existing) {
            return existing;
        }
        const policy = new iam.ManagedPolicy(stack, 'EC2-Key-Pair-Manager-Policy', {
            managedPolicyName: `${this.prefix}-${cleanID}`,
            description: `Used by Lambda ${cleanID}, which is a custom CFN resource, managing EC2 Key Pairs`,
            statements: [
                new statement.Ec2()
                    .allow()
                    .describeKeyPairs()
                    .createKeyPair()
                    .deleteKeyPair(),
                new statement.Secretsmanager().allow().listSecrets(),
                new statement.Secretsmanager()
                    .allow()
                    .createSecret()
                    .tagResource()
                    .ifAwsRequestTag('CreatedBy', ID),
                new statement.Secretsmanager()
                    .allow()
                    .allActions(/^(Describe|Delete|Put|Update)/)
                    .getResourcePolicy()
                    .restoreSecret()
                    .listSecretVersionIds()
                    .untagResource()
                    .ifResourceTag('CreatedBy', ID),
            ],
        });
        const role = new iam.Role(stack, 'EC2-Key-Pair-Manager-Role', {
            roleName: `${this.prefix}-${cleanID}`,
            description: `Used by Lambda ${cleanID}, which is a custom CFN resource, managing EC2 Key Pairs`,
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            managedPolicies: [
                policy,
                iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaBasicExecutionRole'),
            ],
        });
        const fn = new lambda.Function(stack, constructName, {
            functionName: `${this.prefix}-${cleanID}`,
            role: role,
            description: 'Custom CFN resource: Manage EC2 Key Pairs',
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(path.join(__dirname, '../lambda/code.zip')),
            timeout: cdk.Duration.minutes(lambdaTimeout),
        });
        return fn;
    }
    /**
     * Grants read access to the private key in AWS Secrets Manager
     */
    grantRead(grantee) {
        const result = iam.Grant.addToPrincipal({
            grantee,
            actions: [
                'secretsmanager:DescribeSecret',
                'secretsmanager:GetResourcePolicy',
                'secretsmanager:GetSecretValue',
                'secretsmanager:ListSecretVersionIds',
            ],
            resourceArns: [this.arn],
            scope: this,
        });
        return result;
    }
}
exports.KeyPair = KeyPair;
//# sourceMappingURL=data:application/json;base64,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