# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['opyapi',
 'opyapi.console',
 'opyapi.console.dto_generator',
 'opyapi.validators']

package_data = \
{'': ['*']}

install_requires = \
['cleo>=0.8.1,<0.9.0', 'pyyaml>=5.4.1,<6.0.0']

setup_kwargs = {
    'name': 'opyapi',
    'version': '1.1.0',
    'description': 'JsonSchem and OpenAPI tools for python.',
    'long_description': '# OpyAPI - JsonSchema and OpenAPI tools for python <br> [![CI](https://github.com/kodemore/opyapi/actions/workflows/main.yaml/badge.svg)](https://github.com/kodemore/opyapi/actions/workflows/main.yaml) [![Release](https://github.com/kodemore/opyapi/actions/workflows/release.yml/badge.svg)](https://github.com/kodemore/opyapi/actions/workflows/release.yml) [![codecov](https://codecov.io/gh/kodemore/opyapi/branch/main/graph/badge.svg?token=KWFTWSKPKJ)](https://codecov.io/gh/kodemore/opyapi) [![License: MIT](https://img.shields.io/badge/License-MIT-yellow.svg)](https://opensource.org/licenses/MIT)\nJsonSchema and OpenAPI tools for python.\n\n## Features\n- Validation builder for json schema\n- Custom string formatters support\n- Full JsonSchema draft-7 support\n- Support for JsonReferences, $anchor\n- Dataclass generation from OpenAPI schema\n- Great performance\n\n## Installation\n\nWith pip,\n```shell\npip install opyapi\n```\nor through poetry\n```shell\npoetry add opyapi\n```\n\n# Usage\n\n> It is recommended to get familiar with json-schema if you haven\'t yet. \n> [Understanding Json Schema](https://json-schema.org/understanding-json-schema/index.html) is a great place to start and learn Json Schema\'s basics\n## Simple usage\nLibrary can be used to validate data, against defined json-schema like so:\n\n```python\nfrom opyapi import validate\n\nassert validate(\n    {"name": "Test", "age":12}, \n    {\n        "type": "object",\n        "properties": {\n            "name": {\n                "type": "string",\n            },\n            "age": {\n              "type": "integer",  \n            }\n        }\n    }\n)\n```\n\n## Using references\nValidating documents against complex schemas is also possible,\nplease consider the following example with json references:\n\n```python\nfrom opyapi import validate\n\nschema = {\n    "type": "object",\n    "properties": {\n        "name": {\n            "type": "string",\n            "minLength": 2,\n        },\n        "email": {\n            "$ref": "#/$defs/email"\n        }\n    },\n    "$defs": {\n        "email": {\n            "type": "string",\n            "format": "email"\n        }\n    }\n}\n\nvalidate({"name": "Bob", "email": "bob@test.com"}, schema)\n```\n\n## Custom string formats support\n\n```python\nfrom opyapi import StringFormat, validate\nimport re\n\ndef my_format_validator(value: str) -> str:\n    if re.match("^my-", value):\n        return value\n    raise ValueError(f"Could not validate {value}")\n\nStringFormat["my-format"] = my_format_validator\n\nvalidate("my-test", {"type": "string", "format": "my-format"})  # passes\nvalidate("test", {"type": "string", "format": "my-format"})  # fails\n```\n\n> In the above example `opyapi.StringFormat` is used to register new custom format,\n> which is recognised during validation.\n\n## Re-using validators\n\nThere are scenarios where same validator should be used multiple times,\nin these scenarios to improve performance it is better to use `build_validator_for` \nwhich returns a validator function for the passed schema:\n\n```python\nfrom opyapi import build_validator_for\n\nmy_validator = build_validator_for({\n    "type": "object",\n    "properties": {\n        "name": {\n            "type": "string",\n            "minLength": 2,\n        },\n        "email": {\n            "type": "string",\n            "format": "email",\n        }\n    },\n})\n\nassert my_validator({"name": "Bob", "email": "bob@email.com"})\n```\n\n## Catching errors\n\n`opyapi` provides versatile error package, which is really simple to use. You are only required to define try/except \nblock within you can catch one of the errors defined in the package depending on your scenario.\n\n```python\nfrom opyapi import build_validator_for\nfrom opyapi.errors import ValidationError\n\nmy_validator = build_validator_for({\n    "type": "object",\n    "properties": {\n        "name": {\n            "type": "string",\n            "minLength": 2,\n        },\n        "email": {\n            "type": "string",\n            "format": "email",\n        }\n    },\n})\n\ntry:\n    my_validator({"name": "Bob", "email": "invalid"})\nexcept ValidationError as e:\n    print(f"Validation failed from the reason: {e}")\n\n```\n\n### Errors structure\n\nThe following graph represents exception used in `opyapi`. \nAll exceptions are part of `opyapi.errors` module.\n\n```\nValueError\n    ┗━ ValidationError\n            ┣━ TypeValidationError\n            ┣━ EnumValidationError\n            ┣━ FormatValidationError\n            ┣━ ItemsValidationError\n            ┃           ┣━ UniqueItemsValidationError\n            ┃           ┣━ AdditionalItemsValidationError\n            ┃           ┣━ MinimumItemsValidationError\n            ┃           ┗━ MaximumItemsValidationError\n            ┣━ MultipleOfValidationError\n            ┣━ ComparisonValidationError\n            ┃           ┣━ EqualityValidationError\n            ┃           ┗━ RangeValidationError\n            ┃                       ┣━ MinimumValidationError\n            ┃                       ┣━ MaximumValidationError\n            ┃                       ┣━ ExclusiveMinimumValidationError\n            ┃                       ┗━ ExclusiveMaximumValidationError\n            ┗━ ObjectValidationError\n                        ┣━ PropertyValidationError\n                        ┃           ┣━ RequiredPropertyValidationError\n                        ┃           ┣━ PropertyValueValidationError\n                        ┃           ┣━ PropertyNameValidationError\n                        ┃           ┗━ AdditionalPropertiesValidationError\n                        ┣━ ObjectSizeValidationError\n                        ┃           ┣━ MinimumPropertiesValidationError\n                        ┃           ┗━ MaximumPropertiesValidationError\n                        ┗━ DependencyValidationError\n```\n\n#### `opyapi.errors.ValidationError`\nGeneric validation error.\n\n#### `opyapi.errors.TypeValidationError`\nRaised when passed type does not conform type defined in the schema.\n\n> Can be triggered by following example schema:\n> ```json\n> {\n>   "type": "integer"\n> }\n> ```\n\n#### `opyapi.errors.EnumValidationError`\nRaised when value does not conform enum definition.\n\n> Can be triggered by following example schema:\n> ```json\n> {\n>   "enum": ["integer", "string", "boolean"]\n> }\n> ```\n\n#### `opyapi.errors.FormatValidationError`\nRaised when value does not conform defined format in string schema.\n\n> Can be triggered by following example schema:\n> ```json\n> {\n>   "type": "string",\n>   "format": "date-time"\n> }\n> ```\n\n#### `opyapi.errors.ItemsValidationError`\nGeneric Exception raised when validation of an array fails for some reason.\n\n#### `opyapi.errors.UniqueItemsValidationError`\nRaised when items in an array are expected to be unique but passed value does not conform the requirement. \nThis exception extends generic `opyapi.errors.ItemsValidationError` exception.\n\n> Can be triggered by following example schema:\n> ```json\n> {\n>   "type": "array",\n>   "uniqueItems": true\n> }\n> ```\n\n#### `opyapi.errors.AdditionalItemsValidationError`\nRaised when additional items in an array either does not conform the schema or are not expected.\nThis exception extends generic `opyapi.errors.ItemsValidationError` exception.\n\n> Can be triggered by the following schema:\n> ```json\n> {\n>   "type": "array",\n>   "items": [\n>     {"type": "string"},\n>     {"type":  "integer"}        \n>   ],\n>   "additionalItems": false\n> }\n> ```\n\n#### `opyapi.errors.MinimumItemsValidationError`, `opyapi.errors.MaximumItemsValidationError`\nRaised when number of items in an array does not conform maximum or minimum items specified in schema.\nThis exception extends generic `opyapi.errors.ItemsValidationError` exception.\n\n> Can be triggered by following example schema:\n> ```json\n> {\n>   "type": "array",\n>   "minimumItems": 2,\n>   "MaximumItems": 10\n> }\n> ```\n\n\n#### `opyapi.errors.MultipleOfValidationError`\nRaised when validated number is not multiplication of passed value.\n\n> Can be triggered by following example schema:\n> ```json\n> {\n>   "type": "numerical",\n>   "multipleOf": 2\n> }\n> ```\n\n#### `opyapi.errors.ComparisonValidationError`\nRaised when comparison operation fails. This is a generic exception used by other comparison errors.\n\n#### `opyapi.errors.EqualityValidationError`\nRaised when validated value is not the same as defined expected value. This exception extends `opyapi.errors.ComparisonValidationError`.\n\n> Can be triggered by following example schema:\n> ```json\n> {\n>   "const": "test"\n> }\n> ```\n\n#### `opyapi.errors.RangeValidationError`\nRaised when (exclusive) minimum, (exclusive) maximum comparisons fail. This exception extends `opyapi.errors.ComparisonValidationError`.\n\n#### `opyapi.errors.MinimumValidationError`, `opyapi.errors.MaximumValidationError`, `opyapi.errors.ExclusiveMinimumValidationError`, `opyapi.errors.ExclusiveMaximumValidationError`\nRaised when passed numerical (or integer) value is not within expected range defined in schema.\n\n> Can be triggered by the following example schema:\n> ```json\n> {\n>   "type": "numerical",\n>   "minimum": 2\n> }\n> ```\n\n\n#### `opyapi.errors.ObjectValidationError`\nGeneric exception raised when validation of an object fails for some reason.\n\n#### `opyapi.errors.PropertyValidationError`\nGeneric exception raised when validation of object\'s property fails for some reason. \nThis exception extends `opyapi.errors.ObjectValidationError` exception.\n\n#### `opyapi.errors.RequiredPropertyValidationError`\nRaised when required property is not present in passed object.\nThis exception extends `opyapi.errors.PropertyValidationError` exception.\n\n> Can be triggered by the following example schema:\n> ```json\n> {\n>   "type": "object",\n>   "properties": {\n>     "name" : {"type": "string"}\n>   },\n>   "required": ["name"]\n> }\n> ```\n\n#### `opyapi.errors.PropertyValueValidationError`\nRaised when property contains invalid value. \nThis exception extends `opyapi.errors.PropertyValidationError` exception.\n\n> Can be triggered by the following example schema:\n> ```json\n> {\n>   "type": "object",\n>   "properties": {\n>     "name" : {"type": "string"}\n>   }\n> }\n> ```\n\n#### `opyapi.errors.PropertyNameValidationError`\nRaised when property\'s names does not conform defined schema. \nThis exception extends `opyapi.errors.PropertyValidationError` exception.\n\n> Can be triggered by the following example schema:\n> ```json\n> {\n>   "type": "object",\n>   "properties": {\n>     "name" : {"type": "string"}\n>   },\n>   "propertyNames": {\n>     "pattern": "^x-" \n>   }\n> }\n> ```\n\n#### `opyapi.errors.AdditionalPropertiesValidationError`\nRaised when additional properties are not allowed or set schema is not followed. \nThis exception extends `opyapi.errors.PropertyValidationError` exception.\n\n> Can be triggered by the following example schema:\n> ```json\n> {\n>   "type": "object",\n>   "properties": {\n>     "name" : {"type": "string"}\n>   },\n>   "additionalProperties": false\n> }\n> ```\n\n#### `opyapi.errors.ObjectSizeValidationError`\nGeneric exception raised when number of properties does not conform defined schema. \nThis exception extends `opyapi.errors.ObjectValidationError` exception.\n\n#### `opyapi.errors.MinimumPropertiesValidationError`, `opyapi.errors.MaximumPropertiesValidationError`\nGeneric exception raised when number of properties does not conform defined schema. \nThese exceptions extend `opyapi.errors.ObjectSizeValidationError`exception.\n\n> Can be triggered by the following example schema:\n> ```json\n> {\n>   "type": "object",\n>   "properties": {\n>     "name" : {"type": "string"}\n>   },\n>   "minProperties": 2,\n>   "maxProperties": 10\n> }\n> ```\n\n#### `opyapi.errors.DependencyValidationError`\n\nRaised when dependent properties are defined in the schema but not provided in a passed object. \nThis exception extends `opyapi.errors.ObjectValidationError` exception.\n\n> Can be triggered by the following example schema:\n> ```json\n> {\n>   "type": "object",\n>   "properties": {\n>     "name" : {"type": "string"}\n>   },\n>   "dependentRequired": {\n>     "name": ["first_name", "last_name"]\n>   }\n> }\n> ```\n\n## API\n\n### `validate(obj: typing.Any, schema: typing.Union[dict, opyapi.JsonSchema]): typing.Any`\n\nValidates passed object `obj`, and if valid returns the object, otherwise raises a `ValueError` exception.\n\n```python\nfrom opyapi import validate\n\nassert validate(\n    {"name": "Test", "age":12}, \n    {\n        "type": "object",\n        "properties": {\n            "name": {\n                "type": "string",\n            },\n            "age": {\n              "type": "integer",  \n            }\n        }\n    }\n)\n```\n\n### `build_validator_for(schema: typing.Union[dict, JsonSchema]) -> Callable`\n\nCreates validator function for passed json schema and returns it as a result.\n\n```python\nfrom opyapi import build_validator_for\n\nvalidator =  build_validator_for({\n    "type": "object",\n    "properties": {\n        "name": {\n            "type": "string",\n        },\n        "age": {\n          "type": "integer",  \n        }\n    }\n})\n```\n',
    'author': 'Dawid Kraczkowski',
    'author_email': 'dawid.kraczkowski@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/kodemore/opyapi',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
