"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const receipt_rule_1 = require("./receipt-rule");
const ses_generated_1 = require("./ses.generated");
/**
 * A new or imported receipt rule set.
 */
class ReceiptRuleSetBase extends core_1.Resource {
    /**
     * Adds a new receipt rule in this rule set. The new rule is added after
     * the last added rule unless `after` is specified.
     */
    addRule(id, options) {
        this.lastAddedRule = new receipt_rule_1.ReceiptRule(this, id, {
            after: this.lastAddedRule ? this.lastAddedRule : undefined,
            ruleSet: this,
            ...options,
        });
        return this.lastAddedRule;
    }
    /**
     * Adds a drop spam rule
     */
    addDropSpamRule() {
        const dropSpam = new receipt_rule_1.DropSpamReceiptRule(this, 'DropSpam', {
            ruleSet: this,
        });
        this.lastAddedRule = dropSpam.rule;
    }
}
/**
 * A new receipt rule set.
 */
class ReceiptRuleSet extends ReceiptRuleSetBase {
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.receiptRuleSetName,
        });
        const resource = new ses_generated_1.CfnReceiptRuleSet(this, 'Resource', {
            ruleSetName: this.physicalName,
        });
        this.receiptRuleSetName = resource.ref;
        if (props) {
            const rules = props.rules || [];
            rules.forEach((ruleOption, idx) => this.addRule(`Rule${idx}`, ruleOption));
            if (props.dropSpam) {
                this.addDropSpamRule();
            }
        }
    }
    /**
     * Import an exported receipt rule set.
     */
    static fromReceiptRuleSetName(scope, id, receiptRuleSetName) {
        class Import extends ReceiptRuleSetBase {
            constructor() {
                super(...arguments);
                this.receiptRuleSetName = receiptRuleSetName;
            }
        }
        return new Import(scope, id);
    }
}
exports.ReceiptRuleSet = ReceiptRuleSet;
//# sourceMappingURL=data:application/json;base64,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