/*
 * Decompiled with CFR 0.152.
 */
package org.infinispan.util.concurrent;

import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Queue;
import java.util.concurrent.AbstractExecutorService;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import org.infinispan.commons.time.TimeService;
import org.infinispan.factories.annotations.Stop;
import org.infinispan.factories.scopes.Scope;
import org.infinispan.factories.scopes.Scopes;
import org.infinispan.util.concurrent.BlockingRunnable;
import org.infinispan.util.concurrent.BlockingTaskAwareExecutorService;
import org.infinispan.util.logging.Log;
import org.infinispan.util.logging.LogFactory;

@Scope(value=Scopes.GLOBAL)
public class BlockingTaskAwareExecutorServiceImpl
extends AbstractExecutorService
implements BlockingTaskAwareExecutorService {
    private static final Log log = LogFactory.getLog(BlockingTaskAwareExecutorServiceImpl.class);
    private final Queue<BlockingRunnable> blockedTasks;
    private final ExecutorService executorService;
    private final TimeService timeService;
    private volatile boolean shutdown;
    private final AtomicInteger requestCounter = new AtomicInteger();

    public BlockingTaskAwareExecutorServiceImpl(ExecutorService executorService, TimeService timeService) {
        this.blockedTasks = new ConcurrentLinkedQueue<BlockingRunnable>();
        this.executorService = executorService;
        this.timeService = timeService;
        this.shutdown = false;
    }

    @Stop
    void stop() {
        this.shutdown = true;
    }

    @Override
    public final void execute(BlockingRunnable runnable2) {
        if (this.shutdown) {
            throw new RejectedExecutionException("Executor Service is already shutdown");
        }
        if (runnable2.isReady()) {
            this.doExecute(runnable2);
            if (log.isTraceEnabled()) {
                log.tracef("Added a new task directly: %d task(s) are waiting", this.blockedTasks.size());
            }
        } else {
            this.blockedTasks.offer(runnable2);
            this.checkForReadyTasks();
            if (log.isTraceEnabled()) {
                log.tracef("Added a new task to the queue: %d task(s) are waiting", this.blockedTasks.size());
            }
        }
    }

    @Override
    public void shutdown() {
        this.shutdown = true;
        this.executorService.shutdown();
    }

    @Override
    public List<Runnable> shutdownNow() {
        this.shutdown = true;
        LinkedList<Runnable> runnableList = new LinkedList<Runnable>();
        runnableList.addAll(this.executorService.shutdownNow());
        runnableList.addAll(this.blockedTasks);
        return runnableList;
    }

    @Override
    public boolean isShutdown() {
        return this.shutdown;
    }

    @Override
    public boolean isTerminated() {
        return this.shutdown && this.blockedTasks.isEmpty() && this.executorService.isTerminated();
    }

    @Override
    public boolean awaitTermination(long timeout2, TimeUnit unit) throws InterruptedException {
        long endTime = this.timeService.expectedEndTime(timeout2, unit);
        long waitTime = this.timeService.remainingTime(endTime, TimeUnit.MILLISECONDS);
        while (!this.blockedTasks.isEmpty() && waitTime > 0L) {
            Thread.sleep(waitTime);
            waitTime = this.timeService.remainingTime(endTime, TimeUnit.MILLISECONDS);
        }
        return this.isTerminated();
    }

    @Override
    public final void checkForReadyTasks() {
        if (!this.blockedTasks.isEmpty()) {
            this.tryBlockedTasks();
        }
    }

    @Override
    public void execute(Runnable command) {
        if (this.shutdown) {
            throw new RejectedExecutionException("Executor Service is already shutdown");
        }
        if (command instanceof BlockingRunnable) {
            this.execute((BlockingRunnable)command);
        } else {
            try {
                this.executorService.execute(command);
            }
            catch (RejectedExecutionException rejected) {
                this.blockedTasks.offer(new RunnableWrapper(command));
                this.checkForReadyTasks();
            }
        }
    }

    public ExecutorService getExecutorService() {
        return this.executorService;
    }

    private void tryBlockedTasks() {
        int counter = this.requestCounter.getAndIncrement();
        if (counter == 0) {
            do {
                int taskExecutionCount = 0;
                int remaining = 0;
                Iterator iterator2 = this.blockedTasks.iterator();
                while (iterator2.hasNext()) {
                    boolean ready;
                    BlockingRunnable runnable2 = (BlockingRunnable)iterator2.next();
                    try {
                        ready = runnable2.isReady();
                    }
                    catch (Exception e) {
                        log.debugf((Throwable)e, "Failed to check ready state of %s, dropping.", (Object)runnable2);
                        iterator2.remove();
                        continue;
                    }
                    boolean executed = false;
                    if (ready) {
                        iterator2.remove();
                        executed = this.doExecute(runnable2);
                    }
                    if (executed) {
                        ++taskExecutionCount;
                        continue;
                    }
                    ++remaining;
                }
                if (!log.isTraceEnabled()) continue;
                log.tracef("Tasks executed=%s, still pending=~%s", taskExecutionCount, remaining);
            } while ((counter = this.requestCounter.addAndGet(-counter)) != 0);
        }
    }

    private boolean doExecute(BlockingRunnable runnable2) {
        try {
            this.executorService.execute(runnable2);
            return true;
        }
        catch (RejectedExecutionException rejected) {
            if (!this.shutdown) {
                this.blockedTasks.offer(runnable2);
                this.requestCounter.incrementAndGet();
            }
            return false;
        }
    }

    private static class RunnableWrapper
    implements BlockingRunnable {
        private final Runnable runnable;

        private RunnableWrapper(Runnable runnable2) {
            this.runnable = runnable2;
        }

        @Override
        public boolean isReady() {
            return true;
        }

        @Override
        public void run() {
            this.runnable.run();
        }

        public String toString() {
            return "RunnableWrapper(" + this.runnable + ")";
        }
    }
}

