"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const sfn = require("@aws-cdk/aws-stepfunctions");
const core_1 = require("@aws-cdk/core");
const start_execution_1 = require("../../lib/stepfunctions/start-execution");
let stack;
let child;
beforeEach(() => {
    stack = new core_1.Stack();
    child = new sfn.StateMachine(stack, 'ChildStateMachine', {
        definition: sfn.Chain.start(new sfn.Pass(stack, 'PassState')),
    });
});
test('Execute State Machine - Default - Request Response', () => {
    const task = new start_execution_1.StepFunctionsStartExecution(stack, 'ChildTask', {
        stateMachine: child,
        input: sfn.TaskInput.fromObject({
            foo: 'bar',
        }),
        name: 'myExecutionName',
    });
    new sfn.StateMachine(stack, 'ParentStateMachine', {
        definition: task,
    });
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::states:startExecution',
                ],
            ],
        },
        End: true,
        Parameters: {
            Input: {
                foo: 'bar',
            },
            Name: 'myExecutionName',
            StateMachineArn: {
                Ref: 'ChildStateMachine9133117F',
            },
        },
    });
});
test('Execute State Machine - Run Job', () => {
    const task = new start_execution_1.StepFunctionsStartExecution(stack, 'ChildTask', {
        stateMachine: child,
        integrationPattern: sfn.IntegrationPattern.RUN_JOB,
    });
    new sfn.StateMachine(stack, 'ParentStateMachine', {
        definition: task,
    });
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::states:startExecution.sync:2',
                ],
            ],
        },
        End: true,
        Parameters: {
            'Input.$': '$',
            'StateMachineArn': {
                Ref: 'ChildStateMachine9133117F',
            },
        },
    });
    expect(stack).toHaveResource('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: 'states:StartExecution',
                    Effect: 'Allow',
                    Resource: {
                        Ref: 'ChildStateMachine9133117F',
                    },
                },
                {
                    Action: ['states:DescribeExecution', 'states:StopExecution'],
                    Effect: 'Allow',
                    Resource: {
                        'Fn::Join': [
                            '',
                            [
                                'arn:',
                                {
                                    Ref: 'AWS::Partition',
                                },
                                ':states:',
                                {
                                    Ref: 'AWS::Region',
                                },
                                ':',
                                {
                                    Ref: 'AWS::AccountId',
                                },
                                ':execution:',
                                {
                                    'Fn::Select': [
                                        6,
                                        {
                                            'Fn::Split': [
                                                ':',
                                                {
                                                    Ref: 'ChildStateMachine9133117F',
                                                },
                                            ],
                                        },
                                    ],
                                },
                                '*',
                            ],
                        ],
                    },
                },
                {
                    Action: ['events:PutTargets', 'events:PutRule', 'events:DescribeRule'],
                    Effect: 'Allow',
                    Resource: {
                        'Fn::Join': [
                            '',
                            [
                                'arn:',
                                {
                                    Ref: 'AWS::Partition',
                                },
                                ':events:',
                                {
                                    Ref: 'AWS::Region',
                                },
                                ':',
                                {
                                    Ref: 'AWS::AccountId',
                                },
                                ':rule/StepFunctionsGetEventsForStepFunctionsExecutionRule',
                            ],
                        ],
                    },
                },
            ],
            Version: '2012-10-17',
        },
        Roles: [
            {
                Ref: 'ParentStateMachineRoleE902D002',
            },
        ],
    });
});
test('Execute State Machine - Wait For Task Token', () => {
    const task = new start_execution_1.StepFunctionsStartExecution(stack, 'ChildTask', {
        stateMachine: child,
        integrationPattern: sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
        input: sfn.TaskInput.fromObject({
            token: sfn.Context.taskToken,
        }),
    });
    new sfn.StateMachine(stack, 'ParentStateMachine', {
        definition: task,
    });
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::states:startExecution.waitForTaskToken',
                ],
            ],
        },
        End: true,
        Parameters: {
            Input: {
                'token.$': '$$.Task.Token',
            },
            StateMachineArn: {
                Ref: 'ChildStateMachine9133117F',
            },
        },
    });
});
test('Execute State Machine - Wait For Task Token - Missing Task Token', () => {
    expect(() => {
        new start_execution_1.StepFunctionsStartExecution(stack, 'ChildTask', {
            stateMachine: child,
            integrationPattern: sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
        });
    }).toThrow('Task Token is required in `input` for callback. Use Context.taskToken to set the token.');
});
//# sourceMappingURL=data:application/json;base64,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