"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const sfn = require("@aws-cdk/aws-stepfunctions");
const core_1 = require("@aws-cdk/core");
const tasks = require("../../lib");
const jobName = 'GlueJob';
let stack;
beforeEach(() => {
    stack = new core_1.Stack();
});
test('Invoke glue job with just job ARN', () => {
    const task = new sfn.Task(stack, 'Task', {
        task: new tasks.RunGlueJobTask(jobName),
    });
    new sfn.StateMachine(stack, 'SM', {
        definition: task,
    });
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::glue:startJobRun',
                ],
            ],
        },
        End: true,
        Parameters: {
            JobName: jobName,
        },
    });
});
test('Invoke glue job with full properties', () => {
    const jobArguments = {
        key: 'value',
    };
    const timeoutMinutes = 1440;
    const timeout = core_1.Duration.minutes(timeoutMinutes);
    const securityConfiguration = 'securityConfiguration';
    const notifyDelayAfterMinutes = 10;
    const notifyDelayAfter = core_1.Duration.minutes(notifyDelayAfterMinutes);
    const task = new sfn.Task(stack, 'Task', {
        task: new tasks.RunGlueJobTask(jobName, {
            integrationPattern: sfn.ServiceIntegrationPattern.SYNC,
            arguments: jobArguments,
            timeout,
            securityConfiguration,
            notifyDelayAfter,
        }),
    });
    new sfn.StateMachine(stack, 'SM', {
        definition: task,
    });
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::glue:startJobRun.sync',
                ],
            ],
        },
        End: true,
        Parameters: {
            JobName: jobName,
            Arguments: jobArguments,
            Timeout: timeoutMinutes,
            SecurityConfiguration: securityConfiguration,
            NotificationProperty: {
                NotifyDelayAfter: notifyDelayAfterMinutes,
            },
        },
    });
});
test('permitted role actions limited to start job run if service integration pattern is FIRE_AND_FORGET', () => {
    const task = new sfn.Task(stack, 'Task', {
        task: new tasks.RunGlueJobTask(jobName, {
            integrationPattern: sfn.ServiceIntegrationPattern.FIRE_AND_FORGET,
        }),
    });
    new sfn.StateMachine(stack, 'SM', {
        definition: task,
    });
    expect(stack).toHaveResourceLike('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [{
                    Action: 'glue:StartJobRun',
                }],
        },
    });
});
test('permitted role actions include start, get, and stop job run if service integration pattern is SYNC', () => {
    const task = new sfn.Task(stack, 'Task', {
        task: new tasks.RunGlueJobTask(jobName, {
            integrationPattern: sfn.ServiceIntegrationPattern.SYNC,
        }),
    });
    new sfn.StateMachine(stack, 'SM', {
        definition: task,
    });
    expect(stack).toHaveResourceLike('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [{
                    Action: [
                        'glue:StartJobRun',
                        'glue:GetJobRun',
                        'glue:GetJobRuns',
                        'glue:BatchStopJobRun',
                    ],
                }],
        },
    });
});
test('Task throws if WAIT_FOR_TASK_TOKEN is supplied as service integration pattern', () => {
    expect(() => {
        new sfn.Task(stack, 'Task', {
            task: new tasks.RunGlueJobTask(jobName, {
                integrationPattern: sfn.ServiceIntegrationPattern.WAIT_FOR_TASK_TOKEN,
            }),
        });
    }).toThrow(/Invalid Service Integration Pattern: WAIT_FOR_TASK_TOKEN is not supported to call Glue./i);
});
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicnVuLWdsdWUtam9iLXRhc2sudGVzdC5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbInJ1bi1nbHVlLWpvYi10YXNrLnRlc3QudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7QUFBQSxnQ0FBOEI7QUFDOUIsa0RBQWtEO0FBQ2xELHdDQUFnRDtBQUNoRCxtQ0FBbUM7QUFFbkMsTUFBTSxPQUFPLEdBQUcsU0FBUyxDQUFDO0FBQzFCLElBQUksS0FBWSxDQUFDO0FBQ2pCLFVBQVUsQ0FBQyxHQUFHLEVBQUU7SUFDZCxLQUFLLEdBQUcsSUFBSSxZQUFLLEVBQUUsQ0FBQztBQUN0QixDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyxtQ0FBbUMsRUFBRSxHQUFHLEVBQUU7SUFDN0MsTUFBTSxJQUFJLEdBQUcsSUFBSSxHQUFHLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRSxNQUFNLEVBQUU7UUFDdkMsSUFBSSxFQUFFLElBQUksS0FBSyxDQUFDLGNBQWMsQ0FBQyxPQUFPLENBQUM7S0FDeEMsQ0FBQyxDQUFDO0lBQ0gsSUFBSSxHQUFHLENBQUMsWUFBWSxDQUFDLEtBQUssRUFBRSxJQUFJLEVBQUU7UUFDaEMsVUFBVSxFQUFFLElBQUk7S0FDakIsQ0FBQyxDQUFDO0lBRUgsTUFBTSxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLFdBQVcsRUFBRSxDQUFDLENBQUMsQ0FBQyxPQUFPLENBQUM7UUFDaEQsSUFBSSxFQUFFLE1BQU07UUFDWixRQUFRLEVBQUU7WUFDUixVQUFVLEVBQUU7Z0JBQ1YsRUFBRTtnQkFDRjtvQkFDRSxNQUFNO29CQUNOO3dCQUNFLEdBQUcsRUFBRSxnQkFBZ0I7cUJBQ3RCO29CQUNELDRCQUE0QjtpQkFDN0I7YUFDRjtTQUNGO1FBQ0QsR0FBRyxFQUFFLElBQUk7UUFDVCxVQUFVLEVBQUU7WUFDVixPQUFPLEVBQUUsT0FBTztTQUNqQjtLQUNGLENBQUMsQ0FBQztBQUNMLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLHNDQUFzQyxFQUFFLEdBQUcsRUFBRTtJQUNoRCxNQUFNLFlBQVksR0FBRztRQUNuQixHQUFHLEVBQUUsT0FBTztLQUNiLENBQUM7SUFDRixNQUFNLGNBQWMsR0FBRyxJQUFJLENBQUM7SUFDNUIsTUFBTSxPQUFPLEdBQUcsZUFBUSxDQUFDLE9BQU8sQ0FBQyxjQUFjLENBQUMsQ0FBQztJQUNqRCxNQUFNLHFCQUFxQixHQUFHLHVCQUF1QixDQUFDO0lBQ3RELE1BQU0sdUJBQXVCLEdBQUcsRUFBRSxDQUFDO0lBQ25DLE1BQU0sZ0JBQWdCLEdBQUcsZUFBUSxDQUFDLE9BQU8sQ0FBQyx1QkFBdUIsQ0FBQyxDQUFDO0lBQ25FLE1BQU0sSUFBSSxHQUFHLElBQUksR0FBRyxDQUFDLElBQUksQ0FBQyxLQUFLLEVBQUUsTUFBTSxFQUFFO1FBQ3ZDLElBQUksRUFBRSxJQUFJLEtBQUssQ0FBQyxjQUFjLENBQUMsT0FBTyxFQUFFO1lBQ3RDLGtCQUFrQixFQUFFLEdBQUcsQ0FBQyx5QkFBeUIsQ0FBQyxJQUFJO1lBQ3RELFNBQVMsRUFBRSxZQUFZO1lBQ3ZCLE9BQU87WUFDUCxxQkFBcUI7WUFDckIsZ0JBQWdCO1NBQ2pCLENBQUM7S0FDSCxDQUFDLENBQUM7SUFDSCxJQUFJLEdBQUcsQ0FBQyxZQUFZLENBQUMsS0FBSyxFQUFFLElBQUksRUFBRTtRQUNoQyxVQUFVLEVBQUUsSUFBSTtLQUNqQixDQUFDLENBQUM7SUFFSCxNQUFNLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsV0FBVyxFQUFFLENBQUMsQ0FBQyxDQUFDLE9BQU8sQ0FBQztRQUNoRCxJQUFJLEVBQUUsTUFBTTtRQUNaLFFBQVEsRUFBRTtZQUNSLFVBQVUsRUFBRTtnQkFDVixFQUFFO2dCQUNGO29CQUNFLE1BQU07b0JBQ047d0JBQ0UsR0FBRyxFQUFFLGdCQUFnQjtxQkFDdEI7b0JBQ0QsaUNBQWlDO2lCQUNsQzthQUNGO1NBQ0Y7UUFDRCxHQUFHLEVBQUUsSUFBSTtRQUNULFVBQVUsRUFBRTtZQUNWLE9BQU8sRUFBRSxPQUFPO1lBQ2hCLFNBQVMsRUFBRSxZQUFZO1lBQ3ZCLE9BQU8sRUFBRSxjQUFjO1lBQ3ZCLHFCQUFxQixFQUFFLHFCQUFxQjtZQUM1QyxvQkFBb0IsRUFBRTtnQkFDcEIsZ0JBQWdCLEVBQUUsdUJBQXVCO2FBQzFDO1NBQ0Y7S0FDRixDQUFDLENBQUM7QUFDTCxDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyxtR0FBbUcsRUFBRSxHQUFHLEVBQUU7SUFDN0csTUFBTSxJQUFJLEdBQUcsSUFBSSxHQUFHLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRSxNQUFNLEVBQUU7UUFDdkMsSUFBSSxFQUFFLElBQUksS0FBSyxDQUFDLGNBQWMsQ0FBQyxPQUFPLEVBQUU7WUFDdEMsa0JBQWtCLEVBQUUsR0FBRyxDQUFDLHlCQUF5QixDQUFDLGVBQWU7U0FDbEUsQ0FBQztLQUNILENBQUMsQ0FBQztJQUNILElBQUksR0FBRyxDQUFDLFlBQVksQ0FBQyxLQUFLLEVBQUUsSUFBSSxFQUFFO1FBQ2hDLFVBQVUsRUFBRSxJQUFJO0tBQ2pCLENBQUMsQ0FBQztJQUVILE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxrQkFBa0IsQ0FBQyxrQkFBa0IsRUFBRTtRQUNuRCxjQUFjLEVBQUU7WUFDZCxTQUFTLEVBQUUsQ0FBQztvQkFDVixNQUFNLEVBQUUsa0JBQWtCO2lCQUMzQixDQUFDO1NBQ0g7S0FDRixDQUFDLENBQUM7QUFDTCxDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyxvR0FBb0csRUFBRSxHQUFHLEVBQUU7SUFDOUcsTUFBTSxJQUFJLEdBQUcsSUFBSSxHQUFHLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRSxNQUFNLEVBQUU7UUFDdkMsSUFBSSxFQUFFLElBQUksS0FBSyxDQUFDLGNBQWMsQ0FBQyxPQUFPLEVBQUU7WUFDdEMsa0JBQWtCLEVBQUUsR0FBRyxDQUFDLHlCQUF5QixDQUFDLElBQUk7U0FDdkQsQ0FBQztLQUNILENBQUMsQ0FBQztJQUNILElBQUksR0FBRyxDQUFDLFlBQVksQ0FBQyxLQUFLLEVBQUUsSUFBSSxFQUFFO1FBQ2hDLFVBQVUsRUFBRSxJQUFJO0tBQ2pCLENBQUMsQ0FBQztJQUVILE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxrQkFBa0IsQ0FBQyxrQkFBa0IsRUFBRTtRQUNuRCxjQUFjLEVBQUU7WUFDZCxTQUFTLEVBQUUsQ0FBQztvQkFDVixNQUFNLEVBQUU7d0JBQ04sa0JBQWtCO3dCQUNsQixnQkFBZ0I7d0JBQ2hCLGlCQUFpQjt3QkFDakIsc0JBQXNCO3FCQUN2QjtpQkFDRixDQUFDO1NBQ0g7S0FDRixDQUFDLENBQUM7QUFDTCxDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQywrRUFBK0UsRUFBRSxHQUFHLEVBQUU7SUFDekYsTUFBTSxDQUFDLEdBQUcsRUFBRTtRQUNWLElBQUksR0FBRyxDQUFDLElBQUksQ0FBQyxLQUFLLEVBQUUsTUFBTSxFQUFFO1lBQzFCLElBQUksRUFBRSxJQUFJLEtBQUssQ0FBQyxjQUFjLENBQUMsT0FBTyxFQUFFO2dCQUN0QyxrQkFBa0IsRUFBRSxHQUFHLENBQUMseUJBQXlCLENBQUMsbUJBQW1CO2FBQ3RFLENBQUM7U0FDSCxDQUFDLENBQUM7SUFDTCxDQUFDLENBQUMsQ0FBQyxPQUFPLENBQUMsMEZBQTBGLENBQUMsQ0FBQztBQUN6RyxDQUFDLENBQUMsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAnQGF3cy1jZGsvYXNzZXJ0L2plc3QnO1xuaW1wb3J0ICogYXMgc2ZuIGZyb20gJ0Bhd3MtY2RrL2F3cy1zdGVwZnVuY3Rpb25zJztcbmltcG9ydCB7IER1cmF0aW9uLCBTdGFjayB9IGZyb20gJ0Bhd3MtY2RrL2NvcmUnO1xuaW1wb3J0ICogYXMgdGFza3MgZnJvbSAnLi4vLi4vbGliJztcblxuY29uc3Qgam9iTmFtZSA9ICdHbHVlSm9iJztcbmxldCBzdGFjazogU3RhY2s7XG5iZWZvcmVFYWNoKCgpID0+IHtcbiAgc3RhY2sgPSBuZXcgU3RhY2soKTtcbn0pO1xuXG50ZXN0KCdJbnZva2UgZ2x1ZSBqb2Igd2l0aCBqdXN0IGpvYiBBUk4nLCAoKSA9PiB7XG4gIGNvbnN0IHRhc2sgPSBuZXcgc2ZuLlRhc2soc3RhY2ssICdUYXNrJywge1xuICAgIHRhc2s6IG5ldyB0YXNrcy5SdW5HbHVlSm9iVGFzayhqb2JOYW1lKSxcbiAgfSk7XG4gIG5ldyBzZm4uU3RhdGVNYWNoaW5lKHN0YWNrLCAnU00nLCB7XG4gICAgZGVmaW5pdGlvbjogdGFzayxcbiAgfSk7XG5cbiAgZXhwZWN0KHN0YWNrLnJlc29sdmUodGFzay50b1N0YXRlSnNvbigpKSkudG9FcXVhbCh7XG4gICAgVHlwZTogJ1Rhc2snLFxuICAgIFJlc291cmNlOiB7XG4gICAgICAnRm46OkpvaW4nOiBbXG4gICAgICAgICcnLFxuICAgICAgICBbXG4gICAgICAgICAgJ2FybjonLFxuICAgICAgICAgIHtcbiAgICAgICAgICAgIFJlZjogJ0FXUzo6UGFydGl0aW9uJyxcbiAgICAgICAgICB9LFxuICAgICAgICAgICc6c3RhdGVzOjo6Z2x1ZTpzdGFydEpvYlJ1bicsXG4gICAgICAgIF0sXG4gICAgICBdLFxuICAgIH0sXG4gICAgRW5kOiB0cnVlLFxuICAgIFBhcmFtZXRlcnM6IHtcbiAgICAgIEpvYk5hbWU6IGpvYk5hbWUsXG4gICAgfSxcbiAgfSk7XG59KTtcblxudGVzdCgnSW52b2tlIGdsdWUgam9iIHdpdGggZnVsbCBwcm9wZXJ0aWVzJywgKCkgPT4ge1xuICBjb25zdCBqb2JBcmd1bWVudHMgPSB7XG4gICAga2V5OiAndmFsdWUnLFxuICB9O1xuICBjb25zdCB0aW1lb3V0TWludXRlcyA9IDE0NDA7XG4gIGNvbnN0IHRpbWVvdXQgPSBEdXJhdGlvbi5taW51dGVzKHRpbWVvdXRNaW51dGVzKTtcbiAgY29uc3Qgc2VjdXJpdHlDb25maWd1cmF0aW9uID0gJ3NlY3VyaXR5Q29uZmlndXJhdGlvbic7XG4gIGNvbnN0IG5vdGlmeURlbGF5QWZ0ZXJNaW51dGVzID0gMTA7XG4gIGNvbnN0IG5vdGlmeURlbGF5QWZ0ZXIgPSBEdXJhdGlvbi5taW51dGVzKG5vdGlmeURlbGF5QWZ0ZXJNaW51dGVzKTtcbiAgY29uc3QgdGFzayA9IG5ldyBzZm4uVGFzayhzdGFjaywgJ1Rhc2snLCB7XG4gICAgdGFzazogbmV3IHRhc2tzLlJ1bkdsdWVKb2JUYXNrKGpvYk5hbWUsIHtcbiAgICAgIGludGVncmF0aW9uUGF0dGVybjogc2ZuLlNlcnZpY2VJbnRlZ3JhdGlvblBhdHRlcm4uU1lOQyxcbiAgICAgIGFyZ3VtZW50czogam9iQXJndW1lbnRzLFxuICAgICAgdGltZW91dCxcbiAgICAgIHNlY3VyaXR5Q29uZmlndXJhdGlvbixcbiAgICAgIG5vdGlmeURlbGF5QWZ0ZXIsXG4gICAgfSksXG4gIH0pO1xuICBuZXcgc2ZuLlN0YXRlTWFjaGluZShzdGFjaywgJ1NNJywge1xuICAgIGRlZmluaXRpb246IHRhc2ssXG4gIH0pO1xuXG4gIGV4cGVjdChzdGFjay5yZXNvbHZlKHRhc2sudG9TdGF0ZUpzb24oKSkpLnRvRXF1YWwoe1xuICAgIFR5cGU6ICdUYXNrJyxcbiAgICBSZXNvdXJjZToge1xuICAgICAgJ0ZuOjpKb2luJzogW1xuICAgICAgICAnJyxcbiAgICAgICAgW1xuICAgICAgICAgICdhcm46JyxcbiAgICAgICAgICB7XG4gICAgICAgICAgICBSZWY6ICdBV1M6OlBhcnRpdGlvbicsXG4gICAgICAgICAgfSxcbiAgICAgICAgICAnOnN0YXRlczo6OmdsdWU6c3RhcnRKb2JSdW4uc3luYycsXG4gICAgICAgIF0sXG4gICAgICBdLFxuICAgIH0sXG4gICAgRW5kOiB0cnVlLFxuICAgIFBhcmFtZXRlcnM6IHtcbiAgICAgIEpvYk5hbWU6IGpvYk5hbWUsXG4gICAgICBBcmd1bWVudHM6IGpvYkFyZ3VtZW50cyxcbiAgICAgIFRpbWVvdXQ6IHRpbWVvdXRNaW51dGVzLFxuICAgICAgU2VjdXJpdHlDb25maWd1cmF0aW9uOiBzZWN1cml0eUNvbmZpZ3VyYXRpb24sXG4gICAgICBOb3RpZmljYXRpb25Qcm9wZXJ0eToge1xuICAgICAgICBOb3RpZnlEZWxheUFmdGVyOiBub3RpZnlEZWxheUFmdGVyTWludXRlcyxcbiAgICAgIH0sXG4gICAgfSxcbiAgfSk7XG59KTtcblxudGVzdCgncGVybWl0dGVkIHJvbGUgYWN0aW9ucyBsaW1pdGVkIHRvIHN0YXJ0IGpvYiBydW4gaWYgc2VydmljZSBpbnRlZ3JhdGlvbiBwYXR0ZXJuIGlzIEZJUkVfQU5EX0ZPUkdFVCcsICgpID0+IHtcbiAgY29uc3QgdGFzayA9IG5ldyBzZm4uVGFzayhzdGFjaywgJ1Rhc2snLCB7XG4gICAgdGFzazogbmV3IHRhc2tzLlJ1bkdsdWVKb2JUYXNrKGpvYk5hbWUsIHtcbiAgICAgIGludGVncmF0aW9uUGF0dGVybjogc2ZuLlNlcnZpY2VJbnRlZ3JhdGlvblBhdHRlcm4uRklSRV9BTkRfRk9SR0VULFxuICAgIH0pLFxuICB9KTtcbiAgbmV3IHNmbi5TdGF0ZU1hY2hpbmUoc3RhY2ssICdTTScsIHtcbiAgICBkZWZpbml0aW9uOiB0YXNrLFxuICB9KTtcblxuICBleHBlY3Qoc3RhY2spLnRvSGF2ZVJlc291cmNlTGlrZSgnQVdTOjpJQU06OlBvbGljeScsIHtcbiAgICBQb2xpY3lEb2N1bWVudDoge1xuICAgICAgU3RhdGVtZW50OiBbe1xuICAgICAgICBBY3Rpb246ICdnbHVlOlN0YXJ0Sm9iUnVuJyxcbiAgICAgIH1dLFxuICAgIH0sXG4gIH0pO1xufSk7XG5cbnRlc3QoJ3Blcm1pdHRlZCByb2xlIGFjdGlvbnMgaW5jbHVkZSBzdGFydCwgZ2V0LCBhbmQgc3RvcCBqb2IgcnVuIGlmIHNlcnZpY2UgaW50ZWdyYXRpb24gcGF0dGVybiBpcyBTWU5DJywgKCkgPT4ge1xuICBjb25zdCB0YXNrID0gbmV3IHNmbi5UYXNrKHN0YWNrLCAnVGFzaycsIHtcbiAgICB0YXNrOiBuZXcgdGFza3MuUnVuR2x1ZUpvYlRhc2soam9iTmFtZSwge1xuICAgICAgaW50ZWdyYXRpb25QYXR0ZXJuOiBzZm4uU2VydmljZUludGVncmF0aW9uUGF0dGVybi5TWU5DLFxuICAgIH0pLFxuICB9KTtcbiAgbmV3IHNmbi5TdGF0ZU1hY2hpbmUoc3RhY2ssICdTTScsIHtcbiAgICBkZWZpbml0aW9uOiB0YXNrLFxuICB9KTtcblxuICBleHBlY3Qoc3RhY2spLnRvSGF2ZVJlc291cmNlTGlrZSgnQVdTOjpJQU06OlBvbGljeScsIHtcbiAgICBQb2xpY3lEb2N1bWVudDoge1xuICAgICAgU3RhdGVtZW50OiBbe1xuICAgICAgICBBY3Rpb246IFtcbiAgICAgICAgICAnZ2x1ZTpTdGFydEpvYlJ1bicsXG4gICAgICAgICAgJ2dsdWU6R2V0Sm9iUnVuJyxcbiAgICAgICAgICAnZ2x1ZTpHZXRKb2JSdW5zJyxcbiAgICAgICAgICAnZ2x1ZTpCYXRjaFN0b3BKb2JSdW4nLFxuICAgICAgICBdLFxuICAgICAgfV0sXG4gICAgfSxcbiAgfSk7XG59KTtcblxudGVzdCgnVGFzayB0aHJvd3MgaWYgV0FJVF9GT1JfVEFTS19UT0tFTiBpcyBzdXBwbGllZCBhcyBzZXJ2aWNlIGludGVncmF0aW9uIHBhdHRlcm4nLCAoKSA9PiB7XG4gIGV4cGVjdCgoKSA9PiB7XG4gICAgbmV3IHNmbi5UYXNrKHN0YWNrLCAnVGFzaycsIHtcbiAgICAgIHRhc2s6IG5ldyB0YXNrcy5SdW5HbHVlSm9iVGFzayhqb2JOYW1lLCB7XG4gICAgICAgIGludGVncmF0aW9uUGF0dGVybjogc2ZuLlNlcnZpY2VJbnRlZ3JhdGlvblBhdHRlcm4uV0FJVF9GT1JfVEFTS19UT0tFTixcbiAgICAgIH0pLFxuICAgIH0pO1xuICB9KS50b1Rocm93KC9JbnZhbGlkIFNlcnZpY2UgSW50ZWdyYXRpb24gUGF0dGVybjogV0FJVF9GT1JfVEFTS19UT0tFTiBpcyBub3Qgc3VwcG9ydGVkIHRvIGNhbGwgR2x1ZS4vaSk7XG59KTtcbiJdfQ==