"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const sfn = require("@aws-cdk/aws-stepfunctions");
const cdk = require("@aws-cdk/core");
const tasks = require("../../lib");
let stack;
beforeEach(() => {
    // GIVEN
    stack = new cdk.Stack();
});
test('Modify an InstanceGroup with static ClusterId, InstanceGroupName, and InstanceGroup', () => {
    // WHEN
    const task = new tasks.EmrModifyInstanceGroupByName(stack, 'Task', {
        clusterId: 'ClusterId',
        instanceGroupName: 'InstanceGroupName',
        instanceGroup: {
            configurations: [{
                    classification: 'Classification',
                    properties: {
                        Key: 'Value',
                    },
                }],
            eC2InstanceIdsToTerminate: ['InstanceToTerminate'],
            instanceCount: 1,
            shrinkPolicy: {
                decommissionTimeout: cdk.Duration.seconds(1),
                instanceResizePolicy: {
                    instanceTerminationTimeout: cdk.Duration.seconds(1),
                    instancesToProtect: ['InstanceToProtect'],
                    instancesToTerminate: ['InstanceToTerminate'],
                },
            },
        },
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::elasticmapreduce:modifyInstanceGroupByName',
                ],
            ],
        },
        End: true,
        Parameters: {
            ClusterId: 'ClusterId',
            InstanceGroupName: 'InstanceGroupName',
            InstanceGroup: {
                Configurations: [{
                        Classification: 'Classification',
                        Properties: {
                            Key: 'Value',
                        },
                    }],
                EC2InstanceIdsToTerminate: ['InstanceToTerminate'],
                InstanceCount: 1,
                ShrinkPolicy: {
                    DecommissionTimeout: 1,
                    InstanceResizePolicy: {
                        InstanceTerminationTimeout: 1,
                        InstancesToProtect: ['InstanceToProtect'],
                        InstancesToTerminate: ['InstanceToTerminate'],
                    },
                },
            },
        },
    });
});
test('Modify an InstanceGroup with ClusterId from payload and static InstanceGroupName and InstanceGroupConfiguration', () => {
    // WHEN
    const task = new tasks.EmrModifyInstanceGroupByName(stack, 'Task', {
        clusterId: sfn.Data.stringAt('$.ClusterId'),
        instanceGroupName: 'InstanceGroupName',
        instanceGroup: {
            instanceCount: 1,
        },
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::elasticmapreduce:modifyInstanceGroupByName',
                ],
            ],
        },
        End: true,
        Parameters: {
            'ClusterId.$': '$.ClusterId',
            'InstanceGroupName': 'InstanceGroupName',
            'InstanceGroup': {
                InstanceCount: 1,
            },
        },
    });
});
test('Modify an InstanceGroup with static ClusterId and InstanceGroupConfigurateion and InstanceGroupName from payload', () => {
    // WHEN
    const task = new tasks.EmrModifyInstanceGroupByName(stack, 'Task', {
        clusterId: 'ClusterId',
        instanceGroupName: sfn.Data.stringAt('$.InstanceGroupName'),
        instanceGroup: {
            instanceCount: 1,
        },
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::elasticmapreduce:modifyInstanceGroupByName',
                ],
            ],
        },
        End: true,
        Parameters: {
            'ClusterId': 'ClusterId',
            'InstanceGroupName.$': '$.InstanceGroupName',
            'InstanceGroup': {
                InstanceCount: 1,
            },
        },
    });
});
test('Modify an InstanceGroup with static ClusterId and InstanceGroupName and InstanceCount from payload', () => {
    // WHEN
    const task = new tasks.EmrModifyInstanceGroupByName(stack, 'Task', {
        clusterId: 'ClusterId',
        instanceGroupName: 'InstanceGroupName',
        instanceGroup: {
            instanceCount: sfn.Data.numberAt('$.InstanceCount'),
        },
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::elasticmapreduce:modifyInstanceGroupByName',
                ],
            ],
        },
        End: true,
        Parameters: {
            ClusterId: 'ClusterId',
            InstanceGroupName: 'InstanceGroupName',
            InstanceGroup: {
                'InstanceCount.$': '$.InstanceCount',
            },
        },
    });
});
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZW1yLW1vZGlmeS1pbnN0YW5jZS1ncm91cC1ieS1uYW1lLnRlc3QuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJlbXItbW9kaWZ5LWluc3RhbmNlLWdyb3VwLWJ5LW5hbWUudGVzdC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOztBQUFBLGtEQUFrRDtBQUNsRCxxQ0FBcUM7QUFDckMsbUNBQW1DO0FBRW5DLElBQUksS0FBZ0IsQ0FBQztBQUVyQixVQUFVLENBQUMsR0FBRyxFQUFFO0lBQ2QsUUFBUTtJQUNSLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztBQUMxQixDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyxxRkFBcUYsRUFBRSxHQUFHLEVBQUU7SUFDL0YsT0FBTztJQUNQLE1BQU0sSUFBSSxHQUFHLElBQUksS0FBSyxDQUFDLDRCQUE0QixDQUFDLEtBQUssRUFBRSxNQUFNLEVBQUU7UUFDakUsU0FBUyxFQUFFLFdBQVc7UUFDdEIsaUJBQWlCLEVBQUUsbUJBQW1CO1FBQ3RDLGFBQWEsRUFBRTtZQUNiLGNBQWMsRUFBRSxDQUFDO29CQUNmLGNBQWMsRUFBRSxnQkFBZ0I7b0JBQ2hDLFVBQVUsRUFBRTt3QkFDVixHQUFHLEVBQUUsT0FBTztxQkFDYjtpQkFDRixDQUFDO1lBQ0YseUJBQXlCLEVBQUUsQ0FBQyxxQkFBcUIsQ0FBQztZQUNsRCxhQUFhLEVBQUUsQ0FBQztZQUNoQixZQUFZLEVBQUU7Z0JBQ1osbUJBQW1CLEVBQUUsR0FBRyxDQUFDLFFBQVEsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDO2dCQUM1QyxvQkFBb0IsRUFBRTtvQkFDcEIsMEJBQTBCLEVBQUUsR0FBRyxDQUFDLFFBQVEsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDO29CQUNuRCxrQkFBa0IsRUFBRSxDQUFDLG1CQUFtQixDQUFDO29CQUN6QyxvQkFBb0IsRUFBRSxDQUFDLHFCQUFxQixDQUFDO2lCQUM5QzthQUNGO1NBQ0Y7S0FDRixDQUFDLENBQUM7SUFFSCxPQUFPO0lBQ1AsTUFBTSxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLFdBQVcsRUFBRSxDQUFDLENBQUMsQ0FBQyxPQUFPLENBQUM7UUFDaEQsSUFBSSxFQUFFLE1BQU07UUFDWixRQUFRLEVBQUU7WUFDUixVQUFVLEVBQUU7Z0JBQ1YsRUFBRTtnQkFDRjtvQkFDRSxNQUFNO29CQUNOO3dCQUNFLEdBQUcsRUFBRSxnQkFBZ0I7cUJBQ3RCO29CQUNELHNEQUFzRDtpQkFDdkQ7YUFDRjtTQUNGO1FBQ0QsR0FBRyxFQUFFLElBQUk7UUFDVCxVQUFVLEVBQUU7WUFDVixTQUFTLEVBQUUsV0FBVztZQUN0QixpQkFBaUIsRUFBRSxtQkFBbUI7WUFDdEMsYUFBYSxFQUFFO2dCQUNiLGNBQWMsRUFBRSxDQUFDO3dCQUNmLGNBQWMsRUFBRSxnQkFBZ0I7d0JBQ2hDLFVBQVUsRUFBRTs0QkFDVixHQUFHLEVBQUUsT0FBTzt5QkFDYjtxQkFDRixDQUFDO2dCQUNGLHlCQUF5QixFQUFFLENBQUMscUJBQXFCLENBQUM7Z0JBQ2xELGFBQWEsRUFBRSxDQUFDO2dCQUNoQixZQUFZLEVBQUU7b0JBQ1osbUJBQW1CLEVBQUUsQ0FBQztvQkFDdEIsb0JBQW9CLEVBQUU7d0JBQ3BCLDBCQUEwQixFQUFFLENBQUM7d0JBQzdCLGtCQUFrQixFQUFFLENBQUMsbUJBQW1CLENBQUM7d0JBQ3pDLG9CQUFvQixFQUFFLENBQUMscUJBQXFCLENBQUM7cUJBQzlDO2lCQUNGO2FBQ0Y7U0FDRjtLQUNGLENBQUMsQ0FBQztBQUNMLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLGlIQUFpSCxFQUFFLEdBQUcsRUFBRTtJQUMzSCxPQUFPO0lBQ1AsTUFBTSxJQUFJLEdBQUcsSUFBSSxLQUFLLENBQUMsNEJBQTRCLENBQUMsS0FBSyxFQUFFLE1BQU0sRUFBRTtRQUNqRSxTQUFTLEVBQUUsR0FBRyxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsYUFBYSxDQUFDO1FBQzNDLGlCQUFpQixFQUFFLG1CQUFtQjtRQUN0QyxhQUFhLEVBQUU7WUFDYixhQUFhLEVBQUUsQ0FBQztTQUNqQjtLQUNGLENBQUMsQ0FBQztJQUVILE9BQU87SUFDUCxNQUFNLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsV0FBVyxFQUFFLENBQUMsQ0FBQyxDQUFDLE9BQU8sQ0FBQztRQUNoRCxJQUFJLEVBQUUsTUFBTTtRQUNaLFFBQVEsRUFBRTtZQUNSLFVBQVUsRUFBRTtnQkFDVixFQUFFO2dCQUNGO29CQUNFLE1BQU07b0JBQ047d0JBQ0UsR0FBRyxFQUFFLGdCQUFnQjtxQkFDdEI7b0JBQ0Qsc0RBQXNEO2lCQUN2RDthQUNGO1NBQ0Y7UUFDRCxHQUFHLEVBQUUsSUFBSTtRQUNULFVBQVUsRUFBRTtZQUNWLGFBQWEsRUFBRSxhQUFhO1lBQzVCLG1CQUFtQixFQUFFLG1CQUFtQjtZQUN4QyxlQUFlLEVBQUU7Z0JBQ2YsYUFBYSxFQUFFLENBQUM7YUFDakI7U0FDRjtLQUNGLENBQUMsQ0FBQztBQUNMLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLGtIQUFrSCxFQUFFLEdBQUcsRUFBRTtJQUM1SCxPQUFPO0lBQ1AsTUFBTSxJQUFJLEdBQUcsSUFBSSxLQUFLLENBQUMsNEJBQTRCLENBQUMsS0FBSyxFQUFFLE1BQU0sRUFBRTtRQUNqRSxTQUFTLEVBQUUsV0FBVztRQUN0QixpQkFBaUIsRUFBRSxHQUFHLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxxQkFBcUIsQ0FBQztRQUMzRCxhQUFhLEVBQUU7WUFDYixhQUFhLEVBQUUsQ0FBQztTQUNqQjtLQUNGLENBQUMsQ0FBQztJQUVILE9BQU87SUFDUCxNQUFNLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsV0FBVyxFQUFFLENBQUMsQ0FBQyxDQUFDLE9BQU8sQ0FBQztRQUNoRCxJQUFJLEVBQUUsTUFBTTtRQUNaLFFBQVEsRUFBRTtZQUNSLFVBQVUsRUFBRTtnQkFDVixFQUFFO2dCQUNGO29CQUNFLE1BQU07b0JBQ047d0JBQ0UsR0FBRyxFQUFFLGdCQUFnQjtxQkFDdEI7b0JBQ0Qsc0RBQXNEO2lCQUN2RDthQUNGO1NBQ0Y7UUFDRCxHQUFHLEVBQUUsSUFBSTtRQUNULFVBQVUsRUFBRTtZQUNWLFdBQVcsRUFBRSxXQUFXO1lBQ3hCLHFCQUFxQixFQUFFLHFCQUFxQjtZQUM1QyxlQUFlLEVBQUU7Z0JBQ2YsYUFBYSxFQUFFLENBQUM7YUFDakI7U0FDRjtLQUNGLENBQUMsQ0FBQztBQUNMLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLG9HQUFvRyxFQUFFLEdBQUcsRUFBRTtJQUM5RyxPQUFPO0lBQ1AsTUFBTSxJQUFJLEdBQUcsSUFBSSxLQUFLLENBQUMsNEJBQTRCLENBQUMsS0FBSyxFQUFFLE1BQU0sRUFBRTtRQUNqRSxTQUFTLEVBQUUsV0FBVztRQUN0QixpQkFBaUIsRUFBRSxtQkFBbUI7UUFDdEMsYUFBYSxFQUFFO1lBQ2IsYUFBYSxFQUFFLEdBQUcsQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLGlCQUFpQixDQUFDO1NBQ3BEO0tBQ0YsQ0FBQyxDQUFDO0lBRUgsT0FBTztJQUNQLE1BQU0sQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxXQUFXLEVBQUUsQ0FBQyxDQUFDLENBQUMsT0FBTyxDQUFDO1FBQ2hELElBQUksRUFBRSxNQUFNO1FBQ1osUUFBUSxFQUFFO1lBQ1IsVUFBVSxFQUFFO2dCQUNWLEVBQUU7Z0JBQ0Y7b0JBQ0UsTUFBTTtvQkFDTjt3QkFDRSxHQUFHLEVBQUUsZ0JBQWdCO3FCQUN0QjtvQkFDRCxzREFBc0Q7aUJBQ3ZEO2FBQ0Y7U0FDRjtRQUNELEdBQUcsRUFBRSxJQUFJO1FBQ1QsVUFBVSxFQUFFO1lBQ1YsU0FBUyxFQUFFLFdBQVc7WUFDdEIsaUJBQWlCLEVBQUUsbUJBQW1CO1lBQ3RDLGFBQWEsRUFBRTtnQkFDYixpQkFBaUIsRUFBRSxpQkFBaUI7YUFDckM7U0FDRjtLQUNGLENBQUMsQ0FBQztBQUNMLENBQUMsQ0FBQyxDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0ICogYXMgc2ZuIGZyb20gJ0Bhd3MtY2RrL2F3cy1zdGVwZnVuY3Rpb25zJztcbmltcG9ydCAqIGFzIGNkayBmcm9tICdAYXdzLWNkay9jb3JlJztcbmltcG9ydCAqIGFzIHRhc2tzIGZyb20gJy4uLy4uL2xpYic7XG5cbmxldCBzdGFjazogY2RrLlN0YWNrO1xuXG5iZWZvcmVFYWNoKCgpID0+IHtcbiAgLy8gR0lWRU5cbiAgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG59KTtcblxudGVzdCgnTW9kaWZ5IGFuIEluc3RhbmNlR3JvdXAgd2l0aCBzdGF0aWMgQ2x1c3RlcklkLCBJbnN0YW5jZUdyb3VwTmFtZSwgYW5kIEluc3RhbmNlR3JvdXAnLCAoKSA9PiB7XG4gIC8vIFdIRU5cbiAgY29uc3QgdGFzayA9IG5ldyB0YXNrcy5FbXJNb2RpZnlJbnN0YW5jZUdyb3VwQnlOYW1lKHN0YWNrLCAnVGFzaycsIHtcbiAgICBjbHVzdGVySWQ6ICdDbHVzdGVySWQnLFxuICAgIGluc3RhbmNlR3JvdXBOYW1lOiAnSW5zdGFuY2VHcm91cE5hbWUnLFxuICAgIGluc3RhbmNlR3JvdXA6IHtcbiAgICAgIGNvbmZpZ3VyYXRpb25zOiBbe1xuICAgICAgICBjbGFzc2lmaWNhdGlvbjogJ0NsYXNzaWZpY2F0aW9uJyxcbiAgICAgICAgcHJvcGVydGllczoge1xuICAgICAgICAgIEtleTogJ1ZhbHVlJyxcbiAgICAgICAgfSxcbiAgICAgIH1dLFxuICAgICAgZUMySW5zdGFuY2VJZHNUb1Rlcm1pbmF0ZTogWydJbnN0YW5jZVRvVGVybWluYXRlJ10sXG4gICAgICBpbnN0YW5jZUNvdW50OiAxLFxuICAgICAgc2hyaW5rUG9saWN5OiB7XG4gICAgICAgIGRlY29tbWlzc2lvblRpbWVvdXQ6IGNkay5EdXJhdGlvbi5zZWNvbmRzKDEpLFxuICAgICAgICBpbnN0YW5jZVJlc2l6ZVBvbGljeToge1xuICAgICAgICAgIGluc3RhbmNlVGVybWluYXRpb25UaW1lb3V0OiBjZGsuRHVyYXRpb24uc2Vjb25kcygxKSxcbiAgICAgICAgICBpbnN0YW5jZXNUb1Byb3RlY3Q6IFsnSW5zdGFuY2VUb1Byb3RlY3QnXSxcbiAgICAgICAgICBpbnN0YW5jZXNUb1Rlcm1pbmF0ZTogWydJbnN0YW5jZVRvVGVybWluYXRlJ10sXG4gICAgICAgIH0sXG4gICAgICB9LFxuICAgIH0sXG4gIH0pO1xuXG4gIC8vIFRIRU5cbiAgZXhwZWN0KHN0YWNrLnJlc29sdmUodGFzay50b1N0YXRlSnNvbigpKSkudG9FcXVhbCh7XG4gICAgVHlwZTogJ1Rhc2snLFxuICAgIFJlc291cmNlOiB7XG4gICAgICAnRm46OkpvaW4nOiBbXG4gICAgICAgICcnLFxuICAgICAgICBbXG4gICAgICAgICAgJ2FybjonLFxuICAgICAgICAgIHtcbiAgICAgICAgICAgIFJlZjogJ0FXUzo6UGFydGl0aW9uJyxcbiAgICAgICAgICB9LFxuICAgICAgICAgICc6c3RhdGVzOjo6ZWxhc3RpY21hcHJlZHVjZTptb2RpZnlJbnN0YW5jZUdyb3VwQnlOYW1lJyxcbiAgICAgICAgXSxcbiAgICAgIF0sXG4gICAgfSxcbiAgICBFbmQ6IHRydWUsXG4gICAgUGFyYW1ldGVyczoge1xuICAgICAgQ2x1c3RlcklkOiAnQ2x1c3RlcklkJyxcbiAgICAgIEluc3RhbmNlR3JvdXBOYW1lOiAnSW5zdGFuY2VHcm91cE5hbWUnLFxuICAgICAgSW5zdGFuY2VHcm91cDoge1xuICAgICAgICBDb25maWd1cmF0aW9uczogW3tcbiAgICAgICAgICBDbGFzc2lmaWNhdGlvbjogJ0NsYXNzaWZpY2F0aW9uJyxcbiAgICAgICAgICBQcm9wZXJ0aWVzOiB7XG4gICAgICAgICAgICBLZXk6ICdWYWx1ZScsXG4gICAgICAgICAgfSxcbiAgICAgICAgfV0sXG4gICAgICAgIEVDMkluc3RhbmNlSWRzVG9UZXJtaW5hdGU6IFsnSW5zdGFuY2VUb1Rlcm1pbmF0ZSddLFxuICAgICAgICBJbnN0YW5jZUNvdW50OiAxLFxuICAgICAgICBTaHJpbmtQb2xpY3k6IHtcbiAgICAgICAgICBEZWNvbW1pc3Npb25UaW1lb3V0OiAxLFxuICAgICAgICAgIEluc3RhbmNlUmVzaXplUG9saWN5OiB7XG4gICAgICAgICAgICBJbnN0YW5jZVRlcm1pbmF0aW9uVGltZW91dDogMSxcbiAgICAgICAgICAgIEluc3RhbmNlc1RvUHJvdGVjdDogWydJbnN0YW5jZVRvUHJvdGVjdCddLFxuICAgICAgICAgICAgSW5zdGFuY2VzVG9UZXJtaW5hdGU6IFsnSW5zdGFuY2VUb1Rlcm1pbmF0ZSddLFxuICAgICAgICAgIH0sXG4gICAgICAgIH0sXG4gICAgICB9LFxuICAgIH0sXG4gIH0pO1xufSk7XG5cbnRlc3QoJ01vZGlmeSBhbiBJbnN0YW5jZUdyb3VwIHdpdGggQ2x1c3RlcklkIGZyb20gcGF5bG9hZCBhbmQgc3RhdGljIEluc3RhbmNlR3JvdXBOYW1lIGFuZCBJbnN0YW5jZUdyb3VwQ29uZmlndXJhdGlvbicsICgpID0+IHtcbiAgLy8gV0hFTlxuICBjb25zdCB0YXNrID0gbmV3IHRhc2tzLkVtck1vZGlmeUluc3RhbmNlR3JvdXBCeU5hbWUoc3RhY2ssICdUYXNrJywge1xuICAgIGNsdXN0ZXJJZDogc2ZuLkRhdGEuc3RyaW5nQXQoJyQuQ2x1c3RlcklkJyksXG4gICAgaW5zdGFuY2VHcm91cE5hbWU6ICdJbnN0YW5jZUdyb3VwTmFtZScsXG4gICAgaW5zdGFuY2VHcm91cDoge1xuICAgICAgaW5zdGFuY2VDb3VudDogMSxcbiAgICB9LFxuICB9KTtcblxuICAvLyBUSEVOXG4gIGV4cGVjdChzdGFjay5yZXNvbHZlKHRhc2sudG9TdGF0ZUpzb24oKSkpLnRvRXF1YWwoe1xuICAgIFR5cGU6ICdUYXNrJyxcbiAgICBSZXNvdXJjZToge1xuICAgICAgJ0ZuOjpKb2luJzogW1xuICAgICAgICAnJyxcbiAgICAgICAgW1xuICAgICAgICAgICdhcm46JyxcbiAgICAgICAgICB7XG4gICAgICAgICAgICBSZWY6ICdBV1M6OlBhcnRpdGlvbicsXG4gICAgICAgICAgfSxcbiAgICAgICAgICAnOnN0YXRlczo6OmVsYXN0aWNtYXByZWR1Y2U6bW9kaWZ5SW5zdGFuY2VHcm91cEJ5TmFtZScsXG4gICAgICAgIF0sXG4gICAgICBdLFxuICAgIH0sXG4gICAgRW5kOiB0cnVlLFxuICAgIFBhcmFtZXRlcnM6IHtcbiAgICAgICdDbHVzdGVySWQuJCc6ICckLkNsdXN0ZXJJZCcsXG4gICAgICAnSW5zdGFuY2VHcm91cE5hbWUnOiAnSW5zdGFuY2VHcm91cE5hbWUnLFxuICAgICAgJ0luc3RhbmNlR3JvdXAnOiB7XG4gICAgICAgIEluc3RhbmNlQ291bnQ6IDEsXG4gICAgICB9LFxuICAgIH0sXG4gIH0pO1xufSk7XG5cbnRlc3QoJ01vZGlmeSBhbiBJbnN0YW5jZUdyb3VwIHdpdGggc3RhdGljIENsdXN0ZXJJZCBhbmQgSW5zdGFuY2VHcm91cENvbmZpZ3VyYXRlaW9uIGFuZCBJbnN0YW5jZUdyb3VwTmFtZSBmcm9tIHBheWxvYWQnLCAoKSA9PiB7XG4gIC8vIFdIRU5cbiAgY29uc3QgdGFzayA9IG5ldyB0YXNrcy5FbXJNb2RpZnlJbnN0YW5jZUdyb3VwQnlOYW1lKHN0YWNrLCAnVGFzaycsIHtcbiAgICBjbHVzdGVySWQ6ICdDbHVzdGVySWQnLFxuICAgIGluc3RhbmNlR3JvdXBOYW1lOiBzZm4uRGF0YS5zdHJpbmdBdCgnJC5JbnN0YW5jZUdyb3VwTmFtZScpLFxuICAgIGluc3RhbmNlR3JvdXA6IHtcbiAgICAgIGluc3RhbmNlQ291bnQ6IDEsXG4gICAgfSxcbiAgfSk7XG5cbiAgLy8gVEhFTlxuICBleHBlY3Qoc3RhY2sucmVzb2x2ZSh0YXNrLnRvU3RhdGVKc29uKCkpKS50b0VxdWFsKHtcbiAgICBUeXBlOiAnVGFzaycsXG4gICAgUmVzb3VyY2U6IHtcbiAgICAgICdGbjo6Sm9pbic6IFtcbiAgICAgICAgJycsXG4gICAgICAgIFtcbiAgICAgICAgICAnYXJuOicsXG4gICAgICAgICAge1xuICAgICAgICAgICAgUmVmOiAnQVdTOjpQYXJ0aXRpb24nLFxuICAgICAgICAgIH0sXG4gICAgICAgICAgJzpzdGF0ZXM6OjplbGFzdGljbWFwcmVkdWNlOm1vZGlmeUluc3RhbmNlR3JvdXBCeU5hbWUnLFxuICAgICAgICBdLFxuICAgICAgXSxcbiAgICB9LFxuICAgIEVuZDogdHJ1ZSxcbiAgICBQYXJhbWV0ZXJzOiB7XG4gICAgICAnQ2x1c3RlcklkJzogJ0NsdXN0ZXJJZCcsXG4gICAgICAnSW5zdGFuY2VHcm91cE5hbWUuJCc6ICckLkluc3RhbmNlR3JvdXBOYW1lJyxcbiAgICAgICdJbnN0YW5jZUdyb3VwJzoge1xuICAgICAgICBJbnN0YW5jZUNvdW50OiAxLFxuICAgICAgfSxcbiAgICB9LFxuICB9KTtcbn0pO1xuXG50ZXN0KCdNb2RpZnkgYW4gSW5zdGFuY2VHcm91cCB3aXRoIHN0YXRpYyBDbHVzdGVySWQgYW5kIEluc3RhbmNlR3JvdXBOYW1lIGFuZCBJbnN0YW5jZUNvdW50IGZyb20gcGF5bG9hZCcsICgpID0+IHtcbiAgLy8gV0hFTlxuICBjb25zdCB0YXNrID0gbmV3IHRhc2tzLkVtck1vZGlmeUluc3RhbmNlR3JvdXBCeU5hbWUoc3RhY2ssICdUYXNrJywge1xuICAgIGNsdXN0ZXJJZDogJ0NsdXN0ZXJJZCcsXG4gICAgaW5zdGFuY2VHcm91cE5hbWU6ICdJbnN0YW5jZUdyb3VwTmFtZScsXG4gICAgaW5zdGFuY2VHcm91cDoge1xuICAgICAgaW5zdGFuY2VDb3VudDogc2ZuLkRhdGEubnVtYmVyQXQoJyQuSW5zdGFuY2VDb3VudCcpLFxuICAgIH0sXG4gIH0pO1xuXG4gIC8vIFRIRU5cbiAgZXhwZWN0KHN0YWNrLnJlc29sdmUodGFzay50b1N0YXRlSnNvbigpKSkudG9FcXVhbCh7XG4gICAgVHlwZTogJ1Rhc2snLFxuICAgIFJlc291cmNlOiB7XG4gICAgICAnRm46OkpvaW4nOiBbXG4gICAgICAgICcnLFxuICAgICAgICBbXG4gICAgICAgICAgJ2FybjonLFxuICAgICAgICAgIHtcbiAgICAgICAgICAgIFJlZjogJ0FXUzo6UGFydGl0aW9uJyxcbiAgICAgICAgICB9LFxuICAgICAgICAgICc6c3RhdGVzOjo6ZWxhc3RpY21hcHJlZHVjZTptb2RpZnlJbnN0YW5jZUdyb3VwQnlOYW1lJyxcbiAgICAgICAgXSxcbiAgICAgIF0sXG4gICAgfSxcbiAgICBFbmQ6IHRydWUsXG4gICAgUGFyYW1ldGVyczoge1xuICAgICAgQ2x1c3RlcklkOiAnQ2x1c3RlcklkJyxcbiAgICAgIEluc3RhbmNlR3JvdXBOYW1lOiAnSW5zdGFuY2VHcm91cE5hbWUnLFxuICAgICAgSW5zdGFuY2VHcm91cDoge1xuICAgICAgICAnSW5zdGFuY2VDb3VudC4kJzogJyQuSW5zdGFuY2VDb3VudCcsXG4gICAgICB9LFxuICAgIH0sXG4gIH0pO1xufSk7XG4iXX0=