"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.EvaluateExpression = void 0;
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const sfn = require("@aws-cdk/aws-stepfunctions");
const path = require("path");
/**
 * A Step Functions Task to evaluate an expression
 *
 * OUTPUT: the output of this task is the evaluated expression.
 *
 * @experimental
 */
class EvaluateExpression {
    constructor(props) {
        this.props = props;
    }
    bind(task) {
        const matches = this.props.expression.match(/\$[.\[][.a-zA-Z[\]0-9]+/g);
        let expressionAttributeValues = {};
        if (matches) {
            expressionAttributeValues = matches.reduce((acc, m) => ({
                ...acc,
                [m]: sfn.Data.stringAt(m),
            }), {});
        }
        const evalFn = createEvalFn(this.props.runtime || lambda.Runtime.NODEJS_10_X, task);
        const parameters = {
            expression: this.props.expression,
            expressionAttributeValues,
        };
        return {
            resourceArn: evalFn.functionArn,
            policyStatements: [new iam.PolicyStatement({
                    resources: [evalFn.functionArn],
                    actions: ['lambda:InvokeFunction'],
                })],
            parameters,
        };
    }
}
exports.EvaluateExpression = EvaluateExpression;
function createEvalFn(runtime, scope) {
    const code = lambda.Code.asset(path.join(__dirname, `eval-${runtime.name}-handler`));
    const lambdaPurpose = 'Eval';
    switch (runtime) {
        case lambda.Runtime.NODEJS_10_X:
            return new lambda.SingletonFunction(scope, 'EvalFunction', {
                runtime,
                handler: 'index.handler',
                uuid: 'a0d2ce44-871b-4e74-87a1-f5e63d7c3bdc',
                lambdaPurpose,
                code,
            });
        // TODO: implement other runtimes
        default:
            throw new Error(`The runtime ${runtime.name} is currently not supported.`);
    }
}
//# sourceMappingURL=data:application/json;base64,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