"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.EmrCreateCluster = void 0;
const iam = require("@aws-cdk/aws-iam");
const sfn = require("@aws-cdk/aws-stepfunctions");
const cdk = require("@aws-cdk/core");
const task_utils_1 = require("../private/task-utils");
const cluster_utils_1 = require("./private/cluster-utils");
/**
 * A Step Functions Task to create an EMR Cluster.
 *
 * The ClusterConfiguration is defined as Parameters in the state machine definition.
 *
 * OUTPUT: the ClusterId.
 *
 * @experimental
 */
class EmrCreateCluster extends sfn.TaskStateBase {
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id, props);
        this.props = props;
        this.visibleToAllUsers = this.props.visibleToAllUsers !== undefined ? this.props.visibleToAllUsers : true;
        this.integrationPattern = props.integrationPattern || sfn.IntegrationPattern.RUN_JOB;
        task_utils_1.validatePatternSupported(this.integrationPattern, EmrCreateCluster.SUPPORTED_INTEGRATION_PATTERNS);
        this._autoScalingRole = this.props.autoScalingRole;
        // If the Roles are undefined then they weren't provided, so create them
        this._serviceRole = (_a = this.props.serviceRole) !== null && _a !== void 0 ? _a : this.createServiceRole();
        this._clusterRole = (_b = this.props.clusterRole) !== null && _b !== void 0 ? _b : this.createClusterRole();
        // AutoScaling roles are not valid with InstanceFleet clusters.
        // Attempt to create only if .instances.instanceFleets is undefined or empty
        if (this.props.instances.instanceFleets === undefined || this.props.instances.instanceFleets.length === 0) {
            this._autoScalingRole = this._autoScalingRole || this.createAutoScalingRole();
            // If InstanceFleets are used and an AutoScaling Role is specified, throw an error
        }
        else if (this._autoScalingRole !== undefined) {
            throw new Error('Auto Scaling roles can not be specified with instance fleets.');
        }
        this.taskPolicies = this.createPolicyStatements(this._serviceRole, this._clusterRole, this._autoScalingRole);
    }
    /**
     * The service role for the EMR Cluster.
     *
     * Only available after task has been added to a state machine.
     */
    get serviceRole() {
        if (this._serviceRole === undefined) {
            throw new Error('role not available yet--use the object in a Task first');
        }
        return this._serviceRole;
    }
    /**
     * The instance role for the EMR Cluster.
     *
     * Only available after task has been added to a state machine.
     */
    get clusterRole() {
        if (this._clusterRole === undefined) {
            throw new Error('role not available yet--use the object in a Task first');
        }
        return this._clusterRole;
    }
    /**
     * The autoscaling role for the EMR Cluster.
     *
     * Only available after task has been added to a state machine.
     */
    get autoScalingRole() {
        if (this._autoScalingRole === undefined) {
            throw new Error('role not available yet--use the object in a Task first');
        }
        return this._autoScalingRole;
    }
    renderTask() {
        var _a, _b, _c;
        return {
            Resource: task_utils_1.integrationResourceArn('elasticmapreduce', 'createCluster', this.integrationPattern),
            Parameters: sfn.FieldUtils.renderObject({
                Instances: cluster_utils_1.InstancesConfigPropertyToJson(this.props.instances),
                JobFlowRole: cdk.stringToCloudFormation(this._clusterRole.roleName),
                Name: cdk.stringToCloudFormation(this.props.name),
                ServiceRole: cdk.stringToCloudFormation(this._serviceRole.roleName),
                AdditionalInfo: cdk.stringToCloudFormation(this.props.additionalInfo),
                Applications: cdk.listMapper(cluster_utils_1.ApplicationConfigPropertyToJson)(this.props.applications),
                AutoScalingRole: cdk.stringToCloudFormation((_a = this._autoScalingRole) === null || _a === void 0 ? void 0 : _a.roleName),
                BootstrapActions: cdk.listMapper(cluster_utils_1.BootstrapActionConfigToJson)(this.props.bootstrapActions),
                Configurations: cdk.listMapper(cluster_utils_1.ConfigurationPropertyToJson)(this.props.configurations),
                CustomAmiId: cdk.stringToCloudFormation(this.props.customAmiId),
                EbsRootVolumeSize: (_b = this.props.ebsRootVolumeSize) === null || _b === void 0 ? void 0 : _b.toGibibytes(),
                KerberosAttributes: this.props.kerberosAttributes ? cluster_utils_1.KerberosAttributesPropertyToJson(this.props.kerberosAttributes) : undefined,
                LogUri: cdk.stringToCloudFormation(this.props.logUri),
                ReleaseLabel: cdk.stringToCloudFormation(this.props.releaseLabel),
                ScaleDownBehavior: cdk.stringToCloudFormation((_c = this.props.scaleDownBehavior) === null || _c === void 0 ? void 0 : _c.valueOf()),
                SecurityConfiguration: cdk.stringToCloudFormation(this.props.securityConfiguration),
                ...(this.props.tags ? this.renderTags(this.props.tags) : undefined),
                VisibleToAllUsers: cdk.booleanToCloudFormation(this.visibleToAllUsers),
            }),
        };
    }
    renderTags(tags) {
        return tags ? { Tags: Object.keys(tags).map((key) => ({ Key: key, Value: tags[key] })) } : {};
    }
    /**
     * This generates the PolicyStatements required by the Task to call CreateCluster.
     */
    createPolicyStatements(serviceRole, clusterRole, autoScalingRole) {
        const stack = cdk.Stack.of(this);
        const policyStatements = [
            new iam.PolicyStatement({
                actions: ['elasticmapreduce:RunJobFlow', 'elasticmapreduce:DescribeCluster', 'elasticmapreduce:TerminateJobFlows'],
                resources: ['*'],
            }),
        ];
        // Allow the StateMachine to PassRole to Cluster roles
        policyStatements.push(new iam.PolicyStatement({
            actions: ['iam:PassRole'],
            resources: [serviceRole.roleArn, clusterRole.roleArn],
        }));
        if (autoScalingRole !== undefined) {
            policyStatements.push(new iam.PolicyStatement({
                actions: ['iam:PassRole'],
                resources: [autoScalingRole.roleArn],
            }));
        }
        if (this.integrationPattern === sfn.IntegrationPattern.RUN_JOB) {
            policyStatements.push(new iam.PolicyStatement({
                actions: ['events:PutTargets', 'events:PutRule', 'events:DescribeRule'],
                resources: [
                    stack.formatArn({
                        service: 'events',
                        resource: 'rule',
                        resourceName: 'StepFunctionsGetEventForEMRRunJobFlowRule',
                    }),
                ],
            }));
        }
        return policyStatements;
    }
    /**
     * Generate the Role used by the EMR Service
     */
    createServiceRole() {
        return new iam.Role(this, 'ServiceRole', {
            assumedBy: new iam.ServicePrincipal('elasticmapreduce.amazonaws.com'),
            managedPolicies: [iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonElasticMapReduceRole')],
        });
    }
    /**
     * Generate the Role and Instance Profile used by the EC2 instances
     *
     * Data access permissions will need to be updated by the user
     */
    createClusterRole() {
        const role = new iam.Role(this, 'InstanceRole', {
            assumedBy: new iam.ServicePrincipal('ec2.amazonaws.com'),
        });
        new iam.CfnInstanceProfile(this, 'InstanceProfile', {
            roles: [role.roleName],
            instanceProfileName: role.roleName,
        });
        return role;
    }
    /**
     * Generate the Role used to AutoScale the Cluster
     */
    createAutoScalingRole() {
        var _a;
        const role = new iam.Role(this, 'AutoScalingRole', {
            assumedBy: new iam.ServicePrincipal('elasticmapreduce.amazonaws.com'),
            managedPolicies: [iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonElasticMapReduceforAutoScalingRole')],
        });
        (_a = role.assumeRolePolicy) === null || _a === void 0 ? void 0 : _a.addStatements(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            principals: [new iam.ServicePrincipal('application-autoscaling.amazonaws.com')],
            actions: ['sts:AssumeRole'],
        }));
        return role;
    }
}
exports.EmrCreateCluster = EmrCreateCluster;
EmrCreateCluster.SUPPORTED_INTEGRATION_PATTERNS = [
    sfn.IntegrationPattern.REQUEST_RESPONSE,
    sfn.IntegrationPattern.RUN_JOB,
];
(function (EmrCreateCluster) {
    /**
     * Valid valus for the Cluster ScaleDownBehavior
     *
     * @experimental
     */
    let EmrClusterScaleDownBehavior;
    (function (EmrClusterScaleDownBehavior) {
        /**
         * Indicates that Amazon EMR terminates nodes at the instance-hour boundary, regardless of when the request to terminate the instance was
         * submitted. This option is only available with Amazon EMR 5.1.0 and later and is the default for clusters created using that version
         */
        EmrClusterScaleDownBehavior["TERMINATE_AT_INSTANCE_HOUR"] = "TERMINATE_AT_INSTANCE_HOUR";
        /**
         * Indicates that Amazon EMR blacklists and drains tasks from nodes before terminating the Amazon EC2 instances, regardless of the
         * instance-hour boundary.
         */
        EmrClusterScaleDownBehavior["TERMINATE_AT_TASK_COMPLETION"] = "TERMINATE_AT_TASK_COMPLETION";
    })(EmrClusterScaleDownBehavior = EmrCreateCluster.EmrClusterScaleDownBehavior || (EmrCreateCluster.EmrClusterScaleDownBehavior = {}));
    /**
     * Instance Role Types
     *
     * @experimental
     */
    let InstanceRoleType;
    (function (InstanceRoleType) {
        /**
         * Master Node
         */
        InstanceRoleType["MASTER"] = "MASTER";
        /**
         * Core Node
         */
        InstanceRoleType["CORE"] = "CORE";
        /**
         * Task Node
         */
        InstanceRoleType["TASK"] = "TASK";
    })(InstanceRoleType = EmrCreateCluster.InstanceRoleType || (EmrCreateCluster.InstanceRoleType = {}));
    /**
     * EBS Volume Types
     *
     * @experimental
     */
    let EbsBlockDeviceVolumeType;
    (function (EbsBlockDeviceVolumeType) {
        /**
         * gp2 Volume Type
         */
        EbsBlockDeviceVolumeType["GP2"] = "gp2";
        /**
         * io1 Volume Type
         */
        EbsBlockDeviceVolumeType["IO1"] = "io1";
        /**
         * Standard Volume Type
         */
        EbsBlockDeviceVolumeType["STANDARD"] = "standard";
    })(EbsBlockDeviceVolumeType = EmrCreateCluster.EbsBlockDeviceVolumeType || (EmrCreateCluster.EbsBlockDeviceVolumeType = {}));
    /**
     * Spot Timeout Actions
     *
     * @experimental
     */
    let SpotTimeoutAction;
    (function (SpotTimeoutAction) {
        /**\
         * SWITCH_TO_ON_DEMAND
         */
        SpotTimeoutAction["SWITCH_TO_ON_DEMAND"] = "SWITCH_TO_ON_DEMAND";
        /**
         * TERMINATE_CLUSTER
         */
        SpotTimeoutAction["TERMINATE_CLUSTER"] = "TERMINATE_CLUSTER";
    })(SpotTimeoutAction = EmrCreateCluster.SpotTimeoutAction || (EmrCreateCluster.SpotTimeoutAction = {}));
    /**
     * CloudWatch Alarm Comparison Operators
     *
     * @experimental
     */
    let CloudWatchAlarmComparisonOperator;
    (function (CloudWatchAlarmComparisonOperator) {
        /**
         * GREATER_THAN_OR_EQUAL
         */
        CloudWatchAlarmComparisonOperator["GREATER_THAN_OR_EQUAL"] = "GREATER_THAN_OR_EQUAL";
        /**
         * GREATER_THAN
         */
        CloudWatchAlarmComparisonOperator["GREATER_THAN"] = "GREATER_THAN";
        /**
         * LESS_THAN
         */
        CloudWatchAlarmComparisonOperator["LESS_THAN"] = "LESS_THAN";
        /**
         * LESS_THAN_OR_EQUAL
         */
        CloudWatchAlarmComparisonOperator["LESS_THAN_OR_EQUAL"] = "LESS_THAN_OR_EQUAL";
    })(CloudWatchAlarmComparisonOperator = EmrCreateCluster.CloudWatchAlarmComparisonOperator || (EmrCreateCluster.CloudWatchAlarmComparisonOperator = {}));
    /**
     * CloudWatch Alarm Statistics
     *
     * @experimental
     */
    let CloudWatchAlarmStatistic;
    (function (CloudWatchAlarmStatistic) {
        /**
         * SAMPLE_COUNT
         */
        CloudWatchAlarmStatistic["SAMPLE_COUNT"] = "SAMPLE_COUNT";
        /**
         * AVERAGE
         */
        CloudWatchAlarmStatistic["AVERAGE"] = "AVERAGE";
        /**
         * SUM
         */
        CloudWatchAlarmStatistic["SUM"] = "SUM";
        /**
         * MINIMUM
         */
        CloudWatchAlarmStatistic["MINIMUM"] = "MINIMUM";
        /**
         * MAXIMUM
         */
        CloudWatchAlarmStatistic["MAXIMUM"] = "MAXIMUM";
    })(CloudWatchAlarmStatistic = EmrCreateCluster.CloudWatchAlarmStatistic || (EmrCreateCluster.CloudWatchAlarmStatistic = {}));
    /**
     * CloudWatch Alarm Units
     *
     * @experimental
     */
    let CloudWatchAlarmUnit;
    (function (CloudWatchAlarmUnit) {
        /**
         * NONE
         */
        CloudWatchAlarmUnit["NONE"] = "NONE";
        /**
         * SECONDS
         */
        CloudWatchAlarmUnit["SECONDS"] = "SECONDS";
        /**
         * MICRO_SECONDS
         */
        CloudWatchAlarmUnit["MICRO_SECONDS"] = "MICRO_SECONDS";
        /**
         * MILLI_SECONDS
         */
        CloudWatchAlarmUnit["MILLI_SECONDS"] = "MILLI_SECONDS";
        /**
         * BYTES
         */
        CloudWatchAlarmUnit["BYTES"] = "BYTES";
        /**
         * KILO_BYTES
         */
        CloudWatchAlarmUnit["KILO_BYTES"] = "KILO_BYTES";
        /**
         * MEGA_BYTES
         */
        CloudWatchAlarmUnit["MEGA_BYTES"] = "MEGA_BYTES";
        /**
         * GIGA_BYTES
         */
        CloudWatchAlarmUnit["GIGA_BYTES"] = "GIGA_BYTES";
        /**
         * TERA_BYTES
         */
        CloudWatchAlarmUnit["TERA_BYTES"] = "TERA_BYTES";
        /**
         * BITS
         */
        CloudWatchAlarmUnit["BITS"] = "BITS";
        /**
         * KILO_BITS
         */
        CloudWatchAlarmUnit["KILO_BITS"] = "KILO_BITS";
        /**
         * MEGA_BITS
         */
        CloudWatchAlarmUnit["MEGA_BITS"] = "MEGA_BITS";
        /**
         * GIGA_BITS
         */
        CloudWatchAlarmUnit["GIGA_BITS"] = "GIGA_BITS";
        /**
         * TERA_BITS
         */
        CloudWatchAlarmUnit["TERA_BITS"] = "TERA_BITS";
        /**
         * PERCENT
         */
        CloudWatchAlarmUnit["PERCENT"] = "PERCENT";
        /**
         * COUNT
         */
        CloudWatchAlarmUnit["COUNT"] = "COUNT";
        /**
         * BYTES_PER_SECOND
         */
        CloudWatchAlarmUnit["BYTES_PER_SECOND"] = "BYTES_PER_SECOND";
        /**
         * KILO_BYTES_PER_SECOND
         */
        CloudWatchAlarmUnit["KILO_BYTES_PER_SECOND"] = "KILO_BYTES_PER_SECOND";
        /**
         * MEGA_BYTES_PER_SECOND
         */
        CloudWatchAlarmUnit["MEGA_BYTES_PER_SECOND"] = "MEGA_BYTES_PER_SECOND";
        /**
         * GIGA_BYTES_PER_SECOND
         */
        CloudWatchAlarmUnit["GIGA_BYTES_PER_SECOND"] = "GIGA_BYTES_PER_SECOND";
        /**
         * TERA_BYTES_PER_SECOND
         */
        CloudWatchAlarmUnit["TERA_BYTES_PER_SECOND"] = "TERA_BYTES_PER_SECOND";
        /**
         * BITS_PER_SECOND
         */
        CloudWatchAlarmUnit["BITS_PER_SECOND"] = "BITS_PER_SECOND";
        /**
         * KILO_BITS_PER_SECOND
         */
        CloudWatchAlarmUnit["KILO_BITS_PER_SECOND"] = "KILO_BITS_PER_SECOND";
        /**
         * MEGA_BITS_PER_SECOND
         */
        CloudWatchAlarmUnit["MEGA_BITS_PER_SECOND"] = "MEGA_BITS_PER_SECOND";
        /**
         * GIGA_BITS_PER_SECOND
         */
        CloudWatchAlarmUnit["GIGA_BITS_PER_SECOND"] = "GIGA_BITS_PER_SECOND";
        /**
         * TERA_BITS_PER_SECOND
         */
        CloudWatchAlarmUnit["TERA_BITS_PER_SECOND"] = "TERA_BITS_PER_SECOND";
        /**
         * COUNT_PER_SECOND
         */
        CloudWatchAlarmUnit["COUNT_PER_SECOND"] = "COUNT_PER_SECOND";
    })(CloudWatchAlarmUnit = EmrCreateCluster.CloudWatchAlarmUnit || (EmrCreateCluster.CloudWatchAlarmUnit = {}));
    /**
     * EC2 Instance Market
     *
     * @experimental
     */
    let InstanceMarket;
    (function (InstanceMarket) {
        /**
         * On Demand Instance
         */
        InstanceMarket["ON_DEMAND"] = "ON_DEMAND";
        /**
         * Spot Instance
         */
        InstanceMarket["SPOT"] = "SPOT";
    })(InstanceMarket = EmrCreateCluster.InstanceMarket || (EmrCreateCluster.InstanceMarket = {}));
    /**
     * AutoScaling Adjustment Type
     *
     * @experimental
     */
    let ScalingAdjustmentType;
    (function (ScalingAdjustmentType) {
        /**
         * CHANGE_IN_CAPACITY
         */
        ScalingAdjustmentType["CHANGE_IN_CAPACITY"] = "CHANGE_IN_CAPACITY";
        /**
         * PERCENT_CHANGE_IN_CAPACITY
         */
        ScalingAdjustmentType["PERCENT_CHANGE_IN_CAPACITY"] = "PERCENT_CHANGE_IN_CAPACITY";
        /**
         * EXACT_CAPACITY
         */
        ScalingAdjustmentType["EXACT_CAPACITY"] = "EXACT_CAPACITY";
    })(ScalingAdjustmentType = EmrCreateCluster.ScalingAdjustmentType || (EmrCreateCluster.ScalingAdjustmentType = {}));
})(EmrCreateCluster = exports.EmrCreateCluster || (exports.EmrCreateCluster = {}));
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZW1yLWNyZWF0ZS1jbHVzdGVyLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiZW1yLWNyZWF0ZS1jbHVzdGVyLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUFBLHdDQUF3QztBQUN4QyxrREFBa0Q7QUFDbEQscUNBQXFDO0FBQ3JDLHNEQUF5RjtBQUN6RiwyREFNaUM7QUF5SWpDOzs7Ozs7OztHQVFHO0FBQ0gsTUFBYSxnQkFBaUIsU0FBUSxHQUFHLENBQUMsYUFBYTtJQWdCckQsWUFBWSxLQUFvQixFQUFFLEVBQVUsRUFBbUIsS0FBNEI7O1FBQ3pGLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxFQUFFLEtBQUssQ0FBQyxDQUFDO1FBRHFDLFVBQUssR0FBTCxLQUFLLENBQXVCO1FBRXpGLElBQUksQ0FBQyxpQkFBaUIsR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLGlCQUFpQixLQUFLLFNBQVMsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxpQkFBaUIsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDO1FBQzFHLElBQUksQ0FBQyxrQkFBa0IsR0FBRyxLQUFLLENBQUMsa0JBQWtCLElBQUksR0FBRyxDQUFDLGtCQUFrQixDQUFDLE9BQU8sQ0FBQztRQUNyRixxQ0FBd0IsQ0FBQyxJQUFJLENBQUMsa0JBQWtCLEVBQUUsZ0JBQWdCLENBQUMsOEJBQThCLENBQUMsQ0FBQztRQUVuRyxJQUFJLENBQUMsZ0JBQWdCLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxlQUFlLENBQUM7UUFFbkQsd0VBQXdFO1FBQ3hFLElBQUksQ0FBQyxZQUFZLFNBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxXQUFXLG1DQUFJLElBQUksQ0FBQyxpQkFBaUIsRUFBRSxDQUFDO1FBQ3ZFLElBQUksQ0FBQyxZQUFZLFNBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxXQUFXLG1DQUFJLElBQUksQ0FBQyxpQkFBaUIsRUFBRSxDQUFDO1FBRXZFLCtEQUErRDtRQUMvRCw0RUFBNEU7UUFDNUUsSUFBSSxJQUFJLENBQUMsS0FBSyxDQUFDLFNBQVMsQ0FBQyxjQUFjLEtBQUssU0FBUyxJQUFJLElBQUksQ0FBQyxLQUFLLENBQUMsU0FBUyxDQUFDLGNBQWMsQ0FBQyxNQUFNLEtBQUssQ0FBQyxFQUFFO1lBQ3pHLElBQUksQ0FBQyxnQkFBZ0IsR0FBRyxJQUFJLENBQUMsZ0JBQWdCLElBQUksSUFBSSxDQUFDLHFCQUFxQixFQUFFLENBQUM7WUFDOUUsa0ZBQWtGO1NBQ25GO2FBQU0sSUFBSSxJQUFJLENBQUMsZ0JBQWdCLEtBQUssU0FBUyxFQUFFO1lBQzlDLE1BQU0sSUFBSSxLQUFLLENBQUMsK0RBQStELENBQUMsQ0FBQztTQUNsRjtRQUVELElBQUksQ0FBQyxZQUFZLEdBQUcsSUFBSSxDQUFDLHNCQUFzQixDQUFDLElBQUksQ0FBQyxZQUFZLEVBQUUsSUFBSSxDQUFDLFlBQVksRUFBRSxJQUFJLENBQUMsZ0JBQWdCLENBQUMsQ0FBQztJQUMvRyxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNILElBQVcsV0FBVztRQUNwQixJQUFJLElBQUksQ0FBQyxZQUFZLEtBQUssU0FBUyxFQUFFO1lBQ25DLE1BQU0sSUFBSSxLQUFLLENBQUMsd0RBQXdELENBQUMsQ0FBQztTQUMzRTtRQUNELE9BQU8sSUFBSSxDQUFDLFlBQVksQ0FBQztJQUMzQixDQUFDO0lBRUQ7Ozs7T0FJRztJQUNILElBQVcsV0FBVztRQUNwQixJQUFJLElBQUksQ0FBQyxZQUFZLEtBQUssU0FBUyxFQUFFO1lBQ25DLE1BQU0sSUFBSSxLQUFLLENBQUMsd0RBQXdELENBQUMsQ0FBQztTQUMzRTtRQUNELE9BQU8sSUFBSSxDQUFDLFlBQVksQ0FBQztJQUMzQixDQUFDO0lBRUQ7Ozs7T0FJRztJQUNILElBQVcsZUFBZTtRQUN4QixJQUFJLElBQUksQ0FBQyxnQkFBZ0IsS0FBSyxTQUFTLEVBQUU7WUFDdkMsTUFBTSxJQUFJLEtBQUssQ0FBQyx3REFBd0QsQ0FBQyxDQUFDO1NBQzNFO1FBQ0QsT0FBTyxJQUFJLENBQUMsZ0JBQWdCLENBQUM7SUFDL0IsQ0FBQztJQUVTLFVBQVU7O1FBQ2xCLE9BQU87WUFDTCxRQUFRLEVBQUUsbUNBQXNCLENBQUMsa0JBQWtCLEVBQUUsZUFBZSxFQUFFLElBQUksQ0FBQyxrQkFBa0IsQ0FBQztZQUM5RixVQUFVLEVBQUUsR0FBRyxDQUFDLFVBQVUsQ0FBQyxZQUFZLENBQUM7Z0JBQ3RDLFNBQVMsRUFBRSw2Q0FBNkIsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLFNBQVMsQ0FBQztnQkFDOUQsV0FBVyxFQUFFLEdBQUcsQ0FBQyxzQkFBc0IsQ0FBQyxJQUFJLENBQUMsWUFBWSxDQUFDLFFBQVEsQ0FBQztnQkFDbkUsSUFBSSxFQUFFLEdBQUcsQ0FBQyxzQkFBc0IsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQztnQkFDakQsV0FBVyxFQUFFLEdBQUcsQ0FBQyxzQkFBc0IsQ0FBQyxJQUFJLENBQUMsWUFBWSxDQUFDLFFBQVEsQ0FBQztnQkFDbkUsY0FBYyxFQUFFLEdBQUcsQ0FBQyxzQkFBc0IsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLGNBQWMsQ0FBQztnQkFDckUsWUFBWSxFQUFFLEdBQUcsQ0FBQyxVQUFVLENBQUMsK0NBQStCLENBQUMsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLFlBQVksQ0FBQztnQkFDdEYsZUFBZSxFQUFFLEdBQUcsQ0FBQyxzQkFBc0IsT0FBQyxJQUFJLENBQUMsZ0JBQWdCLDBDQUFFLFFBQVEsQ0FBQztnQkFDNUUsZ0JBQWdCLEVBQUUsR0FBRyxDQUFDLFVBQVUsQ0FBQywyQ0FBMkIsQ0FBQyxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsZ0JBQWdCLENBQUM7Z0JBQzFGLGNBQWMsRUFBRSxHQUFHLENBQUMsVUFBVSxDQUFDLDJDQUEyQixDQUFDLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxjQUFjLENBQUM7Z0JBQ3RGLFdBQVcsRUFBRSxHQUFHLENBQUMsc0JBQXNCLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxXQUFXLENBQUM7Z0JBQy9ELGlCQUFpQixRQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsaUJBQWlCLDBDQUFFLFdBQVcsRUFBRTtnQkFDOUQsa0JBQWtCLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDLENBQUMsZ0RBQWdDLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDLENBQUMsQ0FBQyxTQUFTO2dCQUMvSCxNQUFNLEVBQUUsR0FBRyxDQUFDLHNCQUFzQixDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDO2dCQUNyRCxZQUFZLEVBQUUsR0FBRyxDQUFDLHNCQUFzQixDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsWUFBWSxDQUFDO2dCQUNqRSxpQkFBaUIsRUFBRSxHQUFHLENBQUMsc0JBQXNCLE9BQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxpQkFBaUIsMENBQUUsT0FBTyxHQUFHO2dCQUN0RixxQkFBcUIsRUFBRSxHQUFHLENBQUMsc0JBQXNCLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxxQkFBcUIsQ0FBQztnQkFDbkYsR0FBRyxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxDQUFDLFNBQVMsQ0FBQztnQkFDbkUsaUJBQWlCLEVBQUUsR0FBRyxDQUFDLHVCQUF1QixDQUFDLElBQUksQ0FBQyxpQkFBaUIsQ0FBQzthQUN2RSxDQUFDO1NBQ0gsQ0FBQztJQUNKLENBQUM7SUFFTyxVQUFVLENBQUMsSUFBd0M7UUFDekQsT0FBTyxJQUFJLENBQUMsQ0FBQyxDQUFDLEVBQUUsSUFBSSxFQUFFLE1BQU0sQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUMsR0FBRyxDQUFDLENBQUMsR0FBRyxFQUFFLEVBQUUsQ0FBQyxDQUFDLEVBQUUsR0FBRyxFQUFFLEdBQUcsRUFBRSxLQUFLLEVBQUUsSUFBSSxDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQztJQUNoRyxDQUFDO0lBRUQ7O09BRUc7SUFDSyxzQkFBc0IsQ0FBQyxXQUFzQixFQUFFLFdBQXNCLEVBQUUsZUFBMkI7UUFDeEcsTUFBTSxLQUFLLEdBQUcsR0FBRyxDQUFDLEtBQUssQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUM7UUFFakMsTUFBTSxnQkFBZ0IsR0FBRztZQUN2QixJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUM7Z0JBQ3RCLE9BQU8sRUFBRSxDQUFDLDZCQUE2QixFQUFFLGtDQUFrQyxFQUFFLG9DQUFvQyxDQUFDO2dCQUNsSCxTQUFTLEVBQUUsQ0FBQyxHQUFHLENBQUM7YUFDakIsQ0FBQztTQUNILENBQUM7UUFFRixzREFBc0Q7UUFDdEQsZ0JBQWdCLENBQUMsSUFBSSxDQUNuQixJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUM7WUFDdEIsT0FBTyxFQUFFLENBQUMsY0FBYyxDQUFDO1lBQ3pCLFNBQVMsRUFBRSxDQUFDLFdBQVcsQ0FBQyxPQUFPLEVBQUUsV0FBVyxDQUFDLE9BQU8sQ0FBQztTQUN0RCxDQUFDLENBQ0gsQ0FBQztRQUNGLElBQUksZUFBZSxLQUFLLFNBQVMsRUFBRTtZQUNqQyxnQkFBZ0IsQ0FBQyxJQUFJLENBQ25CLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQztnQkFDdEIsT0FBTyxFQUFFLENBQUMsY0FBYyxDQUFDO2dCQUN6QixTQUFTLEVBQUUsQ0FBQyxlQUFlLENBQUMsT0FBTyxDQUFDO2FBQ3JDLENBQUMsQ0FDSCxDQUFDO1NBQ0g7UUFFRCxJQUFJLElBQUksQ0FBQyxrQkFBa0IsS0FBSyxHQUFHLENBQUMsa0JBQWtCLENBQUMsT0FBTyxFQUFFO1lBQzlELGdCQUFnQixDQUFDLElBQUksQ0FDbkIsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDO2dCQUN0QixPQUFPLEVBQUUsQ0FBQyxtQkFBbUIsRUFBRSxnQkFBZ0IsRUFBRSxxQkFBcUIsQ0FBQztnQkFDdkUsU0FBUyxFQUFFO29CQUNULEtBQUssQ0FBQyxTQUFTLENBQUM7d0JBQ2QsT0FBTyxFQUFFLFFBQVE7d0JBQ2pCLFFBQVEsRUFBRSxNQUFNO3dCQUNoQixZQUFZLEVBQUUsMkNBQTJDO3FCQUMxRCxDQUFDO2lCQUNIO2FBQ0YsQ0FBQyxDQUNILENBQUM7U0FDSDtRQUVELE9BQU8sZ0JBQWdCLENBQUM7SUFDMUIsQ0FBQztJQUVEOztPQUVHO0lBQ0ssaUJBQWlCO1FBQ3ZCLE9BQU8sSUFBSSxHQUFHLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSxhQUFhLEVBQUU7WUFDdkMsU0FBUyxFQUFFLElBQUksR0FBRyxDQUFDLGdCQUFnQixDQUFDLGdDQUFnQyxDQUFDO1lBQ3JFLGVBQWUsRUFBRSxDQUFDLEdBQUcsQ0FBQyxhQUFhLENBQUMsd0JBQXdCLENBQUMseUNBQXlDLENBQUMsQ0FBQztTQUN6RyxDQUFDLENBQUM7SUFDTCxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNLLGlCQUFpQjtRQUN2QixNQUFNLElBQUksR0FBRyxJQUFJLEdBQUcsQ0FBQyxJQUFJLENBQUMsSUFBSSxFQUFFLGNBQWMsRUFBRTtZQUM5QyxTQUFTLEVBQUUsSUFBSSxHQUFHLENBQUMsZ0JBQWdCLENBQUMsbUJBQW1CLENBQUM7U0FDekQsQ0FBQyxDQUFDO1FBRUgsSUFBSSxHQUFHLENBQUMsa0JBQWtCLENBQUMsSUFBSSxFQUFFLGlCQUFpQixFQUFFO1lBQ2xELEtBQUssRUFBRSxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUM7WUFDdEIsbUJBQW1CLEVBQUUsSUFBSSxDQUFDLFFBQVE7U0FDbkMsQ0FBQyxDQUFDO1FBRUgsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7O09BRUc7SUFDSyxxQkFBcUI7O1FBQzNCLE1BQU0sSUFBSSxHQUFHLElBQUksR0FBRyxDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUUsaUJBQWlCLEVBQUU7WUFDakQsU0FBUyxFQUFFLElBQUksR0FBRyxDQUFDLGdCQUFnQixDQUFDLGdDQUFnQyxDQUFDO1lBQ3JFLGVBQWUsRUFBRSxDQUFDLEdBQUcsQ0FBQyxhQUFhLENBQUMsd0JBQXdCLENBQUMsdURBQXVELENBQUMsQ0FBQztTQUN2SCxDQUFDLENBQUM7UUFFSCxNQUFBLElBQUksQ0FBQyxnQkFBZ0IsMENBQUUsYUFBYSxDQUNsQyxJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUM7WUFDdEIsTUFBTSxFQUFFLEdBQUcsQ0FBQyxNQUFNLENBQUMsS0FBSztZQUN4QixVQUFVLEVBQUUsQ0FBQyxJQUFJLEdBQUcsQ0FBQyxnQkFBZ0IsQ0FBQyx1Q0FBdUMsQ0FBQyxDQUFDO1lBQy9FLE9BQU8sRUFBRSxDQUFDLGdCQUFnQixDQUFDO1NBQzVCLENBQUMsRUFDRjtRQUVGLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQzs7QUF2TUgsNENBd01DO0FBdk15QiwrQ0FBOEIsR0FBNkI7SUFDakYsR0FBRyxDQUFDLGtCQUFrQixDQUFDLGdCQUFnQjtJQUN2QyxHQUFHLENBQUMsa0JBQWtCLENBQUMsT0FBTztDQUMvQixDQUFDO0FBc01KLFdBQWlCLGdCQUFnQjtJQUMvQjs7OztPQUlHO0lBQ0gsSUFBWSwyQkFZWDtJQVpELFdBQVksMkJBQTJCO1FBQ3JDOzs7V0FHRztRQUNILHdGQUF5RCxDQUFBO1FBRXpEOzs7V0FHRztRQUNILDRGQUE2RCxDQUFBO0lBQy9ELENBQUMsRUFaVywyQkFBMkIsR0FBM0IsNENBQTJCLEtBQTNCLDRDQUEyQixRQVl0QztJQUVEOzs7O09BSUc7SUFDSCxJQUFZLGdCQWFYO0lBYkQsV0FBWSxnQkFBZ0I7UUFDMUI7O1dBRUc7UUFDSCxxQ0FBaUIsQ0FBQTtRQUNqQjs7V0FFRztRQUNILGlDQUFhLENBQUE7UUFDYjs7V0FFRztRQUNILGlDQUFhLENBQUE7SUFDZixDQUFDLEVBYlcsZ0JBQWdCLEdBQWhCLGlDQUFnQixLQUFoQixpQ0FBZ0IsUUFhM0I7SUFFRDs7OztPQUlHO0lBQ0gsSUFBWSx3QkFhWDtJQWJELFdBQVksd0JBQXdCO1FBQ2xDOztXQUVHO1FBQ0gsdUNBQVcsQ0FBQTtRQUNYOztXQUVHO1FBQ0gsdUNBQVcsQ0FBQTtRQUNYOztXQUVHO1FBQ0gsaURBQXFCLENBQUE7SUFDdkIsQ0FBQyxFQWJXLHdCQUF3QixHQUF4Qix5Q0FBd0IsS0FBeEIseUNBQXdCLFFBYW5DO0lBZ0lEOzs7O09BSUc7SUFDSCxJQUFZLGlCQVNYO0lBVEQsV0FBWSxpQkFBaUI7UUFDM0I7O1dBRUc7UUFDSCxnRUFBMkMsQ0FBQTtRQUMzQzs7V0FFRztRQUNILDREQUF1QyxDQUFBO0lBQ3pDLENBQUMsRUFUVyxpQkFBaUIsR0FBakIsa0NBQWlCLEtBQWpCLGtDQUFpQixRQVM1QjtJQTJGRDs7OztPQUlHO0lBQ0gsSUFBWSxpQ0FpQlg7SUFqQkQsV0FBWSxpQ0FBaUM7UUFDM0M7O1dBRUc7UUFDSCxvRkFBK0MsQ0FBQTtRQUMvQzs7V0FFRztRQUNILGtFQUE2QixDQUFBO1FBQzdCOztXQUVHO1FBQ0gsNERBQXVCLENBQUE7UUFDdkI7O1dBRUc7UUFDSCw4RUFBeUMsQ0FBQTtJQUMzQyxDQUFDLEVBakJXLGlDQUFpQyxHQUFqQyxrREFBaUMsS0FBakMsa0RBQWlDLFFBaUI1QztJQUVEOzs7O09BSUc7SUFDSCxJQUFZLHdCQXFCWDtJQXJCRCxXQUFZLHdCQUF3QjtRQUNsQzs7V0FFRztRQUNILHlEQUE2QixDQUFBO1FBQzdCOztXQUVHO1FBQ0gsK0NBQW1CLENBQUE7UUFDbkI7O1dBRUc7UUFDSCx1Q0FBVyxDQUFBO1FBQ1g7O1dBRUc7UUFDSCwrQ0FBbUIsQ0FBQTtRQUNuQjs7V0FFRztRQUNILCtDQUFtQixDQUFBO0lBQ3JCLENBQUMsRUFyQlcsd0JBQXdCLEdBQXhCLHlDQUF3QixLQUF4Qix5Q0FBd0IsUUFxQm5DO0lBRUQ7Ozs7T0FJRztJQUNILElBQVksbUJBNkdYO0lBN0dELFdBQVksbUJBQW1CO1FBQzdCOztXQUVHO1FBQ0gsb0NBQWEsQ0FBQTtRQUNiOztXQUVHO1FBQ0gsMENBQW1CLENBQUE7UUFDbkI7O1dBRUc7UUFDSCxzREFBK0IsQ0FBQTtRQUMvQjs7V0FFRztRQUNILHNEQUErQixDQUFBO1FBQy9COztXQUVHO1FBQ0gsc0NBQWUsQ0FBQTtRQUNmOztXQUVHO1FBQ0gsZ0RBQXlCLENBQUE7UUFDekI7O1dBRUc7UUFDSCxnREFBeUIsQ0FBQTtRQUN6Qjs7V0FFRztRQUNILGdEQUF5QixDQUFBO1FBQ3pCOztXQUVHO1FBQ0gsZ0RBQXlCLENBQUE7UUFDekI7O1dBRUc7UUFDSCxvQ0FBYSxDQUFBO1FBQ2I7O1dBRUc7UUFDSCw4Q0FBdUIsQ0FBQTtRQUN2Qjs7V0FFRztRQUNILDhDQUF1QixDQUFBO1FBQ3ZCOztXQUVHO1FBQ0gsOENBQXVCLENBQUE7UUFDdkI7O1dBRUc7UUFDSCw4Q0FBdUIsQ0FBQTtRQUN2Qjs7V0FFRztRQUNILDBDQUFtQixDQUFBO1FBQ25COztXQUVHO1FBQ0gsc0NBQWUsQ0FBQTtRQUNmOztXQUVHO1FBQ0gsNERBQXFDLENBQUE7UUFDckM7O1dBRUc7UUFDSCxzRUFBK0MsQ0FBQTtRQUMvQzs7V0FFRztRQUNILHNFQUErQyxDQUFBO1FBQy9DOztXQUVHO1FBQ0gsc0VBQStDLENBQUE7UUFDL0M7O1dBRUc7UUFDSCxzRUFBK0MsQ0FBQTtRQUMvQzs7V0FFRztRQUNILDBEQUFtQyxDQUFBO1FBQ25DOztXQUVHO1FBQ0gsb0VBQTZDLENBQUE7UUFDN0M7O1dBRUc7UUFDSCxvRUFBNkMsQ0FBQTtRQUM3Qzs7V0FFRztRQUNILG9FQUE2QyxDQUFBO1FBQzdDOztXQUVHO1FBQ0gsb0VBQTZDLENBQUE7UUFDN0M7O1dBRUc7UUFDSCw0REFBcUMsQ0FBQTtJQUN2QyxDQUFDLEVBN0dXLG1CQUFtQixHQUFuQixvQ0FBbUIsS0FBbkIsb0NBQW1CLFFBNkc5QjtJQTZHRDs7OztPQUlHO0lBQ0gsSUFBWSxjQVNYO0lBVEQsV0FBWSxjQUFjO1FBQ3hCOztXQUVHO1FBQ0gseUNBQXVCLENBQUE7UUFDdkI7O1dBRUc7UUFDSCwrQkFBYSxDQUFBO0lBQ2YsQ0FBQyxFQVRXLGNBQWMsR0FBZCwrQkFBYyxLQUFkLCtCQUFjLFFBU3pCO0lBRUQ7Ozs7T0FJRztJQUNILElBQVkscUJBYVg7SUFiRCxXQUFZLHFCQUFxQjtRQUMvQjs7V0FFRztRQUNILGtFQUF5QyxDQUFBO1FBQ3pDOztXQUVHO1FBQ0gsa0ZBQXlELENBQUE7UUFDekQ7O1dBRUc7UUFDSCwwREFBaUMsQ0FBQTtJQUNuQyxDQUFDLEVBYlcscUJBQXFCLEdBQXJCLHNDQUFxQixLQUFyQixzQ0FBcUIsUUFhaEM7QUErZkgsQ0FBQyxFQXZsQ2dCLGdCQUFnQixHQUFoQix3QkFBZ0IsS0FBaEIsd0JBQWdCLFFBdWxDaEMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgKiBhcyBpYW0gZnJvbSAnQGF3cy1jZGsvYXdzLWlhbSc7XG5pbXBvcnQgKiBhcyBzZm4gZnJvbSAnQGF3cy1jZGsvYXdzLXN0ZXBmdW5jdGlvbnMnO1xuaW1wb3J0ICogYXMgY2RrIGZyb20gJ0Bhd3MtY2RrL2NvcmUnO1xuaW1wb3J0IHsgaW50ZWdyYXRpb25SZXNvdXJjZUFybiwgdmFsaWRhdGVQYXR0ZXJuU3VwcG9ydGVkIH0gZnJvbSAnLi4vcHJpdmF0ZS90YXNrLXV0aWxzJztcbmltcG9ydCB7XG4gIEFwcGxpY2F0aW9uQ29uZmlnUHJvcGVydHlUb0pzb24sXG4gIEJvb3RzdHJhcEFjdGlvbkNvbmZpZ1RvSnNvbixcbiAgQ29uZmlndXJhdGlvblByb3BlcnR5VG9Kc29uLFxuICBJbnN0YW5jZXNDb25maWdQcm9wZXJ0eVRvSnNvbixcbiAgS2VyYmVyb3NBdHRyaWJ1dGVzUHJvcGVydHlUb0pzb24sXG59IGZyb20gJy4vcHJpdmF0ZS9jbHVzdGVyLXV0aWxzJztcblxuLyoqXG4gKiBQcm9wZXJ0aWVzIGZvciBFbXJDcmVhdGVDbHVzdGVyXG4gKlxuICogU2VlIHRoZSBSdW5Kb2JGbG93IEFQSSBmb3IgY29tcGxldGUgZG9jdW1lbnRhdGlvbiBvbiBpbnB1dCBwYXJhbWV0ZXJzXG4gKlxuICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZW1yL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1J1bkpvYkZsb3cuaHRtbFxuICpcbiAqIEBleHBlcmltZW50YWxcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBFbXJDcmVhdGVDbHVzdGVyUHJvcHMgZXh0ZW5kcyBzZm4uVGFza1N0YXRlQmFzZVByb3BzIHtcbiAgLyoqXG4gICAqIEEgc3BlY2lmaWNhdGlvbiBvZiB0aGUgbnVtYmVyIGFuZCB0eXBlIG9mIEFtYXpvbiBFQzIgaW5zdGFuY2VzLlxuICAgKi9cbiAgcmVhZG9ubHkgaW5zdGFuY2VzOiBFbXJDcmVhdGVDbHVzdGVyLkluc3RhbmNlc0NvbmZpZ1Byb3BlcnR5O1xuXG4gIC8qKlxuICAgKiBBbHNvIGNhbGxlZCBpbnN0YW5jZSBwcm9maWxlIGFuZCBFQzIgcm9sZS4gQW4gSUFNIHJvbGUgZm9yIGFuIEVNUiBjbHVzdGVyLiBUaGUgRUMyIGluc3RhbmNlcyBvZiB0aGUgY2x1c3RlciBhc3N1bWUgdGhpcyByb2xlLlxuICAgKlxuICAgKiBUaGlzIGF0dHJpYnV0ZSBoYXMgYmVlbiByZW5hbWVkIGZyb20gam9iRmxvd1JvbGUgdG8gY2x1c3RlclJvbGUgdG8gYWxpZ24gd2l0aCBvdGhlciBFUk0vU3RlcEZ1bmN0aW9uIGludGVncmF0aW9uIHBhcmFtZXRlcnMuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gKiBBIFJvbGUgd2lsbCBiZSBjcmVhdGVkXG4gICAqL1xuICByZWFkb25seSBjbHVzdGVyUm9sZT86IGlhbS5JUm9sZTtcblxuICAvKipcbiAgICogVGhlIE5hbWUgb2YgdGhlIENsdXN0ZXJcbiAgICovXG4gIHJlYWRvbmx5IG5hbWU6IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIElBTSByb2xlIHRoYXQgd2lsbCBiZSBhc3N1bWVkIGJ5IHRoZSBBbWF6b24gRU1SIHNlcnZpY2UgdG8gYWNjZXNzIEFXUyByZXNvdXJjZXMgb24geW91ciBiZWhhbGYuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gQSByb2xlIHdpbGwgYmUgY3JlYXRlZCB0aGF0IEFtYXpvbiBFTVIgc2VydmljZSBjYW4gYXNzdW1lLlxuICAgKi9cbiAgcmVhZG9ubHkgc2VydmljZVJvbGU/OiBpYW0uSVJvbGU7XG5cbiAgLyoqXG4gICAqIEEgSlNPTiBzdHJpbmcgZm9yIHNlbGVjdGluZyBhZGRpdGlvbmFsIGZlYXR1cmVzLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIE5vbmVcbiAgICovXG4gIHJlYWRvbmx5IGFkZGl0aW9uYWxJbmZvPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBBIGNhc2UtaW5zZW5zaXRpdmUgbGlzdCBvZiBhcHBsaWNhdGlvbnMgZm9yIEFtYXpvbiBFTVIgdG8gaW5zdGFsbCBhbmQgY29uZmlndXJlIHdoZW4gbGF1bmNoaW5nIHRoZSBjbHVzdGVyLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIEVNUiBzZWxlY3RlZCBkZWZhdWx0XG4gICAqL1xuICByZWFkb25seSBhcHBsaWNhdGlvbnM/OiBFbXJDcmVhdGVDbHVzdGVyLkFwcGxpY2F0aW9uQ29uZmlnUHJvcGVydHlbXTtcblxuICAvKipcbiAgICogQW4gSUFNIHJvbGUgZm9yIGF1dG9tYXRpYyBzY2FsaW5nIHBvbGljaWVzLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIEEgcm9sZSB3aWxsIGJlIGNyZWF0ZWQuXG4gICAqL1xuICByZWFkb25seSBhdXRvU2NhbGluZ1JvbGU/OiBpYW0uSVJvbGU7XG5cbiAgLyoqXG4gICAqIEEgbGlzdCBvZiBib290c3RyYXAgYWN0aW9ucyB0byBydW4gYmVmb3JlIEhhZG9vcCBzdGFydHMgb24gdGhlIGNsdXN0ZXIgbm9kZXMuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gTm9uZVxuICAgKi9cbiAgcmVhZG9ubHkgYm9vdHN0cmFwQWN0aW9ucz86IEVtckNyZWF0ZUNsdXN0ZXIuQm9vdHN0cmFwQWN0aW9uQ29uZmlnUHJvcGVydHlbXTtcblxuICAvKipcbiAgICogVGhlIGxpc3Qgb2YgY29uZmlndXJhdGlvbnMgc3VwcGxpZWQgZm9yIHRoZSBFTVIgY2x1c3RlciB5b3UgYXJlIGNyZWF0aW5nLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIE5vbmVcbiAgICovXG4gIHJlYWRvbmx5IGNvbmZpZ3VyYXRpb25zPzogRW1yQ3JlYXRlQ2x1c3Rlci5Db25maWd1cmF0aW9uUHJvcGVydHlbXTtcblxuICAvKipcbiAgICogVGhlIElEIG9mIGEgY3VzdG9tIEFtYXpvbiBFQlMtYmFja2VkIExpbnV4IEFNSS5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBOb25lXG4gICAqL1xuICByZWFkb25seSBjdXN0b21BbWlJZD86IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIHNpemUgb2YgdGhlIEVCUyByb290IGRldmljZSB2b2x1bWUgb2YgdGhlIExpbnV4IEFNSSB0aGF0IGlzIHVzZWQgZm9yIGVhY2ggRUMyIGluc3RhbmNlLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIEVNUiBzZWxlY3RlZCBkZWZhdWx0XG4gICAqL1xuICByZWFkb25seSBlYnNSb290Vm9sdW1lU2l6ZT86IGNkay5TaXplO1xuXG4gIC8qKlxuICAgKiBBdHRyaWJ1dGVzIGZvciBLZXJiZXJvcyBjb25maWd1cmF0aW9uIHdoZW4gS2VyYmVyb3MgYXV0aGVudGljYXRpb24gaXMgZW5hYmxlZCB1c2luZyBhIHNlY3VyaXR5IGNvbmZpZ3VyYXRpb24uXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gTm9uZVxuICAgKi9cbiAgcmVhZG9ubHkga2VyYmVyb3NBdHRyaWJ1dGVzPzogRW1yQ3JlYXRlQ2x1c3Rlci5LZXJiZXJvc0F0dHJpYnV0ZXNQcm9wZXJ0eTtcblxuICAvKipcbiAgICogVGhlIGxvY2F0aW9uIGluIEFtYXpvbiBTMyB0byB3cml0ZSB0aGUgbG9nIGZpbGVzIG9mIHRoZSBqb2IgZmxvdy5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBOb25lXG4gICAqL1xuICByZWFkb25seSBsb2dVcmk/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBBbWF6b24gRU1SIHJlbGVhc2UgbGFiZWwsIHdoaWNoIGRldGVybWluZXMgdGhlIHZlcnNpb24gb2Ygb3Blbi1zb3VyY2UgYXBwbGljYXRpb24gcGFja2FnZXMgaW5zdGFsbGVkIG9uIHRoZSBjbHVzdGVyLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIEVNUiBzZWxlY3RlZCBkZWZhdWx0XG4gICAqL1xuICByZWFkb25seSByZWxlYXNlTGFiZWw/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFNwZWNpZmllcyB0aGUgd2F5IHRoYXQgaW5kaXZpZHVhbCBBbWF6b24gRUMyIGluc3RhbmNlcyB0ZXJtaW5hdGUgd2hlbiBhbiBhdXRvbWF0aWMgc2NhbGUtaW4gYWN0aXZpdHkgb2NjdXJzIG9yIGFuIGluc3RhbmNlIGdyb3VwIGlzIHJlc2l6ZWQuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gRU1SIHNlbGVjdGVkIGRlZmF1bHRcbiAgICovXG4gIHJlYWRvbmx5IHNjYWxlRG93bkJlaGF2aW9yPzogRW1yQ3JlYXRlQ2x1c3Rlci5FbXJDbHVzdGVyU2NhbGVEb3duQmVoYXZpb3I7XG5cbiAgLyoqXG4gICAqIFRoZSBuYW1lIG9mIGEgc2VjdXJpdHkgY29uZmlndXJhdGlvbiB0byBhcHBseSB0byB0aGUgY2x1c3Rlci5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBOb25lXG4gICAqL1xuICByZWFkb25seSBzZWN1cml0eUNvbmZpZ3VyYXRpb24/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIEEgbGlzdCBvZiB0YWdzIHRvIGFzc29jaWF0ZSB3aXRoIGEgY2x1c3RlciBhbmQgcHJvcGFnYXRlIHRvIEFtYXpvbiBFQzIgaW5zdGFuY2VzLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIE5vbmVcbiAgICovXG4gIHJlYWRvbmx5IHRhZ3M/OiB7IFtrZXk6IHN0cmluZ106IHN0cmluZyB9O1xuXG4gIC8qKlxuICAgKiBBIHZhbHVlIG9mIHRydWUgaW5kaWNhdGVzIHRoYXQgYWxsIElBTSB1c2VycyBpbiB0aGUgQVdTIGFjY291bnQgY2FuIHBlcmZvcm0gY2x1c3RlciBhY3Rpb25zIGlmIHRoZXkgaGF2ZSB0aGUgcHJvcGVyIElBTSBwb2xpY3kgcGVybWlzc2lvbnMuXG4gICAqXG4gICAqIEBkZWZhdWx0IHRydWVcbiAgICovXG4gIHJlYWRvbmx5IHZpc2libGVUb0FsbFVzZXJzPzogYm9vbGVhbjtcbn1cblxuLyoqXG4gKiBBIFN0ZXAgRnVuY3Rpb25zIFRhc2sgdG8gY3JlYXRlIGFuIEVNUiBDbHVzdGVyLlxuICpcbiAqIFRoZSBDbHVzdGVyQ29uZmlndXJhdGlvbiBpcyBkZWZpbmVkIGFzIFBhcmFtZXRlcnMgaW4gdGhlIHN0YXRlIG1hY2hpbmUgZGVmaW5pdGlvbi5cbiAqXG4gKiBPVVRQVVQ6IHRoZSBDbHVzdGVySWQuXG4gKlxuICogQGV4cGVyaW1lbnRhbFxuICovXG5leHBvcnQgY2xhc3MgRW1yQ3JlYXRlQ2x1c3RlciBleHRlbmRzIHNmbi5UYXNrU3RhdGVCYXNlIHtcbiAgcHJpdmF0ZSBzdGF0aWMgcmVhZG9ubHkgU1VQUE9SVEVEX0lOVEVHUkFUSU9OX1BBVFRFUk5TOiBzZm4uSW50ZWdyYXRpb25QYXR0ZXJuW10gPSBbXG4gICAgc2ZuLkludGVncmF0aW9uUGF0dGVybi5SRVFVRVNUX1JFU1BPTlNFLFxuICAgIHNmbi5JbnRlZ3JhdGlvblBhdHRlcm4uUlVOX0pPQixcbiAgXTtcblxuICBwcm90ZWN0ZWQgcmVhZG9ubHkgdGFza1BvbGljaWVzPzogaWFtLlBvbGljeVN0YXRlbWVudFtdO1xuICBwcm90ZWN0ZWQgcmVhZG9ubHkgdGFza01ldHJpY3M/OiBzZm4uVGFza01ldHJpY3NDb25maWc7XG5cbiAgcHJpdmF0ZSByZWFkb25seSB2aXNpYmxlVG9BbGxVc2VyczogYm9vbGVhbjtcbiAgcHJpdmF0ZSByZWFkb25seSBpbnRlZ3JhdGlvblBhdHRlcm46IHNmbi5JbnRlZ3JhdGlvblBhdHRlcm47XG5cbiAgcHJpdmF0ZSBfc2VydmljZVJvbGU6IGlhbS5JUm9sZTtcbiAgcHJpdmF0ZSBfY2x1c3RlclJvbGU6IGlhbS5JUm9sZTtcbiAgcHJpdmF0ZSBfYXV0b1NjYWxpbmdSb2xlPzogaWFtLklSb2xlO1xuXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBjZGsuQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcml2YXRlIHJlYWRvbmx5IHByb3BzOiBFbXJDcmVhdGVDbHVzdGVyUHJvcHMpIHtcbiAgICBzdXBlcihzY29wZSwgaWQsIHByb3BzKTtcbiAgICB0aGlzLnZpc2libGVUb0FsbFVzZXJzID0gdGhpcy5wcm9wcy52aXNpYmxlVG9BbGxVc2VycyAhPT0gdW5kZWZpbmVkID8gdGhpcy5wcm9wcy52aXNpYmxlVG9BbGxVc2VycyA6IHRydWU7XG4gICAgdGhpcy5pbnRlZ3JhdGlvblBhdHRlcm4gPSBwcm9wcy5pbnRlZ3JhdGlvblBhdHRlcm4gfHwgc2ZuLkludGVncmF0aW9uUGF0dGVybi5SVU5fSk9CO1xuICAgIHZhbGlkYXRlUGF0dGVyblN1cHBvcnRlZCh0aGlzLmludGVncmF0aW9uUGF0dGVybiwgRW1yQ3JlYXRlQ2x1c3Rlci5TVVBQT1JURURfSU5URUdSQVRJT05fUEFUVEVSTlMpO1xuXG4gICAgdGhpcy5fYXV0b1NjYWxpbmdSb2xlID0gdGhpcy5wcm9wcy5hdXRvU2NhbGluZ1JvbGU7XG5cbiAgICAvLyBJZiB0aGUgUm9sZXMgYXJlIHVuZGVmaW5lZCB0aGVuIHRoZXkgd2VyZW4ndCBwcm92aWRlZCwgc28gY3JlYXRlIHRoZW1cbiAgICB0aGlzLl9zZXJ2aWNlUm9sZSA9IHRoaXMucHJvcHMuc2VydmljZVJvbGUgPz8gdGhpcy5jcmVhdGVTZXJ2aWNlUm9sZSgpO1xuICAgIHRoaXMuX2NsdXN0ZXJSb2xlID0gdGhpcy5wcm9wcy5jbHVzdGVyUm9sZSA/PyB0aGlzLmNyZWF0ZUNsdXN0ZXJSb2xlKCk7XG5cbiAgICAvLyBBdXRvU2NhbGluZyByb2xlcyBhcmUgbm90IHZhbGlkIHdpdGggSW5zdGFuY2VGbGVldCBjbHVzdGVycy5cbiAgICAvLyBBdHRlbXB0IHRvIGNyZWF0ZSBvbmx5IGlmIC5pbnN0YW5jZXMuaW5zdGFuY2VGbGVldHMgaXMgdW5kZWZpbmVkIG9yIGVtcHR5XG4gICAgaWYgKHRoaXMucHJvcHMuaW5zdGFuY2VzLmluc3RhbmNlRmxlZXRzID09PSB1bmRlZmluZWQgfHwgdGhpcy5wcm9wcy5pbnN0YW5jZXMuaW5zdGFuY2VGbGVldHMubGVuZ3RoID09PSAwKSB7XG4gICAgICB0aGlzLl9hdXRvU2NhbGluZ1JvbGUgPSB0aGlzLl9hdXRvU2NhbGluZ1JvbGUgfHwgdGhpcy5jcmVhdGVBdXRvU2NhbGluZ1JvbGUoKTtcbiAgICAgIC8vIElmIEluc3RhbmNlRmxlZXRzIGFyZSB1c2VkIGFuZCBhbiBBdXRvU2NhbGluZyBSb2xlIGlzIHNwZWNpZmllZCwgdGhyb3cgYW4gZXJyb3JcbiAgICB9IGVsc2UgaWYgKHRoaXMuX2F1dG9TY2FsaW5nUm9sZSAhPT0gdW5kZWZpbmVkKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ0F1dG8gU2NhbGluZyByb2xlcyBjYW4gbm90IGJlIHNwZWNpZmllZCB3aXRoIGluc3RhbmNlIGZsZWV0cy4nKTtcbiAgICB9XG5cbiAgICB0aGlzLnRhc2tQb2xpY2llcyA9IHRoaXMuY3JlYXRlUG9saWN5U3RhdGVtZW50cyh0aGlzLl9zZXJ2aWNlUm9sZSwgdGhpcy5fY2x1c3RlclJvbGUsIHRoaXMuX2F1dG9TY2FsaW5nUm9sZSk7XG4gIH1cblxuICAvKipcbiAgICogVGhlIHNlcnZpY2Ugcm9sZSBmb3IgdGhlIEVNUiBDbHVzdGVyLlxuICAgKlxuICAgKiBPbmx5IGF2YWlsYWJsZSBhZnRlciB0YXNrIGhhcyBiZWVuIGFkZGVkIHRvIGEgc3RhdGUgbWFjaGluZS5cbiAgICovXG4gIHB1YmxpYyBnZXQgc2VydmljZVJvbGUoKTogaWFtLklSb2xlIHtcbiAgICBpZiAodGhpcy5fc2VydmljZVJvbGUgPT09IHVuZGVmaW5lZCkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdyb2xlIG5vdCBhdmFpbGFibGUgeWV0LS11c2UgdGhlIG9iamVjdCBpbiBhIFRhc2sgZmlyc3QnKTtcbiAgICB9XG4gICAgcmV0dXJuIHRoaXMuX3NlcnZpY2VSb2xlO1xuICB9XG5cbiAgLyoqXG4gICAqIFRoZSBpbnN0YW5jZSByb2xlIGZvciB0aGUgRU1SIENsdXN0ZXIuXG4gICAqXG4gICAqIE9ubHkgYXZhaWxhYmxlIGFmdGVyIHRhc2sgaGFzIGJlZW4gYWRkZWQgdG8gYSBzdGF0ZSBtYWNoaW5lLlxuICAgKi9cbiAgcHVibGljIGdldCBjbHVzdGVyUm9sZSgpOiBpYW0uSVJvbGUge1xuICAgIGlmICh0aGlzLl9jbHVzdGVyUm9sZSA9PT0gdW5kZWZpbmVkKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ3JvbGUgbm90IGF2YWlsYWJsZSB5ZXQtLXVzZSB0aGUgb2JqZWN0IGluIGEgVGFzayBmaXJzdCcpO1xuICAgIH1cbiAgICByZXR1cm4gdGhpcy5fY2x1c3RlclJvbGU7XG4gIH1cblxuICAvKipcbiAgICogVGhlIGF1dG9zY2FsaW5nIHJvbGUgZm9yIHRoZSBFTVIgQ2x1c3Rlci5cbiAgICpcbiAgICogT25seSBhdmFpbGFibGUgYWZ0ZXIgdGFzayBoYXMgYmVlbiBhZGRlZCB0byBhIHN0YXRlIG1hY2hpbmUuXG4gICAqL1xuICBwdWJsaWMgZ2V0IGF1dG9TY2FsaW5nUm9sZSgpOiBpYW0uSVJvbGUge1xuICAgIGlmICh0aGlzLl9hdXRvU2NhbGluZ1JvbGUgPT09IHVuZGVmaW5lZCkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdyb2xlIG5vdCBhdmFpbGFibGUgeWV0LS11c2UgdGhlIG9iamVjdCBpbiBhIFRhc2sgZmlyc3QnKTtcbiAgICB9XG4gICAgcmV0dXJuIHRoaXMuX2F1dG9TY2FsaW5nUm9sZTtcbiAgfVxuXG4gIHByb3RlY3RlZCByZW5kZXJUYXNrKCk6IGFueSB7XG4gICAgcmV0dXJuIHtcbiAgICAgIFJlc291cmNlOiBpbnRlZ3JhdGlvblJlc291cmNlQXJuKCdlbGFzdGljbWFwcmVkdWNlJywgJ2NyZWF0ZUNsdXN0ZXInLCB0aGlzLmludGVncmF0aW9uUGF0dGVybiksXG4gICAgICBQYXJhbWV0ZXJzOiBzZm4uRmllbGRVdGlscy5yZW5kZXJPYmplY3Qoe1xuICAgICAgICBJbnN0YW5jZXM6IEluc3RhbmNlc0NvbmZpZ1Byb3BlcnR5VG9Kc29uKHRoaXMucHJvcHMuaW5zdGFuY2VzKSxcbiAgICAgICAgSm9iRmxvd1JvbGU6IGNkay5zdHJpbmdUb0Nsb3VkRm9ybWF0aW9uKHRoaXMuX2NsdXN0ZXJSb2xlLnJvbGVOYW1lKSxcbiAgICAgICAgTmFtZTogY2RrLnN0cmluZ1RvQ2xvdWRGb3JtYXRpb24odGhpcy5wcm9wcy5uYW1lKSxcbiAgICAgICAgU2VydmljZVJvbGU6IGNkay5zdHJpbmdUb0Nsb3VkRm9ybWF0aW9uKHRoaXMuX3NlcnZpY2VSb2xlLnJvbGVOYW1lKSxcbiAgICAgICAgQWRkaXRpb25hbEluZm86IGNkay5zdHJpbmdUb0Nsb3VkRm9ybWF0aW9uKHRoaXMucHJvcHMuYWRkaXRpb25hbEluZm8pLFxuICAgICAgICBBcHBsaWNhdGlvbnM6IGNkay5saXN0TWFwcGVyKEFwcGxpY2F0aW9uQ29uZmlnUHJvcGVydHlUb0pzb24pKHRoaXMucHJvcHMuYXBwbGljYXRpb25zKSxcbiAgICAgICAgQXV0b1NjYWxpbmdSb2xlOiBjZGsuc3RyaW5nVG9DbG91ZEZvcm1hdGlvbih0aGlzLl9hdXRvU2NhbGluZ1JvbGU/LnJvbGVOYW1lKSxcbiAgICAgICAgQm9vdHN0cmFwQWN0aW9uczogY2RrLmxpc3RNYXBwZXIoQm9vdHN0cmFwQWN0aW9uQ29uZmlnVG9Kc29uKSh0aGlzLnByb3BzLmJvb3RzdHJhcEFjdGlvbnMpLFxuICAgICAgICBDb25maWd1cmF0aW9uczogY2RrLmxpc3RNYXBwZXIoQ29uZmlndXJhdGlvblByb3BlcnR5VG9Kc29uKSh0aGlzLnByb3BzLmNvbmZpZ3VyYXRpb25zKSxcbiAgICAgICAgQ3VzdG9tQW1pSWQ6IGNkay5zdHJpbmdUb0Nsb3VkRm9ybWF0aW9uKHRoaXMucHJvcHMuY3VzdG9tQW1pSWQpLFxuICAgICAgICBFYnNSb290Vm9sdW1lU2l6ZTogdGhpcy5wcm9wcy5lYnNSb290Vm9sdW1lU2l6ZT8udG9HaWJpYnl0ZXMoKSxcbiAgICAgICAgS2VyYmVyb3NBdHRyaWJ1dGVzOiB0aGlzLnByb3BzLmtlcmJlcm9zQXR0cmlidXRlcyA/IEtlcmJlcm9zQXR0cmlidXRlc1Byb3BlcnR5VG9Kc29uKHRoaXMucHJvcHMua2VyYmVyb3NBdHRyaWJ1dGVzKSA6IHVuZGVmaW5lZCxcbiAgICAgICAgTG9nVXJpOiBjZGsuc3RyaW5nVG9DbG91ZEZvcm1hdGlvbih0aGlzLnByb3BzLmxvZ1VyaSksXG4gICAgICAgIFJlbGVhc2VMYWJlbDogY2RrLnN0cmluZ1RvQ2xvdWRGb3JtYXRpb24odGhpcy5wcm9wcy5yZWxlYXNlTGFiZWwpLFxuICAgICAgICBTY2FsZURvd25CZWhhdmlvcjogY2RrLnN0cmluZ1RvQ2xvdWRGb3JtYXRpb24odGhpcy5wcm9wcy5zY2FsZURvd25CZWhhdmlvcj8udmFsdWVPZigpKSxcbiAgICAgICAgU2VjdXJpdHlDb25maWd1cmF0aW9uOiBjZGsuc3RyaW5nVG9DbG91ZEZvcm1hdGlvbih0aGlzLnByb3BzLnNlY3VyaXR5Q29uZmlndXJhdGlvbiksXG4gICAgICAgIC4uLih0aGlzLnByb3BzLnRhZ3MgPyB0aGlzLnJlbmRlclRhZ3ModGhpcy5wcm9wcy50YWdzKSA6IHVuZGVmaW5lZCksXG4gICAgICAgIFZpc2libGVUb0FsbFVzZXJzOiBjZGsuYm9vbGVhblRvQ2xvdWRGb3JtYXRpb24odGhpcy52aXNpYmxlVG9BbGxVc2VycyksXG4gICAgICB9KSxcbiAgICB9O1xuICB9XG5cbiAgcHJpdmF0ZSByZW5kZXJUYWdzKHRhZ3M6IHsgW2tleTogc3RyaW5nXTogYW55IH0gfCB1bmRlZmluZWQpOiB7IFtrZXk6IHN0cmluZ106IGFueSB9IHtcbiAgICByZXR1cm4gdGFncyA/IHsgVGFnczogT2JqZWN0LmtleXModGFncykubWFwKChrZXkpID0+ICh7IEtleToga2V5LCBWYWx1ZTogdGFnc1trZXldIH0pKSB9IDoge307XG4gIH1cblxuICAvKipcbiAgICogVGhpcyBnZW5lcmF0ZXMgdGhlIFBvbGljeVN0YXRlbWVudHMgcmVxdWlyZWQgYnkgdGhlIFRhc2sgdG8gY2FsbCBDcmVhdGVDbHVzdGVyLlxuICAgKi9cbiAgcHJpdmF0ZSBjcmVhdGVQb2xpY3lTdGF0ZW1lbnRzKHNlcnZpY2VSb2xlOiBpYW0uSVJvbGUsIGNsdXN0ZXJSb2xlOiBpYW0uSVJvbGUsIGF1dG9TY2FsaW5nUm9sZT86IGlhbS5JUm9sZSk6IGlhbS5Qb2xpY3lTdGF0ZW1lbnRbXSB7XG4gICAgY29uc3Qgc3RhY2sgPSBjZGsuU3RhY2sub2YodGhpcyk7XG5cbiAgICBjb25zdCBwb2xpY3lTdGF0ZW1lbnRzID0gW1xuICAgICAgbmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgICBhY3Rpb25zOiBbJ2VsYXN0aWNtYXByZWR1Y2U6UnVuSm9iRmxvdycsICdlbGFzdGljbWFwcmVkdWNlOkRlc2NyaWJlQ2x1c3RlcicsICdlbGFzdGljbWFwcmVkdWNlOlRlcm1pbmF0ZUpvYkZsb3dzJ10sXG4gICAgICAgIHJlc291cmNlczogWycqJ10sXG4gICAgICB9KSxcbiAgICBdO1xuXG4gICAgLy8gQWxsb3cgdGhlIFN0YXRlTWFjaGluZSB0byBQYXNzUm9sZSB0byBDbHVzdGVyIHJvbGVzXG4gICAgcG9saWN5U3RhdGVtZW50cy5wdXNoKFxuICAgICAgbmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgICBhY3Rpb25zOiBbJ2lhbTpQYXNzUm9sZSddLFxuICAgICAgICByZXNvdXJjZXM6IFtzZXJ2aWNlUm9sZS5yb2xlQXJuLCBjbHVzdGVyUm9sZS5yb2xlQXJuXSxcbiAgICAgIH0pLFxuICAgICk7XG4gICAgaWYgKGF1dG9TY2FsaW5nUm9sZSAhPT0gdW5kZWZpbmVkKSB7XG4gICAgICBwb2xpY3lTdGF0ZW1lbnRzLnB1c2goXG4gICAgICAgIG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgICBhY3Rpb25zOiBbJ2lhbTpQYXNzUm9sZSddLFxuICAgICAgICAgIHJlc291cmNlczogW2F1dG9TY2FsaW5nUm9sZS5yb2xlQXJuXSxcbiAgICAgICAgfSksXG4gICAgICApO1xuICAgIH1cblxuICAgIGlmICh0aGlzLmludGVncmF0aW9uUGF0dGVybiA9PT0gc2ZuLkludGVncmF0aW9uUGF0dGVybi5SVU5fSk9CKSB7XG4gICAgICBwb2xpY3lTdGF0ZW1lbnRzLnB1c2goXG4gICAgICAgIG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgICBhY3Rpb25zOiBbJ2V2ZW50czpQdXRUYXJnZXRzJywgJ2V2ZW50czpQdXRSdWxlJywgJ2V2ZW50czpEZXNjcmliZVJ1bGUnXSxcbiAgICAgICAgICByZXNvdXJjZXM6IFtcbiAgICAgICAgICAgIHN0YWNrLmZvcm1hdEFybih7XG4gICAgICAgICAgICAgIHNlcnZpY2U6ICdldmVudHMnLFxuICAgICAgICAgICAgICByZXNvdXJjZTogJ3J1bGUnLFxuICAgICAgICAgICAgICByZXNvdXJjZU5hbWU6ICdTdGVwRnVuY3Rpb25zR2V0RXZlbnRGb3JFTVJSdW5Kb2JGbG93UnVsZScsXG4gICAgICAgICAgICB9KSxcbiAgICAgICAgICBdLFxuICAgICAgICB9KSxcbiAgICAgICk7XG4gICAgfVxuXG4gICAgcmV0dXJuIHBvbGljeVN0YXRlbWVudHM7XG4gIH1cblxuICAvKipcbiAgICogR2VuZXJhdGUgdGhlIFJvbGUgdXNlZCBieSB0aGUgRU1SIFNlcnZpY2VcbiAgICovXG4gIHByaXZhdGUgY3JlYXRlU2VydmljZVJvbGUoKTogaWFtLklSb2xlIHtcbiAgICByZXR1cm4gbmV3IGlhbS5Sb2xlKHRoaXMsICdTZXJ2aWNlUm9sZScsIHtcbiAgICAgIGFzc3VtZWRCeTogbmV3IGlhbS5TZXJ2aWNlUHJpbmNpcGFsKCdlbGFzdGljbWFwcmVkdWNlLmFtYXpvbmF3cy5jb20nKSxcbiAgICAgIG1hbmFnZWRQb2xpY2llczogW2lhbS5NYW5hZ2VkUG9saWN5LmZyb21Bd3NNYW5hZ2VkUG9saWN5TmFtZSgnc2VydmljZS1yb2xlL0FtYXpvbkVsYXN0aWNNYXBSZWR1Y2VSb2xlJyldLFxuICAgIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIEdlbmVyYXRlIHRoZSBSb2xlIGFuZCBJbnN0YW5jZSBQcm9maWxlIHVzZWQgYnkgdGhlIEVDMiBpbnN0YW5jZXNcbiAgICpcbiAgICogRGF0YSBhY2Nlc3MgcGVybWlzc2lvbnMgd2lsbCBuZWVkIHRvIGJlIHVwZGF0ZWQgYnkgdGhlIHVzZXJcbiAgICovXG4gIHByaXZhdGUgY3JlYXRlQ2x1c3RlclJvbGUoKTogaWFtLklSb2xlIHtcbiAgICBjb25zdCByb2xlID0gbmV3IGlhbS5Sb2xlKHRoaXMsICdJbnN0YW5jZVJvbGUnLCB7XG4gICAgICBhc3N1bWVkQnk6IG5ldyBpYW0uU2VydmljZVByaW5jaXBhbCgnZWMyLmFtYXpvbmF3cy5jb20nKSxcbiAgICB9KTtcblxuICAgIG5ldyBpYW0uQ2ZuSW5zdGFuY2VQcm9maWxlKHRoaXMsICdJbnN0YW5jZVByb2ZpbGUnLCB7XG4gICAgICByb2xlczogW3JvbGUucm9sZU5hbWVdLFxuICAgICAgaW5zdGFuY2VQcm9maWxlTmFtZTogcm9sZS5yb2xlTmFtZSxcbiAgICB9KTtcblxuICAgIHJldHVybiByb2xlO1xuICB9XG5cbiAgLyoqXG4gICAqIEdlbmVyYXRlIHRoZSBSb2xlIHVzZWQgdG8gQXV0b1NjYWxlIHRoZSBDbHVzdGVyXG4gICAqL1xuICBwcml2YXRlIGNyZWF0ZUF1dG9TY2FsaW5nUm9sZSgpOiBpYW0uSVJvbGUge1xuICAgIGNvbnN0IHJvbGUgPSBuZXcgaWFtLlJvbGUodGhpcywgJ0F1dG9TY2FsaW5nUm9sZScsIHtcbiAgICAgIGFzc3VtZWRCeTogbmV3IGlhbS5TZXJ2aWNlUHJpbmNpcGFsKCdlbGFzdGljbWFwcmVkdWNlLmFtYXpvbmF3cy5jb20nKSxcbiAgICAgIG1hbmFnZWRQb2xpY2llczogW2lhbS5NYW5hZ2VkUG9saWN5LmZyb21Bd3NNYW5hZ2VkUG9saWN5TmFtZSgnc2VydmljZS1yb2xlL0FtYXpvbkVsYXN0aWNNYXBSZWR1Y2Vmb3JBdXRvU2NhbGluZ1JvbGUnKV0sXG4gICAgfSk7XG5cbiAgICByb2xlLmFzc3VtZVJvbGVQb2xpY3k/LmFkZFN0YXRlbWVudHMoXG4gICAgICBuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgIGVmZmVjdDogaWFtLkVmZmVjdC5BTExPVyxcbiAgICAgICAgcHJpbmNpcGFsczogW25ldyBpYW0uU2VydmljZVByaW5jaXBhbCgnYXBwbGljYXRpb24tYXV0b3NjYWxpbmcuYW1hem9uYXdzLmNvbScpXSxcbiAgICAgICAgYWN0aW9uczogWydzdHM6QXNzdW1lUm9sZSddLFxuICAgICAgfSksXG4gICAgKTtcblxuICAgIHJldHVybiByb2xlO1xuICB9XG59XG5cbmV4cG9ydCBuYW1lc3BhY2UgRW1yQ3JlYXRlQ2x1c3RlciB7XG4gIC8qKlxuICAgKiBWYWxpZCB2YWx1cyBmb3IgdGhlIENsdXN0ZXIgU2NhbGVEb3duQmVoYXZpb3JcbiAgICpcbiAgICogQGV4cGVyaW1lbnRhbFxuICAgKi9cbiAgZXhwb3J0IGVudW0gRW1yQ2x1c3RlclNjYWxlRG93bkJlaGF2aW9yIHtcbiAgICAvKipcbiAgICAgKiBJbmRpY2F0ZXMgdGhhdCBBbWF6b24gRU1SIHRlcm1pbmF0ZXMgbm9kZXMgYXQgdGhlIGluc3RhbmNlLWhvdXIgYm91bmRhcnksIHJlZ2FyZGxlc3Mgb2Ygd2hlbiB0aGUgcmVxdWVzdCB0byB0ZXJtaW5hdGUgdGhlIGluc3RhbmNlIHdhc1xuICAgICAqIHN1Ym1pdHRlZC4gVGhpcyBvcHRpb24gaXMgb25seSBhdmFpbGFibGUgd2l0aCBBbWF6b24gRU1SIDUuMS4wIGFuZCBsYXRlciBhbmQgaXMgdGhlIGRlZmF1bHQgZm9yIGNsdXN0ZXJzIGNyZWF0ZWQgdXNpbmcgdGhhdCB2ZXJzaW9uXG4gICAgICovXG4gICAgVEVSTUlOQVRFX0FUX0lOU1RBTkNFX0hPVVIgPSAnVEVSTUlOQVRFX0FUX0lOU1RBTkNFX0hPVVInLFxuXG4gICAgLyoqXG4gICAgICogSW5kaWNhdGVzIHRoYXQgQW1hem9uIEVNUiBibGFja2xpc3RzIGFuZCBkcmFpbnMgdGFza3MgZnJvbSBub2RlcyBiZWZvcmUgdGVybWluYXRpbmcgdGhlIEFtYXpvbiBFQzIgaW5zdGFuY2VzLCByZWdhcmRsZXNzIG9mIHRoZVxuICAgICAqIGluc3RhbmNlLWhvdXIgYm91bmRhcnkuXG4gICAgICovXG4gICAgVEVSTUlOQVRFX0FUX1RBU0tfQ09NUExFVElPTiA9ICdURVJNSU5BVEVfQVRfVEFTS19DT01QTEVUSU9OJyxcbiAgfVxuXG4gIC8qKlxuICAgKiBJbnN0YW5jZSBSb2xlIFR5cGVzXG4gICAqXG4gICAqIEBleHBlcmltZW50YWxcbiAgICovXG4gIGV4cG9ydCBlbnVtIEluc3RhbmNlUm9sZVR5cGUge1xuICAgIC8qKlxuICAgICAqIE1hc3RlciBOb2RlXG4gICAgICovXG4gICAgTUFTVEVSID0gJ01BU1RFUicsXG4gICAgLyoqXG4gICAgICogQ29yZSBOb2RlXG4gICAgICovXG4gICAgQ09SRSA9ICdDT1JFJyxcbiAgICAvKipcbiAgICAgKiBUYXNrIE5vZGVcbiAgICAgKi9cbiAgICBUQVNLID0gJ1RBU0snLFxuICB9XG5cbiAgLyoqXG4gICAqIEVCUyBWb2x1bWUgVHlwZXNcbiAgICpcbiAgICogQGV4cGVyaW1lbnRhbFxuICAgKi9cbiAgZXhwb3J0IGVudW0gRWJzQmxvY2tEZXZpY2VWb2x1bWVUeXBlIHtcbiAgICAvKipcbiAgICAgKiBncDIgVm9sdW1lIFR5cGVcbiAgICAgKi9cbiAgICBHUDIgPSAnZ3AyJyxcbiAgICAvKipcbiAgICAgKiBpbzEgVm9sdW1lIFR5cGVcbiAgICAgKi9cbiAgICBJTzEgPSAnaW8xJyxcbiAgICAvKipcbiAgICAgKiBTdGFuZGFyZCBWb2x1bWUgVHlwZVxuICAgICAqL1xuICAgIFNUQU5EQVJEID0gJ3N0YW5kYXJkJyxcbiAgfVxuXG4gIC8qKlxuICAgKiBFQlMgdm9sdW1lIHNwZWNpZmljYXRpb25zIHN1Y2ggYXMgdm9sdW1lIHR5cGUsIElPUFMsIGFuZCBzaXplIChHaUIpIHRoYXQgd2lsbCBiZSByZXF1ZXN0ZWQgZm9yIHRoZSBFQlMgdm9sdW1lIGF0dGFjaGVkXG4gICAqIHRvIGFuIEVDMiBpbnN0YW5jZSBpbiB0aGUgY2x1c3Rlci5cbiAgICpcbiAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZW1yL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1ZvbHVtZVNwZWNpZmljYXRpb24uaHRtbFxuICAgKlxuICAgKiBAZXhwZXJpbWVudGFsXG4gICAqL1xuICBleHBvcnQgaW50ZXJmYWNlIFZvbHVtZVNwZWNpZmljYXRpb25Qcm9wZXJ0eSB7XG4gICAgLyoqXG4gICAgICogVGhlIG51bWJlciBvZiBJL08gb3BlcmF0aW9ucyBwZXIgc2Vjb25kIChJT1BTKSB0aGF0IHRoZSB2b2x1bWUgc3VwcG9ydHMuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIEVNUiBzZWxlY3RlZCBkZWZhdWx0XG4gICAgICovXG4gICAgcmVhZG9ubHkgaW9wcz86IG51bWJlcjtcblxuICAgIC8qKlxuICAgICAqIFRoZSB2b2x1bWUgc2l6ZS4gSWYgdGhlIHZvbHVtZSB0eXBlIGlzIEVCUy1vcHRpbWl6ZWQsIHRoZSBtaW5pbXVtIHZhbHVlIGlzIDEwR2lCLlxuICAgICAqIE1heGltdW0gc2l6ZSBpcyAxVGlCXG4gICAgICovXG4gICAgcmVhZG9ubHkgdm9sdW1lU2l6ZTogY2RrLlNpemU7XG5cbiAgICAvKipcbiAgICAgKiBUaGUgdm9sdW1lIHR5cGUuIFZvbHVtZSB0eXBlcyBzdXBwb3J0ZWQgYXJlIGdwMiwgaW8xLCBzdGFuZGFyZC5cbiAgICAgKi9cbiAgICByZWFkb25seSB2b2x1bWVUeXBlOiBFYnNCbG9ja0RldmljZVZvbHVtZVR5cGU7XG4gIH1cblxuICAvKipcbiAgICogQ29uZmlndXJhdGlvbiBvZiByZXF1ZXN0ZWQgRUJTIGJsb2NrIGRldmljZSBhc3NvY2lhdGVkIHdpdGggdGhlIGluc3RhbmNlIGdyb3VwIHdpdGggY291bnQgb2Ygdm9sdW1lcyB0aGF0IHdpbGwgYmVcbiAgICogYXNzb2NpYXRlZCB0byBldmVyeSBpbnN0YW5jZS5cbiAgICpcbiAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZW1yL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Vic0Jsb2NrRGV2aWNlQ29uZmlnLmh0bWxcbiAgICpcbiAgICogQGV4cGVyaW1lbnRhbFxuICAgKi9cbiAgZXhwb3J0IGludGVyZmFjZSBFYnNCbG9ja0RldmljZUNvbmZpZ1Byb3BlcnR5IHtcbiAgICAvKipcbiAgICAgKiBFQlMgdm9sdW1lIHNwZWNpZmljYXRpb25zIHN1Y2ggYXMgdm9sdW1lIHR5cGUsIElPUFMsIGFuZCBzaXplIChHaUIpIHRoYXQgd2lsbCBiZSByZXF1ZXN0ZWQgZm9yIHRoZSBFQlMgdm9sdW1lIGF0dGFjaGVkIHRvIGFuIEVDMlxuICAgICAqIGluc3RhbmNlIGluIHRoZSBjbHVzdGVyLlxuICAgICAqL1xuICAgIHJlYWRvbmx5IHZvbHVtZVNwZWNpZmljYXRpb246IFZvbHVtZVNwZWNpZmljYXRpb25Qcm9wZXJ0eTtcblxuICAgIC8qKlxuICAgICAqIE51bWJlciBvZiBFQlMgdm9sdW1lcyB3aXRoIGEgc3BlY2lmaWMgdm9sdW1lIGNvbmZpZ3VyYXRpb24gdGhhdCB3aWxsIGJlIGFzc29jaWF0ZWQgd2l0aCBldmVyeSBpbnN0YW5jZSBpbiB0aGUgaW5zdGFuY2UgZ3JvdXBcbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IEVNUiBzZWxlY3RlZCBkZWZhdWx0XG4gICAgICovXG4gICAgcmVhZG9ubHkgdm9sdW1lc1Blckluc3RhbmNlPzogbnVtYmVyO1xuICB9XG5cbiAgLyoqXG4gICAqIFRoZSBBbWF6b24gRUJTIGNvbmZpZ3VyYXRpb24gb2YgYSBjbHVzdGVyIGluc3RhbmNlLlxuICAgKlxuICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9lbXIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRWJzQ29uZmlndXJhdGlvbi5odG1sXG4gICAqXG4gICAqIEBleHBlcmltZW50YWxcbiAgICovXG4gIGV4cG9ydCBpbnRlcmZhY2UgRWJzQ29uZmlndXJhdGlvblByb3BlcnR5IHtcbiAgICAvKipcbiAgICAgKiBBbiBhcnJheSBvZiBBbWF6b24gRUJTIHZvbHVtZSBzcGVjaWZpY2F0aW9ucyBhdHRhY2hlZCB0byBhIGNsdXN0ZXIgaW5zdGFuY2UuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIE5vbmVcbiAgICAgKi9cbiAgICByZWFkb25seSBlYnNCbG9ja0RldmljZUNvbmZpZ3M/OiBFYnNCbG9ja0RldmljZUNvbmZpZ1Byb3BlcnR5W107XG5cbiAgICAvKipcbiAgICAgKiBJbmRpY2F0ZXMgd2hldGhlciBhbiBBbWF6b24gRUJTIHZvbHVtZSBpcyBFQlMtb3B0aW1pemVkLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBFTVIgc2VsZWN0ZWQgZGVmYXVsdFxuICAgICAqL1xuICAgIHJlYWRvbmx5IGVic09wdGltaXplZD86IGJvb2xlYW47XG4gIH1cblxuICAvKipcbiAgICogQW4gaW5zdGFuY2UgdHlwZSBjb25maWd1cmF0aW9uIGZvciBlYWNoIGluc3RhbmNlIHR5cGUgaW4gYW4gaW5zdGFuY2UgZmxlZXQsIHdoaWNoIGRldGVybWluZXMgdGhlIEVDMiBpbnN0YW5jZXMgQW1hem9uIEVNUiBhdHRlbXB0cyB0b1xuICAgKiBwcm92aXNpb24gdG8gZnVsZmlsbCBPbi1EZW1hbmQgYW5kIFNwb3QgdGFyZ2V0IGNhcGFjaXRpZXMuXG4gICAqXG4gICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2Vtci9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9JbnN0YW5jZVR5cGVDb25maWcuaHRtbFxuICAgKlxuICAgKiBAZXhwZXJpbWVudGFsXG4gICAqL1xuICBleHBvcnQgaW50ZXJmYWNlIEluc3RhbmNlVHlwZUNvbmZpZ1Byb3BlcnR5IHtcbiAgICAvKipcbiAgICAgKiBUaGUgYmlkIHByaWNlIGZvciBlYWNoIEVDMiBTcG90IGluc3RhbmNlIHR5cGUgYXMgZGVmaW5lZCBieSBJbnN0YW5jZVR5cGUuIEV4cHJlc3NlZCBpbiBVU0QuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIE5vbmVcbiAgICAgKi9cbiAgICByZWFkb25seSBiaWRQcmljZT86IHN0cmluZztcblxuICAgIC8qKlxuICAgICAqIFRoZSBiaWQgcHJpY2UsIGFzIGEgcGVyY2VudGFnZSBvZiBPbi1EZW1hbmQgcHJpY2UuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIE5vbmVcbiAgICAgKi9cbiAgICByZWFkb25seSBiaWRQcmljZUFzUGVyY2VudGFnZU9mT25EZW1hbmRQcmljZT86IG51bWJlcjtcblxuICAgIC8qKlxuICAgICAqIEEgY29uZmlndXJhdGlvbiBjbGFzc2lmaWNhdGlvbiB0aGF0IGFwcGxpZXMgd2hlbiBwcm92aXNpb25pbmcgY2x1c3RlciBpbnN0YW5jZXMsIHdoaWNoIGNhbiBpbmNsdWRlIGNvbmZpZ3VyYXRpb25zIGZvciBhcHBsaWNhdGlvbnNcbiAgICAgKiBhbmQgc29mdHdhcmUgdGhhdCBydW4gb24gdGhlIGNsdXN0ZXIuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIE5vbmVcbiAgICAgKi9cbiAgICByZWFkb25seSBjb25maWd1cmF0aW9ucz86IENvbmZpZ3VyYXRpb25Qcm9wZXJ0eVtdO1xuXG4gICAgLyoqXG4gICAgICogVGhlIGNvbmZpZ3VyYXRpb24gb2YgQW1hem9uIEVsYXN0aWMgQmxvY2sgU3RvcmFnZSAoRUJTKSBhdHRhY2hlZCB0byBlYWNoIGluc3RhbmNlIGFzIGRlZmluZWQgYnkgSW5zdGFuY2VUeXBlLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBOb25lXG4gICAgICovXG4gICAgcmVhZG9ubHkgZWJzQ29uZmlndXJhdGlvbj86IEVic0NvbmZpZ3VyYXRpb25Qcm9wZXJ0eTtcblxuICAgIC8qKlxuICAgICAqIEFuIEVDMiBpbnN0YW5jZSB0eXBlXG4gICAgICovXG4gICAgcmVhZG9ubHkgaW5zdGFuY2VUeXBlOiBzdHJpbmc7XG5cbiAgICAvKipcbiAgICAgKiBUaGUgbnVtYmVyIG9mIHVuaXRzIHRoYXQgYSBwcm92aXNpb25lZCBpbnN0YW5jZSBvZiB0aGlzIHR5cGUgcHJvdmlkZXMgdG93YXJkIGZ1bGZpbGxpbmcgdGhlIHRhcmdldCBjYXBhY2l0aWVzIGRlZmluZWRcbiAgICAgKiBpbiB0aGUgSW5zdGFuY2VGbGVldENvbmZpZy5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gTm9uZVxuICAgICAqL1xuICAgIHJlYWRvbmx5IHdlaWdodGVkQ2FwYWNpdHk/OiBudW1iZXI7XG4gIH1cblxuICAvKipcbiAgICogU3BvdCBUaW1lb3V0IEFjdGlvbnNcbiAgICpcbiAgICogQGV4cGVyaW1lbnRhbFxuICAgKi9cbiAgZXhwb3J0IGVudW0gU3BvdFRpbWVvdXRBY3Rpb24ge1xuICAgIC8qKlxcXG4gICAgICogU1dJVENIX1RPX09OX0RFTUFORFxuICAgICAqL1xuICAgIFNXSVRDSF9UT19PTl9ERU1BTkQgPSAnU1dJVENIX1RPX09OX0RFTUFORCcsXG4gICAgLyoqXG4gICAgICogVEVSTUlOQVRFX0NMVVNURVJcbiAgICAgKi9cbiAgICBURVJNSU5BVEVfQ0xVU1RFUiA9ICdURVJNSU5BVEVfQ0xVU1RFUicsXG4gIH1cblxuICAvKipcbiAgICogVGhlIGxhdW5jaCBzcGVjaWZpY2F0aW9uIGZvciBTcG90IGluc3RhbmNlcyBpbiB0aGUgaW5zdGFuY2UgZmxlZXQsIHdoaWNoIGRldGVybWluZXMgdGhlIGRlZmluZWQgZHVyYXRpb24gYW5kIHByb3Zpc2lvbmluZyB0aW1lb3V0IGJlaGF2aW9yLlxuICAgKlxuICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9lbXIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfU3BvdFByb3Zpc2lvbmluZ1NwZWNpZmljYXRpb24uaHRtbFxuICAgKlxuICAgKiBAZXhwZXJpbWVudGFsXG4gICAqL1xuICBleHBvcnQgaW50ZXJmYWNlIFNwb3RQcm92aXNpb25pbmdTcGVjaWZpY2F0aW9uUHJvcGVydHkge1xuICAgIC8qKlxuICAgICAqIFRoZSBkZWZpbmVkIGR1cmF0aW9uIGZvciBTcG90IGluc3RhbmNlcyAoYWxzbyBrbm93biBhcyBTcG90IGJsb2NrcykgaW4gbWludXRlcy5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IE5vIGJsb2NrRHVyYXRpb25NaW51dGVzXG4gICAgICovXG4gICAgcmVhZG9ubHkgYmxvY2tEdXJhdGlvbk1pbnV0ZXM/OiBudW1iZXI7XG5cbiAgICAvKipcbiAgICAgKiBUaGUgYWN0aW9uIHRvIHRha2Ugd2hlbiBUYXJnZXRTcG90Q2FwYWNpdHkgaGFzIG5vdCBiZWVuIGZ1bGZpbGxlZCB3aGVuIHRoZSBUaW1lb3V0RHVyYXRpb25NaW51dGVzIGhhcyBleHBpcmVkXG4gICAgICovXG4gICAgcmVhZG9ubHkgdGltZW91dEFjdGlvbjogU3BvdFRpbWVvdXRBY3Rpb247XG5cbiAgICAvKipcbiAgICAgKiBUaGUgc3BvdCBwcm92aXNpb25pbmcgdGltZW91dCBwZXJpb2QgaW4gbWludXRlcy5cbiAgICAgKi9cbiAgICByZWFkb25seSB0aW1lb3V0RHVyYXRpb25NaW51dGVzOiBudW1iZXI7XG4gIH1cblxuICAvKipcbiAgICogVGhlIGxhdW5jaCBzcGVjaWZpY2F0aW9uIGZvciBTcG90IGluc3RhbmNlcyBpbiB0aGUgZmxlZXQsIHdoaWNoIGRldGVybWluZXMgdGhlIGRlZmluZWQgZHVyYXRpb24gYW5kIHByb3Zpc2lvbmluZyB0aW1lb3V0IGJlaGF2aW9yLlxuICAgKlxuICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9lbXIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfSW5zdGFuY2VGbGVldFByb3Zpc2lvbmluZ1NwZWNpZmljYXRpb25zLmh0bWxcbiAgICpcbiAgICogQGV4cGVyaW1lbnRhbFxuICAgKi9cbiAgZXhwb3J0IGludGVyZmFjZSBJbnN0YW5jZUZsZWV0UHJvdmlzaW9uaW5nU3BlY2lmaWNhdGlvbnNQcm9wZXJ0eSB7XG4gICAgLyoqXG4gICAgICogVGhlIGxhdW5jaCBzcGVjaWZpY2F0aW9uIGZvciBTcG90IGluc3RhbmNlcyBpbiB0aGUgZmxlZXQsIHdoaWNoIGRldGVybWluZXMgdGhlIGRlZmluZWQgZHVyYXRpb24gYW5kIHByb3Zpc2lvbmluZyB0aW1lb3V0IGJlaGF2aW9yLlxuICAgICAqL1xuICAgIHJlYWRvbmx5IHNwb3RTcGVjaWZpY2F0aW9uOiBTcG90UHJvdmlzaW9uaW5nU3BlY2lmaWNhdGlvblByb3BlcnR5O1xuICB9XG5cbiAgLyoqXG4gICAqIFRoZSBjb25maWd1cmF0aW9uIHRoYXQgZGVmaW5lcyBhbiBpbnN0YW5jZSBmbGVldC5cbiAgICpcbiAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZW1yL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0luc3RhbmNlRmxlZXRDb25maWcuaHRtbFxuICAgKlxuICAgKiBAZXhwZXJpbWVudGFsXG4gICAqL1xuICBleHBvcnQgaW50ZXJmYWNlIEluc3RhbmNlRmxlZXRDb25maWdQcm9wZXJ0eSB7XG4gICAgLyoqXG4gICAgICogVGhlIG5vZGUgdHlwZSB0aGF0IHRoZSBpbnN0YW5jZSBmbGVldCBob3N0cy4gVmFsaWQgdmFsdWVzIGFyZSBNQVNURVIsQ09SRSxhbmQgVEFTSy5cbiAgICAgKi9cbiAgICByZWFkb25seSBpbnN0YW5jZUZsZWV0VHlwZTogSW5zdGFuY2VSb2xlVHlwZTtcblxuICAgIC8qKlxuICAgICAqIFRoZSBpbnN0YW5jZSB0eXBlIGNvbmZpZ3VyYXRpb25zIHRoYXQgZGVmaW5lIHRoZSBFQzIgaW5zdGFuY2VzIGluIHRoZSBpbnN0YW5jZSBmbGVldC5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IE5vIGluc3RhbmNlVHBlQ29uZmlnc1xuICAgICAqL1xuICAgIHJlYWRvbmx5IGluc3RhbmNlVHlwZUNvbmZpZ3M/OiBJbnN0YW5jZVR5cGVDb25maWdQcm9wZXJ0eVtdO1xuXG4gICAgLyoqXG4gICAgICogVGhlIGxhdW5jaCBzcGVjaWZpY2F0aW9uIGZvciB0aGUgaW5zdGFuY2UgZmxlZXQuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCBObyBsYXVuY2hTcGVjaWZpY2F0aW9uc1xuICAgICAqL1xuICAgIHJlYWRvbmx5IGxhdW5jaFNwZWNpZmljYXRpb25zPzogSW5zdGFuY2VGbGVldFByb3Zpc2lvbmluZ1NwZWNpZmljYXRpb25zUHJvcGVydHk7XG5cbiAgICAvKipcbiAgICAgKiBUaGUgZnJpZW5kbHkgbmFtZSBvZiB0aGUgaW5zdGFuY2UgZmxlZXQuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCBObyBuYW1lXG4gICAgICovXG4gICAgcmVhZG9ubHkgbmFtZT86IHN0cmluZztcblxuICAgIC8qKlxuICAgICAqIFRoZSB0YXJnZXQgY2FwYWNpdHkgb2YgT24tRGVtYW5kIHVuaXRzIGZvciB0aGUgaW5zdGFuY2UgZmxlZXQsIHdoaWNoIGRldGVybWluZXMgaG93IG1hbnkgT24tRGVtYW5kIGluc3RhbmNlcyB0byBwcm92aXNpb24uXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCBObyB0YXJnZXRPbkRlbWFuZENhcGFjaXR5XG4gICAgICovXG4gICAgcmVhZG9ubHkgdGFyZ2V0T25EZW1hbmRDYXBhY2l0eT86IG51bWJlcjtcblxuICAgIC8qKlxuICAgICAqIFRoZSB0YXJnZXQgY2FwYWNpdHkgb2YgU3BvdCB1bml0cyBmb3IgdGhlIGluc3RhbmNlIGZsZWV0LCB3aGljaCBkZXRlcm1pbmVzIGhvdyBtYW55IFNwb3QgaW5zdGFuY2VzIHRvIHByb3Zpc2lvblxuICAgICAqXG4gICAgICogQGRlZmF1bHQgTm8gdGFyZ2V0U3BvdENhcGFjaXR5XG4gICAgICovXG4gICAgcmVhZG9ubHkgdGFyZ2V0U3BvdENhcGFjaXR5PzogbnVtYmVyO1xuICB9XG5cbiAgLyoqXG4gICAqIENsb3VkV2F0Y2ggQWxhcm0gQ29tcGFyaXNvbiBPcGVyYXRvcnNcbiAgICpcbiAgICogQGV4cGVyaW1lbnRhbFxuICAgKi9cbiAgZXhwb3J0IGVudW0gQ2xvdWRXYXRjaEFsYXJtQ29tcGFyaXNvbk9wZXJhdG9yIHtcbiAgICAvKipcbiAgICAgKiBHUkVBVEVSX1RIQU5fT1JfRVFVQUxcbiAgICAgKi9cbiAgICBHUkVBVEVSX1RIQU5fT1JfRVFVQUwgPSAnR1JFQVRFUl9USEFOX09SX0VRVUFMJyxcbiAgICAvKipcbiAgICAgKiBHUkVBVEVSX1RIQU5cbiAgICAgKi9cbiAgICBHUkVBVEVSX1RIQU4gPSAnR1JFQVRFUl9USEFOJyxcbiAgICAvKipcbiAgICAgKiBMRVNTX1RIQU5cbiAgICAgKi9cbiAgICBMRVNTX1RIQU4gPSAnTEVTU19USEFOJyxcbiAgICAvKipcbiAgICAgKiBMRVNTX1RIQU5fT1JfRVFVQUxcbiAgICAgKi9cbiAgICBMRVNTX1RIQU5fT1JfRVFVQUwgPSAnTEVTU19USEFOX09SX0VRVUFMJyxcbiAgfVxuXG4gIC8qKlxuICAgKiBDbG91ZFdhdGNoIEFsYXJtIFN0YXRpc3RpY3NcbiAgICpcbiAgICogQGV4cGVyaW1lbnRhbFxuICAgKi9cbiAgZXhwb3J0IGVudW0gQ2xvdWRXYXRjaEFsYXJtU3RhdGlzdGljIHtcbiAgICAvKipcbiAgICAgKiBTQU1QTEVfQ09VTlRcbiAgICAgKi9cbiAgICBTQU1QTEVfQ09VTlQgPSAnU0FNUExFX0NPVU5UJyxcbiAgICAvKipcbiAgICAgKiBBVkVSQUdFXG4gICAgICovXG4gICAgQVZFUkFHRSA9ICdBVkVSQUdFJyxcbiAgICAvKipcbiAgICAgKiBTVU1cbiAgICAgKi9cbiAgICBTVU0gPSAnU1VNJyxcbiAgICAvKipcbiAgICAgKiBNSU5JTVVNXG4gICAgICovXG4gICAgTUlOSU1VTSA9ICdNSU5JTVVNJyxcbiAgICAvKipcbiAgICAgKiBNQVhJTVVNXG4gICAgICovXG4gICAgTUFYSU1VTSA9ICdNQVhJTVVNJyxcbiAgfVxuXG4gIC8qKlxuICAgKiBDbG91ZFdhdGNoIEFsYXJtIFVuaXRzXG4gICAqXG4gICAqIEBleHBlcmltZW50YWxcbiAgICovXG4gIGV4cG9ydCBlbnVtIENsb3VkV2F0Y2hBbGFybVVuaXQge1xuICAgIC8qKlxuICAgICAqIE5PTkVcbiAgICAgKi9cbiAgICBOT05FID0gJ05PTkUnLFxuICAgIC8qKlxuICAgICAqIFNFQ09ORFNcbiAgICAgKi9cbiAgICBTRUNPTkRTID0gJ1NFQ09ORFMnLFxuICAgIC8qKlxuICAgICAqIE1JQ1JPX1NFQ09ORFNcbiAgICAgKi9cbiAgICBNSUNST19TRUNPTkRTID0gJ01JQ1JPX1NFQ09ORFMnLFxuICAgIC8qKlxuICAgICAqIE1JTExJX1NFQ09ORFNcbiAgICAgKi9cbiAgICBNSUxMSV9TRUNPTkRTID0gJ01JTExJX1NFQ09ORFMnLFxuICAgIC8qKlxuICAgICAqIEJZVEVTXG4gICAgICovXG4gICAgQllURVMgPSAnQllURVMnLFxuICAgIC8qKlxuICAgICAqIEtJTE9fQllURVNcbiAgICAgKi9cbiAgICBLSUxPX0JZVEVTID0gJ0tJTE9fQllURVMnLFxuICAgIC8qKlxuICAgICAqIE1FR0FfQllURVNcbiAgICAgKi9cbiAgICBNRUdBX0JZVEVTID0gJ01FR0FfQllURVMnLFxuICAgIC8qKlxuICAgICAqIEdJR0FfQllURVNcbiAgICAgKi9cbiAgICBHSUdBX0JZVEVTID0gJ0dJR0FfQllURVMnLFxuICAgIC8qKlxuICAgICAqIFRFUkFfQllURVNcbiAgICAgKi9cbiAgICBURVJBX0JZVEVTID0gJ1RFUkFfQllURVMnLFxuICAgIC8qKlxuICAgICAqIEJJVFNcbiAgICAgKi9cbiAgICBCSVRTID0gJ0JJVFMnLFxuICAgIC8qKlxuICAgICAqIEtJTE9fQklUU1xuICAgICAqL1xuICAgIEtJTE9fQklUUyA9ICdLSUxPX0JJVFMnLFxuICAgIC8qKlxuICAgICAqIE1FR0FfQklUU1xuICAgICAqL1xuICAgIE1FR0FfQklUUyA9ICdNRUdBX0JJVFMnLFxuICAgIC8qKlxuICAgICAqIEdJR0FfQklUU1xuICAgICAqL1xuICAgIEdJR0FfQklUUyA9ICdHSUdBX0JJVFMnLFxuICAgIC8qKlxuICAgICAqIFRFUkFfQklUU1xuICAgICAqL1xuICAgIFRFUkFfQklUUyA9ICdURVJBX0JJVFMnLFxuICAgIC8qKlxuICAgICAqIFBFUkNFTlRcbiAgICAgKi9cbiAgICBQRVJDRU5UID0gJ1BFUkNFTlQnLFxuICAgIC8qKlxuICAgICAqIENPVU5UXG4gICAgICovXG4gICAgQ09VTlQgPSAnQ09VTlQnLFxuICAgIC8qKlxuICAgICAqIEJZVEVTX1BFUl9TRUNPTkRcbiAgICAgKi9cbiAgICBCWVRFU19QRVJfU0VDT05EID0gJ0JZVEVTX1BFUl9TRUNPTkQnLFxuICAgIC8qKlxuICAgICAqIEtJTE9fQllURVNfUEVSX1NFQ09ORFxuICAgICAqL1xuICAgIEtJTE9fQllURVNfUEVSX1NFQ09ORCA9ICdLSUxPX0JZVEVTX1BFUl9TRUNPTkQnLFxuICAgIC8qKlxuICAgICAqIE1FR0FfQllURVNfUEVSX1NFQ09ORFxuICAgICAqL1xuICAgIE1FR0FfQllURVNfUEVSX1NFQ09ORCA9ICdNRUdBX0JZVEVTX1BFUl9TRUNPTkQnLFxuICAgIC8qKlxuICAgICAqIEdJR0FfQllURVNfUEVSX1NFQ09ORFxuICAgICAqL1xuICAgIEdJR0FfQllURVNfUEVSX1NFQ09ORCA9ICdHSUdBX0JZVEVTX1BFUl9TRUNPTkQnLFxuICAgIC8qKlxuICAgICAqIFRFUkFfQllURVNfUEVSX1NFQ09ORFxuICAgICAqL1xuICAgIFRFUkFfQllURVNfUEVSX1NFQ09ORCA9ICdURVJBX0JZVEVTX1BFUl9TRUNPTkQnLFxuICAgIC8qKlxuICAgICAqIEJJVFNfUEVSX1NFQ09ORFxuICAgICAqL1xuICAgIEJJVFNfUEVSX1NFQ09ORCA9ICdCSVRTX1BFUl9TRUNPTkQnLFxuICAgIC8qKlxuICAgICAqIEtJTE9fQklUU19QRVJfU0VDT05EXG4gICAgICovXG4gICAgS0lMT19CSVRTX1BFUl9TRUNPTkQgPSAnS0lMT19CSVRTX1BFUl9TRUNPTkQnLFxuICAgIC8qKlxuICAgICAqIE1FR0FfQklUU19QRVJfU0VDT05EXG4gICAgICovXG4gICAgTUVHQV9CSVRTX1BFUl9TRUNPTkQgPSAnTUVHQV9CSVRTX1BFUl9TRUNPTkQnLFxuICAgIC8qKlxuICAgICAqIEdJR0FfQklUU19QRVJfU0VDT05EXG4gICAgICovXG4gICAgR0lHQV9CSVRTX1BFUl9TRUNPTkQgPSAnR0lHQV9CSVRTX1BFUl9TRUNPTkQnLFxuICAgIC8qKlxuICAgICAqIFRFUkFfQklUU19QRVJfU0VDT05EXG4gICAgICovXG4gICAgVEVSQV9CSVRTX1BFUl9TRUNPTkQgPSAnVEVSQV9CSVRTX1BFUl9TRUNPTkQnLFxuICAgIC8qKlxuICAgICAqIENPVU5UX1BFUl9TRUNPTkRcbiAgICAgKi9cbiAgICBDT1VOVF9QRVJfU0VDT05EID0gJ0NPVU5UX1BFUl9TRUNPTkQnLFxuICB9XG5cbiAgLyoqXG4gICAqIEEgQ2xvdWRXYXRjaCBkaW1lbnNpb24sIHdoaWNoIGlzIHNwZWNpZmllZCB1c2luZyBhIEtleSAoa25vd24gYXMgYSBOYW1lIGluIENsb3VkV2F0Y2gpLCBWYWx1ZSBwYWlyLiBCeSBkZWZhdWx0LCBBbWF6b24gRU1SIHVzZXNcbiAgICogb25lIGRpbWVuc2lvbiB3aG9zZSBLZXkgaXMgSm9iRmxvd0lEIGFuZCBWYWx1ZSBpcyBhIHZhcmlhYmxlIHJlcHJlc2VudGluZyB0aGUgY2x1c3RlciBJRCwgd2hpY2ggaXMgJHtlbXIuY2x1c3RlcklkfS4gVGhpcyBlbmFibGVzXG4gICAqIHRoZSBydWxlIHRvIGJvb3RzdHJhcCB3aGVuIHRoZSBjbHVzdGVyIElEIGJlY29tZXMgYXZhaWxhYmxlXG4gICAqXG4gICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2Vtci9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9NZXRyaWNEaW1lbnNpb24uaHRtbFxuICAgKlxuICAgKiBAZXhwZXJpbWVudGFsXG4gICAqL1xuICBleHBvcnQgaW50ZXJmYWNlIE1ldHJpY0RpbWVuc2lvblByb3BlcnR5IHtcbiAgICAvKipcbiAgICAgKiBUaGUgZGltZW5zaW9uIG5hbWVcbiAgICAgKi9cbiAgICByZWFkb25seSBrZXk6IHN0cmluZztcblxuICAgIC8qKlxuICAgICAqIFRoZSBkaW1lbnNpb24gdmFsdWVcbiAgICAgKi9cbiAgICByZWFkb25seSB2YWx1ZTogc3RyaW5nO1xuICB9XG5cbiAgLyoqXG4gICAqIFRoZSBkZWZpbml0aW9uIG9mIGEgQ2xvdWRXYXRjaCBtZXRyaWMgYWxhcm0sIHdoaWNoIGRldGVybWluZXMgd2hlbiBhbiBhdXRvbWF0aWMgc2NhbGluZyBhY3Rpdml0eSBpcyB0cmlnZ2VyZWQuIFdoZW4gdGhlIGRlZmluZWQgYWxhcm0gY29uZGl0aW9uc1xuICAgKiBhcmUgc2F0aXNmaWVkLCBzY2FsaW5nIGFjdGl2aXR5IGJlZ2lucy5cbiAgICpcbiAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZW1yL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Nsb3VkV2F0Y2hBbGFybURlZmluaXRpb24uaHRtbFxuICAgKlxuICAgKiBAZXhwZXJpbWVudGFsXG4gICAqL1xuICBleHBvcnQgaW50ZXJmYWNlIENsb3VkV2F0Y2hBbGFybURlZmluaXRpb25Qcm9wZXJ0eSB7XG4gICAgLyoqXG4gICAgICogRGV0ZXJtaW5lcyBob3cgdGhlIG1ldHJpYyBzcGVjaWZpZWQgYnkgTWV0cmljTmFtZSBpcyBjb21wYXJlZCB0byB0aGUgdmFsdWUgc3BlY2lmaWVkIGJ5IFRocmVzaG9sZFxuICAgICAqL1xuICAgIHJlYWRvbmx5IGNvbXBhcmlzb25PcGVyYXRvcjogQ2xvdWRXYXRjaEFsYXJtQ29tcGFyaXNvbk9wZXJhdG9yO1xuXG4gICAgLyoqXG4gICAgICogQSBDbG91ZFdhdGNoIG1ldHJpYyBkaW1lbnNpb25cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gTm8gZGltZW5zaW9uc1xuICAgICAqL1xuICAgIHJlYWRvbmx5IGRpbWVuc2lvbnM/OiBNZXRyaWNEaW1lbnNpb25Qcm9wZXJ0eVtdO1xuXG4gICAgLyoqXG4gICAgICogVGhlIG51bWJlciBvZiBwZXJpb2RzLCBpbiBmaXZlLW1pbnV0ZSBpbmNyZW1lbnRzLCBkdXJpbmcgd2hpY2ggdGhlIGFsYXJtIGNvbmRpdGlvbiBtdXN0IGV4aXN0IGJlZm9yZSB0aGUgYWxhcm0gdHJpZ2dlcnMgYXV0b21hdGljXG4gICAgICogc2NhbGluZyBhY3Rpdml0eS5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IDFcbiAgICAgKi9cbiAgICByZWFkb25seSBldmFsdWF0aW9uUGVyaW9kcz86IG51bWJlcjtcblxuICAgIC8qKlxuICAgICAqIFRoZSBuYW1lIG9mIHRoZSBDbG91ZFdhdGNoIG1ldHJpYyB0aGF0IGlzIHdhdGNoZWQgdG8gZGV0ZXJtaW5lIGFuIGFsYXJtIGNvbmRpdGlvbi5cbiAgICAgKi9cbiAgICByZWFkb25seSBtZXRyaWNOYW1lOiBzdHJpbmc7XG5cbiAgICAvKipcbiAgICAgKiBUaGUgbmFtZXNwYWNlIGZvciB0aGUgQ2xvdWRXYXRjaCBtZXRyaWMuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAnQVdTL0VsYXN0aWNNYXBSZWR1Y2UnXG4gICAgICovXG4gICAgcmVhZG9ubHkgbmFtZXNwYWNlPzogc3RyaW5nO1xuXG4gICAgLyoqXG4gICAgICogVGhlIHBlcmlvZCwgaW4gc2Vjb25kcywgb3ZlciB3aGljaCB0aGUgc3RhdGlzdGljIGlzIGFwcGxpZWQuIEVNUiBDbG91ZFdhdGNoIG1ldHJpY3MgYXJlIGVtaXR0ZWQgZXZlcnkgZml2ZSBtaW51dGVzICgzMDAgc2Vjb25kcyksIHNvIGlmXG4gICAgICogYW4gRU1SIENsb3VkV2F0Y2ggbWV0cmljIGlzIHNwZWNpZmllZCwgc3BlY2lmeSAzMDAuXG4gICAgICovXG4gICAgcmVhZG9ubHkgcGVyaW9kOiBjZGsuRHVyYXRpb247XG5cbiAgICAvKipcbiAgICAgKiBUaGUgc3RhdGlzdGljIHRvIGFwcGx5IHRvIHRoZSBtZXRyaWMgYXNzb2NpYXRlZCB3aXRoIHRoZSBhbGFybS5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IENsb3VkV2F0Y2hBbGFybVN0YXRpc3RpYy5BVkVSQUdFXG4gICAgICovXG4gICAgcmVhZG9ubHkgc3RhdGlzdGljPzogQ2xvdWRXYXRjaEFsYXJtU3RhdGlzdGljO1xuXG4gICAgLyoqXG4gICAgICogVGhlIHZhbHVlIGFnYWluc3Qgd2hpY2ggdGhlIHNwZWNpZmllZCBzdGF0aXN0aWMgaXMgY29tcGFyZWQuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIE5vbmVcbiAgICAgKi9cbiAgICByZWFkb25seSB0aHJlc2hvbGQ/OiBudW1iZXI7XG5cbiAgICAvKipcbiAgICAgKiBUaGUgdW5pdCBvZiBtZWFzdXJlIGFzc29jaWF0ZWQgd2l0aCB0aGUgQ2xvdWRXYXRjaCBtZXRyaWMgYmVpbmcgd2F0Y2hlZC4gVGhlIHZhbHVlIHNwZWNpZmllZCBmb3IgVW5pdCBtdXN0IGNvcnJlc3BvbmQgdG8gdGhlIHVuaXRzXG4gICAgICogc3BlY2lmaWVkIGluIHRoZSBDbG91ZFdhdGNoIG1ldHJpYy5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IENsb3VkV2F0Y2hBbGFybVVuaXQuTk9ORVxuICAgICAqL1xuICAgIHJlYWRvbmx5IHVuaXQ/OiBDbG91ZFdhdGNoQWxhcm1Vbml0O1xuICB9XG5cbiAgLyoqXG4gICAqIFRoZSBjb25kaXRpb25zIHRoYXQgdHJpZ2dlciBhbiBhdXRvbWF0aWMgc2NhbGluZyBhY3Rpdml0eSBhbmQgdGhlIGRlZmluaXRpb24gb2YgYSBDbG91ZFdhdGNoIG1ldHJpYyBhbGFybS5cbiAgICogV2hlbiB0aGUgZGVmaW5lZCBhbGFybSBjb25kaXRpb25zIGFyZSBtZXQgYWxvbmcgd2l0aCBvdGhlciB0cmlnZ2VyIHBhcmFtZXRlcnMsIHNjYWxpbmcgYWN0aXZpdHkgYmVnaW5zLlxuICAgKlxuICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9lbXIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfU2NhbGluZ1RyaWdnZXIuaHRtbFxuICAgKlxuICAgKiBAZXhwZXJpbWVudGFsXG4gICAqL1xuICBleHBvcnQgaW50ZXJmYWNlIFNjYWxpbmdUcmlnZ2VyUHJvcGVydHkge1xuICAgIC8qKlxuICAgICAqIFRoZSBkZWZpbml0aW9uIG9mIGEgQ2xvdWRXYXRjaCBtZXRyaWMgYWxhcm0uIFdoZW4gdGhlIGRlZmluZWQgYWxhcm0gY29uZGl0aW9ucyBhcmUgbWV0IGFsb25nIHdpdGggb3RoZXIgdHJpZ2dlciBwYXJhbWV0ZXJzLFxuICAgICAqIHNjYWxpbmcgYWN0aXZpdHkgYmVnaW5zLlxuICAgICAqL1xuICAgIHJlYWRvbmx5IGNsb3VkV2F0Y2hBbGFybURlZmluaXRpb246IENsb3VkV2F0Y2hBbGFybURlZmluaXRpb25Qcm9wZXJ0eTtcbiAgfVxuXG4gIC8qKlxuICAgKiBFQzIgSW5zdGFuY2UgTWFya2V0XG4gICAqXG4gICAqIEBleHBlcmltZW50YWxcbiAgICovXG4gIGV4cG9ydCBlbnVtIEluc3RhbmNlTWFya2V0IHtcbiAgICAvKipcbiAgICAgKiBPbiBEZW1hbmQgSW5zdGFuY2VcbiAgICAgKi9cbiAgICBPTl9ERU1BTkQgPSAnT05fREVNQU5EJyxcbiAgICAvKipcbiAgICAgKiBTcG90IEluc3RhbmNlXG4gICAgICovXG4gICAgU1BPVCA9ICdTUE9UJyxcbiAgfVxuXG4gIC8qKlxuICAgKiBBdXRvU2NhbGluZyBBZGp1c3RtZW50IFR5cGVcbiAgICpcbiAgICogQGV4cGVyaW1lbnRhbFxuICAgKi9cbiAgZXhwb3J0IGVudW0gU2NhbGluZ0FkanVzdG1lbnRUeXBlIHtcbiAgICAvKipcbiAgICAgKiBDSEFOR0VfSU5fQ0FQQUNJVFlcbiAgICAgKi9cbiAgICBDSEFOR0VfSU5fQ0FQQUNJVFkgPSAnQ0hBTkdFX0lOX0NBUEFDSVRZJyxcbiAgICAvKipcbiAgICAgKiBQRVJDRU5UX0NIQU5HRV9JTl9DQVBBQ0lUWVxuICAgICAqL1xuICAgIFBFUkNFTlRfQ0hBTkdFX0lOX0NBUEFDSVRZID0gJ1BFUkNFTlRfQ0hBTkdFX0lOX0NBUEFDSVRZJyxcbiAgICAvKipcbiAgICAgKiBFWEFDVF9DQVBBQ0lUWVxuICAgICAqL1xuICAgIEVYQUNUX0NBUEFDSVRZID0gJ0VYQUNUX0NBUEFDSVRZJyxcbiAgfVxuXG4gIC8qKlxuICAgKiBBbiBhdXRvbWF0aWMgc2NhbGluZyBjb25maWd1cmF0aW9uLCB3aGljaCBkZXNjcmliZXMgaG93IHRoZSBwb2xpY3kgYWRkcyBvciByZW1vdmVzIGluc3RhbmNlcywgdGhlIGNvb2xkb3duIHBlcmlvZCwgYW5kIHRoZSBudW1iZXIgb2YgRUMyXG4gICAqIGluc3RhbmNlcyB0aGF0IHdpbGwgYmUgYWRkZWQgZWFjaCB0aW1lIHRoZSBDbG91ZFdhdGNoIG1ldHJpYyBhbGFybSBjb25kaXRpb24gaXMgc2F0aXNmaWVkLlxuICAgKlxuICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9lbXIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfU2ltcGxlU2NhbGluZ1BvbGljeUNvbmZpZ3VyYXRpb24uaHRtbFxuICAgKlxuICAgKiBAZXhwZXJpbWVudGFsXG4gICAqL1xuICBleHBvcnQgaW50ZXJmYWNlIFNpbXBsZVNjYWxpbmdQb2xpY3lDb25maWd1cmF0aW9uUHJvcGVydHkge1xuICAgIC8qKlxuICAgICAqIFRoZSB3YXkgaW4gd2hpY2ggRUMyIGluc3RhbmNlcyBhcmUgYWRkZWQgKGlmIFNjYWxpbmdBZGp1c3RtZW50IGlzIGEgcG9zaXRpdmUgbnVtYmVyKSBvciB0ZXJtaW5hdGVkIChpZiBTY2FsaW5nQWRqdXN0bWVudCBpcyBhIG5lZ2F0aXZlXG4gICAgICogbnVtYmVyKSBlYWNoIHRpbWUgdGhlIHNjYWxpbmcgYWN0aXZpdHkgaXMgdHJpZ2dlcmVkLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBOb25lXG4gICAgICovXG4gICAgcmVhZG9ubHkgYWRqdXN0bWVudFR5cGU/OiBTY2FsaW5nQWRqdXN0bWVudFR5cGU7XG5cbiAgICAvKipcbiAgICAgKiBUaGUgYW1vdW50IG9mIHRpbWUsIGluIHNlY29uZHMsIGFmdGVyIGEgc2NhbGluZyBhY3Rpdml0eSBjb21wbGV0ZXMgYmVmb3JlIGFueSBmdXJ0aGVyIHRyaWdnZXItcmVsYXRlZCBzY2FsaW5nIGFjdGl2aXRpZXMgY2FuIHN0YXJ0LlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgMFxuICAgICAqL1xuICAgIHJlYWRvbmx5IGNvb2xEb3duPzogbnVtYmVyO1xuXG4gICAgLyoqXG4gICAgICogVGhlIGFtb3VudCBieSB3aGljaCB0byBzY2FsZSBpbiBvciBzY2FsZSBvdXQsIGJhc2VkIG9uIHRoZSBzcGVjaWZpZWQgQWRqdXN0bWVudFR5cGUuIEEgcG9zaXRpdmUgdmFsdWUgYWRkcyB0byB0aGUgaW5zdGFuY2UgZ3JvdXAnc1xuICAgICAqIEVDMiBpbnN0YW5jZSBjb3VudCB3aGlsZSBhIG5lZ2F0aXZlIG51bWJlciByZW1vdmVzIGluc3RhbmNlcy4gSWYgQWRqdXN0bWVudFR5cGUgaXMgc2V0IHRvIEVYQUNUX0NBUEFDSVRZLCB0aGUgbnVtYmVyIHNob3VsZCBvbmx5IGJlXG4gICAgICogYSBwb3NpdGl2ZSBpbnRlZ2VyLlxuICAgICAqL1xuICAgIHJlYWRvbmx5IHNjYWxpbmdBZGp1c3RtZW50OiBudW1iZXI7XG4gIH1cblxuICAvKipcbiAgICogVGhlIHR5cGUgb2YgYWRqdXN0bWVudCB0aGUgYXV0b21hdGljIHNjYWxpbmcgYWN0aXZpdHkgbWFrZXMgd2hlbiB0cmlnZ2VyZWQsIGFuZCB0aGUgcGVyaW9kaWNpdHkgb2YgdGhlIGFkanVzdG1lbnQuXG4gICAqIEFuZCBhbiBhdXRvbWF0aWMgc2NhbGluZyBjb25maWd1cmF0aW9uLCB3aGljaCBkZXNjcmliZXMgaG93IHRoZSBwb2xpY3kgYWRkcyBvciByZW1vdmVzIGluc3RhbmNlcywgdGhlIGNvb2xkb3duIHBlcmlvZCxcbiAgICogYW5kIHRoZSBudW1iZXIgb2YgRUMyIGluc3RhbmNlcyB0aGF0IHdpbGwgYmUgYWRkZWQgZWFjaCB0aW1lIHRoZSBDbG91ZFdhdGNoIG1ldHJpYyBhbGFybSBjb25kaXRpb24gaXMgc2F0aXNmaWVkLlxuICAgKlxuICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9lbXIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfU2NhbGluZ0FjdGlvbi5odG1sXG4gICAqXG4gICAqIEBleHBlcmltZW50YWxcbiAgICovXG4gIGV4cG9ydCBpbnRlcmZhY2UgU2NhbGluZ0FjdGlvblByb3BlcnR5IHtcbiAgICAvKipcbiAgICAgKiBOb3QgYXZhaWxhYmxlIGZvciBpbnN0YW5jZSBncm91cHMuIEluc3RhbmNlIGdyb3VwcyB1c2UgdGhlIG1hcmtldCB0eXBlIHNwZWNpZmllZCBmb3IgdGhlIGdyb3VwLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBFTVIgc2VsZWN0ZWQgZGVmYXVsdFxuICAgICAqL1xuICAgIHJlYWRvbmx5IG1hcmtldD86IEluc3RhbmNlTWFya2V0O1xuXG4gICAgLyoqXG4gICAgICogVGhlIHR5cGUgb2YgYWRqdXN0bWVudCB0aGUgYXV0b21hdGljIHNjYWxpbmcgYWN0aXZpdHkgbWFrZXMgd2hlbiB0cmlnZ2VyZWQsIGFuZCB0aGUgcGVyaW9kaWNpdHkgb2YgdGhlIGFkanVzdG1lbnQuXG4gICAgICovXG4gICAgcmVhZG9ubHkgc2ltcGxlU2NhbGluZ1BvbGljeUNvbmZpZ3VyYXRpb246IFNpbXBsZVNjYWxpbmdQb2xpY3lDb25maWd1cmF0aW9uUHJvcGVydHk7XG4gIH1cblxuICAvKipcbiAgICogQSBzY2FsZS1pbiBvciBzY2FsZS1vdXQgcnVsZSB0aGF0IGRlZmluZXMgc2NhbGluZyBhY3Rpdml0eSwgaW5jbHVkaW5nIHRoZSBDbG91ZFdhdGNoIG1ldHJpYyBhbGFybSB0aGF0IHRyaWdnZXJzIGFjdGl2aXR5LCBob3cgRUMyXG4gICAqIGluc3RhbmNlcyBhcmUgYWRkZWQgb3IgcmVtb3ZlZCwgYW5kIHRoZSBwZXJpb2RpY2l0eSBvZiBhZGp1c3RtZW50cy5cbiAgICpcbiAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZW1yL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1NjYWxpbmdSdWxlLmh0bWxcbiAgICpcbiAgICogQGV4cGVyaW1lbnRhbFxuICAgKi9cbiAgZXhwb3J0IGludGVyZmFjZSBTY2FsaW5nUnVsZVByb3BlcnR5IHtcbiAgICAvKipcbiAgICAgKiBUaGUgY29uZGl0aW9ucyB0aGF0IHRyaWdnZXIgYW4gYXV0b21hdGljIHNjYWxpbmcgYWN0aXZpdHkuXG4gICAgICovXG4gICAgcmVhZG9ubHkgYWN0aW9uOiBTY2FsaW5nQWN0aW9uUHJvcGVydHk7XG5cbiAgICAvKipcbiAgICAgKiBBIGZyaWVuZGx5LCBtb3JlIHZlcmJvc2UgZGVzY3JpcHRpb24gb2YgdGhlIGF1dG9tYXRpYyBzY2FsaW5nIHJ1bGUuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIE5vbmVcbiAgICAgKi9cbiAgICByZWFkb25seSBkZXNjcmlwdGlvbj86IHN0cmluZztcblxuICAgIC8qKlxuICAgICAqIFRoZSBuYW1lIHVzZWQgdG8gaWRlbnRpZnkgYW4gYXV0b21hdGljIHNjYWxpbmcgcnVsZS4gUnVsZSBuYW1lcyBtdXN0IGJlIHVuaXF1ZSB3aXRoaW4gYSBzY2FsaW5nIHBvbGljeS5cbiAgICAgKi9cbiAgICByZWFkb25seSBuYW1lOiBzdHJpbmc7XG5cbiAgICAvKipcbiAgICAgKiBUaGUgQ2xvdWRXYXRjaCBhbGFybSBkZWZpbml0aW9uIHRoYXQgZGV0ZXJtaW5lcyB3aGVuIGF1dG9tYXRpYyBzY2FsaW5nIGFjdGl2aXR5IGlzIHRyaWdnZXJlZC5cbiAgICAgKi9cbiAgICByZWFkb25seSB0cmlnZ2VyOiBTY2FsaW5nVHJpZ2dlclByb3BlcnR5O1xuICB9XG5cbiAgLyoqXG4gICAqIFRoZSB1cHBlciBhbmQgbG93ZXIgRUMyIGluc3RhbmNlIGxpbWl0cyBmb3IgYW4gYXV0b21hdGljIHNjYWxpbmcgcG9saWN5LiBBdXRvbWF0aWMgc2NhbGluZyBhY3Rpdml0aWVzIHRyaWdnZXJlZCBieSBhdXRvbWF0aWMgc2NhbGluZ1xuICAgKiBydWxlcyB3aWxsIG5vdCBjYXVzZSBhbiBpbnN0YW5jZSBncm91cCB0byBncm93IGFib3ZlIG9yIGJlbG93IHRoZXNlIGxpbWl0cy5cbiAgICpcbiAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZW1yL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1NjYWxpbmdDb25zdHJhaW50cy5odG1sXG4gICAqXG4gICAqIEBleHBlcmltZW50YWxcbiAgICovXG4gIGV4cG9ydCBpbnRlcmZhY2UgU2NhbGluZ0NvbnN0cmFpbnRzUHJvcGVydHkge1xuICAgIC8qKlxuICAgICAqIFRoZSB1cHBlciBib3VuZGFyeSBvZiBFQzIgaW5zdGFuY2VzIGluIGFuIGluc3RhbmNlIGdyb3VwIGJleW9uZCB3aGljaCBzY2FsaW5nIGFjdGl2aXRpZXMgYXJlIG5vdCBhbGxvd2VkIHRvIGdyb3cuIFNjYWxlLW91dFxuICAgICAqIGFjdGl2aXRpZXMgd2lsbCBub3QgYWRkIGluc3RhbmNlcyBiZXlvbmQgdGhpcyBib3VuZGFyeS5cbiAgICAgKi9cbiAgICByZWFkb25seSBtYXhDYXBhY2l0eTogbnVtYmVyO1xuXG4gICAgLyoqXG4gICAgICogVGhlIGxvd2VyIGJvdW5kYXJ5IG9mIEVDMiBpbnN0YW5jZXMgaW4gYW4gaW5zdGFuY2UgZ3JvdXAgYmVsb3cgd2hpY2ggc2NhbGluZyBhY3Rpdml0aWVzIGFyZSBub3QgYWxsb3dlZCB0byBzaHJpbmsuIFNjYWxlLWluXG4gICAgICogYWN0aXZpdGllcyB3aWxsIG5vdCB0ZXJtaW5hdGUgaW5zdGFuY2VzIGJlbG93IHRoaXMgYm91bmRhcnkuXG4gICAgICovXG4gICAgcmVhZG9ubHkgbWluQ2FwYWNpdHk6IG51bWJlcjtcbiAgfVxuXG4gIC8qKlxuICAgKiBBbiBhdXRvbWF0aWMgc2NhbGluZyBwb2xpY3kgZm9yIGEgY29yZSBpbnN0YW5jZSBncm91cCBvciB0YXNrIGluc3RhbmNlIGdyb3VwIGluIGFuIEFtYXpvbiBFTVIgY2x1c3Rlci5cbiAgICpcbiAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZW1yL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0F1dG9TY2FsaW5nUG9saWN5Lmh0bWxcbiAgICpcbiAgICogQGV4cGVyaW1lbnRhbFxuICAgKi9cbiAgZXhwb3J0IGludGVyZmFjZSBBdXRvU2NhbGluZ1BvbGljeVByb3BlcnR5IHtcbiAgICAvKipcbiAgICAgKiBUaGUgdXBwZXIgYW5kIGxvd2VyIEVDMiBpbnN0YW5jZSBsaW1pdHMgZm9yIGFuIGF1dG9tYXRpYyBzY2FsaW5nIHBvbGljeS4gQXV0b21hdGljIHNjYWxpbmcgYWN0aXZpdHkgd2lsbCBub3QgY2F1c2UgYW4gaW5zdGFuY2VcbiAgICAgKiBncm91cCB0byBncm93IGFib3ZlIG9yIGJlbG93IHRoZXNlIGxpbWl0cy5cbiAgICAgKi9cbiAgICByZWFkb25seSBjb25zdHJhaW50czogU2NhbGluZ0NvbnN0cmFpbnRzUHJvcGVydHk7XG5cbiAgICAvKipcbiAgICAgKiBUaGUgc2NhbGUtaW4gYW5kIHNjYWxlLW91dCBydWxlcyB0aGF0IGNvbXByaXNlIHRoZSBhdXRvbWF0aWMgc2NhbGluZyBwb2xpY3kuXG4gICAgICovXG4gICAgcmVhZG9ubHkgcnVsZXM6IFNjYWxpbmdSdWxlUHJvcGVydHlbXTtcbiAgfVxuXG4gIC8qKlxuICAgKiBDb25maWd1cmF0aW9uIGRlZmluaW5nIGEgbmV3IGluc3RhbmNlIGdyb3VwLlxuICAgKlxuICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9lbXIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfSW5zdGFuY2VHcm91cENvbmZpZy5odG1sXG4gICAqXG4gICAqIEBleHBlcmltZW50YWxcbiAgICovXG4gIGV4cG9ydCBpbnRlcmZhY2UgSW5zdGFuY2VHcm91cENvbmZpZ1Byb3BlcnR5IHtcbiAgICAvKipcbiAgICAgKiBBbiBhdXRvbWF0aWMgc2NhbGluZyBwb2xpY3kgZm9yIGEgY29yZSBpbnN0YW5jZSBncm91cCBvciB0YXNrIGluc3RhbmNlIGdyb3VwIGluIGFuIEFtYXpvbiBFTVIgY2x1c3Rlci5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gTm9uZVxuICAgICAqL1xuICAgIHJlYWRvbmx5IGF1dG9TY2FsaW5nUG9saWN5PzogQXV0b1NjYWxpbmdQb2xpY3lQcm9wZXJ0eTtcblxuICAgIC8qKlxuICAgICAqIFRoZSBiaWQgcHJpY2UgZm9yIGVhY2ggRUMyIFNwb3QgaW5zdGFuY2UgdHlwZSBhcyBkZWZpbmVkIGJ5IEluc3RhbmNlVHlwZS4gRXhwcmVzc2VkIGluIFVTRC5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gTm9uZVxuICAgICAqL1xuICAgIHJlYWRvbmx5IGJpZFByaWNlPzogc3RyaW5nO1xuXG4gICAgLyoqXG4gICAgICogVGhlIGxpc3Qgb2YgY29uZmlndXJhdGlvbnMgc3VwcGxpZWQgZm9yIGFuIEVNUiBjbHVzdGVyIGluc3RhbmNlIGdyb3VwLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBOb25lXG4gICAgICovXG4gICAgcmVhZG9ubHkgY29uZmlndXJhdGlvbnM/OiBDb25maWd1cmF0aW9uUHJvcGVydHlbXTtcblxuICAgIC8qKlxuICAgICAqIEVCUyBjb25maWd1cmF0aW9ucyB0aGF0IHdpbGwgYmUgYXR0YWNoZWQgdG8gZWFjaCBFQzIgaW5zdGFuY2UgaW4gdGhlIGluc3RhbmNlIGdyb3VwLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBOb25lXG4gICAgICovXG4gICAgcmVhZG9ubHkgZWJzQ29uZmlndXJhdGlvbj86IEVic0NvbmZpZ3VyYXRpb25Qcm9wZXJ0eTtcblxuICAgIC8qKlxuICAgICAqIFRhcmdldCBudW1iZXIgb2YgaW5zdGFuY2VzIGZvciB0aGUgaW5zdGFuY2UgZ3JvdXAuXG4gICAgICovXG4gICAgcmVhZG9ubHkgaW5zdGFuY2VDb3VudDogbnVtYmVyO1xuXG4gICAgLyoqXG4gICAgICogVGhlIHJvbGUgb2YgdGhlIGluc3RhbmNlIGdyb3VwIGluIHRoZSBjbHVzdGVyLlxuICAgICAqL1xuICAgIHJlYWRvbmx5IGluc3RhbmNlUm9sZTogSW5zdGFuY2VSb2xlVHlwZTtcblxuICAgIC8qKlxuICAgICAqIFRoZSBFQzIgaW5zdGFuY2UgdHlwZSBmb3IgYWxsIGluc3RhbmNlcyBpbiB0aGUgaW5zdGFuY2UgZ3JvdXAuXG4gICAgICovXG4gICAgcmVhZG9ubHkgaW5zdGFuY2VUeXBlOiBzdHJpbmc7XG5cbiAgICAvKipcbiAgICAgKiBNYXJrZXQgdHlwZSBvZiB0aGUgRUMyIGluc3RhbmNlcyB1c2VkIHRvIGNyZWF0ZSBhIGNsdXN0ZXIgbm9kZS5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gRU1SIHNlbGVjdGVkIGRlZmF1bHRcbiAgICAgKi9cbiAgICByZWFkb25seSBtYXJrZXQ/OiBJbnN0YW5jZU1hcmtldDtcblxuICAgIC8qKlxuICAgICAqIEZyaWVuZGx5IG5hbWUgZ2l2ZW4gdG8gdGhlIGluc3RhbmNlIGdyb3VwLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBOb25lXG4gICAgICovXG4gICAgcmVhZG9ubHkgbmFtZT86IHN0cmluZztcbiAgfVxuXG4gIC8qKlxuICAgKiBUaGUgQW1hem9uIEVDMiBBdmFpbGFiaWxpdHkgWm9uZSBjb25maWd1cmF0aW9uIG9mIHRoZSBjbHVzdGVyIChqb2IgZmxvdykuXG4gICAqXG4gICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2Vtci9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9QbGFjZW1lbnRUeXBlLmh0bWxcbiAgICpcbiAgICogQGV4cGVyaW1lbnRhbFxuICAgKi9cbiAgZXhwb3J0IGludGVyZmFjZSBQbGFjZW1lbnRUeXBlUHJvcGVydHkge1xuICAgIC8qKlxuICAgICAqIFRoZSBBbWF6b24gRUMyIEF2YWlsYWJpbGl0eSBab25lIGZvciB0aGUgY2x1c3Rlci4gQXZhaWxhYmlsaXR5Wm9uZSBpcyB1c2VkIGZvciB1bmlmb3JtIGluc3RhbmNlIGdyb3Vwcywgd2hpbGUgQXZhaWxhYmlsaXR5Wm9uZXNcbiAgICAgKiAocGx1cmFsKSBpcyB1c2VkIGZvciBpbnN0YW5jZSBmbGVldHMuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIEVNUiBzZWxlY3RlZCBkZWZhdWx0XG4gICAgICovXG4gICAgcmVhZG9ubHkgYXZhaWxhYmlsaXR5Wm9uZT86IHN0cmluZztcblxuICAgIC8qKlxuICAgICAqIFdoZW4gbXVsdGlwbGUgQXZhaWxhYmlsaXR5IFpvbmVzIGFyZSBzcGVjaWZpZWQsIEFtYXpvbiBFTVIgZXZhbHVhdGVzIHRoZW0gYW5kIGxhdW5jaGVzIGluc3RhbmNlcyBpbiB0aGUgb3B0aW1hbCBBdmFpbGFiaWxpdHkgWm9uZS5cbiAgICAgKiBBdmFpbGFiaWxpdHlab25lcyBpcyB1c2VkIGZvciBpbnN0YW5jZSBmbGVldHMsIHdoaWxlIEF2YWlsYWJpbGl0eVpvbmUgKHNpbmd1bGFyKSBpcyB1c2VkIGZvciB1bmlmb3JtIGluc3RhbmNlIGdyb3Vwcy5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gRU1SIHNlbGVjdGVkIGRlZmF1bHRcbiAgICAgKi9cbiAgICByZWFkb25seSBhdmFpbGFiaWxpdHlab25lcz86IHN0cmluZ1tdO1xuICB9XG5cbiAgLyoqXG4gICAqIEEgc3BlY2lmaWNhdGlvbiBvZiB0aGUgbnVtYmVyIGFuZCB0eXBlIG9mIEFtYXpvbiBFQzIgaW5zdGFuY2VzLlxuICAgKlxuICAgKiBTZWUgdGhlIFJ1bkpvYkZsb3cgQVBJIGZvciBjb21wbGV0ZSBkb2N1bWVudGF0aW9uIG9uIGlucHV0IHBhcmFtZXRlcnNcbiAgICpcbiAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZW1yL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0pvYkZsb3dJbnN0YW5jZXNDb25maWcuaHRtbFxuICAgKlxuICAgKiBAZXhwZXJpbWVudGFsXG4gICAqL1xuICBleHBvcnQgaW50ZXJmYWNlIEluc3RhbmNlc0NvbmZpZ1Byb3BlcnR5IHtcbiAgICAvKipcbiAgICAgKiBBIGxpc3Qgb2YgYWRkaXRpb25hbCBBbWF6b24gRUMyIHNlY3VyaXR5IGdyb3VwIElEcyBmb3IgdGhlIG1hc3RlciBub2RlLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBOb25lXG4gICAgICovXG4gICAgcmVhZG9ubHkgYWRkaXRpb25hbE1hc3RlclNlY3VyaXR5R3JvdXBzPzogc3RyaW5nW107XG5cbiAgICAvKipcbiAgICAgKiBBIGxpc3Qgb2YgYWRkaXRpb25hbCBBbWF6b24gRUMyIHNlY3VyaXR5IGdyb3VwIElEcyBmb3IgdGhlIGNvcmUgYW5kIHRhc2sgbm9kZXMuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIE5vbmVcbiAgICAgKi9cbiAgICByZWFkb25seSBhZGRpdGlvbmFsU2xhdmVTZWN1cml0eUdyb3Vwcz86IHN0cmluZ1tdO1xuXG4gICAgLyoqXG4gICAgICogVGhlIG5hbWUgb2YgdGhlIEVDMiBrZXkgcGFpciB0aGF0IGNhbiBiZSB1c2VkIHRvIHNzaCB0byB0aGUgbWFzdGVyIG5vZGUgYXMgdGhlIHVzZXIgY2FsbGVkIFwiaGFkb29wLlwiXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIE5vbmVcbiAgICAgKi9cbiAgICByZWFkb25seSBlYzJLZXlOYW1lPzogc3RyaW5nO1xuXG4gICAgLyoqXG4gICAgICogQXBwbGllcyB0byBjbHVzdGVycyB0aGF0IHVzZSB0aGUgdW5pZm9ybSBpbnN0YW5jZSBncm91cCBjb25maWd1cmF0aW9uLiBUbyBsYXVuY2ggdGhlIGNsdXN0ZXIgaW4gQW1hem9uIFZpcnR1YWwgUHJpdmF0ZSBDbG91ZCAoQW1hem9uIFZQQyksXG4gICAgICogc2V0IHRoaXMgcGFyYW1ldGVyIHRvIHRoZSBpZGVudGlmaWVyIG9mIHRoZSBBbWF6b24gVlBDIHN1Ym5ldCB3aGVyZSB5b3Ugd2FudCB0aGUgY2x1c3RlciB0byBsYXVuY2guXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCBFTVIgc2VsZWN0ZWQgZGVmYXVsdFxuICAgICAqL1xuICAgIHJlYWRvbmx5IGVjMlN1Ym5ldElkPzogc3RyaW5nO1xuXG4gICAgLyoqXG4gICAgICogQXBwbGllcyB0byBjbHVzdGVycyB0aGF0IHVzZSB0aGUgaW5zdGFuY2UgZmxlZXQgY29uZmlndXJhdGlvbi4gV2hlbiBtdWx0aXBsZSBFQzIgc3VibmV0IElEcyBhcmUgc3BlY2lmaWVkLCBBbWF6b24gRU1SIGV2YWx1YXRlcyB0aGVtIGFuZFxuICAgICAqIGxhdW5jaGVzIGluc3RhbmNlcyBpbiB0aGUgb3B0aW1hbCBzdWJuZXQuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCBFTVIgc2VsZWN0ZWQgZGVmYXVsdFxuICAgICAqL1xuICAgIHJlYWRvbmx5IGVjMlN1Ym5ldElkcz86IHN0cmluZ1tdO1xuXG4gICAgLyoqXG4gICAgICogVGhlIGlkZW50aWZpZXIgb2YgdGhlIEFtYXpvbiBFQzIgc2VjdXJpdHkgZ3JvdXAgZm9yIHRoZSBtYXN0ZXIgbm9kZS5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gTm9uZVxuICAgICAqL1xuICAgIHJlYWRvbmx5IGVtck1hbmFnZWRNYXN0ZXJTZWN1cml0eUdyb3VwPzogc3RyaW5nO1xuXG4gICAgLyoqXG4gICAgICogVGhlIGlkZW50aWZpZXIgb2YgdGhlIEFtYXpvbiBFQzIgc2VjdXJpdHkgZ3JvdXAgZm9yIHRoZSBjb3JlIGFuZCB0YXNrIG5vZGVzLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBOb25lXG4gICAgICovXG4gICAgcmVhZG9ubHkgZW1yTWFuYWdlZFNsYXZlU2VjdXJpdHlHcm91cD86IHN0cmluZztcblxuICAgIC8qKlxuICAgICAqIEFwcGxpZXMgb25seSB0byBBbWF6b24gRU1SIHJlbGVhc2UgdmVyc2lvbnMgZWFybGllciB0aGFuIDQuMC4gVGhlIEhhZG9vcCB2ZXJzaW9uIGZvciB0aGUgY2x1c3Rlci5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gMC4xOCBpZiB0aGUgQW1pVmVyc2lvbiBwYXJhbWV0ZXIgaXMgbm90IHNldC4gSWYgQW1pVmVyc2lvbiBpcyBzZXQsIHRoZSB2ZXJzaW9uIG9mIEhhZG9vcCBmb3IgdGhhdCBBTUkgdmVyc2lvbiBpcyB1c2VkLlxuICAgICAqL1xuICAgIHJlYWRvbmx5IGhhZG9vcFZlcnNpb24/OiBzdHJpbmc7XG5cbiAgICAvKipcbiAgICAgKiBUaGUgbnVtYmVyIG9mIEVDMiBpbnN0YW5jZXMgaW4gdGhlIGNsdXN0ZXIuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAwXG4gICAgICovXG4gICAgcmVhZG9ubHkgaW5zdGFuY2VDb3VudD86IG51bWJlcjtcblxuICAgIC8qKlxuICAgICAqIERlc2NyaWJlcyB0aGUgRUMyIGluc3RhbmNlcyBhbmQgaW5zdGFuY2UgY29uZmlndXJhdGlvbnMgZm9yIGNsdXN0ZXJzIHRoYXQgdXNlIHRoZSBpbnN0YW5jZSBmbGVldCBjb25maWd1cmF0aW9uLlxuICAgICAqIFRoZSBpbnN0YW5jZSBmbGVldCBjb25maWd1cmF0aW9uIGlzIGF2YWlsYWJsZSBvbmx5IGluIEFtYXpvbiBFTVIgdmVyc2lvbnMgNC44LjAgYW5kIGxhdGVyLCBleGNsdWRpbmcgNS4wLnggdmVyc2lvbnMuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIE5vbmVcbiAgICAgKi9cbiAgICByZWFkb25seSBpbnN0YW5jZUZsZWV0cz86IEluc3RhbmNlRmxlZXRDb25maWdQcm9wZXJ0eVtdO1xuXG4gICAgLyoqXG4gICAgICogQ29uZmlndXJhdGlvbiBmb3IgdGhlIGluc3RhbmNlIGdyb3VwcyBpbiBhIGNsdXN0ZXIuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIE5vbmVcbiAgICAgKi9cbiAgICByZWFkb25seSBpbnN0YW5jZUdyb3Vwcz86IEluc3RhbmNlR3JvdXBDb25maWdQcm9wZXJ0eVtdO1xuXG4gICAgLyoqXG4gICAgICogVGhlIEVDMiBpbnN0YW5jZSB0eXBlIG9mIHRoZSBtYXN0ZXIgbm9kZS5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gTm9uZVxuICAgICAqL1xuICAgIHJlYWRvbmx5IG1hc3Rlckluc3RhbmNlVHlwZT86IHN0cmluZztcblxuICAgIC8qKlxuICAgICAqIFRoZSBBdmFpbGFiaWxpdHkgWm9uZSBpbiB3aGljaCB0aGUgY2x1c3RlciBydW5zLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBFTVIgc2VsZWN0ZWQgZGVmYXVsdFxuICAgICAqL1xuICAgIHJlYWRvbmx5IHBsYWNlbWVudD86IFBsYWNlbWVudFR5cGVQcm9wZXJ0eTtcblxuICAgIC8qKlxuICAgICAqIFRoZSBpZGVudGlmaWVyIG9mIHRoZSBBbWF6b24gRUMyIHNlY3VyaXR5IGdyb3VwIGZvciB0aGUgQW1hem9uIEVNUiBzZXJ2aWNlIHRvIGFjY2VzcyBjbHVzdGVycyBpbiBWUEMgcHJpdmF0ZSBzdWJuZXRzLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBOb25lXG4gICAgICovXG4gICAgcmVhZG9ubHkgc2VydmljZUFjY2Vzc1NlY3VyaXR5R3JvdXA/OiBzdHJpbmc7XG5cbiAgICAvKipcbiAgICAgKiBUaGUgRUMyIGluc3RhbmNlIHR5cGUgb2YgdGhlIGNvcmUgYW5kIHRhc2sgbm9kZXMuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIE5vbmVcbiAgICAgKi9cbiAgICByZWFkb25seSBzbGF2ZUluc3RhbmNlVHlwZT86IHN0cmluZztcblxuICAgIC8qKlxuICAgICAqIFNwZWNpZmllcyB3aGV0aGVyIHRvIGxvY2sgdGhlIGNsdXN0ZXIgdG8gcHJldmVudCB0aGUgQW1hem9uIEVDMiBpbnN0YW5jZXMgZnJvbSBiZWluZyB0ZXJtaW5hdGVkIGJ5IEFQSSBjYWxsLCB1c2VyIGludGVydmVudGlvbixcbiAgICAgKiBvciBpbiB0aGUgZXZlbnQgb2YgYSBqb2ItZmxvdyBlcnJvci5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IGZhbHNlXG4gICAgICovXG4gICAgcmVhZG9ubHkgdGVybWluYXRpb25Qcm90ZWN0ZWQ/OiBib29sZWFuO1xuICB9XG5cbiAgLyoqXG4gICAqIFByb3BlcnRpZXMgZm9yIHRoZSBFTVIgQ2x1c3RlciBBcHBsaWNhdGlvbnNcbiAgICpcbiAgICogQXBwbGllcyB0byBBbWF6b24gRU1SIHJlbGVhc2VzIDQuMCBhbmQgbGF0ZXIuIEEgY2FzZS1pbnNlbnNpdGl2ZSBsaXN0IG9mIGFwcGxpY2F0aW9ucyBmb3IgQW1hem9uIEVNUiB0byBpbnN0YWxsIGFuZCBjb25maWd1cmUgd2hlbiBsYXVuY2hpbmdcbiAgICogdGhlIGNsdXN0ZXIuXG4gICAqXG4gICAqIFNlZSB0aGUgUnVuSm9iRmxvdyBBUEkgZm9yIGNvbXBsZXRlIGRvY3VtZW50YXRpb24gb24gaW5wdXQgcGFyYW1ldGVyc1xuICAgKlxuICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9lbXIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQXBwbGljYXRpb24uaHRtbFxuICAgKlxuICAgKiBAZXhwZXJpbWVudGFsXG4gICAqL1xuICBleHBvcnQgaW50ZXJmYWNlIEFwcGxpY2F0aW9uQ29uZmlnUHJvcGVydHkge1xuICAgIC8qKlxuICAgICAqIFRoaXMgb3B0aW9uIGlzIGZvciBhZHZhbmNlZCB1c2VycyBvbmx5LiBUaGlzIGlzIG1ldGEgaW5mb3JtYXRpb24gYWJvdXQgdGhpcmQtcGFydHkgYXBwbGljYXRpb25zIHRoYXQgdGhpcmQtcGFydHkgdmVuZG9ycyB1c2VcbiAgICAgKiBmb3IgdGVzdGluZyBwdXJwb3Nlcy5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IE5vIGFkZGl0aW9uYWxJbmZvXG4gICAgICovXG4gICAgcmVhZG9ubHkgYWRkaXRpb25hbEluZm8/OiB7IFtrZXk6IHN0cmluZ106IHN0cmluZyB9O1xuXG4gICAgLyoqXG4gICAgICogQXJndW1lbnRzIGZvciBBbWF6b24gRU1SIHRvIHBhc3MgdG8gdGhlIGFwcGxpY2F0aW9uLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgTm8gYXJnc1xuICAgICAqL1xuICAgIHJlYWRvbmx5IGFyZ3M/OiBzdHJpbmdbXTtcblxuICAgIC8qKlxuICAgICAqIFRoZSBuYW1lIG9mIHRoZSBhcHBsaWNhdGlvbi5cbiAgICAgKi9cbiAgICByZWFkb25seSBuYW1lOiBzdHJpbmc7XG5cbiAgICAvKipcbiAgICAgKiBUaGUgdmVyc2lvbiBvZiB0aGUgYXBwbGljYXRpb24uXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCBObyB2ZXJzaW9uXG4gICAgICovXG4gICAgcmVhZG9ubHkgdmVyc2lvbj86IHN0cmluZztcbiAgfVxuXG4gIC8qKlxuICAgKiBDb25maWd1cmF0aW9uIG9mIHRoZSBzY3JpcHQgdG8gcnVuIGR1cmluZyBhIGJvb3RzdHJhcCBhY3Rpb24uXG4gICAqXG4gICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2Vtci9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9TY3JpcHRCb290c3RyYXBBY3Rpb25Db25maWcuaHRtbFxuICAgKlxuICAgKiBAZXhwZXJpbWVudGFsXG4gICAqL1xuICBleHBvcnQgaW50ZXJmYWNlIFNjcmlwdEJvb3RzdHJhcEFjdGlvbkNvbmZpZ1Byb3BlcnR5IHtcbiAgICAvKipcbiAgICAgKiBMb2NhdGlvbiBvZiB0aGUgc2NyaXB0IHRvIHJ1biBkdXJpbmcgYSBib290c3RyYXAgYWN0aW9uLiBDYW4gYmUgZWl0aGVyIGEgbG9jYXRpb24gaW4gQW1hem9uIFMzIG9yIG9uIGEgbG9jYWwgZmlsZSBzeXN0ZW0uXG4gICAgICovXG4gICAgcmVhZG9ubHkgcGF0aDogc3RyaW5nO1xuXG4gICAgLyoqXG4gICAgICogQSBsaXN0IG9mIGNvbW1hbmQgbGluZSBhcmd1bWVudHMgdG8gcGFzcyB0byB0aGUgYm9vdHN0cmFwIGFjdGlvbiBzY3JpcHQuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCBObyBhcmdzXG4gICAgICovXG4gICAgcmVhZG9ubHkgYXJncz86IHN0cmluZ1tdO1xuICB9XG5cbiAgLyoqXG4gICAqIENvbmZpZ3VyYXRpb24gb2YgYSBib290c3RyYXAgYWN0aW9uLlxuICAgKlxuICAgKiBTZWUgdGhlIFJ1bkpvYkZsb3cgQVBJIGZvciBjb21wbGV0ZSBkb2N1bWVudGF0aW9uIG9uIGlucHV0IHBhcmFtZXRlcnNcbiAgICpcbiAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZW1yL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Jvb3RzdHJhcEFjdGlvbkNvbmZpZy5odG1sXG4gICAqXG4gICAqIEBleHBlcmltZW50YWxcbiAgICovXG4gIGV4cG9ydCBpbnRlcmZhY2UgQm9vdHN0cmFwQWN0aW9uQ29uZmlnUHJvcGVydHkge1xuICAgIC8qKlxuICAgICAqIFRoZSBuYW1lIG9mIHRoZSBib290c3RyYXAgYWN0aW9uLlxuICAgICAqL1xuICAgIHJlYWRvbmx5IG5hbWU6IHN0cmluZztcblxuICAgIC8qKlxuICAgICAqIFRoZSBzY3JpcHQgcnVuIGJ5IHRoZSBib290c3RyYXAgYWN0aW9uXG4gICAgICovXG4gICAgcmVhZG9ubHkgc2NyaXB0Qm9vdHN0cmFwQWN0aW9uOiBTY3JpcHRCb290c3RyYXBBY3Rpb25Db25maWdQcm9wZXJ0eTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBbiBvcHRpb25hbCBjb25maWd1cmF0aW9uIHNwZWNpZmljYXRpb24gdG8gYmUgdXNlZCB3aGVuIHByb3Zpc2lvbmluZyBjbHVzdGVyIGluc3RhbmNlcywgd2hpY2ggY2FuIGluY2x1ZGUgY29uZmlndXJhdGlvbnMgZm9yXG4gICAqIGFwcGxpY2F0aW9ucyBhbmQgc29mdHdhcmUgYnVuZGxlZCB3aXRoIEFtYXpvbiBFTVIuXG4gICAqXG4gICAqIFNlZSB0aGUgUnVuSm9iRmxvdyBBUEkgZm9yIGNvbXBsZXRlIGRvY3VtZW50YXRpb24gb24gaW5wdXQgcGFyYW1ldGVyc1xuICAgKlxuICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9lbXIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQ29uZmlndXJhdGlvbi5odG1sXG4gICAqXG4gICAqIEBleHBlcmltZW50YWxcbiAgICovXG4gIGV4cG9ydCBpbnRlcmZhY2UgQ29uZmlndXJhdGlvblByb3BlcnR5IHtcbiAgICAvKipcbiAgICAgKiBUaGUgY2xhc3NpZmljYXRpb24gd2l0aGluIGEgY29uZmlndXJhdGlvbi5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IE5vIGNsYXNzaWZpY2F0aW9uXG4gICAgICovXG4gICAgcmVhZG9ubHkgY2xhc3NpZmljYXRpb24/OiBzdHJpbmc7XG5cbiAgICAvKipcbiAgICAgKiBBIHNldCBvZiBwcm9wZXJ0aWVzIHNwZWNpZmllZCB3aXRoaW4gYSBjb25maWd1cmF0aW9uIGNsYXNzaWZpY2F0aW9uLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgTm8gcHJvcGVydGllc1xuICAgICAqL1xuICAgIHJlYWRvbmx5IHByb3BlcnRpZXM/OiB7IFtrZXk6IHN0cmluZ106IHN0cmluZyB9O1xuXG4gICAgLyoqXG4gICAgICogQSBsaXN0IG9mIGFkZGl0aW9uYWwgY29uZmlndXJhdGlvbnMgdG8gYXBwbHkgd2l0aGluIGEgY29uZmlndXJhdGlvbiBvYmplY3QuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCBObyBjb25maWd1cmF0aW9uc1xuICAgICAqL1xuICAgIHJlYWRvbmx5IGNvbmZpZ3VyYXRpb25zPzogQ29uZmlndXJhdGlvblByb3BlcnR5W107XG4gIH1cblxuICAvKipcbiAgICogQXR0cmlidXRlcyBmb3IgS2VyYmVyb3MgY29uZmlndXJhdGlvbiB3aGVuIEtlcmJlcm9zIGF1dGhlbnRpY2F0aW9uIGlzIGVuYWJsZWQgdXNpbmcgYSBzZWN1cml0eSBjb25maWd1cmF0aW9uLlxuICAgKlxuICAgKiBTZWUgdGhlIFJ1bkpvYkZsb3cgQVBJIGZvciBjb21wbGV0ZSBkb2N1bWVudGF0aW9uIG9uIGlucHV0IHBhcmFtZXRlcnNcbiAgICpcbiAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZW1yL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0tlcmJlcm9zQXR0cmlidXRlcy5odG1sXG4gICAqXG4gICAqIEBleHBlcmltZW50YWxcbiAgICovXG4gIGV4cG9ydCBpbnRlcmZhY2UgS2VyYmVyb3NBdHRyaWJ1dGVzUHJvcGVydHkge1xuICAgIC8qKlxuICAgICAqIFRoZSBBY3RpdmUgRGlyZWN0b3J5IHBhc3N3b3JkIGZvciBBRERvbWFpbkpvaW5Vc2VyLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgTm8gYWREb21haW5Kb2luUGFzc3dvcmRcbiAgICAgKi9cbiAgICByZWFkb25seSBhZERvbWFpbkpvaW5QYXNzd29yZD86IHN0cmluZztcblxuICAgIC8qKlxuICAgICAqIFJlcXVpcmVkIG9ubHkgd2hlbiBlc3RhYmxpc2hpbmcgYSBjcm9zcy1yZWFsbSB0cnVzdCB3aXRoIGFuIEFjdGl2ZSBEaXJlY3RvcnkgZG9tYWluLiBBIHVzZXIgd2l0aCBzdWZmaWNpZW50IHByaXZpbGVnZXMgdG8gam9pblxuICAgICAqIHJlc291cmNlcyB0byB0aGUgZG9tYWluLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgTm8gYWREb21haW5Kb2luVXNlclxuICAgICAqL1xuICAgIHJlYWRvbmx5IGFkRG9tYWluSm9pblVzZXI/OiBzdHJpbmc7XG5cbiAgICAvKipcbiAgICAgKiBSZXF1aXJlZCBvbmx5IHdoZW4gZXN0YWJsaXNoaW5nIGEgY3Jvc3MtcmVhbG0gdHJ1c3Qgd2l0aCBhIEtEQyBpbiBhIGRpZmZlcmVudCByZWFsbS4gVGhlIGNyb3NzLXJlYWxtIHByaW5jaXBhbCBwYXNzd29yZCwgd2hpY2hcbiAgICAgKiBtdXN0IGJlIGlkZW50aWNhbCBhY3Jvc3MgcmVhbG1zLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgTm8gY3Jvc3NSZWFsbVRydXN0UHJpbmNpcGFsUGFzc3dvcmRcbiAgICAgKi9cbiAgICByZWFkb25seSBjcm9zc1JlYWxtVHJ1c3RQcmluY2lwYWxQYXNzd29yZD86IHN0cmluZztcblxuICAgIC8qKlxuICAgICAqIFRoZSBwYXNzd29yZCB1c2VkIHdpdGhpbiB0aGUgY2x1c3RlciBmb3IgdGhlIGthZG1pbiBzZXJ2aWNlIG9uIHRoZSBjbHVzdGVyLWRlZGljYXRlZCBLREMsIHdoaWNoIG1haW50YWlucyBLZXJiZXJvcyBwcmluY2lwYWxzLFxuICAgICAqIHBhc3N3b3JkIHBvbGljaWVzLCBhbmQga2V5dGFicyBmb3IgdGhlIGNsdXN0ZXIuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCBObyBrZGNBZG1pblBhc3N3b3JkXG4gICAgICovXG4gICAgcmVhZG9ubHkga2RjQWRtaW5QYXNzd29yZD86IHN0cmluZztcblxuICAgIC8qKlxuICAgICAqIFRoZSBuYW1lIG9mIHRoZSBLZXJiZXJvcyByZWFsbSB0byB3aGljaCBhbGwgbm9kZXMgaW4gYSBjbHVzdGVyIGJlbG9uZy4gRm9yIGV4YW1wbGUsIEVDMi5JTlRFUk5BTC5cbiAgICAgKi9cbiAgICByZWFkb25seSByZWFsbTogc3RyaW5nO1xuICB9XG59XG4iXX0=