"use strict";
var _a, _b, _c, _d, _e, _f, _g;
Object.defineProperty(exports, "__esModule", { value: true });
exports.LookupMachineImage = exports.OperatingSystemType = exports.GenericWindowsImage = exports.GenericLinuxImage = exports.AmazonLinuxStorage = exports.AmazonLinuxVirt = exports.AmazonLinuxEdition = exports.AmazonLinuxGeneration = exports.AmazonLinuxImage = exports.AmazonLinuxCpuType = exports.WindowsImage = exports.GenericSSMParameterImage = exports.MachineImage = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ssm = require("@aws-cdk/aws-ssm");
const cxschema = require("@aws-cdk/cloud-assembly-schema");
const core_1 = require("@aws-cdk/core");
const user_data_1 = require("./user-data");
/**
 * Factory functions for standard Amazon Machine Image objects.
 *
 * @stability stable
 */
class MachineImage {
    /**
     * A Windows image that is automatically kept up-to-date.
     *
     * This Machine Image automatically updates to the latest version on every
     * deployment. Be aware this will cause your instances to be replaced when a
     * new version of the image becomes available. Do not store stateful information
     * on the instance if you are using this image.
     *
     * @stability stable
     */
    static latestWindows(version, props) {
        return new WindowsImage(version, props);
    }
    /**
     * An Amazon Linux image that is automatically kept up-to-date.
     *
     * This Machine Image automatically updates to the latest version on every
     * deployment. Be aware this will cause your instances to be replaced when a
     * new version of the image becomes available. Do not store stateful information
     * on the instance if you are using this image.
     *
     * @stability stable
     */
    static latestAmazonLinux(props) {
        return new AmazonLinuxImage(props);
    }
    /**
     * A Linux image where you specify the AMI ID for every region.
     *
     * @param amiMap For every region where you are deploying the stack, specify the AMI ID for that region.
     * @param props Customize the image by supplying additional props.
     * @stability stable
     */
    static genericLinux(amiMap, props) {
        return new GenericLinuxImage(amiMap, props);
    }
    /**
     * A Windows image where you specify the AMI ID for every region.
     *
     * @param amiMap For every region where you are deploying the stack, specify the AMI ID for that region.
     * @param props Customize the image by supplying additional props.
     * @stability stable
     */
    static genericWindows(amiMap, props) {
        return new GenericWindowsImage(amiMap, props);
    }
    /**
     * (deprecated) An image specified in SSM parameter store that is automatically kept up-to-date.
     *
     * This Machine Image automatically updates to the latest version on every
     * deployment. Be aware this will cause your instances to be replaced when a
     * new version of the image becomes available. Do not store stateful information
     * on the instance if you are using this image.
     *
     * @param parameterName The name of SSM parameter containing the AMi id.
     * @param os The operating system type of the AMI.
     * @param userData optional user data for the given image.
     * @deprecated Use `MachineImage.fromSsmParameter()` instead
     */
    static fromSSMParameter(parameterName, os, userData) {
        return new GenericSSMParameterImage(parameterName, os, userData);
    }
    /**
     * An image specified in SSM parameter store.
     *
     * By default, the SSM parameter is refreshed at every deployment,
     * causing your instances to be replaced whenever a new version of the AMI
     * is released.
     *
     * Pass `{ cachedInContext: true }` to keep the AMI ID stable. If you do, you
     * will have to remember to periodically invalidate the context to refresh
     * to the newest AMI ID.
     *
     * @stability stable
     */
    static fromSsmParameter(parameterName, options) {
        return new GenericSsmParameterImage(parameterName, options);
    }
    /**
     * Look up a shared Machine Image using DescribeImages.
     *
     * The most recent, available, launchable image matching the given filter
     * criteria will be used. Looking up AMIs may take a long time; specify
     * as many filter criteria as possible to narrow down the search.
     *
     * The AMI selected will be cached in `cdk.context.json` and the same value
     * will be used on future runs. To refresh the AMI lookup, you will have to
     * evict the value from the cache using the `cdk context` command. See
     * https://docs.aws.amazon.com/cdk/latest/guide/context.html for more information.
     *
     * This function can not be used in environment-agnostic stacks.
     *
     * @stability stable
     */
    static lookup(props) {
        return new LookupMachineImage(props);
    }
}
exports.MachineImage = MachineImage;
_a = JSII_RTTI_SYMBOL_1;
MachineImage[_a] = { fqn: "@aws-cdk/aws-ec2.MachineImage", version: "1.129.0" };
/**
 * Select the image based on a given SSM parameter.
 *
 * This Machine Image automatically updates to the latest version on every
 * deployment. Be aware this will cause your instances to be replaced when a
 * new version of the image becomes available. Do not store stateful information
 * on the instance if you are using this image.
 *
 * The AMI ID is selected using the values published to the SSM parameter store.
 *
 * @stability stable
 */
class GenericSSMParameterImage {
    /**
     * @stability stable
     */
    constructor(parameterName, os, userData) {
        this.os = os;
        this.userData = userData;
        this.parameterName = parameterName;
    }
    /**
     * Return the image to use in the given context.
     *
     * @stability stable
     */
    getImage(scope) {
        var _h;
        const ami = ssm.StringParameter.valueForTypedStringParameter(scope, this.parameterName, ssm.ParameterType.AWS_EC2_IMAGE_ID);
        return {
            imageId: ami,
            osType: this.os,
            userData: (_h = this.userData) !== null && _h !== void 0 ? _h : (this.os === OperatingSystemType.WINDOWS ? user_data_1.UserData.forWindows() : user_data_1.UserData.forLinux()),
        };
    }
}
exports.GenericSSMParameterImage = GenericSSMParameterImage;
_b = JSII_RTTI_SYMBOL_1;
GenericSSMParameterImage[_b] = { fqn: "@aws-cdk/aws-ec2.GenericSSMParameterImage", version: "1.129.0" };
/**
 * Select the image based on a given SSM parameter
 *
 * This Machine Image automatically updates to the latest version on every
 * deployment. Be aware this will cause your instances to be replaced when a
 * new version of the image becomes available. Do not store stateful information
 * on the instance if you are using this image.
 *
 * The AMI ID is selected using the values published to the SSM parameter store.
 */
class GenericSsmParameterImage {
    constructor(parameterName, props = {}) {
        this.parameterName = parameterName;
        this.props = props;
    }
    /**
     * Return the image to use in the given context
     */
    getImage(scope) {
        var _h, _j;
        const imageId = lookupImage(scope, this.props.cachedInContext, this.parameterName);
        const osType = (_h = this.props.os) !== null && _h !== void 0 ? _h : OperatingSystemType.LINUX;
        return {
            imageId,
            osType,
            userData: (_j = this.props.userData) !== null && _j !== void 0 ? _j : (osType === OperatingSystemType.WINDOWS ? user_data_1.UserData.forWindows() : user_data_1.UserData.forLinux()),
        };
    }
}
/**
 * Select the latest version of the indicated Windows version.
 *
 * This Machine Image automatically updates to the latest version on every
 * deployment. Be aware this will cause your instances to be replaced when a
 * new version of the image becomes available. Do not store stateful information
 * on the instance if you are using this image.
 *
 * The AMI ID is selected using the values published to the SSM parameter store.
 *
 * https://aws.amazon.com/blogs/mt/query-for-the-latest-windows-ami-using-systems-manager-parameter-store/
 *
 * @stability stable
 */
class WindowsImage extends GenericSSMParameterImage {
    /**
     * @stability stable
     */
    constructor(version, props = {}) {
        super('/aws/service/ami-windows-latest/' + version, OperatingSystemType.WINDOWS, props.userData);
    }
}
exports.WindowsImage = WindowsImage;
_c = JSII_RTTI_SYMBOL_1;
WindowsImage[_c] = { fqn: "@aws-cdk/aws-ec2.WindowsImage", version: "1.129.0" };
/**
 * CPU type.
 *
 * @stability stable
 */
var AmazonLinuxCpuType;
(function (AmazonLinuxCpuType) {
    AmazonLinuxCpuType["ARM_64"] = "arm64";
    AmazonLinuxCpuType["X86_64"] = "x86_64";
})(AmazonLinuxCpuType = exports.AmazonLinuxCpuType || (exports.AmazonLinuxCpuType = {}));
/**
 * Selects the latest version of Amazon Linux.
 *
 * This Machine Image automatically updates to the latest version on every
 * deployment. Be aware this will cause your instances to be replaced when a
 * new version of the image becomes available. Do not store stateful information
 * on the instance if you are using this image.
 *
 * The AMI ID is selected using the values published to the SSM parameter store.
 *
 * @stability stable
 */
class AmazonLinuxImage extends GenericSSMParameterImage {
    /**
     * @stability stable
     */
    constructor(props = {}) {
        var _h;
        super(AmazonLinuxImage.ssmParameterName(props), OperatingSystemType.LINUX, props.userData);
        this.props = props;
        this.cachedInContext = (_h = props.cachedInContext) !== null && _h !== void 0 ? _h : false;
    }
    /**
     * Return the SSM parameter name that will contain the Amazon Linux image with the given attributes.
     *
     * @stability stable
     */
    static ssmParameterName(props = {}) {
        const generation = (props && props.generation) || AmazonLinuxGeneration.AMAZON_LINUX;
        const edition = (props && props.edition) || AmazonLinuxEdition.STANDARD;
        const virtualization = (props && props.virtualization) || AmazonLinuxVirt.HVM;
        const storage = (props && props.storage) || AmazonLinuxStorage.GENERAL_PURPOSE;
        const cpu = (props && props.cpuType) || AmazonLinuxCpuType.X86_64;
        const parts = [
            generation,
            'ami',
            edition !== AmazonLinuxEdition.STANDARD ? edition : undefined,
            virtualization,
            cpu,
            storage,
        ].filter(x => x !== undefined); // Get rid of undefineds
        return '/aws/service/ami-amazon-linux-latest/' + parts.join('-');
    }
    /**
     * Return the image to use in the given context.
     *
     * @stability stable
     */
    getImage(scope) {
        var _h;
        const imageId = lookupImage(scope, this.cachedInContext, this.parameterName);
        const osType = OperatingSystemType.LINUX;
        return {
            imageId,
            osType,
            userData: (_h = this.props.userData) !== null && _h !== void 0 ? _h : user_data_1.UserData.forLinux(),
        };
    }
}
exports.AmazonLinuxImage = AmazonLinuxImage;
_d = JSII_RTTI_SYMBOL_1;
AmazonLinuxImage[_d] = { fqn: "@aws-cdk/aws-ec2.AmazonLinuxImage", version: "1.129.0" };
/**
 * What generation of Amazon Linux to use.
 *
 * @stability stable
 */
var AmazonLinuxGeneration;
(function (AmazonLinuxGeneration) {
    AmazonLinuxGeneration["AMAZON_LINUX"] = "amzn";
    AmazonLinuxGeneration["AMAZON_LINUX_2"] = "amzn2";
})(AmazonLinuxGeneration = exports.AmazonLinuxGeneration || (exports.AmazonLinuxGeneration = {}));
/**
 * Amazon Linux edition.
 *
 * @stability stable
 */
var AmazonLinuxEdition;
(function (AmazonLinuxEdition) {
    AmazonLinuxEdition["STANDARD"] = "standard";
    AmazonLinuxEdition["MINIMAL"] = "minimal";
})(AmazonLinuxEdition = exports.AmazonLinuxEdition || (exports.AmazonLinuxEdition = {}));
/**
 * Virtualization type for Amazon Linux.
 *
 * @stability stable
 */
var AmazonLinuxVirt;
(function (AmazonLinuxVirt) {
    AmazonLinuxVirt["HVM"] = "hvm";
    AmazonLinuxVirt["PV"] = "pv";
})(AmazonLinuxVirt = exports.AmazonLinuxVirt || (exports.AmazonLinuxVirt = {}));
/**
 * @stability stable
 */
var AmazonLinuxStorage;
(function (AmazonLinuxStorage) {
    AmazonLinuxStorage["EBS"] = "ebs";
    /**
     * S3-backed storage
     */
    AmazonLinuxStorage["S3"] = "ebs";
    AmazonLinuxStorage["GENERAL_PURPOSE"] = "gp2";
})(AmazonLinuxStorage = exports.AmazonLinuxStorage || (exports.AmazonLinuxStorage = {}));
/**
 * Construct a Linux machine image from an AMI map.
 *
 * Linux images IDs are not published to SSM parameter store yet, so you'll have to
 * manually specify an AMI map.
 *
 * @stability stable
 */
class GenericLinuxImage {
    /**
     * @stability stable
     */
    constructor(amiMap, props = {}) {
        this.amiMap = amiMap;
        this.props = props;
    }
    /**
     * Return the image to use in the given context.
     *
     * @stability stable
     */
    getImage(scope) {
        var _h;
        const userData = (_h = this.props.userData) !== null && _h !== void 0 ? _h : user_data_1.UserData.forLinux();
        const osType = OperatingSystemType.LINUX;
        const region = core_1.Stack.of(scope).region;
        if (core_1.Token.isUnresolved(region)) {
            const mapping = {};
            for (const [rgn, ami] of Object.entries(this.amiMap)) {
                mapping[rgn] = { ami };
            }
            const amiMap = new core_1.CfnMapping(scope, 'AmiMap', { mapping });
            return {
                imageId: amiMap.findInMap(core_1.Aws.REGION, 'ami'),
                userData,
                osType,
            };
        }
        const imageId = region !== 'test-region' ? this.amiMap[region] : 'ami-12345';
        if (!imageId) {
            throw new Error(`Unable to find AMI in AMI map: no AMI specified for region '${region}'`);
        }
        return {
            imageId,
            userData,
            osType,
        };
    }
}
exports.GenericLinuxImage = GenericLinuxImage;
_e = JSII_RTTI_SYMBOL_1;
GenericLinuxImage[_e] = { fqn: "@aws-cdk/aws-ec2.GenericLinuxImage", version: "1.129.0" };
/**
 * Construct a Windows machine image from an AMI map.
 *
 * Allows you to create a generic Windows EC2 , manually specify an AMI map.
 *
 * @stability stable
 */
class GenericWindowsImage {
    /**
     * @stability stable
     */
    constructor(amiMap, props = {}) {
        this.amiMap = amiMap;
        this.props = props;
    }
    /**
     * Return the image to use in the given context.
     *
     * @stability stable
     */
    getImage(scope) {
        var _h;
        const userData = (_h = this.props.userData) !== null && _h !== void 0 ? _h : user_data_1.UserData.forWindows();
        const osType = OperatingSystemType.WINDOWS;
        const region = core_1.Stack.of(scope).region;
        if (core_1.Token.isUnresolved(region)) {
            const mapping = {};
            for (const [rgn, ami] of Object.entries(this.amiMap)) {
                mapping[rgn] = { ami };
            }
            const amiMap = new core_1.CfnMapping(scope, 'AmiMap', { mapping });
            return {
                imageId: amiMap.findInMap(core_1.Aws.REGION, 'ami'),
                userData,
                osType,
            };
        }
        const imageId = region !== 'test-region' ? this.amiMap[region] : 'ami-12345';
        if (!imageId) {
            throw new Error(`Unable to find AMI in AMI map: no AMI specified for region '${region}'`);
        }
        return {
            imageId,
            userData,
            osType,
        };
    }
}
exports.GenericWindowsImage = GenericWindowsImage;
_f = JSII_RTTI_SYMBOL_1;
GenericWindowsImage[_f] = { fqn: "@aws-cdk/aws-ec2.GenericWindowsImage", version: "1.129.0" };
/**
 * The OS type of a particular image.
 *
 * @stability stable
 */
var OperatingSystemType;
(function (OperatingSystemType) {
    OperatingSystemType[OperatingSystemType["LINUX"] = 0] = "LINUX";
    OperatingSystemType[OperatingSystemType["WINDOWS"] = 1] = "WINDOWS";
    OperatingSystemType[OperatingSystemType["UNKNOWN"] = 2] = "UNKNOWN";
})(OperatingSystemType = exports.OperatingSystemType || (exports.OperatingSystemType = {}));
/**
 * A machine image whose AMI ID will be searched using DescribeImages.
 *
 * The most recent, available, launchable image matching the given filter
 * criteria will be used. Looking up AMIs may take a long time; specify
 * as many filter criteria as possible to narrow down the search.
 *
 * The AMI selected will be cached in `cdk.context.json` and the same value
 * will be used on future runs. To refresh the AMI lookup, you will have to
 * evict the value from the cache using the `cdk context` command. See
 * https://docs.aws.amazon.com/cdk/latest/guide/context.html for more information.
 *
 * @stability stable
 */
class LookupMachineImage {
    /**
     * @stability stable
     */
    constructor(props) {
        this.props = props;
    }
    /**
     * Return the image to use in the given context.
     *
     * @stability stable
     */
    getImage(scope) {
        var _h;
        // Need to know 'windows' or not before doing the query to return the right
        // osType for the dummy value, so might as well add it to the filter.
        const filters = {
            'name': [this.props.name],
            'state': ['available'],
            'image-type': ['machine'],
            'platform': this.props.windows ? ['windows'] : undefined,
        };
        Object.assign(filters, this.props.filters);
        const value = core_1.ContextProvider.getValue(scope, {
            provider: cxschema.ContextProvider.AMI_PROVIDER,
            props: {
                owners: this.props.owners,
                filters,
            },
            dummyValue: 'ami-1234',
        }).value;
        if (typeof value !== 'string') {
            throw new Error(`Response to AMI lookup invalid, got: ${value}`);
        }
        const osType = this.props.windows ? OperatingSystemType.WINDOWS : OperatingSystemType.LINUX;
        return {
            imageId: value,
            osType,
            userData: (_h = this.props.userData) !== null && _h !== void 0 ? _h : user_data_1.UserData.forOperatingSystem(osType),
        };
    }
}
exports.LookupMachineImage = LookupMachineImage;
_g = JSII_RTTI_SYMBOL_1;
LookupMachineImage[_g] = { fqn: "@aws-cdk/aws-ec2.LookupMachineImage", version: "1.129.0" };
function lookupImage(scope, cachedInContext, parameterName) {
    return cachedInContext
        ? ssm.StringParameter.valueFromLookup(scope, parameterName)
        : ssm.StringParameter.valueForTypedStringParameter(scope, parameterName, ssm.ParameterType.AWS_EC2_IMAGE_ID);
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoibWFjaGluZS1pbWFnZS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIm1hY2hpbmUtaW1hZ2UudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7QUFBQSx3Q0FBd0M7QUFDeEMsMkRBQTJEO0FBQzNELHdDQUErRTtBQUUvRSwyQ0FBdUM7Ozs7OztBQWN2QyxNQUFzQixZQUFZOzs7Ozs7Ozs7OztJQUV6QixNQUFNLENBQUMsYUFBYSxDQUFDLE9BQXVCLEVBQUUsS0FBeUI7UUFDNUUsT0FBTyxJQUFJLFlBQVksQ0FBQyxPQUFPLEVBQUUsS0FBSyxDQUFDLENBQUM7SUFDMUMsQ0FBQzs7Ozs7Ozs7Ozs7SUFHTSxNQUFNLENBQUMsaUJBQWlCLENBQUMsS0FBNkI7UUFDM0QsT0FBTyxJQUFJLGdCQUFnQixDQUFDLEtBQUssQ0FBQyxDQUFDO0lBQ3JDLENBQUM7Ozs7Ozs7O0lBR00sTUFBTSxDQUFDLFlBQVksQ0FBQyxNQUE4QixFQUFFLEtBQThCO1FBQ3ZGLE9BQU8sSUFBSSxpQkFBaUIsQ0FBQyxNQUFNLEVBQUUsS0FBSyxDQUFDLENBQUM7SUFDOUMsQ0FBQzs7Ozs7Ozs7SUFHTSxNQUFNLENBQUMsY0FBYyxDQUFDLE1BQThCLEVBQUUsS0FBZ0M7UUFDM0YsT0FBTyxJQUFJLG1CQUFtQixDQUFDLE1BQU0sRUFBRSxLQUFLLENBQUMsQ0FBQztJQUNoRCxDQUFDOzs7Ozs7Ozs7Ozs7OztJQUdNLE1BQU0sQ0FBQyxnQkFBZ0IsQ0FBQyxhQUFxQixFQUFFLEVBQXVCLEVBQUUsUUFBbUI7UUFDaEcsT0FBTyxJQUFJLHdCQUF3QixDQUFDLGFBQWEsRUFBRSxFQUFFLEVBQUUsUUFBUSxDQUFDLENBQUM7SUFDbkUsQ0FBQzs7Ozs7Ozs7Ozs7Ozs7SUFHTSxNQUFNLENBQUMsZ0JBQWdCLENBQUMsYUFBcUIsRUFBRSxPQUFrQztRQUN0RixPQUFPLElBQUksd0JBQXdCLENBQUMsYUFBYSxFQUFFLE9BQU8sQ0FBQyxDQUFDO0lBQzlELENBQUM7Ozs7Ozs7Ozs7Ozs7Ozs7O0lBR00sTUFBTSxDQUFDLE1BQU0sQ0FBQyxLQUE4QjtRQUNqRCxPQUFPLElBQUksa0JBQWtCLENBQUMsS0FBSyxDQUFDLENBQUM7SUFDdkMsQ0FBQzs7QUFsQ0gsb0NBbUNDOzs7Ozs7Ozs7Ozs7Ozs7QUFlRCxNQUFhLHdCQUF3Qjs7OztJQVFuQyxZQUFZLGFBQXFCLEVBQW1CLEVBQXVCLEVBQW1CLFFBQW1CO1FBQTdELE9BQUUsR0FBRixFQUFFLENBQXFCO1FBQW1CLGFBQVEsR0FBUixRQUFRLENBQVc7UUFDL0csSUFBSSxDQUFDLGFBQWEsR0FBRyxhQUFhLENBQUM7SUFDckMsQ0FBQzs7Ozs7O0lBR00sUUFBUSxDQUFDLEtBQWdCOztRQUM5QixNQUFNLEdBQUcsR0FBRyxHQUFHLENBQUMsZUFBZSxDQUFDLDRCQUE0QixDQUFDLEtBQUssRUFBRSxJQUFJLENBQUMsYUFBYSxFQUFFLEdBQUcsQ0FBQyxhQUFhLENBQUMsZ0JBQWdCLENBQUMsQ0FBQztRQUM1SCxPQUFPO1lBQ0wsT0FBTyxFQUFFLEdBQUc7WUFDWixNQUFNLEVBQUUsSUFBSSxDQUFDLEVBQUU7WUFDZixRQUFRLFFBQUUsSUFBSSxDQUFDLFFBQVEsbUNBQUksQ0FBQyxJQUFJLENBQUMsRUFBRSxLQUFLLG1CQUFtQixDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsb0JBQVEsQ0FBQyxVQUFVLEVBQUUsQ0FBQyxDQUFDLENBQUMsb0JBQVEsQ0FBQyxRQUFRLEVBQUUsQ0FBQztTQUNuSCxDQUFDO0lBQ0osQ0FBQzs7QUFwQkgsNERBcUJDOzs7QUFjRDs7Ozs7Ozs7O0dBU0c7QUFDSCxNQUFNLHdCQUF3QjtJQUM1QixZQUE2QixhQUFxQixFQUFtQixRQUFrQyxFQUFFO1FBQTVFLGtCQUFhLEdBQWIsYUFBYSxDQUFRO1FBQW1CLFVBQUssR0FBTCxLQUFLLENBQStCO0lBQ3pHLENBQUM7SUFFRDs7T0FFRztJQUNJLFFBQVEsQ0FBQyxLQUFnQjs7UUFDOUIsTUFBTSxPQUFPLEdBQUcsV0FBVyxDQUFDLEtBQUssRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLGVBQWUsRUFBRSxJQUFJLENBQUMsYUFBYSxDQUFDLENBQUM7UUFFbkYsTUFBTSxNQUFNLFNBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxFQUFFLG1DQUFJLG1CQUFtQixDQUFDLEtBQUssQ0FBQztRQUMxRCxPQUFPO1lBQ0wsT0FBTztZQUNQLE1BQU07WUFDTixRQUFRLFFBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxRQUFRLG1DQUFJLENBQUMsTUFBTSxLQUFLLG1CQUFtQixDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsb0JBQVEsQ0FBQyxVQUFVLEVBQUUsQ0FBQyxDQUFDLENBQUMsb0JBQVEsQ0FBQyxRQUFRLEVBQUUsQ0FBQztTQUN4SCxDQUFDO0lBQ0osQ0FBQztDQUNGOzs7Ozs7Ozs7Ozs7Ozs7QUFTRCxNQUFhLFlBQWEsU0FBUSx3QkFBd0I7Ozs7SUFDeEQsWUFBWSxPQUF1QixFQUFFLFFBQTJCLEVBQUU7UUFDaEUsS0FBSyxDQUFDLGtDQUFrQyxHQUFHLE9BQU8sRUFBRSxtQkFBbUIsQ0FBQyxPQUFPLEVBQUUsS0FBSyxDQUFDLFFBQVEsQ0FBQyxDQUFDO0lBQ25HLENBQUM7O0FBSEgsb0NBSUM7Ozs7Ozs7O0FBR0QsSUFBWSxrQkFNWDtBQU5ELFdBQVksa0JBQWtCO0lBRTVCLHNDQUFnQixDQUFBO0lBR2hCLHVDQUFpQixDQUFBO0FBQ25CLENBQUMsRUFOVyxrQkFBa0IsR0FBbEIsMEJBQWtCLEtBQWxCLDBCQUFrQixRQU03Qjs7Ozs7Ozs7Ozs7OztBQTJCRCxNQUFhLGdCQUFpQixTQUFRLHdCQUF3Qjs7OztJQXNCNUQsWUFBNkIsUUFBK0IsRUFBRTs7UUFDNUQsS0FBSyxDQUFDLGdCQUFnQixDQUFDLGdCQUFnQixDQUFDLEtBQUssQ0FBQyxFQUFFLG1CQUFtQixDQUFDLEtBQUssRUFBRSxLQUFLLENBQUMsUUFBUSxDQUFDLENBQUM7UUFEaEUsVUFBSyxHQUFMLEtBQUssQ0FBNEI7UUFHNUQsSUFBSSxDQUFDLGVBQWUsU0FBRyxLQUFLLENBQUMsZUFBZSxtQ0FBSSxLQUFLLENBQUM7SUFDeEQsQ0FBQzs7Ozs7O0lBeEJNLE1BQU0sQ0FBQyxnQkFBZ0IsQ0FBQyxRQUErQixFQUFFO1FBQzlELE1BQU0sVUFBVSxHQUFHLENBQUMsS0FBSyxJQUFJLEtBQUssQ0FBQyxVQUFVLENBQUMsSUFBSSxxQkFBcUIsQ0FBQyxZQUFZLENBQUM7UUFDckYsTUFBTSxPQUFPLEdBQUcsQ0FBQyxLQUFLLElBQUksS0FBSyxDQUFDLE9BQU8sQ0FBQyxJQUFJLGtCQUFrQixDQUFDLFFBQVEsQ0FBQztRQUN4RSxNQUFNLGNBQWMsR0FBRyxDQUFDLEtBQUssSUFBSSxLQUFLLENBQUMsY0FBYyxDQUFDLElBQUksZUFBZSxDQUFDLEdBQUcsQ0FBQztRQUM5RSxNQUFNLE9BQU8sR0FBRyxDQUFDLEtBQUssSUFBSSxLQUFLLENBQUMsT0FBTyxDQUFDLElBQUksa0JBQWtCLENBQUMsZUFBZSxDQUFDO1FBQy9FLE1BQU0sR0FBRyxHQUFHLENBQUMsS0FBSyxJQUFJLEtBQUssQ0FBQyxPQUFPLENBQUMsSUFBSSxrQkFBa0IsQ0FBQyxNQUFNLENBQUM7UUFDbEUsTUFBTSxLQUFLLEdBQTRCO1lBQ3JDLFVBQVU7WUFDVixLQUFLO1lBQ0wsT0FBTyxLQUFLLGtCQUFrQixDQUFDLFFBQVEsQ0FBQyxDQUFDLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxTQUFTO1lBQzdELGNBQWM7WUFDZCxHQUFHO1lBQ0gsT0FBTztTQUNSLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxLQUFLLFNBQVMsQ0FBQyxDQUFDLENBQUMsd0JBQXdCO1FBRXhELE9BQU8sdUNBQXVDLEdBQUcsS0FBSyxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUNuRSxDQUFDOzs7Ozs7SUFXTSxRQUFRLENBQUMsS0FBZ0I7O1FBQzlCLE1BQU0sT0FBTyxHQUFHLFdBQVcsQ0FBQyxLQUFLLEVBQUUsSUFBSSxDQUFDLGVBQWUsRUFBRSxJQUFJLENBQUMsYUFBYSxDQUFDLENBQUM7UUFFN0UsTUFBTSxNQUFNLEdBQUcsbUJBQW1CLENBQUMsS0FBSyxDQUFDO1FBQ3pDLE9BQU87WUFDTCxPQUFPO1lBQ1AsTUFBTTtZQUNOLFFBQVEsUUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLFFBQVEsbUNBQUksb0JBQVEsQ0FBQyxRQUFRLEVBQUU7U0FDckQsQ0FBQztJQUNKLENBQUM7O0FBdENILDRDQXVDQzs7Ozs7Ozs7QUFHRCxJQUFZLHFCQU1YO0FBTkQsV0FBWSxxQkFBcUI7SUFFL0IsOENBQXFCLENBQUE7SUFHckIsaURBQXdCLENBQUE7QUFDMUIsQ0FBQyxFQU5XLHFCQUFxQixHQUFyQiw2QkFBcUIsS0FBckIsNkJBQXFCLFFBTWhDOzs7Ozs7QUFHRCxJQUFZLGtCQU1YO0FBTkQsV0FBWSxrQkFBa0I7SUFFNUIsMkNBQXFCLENBQUE7SUFHckIseUNBQW1CLENBQUE7QUFDckIsQ0FBQyxFQU5XLGtCQUFrQixHQUFsQiwwQkFBa0IsS0FBbEIsMEJBQWtCLFFBTTdCOzs7Ozs7QUFHRCxJQUFZLGVBTVg7QUFORCxXQUFZLGVBQWU7SUFFekIsOEJBQVcsQ0FBQTtJQUdYLDRCQUFTLENBQUE7QUFDWCxDQUFDLEVBTlcsZUFBZSxHQUFmLHVCQUFlLEtBQWYsdUJBQWUsUUFNMUI7Ozs7QUFFRCxJQUFZLGtCQVdYO0FBWEQsV0FBWSxrQkFBa0I7SUFFNUIsaUNBQVcsQ0FBQTtJQUVYOztPQUVHO0lBQ0gsZ0NBQVUsQ0FBQTtJQUdWLDZDQUF1QixDQUFBO0FBQ3pCLENBQUMsRUFYVyxrQkFBa0IsR0FBbEIsMEJBQWtCLEtBQWxCLDBCQUFrQixRQVc3Qjs7Ozs7Ozs7O0FBZUQsTUFBYSxpQkFBaUI7Ozs7SUFDNUIsWUFBNkIsTUFBb0MsRUFBbUIsUUFBZ0MsRUFBRTtRQUF6RixXQUFNLEdBQU4sTUFBTSxDQUE4QjtRQUFtQixVQUFLLEdBQUwsS0FBSyxDQUE2QjtJQUN0SCxDQUFDOzs7Ozs7SUFFTSxRQUFRLENBQUMsS0FBZ0I7O1FBQzlCLE1BQU0sUUFBUSxTQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsUUFBUSxtQ0FBSSxvQkFBUSxDQUFDLFFBQVEsRUFBRSxDQUFDO1FBQzVELE1BQU0sTUFBTSxHQUFHLG1CQUFtQixDQUFDLEtBQUssQ0FBQztRQUN6QyxNQUFNLE1BQU0sR0FBRyxZQUFLLENBQUMsRUFBRSxDQUFDLEtBQUssQ0FBQyxDQUFDLE1BQU0sQ0FBQztRQUN0QyxJQUFJLFlBQUssQ0FBQyxZQUFZLENBQUMsTUFBTSxDQUFDLEVBQUU7WUFDOUIsTUFBTSxPQUFPLEdBQTRDLEVBQUUsQ0FBQztZQUM1RCxLQUFLLE1BQU0sQ0FBQyxHQUFHLEVBQUUsR0FBRyxDQUFDLElBQUksTUFBTSxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLEVBQUU7Z0JBQ3BELE9BQU8sQ0FBQyxHQUFHLENBQUMsR0FBRyxFQUFFLEdBQUcsRUFBRSxDQUFDO2FBQ3hCO1lBQ0QsTUFBTSxNQUFNLEdBQUcsSUFBSSxpQkFBVSxDQUFDLEtBQUssRUFBRSxRQUFRLEVBQUUsRUFBRSxPQUFPLEVBQUUsQ0FBQyxDQUFDO1lBQzVELE9BQU87Z0JBQ0wsT0FBTyxFQUFFLE1BQU0sQ0FBQyxTQUFTLENBQUMsVUFBRyxDQUFDLE1BQU0sRUFBRSxLQUFLLENBQUM7Z0JBQzVDLFFBQVE7Z0JBQ1IsTUFBTTthQUNQLENBQUM7U0FDSDtRQUNELE1BQU0sT0FBTyxHQUFHLE1BQU0sS0FBSyxhQUFhLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxDQUFDLFdBQVcsQ0FBQztRQUM3RSxJQUFJLENBQUMsT0FBTyxFQUFFO1lBQ1osTUFBTSxJQUFJLEtBQUssQ0FBQywrREFBK0QsTUFBTSxHQUFHLENBQUMsQ0FBQztTQUMzRjtRQUNELE9BQU87WUFDTCxPQUFPO1lBQ1AsUUFBUTtZQUNSLE1BQU07U0FDUCxDQUFDO0lBQ0osQ0FBQzs7QUE3QkgsOENBOEJDOzs7Ozs7Ozs7O0FBR0QsTUFBYSxtQkFBbUI7Ozs7SUFDOUIsWUFBNkIsTUFBa0MsRUFBbUIsUUFBa0MsRUFBRTtRQUF6RixXQUFNLEdBQU4sTUFBTSxDQUE0QjtRQUFtQixVQUFLLEdBQUwsS0FBSyxDQUErQjtJQUN0SCxDQUFDOzs7Ozs7SUFFTSxRQUFRLENBQUMsS0FBZ0I7O1FBQzlCLE1BQU0sUUFBUSxTQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsUUFBUSxtQ0FBSSxvQkFBUSxDQUFDLFVBQVUsRUFBRSxDQUFDO1FBQzlELE1BQU0sTUFBTSxHQUFHLG1CQUFtQixDQUFDLE9BQU8sQ0FBQztRQUMzQyxNQUFNLE1BQU0sR0FBRyxZQUFLLENBQUMsRUFBRSxDQUFDLEtBQUssQ0FBQyxDQUFDLE1BQU0sQ0FBQztRQUN0QyxJQUFJLFlBQUssQ0FBQyxZQUFZLENBQUMsTUFBTSxDQUFDLEVBQUU7WUFDOUIsTUFBTSxPQUFPLEdBQTRDLEVBQUUsQ0FBQztZQUM1RCxLQUFLLE1BQU0sQ0FBQyxHQUFHLEVBQUUsR0FBRyxDQUFDLElBQUksTUFBTSxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLEVBQUU7Z0JBQ3BELE9BQU8sQ0FBQyxHQUFHLENBQUMsR0FBRyxFQUFFLEdBQUcsRUFBRSxDQUFDO2FBQ3hCO1lBQ0QsTUFBTSxNQUFNLEdBQUcsSUFBSSxpQkFBVSxDQUFDLEtBQUssRUFBRSxRQUFRLEVBQUUsRUFBRSxPQUFPLEVBQUUsQ0FBQyxDQUFDO1lBQzVELE9BQU87Z0JBQ0wsT0FBTyxFQUFFLE1BQU0sQ0FBQyxTQUFTLENBQUMsVUFBRyxDQUFDLE1BQU0sRUFBRSxLQUFLLENBQUM7Z0JBQzVDLFFBQVE7Z0JBQ1IsTUFBTTthQUNQLENBQUM7U0FDSDtRQUNELE1BQU0sT0FBTyxHQUFHLE1BQU0sS0FBSyxhQUFhLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxDQUFDLFdBQVcsQ0FBQztRQUM3RSxJQUFJLENBQUMsT0FBTyxFQUFFO1lBQ1osTUFBTSxJQUFJLEtBQUssQ0FBQywrREFBK0QsTUFBTSxHQUFHLENBQUMsQ0FBQztTQUMzRjtRQUNELE9BQU87WUFDTCxPQUFPO1lBQ1AsUUFBUTtZQUNSLE1BQU07U0FDUCxDQUFDO0lBQ0osQ0FBQzs7QUE3Qkgsa0RBOEJDOzs7Ozs7OztBQUdELElBQVksbUJBS1g7QUFMRCxXQUFZLG1CQUFtQjtJQUM3QiwrREFBSyxDQUFBO0lBQ0wsbUVBQU8sQ0FBQTtJQUVQLG1FQUFPLENBQUE7QUFDVCxDQUFDLEVBTFcsbUJBQW1CLEdBQW5CLDJCQUFtQixLQUFuQiwyQkFBbUIsUUFLOUI7Ozs7Ozs7Ozs7Ozs7OztBQUdELE1BQWEsa0JBQWtCOzs7O0lBQzdCLFlBQTZCLEtBQThCO1FBQTlCLFVBQUssR0FBTCxLQUFLLENBQXlCO0lBQzNELENBQUM7Ozs7OztJQUVNLFFBQVEsQ0FBQyxLQUFnQjs7UUFDOUIsMkVBQTJFO1FBQzNFLHFFQUFxRTtRQUNyRSxNQUFNLE9BQU8sR0FBeUM7WUFDcEQsTUFBTSxFQUFFLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUM7WUFDekIsT0FBTyxFQUFFLENBQUMsV0FBVyxDQUFDO1lBQ3RCLFlBQVksRUFBRSxDQUFDLFNBQVMsQ0FBQztZQUN6QixVQUFVLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQyxDQUFDLFNBQVM7U0FDekQsQ0FBQztRQUNGLE1BQU0sQ0FBQyxNQUFNLENBQUMsT0FBTyxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLENBQUM7UUFFM0MsTUFBTSxLQUFLLEdBQUcsc0JBQWUsQ0FBQyxRQUFRLENBQUMsS0FBSyxFQUFFO1lBQzVDLFFBQVEsRUFBRSxRQUFRLENBQUMsZUFBZSxDQUFDLFlBQVk7WUFDL0MsS0FBSyxFQUFFO2dCQUNMLE1BQU0sRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLE1BQU07Z0JBQ3pCLE9BQU87YUFDb0I7WUFDN0IsVUFBVSxFQUFFLFVBQVU7U0FDdkIsQ0FBQyxDQUFDLEtBQWlDLENBQUM7UUFFckMsSUFBSSxPQUFPLEtBQUssS0FBSyxRQUFRLEVBQUU7WUFDN0IsTUFBTSxJQUFJLEtBQUssQ0FBQyx3Q0FBd0MsS0FBSyxFQUFFLENBQUMsQ0FBQztTQUNsRTtRQUVELE1BQU0sTUFBTSxHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxtQkFBbUIsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLG1CQUFtQixDQUFDLEtBQUssQ0FBQztRQUU1RixPQUFPO1lBQ0wsT0FBTyxFQUFFLEtBQUs7WUFDZCxNQUFNO1lBQ04sUUFBUSxRQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsUUFBUSxtQ0FBSSxvQkFBUSxDQUFDLGtCQUFrQixDQUFDLE1BQU0sQ0FBQztTQUNyRSxDQUFDO0lBQ0osQ0FBQzs7QUFuQ0gsZ0RBb0NDOzs7QUFvQkQsU0FBUyxXQUFXLENBQUMsS0FBZ0IsRUFBRSxlQUFvQyxFQUFFLGFBQXFCO0lBQ2hHLE9BQU8sZUFBZTtRQUNwQixDQUFDLENBQUMsR0FBRyxDQUFDLGVBQWUsQ0FBQyxlQUFlLENBQUMsS0FBSyxFQUFFLGFBQWEsQ0FBQztRQUMzRCxDQUFDLENBQUMsR0FBRyxDQUFDLGVBQWUsQ0FBQyw0QkFBNEIsQ0FBQyxLQUFLLEVBQUUsYUFBYSxFQUFFLEdBQUcsQ0FBQyxhQUFhLENBQUMsZ0JBQWdCLENBQUMsQ0FBQztBQUNqSCxDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0ICogYXMgc3NtIGZyb20gJ0Bhd3MtY2RrL2F3cy1zc20nO1xuaW1wb3J0ICogYXMgY3hzY2hlbWEgZnJvbSAnQGF3cy1jZGsvY2xvdWQtYXNzZW1ibHktc2NoZW1hJztcbmltcG9ydCB7IENvbnRleHRQcm92aWRlciwgQ2ZuTWFwcGluZywgQXdzLCBTdGFjaywgVG9rZW4gfSBmcm9tICdAYXdzLWNkay9jb3JlJztcbmltcG9ydCAqIGFzIGN4YXBpIGZyb20gJ0Bhd3MtY2RrL2N4LWFwaSc7XG5pbXBvcnQgeyBVc2VyRGF0YSB9IGZyb20gJy4vdXNlci1kYXRhJztcbmltcG9ydCB7IFdpbmRvd3NWZXJzaW9uIH0gZnJvbSAnLi93aW5kb3dzLXZlcnNpb25zJztcblxuLy8ga2VlcCB0aGlzIGltcG9ydCBzZXBhcmF0ZSBmcm9tIG90aGVyIGltcG9ydHMgdG8gcmVkdWNlIGNoYW5jZSBmb3IgbWVyZ2UgY29uZmxpY3RzIHdpdGggdjItbWFpblxuLy8gZXNsaW50LWRpc2FibGUtbmV4dC1saW5lIG5vLWR1cGxpY2F0ZS1pbXBvcnRzLCBpbXBvcnQvb3JkZXJcbmltcG9ydCB7IENvbnN0cnVjdCB9IGZyb20gJ0Bhd3MtY2RrL2NvcmUnO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBpbnRlcmZhY2UgSU1hY2hpbmVJbWFnZSB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIGdldEltYWdlKHNjb3BlOiBDb25zdHJ1Y3QpOiBNYWNoaW5lSW1hZ2VDb25maWc7XG59XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgYWJzdHJhY3QgY2xhc3MgTWFjaGluZUltYWdlIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHN0YXRpYyBsYXRlc3RXaW5kb3dzKHZlcnNpb246IFdpbmRvd3NWZXJzaW9uLCBwcm9wcz86IFdpbmRvd3NJbWFnZVByb3BzKTogSU1hY2hpbmVJbWFnZSB7XG4gICAgcmV0dXJuIG5ldyBXaW5kb3dzSW1hZ2UodmVyc2lvbiwgcHJvcHMpO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHN0YXRpYyBsYXRlc3RBbWF6b25MaW51eChwcm9wcz86IEFtYXpvbkxpbnV4SW1hZ2VQcm9wcyk6IElNYWNoaW5lSW1hZ2Uge1xuICAgIHJldHVybiBuZXcgQW1hem9uTGludXhJbWFnZShwcm9wcyk7XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgZ2VuZXJpY0xpbnV4KGFtaU1hcDogUmVjb3JkPHN0cmluZywgc3RyaW5nPiwgcHJvcHM/OiBHZW5lcmljTGludXhJbWFnZVByb3BzKTogSU1hY2hpbmVJbWFnZSB7XG4gICAgcmV0dXJuIG5ldyBHZW5lcmljTGludXhJbWFnZShhbWlNYXAsIHByb3BzKTtcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgc3RhdGljIGdlbmVyaWNXaW5kb3dzKGFtaU1hcDogUmVjb3JkPHN0cmluZywgc3RyaW5nPiwgcHJvcHM/OiBHZW5lcmljV2luZG93c0ltYWdlUHJvcHMpOiBJTWFjaGluZUltYWdlIHtcbiAgICByZXR1cm4gbmV3IEdlbmVyaWNXaW5kb3dzSW1hZ2UoYW1pTWFwLCBwcm9wcyk7XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHN0YXRpYyBmcm9tU1NNUGFyYW1ldGVyKHBhcmFtZXRlck5hbWU6IHN0cmluZywgb3M6IE9wZXJhdGluZ1N5c3RlbVR5cGUsIHVzZXJEYXRhPzogVXNlckRhdGEpOiBJTWFjaGluZUltYWdlIHtcbiAgICByZXR1cm4gbmV3IEdlbmVyaWNTU01QYXJhbWV0ZXJJbWFnZShwYXJhbWV0ZXJOYW1lLCBvcywgdXNlckRhdGEpO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgZnJvbVNzbVBhcmFtZXRlcihwYXJhbWV0ZXJOYW1lOiBzdHJpbmcsIG9wdGlvbnM/OiBTc21QYXJhbWV0ZXJJbWFnZU9wdGlvbnMpOiBJTWFjaGluZUltYWdlIHtcbiAgICByZXR1cm4gbmV3IEdlbmVyaWNTc21QYXJhbWV0ZXJJbWFnZShwYXJhbWV0ZXJOYW1lLCBvcHRpb25zKTtcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgbG9va3VwKHByb3BzOiBMb29rdXBNYWNoaW5lSW1hZ2VQcm9wcyk6IElNYWNoaW5lSW1hZ2Uge1xuICAgIHJldHVybiBuZXcgTG9va3VwTWFjaGluZUltYWdlKHByb3BzKTtcbiAgfVxufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGludGVyZmFjZSBNYWNoaW5lSW1hZ2VDb25maWcge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgaW1hZ2VJZDogc3RyaW5nO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBvc1R5cGU6IE9wZXJhdGluZ1N5c3RlbVR5cGU7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSB1c2VyRGF0YTogVXNlckRhdGE7XG59XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGNsYXNzIEdlbmVyaWNTU01QYXJhbWV0ZXJJbWFnZSBpbXBsZW1lbnRzIElNYWNoaW5lSW1hZ2Uge1xuICAvLyBGSVhNRTogdGhpcyBjbGFzcyBvdWdodCB0byBiZSBgQGRlcHJlY2F0ZWRgIGFuZCByZW1vdmVkIGZyb20gdjIsIGJ1dCB0aGF0XG4gIC8vIGlzIGNhdXNpbmcgYnVpbGQgZmFpbHVyZSByaWdodCBub3cuIFJlZjogaHR0cHM6Ly9naXRodWIuY29tL2F3cy9qc2lpL2lzc3Vlcy8zMDI1XG4gIC8vIEAtZGVwcmVjYXRlZCBVc2UgYE1hY2hpbmVJbWFnZS5mcm9tU3NtUGFyYW1ldGVyKClgIGluc3RlYWRcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHJlYWRvbmx5IHBhcmFtZXRlck5hbWU6IHN0cmluZztcblxuICBjb25zdHJ1Y3RvcihwYXJhbWV0ZXJOYW1lOiBzdHJpbmcsIHByaXZhdGUgcmVhZG9ubHkgb3M6IE9wZXJhdGluZ1N5c3RlbVR5cGUsIHByaXZhdGUgcmVhZG9ubHkgdXNlckRhdGE/OiBVc2VyRGF0YSkge1xuICAgIHRoaXMucGFyYW1ldGVyTmFtZSA9IHBhcmFtZXRlck5hbWU7XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgZ2V0SW1hZ2Uoc2NvcGU6IENvbnN0cnVjdCk6IE1hY2hpbmVJbWFnZUNvbmZpZyB7XG4gICAgY29uc3QgYW1pID0gc3NtLlN0cmluZ1BhcmFtZXRlci52YWx1ZUZvclR5cGVkU3RyaW5nUGFyYW1ldGVyKHNjb3BlLCB0aGlzLnBhcmFtZXRlck5hbWUsIHNzbS5QYXJhbWV0ZXJUeXBlLkFXU19FQzJfSU1BR0VfSUQpO1xuICAgIHJldHVybiB7XG4gICAgICBpbWFnZUlkOiBhbWksXG4gICAgICBvc1R5cGU6IHRoaXMub3MsXG4gICAgICB1c2VyRGF0YTogdGhpcy51c2VyRGF0YSA/PyAodGhpcy5vcyA9PT0gT3BlcmF0aW5nU3lzdGVtVHlwZS5XSU5ET1dTID8gVXNlckRhdGEuZm9yV2luZG93cygpIDogVXNlckRhdGEuZm9yTGludXgoKSksXG4gICAgfTtcbiAgfVxufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGludGVyZmFjZSBTc21QYXJhbWV0ZXJJbWFnZU9wdGlvbnMge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IG9zPzogT3BlcmF0aW5nU3lzdGVtVHlwZTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSB1c2VyRGF0YT86IFVzZXJEYXRhO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBjYWNoZWRJbkNvbnRleHQ/OiBib29sZWFuO1xufVxuXG4vKipcbiAqIFNlbGVjdCB0aGUgaW1hZ2UgYmFzZWQgb24gYSBnaXZlbiBTU00gcGFyYW1ldGVyXG4gKlxuICogVGhpcyBNYWNoaW5lIEltYWdlIGF1dG9tYXRpY2FsbHkgdXBkYXRlcyB0byB0aGUgbGF0ZXN0IHZlcnNpb24gb24gZXZlcnlcbiAqIGRlcGxveW1lbnQuIEJlIGF3YXJlIHRoaXMgd2lsbCBjYXVzZSB5b3VyIGluc3RhbmNlcyB0byBiZSByZXBsYWNlZCB3aGVuIGFcbiAqIG5ldyB2ZXJzaW9uIG9mIHRoZSBpbWFnZSBiZWNvbWVzIGF2YWlsYWJsZS4gRG8gbm90IHN0b3JlIHN0YXRlZnVsIGluZm9ybWF0aW9uXG4gKiBvbiB0aGUgaW5zdGFuY2UgaWYgeW91IGFyZSB1c2luZyB0aGlzIGltYWdlLlxuICpcbiAqIFRoZSBBTUkgSUQgaXMgc2VsZWN0ZWQgdXNpbmcgdGhlIHZhbHVlcyBwdWJsaXNoZWQgdG8gdGhlIFNTTSBwYXJhbWV0ZXIgc3RvcmUuXG4gKi9cbmNsYXNzIEdlbmVyaWNTc21QYXJhbWV0ZXJJbWFnZSBpbXBsZW1lbnRzIElNYWNoaW5lSW1hZ2Uge1xuICBjb25zdHJ1Y3Rvcihwcml2YXRlIHJlYWRvbmx5IHBhcmFtZXRlck5hbWU6IHN0cmluZywgcHJpdmF0ZSByZWFkb25seSBwcm9wczogU3NtUGFyYW1ldGVySW1hZ2VPcHRpb25zID0ge30pIHtcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm4gdGhlIGltYWdlIHRvIHVzZSBpbiB0aGUgZ2l2ZW4gY29udGV4dFxuICAgKi9cbiAgcHVibGljIGdldEltYWdlKHNjb3BlOiBDb25zdHJ1Y3QpOiBNYWNoaW5lSW1hZ2VDb25maWcge1xuICAgIGNvbnN0IGltYWdlSWQgPSBsb29rdXBJbWFnZShzY29wZSwgdGhpcy5wcm9wcy5jYWNoZWRJbkNvbnRleHQsIHRoaXMucGFyYW1ldGVyTmFtZSk7XG5cbiAgICBjb25zdCBvc1R5cGUgPSB0aGlzLnByb3BzLm9zID8/IE9wZXJhdGluZ1N5c3RlbVR5cGUuTElOVVg7XG4gICAgcmV0dXJuIHtcbiAgICAgIGltYWdlSWQsXG4gICAgICBvc1R5cGUsXG4gICAgICB1c2VyRGF0YTogdGhpcy5wcm9wcy51c2VyRGF0YSA/PyAob3NUeXBlID09PSBPcGVyYXRpbmdTeXN0ZW1UeXBlLldJTkRPV1MgPyBVc2VyRGF0YS5mb3JXaW5kb3dzKCkgOiBVc2VyRGF0YS5mb3JMaW51eCgpKSxcbiAgICB9O1xuICB9XG59XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBpbnRlcmZhY2UgV2luZG93c0ltYWdlUHJvcHMge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSB1c2VyRGF0YT86IFVzZXJEYXRhO1xufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgY2xhc3MgV2luZG93c0ltYWdlIGV4dGVuZHMgR2VuZXJpY1NTTVBhcmFtZXRlckltYWdlIHtcbiAgY29uc3RydWN0b3IodmVyc2lvbjogV2luZG93c1ZlcnNpb24sIHByb3BzOiBXaW5kb3dzSW1hZ2VQcm9wcyA9IHt9KSB7XG4gICAgc3VwZXIoJy9hd3Mvc2VydmljZS9hbWktd2luZG93cy1sYXRlc3QvJyArIHZlcnNpb24sIE9wZXJhdGluZ1N5c3RlbVR5cGUuV0lORE9XUywgcHJvcHMudXNlckRhdGEpO1xuICB9XG59XG5cbiAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBlbnVtIEFtYXpvbkxpbnV4Q3B1VHlwZSB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIEFSTV82NCA9ICdhcm02NCcsXG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIFg4Nl82NCA9ICd4ODZfNjQnLFxufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgaW50ZXJmYWNlIEFtYXpvbkxpbnV4SW1hZ2VQcm9wcyB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBnZW5lcmF0aW9uPzogQW1hem9uTGludXhHZW5lcmF0aW9uO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBlZGl0aW9uPzogQW1hem9uTGludXhFZGl0aW9uO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSB2aXJ0dWFsaXphdGlvbj86IEFtYXpvbkxpbnV4VmlydDtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgc3RvcmFnZT86IEFtYXpvbkxpbnV4U3RvcmFnZTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgdXNlckRhdGE/OiBVc2VyRGF0YTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGNwdVR5cGU/OiBBbWF6b25MaW51eENwdVR5cGU7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGNhY2hlZEluQ29udGV4dD86IGJvb2xlYW47XG59XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBjbGFzcyBBbWF6b25MaW51eEltYWdlIGV4dGVuZHMgR2VuZXJpY1NTTVBhcmFtZXRlckltYWdlIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgc3NtUGFyYW1ldGVyTmFtZShwcm9wczogQW1hem9uTGludXhJbWFnZVByb3BzID0ge30pIHtcbiAgICBjb25zdCBnZW5lcmF0aW9uID0gKHByb3BzICYmIHByb3BzLmdlbmVyYXRpb24pIHx8IEFtYXpvbkxpbnV4R2VuZXJhdGlvbi5BTUFaT05fTElOVVg7XG4gICAgY29uc3QgZWRpdGlvbiA9IChwcm9wcyAmJiBwcm9wcy5lZGl0aW9uKSB8fCBBbWF6b25MaW51eEVkaXRpb24uU1RBTkRBUkQ7XG4gICAgY29uc3QgdmlydHVhbGl6YXRpb24gPSAocHJvcHMgJiYgcHJvcHMudmlydHVhbGl6YXRpb24pIHx8IEFtYXpvbkxpbnV4VmlydC5IVk07XG4gICAgY29uc3Qgc3RvcmFnZSA9IChwcm9wcyAmJiBwcm9wcy5zdG9yYWdlKSB8fCBBbWF6b25MaW51eFN0b3JhZ2UuR0VORVJBTF9QVVJQT1NFO1xuICAgIGNvbnN0IGNwdSA9IChwcm9wcyAmJiBwcm9wcy5jcHVUeXBlKSB8fCBBbWF6b25MaW51eENwdVR5cGUuWDg2XzY0O1xuICAgIGNvbnN0IHBhcnRzOiBBcnJheTxzdHJpbmd8dW5kZWZpbmVkPiA9IFtcbiAgICAgIGdlbmVyYXRpb24sXG4gICAgICAnYW1pJyxcbiAgICAgIGVkaXRpb24gIT09IEFtYXpvbkxpbnV4RWRpdGlvbi5TVEFOREFSRCA/IGVkaXRpb24gOiB1bmRlZmluZWQsXG4gICAgICB2aXJ0dWFsaXphdGlvbixcbiAgICAgIGNwdSxcbiAgICAgIHN0b3JhZ2UsXG4gICAgXS5maWx0ZXIoeCA9PiB4ICE9PSB1bmRlZmluZWQpOyAvLyBHZXQgcmlkIG9mIHVuZGVmaW5lZHNcblxuICAgIHJldHVybiAnL2F3cy9zZXJ2aWNlL2FtaS1hbWF6b24tbGludXgtbGF0ZXN0LycgKyBwYXJ0cy5qb2luKCctJyk7XG4gIH1cblxuICBwcml2YXRlIHJlYWRvbmx5IGNhY2hlZEluQ29udGV4dDogYm9vbGVhbjtcblxuICBjb25zdHJ1Y3Rvcihwcml2YXRlIHJlYWRvbmx5IHByb3BzOiBBbWF6b25MaW51eEltYWdlUHJvcHMgPSB7fSkge1xuICAgIHN1cGVyKEFtYXpvbkxpbnV4SW1hZ2Uuc3NtUGFyYW1ldGVyTmFtZShwcm9wcyksIE9wZXJhdGluZ1N5c3RlbVR5cGUuTElOVVgsIHByb3BzLnVzZXJEYXRhKTtcblxuICAgIHRoaXMuY2FjaGVkSW5Db250ZXh0ID0gcHJvcHMuY2FjaGVkSW5Db250ZXh0ID8/IGZhbHNlO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIGdldEltYWdlKHNjb3BlOiBDb25zdHJ1Y3QpOiBNYWNoaW5lSW1hZ2VDb25maWcge1xuICAgIGNvbnN0IGltYWdlSWQgPSBsb29rdXBJbWFnZShzY29wZSwgdGhpcy5jYWNoZWRJbkNvbnRleHQsIHRoaXMucGFyYW1ldGVyTmFtZSk7XG5cbiAgICBjb25zdCBvc1R5cGUgPSBPcGVyYXRpbmdTeXN0ZW1UeXBlLkxJTlVYO1xuICAgIHJldHVybiB7XG4gICAgICBpbWFnZUlkLFxuICAgICAgb3NUeXBlLFxuICAgICAgdXNlckRhdGE6IHRoaXMucHJvcHMudXNlckRhdGEgPz8gVXNlckRhdGEuZm9yTGludXgoKSxcbiAgICB9O1xuICB9XG59XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBlbnVtIEFtYXpvbkxpbnV4R2VuZXJhdGlvbiB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBBTUFaT05fTElOVVggPSAnYW16bicsXG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgQU1BWk9OX0xJTlVYXzIgPSAnYW16bjInLFxufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgZW51bSBBbWF6b25MaW51eEVkaXRpb24ge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIFNUQU5EQVJEID0gJ3N0YW5kYXJkJyxcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgTUlOSU1BTCA9ICdtaW5pbWFsJ1xufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGVudW0gQW1hem9uTGludXhWaXJ0IHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgSFZNID0gJ2h2bScsXG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgUFYgPSAncHYnXG59XG5cbmV4cG9ydCBlbnVtIEFtYXpvbkxpbnV4U3RvcmFnZSB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBFQlMgPSAnZWJzJyxcblxuICAvKipcbiAgICogUzMtYmFja2VkIHN0b3JhZ2VcbiAgICovXG4gIFMzID0gJ2VicycsXG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBHRU5FUkFMX1BVUlBPU0UgPSAnZ3AyJyxcbn1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgaW50ZXJmYWNlIEdlbmVyaWNMaW51eEltYWdlUHJvcHMge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgdXNlckRhdGE/OiBVc2VyRGF0YTtcbn1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBpbnRlcmZhY2UgR2VuZXJpY1dpbmRvd3NJbWFnZVByb3BzIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgdXNlckRhdGE/OiBVc2VyRGF0YTtcbn1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgY2xhc3MgR2VuZXJpY0xpbnV4SW1hZ2UgaW1wbGVtZW50cyBJTWFjaGluZUltYWdlIHtcbiAgY29uc3RydWN0b3IocHJpdmF0ZSByZWFkb25seSBhbWlNYXA6IHsgW3JlZ2lvbjogc3RyaW5nXTogc3RyaW5nIH0sIHByaXZhdGUgcmVhZG9ubHkgcHJvcHM6IEdlbmVyaWNMaW51eEltYWdlUHJvcHMgPSB7fSkge1xuICB9XG5cbiAgcHVibGljIGdldEltYWdlKHNjb3BlOiBDb25zdHJ1Y3QpOiBNYWNoaW5lSW1hZ2VDb25maWcge1xuICAgIGNvbnN0IHVzZXJEYXRhID0gdGhpcy5wcm9wcy51c2VyRGF0YSA/PyBVc2VyRGF0YS5mb3JMaW51eCgpO1xuICAgIGNvbnN0IG9zVHlwZSA9IE9wZXJhdGluZ1N5c3RlbVR5cGUuTElOVVg7XG4gICAgY29uc3QgcmVnaW9uID0gU3RhY2sub2Yoc2NvcGUpLnJlZ2lvbjtcbiAgICBpZiAoVG9rZW4uaXNVbnJlc29sdmVkKHJlZ2lvbikpIHtcbiAgICAgIGNvbnN0IG1hcHBpbmc6IHsgW2sxOiBzdHJpbmddOiB7IFtrMjogc3RyaW5nXTogYW55IH0gfSA9IHt9O1xuICAgICAgZm9yIChjb25zdCBbcmduLCBhbWldIG9mIE9iamVjdC5lbnRyaWVzKHRoaXMuYW1pTWFwKSkge1xuICAgICAgICBtYXBwaW5nW3Jnbl0gPSB7IGFtaSB9O1xuICAgICAgfVxuICAgICAgY29uc3QgYW1pTWFwID0gbmV3IENmbk1hcHBpbmcoc2NvcGUsICdBbWlNYXAnLCB7IG1hcHBpbmcgfSk7XG4gICAgICByZXR1cm4ge1xuICAgICAgICBpbWFnZUlkOiBhbWlNYXAuZmluZEluTWFwKEF3cy5SRUdJT04sICdhbWknKSxcbiAgICAgICAgdXNlckRhdGEsXG4gICAgICAgIG9zVHlwZSxcbiAgICAgIH07XG4gICAgfVxuICAgIGNvbnN0IGltYWdlSWQgPSByZWdpb24gIT09ICd0ZXN0LXJlZ2lvbicgPyB0aGlzLmFtaU1hcFtyZWdpb25dIDogJ2FtaS0xMjM0NSc7XG4gICAgaWYgKCFpbWFnZUlkKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYFVuYWJsZSB0byBmaW5kIEFNSSBpbiBBTUkgbWFwOiBubyBBTUkgc3BlY2lmaWVkIGZvciByZWdpb24gJyR7cmVnaW9ufSdgKTtcbiAgICB9XG4gICAgcmV0dXJuIHtcbiAgICAgIGltYWdlSWQsXG4gICAgICB1c2VyRGF0YSxcbiAgICAgIG9zVHlwZSxcbiAgICB9O1xuICB9XG59XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgY2xhc3MgR2VuZXJpY1dpbmRvd3NJbWFnZSBpbXBsZW1lbnRzIElNYWNoaW5lSW1hZ2Uge1xuICBjb25zdHJ1Y3Rvcihwcml2YXRlIHJlYWRvbmx5IGFtaU1hcDoge1tyZWdpb246IHN0cmluZ106IHN0cmluZ30sIHByaXZhdGUgcmVhZG9ubHkgcHJvcHM6IEdlbmVyaWNXaW5kb3dzSW1hZ2VQcm9wcyA9IHt9KSB7XG4gIH1cblxuICBwdWJsaWMgZ2V0SW1hZ2Uoc2NvcGU6IENvbnN0cnVjdCk6IE1hY2hpbmVJbWFnZUNvbmZpZyB7XG4gICAgY29uc3QgdXNlckRhdGEgPSB0aGlzLnByb3BzLnVzZXJEYXRhID8/IFVzZXJEYXRhLmZvcldpbmRvd3MoKTtcbiAgICBjb25zdCBvc1R5cGUgPSBPcGVyYXRpbmdTeXN0ZW1UeXBlLldJTkRPV1M7XG4gICAgY29uc3QgcmVnaW9uID0gU3RhY2sub2Yoc2NvcGUpLnJlZ2lvbjtcbiAgICBpZiAoVG9rZW4uaXNVbnJlc29sdmVkKHJlZ2lvbikpIHtcbiAgICAgIGNvbnN0IG1hcHBpbmc6IHsgW2sxOiBzdHJpbmddOiB7IFtrMjogc3RyaW5nXTogYW55IH0gfSA9IHt9O1xuICAgICAgZm9yIChjb25zdCBbcmduLCBhbWldIG9mIE9iamVjdC5lbnRyaWVzKHRoaXMuYW1pTWFwKSkge1xuICAgICAgICBtYXBwaW5nW3Jnbl0gPSB7IGFtaSB9O1xuICAgICAgfVxuICAgICAgY29uc3QgYW1pTWFwID0gbmV3IENmbk1hcHBpbmcoc2NvcGUsICdBbWlNYXAnLCB7IG1hcHBpbmcgfSk7XG4gICAgICByZXR1cm4ge1xuICAgICAgICBpbWFnZUlkOiBhbWlNYXAuZmluZEluTWFwKEF3cy5SRUdJT04sICdhbWknKSxcbiAgICAgICAgdXNlckRhdGEsXG4gICAgICAgIG9zVHlwZSxcbiAgICAgIH07XG4gICAgfVxuICAgIGNvbnN0IGltYWdlSWQgPSByZWdpb24gIT09ICd0ZXN0LXJlZ2lvbicgPyB0aGlzLmFtaU1hcFtyZWdpb25dIDogJ2FtaS0xMjM0NSc7XG4gICAgaWYgKCFpbWFnZUlkKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYFVuYWJsZSB0byBmaW5kIEFNSSBpbiBBTUkgbWFwOiBubyBBTUkgc3BlY2lmaWVkIGZvciByZWdpb24gJyR7cmVnaW9ufSdgKTtcbiAgICB9XG4gICAgcmV0dXJuIHtcbiAgICAgIGltYWdlSWQsXG4gICAgICB1c2VyRGF0YSxcbiAgICAgIG9zVHlwZSxcbiAgICB9O1xuICB9XG59XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgZW51bSBPcGVyYXRpbmdTeXN0ZW1UeXBlIHtcbiAgTElOVVgsXG4gIFdJTkRPV1MsXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgVU5LTk9XTixcbn1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBjbGFzcyBMb29rdXBNYWNoaW5lSW1hZ2UgaW1wbGVtZW50cyBJTWFjaGluZUltYWdlIHtcbiAgY29uc3RydWN0b3IocHJpdmF0ZSByZWFkb25seSBwcm9wczogTG9va3VwTWFjaGluZUltYWdlUHJvcHMpIHtcbiAgfVxuXG4gIHB1YmxpYyBnZXRJbWFnZShzY29wZTogQ29uc3RydWN0KTogTWFjaGluZUltYWdlQ29uZmlnIHtcbiAgICAvLyBOZWVkIHRvIGtub3cgJ3dpbmRvd3MnIG9yIG5vdCBiZWZvcmUgZG9pbmcgdGhlIHF1ZXJ5IHRvIHJldHVybiB0aGUgcmlnaHRcbiAgICAvLyBvc1R5cGUgZm9yIHRoZSBkdW1teSB2YWx1ZSwgc28gbWlnaHQgYXMgd2VsbCBhZGQgaXQgdG8gdGhlIGZpbHRlci5cbiAgICBjb25zdCBmaWx0ZXJzOiBSZWNvcmQ8c3RyaW5nLCBzdHJpbmdbXSB8IHVuZGVmaW5lZD4gPSB7XG4gICAgICAnbmFtZSc6IFt0aGlzLnByb3BzLm5hbWVdLFxuICAgICAgJ3N0YXRlJzogWydhdmFpbGFibGUnXSxcbiAgICAgICdpbWFnZS10eXBlJzogWydtYWNoaW5lJ10sXG4gICAgICAncGxhdGZvcm0nOiB0aGlzLnByb3BzLndpbmRvd3MgPyBbJ3dpbmRvd3MnXSA6IHVuZGVmaW5lZCxcbiAgICB9O1xuICAgIE9iamVjdC5hc3NpZ24oZmlsdGVycywgdGhpcy5wcm9wcy5maWx0ZXJzKTtcblxuICAgIGNvbnN0IHZhbHVlID0gQ29udGV4dFByb3ZpZGVyLmdldFZhbHVlKHNjb3BlLCB7XG4gICAgICBwcm92aWRlcjogY3hzY2hlbWEuQ29udGV4dFByb3ZpZGVyLkFNSV9QUk9WSURFUixcbiAgICAgIHByb3BzOiB7XG4gICAgICAgIG93bmVyczogdGhpcy5wcm9wcy5vd25lcnMsXG4gICAgICAgIGZpbHRlcnMsXG4gICAgICB9IGFzIGN4c2NoZW1hLkFtaUNvbnRleHRRdWVyeSxcbiAgICAgIGR1bW15VmFsdWU6ICdhbWktMTIzNCcsXG4gICAgfSkudmFsdWUgYXMgY3hhcGkuQW1pQ29udGV4dFJlc3BvbnNlO1xuXG4gICAgaWYgKHR5cGVvZiB2YWx1ZSAhPT0gJ3N0cmluZycpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihgUmVzcG9uc2UgdG8gQU1JIGxvb2t1cCBpbnZhbGlkLCBnb3Q6ICR7dmFsdWV9YCk7XG4gICAgfVxuXG4gICAgY29uc3Qgb3NUeXBlID0gdGhpcy5wcm9wcy53aW5kb3dzID8gT3BlcmF0aW5nU3lzdGVtVHlwZS5XSU5ET1dTIDogT3BlcmF0aW5nU3lzdGVtVHlwZS5MSU5VWDtcblxuICAgIHJldHVybiB7XG4gICAgICBpbWFnZUlkOiB2YWx1ZSxcbiAgICAgIG9zVHlwZSxcbiAgICAgIHVzZXJEYXRhOiB0aGlzLnByb3BzLnVzZXJEYXRhID8/IFVzZXJEYXRhLmZvck9wZXJhdGluZ1N5c3RlbShvc1R5cGUpLFxuICAgIH07XG4gIH1cbn1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgaW50ZXJmYWNlIExvb2t1cE1hY2hpbmVJbWFnZVByb3BzIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgbmFtZTogc3RyaW5nO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBvd25lcnM/OiBzdHJpbmdbXTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBmaWx0ZXJzPzoge1trZXk6IHN0cmluZ106IHN0cmluZ1tdfTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgd2luZG93cz86IGJvb2xlYW47XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgdXNlckRhdGE/OiBVc2VyRGF0YTtcbn1cblxuZnVuY3Rpb24gbG9va3VwSW1hZ2Uoc2NvcGU6IENvbnN0cnVjdCwgY2FjaGVkSW5Db250ZXh0OiBib29sZWFuIHwgdW5kZWZpbmVkLCBwYXJhbWV0ZXJOYW1lOiBzdHJpbmcpIHtcbiAgcmV0dXJuIGNhY2hlZEluQ29udGV4dFxuICAgID8gc3NtLlN0cmluZ1BhcmFtZXRlci52YWx1ZUZyb21Mb29rdXAoc2NvcGUsIHBhcmFtZXRlck5hbWUpXG4gICAgOiBzc20uU3RyaW5nUGFyYW1ldGVyLnZhbHVlRm9yVHlwZWRTdHJpbmdQYXJhbWV0ZXIoc2NvcGUsIHBhcmFtZXRlck5hbWUsIHNzbS5QYXJhbWV0ZXJUeXBlLkFXU19FQzJfSU1BR0VfSUQpO1xufVxuIl19