"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.NetworkMultipleTargetGroupsServiceBase = void 0;
const aws_ecs_1 = require("@aws-cdk/aws-ecs");
const aws_elasticloadbalancingv2_1 = require("@aws-cdk/aws-elasticloadbalancingv2");
const aws_route53_1 = require("@aws-cdk/aws-route53");
const aws_route53_targets_1 = require("@aws-cdk/aws-route53-targets");
const core_1 = require("@aws-cdk/core");
/**
 * The base class for NetworkMultipleTargetGroupsEc2Service and NetworkMultipleTargetGroupsFargateService classes.
 */
class NetworkMultipleTargetGroupsServiceBase extends core_1.Construct {
    /**
     * Constructs a new instance of the NetworkMultipleTargetGroupsServiceBase class.
     */
    constructor(scope, id, props = {}) {
        super(scope, id);
        this.listeners = new Array();
        this.targetGroups = new Array();
        this.loadBalancers = new Array();
        this.validateInput(props);
        this.cluster = props.cluster || this.getDefaultCluster(this, props.vpc);
        this.desiredCount = props.desiredCount || 1;
        if (props.taskImageOptions) {
            this.logDriver = this.createLogDriver(props.taskImageOptions.enableLogging, props.taskImageOptions.logDriver);
        }
        if (props.loadBalancers) {
            for (const lbProps of props.loadBalancers) {
                const lb = this.createLoadBalancer(lbProps.name, lbProps.publicLoadBalancer);
                this.loadBalancers.push(lb);
                for (const listenerProps of lbProps.listeners) {
                    const listener = this.createListener(listenerProps.name, lb, listenerProps.port || 80);
                    this.listeners.push(listener);
                }
                this.createDomainName(lb, lbProps.domainName, lbProps.domainZone);
                new core_1.CfnOutput(this, `LoadBalancerDNS${lb.node.id}`, { value: lb.loadBalancerDnsName });
            }
            // set up default load balancer and listener.
            this.loadBalancer = this.loadBalancers[0];
            this.listener = this.listeners[0];
        }
        else {
            this.loadBalancer = this.createLoadBalancer('LB');
            this.listener = this.createListener('PublicListener', this.loadBalancer, 80);
            this.createDomainName(this.loadBalancer);
            new core_1.CfnOutput(this, 'LoadBalancerDNS', { value: this.loadBalancer.loadBalancerDnsName });
        }
    }
    /**
     * Returns the default cluster.
     */
    getDefaultCluster(scope, vpc) {
        // magic string to avoid collision with user-defined constructs.
        const DEFAULT_CLUSTER_ID = `EcsDefaultClusterMnL3mNNYN${vpc ? vpc.node.id : ''}`;
        const stack = core_1.Stack.of(scope);
        return stack.node.tryFindChild(DEFAULT_CLUSTER_ID) || new aws_ecs_1.Cluster(stack, DEFAULT_CLUSTER_ID, { vpc });
    }
    createAWSLogDriver(prefix) {
        return new aws_ecs_1.AwsLogDriver({ streamPrefix: prefix });
    }
    findListener(name) {
        if (!name) {
            return this.listener;
        }
        for (const listener of this.listeners) {
            if (listener.node.id === name) {
                return listener;
            }
        }
        throw new Error(`Listener ${name} is not defined. Did you define listener with name ${name}?`);
    }
    registerECSTargets(service, container, targets) {
        for (const targetProps of targets) {
            const targetGroup = this.findListener(targetProps.listener).addTargets(`ECSTargetGroup${container.containerName}${targetProps.containerPort}`, {
                port: 80,
                targets: [
                    service.loadBalancerTarget({
                        containerName: container.containerName,
                        containerPort: targetProps.containerPort,
                    }),
                ],
            });
            this.targetGroups.push(targetGroup);
        }
        if (this.targetGroups.length === 0) {
            throw new Error('At least one target group should be specified.');
        }
        return this.targetGroups[0];
    }
    addPortMappingForTargets(container, targets) {
        for (const target of targets) {
            if (!container.findPortMapping(target.containerPort, aws_ecs_1.Protocol.TCP)) {
                container.addPortMappings({
                    containerPort: target.containerPort,
                });
            }
        }
    }
    /**
     * Create log driver if logging is enabled.
     */
    createLogDriver(enableLoggingProp, logDriverProp) {
        const enableLogging = enableLoggingProp !== undefined ? enableLoggingProp : true;
        const logDriver = logDriverProp !== undefined
            ? logDriverProp : enableLogging
            ? this.createAWSLogDriver(this.node.id) : undefined;
        return logDriver;
    }
    validateInput(props) {
        if (props.cluster && props.vpc) {
            throw new Error('You can only specify either vpc or cluster. Alternatively, you can leave both blank');
        }
        if (props.desiredCount !== undefined && props.desiredCount < 1) {
            throw new Error('You must specify a desiredCount greater than 0');
        }
        if (props.loadBalancers) {
            if (props.loadBalancers.length === 0) {
                throw new Error('At least one load balancer must be specified');
            }
            for (const lbProps of props.loadBalancers) {
                if (lbProps.listeners.length === 0) {
                    throw new Error('At least one listener must be specified');
                }
            }
        }
    }
    createLoadBalancer(name, publicLoadBalancer) {
        const internetFacing = publicLoadBalancer !== undefined ? publicLoadBalancer : true;
        const lbProps = {
            vpc: this.cluster.vpc,
            internetFacing,
        };
        return new aws_elasticloadbalancingv2_1.NetworkLoadBalancer(this, name, lbProps);
    }
    createListener(name, lb, port) {
        return lb.addListener(name, {
            port,
        });
    }
    createDomainName(loadBalancer, name, zone) {
        if (typeof name !== 'undefined') {
            if (typeof zone === 'undefined') {
                throw new Error('A Route53 hosted domain zone name is required to configure the specified domain name');
            }
            new aws_route53_1.ARecord(this, `DNS${loadBalancer.node.id}`, {
                zone,
                recordName: name,
                target: aws_route53_1.RecordTarget.fromAlias(new aws_route53_targets_1.LoadBalancerTarget(loadBalancer)),
            });
        }
    }
}
exports.NetworkMultipleTargetGroupsServiceBase = NetworkMultipleTargetGroupsServiceBase;
//# sourceMappingURL=data:application/json;base64,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