#!/usr/bin/env python3
# MIT License
#
# Copyright (c) 2020 FABRIC Testbed
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.
#
#
# Author: Komal Thareja (kthare10@renci.org)
import json
import os
import traceback
from datetime import datetime

from fss_utils.sshkey import FABRICSSHKey
from atomicwrites import atomic_write
import wget
import tarfile
import re
import pip


class JupyterStartup:
    DEFAULT_NOTEBOOK_LOCATION = "/home/fabric/work/"
    DEFAULT_FABRIC_CONFIG_LOCATION = "/home/fabric/work/fabric_config"
    DEFAULT_REQUIREMENTS_LOCATION = "/home/fabric/work/fabric_config/requirements.tx"
    DEFAULT_FABRIC_CONFIG_JSON_LOCATION = "/home/fabric/work/fabric_config/fabric_config.json"
    TOKENS_LOCATION = "/home/fabric/.tokens.json"
    TAGS = "rel1.3"
    REPO_URL = "https://github.com/fabric-testbed/jupyter-examples/archive/refs/tags/"
    EXAMPLES = "examples"
    URL = "url"
    LOCATION = "location"
    REFRESH_TOKEN = "refresh_token"
    TIME_FORMAT = "%Y-%m-%d %H:%M:%S"
    CREATED_AT = "created_at"
    DEFAULT_PRIVATE_SSH_KEY = "/home/fabric/.ssh/id_rsa"
    DEFAULT_PUBLIC_SSH_KEY = "/home/fabric/.ssh/id_rsa.pub"
    DEFAULT_FABRIC_LOG_LEVEL = "INFO"
    DEFAULT_FABRIC_LOG_FILE = "/tmp/fablib/fablib.log"

    FABRIC_CREDMGR_HOST = "FABRIC_CREDMGR_HOST"
    FABRIC_ORCHESTRATOR_HOST = "FABRIC_ORCHESTRATOR_HOST"
    FABRIC_TOKEN_LOCATION = "FABRIC_TOKEN_LOCATION"
    FABRIC_PROJECT_ID = "FABRIC_PROJECT_ID"
    FABRIC_NOTEBOOK_LOCATION = "FABRIC_NOTEBOOK_LOCATION"
    FABRIC_NOTEBOOK_TAGS = "FABRIC_NOTEBOOK_TAGS"
    FABRIC_NOTEBOOK_REPO_URL = "FABRIC_NOTEBOOK_REPO_URL"
    FABRIC_CONFIG_LOCATION = "FABRIC_CONFIG_LOCATION"
    FABRIC_REQUIREMENTS_LOCATION = "FABRIC_REQUIREMENTS_LOCATION"
    FABRIC_CONFIG_JSON_LOCATION = "FABRIC_CONFIG_JSON_LOCATION"
    FABRIC_BASTION_HOST = "FABRIC_BASTION_HOST"
    FABRIC_BASTION_USERNAME = "FABRIC_BASTION_USERNAME"
    FABRIC_BASTION_KEY_LOCATION = "FABRIC_BASTION_KEY_LOCATION"
    FABRIC_SLICE_PRIVATE_KEY_FILE = "FABRIC_SLICE_PRIVATE_KEY_FILE"
    FABRIC_SLICE_PUBLIC_KEY_FILE = "FABRIC_SLICE_PUBLIC_KEY_FILE"
    FABRIC_SLICE_PRIVATE_KEY_PASSPHRASE = "FABRIC_SLICE_PRIVATE_KEY_PASSPHRASE"
    FABRIC_BASTION_PRIVATE_KEY_NAME = "FABRIC_BASTION_PRIVATE_KEY_NAME"
    FABRIC_SLICE_PRIVATE_KEY_NAME = "FABRIC_SLICE_PRIVATE_KEY_NAME"
    FABRIC_SLICE_PUBLIC_KEY_NAME = "FABRIC_SLICE_PUBLIC_KEY_NAME"
    FABRIC_LOG_FILE = "FABRIC_LOG_FILE"
    FABRIC_LOG_LEVEL = "FABRIC_LOG_LEVEL"

    def __init__(self):
        self.notebook_location = os.environ.get(self.FABRIC_NOTEBOOK_LOCATION)
        if self.notebook_location is None:
            self.notebook_location = self.DEFAULT_NOTEBOOK_LOCATION

        self.token_location = os.environ.get(self.FABRIC_TOKEN_LOCATION)
        if self.token_location is None:
            self.token_location = self.TOKENS_LOCATION

        self.tags = os.environ.get(self.FABRIC_NOTEBOOK_TAGS)
        if self.tags is None:
            self.tags = self.TAGS

        self.repo_url = os.environ.get(self.FABRIC_NOTEBOOK_REPO_URL)
        if self.repo_url is None:
            self.repo_url = self.REPO_URL

        self.config_location = os.environ.get(self.FABRIC_CONFIG_LOCATION)
        if self.config_location is None:
            self.config_location = self.DEFAULT_FABRIC_CONFIG_LOCATION

        self.requirements_location = os.environ.get(self.FABRIC_REQUIREMENTS_LOCATION)
        if self.requirements_location is None:
            self.requirements_location = self.DEFAULT_REQUIREMENTS_LOCATION

        self.config_json_location = os.environ.get(self.FABRIC_CONFIG_JSON_LOCATION)
        if self.config_json_location is None:
            self.config_json_location = self.DEFAULT_FABRIC_CONFIG_JSON_LOCATION

    def create_config_dir(self):
        try:
            os.mkdir(self.config_location)
            environment_vars = {
                self.FABRIC_CREDMGR_HOST: os.environ[self.FABRIC_CREDMGR_HOST],
                self.FABRIC_ORCHESTRATOR_HOST: os.environ[self.FABRIC_ORCHESTRATOR_HOST],
                self.FABRIC_BASTION_HOST: os.environ[self.FABRIC_BASTION_HOST],
                self.FABRIC_PROJECT_ID: '<Update Project Id>',
                self.FABRIC_BASTION_USERNAME: '<Update User Name>',
                self.FABRIC_BASTION_KEY_LOCATION: f'{self.config_location}/{os.environ[self.FABRIC_BASTION_PRIVATE_KEY_NAME]}',
                self.FABRIC_SLICE_PRIVATE_KEY_FILE: f'{self.config_location}/{os.environ[self.FABRIC_SLICE_PRIVATE_KEY_NAME]}',
                self.FABRIC_SLICE_PUBLIC_KEY_FILE: f'{self.config_location}/{os.environ[self.FABRIC_SLICE_PUBLIC_KEY_NAME]}',
                self.FABRIC_SLICE_PRIVATE_KEY_PASSPHRASE: '<Update Passphrase>',
                self.FABRIC_LOG_LEVEL: self.DEFAULT_FABRIC_LOG_LEVEL,
                self.FABRIC_LOG_FILE: self.DEFAULT_FABRIC_LOG_FILE
            }
            string_to_write = ""
            for key, value in environment_vars.items():
                if '<' in value and '>':
                    string_to_write += f"#export {key}={value}\n"
                else:
                    string_to_write += f"export {key}={value}\n"

            with atomic_write(f'{self.config_location}/fabric_rc', overwrite=True) as f:
                f.write(string_to_write)

            string_to_write = f"UserKnownHostsFile /dev/null\n" \
                              f"StrictHostKeyChecking no\n" \
                              f"ServerAliveInterval 120 \n" \
                              f"Host bastion-?.fabric-testbed.net\n" \
                              f"User <Update Bastion User Name>\n" \
                              f"ForwardAgent yes\n" \
                              f"Hostname %h\n" \
                              f"IdentityFile {self.config_location}/{os.environ[self.FABRIC_BASTION_PRIVATE_KEY_NAME]}\n" \
                              f"IdentitiesOnly yes\n" \
                              f"Host * !bastion-?.fabric-testbed.net\n" \
                              f"ProxyJump <Update Bastion User Name>@{os.environ[self.FABRIC_BASTION_HOST]}:22\n"
            with atomic_write(f'{self.config_location}/ssh_config', overwrite=True) as f:
                f.write(string_to_write)
        except Exception as e:
            print("Failed to create config directory and default environment file")
            print("Exception: " + str(e))
            traceback.print_exc()

    @staticmethod
    def get_url_file_name(*, url: str):
        url = url.split("#")[0]
        url = url.split("?")[0]
        return os.path.basename(url)

    def download_notebooks(self):
        try:
            config_json = None
            with open(self.config_json_location) as f:
                config_json = json.load(f)

            notebook_location = self.notebook_location
            tags = self.tags
            file_name_release = f"{self.repo_url}/{self.tags}.tar.gz"

            if config_json is not None and config_json.get(self.EXAMPLES) is not None:
                if config_json.get(self.EXAMPLES)[self.LOCATION] is not None:
                    notebook_location = config_json.get(self.EXAMPLES)[self.LOCATION]
                if config_json.get(self.EXAMPLES)[self.URL] is not None:
                    file_name_release = config_json.get(self.EXAMPLES)[self.URL]
                    file_name = self.get_url_file_name(url=file_name_release)
                    tags = re.sub('.tar.gz|.zip', '', file_name)

            if os.path.exists(f"{notebook_location}/jupyter-examples-{tags}"):
                return

            print(f"Downloading the {file_name_release}")
            file_name = wget.download(file_name_release, notebook_location)
            print(f"Extracting the tarball for the Downloaded code: {file_name}")
            with tarfile.open(file_name) as f:
                f.extractall()
            print(f"Removing the downloaded tarball")
            os.remove(file_name)

        except Exception as e:
            print("Failed to download github repository for notebooks")
            print("Exception: " + str(e))
            traceback.print_exc()

    def create_tokens_file(self):
        try:
            tokens_json = {self.REFRESH_TOKEN: os.environ.get("CILOGON_REFRESH_TOKEN"),
                           self.CREATED_AT: datetime.strftime(datetime.utcnow(), self.TIME_FORMAT)}

            with atomic_write(self.token_location, overwrite=True) as f:
                json.dump(tokens_json, f)
        except Exception as e:
            print("Failed to create tokens file")
            print("Exception: " + str(e))
            traceback.print_exc()

    def create_requirements_file(self):
        try:
            with atomic_write(self.requirements_location, overwrite=True) as f:
                f.write("")
        except Exception as e:
            print("Failed to create tokens file")
            print("Exception: " + str(e))
            traceback.print_exc()

    def create_config_file(self):
        try:
            config_json = {
                self.EXAMPLES: {
                    self.URL: f"{self.repo_url}/{self.tags}.tar.gz",
                    self.LOCATION: self.notebook_location
                }
            }

            with atomic_write(self.token_location, overwrite=True) as f:
                json.dump(config_json, f)
        except Exception as e:
            print("Failed to create tokens file")
            print("Exception: " + str(e))
            traceback.print_exc()

    def custom_install_packages(self):
        try:
            if os.path.exists(self.requirements_location):
                pip.main(["install", "-r", self.requirements_location])
        except Exception as e:
            print("Failed to install user specified packages!")
            print("Exception: " + str(e))
            traceback.print_exc()

    def initialize(self):
        """
        Initialize Jupyter Notebook Container
        """
        if not os.path.exists(self.token_location):
            # New jupyternb container has been created
            # Create a token file
            print("Creating token file")
            self.create_tokens_file()

        if not os.path.exists(self.config_location):
            print("Creating config directory and all files")
            self.create_config_dir()

        if not os.path.exists(self.requirements_location):
            print("Creating default requirements.txt")
            self.create_requirements_file()

        if not os.path.exists(self.config_json_location):
            print("Creating default fabric_config.json")
            self.create_config_file()

        # Download the notebooks
        self.download_notebooks()

        # Create SSH Keys
        ssh_key = FABRICSSHKey.generate(comment="fabric@localhost", algorithm="rsa")
        with atomic_write(f'{self.DEFAULT_PRIVATE_SSH_KEY}', overwrite=True) as f:
            f.write(ssh_key.private_key)
        with atomic_write(f'{self.DEFAULT_PUBLIC_SSH_KEY}', overwrite=True) as f:
            f.write(f'{ssh_key.name} {ssh_key.public_key} {ssh_key.comment}')

        # Default key in config directory
        default_ssh_priv_key_config = f'{self.config_location}/{os.environ[self.FABRIC_SLICE_PRIVATE_KEY_NAME]}'
        default_ssh_pub_key_config = f'{self.config_location}/{os.environ[self.FABRIC_SLICE_PUBLIC_KEY_NAME]}'
 
        if not os.path.exists(default_ssh_priv_key_config):
            with atomic_write(default_ssh_priv_key_config, overwrite=True) as f:
                f.write(ssh_key.private_key)

        if not os.path.exists(default_ssh_pub_key_config):
            with atomic_write(default_ssh_pub_key_config, overwrite=True) as f:
                f.write(f'{ssh_key.name} {ssh_key.public_key} {ssh_key.comment}')

        os.chmod(f"{self.DEFAULT_PRIVATE_SSH_KEY}", 600)
        os.chmod(f"{self.DEFAULT_PUBLIC_SSH_KEY}", 644)
        os.chmod(f"{default_ssh_priv_key_config}", 600)
        os.chmod(f"{default_ssh_pub_key_config}", 644)

        self.custom_install_packages()


if __name__ == "__main__":
    js = JupyterStartup()
    js.initialize()

