"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const yaml = require("yaml");
const __1 = require("..");
const deps_1 = require("../deps");
const logging = require("../logging");
const node_package_1 = require("../node-package");
const util_1 = require("./util");
logging.disable();
test('license file is added by default', () => {
    // WHEN
    const project = new TestNodeProject();
    // THEN
    expect(util_1.synthSnapshot(project).LICENSE).toContain('Apache License');
});
test('license file is not added if licensed is false', () => {
    // WHEN
    const project = new TestNodeProject({
        licensed: false,
    });
    // THEN
    const snapshot = util_1.synthSnapshot(project);
    expect(snapshot.LICENSE).toBeUndefined();
    expect(snapshot['.gitignore']).not.toContain('LICENSE');
    expect(snapshot['package.json'].license).toEqual('UNLICENSED');
});
describe('deps', () => {
    test('runtime deps', () => {
        // GIVEN
        const project = new TestNodeProject({
            deps: [
                'aaa@^1.2.3',
                'bbb@~4.5.6',
            ],
        });
        // WHEN
        project.addDeps('ccc');
        project.deps.addDependency('ddd', deps_1.DependencyType.RUNTIME);
        // THEN
        const pkgjson = packageJson(project);
        expect(pkgjson.dependencies).toStrictEqual({
            aaa: '^1.2.3',
            bbb: '~4.5.6',
            ccc: '*',
            ddd: '*',
        });
        expect(pkgjson.peerDependencies).toStrictEqual({});
    });
    test('dev dependencies', () => {
        // GIVEN
        const project = new TestNodeProject({
            devDeps: [
                'aaa@^1.2.3',
                'bbb@~4.5.6',
            ],
        });
        // WHEN
        project.addDevDeps('ccc');
        project.deps.addDependency('ddd', deps_1.DependencyType.TEST);
        project.deps.addDependency('eee@^1', deps_1.DependencyType.DEVENV);
        project.deps.addDependency('fff@^2', deps_1.DependencyType.BUILD);
        // THEN
        const pkgjson = packageJson(project);
        expect(pkgjson.devDependencies.aaa).toStrictEqual('^1.2.3');
        expect(pkgjson.devDependencies.bbb).toStrictEqual('~4.5.6');
        expect(pkgjson.devDependencies.ccc).toStrictEqual('*');
        expect(pkgjson.devDependencies.ddd).toStrictEqual('*');
        expect(pkgjson.devDependencies.eee).toStrictEqual('^1');
        expect(pkgjson.devDependencies.fff).toStrictEqual('^2');
        expect(pkgjson.peerDependencies).toStrictEqual({});
        expect(pkgjson.dependencieds).toBeUndefined();
    });
    test('peerDependencies', () => {
        // GIVEN
        const project = new TestNodeProject({
            peerDeps: [
                'aaa@^1.2.3',
                'bbb@~4.5.6',
            ],
        });
        // WHEN
        project.addPeerDeps('ccc');
        project.deps.addDependency('ddd', deps_1.DependencyType.PEER);
        // THEN
        const pkgjson = packageJson(project);
        expect(pkgjson.peerDependencies).toStrictEqual({
            aaa: '^1.2.3',
            bbb: '~4.5.6',
            ccc: '*',
            ddd: '*',
        });
        // devDependencies are added with pinned versions
        expect(pkgjson.devDependencies.aaa).toStrictEqual('1.2.3');
        expect(pkgjson.devDependencies.bbb).toStrictEqual('4.5.6');
        expect(pkgjson.devDependencies.ccc).toStrictEqual('*');
        expect(pkgjson.devDependencies.ddd).toStrictEqual('*');
        expect(pkgjson.dependencieds).toBeUndefined();
    });
    test('peerDependencies without pinnedDevDep', () => {
        // GIVEN
        const project = new TestNodeProject({
            peerDependencyOptions: {
                pinnedDevDependency: false,
            },
            peerDeps: [
                'aaa@^1.2.3',
                'bbb@~4.5.6',
            ],
        });
        // WHEN
        project.addPeerDeps('ccc');
        project.deps.addDependency('ddd', deps_1.DependencyType.PEER);
        // THEN
        const pkgjson = packageJson(project);
        expect(pkgjson.peerDependencies).toStrictEqual({
            aaa: '^1.2.3',
            bbb: '~4.5.6',
            ccc: '*',
            ddd: '*',
        });
        // sanitize
        ['jest', 'jest-junit', 'projen', 'standard-version'].forEach(d => delete pkgjson.devDependencies[d]);
        expect(pkgjson.devDependencies).toStrictEqual({});
        expect(pkgjson.dependencieds).toBeUndefined();
    });
    test('bundled deps are automatically added as normal deps', () => {
        // GIVEN
        const project = new TestNodeProject({
            bundledDeps: ['hey@2.1.1'],
        });
        // WHEN
        project.addBundledDeps('foo@^1.2.3');
        project.deps.addDependency('bar@~1.0.0', deps_1.DependencyType.BUNDLED);
        // THEN
        const pkgjson = packageJson(project);
        expect(pkgjson.dependencies).toStrictEqual({
            hey: '2.1.1',
            foo: '^1.2.3',
            bar: '~1.0.0',
        });
        expect(pkgjson.bundledDependencies).toStrictEqual([
            'bar',
            'foo',
            'hey',
        ]);
    });
});
describe('npm publishing options', () => {
    test('defaults', () => {
        // GIVEN
        const project = new util_1.TestProject();
        // WHEN
        const npm = new node_package_1.NodePackage(project, {
            packageName: 'my-package',
        });
        // THEN
        expect(npm.npmAccess).toStrictEqual(node_package_1.NpmAccess.PUBLIC);
        expect(npm.npmDistTag).toStrictEqual('latest');
        expect(npm.npmRegistry).toStrictEqual('registry.npmjs.org');
        expect(npm.npmRegistryUrl).toStrictEqual('https://registry.npmjs.org/');
        // since these are all defaults, publishConfig is not defined.
        expect(util_1.synthSnapshot(project)['package.json'].publishConfig).toBeUndefined();
    });
    test('scoped packages default to RESTRICTED access', () => {
        // GIVEN
        const project = new util_1.TestProject();
        // WHEN
        const npm = new node_package_1.NodePackage(project, {
            packageName: 'scoped@my-package',
        });
        // THEN
        expect(npm.npmAccess).toStrictEqual(node_package_1.NpmAccess.RESTRICTED);
        // since these are all defaults, publishConfig is not defined.
        expect(packageJson(project).publishConfig).toBeUndefined();
    });
    test('non-scoped package cannot be RESTRICTED', () => {
        // GIVEN
        const project = new util_1.TestProject();
        // THEN
        expect(() => new node_package_1.NodePackage(project, {
            packageName: 'my-package',
            npmAccess: node_package_1.NpmAccess.RESTRICTED,
        })).toThrow(/"npmAccess" cannot be RESTRICTED for non-scoped npm package/);
    });
    test('custom settings', () => {
        // GIVEN
        const project = new util_1.TestProject();
        // WHEN
        const npm = new node_package_1.NodePackage(project, {
            packageName: 'scoped@my-package',
            npmDistTag: 'next',
            npmRegistryUrl: 'https://foo.bar',
            npmAccess: node_package_1.NpmAccess.PUBLIC,
        });
        // THEN
        expect(npm.npmDistTag).toStrictEqual('next');
        expect(npm.npmRegistry).toStrictEqual('foo.bar');
        expect(npm.npmRegistryUrl).toStrictEqual('https://foo.bar/');
        expect(npm.npmAccess).toStrictEqual(node_package_1.NpmAccess.PUBLIC);
        expect(packageJson(project).publishConfig).toStrictEqual({
            access: 'public',
            registry: 'https://foo.bar/',
            tag: 'next',
        });
    });
    test('deprecated npmRegistry can be used instead of npmRegistryUrl and then https:// is assumed', () => {
        // GIVEN
        const project = new util_1.TestProject();
        // WHEN
        const npm = new node_package_1.NodePackage(project, {
            packageName: 'scoped@my-package',
            npmRegistry: 'foo.bar.com',
        });
        // THEN
        expect(npm.npmRegistry).toStrictEqual('foo.bar.com');
        expect(npm.npmRegistryUrl).toStrictEqual('https://foo.bar.com/');
        expect(packageJson(project).publishConfig).toStrictEqual({
            registry: 'https://foo.bar.com/',
        });
    });
});
test('extend github release workflow', () => {
    var _a;
    const project = new TestNodeProject();
    (_a = project.releaseWorkflow) === null || _a === void 0 ? void 0 : _a.addJobs({
        publish_docker_hub: {
            'runs-on': 'ubuntu-latest',
            'env': {
                CI: 'true',
            },
            'steps': [
                {
                    name: 'Check out the repo',
                    uses: 'actions/checkout@v2',
                },
                {
                    name: 'Push to Docker Hub',
                    uses: 'docker/build-push-action@v1',
                    with: {
                        username: '${{ secrets.DOCKER_USERNAME }}',
                        password: '${{ secrets.DOCKER_PASSWORD }}',
                        repository: 'projen/projen-docker',
                        tag_with_ref: true,
                    },
                },
            ],
        },
    });
    const workflow = util_1.synthSnapshot(project)['.github/workflows/release.yml'];
    expect(workflow).toContain('publish_docker_hub:\n    runs-on: ubuntu-latest\n');
    expect(workflow).toContain('username: ${{ secrets.DOCKER_USERNAME }}\n          password: ${{ secrets.DOCKER_PASSWORD }}');
});
describe('scripts', () => {
    test('removeScript will remove tasks and scripts', () => {
        const p = new TestNodeProject();
        p.addTask('chortle', { exec: 'echo "frabjous day!"' });
        p.setScript('slithy-toves', 'gyre && gimble');
        expect(packageJson(p).scripts).toHaveProperty('chortle');
        expect(packageJson(p).scripts).toHaveProperty('slithy-toves');
        p.removeScript('chortle');
        p.removeScript('slithy-toves');
        expect(packageJson(p).scripts).not.toHaveProperty('chortle');
        expect(packageJson(p).scripts).not.toHaveProperty('slithy-toves');
    });
});
test('buildWorkflowMutable will push changes to PR branches', () => {
    // WHEN
    const project = new TestNodeProject({
        mutableBuild: true,
    });
    // THEN
    const workflowYaml = util_1.synthSnapshot(project)['.github/workflows/build.yml'];
    const workflow = yaml.parse(workflowYaml);
    expect(workflow.jobs.build.steps).toMatchSnapshot();
});
function packageJson(project) {
    return util_1.synthSnapshot(project)['package.json'];
}
class TestNodeProject extends __1.NodeProject {
    constructor(options = {}) {
        super({
            outdir: util_1.mkdtemp(),
            name: 'test-node-project',
            logging: {
                level: __1.LogLevel.OFF,
            },
            defaultReleaseBranch: 'master',
            ...options,
        });
    }
}
//# sourceMappingURL=data:application/json;base64,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