from os import listdir
from os.path import isfile, isdir, join
import json
import pandas as pd
from ..selections import saveAndOpenSelection

class Dataset():
    """Dataset object allow to facilitate manipulation of data generated by bacmman.

    Parameters
    ----------
    path : str
        path to the folder containing the configuration file
    filter : str or callable
        filter on the name of the dataset (optional)

    Attributes
    ----------
    name : str
        dataset name
    object_class_names : list of str
        names of the object classes of the dataset. By default names are found in the configuration file, but can be renamed using the set_object_class_name method
    data : dict
        datatable for each object class
    path : str
        path to the folder containing the configuration file

    """
    def __init__(self, path:str, filter = None):
        self.path = path
        self.name = get_DS_name(path, filter)
        if self.name is not None: # inspect config file
            cf = join(path, self.name+"_config.json")
            with open(cf, errors='ignore') as f:
                try:
                    conf = json.load(f)
                    self.object_class_names = [c["name"] for c in conf["structures"]["list"]]
                    # ensure unicity of names
                    seen = set()
                    self.object_class_names = [_increment_name(x, seen) for x in self.object_class_names]
                except Exception as e:
                    print(f"Error trying to load configuration file: {f}")
                    raise e
        self.data = {}

    def set_object_class_name(self, old_name, new_name:str):
        """Modifies the name of an object class.

        Parameters
        ----------
        old_name : str or int
            either name or index of the old object class to modify
        new_name : str
            new name of the object class

        """
        old_name = self._get_object_class_index(old_name)
        assert old_name < len(self.object_class_names), f"invalid object class idx, should be <{len(self.object_class_names)}"
        if self.object_class_names[old_name]!=new_name:
            assert new_name not in self.object_class_names, "invalid new object_class name: already exists"
            self.object_class_names[old_name] = new_name

    def _get_object_class_name(self, object_class):
        if not isinstance(object_class, str):
            assert object_class<len(self.object_class_names), f"invalid object_class index, should be < {len(self.object_class_names)}"
            return self.object_class_names[object_class]
        else:
            return object_class

    def _get_object_class_index(self, object_class):
        if isinstance(object_class, str):
            assert object_class in self.object_class_names, f"invalid object_class name, should be in {self.object_class_names}"
            return self.object_class_names.index(object_class)
        else:
            return object_class

    def _get_data_file_path(self, object_class):
        return join(self.path, f"{self.name}_{self._get_object_class_index(object_class)}.csv")

    def _open_data(self, object_class, add_dataset_name_column=False):
        data = pd.read_csv(self._get_data_file_path(object_class), sep=';')
        if add_dataset_name_column:
            data["DatasetName"] = self.name
        return data

    def get_data(self, object_class):
        """Open measurement table of an object class as pandas dataframe.
           the file is supposed to be located in the dataset path.
           Note that when calling this method, the dataframe is stored in the dataset object. call _open_data to open the dataframe without storing it.

        Parameters
        ----------
        object_class : str or int
            either name or index of the object class

        Returns
        -------
        pandas dataframe
            Description of returned object.

        """
        object_class_name = self._get_object_class_name(object_class)
        if object_class_name not in self.data:
            self.data[object_class_name] = self._open_data(object_class)
        return self.data[object_class_name]

    def save_selection(self, selection, object_class, name:str, **kwargs):
        """Save a selection (subset of objects) to BACMMAN software.

        Parameters
        ----------
        object_class : str or int
            either name or index of the object class
        selection : pandas dataframe
            dataframe containing indices and positions of objects to include in the selection
        name : name of the selection
        **kwargs : dict
            extra arguments passed to saveAndOpenSelection method, such as port, python_proxy_port, address

        """
        object_class_idx = self._get_object_class_index(object_class)
        saveAndOpenSelection(selection, self.name, objectClassIdx=object_class_idx, selectionName=name, dsPath=self.path, **kwargs)

    def __str__(self):
        return f"{self.name} oc={self.object_class_names} path={self.path}"

class DatasetList(Dataset):
    def __init__(self, dataset_list:list = None, path:str = None, filter = None, object_class_name_mapping:dict = None):
        if dataset_list is not None:
            self.datasets = {d.name:d for d in dataset_list}
        elif path is not None:
            dataset_list = [Dataset(join(path, f), filter) for f in listdir(path) if isdir(join(path, f))]
            self.datasets = {d.name:d for d in dataset_list if d.name is not None}
            if object_class_name_mapping is not None:
                for d in self.datasets.values():
                    for i, n in enumerate(d.object_class_names):
                        if n in object_class_name_mapping:
                            d.object_class_names[i] = object_class_name_mapping[n]

        assert len(self.datasets)>1, "no datasets where found"
        # object_class names are the common object_class names between all object_class names
        self.object_class_names = None
        for d in self.datasets.values():
            if self.object_class_names is None:
                self.object_class_names = d.object_class_names
            else:
                inter = set(self.object_class_names).intersection(d.object_class_names)
                self.object_class_names = [n for n in self.object_class_names if n in inter]
        assert len(self.object_class_names)>0, f"no object_class names in common between datasets : {[str(d) for d in self.datasets.values()]}"
        self.data = {}

    def set_object_class_name(self, old_object_class, new_object_class_name):
        old_object_class_idx = self._get_object_class_index(old_object_class)
        old_object_class_name = self._get_object_class_name(old_object_class)
        assert old_object_class_idx < len(self.object_class_names), f"invalid object_class idx, should be <{len(self.object_class_names)}"
        self.object_class_names[old_object_class_idx] = new_object_class_name
        for d in self.datasets.values():
            d.set_object_class_name(old_object_class_name, new_object_class_name)

    def _open_data(self, object_class):
        object_class = self._get_object_class_name(object_class)
        return pd.concat([d._open_data(object_class, add_dataset_name_column=True) for d in self.datasets.values()]) #open_data to avoid storing the data in each dataset

    def save_selection(self, selection, object_class, name:str, dataset_column="DatasetName", **kwargs):
        object_class_idx = self._get_object_class_index(object_class)
        object_class_name = self._get_object_class_name(object_class)
        for ds_name, sel in selection.groupby(dataset_column):
            if ds_name not in self.datasets:
                print(f"Error: dataset {ds_name} not in datasets: {str(self)}")
            else:
                self.datasets[ds_name].save_selection(sel, object_class_name, name, **kwargs)

    def __str__(self):
        return f"{[n for n in self.datasets.keys()]} oc={self.object_class_names}"

# util functions
def get_DS_name(path, filter):
    # must contain a file ending by _config.json
    if isinstance(filter, str):
        _filter = lambda file_name:filter in file_name
    else:
        _filter = filter
    if isdir(path):
        for f in listdir(path):
            if f.endswith("_config.json") and (filter is None or _filter(f)) and isfile(join(path, f)):
                return f[:-12]
    return None

def _increment_name(name, seen):
    if name in seen:
        new_name = name+"_1"
        inc = 2
        while new_name in seen:
            new_name = f"{name}_{inc}"
            inc += 1
        name = new_name
    seen.add(name)
    return name
