"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Document = void 0;
const cfn = require("@aws-cdk/aws-cloudformation");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/core");
const statement = require("cdk-iam-floyd");
const yaml = require("js-yaml");
const path = require("path");
const resourceType = 'Custom::SSM-Document';
const ID = `CFN::Resource::${resourceType}`;
const createdByTag = 'CreatedByCfnCustomResource';
const cleanID = ID.replace(/:+/g, '-');
const lambdaTimeout = 3; // minutes
/**
 * An SSM document.
 *
 * @stability stable
 */
class Document extends cdk.Construct {
    /**
     * Defines a new SSM document.
     *
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * Name of the document.
         *
         * @stability stable
         */
        this.name = '';
        this.tags = new cdk.TagManager(cdk.TagType.MAP, 'Custom::SSM-Document');
        this.tags.setTag(createdByTag, ID);
        const stack = cdk.Stack.of(this).stackName;
        const fn = this.ensureLambda();
        const name = this.fixDocumentName(props.name);
        if (name.length < 3 || name.length > 128) {
            this.node.addError(`SSM Document name ${name} is invalid. The name must be between 3 and 128 characters.`);
            return;
        }
        let content = props.content;
        if (typeof content === 'string') {
            content = yaml.safeLoad(content);
        }
        const document = new cfn.CustomResource(this, `SSM-Document-${name}`, {
            provider: cfn.CustomResourceProvider.fromLambda(fn),
            resourceType: resourceType,
            properties: {
                updateDefaultVersion: props.updateDefaultVersion || true,
                name: name,
                content: content,
                documentType: props.documentType || 'Command',
                targetType: props.targetType || '/',
                StackName: stack,
                tags: cdk.Lazy.anyValue({
                    produce: () => this.tags.renderTags(),
                }),
            },
        });
        this.name = document.getAttString('Name');
    }
    ensureLambda() {
        const stack = cdk.Stack.of(this);
        const constructName = 'SSM-Document-Manager-Lambda';
        const existing = stack.node.tryFindChild(constructName);
        if (existing) {
            return existing;
        }
        const policy = new iam.ManagedPolicy(stack, 'SSM-Document-Manager-Policy', {
            managedPolicyName: `${stack.stackName}-${cleanID}`,
            description: `Used by Lambda ${cleanID}, which is a custom CFN resource, managing SSM documents`,
            statements: [
                new statement.Ssm().allow().toListDocuments().toListTagsForResource(),
                new statement.Ssm()
                    .allow()
                    .toCreateDocument()
                    .toAddTagsToResource()
                    .ifAwsRequestTag(createdByTag, ID),
                new statement.Ssm()
                    .allow()
                    .toDeleteDocument()
                    .toDescribeDocument()
                    .toGetDocument()
                    .toListDocumentVersions()
                    .toModifyDocumentPermission()
                    .toUpdateDocument()
                    .toUpdateDocumentDefaultVersion()
                    .toAddTagsToResource()
                    .toRemoveTagsFromResource()
                    .ifResourceTag(createdByTag, ID),
            ],
        });
        const role = new iam.Role(stack, 'SSM-Document-Manager-Role', {
            roleName: `${stack.stackName}-${cleanID}`,
            description: `Used by Lambda ${cleanID}, which is a custom CFN resource, managing SSM documents`,
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            managedPolicies: [
                policy,
                iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaBasicExecutionRole'),
            ],
        });
        const fn = new lambda.Function(stack, constructName, {
            functionName: `${stack.stackName}-${cleanID}`,
            role: role,
            description: 'Custom CFN resource: Manage SSM Documents',
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(path.join(__dirname, '../lambda/code.zip')),
            timeout: cdk.Duration.minutes(lambdaTimeout),
        });
        return fn;
    }
    fixDocumentName(name) {
        return name.replace(/[^a-zA-Z0-9_.-]+/g, '-');
    }
}
exports.Document = Document;
//# sourceMappingURL=data:application/json;base64,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