"use strict";
const assert_1 = require("@aws-cdk/assert");
const codebuild = require("@aws-cdk/aws-codebuild");
const codepipeline = require("@aws-cdk/aws-codepipeline");
const core_1 = require("@aws-cdk/core");
const cpactions = require("../../lib");
module.exports = {
    'GitHub source Action': {
        'exposes variables for other actions to consume'(test) {
            const stack = new core_1.Stack();
            const sourceOutput = new codepipeline.Artifact();
            const gitHubSourceAction = new cpactions.GitHubSourceAction({
                actionName: 'Source',
                owner: 'aws',
                repo: 'aws-cdk',
                output: sourceOutput,
                oauthToken: core_1.SecretValue.plainText('secret'),
            });
            new codepipeline.Pipeline(stack, 'Pipeline', {
                stages: [
                    {
                        stageName: 'Source',
                        actions: [gitHubSourceAction],
                    },
                    {
                        stageName: 'Build',
                        actions: [
                            new cpactions.CodeBuildAction({
                                actionName: 'Build',
                                project: new codebuild.PipelineProject(stack, 'MyProject'),
                                input: sourceOutput,
                                environmentVariables: {
                                    CommitUrl: { value: gitHubSourceAction.variables.commitUrl },
                                },
                            }),
                        ],
                    },
                ],
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
                "Stages": [
                    {
                        "Name": "Source",
                    },
                    {
                        "Name": "Build",
                        "Actions": [
                            {
                                "Name": "Build",
                                "Configuration": {
                                    "EnvironmentVariables": '[{"name":"CommitUrl","type":"PLAINTEXT","value":"#{Source_Source_NS.CommitUrl}"}]',
                                },
                            },
                        ],
                    },
                ],
            }));
            test.done();
        },
        'always renders the customer-supplied namespace, even if none of the variables are used'(test) {
            const stack = new core_1.Stack();
            const sourceOutput = new codepipeline.Artifact();
            new codepipeline.Pipeline(stack, 'Pipeline', {
                stages: [
                    {
                        stageName: 'Source',
                        actions: [
                            new cpactions.GitHubSourceAction({
                                actionName: 'Source',
                                owner: 'aws',
                                repo: 'aws-cdk',
                                output: sourceOutput,
                                oauthToken: core_1.SecretValue.plainText('secret'),
                                variablesNamespace: 'MyNamespace',
                            }),
                        ],
                    },
                    {
                        stageName: 'Build',
                        actions: [
                            new cpactions.CodeBuildAction({
                                actionName: 'Build',
                                project: new codebuild.PipelineProject(stack, 'MyProject'),
                                input: sourceOutput,
                            }),
                        ],
                    },
                ],
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
                "Stages": [
                    {
                        "Name": "Source",
                        "Actions": [
                            {
                                "Name": "Source",
                                "Namespace": "MyNamespace",
                            },
                        ],
                    },
                    {},
                ],
            }));
            test.done();
        },
        'fails if a variable from an action without a namespace set that is not part of a pipeline is referenced'(test) {
            const stack = new core_1.Stack();
            const unusedSourceAction = new cpactions.GitHubSourceAction({
                actionName: 'Source2',
                owner: 'aws',
                repo: 'aws-cdk',
                output: new codepipeline.Artifact(),
                oauthToken: core_1.SecretValue.plainText('secret'),
            });
            const sourceOutput = new codepipeline.Artifact();
            new codepipeline.Pipeline(stack, 'Pipeline', {
                stages: [
                    {
                        stageName: 'Source',
                        actions: [new cpactions.GitHubSourceAction({
                                actionName: 'Source1',
                                owner: 'aws',
                                repo: 'aws-cdk',
                                output: sourceOutput,
                                oauthToken: core_1.SecretValue.plainText('secret'),
                            })],
                    },
                    {
                        stageName: 'Build',
                        actions: [
                            new cpactions.CodeBuildAction({
                                actionName: 'Build',
                                project: new codebuild.PipelineProject(stack, 'MyProject'),
                                input: sourceOutput,
                                environmentVariables: {
                                    'VAR1': { value: unusedSourceAction.variables.authorDate },
                                },
                            }),
                        ],
                    },
                ],
            });
            test.throws(() => {
                assert_1.SynthUtils.synthesize(stack);
            }, /Cannot reference variables of action 'Source2', as that action was never added to a pipeline/);
            test.done();
        },
        'fails if a variable from an action with a namespace set that is not part of a pipeline is referenced'(test) {
            const stack = new core_1.Stack();
            const unusedSourceAction = new cpactions.GitHubSourceAction({
                actionName: 'Source2',
                owner: 'aws',
                repo: 'aws-cdk',
                output: new codepipeline.Artifact(),
                oauthToken: core_1.SecretValue.plainText('secret'),
                variablesNamespace: 'MyNamespace',
            });
            const sourceOutput = new codepipeline.Artifact();
            new codepipeline.Pipeline(stack, 'Pipeline', {
                stages: [
                    {
                        stageName: 'Source',
                        actions: [new cpactions.GitHubSourceAction({
                                actionName: 'Source1',
                                owner: 'aws',
                                repo: 'aws-cdk',
                                output: sourceOutput,
                                oauthToken: core_1.SecretValue.plainText('secret'),
                            })],
                    },
                    {
                        stageName: 'Build',
                        actions: [
                            new cpactions.CodeBuildAction({
                                actionName: 'Build',
                                project: new codebuild.PipelineProject(stack, 'MyProject'),
                                input: sourceOutput,
                                environmentVariables: {
                                    'VAR1': { value: unusedSourceAction.variables.authorDate },
                                },
                            }),
                        ],
                    },
                ],
            });
            test.throws(() => {
                assert_1.SynthUtils.synthesize(stack);
            }, /Cannot reference variables of action 'Source2', as that action was never added to a pipeline/);
            test.done();
        },
    },
};
//# sourceMappingURL=data:application/json;base64,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