"use strict";
const assert_1 = require("@aws-cdk/assert");
const codedeploy = require("@aws-cdk/aws-codedeploy");
const codepipeline = require("@aws-cdk/aws-codepipeline");
const cdk = require("@aws-cdk/core");
const cpactions = require("../../lib");
function addEcsDeploymentGroup(stack) {
    return codedeploy.EcsDeploymentGroup.fromEcsDeploymentGroupAttributes(stack, 'EDG', {
        application: codedeploy.EcsApplication.fromEcsApplicationName(stack, 'EA', 'MyApplication'),
        deploymentGroupName: 'MyDeploymentGroup'
    });
}
function addCodeDeployECSCodePipeline(stack, props) {
    new codepipeline.Pipeline(stack, 'Pipeline', {
        stages: [
            {
                stageName: 'Source',
                actions: [
                    new cpactions.GitHubSourceAction({
                        actionName: 'GitHub',
                        output: props.taskDefinitionTemplateInput || props.taskDefinitionTemplateFile.artifact,
                        oauthToken: cdk.SecretValue.plainText('secret'),
                        owner: 'awslabs',
                        repo: 'aws-cdk',
                    }),
                    new cpactions.GitHubSourceAction({
                        actionName: 'GitHub2',
                        output: props.appSpecTemplateInput || props.appSpecTemplateFile.artifact,
                        oauthToken: cdk.SecretValue.plainText('secret'),
                        owner: 'awslabs',
                        repo: 'aws-cdk-2',
                    }),
                ],
            },
            {
                stageName: 'Invoke',
                actions: [
                    new cpactions.CodeDeployEcsDeployAction(props),
                ],
            },
        ],
    });
}
module.exports = {
    'CodeDeploy ECS Deploy Action': {
        'throws an exception if more than 4 container image inputs are provided'(test) {
            const stack = new cdk.Stack();
            const deploymentGroup = addEcsDeploymentGroup(stack);
            const artifact = new codepipeline.Artifact('Artifact');
            const containerImageInputs = [];
            for (let i = 0; i < 5; i++) {
                containerImageInputs.push({
                    input: artifact
                });
            }
            test.throws(() => {
                new cpactions.CodeDeployEcsDeployAction({
                    actionName: 'DeployToECS',
                    deploymentGroup,
                    taskDefinitionTemplateInput: artifact,
                    appSpecTemplateInput: artifact,
                    containerImageInputs,
                });
            }, /Action cannot have more than 4 container image inputs, got: 5/);
            test.done();
        },
        'throws an exception if both appspec artifact input and file are specified'(test) {
            const stack = new cdk.Stack();
            const deploymentGroup = addEcsDeploymentGroup(stack);
            const artifact = new codepipeline.Artifact('Artifact');
            const artifactPath = new codepipeline.ArtifactPath(artifact, 'hello');
            test.throws(() => {
                new cpactions.CodeDeployEcsDeployAction({
                    actionName: 'DeployToECS',
                    deploymentGroup,
                    taskDefinitionTemplateInput: artifact,
                    appSpecTemplateInput: artifact,
                    appSpecTemplateFile: artifactPath,
                });
            }, /Exactly one of 'appSpecTemplateInput' or 'appSpecTemplateFile' can be provided in the ECS CodeDeploy Action/);
            test.done();
        },
        'throws an exception if neither appspec artifact input nor file are specified'(test) {
            const stack = new cdk.Stack();
            const deploymentGroup = addEcsDeploymentGroup(stack);
            const artifact = new codepipeline.Artifact('Artifact');
            test.throws(() => {
                new cpactions.CodeDeployEcsDeployAction({
                    actionName: 'DeployToECS',
                    deploymentGroup,
                    taskDefinitionTemplateInput: artifact,
                });
            }, /Specifying one of 'appSpecTemplateInput' or 'appSpecTemplateFile' is required for the ECS CodeDeploy Action/);
            test.done();
        },
        'throws an exception if both task definition artifact input and file are specified'(test) {
            const stack = new cdk.Stack();
            const deploymentGroup = addEcsDeploymentGroup(stack);
            const artifact = new codepipeline.Artifact('Artifact');
            const artifactPath = new codepipeline.ArtifactPath(artifact, 'hello');
            test.throws(() => {
                new cpactions.CodeDeployEcsDeployAction({
                    actionName: 'DeployToECS',
                    deploymentGroup,
                    taskDefinitionTemplateInput: artifact,
                    taskDefinitionTemplateFile: artifactPath,
                    appSpecTemplateInput: artifact,
                });
            }, /Exactly one of 'taskDefinitionTemplateInput' or 'taskDefinitionTemplateFile' can be provided in the ECS CodeDeploy Action/);
            test.done();
        },
        'throws an exception if neither task definition artifact input nor file are specified'(test) {
            const stack = new cdk.Stack();
            const deploymentGroup = addEcsDeploymentGroup(stack);
            const artifact = new codepipeline.Artifact('Artifact');
            test.throws(() => {
                new cpactions.CodeDeployEcsDeployAction({
                    actionName: 'DeployToECS',
                    deploymentGroup,
                    appSpecTemplateInput: artifact,
                });
            }, /Specifying one of 'taskDefinitionTemplateInput' or 'taskDefinitionTemplateFile' is required for the ECS CodeDeploy Action/);
            test.done();
        },
        'defaults task definition and appspec template paths'(test) {
            const stack = new cdk.Stack();
            const deploymentGroup = addEcsDeploymentGroup(stack);
            addCodeDeployECSCodePipeline(stack, {
                actionName: 'DeployToECS',
                deploymentGroup,
                taskDefinitionTemplateInput: new codepipeline.Artifact('TaskDefArtifact'),
                appSpecTemplateInput: new codepipeline.Artifact('AppSpecArtifact')
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
                Stages: [
                    {},
                    {
                        Actions: [
                            {
                                Configuration: {
                                    ApplicationName: 'MyApplication',
                                    DeploymentGroupName: 'MyDeploymentGroup',
                                    TaskDefinitionTemplateArtifact: 'TaskDefArtifact',
                                    AppSpecTemplateArtifact: 'AppSpecArtifact',
                                    TaskDefinitionTemplatePath: 'taskdef.json',
                                    AppSpecTemplatePath: 'appspec.yaml'
                                },
                                InputArtifacts: [
                                    {
                                        Name: 'TaskDefArtifact'
                                    },
                                    {
                                        Name: 'AppSpecArtifact'
                                    }
                                ]
                            },
                        ],
                    },
                ],
            }));
            test.done();
        },
        'defaults task definition placeholder string'(test) {
            const stack = new cdk.Stack();
            const deploymentGroup = addEcsDeploymentGroup(stack);
            const artifact1 = new codepipeline.Artifact('Artifact1');
            const artifact2 = new codepipeline.Artifact('Artifact2');
            addCodeDeployECSCodePipeline(stack, {
                actionName: 'DeployToECS',
                deploymentGroup,
                taskDefinitionTemplateFile: new codepipeline.ArtifactPath(artifact1, 'task-definition.json'),
                appSpecTemplateFile: new codepipeline.ArtifactPath(artifact2, 'appspec-test.yaml'),
                containerImageInputs: [
                    {
                        input: artifact1
                    },
                    {
                        input: artifact2
                    }
                ]
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
                Stages: [
                    {},
                    {
                        Actions: [
                            {
                                Configuration: {
                                    ApplicationName: 'MyApplication',
                                    DeploymentGroupName: 'MyDeploymentGroup',
                                    TaskDefinitionTemplateArtifact: 'Artifact1',
                                    AppSpecTemplateArtifact: 'Artifact2',
                                    TaskDefinitionTemplatePath: 'task-definition.json',
                                    AppSpecTemplatePath: 'appspec-test.yaml',
                                    Image1ArtifactName: 'Artifact1',
                                    Image1ContainerName: 'IMAGE',
                                    Image2ArtifactName: 'Artifact2',
                                    Image2ContainerName: 'IMAGE'
                                },
                                InputArtifacts: [
                                    {
                                        Name: 'Artifact1'
                                    },
                                    {
                                        Name: 'Artifact2'
                                    }
                                ]
                            },
                        ],
                    },
                ],
            }));
            test.done();
        },
    },
};
//# sourceMappingURL=data:application/json;base64,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