"use strict";
const codepipeline = require("@aws-cdk/aws-codepipeline");
const iam = require("@aws-cdk/aws-iam");
const s3 = require("@aws-cdk/aws-s3");
const cdk = require("@aws-cdk/core");
const _ = require("lodash");
const nodeunit = require("nodeunit");
const cpactions = require("../../lib");
function _assertActionMatches(test, stack, actions, provider, category, configuration) {
    const configurationStr = configuration
        ? `, configuration including ${JSON.stringify(stack.resolve(configuration), null, 2)}`
        : '';
    const actionsStr = JSON.stringify(actions.map(a => ({ owner: a.actionProperties.owner, provider: a.actionProperties.provider,
        category: a.actionProperties.category, configuration: stack.resolve(a.actionConfig.configuration)
    })), null, 2);
    test.ok(_hasAction(stack, actions, provider, category, configuration), `Expected to find an action with provider ${provider}, category ${category}${configurationStr}, but found ${actionsStr}`);
}
function _hasAction(stack, actions, provider, category, configuration) {
    for (const action of actions) {
        if (action.actionProperties.provider !== provider) {
            continue;
        }
        if (action.actionProperties.category !== category) {
            continue;
        }
        if (configuration && !action.actionConfig.configuration) {
            continue;
        }
        if (configuration) {
            for (const key of Object.keys(configuration)) {
                if (!_.isEqual(stack.resolve(action.actionConfig.configuration[key]), stack.resolve(configuration[key]))) {
                    continue;
                }
            }
        }
        return true;
    }
    return false;
}
function _assertPermissionGranted(test, stack, statements, action, resource, conditions) {
    const conditionStr = conditions
        ? ` with condition(s) ${JSON.stringify(stack.resolve(conditions))}`
        : '';
    const resolvedStatements = stack.resolve(statements.map(s => s.toStatementJson()));
    const statementsStr = JSON.stringify(resolvedStatements, null, 2);
    test.ok(_grantsPermission(stack, resolvedStatements, action, resource, conditions), `Expected to find a statement granting ${action} on ${JSON.stringify(stack.resolve(resource))}${conditionStr}, found:\n${statementsStr}`);
}
function _grantsPermission(stack, statements, action, resource, conditions) {
    for (const statement of statements.filter(s => s.Effect === 'Allow')) {
        if (!_isOrContains(stack, statement.Action, action)) {
            continue;
        }
        if (!_isOrContains(stack, statement.Resource, resource)) {
            continue;
        }
        if (conditions && !_isOrContains(stack, statement.Condition, conditions)) {
            continue;
        }
        return true;
    }
    return false;
}
function _isOrContains(stack, entity, value) {
    const resolvedValue = stack.resolve(value);
    const resolvedEntity = stack.resolve(entity);
    if (_.isEqual(resolvedEntity, resolvedValue)) {
        return true;
    }
    if (!Array.isArray(resolvedEntity)) {
        return false;
    }
    for (const tested of entity) {
        if (_.isEqual(tested, resolvedValue)) {
            return true;
        }
    }
    return false;
}
function _stackArn(stackName, scope) {
    return cdk.Stack.of(scope).formatArn({
        service: 'cloudformation',
        resource: 'stack',
        resourceName: `${stackName}/*`,
    });
}
class PipelineDouble extends cdk.Resource {
    constructor(scope, id, { pipelineName, role }) {
        super(scope, id);
        this.pipelineName = pipelineName || 'TestPipeline';
        this.pipelineArn = cdk.Stack.of(this).formatArn({ service: 'codepipeline', resource: 'pipeline', resourceName: this.pipelineName });
        this.role = role;
        this.artifactBucket = new BucketDouble(scope, 'BucketDouble');
    }
    onEvent(_id, _options) {
        throw new Error("Method not implemented.");
    }
    onStateChange(_id, _options) {
        throw new Error("Method not implemented.");
    }
}
class FullAction {
    constructor(actionProperties, actionConfig) {
        this.actionProperties = actionProperties;
        this.actionConfig = actionConfig;
        // empty
    }
}
class StageDouble {
    constructor({ name, pipeline, actions }) {
        this.actions = [];
        this.stageName = name || 'TestStage';
        this.pipeline = pipeline;
        const stageParent = new cdk.Construct(pipeline, this.stageName);
        const fullActions = new Array();
        for (const action of actions) {
            const actionParent = new cdk.Construct(stageParent, action.actionProperties.actionName);
            fullActions.push(new FullAction(action.actionProperties, action.bind(actionParent, this, {
                role: pipeline.role,
                bucket: pipeline.artifactBucket,
            })));
        }
        this.fullActions = fullActions;
    }
    get node() {
        throw new Error('StageDouble is not a real construct');
    }
    addAction(_action) {
        throw new Error('addAction() is not supported on StageDouble');
    }
    onStateChange(_name, _target, _options) {
        throw new Error('onStateChange() is not supported on StageDouble');
    }
}
class RoleDouble extends iam.Role {
    constructor(scope, id, props = { assumedBy: new iam.ServicePrincipal('test') }) {
        super(scope, id, props);
        this.statements = new Array();
    }
    addToPolicy(statement) {
        super.addToPolicy(statement);
        this.statements.push(statement);
        return true;
    }
}
class BucketDouble extends s3.Bucket {
    grantRead(identity, _objectsKeyPattern = '*') {
        return iam.Grant.drop(identity, '');
    }
    grantWrite(identity, _objectsKeyPattern = '*') {
        return iam.Grant.drop(identity, '');
    }
    grantReadWrite(identity, _objectsKeyPattern = '*') {
        return iam.Grant.drop(identity, '');
    }
}
module.exports = nodeunit.testCase({
    'CreateReplaceChangeSet': {
        'works'(test) {
            const app = new cdk.App();
            const stack = new cdk.Stack(app, 'Stack');
            const pipelineRole = new RoleDouble(stack, 'PipelineRole');
            const artifact = new codepipeline.Artifact('TestArtifact');
            const action = new cpactions.CloudFormationCreateReplaceChangeSetAction({
                actionName: 'Action',
                changeSetName: 'MyChangeSet',
                stackName: 'MyStack',
                templatePath: artifact.atPath('path/to/file'),
                adminPermissions: false,
            });
            const stage = new StageDouble({
                pipeline: new PipelineDouble(stack, 'Pipeline', { role: pipelineRole }),
                actions: [action],
            });
            cdk.ConstructNode.prepare(stack.node);
            _assertPermissionGranted(test, stack, pipelineRole.statements, 'iam:PassRole', action.deploymentRole.roleArn);
            const stackArn = _stackArn('MyStack', stack);
            const changeSetCondition = { StringEqualsIfExists: { 'cloudformation:ChangeSetName': 'MyChangeSet' } };
            _assertPermissionGranted(test, stack, pipelineRole.statements, 'cloudformation:DescribeStacks', stackArn, changeSetCondition);
            _assertPermissionGranted(test, stack, pipelineRole.statements, 'cloudformation:DescribeChangeSet', stackArn, changeSetCondition);
            _assertPermissionGranted(test, stack, pipelineRole.statements, 'cloudformation:CreateChangeSet', stackArn, changeSetCondition);
            _assertPermissionGranted(test, stack, pipelineRole.statements, 'cloudformation:DeleteChangeSet', stackArn, changeSetCondition);
            // TODO: revert "as any" once we move all actions into a single package.
            test.deepEqual(stage.fullActions[0].actionProperties.inputs, [artifact], 'The input was correctly registered');
            _assertActionMatches(test, stack, stage.fullActions, 'CloudFormation', 'Deploy', {
                ActionMode: 'CHANGE_SET_CREATE_REPLACE',
                StackName: 'MyStack',
                ChangeSetName: 'MyChangeSet'
            });
            test.done();
        },
        'uses a single permission statement if the same ChangeSet name is used'(test) {
            const stack = new cdk.Stack();
            const pipelineRole = new RoleDouble(stack, 'PipelineRole');
            const artifact = new codepipeline.Artifact('TestArtifact');
            new StageDouble({
                pipeline: new PipelineDouble(stack, 'Pipeline', { role: pipelineRole }),
                actions: [
                    new cpactions.CloudFormationCreateReplaceChangeSetAction({
                        actionName: 'ActionA',
                        changeSetName: 'MyChangeSet',
                        stackName: 'StackA',
                        adminPermissions: false,
                        templatePath: artifact.atPath('path/to/file')
                    }),
                    new cpactions.CloudFormationCreateReplaceChangeSetAction({
                        actionName: 'ActionB',
                        changeSetName: 'MyChangeSet',
                        stackName: 'StackB',
                        adminPermissions: false,
                        templatePath: artifact.atPath('path/to/other/file')
                    }),
                ],
            });
            test.deepEqual(stack.resolve(pipelineRole.statements.map(s => s.toStatementJson())), [
                {
                    Action: 'iam:PassRole',
                    Effect: 'Allow',
                    Resource: [
                        { 'Fn::GetAtt': ['PipelineTestStageActionARole9283FBE3', 'Arn'] },
                        { 'Fn::GetAtt': ['PipelineTestStageActionBRoleCABC8FA5', 'Arn'] }
                    ],
                },
                {
                    Action: [
                        'cloudformation:CreateChangeSet',
                        'cloudformation:DeleteChangeSet',
                        'cloudformation:DescribeChangeSet',
                        'cloudformation:DescribeStacks'
                    ],
                    Condition: { StringEqualsIfExists: { 'cloudformation:ChangeSetName': 'MyChangeSet' } },
                    Effect: 'Allow',
                    Resource: [
                        // tslint:disable-next-line:max-line-length
                        { 'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':cloudformation:', { Ref: 'AWS::Region' }, ':', { Ref: 'AWS::AccountId' }, ':stack/StackA/*']] },
                        // tslint:disable-next-line:max-line-length
                        { 'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':cloudformation:', { Ref: 'AWS::Region' }, ':', { Ref: 'AWS::AccountId' }, ':stack/StackB/*']] }
                    ],
                }
            ]);
            test.done();
        }
    },
    'ExecuteChangeSet': {
        'works'(test) {
            const stack = new cdk.Stack();
            const pipelineRole = new RoleDouble(stack, 'PipelineRole');
            const stage = new StageDouble({
                pipeline: new PipelineDouble(stack, 'Pipeline', { role: pipelineRole }),
                actions: [
                    new cpactions.CloudFormationExecuteChangeSetAction({
                        actionName: 'Action',
                        changeSetName: 'MyChangeSet',
                        stackName: 'MyStack',
                    }),
                ],
            });
            const stackArn = _stackArn('MyStack', stack);
            _assertPermissionGranted(test, stack, pipelineRole.statements, 'cloudformation:ExecuteChangeSet', stackArn, { StringEqualsIfExists: { 'cloudformation:ChangeSetName': 'MyChangeSet' } });
            _assertActionMatches(test, stack, stage.fullActions, 'CloudFormation', 'Deploy', {
                ActionMode: 'CHANGE_SET_EXECUTE',
                StackName: 'MyStack',
                ChangeSetName: 'MyChangeSet'
            });
            test.done();
        },
        'uses a single permission statement if the same ChangeSet name is used'(test) {
            const stack = new cdk.Stack();
            const pipelineRole = new RoleDouble(stack, 'PipelineRole');
            new StageDouble({
                pipeline: new PipelineDouble(stack, 'Pipeline', { role: pipelineRole }),
                actions: [
                    new cpactions.CloudFormationExecuteChangeSetAction({
                        actionName: 'ActionA',
                        changeSetName: 'MyChangeSet',
                        stackName: 'StackA',
                    }),
                    new cpactions.CloudFormationExecuteChangeSetAction({
                        actionName: 'ActionB',
                        changeSetName: 'MyChangeSet',
                        stackName: 'StackB',
                    }),
                ],
            });
            test.deepEqual(stack.resolve(pipelineRole.statements.map(s => s.toStatementJson())), [
                {
                    Action: [
                        'cloudformation:DescribeChangeSet',
                        'cloudformation:DescribeStacks',
                        'cloudformation:ExecuteChangeSet',
                    ],
                    Condition: { StringEqualsIfExists: { 'cloudformation:ChangeSetName': 'MyChangeSet' } },
                    Effect: 'Allow',
                    Resource: [
                        // tslint:disable-next-line:max-line-length
                        { 'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':cloudformation:', { Ref: 'AWS::Region' }, ':', { Ref: 'AWS::AccountId' }, ':stack/StackA/*']] },
                        // tslint:disable-next-line:max-line-length
                        { 'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':cloudformation:', { Ref: 'AWS::Region' }, ':', { Ref: 'AWS::AccountId' }, ':stack/StackB/*']] }
                    ],
                }
            ]);
            test.done();
        }
    },
    'the CreateUpdateStack Action sets the DescribeStack*, Create/Update/DeleteStack & PassRole permissions'(test) {
        const stack = new cdk.Stack();
        const pipelineRole = new RoleDouble(stack, 'PipelineRole');
        const action = new cpactions.CloudFormationCreateUpdateStackAction({
            actionName: 'Action',
            templatePath: new codepipeline.Artifact('TestArtifact').atPath('some/file'),
            stackName: 'MyStack',
            adminPermissions: false,
            replaceOnFailure: true,
        });
        new StageDouble({
            pipeline: new PipelineDouble(stack, 'Pipeline', { role: pipelineRole }),
            actions: [action],
        });
        const stackArn = _stackArn('MyStack', stack);
        _assertPermissionGranted(test, stack, pipelineRole.statements, 'cloudformation:DescribeStack*', stackArn);
        _assertPermissionGranted(test, stack, pipelineRole.statements, 'cloudformation:CreateStack', stackArn);
        _assertPermissionGranted(test, stack, pipelineRole.statements, 'cloudformation:UpdateStack', stackArn);
        _assertPermissionGranted(test, stack, pipelineRole.statements, 'cloudformation:DeleteStack', stackArn);
        _assertPermissionGranted(test, stack, pipelineRole.statements, 'iam:PassRole', action.deploymentRole.roleArn);
        test.done();
    },
    'the DeleteStack Action sets the DescribeStack*, DeleteStack & PassRole permissions'(test) {
        const stack = new cdk.Stack();
        const pipelineRole = new RoleDouble(stack, 'PipelineRole');
        const action = new cpactions.CloudFormationDeleteStackAction({
            actionName: 'Action',
            adminPermissions: false,
            stackName: 'MyStack',
        });
        new StageDouble({
            pipeline: new PipelineDouble(stack, 'Pipeline', { role: pipelineRole }),
            actions: [action],
        });
        const stackArn = _stackArn('MyStack', stack);
        _assertPermissionGranted(test, stack, pipelineRole.statements, 'cloudformation:DescribeStack*', stackArn);
        _assertPermissionGranted(test, stack, pipelineRole.statements, 'cloudformation:DeleteStack', stackArn);
        _assertPermissionGranted(test, stack, pipelineRole.statements, 'iam:PassRole', action.deploymentRole.roleArn);
        test.done();
    },
});
//# sourceMappingURL=data:application/json;base64,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