"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const codepipeline = require("@aws-cdk/aws-codepipeline");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const action_1 = require("../action");
/**
 * CodePipeline invoke Action that is provided by an AWS Lambda function.
 *
 * @see https://docs.aws.amazon.com/codepipeline/latest/userguide/actions-invoke-lambda-function.html
 */
class LambdaInvokeAction extends action_1.Action {
    constructor(props) {
        super({
            ...props,
            resource: props.lambda,
            category: codepipeline.ActionCategory.INVOKE,
            provider: 'Lambda',
            artifactBounds: {
                minInputs: 0,
                maxInputs: 5,
                minOutputs: 0,
                maxOutputs: 5,
            },
        });
        this.props = props;
    }
    /**
     * Reference a CodePipeline variable defined by the Lambda function this action points to.
     * Variables in Lambda invoke actions are defined by calling the PutJobSuccessResult CodePipeline API call
     * with the 'outputVariables' property filled.
     *
     * @param variableName the name of the variable to reference.
     *   A variable by this name must be present in the 'outputVariables' section of the PutJobSuccessResult
     *   request that the Lambda function calls when the action is invoked
     *
     * @see https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_PutJobSuccessResult.html
     */
    variable(variableName) {
        return this.variableExpression(variableName);
    }
    bound(scope, _stage, options) {
        // allow pipeline to list functions
        options.role.addToPolicy(new iam.PolicyStatement({
            actions: ['lambda:ListFunctions'],
            resources: ['*']
        }));
        // allow pipeline to invoke this lambda functionn
        options.role.addToPolicy(new iam.PolicyStatement({
            actions: ['lambda:InvokeFunction'],
            resources: [this.props.lambda.functionArn]
        }));
        // allow the Role access to the Bucket, if there are any inputs/outputs
        if ((this.actionProperties.inputs || []).length > 0) {
            options.bucket.grantRead(options.role);
        }
        if ((this.actionProperties.outputs || []).length > 0) {
            options.bucket.grantWrite(options.role);
        }
        // allow lambda to put job results for this pipeline
        // CodePipeline requires this to be granted to '*'
        // (the Pipeline ARN will not be enough)
        this.props.lambda.addToRolePolicy(new iam.PolicyStatement({
            resources: ['*'],
            actions: ['codepipeline:PutJobSuccessResult', 'codepipeline:PutJobFailureResult']
        }));
        return {
            configuration: {
                FunctionName: this.props.lambda.functionName,
                UserParameters: core_1.Stack.of(scope).toJsonString(this.props.userParameters),
            },
        };
    }
}
exports.LambdaInvokeAction = LambdaInvokeAction;
//# sourceMappingURL=data:application/json;base64,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