"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const codepipeline = require("@aws-cdk/aws-codepipeline");
const action_1 = require("../action");
const jenkins_provider_1 = require("./jenkins-provider");
/**
 * The type of the Jenkins Action that determines its CodePipeline Category -
 * Build, or Test.
 * Note that a Jenkins provider, even if it has the same name,
 * must be separately registered for each type.
 */
var JenkinsActionType;
(function (JenkinsActionType) {
    /**
     * The Action will have the Build Category.
     */
    JenkinsActionType[JenkinsActionType["BUILD"] = 0] = "BUILD";
    /**
     * The Action will have the Test Category.
     */
    JenkinsActionType[JenkinsActionType["TEST"] = 1] = "TEST";
})(JenkinsActionType = exports.JenkinsActionType || (exports.JenkinsActionType = {}));
/**
 * Jenkins build CodePipeline Action.
 *
 * @see https://docs.aws.amazon.com/codepipeline/latest/userguide/tutorials-four-stage-pipeline.html
 */
class JenkinsAction extends action_1.Action {
    constructor(props) {
        super({
            ...props,
            category: props.type === JenkinsActionType.BUILD
                ? codepipeline.ActionCategory.BUILD
                : codepipeline.ActionCategory.TEST,
            provider: props.jenkinsProvider.providerName,
            owner: 'Custom',
            artifactBounds: jenkins_provider_1.jenkinsArtifactsBounds,
            version: props.jenkinsProvider.version,
        });
        this.props = props;
    }
    bound(_scope, _stage, _options) {
        if (this.actionProperties.category === codepipeline.ActionCategory.BUILD) {
            this.props.jenkinsProvider._registerBuildProvider();
        }
        else {
            this.props.jenkinsProvider._registerTestProvider();
        }
        return {
            configuration: {
                ProjectName: this.props.projectName,
            },
        };
    }
}
exports.JenkinsAction = JenkinsAction;
//# sourceMappingURL=data:application/json;base64,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