"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cloudformation = require("@aws-cdk/aws-cloudformation");
const codepipeline = require("@aws-cdk/aws-codepipeline");
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const action_1 = require("../action");
/**
 * Base class for Actions that execute CloudFormation
 */
class CloudFormationAction extends action_1.Action {
    constructor(props, inputs) {
        super({
            ...props,
            provider: 'CloudFormation',
            category: codepipeline.ActionCategory.DEPLOY,
            artifactBounds: {
                minInputs: 0,
                maxInputs: 10,
                minOutputs: 0,
                maxOutputs: 1,
            },
            inputs,
            outputs: props.outputFileName
                ? [props.output || new codepipeline.Artifact(`${props.actionName}_${props.stackName}_Artifact`)]
                : undefined,
        });
        this.props = props;
    }
    bound(_scope, _stage, options) {
        const singletonPolicy = SingletonPolicy.forRole(options.role);
        if ((this.actionProperties.outputs || []).length > 0) {
            options.bucket.grantReadWrite(singletonPolicy);
        }
        else if ((this.actionProperties.inputs || []).length > 0) {
            options.bucket.grantRead(singletonPolicy);
        }
        return {
            configuration: {
                StackName: this.props.stackName,
                OutputFileName: this.props.outputFileName,
            },
        };
    }
}
/**
 * CodePipeline action to execute a prepared change set.
 */
class CloudFormationExecuteChangeSetAction extends CloudFormationAction {
    constructor(props) {
        super(props, undefined);
        this.props2 = props;
    }
    bound(scope, stage, options) {
        SingletonPolicy.forRole(options.role).grantExecuteChangeSet(this.props2);
        const actionConfig = super.bound(scope, stage, options);
        return {
            ...actionConfig,
            configuration: {
                ...actionConfig.configuration,
                ActionMode: 'CHANGE_SET_EXECUTE',
                ChangeSetName: this.props2.changeSetName,
            },
        };
    }
}
exports.CloudFormationExecuteChangeSetAction = CloudFormationExecuteChangeSetAction;
// tslint:enable:max-line-length
/**
 * Base class for all CloudFormation actions that execute or stage deployments.
 */
class CloudFormationDeployAction extends CloudFormationAction {
    constructor(props, inputs) {
        super(props, (props.extraInputs || []).concat(inputs || []));
        this.props2 = props;
    }
    /**
     * Add statement to the service role assumed by CloudFormation while executing this action.
     */
    addToDeploymentRolePolicy(statement) {
        return this.getDeploymentRole('method addToRolePolicy()').addToPolicy(statement);
    }
    get deploymentRole() {
        return this.getDeploymentRole('property role()');
    }
    bound(scope, stage, options) {
        if (this.props2.deploymentRole) {
            this._deploymentRole = this.props2.deploymentRole;
        }
        else {
            const roleStack = cdk.Stack.of(options.role);
            const pipelineStack = cdk.Stack.of(scope);
            if (roleStack.account !== pipelineStack.account) {
                // pass role is not allowed for cross-account access - so,
                // create the deployment Role in the other account!
                this._deploymentRole = new iam.Role(roleStack, `${stage.pipeline.node.uniqueId}-${stage.stageName}-${this.actionProperties.actionName}-DeploymentRole`, {
                    assumedBy: new iam.ServicePrincipal('cloudformation.amazonaws.com'),
                    roleName: cdk.PhysicalName.GENERATE_IF_NEEDED,
                });
            }
            else {
                this._deploymentRole = new iam.Role(scope, 'Role', {
                    assumedBy: new iam.ServicePrincipal('cloudformation.amazonaws.com')
                });
            }
            // the deployment role might need read access to the pipeline's bucket
            // (for example, if it's deploying a Lambda function),
            // and even if it has admin permissions, it won't be enough,
            // as it needs to be added to the key's resource policy
            // (and the bucket's, if the access is cross-account)
            options.bucket.grantRead(this._deploymentRole);
            if (this.props2.adminPermissions) {
                this._deploymentRole.addToPolicy(new iam.PolicyStatement({
                    actions: ['*'],
                    resources: ['*'],
                }));
            }
        }
        SingletonPolicy.forRole(options.role).grantPassRole(this._deploymentRole);
        const capabilities = this.props2.adminPermissions && this.props2.capabilities === undefined
            ? [cloudformation.CloudFormationCapabilities.NAMED_IAM]
            : this.props2.capabilities;
        const actionConfig = super.bound(scope, stage, options);
        return {
            ...actionConfig,
            configuration: {
                ...actionConfig.configuration,
                // None evaluates to empty string which is falsey and results in undefined
                Capabilities: parseCapabilities(capabilities),
                RoleArn: this.deploymentRole.roleArn,
                ParameterOverrides: cdk.Stack.of(scope).toJsonString(this.props2.parameterOverrides),
                TemplateConfiguration: this.props2.templateConfiguration
                    ? this.props2.templateConfiguration.location
                    : undefined,
                StackName: this.props2.stackName,
            },
        };
    }
    getDeploymentRole(member) {
        if (this._deploymentRole) {
            return this._deploymentRole;
        }
        else {
            throw new Error(`Cannot use the ${member} before the Action has been added to a Pipeline`);
        }
    }
}
/**
 * CodePipeline action to prepare a change set.
 *
 * Creates the change set if it doesn't exist based on the stack name and template that you submit.
 * If the change set exists, AWS CloudFormation deletes it, and then creates a new one.
 */
class CloudFormationCreateReplaceChangeSetAction extends CloudFormationDeployAction {
    constructor(props) {
        super(props, props.templateConfiguration
            ? [props.templatePath.artifact, props.templateConfiguration.artifact]
            : [props.templatePath.artifact]);
        this.props3 = props;
    }
    bound(scope, stage, options) {
        // the super call order is to preserve the existing order of statements in policies
        const actionConfig = super.bound(scope, stage, options);
        SingletonPolicy.forRole(options.role).grantCreateReplaceChangeSet(this.props3);
        return {
            ...actionConfig,
            configuration: {
                ...actionConfig.configuration,
                ActionMode: 'CHANGE_SET_REPLACE',
                ChangeSetName: this.props3.changeSetName,
                TemplatePath: this.props3.templatePath.location,
            },
        };
    }
}
exports.CloudFormationCreateReplaceChangeSetAction = CloudFormationCreateReplaceChangeSetAction;
/**
 * CodePipeline action to deploy a stack.
 *
 * Creates the stack if the specified stack doesn't exist. If the stack exists,
 * AWS CloudFormation updates the stack. Use this action to update existing
 * stacks.
 *
 * AWS CodePipeline won't replace the stack, and will fail deployment if the
 * stack is in a failed state. Use `ReplaceOnFailure` for an action that
 * will delete and recreate the stack to try and recover from failed states.
 *
 * Use this action to automatically replace failed stacks without recovering or
 * troubleshooting them. You would typically choose this mode for testing.
 */
class CloudFormationCreateUpdateStackAction extends CloudFormationDeployAction {
    constructor(props) {
        super(props, props.templateConfiguration
            ? [props.templatePath.artifact, props.templateConfiguration.artifact]
            : [props.templatePath.artifact]);
        this.props3 = props;
    }
    bound(scope, stage, options) {
        // the super call order is to preserve the existing order of statements in policies
        const actionConfig = super.bound(scope, stage, options);
        SingletonPolicy.forRole(options.role).grantCreateUpdateStack(this.props3);
        return {
            ...actionConfig,
            configuration: {
                ...actionConfig.configuration,
                ActionMode: this.props3.replaceOnFailure ? 'REPLACE_ON_FAILURE' : 'CREATE_UPDATE',
                TemplatePath: this.props3.templatePath.location,
            },
        };
    }
}
exports.CloudFormationCreateUpdateStackAction = CloudFormationCreateUpdateStackAction;
/**
 * CodePipeline action to delete a stack.
 *
 * Deletes a stack. If you specify a stack that doesn't exist, the action completes successfully
 * without deleting a stack.
 */
class CloudFormationDeleteStackAction extends CloudFormationDeployAction {
    constructor(props) {
        super(props, undefined);
        this.props3 = props;
    }
    bound(scope, stage, options) {
        // the super call order is to preserve the existing order of statements in policies
        const actionConfig = super.bound(scope, stage, options);
        SingletonPolicy.forRole(options.role).grantDeleteStack(this.props3);
        return {
            ...actionConfig,
            configuration: {
                ...actionConfig.configuration,
                ActionMode: 'DELETE_ONLY',
            },
        };
    }
}
exports.CloudFormationDeleteStackAction = CloudFormationDeleteStackAction;
/**
 * Manages a bunch of singleton-y statements on the policy of an IAM Role.
 * Dedicated methods can be used to add specific permissions to the role policy
 * using as few statements as possible (adding resources to existing compatible
 * statements instead of adding new statements whenever possible).
 *
 * Statements created outside of this class are not considered when adding new
 * permissions.
 */
class SingletonPolicy extends cdk.Construct {
    constructor(role) {
        super(role, SingletonPolicy.UUID);
        this.role = role;
        this.statements = {};
        this.grantPrincipal = role;
    }
    /**
     * Obtain a SingletonPolicy for a given role.
     * @param role the Role this policy is bound to.
     * @returns the SingletonPolicy for this role.
     */
    static forRole(role) {
        const found = role.node.tryFindChild(SingletonPolicy.UUID);
        return found || new SingletonPolicy(role);
    }
    grantExecuteChangeSet(props) {
        this.statementFor({
            actions: [
                'cloudformation:DescribeStacks',
                'cloudformation:DescribeChangeSet',
                'cloudformation:ExecuteChangeSet',
            ],
            conditions: { StringEqualsIfExists: { 'cloudformation:ChangeSetName': props.changeSetName } },
        }).addResources(this.stackArnFromProps(props));
    }
    grantCreateReplaceChangeSet(props) {
        this.statementFor({
            actions: [
                'cloudformation:CreateChangeSet',
                'cloudformation:DeleteChangeSet',
                'cloudformation:DescribeChangeSet',
                'cloudformation:DescribeStacks',
            ],
            conditions: { StringEqualsIfExists: { 'cloudformation:ChangeSetName': props.changeSetName } },
        }).addResources(this.stackArnFromProps(props));
    }
    grantCreateUpdateStack(props) {
        const actions = [
            'cloudformation:DescribeStack*',
            'cloudformation:CreateStack',
            'cloudformation:UpdateStack',
            'cloudformation:GetTemplate*',
            'cloudformation:ValidateTemplate',
            'cloudformation:GetStackPolicy',
            'cloudformation:SetStackPolicy',
        ];
        if (props.replaceOnFailure) {
            actions.push('cloudformation:DeleteStack');
        }
        this.statementFor({ actions }).addResources(this.stackArnFromProps(props));
    }
    grantDeleteStack(props) {
        this.statementFor({
            actions: [
                'cloudformation:DescribeStack*',
                'cloudformation:DeleteStack',
            ]
        }).addResources(this.stackArnFromProps(props));
    }
    grantPassRole(role) {
        this.statementFor({ actions: ['iam:PassRole'] }).addResources(role.roleArn);
    }
    statementFor(template) {
        const key = keyFor(template);
        if (!(key in this.statements)) {
            this.statements[key] = new iam.PolicyStatement({ actions: template.actions });
            if (template.conditions) {
                this.statements[key].addConditions(template.conditions);
            }
            this.role.addToPolicy(this.statements[key]);
        }
        return this.statements[key];
        function keyFor(props) {
            const actions = `${props.actions.sort().join('\x1F')}`;
            const conditions = formatConditions(props.conditions);
            return `${actions}\x1D${conditions}`;
            function formatConditions(cond) {
                if (cond == null) {
                    return '';
                }
                let result = '';
                for (const op of Object.keys(cond).sort()) {
                    result += `${op}\x1E`;
                    const condition = cond[op];
                    for (const attribute of Object.keys(condition).sort()) {
                        const value = condition[attribute];
                        result += `${value}\x1F`;
                    }
                }
                return result;
            }
        }
    }
    stackArnFromProps(props) {
        return cdk.Stack.of(this).formatArn({
            region: props.region,
            service: 'cloudformation',
            resource: 'stack',
            resourceName: `${props.stackName}/*`
        });
    }
}
SingletonPolicy.UUID = '8389e75f-0810-4838-bf64-d6f85a95cf83';
function parseCapabilities(capabilities) {
    if (capabilities === undefined) {
        return undefined;
    }
    else if (capabilities.length === 1) {
        const capability = capabilities.toString();
        return (capability === '') ? undefined : capability;
    }
    else if (capabilities.length > 1) {
        return capabilities.join(',');
    }
    return undefined;
}
//# sourceMappingURL=data:application/json;base64,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