"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const https = require("https");
const URL = require("url");
/**
 * Custom CloudFormation resource helper
 */
class CustomResource {
    constructor(event, context, callback, logger) {
        /**
         * Stores functions executed when resource creation is requested
         */
        this.createFunctions = [];
        /**
         * Stores functions executed when resource update is requested
         */
        this.updateFunctions = [];
        /**
         * Stores functions executed when resource deletion is requested
         */
        this.deleteFunctions = [];
        /**
         * Stores values returned to CloudFormation
         */
        this.ResponseData = {};
        if (typeof event.ResponseURL === 'undefined') {
            throw new Error('ResponseURL missing');
        }
        this.event = event;
        this.context = context;
        this.callback = callback;
        this.logger = logger || new StandardLogger();
        this.logger.debug(`REQUEST RECEIVED:\n${JSON.stringify(event)}`);
        this.timeout();
    }
    /**
     * Adds a function to the CREATE queue
     */
    onCreate(func) {
        this.createFunctions.push(func);
        return this;
    }
    /**
     * Adds a function to the UPDATE queue
     */
    onUpdate(func) {
        this.updateFunctions.push(func);
        return this;
    }
    /**
     * Adds a function to the DELETE queue
     */
    onDelete(func) {
        this.deleteFunctions.push(func);
        return this;
    }
    /**
     * Handles the Lambda event
     */
    handle(event) {
        const self = this;
        event.addResponseValue = (key, value) => {
            self.ResponseData[key] = value;
        };
        try {
            let queue;
            if (this.event.RequestType == 'Create')
                queue = this.createFunctions;
            else if (this.event.RequestType == 'Update')
                queue = this.updateFunctions;
            else if (this.event.RequestType == 'Delete')
                queue = this.deleteFunctions;
            else {
                this.sendResponse('FAILED', `Unexpected request type: ${this.event.RequestType}`);
                return this;
            }
            let result = queue.reduce((current, next) => {
                return current.then((value) => {
                    return next(value);
                });
            }, Promise.resolve(event));
            result
                .then(function (event) {
                self.logger.debug(event);
                self.sendResponse('SUCCESS', `${self.event.RequestType} completed successfully`);
            })
                .catch(function (err) {
                self.logger.error(err, err.stack);
                self.sendResponse('FAILED', err.message || err.code);
            });
        }
        catch (err) {
            this.sendResponse('FAILED', err.message || err.code);
        }
        return this;
    }
    /**
     * Sends CloudFormation response just before the Lambda times out
     */
    timeout() {
        const self = this;
        const handler = () => {
            self.logger.error('Timeout FAILURE!');
            new Promise(() => self.sendResponse('FAILED', 'Function timed out')).then(() => self.callback(new Error('Function timed out')));
        };
        setTimeout(handler, this.context.getRemainingTimeInMillis() - 1000);
    }
    /**
     * Sends CloudFormation response
     */
    sendResponse(responseStatus, responseData) {
        const self = this;
        this.logger.debug(`Sending response ${responseStatus}:\n${JSON.stringify(responseData)}`);
        const data = this.ResponseData;
        data['Message'] = responseData;
        const body = JSON.stringify({
            Status: responseStatus,
            Reason: `${responseData} | Full error in CloudWatch ${this.context.logStreamName}`,
            PhysicalResourceId: this.event.ResourceProperties.Name,
            StackId: this.event.StackId,
            RequestId: this.event.RequestId,
            LogicalResourceId: this.event.LogicalResourceId,
            Data: data,
        });
        this.logger.debug('RESPONSE BODY:\n', body);
        const url = URL.parse(this.event.ResponseURL);
        const options = {
            hostname: url.hostname,
            port: 443,
            path: url.path,
            method: 'PUT',
            headers: {
                'content-type': '',
                'content-length': body.length,
            },
        };
        this.logger.info('SENDING RESPONSE...\n');
        const request = https.request(options, function (response) {
            self.logger.debug(`STATUS: ${response.statusCode}`);
            self.logger.debug(`HEADERS: ${JSON.stringify(response.headers)}`);
            self.context.done();
        });
        request.on('error', function (error) {
            self.logger.error(`sendResponse Error: ${error}`);
            self.context.done();
        });
        request.write(body);
        request.end();
    }
}
exports.CustomResource = CustomResource;
/**
 * Standard logger class
 */
class StandardLogger {
    constructor(level) {
        this.level = level || 1 /* WARN */;
    }
    /**
     * Logs message with level ERROR
     */
    error(message, ...optionalParams) {
        if (this.level < 0 /* ERROR */)
            return;
        console.error(message, ...optionalParams);
    }
    /**
     * Logs message with level WARN
     */
    warn(message, ...optionalParams) {
        if (this.level < 1 /* WARN */)
            return;
        console.warn(message, ...optionalParams);
    }
    /**
     * Logs message with level INFO
     */
    info(message, ...optionalParams) {
        if (this.level < 2 /* INFO */)
            return;
        console.info(message, ...optionalParams);
    }
    /**
     * Logs message with level DEBUG
     */
    debug(message, ...optionalParams) {
        if (this.level < 3 /* DEBUG */)
            return;
        console.debug(message, ...optionalParams);
    }
    /**
     * Alias for info
     */
    log(message, ...optionalParams) {
        this.info(message, ...optionalParams);
    }
}
exports.StandardLogger = StandardLogger;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW5kZXguanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi9pbmRleC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOztBQUVBLCtCQUFnQztBQUNoQywyQkFBNEI7QUF5QjVCOztHQUVHO0FBQ0gsTUFBYSxjQUFjO0lBMkN2QixZQUNJLEtBQWtCLEVBQ2xCLE9BQWdCLEVBQ2hCLFFBQWtCLEVBQ2xCLE1BQWU7UUE5Q25COztXQUVHO1FBQ0gsb0JBQWUsR0FBVyxFQUFFLENBQUM7UUFFN0I7O1dBRUc7UUFDSCxvQkFBZSxHQUFXLEVBQUUsQ0FBQztRQUU3Qjs7V0FFRztRQUNILG9CQUFlLEdBQVcsRUFBRSxDQUFDO1FBaUI3Qjs7V0FFRztRQUNILGlCQUFZLEdBRVIsRUFBRSxDQUFDO1FBYUgsSUFBSSxPQUFPLEtBQUssQ0FBQyxXQUFXLEtBQUssV0FBVyxFQUFFO1lBQzFDLE1BQU0sSUFBSSxLQUFLLENBQUMscUJBQXFCLENBQUMsQ0FBQztTQUMxQztRQUNELElBQUksQ0FBQyxLQUFLLEdBQUcsS0FBSyxDQUFDO1FBQ25CLElBQUksQ0FBQyxPQUFPLEdBQUcsT0FBTyxDQUFDO1FBQ3ZCLElBQUksQ0FBQyxRQUFRLEdBQUcsUUFBUSxDQUFDO1FBQ3pCLElBQUksQ0FBQyxNQUFNLEdBQUcsTUFBTSxJQUFJLElBQUksY0FBYyxFQUFFLENBQUM7UUFDN0MsSUFBSSxDQUFDLE1BQU0sQ0FBQyxLQUFLLENBQUMsc0JBQXNCLElBQUksQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLEVBQUUsQ0FBQyxDQUFDO1FBQ2pFLElBQUksQ0FBQyxPQUFPLEVBQUUsQ0FBQztJQUNuQixDQUFDO0lBRUQ7O09BRUc7SUFDSCxRQUFRLENBQUMsSUFBVTtRQUNmLElBQUksQ0FBQyxlQUFlLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQ2hDLE9BQU8sSUFBSSxDQUFDO0lBQ2hCLENBQUM7SUFFRDs7T0FFRztJQUNILFFBQVEsQ0FBQyxJQUFVO1FBQ2YsSUFBSSxDQUFDLGVBQWUsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUM7UUFDaEMsT0FBTyxJQUFJLENBQUM7SUFDaEIsQ0FBQztJQUVEOztPQUVHO0lBQ0gsUUFBUSxDQUFDLElBQVU7UUFDZixJQUFJLENBQUMsZUFBZSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUNoQyxPQUFPLElBQUksQ0FBQztJQUNoQixDQUFDO0lBRUQ7O09BRUc7SUFDSCxNQUFNLENBQUMsS0FBa0I7UUFDckIsTUFBTSxJQUFJLEdBQUcsSUFBSSxDQUFDO1FBRWxCLEtBQUssQ0FBQyxnQkFBZ0IsR0FBRyxDQUFDLEdBQVcsRUFBRSxLQUFVLEVBQUUsRUFBRTtZQUNqRCxJQUFJLENBQUMsWUFBWSxDQUFDLEdBQUcsQ0FBQyxHQUFHLEtBQUssQ0FBQztRQUNuQyxDQUFDLENBQUM7UUFFRixJQUFJO1lBQ0EsSUFBSSxLQUFhLENBQUM7WUFDbEIsSUFBSSxJQUFJLENBQUMsS0FBSyxDQUFDLFdBQVcsSUFBSSxRQUFRO2dCQUNsQyxLQUFLLEdBQUcsSUFBSSxDQUFDLGVBQWUsQ0FBQztpQkFDNUIsSUFBSSxJQUFJLENBQUMsS0FBSyxDQUFDLFdBQVcsSUFBSSxRQUFRO2dCQUN2QyxLQUFLLEdBQUcsSUFBSSxDQUFDLGVBQWUsQ0FBQztpQkFDNUIsSUFBSSxJQUFJLENBQUMsS0FBSyxDQUFDLFdBQVcsSUFBSSxRQUFRO2dCQUN2QyxLQUFLLEdBQUcsSUFBSSxDQUFDLGVBQWUsQ0FBQztpQkFDNUI7Z0JBQ0QsSUFBSSxDQUFDLFlBQVksQ0FDYixRQUFRLEVBQ1IsNEJBQTRCLElBQUksQ0FBQyxLQUFLLENBQUMsV0FBVyxFQUFFLENBQ3ZELENBQUM7Z0JBQ0YsT0FBTyxJQUFJLENBQUM7YUFDZjtZQUVELElBQUksTUFBTSxHQUFHLEtBQUssQ0FBQyxNQUFNLENBQ3JCLENBQUMsT0FBNkMsRUFBRSxJQUFVLEVBQUUsRUFBRTtnQkFDMUQsT0FBUSxPQUEwQixDQUFDLElBQUksQ0FBQyxDQUFDLEtBQVksRUFBRSxFQUFFO29CQUNyRCxPQUFPLElBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQztnQkFDdkIsQ0FBQyxDQUFDLENBQUM7WUFDUCxDQUFDLEVBQ0QsT0FBTyxDQUFDLE9BQU8sQ0FBQyxLQUFjLENBQUMsQ0FDbEMsQ0FBQztZQUVGLE1BQU07aUJBQ0QsSUFBSSxDQUFDLFVBQVUsS0FBMkI7Z0JBQ3ZDLElBQUksQ0FBQyxNQUFNLENBQUMsS0FBSyxDQUFDLEtBQUssQ0FBQyxDQUFDO2dCQUN6QixJQUFJLENBQUMsWUFBWSxDQUNiLFNBQVMsRUFDVCxHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsV0FBVyx5QkFBeUIsQ0FDckQsQ0FBQztZQUNOLENBQUMsQ0FBQztpQkFDRCxLQUFLLENBQUMsVUFBVSxHQUFpQjtnQkFDOUIsSUFBSSxDQUFDLE1BQU0sQ0FBQyxLQUFLLENBQUMsR0FBRyxFQUFFLEdBQUcsQ0FBQyxLQUFLLENBQUMsQ0FBQztnQkFDbEMsSUFBSSxDQUFDLFlBQVksQ0FBQyxRQUFRLEVBQUUsR0FBRyxDQUFDLE9BQU8sSUFBSSxHQUFHLENBQUMsSUFBSSxDQUFDLENBQUM7WUFDekQsQ0FBQyxDQUFDLENBQUM7U0FDVjtRQUFDLE9BQU8sR0FBRyxFQUFFO1lBQ1YsSUFBSSxDQUFDLFlBQVksQ0FBQyxRQUFRLEVBQUUsR0FBRyxDQUFDLE9BQU8sSUFBSSxHQUFHLENBQUMsSUFBSSxDQUFDLENBQUM7U0FDeEQ7UUFDRCxPQUFPLElBQUksQ0FBQztJQUNoQixDQUFDO0lBRUQ7O09BRUc7SUFDSCxPQUFPO1FBQ0gsTUFBTSxJQUFJLEdBQUcsSUFBSSxDQUFDO1FBQ2xCLE1BQU0sT0FBTyxHQUFHLEdBQUcsRUFBRTtZQUNqQixJQUFJLENBQUMsTUFBTSxDQUFDLEtBQUssQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDO1lBQ3RDLElBQUksT0FBTyxDQUFDLEdBQUcsRUFBRSxDQUNiLElBQUksQ0FBQyxZQUFZLENBQUMsUUFBUSxFQUFFLG9CQUFvQixDQUFDLENBQ3BELENBQUMsSUFBSSxDQUFDLEdBQUcsRUFBRSxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsSUFBSSxLQUFLLENBQUMsb0JBQW9CLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFDakUsQ0FBQyxDQUFDO1FBQ0YsVUFBVSxDQUFDLE9BQU8sRUFBRSxJQUFJLENBQUMsT0FBTyxDQUFDLHdCQUF3QixFQUFFLEdBQUcsSUFBSSxDQUFDLENBQUM7SUFDeEUsQ0FBQztJQUVEOztPQUVHO0lBQ0gsWUFBWSxDQUFDLGNBQXNCLEVBQUUsWUFBb0I7UUFDckQsTUFBTSxJQUFJLEdBQUcsSUFBSSxDQUFDO1FBQ2xCLElBQUksQ0FBQyxNQUFNLENBQUMsS0FBSyxDQUNiLG9CQUFvQixjQUFjLE1BQU0sSUFBSSxDQUFDLFNBQVMsQ0FDbEQsWUFBWSxDQUNmLEVBQUUsQ0FDTixDQUFDO1FBRUYsTUFBTSxJQUFJLEdBQUcsSUFBSSxDQUFDLFlBQVksQ0FBQztRQUMvQixJQUFJLENBQUMsU0FBUyxDQUFDLEdBQUcsWUFBWSxDQUFDO1FBRS9CLE1BQU0sSUFBSSxHQUFHLElBQUksQ0FBQyxTQUFTLENBQUM7WUFDeEIsTUFBTSxFQUFFLGNBQWM7WUFDdEIsTUFBTSxFQUFFLEdBQUcsWUFBWSwrQkFBK0IsSUFBSSxDQUFDLE9BQU8sQ0FBQyxhQUFhLEVBQUU7WUFDbEYsa0JBQWtCLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxrQkFBa0IsQ0FBQyxJQUFJO1lBQ3RELE9BQU8sRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLE9BQU87WUFDM0IsU0FBUyxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsU0FBUztZQUMvQixpQkFBaUIsRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLGlCQUFpQjtZQUMvQyxJQUFJLEVBQUUsSUFBSTtTQUNiLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxNQUFNLENBQUMsS0FBSyxDQUFDLGtCQUFrQixFQUFFLElBQUksQ0FBQyxDQUFDO1FBRTVDLE1BQU0sR0FBRyxHQUFHLEdBQUcsQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxXQUFXLENBQUMsQ0FBQztRQUM5QyxNQUFNLE9BQU8sR0FBRztZQUNaLFFBQVEsRUFBRSxHQUFHLENBQUMsUUFBUTtZQUN0QixJQUFJLEVBQUUsR0FBRztZQUNULElBQUksRUFBRSxHQUFHLENBQUMsSUFBSTtZQUNkLE1BQU0sRUFBRSxLQUFLO1lBQ2IsT0FBTyxFQUFFO2dCQUNMLGNBQWMsRUFBRSxFQUFFO2dCQUNsQixnQkFBZ0IsRUFBRSxJQUFJLENBQUMsTUFBTTthQUNoQztTQUNKLENBQUM7UUFFRixJQUFJLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyx1QkFBdUIsQ0FBQyxDQUFDO1FBRTFDLE1BQU0sT0FBTyxHQUFHLEtBQUssQ0FBQyxPQUFPLENBQUMsT0FBTyxFQUFFLFVBQVUsUUFBYTtZQUMxRCxJQUFJLENBQUMsTUFBTSxDQUFDLEtBQUssQ0FBQyxXQUFXLFFBQVEsQ0FBQyxVQUFVLEVBQUUsQ0FBQyxDQUFDO1lBQ3BELElBQUksQ0FBQyxNQUFNLENBQUMsS0FBSyxDQUFDLFlBQVksSUFBSSxDQUFDLFNBQVMsQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLEVBQUUsQ0FBQyxDQUFDO1lBQ2xFLElBQUksQ0FBQyxPQUFPLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDeEIsQ0FBQyxDQUFDLENBQUM7UUFFSCxPQUFPLENBQUMsRUFBRSxDQUFDLE9BQU8sRUFBRSxVQUFVLEtBQVk7WUFDdEMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxLQUFLLENBQUMsdUJBQXVCLEtBQUssRUFBRSxDQUFDLENBQUM7WUFDbEQsSUFBSSxDQUFDLE9BQU8sQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUN4QixDQUFDLENBQUMsQ0FBQztRQUVILE9BQU8sQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLENBQUM7UUFDcEIsT0FBTyxDQUFDLEdBQUcsRUFBRSxDQUFDO0lBQ2xCLENBQUM7Q0FDSjtBQTdNRCx3Q0E2TUM7QUF1QkQ7O0dBRUc7QUFDSCxNQUFhLGNBQWM7SUFRdkIsWUFBWSxLQUFnQjtRQUN4QixJQUFJLENBQUMsS0FBSyxHQUFHLEtBQUssZ0JBQWlCLENBQUM7SUFDeEMsQ0FBQztJQUVEOztPQUVHO0lBQ0gsS0FBSyxDQUFDLE9BQVksRUFBRSxHQUFHLGNBQXFCO1FBQ3hDLElBQUksSUFBSSxDQUFDLEtBQUssZ0JBQWlCO1lBQUUsT0FBTztRQUN4QyxPQUFPLENBQUMsS0FBSyxDQUFDLE9BQU8sRUFBRSxHQUFHLGNBQWMsQ0FBQyxDQUFDO0lBQzlDLENBQUM7SUFFRDs7T0FFRztJQUNILElBQUksQ0FBQyxPQUFZLEVBQUUsR0FBRyxjQUFxQjtRQUN2QyxJQUFJLElBQUksQ0FBQyxLQUFLLGVBQWdCO1lBQUUsT0FBTztRQUN2QyxPQUFPLENBQUMsSUFBSSxDQUFDLE9BQU8sRUFBRSxHQUFHLGNBQWMsQ0FBQyxDQUFDO0lBQzdDLENBQUM7SUFFRDs7T0FFRztJQUNILElBQUksQ0FBQyxPQUFZLEVBQUUsR0FBRyxjQUFxQjtRQUN2QyxJQUFJLElBQUksQ0FBQyxLQUFLLGVBQWdCO1lBQUUsT0FBTztRQUN2QyxPQUFPLENBQUMsSUFBSSxDQUFDLE9BQU8sRUFBRSxHQUFHLGNBQWMsQ0FBQyxDQUFDO0lBQzdDLENBQUM7SUFFRDs7T0FFRztJQUNILEtBQUssQ0FBQyxPQUFZLEVBQUUsR0FBRyxjQUFxQjtRQUN4QyxJQUFJLElBQUksQ0FBQyxLQUFLLGdCQUFpQjtZQUFFLE9BQU87UUFDeEMsT0FBTyxDQUFDLEtBQUssQ0FBQyxPQUFPLEVBQUUsR0FBRyxjQUFjLENBQUMsQ0FBQztJQUM5QyxDQUFDO0lBRUQ7O09BRUc7SUFDSCxHQUFHLENBQUMsT0FBWSxFQUFFLEdBQUcsY0FBcUI7UUFDdEMsSUFBSSxDQUFDLElBQUksQ0FBQyxPQUFPLEVBQUUsR0FBRyxjQUFjLENBQUMsQ0FBQztJQUMxQyxDQUFDO0NBQ0o7QUFsREQsd0NBa0RDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgQ2FsbGJhY2ssIENvbnRleHQgfSBmcm9tICdhd3MtbGFtYmRhJztcbmltcG9ydCBBV1MgPSByZXF1aXJlKCdhd3Mtc2RrJyk7XG5pbXBvcnQgaHR0cHMgPSByZXF1aXJlKCdodHRwcycpO1xuaW1wb3J0IFVSTCA9IHJlcXVpcmUoJ3VybCcpO1xuXG4vKipcbiAqIFRoZSBldmVudCBwYXNzZWQgdG8gdGhlIExhbWJkYSBoYW5kbGVyXG4gKi9cblxuZXhwb3J0IGludGVyZmFjZSBMYW1iZGFFdmVudCB7XG4gICAgW2tleTogc3RyaW5nXTogYW55O1xufVxuXG4vKipcbiAqIFRoZSBldmVudCBwYXNzZWQgdGhyb3VnaCB0aGUgcHJvbWlzZXMuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgRXZlbnQgZXh0ZW5kcyBMYW1iZGFFdmVudCB7XG4gICAgLyoqXG4gICAgICogQWRkcyB2YWx1ZXMgdG8gdGhlIHJlc3BvbnNlIHJldHVybmVkIHRvIENsb3VkRm9ybWF0aW9uXG4gICAgICovXG4gICAgYWRkUmVzcG9uc2VWYWx1ZTogKGtleTogc3RyaW5nLCB2YWx1ZTogYW55KSA9PiB2b2lkO1xufVxuXG4vKipcbiAqIEZ1bmN0aW9uIHNpZ25hdHVyZVxuICovXG5leHBvcnQgdHlwZSBmdW5jID0gKGV2ZW50OiBFdmVudCkgPT4gUHJvbWlzZTxFdmVudCB8IEFXUy5BV1NFcnJvcj47XG5cbi8qKlxuICogQ3VzdG9tIENsb3VkRm9ybWF0aW9uIHJlc291cmNlIGhlbHBlclxuICovXG5leHBvcnQgY2xhc3MgQ3VzdG9tUmVzb3VyY2Uge1xuICAgIC8qKlxuICAgICAqIFN0b3JlcyBmdW5jdGlvbnMgZXhlY3V0ZWQgd2hlbiByZXNvdXJjZSBjcmVhdGlvbiBpcyByZXF1ZXN0ZWRcbiAgICAgKi9cbiAgICBjcmVhdGVGdW5jdGlvbnM6IGZ1bmNbXSA9IFtdO1xuXG4gICAgLyoqXG4gICAgICogU3RvcmVzIGZ1bmN0aW9ucyBleGVjdXRlZCB3aGVuIHJlc291cmNlIHVwZGF0ZSBpcyByZXF1ZXN0ZWRcbiAgICAgKi9cbiAgICB1cGRhdGVGdW5jdGlvbnM6IGZ1bmNbXSA9IFtdO1xuXG4gICAgLyoqXG4gICAgICogU3RvcmVzIGZ1bmN0aW9ucyBleGVjdXRlZCB3aGVuIHJlc291cmNlIGRlbGV0aW9uIGlzIHJlcXVlc3RlZFxuICAgICAqL1xuICAgIGRlbGV0ZUZ1bmN0aW9uczogZnVuY1tdID0gW107XG5cbiAgICAvKipcbiAgICAgKiBUaGUgZXZlbnQgcGFzc2VkIHRvIHRoZSBMYW1iZGEgaGFuZGxlclxuICAgICAqL1xuICAgIGV2ZW50OiBMYW1iZGFFdmVudDtcblxuICAgIC8qKlxuICAgICAqIFRoZSBjb250ZXh0IHBhc3NlZCB0byB0aGUgTGFtYmRhIGhhbmRsZXJcbiAgICAgKi9cbiAgICBjb250ZXh0OiBDb250ZXh0O1xuXG4gICAgLyoqXG4gICAgICogVGhlIGNhbGxiYWNrIGZ1bmN0aW9uIHBhc3NlZCB0byB0aGUgTGFtYmRhIGhhbmRsZXJcbiAgICAgKi9cbiAgICBjYWxsYmFjazogQ2FsbGJhY2s7XG5cbiAgICAvKipcbiAgICAgKiBTdG9yZXMgdmFsdWVzIHJldHVybmVkIHRvIENsb3VkRm9ybWF0aW9uXG4gICAgICovXG4gICAgUmVzcG9uc2VEYXRhOiB7XG4gICAgICAgIFtrZXk6IHN0cmluZ106IGFueTtcbiAgICB9ID0ge307XG5cbiAgICAvKipcbiAgICAgKiBMb2dnZXIgY2xhc3NcbiAgICAgKi9cbiAgICBsb2dnZXI6IExvZ2dlcjtcblxuICAgIGNvbnN0cnVjdG9yKFxuICAgICAgICBldmVudDogTGFtYmRhRXZlbnQsXG4gICAgICAgIGNvbnRleHQ6IENvbnRleHQsXG4gICAgICAgIGNhbGxiYWNrOiBDYWxsYmFjayxcbiAgICAgICAgbG9nZ2VyPzogTG9nZ2VyXG4gICAgKSB7XG4gICAgICAgIGlmICh0eXBlb2YgZXZlbnQuUmVzcG9uc2VVUkwgPT09ICd1bmRlZmluZWQnKSB7XG4gICAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ1Jlc3BvbnNlVVJMIG1pc3NpbmcnKTtcbiAgICAgICAgfVxuICAgICAgICB0aGlzLmV2ZW50ID0gZXZlbnQ7XG4gICAgICAgIHRoaXMuY29udGV4dCA9IGNvbnRleHQ7XG4gICAgICAgIHRoaXMuY2FsbGJhY2sgPSBjYWxsYmFjaztcbiAgICAgICAgdGhpcy5sb2dnZXIgPSBsb2dnZXIgfHwgbmV3IFN0YW5kYXJkTG9nZ2VyKCk7XG4gICAgICAgIHRoaXMubG9nZ2VyLmRlYnVnKGBSRVFVRVNUIFJFQ0VJVkVEOlxcbiR7SlNPTi5zdHJpbmdpZnkoZXZlbnQpfWApO1xuICAgICAgICB0aGlzLnRpbWVvdXQoKTtcbiAgICB9XG5cbiAgICAvKipcbiAgICAgKiBBZGRzIGEgZnVuY3Rpb24gdG8gdGhlIENSRUFURSBxdWV1ZVxuICAgICAqL1xuICAgIG9uQ3JlYXRlKGZ1bmM6IGZ1bmMpOiB0aGlzIHtcbiAgICAgICAgdGhpcy5jcmVhdGVGdW5jdGlvbnMucHVzaChmdW5jKTtcbiAgICAgICAgcmV0dXJuIHRoaXM7XG4gICAgfVxuXG4gICAgLyoqXG4gICAgICogQWRkcyBhIGZ1bmN0aW9uIHRvIHRoZSBVUERBVEUgcXVldWVcbiAgICAgKi9cbiAgICBvblVwZGF0ZShmdW5jOiBmdW5jKTogdGhpcyB7XG4gICAgICAgIHRoaXMudXBkYXRlRnVuY3Rpb25zLnB1c2goZnVuYyk7XG4gICAgICAgIHJldHVybiB0aGlzO1xuICAgIH1cblxuICAgIC8qKlxuICAgICAqIEFkZHMgYSBmdW5jdGlvbiB0byB0aGUgREVMRVRFIHF1ZXVlXG4gICAgICovXG4gICAgb25EZWxldGUoZnVuYzogZnVuYyk6IHRoaXMge1xuICAgICAgICB0aGlzLmRlbGV0ZUZ1bmN0aW9ucy5wdXNoKGZ1bmMpO1xuICAgICAgICByZXR1cm4gdGhpcztcbiAgICB9XG5cbiAgICAvKipcbiAgICAgKiBIYW5kbGVzIHRoZSBMYW1iZGEgZXZlbnRcbiAgICAgKi9cbiAgICBoYW5kbGUoZXZlbnQ6IExhbWJkYUV2ZW50KTogdGhpcyB7XG4gICAgICAgIGNvbnN0IHNlbGYgPSB0aGlzO1xuXG4gICAgICAgIGV2ZW50LmFkZFJlc3BvbnNlVmFsdWUgPSAoa2V5OiBzdHJpbmcsIHZhbHVlOiBhbnkpID0+IHtcbiAgICAgICAgICAgIHNlbGYuUmVzcG9uc2VEYXRhW2tleV0gPSB2YWx1ZTtcbiAgICAgICAgfTtcblxuICAgICAgICB0cnkge1xuICAgICAgICAgICAgbGV0IHF1ZXVlOiBmdW5jW107XG4gICAgICAgICAgICBpZiAodGhpcy5ldmVudC5SZXF1ZXN0VHlwZSA9PSAnQ3JlYXRlJylcbiAgICAgICAgICAgICAgICBxdWV1ZSA9IHRoaXMuY3JlYXRlRnVuY3Rpb25zO1xuICAgICAgICAgICAgZWxzZSBpZiAodGhpcy5ldmVudC5SZXF1ZXN0VHlwZSA9PSAnVXBkYXRlJylcbiAgICAgICAgICAgICAgICBxdWV1ZSA9IHRoaXMudXBkYXRlRnVuY3Rpb25zO1xuICAgICAgICAgICAgZWxzZSBpZiAodGhpcy5ldmVudC5SZXF1ZXN0VHlwZSA9PSAnRGVsZXRlJylcbiAgICAgICAgICAgICAgICBxdWV1ZSA9IHRoaXMuZGVsZXRlRnVuY3Rpb25zO1xuICAgICAgICAgICAgZWxzZSB7XG4gICAgICAgICAgICAgICAgdGhpcy5zZW5kUmVzcG9uc2UoXG4gICAgICAgICAgICAgICAgICAgICdGQUlMRUQnLFxuICAgICAgICAgICAgICAgICAgICBgVW5leHBlY3RlZCByZXF1ZXN0IHR5cGU6ICR7dGhpcy5ldmVudC5SZXF1ZXN0VHlwZX1gXG4gICAgICAgICAgICAgICAgKTtcbiAgICAgICAgICAgICAgICByZXR1cm4gdGhpcztcbiAgICAgICAgICAgIH1cblxuICAgICAgICAgICAgbGV0IHJlc3VsdCA9IHF1ZXVlLnJlZHVjZShcbiAgICAgICAgICAgICAgICAoY3VycmVudDogUHJvbWlzZTxFdmVudCB8IEFXUy5BV1NFcnJvcj4gfCBmdW5jLCBuZXh0OiBmdW5jKSA9PiB7XG4gICAgICAgICAgICAgICAgICAgIHJldHVybiAoY3VycmVudCBhcyBQcm9taXNlPEV2ZW50PikudGhlbigodmFsdWU6IEV2ZW50KSA9PiB7XG4gICAgICAgICAgICAgICAgICAgICAgICByZXR1cm4gbmV4dCh2YWx1ZSk7XG4gICAgICAgICAgICAgICAgICAgIH0pO1xuICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgUHJvbWlzZS5yZXNvbHZlKGV2ZW50IGFzIEV2ZW50KVxuICAgICAgICAgICAgKTtcblxuICAgICAgICAgICAgcmVzdWx0XG4gICAgICAgICAgICAgICAgLnRoZW4oZnVuY3Rpb24gKGV2ZW50OiBFdmVudCB8IEFXUy5BV1NFcnJvcikge1xuICAgICAgICAgICAgICAgICAgICBzZWxmLmxvZ2dlci5kZWJ1ZyhldmVudCk7XG4gICAgICAgICAgICAgICAgICAgIHNlbGYuc2VuZFJlc3BvbnNlKFxuICAgICAgICAgICAgICAgICAgICAgICAgJ1NVQ0NFU1MnLFxuICAgICAgICAgICAgICAgICAgICAgICAgYCR7c2VsZi5ldmVudC5SZXF1ZXN0VHlwZX0gY29tcGxldGVkIHN1Y2Nlc3NmdWxseWBcbiAgICAgICAgICAgICAgICAgICAgKTtcbiAgICAgICAgICAgICAgICB9KVxuICAgICAgICAgICAgICAgIC5jYXRjaChmdW5jdGlvbiAoZXJyOiBBV1MuQVdTRXJyb3IpIHtcbiAgICAgICAgICAgICAgICAgICAgc2VsZi5sb2dnZXIuZXJyb3IoZXJyLCBlcnIuc3RhY2spO1xuICAgICAgICAgICAgICAgICAgICBzZWxmLnNlbmRSZXNwb25zZSgnRkFJTEVEJywgZXJyLm1lc3NhZ2UgfHwgZXJyLmNvZGUpO1xuICAgICAgICAgICAgICAgIH0pO1xuICAgICAgICB9IGNhdGNoIChlcnIpIHtcbiAgICAgICAgICAgIHRoaXMuc2VuZFJlc3BvbnNlKCdGQUlMRUQnLCBlcnIubWVzc2FnZSB8fCBlcnIuY29kZSk7XG4gICAgICAgIH1cbiAgICAgICAgcmV0dXJuIHRoaXM7XG4gICAgfVxuXG4gICAgLyoqXG4gICAgICogU2VuZHMgQ2xvdWRGb3JtYXRpb24gcmVzcG9uc2UganVzdCBiZWZvcmUgdGhlIExhbWJkYSB0aW1lcyBvdXRcbiAgICAgKi9cbiAgICB0aW1lb3V0KCkge1xuICAgICAgICBjb25zdCBzZWxmID0gdGhpcztcbiAgICAgICAgY29uc3QgaGFuZGxlciA9ICgpID0+IHtcbiAgICAgICAgICAgIHNlbGYubG9nZ2VyLmVycm9yKCdUaW1lb3V0IEZBSUxVUkUhJyk7XG4gICAgICAgICAgICBuZXcgUHJvbWlzZSgoKSA9PlxuICAgICAgICAgICAgICAgIHNlbGYuc2VuZFJlc3BvbnNlKCdGQUlMRUQnLCAnRnVuY3Rpb24gdGltZWQgb3V0JylcbiAgICAgICAgICAgICkudGhlbigoKSA9PiBzZWxmLmNhbGxiYWNrKG5ldyBFcnJvcignRnVuY3Rpb24gdGltZWQgb3V0JykpKTtcbiAgICAgICAgfTtcbiAgICAgICAgc2V0VGltZW91dChoYW5kbGVyLCB0aGlzLmNvbnRleHQuZ2V0UmVtYWluaW5nVGltZUluTWlsbGlzKCkgLSAxMDAwKTtcbiAgICB9XG5cbiAgICAvKipcbiAgICAgKiBTZW5kcyBDbG91ZEZvcm1hdGlvbiByZXNwb25zZVxuICAgICAqL1xuICAgIHNlbmRSZXNwb25zZShyZXNwb25zZVN0YXR1czogc3RyaW5nLCByZXNwb25zZURhdGE6IHN0cmluZykge1xuICAgICAgICBjb25zdCBzZWxmID0gdGhpcztcbiAgICAgICAgdGhpcy5sb2dnZXIuZGVidWcoXG4gICAgICAgICAgICBgU2VuZGluZyByZXNwb25zZSAke3Jlc3BvbnNlU3RhdHVzfTpcXG4ke0pTT04uc3RyaW5naWZ5KFxuICAgICAgICAgICAgICAgIHJlc3BvbnNlRGF0YVxuICAgICAgICAgICAgKX1gXG4gICAgICAgICk7XG5cbiAgICAgICAgY29uc3QgZGF0YSA9IHRoaXMuUmVzcG9uc2VEYXRhO1xuICAgICAgICBkYXRhWydNZXNzYWdlJ10gPSByZXNwb25zZURhdGE7XG5cbiAgICAgICAgY29uc3QgYm9keSA9IEpTT04uc3RyaW5naWZ5KHtcbiAgICAgICAgICAgIFN0YXR1czogcmVzcG9uc2VTdGF0dXMsXG4gICAgICAgICAgICBSZWFzb246IGAke3Jlc3BvbnNlRGF0YX0gfCBGdWxsIGVycm9yIGluIENsb3VkV2F0Y2ggJHt0aGlzLmNvbnRleHQubG9nU3RyZWFtTmFtZX1gLFxuICAgICAgICAgICAgUGh5c2ljYWxSZXNvdXJjZUlkOiB0aGlzLmV2ZW50LlJlc291cmNlUHJvcGVydGllcy5OYW1lLFxuICAgICAgICAgICAgU3RhY2tJZDogdGhpcy5ldmVudC5TdGFja0lkLFxuICAgICAgICAgICAgUmVxdWVzdElkOiB0aGlzLmV2ZW50LlJlcXVlc3RJZCxcbiAgICAgICAgICAgIExvZ2ljYWxSZXNvdXJjZUlkOiB0aGlzLmV2ZW50LkxvZ2ljYWxSZXNvdXJjZUlkLFxuICAgICAgICAgICAgRGF0YTogZGF0YSxcbiAgICAgICAgfSk7XG5cbiAgICAgICAgdGhpcy5sb2dnZXIuZGVidWcoJ1JFU1BPTlNFIEJPRFk6XFxuJywgYm9keSk7XG5cbiAgICAgICAgY29uc3QgdXJsID0gVVJMLnBhcnNlKHRoaXMuZXZlbnQuUmVzcG9uc2VVUkwpO1xuICAgICAgICBjb25zdCBvcHRpb25zID0ge1xuICAgICAgICAgICAgaG9zdG5hbWU6IHVybC5ob3N0bmFtZSxcbiAgICAgICAgICAgIHBvcnQ6IDQ0MyxcbiAgICAgICAgICAgIHBhdGg6IHVybC5wYXRoLFxuICAgICAgICAgICAgbWV0aG9kOiAnUFVUJyxcbiAgICAgICAgICAgIGhlYWRlcnM6IHtcbiAgICAgICAgICAgICAgICAnY29udGVudC10eXBlJzogJycsXG4gICAgICAgICAgICAgICAgJ2NvbnRlbnQtbGVuZ3RoJzogYm9keS5sZW5ndGgsXG4gICAgICAgICAgICB9LFxuICAgICAgICB9O1xuXG4gICAgICAgIHRoaXMubG9nZ2VyLmluZm8oJ1NFTkRJTkcgUkVTUE9OU0UuLi5cXG4nKTtcblxuICAgICAgICBjb25zdCByZXF1ZXN0ID0gaHR0cHMucmVxdWVzdChvcHRpb25zLCBmdW5jdGlvbiAocmVzcG9uc2U6IGFueSkge1xuICAgICAgICAgICAgc2VsZi5sb2dnZXIuZGVidWcoYFNUQVRVUzogJHtyZXNwb25zZS5zdGF0dXNDb2RlfWApO1xuICAgICAgICAgICAgc2VsZi5sb2dnZXIuZGVidWcoYEhFQURFUlM6ICR7SlNPTi5zdHJpbmdpZnkocmVzcG9uc2UuaGVhZGVycyl9YCk7XG4gICAgICAgICAgICBzZWxmLmNvbnRleHQuZG9uZSgpO1xuICAgICAgICB9KTtcblxuICAgICAgICByZXF1ZXN0Lm9uKCdlcnJvcicsIGZ1bmN0aW9uIChlcnJvcjogRXJyb3IpIHtcbiAgICAgICAgICAgIHNlbGYubG9nZ2VyLmVycm9yKGBzZW5kUmVzcG9uc2UgRXJyb3I6ICR7ZXJyb3J9YCk7XG4gICAgICAgICAgICBzZWxmLmNvbnRleHQuZG9uZSgpO1xuICAgICAgICB9KTtcblxuICAgICAgICByZXF1ZXN0LndyaXRlKGJvZHkpO1xuICAgICAgICByZXF1ZXN0LmVuZCgpO1xuICAgIH1cbn1cblxuLyoqXG4gKiBMb2dnZXIgY2xhc3NcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBMb2dnZXIge1xuICAgIGxvZyhtZXNzYWdlOiBhbnksIC4uLm9wdGlvbmFsUGFyYW1zOiBhbnlbXSk6IHZvaWQ7XG4gICAgaW5mbyhtZXNzYWdlOiBhbnksIC4uLm9wdGlvbmFsUGFyYW1zOiBhbnlbXSk6IHZvaWQ7XG4gICAgZGVidWcobWVzc2FnZTogYW55LCAuLi5vcHRpb25hbFBhcmFtczogYW55W10pOiB2b2lkO1xuICAgIHdhcm4obWVzc2FnZTogYW55LCAuLi5vcHRpb25hbFBhcmFtczogYW55W10pOiB2b2lkO1xuICAgIGVycm9yKG1lc3NhZ2U6IGFueSwgLi4ub3B0aW9uYWxQYXJhbXM6IGFueVtdKTogdm9pZDtcbn1cblxuLyoqXG4gKiBMb2dMZXZlbHMgc3VwcG9ydGVkIGJ5IHRoZSBsb2dnZXJcbiAqL1xuZXhwb3J0IGNvbnN0IGVudW0gTG9nTGV2ZWwge1xuICAgIEVSUk9SLFxuICAgIFdBUk4sXG4gICAgSU5GTyxcbiAgICBERUJVRyxcbn1cblxuLyoqXG4gKiBTdGFuZGFyZCBsb2dnZXIgY2xhc3NcbiAqL1xuZXhwb3J0IGNsYXNzIFN0YW5kYXJkTG9nZ2VyIHtcbiAgICAvKipcbiAgICAgKiBUaGUgbG9nIGxldmVsXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCBMb2dMZXZlbC5XQVJOXG4gICAgICovXG4gICAgbGV2ZWw6IExvZ0xldmVsO1xuXG4gICAgY29uc3RydWN0b3IobGV2ZWw/OiBMb2dMZXZlbCkge1xuICAgICAgICB0aGlzLmxldmVsID0gbGV2ZWwgfHwgTG9nTGV2ZWwuV0FSTjtcbiAgICB9XG5cbiAgICAvKipcbiAgICAgKiBMb2dzIG1lc3NhZ2Ugd2l0aCBsZXZlbCBFUlJPUlxuICAgICAqL1xuICAgIGVycm9yKG1lc3NhZ2U6IGFueSwgLi4ub3B0aW9uYWxQYXJhbXM6IGFueVtdKSB7XG4gICAgICAgIGlmICh0aGlzLmxldmVsIDwgTG9nTGV2ZWwuRVJST1IpIHJldHVybjtcbiAgICAgICAgY29uc29sZS5lcnJvcihtZXNzYWdlLCAuLi5vcHRpb25hbFBhcmFtcyk7XG4gICAgfVxuXG4gICAgLyoqXG4gICAgICogTG9ncyBtZXNzYWdlIHdpdGggbGV2ZWwgV0FSTlxuICAgICAqL1xuICAgIHdhcm4obWVzc2FnZTogYW55LCAuLi5vcHRpb25hbFBhcmFtczogYW55W10pIHtcbiAgICAgICAgaWYgKHRoaXMubGV2ZWwgPCBMb2dMZXZlbC5XQVJOKSByZXR1cm47XG4gICAgICAgIGNvbnNvbGUud2FybihtZXNzYWdlLCAuLi5vcHRpb25hbFBhcmFtcyk7XG4gICAgfVxuXG4gICAgLyoqXG4gICAgICogTG9ncyBtZXNzYWdlIHdpdGggbGV2ZWwgSU5GT1xuICAgICAqL1xuICAgIGluZm8obWVzc2FnZTogYW55LCAuLi5vcHRpb25hbFBhcmFtczogYW55W10pIHtcbiAgICAgICAgaWYgKHRoaXMubGV2ZWwgPCBMb2dMZXZlbC5JTkZPKSByZXR1cm47XG4gICAgICAgIGNvbnNvbGUuaW5mbyhtZXNzYWdlLCAuLi5vcHRpb25hbFBhcmFtcyk7XG4gICAgfVxuXG4gICAgLyoqXG4gICAgICogTG9ncyBtZXNzYWdlIHdpdGggbGV2ZWwgREVCVUdcbiAgICAgKi9cbiAgICBkZWJ1ZyhtZXNzYWdlOiBhbnksIC4uLm9wdGlvbmFsUGFyYW1zOiBhbnlbXSkge1xuICAgICAgICBpZiAodGhpcy5sZXZlbCA8IExvZ0xldmVsLkRFQlVHKSByZXR1cm47XG4gICAgICAgIGNvbnNvbGUuZGVidWcobWVzc2FnZSwgLi4ub3B0aW9uYWxQYXJhbXMpO1xuICAgIH1cblxuICAgIC8qKlxuICAgICAqIEFsaWFzIGZvciBpbmZvXG4gICAgICovXG4gICAgbG9nKG1lc3NhZ2U6IGFueSwgLi4ub3B0aW9uYWxQYXJhbXM6IGFueVtdKSB7XG4gICAgICAgIHRoaXMuaW5mbyhtZXNzYWdlLCAuLi5vcHRpb25hbFBhcmFtcyk7XG4gICAgfVxufVxuIl19