"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AutomatedWaf = exports.LogLevel = exports.Waf2ScopeOption = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const glue = require("@aws-cdk/aws-glue-alpha");
const cdk = require("aws-cdk-lib");
const apigateway = require("aws-cdk-lib/aws-apigateway");
const athena = require("aws-cdk-lib/aws-athena");
const cloudwatch = require("aws-cdk-lib/aws-cloudwatch");
const events = require("aws-cdk-lib/aws-events");
const targets = require("aws-cdk-lib/aws-events-targets");
const cfnGlue = require("aws-cdk-lib/aws-glue");
const iam = require("aws-cdk-lib/aws-iam");
const firehose = require("aws-cdk-lib/aws-kinesisfirehose");
const lambda = require("aws-cdk-lib/aws-lambda");
const s3 = require("aws-cdk-lib/aws-s3");
const s3n = require("aws-cdk-lib/aws-s3-notifications");
const wafv2 = require("aws-cdk-lib/aws-wafv2");
const cr = require("aws-cdk-lib/custom-resources");
const constructs_1 = require("constructs");
/**
 * @experimental
 */
var Waf2ScopeOption;
(function (Waf2ScopeOption) {
    Waf2ScopeOption["CLOUDFRONT"] = "CLOUDFRONT";
    Waf2ScopeOption["REGIONAL"] = "REGIONAL";
})(Waf2ScopeOption = exports.Waf2ScopeOption || (exports.Waf2ScopeOption = {}));
/**
 * @experimental
 */
var LogLevel;
(function (LogLevel) {
    LogLevel["DEBUG"] = "DEBUG";
    LogLevel["INFO"] = "INFO";
    LogLevel["WARNING"] = "WARNING";
    LogLevel["ERROR"] = "ERROR";
    LogLevel["CRITICAL"] = "CRITICAL";
})(LogLevel = exports.LogLevel || (exports.LogLevel = {}));
const Elastic_Load_Balancing_account_ID = {
    'us-east-1': '127311923021',
    'us-east-2': '033677994240',
    'us-west-1': '027434742980',
    'us-west-2': '797873946194',
    'af-south-1': '098369216593',
    'ca-central-1': '985666609251',
    'eu-central-1': '054676820928',
    'eu-west-1': '156460612806',
    'eu-west-2': '652711504416',
    'eu-south-1': '635631232127',
    'eu-west-3': '009996457667',
    'eu-north-1': '897822967062',
    'ap-east-1': '754344448648',
    'ap-northeast-1': '582318560864',
    'ap-northeast-2': '600734575887',
    'ap-northeast-3': '383597477331',
    'ap-southeast-1': '114774131450',
    'ap-southeast-2': '783225319266',
    'ap-south-1': '718504428378',
    'me-south-1': '076674570225',
    'sa-east-1': '507241528517',
    'us-gov-west-1': '048591011584',
    'us-gov-east-1': '190560391635',
    'cn-north-1': '638102146993',
    'cn-northwest-1': '037604701340',
};
/**
 * @experimental
 */
class AutomatedWaf extends constructs_1.Construct {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        var _b, _c, _d, _e, _f, _g, _h;
        super(scope, id);
        const stack = cdk.Stack.of(this);
        if (props.resourceNamingPrefix &&
            !this.validateResourceNamingPrefix(props.resourceNamingPrefix)) {
            throw new Error('resourceNamingPrefix property only allow alphanumeric and "_" symbol because glue database naming is needed');
        }
        //default value
        const errorThreshold = (_b = props.errorThreshold) !== null && _b !== void 0 ? _b : 200;
        const countMode = (_c = props.countMode) !== null && _c !== void 0 ? _c : false;
        const requestThreshold = (_d = props.requestThreshold) !== null && _d !== void 0 ? _d : 1000;
        const blockPeriod = (_e = props.blockPeriod) !== null && _e !== void 0 ? _e : 240;
        const enableShieldAdvancedLambda = (_f = props.enableShieldAdvancedLambda) !== null && _f !== void 0 ? _f : false;
        const logLevel = (_g = props.logLevel) !== null && _g !== void 0 ? _g : 'INFO';
        const cloudWatchDashboardName = `WAFMonitoringDashboard${props.resourceNamingPrefix}${cdk.Aws.REGION}`;
        const reputationListName = cdk.Fn.ref('AWS::StackName') + 'IPReputationListsRule';
        const allowListName = cdk.Fn.ref('AWS::StackName') + 'WhitelistRule';
        const blacklistRuleName = cdk.Fn.ref('AWS::StackName') + 'BlacklistRule';
        const httpFloodRegularRuleName = cdk.Fn.ref('AWS::StackName') + 'HttpFloodRegularRule';
        const scannersProbesRuleName = cdk.Fn.ref('AWS::StackName') + 'ScannersProbesRule';
        const badBotRuleName = cdk.Fn.ref('AWS::StackName') + 'BadBotRule';
        const sqlInjectionRuleName = cdk.Fn.ref('AWS::StackName') + 'SqlInjectionRule';
        const xssRuleName = cdk.Fn.ref('AWS::StackName') + 'XssRule';
        const wafLogBucketAppendProps = props.wafLogBucketName
            ? { bucketName: props.wafLogBucketName }
            : {};
        //WafLogBucket
        const wafLogBucket = new s3.Bucket(this, 'WafLogBucket', {
            ...wafLogBucketAppendProps,
            publicReadAccess: false,
            encryption: s3.BucketEncryption.KMS_MANAGED,
            removalPolicy: cdk.RemovalPolicy.DESTROY,
            autoDeleteObjects: true,
        });
        const accessLogBucketAppendProps = props.appAccessLogBucketName
            ? { bucketName: props.appAccessLogBucketName }
            : {};
        const accessLogBucket = new s3.Bucket(this, 'AccessLogBucket', {
            ...accessLogBucketAppendProps,
            publicReadAccess: false,
            encryption: s3.BucketEncryption.S3_MANAGED,
            accessControl: props.waf2Scope == Waf2ScopeOption.CLOUDFRONT ? s3.BucketAccessControl.LOG_DELIVERY_WRITE : s3.BucketAccessControl.PRIVATE,
            removalPolicy: cdk.RemovalPolicy.DESTROY,
            autoDeleteObjects: true,
        });
        if (props.waf2Scope == Waf2ScopeOption.REGIONAL) {
            accessLogBucket.addToResourcePolicy(new iam.PolicyStatement({
                effect: iam.Effect.ALLOW,
                actions: ['s3:PutObject'],
                resources: [
                    `${accessLogBucket.bucketArn}/AWSLogs/${cdk.Aws.ACCOUNT_ID}/*`,
                ],
                principals: [
                    new iam.AccountPrincipal(Elastic_Load_Balancing_account_ID[stack.region]),
                ],
            }));
            accessLogBucket.addToResourcePolicy(new iam.PolicyStatement({
                effect: iam.Effect.ALLOW,
                actions: ['s3:PutObject'],
                resources: [
                    `${accessLogBucket.bucketArn}/AWSLogs/${cdk.Aws.ACCOUNT_ID}/*`,
                ],
                principals: [new iam.ServicePrincipal('delivery.logs.amazonaws.com')],
                conditions: {
                    StringEquals: {
                        's3:x-amz-acl': 'bucket-owner-full-control',
                    },
                },
            }));
            accessLogBucket.addToResourcePolicy(new iam.PolicyStatement({
                effect: iam.Effect.ALLOW,
                actions: ['s3:GetBucketAcl'],
                resources: [accessLogBucket.bucketArn],
                principals: [new iam.ServicePrincipal('delivery.logs.amazonaws.com')],
            }));
        }
        // Setup Whitelist IP Set
        const whitelistIpSetV4 = new wafv2.CfnIPSet(this, 'WhitelistSetIPV4', {
            addresses: [],
            ipAddressVersion: 'IPV4',
            scope: props.waf2Scope,
            name: cdk.Fn.ref('AWS::StackName') + 'WhitelistSetIPV4',
            description: 'Allow whitelist for IPV4 addresses',
        });
        const whitelistIpSetV6 = new wafv2.CfnIPSet(this, 'WhitelistSetIPV6', {
            addresses: [],
            ipAddressVersion: 'IPV6',
            scope: props.waf2Scope,
            name: cdk.Fn.ref('AWS::StackName') + 'WhitelistSetIPV6',
            description: 'Allow whitelist for IPV6 addresses',
        });
        // Setup Blacklist IP Set
        const blacklistIpSetV4 = new wafv2.CfnIPSet(this, 'BlacklistSetIPV4', {
            addresses: [],
            ipAddressVersion: 'IPV4',
            scope: props.waf2Scope,
            name: cdk.Fn.ref('AWS::StackName') + 'BlacklistSetIPV4',
            description: 'Allow blacklist for IPV4 addresses',
        });
        const blacklistIpSetV6 = new wafv2.CfnIPSet(this, 'BlacklistSetIPV6', {
            addresses: [],
            ipAddressVersion: 'IPV6',
            scope: props.waf2Scope,
            name: cdk.Fn.ref('AWS::StackName') + 'BlacklistSetIPV6',
            description: 'Allow blacklist for IPV6 addresses',
        });
        // Setup HTTP Flood IP Set
        const httpFloodIpSetV4 = new wafv2.CfnIPSet(this, 'HTTPFloodSetIPV4', {
            addresses: [],
            ipAddressVersion: 'IPV4',
            scope: props.waf2Scope,
            name: cdk.Fn.ref('AWS::StackName') + 'HTTPFloodSetIPV4',
            description: 'Block HTTP Flood IPV4 addresses',
        });
        const httpFloodIpSetV6 = new wafv2.CfnIPSet(this, 'HTTPFloodSetIPV6', {
            addresses: [],
            ipAddressVersion: 'IPV6',
            scope: props.waf2Scope,
            name: cdk.Fn.ref('AWS::StackName') + 'HTTPFloodSetIPV6',
            description: 'Block HTTP Flood IPV6 addresses',
        });
        // Block Scanners/Probes IP Set
        const scannersProbesIpSetV4 = new wafv2.CfnIPSet(this, 'ScannersProbesSetIPV4', {
            addresses: [],
            ipAddressVersion: 'IPV4',
            scope: props.waf2Scope,
            name: cdk.Fn.ref('AWS::StackName') + 'ScannersProbesSetIPV4',
            description: 'Block Scanners/Probes IPV4 addresses',
        });
        const scannersProbesIpSetV6 = new wafv2.CfnIPSet(this, 'ScannersProbesSetIPV6', {
            addresses: [],
            ipAddressVersion: 'IPV6',
            scope: props.waf2Scope,
            name: cdk.Fn.ref('AWS::StackName') + 'ScannersProbesSetIPV6',
            description: 'Block Scanners/Probes IPV6 addresses',
        });
        // Block Reputation List IP Set
        const reputationListsIpSetV4 = new wafv2.CfnIPSet(this, 'IPReputationListsSetIPV4', {
            addresses: [],
            ipAddressVersion: 'IPV4',
            scope: props.waf2Scope,
            name: cdk.Fn.ref('AWS::StackName') + 'IPReputationListsSetIPV4',
            description: 'Block Reputation List IPV4 addresses',
        });
        const reputationListsIpSetV6 = new wafv2.CfnIPSet(this, 'IPReputationListsSetIPV6', {
            addresses: [],
            ipAddressVersion: 'IPV6',
            scope: props.waf2Scope,
            name: cdk.Fn.ref('AWS::StackName') + 'IPReputationListsSetIPV6',
            description: 'Block Reputation List IPV6 addresses',
        });
        // Block Bad Bot IP Set
        const badBotIpSetV4 = new wafv2.CfnIPSet(this, 'IPBadBotSetIPV4', {
            addresses: [],
            ipAddressVersion: 'IPV4',
            scope: props.waf2Scope,
            name: cdk.Fn.ref('AWS::StackName') + 'IPBadBotSetIPV4',
            description: 'Block Bad Bot IPV4 addresses',
        });
        const badBotIpSetV6 = new wafv2.CfnIPSet(this, 'IPBadBotSetIPV6', {
            addresses: [],
            ipAddressVersion: 'IPV6',
            scope: props.waf2Scope,
            name: cdk.Fn.ref('AWS::StackName') + 'IPBadBotSetIPV6',
            description: 'Block Bad Bot IPV6 addresses',
        });
        const badBotWafRules = [];
        if (props.waf2Scope == Waf2ScopeOption.CLOUDFRONT) {
            badBotWafRules.push({
                name: badBotRuleName,
                priority: 7,
                action: !countMode
                    ? {
                        block: {},
                    }
                    : { count: {} },
                visibilityConfig: {
                    sampledRequestsEnabled: true,
                    cloudWatchMetricsEnabled: true,
                    metricName: badBotRuleName,
                },
                statement: {
                    orStatement: {
                        statements: [
                            {
                                ipSetReferenceStatement: {
                                    arn: cdk.Fn.getAtt(badBotIpSetV4.logicalId, 'Arn').toString(),
                                },
                            },
                            {
                                ipSetReferenceStatement: {
                                    arn: cdk.Fn.getAtt(badBotIpSetV6.logicalId, 'Arn').toString(),
                                },
                            },
                        ],
                    },
                },
            });
        }
        // WAF Web ACL
        const wafweb = new wafv2.CfnWebACL(this, 'wafweb', {
            name: props.resourceNamingPrefix
                ? `${props.resourceNamingPrefix}-WAF`
                : 'CloudFront-Web-WAF',
            description: 'Custom WAFWebACL',
            defaultAction: {
                allow: {},
            },
            scope: props.waf2Scope,
            visibilityConfig: {
                cloudWatchMetricsEnabled: true,
                sampledRequestsEnabled: true,
                metricName: props.resourceNamingPrefix
                    ? `${props.resourceNamingPrefix}-WAF`
                    : 'CloudFront-Web-WAF',
            },
            rules: [
                {
                    name: 'AWS-AWSManagedRulesCommonRuleSet',
                    priority: 0,
                    overrideAction: !countMode
                        ? {
                            none: {},
                        }
                        : { count: {} },
                    visibilityConfig: {
                        cloudWatchMetricsEnabled: true,
                        sampledRequestsEnabled: true,
                        metricName: props.resourceNamingPrefix
                            ? `${props.resourceNamingPrefix}-waf-ipset-metrics`
                            : 'cloudfront-waf-ipset-metrics',
                    },
                    statement: {
                        managedRuleGroupStatement: {
                            vendorName: 'AWS',
                            name: 'AWSManagedRulesCommonRuleSet',
                        },
                    },
                },
                {
                    name: allowListName,
                    priority: 1,
                    action: !countMode
                        ? {
                            allow: {},
                        }
                        : { count: {} },
                    visibilityConfig: {
                        sampledRequestsEnabled: true,
                        cloudWatchMetricsEnabled: true,
                        metricName: allowListName,
                    },
                    statement: {
                        orStatement: {
                            statements: [
                                {
                                    ipSetReferenceStatement: {
                                        arn: cdk.Fn.getAtt(whitelistIpSetV4.logicalId, 'Arn').toString(),
                                    },
                                },
                                {
                                    ipSetReferenceStatement: {
                                        arn: cdk.Fn.getAtt(whitelistIpSetV6.logicalId, 'Arn').toString(),
                                    },
                                },
                            ],
                        },
                    },
                },
                {
                    name: blacklistRuleName,
                    priority: 2,
                    action: !countMode
                        ? {
                            block: {},
                        }
                        : { count: {} },
                    visibilityConfig: {
                        sampledRequestsEnabled: true,
                        cloudWatchMetricsEnabled: true,
                        metricName: blacklistRuleName,
                    },
                    statement: {
                        orStatement: {
                            statements: [
                                {
                                    ipSetReferenceStatement: {
                                        arn: cdk.Fn.getAtt(blacklistIpSetV4.logicalId, 'Arn').toString(),
                                    },
                                },
                                {
                                    ipSetReferenceStatement: {
                                        arn: cdk.Fn.getAtt(blacklistIpSetV6.logicalId, 'Arn').toString(),
                                    },
                                },
                            ],
                        },
                    },
                },
                {
                    name: httpFloodRegularRuleName,
                    priority: 3,
                    action: !countMode
                        ? {
                            block: {},
                        }
                        : { count: {} },
                    visibilityConfig: {
                        sampledRequestsEnabled: true,
                        cloudWatchMetricsEnabled: true,
                        metricName: httpFloodRegularRuleName,
                    },
                    statement: {
                        orStatement: {
                            statements: [
                                {
                                    ipSetReferenceStatement: {
                                        arn: cdk.Fn.getAtt(httpFloodIpSetV4.logicalId, 'Arn').toString(),
                                    },
                                },
                                {
                                    ipSetReferenceStatement: {
                                        arn: cdk.Fn.getAtt(httpFloodIpSetV6.logicalId, 'Arn').toString(),
                                    },
                                },
                            ],
                        },
                    },
                },
                {
                    name: scannersProbesRuleName,
                    priority: 5,
                    action: !countMode
                        ? {
                            block: {},
                        }
                        : { count: {} },
                    visibilityConfig: {
                        sampledRequestsEnabled: true,
                        cloudWatchMetricsEnabled: true,
                        metricName: scannersProbesRuleName,
                    },
                    statement: {
                        orStatement: {
                            statements: [
                                {
                                    ipSetReferenceStatement: {
                                        arn: cdk.Fn.getAtt(scannersProbesIpSetV4.logicalId, 'Arn').toString(),
                                    },
                                },
                                {
                                    ipSetReferenceStatement: {
                                        arn: cdk.Fn.getAtt(scannersProbesIpSetV6.logicalId, 'Arn').toString(),
                                    },
                                },
                            ],
                        },
                    },
                },
                {
                    name: reputationListName,
                    priority: 6,
                    action: !countMode
                        ? {
                            block: {},
                        }
                        : { count: {} },
                    visibilityConfig: {
                        sampledRequestsEnabled: true,
                        cloudWatchMetricsEnabled: true,
                        metricName: reputationListName,
                    },
                    statement: {
                        orStatement: {
                            statements: [
                                {
                                    ipSetReferenceStatement: {
                                        arn: cdk.Fn.getAtt(reputationListsIpSetV4.logicalId, 'Arn').toString(),
                                    },
                                },
                                {
                                    ipSetReferenceStatement: {
                                        arn: cdk.Fn.getAtt(reputationListsIpSetV6.logicalId, 'Arn').toString(),
                                    },
                                },
                            ],
                        },
                    },
                },
                {
                    name: sqlInjectionRuleName,
                    priority: 20,
                    action: !countMode
                        ? {
                            block: {},
                        }
                        : { count: {} },
                    visibilityConfig: {
                        sampledRequestsEnabled: true,
                        cloudWatchMetricsEnabled: true,
                        metricName: sqlInjectionRuleName,
                    },
                    statement: {
                        orStatement: {
                            statements: [
                                {
                                    sqliMatchStatement: {
                                        fieldToMatch: {
                                            queryString: {},
                                        },
                                        textTransformations: [
                                            {
                                                priority: 1,
                                                type: 'URL_DECODE',
                                            },
                                            {
                                                priority: 2,
                                                type: 'HTML_ENTITY_DECODE',
                                            },
                                        ],
                                    },
                                },
                                {
                                    sqliMatchStatement: {
                                        fieldToMatch: {
                                            body: {},
                                        },
                                        textTransformations: [
                                            {
                                                priority: 1,
                                                type: 'URL_DECODE',
                                            },
                                            {
                                                priority: 2,
                                                type: 'HTML_ENTITY_DECODE',
                                            },
                                        ],
                                    },
                                },
                                {
                                    sqliMatchStatement: {
                                        fieldToMatch: {
                                            uriPath: {},
                                        },
                                        textTransformations: [
                                            {
                                                priority: 1,
                                                type: 'URL_DECODE',
                                            },
                                            {
                                                priority: 2,
                                                type: 'HTML_ENTITY_DECODE',
                                            },
                                        ],
                                    },
                                },
                                {
                                    sqliMatchStatement: {
                                        fieldToMatch: {
                                            singleHeader: { Name: 'Authorization' },
                                        },
                                        textTransformations: [
                                            {
                                                priority: 1,
                                                type: 'URL_DECODE',
                                            },
                                            {
                                                priority: 2,
                                                type: 'HTML_ENTITY_DECODE',
                                            },
                                        ],
                                    },
                                },
                                {
                                    sqliMatchStatement: {
                                        fieldToMatch: {
                                            singleHeader: { Name: 'Cookie' },
                                        },
                                        textTransformations: [
                                            {
                                                priority: 1,
                                                type: 'URL_DECODE',
                                            },
                                            {
                                                priority: 2,
                                                type: 'HTML_ENTITY_DECODE',
                                            },
                                        ],
                                    },
                                },
                            ],
                        },
                    },
                },
                {
                    name: xssRuleName,
                    priority: 30,
                    action: !countMode
                        ? {
                            block: {},
                        }
                        : { count: {} },
                    visibilityConfig: {
                        sampledRequestsEnabled: true,
                        cloudWatchMetricsEnabled: true,
                        metricName: xssRuleName,
                    },
                    statement: {
                        orStatement: {
                            statements: [
                                {
                                    xssMatchStatement: {
                                        fieldToMatch: {
                                            queryString: {},
                                        },
                                        textTransformations: [
                                            {
                                                priority: 1,
                                                type: 'URL_DECODE',
                                            },
                                            {
                                                priority: 2,
                                                type: 'HTML_ENTITY_DECODE',
                                            },
                                        ],
                                    },
                                },
                                {
                                    xssMatchStatement: {
                                        fieldToMatch: {
                                            body: {},
                                        },
                                        textTransformations: [
                                            {
                                                priority: 1,
                                                type: 'URL_DECODE',
                                            },
                                            {
                                                priority: 2,
                                                type: 'HTML_ENTITY_DECODE',
                                            },
                                        ],
                                    },
                                },
                                {
                                    xssMatchStatement: {
                                        fieldToMatch: {
                                            uriPath: {},
                                        },
                                        textTransformations: [
                                            {
                                                priority: 1,
                                                type: 'URL_DECODE',
                                            },
                                            {
                                                priority: 2,
                                                type: 'HTML_ENTITY_DECODE',
                                            },
                                        ],
                                    },
                                },
                                {
                                    xssMatchStatement: {
                                        fieldToMatch: {
                                            singleHeader: {
                                                name: 'Cookie',
                                            },
                                        },
                                        textTransformations: [
                                            {
                                                priority: 1,
                                                type: 'URL_DECODE',
                                            },
                                            {
                                                priority: 2,
                                                type: 'HTML_ENTITY_DECODE',
                                            },
                                        ],
                                    },
                                },
                            ],
                        },
                    },
                },
                ...badBotWafRules,
            ],
        });
        if (props.associatedResourceArn) {
            new wafv2.CfnWebACLAssociation(this, 'wafAssociation', {
                webAclArn: wafweb.attrArn,
                resourceArn: props.associatedResourceArn,
            });
        }
        //Lambda
        new lambda.DockerImageFunction(this, 'Helper', {
            description: "This lambda function verifies the main project's dependencies, requirements and implement auxiliary functions.",
            code: lambda.DockerImageCode.fromImageAsset(path.join(__dirname, '../source/helper'), {
                cmd: ['helper.lambda_handler'],
            }),
            memorySize: 512,
            timeout: cdk.Duration.seconds(300),
            environment: {
                LOG_LEVEL: logLevel,
                SCOPE: props.waf2Scope,
            },
        });
        const logParserLambda = new lambda.DockerImageFunction(this, 'LogParser', {
            description: 'This function parses access logs to identify suspicious behavior, such as an abnormal amount of errors. It then blocks those IP addresses for a customer-defined period of time.',
            code: lambda.DockerImageCode.fromImageAsset(path.join(__dirname, '../source/log_parser'), {
                cmd: ['log-parser.lambda_handler'],
            }),
            memorySize: 512,
            timeout: cdk.Duration.seconds(300),
            environment: {
                APP_ACCESS_LOG_BUCKET: accessLogBucket.bucketName,
                WAF_ACCESS_LOG_BUCKET: wafLogBucket.bucketName,
                SEND_ANONYMOUS_USAGE_DATA: 'No',
                UUID: 'CreateUniqueID.UUID',
                LIMIT_IP_ADDRESS_RANGES_PER_IP_MATCH_CONDITION: '10000',
                MAX_AGE_TO_UPDATE: '30',
                REGION: 'AWS::Region',
                SCOPE: props.waf2Scope,
                LOG_TYPE: props.waf2Scope == Waf2ScopeOption.CLOUDFRONT ? 'cloudfront' : 'alb',
                METRIC_NAME_PREFIX: cdk.Fn.ref('AWS::StackName'),
                LOG_LEVEL: logLevel,
                STACK_NAME: cdk.Fn.ref('AWS::StackName'),
                IP_SET_ID_HTTP_FLOODV4: httpFloodIpSetV4.attrArn,
                IP_SET_ID_HTTP_FLOODV6: httpFloodIpSetV6.attrArn,
                IP_SET_NAME_HTTP_FLOODV4: httpFloodIpSetV4.name,
                IP_SET_NAME_HTTP_FLOODV6: httpFloodIpSetV6.name,
                IP_SET_ID_SCANNERS_PROBESV4: scannersProbesIpSetV4.attrArn,
                IP_SET_ID_SCANNERS_PROBESV6: scannersProbesIpSetV6.attrArn,
                IP_SET_NAME_SCANNERS_PROBESV4: scannersProbesIpSetV4.name,
                IP_SET_NAME_SCANNERS_PROBESV6: scannersProbesIpSetV6.name,
                WAF_BLOCK_PERIOD: String(blockPeriod),
                ERROR_THRESHOLD: String(errorThreshold),
                REQUEST_THRESHOLD: String(requestThreshold),
                SOLUTION_ID: 'SO8128',
                METRICS_URL: 'https://metrics.awssolutionsbuilder.com/generic',
            },
        });
        const moveLogToPartitionLambda = new lambda.DockerImageFunction(this, 'MoveS3LogsForPartition', {
            description: 'This function is triggered by S3 event to move log files(upon their arrival in s3) from their original location to a partitioned folder structure created per timestamps in file names, hence allowing the usage of partitioning within AWS Athena.',
            code: lambda.DockerImageCode.fromImageAsset(path.join(__dirname, '../source/log_parser'), {
                cmd: ['partition_s3_logs.lambda_handler'],
            }),
            memorySize: 512,
            timeout: cdk.Duration.seconds(300),
            environment: {
                ENDPOINT: props.waf2Scope == Waf2ScopeOption.CLOUDFRONT
                    ? 'CloudFront'
                    : 'Alb',
                KEEP_ORIGINAL_DATA: 'No',
                LOG_LEVEL: logLevel,
            },
        });
        new lambda.DockerImageFunction(this, 'CustomTimer', {
            description: 'This lambda function counts X seconds and can be used to slow down component creation in CloudFormation',
            code: lambda.DockerImageCode.fromImageAsset(path.join(__dirname, '../source/timer'), {
                cmd: ['timer.lambda_handler'],
            }),
            memorySize: 128,
            timeout: cdk.Duration.seconds(300),
            environment: {
                LOG_LEVEL: logLevel,
                SECONDS: '2',
            },
        });
        //IP reputation list Lambda
        const reputationListRole = new iam.Role(this, 'LambdaRoleReputationListsParser', {
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
        });
        reputationListRole.attachInlinePolicy(new iam.Policy(this, 'CloudWatchLogs', {
            policyName: 'CloudWatchLogs',
            statements: [
                new iam.PolicyStatement({
                    effect: iam.Effect.ALLOW,
                    resources: ['*'],
                    actions: [
                        'logs:CreateLogGroup',
                        'logs:CreateLogStream',
                        'logs:PutLogEvents',
                    ],
                }),
            ],
        }));
        reputationListRole.attachInlinePolicy(new iam.Policy(this, 'WAFGetAndUpdateIPSet', {
            policyName: 'WAFGetAndUpdateIPSet',
            statements: [
                new iam.PolicyStatement({
                    effect: iam.Effect.ALLOW,
                    resources: ['*'],
                    actions: ['wafv2:GetIPSet', 'wafv2:UpdateIPSet'],
                }),
            ],
        }));
        reputationListRole.attachInlinePolicy(new iam.Policy(this, 'CloudFormationAccess', {
            policyName: 'CloudFormationAccess',
            statements: [
                new iam.PolicyStatement({
                    effect: iam.Effect.ALLOW,
                    resources: ['*'],
                    actions: ['cloudformation:DescribeStacks'],
                }),
            ],
        }));
        reputationListRole.attachInlinePolicy(new iam.Policy(this, 'CloudWatchAccess', {
            policyName: 'CloudWatchAccess',
            statements: [
                new iam.PolicyStatement({
                    effect: iam.Effect.ALLOW,
                    resources: ['*'],
                    actions: ['cloudwatch:GetMetricStatistics'],
                }),
            ],
        }));
        const reputationListsParserLambda = new lambda.DockerImageFunction(this, 'ReputationListsParser', {
            description: 'This lambda function checks third-party IP reputation lists hourly for new IP ranges to block. These lists include the Spamhaus Dont Route Or Peer (DROP) and Extended Drop (EDROP) lists, the Proofpoint Emerging Threats IP list, and the Tor exit node list.',
            code: lambda.DockerImageCode.fromImageAsset(path.join(__dirname, '../source/reputation_lists_parser'), {
                cmd: ['reputation-lists.lambda_handler'],
            }),
            memorySize: 512,
            timeout: cdk.Duration.seconds(300),
            role: reputationListRole,
            environment: {
                IP_SET_ID_REPUTATIONV4: reputationListsIpSetV4.attrArn,
                IP_SET_ID_REPUTATIONV6: reputationListsIpSetV6.attrArn,
                IP_SET_NAME_REPUTATIONV4: reputationListsIpSetV4.name,
                IP_SET_NAME_REPUTATIONV6: reputationListsIpSetV6.name,
                SCOPE: props.waf2Scope,
                LOG_LEVEL: logLevel,
                URL_LIST: '[{"url":"https://www.spamhaus.org/drop/drop.txt"},{"url":"https://www.spamhaus.org/drop/edrop.txt"},{"url":"https://check.torproject.org/exit-addresses", "prefix":"ExitAddress"},{"url":"https://rules.emergingthreats.net/fwrules/emerging-Block-IPs.txt"}]',
                SOLUTION_ID: 'SO8128',
                METRICS_URL: 'https://metrics.awssolutionsbuilder.com/generic',
                STACK_NAME: cdk.Fn.ref('AWS::StackName'),
                LOG_TYPE: props.waf2Scope == Waf2ScopeOption.CLOUDFRONT
                    ? 'cloudfront'
                    : 'alb',
                SEND_ANONYMOUS_USAGE_DATA: 'No',
                IPREPUTATIONLIST_METRICNAME: reputationListName,
            },
        });
        //Add CloudWatch event to Lambda
        const reputationListsParserRule = new events.Rule(this, 'ReputationListsParserEventsRule', {
            description: 'Security Automation - WAF Reputation Lists',
            schedule: events.Schedule.expression('rate(1 hour)'),
        });
        const reputationListsParserRuleInput = {
            URL_LIST: [
                {
                    url: 'https://www.spamhaus.org/drop/drop.txt',
                },
                {
                    url: 'https://www.spamhaus.org/drop/edrop.txt',
                },
                {
                    url: 'https://check.torproject.org/exit-addresses',
                    prefix: 'ExitAddress',
                },
                {
                    url: 'https://rules.emergingthreats.net/fwrules/emerging-Block-IPs.txt',
                },
            ],
            IP_SET_ID_REPUTATIONV4: reputationListsIpSetV4.attrArn,
            IP_SET_ID_REPUTATIONV6: reputationListsIpSetV6.attrArn,
            IP_SET_NAME_REPUTATIONV4: reputationListsIpSetV4.name,
            IP_SET_NAME_REPUTATIONV6: reputationListsIpSetV6.name,
            SCOPE: props.waf2Scope,
        };
        reputationListsParserRule.addTarget(new targets.LambdaFunction(reputationListsParserLambda, {
            event: events.RuleTargetInput.fromObject(reputationListsParserRuleInput),
        }));
        if (enableShieldAdvancedLambda) {
            //AWS Shield Advanced Lambda
            const shieldRole = new iam.Role(this, 'ShieldAdvanceRole', {
                assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            });
            shieldRole.attachInlinePolicy(new iam.Policy(this, 'ShieldAdvanceAccess', {
                policyName: 'CloudFrontShieldAdvanceAccess',
                statements: [
                    new iam.PolicyStatement({
                        effect: iam.Effect.ALLOW,
                        resources: ['*'],
                        actions: ['shield:Create*'],
                    }),
                ],
            }));
            shieldRole.attachInlinePolicy(new iam.Policy(this, 'CloudFrontDistributionAccess', {
                policyName: 'CloudFrontDistributionAccess',
                statements: [
                    new iam.PolicyStatement({
                        effect: iam.Effect.ALLOW,
                        resources: ['*'],
                        actions: ['cloudfront:GetDistribution'],
                    }),
                ],
            }));
            shieldRole.attachInlinePolicy(new iam.Policy(this, 'ShieldLogAccess', {
                policyName: 'ShieldLogAccess',
                statements: [
                    new iam.PolicyStatement({
                        effect: iam.Effect.ALLOW,
                        resources: ['*'],
                        actions: [
                            'logs:CreateLogGroup',
                            'logs:CreateLogStream',
                            'logs:PutLogEvents',
                        ],
                    }),
                ],
            }));
            new lambda.DockerImageFunction(this, 'ShieldAdvancedLambda', {
                description: 'This lambda function create an AWS Shield resource protection and protection group for the cloudfront resource.',
                code: lambda.DockerImageCode.fromImageAsset(path.join(__dirname, '../source/shield_protection'), {
                    cmd: ['shield-protection.lambda_handler'],
                }),
                memorySize: 512,
                timeout: cdk.Duration.seconds(300),
                role: shieldRole,
                environment: {
                    SCOPE: props.waf2Scope,
                    LOG_LEVEL: logLevel,
                    SOLUTION_ID: 'SO8128',
                },
            });
        }
        if (props.waf2Scope == Waf2ScopeOption.CLOUDFRONT) {
            //Badbot protection Lambda
            const badBotRole = new iam.Role(this, 'BadBotRole', {
                assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            });
            badBotRole.attachInlinePolicy(new iam.Policy(this, 'BadBotLogsAccess', {
                policyName: 'LogsAccess',
                statements: [
                    new iam.PolicyStatement({
                        effect: iam.Effect.ALLOW,
                        resources: ['*'],
                        actions: [
                            'logs:CreateLogGroup',
                            'logs:CreateLogStream',
                            'logs:PutLogEvents',
                        ],
                    }),
                ],
            }));
            badBotRole.attachInlinePolicy(new iam.Policy(this, 'BadBotCloudFormationAccess', {
                policyName: 'CloudFormationAccess',
                statements: [
                    new iam.PolicyStatement({
                        effect: iam.Effect.ALLOW,
                        resources: ['*'],
                        actions: ['cloudformation:DescribeStacks'],
                    }),
                ],
            }));
            badBotRole.attachInlinePolicy(new iam.Policy(this, 'BadBotCloudWatchAccess', {
                policyName: 'CloudWatchAccess',
                statements: [
                    new iam.PolicyStatement({
                        effect: iam.Effect.ALLOW,
                        resources: ['*'],
                        actions: ['cloudwatch:GetMetricStatistics'],
                    }),
                ],
            }));
            badBotRole.attachInlinePolicy(new iam.Policy(this, 'BadBotWAFGetAndUpdateIPSet', {
                policyName: 'WAFGetAndUpdateIPSet',
                statements: [
                    new iam.PolicyStatement({
                        effect: iam.Effect.ALLOW,
                        resources: ['*'],
                        actions: ['wafv2:GetIPSet', 'wafv2:UpdateIPSet'],
                    }),
                ],
            }));
            const badBotParserLambda = new lambda.DockerImageFunction(this, 'BadBotParser', {
                description: 'This lambda function will intercepts and inspects trap endpoint requests to extract its IP address, and then add it to an AWS WAF block list.',
                role: badBotRole,
                code: lambda.DockerImageCode.fromImageAsset(path.join(__dirname, '../source/access_handler'), {
                    cmd: ['access-handler.lambda_handler'],
                }),
                memorySize: 512,
                timeout: cdk.Duration.seconds(300),
                environment: {
                    SCOPE: props.waf2Scope,
                    IP_SET_ID_BAD_BOTV4: badBotIpSetV4.attrArn,
                    IP_SET_ID_BAD_BOTV6: badBotIpSetV6.attrArn,
                    IP_SET_NAME_BAD_BOTV4: badBotIpSetV4.name,
                    IP_SET_NAME_BAD_BOTV6: badBotIpSetV6.name,
                    SEND_ANONYMOUS_USAGE_DATA: 'No',
                    UUID: 'CreateUniqueID.UUID',
                    REGION: cdk.Fn.ref('AWS::Region'),
                    LOG_TYPE: 'cloudfront',
                    METRIC_NAME_PREFIX: cdk.Fn.ref('AWS::StackName'),
                    LOG_LEVEL: logLevel,
                    SOLUTION_ID: 'SO8128',
                    METRICS_URL: 'https://metrics.awssolutionsbuilder.com/generic',
                    STACK_NAME: cdk.Fn.ref('AWS::StackName'),
                },
            });
            //API Gateway for badbot detection
            const badBotApi = new apigateway.RestApi(this, 'ApiGatewayBadBot', {
                restApiName: 'Security Automation - WAF Bad Bot API',
                description: 'API created by AWS WAF Security Automation CloudFormation template. This endpoint will be used to capture bad bots.',
            });
            const integration = new apigateway.LambdaIntegration(badBotParserLambda);
            badBotApi.root.addMethod('ANY', integration, {
                requestParameters: {
                    'method.request.header.X-Forwarded-For': false,
                },
            });
            const badBotProxy = badBotApi.root.addResource('{proxy+}');
            badBotProxy.addMethod('ANY', integration, {
                requestParameters: {
                    'method.request.header.X-Forwarded-For': false,
                },
            });
        }
        const customResourceLambda = new lambda.DockerImageFunction(this, 'CustomResource', {
            description: 'This lambda function configures the Web ACL rules based on the features enabled in the CloudFormation template.',
            code: lambda.DockerImageCode.fromImageAsset(path.join(__dirname, '../source/custom_resource'), {
                cmd: ['custom-resource.lambda_handler'],
            }),
            memorySize: 512,
            timeout: cdk.Duration.seconds(300),
            environment: {
                LOG_LEVEL: logLevel,
                SCOPE: props.waf2Scope,
                SOLUTION_ID: 'SO8128',
                METRICS_URL: 'https://metrics.awssolutionsbuilder.com/generic',
            },
        });
        //Kinesis Data Firehose
        const firehoseRole = new iam.Role(this, 'FirehoseRole', {
            assumedBy: new iam.ServicePrincipal('firehose.amazonaws.com'),
        });
        firehoseRole.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'kinesis:DescribeStream',
                'kinesis:GetShardIterator',
                'kinesis:GetRecords',
            ],
            resources: ['*'],
        }));
        firehoseRole.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                's3:AbortMultipartUpload',
                's3:GetBucketLocation',
                's3:GetObject',
                's3:ListBucket',
                's3:ListBucketMultipartUploads',
                's3:PutObject',
            ],
            resources: [wafLogBucket.bucketArn, wafLogBucket.arnForObjects('*')],
        }));
        const firehoseStream = new firehose.CfnDeliveryStream(this, 'FirehoseWAFLogsDeliveryStream', {
            deliveryStreamName: 'aws-waf-logs-' + stack.stackName,
            deliveryStreamType: 'DirectPut',
            extendedS3DestinationConfiguration: {
                bucketArn: wafLogBucket.bucketArn,
                bufferingHints: {
                    intervalInSeconds: 300,
                    sizeInMBs: 5,
                },
                compressionFormat: 'GZIP',
                prefix: 'AWSLogs/year=!{timestamp:yyyy}/month=!{timestamp:MM}/day=!{timestamp:dd}/hour=!{timestamp:HH}/',
                errorOutputPrefix: 'AWSErrorLogs/result=!{firehose:error-output-type}/year=!{timestamp:yyyy}/month=!{timestamp:MM}/day=!{timestamp:dd}/hour=!{timestamp:HH}/',
                roleArn: firehoseRole.roleArn,
            },
        });
        //Glue DB & table
        const glueDatabasePostfix = props.resourceNamingPrefix
            ? `_${props.resourceNamingPrefix.toLowerCase()}`
            : '';
        const glueAccessLogsDatabase = new glue.Database(this, 'GlueAccessLogsDatabase', {
            databaseName: `glue_accesslogs_database${glueDatabasePostfix}`,
        });
        const glueWafAccessLogsTable = new glue.Table(this, 'GlueWafAccessLogsTable', {
            database: glueAccessLogsDatabase,
            tableName: 'waf_access_logs',
            description: stack.stackName + ' - WAF Logs',
            bucket: wafLogBucket,
            s3Prefix: 'AWSLogs/',
            columns: [
                {
                    name: 'timestamp',
                    type: glue.Schema.BIG_INT,
                },
                {
                    name: 'formatversion',
                    type: glue.Schema.INTEGER,
                },
                {
                    name: 'webaclid',
                    type: glue.Schema.STRING,
                },
                {
                    name: 'terminatingruleid',
                    type: glue.Schema.STRING,
                },
                {
                    name: 'terminatingruletype',
                    type: glue.Schema.STRING,
                },
                {
                    name: 'action',
                    type: glue.Schema.STRING,
                },
                {
                    name: 'httpsourcename',
                    type: glue.Schema.STRING,
                },
                {
                    name: 'httpsourceid',
                    type: glue.Schema.STRING,
                },
                {
                    name: 'rulegrouplist',
                    type: glue.Schema.array(glue.Schema.STRING),
                },
                {
                    name: 'ratebasedrulelist',
                    type: glue.Schema.array(glue.Schema.STRING),
                },
                {
                    name: 'nonterminatingmatchingrules',
                    type: glue.Schema.array(glue.Schema.STRING),
                },
                {
                    name: 'httprequest',
                    type: glue.Schema.struct([
                        {
                            name: 'clientip',
                            type: glue.Schema.STRING,
                        },
                        {
                            name: 'country',
                            type: glue.Schema.STRING,
                        },
                        {
                            name: 'headers',
                            type: glue.Schema.array(glue.Schema.struct([
                                {
                                    name: 'name',
                                    type: glue.Schema.STRING,
                                },
                                {
                                    name: 'value',
                                    type: glue.Schema.STRING,
                                },
                            ])),
                        },
                        {
                            name: 'uri',
                            type: glue.Schema.STRING,
                        },
                        {
                            name: 'args',
                            type: glue.Schema.STRING,
                        },
                        {
                            name: 'httpversion',
                            type: glue.Schema.STRING,
                        },
                        {
                            name: 'httpmethod',
                            type: glue.Schema.STRING,
                        },
                        {
                            name: 'requestid',
                            type: glue.Schema.STRING,
                        },
                    ]),
                },
            ],
            partitionKeys: [
                {
                    name: 'year',
                    type: glue.Schema.INTEGER,
                },
                {
                    name: 'month',
                    type: glue.Schema.INTEGER,
                },
                {
                    name: 'day',
                    type: glue.Schema.INTEGER,
                },
                {
                    name: 'hour',
                    type: glue.Schema.INTEGER,
                },
            ],
            dataFormat: {
                inputFormat: new glue.InputFormat('org.apache.hadoop.mapred.TextInputFormat'),
                outputFormat: new glue.OutputFormat('org.apache.hadoop.hive.ql.io.HiveIgnoreKeyTextOutputFormat'),
                serializationLibrary: new glue.SerializationLibrary('org.openx.data.jsonserde.JsonSerDe'),
            },
        });
        // const df = new glue.DataFormat({
        //   inputFormat: new glue.InputFormat('org.apache.hadoop.mapred.TextInputFormat'),
        //   outputFormat: new glue.OutputFormat('org.apache.hadoop.hive.ql.io.HiveIgnoreKeyTextOutputFormat'),
        //   serializationLibrary: new glue.SerializationLibrary('org.openx.data.jsonserde.JsonSerDe'),
        // });
        if (props.waf2Scope == Waf2ScopeOption.CLOUDFRONT) {
            new cfnGlue.CfnTable(this, 'glueAppAccessLogsTable', {
                databaseName: glueAccessLogsDatabase.databaseName,
                catalogId: cdk.Aws.ACCOUNT_ID,
                tableInput: {
                    name: 'app_access_logs',
                    description: stack.stackName + ' - APP Access Logs',
                    parameters: {
                        'skip.header.line.count': '2',
                        'EXTERNAL': 'TRUE',
                    },
                    storageDescriptor: {
                        columns: [
                            {
                                name: 'date',
                                type: glue.Schema.DATE.inputString,
                            },
                            {
                                name: 'time',
                                type: glue.Schema.STRING.inputString,
                            },
                            {
                                name: 'location',
                                type: glue.Schema.STRING.inputString,
                            },
                            {
                                name: 'bytes',
                                type: glue.Schema.BIG_INT.inputString,
                            },
                            {
                                name: 'requestip',
                                type: glue.Schema.STRING.inputString,
                            },
                            {
                                name: 'method',
                                type: glue.Schema.STRING.inputString,
                            },
                            {
                                name: 'host',
                                type: glue.Schema.STRING.inputString,
                            },
                            {
                                name: 'uri',
                                type: glue.Schema.STRING.inputString,
                            },
                            {
                                name: 'status',
                                type: glue.Schema.INTEGER.inputString,
                            },
                            {
                                name: 'referrer',
                                type: glue.Schema.STRING.inputString,
                            },
                            {
                                name: 'useragent',
                                type: glue.Schema.STRING.inputString,
                            },
                            {
                                name: 'querystring',
                                type: glue.Schema.STRING.inputString,
                            },
                            {
                                name: 'cookie',
                                type: glue.Schema.STRING.inputString,
                            },
                            {
                                name: 'resulttype',
                                type: glue.Schema.STRING.inputString,
                            },
                            {
                                name: 'requestid',
                                type: glue.Schema.STRING.inputString,
                            },
                            {
                                name: 'hostheader',
                                type: glue.Schema.STRING.inputString,
                            },
                            {
                                name: 'requestprotocol',
                                type: glue.Schema.STRING.inputString,
                            },
                            {
                                name: 'requestbytes',
                                type: glue.Schema.BIG_INT.inputString,
                            },
                            {
                                name: 'timetaken',
                                type: glue.Schema.FLOAT.inputString,
                            },
                            {
                                name: 'xforwardedfor',
                                type: glue.Schema.STRING.inputString,
                            },
                            {
                                name: 'sslprotocol',
                                type: glue.Schema.STRING.inputString,
                            },
                            {
                                name: 'sslcipher',
                                type: glue.Schema.STRING.inputString,
                            },
                            {
                                name: 'responseresulttype',
                                type: glue.Schema.STRING.inputString,
                            },
                            {
                                name: 'httpversion',
                                type: glue.Schema.STRING.inputString,
                            },
                            {
                                name: 'filestatus',
                                type: glue.Schema.STRING.inputString,
                            },
                            {
                                name: 'encryptedfields',
                                type: glue.Schema.INTEGER.inputString,
                            },
                        ],
                        compressed: false,
                        inputFormat: 'org.apache.hadoop.mapred.TextInputFormat',
                        location: 's3://' + accessLogBucket.bucketName + '/AWSLogs-Partitioned/',
                        outputFormat: 'org.apache.hadoop.hive.ql.io.HiveIgnoreKeyTextOutputFormat',
                        serdeInfo: {
                            serializationLibrary: 'org.apache.hadoop.hive.serde2.lazy.LazySimpleSerDe',
                            parameters: {
                                'field.delim': '	',
                                'serialization.format': '	',
                            },
                        },
                        storedAsSubDirectories: false,
                    },
                    tableType: 'EXTERNAL_TABLE',
                    partitionKeys: [
                        {
                            name: 'year',
                            type: glue.Schema.INTEGER.inputString,
                        },
                        {
                            name: 'month',
                            type: glue.Schema.INTEGER.inputString,
                        },
                        {
                            name: 'day',
                            type: glue.Schema.INTEGER.inputString,
                        },
                        {
                            name: 'hour',
                            type: glue.Schema.INTEGER.inputString,
                        },
                    ],
                },
            });
        }
        else {
            // glue access log format for Alb
            new cfnGlue.CfnTable(this, 'glueAppAccessLogsTable', {
                databaseName: glueAccessLogsDatabase.databaseName,
                catalogId: cdk.Aws.ACCOUNT_ID,
                tableInput: {
                    name: 'app_access_logs',
                    description: stack.stackName + ' - APP Access Logs',
                    parameters: {
                        EXTERNAL: 'TRUE',
                    },
                    storageDescriptor: {
                        columns: [
                            {
                                name: 'type',
                                type: glue.Schema.STRING.inputString,
                            },
                            {
                                name: 'time',
                                type: glue.Schema.STRING.inputString,
                            },
                            {
                                name: 'elb',
                                type: glue.Schema.STRING.inputString,
                            },
                            {
                                name: 'client_ip',
                                type: glue.Schema.STRING.inputString,
                            },
                            {
                                name: 'client_port',
                                type: glue.Schema.INTEGER.inputString,
                            },
                            {
                                name: 'target_ip',
                                type: glue.Schema.STRING.inputString,
                            },
                            {
                                name: 'target_port',
                                type: glue.Schema.INTEGER.inputString,
                            },
                            {
                                name: 'request_processing_time',
                                type: glue.Schema.DOUBLE.inputString,
                            },
                            {
                                name: 'target_processing_time',
                                type: glue.Schema.DOUBLE.inputString,
                            },
                            {
                                name: 'response_processing_time',
                                type: glue.Schema.DOUBLE.inputString,
                            },
                            {
                                name: 'elb_status_code',
                                type: glue.Schema.STRING.inputString,
                            },
                            {
                                name: 'target_status_code',
                                type: glue.Schema.STRING.inputString,
                            },
                            {
                                name: 'received_bytes',
                                type: glue.Schema.BIG_INT.inputString,
                            },
                            {
                                name: 'sent_bytes',
                                type: glue.Schema.BIG_INT.inputString,
                            },
                            {
                                name: 'request_verb',
                                type: glue.Schema.STRING.inputString,
                            },
                            {
                                name: 'request_url',
                                type: glue.Schema.STRING.inputString,
                            },
                            {
                                name: 'request_proto',
                                type: glue.Schema.STRING.inputString,
                            },
                            {
                                name: 'user_agent',
                                type: glue.Schema.STRING.inputString,
                            },
                            {
                                name: 'ssl_cipher',
                                type: glue.Schema.STRING.inputString,
                            },
                            {
                                name: 'ssl_protocol',
                                type: glue.Schema.STRING.inputString,
                            },
                            {
                                name: 'target_group_arn',
                                type: glue.Schema.STRING.inputString,
                            },
                            {
                                name: 'trace_id',
                                type: glue.Schema.STRING.inputString,
                            },
                            {
                                name: 'domain_name',
                                type: glue.Schema.STRING.inputString,
                            },
                            {
                                name: 'chosen_cert_arn',
                                type: glue.Schema.STRING.inputString,
                            },
                            {
                                name: 'matched_rule_priority',
                                type: glue.Schema.STRING.inputString,
                            },
                            {
                                name: 'request_creation_time',
                                type: glue.Schema.STRING.inputString,
                            },
                            {
                                name: 'actions_executed',
                                type: glue.Schema.STRING.inputString,
                            },
                            {
                                name: 'redirect_url',
                                type: glue.Schema.STRING.inputString,
                            },
                            {
                                name: 'error_reason',
                                type: glue.Schema.STRING.inputString,
                            },
                        ],
                        compressed: false,
                        inputFormat: 'org.apache.hadoop.mapred.TextInputFormat',
                        location: 's3://' + accessLogBucket.bucketName + '/AWSLogs-Partitioned/',
                        outputFormat: 'org.apache.hadoop.hive.ql.io.HiveIgnoreKeyTextOutputFormat',
                        serdeInfo: {
                            serializationLibrary: 'org.apache.hadoop.hive.serde2.RegexSerDe',
                            parameters: {
                                'serialization.format': '1',
                                'input.regex': '([^ ]*) ([^ ]*) ([^ ]*) ([^ ]*):([0-9]*) ([^ ]*)[:-]([0-9]*) ([-.0-9]*) ([-.0-9]*) ([-.0-9]*) (|[-0-9]*) (-|[-0-9]*) ([-0-9]*) ([-0-9]*) "([^ ]*) ([^ ]*) (- |[^ ]*)" "([^"]*)" ([A-Z0-9-]+) ([A-Za-z0-9.-]*) ([^ ]*) "([^"]*)" "([^"]*)" "([^"]*)" ([-.0-9]*) ([^ ]*) "([^"]*)" "([^ ]*)" "([^ ].*)"',
                            },
                        },
                        storedAsSubDirectories: false,
                    },
                    tableType: 'EXTERNAL_TABLE',
                    partitionKeys: [
                        {
                            name: 'year',
                            type: glue.Schema.INTEGER.inputString,
                        },
                        {
                            name: 'month',
                            type: glue.Schema.INTEGER.inputString,
                        },
                        {
                            name: 'day',
                            type: glue.Schema.INTEGER.inputString,
                        },
                        {
                            name: 'hour',
                            type: glue.Schema.INTEGER.inputString,
                        },
                    ],
                },
            });
        }
        //Athena
        const athenaWorkGroupPostfix = (_h = props.resourceNamingPrefix) !== null && _h !== void 0 ? _h : '';
        const addPartitionAthenaQueryWorkGroup = new athena.CfnWorkGroup(this, 'WAFAddPartitionAthenaQueryWorkGroup', {
            name: `WAFAddPartitionAthenaQueryWorkGroup${athenaWorkGroupPostfix}`,
            description: 'Athena WorkGroup for adding Athena partition queries used by AWS WAF Security Automations Solution',
            state: 'ENABLED',
            recursiveDeleteOption: true,
            workGroupConfiguration: {
                publishCloudWatchMetricsEnabled: true,
                enforceWorkGroupConfiguration: false,
            },
        });
        const wAFLogAthenaQueryWorkGroup = new athena.CfnWorkGroup(this, 'WAFLogAthenaQueryWorkGroup', {
            name: `WAFLogAthenaQueryWorkGroup${athenaWorkGroupPostfix}`,
            description: 'Athena WorkGroup for WAF log queries used by AWS WAF Security Automations Solution',
            state: 'ENABLED',
            recursiveDeleteOption: true,
            workGroupConfiguration: {
                publishCloudWatchMetricsEnabled: true,
            },
        });
        const wAFAppAccessLogAthenaQueryWorkGroup = new athena.CfnWorkGroup(this, 'WAFAppAccessLogAthenaQueryWorkGroup', {
            name: `WAFAppAccessLogAthenaQueryWorkGroup${athenaWorkGroupPostfix}`,
            description: 'Athena WorkGroup for CloudFront or ALB application access log queries used by AWS WAF Security Automations Solution',
            state: 'ENABLED',
            recursiveDeleteOption: true,
            workGroupConfiguration: {
                publishCloudWatchMetricsEnabled: true,
            },
        });
        //Cloudwatch Dashboard
        new cloudwatch.CfnDashboard(this, 'MonitoringDashboard', {
            dashboardName: cloudWatchDashboardName,
            dashboardBody: JSON.stringify({
                widgets: [
                    {
                        type: 'metric',
                        x: 0,
                        y: 0,
                        width: 15,
                        height: 10,
                        properties: {
                            view: 'timeSeries',
                            stacked: false,
                            metrics: [
                                [
                                    'WAF',
                                    'BlockedRequests',
                                    'WebACL',
                                    'WAFWebACLMetricName',
                                    'Rule',
                                    'ALL',
                                ],
                                [
                                    'WAF',
                                    'AllowedRequests',
                                    'WebACL',
                                    'WAFWebACLMetricName',
                                    'Rule',
                                    'ALL',
                                ],
                            ],
                            region: 'us-east-1',
                            period: 300,
                        },
                    },
                ],
            }),
        });
        //Add CloudWatch event to Lambda LogParser
        const logParserWafLogRuleInput = {
            resourceType: 'LambdaAthenaWAFLogParser',
            glueAccessLogsDatabase: glueAccessLogsDatabase.databaseName,
            accessLogBucket: wafLogBucket.bucketName,
            glueWafAccessLogsTable: glueWafAccessLogsTable.tableName,
            athenaWorkGroup: wAFLogAthenaQueryWorkGroup.name,
        };
        new events.Rule(this, 'lambdaAthenaWAFLogParserRule', {
            description: 'Security Automation - WAF Logs Athena parser',
            schedule: events.Schedule.expression('rate(5 minutes)'),
            targets: [
                new targets.LambdaFunction(logParserLambda, {
                    event: events.RuleTargetInput.fromObject(logParserWafLogRuleInput),
                }),
            ],
        });
        const lambdaAthenaAppLogParserRuleInput = {
            resourceType: 'LambdaAthenaAppLogParser',
            glueAccessLogsDatabase: glueAccessLogsDatabase.databaseName,
            accessLogBucket: accessLogBucket.bucketName,
            glueAppAccessLogsTable: 'app_access_logs',
            athenaWorkGroup: wAFAppAccessLogAthenaQueryWorkGroup.name,
        };
        new events.Rule(this, 'lambdaAthenaAppLogParserRule', {
            description: 'Security Automation - App Logs Athena parser',
            schedule: events.Schedule.expression('rate(5 minutes)'),
            targets: [
                new targets.LambdaFunction(logParserLambda, {
                    event: events.RuleTargetInput.fromObject(lambdaAthenaAppLogParserRuleInput),
                }),
            ],
        });
        const addAthenaPartitionsLambda = new lambda.DockerImageFunction(this, 'AddAthenaPartitionsFunction', {
            description: 'This function adds a new hourly partition to athena table. It runs every hour, triggered by a CloudWatch event.',
            code: lambda.DockerImageCode.fromImageAsset(path.join(__dirname, '../source/log_parser'), {
                cmd: ['add_athena_partitions.lambda_handler'],
            }),
            memorySize: 512,
            timeout: cdk.Duration.minutes(5),
            maxEventAge: cdk.Duration.hours(6),
            retryAttempts: 2,
            environment: {
                LOG_LEVEL: logLevel,
            },
        });
        const addAthenaPartitionsRuleInput = {
            resourceType: 'LambdaAddAthenaPartitionsEventsRule',
            glueAccessLogsDatabase: glueAccessLogsDatabase.databaseName,
            accessLogBucket: accessLogBucket.bucketName,
            glueAppAccessLogsTable: 'app_access_logs',
            glueWafAccessLogsTable: glueWafAccessLogsTable.tableName,
            wafLogBucket: wafLogBucket.bucketName,
            athenaWorkGroup: addPartitionAthenaQueryWorkGroup.name,
        };
        new events.Rule(this, 'lambdaAddAthenaPartition', {
            description: 'Security Automations - Add partitions to Athena table',
            schedule: events.Schedule.expression('rate(1 hour)'),
            targets: [
                new targets.LambdaFunction(addAthenaPartitionsLambda, {
                    event: events.RuleTargetInput.fromObject(addAthenaPartitionsRuleInput),
                }),
            ],
        });
        addAthenaPartitionsLambda.addToRolePolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                's3:AbortMultipartUpload',
                's3:GetBucketLocation',
                's3:GetObject',
                's3:ListBucket',
                's3:ListBucketMultipartUploads',
                's3:PutObject',
                'glue:GetTable',
                'glue:GetDatabase',
                'glue:UpdateDatabase',
                'glue:CreateDatabase',
                'glue:BatchCreatePartition',
                'logs:CreateLogGroup',
                'logs:CreateLogStream',
                'logs:PutLogEvents',
            ],
            resources: ['*'],
        }));
        addAthenaPartitionsLambda.addToRolePolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['athena:StartQueryExecution'],
            resources: ['*'],
        }));
        logParserLambda.addToRolePolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['glue:GetTable', 'glue:GetPartitions'],
            resources: [
                glueAccessLogsDatabase.catalogArn,
                glueAccessLogsDatabase.databaseArn,
                `arn:${cdk.Aws.PARTITION}:glue:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:table/${glueAccessLogsDatabase.databaseName}/waf_access_logs`,
                `arn:${cdk.Aws.PARTITION}:glue:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:table/${glueAccessLogsDatabase.databaseName}/app_access_logs`,
            ],
        }));
        logParserLambda.addToRolePolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['wafv2:GetIPSet', 'wafv2:UpdateIPSet'],
            resources: [
                httpFloodIpSetV4.attrArn,
                httpFloodIpSetV6.attrArn,
                scannersProbesIpSetV4.attrArn,
                scannersProbesIpSetV6.attrArn,
            ],
        }));
        logParserLambda.addToRolePolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['athena:GetNamedQuery', 'athena:StartQueryExecution'],
            resources: [
                `arn:${cdk.Aws.PARTITION}:athena:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:workgroup/WAF*`,
            ],
        }));
        logParserLambda.addToRolePolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['cloudwatch:GetMetricStatistics'],
            resources: ['*'],
        }));
        logParserLambda.addToRolePolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'logs:CreateLogGroup',
                'logs:CreateLogStream',
                'logs:PutLogEvents',
            ],
            resources: [
                `arn:${cdk.Aws.PARTITION}:athena:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:log-group:/aws/lambda/*LogParser*`,
            ],
        }));
        logParserLambda.addToRolePolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                's3:GetBucketLocation',
                's3:GetObject',
                's3:ListBucket',
                's3:ListBucketMultipartUploads',
                's3:ListMultipartUploadParts',
                's3:AbortMultipartUpload',
                's3:CreateBucket',
                's3:PutObject',
            ],
            resources: [
                accessLogBucket.bucketArn + '/athena_results/*',
                accessLogBucket.bucketArn + '/*',
                accessLogBucket.bucketArn,
                wafLogBucket.bucketArn + '/athena_results/*',
                wafLogBucket.bucketArn + '/*',
                wafLogBucket.bucketArn,
            ],
        }));
        // S3
        logParserLambda.addToRolePolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                's3:GetBucketLocation',
                's3:GetObject',
                's3:ListBucket',
                's3:ListBucketMultipartUploads',
                's3:ListMultipartUploadParts',
                's3:AbortMultipartUpload',
                's3:CreateBucket',
                's3:PutObject',
            ],
            resources: [
                wafLogBucket.bucketArn + '/athena_results/*',
                accessLogBucket.bucketArn + '/athena_results/*',
                wafLogBucket.bucketArn,
                accessLogBucket.bucketArn,
            ],
        }));
        logParserLambda.addToRolePolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['s3:GetObject'],
            resources: [
                wafLogBucket.bucketArn + '/*',
                accessLogBucket.bucketArn + '/*',
            ],
        }));
        logParserLambda.addToRolePolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['s3:PutObject'],
            resources: [
                wafLogBucket.bucketArn + '/' + stack.stackName + '-waf_log_out.json',
                wafLogBucket.bucketArn + '/' + stack.stackName + '-waf_log_conf.json',
                accessLogBucket.bucketArn +
                    '/' +
                    stack.stackName +
                    '-waf_log_out.json',
                accessLogBucket.bucketArn +
                    '/' +
                    stack.stackName +
                    '-waf_log_conf.json',
            ],
        }));
        moveLogToPartitionLambda.addToRolePolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'logs:CreateLogGroup',
                'logs:CreateLogStream',
                'logs:PutLogEvents',
            ],
            resources: [
                `arn:${cdk.Aws.PARTITION}:athena:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:log-group:/aws/lambda/*MoveS3LogsForPartition*`,
            ],
        }));
        moveLogToPartitionLambda.addToRolePolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['s3:GetObject', 's3:DeleteObject', 's3:PutObject'],
            resources: [accessLogBucket.bucketArn + '/*'],
        }));
        customResourceLambda.addToRolePolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['cloudformation:DescribeStacks'],
            resources: [
                `arn:${cdk.Aws.PARTITION}:cloudformation:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:stack/${cdk.Stack.name}/*`,
            ],
        }));
        customResourceLambda.addToRolePolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['lambda:InvokeFunction'],
            resources: [
                `arn:${cdk.Aws.PARTITION}:lambda:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:function:${stack.stackName}*`,
            ],
        }));
        customResourceLambda.addToRolePolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'wafv2:GetIPSet',
                'wafv2:DeleteIPSet',
                'wafv2:GetWebACL',
                'wafv2:UpdateWebACL',
                'wafv2:DeleteLoggingConfiguration',
                'wafv2:PutLoggingConfiguration',
            ],
            resources: [wafweb.attrArn],
        }));
        customResourceLambda.addToRolePolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['wafv2:GetIPSet', 'wafv2:DeleteIPSet'],
            resources: [
                `arn:${cdk.Aws.PARTITION}:logs:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:regional/ipset/${stack.stackName}*`,
                `arn:${cdk.Aws.PARTITION}:logs:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:global/ipset/${stack.stackName}*`,
            ],
        }));
        customResourceLambda.addToRolePolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'logs:CreateLogGroup',
                'logs:CreateLogStream',
                'logs:PutLogEvents',
            ],
            resources: [
                `arn:${cdk.Aws.PARTITION}:logs:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:log-group:/aws/lambda/*CustomResource*`,
            ],
        }));
        customResourceLambda.addToRolePolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                's3:GetBucketLocation',
                's3:ListBucket',
                's3:CreateBucket',
                's3:GetBucketNotification',
                's3:PutBucketNotification',
                's3:PutEncryptionConfiguration',
                's3:PutBucketPublicAccessBlock',
                's3:GetObject*',
                's3:PutObject*',
                's3:DeleteObject*',
            ],
            resources: [
                accessLogBucket.bucketArn,
                `${accessLogBucket.bucketArn}/*`,
            ],
        }));
        customResourceLambda.addToRolePolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                's3:CreateBucket',
                's3:GetBucketNotification',
                's3:PutBucketNotification',
                's3:GetObject*',
                's3:PutObject*',
                's3:DeleteObject*',
            ],
            resources: [wafLogBucket.bucketArn, `${wafLogBucket.bucketArn}/*`],
        }));
        customResourceLambda.addToRolePolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['iam:CreateServiceLinkedRole'],
            resources: [
                'arn:aws:iam::*:role/aws-service-role/wafv2.amazonaws.com/AWSServiceRoleForWAFV2Logging',
            ],
        }));
        wafLogBucket.addEventNotification(s3.EventType.OBJECT_CREATED, new s3n.LambdaDestination(logParserLambda), {
            prefix: 'athena_results/',
            suffix: 'csv',
        });
        accessLogBucket.addEventNotification(s3.EventType.OBJECT_CREATED, new s3n.LambdaDestination(moveLogToPartitionLambda), {
            prefix: 'AWSLogs/',
            suffix: 'gz',
        });
        accessLogBucket.addEventNotification(s3.EventType.OBJECT_CREATED, new s3n.LambdaDestination(logParserLambda), {
            prefix: 'athena_results/',
            suffix: 'csv',
        });
        const customResourceProvider = new cr.Provider(this, 'customResourceProvider', {
            onEventHandler: customResourceLambda,
        });
        new cdk.CustomResource(this, 'ConfigureAWSWAFLogs', {
            serviceToken: customResourceProvider.serviceToken,
            resourceType: 'Custom::ConfigureAWSWAFLogs',
            properties: {
                WAFWebACLArn: wafweb.attrArn,
                DeliveryStreamArn: firehoseStream.attrArn,
            },
        });
        new cdk.CustomResource(this, 'ConfigureAppAccessLogBucket', {
            serviceToken: customResourceProvider.serviceToken,
            resourceType: 'Custom::ConfigureAppAccessLogBucket',
            properties: {
                Region: cdk.Aws.REGION,
                AppAccessLogBucket: accessLogBucket.bucketName,
                LogParser: logParserLambda.functionArn,
                ScannersProbesAthenaLogParser: 'yes',
                MoveS3LogsForPartition: moveLogToPartitionLambda.functionName,
            },
        });
        new cdk.CustomResource(this, 'ConfigureWafLogBucket', {
            serviceToken: customResourceProvider.serviceToken,
            resourceType: 'Custom::ConfigureWafLogBucket',
            properties: {
                Region: cdk.Aws.REGION,
                WafLogBucket: wafLogBucket.bucketName,
                LogParser: logParserLambda.functionArn,
                HttpFloodAthenaLogParser: 'yes',
            },
        });
        new cdk.CustomResource(this, 'ConfigureWebAcl', {
            serviceToken: customResourceProvider.serviceToken,
            resourceType: 'Custom::ConfigureWebAcl',
            properties: {
                WAFWhitelistSetIPV4: whitelistIpSetV4.attrId,
                WAFWhitelistSetIPV4Name: whitelistIpSetV4.name,
            },
        });
        new cdk.CustomResource(this, 'GenerateAppLogParserConfFile', {
            serviceToken: customResourceProvider.serviceToken,
            resourceType: 'Custom::GenerateAppLogParserConfFile',
            properties: {
                StackName: stack.stackName,
                ErrorThreshold: errorThreshold,
                WAFBlockPeriod: blockPeriod,
                AppAccessLogBucket: accessLogBucket.bucketName,
            },
        });
        new cdk.CustomResource(this, 'GenerateWafLogParserConfFile', {
            serviceToken: customResourceProvider.serviceToken,
            resourceType: 'Custom::GenerateWafLogParserConfFile',
            properties: {
                StackName: stack.stackName,
                RequestThreshold: requestThreshold,
                WAFBlockPeriod: blockPeriod,
                WafAccessLogBucket: wafLogBucket.bucketName,
            },
        });
        new cdk.CustomResource(this, 'AddAthenaPartitions', {
            serviceToken: customResourceProvider.serviceToken,
            resourceType: 'Custom::AddAthenaPartitions',
            properties: {
                ResourceType: 'Custom::AddAthenaPartitions',
                AddAthenaPartitionsLambda: addAthenaPartitionsLambda.functionName,
                GlueAccessLogsDatabase: glueAccessLogsDatabase.databaseName,
                AppAccessLogBucket: accessLogBucket.bucketName,
                GlueAppAccessLogsTable: 'app_access_logs',
                GlueWafAccessLogsTable: glueWafAccessLogsTable.tableName,
                WafLogBucket: wafLogBucket.bucketName,
                AthenaWorkGroup: addPartitionAthenaQueryWorkGroup.name,
            },
        });
        new cdk.CfnOutput(this, 'AppAccessLogBucketName', {
            value: accessLogBucket.bucketName,
        });
        new cdk.CfnOutput(this, 'WafLogBucketName', {
            value: wafLogBucket.bucketName,
        });
        new cdk.CfnOutput(this, 'WAFWebACLName', { value: wafweb.attrId });
        new cdk.CfnOutput(this, 'WAFWebACLArn', { value: wafweb.attrArn });
    }
    /**
     * @experimental
     */
    validateResourceNamingPrefix(resourceNamingPrefix) {
        return /^[a-zA-Z]+[a-zA-Z0-9_]+$/.test(resourceNamingPrefix);
    }
}
exports.AutomatedWaf = AutomatedWaf;
_a = JSII_RTTI_SYMBOL_1;
AutomatedWaf[_a] = { fqn: "@ikala-cloud/aws-waf-solution.AutomatedWaf", version: "2.0.15" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoibWFpbi5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uL3NyYy9tYWluLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7Ozs7O0FBQUEsNkJBQTZCO0FBQzdCLGdEQUFnRDtBQUNoRCxtQ0FBbUM7QUFDbkMseURBQXlEO0FBQ3pELGlEQUFpRDtBQUNqRCx5REFBeUQ7QUFDekQsaURBQWlEO0FBQ2pELDBEQUEwRDtBQUMxRCxnREFBZ0Q7QUFDaEQsMkNBQTJDO0FBQzNDLDREQUE0RDtBQUM1RCxpREFBaUQ7QUFDakQseUNBQXlDO0FBQ3pDLHdEQUF3RDtBQUN4RCwrQ0FBK0M7QUFDL0MsbURBQW1EO0FBQ25ELDJDQUF1Qzs7OztBQUV2QyxJQUFZLGVBR1g7QUFIRCxXQUFZLGVBQWU7SUFDekIsNENBQXlCLENBQUE7SUFDekIsd0NBQXFCLENBQUE7QUFDdkIsQ0FBQyxFQUhXLGVBQWUsR0FBZix1QkFBZSxLQUFmLHVCQUFlLFFBRzFCOzs7O0FBRUQsSUFBWSxRQU1YO0FBTkQsV0FBWSxRQUFRO0lBQ2xCLDJCQUFlLENBQUE7SUFDZix5QkFBYSxDQUFBO0lBQ2IsK0JBQW1CLENBQUE7SUFDbkIsMkJBQWUsQ0FBQTtJQUNmLGlDQUFxQixDQUFBO0FBQ3ZCLENBQUMsRUFOVyxRQUFRLEdBQVIsZ0JBQVEsS0FBUixnQkFBUSxRQU1uQjtBQUVELE1BQU0saUNBQWlDLEdBQThCO0lBQ25FLFdBQVcsRUFBRSxjQUFjO0lBQzNCLFdBQVcsRUFBRSxjQUFjO0lBQzNCLFdBQVcsRUFBRSxjQUFjO0lBQzNCLFdBQVcsRUFBRSxjQUFjO0lBQzNCLFlBQVksRUFBRSxjQUFjO0lBQzVCLGNBQWMsRUFBRSxjQUFjO0lBQzlCLGNBQWMsRUFBRSxjQUFjO0lBQzlCLFdBQVcsRUFBRSxjQUFjO0lBQzNCLFdBQVcsRUFBRSxjQUFjO0lBQzNCLFlBQVksRUFBRSxjQUFjO0lBQzVCLFdBQVcsRUFBRSxjQUFjO0lBQzNCLFlBQVksRUFBRSxjQUFjO0lBQzVCLFdBQVcsRUFBRSxjQUFjO0lBQzNCLGdCQUFnQixFQUFFLGNBQWM7SUFDaEMsZ0JBQWdCLEVBQUUsY0FBYztJQUNoQyxnQkFBZ0IsRUFBRSxjQUFjO0lBQ2hDLGdCQUFnQixFQUFFLGNBQWM7SUFDaEMsZ0JBQWdCLEVBQUUsY0FBYztJQUNoQyxZQUFZLEVBQUUsY0FBYztJQUM1QixZQUFZLEVBQUUsY0FBYztJQUM1QixXQUFXLEVBQUUsY0FBYztJQUMzQixlQUFlLEVBQUUsY0FBYztJQUMvQixlQUFlLEVBQUUsY0FBYztJQUMvQixZQUFZLEVBQUUsY0FBYztJQUM1QixnQkFBZ0IsRUFBRSxjQUFjO0NBQ2pDLENBQUM7Ozs7QUFtQ0YsTUFBYSxZQUFhLFNBQVEsc0JBQVM7Ozs7SUFDekMsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUF3Qjs7UUFDaEUsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztRQUVqQixNQUFNLEtBQUssR0FBRyxHQUFHLENBQUMsS0FBSyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUNqQyxJQUNFLEtBQUssQ0FBQyxvQkFBb0I7WUFDMUIsQ0FBQyxJQUFJLENBQUMsNEJBQTRCLENBQUMsS0FBSyxDQUFDLG9CQUFvQixDQUFDLEVBQzlEO1lBQ0EsTUFBTSxJQUFJLEtBQUssQ0FDYiw2R0FBNkcsQ0FDOUcsQ0FBQztTQUNIO1FBRUQsZUFBZTtRQUNmLE1BQU0sY0FBYyxTQUFHLEtBQUssQ0FBQyxjQUFjLG1DQUFJLEdBQUcsQ0FBQztRQUNuRCxNQUFNLFNBQVMsU0FBRyxLQUFLLENBQUMsU0FBUyxtQ0FBSSxLQUFLLENBQUM7UUFDM0MsTUFBTSxnQkFBZ0IsU0FBRyxLQUFLLENBQUMsZ0JBQWdCLG1DQUFJLElBQUksQ0FBQztRQUN4RCxNQUFNLFdBQVcsU0FBRyxLQUFLLENBQUMsV0FBVyxtQ0FBSSxHQUFHLENBQUM7UUFDN0MsTUFBTSwwQkFBMEIsU0FDOUIsS0FBSyxDQUFDLDBCQUEwQixtQ0FBSSxLQUFLLENBQUM7UUFFNUMsTUFBTSxRQUFRLFNBQUcsS0FBSyxDQUFDLFFBQVEsbUNBQUksTUFBTSxDQUFDO1FBRTFDLE1BQU0sdUJBQXVCLEdBQUcseUJBQXlCLEtBQUssQ0FBQyxvQkFBb0IsR0FBRyxHQUFHLENBQUMsR0FBRyxDQUFDLE1BQU0sRUFBRSxDQUFDO1FBQ3ZHLE1BQU0sa0JBQWtCLEdBQ3RCLEdBQUcsQ0FBQyxFQUFFLENBQUMsR0FBRyxDQUFDLGdCQUFnQixDQUFDLEdBQUcsdUJBQXVCLENBQUM7UUFDekQsTUFBTSxhQUFhLEdBQUcsR0FBRyxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsZ0JBQWdCLENBQUMsR0FBRyxlQUFlLENBQUM7UUFDckUsTUFBTSxpQkFBaUIsR0FBRyxHQUFHLENBQUMsRUFBRSxDQUFDLEdBQUcsQ0FBQyxnQkFBZ0IsQ0FBQyxHQUFHLGVBQWUsQ0FBQztRQUN6RSxNQUFNLHdCQUF3QixHQUM1QixHQUFHLENBQUMsRUFBRSxDQUFDLEdBQUcsQ0FBQyxnQkFBZ0IsQ0FBQyxHQUFHLHNCQUFzQixDQUFDO1FBQ3hELE1BQU0sc0JBQXNCLEdBQzFCLEdBQUcsQ0FBQyxFQUFFLENBQUMsR0FBRyxDQUFDLGdCQUFnQixDQUFDLEdBQUcsb0JBQW9CLENBQUM7UUFDdEQsTUFBTSxjQUFjLEdBQUcsR0FBRyxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsZ0JBQWdCLENBQUMsR0FBRyxZQUFZLENBQUM7UUFDbkUsTUFBTSxvQkFBb0IsR0FDeEIsR0FBRyxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsZ0JBQWdCLENBQUMsR0FBRyxrQkFBa0IsQ0FBQztRQUNwRCxNQUFNLFdBQVcsR0FBRyxHQUFHLENBQUMsRUFBRSxDQUFDLEdBQUcsQ0FBQyxnQkFBZ0IsQ0FBQyxHQUFHLFNBQVMsQ0FBQztRQUU3RCxNQUFNLHVCQUF1QixHQUFHLEtBQUssQ0FBQyxnQkFBZ0I7WUFDcEQsQ0FBQyxDQUFDLEVBQUUsVUFBVSxFQUFFLEtBQUssQ0FBQyxnQkFBZ0IsRUFBRTtZQUN4QyxDQUFDLENBQUMsRUFBRSxDQUFDO1FBRVAsY0FBYztRQUNkLE1BQU0sWUFBWSxHQUFHLElBQUksRUFBRSxDQUFDLE1BQU0sQ0FBQyxJQUFJLEVBQUUsY0FBYyxFQUFFO1lBQ3ZELEdBQUcsdUJBQXVCO1lBQzFCLGdCQUFnQixFQUFFLEtBQUs7WUFDdkIsVUFBVSxFQUFFLEVBQUUsQ0FBQyxnQkFBZ0IsQ0FBQyxXQUFXO1lBQzNDLGFBQWEsRUFBRSxHQUFHLENBQUMsYUFBYSxDQUFDLE9BQU87WUFDeEMsaUJBQWlCLEVBQUUsSUFBSTtTQUN4QixDQUFDLENBQUM7UUFFSCxNQUFNLDBCQUEwQixHQUFHLEtBQUssQ0FBQyxzQkFBc0I7WUFDN0QsQ0FBQyxDQUFDLEVBQUUsVUFBVSxFQUFFLEtBQUssQ0FBQyxzQkFBc0IsRUFBRTtZQUM5QyxDQUFDLENBQUMsRUFBRSxDQUFDO1FBRVAsTUFBTSxlQUFlLEdBQUcsSUFBSSxFQUFFLENBQUMsTUFBTSxDQUFDLElBQUksRUFBRSxpQkFBaUIsRUFBRTtZQUM3RCxHQUFHLDBCQUEwQjtZQUM3QixnQkFBZ0IsRUFBRSxLQUFLO1lBQ3ZCLFVBQVUsRUFBRSxFQUFFLENBQUMsZ0JBQWdCLENBQUMsVUFBVTtZQUMxQyxhQUFhLEVBQ1gsS0FBSyxDQUFDLFNBQVMsSUFBSSxlQUFlLENBQUMsVUFBVSxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsbUJBQW1CLENBQUMsa0JBQWtCLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxtQkFBbUIsQ0FBQyxPQUFPO1lBQzVILGFBQWEsRUFBRSxHQUFHLENBQUMsYUFBYSxDQUFDLE9BQU87WUFDeEMsaUJBQWlCLEVBQUUsSUFBSTtTQUN4QixDQUFDLENBQUM7UUFFSCxJQUFJLEtBQUssQ0FBQyxTQUFTLElBQUksZUFBZSxDQUFDLFFBQVEsRUFBRTtZQUMvQyxlQUFlLENBQUMsbUJBQW1CLENBQ2pDLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQztnQkFDdEIsTUFBTSxFQUFFLEdBQUcsQ0FBQyxNQUFNLENBQUMsS0FBSztnQkFDeEIsT0FBTyxFQUFFLENBQUMsY0FBYyxDQUFDO2dCQUN6QixTQUFTLEVBQUU7b0JBQ1QsR0FBRyxlQUFlLENBQUMsU0FBUyxZQUFZLEdBQUcsQ0FBQyxHQUFHLENBQUMsVUFBVSxJQUFJO2lCQUMvRDtnQkFDRCxVQUFVLEVBQUU7b0JBQ1YsSUFBSSxHQUFHLENBQUMsZ0JBQWdCLENBQ3RCLGlDQUFpQyxDQUFDLEtBQUssQ0FBQyxNQUFNLENBQUMsQ0FDaEQ7aUJBQ0Y7YUFDRixDQUFDLENBQ0gsQ0FBQztZQUNGLGVBQWUsQ0FBQyxtQkFBbUIsQ0FDakMsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDO2dCQUN0QixNQUFNLEVBQUUsR0FBRyxDQUFDLE1BQU0sQ0FBQyxLQUFLO2dCQUN4QixPQUFPLEVBQUUsQ0FBQyxjQUFjLENBQUM7Z0JBQ3pCLFNBQVMsRUFBRTtvQkFDVCxHQUFHLGVBQWUsQ0FBQyxTQUFTLFlBQVksR0FBRyxDQUFDLEdBQUcsQ0FBQyxVQUFVLElBQUk7aUJBQy9EO2dCQUNELFVBQVUsRUFBRSxDQUFDLElBQUksR0FBRyxDQUFDLGdCQUFnQixDQUFDLDZCQUE2QixDQUFDLENBQUM7Z0JBQ3JFLFVBQVUsRUFBRTtvQkFDVixZQUFZLEVBQUU7d0JBQ1osY0FBYyxFQUFFLDJCQUEyQjtxQkFDNUM7aUJBQ0Y7YUFDRixDQUFDLENBQ0gsQ0FBQztZQUNGLGVBQWUsQ0FBQyxtQkFBbUIsQ0FDakMsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDO2dCQUN0QixNQUFNLEVBQUUsR0FBRyxDQUFDLE1BQU0sQ0FBQyxLQUFLO2dCQUN4QixPQUFPLEVBQUUsQ0FBQyxpQkFBaUIsQ0FBQztnQkFDNUIsU0FBUyxFQUFFLENBQUMsZUFBZSxDQUFDLFNBQVMsQ0FBQztnQkFDdEMsVUFBVSxFQUFFLENBQUMsSUFBSSxHQUFHLENBQUMsZ0JBQWdCLENBQUMsNkJBQTZCLENBQUMsQ0FBQzthQUN0RSxDQUFDLENBQ0gsQ0FBQztTQUNIO1FBRUQseUJBQXlCO1FBQ3pCLE1BQU0sZ0JBQWdCLEdBQUcsSUFBSSxLQUFLLENBQUMsUUFBUSxDQUFDLElBQUksRUFBRSxrQkFBa0IsRUFBRTtZQUNwRSxTQUFTLEVBQUUsRUFBRTtZQUNiLGdCQUFnQixFQUFFLE1BQU07WUFDeEIsS0FBSyxFQUFFLEtBQUssQ0FBQyxTQUFTO1lBQ3RCLElBQUksRUFBRSxHQUFHLENBQUMsRUFBRSxDQUFDLEdBQUcsQ0FBQyxnQkFBZ0IsQ0FBQyxHQUFHLGtCQUFrQjtZQUN2RCxXQUFXLEVBQUUsb0NBQW9DO1NBQ2xELENBQUMsQ0FBQztRQUVILE1BQU0sZ0JBQWdCLEdBQUcsSUFBSSxLQUFLLENBQUMsUUFBUSxDQUFDLElBQUksRUFBRSxrQkFBa0IsRUFBRTtZQUNwRSxTQUFTLEVBQUUsRUFBRTtZQUNiLGdCQUFnQixFQUFFLE1BQU07WUFDeEIsS0FBSyxFQUFFLEtBQUssQ0FBQyxTQUFTO1lBQ3RCLElBQUksRUFBRSxHQUFHLENBQUMsRUFBRSxDQUFDLEdBQUcsQ0FBQyxnQkFBZ0IsQ0FBQyxHQUFHLGtCQUFrQjtZQUN2RCxXQUFXLEVBQUUsb0NBQW9DO1NBQ2xELENBQUMsQ0FBQztRQUVILHlCQUF5QjtRQUN6QixNQUFNLGdCQUFnQixHQUFHLElBQUksS0FBSyxDQUFDLFFBQVEsQ0FBQyxJQUFJLEVBQUUsa0JBQWtCLEVBQUU7WUFDcEUsU0FBUyxFQUFFLEVBQUU7WUFDYixnQkFBZ0IsRUFBRSxNQUFNO1lBQ3hCLEtBQUssRUFBRSxLQUFLLENBQUMsU0FBUztZQUN0QixJQUFJLEVBQUUsR0FBRyxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsZ0JBQWdCLENBQUMsR0FBRyxrQkFBa0I7WUFDdkQsV0FBVyxFQUFFLG9DQUFvQztTQUNsRCxDQUFDLENBQUM7UUFFSCxNQUFNLGdCQUFnQixHQUFHLElBQUksS0FBSyxDQUFDLFFBQVEsQ0FBQyxJQUFJLEVBQUUsa0JBQWtCLEVBQUU7WUFDcEUsU0FBUyxFQUFFLEVBQUU7WUFDYixnQkFBZ0IsRUFBRSxNQUFNO1lBQ3hCLEtBQUssRUFBRSxLQUFLLENBQUMsU0FBUztZQUN0QixJQUFJLEVBQUUsR0FBRyxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsZ0JBQWdCLENBQUMsR0FBRyxrQkFBa0I7WUFDdkQsV0FBVyxFQUFFLG9DQUFvQztTQUNsRCxDQUFDLENBQUM7UUFFSCwwQkFBMEI7UUFDMUIsTUFBTSxnQkFBZ0IsR0FBRyxJQUFJLEtBQUssQ0FBQyxRQUFRLENBQUMsSUFBSSxFQUFFLGtCQUFrQixFQUFFO1lBQ3BFLFNBQVMsRUFBRSxFQUFFO1lBQ2IsZ0JBQWdCLEVBQUUsTUFBTTtZQUN4QixLQUFLLEVBQUUsS0FBSyxDQUFDLFNBQVM7WUFDdEIsSUFBSSxFQUFFLEdBQUcsQ0FBQyxFQUFFLENBQUMsR0FBRyxDQUFDLGdCQUFnQixDQUFDLEdBQUcsa0JBQWtCO1lBQ3ZELFdBQVcsRUFBRSxpQ0FBaUM7U0FDL0MsQ0FBQyxDQUFDO1FBRUgsTUFBTSxnQkFBZ0IsR0FBRyxJQUFJLEtBQUssQ0FBQyxRQUFRLENBQUMsSUFBSSxFQUFFLGtCQUFrQixFQUFFO1lBQ3BFLFNBQVMsRUFBRSxFQUFFO1lBQ2IsZ0JBQWdCLEVBQUUsTUFBTTtZQUN4QixLQUFLLEVBQUUsS0FBSyxDQUFDLFNBQVM7WUFDdEIsSUFBSSxFQUFFLEdBQUcsQ0FBQyxFQUFFLENBQUMsR0FBRyxDQUFDLGdCQUFnQixDQUFDLEdBQUcsa0JBQWtCO1lBQ3ZELFdBQVcsRUFBRSxpQ0FBaUM7U0FDL0MsQ0FBQyxDQUFDO1FBRUgsK0JBQStCO1FBQy9CLE1BQU0scUJBQXFCLEdBQUcsSUFBSSxLQUFLLENBQUMsUUFBUSxDQUM5QyxJQUFJLEVBQ0osdUJBQXVCLEVBQ3ZCO1lBQ0UsU0FBUyxFQUFFLEVBQUU7WUFDYixnQkFBZ0IsRUFBRSxNQUFNO1lBQ3hCLEtBQUssRUFBRSxLQUFLLENBQUMsU0FBUztZQUN0QixJQUFJLEVBQUUsR0FBRyxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsZ0JBQWdCLENBQUMsR0FBRyx1QkFBdUI7WUFDNUQsV0FBVyxFQUFFLHNDQUFzQztTQUNwRCxDQUNGLENBQUM7UUFFRixNQUFNLHFCQUFxQixHQUFHLElBQUksS0FBSyxDQUFDLFFBQVEsQ0FDOUMsSUFBSSxFQUNKLHVCQUF1QixFQUN2QjtZQUNFLFNBQVMsRUFBRSxFQUFFO1lBQ2IsZ0JBQWdCLEVBQUUsTUFBTTtZQUN4QixLQUFLLEVBQUUsS0FBSyxDQUFDLFNBQVM7WUFDdEIsSUFBSSxFQUFFLEdBQUcsQ0FBQyxFQUFFLENBQUMsR0FBRyxDQUFDLGdCQUFnQixDQUFDLEdBQUcsdUJBQXVCO1lBQzVELFdBQVcsRUFBRSxzQ0FBc0M7U0FDcEQsQ0FDRixDQUFDO1FBRUYsK0JBQStCO1FBQy9CLE1BQU0sc0JBQXNCLEdBQUcsSUFBSSxLQUFLLENBQUMsUUFBUSxDQUMvQyxJQUFJLEVBQ0osMEJBQTBCLEVBQzFCO1lBQ0UsU0FBUyxFQUFFLEVBQUU7WUFDYixnQkFBZ0IsRUFBRSxNQUFNO1lBQ3hCLEtBQUssRUFBRSxLQUFLLENBQUMsU0FBUztZQUN0QixJQUFJLEVBQUUsR0FBRyxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsZ0JBQWdCLENBQUMsR0FBRywwQkFBMEI7WUFDL0QsV0FBVyxFQUFFLHNDQUFzQztTQUNwRCxDQUNGLENBQUM7UUFFRixNQUFNLHNCQUFzQixHQUFHLElBQUksS0FBSyxDQUFDLFFBQVEsQ0FDL0MsSUFBSSxFQUNKLDBCQUEwQixFQUMxQjtZQUNFLFNBQVMsRUFBRSxFQUFFO1lBQ2IsZ0JBQWdCLEVBQUUsTUFBTTtZQUN4QixLQUFLLEVBQUUsS0FBSyxDQUFDLFNBQVM7WUFDdEIsSUFBSSxFQUFFLEdBQUcsQ0FBQyxFQUFFLENBQUMsR0FBRyxDQUFDLGdCQUFnQixDQUFDLEdBQUcsMEJBQTBCO1lBQy9ELFdBQVcsRUFBRSxzQ0FBc0M7U0FDcEQsQ0FDRixDQUFDO1FBRUYsdUJBQXVCO1FBQ3ZCLE1BQU0sYUFBYSxHQUFHLElBQUksS0FBSyxDQUFDLFFBQVEsQ0FBQyxJQUFJLEVBQUUsaUJBQWlCLEVBQUU7WUFDaEUsU0FBUyxFQUFFLEVBQUU7WUFDYixnQkFBZ0IsRUFBRSxNQUFNO1lBQ3hCLEtBQUssRUFBRSxLQUFLLENBQUMsU0FBUztZQUN0QixJQUFJLEVBQUUsR0FBRyxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsZ0JBQWdCLENBQUMsR0FBRyxpQkFBaUI7WUFDdEQsV0FBVyxFQUFFLDhCQUE4QjtTQUM1QyxDQUFDLENBQUM7UUFFSCxNQUFNLGFBQWEsR0FBRyxJQUFJLEtBQUssQ0FBQyxRQUFRLENBQUMsSUFBSSxFQUFFLGlCQUFpQixFQUFFO1lBQ2hFLFNBQVMsRUFBRSxFQUFFO1lBQ2IsZ0JBQWdCLEVBQUUsTUFBTTtZQUN4QixLQUFLLEVBQUUsS0FBSyxDQUFDLFNBQVM7WUFDdEIsSUFBSSxFQUFFLEdBQUcsQ0FBQyxFQUFFLENBQUMsR0FBRyxDQUFDLGdCQUFnQixDQUFDLEdBQUcsaUJBQWlCO1lBQ3RELFdBQVcsRUFBRSw4QkFBOEI7U0FDNUMsQ0FBQyxDQUFDO1FBRUgsTUFBTSxjQUFjLEdBQUcsRUFBRSxDQUFDO1FBQzFCLElBQUksS0FBSyxDQUFDLFNBQVMsSUFBSSxlQUFlLENBQUMsVUFBVSxFQUFFO1lBQ2pELGNBQWMsQ0FBQyxJQUFJLENBQUM7Z0JBQ2xCLElBQUksRUFBRSxjQUFjO2dCQUNwQixRQUFRLEVBQUUsQ0FBQztnQkFDWCxNQUFNLEVBQUUsQ0FBQyxTQUFTO29CQUNoQixDQUFDLENBQUM7d0JBQ0EsS0FBSyxFQUFFLEVBQUU7cUJBQ1Y7b0JBQ0QsQ0FBQyxDQUFDLEVBQUUsS0FBSyxFQUFFLEVBQUUsRUFBRTtnQkFDakIsZ0JBQWdCLEVBQUU7b0JBQ2hCLHNCQUFzQixFQUFFLElBQUk7b0JBQzVCLHdCQUF3QixFQUFFLElBQUk7b0JBQzlCLFVBQVUsRUFBRSxjQUFjO2lCQUMzQjtnQkFDRCxTQUFTLEVBQUU7b0JBQ1QsV0FBVyxFQUFFO3dCQUNYLFVBQVUsRUFBRTs0QkFDVjtnQ0FDRSx1QkFBdUIsRUFBRTtvQ0FDdkIsR0FBRyxFQUFFLEdBQUcsQ0FBQyxFQUFFLENBQUMsTUFBTSxDQUFDLGFBQWEsQ0FBQyxTQUFTLEVBQUUsS0FBSyxDQUFDLENBQUMsUUFBUSxFQUFFO2lDQUM5RDs2QkFDRjs0QkFDRDtnQ0FDRSx1QkFBdUIsRUFBRTtvQ0FDdkIsR0FBRyxFQUFFLEdBQUcsQ0FBQyxFQUFFLENBQUMsTUFBTSxDQUFDLGFBQWEsQ0FBQyxTQUFTLEVBQUUsS0FBSyxDQUFDLENBQUMsUUFBUSxFQUFFO2lDQUM5RDs2QkFDRjt5QkFDRjtxQkFDRjtpQkFDRjthQUNGLENBQUMsQ0FBQztTQUNKO1FBRUQsY0FBYztRQUNkLE1BQU0sTUFBTSxHQUFHLElBQUksS0FBSyxDQUFDLFNBQVMsQ0FBQyxJQUFJLEVBQUUsUUFBUSxFQUFFO1lBQ2pELElBQUksRUFBRSxLQUFLLENBQUMsb0JBQW9CO2dCQUM5QixDQUFDLENBQUMsR0FBRyxLQUFLLENBQUMsb0JBQW9CLE1BQU07Z0JBQ3JDLENBQUMsQ0FBQyxvQkFBb0I7WUFDeEIsV0FBVyxFQUFFLGtCQUFrQjtZQUMvQixhQUFhLEVBQUU7Z0JBQ2IsS0FBSyxFQUFFLEVBQUU7YUFDVjtZQUNELEtBQUssRUFBRSxLQUFLLENBQUMsU0FBUztZQUN0QixnQkFBZ0IsRUFBRTtnQkFDaEIsd0JBQXdCLEVBQUUsSUFBSTtnQkFDOUIsc0JBQXNCLEVBQUUsSUFBSTtnQkFDNUIsVUFBVSxFQUFFLEtBQUssQ0FBQyxvQkFBb0I7b0JBQ3BDLENBQUMsQ0FBQyxHQUFHLEtBQUssQ0FBQyxvQkFBb0IsTUFBTTtvQkFDckMsQ0FBQyxDQUFDLG9CQUFvQjthQUN6QjtZQUNELEtBQUssRUFBRTtnQkFDTDtvQkFDRSxJQUFJLEVBQUUsa0NBQWtDO29CQUN4QyxRQUFRLEVBQUUsQ0FBQztvQkFDWCxjQUFjLEVBQUUsQ0FBQyxTQUFTO3dCQUN4QixDQUFDLENBQUM7NEJBQ0EsSUFBSSxFQUFFLEVBQUU7eUJBQ1Q7d0JBQ0QsQ0FBQyxDQUFDLEVBQUUsS0FBSyxFQUFFLEVBQUUsRUFBRTtvQkFDakIsZ0JBQWdCLEVBQUU7d0JBQ2hCLHdCQUF3QixFQUFFLElBQUk7d0JBQzlCLHNCQUFzQixFQUFFLElBQUk7d0JBQzVCLFVBQVUsRUFBRSxLQUFLLENBQUMsb0JBQW9COzRCQUNwQyxDQUFDLENBQUMsR0FBRyxLQUFLLENBQUMsb0JBQW9CLG9CQUFvQjs0QkFDbkQsQ0FBQyxDQUFDLDhCQUE4QjtxQkFDbkM7b0JBQ0QsU0FBUyxFQUFFO3dCQUNULHlCQUF5QixFQUFFOzRCQUN6QixVQUFVLEVBQUUsS0FBSzs0QkFDakIsSUFBSSxFQUFFLDhCQUE4Qjt5QkFDckM7cUJBQ0Y7aUJBQ0Y7Z0JBQ0Q7b0JBQ0UsSUFBSSxFQUFFLGFBQWE7b0JBQ25CLFFBQVEsRUFBRSxDQUFDO29CQUNYLE1BQU0sRUFBRSxDQUFDLFNBQVM7d0JBQ2hCLENBQUMsQ0FBQzs0QkFDQSxLQUFLLEVBQUUsRUFBRTt5QkFDVjt3QkFDRCxDQUFDLENBQUMsRUFBRSxLQUFLLEVBQUUsRUFBRSxFQUFFO29CQUNqQixnQkFBZ0IsRUFBRTt3QkFDaEIsc0JBQXNCLEVBQUUsSUFBSTt3QkFDNUIsd0JBQXdCLEVBQUUsSUFBSTt3QkFDOUIsVUFBVSxFQUFFLGFBQWE7cUJBQzFCO29CQUNELFNBQVMsRUFBRTt3QkFDVCxXQUFXLEVBQUU7NEJBQ1gsVUFBVSxFQUFFO2dDQUNWO29DQUNFLHVCQUF1QixFQUFFO3dDQUN2QixHQUFHLEVBQUUsR0FBRyxDQUFDLEVBQUUsQ0FBQyxNQUFNLENBQ2hCLGdCQUFnQixDQUFDLFNBQVMsRUFDMUIsS0FBSyxDQUNOLENBQUMsUUFBUSxFQUFFO3FDQUNiO2lDQUNGO2dDQUNEO29DQUNFLHVCQUF1QixFQUFFO3dDQUN2QixHQUFHLEVBQUUsR0FBRyxDQUFDLEVBQUUsQ0FBQyxNQUFNLENBQ2hCLGdCQUFnQixDQUFDLFNBQVMsRUFDMUIsS0FBSyxDQUNOLENBQUMsUUFBUSxFQUFFO3FDQUNiO2lDQUNGOzZCQUNGO3lCQUNGO3FCQUNGO2lCQUNGO2dCQUNEO29CQUNFLElBQUksRUFBRSxpQkFBaUI7b0JBQ3ZCLFFBQVEsRUFBRSxDQUFDO29CQUNYLE1BQU0sRUFBRSxDQUFDLFNBQVM7d0JBQ2hCLENBQUMsQ0FBQzs0QkFDQSxLQUFLLEVBQUUsRUFBRTt5QkFDVjt3QkFDRCxDQUFDLENBQUMsRUFBRSxLQUFLLEVBQUUsRUFBRSxFQUFFO29CQUVqQixnQkFBZ0IsRUFBRTt3QkFDaEIsc0JBQXNCLEVBQUUsSUFBSTt3QkFDNUIsd0JBQXdCLEVBQUUsSUFBSTt3QkFDOUIsVUFBVSxFQUFFLGlCQUFpQjtxQkFDOUI7b0JBQ0QsU0FBUyxFQUFFO3dCQUNULFdBQVcsRUFBRTs0QkFDWCxVQUFVLEVBQUU7Z0NBQ1Y7b0NBQ0UsdUJBQXVCLEVBQUU7d0NBQ3ZCLEdBQUcsRUFBRSxHQUFHLENBQUMsRUFBRSxDQUFDLE1BQU0sQ0FDaEIsZ0JBQWdCLENBQUMsU0FBUyxFQUMxQixLQUFLLENBQ04sQ0FBQyxRQUFRLEVBQUU7cUNBQ2I7aUNBQ0Y7Z0NBQ0Q7b0NBQ0UsdUJBQXVCLEVBQUU7d0NBQ3ZCLEdBQUcsRUFBRSxHQUFHLENBQUMsRUFBRSxDQUFDLE1BQU0sQ0FDaEIsZ0JBQWdCLENBQUMsU0FBUyxFQUMxQixLQUFLLENBQ04sQ0FBQyxRQUFRLEVBQUU7cUNBQ2I7aUNBQ0Y7NkJBQ0Y7eUJBQ0Y7cUJBQ0Y7aUJBQ0Y7Z0JBQ0Q7b0JBQ0UsSUFBSSxFQUFFLHdCQUF3QjtvQkFDOUIsUUFBUSxFQUFFLENBQUM7b0JBQ1gsTUFBTSxFQUFFLENBQUMsU0FBUzt3QkFDaEIsQ0FBQyxDQUFDOzRCQUNBLEtBQUssRUFBRSxFQUFFO3lCQUNWO3dCQUNELENBQUMsQ0FBQyxFQUFFLEtBQUssRUFBRSxFQUFFLEVBQUU7b0JBQ2pCLGdCQUFnQixFQUFFO3dCQUNoQixzQkFBc0IsRUFBRSxJQUFJO3dCQUM1Qix3QkFBd0IsRUFBRSxJQUFJO3dCQUM5QixVQUFVLEVBQUUsd0JBQXdCO3FCQUNyQztvQkFDRCxTQUFTLEVBQUU7d0JBQ1QsV0FBVyxFQUFFOzRCQUNYLFVBQVUsRUFBRTtnQ0FDVjtvQ0FDRSx1QkFBdUIsRUFBRTt3Q0FDdkIsR0FBRyxFQUFFLEdBQUcsQ0FBQyxFQUFFLENBQUMsTUFBTSxDQUNoQixnQkFBZ0IsQ0FBQyxTQUFTLEVBQzFCLEtBQUssQ0FDTixDQUFDLFFBQVEsRUFBRTtxQ0FDYjtpQ0FDRjtnQ0FDRDtvQ0FDRSx1QkFBdUIsRUFBRTt3Q0FDdkIsR0FBRyxFQUFFLEdBQUcsQ0FBQyxFQUFFLENBQUMsTUFBTSxDQUNoQixnQkFBZ0IsQ0FBQyxTQUFTLEVBQzFCLEtBQUssQ0FDTixDQUFDLFFBQVEsRUFBRTtxQ0FDYjtpQ0FDRjs2QkFDRjt5QkFDRjtxQkFDRjtpQkFDRjtnQkFDRDtvQkFDRSxJQUFJLEVBQUUsc0JBQXNCO29CQUM1QixRQUFRLEVBQUUsQ0FBQztvQkFDWCxNQUFNLEVBQUUsQ0FBQyxTQUFTO3dCQUNoQixDQUFDLENBQUM7NEJBQ0EsS0FBSyxFQUFFLEVBQUU7eUJBQ1Y7d0JBQ0QsQ0FBQyxDQUFDLEVBQUUsS0FBSyxFQUFFLEVBQUUsRUFBRTtvQkFFakIsZ0JBQWdCLEVBQUU7d0JBQ2hCLHNCQUFzQixFQUFFLElBQUk7d0JBQzVCLHdCQUF3QixFQUFFLElBQUk7d0JBQzlCLFVBQVUsRUFBRSxzQkFBc0I7cUJBQ25DO29CQUNELFNBQVMsRUFBRTt3QkFDVCxXQUFXLEVBQUU7NEJBQ1gsVUFBVSxFQUFFO2dDQUNWO29DQUNFLHVCQUF1QixFQUFFO3dDQUN2QixHQUFHLEVBQUUsR0FBRyxDQUFDLEVBQUUsQ0FBQyxNQUFNLENBQ2hCLHFCQUFxQixDQUFDLFNBQVMsRUFDL0IsS0FBSyxDQUNOLENBQUMsUUFBUSxFQUFFO3FDQUNiO2lDQUNGO2dDQUNEO29DQUNFLHVCQUF1QixFQUFFO3dDQUN2QixHQUFHLEVBQUUsR0FBRyxDQUFDLEVBQUUsQ0FBQyxNQUFNLENBQ2hCLHFCQUFxQixDQUFDLFNBQVMsRUFDL0IsS0FBSyxDQUNOLENBQUMsUUFBUSxFQUFFO3FDQUNiO2lDQUNGOzZCQUNGO3lCQUNGO3FCQUNGO2lCQUNGO2dCQUNEO29CQUNFLElBQUksRUFBRSxrQkFBa0I7b0JBQ3hCLFFBQVEsRUFBRSxDQUFDO29CQUNYLE1BQU0sRUFBRSxDQUFDLFNBQVM7d0JBQ2hCLENBQUMsQ0FBQzs0QkFDQSxLQUFLLEVBQUUsRUFBRTt5QkFDVjt3QkFDRCxDQUFDLENBQUMsRUFBRSxLQUFLLEVBQUUsRUFBRSxFQUFFO29CQUNqQixnQkFBZ0IsRUFBRTt3QkFDaEIsc0JBQXNCLEVBQUUsSUFBSTt3QkFDNUIsd0JBQXdCLEVBQUUsSUFBSTt3QkFDOUIsVUFBVSxFQUFFLGtCQUFrQjtxQkFDL0I7b0JBQ0QsU0FBUyxFQUFFO3dCQUNULFdBQVcsRUFBRTs0QkFDWCxVQUFVLEVBQUU7Z0NBQ1Y7b0NBQ0UsdUJBQXVCLEVBQUU7d0NBQ3ZCLEdBQUcsRUFBRSxHQUFHLENBQUMsRUFBRSxDQUFDLE1BQU0sQ0FDaEIsc0JBQXNCLENBQUMsU0FBUyxFQUNoQyxLQUFLLENBQ04sQ0FBQyxRQUFRLEVBQUU7cUNBQ2I7aUNBQ0Y7Z0NBQ0Q7b0NBQ0UsdUJBQXVCLEVBQUU7d0NBQ3ZCLEdBQUcsRUFBRSxHQUFHLENBQUMsRUFBRSxDQUFDLE1BQU0sQ0FDaEIsc0JBQXNCLENBQUMsU0FBUyxFQUNoQyxLQUFLLENBQ04sQ0FBQyxRQUFRLEVBQUU7cUNBQ2I7aUNBQ0Y7NkJBQ0Y7eUJBQ0Y7cUJBQ0Y7aUJBQ0Y7Z0JBQ0Q7b0JBQ0UsSUFBSSxFQUFFLG9CQUFvQjtvQkFDMUIsUUFBUSxFQUFFLEVBQUU7b0JBQ1osTUFBTSxFQUFFLENBQUMsU0FBUzt3QkFDaEIsQ0FBQyxDQUFDOzRCQUNBLEtBQUssRUFBRSxFQUFFO3lCQUNWO3dCQUNELENBQUMsQ0FBQyxFQUFFLEtBQUssRUFBRSxFQUFFLEVBQUU7b0JBQ2pCLGdCQUFnQixFQUFFO3dCQUNoQixzQkFBc0IsRUFBRSxJQUFJO3dCQUM1Qix3QkFBd0IsRUFBRSxJQUFJO3dCQUM5QixVQUFVLEVBQUUsb0JBQW9CO3FCQUNqQztvQkFDRCxTQUFTLEVBQUU7d0JBQ1QsV0FBVyxFQUFFOzRCQUNYLFVBQVUsRUFBRTtnQ0FDVjtvQ0FDRSxrQkFBa0IsRUFBRTt3Q0FDbEIsWUFBWSxFQUFFOzRDQUNaLFdBQVcsRUFBRSxFQUFFO3lDQUNoQjt3Q0FDRCxtQkFBbUIsRUFBRTs0Q0FDbkI7Z0RBQ0UsUUFBUSxFQUFFLENBQUM7Z0RBQ1gsSUFBSSxFQUFFLFlBQVk7NkNBQ25COzRDQUNEO2dEQUNFLFFBQVEsRUFBRSxDQUFDO2dEQUNYLElBQUksRUFBRSxvQkFBb0I7NkNBQzNCO3lDQUNGO3FDQUNGO2lDQUNGO2dDQUNEO29DQUNFLGtCQUFrQixFQUFFO3dDQUNsQixZQUFZLEVBQUU7NENBQ1osSUFBSSxFQUFFLEVBQUU7eUNBQ1Q7d0NBQ0QsbUJBQW1CLEVBQUU7NENBQ25CO2dEQUNFLFFBQVEsRUFBRSxDQUFDO2dEQUNYLElBQUksRUFBRSxZQUFZOzZDQUNuQjs0Q0FDRDtnREFDRSxRQUFRLEVBQUUsQ0FBQztnREFDWCxJQUFJLEVBQUUsb0JBQW9COzZDQUMzQjt5Q0FDRjtxQ0FDRjtpQ0FDRjtnQ0FDRDtvQ0FDRSxrQkFBa0IsRUFBRTt3Q0FDbEIsWUFBWSxFQUFFOzRDQUNaLE9BQU8sRUFBRSxFQUFFO3lDQUNaO3dDQUNELG1CQUFtQixFQUFFOzRDQUNuQjtnREFDRSxRQUFRLEVBQUUsQ0FBQztnREFDWCxJQUFJLEVBQUUsWUFBWTs2Q0FDbkI7NENBQ0Q7Z0RBQ0UsUUFBUSxFQUFFLENBQUM7Z0RBQ1gsSUFBSSxFQUFFLG9CQUFvQjs2Q0FDM0I7eUNBQ0Y7cUNBQ0Y7aUNBQ0Y7Z0NBQ0Q7b0NBQ0Usa0JBQWtCLEVBQUU7d0NBQ2xCLFlBQVksRUFBRTs0Q0FDWixZQUFZLEVBQUUsRUFBRSxJQUFJLEVBQUUsZUFBZSxFQUFFO3lDQUN4Qzt3Q0FDRCxtQkFBbUIsRUFBRTs0Q0FDbkI7Z0RBQ0UsUUFBUSxFQUFFLENBQUM7Z0RBQ1gsSUFBSSxFQUFFLFlBQVk7NkNBQ25COzRDQUNEO2dEQUNFLFFBQVEsRUFBRSxDQUFDO2dEQUNYLElBQUksRUFBRSxvQkFBb0I7NkNBQzNCO3lDQUNGO3FDQUNGO2lDQUNGO2dDQUNEO29DQUNFLGtCQUFrQixFQUFFO3dDQUNsQixZQUFZLEVBQUU7NENBQ1osWUFBWSxFQUFFLEVBQUUsSUFBSSxFQUFFLFFBQVEsRUFBRTt5Q0FDakM7d0NBQ0QsbUJBQW1CLEVBQUU7NENBQ25CO2dEQUNFLFFBQVEsRUFBRSxDQUFDO2dEQUNYLElBQUksRUFBRSxZQUFZOzZDQUNuQjs0Q0FDRDtnREFDRSxRQUFRLEVBQUUsQ0FBQztnREFDWCxJQUFJLEVBQUUsb0JBQW9COzZDQUMzQjt5Q0FDRjtxQ0FDRjtpQ0FDRjs2QkFDRjt5QkFDRjtxQkFDRjtpQkFDRjtnQkFDRDtvQkFDRSxJQUFJLEVBQUUsV0FBVztvQkFDakIsUUFBUSxFQUFFLEVBQUU7b0JBQ1osTUFBTSxFQUFFLENBQUMsU0FBUzt3QkFDaEIsQ0FBQyxDQUFDOzRCQUNBLEtBQUssRUFBRSxFQUFFO3lCQUNWO3dCQUNELENBQUMsQ0FBQyxFQUFFLEtBQUssRUFBRSxFQUFFLEVBQUU7b0JBQ2pCLGdCQUFnQixFQUFFO3dCQUNoQixzQkFBc0IsRUFBRSxJQUFJO3dCQUM1Qix3QkFBd0IsRUFBRSxJQUFJO3dCQUM5QixVQUFVLEVBQUUsV0FBVztxQkFDeEI7b0JBQ0QsU0FBUyxFQUFFO3dCQUNULFdBQVcsRUFBRTs0QkFDWCxVQUFVLEVBQUU7Z0NBQ1Y7b0NBQ0UsaUJBQWlCLEVBQUU7d0NBQ2pCLFlBQVksRUFBRTs0Q0FDWixXQUFXLEVBQUUsRUFBRTt5Q0FDaEI7d0NBQ0QsbUJBQW1CLEVBQUU7NENBQ25CO2dEQUNFLFFBQVEsRUFBRSxDQUFDO2dEQUNYLElBQUksRUFBRSxZQUFZOzZDQUNuQjs0Q0FDRDtnREFDRSxRQUFRLEVBQUUsQ0FBQztnREFDWCxJQUFJLEVBQUUsb0JBQW9COzZDQUMzQjt5Q0FDRjtxQ0FDRjtpQ0FDRjtnQ0FDRDtvQ0FDRSxpQkFBaUIsRUFBRTt3Q0FDakIsWUFBWSxFQUFFOzRDQUNaLElBQUksRUFBRSxFQUFFO3lDQUNUO3dDQUNELG1CQUFtQixFQUFFOzRDQUNuQjtnREFDRSxRQUFRLEVBQUUsQ0FBQztnREFDWCxJQUFJLEVBQUUsWUFBWTs2Q0FDbkI7NENBQ0Q7Z0RBQ0UsUUFBUSxFQUFFLENBQUM7Z0RBQ1gsSUFBSSxFQUFFLG9CQUFvQjs2Q0FDM0I7eUNBQ0Y7cUNBQ0Y7aUNBQ0Y7Z0NBQ0Q7b0NBQ0UsaUJBQWlCLEVBQUU7d0NBQ2pCLFlBQVksRUFBRTs0Q0FDWixPQUFPLEVBQUUsRUFBRTt5Q0FDWjt3Q0FDRCxtQkFBbUIsRUFBRTs0Q0FDbkI7Z0RBQ0UsUUFBUSxFQUFFLENBQUM7Z0RBQ1gsSUFBSSxFQUFFLFlBQVk7NkNBQ25COzRDQUNEO2dEQUNFLFFBQVEsRUFBRSxDQUFDO2dEQUNYLElBQUksRUFBRSxvQkFBb0I7NkNBQzNCO3lDQUNGO3FDQUNGO2lDQUNGO2dDQUNEO29DQUNFLGlCQUFpQixFQUFFO3dDQUNqQixZQUFZLEVBQUU7NENBQ1osWUFBWSxFQUFFO2dEQUNaLElBQUksRUFBRSxRQUFROzZDQUNmO3lDQUNGO3dDQUNELG1CQUFtQixFQUFFOzRDQUNuQjtnREFDRSxRQUFRLEVBQUUsQ0FBQztnREFDWCxJQUFJLEVBQUUsWUFBWTs2Q0FDbkI7NENBQ0Q7Z0RBQ0UsUUFBUSxFQUFFLENBQUM7Z0RBQ1gsSUFBSSxFQUFFLG9CQUFvQjs2Q0FDM0I7eUNBQ0Y7cUNBQ0Y7aUNBQ0Y7NkJBQ0Y7eUJBQ0Y7cUJBQ0Y7aUJBQ0Y7Z0JBQ0QsR0FBRyxjQUFjO2FBQ2xCO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsSUFBSSxLQUFLLENBQUMscUJBQXFCLEVBQUU7WUFDL0IsSUFBSSxLQUFLLENBQUMsb0JBQW9CLENBQUMsSUFBSSxFQUFFLGdCQUFnQixFQUFFO2dCQUNyRCxTQUFTLEVBQUUsTUFBTSxDQUFDLE9BQU87Z0JBQ3pCLFdBQVcsRUFBRSxLQUFLLENBQUMscUJBQXFCO2FBQ3pDLENBQUMsQ0FBQztTQUNKO1FBRUQsUUFBUTtRQUNSLElBQUksTUFBTSxDQUFDLG1CQUFtQixDQUFDLElBQUksRUFBRSxRQUFRLEVBQUU7WUFDN0MsV0FBVyxFQUNULGdIQUFnSDtZQUNsSCxJQUFJLEVBQUUsTUFBTSxDQUFDLGVBQWUsQ0FBQyxjQUFjLENBQ3pDLElBQUksQ0FBQyxJQUFJLENBQUMsU0FBUyxFQUFFLGtCQUFrQixDQUFDLEVBQ3hDO2dCQUNFLEdBQUcsRUFBRSxDQUFDLHVCQUF1QixDQUFDO2FBQy9CLENBQ0Y7WUFDRCxVQUFVLEVBQUUsR0FBRztZQUNmLE9BQU8sRUFBRSxHQUFHLENBQUMsUUFBUSxDQUFDLE9BQU8sQ0FBQyxHQUFHLENBQUM7WUFDbEMsV0FBVyxFQUFFO2dCQUNYLFNBQVMsRUFBRSxRQUFRO2dCQUNuQixLQUFLLEVBQUUsS0FBSyxDQUFDLFNBQVM7YUFDdkI7U0FDRixDQUFDLENBQUM7UUFFSCxNQUFNLGVBQWUsR0FBRyxJQUFJLE1BQU0sQ0FBQyxtQkFBbUIsQ0FBQyxJQUFJLEVBQUUsV0FBVyxFQUFFO1lBQ3hFLFdBQVcsRUFDVCxrTEFBa0w7WUFDcEwsSUFBSSxFQUFFLE1BQU0sQ0FBQyxlQUFlLENBQUMsY0FBYyxDQUN6QyxJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsRUFBRSxzQkFBc0IsQ0FBQyxFQUM1QztnQkFDRSxHQUFHLEVBQUUsQ0FBQywyQkFBMkIsQ0FBQzthQUNuQyxDQUNGO1lBQ0QsVUFBVSxFQUFFLEdBQUc7WUFDZixPQUFPLEVBQUUsR0FBRyxDQUFDLFFBQVEsQ0FBQyxPQUFPLENBQUMsR0FBRyxDQUFDO1lBQ2xDLFdBQVcsRUFBRTtnQkFDWCxxQkFBcUIsRUFBRSxlQUFlLENBQUMsVUFBVTtnQkFDakQscUJBQXFCLEVBQUUsWUFBWSxDQUFDLFVBQVU7Z0JBQzlDLHlCQUF5QixFQUFFLElBQUk7Z0JBQy9CLElBQUksRUFBRSxxQkFBcUI7Z0JBQzNCLDhDQUE4QyxFQUFFLE9BQU87Z0JBQ3ZELGlCQUFpQixFQUFFLElBQUk7Z0JBQ3ZCLE1BQU0sRUFBRSxhQUFhO2dCQUNyQixLQUFLLEVBQUUsS0FBSyxDQUFDLFNBQVM7Z0JBQ3RCLFFBQVEsRUFDTixLQUFLLENBQUMsU0FBUyxJQUFJLGVBQWUsQ0FBQyxVQUFVLENBQUMsQ0FBQyxDQUFDLFlBQVksQ0FBQyxDQUFDLENBQUMsS0FBSztnQkFDdEUsa0JBQWtCLEVBQUUsR0FBRyxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsZ0JBQWdCLENBQUM7Z0JBQ2hELFNBQVMsRUFBRSxRQUFRO2dCQUNuQixVQUFVLEVBQUUsR0FBRyxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsZ0JBQWdCLENBQUM7Z0JBQ3hDLHNCQUFzQixFQUFFLGdCQUFnQixDQUFDLE9BQU87Z0JBQ2hELHNCQUFzQixFQUFFLGdCQUFnQixDQUFDLE9BQU87Z0JBQ2hELHdCQUF3QixFQUFFLGdCQUFnQixDQUFDLElBQUs7Z0JBQ2hELHdCQUF3QixFQUFFLGdCQUFnQixDQUFDLElBQUs7Z0JBQ2hELDJCQUEyQixFQUFFLHFCQUFxQixDQUFDLE9BQU87Z0JBQzFELDJCQUEyQixFQUFFLHFCQUFxQixDQUFDLE9BQU87Z0JBQzFELDZCQUE2QixFQUFFLHFCQUFxQixDQUFDLElBQUs7Z0JBQzFELDZCQUE2QixFQUFFLHFCQUFxQixDQUFDLElBQUs7Z0JBQzFELGdCQUFnQixFQUFFLE1BQU0sQ0FBQyxXQUFXLENBQUM7Z0JBQ3JDLGVBQWUsRUFBRSxNQUFNLENBQUMsY0FBYyxDQUFDO2dCQUN2QyxpQkFBaUIsRUFBRSxNQUFNLENBQUMsZ0JBQWdCLENBQUM7Z0JBQzNDLFdBQVcsRUFBRSxRQUFRO2dCQUNyQixXQUFXLEVBQUUsaURBQWlEO2FBQy9EO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsTUFBTSx3QkFBd0IsR0FBRyxJQUFJLE1BQU0sQ0FBQyxtQkFBbUIsQ0FDN0QsSUFBSSxFQUNKLHdCQUF3QixFQUN4QjtZQUNFLFdBQVcsRUFDVCxxUEFBcVA7WUFDdlAsSUFBSSxFQUFFLE1BQU0sQ0FBQyxlQUFlLENBQUMsY0FBYyxDQUN6QyxJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsRUFBRSxzQkFBc0IsQ0FBQyxFQUM1QztnQkFDRSxHQUFHLEVBQUUsQ0FBQyxrQ0FBa0MsQ0FBQzthQUMxQyxDQUNGO1lBQ0QsVUFBVSxFQUFFLEdBQUc7WUFDZixPQUFPLEVBQUUsR0FBRyxDQUFDLFFBQVEsQ0FBQyxPQUFPLENBQUMsR0FBRyxDQUFDO1lBQ2xDLFdBQVcsRUFBRTtnQkFDWCxRQUFRLEVBQ04sS0FBSyxDQUFDLFNBQVMsSUFBSSxlQUFlLENBQUMsVUFBVTtvQkFDM0MsQ0FBQyxDQUFDLFlBQVk7b0JBQ2QsQ0FBQyxDQUFDLEtBQUs7Z0JBQ1gsa0JBQWtCLEVBQUUsSUFBSTtnQkFDeEIsU0FBUyxFQUFFLFFBQVE7YUFDcEI7U0FDRixDQUNGLENBQUM7UUFFRixJQUFJLE1BQU0sQ0FBQyxtQkFBbUIsQ0FBQyxJQUFJLEVBQUUsYUFBYSxFQUFFO1lBQ2xELFdBQVcsRUFDVCx5R0FBeUc7WUFDM0csSUFBSSxFQUFFLE1BQU0sQ0FBQyxlQUFlLENBQUMsY0FBYyxDQUN6QyxJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsRUFBRSxpQkFBaUIsQ0FBQyxFQUN2QztnQkFDRSxHQUFHLEVBQUUsQ0FBQyxzQkFBc0IsQ0FBQzthQUM5QixDQUNGO1lBQ0QsVUFBVSxFQUFFLEdBQUc7WUFDZixPQUFPLEVBQUUsR0FBRyxDQUFDLFFBQVEsQ0FBQyxPQUFPLENBQUMsR0FBRyxDQUFDO1lBQ2xDLFdBQVcsRUFBRTtnQkFDWCxTQUFTLEVBQUUsUUFBUTtnQkFDbkIsT0FBTyxFQUFFLEdBQUc7YUFDYjtTQUNGLENBQUMsQ0FBQztRQUVILDJCQUEyQjtRQUMzQixNQUFNLGtCQUFrQixHQUFHLElBQUksR0FBRyxDQUFDLElBQUksQ0FDckMsSUFBSSxFQUNKLGlDQUFpQyxFQUNqQztZQUNFLFNBQVMsRUFBRSxJQUFJLEdBQUcsQ0FBQyxnQkFBZ0IsQ0FBQyxzQkFBc0IsQ0FBQztTQUM1RCxDQUNGLENBQUM7UUFDRixrQkFBa0IsQ0FBQyxrQkFBa0IsQ0FDbkMsSUFBSSxHQUFHLENBQUMsTUFBTSxDQUFDLElBQUksRUFBRSxnQkFBZ0IsRUFBRTtZQUNyQyxVQUFVLEVBQUUsZ0JBQWdCO1lBQzVCLFVBQVUsRUFBRTtnQkFDVixJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUM7b0JBQ3RCLE1BQU0sRUFBRSxHQUFHLENBQUMsTUFBTSxDQUFDLEtBQUs7b0JBQ3hCLFNBQVMsRUFBRSxDQUFDLEdBQUcsQ0FBQztvQkFDaEIsT0FBTyxFQUFFO3dCQUNQLHFCQUFxQjt3QkFDckIsc0JBQXNCO3dCQUN0QixtQkFBbUI7cUJBQ3BCO2lCQUNGLENBQUM7YUFDSDtTQUNGLENBQUMsQ0FDSCxDQUFDO1FBQ0Ysa0JBQWtCLENBQUMsa0JBQWtCLENBQ25DLElBQUksR0FBRyxDQUFDLE1BQU0sQ0FBQyxJQUFJLEVBQUUsc0JBQXNCLEVBQUU7WUFDM0MsVUFBVSxFQUFFLHNCQUFzQjtZQUNsQyxVQUFVLEVBQUU7Z0JBQ1YsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDO29CQUN0QixNQUFNLEVBQUUsR0FBRyxDQUFDLE1BQU0sQ0FBQyxLQUFLO29CQUN4QixTQUFTLEVBQUUsQ0FBQyxHQUFHLENBQUM7b0JBQ2hCLE9BQU8sRUFBRSxDQUFDLGdCQUFnQixFQUFFLG1CQUFtQixDQUFDO2lCQUNqRCxDQUFDO2FBQ0g7U0FDRixDQUFDLENBQ0gsQ0FBQztRQUNGLGtCQUFrQixDQUFDLGtCQUFrQixDQUNuQyxJQUFJLEdBQUcsQ0FBQyxNQUFNLENBQUMsSUFBSSxFQUFFLHNCQUFzQixFQUFFO1lBQzNDLFVBQVUsRUFBRSxzQkFBc0I7WUFDbEMsVUFBVSxFQUFFO2dCQUNWLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQztvQkFDdEIsTUFBTSxFQUFFLEdBQUcsQ0FBQyxNQUFNLENBQUMsS0FBSztvQkFDeEIsU0FBUyxFQUFFLENBQUMsR0FBRyxDQUFDO29CQUNoQixPQUFPLEVBQUUsQ0FBQywrQkFBK0IsQ0FBQztpQkFDM0MsQ0FBQzthQUNIO1NBQ0YsQ0FBQyxDQUNILENBQUM7UUFDRixrQkFBa0IsQ0FBQyxrQkFBa0IsQ0FDbkMsSUFBSSxHQUFHLENBQUMsTUFBTSxDQUFDLElBQUksRUFBRSxrQkFBa0IsRUFBRTtZQUN2QyxVQUFVLEVBQUUsa0JBQWtCO1lBQzlCLFVBQVUsRUFBRTtnQkFDVixJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUM7b0JBQ3RCLE1BQU0sRUFBRSxHQUFHLENBQUMsTUFBTSxDQUFDLEtBQUs7b0JBQ3hCLFNBQVMsRUFBRSxDQUFDLEdBQUcsQ0FBQztvQkFDaEIsT0FBTyxFQUFFLENBQUMsZ0NBQWdDLENBQUM7aUJBQzVDLENBQUM7YUFDSDtTQUNGLENBQUMsQ0FDSCxDQUFDO1FBRUYsTUFBTSwyQkFBMkIsR0FBRyxJQUFJLE1BQU0sQ0FBQyxtQkFBbUIsQ0FDaEUsSUFBSSxFQUNKLHVCQUF1QixFQUN2QjtZQUNFLFdBQVcsRUFDVCxpUUFBaVE7WUFDblEsSUFBSSxFQUFFLE1BQU0sQ0FBQyxlQUFlLENBQUMsY0FBYyxDQUN6QyxJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsRUFBRSxtQ0FBbUMsQ0FBQyxFQUN6RDtnQkFDRSxHQUFHLEVBQUUsQ0FBQyxpQ0FBaUMsQ0FBQzthQUN6QyxDQUNGO1lBQ0QsVUFBVSxFQUFFLEdBQUc7WUFDZixPQUFPLEVBQUUsR0FBRyxDQUFDLFFBQVEsQ0FBQyxPQUFPLENBQUMsR0FBRyxDQUFDO1lBQ2xDLElBQUksRUFBRSxrQkFBa0I7WUFDeEIsV0FBVyxFQUFFO2dCQUNYLHNCQUFzQixFQUFFLHNCQUFzQixDQUFDLE9BQU87Z0JBQ3RELHNCQUFzQixFQUFFLHNCQUFzQixDQUFDLE9BQU87Z0JBQ3RELHdCQUF3QixFQUFFLHNCQUFzQixDQUFDLElBQUs7Z0JBQ3RELHdCQUF3QixFQUFFLHNCQUFzQixDQUFDLElBQUs7Z0JBQ3RELEtBQUssRUFBRSxLQUFLLENBQUMsU0FBUztnQkFDdEIsU0FBUyxFQUFFLFFBQVE7Z0JBQ25CLFFBQVEsRUFDTiwrUEFBK1A7Z0JBQ2pRLFdBQVcsRUFBRSxRQUFRO2dCQUNyQixXQUFXLEVBQUUsaURBQWlEO2dCQUM5RCxVQUFVLEVBQUUsR0FBRyxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsZ0JBQWdCLENBQUM7Z0JBQ3hDLFFBQVEsRUFDTixLQUFLLENBQUMsU0FBUyxJQUFJLGVBQWUsQ0FBQyxVQUFVO29CQUMzQyxDQUFDLENBQUMsWUFBWTtvQkFDZCxDQUFDLENBQUMsS0FBSztnQkFDWCx5QkFBeUIsRUFBRSxJQUFJO2dCQUMvQiwyQkFBMkIsRUFBRSxrQkFBa0I7YUFDaEQ7U0FDRixDQUNGLENBQUM7UUFFRixnQ0FBZ0M7UUFDaEMsTUFBTSx5QkFBeUIsR0FBRyxJQUFJLE1BQU0sQ0FBQyxJQUFJLENBQy9DLElBQUksRUFDSixpQ0FBaUMsRUFDakM7WUFDRSxXQUFXLEVBQUUsNENBQTRDO1lBQ3pELFFBQVEsRUFBRSxNQUFNLENBQUMsUUFBUSxDQUFDLFVBQVUsQ0FBQyxjQUFjLENBQUM7U0FDckQsQ0FDRixDQUFDO1FBQ0YsTUFBTSw4QkFBOEIsR0FBRztZQUNyQyxRQUFRLEVBQUU7Z0JBQ1I7b0JBQ0UsR0FBRyxFQUFFLHdDQUF3QztpQkFDOUM7Z0JBQ0Q7b0JBQ0UsR0FBRyxFQUFFLHlDQUF5QztpQkFDL0M7Z0JBQ0Q7b0JBQ0UsR0FBRyxFQUFFLDZDQUE2QztvQkFDbEQsTUFBTSxFQUFFLGFBQWE7aUJBQ3RCO2dCQUNEO29CQUNFLEdBQUcsRUFBRSxrRUFBa0U7aUJBQ3hFO2FBQ0Y7WUFDRCxzQkFBc0IsRUFBRSxzQkFBc0IsQ0FBQyxPQUFPO1lBQ3RELHNCQUFzQixFQUFFLHNCQUFzQixDQUFDLE9BQU87WUFDdEQsd0JBQXdCLEVBQUUsc0JBQXNCLENBQUMsSUFBSztZQUN0RCx3QkFBd0IsRUFBRSxzQkFBc0IsQ0FBQyxJQUFLO1lBQ3RELEtBQUssRUFBRSxLQUFLLENBQUMsU0FBUztTQUN2QixDQUFDO1FBQ0YseUJBQXlCLENBQUMsU0FBUyxDQUNqQyxJQUFJLE9BQU8sQ0FBQyxjQUFjLENBQUMsMkJBQTJCLEVBQUU7WUFDdEQsS0FBSyxFQUFFLE1BQU0sQ0FBQyxlQUFlLENBQUMsVUFBVSxDQUN0Qyw4QkFBOEIsQ0FDL0I7U0FDRixDQUFDLENBQ0gsQ0FBQztRQUVGLElBQUksMEJBQTBCLEVBQUU7WUFDOUIsNEJBQTRCO1lBQzVCLE1BQU0sVUFBVSxHQUFHLElBQUksR0FBRyxDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUUsbUJBQW1CLEVBQUU7Z0JBQ3pELFNBQVMsRUFBRSxJQUFJLEdBQUcsQ0FBQyxnQkFBZ0IsQ0FBQyxzQkFBc0IsQ0FBQzthQUM1RCxDQUFDLENBQUM7WUFDSCxVQUFVLENBQUMsa0JBQWtCLENBQzNCLElBQUksR0FBRyxDQUFDLE1BQU0sQ0FBQyxJQUFJLEVBQUUscUJBQXFCLEVBQUU7Z0JBQzFDLFVBQVUsRUFBRSwrQkFBK0I7Z0JBQzNDLFVBQVUsRUFBRTtvQkFDVixJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUM7d0JBQ3RCLE1BQU0sRUFBRSxHQUFHLENBQUMsTUFBTSxDQUFDLEtBQUs7d0JBQ3hCLFNBQVMsRUFBRSxDQUFDLEdBQUcsQ0FBQzt3QkFDaEIsT0FBTyxFQUFFLENBQUMsZ0JBQWdCLENBQUM7cUJBQzVCLENBQUM7aUJBQ0g7YUFDRixDQUFDLENBQ0gsQ0FBQztZQUNGLFVBQVUsQ0FBQyxrQkFBa0IsQ0FDM0IsSUFBSSxHQUFHLENBQUMsTUFBTSxDQUFDLElBQUksRUFBRSw4QkFBOEIsRUFBRTtnQkFDbkQsVUFBVSxFQUFFLDhCQUE4QjtnQkFDMUMsVUFBVSxFQUFFO29CQUNWLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQzt3QkFDdEIsTUFBTSxFQUFFLEdBQUcsQ0FBQyxNQUFNLENBQUMsS0FBSzt3QkFDeEIsU0FBUyxFQUFFLENBQUMsR0FBRyxDQUFDO3dCQUNoQixPQUFPLEVBQUUsQ0FBQyw0QkFBNEIsQ0FBQztxQkFDeEMsQ0FBQztpQkFDSDthQUNGLENBQUMsQ0FDSCxDQUFDO1lBQ0YsVUFBVSxDQUFDLGtCQUFrQixDQUMzQixJQUFJLEdBQUcsQ0FBQyxNQUFNLENBQUMsSUFBSSxFQUFFLGlCQUFpQixFQUFFO2dCQUN0QyxVQUFVLEVBQUUsaUJBQWlCO2dCQUM3QixVQUFVLEVBQUU7b0JBQ1YsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDO3dCQUN0QixNQUFNLEVBQUUsR0FBRyxDQUFDLE1BQU0sQ0FBQyxLQUFLO3dCQUN4QixTQUFTLEVBQUUsQ0FBQyxHQUFHLENBQUM7d0JBQ2hCLE9BQU8sRUFBRTs0QkFDUCxxQkFBcUI7NEJBQ3JCLHNCQUFzQjs0QkFDdEIsbUJBQW1CO3lCQUNwQjtxQkFDRixDQUFDO2lCQUNIO2FBQ0YsQ0FBQyxDQUNILENBQUM7WUFFRixJQUFJLE1BQU0sQ0FBQyxtQkFBbUIsQ0FBQyxJQUFJLEVBQUUsc0JBQXNCLEVBQUU7Z0JBQzNELFdBQVcsRUFDVCxpSEFBaUg7Z0JBQ25ILElBQUksRUFBRSxNQUFNLENBQUMsZUFBZSxDQUFDLGNBQWMsQ0FDekMsSUFBSSxDQUFDLElBQUksQ0FBQyxTQUFTLEVBQUUsNkJBQTZCLENBQUMsRUFDbkQ7b0JBQ0UsR0FBRyxFQUFFLENBQUMsa0NBQWtDLENBQUM7aUJBQzFDLENBQ0Y7Z0JBQ0QsVUFBVSxFQUFFLEdBQUc7Z0JBQ2YsT0FBTyxFQUFFLEdBQUcsQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLEdBQUcsQ0FBQztnQkFDbEMsSUFBSSxFQUFFLFVBQVU7Z0JBQ2hCLFdBQVcsRUFBRTtvQkFDWCxLQUFLLEVBQUUsS0FBSyxDQUFDLFNBQVM7b0JBQ3RCLFNBQVMsRUFBRSxRQUFRO29CQUNuQixXQUFXLEVBQUUsUUFBUTtpQkFDdEI7YUFDRixDQUFDLENBQUM7U0FDSjtRQUVELElBQUksS0FBSyxDQUFDLFNBQVMsSUFBSSxlQUFlLENBQUMsVUFBVSxFQUFFO1lBQ2pELDBCQUEwQjtZQUMxQixNQUFNLFVBQVUsR0FBRyxJQUFJLEdBQUcsQ0FBQyxJQUFJLENBQUMsSUFBSSxFQUFFLFlBQVksRUFBRTtnQkFDbEQsU0FBUyxFQUFFLElBQUksR0FBRyxDQUFDLGdCQUFnQixDQUFDLHNCQUFzQixDQUFDO2FBQzVELENBQUMsQ0FBQztZQUNILFVBQVUsQ0FBQyxrQkFBa0IsQ0FDM0IsSUFBSSxHQUFHLENBQUMsTUFBTSxDQUFDLElBQUksRUFBRSxrQkFBa0IsRUFBRTtnQkFDdkMsVUFBVSxFQUFFLFlBQVk7Z0JBQ3hCLFVBQVUsRUFBRTtvQkFDVixJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUM7d0JBQ3RCLE1BQU0sRUFBRSxHQUFHLENBQUMsTUFBTSxDQUFDLEtBQUs7d0JBQ3hCLFNBQVMsRUFBRSxDQUFDLEdBQUcsQ0FBQzt3QkFDaEIsT0FBTyxFQUFFOzRCQUNQLHFCQUFxQjs0QkFDckIsc0JBQXNCOzRCQUN0QixtQkFBbUI7eUJBQ3BCO3FCQUNGLENBQUM7aUJBQ0g7YUFDRixDQUFDLENBQ0gsQ0FBQztZQUNGLFVBQVUsQ0FBQyxrQkFBa0IsQ0FDM0IsSUFBSSxHQUFHLENBQUMsTUFBTSxDQUFDLElBQUksRUFBRSw0QkFBNEIsRUFBRTtnQkFDakQsVUFBVSxFQUFFLHNCQUFzQjtnQkFDbEMsVUFBVSxFQUFFO29CQUNWLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQzt3QkFDdEIsTUFBTSxFQUFFLEdBQUcsQ0FBQyxNQUFNLENBQUMsS0FBSzt3QkFDeEIsU0FBUyxFQUFFLENBQUMsR0FBRyxDQUFDO3dCQUNoQixPQUFPLEVBQUUsQ0FBQywrQkFBK0IsQ0FBQztxQkFDM0MsQ0FBQztpQkFDSDthQUNGLENBQUMsQ0FDSCxDQUFDO1lBQ0YsVUFBVSxDQUFDLGtCQUFrQixDQUMzQixJQUFJLEdBQUcsQ0FBQyxNQUFNLENBQUMsSUFBSSxFQUFFLHdCQUF3QixFQUFFO2dCQUM3QyxVQUFVLEVBQUUsa0JBQWtCO2dCQUM5QixVQUFVLEVBQUU7b0JBQ1YsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDO3dCQUN0QixNQUFNLEVBQUUsR0FBRyxDQUFDLE1BQU0sQ0FBQyxLQUFLO3dCQUN4QixTQUFTLEVBQUUsQ0FBQyxHQUFHLENBQUM7d0JBQ2hCLE9BQU8sRUFBRSxDQUFDLGdDQUFnQyxDQUFDO3FCQUM1QyxDQUFDO2lCQUNIO2FBQ0YsQ0FBQyxDQUNILENBQUM7WUFDRixVQUFVLENBQUMsa0JBQWtCLENBQzNCLElBQUksR0FBRyxDQUFDLE1BQU0sQ0FBQyxJQUFJLEVBQUUsNEJBQTRCLEVBQUU7Z0JBQ2pELFVBQVUsRUFBRSxzQkFBc0I7Z0JBQ2xDLFVBQVUsRUFBRTtvQkFDVixJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUM7d0JBQ3RCLE1BQU0sRUFBRSxHQUFHLENBQUMsTUFBTSxDQUFDLEtBQUs7d0JBQ3hCLFNBQVMsRUFBRSxDQUFDLEdBQUcsQ0FBQzt3QkFDaEIsT0FBTyxFQUFFLENBQUMsZ0JBQWdCLEVBQUUsbUJBQW1CLENBQUM7cUJBQ2pELENBQUM7aUJBQ0g7YUFDRixDQUFDLENBQ0gsQ0FBQztZQUVGLE1BQU0sa0JBQWtCLEdBQUcsSUFBSSxNQUFNLENBQUMsbUJBQW1CLENBQ3ZELElBQUksRUFDSixjQUFjLEVBQ2Q7Z0JBQ0UsV0FBVyxFQUNULCtJQUErSTtnQkFDakosSUFBSSxFQUFFLFVBQVU7Z0JBQ2hCLElBQUksRUFBRSxNQUFNLENBQUMsZUFBZSxDQUFDLGNBQWMsQ0FDekMsSUFBSSxDQUFDLElBQUksQ0FBQyxTQUFTLEVBQUUsMEJBQTBCLENBQUMsRUFDaEQ7b0JBQ0UsR0FBRyxFQUFFLENBQUMsK0JBQStCLENBQUM7aUJBQ3ZDLENBQ0Y7Z0JBQ0QsVUFBVSxFQUFFLEdBQUc7Z0JBQ2YsT0FBTyxFQUFFLEdBQUcsQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLEdBQUcsQ0FBQztnQkFDbEMsV0FBVyxFQUFFO29CQUNYLEtBQUssRUFBRSxLQUFLLENBQUMsU0FBUztvQkFDdEIsbUJBQW1CLEVBQUUsYUFBYSxDQUFDLE9BQU87b0JBQzFDLG1CQUFtQixFQUFFLGFBQWEsQ0FBQyxPQUFPO29CQUMxQyxxQkFBcUIsRUFBRSxhQUFhLENBQUMsSUFBSztvQkFDMUMscUJBQXFCLEVBQUUsYUFBYSxDQUFDLElBQUs7b0JBQzFDLHlCQUF5QixFQUFFLElBQUk7b0JBQy9CLElBQUksRUFBRSxxQkFBcUI7b0JBQzNCLE1BQU0sRUFBRSxHQUFHLENBQUMsRUFBRSxDQUFDLEdBQUcsQ0FBQyxhQUFhLENBQUM7b0JBQ2pDLFFBQVEsRUFBRSxZQUFZO29CQUN0QixrQkFBa0IsRUFBRSxHQUFHLENBQUMsRUFBRSxDQUFDLEdBQUcsQ0FBQyxnQkFBZ0IsQ0FBQztvQkFDaEQsU0FBUyxFQUFFLFFBQVE7b0JBQ25CLFdBQVcsRUFBRSxRQUFRO29CQUNyQixXQUFXLEVBQUUsaURBQWlEO29CQUM5RCxVQUFVLEVBQUUsR0FBRyxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsZ0JBQWdCLENBQUM7aUJBQ3pDO2FBQ0YsQ0FDRixDQUFDO1lBRUYsa0NBQWtDO1lBQ2xDLE1BQU0sU0FBUyxHQUFHLElBQUksVUFBVSxDQUFDLE9BQU8sQ0FBQyxJQUFJLEVBQUUsa0JBQWtCLEVBQUU7Z0JBQ2pFLFdBQVcsRUFBRSx1Q0FBdUM7Z0JBQ3BELFdBQVcsRUFDVCxxSEFBcUg7YUFDeEgsQ0FBQyxDQUFDO1lBQ0gsTUFBTSxXQUFXLEdBQUcsSUFBSSxVQUFVLENBQUMsaUJBQWlCLENBQUMsa0JBQWtCLENBQUMsQ0FBQztZQUN6RSxTQUFTLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxLQUFLLEVBQUUsV0FBVyxFQUFFO2dCQUMzQyxpQkFBaUIsRUFBRTtvQkFDakIsdUNBQXVDLEVBQUUsS0FBSztpQkFDL0M7YUFDRixDQUFDLENBQUM7WUFDSCxNQUFNLFdBQVcsR0FBRyxTQUFTLENBQUMsSUFBSSxDQUFDLFdBQVcsQ0FBQyxVQUFVLENBQUMsQ0FBQztZQUMzRCxXQUFXLENBQUMsU0FBUyxDQUFDLEtBQUssRUFBRSxXQUFXLEVBQUU7Z0JBQ3hDLGlCQUFpQixFQUFFO29CQUNqQix1Q0FBdUMsRUFBRSxLQUFLO2lCQUMvQzthQUNGLENBQUMsQ0FBQztTQUNKO1FBRUQsTUFBTSxvQkFBb0IsR0FBRyxJQUFJLE1BQU0sQ0FBQyxtQkFBbUIsQ0FDekQsSUFBSSxFQUNKLGdCQUFnQixFQUNoQjtZQUNFLFdBQVcsRUFDVCxpSEFBaUg7WUFDbkgsSUFBSSxFQUFFLE1BQU0sQ0FBQyxlQUFlLENBQUMsY0FBYyxDQUN6QyxJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsRUFBRSwyQkFBMkIsQ0FBQyxFQUNqRDtnQkFDRSxHQUFHLEVBQUUsQ0FBQyxnQ0FBZ0MsQ0FBQzthQUN4QyxDQUNGO1lBQ0QsVUFBVSxFQUFFLEdBQUc7WUFDZixPQUFPLEVBQUUsR0FBRyxDQUFDLFFBQVEsQ0FBQyxPQUFPLENBQUMsR0FBRyxDQUFDO1lBQ2xDLFdBQVcsRUFBRTtnQkFDWCxTQUFTLEVBQUUsUUFBUTtnQkFDbkIsS0FBSyxFQUFFLEtBQUssQ0FBQyxTQUFTO2dCQUN0QixXQUFXLEVBQUUsUUFBUTtnQkFDckIsV0FBVyxFQUFFLGlEQUFpRDthQUMvRDtTQUNGLENBQ0YsQ0FBQztRQUVGLHVCQUF1QjtRQUN2QixNQUFNLFlBQVksR0FBRyxJQUFJLEdBQUcsQ0FBQyxJQUFJLENBQUMsSUFBSSxFQUFFLGNBQWMsRUFBRTtZQUN0RCxTQUFTLEVBQUUsSUFBSSxHQUFHLENBQUMsZ0JBQWdCLENBQUMsd0JBQXdCLENBQUM7U0FDOUQsQ0FBQyxDQUFDO1FBRUgsWUFBWSxDQUFDLFdBQVcsQ0FDdEIsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDO1lBQ3RCLE1BQU0sRUFBRSxHQUFHLENBQUMsTUFBTSxDQUFDLEtBQUs7WUFDeEIsT0FBTyxFQUFFO2dCQUNQLHdCQUF3QjtnQkFDeEIsMEJBQTBCO2dCQUMxQixvQkFBb0I7YUFDckI7WUFDRCxTQUFTLEVBQUUsQ0FBQyxHQUFHLENBQUM7U0FDakIsQ0FBQyxDQUNILENBQUM7UUFFRixZQUFZLENBQUMsV0FBVyxDQUN0QixJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUM7WUFDdEIsTUFBTSxFQUFFLEdBQUcsQ0FBQyxNQUFNLENBQUMsS0FBSztZQUN4QixPQUFPLEVBQUU7Z0JBQ1AseUJBQXlCO2dCQUN6QixzQkFBc0I7Z0JBQ3RCLGNBQWM7Z0JBQ2QsZUFBZTtnQkFDZiwrQkFBK0I7Z0JBQy9CLGNBQWM7YUFDZjtZQUNELFNBQVMsRUFBRSxDQUFDLFlBQVksQ0FBQyxTQUFTLEVBQUUsWUFBWSxDQUFDLGFBQWEsQ0FBQyxHQUFHLENBQUMsQ0FBQztTQUNyRSxDQUFDLENBQ0gsQ0FBQztRQUVGLE1BQU0sY0FBYyxHQUFHLElBQUksUUFBUSxDQUFDLGlCQUFpQixDQUNuRCxJQUFJLEVBQ0osK0JBQStCLEVBQy9CO1lBQ0Usa0JBQWtCLEVBQUUsZUFBZSxHQUFHLEtBQUssQ0FBQyxTQUFTO1lBQ3JELGtCQUFrQixFQUFFLFdBQVc7WUFDL0Isa0NBQWtDLEVBQUU7Z0JBQ2xDLFNBQVMsRUFBRSxZQUFZLENBQUMsU0FBUztnQkFDakMsY0FBYyxFQUFFO29CQUNkLGlCQUFpQixFQUFFLEdBQUc7b0JBQ3RCLFNBQVMsRUFBRSxDQUFDO2lCQUNiO2dCQUNELGlCQUFpQixFQUFFLE1BQU07Z0JBQ3pCLE1BQU0sRUFDSixnR0FBZ0c7Z0JBQ2xHLGlCQUFpQixFQUNmLDBJQUEwSTtnQkFDNUksT0FBTyxFQUFFLFlBQVksQ0FBQyxPQUFPO2FBQzlCO1NBQ0YsQ0FDRixDQUFDO1FBRUYsaUJBQWlCO1FBQ2pCLE1BQU0sbUJBQW1CLEdBQUcsS0FBSyxDQUFDLG9CQUFvQjtZQUNwRCxDQUFDLENBQUMsSUFBSSxLQUFLLENBQUMsb0JBQW9CLENBQUMsV0FBVyxFQUFFLEVBQUU7WUFDaEQsQ0FBQyxDQUFDLEVBQUUsQ0FBQztRQUNQLE1BQU0sc0JBQXNCLEdBQUcsSUFBSSxJQUFJLENBQUMsUUFBUSxDQUM5QyxJQUFJLEVBQ0osd0JBQXdCLEVBQ3hCO1lBQ0UsWUFBWSxFQUFFLDJCQUEyQixtQkFBbUIsRUFBRTtTQUMvRCxDQUNGLENBQUM7UUFFRixNQUFNLHNCQUFzQixHQUFHLElBQUksSUFBSSxDQUFDLEtBQUssQ0FDM0MsSUFBSSxFQUNKLHdCQUF3QixFQUN4QjtZQUNFLFFBQVEsRUFBRSxzQkFBc0I7WUFDaEMsU0FBUyxFQUFFLGlCQUFpQjtZQUM1QixXQUFXLEVBQUUsS0FBSyxDQUFDLFNBQVMsR0FBRyxhQUFhO1lBQzVDLE1BQU0sRUFBRSxZQUFZO1lBQ3BCLFFBQVEsRUFBRSxVQUFVO1lBQ3BCLE9BQU8sRUFBRTtnQkFDUDtvQkFDRSxJQUFJLEVBQUUsV0FBVztvQkFDakIsSUFBSSxFQUFFLElBQUksQ0FBQyxNQUFNLENBQUMsT0FBTztpQkFDMUI7Z0JBQ0Q7b0JBQ0UsSUFBSSxFQUFFLGVBQWU7b0JBQ3JCLElBQUksRUFBRSxJQUFJLENBQUMsTUFBTSxDQUFDLE9BQU87aUJBQzFCO2dCQUNEO29CQUNFLElBQUksRUFBRSxVQUFVO29CQUNoQixJQUFJLEVBQUUsSUFBSSxDQUFDLE1BQU0sQ0FBQyxNQUFNO2lCQUN6QjtnQkFDRDtvQkFDRSxJQUFJLEVBQUUsbUJBQW1CO29CQUN6QixJQUFJLEVBQUUsSUFBSSxDQUFDLE1BQU0sQ0FBQyxNQUFNO2lCQUN6QjtnQkFDRDtvQkFDRSxJQUFJLEVBQUUscUJBQXFCO29CQUMzQixJQUFJLEVBQUUsSUFBSSxDQUFDLE1BQU0sQ0FBQyxNQUFNO2lCQUN6QjtnQkFDRDtvQkFDRSxJQUFJLEVBQUUsUUFBUTtvQkFDZCxJQUFJLEVBQUUsSUFBSSxDQUFDLE1BQU0sQ0FBQyxNQUFNO2lCQUN6QjtnQkFDRDtvQkFDRSxJQUFJLEVBQUUsZ0JBQWdCO29CQUN0QixJQUFJLEVBQUUsSUFBSSxDQUFDLE1BQU0sQ0FBQyxNQUFNO2lCQUN6QjtnQkFDRDtvQkFDRSxJQUFJLEVBQUUsY0FBYztvQkFDcEIsSUFBSSxFQUFFLElBQUksQ0FBQyxNQUFNLENBQUMsTUFBTTtpQkFDekI7Z0JBQ0Q7b0JBQ0UsSUFBSSxFQUFFLGVBQWU7b0JBQ3JCLElBQUksRUFBRSxJQUFJLENBQUMsTUFBTSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLE1BQU0sQ0FBQztpQkFDNUM7Z0JBQ0Q7b0JBQ0UsSUFBSSxFQUFFLG1CQUFtQjtvQkFDekIsSUFBSSxFQUFFLElBQUksQ0FBQyxNQUFNLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsTUFBTSxDQUFDO2lCQUM1QztnQkFDRDtvQkFDRSxJQUFJLEVBQUUsNkJBQTZCO29CQUNuQyxJQUFJLEVBQUUsSUFBSSxDQUFDLE1BQU0sQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxNQUFNLENBQUM7aUJBQzVDO2dCQUNEO29CQUNFLElBQUksRUFBRSxhQUFhO29CQUNuQixJQUFJLEVBQUUsSUFBSSxDQUFDLE1BQU0sQ0FBQyxNQUFNLENBQUM7d0JBQ3ZCOzRCQUNFLElBQUksRUFBRSxVQUFVOzRCQUNoQixJQUFJLEVBQUUsSUFBSSxDQUFDLE1BQU0sQ0FBQyxNQUFNO3lCQUN6Qjt3QkFDRDs0QkFDRSxJQUFJLEVBQUUsU0FBUzs0QkFDZixJQUFJLEVBQUUsSUFBSSxDQUFDLE1BQU0sQ0FBQyxNQUFNO3lCQUN6Qjt3QkFDRDs0QkFDRSxJQUFJLEVBQUUsU0FBUzs0QkFDZixJQUFJLEVBQUUsSUFBSSxDQUFDLE1BQU0sQ0FBQyxLQUFLLENBQ3JCLElBQUksQ0FBQyxNQUFNLENBQUMsTUFBTSxDQUFDO2dDQUNqQjtvQ0FDRSxJQUFJLEVBQUUsTUFBTTtvQ0FDWixJQUFJLEVBQUUsSUFBSSxDQUFDLE1BQU0sQ0FBQyxNQUFNO2lDQUN6QjtnQ0FDRDtvQ0FDRSxJQUFJLEVBQUUsT0FBTztvQ0FDYixJQUFJLEVBQUUsSUFBSSxDQUFDLE1BQU0sQ0FBQyxNQUFNO2lDQUN6Qjs2QkFDRixDQUFDLENBQ0g7eUJBQ0Y7d0JBQ0Q7NEJBQ0UsSUFBSSxFQUFFLEtBQUs7NEJBQ1gsSUFBSSxFQUFFLElBQUksQ0FBQyxNQUFNLENBQUMsTUFBTTt5QkFDekI7d0JBQ0Q7NEJBQ0UsSUFBSSxFQUFFLE1BQU07NEJBQ1osSUFBSSxFQUFFLElBQUksQ0FBQyxNQUFNLENBQUMsTUFBTTt5QkFDekI7d0JBQ0Q7NEJBQ0UsSUFBSSxFQUFFLGFBQWE7NEJBQ25CLElBQUksRUFBRSxJQUFJLENBQUMsTUFBTSxDQUFDLE1BQU07eUJBQ3pCO3dCQUNEOzRCQUNFLElBQUksRUFBRSxZQUFZOzRCQUNsQixJQUFJLEVBQUUsSUFBSSxDQUFDLE1BQU0sQ0FBQyxNQUFNO3lCQUN6Qjt3QkFDRDs0QkFDRSxJQUFJLEVBQUUsV0FBVzs0QkFDakIsSUFBSSxFQUFFLElBQUksQ0FBQyxNQUFNLENBQUMsTUFBTTt5QkFDekI7cUJBQ0YsQ0FBQztpQkFDSDthQUNGO1lBQ0QsYUFBYSxFQUFFO2dCQUNiO29CQUNFLElBQUksRUFBRSxNQUFNO29CQUNaLElBQUksRUFBRSxJQUFJLENBQUMsTUFBTSxDQUFDLE9BQU87aUJBQzFCO2dCQUNEO29CQUNFLElBQUksRUFBRSxPQUFPO29CQUNiLElBQUksRUFBRSxJQUFJLENBQUMsTUFBTSxDQUFDLE9BQU87aUJBQzFCO2dCQUNEO29CQUNFLElBQUksRUFBRSxLQUFLO29CQUNYLElBQUksRUFBRSxJQUFJLENBQUMsTUFBTSxDQUFDLE9BQU87aUJBQzFCO2dCQUNEO29CQUNFLElBQUksRUFBRSxNQUFNO29CQUNaLElBQUksRUFBRSxJQUFJLENBQUMsTUFBTSxDQUFDLE9BQU87aUJBQzFCO2FBQ0Y7WUFDRCxVQUFVLEVBQUU7Z0JBQ1YsV0FBVyxFQUFFLElBQUksSUFBSSxDQUFDLFdBQVcsQ0FDL0IsMENBQTBDLENBQzNDO2dCQUNELFlBQVksRUFBRSxJQUFJLElBQUksQ0FBQyxZQUFZLENBQ2pDLDREQUE0RCxDQUM3RDtnQkFDRCxvQkFBb0IsRUFBRSxJQUFJLElBQUksQ0FBQyxvQkFBb0IsQ0FDakQsb0NBQW9DLENBQ3JDO2FBQ0Y7U0FDRixDQUNGLENBQUM7UUFFRixtQ0FBbUM7UUFDbkMsbUZBQW1GO1FBQ25GLHVHQUF1RztRQUN2RywrRkFBK0Y7UUFDL0YsTUFBTTtRQUVOLElBQUksS0FBSyxDQUFDLFNBQVMsSUFBSSxlQUFlLENBQUMsVUFBVSxFQUFFO1lBQ2pELElBQUksT0FBTyxDQUFDLFFBQVEsQ0FBQyxJQUFJLEVBQUUsd0JBQXdCLEVBQUU7Z0JBQ25ELFlBQVksRUFBRSxzQkFBc0IsQ0FBQyxZQUFZO2dCQUNqRCxTQUFTLEVBQUUsR0FBRyxDQUFDLEdBQUcsQ0FBQyxVQUFVO2dCQUM3QixVQUFVLEVBQUU7b0JBQ1YsSUFBSSxFQUFFLGlCQUFpQjtvQkFDdkIsV0FBVyxFQUFFLEtBQUssQ0FBQyxTQUFTLEdBQUcsb0JBQW9CO29CQUNuRCxVQUFVLEVBQUU7d0JBQ1Ysd0JBQXdCLEVBQUUsR0FBRzt3QkFDN0IsVUFBVSxFQUFFLE1BQU07cUJBQ25CO29CQUNELGlCQUFpQixFQUFFO3dCQUNqQixPQUFPLEVBQUU7NEJBQ1A7Z0NBQ0UsSUFBSSxFQUFFLE1BQU07Z0NBQ1osSUFBSSxFQUFFLElBQUksQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLFdBQVc7NkJBQ25DOzRCQUNEO2dDQUNFLElBQUksRUFBRSxNQUFNO2dDQUNaLElBQUksRUFBRSxJQUFJLENBQUMsTUFBTSxDQUFDLE1BQU0sQ0FBQyxXQUFXOzZCQUNyQzs0QkFDRDtnQ0FDRSxJQUFJLEVBQUUsVUFBVTtnQ0FDaEIsSUFBSSxFQUFFLElBQUksQ0FBQyxNQUFNLENBQUMsTUFBTSxDQUFDLFdBQVc7NkJBQ3JDOzRCQUNEO2dDQUNFLElBQUksRUFBRSxPQUFPO2dDQUNiLElBQUksRUFBRSxJQUFJLENBQUMsTUFBTSxDQUFDLE9BQU8sQ0FBQyxXQUFXOzZCQUN0Qzs0QkFDRDtnQ0FDRSxJQUFJLEVBQUUsV0FBVztnQ0FDakIsSUFBSSxFQUFFLElBQUksQ0FBQyxNQUFNLENBQUMsTUFBTSxDQUFDLFdBQVc7NkJBQ3JDOzRCQUNEO2dDQUNFLElBQUksRUFBRSxRQUFRO2dDQUNkLElBQUksRUFBRSxJQUFJLENBQUMsTUFBTSxDQUFDLE1BQU0sQ0FBQyxXQUFXOzZCQUNyQzs0QkFDRDtnQ0FDRSxJQUFJLEVBQUUsTUFBTTtnQ0FDWixJQUFJLEVBQUUsSUFBSSxDQUFDLE1BQU0sQ0FBQyxNQUFNLENBQUMsV0FBVzs2QkFDckM7NEJBQ0Q7Z0NBQ0UsSUFBSSxFQUFFLEtBQUs7Z0NBQ1gsSUFBSSxFQUFFLElBQUksQ0FBQyxNQUFNLENBQUMsTUFBTSxDQUFDLFdBQVc7NkJBQ3JDOzRCQUNEO2dDQUNFLElBQUksRUFBRSxRQUFRO2dDQUNkLElBQUksRUFBRSxJQUFJLENBQUMsTUFBTSxDQUFDLE9BQU8sQ0FBQyxXQUFXOzZCQUN0Qzs0QkFDRDtnQ0FDRSxJQUFJLEVBQUUsVUFBVTtnQ0FDaEIsSUFBSSxFQUFFLElBQUksQ0FBQyxNQUFNLENBQUMsTUFBTSxDQUFDLFdBQVc7NkJBQ3JDOzRCQUNEO2dDQUNFLElBQUksRUFBRSxXQUFXO2dDQUNqQixJQUFJLEVBQUUsSUFBSSxDQUFDLE1BQU0sQ0FBQyxNQUFNLENBQUMsV0FBVzs2QkFDckM7NEJBQ0Q7Z0NBQ0UsSUFBSSxFQUFFLGFBQWE7Z0NBQ25CLElBQUksRUFBRSxJQUFJLENBQUMsTUFBTSxDQUFDLE1BQU0sQ0FBQyxXQUFXOzZCQUNyQzs0QkFDRDtnQ0FDRSxJQUFJLEVBQUUsUUFBUTtnQ0FDZCxJQUFJLEVBQUUsSUFBSSxDQUFDLE1BQU0sQ0FBQyxNQUFNLENBQUMsV0FBVzs2QkFDckM7NEJBQ0Q7Z0NBQ0UsSUFBSSxFQUFFLFlBQVk7Z0NBQ2xCLElBQUksRUFBRSxJQUFJLENBQUMsTUFBTSxDQUFDLE1BQU0sQ0FBQyxXQUFXOzZCQUNyQzs0QkFDRDtnQ0FDRSxJQUFJLEVBQUUsV0FBVztnQ0FDakIsSUFBSSxFQUFFLElBQUksQ0FBQyxNQUFNLENBQUMsTUFBTSxDQUFDLFdBQVc7NkJBQ3JDOzRCQUNEO2dDQUNFLElBQUksRUFBRSxZQUFZO2dDQUNsQixJQUFJLEVBQUUsSUFBSSxDQUFDLE1BQU0sQ0FBQyxNQUFNLENBQUMsV0FBVzs2QkFDckM7NEJBQ0Q7Z0NBQ0UsSUFBSSxFQUFFLGlCQUFpQjtnQ0FDdkIsSUFBSSxFQUFFLElBQUksQ0FBQyxNQUFNLENBQUMsTUFBTSxDQUFDLFdBQVc7NkJBQ3JDOzRCQUNEO2dDQUNFLElBQUksRUFBRSxjQUFjO2dDQUNwQixJQUFJLEVBQUUsSUFBSSxDQUFDLE1BQU0sQ0FBQyxPQUFPLENBQUMsV0FBVzs2QkFDdEM7NEJBQ0Q7Z0NBQ0UsSUFBSSxFQUFFLFdBQVc7Z0NBQ2pCLElBQUksRUFBRSxJQUFJLENBQUMsTUFBTSxDQUFDLEtBQUssQ0FBQyxXQUFXOzZCQUNwQzs0QkFDRDtnQ0FDRSxJQUFJLEVBQUUsZUFBZTtnQ0FDckIsSUFBSSxFQUFFLElBQUksQ0FBQyxNQUFNLENBQUMsTUFBTSxDQUFDLFdBQVc7NkJBQ3JDOzRCQUNEO2dDQUNFLElBQUksRUFBRSxhQUFhO2dDQUNuQixJQUFJLEVBQUUsSUFBSSxDQUFDLE1BQU0sQ0FBQyxNQUFNLENBQUMsV0FBVzs2QkFDckM7NEJBQ0Q7Z0NBQ0UsSUFBSSxFQUFFLFdBQVc7Z0NBQ2pCLElBQUksRUFBRSxJQUFJLENBQUMsTUFBTSxDQUFDLE1BQU0sQ0FBQyxXQUFXOzZCQUNyQzs0QkFDRDtnQ0FDRSxJQUFJLEVBQUUsb0JBQW9CO2dDQUMxQixJQUFJLEVBQUUsSUFBSSxDQUFDLE1BQU0sQ0FBQyxNQUFNLENBQUMsV0FBVzs2QkFDckM7NEJBQ0Q7Z0NBQ0UsSUFBSSxFQUFFLGFBQWE7Z0NBQ25CLElBQUksRUFBRSxJQUFJLENBQUMsTUFBTSxDQUFDLE1BQU0sQ0FBQyxXQUFXOzZCQUNyQzs0QkFDRDtnQ0FDRSxJQUFJLEVBQUUsWUFBWTtnQ0FDbEIsSUFBSSxFQUFFLElBQUksQ0FBQyxNQUFNLENBQUMsTUFBTSxDQUFDLFdBQVc7NkJBQ3JDOzRCQUNEO2dDQUNFLElBQUksRUFBRSxpQkFBaUI7Z0NBQ3ZCLElBQUksRUFBRSxJQUFJLENBQUMsTUFBTSxDQUFDLE9BQU8sQ0FBQyxXQUFXOzZCQUN0Qzt5QkFDRjt3QkFDRCxVQUFVLEVBQUUsS0FBSzt3QkFDakIsV0FBVyxFQUFFLDBDQUEwQzt3QkFDdkQsUUFBUSxFQUNOLE9BQU8sR0FBRyxlQUFlLENBQUMsVUFBVSxHQUFHLHVCQUF1Qjt3QkFDaEUsWUFBWSxFQUNWLDREQUE0RDt3QkFDOUQsU0FBUyxFQUFFOzRCQUNULG9CQUFvQixFQUNsQixvREFBb0Q7NEJBQ3RELFVBQVUsRUFBRTtnQ0FDVixhQUFhLEVBQUUsR0FBRztnQ0FDbEIsc0JBQXNCLEVBQUUsR0FBRzs2QkFDNUI7eUJBQ0Y7d0JBQ0Qsc0JBQXNCLEVBQUUsS0FBSztxQkFDOUI7b0JBQ0QsU0FBUyxFQUFFLGdCQUFnQjtvQkFDM0IsYUFBYSxFQUFFO3dCQUNiOzRCQUNFLElBQUksRUFBRSxNQUFNOzRCQUNaLElBQUksRUFBRSxJQUFJLENBQUMsTUFBTSxDQUFDLE9BQU8sQ0FBQyxXQUFXO3lCQUN0Qzt3QkFDRDs0QkFDRSxJQUFJLEVBQUUsT0FBTzs0QkFDYixJQUFJLEVBQUUsSUFBSSxDQUFDLE1BQU0sQ0FBQyxPQUFPLENBQUMsV0FBVzt5QkFDdEM7d0JBQ0Q7NEJBQ0UsSUFBSSxFQUFFLEtBQUs7NEJBQ1gsSUFBSSxFQUFFLElBQUksQ0FBQyxNQUFNLENBQUMsT0FBTyxDQUFDLFdBQVc7eUJBQ3RDO3dCQUNEOzRCQUNFLElBQUksRUFBRSxNQUFNOzRCQUNaLElBQUksRUFBRSxJQUFJLENBQUMsTUFBTSxDQUFDLE9BQU8sQ0FBQyxXQUFXO3lCQUN0QztxQkFDRjtpQkFDRjthQUNGLENBQUMsQ0FBQztTQUNKO2FBQU07WUFDTCxpQ0FBaUM7WUFDakMsSUFBSSxPQUFPLENBQUMsUUFBUSxDQUFDLElBQUksRUFBRSx3QkFBd0IsRUFBRTtnQkFDbkQsWUFBWSxFQUFFLHNCQUFzQixDQUFDLFlBQVk7Z0JBQ2pELFNBQVMsRUFBRSxHQUFHLENBQUMsR0FBRyxDQUFDLFVBQVU7Z0JBQzdCLFVBQVUsRUFBRTtvQkFDVixJQUFJLEVBQUUsaUJBQWlCO29CQUN2QixXQUFXLEVBQUUsS0FBSyxDQUFDLFNBQVMsR0FBRyxvQkFBb0I7b0JBQ25ELFVBQVUsRUFBRTt3QkFDVixRQUFRLEVBQUUsTUFBTTtxQkFDakI7b0JBQ0QsaUJBQWlCLEVBQUU7d0JBQ2pCLE9BQU8sRUFBRTs0QkFDUDtnQ0FDRSxJQUFJLEVBQUUsTUFBTTtnQ0FDWixJQUFJLEVBQUUsSUFBSSxDQUFDLE1BQU0sQ0FBQyxNQUFNLENBQUMsV0FBVzs2QkFDckM7NEJBQ0Q7Z0NBQ0UsSUFBSSxFQUFFLE1BQU07Z0NBQ1osSUFBSSxFQUFFLElBQUksQ0FBQyxNQUFNLENBQUMsTUFBTSxDQUFDLFdBQVc7NkJBQ3JDOzRCQUNEO2dDQUNFLElBQUksRUFBRSxLQUFLO2dDQUNYLElBQUksRUFBRSxJQUFJLENBQUMsTUFBTSxDQUFDLE1BQU0sQ0FBQyxXQUFXOzZCQUNyQzs0QkFDRDtnQ0FDRSxJQUFJLEVBQUUsV0FBVztnQ0FDakIsSUFBSSxFQUFFLElBQUksQ0FBQyxNQUFNLENBQUMsTUFBTSxDQUFDLFdBQVc7NkJBQ3JDOzRCQUNEO2dDQUNFLElBQUksRUFBRSxhQUFhO2dDQUNuQixJQUFJLEVBQUUsSUFBSSxDQUFDLE1BQU0sQ0FBQyxPQUFPLENBQUMsV0FBVzs2QkFDdEM7NEJBQ0Q7Z0NBQ0UsSUFBSSxFQUFFLFdBQVc7Z0NBQ2pCLElBQUksRUFBRSxJQUFJLENBQUMsTUFBTSxDQUFDLE1BQU0sQ0FBQyxXQUFXOzZCQUNyQzs0QkFDRDtnQ0FDRSxJQUFJLEVBQUUsYUFBYTtnQ0FDbkIsSUFBSSxFQUFFLElBQUksQ0FBQyxNQUFNLENBQUMsT0FBTyxDQUFDLFdBQVc7NkJBQ3RDOzRCQUNEO2dDQUNFLElBQUksRUFBRSx5QkFBeUI7Z0NBQy9CLElBQUksRUFBRSxJQUFJLENBQUMsTUFBTSxDQUFDLE1BQU0sQ0FBQyxXQUFXOzZCQUNyQzs0QkFDRDtnQ0FDRSxJQUFJLEVBQUUsd0JBQXdCO2dDQUM5QixJQUFJLEVBQUUsSUFBSSxDQUFDLE1BQU0sQ0FBQyxNQUFNLENBQUMsV0FBVzs2QkFDckM7NEJBQ0Q7Z0NBQ0UsSUFBSSxFQUFFLDBCQUEwQjtnQ0FDaEMsSUFBSSxFQUFFLElBQUksQ0FBQyxNQUFNLENBQUMsTUFBTSxDQUFDLFdBQVc7NkJBQ3JDOzRCQUNEO2dDQUNFLElBQUksRUFBRSxpQkFBaUI7Z0NBQ3ZCLElBQUksRUFBRSxJQUFJLENBQUMsTUFBTSxDQUFDLE1BQU0sQ0FBQyxXQUFXOzZCQUNyQzs0QkFDRDtnQ0FDRSxJQUFJLEVBQUUsb0JBQW9CO2dDQUMxQixJQUFJLEVBQUUsSUFBSSxDQUFDLE1BQU0sQ0FBQyxNQUFNLENBQUMsV0FBVzs2QkFDckM7NEJBQ0Q7Z0NBQ0UsSUFBSSxFQUFFLGdCQUFnQjtnQ0FDdEIsSUFBSSxFQUFFLElBQUksQ0FBQyxNQUFNLENBQUMsT0FBTyxDQUFDLFdBQVc7NkJBQ3RDOzRCQUNEO2dDQUNFLElBQUksRUFBRSxZQUFZO2dDQUNsQixJQUFJLEVBQUUsSUFBSSxDQUFDLE1BQU0sQ0FBQyxPQUFPLENBQUMsV0FBVzs2QkFDdEM7NEJBQ0Q7Z0NBQ0UsSUFBSSxFQUFFLGNBQWM7Z0NBQ3BCLElBQUksRUFBRSxJQUFJLENBQUMsTUFBTSxDQUFDLE1BQU0sQ0FBQyxXQUFXOzZCQUNyQzs0QkFDRDtnQ0FDRSxJQUFJLEVBQUUsYUFBYTtnQ0FDbkIsSUFBSSxFQUFFLElBQUksQ0FBQyxNQUFNLENBQUMsTUFBTSxDQUFDLFdBQVc7NkJBQ3JDOzRCQUNEO2dDQUNFLElBQUksRUFBRSxlQUFlO2dDQUNyQixJQUFJLEVBQUUsSUFBSSxDQUFDLE1BQU0sQ0FBQyxNQUFNLENBQUMsV0FBVzs2QkFDckM7NEJBQ0Q7Z0NBQ0UsSUFBSSxFQUFFLFlBQVk7Z0NBQ2xCLElBQUksRUFBRSxJQUFJLENBQUMsTUFBTSxDQUFDLE1BQU0sQ0FBQyxXQUFXOzZCQUNyQzs0QkFDRDtnQ0FDRSxJQUFJLEVBQUUsWUFBWTtnQ0FDbEIsSUFBSSxFQUFFLElBQUksQ0FBQyxNQUFNLENBQUMsTUFBTSxDQUFDLFdBQVc7NkJBQ3JDOzRCQUNEO2dDQUNFLElBQUksRUFBRSxjQUFjO2dDQUNwQixJQUFJLEVBQUUsSUFBSSxDQUFDLE1BQU0sQ0FBQyxNQUFNLENBQUMsV0FBVzs2QkFDckM7NEJBQ0Q7Z0NBQ0UsSUFBSSxFQUFFLGtCQUFrQjtnQ0FDeEIsSUFBSSxFQUFFLElBQUksQ0FBQyxNQUFNLENBQUMsTUFBTSxDQUFDLFdBQVc7NkJBQ3JDOzRCQUNEO2dDQUNFLElBQUksRUFBRSxVQUFVO2dDQUNoQixJQUFJLEVBQUUsSUFBSSxDQUFDLE1BQU0sQ0FBQyxNQUFNLENBQUMsV0FBVzs2QkFDckM7NEJBQ0Q7Z0NBQ0UsSUFBSSxFQUFFLGFBQWE7Z0NBQ25CLElBQUksRUFBRSxJQUFJLENBQUMsTUFBTSxDQUFDLE1BQU0sQ0FBQyxXQUFXOzZCQUNyQzs0QkFDRDtnQ0FDRSxJQUFJLEVBQUUsaUJBQWlCO2dDQUN2QixJQUFJLEVBQUUsSUFBSSxDQUFDLE1BQU0sQ0FBQyxNQUFNLENBQUMsV0FBVzs2QkFDckM7NEJBQ0Q7Z0NBQ0UsSUFBSSxFQUFFLHVCQUF1QjtnQ0FDN0IsSUFBSSxFQUFFLElBQUksQ0FBQyxNQUFNLENBQUMsTUFBTSxDQUFDLFdBQVc7NkJBQ3JDOzRCQUNEO2dDQUNFLElBQUksRUFBRSx1QkFBdUI7Z0NBQzdCLElBQUksRUFBRSxJQUFJLENBQUMsTUFBTSxDQUFDLE1BQU0sQ0FBQyxXQUFXOzZCQUNyQzs0QkFDRDtnQ0FDRSxJQUFJLEVBQUUsa0JBQWtCO2dDQUN4QixJQUFJLEVBQUUsSUFBSSxDQUFDLE1BQU0sQ0FBQyxNQUFNLENBQUMsV0FBVzs2QkFDckM7NEJBQ0Q7Z0NBQ0UsSUFBSSxFQUFFLGNBQWM7Z0NBQ3BCLElBQUksRUFBRSxJQUFJLENBQUMsTUFBTSxDQUFDLE1BQU0sQ0FBQyxXQUFXOzZCQUNyQzs0QkFDRDtnQ0FDRSxJQUFJLEVBQUUsY0FBYztnQ0FDcEIsSUFBSSxFQUFFLElBQUksQ0FBQyxNQUFNLENBQUMsTUFBTSxDQUFDLFdBQVc7NkJBQ3JDO3lCQUNGO3dCQUNELFVBQVUsRUFBRSxLQUFLO3dCQUNqQixXQUFXLEVBQUUsMENBQTBDO3dCQUN2RCxRQUFRLEVBQ04sT0FBTyxHQUFHLGVBQWUsQ0FBQyxVQUFVLEdBQUcsdUJBQXVCO3dCQUNoRSxZQUFZLEVBQ1YsNERBQTREO3dCQUM5RCxTQUFTLEVBQUU7NEJBQ1Qsb0JBQW9CLEVBQUUsMENBQTBDOzRCQUNoRSxVQUFVLEVBQUU7Z0NBQ1Ysc0JBQXNCLEVBQUUsR0FBRztnQ0FDM0IsYUFBYSxFQUNYLHVTQUF1Uzs2QkFDMVM7eUJBQ0Y7d0JBQ0Qsc0JBQXNCLEVBQUUsS0FBSztxQkFDOUI7b0JBQ0QsU0FBUyxFQUFFLGdCQUFnQjtvQkFDM0IsYUFBYSxFQUFFO3dCQUNiOzRCQUNFLElBQUksRUFBRSxNQUFNOzRCQUNaLElBQUksRUFBRSxJQUFJLENBQUMsTUFBTSxDQUFDLE9BQU8sQ0FBQyxXQUFXO3lCQUN0Qzt3QkFDRDs0QkFDRSxJQUFJLEVBQUUsT0FBTzs0QkFDYixJQUFJLEVBQUUsSUFBSSxDQUFDLE1BQU0sQ0FBQyxPQUFPLENBQUMsV0FBVzt5QkFDdEM7d0JBQ0Q7NEJBQ0UsSUFBSSxFQUFFLEtBQUs7NEJBQ1gsSUFBSSxFQUFFLElBQUksQ0FBQyxNQUFNLENBQUMsT0FBTyxDQUFDLFdBQVc7eUJBQ3RDO3dCQUNEOzRCQUNFLElBQUksRUFBRSxNQUFNOzRCQUNaLElBQUksRUFBRSxJQUFJLENBQUMsTUFBTSxDQUFDLE9BQU8sQ0FBQyxXQUFXO3lCQUN0QztxQkFDRjtpQkFDRjthQUNGLENBQUMsQ0FBQztTQUNKO1FBRUQsUUFBUTtRQUNSLE1BQU0sc0JBQXNCLFNBQUcsS0FBSyxDQUFDLG9CQUFvQixtQ0FBSSxFQUFFLENBQUM7UUFDaEUsTUFBTSxnQ0FBZ0MsR0FBRyxJQUFJLE1BQU0sQ0FBQyxZQUFZLENBQzlELElBQUksRUFDSixxQ0FBcUMsRUFDckM7WUFDRSxJQUFJLEVBQUUsc0NBQXNDLHNCQUFzQixFQUFFO1lBQ3BFLFdBQVcsRUFDVCxvR0FBb0c7WUFDdEcsS0FBSyxFQUFFLFNBQVM7WUFDaEIscUJBQXFCLEVBQUUsSUFBSTtZQUMzQixzQkFBc0IsRUFBRTtnQkFDdEIsK0JBQStCLEVBQUUsSUFBSTtnQkFDckMsNkJBQTZCLEVBQUUsS0FBSzthQUNyQztTQUNGLENBQ0YsQ0FBQztRQUVGLE1BQU0sMEJBQTBCLEdBQUcsSUFBSSxNQUFNLENBQUMsWUFBWSxDQUN4RCxJQUFJLEVBQ0osNEJBQTRCLEVBQzVCO1lBQ0UsSUFBSSxFQUFFLDZCQUE2QixzQkFBc0IsRUFBRTtZQUMzRCxXQUFXLEVBQ1Qsb0ZBQW9GO1lBQ3RGLEtBQUssRUFBRSxTQUFTO1lBQ2hCLHFCQUFxQixFQUFFLElBQUk7WUFDM0Isc0JBQXNCLEVBQUU7Z0JBQ3RCLCtCQUErQixFQUFFLElBQUk7YUFDdEM7U0FDRixDQUNGLENBQUM7UUFFRixNQUFNLG1DQUFtQyxHQUFHLElBQUksTUFBTSxDQUFDLFlBQVksQ0FDakUsSUFBSSxFQUNKLHFDQUFxQyxFQUNyQztZQUNFLElBQUksRUFBRSxzQ0FBc0Msc0JBQXNCLEVBQUU7WUFDcEUsV0FBVyxFQUNULHFIQUFxSDtZQUN2SCxLQUFLLEVBQUUsU0FBUztZQUNoQixxQkFBcUIsRUFBRSxJQUFJO1lBQzNCLHNCQUFzQixFQUFFO2dCQUN0QiwrQkFBK0IsRUFBRSxJQUFJO2FBQ3RDO1NBQ0YsQ0FDRixDQUFDO1FBRUYsc0JBQXNCO1FBQ3RCLElBQUksVUFBVSxDQUFDLFlBQVksQ0FBQyxJQUFJLEVBQUUscUJBQXFCLEVBQUU7WUFDdkQsYUFBYSxFQUFFLHVCQUF1QjtZQUN0QyxhQUFhLEVBQUUsSUFBSSxDQUFDLFNBQVMsQ0FBQztnQkFDNUIsT0FBTyxFQUFFO29CQUNQO3dCQUNFLElBQUksRUFBRSxRQUFRO3dCQUNkLENBQUMsRUFBRSxDQUFDO3dCQUNKLENBQUMsRUFBRSxDQUFDO3dCQUNKLEtBQUssRUFBRSxFQUFFO3dCQUNULE1BQU0sRUFBRSxFQUFFO3dCQUNWLFVBQVUsRUFBRTs0QkFDVixJQUFJLEVBQUUsWUFBWTs0QkFDbEIsT0FBTyxFQUFFLEtBQUs7NEJBQ2QsT0FBTyxFQUFFO2dDQUNQO29DQUNFLEtBQUs7b0NBQ0wsaUJBQWlCO29DQUNqQixRQUFRO29DQUNSLHFCQUFxQjtvQ0FDckIsTUFBTTtvQ0FDTixLQUFLO2lDQUNOO2dDQUNEO29DQUNFLEtBQUs7b0NBQ0wsaUJBQWlCO29DQUNqQixRQUFRO29DQUNSLHFCQUFxQjtvQ0FDckIsTUFBTTtvQ0FDTixLQUFLO2lDQUNOOzZCQUNGOzRCQUNELE1BQU0sRUFBRSxXQUFXOzRCQUNuQixNQUFNLEVBQUUsR0FBRzt5QkFDWjtxQkFDRjtpQkFDRjthQUNGLENBQUM7U0FDSCxDQUFDLENBQUM7UUFFSCwwQ0FBMEM7UUFDMUMsTUFBTSx3QkFBd0IsR0FBRztZQUMvQixZQUFZLEVBQUUsMEJBQTBCO1lBQ3hDLHNCQUFzQixFQUFFLHNCQUFzQixDQUFDLFlBQVk7WUFDM0QsZUFBZSxFQUFFLFlBQVksQ0FBQyxVQUFVO1lBQ3hDLHNCQUFzQixFQUFFLHNCQUFzQixDQUFDLFNBQVM7WUFDeEQsZUFBZSxFQUFFLDBCQUEwQixDQUFDLElBQUk7U0FDakQsQ0FBQztRQUVGLElBQUksTUFBTSxDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUUsOEJBQThCLEVBQUU7WUFDcEQsV0FBVyxFQUFFLDhDQUE4QztZQUMzRCxRQUFRLEVBQUUsTUFBTSxDQUFDLFFBQVEsQ0FBQyxVQUFVLENBQUMsaUJBQWlCLENBQUM7WUFDdkQsT0FBTyxFQUFFO2dCQUNQLElBQUksT0FBTyxDQUFDLGNBQWMsQ0FBQyxlQUFlLEVBQUU7b0JBQzFDLEtBQUssRUFBRSxNQUFNLENBQUMsZUFBZSxDQUFDLFVBQVUsQ0FBQyx3QkFBd0IsQ0FBQztpQkFDbkUsQ0FBQzthQUNIO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsTUFBTSxpQ0FBaUMsR0FBRztZQUN4QyxZQUFZLEVBQUUsMEJBQTBCO1lBQ3hDLHNCQUFzQixFQUFFLHNCQUFzQixDQUFDLFlBQVk7WUFDM0QsZUFBZSxFQUFFLGVBQWUsQ0FBQyxVQUFVO1lBQzNDLHNCQUFzQixFQUFFLGlCQUFpQjtZQUN6QyxlQUFlLEVBQUUsbUNBQW1DLENBQUMsSUFBSTtTQUMxRCxDQUFDO1FBRUYsSUFBSSxNQUFNLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSw4QkFBOEIsRUFBRTtZQUNwRCxXQUFXLEVBQUUsOENBQThDO1lBQzNELFFBQVEsRUFBRSxNQUFNLENBQUMsUUFBUSxDQUFDLFVBQVUsQ0FBQyxpQkFBaUIsQ0FBQztZQUN2RCxPQUFPLEVBQUU7Z0JBQ1AsSUFBSSxPQUFPLENBQUMsY0FBYyxDQUFDLGVBQWUsRUFBRTtvQkFDMUMsS0FBSyxFQUFFLE1BQU0sQ0FBQyxlQUFlLENBQUMsVUFBVSxDQUN0QyxpQ0FBaUMsQ0FDbEM7aUJBQ0YsQ0FBQzthQUNIO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsTUFBTSx5QkFBeUIsR0FBRyxJQUFJLE1BQU0sQ0FBQyxtQkFBbUIsQ0FDOUQsSUFBSSxFQUNKLDZCQUE2QixFQUM3QjtZQUNFLFdBQVcsRUFDVCxpSEFBaUg7WUFDbkgsSUFBSSxFQUFFLE1BQU0sQ0FBQyxlQUFlLENBQUMsY0FBYyxDQUN6QyxJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsRUFBRSxzQkFBc0IsQ0FBQyxFQUM1QztnQkFDRSxHQUFHLEVBQUUsQ0FBQyxzQ0FBc0MsQ0FBQzthQUM5QyxDQUNGO1lBQ0QsVUFBVSxFQUFFLEdBQUc7WUFDZixPQUFPLEVBQUUsR0FBRyxDQUFDLFFBQVEsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDO1lBQ2hDLFdBQVcsRUFBRSxHQUFHLENBQUMsUUFBUSxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUM7WUFDbEMsYUFBYSxFQUFFLENBQUM7WUFDaEIsV0FBVyxFQUFFO2dCQUNYLFNBQVMsRUFBRSxRQUFRO2FBQ3BCO1NBQ0YsQ0FDRixDQUFDO1FBRUYsTUFBTSw0QkFBNEIsR0FBRztZQUNuQyxZQUFZLEVBQUUscUNBQXFDO1lBQ25ELHNCQUFzQixFQUFFLHNCQUFzQixDQUFDLFlBQVk7WUFDM0QsZUFBZSxFQUFFLGVBQWUsQ0FBQyxVQUFVO1lBQzNDLHNCQUFzQixFQUFFLGlCQUFpQjtZQUN6QyxzQkFBc0IsRUFBRSxzQkFBc0IsQ0FBQyxTQUFTO1lBQ3hELFlBQVksRUFBRSxZQUFZLENBQUMsVUFBVTtZQUNyQyxlQUFlLEVBQUUsZ0NBQWdDLENBQUMsSUFBSTtTQUN2RCxDQUFDO1FBRUYsSUFBSSxNQUFNLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSwwQkFBMEIsRUFBRTtZQUNoRCxXQUFXLEVBQUUsdURBQXVEO1lBQ3BFLFFBQVEsRUFBRSxNQUFNLENBQUMsUUFBUSxDQUFDLFVBQVUsQ0FBQyxjQUFjLENBQUM7WUFDcEQsT0FBTyxFQUFFO2dCQUNQLElBQUksT0FBTyxDQUFDLGNBQWMsQ0FBQyx5QkFBeUIsRUFBRTtvQkFDcEQsS0FBSyxFQUFFLE1BQU0sQ0FBQyxlQUFlLENBQUMsVUFBVSxDQUN0Qyw0QkFBNEIsQ0FDN0I7aUJBQ0YsQ0FBQzthQUNIO1NBQ0YsQ0FBQyxDQUFDO1FBRUgseUJBQXlCLENBQUMsZUFBZSxDQUN2QyxJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUM7WUFDdEIsTUFBTSxFQUFFLEdBQUcsQ0FBQyxNQUFNLENBQUMsS0FBSztZQUN4QixPQUFPLEVBQUU7Z0JBQ1AseUJBQXlCO2dCQUN6QixzQkFBc0I7Z0JBQ3RCLGNBQWM7Z0JBQ2QsZUFBZTtnQkFDZiwrQkFBK0I7Z0JBQy9CLGNBQWM7Z0JBQ2QsZUFBZTtnQkFDZixrQkFBa0I7Z0JBQ2xCLHFCQUFxQjtnQkFDckIscUJBQXFCO2dCQUNyQiwyQkFBMkI7Z0JBQzNCLHFCQUFxQjtnQkFDckIsc0JBQXNCO2dCQUN0QixtQkFBbUI7YUFDcEI7WUFDRCxTQUFTLEVBQUUsQ0FBQyxHQUFHLENBQUM7U0FDakIsQ0FBQyxDQUNILENBQUM7UUFFRix5QkFBeUIsQ0FBQyxlQUFlLENBQ3ZDLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQztZQUN0QixNQUFNLEVBQUUsR0FBRyxDQUFDLE1BQU0sQ0FBQyxLQUFLO1lBQ3hCLE9BQU8sRUFBRSxDQUFDLDRCQUE0QixDQUFDO1lBQ3ZDLFNBQVMsRUFBRSxDQUFDLEdBQUcsQ0FBQztTQUNqQixDQUFDLENBQ0gsQ0FBQztRQUVGLGVBQWUsQ0FBQyxlQUFlLENBQzdCLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQztZQUN0QixNQUFNLEVBQUUsR0FBRyxDQUFDLE1BQU0sQ0FBQyxLQUFLO1lBQ3hCLE9BQU8sRUFBRSxDQUFDLGVBQWUsRUFBRSxvQkFBb0IsQ0FBQztZQUNoRCxTQUFTLEVBQUU7Z0JBQ1Qsc0JBQXNCLENBQUMsVUFBVTtnQkFDakMsc0JBQXNCLENBQUMsV0FBVztnQkFDbEMsT0FBTyxHQUFHLENBQUMsR0FBRyxDQUFDLFNBQVMsU0FBUyxHQUFHLENBQUMsR0FBRyxDQUFDLE1BQU0sSUFBSSxHQUFHLENBQUMsR0FBRyxDQUFDLFVBQVUsVUFBVSxzQkFBc0IsQ0FBQyxZQUFZLGtCQUFrQjtnQkFDcEksT0FBTyxHQUFHLENBQUMsR0FBRyxDQUFDLFNBQVMsU0FBUyxHQUFHLENBQUMsR0FBRyxDQUFDLE1BQU0sSUFBSSxHQUFHLENBQUMsR0FBRyxDQUFDLFVBQVUsVUFBVSxzQkFBc0IsQ0FBQyxZQUFZLGtCQUFrQjthQUNySTtTQUNGLENBQUMsQ0FDSCxDQUFDO1FBRUYsZUFBZSxDQUFDLGVBQWUsQ0FDN0IsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDO1lBQ3RCLE1BQU0sRUFBRSxHQUFHLENBQUMsTUFBTSxDQUFDLEtBQUs7WUFDeEIsT0FBTyxFQUFFLENBQUMsZ0JBQWdCLEVBQUUsbUJBQW1CLENBQUM7WUFDaEQsU0FBUyxFQUFFO2dCQUNULGdCQUFnQixDQUFDLE9BQU87Z0JBQ3hCLGdCQUFnQixDQUFDLE9BQU87Z0JBQ3hCLHFCQUFxQixDQUFDLE9BQU87Z0JBQzdCLHFCQUFxQixDQUFDLE9BQU87YUFDOUI7U0FDRixDQUFDLENBQ0gsQ0FBQztRQUVGLGVBQWUsQ0FBQyxlQUFlLENBQzdCLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQztZQUN0QixNQUFNLEVBQUUsR0FBRyxDQUFDLE1BQU0sQ0FBQyxLQUFLO1lBQ3hCLE9BQU8sRUFBRSxDQUFDLHNCQUFzQixFQUFFLDRCQUE0QixDQUFDO1lBQy9ELFNBQVMsRUFBRTtnQkFDVCxPQUFPLEdBQUcsQ0FBQyxHQUFHLENBQUMsU0FBUyxXQUFXLEdBQUcsQ0FBQyxHQUFHLENBQUMsTUFBTSxJQUFJLEdBQUcsQ0FBQyxHQUFHLENBQUMsVUFBVSxpQkFBaUI7YUFDekY7U0FDRixDQUFDLENBQ0gsQ0FBQztRQUVGLGVBQWUsQ0FBQyxlQUFlLENBQzdCLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQztZQUN0QixNQUFNLEVBQUUsR0FBRyxDQUFDLE1BQU0sQ0FBQyxLQUFLO1lBQ3hCLE9BQU8sRUFBRSxDQUFDLGdDQUFnQyxDQUFDO1lBQzNDLFNBQVMsRUFBRSxDQUFDLEdBQUcsQ0FBQztTQUNqQixDQUFDLENBQ0gsQ0FBQztRQUVGLGVBQWUsQ0FBQyxlQUFlLENBQzdCLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQztZQUN0QixNQUFNLEVBQUUsR0FBRyxDQUFDLE1BQU0sQ0FBQyxLQUFLO1lBQ3hCLE9BQU8sRUFBRTtnQkFDUCxxQkFBcUI7Z0JBQ3JCLHNCQUFzQjtnQkFDdEIsbUJBQW1CO2FBQ3BCO1lBQ0QsU0FBUyxFQUFFO2dCQUNULE9BQU8sR0FBRyxDQUFDLEdBQUcsQ0FBQyxTQUFTLFdBQVcsR0FBRyxDQUFDLEdBQUcsQ0FBQyxNQUFNLElBQUksR0FBRyxDQUFDLEdBQUcsQ0FBQyxVQUFVLG9DQUFvQzthQUM1RztTQUNGLENBQUMsQ0FDSCxDQUFDO1FBRUYsZUFBZSxDQUFDLGVBQWUsQ0FDN0IsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDO1lBQ3RCLE1BQU0sRUFBRSxHQUFHLENBQUMsTUFBTSxDQUFDLEtBQUs7WUFDeEIsT0FBTyxFQUFFO2dCQUNQLHNCQUFzQjtnQkFDdEIsY0FBYztnQkFDZCxlQUFlO2dCQUNmLCtCQUErQjtnQkFDL0IsNkJBQTZCO2dCQUM3Qix5QkFBeUI7Z0JBQ3pCLGlCQUFpQjtnQkFDakIsY0FBYzthQUNmO1lBQ0QsU0FBUyxFQUFFO2dCQUNULGVBQWUsQ0FBQyxTQUFTLEdBQUcsbUJBQW1CO2dCQUMvQyxlQUFlLENBQUMsU0FBUyxHQUFHLElBQUk7Z0JBQ2hDLGVBQWUsQ0FBQyxTQUFTO2dCQUN6QixZQUFZLENBQUMsU0FBUyxHQUFHLG1CQUFtQjtnQkFDNUMsWUFBWSxDQUFDLFNBQVMsR0FBRyxJQUFJO2dCQUM3QixZQUFZLENBQUMsU0FBUzthQUN2QjtTQUNGLENBQUMsQ0FDSCxDQUFDO1FBRUYsS0FBSztRQUNMLGVBQWUsQ0FBQyxlQUFlLENBQzdCLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQztZQUN0QixNQUFNLEVBQUUsR0FBRyxDQUFDLE1BQU0sQ0FBQyxLQUFLO1lBQ3hCLE9BQU8sRUFBRTtnQkFDUCxzQkFBc0I7Z0JBQ3RCLGNBQWM7Z0JBQ2QsZUFBZTtnQkFDZiwrQkFBK0I7Z0JBQy9CLDZCQUE2QjtnQkFDN0IseUJBQXlCO2dCQUN6QixpQkFBaUI7Z0JBQ2pCLGNBQWM7YUFDZjtZQUNELFNBQVMsRUFBRTtnQkFDVCxZQUFZLENBQUMsU0FBUyxHQUFHLG1CQUFtQjtnQkFDNUMsZUFBZSxDQUFDLFNBQVMsR0FBRyxtQkFBbUI7Z0JBQy9DLFlBQVksQ0FBQyxTQUFTO2dCQUN0QixlQUFlLENBQUMsU0FBUzthQUMxQjtTQUNGLENBQUMsQ0FDSCxDQUFDO1FBRUYsZUFBZSxDQUFDLGVBQWUsQ0FDN0IsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDO1lBQ3RCLE1BQU0sRUFBRSxHQUFHLENBQUMsTUFBTSxDQUFDLEtBQUs7WUFDeEIsT0FBTyxFQUFFLENBQUMsY0FBYyxDQUFDO1lBQ3pCLFNBQVMsRUFBRTtnQkFDVCxZQUFZLENBQUMsU0FBUyxHQUFHLElBQUk7Z0JBQzdCLGVBQWUsQ0FBQyxTQUFTLEdBQUcsSUFBSTthQUNqQztTQUNGLENBQUMsQ0FDSCxDQUFDO1FBRUYsZUFBZSxDQUFDLGVBQWUsQ0FDN0IsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDO1lBQ3RCLE1BQU0sRUFBRSxHQUFHLENBQUMsTUFBTSxDQUFDLEtBQUs7WUFDeEIsT0FBTyxFQUFFLENBQUMsY0FBYyxDQUFDO1lBQ3pCLFNBQVMsRUFBRTtnQkFDVCxZQUFZLENBQUMsU0FBUyxHQUFHLEdBQUcsR0FBRyxLQUFLLENBQUMsU0FBUyxHQUFHLG1CQUFtQjtnQkFDcEUsWUFBWSxDQUFDLFNBQVMsR0FBRyxHQUFHLEdBQUcsS0FBSyxDQUFDLFNBQVMsR0FBRyxvQkFBb0I7Z0JBQ3JFLGVBQWUsQ0FBQyxTQUFTO29CQUN2QixHQUFHO29CQUNILEtBQUssQ0FBQyxTQUFTO29CQUNmLG1CQUFtQjtnQkFDckIsZUFBZSxDQUFDLFNBQVM7b0JBQ3ZCLEdBQUc7b0JBQ0gsS0FBSyxDQUFDLFNBQVM7b0JBQ2Ysb0JBQW9CO2FBQ3ZCO1NBQ0YsQ0FBQyxDQUNILENBQUM7UUFFRix3QkFBd0IsQ0FBQyxlQUFlLENBQ3RDLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQztZQUN0QixNQUFNLEVBQUUsR0FBRyxDQUFDLE1BQU0sQ0FBQyxLQUFLO1lBQ3hCLE9BQU8sRUFBRTtnQkFDUCxxQkFBcUI7Z0JBQ3JCLHNCQUFzQjtnQkFDdEIsbUJBQW1CO2FBQ3BCO1lBQ0QsU0FBUyxFQUFFO2dCQUNULE9BQU8sR0FBRyxDQUFDLEdBQUcsQ0FBQyxTQUFTLFdBQVcsR0FBRyxDQUFDLEdBQUcsQ0FBQyxNQUFNLElBQUksR0FBRyxDQUFDLEdBQUcsQ0FBQyxVQUFVLGlEQUFpRDthQUN6SDtTQUNGLENBQUMsQ0FDSCxDQUFDO1FBRUYsd0JBQXdCLENBQUMsZUFBZSxDQUN0QyxJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUM7WUFDdEIsTUFBTSxFQUFFLEdBQUcsQ0FBQyxNQUFNLENBQUMsS0FBSztZQUN4QixPQUFPLEVBQUUsQ0FBQyxjQUFjLEVBQUUsaUJBQWlCLEVBQUUsY0FBYyxDQUFDO1lBQzVELFNBQVMsRUFBRSxDQUFDLGVBQWUsQ0FBQyxTQUFTLEdBQUcsSUFBSSxDQUFDO1NBQzlDLENBQUMsQ0FDSCxDQUFDO1FBRUYsb0JBQW9CLENBQUMsZUFBZSxDQUNsQyxJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUM7WUFDdEIsTUFBTSxFQUFFLEdBQUcsQ0FBQyxNQUFNLENBQUMsS0FBSztZQUN4QixPQUFPLEVBQUUsQ0FBQywrQkFBK0IsQ0FBQztZQUMxQyxTQUFTLEVBQUU7Z0JBQ1QsT0FBTyxHQUFHLENBQUMsR0FBRyxDQUFDLFNBQVMsbUJBQW1CLEdBQUcsQ0FBQyxHQUFHLENBQUMsTUFBTSxJQUFJLEdBQUcsQ0FBQyxHQUFHLENBQUMsVUFBVSxVQUFVLEdBQUcsQ0FBQyxLQUFLLENBQUMsSUFBSSxJQUFJO2FBQzVHO1NBQ0YsQ0FBQyxDQUNILENBQUM7UUFFRixvQkFBb0IsQ0FBQyxlQUFlLENBQ2xDLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQztZQUN0QixNQUFNLEVBQUUsR0FBRyxDQUFDLE1BQU0sQ0FBQyxLQUFLO1lBQ3hCLE9BQU8sRUFBRSxDQUFDLHVCQUF1QixDQUFDO1lBQ2xDLFNBQVMsRUFBRTtnQkFDVCxPQUFPLEdBQUcsQ0FBQyxHQUFHLENBQUMsU0FBUyxXQUFXLEdBQUcsQ0FBQyxHQUFHLENBQUMsTUFBTSxJQUFJLEdBQUcsQ0FBQyxHQUFHLENBQUMsVUFBVSxhQUFhLEtBQUssQ0FBQyxTQUFTLEdBQUc7YUFDdkc7U0FDRixDQUFDLENBQ0gsQ0FBQztRQUVGLG9CQUFvQixDQUFDLGVBQWUsQ0FDbEMsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDO1lBQ3RCLE1BQU0sRUFBRSxHQUFHLENBQUMsTUFBTSxDQUFDLEtBQUs7WUFDeEIsT0FBTyxFQUFFO2dCQUNQLGdCQUFnQjtnQkFDaEIsbUJBQW1CO2dCQUNuQixpQkFBaUI7Z0JBQ2pCLG9CQUFvQjtnQkFDcEIsa0NBQWtDO2dCQUNsQywrQkFBK0I7YUFDaEM7WUFDRCxTQUFTLEVBQUUsQ0FBQyxNQUFNLENBQUMsT0FBTyxDQUFDO1NBQzVCLENBQUMsQ0FDSCxDQUFDO1FBRUYsb0JBQW9CLENBQUMsZUFBZSxDQUNsQyxJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUM7WUFDdEIsTUFBTSxFQUFFLEdBQUcsQ0FBQyxNQUFNLENBQUMsS0FBSztZQUN4QixPQUFPLEVBQUUsQ0FBQyxnQkFBZ0IsRUFBRSxtQkFBbUIsQ0FBQztZQUNoRCxTQUFTLEVBQUU7Z0JBQ1QsT0FBTyxHQUFHLENBQUMsR0FBRyxDQUFDLFNBQVMsU0FBUyxHQUFHLENBQUMsR0FBRyxDQUFDLE1BQU0sSUFBSSxHQUFHLENBQUMsR0FBRyxDQUFDLFVBQVUsbUJBQW1CLEtBQUssQ0FBQyxTQUFTLEdBQUc7Z0JBQzFHLE9BQU8sR0FBRyxDQUFDLEdBQUcsQ0FBQyxTQUFTLFNBQVMsR0FBRyxDQUFDLEdBQUcsQ0FBQyxNQUFNLElBQUksR0FBRyxDQUFDLEdBQUcsQ0FBQyxVQUFVLGlCQUFpQixLQUFLLENBQUMsU0FBUyxHQUFHO2FBQ3pHO1NBQ0YsQ0FBQyxDQUNILENBQUM7UUFFRixvQkFBb0IsQ0FBQyxlQUFlLENBQ2xDLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQztZQUN0QixNQUFNLEVBQUUsR0FBRyxDQUFDLE1BQU0sQ0FBQyxLQUFLO1lBQ3hCLE9BQU8sRUFBRTtnQkFDUCxxQkFBcUI7Z0JBQ3JCLHNCQUFzQjtnQkFDdEIsbUJBQW1CO2FBQ3BCO1lBQ0QsU0FBUyxFQUFFO2dCQUNULE9BQU8sR0FBRyxDQUFDLEdBQUcsQ0FBQyxTQUFTLFNBQVMsR0FBRyxDQUFDLEdBQUcsQ0FBQyxNQUFNLElBQUksR0FBRyxDQUFDLEdBQUcsQ0FBQyxVQUFVLHlDQUF5QzthQUMvRztTQUNGLENBQUMsQ0FDSCxDQUFDO1FBRUYsb0JBQW9CLENBQUMsZUFBZSxDQUNsQyxJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUM7WUFDdEIsTUFBTSxFQUFFLEdBQUcsQ0FBQyxNQUFNLENBQUMsS0FBSztZQUN4QixPQUFPLEVBQUU7Z0JBQ1Asc0JBQXNCO2dCQUN0QixlQUFlO2dCQUNmLGlCQUFpQjtnQkFDakIsMEJBQTBCO2dCQUMxQiwwQkFBMEI7Z0JBQzFCLCtCQUErQjtnQkFDL0IsK0JBQStCO2dCQUMvQixlQUFlO2dCQUNmLGVBQWU7Z0JBQ2Ysa0JBQWtCO2FBQ25CO1lBQ0QsU0FBUyxFQUFFO2dCQUNULGVBQWUsQ0FBQyxTQUFTO2dCQUN6QixHQUFHLGVBQWUsQ0FBQyxTQUFTLElBQUk7YUFDakM7U0FDRixDQUFDLENBQ0gsQ0FBQztRQUVGLG9CQUFvQixDQUFDLGVBQWUsQ0FDbEMsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDO1lBQ3RCLE1BQU0sRUFBRSxHQUFHLENBQUMsTUFBTSxDQUFDLEtBQUs7WUFDeEIsT0FBTyxFQUFFO2dCQUNQLGlCQUFpQjtnQkFDakIsMEJBQTBCO2dCQUMxQiwwQkFBMEI7Z0JBQzFCLGVBQWU7Z0JBQ2YsZUFBZTtnQkFDZixrQkFBa0I7YUFDbkI7WUFDRCxTQUFTLEVBQUUsQ0FBQyxZQUFZLENBQUMsU0FBUyxFQUFFLEdBQUcsWUFBWSxDQUFDLFNBQVMsSUFBSSxDQUFDO1NBQ25FLENBQUMsQ0FDSCxDQUFDO1FBRUYsb0JBQW9CLENBQUMsZUFBZSxDQUNsQyxJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUM7WUFDdEIsTUFBTSxFQUFFLEdBQUcsQ0FBQyxNQUFNLENBQUMsS0FBSztZQUN4QixPQUFPLEVBQUUsQ0FBQyw2QkFBNkIsQ0FBQztZQUN4QyxTQUFTLEVBQUU7Z0JBQ1Qsd0ZBQXdGO2FBQ3pGO1NBQ0YsQ0FBQyxDQUNILENBQUM7UUFFRixZQUFZLENBQUMsb0JBQW9CLENBQy9CLEVBQUUsQ0FBQyxTQUFTLENBQUMsY0FBYyxFQUMzQixJQUFJLEdBQUcsQ0FBQyxpQkFBaUIsQ0FBQyxlQUFlLENBQUMsRUFDMUM7WUFDRSxNQUFNLEVBQUUsaUJBQWlCO1lBQ3pCLE1BQU0sRUFBRSxLQUFLO1NBQ2QsQ0FDRixDQUFDO1FBRUYsZUFBZSxDQUFDLG9CQUFvQixDQUNsQyxFQUFFLENBQUMsU0FBUyxDQUFDLGNBQWMsRUFDM0IsSUFBSSxHQUFHLENBQUMsaUJBQWlCLENBQUMsd0JBQXdCLENBQUMsRUFDbkQ7WUFDRSxNQUFNLEVBQUUsVUFBVTtZQUNsQixNQUFNLEVBQUUsSUFBSTtTQUNiLENBQ0YsQ0FBQztRQUVGLGVBQWUsQ0FBQyxvQkFBb0IsQ0FDbEMsRUFBRSxDQUFDLFNBQVMsQ0FBQyxjQUFjLEVBQzNCLElBQUksR0FBRyxDQUFDLGlCQUFpQixDQUFDLGVBQWUsQ0FBQyxFQUMxQztZQUNFLE1BQU0sRUFBRSxpQkFBaUI7WUFDekIsTUFBTSxFQUFFLEtBQUs7U0FDZCxDQUNGLENBQUM7UUFFRixNQUFNLHNCQUFzQixHQUFHLElBQUksRUFBRSxDQUFDLFFBQVEsQ0FDNUMsSUFBSSxFQUNKLHdCQUF3QixFQUN4QjtZQUNFLGNBQWMsRUFBRSxvQkFBb0I7U0FDckMsQ0FDRixDQUFDO1FBRUYsSUFBSSxHQUFHLENBQUMsY0FBYyxDQUFDLElBQUksRUFBRSxxQkFBcUIsRUFBRTtZQUNsRCxZQUFZLEVBQUUsc0JBQXNCLENBQUMsWUFBWTtZQUNqRCxZQUFZLEVBQUUsNkJBQTZCO1lBQzNDLFVBQVUsRUFBRTtnQkFDVixZQUFZLEVBQUUsTUFBTSxDQUFDLE9BQU87Z0JBQzVCLGlCQUFpQixFQUFFLGNBQWMsQ0FBQyxPQUFPO2FBQzFDO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsSUFBSSxHQUFHLENBQUMsY0FBYyxDQUFDLElBQUksRUFBRSw2QkFBNkIsRUFBRTtZQUMxRCxZQUFZLEVBQUUsc0JBQXNCLENBQUMsWUFBWTtZQUNqRCxZQUFZLEVBQUUscUNBQXFDO1lBQ25ELFVBQVUsRUFBRTtnQkFDVixNQUFNLEVBQUUsR0FBRyxDQUFDLEdBQUcsQ0FBQyxNQUFNO2dCQUN0QixrQkFBa0IsRUFBRSxlQUFlLENBQUMsVUFBVTtnQkFDOUMsU0FBUyxFQUFFLGVBQWUsQ0FBQyxXQUFXO2dCQUN0Qyw2QkFBNkIsRUFBRSxLQUFLO2dCQUNwQyxzQkFBc0IsRUFBRSx3QkFBd0IsQ0FBQyxZQUFZO2FBQzlEO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsSUFBSSxHQUFHLENBQUMsY0FBYyxDQUFDLElBQUksRUFBRSx1QkFBdUIsRUFBRTtZQUNwRCxZQUFZLEVBQUUsc0JBQXNCLENBQUMsWUFBWTtZQUNqRCxZQUFZLEVBQUUsK0JBQStCO1lBQzdDLFVBQVUsRUFBRTtnQkFDVixNQUFNLEVBQUUsR0FBRyxDQUFDLEdBQUcsQ0FBQyxNQUFNO2dCQUN0QixZQUFZLEVBQUUsWUFBWSxDQUFDLFVBQVU7Z0JBQ3JDLFNBQVMsRUFBRSxlQUFlLENBQUMsV0FBVztnQkFDdEMsd0JBQXdCLEVBQUUsS0FBSzthQUNoQztTQUNGLENBQUMsQ0FBQztRQUVILElBQUksR0FBRyxDQUFDLGNBQWMsQ0FBQyxJQUFJLEVBQUUsaUJBQWlCLEVBQUU7WUFDOUMsWUFBWSxFQUFFLHNCQUFzQixDQUFDLFlBQVk7WUFDakQsWUFBWSxFQUFFLHlCQUF5QjtZQUN2QyxVQUFVLEVBQUU7Z0JBQ1YsbUJBQW1CLEVBQUUsZ0JBQWdCLENBQUMsTUFBTTtnQkFDNUMsdUJBQXVCLEVBQUUsZ0JBQWdCLENBQUMsSUFBSTthQUMvQztTQUNGLENBQUMsQ0FBQztRQUVILElBQUksR0FBRyxDQUFDLGNBQWMsQ0FBQyxJQUFJLEVBQUUsOEJBQThCLEVBQUU7WUFDM0QsWUFBWSxFQUFFLHNCQUFzQixDQUFDLFlBQVk7WUFDakQsWUFBWSxFQUFFLHNDQUFzQztZQUNwRCxVQUFVLEVBQUU7Z0JBQ1YsU0FBUyxFQUFFLEtBQUssQ0FBQyxTQUFTO2dCQUMxQixjQUFjLEVBQUUsY0FBYztnQkFDOUIsY0FBYyxFQUFFLFdBQVc7Z0JBQzNCLGtCQUFrQixFQUFFLGVBQWUsQ0FBQyxVQUFVO2FBQy9DO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsSUFBSSxHQUFHLENBQUMsY0FBYyxDQUFDLElBQUksRUFBRSw4QkFBOEIsRUFBRTtZQUMzRCxZQUFZLEVBQUUsc0JBQXNCLENBQUMsWUFBWTtZQUNqRCxZQUFZLEVBQUUsc0NBQXNDO1lBQ3BELFVBQVUsRUFBRTtnQkFDVixTQUFTLEVBQUUsS0FBSyxDQUFDLFNBQVM7Z0JBQzFCLGdCQUFnQixFQUFFLGdCQUFnQjtnQkFDbEMsY0FBYyxFQUFFLFdBQVc7Z0JBQzNCLGtCQUFrQixFQUFFLFlBQVksQ0FBQyxVQUFVO2FBQzVDO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsSUFBSSxHQUFHLENBQUMsY0FBYyxDQUFDLElBQUksRUFBRSxxQkFBcUIsRUFBRTtZQUNsRCxZQUFZLEVBQUUsc0JBQXNCLENBQUMsWUFBWTtZQUNqRCxZQUFZLEVBQUUsNkJBQTZCO1lBQzNDLFVBQVUsRUFBRTtnQkFDVixZQUFZLEVBQUUsNkJBQTZCO2dCQUMzQyx5QkFBeUIsRUFBRSx5QkFBeUIsQ0FBQyxZQUFZO2dCQUNqRSxzQkFBc0IsRUFBRSxzQkFBc0IsQ0FBQyxZQUFZO2dCQUMzRCxrQkFBa0IsRUFBRSxlQUFlLENBQUMsVUFBVTtnQkFDOUMsc0JBQXNCLEVBQUUsaUJBQWlCO2dCQUN6QyxzQkFBc0IsRUFBRSxzQkFBc0IsQ0FBQyxTQUFTO2dCQUN4RCxZQUFZLEVBQUUsWUFBWSxDQUFDLFVBQVU7Z0JBQ3JDLGVBQWUsRUFBRSxnQ0FBZ0MsQ0FBQyxJQUFJO2FBQ3ZEO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsSUFBSSxHQUFHLENBQUMsU0FBUyxDQUFDLElBQUksRUFBRSx3QkFBd0IsRUFBRTtZQUNoRCxLQUFLLEVBQUUsZUFBZSxDQUFDLFVBQVU7U0FDbEMsQ0FBQyxDQUFDO1FBRUgsSUFBSSxHQUFHLENBQUMsU0FBUyxDQUFDLElBQUksRUFBRSxrQkFBa0IsRUFBRTtZQUMxQyxLQUFLLEVBQUUsWUFBWSxDQUFDLFVBQVU7U0FDL0IsQ0FBQyxDQUFDO1FBRUgsSUFBSSxHQUFHLENBQUMsU0FBUyxDQUFDLElBQUksRUFBRSxlQUFlLEVBQUUsRUFBRSxLQUFLLEVBQUUsTUFBTSxDQUFDLE1BQU0sRUFBRSxDQUFDLENBQUM7UUFFbkUsSUFBSSxHQUFHLENBQUMsU0FBUyxDQUFDLElBQUksRUFBRSxjQUFjLEVBQUUsRUFBRSxLQUFLLEVBQUUsTUFBTSxDQUFDLE9BQU8sRUFBRSxDQUFDLENBQUM7SUFDckUsQ0FBQzs7OztJQUVELDRCQUE0QixDQUFDLG9CQUE0QjtRQUN2RCxPQUFPLDBCQUEwQixDQUFDLElBQUksQ0FBQyxvQkFBb0IsQ0FBQyxDQUFDO0lBQy9ELENBQUM7O0FBenNFSCxvQ0Ewc0VDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0ICogYXMgcGF0aCBmcm9tICdwYXRoJztcbmltcG9ydCAqIGFzIGdsdWUgZnJvbSAnQGF3cy1jZGsvYXdzLWdsdWUtYWxwaGEnO1xuaW1wb3J0ICogYXMgY2RrIGZyb20gJ2F3cy1jZGstbGliJztcbmltcG9ydCAqIGFzIGFwaWdhdGV3YXkgZnJvbSAnYXdzLWNkay1saWIvYXdzLWFwaWdhdGV3YXknO1xuaW1wb3J0ICogYXMgYXRoZW5hIGZyb20gJ2F3cy1jZGstbGliL2F3cy1hdGhlbmEnO1xuaW1wb3J0ICogYXMgY2xvdWR3YXRjaCBmcm9tICdhd3MtY2RrLWxpYi9hd3MtY2xvdWR3YXRjaCc7XG5pbXBvcnQgKiBhcyBldmVudHMgZnJvbSAnYXdzLWNkay1saWIvYXdzLWV2ZW50cyc7XG5pbXBvcnQgKiBhcyB0YXJnZXRzIGZyb20gJ2F3cy1jZGstbGliL2F3cy1ldmVudHMtdGFyZ2V0cyc7XG5pbXBvcnQgKiBhcyBjZm5HbHVlIGZyb20gJ2F3cy1jZGstbGliL2F3cy1nbHVlJztcbmltcG9ydCAqIGFzIGlhbSBmcm9tICdhd3MtY2RrLWxpYi9hd3MtaWFtJztcbmltcG9ydCAqIGFzIGZpcmVob3NlIGZyb20gJ2F3cy1jZGstbGliL2F3cy1raW5lc2lzZmlyZWhvc2UnO1xuaW1wb3J0ICogYXMgbGFtYmRhIGZyb20gJ2F3cy1jZGstbGliL2F3cy1sYW1iZGEnO1xuaW1wb3J0ICogYXMgczMgZnJvbSAnYXdzLWNkay1saWIvYXdzLXMzJztcbmltcG9ydCAqIGFzIHMzbiBmcm9tICdhd3MtY2RrLWxpYi9hd3MtczMtbm90aWZpY2F0aW9ucyc7XG5pbXBvcnQgKiBhcyB3YWZ2MiBmcm9tICdhd3MtY2RrLWxpYi9hd3Mtd2FmdjInO1xuaW1wb3J0ICogYXMgY3IgZnJvbSAnYXdzLWNkay1saWIvY3VzdG9tLXJlc291cmNlcyc7XG5pbXBvcnQgeyBDb25zdHJ1Y3QgfSBmcm9tICdjb25zdHJ1Y3RzJztcblxuZXhwb3J0IGVudW0gV2FmMlNjb3BlT3B0aW9uIHtcbiAgQ0xPVURGUk9OVCA9ICdDTE9VREZST05UJyxcbiAgUkVHSU9OQUwgPSAnUkVHSU9OQUwnLFxufVxuXG5leHBvcnQgZW51bSBMb2dMZXZlbCB7XG4gIERFQlVHID0gJ0RFQlVHJyxcbiAgSU5GTyA9ICdJTkZPJyxcbiAgV0FSTklORyA9ICdXQVJOSU5HJyxcbiAgRVJST1IgPSAnRVJST1InLFxuICBDUklUSUNBTCA9ICdDUklUSUNBTCcsXG59XG5cbmNvbnN0IEVsYXN0aWNfTG9hZF9CYWxhbmNpbmdfYWNjb3VudF9JRDogeyBba2V5OiBzdHJpbmddOiBzdHJpbmcgfSA9IHtcbiAgJ3VzLWVhc3QtMSc6ICcxMjczMTE5MjMwMjEnLFxuICAndXMtZWFzdC0yJzogJzAzMzY3Nzk5NDI0MCcsXG4gICd1cy13ZXN0LTEnOiAnMDI3NDM0NzQyOTgwJyxcbiAgJ3VzLXdlc3QtMic6ICc3OTc4NzM5NDYxOTQnLFxuICAnYWYtc291dGgtMSc6ICcwOTgzNjkyMTY1OTMnLFxuICAnY2EtY2VudHJhbC0xJzogJzk4NTY2NjYwOTI1MScsXG4gICdldS1jZW50cmFsLTEnOiAnMDU0Njc2ODIwOTI4JyxcbiAgJ2V1LXdlc3QtMSc6ICcxNTY0NjA2MTI4MDYnLFxuICAnZXUtd2VzdC0yJzogJzY1MjcxMTUwNDQxNicsXG4gICdldS1zb3V0aC0xJzogJzYzNTYzMTIzMjEyNycsXG4gICdldS13ZXN0LTMnOiAnMDA5OTk2NDU3NjY3JyxcbiAgJ2V1LW5vcnRoLTEnOiAnODk3ODIyOTY3MDYyJyxcbiAgJ2FwLWVhc3QtMSc6ICc3NTQzNDQ0NDg2NDgnLFxuICAnYXAtbm9ydGhlYXN0LTEnOiAnNTgyMzE4NTYwODY0JyxcbiAgJ2FwLW5vcnRoZWFzdC0yJzogJzYwMDczNDU3NTg4NycsXG4gICdhcC1ub3J0aGVhc3QtMyc6ICczODM1OTc0NzczMzEnLFxuICAnYXAtc291dGhlYXN0LTEnOiAnMTE0Nzc0MTMxNDUwJyxcbiAgJ2FwLXNvdXRoZWFzdC0yJzogJzc4MzIyNTMxOTI2NicsXG4gICdhcC1zb3V0aC0xJzogJzcxODUwNDQyODM3OCcsXG4gICdtZS1zb3V0aC0xJzogJzA3NjY3NDU3MDIyNScsXG4gICdzYS1lYXN0LTEnOiAnNTA3MjQxNTI4NTE3JyxcbiAgJ3VzLWdvdi13ZXN0LTEnOiAnMDQ4NTkxMDExNTg0JyxcbiAgJ3VzLWdvdi1lYXN0LTEnOiAnMTkwNTYwMzkxNjM1JyxcbiAgJ2NuLW5vcnRoLTEnOiAnNjM4MTAyMTQ2OTkzJyxcbiAgJ2NuLW5vcnRod2VzdC0xJzogJzAzNzYwNDcwMTM0MCcsXG59O1xuXG5leHBvcnQgaW50ZXJmYWNlIEF1dG9tYXRlZFdhZlByb3BzIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSB3YWYyU2NvcGU6IFdhZjJTY29wZU9wdGlvbjtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGNvdW50TW9kZT86IGJvb2xlYW47XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBhc3NvY2lhdGVkUmVzb3VyY2VBcm4/OiBzdHJpbmc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSByZXNvdXJjZU5hbWluZ1ByZWZpeD86IHN0cmluZztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgZXJyb3JUaHJlc2hvbGQ/OiBudW1iZXI7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHJlcXVlc3RUaHJlc2hvbGQ/OiBudW1iZXI7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGJsb2NrUGVyaW9kPzogbnVtYmVyO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgZW5hYmxlU2hpZWxkQWR2YW5jZWRMYW1iZGE/OiBib29sZWFuO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgbG9nTGV2ZWw/OiBMb2dMZXZlbDtcblxuICByZWFkb25seSB3YWZMb2dCdWNrZXROYW1lPzogc3RyaW5nO1xuXG4gIHJlYWRvbmx5IGFwcEFjY2Vzc0xvZ0J1Y2tldE5hbWU/OiBzdHJpbmc7XG59XG5cbmV4cG9ydCBjbGFzcyBBdXRvbWF0ZWRXYWYgZXh0ZW5kcyBDb25zdHJ1Y3Qge1xuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogQXV0b21hdGVkV2FmUHJvcHMpIHtcbiAgICBzdXBlcihzY29wZSwgaWQpO1xuXG4gICAgY29uc3Qgc3RhY2sgPSBjZGsuU3RhY2sub2YodGhpcyk7XG4gICAgaWYgKFxuICAgICAgcHJvcHMucmVzb3VyY2VOYW1pbmdQcmVmaXggJiZcbiAgICAgICF0aGlzLnZhbGlkYXRlUmVzb3VyY2VOYW1pbmdQcmVmaXgocHJvcHMucmVzb3VyY2VOYW1pbmdQcmVmaXgpXG4gICAgKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoXG4gICAgICAgICdyZXNvdXJjZU5hbWluZ1ByZWZpeCBwcm9wZXJ0eSBvbmx5IGFsbG93IGFscGhhbnVtZXJpYyBhbmQgXCJfXCIgc3ltYm9sIGJlY2F1c2UgZ2x1ZSBkYXRhYmFzZSBuYW1pbmcgaXMgbmVlZGVkJ1xuICAgICAgKTtcbiAgICB9XG5cbiAgICAvL2RlZmF1bHQgdmFsdWVcbiAgICBjb25zdCBlcnJvclRocmVzaG9sZCA9IHByb3BzLmVycm9yVGhyZXNob2xkID8/IDIwMDtcbiAgICBjb25zdCBjb3VudE1vZGUgPSBwcm9wcy5jb3VudE1vZGUgPz8gZmFsc2U7XG4gICAgY29uc3QgcmVxdWVzdFRocmVzaG9sZCA9IHByb3BzLnJlcXVlc3RUaHJlc2hvbGQgPz8gMTAwMDtcbiAgICBjb25zdCBibG9ja1BlcmlvZCA9IHByb3BzLmJsb2NrUGVyaW9kID8/IDI0MDtcbiAgICBjb25zdCBlbmFibGVTaGllbGRBZHZhbmNlZExhbWJkYSA9XG4gICAgICBwcm9wcy5lbmFibGVTaGllbGRBZHZhbmNlZExhbWJkYSA/PyBmYWxzZTtcblxuICAgIGNvbnN0IGxvZ0xldmVsID0gcHJvcHMubG9nTGV2ZWwgPz8gJ0lORk8nO1xuXG4gICAgY29uc3QgY2xvdWRXYXRjaERhc2hib2FyZE5hbWUgPSBgV0FGTW9uaXRvcmluZ0Rhc2hib2FyZCR7cHJvcHMucmVzb3VyY2VOYW1pbmdQcmVmaXh9JHtjZGsuQXdzLlJFR0lPTn1gO1xuICAgIGNvbnN0IHJlcHV0YXRpb25MaXN0TmFtZSA9XG4gICAgICBjZGsuRm4ucmVmKCdBV1M6OlN0YWNrTmFtZScpICsgJ0lQUmVwdXRhdGlvbkxpc3RzUnVsZSc7XG4gICAgY29uc3QgYWxsb3dMaXN0TmFtZSA9IGNkay5Gbi5yZWYoJ0FXUzo6U3RhY2tOYW1lJykgKyAnV2hpdGVsaXN0UnVsZSc7XG4gICAgY29uc3QgYmxhY2tsaXN0UnVsZU5hbWUgPSBjZGsuRm4ucmVmKCdBV1M6OlN0YWNrTmFtZScpICsgJ0JsYWNrbGlzdFJ1bGUnO1xuICAgIGNvbnN0IGh0dHBGbG9vZFJlZ3VsYXJSdWxlTmFtZSA9XG4gICAgICBjZGsuRm4ucmVmKCdBV1M6OlN0YWNrTmFtZScpICsgJ0h0dHBGbG9vZFJlZ3VsYXJSdWxlJztcbiAgICBjb25zdCBzY2FubmVyc1Byb2Jlc1J1bGVOYW1lID1cbiAgICAgIGNkay5Gbi5yZWYoJ0FXUzo6U3RhY2tOYW1lJykgKyAnU2Nhbm5lcnNQcm9iZXNSdWxlJztcbiAgICBjb25zdCBiYWRCb3RSdWxlTmFtZSA9IGNkay5Gbi5yZWYoJ0FXUzo6U3RhY2tOYW1lJykgKyAnQmFkQm90UnVsZSc7XG4gICAgY29uc3Qgc3FsSW5qZWN0aW9uUnVsZU5hbWUgPVxuICAgICAgY2RrLkZuLnJlZignQVdTOjpTdGFja05hbWUnKSArICdTcWxJbmplY3Rpb25SdWxlJztcbiAgICBjb25zdCB4c3NSdWxlTmFtZSA9IGNkay5Gbi5yZWYoJ0FXUzo6U3RhY2tOYW1lJykgKyAnWHNzUnVsZSc7XG5cbiAgICBjb25zdCB3YWZMb2dCdWNrZXRBcHBlbmRQcm9wcyA9IHByb3BzLndhZkxvZ0J1Y2tldE5hbWVcbiAgICAgID8geyBidWNrZXROYW1lOiBwcm9wcy53YWZMb2dCdWNrZXROYW1lIH1cbiAgICAgIDoge307XG5cbiAgICAvL1dhZkxvZ0J1Y2tldFxuICAgIGNvbnN0IHdhZkxvZ0J1Y2tldCA9IG5ldyBzMy5CdWNrZXQodGhpcywgJ1dhZkxvZ0J1Y2tldCcsIHtcbiAgICAgIC4uLndhZkxvZ0J1Y2tldEFwcGVuZFByb3BzLFxuICAgICAgcHVibGljUmVhZEFjY2VzczogZmFsc2UsXG4gICAgICBlbmNyeXB0aW9uOiBzMy5CdWNrZXRFbmNyeXB0aW9uLktNU19NQU5BR0VELFxuICAgICAgcmVtb3ZhbFBvbGljeTogY2RrLlJlbW92YWxQb2xpY3kuREVTVFJPWSxcbiAgICAgIGF1dG9EZWxldGVPYmplY3RzOiB0cnVlLFxuICAgIH0pO1xuXG4gICAgY29uc3QgYWNjZXNzTG9nQnVja2V0QXBwZW5kUHJvcHMgPSBwcm9wcy5hcHBBY2Nlc3NMb2dCdWNrZXROYW1lXG4gICAgICA/IHsgYnVja2V0TmFtZTogcHJvcHMuYXBwQWNjZXNzTG9nQnVja2V0TmFtZSB9XG4gICAgICA6IHt9O1xuXG4gICAgY29uc3QgYWNjZXNzTG9nQnVja2V0ID0gbmV3IHMzLkJ1Y2tldCh0aGlzLCAnQWNjZXNzTG9nQnVja2V0Jywge1xuICAgICAgLi4uYWNjZXNzTG9nQnVja2V0QXBwZW5kUHJvcHMsXG4gICAgICBwdWJsaWNSZWFkQWNjZXNzOiBmYWxzZSxcbiAgICAgIGVuY3J5cHRpb246IHMzLkJ1Y2tldEVuY3J5cHRpb24uUzNfTUFOQUdFRCxcbiAgICAgIGFjY2Vzc0NvbnRyb2w6XG4gICAgICAgIHByb3BzLndhZjJTY29wZSA9PSBXYWYyU2NvcGVPcHRpb24uQ0xPVURGUk9OVCA/IHMzLkJ1Y2tldEFjY2Vzc0NvbnRyb2wuTE9HX0RFTElWRVJZX1dSSVRFIDogczMuQnVja2V0QWNjZXNzQ29udHJvbC5QUklWQVRFLFxuICAgICAgcmVtb3ZhbFBvbGljeTogY2RrLlJlbW92YWxQb2xpY3kuREVTVFJPWSxcbiAgICAgIGF1dG9EZWxldGVPYmplY3RzOiB0cnVlLFxuICAgIH0pO1xuXG4gICAgaWYgKHByb3BzLndhZjJTY29wZSA9PSBXYWYyU2NvcGVPcHRpb24uUkVHSU9OQUwpIHtcbiAgICAgIGFjY2Vzc0xvZ0J1Y2tldC5hZGRUb1Jlc291cmNlUG9saWN5KFxuICAgICAgICBuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgICAgZWZmZWN0OiBpYW0uRWZmZWN0LkFMTE9XLFxuICAgICAgICAgIGFjdGlvbnM6IFsnczM6UHV0T2JqZWN0J10sXG4gICAgICAgICAgcmVzb3VyY2VzOiBbXG4gICAgICAgICAgICBgJHthY2Nlc3NMb2dCdWNrZXQuYnVja2V0QXJufS9BV1NMb2dzLyR7Y2RrLkF3cy5BQ0NPVU5UX0lEfS8qYCxcbiAgICAgICAgICBdLFxuICAgICAgICAgIHByaW5jaXBhbHM6IFtcbiAgICAgICAgICAgIG5ldyBpYW0uQWNjb3VudFByaW5jaXBhbChcbiAgICAgICAgICAgICAgRWxhc3RpY19Mb2FkX0JhbGFuY2luZ19hY2NvdW50X0lEW3N0YWNrLnJlZ2lvbl1cbiAgICAgICAgICAgICksXG4gICAgICAgICAgXSxcbiAgICAgICAgfSlcbiAgICAgICk7XG4gICAgICBhY2Nlc3NMb2dCdWNrZXQuYWRkVG9SZXNvdXJjZVBvbGljeShcbiAgICAgICAgbmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgICAgIGVmZmVjdDogaWFtLkVmZmVjdC5BTExPVyxcbiAgICAgICAgICBhY3Rpb25zOiBbJ3MzOlB1dE9iamVjdCddLFxuICAgICAgICAgIHJlc291cmNlczogW1xuICAgICAgICAgICAgYCR7YWNjZXNzTG9nQnVja2V0LmJ1Y2tldEFybn0vQVdTTG9ncy8ke2Nkay5Bd3MuQUNDT1VOVF9JRH0vKmAsXG4gICAgICAgICAgXSxcbiAgICAgICAgICBwcmluY2lwYWxzOiBbbmV3IGlhbS5TZXJ2aWNlUHJpbmNpcGFsKCdkZWxpdmVyeS5sb2dzLmFtYXpvbmF3cy5jb20nKV0sXG4gICAgICAgICAgY29uZGl0aW9uczoge1xuICAgICAgICAgICAgU3RyaW5nRXF1YWxzOiB7XG4gICAgICAgICAgICAgICdzMzp4LWFtei1hY2wnOiAnYnVja2V0LW93bmVyLWZ1bGwtY29udHJvbCcsXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIH0sXG4gICAgICAgIH0pXG4gICAgICApO1xuICAgICAgYWNjZXNzTG9nQnVja2V0LmFkZFRvUmVzb3VyY2VQb2xpY3koXG4gICAgICAgIG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgICBlZmZlY3Q6IGlhbS5FZmZlY3QuQUxMT1csXG4gICAgICAgICAgYWN0aW9uczogWydzMzpHZXRCdWNrZXRBY2wnXSxcbiAgICAgICAgICByZXNvdXJjZXM6IFthY2Nlc3NMb2dCdWNrZXQuYnVja2V0QXJuXSxcbiAgICAgICAgICBwcmluY2lwYWxzOiBbbmV3IGlhbS5TZXJ2aWNlUHJpbmNpcGFsKCdkZWxpdmVyeS5sb2dzLmFtYXpvbmF3cy5jb20nKV0sXG4gICAgICAgIH0pXG4gICAgICApO1xuICAgIH1cblxuICAgIC8vIFNldHVwIFdoaXRlbGlzdCBJUCBTZXRcbiAgICBjb25zdCB3aGl0ZWxpc3RJcFNldFY0ID0gbmV3IHdhZnYyLkNmbklQU2V0KHRoaXMsICdXaGl0ZWxpc3RTZXRJUFY0Jywge1xuICAgICAgYWRkcmVzc2VzOiBbXSxcbiAgICAgIGlwQWRkcmVzc1ZlcnNpb246ICdJUFY0JyxcbiAgICAgIHNjb3BlOiBwcm9wcy53YWYyU2NvcGUsXG4gICAgICBuYW1lOiBjZGsuRm4ucmVmKCdBV1M6OlN0YWNrTmFtZScpICsgJ1doaXRlbGlzdFNldElQVjQnLFxuICAgICAgZGVzY3JpcHRpb246ICdBbGxvdyB3aGl0ZWxpc3QgZm9yIElQVjQgYWRkcmVzc2VzJyxcbiAgICB9KTtcblxuICAgIGNvbnN0IHdoaXRlbGlzdElwU2V0VjYgPSBuZXcgd2FmdjIuQ2ZuSVBTZXQodGhpcywgJ1doaXRlbGlzdFNldElQVjYnLCB7XG4gICAgICBhZGRyZXNzZXM6IFtdLFxuICAgICAgaXBBZGRyZXNzVmVyc2lvbjogJ0lQVjYnLFxuICAgICAgc2NvcGU6IHByb3BzLndhZjJTY29wZSxcbiAgICAgIG5hbWU6IGNkay5Gbi5yZWYoJ0FXUzo6U3RhY2tOYW1lJykgKyAnV2hpdGVsaXN0U2V0SVBWNicsXG4gICAgICBkZXNjcmlwdGlvbjogJ0FsbG93IHdoaXRlbGlzdCBmb3IgSVBWNiBhZGRyZXNzZXMnLFxuICAgIH0pO1xuXG4gICAgLy8gU2V0dXAgQmxhY2tsaXN0IElQIFNldFxuICAgIGNvbnN0IGJsYWNrbGlzdElwU2V0VjQgPSBuZXcgd2FmdjIuQ2ZuSVBTZXQodGhpcywgJ0JsYWNrbGlzdFNldElQVjQnLCB7XG4gICAgICBhZGRyZXNzZXM6IFtdLFxuICAgICAgaXBBZGRyZXNzVmVyc2lvbjogJ0lQVjQnLFxuICAgICAgc2NvcGU6IHByb3BzLndhZjJTY29wZSxcbiAgICAgIG5hbWU6IGNkay5Gbi5yZWYoJ0FXUzo6U3RhY2tOYW1lJykgKyAnQmxhY2tsaXN0U2V0SVBWNCcsXG4gICAgICBkZXNjcmlwdGlvbjogJ0FsbG93IGJsYWNrbGlzdCBmb3IgSVBWNCBhZGRyZXNzZXMnLFxuICAgIH0pO1xuXG4gICAgY29uc3QgYmxhY2tsaXN0SXBTZXRWNiA9IG5ldyB3YWZ2Mi5DZm5JUFNldCh0aGlzLCAnQmxhY2tsaXN0U2V0SVBWNicsIHtcbiAgICAgIGFkZHJlc3NlczogW10sXG4gICAgICBpcEFkZHJlc3NWZXJzaW9uOiAnSVBWNicsXG4gICAgICBzY29wZTogcHJvcHMud2FmMlNjb3BlLFxuICAgICAgbmFtZTogY2RrLkZuLnJlZignQVdTOjpTdGFja05hbWUnKSArICdCbGFja2xpc3RTZXRJUFY2JyxcbiAgICAgIGRlc2NyaXB0aW9uOiAnQWxsb3cgYmxhY2tsaXN0IGZvciBJUFY2IGFkZHJlc3NlcycsXG4gICAgfSk7XG5cbiAgICAvLyBTZXR1cCBIVFRQIEZsb29kIElQIFNldFxuICAgIGNvbnN0IGh0dHBGbG9vZElwU2V0VjQgPSBuZXcgd2FmdjIuQ2ZuSVBTZXQodGhpcywgJ0hUVFBGbG9vZFNldElQVjQnLCB7XG4gICAgICBhZGRyZXNzZXM6IFtdLFxuICAgICAgaXBBZGRyZXNzVmVyc2lvbjogJ0lQVjQnLFxuICAgICAgc2NvcGU6IHByb3BzLndhZjJTY29wZSxcbiAgICAgIG5hbWU6IGNkay5Gbi5yZWYoJ0FXUzo6U3RhY2tOYW1lJykgKyAnSFRUUEZsb29kU2V0SVBWNCcsXG4gICAgICBkZXNjcmlwdGlvbjogJ0Jsb2NrIEhUVFAgRmxvb2QgSVBWNCBhZGRyZXNzZXMnLFxuICAgIH0pO1xuXG4gICAgY29uc3QgaHR0cEZsb29kSXBTZXRWNiA9IG5ldyB3YWZ2Mi5DZm5JUFNldCh0aGlzLCAnSFRUUEZsb29kU2V0SVBWNicsIHtcbiAgICAgIGFkZHJlc3NlczogW10sXG4gICAgICBpcEFkZHJlc3NWZXJzaW9uOiAnSVBWNicsXG4gICAgICBzY29wZTogcHJvcHMud2FmMlNjb3BlLFxuICAgICAgbmFtZTogY2RrLkZuLnJlZignQVdTOjpTdGFja05hbWUnKSArICdIVFRQRmxvb2RTZXRJUFY2JyxcbiAgICAgIGRlc2NyaXB0aW9uOiAnQmxvY2sgSFRUUCBGbG9vZCBJUFY2IGFkZHJlc3NlcycsXG4gICAgfSk7XG5cbiAgICAvLyBCbG9jayBTY2FubmVycy9Qcm9iZXMgSVAgU2V0XG4gICAgY29uc3Qgc2Nhbm5lcnNQcm9iZXNJcFNldFY0ID0gbmV3IHdhZnYyLkNmbklQU2V0KFxuICAgICAgdGhpcyxcbiAgICAgICdTY2FubmVyc1Byb2Jlc1NldElQVjQnLFxuICAgICAge1xuICAgICAgICBhZGRyZXNzZXM6IFtdLFxuICAgICAgICBpcEFkZHJlc3NWZXJzaW9uOiAnSVBWNCcsXG4gICAgICAgIHNjb3BlOiBwcm9wcy53YWYyU2NvcGUsXG4gICAgICAgIG5hbWU6IGNkay5Gbi5yZWYoJ0FXUzo6U3RhY2tOYW1lJykgKyAnU2Nhbm5lcnNQcm9iZXNTZXRJUFY0JyxcbiAgICAgICAgZGVzY3JpcHRpb246ICdCbG9jayBTY2FubmVycy9Qcm9iZXMgSVBWNCBhZGRyZXNzZXMnLFxuICAgICAgfVxuICAgICk7XG5cbiAgICBjb25zdCBzY2FubmVyc1Byb2Jlc0lwU2V0VjYgPSBuZXcgd2FmdjIuQ2ZuSVBTZXQoXG4gICAgICB0aGlzLFxuICAgICAgJ1NjYW5uZXJzUHJvYmVzU2V0SVBWNicsXG4gICAgICB7XG4gICAgICAgIGFkZHJlc3NlczogW10sXG4gICAgICAgIGlwQWRkcmVzc1ZlcnNpb246ICdJUFY2JyxcbiAgICAgICAgc2NvcGU6IHByb3BzLndhZjJTY29wZSxcbiAgICAgICAgbmFtZTogY2RrLkZuLnJlZignQVdTOjpTdGFja05hbWUnKSArICdTY2FubmVyc1Byb2Jlc1NldElQVjYnLFxuICAgICAgICBkZXNjcmlwdGlvbjogJ0Jsb2NrIFNjYW5uZXJzL1Byb2JlcyBJUFY2IGFkZHJlc3NlcycsXG4gICAgICB9XG4gICAgKTtcblxuICAgIC8vIEJsb2NrIFJlcHV0YXRpb24gTGlzdCBJUCBTZXRcbiAgICBjb25zdCByZXB1dGF0aW9uTGlzdHNJcFNldFY0ID0gbmV3IHdhZnYyLkNmbklQU2V0KFxuICAgICAgdGhpcyxcbiAgICAgICdJUFJlcHV0YXRpb25MaXN0c1NldElQVjQnLFxuICAgICAge1xuICAgICAgICBhZGRyZXNzZXM6IFtdLFxuICAgICAgICBpcEFkZHJlc3NWZXJzaW9uOiAnSVBWNCcsXG4gICAgICAgIHNjb3BlOiBwcm9wcy53YWYyU2NvcGUsXG4gICAgICAgIG5hbWU6IGNkay5Gbi5yZWYoJ0FXUzo6U3RhY2tOYW1lJykgKyAnSVBSZXB1dGF0aW9uTGlzdHNTZXRJUFY0JyxcbiAgICAgICAgZGVzY3JpcHRpb246ICdCbG9jayBSZXB1dGF0aW9uIExpc3QgSVBWNCBhZGRyZXNzZXMnLFxuICAgICAgfVxuICAgICk7XG5cbiAgICBjb25zdCByZXB1dGF0aW9uTGlzdHNJcFNldFY2ID0gbmV3IHdhZnYyLkNmbklQU2V0KFxuICAgICAgdGhpcyxcbiAgICAgICdJUFJlcHV0YXRpb25MaXN0c1NldElQVjYnLFxuICAgICAge1xuICAgICAgICBhZGRyZXNzZXM6IFtdLFxuICAgICAgICBpcEFkZHJlc3NWZXJzaW9uOiAnSVBWNicsXG4gICAgICAgIHNjb3BlOiBwcm9wcy53YWYyU2NvcGUsXG4gICAgICAgIG5hbWU6IGNkay5Gbi5yZWYoJ0FXUzo6U3RhY2tOYW1lJykgKyAnSVBSZXB1dGF0aW9uTGlzdHNTZXRJUFY2JyxcbiAgICAgICAgZGVzY3JpcHRpb246ICdCbG9jayBSZXB1dGF0aW9uIExpc3QgSVBWNiBhZGRyZXNzZXMnLFxuICAgICAgfVxuICAgICk7XG5cbiAgICAvLyBCbG9jayBCYWQgQm90IElQIFNldFxuICAgIGNvbnN0IGJhZEJvdElwU2V0VjQgPSBuZXcgd2FmdjIuQ2ZuSVBTZXQodGhpcywgJ0lQQmFkQm90U2V0SVBWNCcsIHtcbiAgICAgIGFkZHJlc3NlczogW10sXG4gICAgICBpcEFkZHJlc3NWZXJzaW9uOiAnSVBWNCcsXG4gICAgICBzY29wZTogcHJvcHMud2FmMlNjb3BlLFxuICAgICAgbmFtZTogY2RrLkZuLnJlZignQVdTOjpTdGFja05hbWUnKSArICdJUEJhZEJvdFNldElQVjQnLFxuICAgICAgZGVzY3JpcHRpb246ICdCbG9jayBCYWQgQm90IElQVjQgYWRkcmVzc2VzJyxcbiAgICB9KTtcblxuICAgIGNvbnN0IGJhZEJvdElwU2V0VjYgPSBuZXcgd2FmdjIuQ2ZuSVBTZXQodGhpcywgJ0lQQmFkQm90U2V0SVBWNicsIHtcbiAgICAgIGFkZHJlc3NlczogW10sXG4gICAgICBpcEFkZHJlc3NWZXJzaW9uOiAnSVBWNicsXG4gICAgICBzY29wZTogcHJvcHMud2FmMlNjb3BlLFxuICAgICAgbmFtZTogY2RrLkZuLnJlZignQVdTOjpTdGFja05hbWUnKSArICdJUEJhZEJvdFNldElQVjYnLFxuICAgICAgZGVzY3JpcHRpb246ICdCbG9jayBCYWQgQm90IElQVjYgYWRkcmVzc2VzJyxcbiAgICB9KTtcblxuICAgIGNvbnN0IGJhZEJvdFdhZlJ1bGVzID0gW107XG4gICAgaWYgKHByb3BzLndhZjJTY29wZSA9PSBXYWYyU2NvcGVPcHRpb24uQ0xPVURGUk9OVCkge1xuICAgICAgYmFkQm90V2FmUnVsZXMucHVzaCh7XG4gICAgICAgIG5hbWU6IGJhZEJvdFJ1bGVOYW1lLFxuICAgICAgICBwcmlvcml0eTogNyxcbiAgICAgICAgYWN0aW9uOiAhY291bnRNb2RlXG4gICAgICAgICAgPyB7XG4gICAgICAgICAgICBibG9jazoge30sXG4gICAgICAgICAgfVxuICAgICAgICAgIDogeyBjb3VudDoge30gfSxcbiAgICAgICAgdmlzaWJpbGl0eUNvbmZpZzoge1xuICAgICAgICAgIHNhbXBsZWRSZXF1ZXN0c0VuYWJsZWQ6IHRydWUsXG4gICAgICAgICAgY2xvdWRXYXRjaE1ldHJpY3NFbmFibGVkOiB0cnVlLFxuICAgICAgICAgIG1ldHJpY05hbWU6IGJhZEJvdFJ1bGVOYW1lLFxuICAgICAgICB9LFxuICAgICAgICBzdGF0ZW1lbnQ6IHtcbiAgICAgICAgICBvclN0YXRlbWVudDoge1xuICAgICAgICAgICAgc3RhdGVtZW50czogW1xuICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgaXBTZXRSZWZlcmVuY2VTdGF0ZW1lbnQ6IHtcbiAgICAgICAgICAgICAgICAgIGFybjogY2RrLkZuLmdldEF0dChiYWRCb3RJcFNldFY0LmxvZ2ljYWxJZCwgJ0FybicpLnRvU3RyaW5nKCksXG4gICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgIGlwU2V0UmVmZXJlbmNlU3RhdGVtZW50OiB7XG4gICAgICAgICAgICAgICAgICBhcm46IGNkay5Gbi5nZXRBdHQoYmFkQm90SXBTZXRWNi5sb2dpY2FsSWQsICdBcm4nKS50b1N0cmluZygpLFxuICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICBdLFxuICAgICAgICAgIH0sXG4gICAgICAgIH0sXG4gICAgICB9KTtcbiAgICB9XG5cbiAgICAvLyBXQUYgV2ViIEFDTFxuICAgIGNvbnN0IHdhZndlYiA9IG5ldyB3YWZ2Mi5DZm5XZWJBQ0wodGhpcywgJ3dhZndlYicsIHtcbiAgICAgIG5hbWU6IHByb3BzLnJlc291cmNlTmFtaW5nUHJlZml4XG4gICAgICAgID8gYCR7cHJvcHMucmVzb3VyY2VOYW1pbmdQcmVmaXh9LVdBRmBcbiAgICAgICAgOiAnQ2xvdWRGcm9udC1XZWItV0FGJyxcbiAgICAgIGRlc2NyaXB0aW9uOiAnQ3VzdG9tIFdBRldlYkFDTCcsXG4gICAgICBkZWZhdWx0QWN0aW9uOiB7XG4gICAgICAgIGFsbG93OiB7fSxcbiAgICAgIH0sXG4gICAgICBzY29wZTogcHJvcHMud2FmMlNjb3BlLFxuICAgICAgdmlzaWJpbGl0eUNvbmZpZzoge1xuICAgICAgICBjbG91ZFdhdGNoTWV0cmljc0VuYWJsZWQ6IHRydWUsXG4gICAgICAgIHNhbXBsZWRSZXF1ZXN0c0VuYWJsZWQ6IHRydWUsXG4gICAgICAgIG1ldHJpY05hbWU6IHByb3BzLnJlc291cmNlTmFtaW5nUHJlZml4XG4gICAgICAgICAgPyBgJHtwcm9wcy5yZXNvdXJjZU5hbWluZ1ByZWZpeH0tV0FGYFxuICAgICAgICAgIDogJ0Nsb3VkRnJvbnQtV2ViLVdBRicsXG4gICAgICB9LFxuICAgICAgcnVsZXM6IFtcbiAgICAgICAge1xuICAgICAgICAgIG5hbWU6ICdBV1MtQVdTTWFuYWdlZFJ1bGVzQ29tbW9uUnVsZVNldCcsXG4gICAgICAgICAgcHJpb3JpdHk6IDAsXG4gICAgICAgICAgb3ZlcnJpZGVBY3Rpb246ICFjb3VudE1vZGVcbiAgICAgICAgICAgID8ge1xuICAgICAgICAgICAgICBub25lOiB7fSxcbiAgICAgICAgICAgIH1cbiAgICAgICAgICAgIDogeyBjb3VudDoge30gfSxcbiAgICAgICAgICB2aXNpYmlsaXR5Q29uZmlnOiB7XG4gICAgICAgICAgICBjbG91ZFdhdGNoTWV0cmljc0VuYWJsZWQ6IHRydWUsXG4gICAgICAgICAgICBzYW1wbGVkUmVxdWVzdHNFbmFibGVkOiB0cnVlLFxuICAgICAgICAgICAgbWV0cmljTmFtZTogcHJvcHMucmVzb3VyY2VOYW1pbmdQcmVmaXhcbiAgICAgICAgICAgICAgPyBgJHtwcm9wcy5yZXNvdXJjZU5hbWluZ1ByZWZpeH0td2FmLWlwc2V0LW1ldHJpY3NgXG4gICAgICAgICAgICAgIDogJ2Nsb3VkZnJvbnQtd2FmLWlwc2V0LW1ldHJpY3MnLFxuICAgICAgICAgIH0sXG4gICAgICAgICAgc3RhdGVtZW50OiB7XG4gICAgICAgICAgICBtYW5hZ2VkUnVsZUdyb3VwU3RhdGVtZW50OiB7XG4gICAgICAgICAgICAgIHZlbmRvck5hbWU6ICdBV1MnLFxuICAgICAgICAgICAgICBuYW1lOiAnQVdTTWFuYWdlZFJ1bGVzQ29tbW9uUnVsZVNldCcsXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIH0sXG4gICAgICAgIH0sXG4gICAgICAgIHtcbiAgICAgICAgICBuYW1lOiBhbGxvd0xpc3ROYW1lLFxuICAgICAgICAgIHByaW9yaXR5OiAxLFxuICAgICAgICAgIGFjdGlvbjogIWNvdW50TW9kZVxuICAgICAgICAgICAgPyB7XG4gICAgICAgICAgICAgIGFsbG93OiB7fSxcbiAgICAgICAgICAgIH1cbiAgICAgICAgICAgIDogeyBjb3VudDoge30gfSxcbiAgICAgICAgICB2aXNpYmlsaXR5Q29uZmlnOiB7XG4gICAgICAgICAgICBzYW1wbGVkUmVxdWVzdHNFbmFibGVkOiB0cnVlLFxuICAgICAgICAgICAgY2xvdWRXYXRjaE1ldHJpY3NFbmFibGVkOiB0cnVlLFxuICAgICAgICAgICAgbWV0cmljTmFtZTogYWxsb3dMaXN0TmFtZSxcbiAgICAgICAgICB9LFxuICAgICAgICAgIHN0YXRlbWVudDoge1xuICAgICAgICAgICAgb3JTdGF0ZW1lbnQ6IHtcbiAgICAgICAgICAgICAgc3RhdGVtZW50czogW1xuICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgIGlwU2V0UmVmZXJlbmNlU3RhdGVtZW50OiB7XG4gICAgICAgICAgICAgICAgICAgIGFybjogY2RrLkZuLmdldEF0dChcbiAgICAgICAgICAgICAgICAgICAgICB3aGl0ZWxpc3RJcFNldFY0LmxvZ2ljYWxJZCxcbiAgICAgICAgICAgICAgICAgICAgICAnQXJuJ1xuICAgICAgICAgICAgICAgICAgICApLnRvU3RyaW5nKCksXG4gICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgaXBTZXRSZWZlcmVuY2VTdGF0ZW1lbnQ6IHtcbiAgICAgICAgICAgICAgICAgICAgYXJuOiBjZGsuRm4uZ2V0QXR0KFxuICAgICAgICAgICAgICAgICAgICAgIHdoaXRlbGlzdElwU2V0VjYubG9naWNhbElkLFxuICAgICAgICAgICAgICAgICAgICAgICdBcm4nXG4gICAgICAgICAgICAgICAgICAgICkudG9TdHJpbmcoKSxcbiAgICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgfSxcbiAgICAgICAgfSxcbiAgICAgICAge1xuICAgICAgICAgIG5hbWU6IGJsYWNrbGlzdFJ1bGVOYW1lLFxuICAgICAgICAgIHByaW9yaXR5OiAyLFxuICAgICAgICAgIGFjdGlvbjogIWNvdW50TW9kZVxuICAgICAgICAgICAgPyB7XG4gICAgICAgICAgICAgIGJsb2NrOiB7fSxcbiAgICAgICAgICAgIH1cbiAgICAgICAgICAgIDogeyBjb3VudDoge30gfSxcblxuICAgICAgICAgIHZpc2liaWxpdHlDb25maWc6IHtcbiAgICAgICAgICAgIHNhbXBsZWRSZXF1ZXN0c0VuYWJsZWQ6IHRydWUsXG4gICAgICAgICAgICBjbG91ZFdhdGNoTWV0cmljc0VuYWJsZWQ6IHRydWUsXG4gICAgICAgICAgICBtZXRyaWNOYW1lOiBibGFja2xpc3RSdWxlTmFtZSxcbiAgICAgICAgICB9LFxuICAgICAgICAgIHN0YXRlbWVudDoge1xuICAgICAgICAgICAgb3JTdGF0ZW1lbnQ6IHtcbiAgICAgICAgICAgICAgc3RhdGVtZW50czogW1xuICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgIGlwU2V0UmVmZXJlbmNlU3RhdGVtZW50OiB7XG4gICAgICAgICAgICAgICAgICAgIGFybjogY2RrLkZuLmdldEF0dChcbiAgICAgICAgICAgICAgICAgICAgICBibGFja2xpc3RJcFNldFY0LmxvZ2ljYWxJZCxcbiAgICAgICAgICAgICAgICAgICAgICAnQXJuJ1xuICAgICAgICAgICAgICAgICAgICApLnRvU3RyaW5nKCksXG4gICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgaXBTZXRSZWZlcmVuY2VTdGF0ZW1lbnQ6IHtcbiAgICAgICAgICAgICAgICAgICAgYXJuOiBjZGsuRm4uZ2V0QXR0KFxuICAgICAgICAgICAgICAgICAgICAgIGJsYWNrbGlzdElwU2V0VjYubG9naWNhbElkLFxuICAgICAgICAgICAgICAgICAgICAgICdBcm4nXG4gICAgICAgICAgICAgICAgICAgICkudG9TdHJpbmcoKSxcbiAgICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgfSxcbiAgICAgICAgfSxcbiAgICAgICAge1xuICAgICAgICAgIG5hbWU6IGh0dHBGbG9vZFJlZ3VsYXJSdWxlTmFtZSxcbiAgICAgICAgICBwcmlvcml0eTogMyxcbiAgICAgICAgICBhY3Rpb246ICFjb3VudE1vZGVcbiAgICAgICAgICAgID8ge1xuICAgICAgICAgICAgICBibG9jazoge30sXG4gICAgICAgICAgICB9XG4gICAgICAgICAgICA6IHsgY291bnQ6IHt9IH0sXG4gICAgICAgICAgdmlzaWJpbGl0eUNvbmZpZzoge1xuICAgICAgICAgICAgc2FtcGxlZFJlcXVlc3RzRW5hYmxlZDogdHJ1ZSxcbiAgICAgICAgICAgIGNsb3VkV2F0Y2hNZXRyaWNzRW5hYmxlZDogdHJ1ZSxcbiAgICAgICAgICAgIG1ldHJpY05hbWU6IGh0dHBGbG9vZFJlZ3VsYXJSdWxlTmFtZSxcbiAgICAgICAgICB9LFxuICAgICAgICAgIHN0YXRlbWVudDoge1xuICAgICAgICAgICAgb3JTdGF0ZW1lbnQ6IHtcbiAgICAgICAgICAgICAgc3RhdGVtZW50czogW1xuICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgIGlwU2V0UmVmZXJlbmNlU3RhdGVtZW50OiB7XG4gICAgICAgICAgICAgICAgICAgIGFybjogY2RrLkZuLmdldEF0dChcbiAgICAgICAgICAgICAgICAgICAgICBodHRwRmxvb2RJcFNldFY0LmxvZ2ljYWxJZCxcbiAgICAgICAgICAgICAgICAgICAgICAnQXJuJ1xuICAgICAgICAgICAgICAgICAgICApLnRvU3RyaW5nKCksXG4gICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgaXBTZXRSZWZlcmVuY2VTdGF0ZW1lbnQ6IHtcbiAgICAgICAgICAgICAgICAgICAgYXJuOiBjZGsuRm4uZ2V0QXR0KFxuICAgICAgICAgICAgICAgICAgICAgIGh0dHBGbG9vZElwU2V0VjYubG9naWNhbElkLFxuICAgICAgICAgICAgICAgICAgICAgICdBcm4nXG4gICAgICAgICAgICAgICAgICAgICkudG9TdHJpbmcoKSxcbiAgICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgfSxcbiAgICAgICAgfSxcbiAgICAgICAge1xuICAgICAgICAgIG5hbWU6IHNjYW5uZXJzUHJvYmVzUnVsZU5hbWUsXG4gICAgICAgICAgcHJpb3JpdHk6IDUsXG4gICAgICAgICAgYWN0aW9uOiAhY291bnRNb2RlXG4gICAgICAgICAgICA/IHtcbiAgICAgICAgICAgICAgYmxvY2s6IHt9LFxuICAgICAgICAgICAgfVxuICAgICAgICAgICAgOiB7IGNvdW50OiB7fSB9LFxuXG4gICAgICAgICAgdmlzaWJpbGl0eUNvbmZpZzoge1xuICAgICAgICAgICAgc2FtcGxlZFJlcXVlc3RzRW5hYmxlZDogdHJ1ZSxcbiAgICAgICAgICAgIGNsb3VkV2F0Y2hNZXRyaWNzRW5hYmxlZDogdHJ1ZSxcbiAgICAgICAgICAgIG1ldHJpY05hbWU6IHNjYW5uZXJzUHJvYmVzUnVsZU5hbWUsXG4gICAgICAgICAgfSxcbiAgICAgICAgICBzdGF0ZW1lbnQ6IHtcbiAgICAgICAgICAgIG9yU3RhdGVtZW50OiB7XG4gICAgICAgICAgICAgIHN0YXRlbWVudHM6IFtcbiAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICBpcFNldFJlZmVyZW5jZVN0YXRlbWVudDoge1xuICAgICAgICAgICAgICAgICAgICBhcm46IGNkay5Gbi5nZXRBdHQoXG4gICAgICAgICAgICAgICAgICAgICAgc2Nhbm5lcnNQcm9iZXNJcFNldFY0LmxvZ2ljYWxJZCxcbiAgICAgICAgICAgICAgICAgICAgICAnQXJuJ1xuICAgICAgICAgICAgICAgICAgICApLnRvU3RyaW5nKCksXG4gICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgaXBTZXRSZWZlcmVuY2VTdGF0ZW1lbnQ6IHtcbiAgICAgICAgICAgICAgICAgICAgYXJuOiBjZGsuRm4uZ2V0QXR0KFxuICAgICAgICAgICAgICAgICAgICAgIHNjYW5uZXJzUHJvYmVzSXBTZXRWNi5sb2dpY2FsSWQsXG4gICAgICAgICAgICAgICAgICAgICAgJ0FybidcbiAgICAgICAgICAgICAgICAgICAgKS50b1N0cmluZygpLFxuICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICB9LFxuICAgICAgICB9LFxuICAgICAgICB7XG4gICAgICAgICAgbmFtZTogcmVwdXRhdGlvbkxpc3ROYW1lLFxuICAgICAgICAgIHByaW9yaXR5OiA2LFxuICAgICAgICAgIGFjdGlvbjogIWNvdW50TW9kZVxuICAgICAgICAgICAgPyB7XG4gICAgICAgICAgICAgIGJsb2NrOiB7fSxcbiAgICAgICAgICAgIH1cbiAgICAgICAgICAgIDogeyBjb3VudDoge30gfSxcbiAgICAgICAgICB2aXNpYmlsaXR5Q29uZmlnOiB7XG4gICAgICAgICAgICBzYW1wbGVkUmVxdWVzdHNFbmFibGVkOiB0cnVlLFxuICAgICAgICAgICAgY2xvdWRXYXRjaE1ldHJpY3NFbmFibGVkOiB0cnVlLFxuICAgICAgICAgICAgbWV0cmljTmFtZTogcmVwdXRhdGlvbkxpc3ROYW1lLFxuICAgICAgICAgIH0sXG4gICAgICAgICAgc3RhdGVtZW50OiB7XG4gICAgICAgICAgICBvclN0YXRlbWVudDoge1xuICAgICAgICAgICAgICBzdGF0ZW1lbnRzOiBbXG4gICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgaXBTZXRSZWZlcmVuY2VTdGF0ZW1lbnQ6IHtcbiAgICAgICAgICAgICAgICAgICAgYXJuOiBjZGsuRm4uZ2V0QXR0KFxuICAgICAgICAgICAgICAgICAgICAgIHJlcHV0YXRpb25MaXN0c0lwU2V0VjQubG9naWNhbElkLFxuICAgICAgICAgICAgICAgICAgICAgICdBcm4nXG4gICAgICAgICAgICAgICAgICAgICkudG9TdHJpbmcoKSxcbiAgICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICBpcFNldFJlZmVyZW5jZVN0YXRlbWVudDoge1xuICAgICAgICAgICAgICAgICAgICBhcm46IGNkay5Gbi5nZXRBdHQoXG4gICAgICAgICAgICAgICAgICAgICAgcmVwdXRhdGlvbkxpc3RzSXBTZXRWNi5sb2dpY2FsSWQsXG4gICAgICAgICAgICAgICAgICAgICAgJ0FybidcbiAgICAgICAgICAgICAgICAgICAgKS50b1N0cmluZygpLFxuICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICB9LFxuICAgICAgICB9LFxuICAgICAgICB7XG4gICAgICAgICAgbmFtZTogc3FsSW5qZWN0aW9uUnVsZU5hbWUsXG4gICAgICAgICAgcHJpb3JpdHk6IDIwLFxuICAgICAgICAgIGFjdGlvbjogIWNvdW50TW9kZVxuICAgICAgICAgICAgPyB7XG4gICAgICAgICAgICAgIGJsb2NrOiB7fSxcbiAgICAgICAgICAgIH1cbiAgICAgICAgICAgIDogeyBjb3VudDoge30gfSxcbiAgICAgICAgICB2aXNpYmlsaXR5Q29uZmlnOiB7XG4gICAgICAgICAgICBzYW1wbGVkUmVxdWVzdHNFbmFibGVkOiB0cnVlLFxuICAgICAgICAgICAgY2xvdWRXYXRjaE1ldHJpY3NFbmFibGVkOiB0cnVlLFxuICAgICAgICAgICAgbWV0cmljTmFtZTogc3FsSW5qZWN0aW9uUnVsZU5hbWUsXG4gICAgICAgICAgfSxcbiAgICAgICAgICBzdGF0ZW1lbnQ6IHtcbiAgICAgICAgICAgIG9yU3RhdGVtZW50OiB7XG4gICAgICAgICAgICAgIHN0YXRlbWVudHM6IFtcbiAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICBzcWxpTWF0Y2hTdGF0ZW1lbnQ6IHtcbiAgICAgICAgICAgICAgICAgICAgZmllbGRUb01hdGNoOiB7XG4gICAgICAgICAgICAgICAgICAgICAgcXVlcnlTdHJpbmc6IHt9LFxuICAgICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgICAgICB0ZXh0VHJhbnNmb3JtYXRpb25zOiBbXG4gICAgICAgICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgICAgICAgcHJpb3JpdHk6IDEsXG4gICAgICAgICAgICAgICAgICAgICAgICB0eXBlOiAnVVJMX0RFQ09ERScsXG4gICAgICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICAgICAgICBwcmlvcml0eTogMixcbiAgICAgICAgICAgICAgICAgICAgICAgIHR5cGU6ICdIVE1MX0VOVElUWV9ERUNPREUnLFxuICAgICAgICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgc3FsaU1hdGNoU3RhdGVtZW50OiB7XG4gICAgICAgICAgICAgICAgICAgIGZpZWxkVG9NYXRjaDoge1xuICAgICAgICAgICAgICAgICAgICAgIGJvZHk6IHt9LFxuICAgICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgICAgICB0ZXh0VHJhbnNmb3JtYXRpb25zOiBbXG4gICAgICAgICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgICAgICAgcHJpb3JpdHk6IDEsXG4gICAgICAgICAgICAgICAgICAgICAgICB0eXBlOiAnVVJMX0RFQ09ERScsXG4gICAgICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICAgICAgICBwcmlvcml0eTogMixcbiAgICAgICAgICAgICAgICAgICAgICAgIHR5cGU6ICdIVE1MX0VOVElUWV9ERUNPREUnLFxuICAgICAgICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgc3FsaU1hdGNoU3RhdGVtZW50OiB7XG4gICAgICAgICAgICAgICAgICAgIGZpZWxkVG9NYXRjaDoge1xuICAgICAgICAgICAgICAgICAgICAgIHVyaVBhdGg6IHt9LFxuICAgICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgICAgICB0ZXh0VHJhbnNmb3JtYXRpb25zOiBbXG4gICAgICAgICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgICAgICAgcHJpb3JpdHk6IDEsXG4gICAgICAgICAgICAgICAgICAgICAgICB0eXBlOiAnVVJMX0RFQ09ERScsXG4gICAgICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICAgICAgICBwcmlvcml0eTogMixcbiAgICAgICAgICAgICAgICAgICAgICAgIHR5cGU6ICdIVE1MX0VOVElUWV9ERUNPREUnLFxuICAgICAgICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgc3FsaU1hdGNoU3RhdGVtZW50OiB7XG4gICAgICAgICAgICAgICAgICAgIGZpZWxkVG9NYXRjaDoge1xuICAgICAgICAgICAgICAgICAgICAgIHNpbmdsZUhlYWRlcjogeyBOYW1lOiAnQXV0aG9yaXphdGlvbicgfSxcbiAgICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICAgICAgdGV4dFRyYW5zZm9ybWF0aW9uczogW1xuICAgICAgICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgICAgICAgIHByaW9yaXR5OiAxLFxuICAgICAgICAgICAgICAgICAgICAgICAgdHlwZTogJ1VSTF9ERUNPREUnLFxuICAgICAgICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgICAgICAgcHJpb3JpdHk6IDIsXG4gICAgICAgICAgICAgICAgICAgICAgICB0eXBlOiAnSFRNTF9FTlRJVFlfREVDT0RFJyxcbiAgICAgICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgIHNxbGlNYXRjaFN0YXRlbWVudDoge1xuICAgICAgICAgICAgICAgICAgICBmaWVsZFRvTWF0Y2g6IHtcbiAgICAgICAgICAgICAgICAgICAgICBzaW5nbGVIZWFkZXI6IHsgTmFtZTogJ0Nvb2tpZScgfSxcbiAgICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICAgICAgdGV4dFRyYW5zZm9ybWF0aW9uczogW1xuICAgICAgICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgICAgICAgIHByaW9yaXR5OiAxLFxuICAgICAgICAgICAgICAgICAgICAgICAgdHlwZTogJ1VSTF9ERUNPREUnLFxuICAgICAgICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgICAgICAgcHJpb3JpdHk6IDIsXG4gICAgICAgICAgICAgICAgICAgICAgICB0eXBlOiAnSFRNTF9FTlRJVFlfREVDT0RFJyxcbiAgICAgICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICB9LFxuICAgICAgICB9LFxuICAgICAgICB7XG4gICAgICAgICAgbmFtZTogeHNzUnVsZU5hbWUsXG4gICAgICAgICAgcHJpb3JpdHk6IDMwLFxuICAgICAgICAgIGFjdGlvbjogIWNvdW50TW9kZVxuICAgICAgICAgICAgPyB7XG4gICAgICAgICAgICAgIGJsb2NrOiB7fSxcbiAgICAgICAgICAgIH1cbiAgICAgICAgICAgIDogeyBjb3VudDoge30gfSxcbiAgICAgICAgICB2aXNpYmlsaXR5Q29uZmlnOiB7XG4gICAgICAgICAgICBzYW1wbGVkUmVxdWVzdHNFbmFibGVkOiB0cnVlLFxuICAgICAgICAgICAgY2xvdWRXYXRjaE1ldHJpY3NFbmFibGVkOiB0cnVlLFxuICAgICAgICAgICAgbWV0cmljTmFtZTogeHNzUnVsZU5hbWUsXG4gICAgICAgICAgfSxcbiAgICAgICAgICBzdGF0ZW1lbnQ6IHtcbiAgICAgICAgICAgIG9yU3RhdGVtZW50OiB7XG4gICAgICAgICAgICAgIHN0YXRlbWVudHM6IFtcbiAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICB4c3NNYXRjaFN0YXRlbWVudDoge1xuICAgICAgICAgICAgICAgICAgICBmaWVsZFRvTWF0Y2g6IHtcbiAgICAgICAgICAgICAgICAgICAgICBxdWVyeVN0cmluZzoge30sXG4gICAgICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgICAgIHRleHRUcmFuc2Zvcm1hdGlvbnM6IFtcbiAgICAgICAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICAgICAgICBwcmlvcml0eTogMSxcbiAgICAgICAgICAgICAgICAgICAgICAgIHR5cGU6ICdVUkxfREVDT0RFJyxcbiAgICAgICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgICAgICAgIHByaW9yaXR5OiAyLFxuICAgICAgICAgICAgICAgICAgICAgICAgdHlwZTogJ0hUTUxfRU5USVRZX0RFQ09ERScsXG4gICAgICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICB4c3NNYXRjaFN0YXRlbWVudDoge1xuICAgICAgICAgICAgICAgICAgICBmaWVsZFRvTWF0Y2g6IHtcbiAgICAgICAgICAgICAgICAgICAgICBib2R5OiB7fSxcbiAgICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICAgICAgdGV4dFRyYW5zZm9ybWF0aW9uczogW1xuICAgICAgICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgICAgICAgIHByaW9yaXR5OiAxLFxuICAgICAgICAgICAgICAgICAgICAgICAgdHlwZTogJ1VSTF9ERUNPREUnLFxuICAgICAgICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgICAgICAgcHJpb3JpdHk6IDIsXG4gICAgICAgICAgICAgICAgICAgICAgICB0eXBlOiAnSFRNTF9FTlRJVFlfREVDT0RFJyxcbiAgICAgICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgIHhzc01hdGNoU3RhdGVtZW50OiB7XG4gICAgICAgICAgICAgICAgICAgIGZpZWxkVG9NYXRjaDoge1xuICAgICAgICAgICAgICAgICAgICAgIHVyaVBhdGg6IHt9LFxuICAgICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgICAgICB0ZXh0VHJhbnNmb3JtYXRpb25zOiBbXG4gICAgICAgICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgICAgICAgcHJpb3JpdHk6IDEsXG4gICAgICAgICAgICAgICAgICAgICAgICB0eXBlOiAnVVJMX0RFQ09ERScsXG4gICAgICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICAgICAgICBwcmlvcml0eTogMixcbiAgICAgICAgICAgICAgICAgICAgICAgIHR5cGU6ICdIVE1MX0VOVElUWV9ERUNPREUnLFxuICAgICAgICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgeHNzTWF0Y2hTdGF0ZW1lbnQ6IHtcbiAgICAgICAgICAgICAgICAgICAgZmllbGRUb01hdGNoOiB7XG4gICAgICAgICAgICAgICAgICAgICAgc2luZ2xlSGVhZGVyOiB7XG4gICAgICAgICAgICAgICAgICAgICAgICBuYW1lOiAnQ29va2llJyxcbiAgICAgICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgICAgICB0ZXh0VHJhbnNmb3JtYXRpb25zOiBbXG4gICAgICAgICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgICAgICAgcHJpb3JpdHk6IDEsXG4gICAgICAgICAgICAgICAgICAgICAgICB0eXBlOiAnVVJMX0RFQ09ERScsXG4gICAgICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICAgICAgICBwcmlvcml0eTogMixcbiAgICAgICAgICAgICAgICAgICAgICAgIHR5cGU6ICdIVE1MX0VOVElUWV9ERUNPREUnLFxuICAgICAgICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIH0sXG4gICAgICAgIH0sXG4gICAgICAgIC4uLmJhZEJvdFdhZlJ1bGVzLFxuICAgICAgXSxcbiAgICB9KTtcblxuICAgIGlmIChwcm9wcy5hc3NvY2lhdGVkUmVzb3VyY2VBcm4pIHtcbiAgICAgIG5ldyB3YWZ2Mi5DZm5XZWJBQ0xBc3NvY2lhdGlvbih0aGlzLCAnd2FmQXNzb2NpYXRpb24nLCB7XG4gICAgICAgIHdlYkFjbEFybjogd2Fmd2ViLmF0dHJBcm4sXG4gICAgICAgIHJlc291cmNlQXJuOiBwcm9wcy5hc3NvY2lhdGVkUmVzb3VyY2VBcm4sXG4gICAgICB9KTtcbiAgICB9XG5cbiAgICAvL0xhbWJkYVxuICAgIG5ldyBsYW1iZGEuRG9ja2VySW1hZ2VGdW5jdGlvbih0aGlzLCAnSGVscGVyJywge1xuICAgICAgZGVzY3JpcHRpb246XG4gICAgICAgIFwiVGhpcyBsYW1iZGEgZnVuY3Rpb24gdmVyaWZpZXMgdGhlIG1haW4gcHJvamVjdCdzIGRlcGVuZGVuY2llcywgcmVxdWlyZW1lbnRzIGFuZCBpbXBsZW1lbnQgYXV4aWxpYXJ5IGZ1bmN0aW9ucy5cIixcbiAgICAgIGNvZGU6IGxhbWJkYS5Eb2NrZXJJbWFnZUNvZGUuZnJvbUltYWdlQXNzZXQoXG4gICAgICAgIHBhdGguam9pbihfX2Rpcm5hbWUsICcuLi9zb3VyY2UvaGVscGVyJyksXG4gICAgICAgIHtcbiAgICAgICAgICBjbWQ6IFsnaGVscGVyLmxhbWJkYV9oYW5kbGVyJ10sXG4gICAgICAgIH1cbiAgICAgICksXG4gICAgICBtZW1vcnlTaXplOiA1MTIsXG4gICAgICB0aW1lb3V0OiBjZGsuRHVyYXRpb24uc2Vjb25kcygzMDApLFxuICAgICAgZW52aXJvbm1lbnQ6IHtcbiAgICAgICAgTE9HX0xFVkVMOiBsb2dMZXZlbCxcbiAgICAgICAgU0NPUEU6IHByb3BzLndhZjJTY29wZSxcbiAgICAgIH0sXG4gICAgfSk7XG5cbiAgICBjb25zdCBsb2dQYXJzZXJMYW1iZGEgPSBuZXcgbGFtYmRhLkRvY2tlckltYWdlRnVuY3Rpb24odGhpcywgJ0xvZ1BhcnNlcicsIHtcbiAgICAgIGRlc2NyaXB0aW9uOlxuICAgICAgICAnVGhpcyBmdW5jdGlvbiBwYXJzZXMgYWNjZXNzIGxvZ3MgdG8gaWRlbnRpZnkgc3VzcGljaW91cyBiZWhhdmlvciwgc3VjaCBhcyBhbiBhYm5vcm1hbCBhbW91bnQgb2YgZXJyb3JzLiBJdCB0aGVuIGJsb2NrcyB0aG9zZSBJUCBhZGRyZXNzZXMgZm9yIGEgY3VzdG9tZXItZGVmaW5lZCBwZXJpb2Qgb2YgdGltZS4nLFxuICAgICAgY29kZTogbGFtYmRhLkRvY2tlckltYWdlQ29kZS5mcm9tSW1hZ2VBc3NldChcbiAgICAgICAgcGF0aC5qb2luKF9fZGlybmFtZSwgJy4uL3NvdXJjZS9sb2dfcGFyc2VyJyksXG4gICAgICAgIHtcbiAgICAgICAgICBjbWQ6IFsnbG9nLXBhcnNlci5sYW1iZGFfaGFuZGxlciddLFxuICAgICAgICB9XG4gICAgICApLFxuICAgICAgbWVtb3J5U2l6ZTogNTEyLFxuICAgICAgdGltZW91dDogY2RrLkR1cmF0aW9uLnNlY29uZHMoMzAwKSxcbiAgICAgIGVudmlyb25tZW50OiB7XG4gICAgICAgIEFQUF9BQ0NFU1NfTE9HX0JVQ0tFVDogYWNjZXNzTG9nQnVja2V0LmJ1Y2tldE5hbWUsXG4gICAgICAgIFdBRl9BQ0NFU1NfTE9HX0JVQ0tFVDogd2FmTG9nQnVja2V0LmJ1Y2tldE5hbWUsXG4gICAgICAgIFNFTkRfQU5PTllNT1VTX1VTQUdFX0RBVEE6ICdObycsXG4gICAgICAgIFVVSUQ6ICdDcmVhdGVVbmlxdWVJRC5VVUlEJyxcbiAgICAgICAgTElNSVRfSVBfQUREUkVTU19SQU5HRVNfUEVSX0lQX01BVENIX0NPTkRJVElPTjogJzEwMDAwJyxcbiAgICAgICAgTUFYX0FHRV9UT19VUERBVEU6ICczMCcsXG4gICAgICAgIFJFR0lPTjogJ0FXUzo6UmVnaW9uJyxcbiAgICAgICAgU0NPUEU6IHByb3BzLndhZjJTY29wZSxcbiAgICAgICAgTE9HX1RZUEU6XG4gICAgICAgICAgcHJvcHMud2FmMlNjb3BlID09IFdhZjJTY29wZU9wdGlvbi5DTE9VREZST05UID8gJ2Nsb3VkZnJvbnQnIDogJ2FsYicsXG4gICAgICAgIE1FVFJJQ19OQU1FX1BSRUZJWDogY2RrLkZuLnJlZignQVdTOjpTdGFja05hbWUnKSxcbiAgICAgICAgTE9HX0xFVkVMOiBsb2dMZXZlbCxcbiAgICAgICAgU1RBQ0tfTkFNRTogY2RrLkZuLnJlZignQVdTOjpTdGFja05hbWUnKSxcbiAgICAgICAgSVBfU0VUX0lEX0hUVFBfRkxPT0RWNDogaHR0cEZsb29kSXBTZXRWNC5hdHRyQXJuLFxuICAgICAgICBJUF9TRVRfSURfSFRUUF9GTE9PRFY2OiBodHRwRmxvb2RJcFNldFY2LmF0dHJBcm4sXG4gICAgICAgIElQX1NFVF9OQU1FX0hUVFBfRkxPT0RWNDogaHR0cEZsb29kSXBTZXRWNC5uYW1lISxcbiAgICAgICAgSVBfU0VUX05BTUVfSFRUUF9GTE9PRFY2OiBodHRwRmxvb2RJcFNldFY2Lm5hbWUhLFxuICAgICAgICBJUF9TRVRfSURfU0NBTk5FUlNfUFJPQkVTVjQ6IHNjYW5uZXJzUHJvYmVzSXBTZXRWNC5hdHRyQXJuLFxuICAgICAgICBJUF9TRVRfSURfU0NBTk5FUlNfUFJPQkVTVjY6IHNjYW5uZXJzUHJvYmVzSXBTZXRWNi5hdHRyQXJuLFxuICAgICAgICBJUF9TRVRfTkFNRV9TQ0FOTkVSU19QUk9CRVNWNDogc2Nhbm5lcnNQcm9iZXNJcFNldFY0Lm5hbWUhLFxuICAgICAgICBJUF9TRVRfTkFNRV9TQ0FOTkVSU19QUk9CRVNWNjogc2Nhbm5lcnNQcm9iZXNJcFNldFY2Lm5hbWUhLFxuICAgICAgICBXQUZfQkxPQ0tfUEVSSU9EOiBTdHJpbmcoYmxvY2tQZXJpb2QpLFxuICAgICAgICBFUlJPUl9USFJFU0hPTEQ6IFN0cmluZyhlcnJvclRocmVzaG9sZCksXG4gICAgICAgIFJFUVVFU1RfVEhSRVNIT0xEOiBTdHJpbmcocmVxdWVzdFRocmVzaG9sZCksXG4gICAgICAgIFNPTFVUSU9OX0lEOiAnU084MTI4JyxcbiAgICAgICAgTUVUUklDU19VUkw6ICdodHRwczovL21ldHJpY3MuYXdzc29sdXRpb25zYnVpbGRlci5jb20vZ2VuZXJpYycsXG4gICAgICB9LFxuICAgIH0pO1xuXG4gICAgY29uc3QgbW92ZUxvZ1RvUGFydGl0aW9uTGFtYmRhID0gbmV3IGxhbWJkYS5Eb2NrZXJJbWFnZUZ1bmN0aW9uKFxuICAgICAgdGhpcyxcbiAgICAgICdNb3ZlUzNMb2dzRm9yUGFydGl0aW9uJyxcbiAgICAgIHtcbiAgICAgICAgZGVzY3JpcHRpb246XG4gICAgICAgICAgJ1RoaXMgZnVuY3Rpb24gaXMgdHJpZ2dlcmVkIGJ5IFMzIGV2ZW50IHRvIG1vdmUgbG9nIGZpbGVzKHVwb24gdGhlaXIgYXJyaXZhbCBpbiBzMykgZnJvbSB0aGVpciBvcmlnaW5hbCBsb2NhdGlvbiB0byBhIHBhcnRpdGlvbmVkIGZvbGRlciBzdHJ1Y3R1cmUgY3JlYXRlZCBwZXIgdGltZXN0YW1wcyBpbiBmaWxlIG5hbWVzLCBoZW5jZSBhbGxvd2luZyB0aGUgdXNhZ2Ugb2YgcGFydGl0aW9uaW5nIHdpdGhpbiBBV1MgQXRoZW5hLicsXG4gICAgICAgIGNvZGU6IGxhbWJkYS5Eb2NrZXJJbWFnZUNvZGUuZnJvbUltYWdlQXNzZXQoXG4gICAgICAgICAgcGF0aC5qb2luKF9fZGlybmFtZSwgJy4uL3NvdXJjZS9sb2dfcGFyc2VyJyksXG4gICAgICAgICAge1xuICAgICAgICAgICAgY21kOiBbJ3BhcnRpdGlvbl9zM19sb2dzLmxhbWJkYV9oYW5kbGVyJ10sXG4gICAgICAgICAgfVxuICAgICAgICApLFxuICAgICAgICBtZW1vcnlTaXplOiA1MTIsXG4gICAgICAgIHRpbWVvdXQ6IGNkay5EdXJhdGlvbi5zZWNvbmRzKDMwMCksXG4gICAgICAgIGVudmlyb25tZW50OiB7XG4gICAgICAgICAgRU5EUE9JTlQ6XG4gICAgICAgICAgICBwcm9wcy53YWYyU2NvcGUgPT0gV2FmMlNjb3BlT3B0aW9uLkNMT1VERlJPTlRcbiAgICAgICAgICAgICAgPyAnQ2xvdWRGcm9udCdcbiAgICAgICAgICAgICAgOiAnQWxiJyxcbiAgICAgICAgICBLRUVQX09SSUdJTkFMX0RBVEE6ICdObycsXG4gICAgICAgICAgTE9HX0xFVkVMOiBsb2dMZXZlbCxcbiAgICAgICAgfSxcbiAgICAgIH1cbiAgICApO1xuXG4gICAgbmV3IGxhbWJkYS5Eb2NrZXJJbWFnZUZ1bmN0aW9uKHRoaXMsICdDdXN0b21UaW1lcicsIHtcbiAgICAgIGRlc2NyaXB0aW9uOlxuICAgICAgICAnVGhpcyBsYW1iZGEgZnVuY3Rpb24gY291bnRzIFggc2Vjb25kcyBhbmQgY2FuIGJlIHVzZWQgdG8gc2xvdyBkb3duIGNvbXBvbmVudCBjcmVhdGlvbiBpbiBDbG91ZEZvcm1hdGlvbicsXG4gICAgICBjb2RlOiBsYW1iZGEuRG9ja2VySW1hZ2VDb2RlLmZyb21JbWFnZUFzc2V0KFxuICAgICAgICBwYXRoLmpvaW4oX19kaXJuYW1lLCAnLi4vc291cmNlL3RpbWVyJyksXG4gICAgICAgIHtcbiAgICAgICAgICBjbWQ6IFsndGltZXIubGFtYmRhX2hhbmRsZXInXSxcbiAgICAgICAgfVxuICAgICAgKSxcbiAgICAgIG1lbW9yeVNpemU6IDEyOCxcbiAgICAgIHRpbWVvdXQ6IGNkay5EdXJhdGlvbi5zZWNvbmRzKDMwMCksXG4gICAgICBlbnZpcm9ubWVudDoge1xuICAgICAgICBMT0dfTEVWRUw6IGxvZ0xldmVsLFxuICAgICAgICBTRUNPTkRTOiAnMicsXG4gICAgICB9LFxuICAgIH0pO1xuXG4gICAgLy9JUCByZXB1dGF0aW9uIGxpc3QgTGFtYmRhXG4gICAgY29uc3QgcmVwdXRhdGlvbkxpc3RSb2xlID0gbmV3IGlhbS5Sb2xlKFxuICAgICAgdGhpcyxcbiAgICAgICdMYW1iZGFSb2xlUmVwdXRhdGlvbkxpc3RzUGFyc2VyJyxcbiAgICAgIHtcbiAgICAgICAgYXNzdW1lZEJ5OiBuZXcgaWFtLlNlcnZpY2VQcmluY2lwYWwoJ2xhbWJkYS5hbWF6b25hd3MuY29tJyksXG4gICAgICB9XG4gICAgKTtcbiAgICByZXB1dGF0aW9uTGlzdFJvbGUuYXR0YWNoSW5saW5lUG9saWN5KFxuICAgICAgbmV3IGlhbS5Qb2xpY3kodGhpcywgJ0Nsb3VkV2F0Y2hMb2dzJywge1xuICAgICAgICBwb2xpY3lOYW1lOiAnQ2xvdWRXYXRjaExvZ3MnLFxuICAgICAgICBzdGF0ZW1lbnRzOiBbXG4gICAgICAgICAgbmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgICAgICAgZWZmZWN0OiBpYW0uRWZmZWN0LkFMTE9XLFxuICAgICAgICAgICAgcmVzb3VyY2VzOiBbJyonXSxcbiAgICAgICAgICAgIGFjdGlvbnM6IFtcbiAgICAgICAgICAgICAgJ2xvZ3M6Q3JlYXRlTG9nR3JvdXAnLFxuICAgICAgICAgICAgICAnbG9nczpDcmVhdGVMb2dTdHJlYW0nLFxuICAgICAgICAgICAgICAnbG9nczpQdXRMb2dFdmVudHMnLFxuICAgICAgICAgICAgXSxcbiAgICAgICAgICB9KSxcbiAgICAgICAgXSxcbiAgICAgIH0pXG4gICAgKTtcbiAgICByZXB1dGF0aW9uTGlzdFJvbGUuYXR0YWNoSW5saW5lUG9saWN5KFxuICAgICAgbmV3IGlhbS5Qb2xpY3kodGhpcywgJ1dBRkdldEFuZFVwZGF0ZUlQU2V0Jywge1xuICAgICAgICBwb2xpY3lOYW1lOiAnV0FGR2V0QW5kVXBkYXRlSVBTZXQnLFxuICAgICAgICBzdGF0ZW1lbnRzOiBbXG4gICAgICAgICAgbmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgICAgICAgZWZmZWN0OiBpYW0uRWZmZWN0LkFMTE9XLFxuICAgICAgICAgICAgcmVzb3VyY2VzOiBbJyonXSxcbiAgICAgICAgICAgIGFjdGlvbnM6IFsnd2FmdjI6R2V0SVBTZXQnLCAnd2FmdjI6VXBkYXRlSVBTZXQnXSxcbiAgICAgICAgICB9KSxcbiAgICAgICAgXSxcbiAgICAgIH0pXG4gICAgKTtcbiAgICByZXB1dGF0aW9uTGlzdFJvbGUuYXR0YWNoSW5saW5lUG9saWN5KFxuICAgICAgbmV3IGlhbS5Qb2xpY3kodGhpcywgJ0Nsb3VkRm9ybWF0aW9uQWNjZXNzJywge1xuICAgICAgICBwb2xpY3lOYW1lOiAnQ2xvdWRGb3JtYXRpb25BY2Nlc3MnLFxuICAgICAgICBzdGF0ZW1lbnRzOiBbXG4gICAgICAgICAgbmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgICAgICAgZWZmZWN0OiBpYW0uRWZmZWN0LkFMTE9XLFxuICAgICAgICAgICAgcmVzb3VyY2VzOiBbJyonXSxcbiAgICAgICAgICAgIGFjdGlvbnM6IFsnY2xvdWRmb3JtYXRpb246RGVzY3JpYmVTdGFja3MnXSxcbiAgICAgICAgICB9KSxcbiAgICAgICAgXSxcbiAgICAgIH0pXG4gICAgKTtcbiAgICByZXB1dGF0aW9uTGlzdFJvbGUuYXR0YWNoSW5saW5lUG9saWN5KFxuICAgICAgbmV3IGlhbS5Qb2xpY3kodGhpcywgJ0Nsb3VkV2F0Y2hBY2Nlc3MnLCB7XG4gICAgICAgIHBvbGljeU5hbWU6ICdDbG91ZFdhdGNoQWNjZXNzJyxcbiAgICAgICAgc3RhdGVtZW50czogW1xuICAgICAgICAgIG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgICAgIGVmZmVjdDogaWFtLkVmZmVjdC5BTExPVyxcbiAgICAgICAgICAgIHJlc291cmNlczogWycqJ10sXG4gICAgICAgICAgICBhY3Rpb25zOiBbJ2Nsb3Vkd2F0Y2g6R2V0TWV0cmljU3RhdGlzdGljcyddLFxuICAgICAgICAgIH0pLFxuICAgICAgICBdLFxuICAgICAgfSlcbiAgICApO1xuXG4gICAgY29uc3QgcmVwdXRhdGlvbkxpc3RzUGFyc2VyTGFtYmRhID0gbmV3IGxhbWJkYS5Eb2NrZXJJbWFnZUZ1bmN0aW9uKFxuICAgICAgdGhpcyxcbiAgICAgICdSZXB1dGF0aW9uTGlzdHNQYXJzZXInLFxuICAgICAge1xuICAgICAgICBkZXNjcmlwdGlvbjpcbiAgICAgICAgICAnVGhpcyBsYW1iZGEgZnVuY3Rpb24gY2hlY2tzIHRoaXJkLXBhcnR5IElQIHJlcHV0YXRpb24gbGlzdHMgaG91cmx5IGZvciBuZXcgSVAgcmFuZ2VzIHRvIGJsb2NrLiBUaGVzZSBsaXN0cyBpbmNsdWRlIHRoZSBTcGFtaGF1cyBEb250IFJvdXRlIE9yIFBlZXIgKERST1ApIGFuZCBFeHRlbmRlZCBEcm9wIChFRFJPUCkgbGlzdHMsIHRoZSBQcm9vZnBvaW50IEVtZXJnaW5nIFRocmVhdHMgSVAgbGlzdCwgYW5kIHRoZSBUb3IgZXhpdCBub2RlIGxpc3QuJyxcbiAgICAgICAgY29kZTogbGFtYmRhLkRvY2tlckltYWdlQ29kZS5mcm9tSW1hZ2VBc3NldChcbiAgICAgICAgICBwYXRoLmpvaW4oX19kaXJuYW1lLCAnLi4vc291cmNlL3JlcHV0YXRpb25fbGlzdHNfcGFyc2VyJyksXG4gICAgICAgICAge1xuICAgICAgICAgICAgY21kOiBbJ3JlcHV0YXRpb24tbGlzdHMubGFtYmRhX2hhbmRsZXInXSxcbiAgICAgICAgICB9XG4gICAgICAgICksXG4gICAgICAgIG1lbW9yeVNpemU6IDUxMixcbiAgICAgICAgdGltZW91dDogY2RrLkR1cmF0aW9uLnNlY29uZHMoMzAwKSxcbiAgICAgICAgcm9sZTogcmVwdXRhdGlvbkxpc3RSb2xlLFxuICAgICAgICBlbnZpcm9ubWVudDoge1xuICAgICAgICAgIElQX1NFVF9JRF9SRVBVVEFUSU9OVjQ6IHJlcHV0YXRpb25MaXN0c0lwU2V0VjQuYXR0ckFybixcbiAgICAgICAgICBJUF9TRVRfSURfUkVQVVRBVElPTlY2OiByZXB1dGF0aW9uTGlzdHNJcFNldFY2LmF0dHJBcm4sXG4gICAgICAgICAgSVBfU0VUX05BTUVfUkVQVVRBVElPTlY0OiByZXB1dGF0aW9uTGlzdHNJcFNldFY0Lm5hbWUhLFxuICAgICAgICAgIElQX1NFVF9OQU1FX1JFUFVUQVRJT05WNjogcmVwdXRhdGlvbkxpc3RzSXBTZXRWNi5uYW1lISxcbiAgICAgICAgICBTQ09QRTogcHJvcHMud2FmMlNjb3BlLFxuICAgICAgICAgIExPR19MRVZFTDogbG9nTGV2ZWwsXG4gICAgICAgICAgVVJMX0xJU1Q6XG4gICAgICAgICAgICAnW3tcInVybFwiOlwiaHR0cHM6Ly93d3cuc3BhbWhhdXMub3JnL2Ryb3AvZHJvcC50eHRcIn0se1widXJsXCI6XCJodHRwczovL3d3dy5zcGFtaGF1cy5vcmcvZHJvcC9lZHJvcC50eHRcIn0se1widXJsXCI6XCJodHRwczovL2NoZWNrLnRvcnByb2plY3Qub3JnL2V4aXQtYWRkcmVzc2VzXCIsIFwicHJlZml4XCI6XCJFeGl0QWRkcmVzc1wifSx7XCJ1cmxcIjpcImh0dHBzOi8vcnVsZXMuZW1lcmdpbmd0aHJlYXRzLm5ldC9md3J1bGVzL2VtZXJnaW5nLUJsb2NrLUlQcy50eHRcIn1dJyxcbiAgICAgICAgICBTT0xVVElPTl9JRDogJ1NPODEyOCcsXG4gICAgICAgICAgTUVUUklDU19VUkw6ICdodHRwczovL21ldHJpY3MuYXdzc29sdXRpb25zYnVpbGRlci5jb20vZ2VuZXJpYycsXG4gICAgICAgICAgU1RBQ0tfTkFNRTogY2RrLkZuLnJlZignQVdTOjpTdGFja05hbWUnKSxcbiAgICAgICAgICBMT0dfVFlQRTpcbiAgICAgICAgICAgIHByb3BzLndhZjJTY29wZSA9PSBXYWYyU2NvcGVPcHRpb24uQ0xPVURGUk9OVFxuICAgICAgICAgICAgICA/ICdjbG91ZGZyb250J1xuICAgICAgICAgICAgICA6ICdhbGInLFxuICAgICAgICAgIFNFTkRfQU5PTllNT1VTX1VTQUdFX0RBVEE6ICdObycsXG4gICAgICAgICAgSVBSRVBVVEFUSU9OTElTVF9NRVRSSUNOQU1FOiByZXB1dGF0aW9uTGlzdE5hbWUsXG4gICAgICAgIH0sXG4gICAgICB9XG4gICAgKTtcblxuICAgIC8vQWRkIENsb3VkV2F0Y2ggZXZlbnQgdG8gTGFtYmRhXG4gICAgY29uc3QgcmVwdXRhdGlvbkxpc3RzUGFyc2VyUnVsZSA9IG5ldyBldmVudHMuUnVsZShcbiAgICAgIHRoaXMsXG4gICAgICAnUmVwdXRhdGlvbkxpc3RzUGFyc2VyRXZlbnRzUnVsZScsXG4gICAgICB7XG4gICAgICAgIGRlc2NyaXB0aW9uOiAnU2VjdXJpdHkgQXV0b21hdGlvbiAtIFdBRiBSZXB1dGF0aW9uIExpc3RzJyxcbiAgICAgICAgc2NoZWR1bGU6IGV2ZW50cy5TY2hlZHVsZS5leHByZXNzaW9uKCdyYXRlKDEgaG91ciknKSxcbiAgICAgIH1cbiAgICApO1xuICAgIGNvbnN0IHJlcHV0YXRpb25MaXN0c1BhcnNlclJ1bGVJbnB1dCA9IHtcbiAgICAgIFVSTF9MSVNUOiBbXG4gICAgICAgIHtcbiAgICAgICAgICB1cmw6ICdodHRwczovL3d3dy5zcGFtaGF1cy5vcmcvZHJvcC9kcm9wLnR4dCcsXG4gICAgICAgIH0sXG4gICAgICAgIHtcbiAgICAgICAgICB1cmw6ICdodHRwczovL3d3dy5zcGFtaGF1cy5vcmcvZHJvcC9lZHJvcC50eHQnLFxuICAgICAgICB9LFxuICAgICAgICB7XG4gICAgICAgICAgdXJsOiAnaHR0cHM6Ly9jaGVjay50b3Jwcm9qZWN0Lm9yZy9leGl0LWFkZHJlc3NlcycsXG4gICAgICAgICAgcHJlZml4OiAnRXhpdEFkZHJlc3MnLFxuICAgICAgICB9LFxuICAgICAgICB7XG4gICAgICAgICAgdXJsOiAnaHR0cHM6Ly9ydWxlcy5lbWVyZ2luZ3RocmVhdHMubmV0L2Z3cnVsZXMvZW1lcmdpbmctQmxvY2stSVBzLnR4dCcsXG4gICAgICAgIH0sXG4gICAgICBdLFxuICAgICAgSVBfU0VUX0lEX1JFUFVUQVRJT05WNDogcmVwdXRhdGlvbkxpc3RzSXBTZXRWNC5hdHRyQXJuLFxuICAgICAgSVBfU0VUX0lEX1JFUFVUQVRJT05WNjogcmVwdXRhdGlvbkxpc3RzSXBTZXRWNi5hdHRyQXJuLFxuICAgICAgSVBfU0VUX05BTUVfUkVQVVRBVElPTlY0OiByZXB1dGF0aW9uTGlzdHNJcFNldFY0Lm5hbWUhLFxuICAgICAgSVBfU0VUX05BTUVfUkVQVVRBVElPTlY2OiByZXB1dGF0aW9uTGlzdHNJcFNldFY2Lm5hbWUhLFxuICAgICAgU0NPUEU6IHByb3BzLndhZjJTY29wZSxcbiAgICB9O1xuICAgIHJlcHV0YXRpb25MaXN0c1BhcnNlclJ1bGUuYWRkVGFyZ2V0KFxuICAgICAgbmV3IHRhcmdldHMuTGFtYmRhRnVuY3Rpb24ocmVwdXRhdGlvbkxpc3RzUGFyc2VyTGFtYmRhLCB7XG4gICAgICAgIGV2ZW50OiBldmVudHMuUnVsZVRhcmdldElucHV0LmZyb21PYmplY3QoXG4gICAgICAgICAgcmVwdXRhdGlvbkxpc3RzUGFyc2VyUnVsZUlucHV0XG4gICAgICAgICksXG4gICAgICB9KVxuICAgICk7XG5cbiAgICBpZiAoZW5hYmxlU2hpZWxkQWR2YW5jZWRMYW1iZGEpIHtcbiAgICAgIC8vQVdTIFNoaWVsZCBBZHZhbmNlZCBMYW1iZGFcbiAgICAgIGNvbnN0IHNoaWVsZFJvbGUgPSBuZXcgaWFtLlJvbGUodGhpcywgJ1NoaWVsZEFkdmFuY2VSb2xlJywge1xuICAgICAgICBhc3N1bWVkQnk6IG5ldyBpYW0uU2VydmljZVByaW5jaXBhbCgnbGFtYmRhLmFtYXpvbmF3cy5jb20nKSxcbiAgICAgIH0pO1xuICAgICAgc2hpZWxkUm9sZS5hdHRhY2hJbmxpbmVQb2xpY3koXG4gICAgICAgIG5ldyBpYW0uUG9saWN5KHRoaXMsICdTaGllbGRBZHZhbmNlQWNjZXNzJywge1xuICAgICAgICAgIHBvbGljeU5hbWU6ICdDbG91ZEZyb250U2hpZWxkQWR2YW5jZUFjY2VzcycsXG4gICAgICAgICAgc3RhdGVtZW50czogW1xuICAgICAgICAgICAgbmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgICAgICAgICBlZmZlY3Q6IGlhbS5FZmZlY3QuQUxMT1csXG4gICAgICAgICAgICAgIHJlc291cmNlczogWycqJ10sXG4gICAgICAgICAgICAgIGFjdGlvbnM6IFsnc2hpZWxkOkNyZWF0ZSonXSxcbiAgICAgICAgICAgIH0pLFxuICAgICAgICAgIF0sXG4gICAgICAgIH0pXG4gICAgICApO1xuICAgICAgc2hpZWxkUm9sZS5hdHRhY2hJbmxpbmVQb2xpY3koXG4gICAgICAgIG5ldyBpYW0uUG9saWN5KHRoaXMsICdDbG91ZEZyb250RGlzdHJpYnV0aW9uQWNjZXNzJywge1xuICAgICAgICAgIHBvbGljeU5hbWU6ICdDbG91ZEZyb250RGlzdHJpYnV0aW9uQWNjZXNzJyxcbiAgICAgICAgICBzdGF0ZW1lbnRzOiBbXG4gICAgICAgICAgICBuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgICAgICAgIGVmZmVjdDogaWFtLkVmZmVjdC5BTExPVyxcbiAgICAgICAgICAgICAgcmVzb3VyY2VzOiBbJyonXSxcbiAgICAgICAgICAgICAgYWN0aW9uczogWydjbG91ZGZyb250OkdldERpc3RyaWJ1dGlvbiddLFxuICAgICAgICAgICAgfSksXG4gICAgICAgICAgXSxcbiAgICAgICAgfSlcbiAgICAgICk7XG4gICAgICBzaGllbGRSb2xlLmF0dGFjaElubGluZVBvbGljeShcbiAgICAgICAgbmV3IGlhbS5Qb2xpY3kodGhpcywgJ1NoaWVsZExvZ0FjY2VzcycsIHtcbiAgICAgICAgICBwb2xpY3lOYW1lOiAnU2hpZWxkTG9nQWNjZXNzJyxcbiAgICAgICAgICBzdGF0ZW1lbnRzOiBbXG4gICAgICAgICAgICBuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgICAgICAgIGVmZmVjdDogaWFtLkVmZmVjdC5BTExPVyxcbiAgICAgICAgICAgICAgcmVzb3VyY2VzOiBbJyonXSxcbiAgICAgICAgICAgICAgYWN0aW9uczogW1xuICAgICAgICAgICAgICAgICdsb2dzOkNyZWF0ZUxvZ0dyb3VwJyxcbiAgICAgICAgICAgICAgICAnbG9nczpDcmVhdGVMb2dTdHJlYW0nLFxuICAgICAgICAgICAgICAgICdsb2dzOlB1dExvZ0V2ZW50cycsXG4gICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICB9KSxcbiAgICAgICAgICBdLFxuICAgICAgICB9KVxuICAgICAgKTtcblxuICAgICAgbmV3IGxhbWJkYS5Eb2NrZXJJbWFnZUZ1bmN0aW9uKHRoaXMsICdTaGllbGRBZHZhbmNlZExhbWJkYScsIHtcbiAgICAgICAgZGVzY3JpcHRpb246XG4gICAgICAgICAgJ1RoaXMgbGFtYmRhIGZ1bmN0aW9uIGNyZWF0ZSBhbiBBV1MgU2hpZWxkIHJlc291cmNlIHByb3RlY3Rpb24gYW5kIHByb3RlY3Rpb24gZ3JvdXAgZm9yIHRoZSBjbG91ZGZyb250IHJlc291cmNlLicsXG4gICAgICAgIGNvZGU6IGxhbWJkYS5Eb2NrZXJJbWFnZUNvZGUuZnJvbUltYWdlQXNzZXQoXG4gICAgICAgICAgcGF0aC5qb2luKF9fZGlybmFtZSwgJy4uL3NvdXJjZS9zaGllbGRfcHJvdGVjdGlvbicpLFxuICAgICAgICAgIHtcbiAgICAgICAgICAgIGNtZDogWydzaGllbGQtcHJvdGVjdGlvbi5sYW1iZGFfaGFuZGxlciddLFxuICAgICAgICAgIH1cbiAgICAgICAgKSxcbiAgICAgICAgbWVtb3J5U2l6ZTogNTEyLFxuICAgICAgICB0aW1lb3V0OiBjZGsuRHVyYXRpb24uc2Vjb25kcygzMDApLFxuICAgICAgICByb2xlOiBzaGllbGRSb2xlLFxuICAgICAgICBlbnZpcm9ubWVudDoge1xuICAgICAgICAgIFNDT1BFOiBwcm9wcy53YWYyU2NvcGUsXG4gICAgICAgICAgTE9HX0xFVkVMOiBsb2dMZXZlbCxcbiAgICAgICAgICBTT0xVVElPTl9JRDogJ1NPODEyOCcsXG4gICAgICAgIH0sXG4gICAgICB9KTtcbiAgICB9XG5cbiAgICBpZiAocHJvcHMud2FmMlNjb3BlID09IFdhZjJTY29wZU9wdGlvbi5DTE9VREZST05UKSB7XG4gICAgICAvL0JhZGJvdCBwcm90ZWN0aW9uIExhbWJkYVxuICAgICAgY29uc3QgYmFkQm90Um9sZSA9IG5ldyBpYW0uUm9sZSh0aGlzLCAnQmFkQm90Um9sZScsIHtcbiAgICAgICAgYXNzdW1lZEJ5OiBuZXcgaWFtLlNlcnZpY2VQcmluY2lwYWwoJ2xhbWJkYS5hbWF6b25hd3MuY29tJyksXG4gICAgICB9KTtcbiAgICAgIGJhZEJvdFJvbGUuYXR0YWNoSW5saW5lUG9saWN5KFxuICAgICAgICBuZXcgaWFtLlBvbGljeSh0aGlzLCAnQmFkQm90TG9nc0FjY2VzcycsIHtcbiAgICAgICAgICBwb2xpY3lOYW1lOiAnTG9nc0FjY2VzcycsXG4gICAgICAgICAgc3RhdGVtZW50czogW1xuICAgICAgICAgICAgbmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgICAgICAgICBlZmZlY3Q6IGlhbS5FZmZlY3QuQUxMT1csXG4gICAgICAgICAgICAgIHJlc291cmNlczogWycqJ10sXG4gICAgICAgICAgICAgIGFjdGlvbnM6IFtcbiAgICAgICAgICAgICAgICAnbG9nczpDcmVhdGVMb2dHcm91cCcsXG4gICAgICAgICAgICAgICAgJ2xvZ3M6Q3JlYXRlTG9nU3RyZWFtJyxcbiAgICAgICAgICAgICAgICAnbG9nczpQdXRMb2dFdmVudHMnLFxuICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgfSksXG4gICAgICAgICAgXSxcbiAgICAgICAgfSlcbiAgICAgICk7XG4gICAgICBiYWRCb3RSb2xlLmF0dGFjaElubGluZVBvbGljeShcbiAgICAgICAgbmV3IGlhbS5Qb2xpY3kodGhpcywgJ0JhZEJvdENsb3VkRm9ybWF0aW9uQWNjZXNzJywge1xuICAgICAgICAgIHBvbGljeU5hbWU6ICdDbG91ZEZvcm1hdGlvbkFjY2VzcycsXG4gICAgICAgICAgc3RhdGVtZW50czogW1xuICAgICAgICAgICAgbmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgICAgICAgICBlZmZlY3Q6IGlhbS5FZmZlY3QuQUxMT1csXG4gICAgICAgICAgICAgIHJlc291cmNlczogWycqJ10sXG4gICAgICAgICAgICAgIGFjdGlvbnM6IFsnY2xvdWRmb3JtYXRpb246RGVzY3JpYmVTdGFja3MnXSxcbiAgICAgICAgICAgIH0pLFxuICAgICAgICAgIF0sXG4gICAgICAgIH0pXG4gICAgICApO1xuICAgICAgYmFkQm90Um9sZS5hdHRhY2hJbmxpbmVQb2xpY3koXG4gICAgICAgIG5ldyBpYW0uUG9saWN5KHRoaXMsICdCYWRCb3RDbG91ZFdhdGNoQWNjZXNzJywge1xuICAgICAgICAgIHBvbGljeU5hbWU6ICdDbG91ZFdhdGNoQWNjZXNzJyxcbiAgICAgICAgICBzdGF0ZW1lbnRzOiBbXG4gICAgICAgICAgICBuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgICAgICAgIGVmZmVjdDogaWFtLkVmZmVjdC5BTExPVyxcbiAgICAgICAgICAgICAgcmVzb3VyY2VzOiBbJyonXSxcbiAgICAgICAgICAgICAgYWN0aW9uczogWydjbG91ZHdhdGNoOkdldE1ldHJpY1N0YXRpc3RpY3MnXSxcbiAgICAgICAgICAgIH0pLFxuICAgICAgICAgIF0sXG4gICAgICAgIH0pXG4gICAgICApO1xuICAgICAgYmFkQm90Um9sZS5hdHRhY2hJbmxpbmVQb2xpY3koXG4gICAgICAgIG5ldyBpYW0uUG9saWN5KHRoaXMsICdCYWRCb3RXQUZHZXRBbmRVcGRhdGVJUFNldCcsIHtcbiAgICAgICAgICBwb2xpY3lOYW1lOiAnV0FGR2V0QW5kVXBkYXRlSVBTZXQnLFxuICAgICAgICAgIHN0YXRlbWVudHM6IFtcbiAgICAgICAgICAgIG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgICAgICAgZWZmZWN0OiBpYW0uRWZmZWN0LkFMTE9XLFxuICAgICAgICAgICAgICByZXNvdXJjZXM6IFsnKiddLFxuICAgICAgICAgICAgICBhY3Rpb25zOiBbJ3dhZnYyOkdldElQU2V0JywgJ3dhZnYyOlVwZGF0ZUlQU2V0J10sXG4gICAgICAgICAgICB9KSxcbiAgICAgICAgICBdLFxuICAgICAgICB9KVxuICAgICAgKTtcblxuICAgICAgY29uc3QgYmFkQm90UGFyc2VyTGFtYmRhID0gbmV3IGxhbWJkYS5Eb2NrZXJJbWFnZUZ1bmN0aW9uKFxuICAgICAgICB0aGlzLFxuICAgICAgICAnQmFkQm90UGFyc2VyJyxcbiAgICAgICAge1xuICAgICAgICAgIGRlc2NyaXB0aW9uOlxuICAgICAgICAgICAgJ1RoaXMgbGFtYmRhIGZ1bmN0aW9uIHdpbGwgaW50ZXJjZXB0cyBhbmQgaW5zcGVjdHMgdHJhcCBlbmRwb2ludCByZXF1ZXN0cyB0byBleHRyYWN0IGl0cyBJUCBhZGRyZXNzLCBhbmQgdGhlbiBhZGQgaXQgdG8gYW4gQVdTIFdBRiBibG9jayBsaXN0LicsXG4gICAgICAgICAgcm9sZTogYmFkQm90Um9sZSxcbiAgICAgICAgICBjb2RlOiBsYW1iZGEuRG9ja2VySW1hZ2VDb2RlLmZyb21JbWFnZUFzc2V0KFxuICAgICAgICAgICAgcGF0aC5qb2luKF9fZGlybmFtZSwgJy4uL3NvdXJjZS9hY2Nlc3NfaGFuZGxlcicpLFxuICAgICAgICAgICAge1xuICAgICAgICAgICAgICBjbWQ6IFsnYWNjZXNzLWhhbmRsZXIubGFtYmRhX2hhbmRsZXInXSxcbiAgICAgICAgICAgIH1cbiAgICAgICAgICApLFxuICAgICAgICAgIG1lbW9yeVNpemU6IDUxMixcbiAgICAgICAgICB0aW1lb3V0OiBjZGsuRHVyYXRpb24uc2Vjb25kcygzMDApLFxuICAgICAgICAgIGVudmlyb25tZW50OiB7XG4gICAgICAgICAgICBTQ09QRTogcHJvcHMud2FmMlNjb3BlLFxuICAgICAgICAgICAgSVBfU0VUX0lEX0JBRF9CT1RWNDogYmFkQm90SXBTZXRWNC5hdHRyQXJuLFxuICAgICAgICAgICAgSVBfU0VUX0lEX0JBRF9CT1RWNjogYmFkQm90SXBTZXRWNi5hdHRyQXJuLFxuICAgICAgICAgICAgSVBfU0VUX05BTUVfQkFEX0JPVFY0OiBiYWRCb3RJcFNldFY0Lm5hbWUhLFxuICAgICAgICAgICAgSVBfU0VUX05BTUVfQkFEX0JPVFY2OiBiYWRCb3RJcFNldFY2Lm5hbWUhLFxuICAgICAgICAgICAgU0VORF9BTk9OWU1PVVNfVVNBR0VfREFUQTogJ05vJyxcbiAgICAgICAgICAgIFVVSUQ6ICdDcmVhdGVVbmlxdWVJRC5VVUlEJyxcbiAgICAgICAgICAgIFJFR0lPTjogY2RrLkZuLnJlZignQVdTOjpSZWdpb24nKSxcbiAgICAgICAgICAgIExPR19UWVBFOiAnY2xvdWRmcm9udCcsXG4gICAgICAgICAgICBNRVRSSUNfTkFNRV9QUkVGSVg6IGNkay5Gbi5yZWYoJ0FXUzo6U3RhY2tOYW1lJyksXG4gICAgICAgICAgICBMT0dfTEVWRUw6IGxvZ0xldmVsLFxuICAgICAgICAgICAgU09MVVRJT05fSUQ6ICdTTzgxMjgnLFxuICAgICAgICAgICAgTUVUUklDU19VUkw6ICdodHRwczovL21ldHJpY3MuYXdzc29sdXRpb25zYnVpbGRlci5jb20vZ2VuZXJpYycsXG4gICAgICAgICAgICBTVEFDS19OQU1FOiBjZGsuRm4ucmVmKCdBV1M6OlN0YWNrTmFtZScpLFxuICAgICAgICAgIH0sXG4gICAgICAgIH1cbiAgICAgICk7XG5cbiAgICAgIC8vQVBJIEdhdGV3YXkgZm9yIGJhZGJvdCBkZXRlY3Rpb25cbiAgICAgIGNvbnN0IGJhZEJvdEFwaSA9IG5ldyBhcGlnYXRld2F5LlJlc3RBcGkodGhpcywgJ0FwaUdhdGV3YXlCYWRCb3QnLCB7XG4gICAgICAgIHJlc3RBcGlOYW1lOiAnU2VjdXJpdHkgQXV0b21hdGlvbiAtIFdBRiBCYWQgQm90IEFQSScsXG4gICAgICAgIGRlc2NyaXB0aW9uOlxuICAgICAgICAgICdBUEkgY3JlYXRlZCBieSBBV1MgV0FGIFNlY3VyaXR5IEF1dG9tYXRpb24gQ2xvdWRGb3JtYXRpb24gdGVtcGxhdGUuIFRoaXMgZW5kcG9pbnQgd2lsbCBiZSB1c2VkIHRvIGNhcHR1cmUgYmFkIGJvdHMuJyxcbiAgICAgIH0pO1xuICAgICAgY29uc3QgaW50ZWdyYXRpb24gPSBuZXcgYXBpZ2F0ZXdheS5MYW1iZGFJbnRlZ3JhdGlvbihiYWRCb3RQYXJzZXJMYW1iZGEpO1xuICAgICAgYmFkQm90QXBpLnJvb3QuYWRkTWV0aG9kKCdBTlknLCBpbnRlZ3JhdGlvbiwge1xuICAgICAgICByZXF1ZXN0UGFyYW1ldGVyczoge1xuICAgICAgICAgICdtZXRob2QucmVxdWVzdC5oZWFkZXIuWC1Gb3J3YXJkZWQtRm9yJzogZmFsc2UsXG4gICAgICAgIH0sXG4gICAgICB9KTtcbiAgICAgIGNvbnN0IGJhZEJvdFByb3h5ID0gYmFkQm90QXBpLnJvb3QuYWRkUmVzb3VyY2UoJ3twcm94eSt9Jyk7XG4gICAgICBiYWRCb3RQcm94eS5hZGRNZXRob2QoJ0FOWScsIGludGVncmF0aW9uLCB7XG4gICAgICAgIHJlcXVlc3RQYXJhbWV0ZXJzOiB7XG4gICAgICAgICAgJ21ldGhvZC5yZXF1ZXN0LmhlYWRlci5YLUZvcndhcmRlZC1Gb3InOiBmYWxzZSxcbiAgICAgICAgfSxcbiAgICAgIH0pO1xuICAgIH1cblxuICAgIGNvbnN0IGN1c3RvbVJlc291cmNlTGFtYmRhID0gbmV3IGxhbWJkYS5Eb2NrZXJJbWFnZUZ1bmN0aW9uKFxuICAgICAgdGhpcyxcbiAgICAgICdDdXN0b21SZXNvdXJjZScsXG4gICAgICB7XG4gICAgICAgIGRlc2NyaXB0aW9uOlxuICAgICAgICAgICdUaGlzIGxhbWJkYSBmdW5jdGlvbiBjb25maWd1cmVzIHRoZSBXZWIgQUNMIHJ1bGVzIGJhc2VkIG9uIHRoZSBmZWF0dXJlcyBlbmFibGVkIGluIHRoZSBDbG91ZEZvcm1hdGlvbiB0ZW1wbGF0ZS4nLFxuICAgICAgICBjb2RlOiBsYW1iZGEuRG9ja2VySW1hZ2VDb2RlLmZyb21JbWFnZUFzc2V0KFxuICAgICAgICAgIHBhdGguam9pbihfX2Rpcm5hbWUsICcuLi9zb3VyY2UvY3VzdG9tX3Jlc291cmNlJyksXG4gICAgICAgICAge1xuICAgICAgICAgICAgY21kOiBbJ2N1c3RvbS1yZXNvdXJjZS5sYW1iZGFfaGFuZGxlciddLFxuICAgICAgICAgIH1cbiAgICAgICAgKSxcbiAgICAgICAgbWVtb3J5U2l6ZTogNTEyLFxuICAgICAgICB0aW1lb3V0OiBjZGsuRHVyYXRpb24uc2Vjb25kcygzMDApLFxuICAgICAgICBlbnZpcm9ubWVudDoge1xuICAgICAgICAgIExPR19MRVZFTDogbG9nTGV2ZWwsXG4gICAgICAgICAgU0NPUEU6IHByb3BzLndhZjJTY29wZSxcbiAgICAgICAgICBTT0xVVElPTl9JRDogJ1NPODEyOCcsXG4gICAgICAgICAgTUVUUklDU19VUkw6ICdodHRwczovL21ldHJpY3MuYXdzc29sdXRpb25zYnVpbGRlci5jb20vZ2VuZXJpYycsXG4gICAgICAgIH0sXG4gICAgICB9XG4gICAgKTtcblxuICAgIC8vS2luZXNpcyBEYXRhIEZpcmVob3NlXG4gICAgY29uc3QgZmlyZWhvc2VSb2xlID0gbmV3IGlhbS5Sb2xlKHRoaXMsICdGaXJlaG9zZVJvbGUnLCB7XG4gICAgICBhc3N1bWVkQnk6IG5ldyBpYW0uU2VydmljZVByaW5jaXBhbCgnZmlyZWhvc2UuYW1hem9uYXdzLmNvbScpLFxuICAgIH0pO1xuXG4gICAgZmlyZWhvc2VSb2xlLmFkZFRvUG9saWN5KFxuICAgICAgbmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgICBlZmZlY3Q6IGlhbS5FZmZlY3QuQUxMT1csXG4gICAgICAgIGFjdGlvbnM6IFtcbiAgICAgICAgICAna2luZXNpczpEZXNjcmliZVN0cmVhbScsXG4gICAgICAgICAgJ2tpbmVzaXM6R2V0U2hhcmRJdGVyYXRvcicsXG4gICAgICAgICAgJ2tpbmVzaXM6R2V0UmVjb3JkcycsXG4gICAgICAgIF0sXG4gICAgICAgIHJlc291cmNlczogWycqJ10sXG4gICAgICB9KVxuICAgICk7XG5cbiAgICBmaXJlaG9zZVJvbGUuYWRkVG9Qb2xpY3koXG4gICAgICBuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgIGVmZmVjdDogaWFtLkVmZmVjdC5BTExPVyxcbiAgICAgICAgYWN0aW9uczogW1xuICAgICAgICAgICdzMzpBYm9ydE11bHRpcGFydFVwbG9hZCcsXG4gICAgICAgICAgJ3MzOkdldEJ1Y2tldExvY2F0aW9uJyxcbiAgICAgICAgICAnczM6R2V0T2JqZWN0JyxcbiAgICAgICAgICAnczM6TGlzdEJ1Y2tldCcsXG4gICAgICAgICAgJ3MzOkxpc3RCdWNrZXRNdWx0aXBhcnRVcGxvYWRzJyxcbiAgICAgICAgICAnczM6UHV0T2JqZWN0JyxcbiAgICAgICAgXSxcbiAgICAgICAgcmVzb3VyY2VzOiBbd2FmTG9nQnVja2V0LmJ1Y2tldEFybiwgd2FmTG9nQnVja2V0LmFybkZvck9iamVjdHMoJyonKV0sXG4gICAgICB9KVxuICAgICk7XG5cbiAgICBjb25zdCBmaXJlaG9zZVN0cmVhbSA9IG5ldyBmaXJlaG9zZS5DZm5EZWxpdmVyeVN0cmVhbShcbiAgICAgIHRoaXMsXG4gICAgICAnRmlyZWhvc2VXQUZMb2dzRGVsaXZlcnlTdHJlYW0nLFxuICAgICAge1xuICAgICAgICBkZWxpdmVyeVN0cmVhbU5hbWU6ICdhd3Mtd2FmLWxvZ3MtJyArIHN0YWNrLnN0YWNrTmFtZSxcbiAgICAgICAgZGVsaXZlcnlTdHJlYW1UeXBlOiAnRGlyZWN0UHV0JyxcbiAgICAgICAgZXh0ZW5kZWRTM0Rlc3RpbmF0aW9uQ29uZmlndXJhdGlvbjoge1xuICAgICAgICAgIGJ1Y2tldEFybjogd2FmTG9nQnVja2V0LmJ1Y2tldEFybixcbiAgICAgICAgICBidWZmZXJpbmdIaW50czoge1xuICAgICAgICAgICAgaW50ZXJ2YWxJblNlY29uZHM6IDMwMCxcbiAgICAgICAgICAgIHNpemVJbk1CczogNSxcbiAgICAgICAgICB9LFxuICAgICAgICAgIGNvbXByZXNzaW9uRm9ybWF0OiAnR1pJUCcsXG4gICAgICAgICAgcHJlZml4OlxuICAgICAgICAgICAgJ0FXU0xvZ3MveWVhcj0he3RpbWVzdGFtcDp5eXl5fS9tb250aD0he3RpbWVzdGFtcDpNTX0vZGF5PSF7dGltZXN0YW1wOmRkfS9ob3VyPSF7dGltZXN0YW1wOkhIfS8nLFxuICAgICAgICAgIGVycm9yT3V0cHV0UHJlZml4OlxuICAgICAgICAgICAgJ0FXU0Vycm9yTG9ncy9yZXN1bHQ9IXtmaXJlaG9zZTplcnJvci1vdXRwdXQtdHlwZX0veWVhcj0he3RpbWVzdGFtcDp5eXl5fS9tb250aD0he3RpbWVzdGFtcDpNTX0vZGF5PSF7dGltZXN0YW1wOmRkfS9ob3VyPSF7dGltZXN0YW1wOkhIfS8nLFxuICAgICAgICAgIHJvbGVBcm46IGZpcmVob3NlUm9sZS5yb2xlQXJuLFxuICAgICAgICB9LFxuICAgICAgfVxuICAgICk7XG5cbiAgICAvL0dsdWUgREIgJiB0YWJsZVxuICAgIGNvbnN0IGdsdWVEYXRhYmFzZVBvc3RmaXggPSBwcm9wcy5yZXNvdXJjZU5hbWluZ1ByZWZpeFxuICAgICAgPyBgXyR7cHJvcHMucmVzb3VyY2VOYW1pbmdQcmVmaXgudG9Mb3dlckNhc2UoKX1gXG4gICAgICA6ICcnO1xuICAgIGNvbnN0IGdsdWVBY2Nlc3NMb2dzRGF0YWJhc2UgPSBuZXcgZ2x1ZS5EYXRhYmFzZShcbiAgICAgIHRoaXMsXG4gICAgICAnR2x1ZUFjY2Vzc0xvZ3NEYXRhYmFzZScsXG4gICAgICB7XG4gICAgICAgIGRhdGFiYXNlTmFtZTogYGdsdWVfYWNjZXNzbG9nc19kYXRhYmFzZSR7Z2x1ZURhdGFiYXNlUG9zdGZpeH1gLFxuICAgICAgfVxuICAgICk7XG5cbiAgICBjb25zdCBnbHVlV2FmQWNjZXNzTG9nc1RhYmxlID0gbmV3IGdsdWUuVGFibGUoXG4gICAgICB0aGlzLFxuICAgICAgJ0dsdWVXYWZBY2Nlc3NMb2dzVGFibGUnLFxuICAgICAge1xuICAgICAgICBkYXRhYmFzZTogZ2x1ZUFjY2Vzc0xvZ3NEYXRhYmFzZSxcbiAgICAgICAgdGFibGVOYW1lOiAnd2FmX2FjY2Vzc19sb2dzJyxcbiAgICAgICAgZGVzY3JpcHRpb246IHN0YWNrLnN0YWNrTmFtZSArICcgLSBXQUYgTG9ncycsXG4gICAgICAgIGJ1Y2tldDogd2FmTG9nQnVja2V0LFxuICAgICAgICBzM1ByZWZpeDogJ0FXU0xvZ3MvJyxcbiAgICAgICAgY29sdW1uczogW1xuICAgICAgICAgIHtcbiAgICAgICAgICAgIG5hbWU6ICd0aW1lc3RhbXAnLFxuICAgICAgICAgICAgdHlwZTogZ2x1ZS5TY2hlbWEuQklHX0lOVCxcbiAgICAgICAgICB9LFxuICAgICAgICAgIHtcbiAgICAgICAgICAgIG5hbWU6ICdmb3JtYXR2ZXJzaW9uJyxcbiAgICAgICAgICAgIHR5cGU6IGdsdWUuU2NoZW1hLklOVEVHRVIsXG4gICAgICAgICAgfSxcbiAgICAgICAgICB7XG4gICAgICAgICAgICBuYW1lOiAnd2ViYWNsaWQnLFxuICAgICAgICAgICAgdHlwZTogZ2x1ZS5TY2hlbWEuU1RSSU5HLFxuICAgICAgICAgIH0sXG4gICAgICAgICAge1xuICAgICAgICAgICAgbmFtZTogJ3Rlcm1pbmF0aW5ncnVsZWlkJyxcbiAgICAgICAgICAgIHR5cGU6IGdsdWUuU2NoZW1hLlNUUklORyxcbiAgICAgICAgICB9LFxuICAgICAgICAgIHtcbiAgICAgICAgICAgIG5hbWU6ICd0ZXJtaW5hdGluZ3J1bGV0eXBlJyxcbiAgICAgICAgICAgIHR5cGU6IGdsdWUuU2NoZW1hLlNUUklORyxcbiAgICAgICAgICB9LFxuICAgICAgICAgIHtcbiAgICAgICAgICAgIG5hbWU6ICdhY3Rpb24nLFxuICAgICAgICAgICAgdHlwZTogZ2x1ZS5TY2hlbWEuU1RSSU5HLFxuICAgICAgICAgIH0sXG4gICAgICAgICAge1xuICAgICAgICAgICAgbmFtZTogJ2h0dHBzb3VyY2VuYW1lJyxcbiAgICAgICAgICAgIHR5cGU6IGdsdWUuU2NoZW1hLlNUUklORyxcbiAgICAgICAgICB9LFxuICAgICAgICAgIHtcbiAgICAgICAgICAgIG5hbWU6ICdodHRwc291cmNlaWQnLFxuICAgICAgICAgICAgdHlwZTogZ2x1ZS5TY2hlbWEuU1RSSU5HLFxuICAgICAgICAgIH0sXG4gICAgICAgICAge1xuICAgICAgICAgICAgbmFtZTogJ3J1bGVncm91cGxpc3QnLFxuICAgICAgICAgICAgdHlwZTogZ2x1ZS5TY2hlbWEuYXJyYXkoZ2x1ZS5TY2hlbWEuU1RSSU5HKSxcbiAgICAgICAgICB9LFxuICAgICAgICAgIHtcbiAgICAgICAgICAgIG5hbWU6ICdyYXRlYmFzZWRydWxlbGlzdCcsXG4gICAgICAgICAgICB0eXBlOiBnbHVlLlNjaGVtYS5hcnJheShnbHVlLlNjaGVtYS5TVFJJTkcpLFxuICAgICAgICAgIH0sXG4gICAgICAgICAge1xuICAgICAgICAgICAgbmFtZTogJ25vbnRlcm1pbmF0aW5nbWF0Y2hpbmdydWxlcycsXG4gICAgICAgICAgICB0eXBlOiBnbHVlLlNjaGVtYS5hcnJheShnbHVlLlNjaGVtYS5TVFJJTkcpLFxuICAgICAgICAgIH0sXG4gICAgICAgICAge1xuICAgICAgICAgICAgbmFtZTogJ2h0dHByZXF1ZXN0JyxcbiAgICAgICAgICAgIHR5cGU6IGdsdWUuU2NoZW1hLnN0cnVjdChbXG4gICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICBuYW1lOiAnY2xpZW50aXAnLFxuICAgICAgICAgICAgICAgIHR5cGU6IGdsdWUuU2NoZW1hLlNUUklORyxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgIG5hbWU6ICdjb3VudHJ5JyxcbiAgICAgICAgICAgICAgICB0eXBlOiBnbHVlLlNjaGVtYS5TVFJJTkcsXG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICBuYW1lOiAnaGVhZGVycycsXG4gICAgICAgICAgICAgICAgdHlwZTogZ2x1ZS5TY2hlbWEuYXJyYXkoXG4gICAgICAgICAgICAgICAgICBnbHVlLlNjaGVtYS5zdHJ1Y3QoW1xuICAgICAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICAgICAgbmFtZTogJ25hbWUnLFxuICAgICAgICAgICAgICAgICAgICAgIHR5cGU6IGdsdWUuU2NoZW1hLlNUUklORyxcbiAgICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgICAgIG5hbWU6ICd2YWx1ZScsXG4gICAgICAgICAgICAgICAgICAgICAgdHlwZTogZ2x1ZS5TY2hlbWEuU1RSSU5HLFxuICAgICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgICAgXSlcbiAgICAgICAgICAgICAgICApLFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgbmFtZTogJ3VyaScsXG4gICAgICAgICAgICAgICAgdHlwZTogZ2x1ZS5TY2hlbWEuU1RSSU5HLFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgbmFtZTogJ2FyZ3MnLFxuICAgICAgICAgICAgICAgIHR5cGU6IGdsdWUuU2NoZW1hLlNUUklORyxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgIG5hbWU6ICdodHRwdmVyc2lvbicsXG4gICAgICAgICAgICAgICAgdHlwZTogZ2x1ZS5TY2hlbWEuU1RSSU5HLFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgbmFtZTogJ2h0dHBtZXRob2QnLFxuICAgICAgICAgICAgICAgIHR5cGU6IGdsdWUuU2NoZW1hLlNUUklORyxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgIG5hbWU6ICdyZXF1ZXN0aWQnLFxuICAgICAgICAgICAgICAgIHR5cGU6IGdsdWUuU2NoZW1hLlNUUklORyxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIF0pLFxuICAgICAgICAgIH0sXG4gICAgICAgIF0sXG4gICAgICAgIHBhcnRpdGlvbktleXM6IFtcbiAgICAgICAgICB7XG4gICAgICAgICAgICBuYW1lOiAneWVhcicsXG4gICAgICAgICAgICB0eXBlOiBnbHVlLlNjaGVtYS5JTlRFR0VSLFxuICAgICAgICAgIH0sXG4gICAgICAgICAge1xuICAgICAgICAgICAgbmFtZTogJ21vbnRoJyxcbiAgICAgICAgICAgIHR5cGU6IGdsdWUuU2NoZW1hLklOVEVHRVIsXG4gICAgICAgICAgfSxcbiAgICAgICAgICB7XG4gICAgICAgICAgICBuYW1lOiAnZGF5JyxcbiAgICAgICAgICAgIHR5cGU6IGdsdWUuU2NoZW1hLklOVEVHRVIsXG4gICAgICAgICAgfSxcbiAgICAgICAgICB7XG4gICAgICAgICAgICBuYW1lOiAnaG91cicsXG4gICAgICAgICAgICB0eXBlOiBnbHVlLlNjaGVtYS5JTlRFR0VSLFxuICAgICAgICAgIH0sXG4gICAgICAgIF0sXG4gICAgICAgIGRhdGFGb3JtYXQ6IHtcbiAgICAgICAgICBpbnB1dEZvcm1hdDogbmV3IGdsdWUuSW5wdXRGb3JtYXQoXG4gICAgICAgICAgICAnb3JnLmFwYWNoZS5oYWRvb3AubWFwcmVkLlRleHRJbnB1dEZvcm1hdCdcbiAgICAgICAgICApLFxuICAgICAgICAgIG91dHB1dEZvcm1hdDogbmV3IGdsdWUuT3V0cHV0Rm9ybWF0KFxuICAgICAgICAgICAgJ29yZy5hcGFjaGUuaGFkb29wLmhpdmUucWwuaW8uSGl2ZUlnbm9yZUtleVRleHRPdXRwdXRGb3JtYXQnXG4gICAgICAgICAgKSxcbiAgICAgICAgICBzZXJpYWxpemF0aW9uTGlicmFyeTogbmV3IGdsdWUuU2VyaWFsaXphdGlvbkxpYnJhcnkoXG4gICAgICAgICAgICAnb3JnLm9wZW54LmRhdGEuanNvbnNlcmRlLkpzb25TZXJEZSdcbiAgICAgICAgICApLFxuICAgICAgICB9LFxuICAgICAgfVxuICAgICk7XG5cbiAgICAvLyBjb25zdCBkZiA9IG5ldyBnbHVlLkRhdGFGb3JtYXQoe1xuICAgIC8vICAgaW5wdXRGb3JtYXQ6IG5ldyBnbHVlLklucHV0Rm9ybWF0KCdvcmcuYXBhY2hlLmhhZG9vcC5tYXByZWQuVGV4dElucHV0Rm9ybWF0JyksXG4gICAgLy8gICBvdXRwdXRGb3JtYXQ6IG5ldyBnbHVlLk91dHB1dEZvcm1hdCgnb3JnLmFwYWNoZS5oYWRvb3AuaGl2ZS5xbC5pby5IaXZlSWdub3JlS2V5VGV4dE91dHB1dEZvcm1hdCcpLFxuICAgIC8vICAgc2VyaWFsaXphdGlvbkxpYnJhcnk6IG5ldyBnbHVlLlNlcmlhbGl6YXRpb25MaWJyYXJ5KCdvcmcub3BlbnguZGF0YS5qc29uc2VyZGUuSnNvblNlckRlJyksXG4gICAgLy8gfSk7XG5cbiAgICBpZiAocHJvcHMud2FmMlNjb3BlID09IFdhZjJTY29wZU9wdGlvbi5DTE9VREZST05UKSB7XG4gICAgICBuZXcgY2ZuR2x1ZS5DZm5UYWJsZSh0aGlzLCAnZ2x1ZUFwcEFjY2Vzc0xvZ3NUYWJsZScsIHtcbiAgICAgICAgZGF0YWJhc2VOYW1lOiBnbHVlQWNjZXNzTG9nc0RhdGFiYXNlLmRhdGFiYXNlTmFtZSxcbiAgICAgICAgY2F0YWxvZ0lkOiBjZGsuQXdzLkFDQ09VTlRfSUQsXG4gICAgICAgIHRhYmxlSW5wdXQ6IHtcbiAgICAgICAgICBuYW1lOiAnYXBwX2FjY2Vzc19sb2dzJyxcbiAgICAgICAgICBkZXNjcmlwdGlvbjogc3RhY2suc3RhY2tOYW1lICsgJyAtIEFQUCBBY2Nlc3MgTG9ncycsXG4gICAgICAgICAgcGFyYW1ldGVyczoge1xuICAgICAgICAgICAgJ3NraXAuaGVhZGVyLmxpbmUuY291bnQnOiAnMicsXG4gICAgICAgICAgICAnRVhURVJOQUwnOiAnVFJVRScsXG4gICAgICAgICAgfSxcbiAgICAgICAgICBzdG9yYWdlRGVzY3JpcHRvcjoge1xuICAgICAgICAgICAgY29sdW1uczogW1xuICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgbmFtZTogJ2RhdGUnLFxuICAgICAgICAgICAgICAgIHR5cGU6IGdsdWUuU2NoZW1hLkRBVEUuaW5wdXRTdHJpbmcsXG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICBuYW1lOiAndGltZScsXG4gICAgICAgICAgICAgICAgdHlwZTogZ2x1ZS5TY2hlbWEuU1RSSU5HLmlucHV0U3RyaW5nLFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgbmFtZTogJ2xvY2F0aW9uJyxcbiAgICAgICAgICAgICAgICB0eXBlOiBnbHVlLlNjaGVtYS5TVFJJTkcuaW5wdXRTdHJpbmcsXG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICBuYW1lOiAnYnl0ZXMnLFxuICAgICAgICAgICAgICAgIHR5cGU6IGdsdWUuU2NoZW1hLkJJR19JTlQuaW5wdXRTdHJpbmcsXG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICBuYW1lOiAncmVxdWVzdGlwJyxcbiAgICAgICAgICAgICAgICB0eXBlOiBnbHVlLlNjaGVtYS5TVFJJTkcuaW5wdXRTdHJpbmcsXG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICBuYW1lOiAnbWV0aG9kJyxcbiAgICAgICAgICAgICAgICB0eXBlOiBnbHVlLlNjaGVtYS5TVFJJTkcuaW5wdXRTdHJpbmcsXG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICBuYW1lOiAnaG9zdCcsXG4gICAgICAgICAgICAgICAgdHlwZTogZ2x1ZS5TY2hlbWEuU1RSSU5HLmlucHV0U3RyaW5nLFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgbmFtZTogJ3VyaScsXG4gICAgICAgICAgICAgICAgdHlwZTogZ2x1ZS5TY2hlbWEuU1RSSU5HLmlucHV0U3RyaW5nLFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgbmFtZTogJ3N0YXR1cycsXG4gICAgICAgICAgICAgICAgdHlwZTogZ2x1ZS5TY2hlbWEuSU5URUdFUi5pbnB1dFN0cmluZyxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgIG5hbWU6ICdyZWZlcnJlcicsXG4gICAgICAgICAgICAgICAgdHlwZTogZ2x1ZS5TY2hlbWEuU1RSSU5HLmlucHV0U3RyaW5nLFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgbmFtZTogJ3VzZXJhZ2VudCcsXG4gICAgICAgICAgICAgICAgdHlwZTogZ2x1ZS5TY2hlbWEuU1RSSU5HLmlucHV0U3RyaW5nLFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgbmFtZTogJ3F1ZXJ5c3RyaW5nJyxcbiAgICAgICAgICAgICAgICB0eXBlOiBnbHVlLlNjaGVtYS5TVFJJTkcuaW5wdXRTdHJpbmcsXG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICBuYW1lOiAnY29va2llJyxcbiAgICAgICAgICAgICAgICB0eXBlOiBnbHVlLlNjaGVtYS5TVFJJTkcuaW5wdXRTdHJpbmcsXG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICBuYW1lOiAncmVzdWx0dHlwZScsXG4gICAgICAgICAgICAgICAgdHlwZTogZ2x1ZS5TY2hlbWEuU1RSSU5HLmlucHV0U3RyaW5nLFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgbmFtZTogJ3JlcXVlc3RpZCcsXG4gICAgICAgICAgICAgICAgdHlwZTogZ2x1ZS5TY2hlbWEuU1RSSU5HLmlucHV0U3RyaW5nLFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgbmFtZTogJ2hvc3RoZWFkZXInLFxuICAgICAgICAgICAgICAgIHR5cGU6IGdsdWUuU2NoZW1hLlNUUklORy5pbnB1dFN0cmluZyxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgIG5hbWU6ICdyZXF1ZXN0cHJvdG9jb2wnLFxuICAgICAgICAgICAgICAgIHR5cGU6IGdsdWUuU2NoZW1hLlNUUklORy5pbnB1dFN0cmluZyxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgIG5hbWU6ICdyZXF1ZXN0Ynl0ZXMnLFxuICAgICAgICAgICAgICAgIHR5cGU6IGdsdWUuU2NoZW1hLkJJR19JTlQuaW5wdXRTdHJpbmcsXG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICBuYW1lOiAndGltZXRha2VuJyxcbiAgICAgICAgICAgICAgICB0eXBlOiBnbHVlLlNjaGVtYS5GTE9BVC5pbnB1dFN0cmluZyxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgIG5hbWU6ICd4Zm9yd2FyZGVkZm9yJyxcbiAgICAgICAgICAgICAgICB0eXBlOiBnbHVlLlNjaGVtYS5TVFJJTkcuaW5wdXRTdHJpbmcsXG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICBuYW1lOiAnc3NscHJvdG9jb2wnLFxuICAgICAgICAgICAgICAgIHR5cGU6IGdsdWUuU2NoZW1hLlNUUklORy5pbnB1dFN0cmluZyxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgIG5hbWU6ICdzc2xjaXBoZXInLFxuICAgICAgICAgICAgICAgIHR5cGU6IGdsdWUuU2NoZW1hLlNUUklORy5pbnB1dFN0cmluZyxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgIG5hbWU6ICdyZXNwb25zZXJlc3VsdHR5cGUnLFxuICAgICAgICAgICAgICAgIHR5cGU6IGdsdWUuU2NoZW1hLlNUUklORy5pbnB1dFN0cmluZyxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgIG5hbWU6ICdodHRwdmVyc2lvbicsXG4gICAgICAgICAgICAgICAgdHlwZTogZ2x1ZS5TY2hlbWEuU1RSSU5HLmlucHV0U3RyaW5nLFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgbmFtZTogJ2ZpbGVzdGF0dXMnLFxuICAgICAgICAgICAgICAgIHR5cGU6IGdsdWUuU2NoZW1hLlNUUklORy5pbnB1dFN0cmluZyxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgIG5hbWU6ICdlbmNyeXB0ZWRmaWVsZHMnLFxuICAgICAgICAgICAgICAgIHR5cGU6IGdsdWUuU2NoZW1hLklOVEVHRVIuaW5wdXRTdHJpbmcsXG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICBdLFxuICAgICAgICAgICAgY29tcHJlc3NlZDogZmFsc2UsXG4gICAgICAgICAgICBpbnB1dEZvcm1hdDogJ29yZy5hcGFjaGUuaGFkb29wLm1hcHJlZC5UZXh0SW5wdXRGb3JtYXQnLFxuICAgICAgICAgICAgbG9jYXRpb246XG4gICAgICAgICAgICAgICdzMzovLycgKyBhY2Nlc3NMb2dCdWNrZXQuYnVja2V0TmFtZSArICcvQVdTTG9ncy1QYXJ0aXRpb25lZC8nLFxuICAgICAgICAgICAgb3V0cHV0Rm9ybWF0OlxuICAgICAgICAgICAgICAnb3JnLmFwYWNoZS5oYWRvb3AuaGl2ZS5xbC5pby5IaXZlSWdub3JlS2V5VGV4dE91dHB1dEZvcm1hdCcsXG4gICAgICAgICAgICBzZXJkZUluZm86IHtcbiAgICAgICAgICAgICAgc2VyaWFsaXphdGlvbkxpYnJhcnk6XG4gICAgICAgICAgICAgICAgJ29yZy5hcGFjaGUuaGFkb29wLmhpdmUuc2VyZGUyLmxhenkuTGF6eVNpbXBsZVNlckRlJyxcbiAgICAgICAgICAgICAgcGFyYW1ldGVyczoge1xuICAgICAgICAgICAgICAgICdmaWVsZC5kZWxpbSc6ICdcdCcsXG4gICAgICAgICAgICAgICAgJ3NlcmlhbGl6YXRpb24uZm9ybWF0JzogJ1x0JyxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgICBzdG9yZWRBc1N1YkRpcmVjdG9yaWVzOiBmYWxzZSxcbiAgICAgICAgICB9LFxuICAgICAgICAgIHRhYmxlVHlwZTogJ0VYVEVSTkFMX1RBQkxFJyxcbiAgICAgICAgICBwYXJ0aXRpb25LZXlzOiBbXG4gICAgICAgICAgICB7XG4gICAgICAgICAgICAgIG5hbWU6ICd5ZWFyJyxcbiAgICAgICAgICAgICAgdHlwZTogZ2x1ZS5TY2hlbWEuSU5URUdFUi5pbnB1dFN0cmluZyxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgICB7XG4gICAgICAgICAgICAgIG5hbWU6ICdtb250aCcsXG4gICAgICAgICAgICAgIHR5cGU6IGdsdWUuU2NoZW1hLklOVEVHRVIuaW5wdXRTdHJpbmcsXG4gICAgICAgICAgICB9LFxuICAgICAgICAgICAge1xuICAgICAgICAgICAgICBuYW1lOiAnZGF5JyxcbiAgICAgICAgICAgICAgdHlwZTogZ2x1ZS5TY2hlbWEuSU5URUdFUi5pbnB1dFN0cmluZyxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgICB7XG4gICAgICAgICAgICAgIG5hbWU6ICdob3VyJyxcbiAgICAgICAgICAgICAgdHlwZTogZ2x1ZS5TY2hlbWEuSU5URUdFUi5pbnB1dFN0cmluZyxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgXSxcbiAgICAgICAgfSxcbiAgICAgIH0pO1xuICAgIH0gZWxzZSB7XG4gICAgICAvLyBnbHVlIGFjY2VzcyBsb2cgZm9ybWF0IGZvciBBbGJcbiAgICAgIG5ldyBjZm5HbHVlLkNmblRhYmxlKHRoaXMsICdnbHVlQXBwQWNjZXNzTG9nc1RhYmxlJywge1xuICAgICAgICBkYXRhYmFzZU5hbWU6IGdsdWVBY2Nlc3NMb2dzRGF0YWJhc2UuZGF0YWJhc2VOYW1lLFxuICAgICAgICBjYXRhbG9nSWQ6IGNkay5Bd3MuQUNDT1VOVF9JRCxcbiAgICAgICAgdGFibGVJbnB1dDoge1xuICAgICAgICAgIG5hbWU6ICdhcHBfYWNjZXNzX2xvZ3MnLFxuICAgICAgICAgIGRlc2NyaXB0aW9uOiBzdGFjay5zdGFja05hbWUgKyAnIC0gQVBQIEFjY2VzcyBMb2dzJyxcbiAgICAgICAgICBwYXJhbWV0ZXJzOiB7XG4gICAgICAgICAgICBFWFRFUk5BTDogJ1RSVUUnLFxuICAgICAgICAgIH0sXG4gICAgICAgICAgc3RvcmFnZURlc2NyaXB0b3I6IHtcbiAgICAgICAgICAgIGNvbHVtbnM6IFtcbiAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgIG5hbWU6ICd0eXBlJyxcbiAgICAgICAgICAgICAgICB0eXBlOiBnbHVlLlNjaGVtYS5TVFJJTkcuaW5wdXRTdHJpbmcsXG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICBuYW1lOiAndGltZScsXG4gICAgICAgICAgICAgICAgdHlwZTogZ2x1ZS5TY2hlbWEuU1RSSU5HLmlucHV0U3RyaW5nLFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgbmFtZTogJ2VsYicsXG4gICAgICAgICAgICAgICAgdHlwZTogZ2x1ZS5TY2hlbWEuU1RSSU5HLmlucHV0U3RyaW5nLFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgbmFtZTogJ2NsaWVudF9pcCcsXG4gICAgICAgICAgICAgICAgdHlwZTogZ2x1ZS5TY2hlbWEuU1RSSU5HLmlucHV0U3RyaW5nLFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgbmFtZTogJ2NsaWVudF9wb3J0JyxcbiAgICAgICAgICAgICAgICB0eXBlOiBnbHVlLlNjaGVtYS5JTlRFR0VSLmlucHV0U3RyaW5nLFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgbmFtZTogJ3RhcmdldF9pcCcsXG4gICAgICAgICAgICAgICAgdHlwZTogZ2x1ZS5TY2hlbWEuU1RSSU5HLmlucHV0U3RyaW5nLFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgbmFtZTogJ3RhcmdldF9wb3J0JyxcbiAgICAgICAgICAgICAgICB0eXBlOiBnbHVlLlNjaGVtYS5JTlRFR0VSLmlucHV0U3RyaW5nLFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgbmFtZTogJ3JlcXVlc3RfcHJvY2Vzc2luZ190aW1lJyxcbiAgICAgICAgICAgICAgICB0eXBlOiBnbHVlLlNjaGVtYS5ET1VCTEUuaW5wdXRTdHJpbmcsXG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICBuYW1lOiAndGFyZ2V0X3Byb2Nlc3NpbmdfdGltZScsXG4gICAgICAgICAgICAgICAgdHlwZTogZ2x1ZS5TY2hlbWEuRE9VQkxFLmlucHV0U3RyaW5nLFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgbmFtZTogJ3Jlc3BvbnNlX3Byb2Nlc3NpbmdfdGltZScsXG4gICAgICAgICAgICAgICAgdHlwZTogZ2x1ZS5TY2hlbWEuRE9VQkxFLmlucHV0U3RyaW5nLFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgbmFtZTogJ2VsYl9zdGF0dXNfY29kZScsXG4gICAgICAgICAgICAgICAgdHlwZTogZ2x1ZS5TY2hlbWEuU1RSSU5HLmlucHV0U3RyaW5nLFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgbmFtZTogJ3RhcmdldF9zdGF0dXNfY29kZScsXG4gICAgICAgICAgICAgICAgdHlwZTogZ2x1ZS5TY2hlbWEuU1RSSU5HLmlucHV0U3RyaW5nLFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgbmFtZTogJ3JlY2VpdmVkX2J5dGVzJyxcbiAgICAgICAgICAgICAgICB0eXBlOiBnbHVlLlNjaGVtYS5CSUdfSU5ULmlucHV0U3RyaW5nLFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgbmFtZTogJ3NlbnRfYnl0ZXMnLFxuICAgICAgICAgICAgICAgIHR5cGU6IGdsdWUuU2NoZW1hLkJJR19JTlQuaW5wdXRTdHJpbmcsXG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICBuYW1lOiAncmVxdWVzdF92ZXJiJyxcbiAgICAgICAgICAgICAgICB0eXBlOiBnbHVlLlNjaGVtYS5TVFJJTkcuaW5wdXRTdHJpbmcsXG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICBuYW1lOiAncmVxdWVzdF91cmwnLFxuICAgICAgICAgICAgICAgIHR5cGU6IGdsdWUuU2NoZW1hLlNUUklORy5pbnB1dFN0cmluZyxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgIG5hbWU6ICdyZXF1ZXN0X3Byb3RvJyxcbiAgICAgICAgICAgICAgICB0eXBlOiBnbHVlLlNjaGVtYS5TVFJJTkcuaW5wdXRTdHJpbmcsXG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICBuYW1lOiAndXNlcl9hZ2VudCcsXG4gICAgICAgICAgICAgICAgdHlwZTogZ2x1ZS5TY2hlbWEuU1RSSU5HLmlucHV0U3RyaW5nLFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgbmFtZTogJ3NzbF9jaXBoZXInLFxuICAgICAgICAgICAgICAgIHR5cGU6IGdsdWUuU2NoZW1hLlNUUklORy5pbnB1dFN0cmluZyxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgIG5hbWU6ICdzc2xfcHJvdG9jb2wnLFxuICAgICAgICAgICAgICAgIHR5cGU6IGdsdWUuU2NoZW1hLlNUUklORy5pbnB1dFN0cmluZyxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgIG5hbWU6ICd0YXJnZXRfZ3JvdXBfYXJuJyxcbiAgICAgICAgICAgICAgICB0eXBlOiBnbHVlLlNjaGVtYS5TVFJJTkcuaW5wdXRTdHJpbmcsXG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICBuYW1lOiAndHJhY2VfaWQnLFxuICAgICAgICAgICAgICAgIHR5cGU6IGdsdWUuU2NoZW1hLlNUUklORy5pbnB1dFN0cmluZyxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgIG5hbWU6ICdkb21haW5fbmFtZScsXG4gICAgICAgICAgICAgICAgdHlwZTogZ2x1ZS5TY2hlbWEuU1RSSU5HLmlucHV0U3RyaW5nLFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgbmFtZTogJ2Nob3Nlbl9jZXJ0X2FybicsXG4gICAgICAgICAgICAgICAgdHlwZTogZ2x1ZS5TY2hlbWEuU1RSSU5HLmlucHV0U3RyaW5nLFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgbmFtZTogJ21hdGNoZWRfcnVsZV9wcmlvcml0eScsXG4gICAgICAgICAgICAgICAgdHlwZTogZ2x1ZS5TY2hlbWEuU1RSSU5HLmlucHV0U3RyaW5nLFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgbmFtZTogJ3JlcXVlc3RfY3JlYXRpb25fdGltZScsXG4gICAgICAgICAgICAgICAgdHlwZTogZ2x1ZS5TY2hlbWEuU1RSSU5HLmlucHV0U3RyaW5nLFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgbmFtZTogJ2FjdGlvbnNfZXhlY3V0ZWQnLFxuICAgICAgICAgICAgICAgIHR5cGU6IGdsdWUuU2NoZW1hLlNUUklORy5pbnB1dFN0cmluZyxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgIG5hbWU6ICdyZWRpcmVjdF91cmwnLFxuICAgICAgICAgICAgICAgIHR5cGU6IGdsdWUuU2NoZW1hLlNUUklORy5pbnB1dFN0cmluZyxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgIG5hbWU6ICdlcnJvcl9yZWFzb24nLFxuICAgICAgICAgICAgICAgIHR5cGU6IGdsdWUuU2NoZW1hLlNUUklORy5pbnB1dFN0cmluZyxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIF0sXG4gICAgICAgICAgICBjb21wcmVzc2VkOiBmYWxzZSxcbiAgICAgICAgICAgIGlucHV0Rm9ybWF0OiAnb3JnLmFwYWNoZS5oYWRvb3AubWFwcmVkLlRleHRJbnB1dEZvcm1hdCcsXG4gICAgICAgICAgICBsb2NhdGlvbjpcbiAgICAgICAgICAgICAgJ3MzOi8vJyArIGFjY2Vzc0xvZ0J1Y2tldC5idWNrZXROYW1lICsgJy9BV1NMb2dzLVBhcnRpdGlvbmVkLycsXG4gICAgICAgICAgICBvdXRwdXRGb3JtYXQ6XG4gICAgICAgICAgICAgICdvcmcuYXBhY2hlLmhhZG9vcC5oaXZlLnFsLmlvLkhpdmVJZ25vcmVLZXlUZXh0T3V0cHV0Rm9ybWF0JyxcbiAgICAgICAgICAgIHNlcmRlSW5mbzoge1xuICAgICAgICAgICAgICBzZXJpYWxpemF0aW9uTGlicmFyeTogJ29yZy5hcGFjaGUuaGFkb29wLmhpdmUuc2VyZGUyLlJlZ2V4U2VyRGUnLFxuICAgICAgICAgICAgICBwYXJhbWV0ZXJzOiB7XG4gICAgICAgICAgICAgICAgJ3NlcmlhbGl6YXRpb24uZm9ybWF0JzogJzEnLFxuICAgICAgICAgICAgICAgICdpbnB1dC5yZWdleCc6XG4gICAgICAgICAgICAgICAgICAnKFteIF0qKSAoW14gXSopIChbXiBdKikgKFteIF0qKTooWzAtOV0qKSAoW14gXSopWzotXShbMC05XSopIChbLS4wLTldKikgKFstLjAtOV0qKSAoWy0uMC05XSopICh8Wy0wLTldKikgKC18Wy0wLTldKikgKFstMC05XSopIChbLTAtOV0qKSBcIihbXiBdKikgKFteIF0qKSAoLSB8W14gXSopXCIgXCIoW15cIl0qKVwiIChbQS1aMC05LV0rKSAoW0EtWmEtejAtOS4tXSopIChbXiBdKikgXCIoW15cIl0qKVwiIFwiKFteXCJdKilcIiBcIihbXlwiXSopXCIgKFstLjAtOV0qKSAoW14gXSopIFwiKFteXCJdKilcIiBcIihbXiBdKilcIiBcIihbXiBdLiopXCInLFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIHN0b3JlZEFzU3ViRGlyZWN0b3JpZXM6IGZhbHNlLFxuICAgICAgICAgIH0sXG4gICAgICAgICAgdGFibGVUeXBlOiAnRVhURVJOQUxfVEFCTEUnLFxuICAgICAgICAgIHBhcnRpdGlvbktleXM6IFtcbiAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgbmFtZTogJ3llYXInLFxuICAgICAgICAgICAgICB0eXBlOiBnbHVlLlNjaGVtYS5JTlRFR0VSLmlucHV0U3RyaW5nLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgbmFtZTogJ21vbnRoJyxcbiAgICAgICAgICAgICAgdHlwZTogZ2x1ZS5TY2hlbWEuSU5URUdFUi5pbnB1dFN0cmluZyxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgICB7XG4gICAgICAgICAgICAgIG5hbWU6ICdkYXknLFxuICAgICAgICAgICAgICB0eXBlOiBnbHVlLlNjaGVtYS5JTlRFR0VSLmlucHV0U3RyaW5nLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgbmFtZTogJ2hvdXInLFxuICAgICAgICAgICAgICB0eXBlOiBnbHVlLlNjaGVtYS5JTlRFR0VSLmlucHV0U3RyaW5nLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICBdLFxuICAgICAgICB9LFxuICAgICAgfSk7XG4gICAgfVxuXG4gICAgLy9BdGhlbmFcbiAgICBjb25zdCBhdGhlbmFXb3JrR3JvdXBQb3N0Zml4ID0gcHJvcHMucmVzb3VyY2VOYW1pbmdQcmVmaXggPz8gJyc7XG4gICAgY29uc3QgYWRkUGFydGl0aW9uQXRoZW5hUXVlcnlXb3JrR3JvdXAgPSBuZXcgYXRoZW5hLkNmbldvcmtHcm91cChcbiAgICAgIHRoaXMsXG4gICAgICAnV0FGQWRkUGFydGl0aW9uQXRoZW5hUXVlcnlXb3JrR3JvdXAnLFxuICAgICAge1xuICAgICAgICBuYW1lOiBgV0FGQWRkUGFydGl0aW9uQXRoZW5hUXVlcnlXb3JrR3JvdXAke2F0aGVuYVdvcmtHcm91cFBvc3RmaXh9YCxcbiAgICAgICAgZGVzY3JpcHRpb246XG4gICAgICAgICAgJ0F0aGVuYSBXb3JrR3JvdXAgZm9yIGFkZGluZyBBdGhlbmEgcGFydGl0aW9uIHF1ZXJpZXMgdXNlZCBieSBBV1MgV0FGIFNlY3VyaXR5IEF1dG9tYXRpb25zIFNvbHV0aW9uJyxcbiAgICAgICAgc3RhdGU6ICdFTkFCTEVEJyxcbiAgICAgICAgcmVjdXJzaXZlRGVsZXRlT3B0aW9uOiB0cnVlLFxuICAgICAgICB3b3JrR3JvdXBDb25maWd1cmF0aW9uOiB7XG4gICAgICAgICAgcHVibGlzaENsb3VkV2F0Y2hNZXRyaWNzRW5hYmxlZDogdHJ1ZSxcbiAgICAgICAgICBlbmZvcmNlV29ya0dyb3VwQ29uZmlndXJhdGlvbjogZmFsc2UsXG4gICAgICAgIH0sXG4gICAgICB9XG4gICAgKTtcblxuICAgIGNvbnN0IHdBRkxvZ0F0aGVuYVF1ZXJ5V29ya0dyb3VwID0gbmV3IGF0aGVuYS5DZm5Xb3JrR3JvdXAoXG4gICAgICB0aGlzLFxuICAgICAgJ1dBRkxvZ0F0aGVuYVF1ZXJ5V29ya0dyb3VwJyxcbiAgICAgIHtcbiAgICAgICAgbmFtZTogYFdBRkxvZ0F0aGVuYVF1ZXJ5V29ya0dyb3VwJHthdGhlbmFXb3JrR3JvdXBQb3N0Zml4fWAsXG4gICAgICAgIGRlc2NyaXB0aW9uOlxuICAgICAgICAgICdBdGhlbmEgV29ya0dyb3VwIGZvciBXQUYgbG9nIHF1ZXJpZXMgdXNlZCBieSBBV1MgV0FGIFNlY3VyaXR5IEF1dG9tYXRpb25zIFNvbHV0aW9uJyxcbiAgICAgICAgc3RhdGU6ICdFTkFCTEVEJyxcbiAgICAgICAgcmVjdXJzaXZlRGVsZXRlT3B0aW9uOiB0cnVlLFxuICAgICAgICB3b3JrR3JvdXBDb25maWd1cmF0aW9uOiB7XG4gICAgICAgICAgcHVibGlzaENsb3VkV2F0Y2hNZXRyaWNzRW5hYmxlZDogdHJ1ZSxcbiAgICAgICAgfSxcbiAgICAgIH1cbiAgICApO1xuXG4gICAgY29uc3Qgd0FGQXBwQWNjZXNzTG9nQXRoZW5hUXVlcnlXb3JrR3JvdXAgPSBuZXcgYXRoZW5hLkNmbldvcmtHcm91cChcbiAgICAgIHRoaXMsXG4gICAgICAnV0FGQXBwQWNjZXNzTG9nQXRoZW5hUXVlcnlXb3JrR3JvdXAnLFxuICAgICAge1xuICAgICAgICBuYW1lOiBgV0FGQXBwQWNjZXNzTG9nQXRoZW5hUXVlcnlXb3JrR3JvdXAke2F0aGVuYVdvcmtHcm91cFBvc3RmaXh9YCxcbiAgICAgICAgZGVzY3JpcHRpb246XG4gICAgICAgICAgJ0F0aGVuYSBXb3JrR3JvdXAgZm9yIENsb3VkRnJvbnQgb3IgQUxCIGFwcGxpY2F0aW9uIGFjY2VzcyBsb2cgcXVlcmllcyB1c2VkIGJ5IEFXUyBXQUYgU2VjdXJpdHkgQXV0b21hdGlvbnMgU29sdXRpb24nLFxuICAgICAgICBzdGF0ZTogJ0VOQUJMRUQnLFxuICAgICAgICByZWN1cnNpdmVEZWxldGVPcHRpb246IHRydWUsXG4gICAgICAgIHdvcmtHcm91cENvbmZpZ3VyYXRpb246IHtcbiAgICAgICAgICBwdWJsaXNoQ2xvdWRXYXRjaE1ldHJpY3NFbmFibGVkOiB0cnVlLFxuICAgICAgICB9LFxuICAgICAgfVxuICAgICk7XG5cbiAgICAvL0Nsb3Vkd2F0Y2ggRGFzaGJvYXJkXG4gICAgbmV3IGNsb3Vkd2F0Y2guQ2ZuRGFzaGJvYXJkKHRoaXMsICdNb25pdG9yaW5nRGFzaGJvYXJkJywge1xuICAgICAgZGFzaGJvYXJkTmFtZTogY2xvdWRXYXRjaERhc2hib2FyZE5hbWUsXG4gICAgICBkYXNoYm9hcmRCb2R5OiBKU09OLnN0cmluZ2lmeSh7XG4gICAgICAgIHdpZGdldHM6IFtcbiAgICAgICAgICB7XG4gICAgICAgICAgICB0eXBlOiAnbWV0cmljJyxcbiAgICAgICAgICAgIHg6IDAsXG4gICAgICAgICAgICB5OiAwLFxuICAgICAgICAgICAgd2lkdGg6IDE1LFxuICAgICAgICAgICAgaGVpZ2h0OiAxMCxcbiAgICAgICAgICAgIHByb3BlcnRpZXM6IHtcbiAgICAgICAgICAgICAgdmlldzogJ3RpbWVTZXJpZXMnLFxuICAgICAgICAgICAgICBzdGFja2VkOiBmYWxzZSxcbiAgICAgICAgICAgICAgbWV0cmljczogW1xuICAgICAgICAgICAgICAgIFtcbiAgICAgICAgICAgICAgICAgICdXQUYnLFxuICAgICAgICAgICAgICAgICAgJ0Jsb2NrZWRSZXF1ZXN0cycsXG4gICAgICAgICAgICAgICAgICAnV2ViQUNMJyxcbiAgICAgICAgICAgICAgICAgICdXQUZXZWJBQ0xNZXRyaWNOYW1lJyxcbiAgICAgICAgICAgICAgICAgICdSdWxlJyxcbiAgICAgICAgICAgICAgICAgICdBTEwnLFxuICAgICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICAgICAgW1xuICAgICAgICAgICAgICAgICAgJ1dBRicsXG4gICAgICAgICAgICAgICAgICAnQWxsb3dlZFJlcXVlc3RzJyxcbiAgICAgICAgICAgICAgICAgICdXZWJBQ0wnLFxuICAgICAgICAgICAgICAgICAgJ1dBRldlYkFDTE1ldHJpY05hbWUnLFxuICAgICAgICAgICAgICAgICAgJ1J1bGUnLFxuICAgICAgICAgICAgICAgICAgJ0FMTCcsXG4gICAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgcmVnaW9uOiAndXMtZWFzdC0xJyxcbiAgICAgICAgICAgICAgcGVyaW9kOiAzMDAsXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIH0sXG4gICAgICAgIF0sXG4gICAgICB9KSxcbiAgICB9KTtcblxuICAgIC8vQWRkIENsb3VkV2F0Y2ggZXZlbnQgdG8gTGFtYmRhIExvZ1BhcnNlclxuICAgIGNvbnN0IGxvZ1BhcnNlcldhZkxvZ1J1bGVJbnB1dCA9IHtcbiAgICAgIHJlc291cmNlVHlwZTogJ0xhbWJkYUF0aGVuYVdBRkxvZ1BhcnNlcicsXG4gICAgICBnbHVlQWNjZXNzTG9nc0RhdGFiYXNlOiBnbHVlQWNjZXNzTG9nc0RhdGFiYXNlLmRhdGFiYXNlTmFtZSxcbiAgICAgIGFjY2Vzc0xvZ0J1Y2tldDogd2FmTG9nQnVja2V0LmJ1Y2tldE5hbWUsXG4gICAgICBnbHVlV2FmQWNjZXNzTG9nc1RhYmxlOiBnbHVlV2FmQWNjZXNzTG9nc1RhYmxlLnRhYmxlTmFtZSxcbiAgICAgIGF0aGVuYVdvcmtHcm91cDogd0FGTG9nQXRoZW5hUXVlcnlXb3JrR3JvdXAubmFtZSxcbiAgICB9O1xuXG4gICAgbmV3IGV2ZW50cy5SdWxlKHRoaXMsICdsYW1iZGFBdGhlbmFXQUZMb2dQYXJzZXJSdWxlJywge1xuICAgICAgZGVzY3JpcHRpb246ICdTZWN1cml0eSBBdXRvbWF0aW9uIC0gV0FGIExvZ3MgQXRoZW5hIHBhcnNlcicsXG4gICAgICBzY2hlZHVsZTogZXZlbnRzLlNjaGVkdWxlLmV4cHJlc3Npb24oJ3JhdGUoNSBtaW51dGVzKScpLFxuICAgICAgdGFyZ2V0czogW1xuICAgICAgICBuZXcgdGFyZ2V0cy5MYW1iZGFGdW5jdGlvbihsb2dQYXJzZXJMYW1iZGEsIHtcbiAgICAgICAgICBldmVudDogZXZlbnRzLlJ1bGVUYXJnZXRJbnB1dC5mcm9tT2JqZWN0KGxvZ1BhcnNlcldhZkxvZ1J1bGVJbnB1dCksXG4gICAgICAgIH0pLFxuICAgICAgXSxcbiAgICB9KTtcblxuICAgIGNvbnN0IGxhbWJkYUF0aGVuYUFwcExvZ1BhcnNlclJ1bGVJbnB1dCA9IHtcbiAgICAgIHJlc291cmNlVHlwZTogJ0xhbWJkYUF0aGVuYUFwcExvZ1BhcnNlcicsXG4gICAgICBnbHVlQWNjZXNzTG9nc0RhdGFiYXNlOiBnbHVlQWNjZXNzTG9nc0RhdGFiYXNlLmRhdGFiYXNlTmFtZSxcbiAgICAgIGFjY2Vzc0xvZ0J1Y2tldDogYWNjZXNzTG9nQnVja2V0LmJ1Y2tldE5hbWUsXG4gICAgICBnbHVlQXBwQWNjZXNzTG9nc1RhYmxlOiAnYXBwX2FjY2Vzc19sb2dzJyxcbiAgICAgIGF0aGVuYVdvcmtHcm91cDogd0FGQXBwQWNjZXNzTG9nQXRoZW5hUXVlcnlXb3JrR3JvdXAubmFtZSxcbiAgICB9O1xuXG4gICAgbmV3IGV2ZW50cy5SdWxlKHRoaXMsICdsYW1iZGFBdGhlbmFBcHBMb2dQYXJzZXJSdWxlJywge1xuICAgICAgZGVzY3JpcHRpb246ICdTZWN1cml0eSBBdXRvbWF0aW9uIC0gQXBwIExvZ3MgQXRoZW5hIHBhcnNlcicsXG4gICAgICBzY2hlZHVsZTogZXZlbnRzLlNjaGVkdWxlLmV4cHJlc3Npb24oJ3JhdGUoNSBtaW51dGVzKScpLFxuICAgICAgdGFyZ2V0czogW1xuICAgICAgICBuZXcgdGFyZ2V0cy5MYW1iZGFGdW5jdGlvbihsb2dQYXJzZXJMYW1iZGEsIHtcbiAgICAgICAgICBldmVudDogZXZlbnRzLlJ1bGVUYXJnZXRJbnB1dC5mcm9tT2JqZWN0KFxuICAgICAgICAgICAgbGFtYmRhQXRoZW5hQXBwTG9nUGFyc2VyUnVsZUlucHV0XG4gICAgICAgICAgKSxcbiAgICAgICAgfSksXG4gICAgICBdLFxuICAgIH0pO1xuXG4gICAgY29uc3QgYWRkQXRoZW5hUGFydGl0aW9uc0xhbWJkYSA9IG5ldyBsYW1iZGEuRG9ja2VySW1hZ2VGdW5jdGlvbihcbiAgICAgIHRoaXMsXG4gICAgICAnQWRkQXRoZW5hUGFydGl0aW9uc0Z1bmN0aW9uJyxcbiAgICAgIHtcbiAgICAgICAgZGVzY3JpcHRpb246XG4gICAgICAgICAgJ1RoaXMgZnVuY3Rpb24gYWRkcyBhIG5ldyBob3VybHkgcGFydGl0aW9uIHRvIGF0aGVuYSB0YWJsZS4gSXQgcnVucyBldmVyeSBob3VyLCB0cmlnZ2VyZWQgYnkgYSBDbG91ZFdhdGNoIGV2ZW50LicsXG4gICAgICAgIGNvZGU6IGxhbWJkYS5Eb2NrZXJJbWFnZUNvZGUuZnJvbUltYWdlQXNzZXQoXG4gICAgICAgICAgcGF0aC5qb2luKF9fZGlybmFtZSwgJy4uL3NvdXJjZS9sb2dfcGFyc2VyJyksXG4gICAgICAgICAge1xuICAgICAgICAgICAgY21kOiBbJ2FkZF9hdGhlbmFfcGFydGl0aW9ucy5sYW1iZGFfaGFuZGxlciddLFxuICAgICAgICAgIH1cbiAgICAgICAgKSxcbiAgICAgICAgbWVtb3J5U2l6ZTogNTEyLFxuICAgICAgICB0aW1lb3V0OiBjZGsuRHVyYXRpb24ubWludXRlcyg1KSxcbiAgICAgICAgbWF4RXZlbnRBZ2U6IGNkay5EdXJhdGlvbi5ob3Vycyg2KSxcbiAgICAgICAgcmV0cnlBdHRlbXB0czogMixcbiAgICAgICAgZW52aXJvbm1lbnQ6IHtcbiAgICAgICAgICBMT0dfTEVWRUw6IGxvZ0xldmVsLFxuICAgICAgICB9LFxuICAgICAgfVxuICAgICk7XG5cbiAgICBjb25zdCBhZGRBdGhlbmFQYXJ0aXRpb25zUnVsZUlucHV0ID0ge1xuICAgICAgcmVzb3VyY2VUeXBlOiAnTGFtYmRhQWRkQXRoZW5hUGFydGl0aW9uc0V2ZW50c1J1bGUnLFxuICAgICAgZ2x1ZUFjY2Vzc0xvZ3NEYXRhYmFzZTogZ2x1ZUFjY2Vzc0xvZ3NEYXRhYmFzZS5kYXRhYmFzZU5hbWUsXG4gICAgICBhY2Nlc3NMb2dCdWNrZXQ6IGFjY2Vzc0xvZ0J1Y2tldC5idWNrZXROYW1lLFxuICAgICAgZ2x1ZUFwcEFjY2Vzc0xvZ3NUYWJsZTogJ2FwcF9hY2Nlc3NfbG9ncycsXG4gICAgICBnbHVlV2FmQWNjZXNzTG9nc1RhYmxlOiBnbHVlV2FmQWNjZXNzTG9nc1RhYmxlLnRhYmxlTmFtZSxcbiAgICAgIHdhZkxvZ0J1Y2tldDogd2FmTG9nQnVja2V0LmJ1Y2tldE5hbWUsXG4gICAgICBhdGhlbmFXb3JrR3JvdXA6IGFkZFBhcnRpdGlvbkF0aGVuYVF1ZXJ5V29ya0dyb3VwLm5hbWUsXG4gICAgfTtcblxuICAgIG5ldyBldmVudHMuUnVsZSh0aGlzLCAnbGFtYmRhQWRkQXRoZW5hUGFydGl0aW9uJywge1xuICAgICAgZGVzY3JpcHRpb246ICdTZWN1cml0eSBBdXRvbWF0aW9ucyAtIEFkZCBwYXJ0aXRpb25zIHRvIEF0aGVuYSB0YWJsZScsXG4gICAgICBzY2hlZHVsZTogZXZlbnRzLlNjaGVkdWxlLmV4cHJlc3Npb24oJ3JhdGUoMSBob3VyKScpLFxuICAgICAgdGFyZ2V0czogW1xuICAgICAgICBuZXcgdGFyZ2V0cy5MYW1iZGFGdW5jdGlvbihhZGRBdGhlbmFQYXJ0aXRpb25zTGFtYmRhLCB7XG4gICAgICAgICAgZXZlbnQ6IGV2ZW50cy5SdWxlVGFyZ2V0SW5wdXQuZnJvbU9iamVjdChcbiAgICAgICAgICAgIGFkZEF0aGVuYVBhcnRpdGlvbnNSdWxlSW5wdXRcbiAgICAgICAgICApLFxuICAgICAgICB9KSxcbiAgICAgIF0sXG4gICAgfSk7XG5cbiAgICBhZGRBdGhlbmFQYXJ0aXRpb25zTGFtYmRhLmFkZFRvUm9sZVBvbGljeShcbiAgICAgIG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgZWZmZWN0OiBpYW0uRWZmZWN0LkFMTE9XLFxuICAgICAgICBhY3Rpb25zOiBbXG4gICAgICAgICAgJ3MzOkFib3J0TXVsdGlwYXJ0VXBsb2FkJyxcbiAgICAgICAgICAnczM6R2V0QnVja2V0TG9jYXRpb24nLFxuICAgICAgICAgICdzMzpHZXRPYmplY3QnLFxuICAgICAgICAgICdzMzpMaXN0QnVja2V0JyxcbiAgICAgICAgICAnczM6TGlzdEJ1Y2tldE11bHRpcGFydFVwbG9hZHMnLFxuICAgICAgICAgICdzMzpQdXRPYmplY3QnLFxuICAgICAgICAgICdnbHVlOkdldFRhYmxlJyxcbiAgICAgICAgICAnZ2x1ZTpHZXREYXRhYmFzZScsXG4gICAgICAgICAgJ2dsdWU6VXBkYXRlRGF0YWJhc2UnLFxuICAgICAgICAgICdnbHVlOkNyZWF0ZURhdGFiYXNlJyxcbiAgICAgICAgICAnZ2x1ZTpCYXRjaENyZWF0ZVBhcnRpdGlvbicsXG4gICAgICAgICAgJ2xvZ3M6Q3JlYXRlTG9nR3JvdXAnLFxuICAgICAgICAgICdsb2dzOkNyZWF0ZUxvZ1N0cmVhbScsXG4gICAgICAgICAgJ2xvZ3M6UHV0TG9nRXZlbnRzJyxcbiAgICAgICAgXSxcbiAgICAgICAgcmVzb3VyY2VzOiBbJyonXSxcbiAgICAgIH0pXG4gICAgKTtcblxuICAgIGFkZEF0aGVuYVBhcnRpdGlvbnNMYW1iZGEuYWRkVG9Sb2xlUG9saWN5KFxuICAgICAgbmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgICBlZmZlY3Q6IGlhbS5FZmZlY3QuQUxMT1csXG4gICAgICAgIGFjdGlvbnM6IFsnYXRoZW5hOlN0YXJ0UXVlcnlFeGVjdXRpb24nXSxcbiAgICAgICAgcmVzb3VyY2VzOiBbJyonXSxcbiAgICAgIH0pXG4gICAgKTtcblxuICAgIGxvZ1BhcnNlckxhbWJkYS5hZGRUb1JvbGVQb2xpY3koXG4gICAgICBuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgIGVmZmVjdDogaWFtLkVmZmVjdC5BTExPVyxcbiAgICAgICAgYWN0aW9uczogWydnbHVlOkdldFRhYmxlJywgJ2dsdWU6R2V0UGFydGl0aW9ucyddLFxuICAgICAgICByZXNvdXJjZXM6IFtcbiAgICAgICAgICBnbHVlQWNjZXNzTG9nc0RhdGFiYXNlLmNhdGFsb2dBcm4sXG4gICAgICAgICAgZ2x1ZUFjY2Vzc0xvZ3NEYXRhYmFzZS5kYXRhYmFzZUFybixcbiAgICAgICAgICBgYXJuOiR7Y2RrLkF3cy5QQVJUSVRJT059OmdsdWU6JHtjZGsuQXdzLlJFR0lPTn06JHtjZGsuQXdzLkFDQ09VTlRfSUR9OnRhYmxlLyR7Z2x1ZUFjY2Vzc0xvZ3NEYXRhYmFzZS5kYXRhYmFzZU5hbWV9L3dhZl9hY2Nlc3NfbG9nc2AsXG4gICAgICAgICAgYGFybjoke2Nkay5Bd3MuUEFSVElUSU9OfTpnbHVlOiR7Y2RrLkF3cy5SRUdJT059OiR7Y2RrLkF3cy5BQ0NPVU5UX0lEfTp0YWJsZS8ke2dsdWVBY2Nlc3NMb2dzRGF0YWJhc2UuZGF0YWJhc2VOYW1lfS9hcHBfYWNjZXNzX2xvZ3NgLFxuICAgICAgICBdLFxuICAgICAgfSlcbiAgICApO1xuXG4gICAgbG9nUGFyc2VyTGFtYmRhLmFkZFRvUm9sZVBvbGljeShcbiAgICAgIG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgZWZmZWN0OiBpYW0uRWZmZWN0LkFMTE9XLFxuICAgICAgICBhY3Rpb25zOiBbJ3dhZnYyOkdldElQU2V0JywgJ3dhZnYyOlVwZGF0ZUlQU2V0J10sXG4gICAgICAgIHJlc291cmNlczogW1xuICAgICAgICAgIGh0dHBGbG9vZElwU2V0VjQuYXR0ckFybixcbiAgICAgICAgICBodHRwRmxvb2RJcFNldFY2LmF0dHJBcm4sXG4gICAgICAgICAgc2Nhbm5lcnNQcm9iZXNJcFNldFY0LmF0dHJBcm4sXG4gICAgICAgICAgc2Nhbm5lcnNQcm9iZXNJcFNldFY2LmF0dHJBcm4sXG4gICAgICAgIF0sXG4gICAgICB9KVxuICAgICk7XG5cbiAgICBsb2dQYXJzZXJMYW1iZGEuYWRkVG9Sb2xlUG9saWN5KFxuICAgICAgbmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgICBlZmZlY3Q6IGlhbS5FZmZlY3QuQUxMT1csXG4gICAgICAgIGFjdGlvbnM6IFsnYXRoZW5hOkdldE5hbWVkUXVlcnknLCAnYXRoZW5hOlN0YXJ0UXVlcnlFeGVjdXRpb24nXSxcbiAgICAgICAgcmVzb3VyY2VzOiBbXG4gICAgICAgICAgYGFybjoke2Nkay5Bd3MuUEFSVElUSU9OfTphdGhlbmE6JHtjZGsuQXdzLlJFR0lPTn06JHtjZGsuQXdzLkFDQ09VTlRfSUR9Ondvcmtncm91cC9XQUYqYCxcbiAgICAgICAgXSxcbiAgICAgIH0pXG4gICAgKTtcblxuICAgIGxvZ1BhcnNlckxhbWJkYS5hZGRUb1JvbGVQb2xpY3koXG4gICAgICBuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgIGVmZmVjdDogaWFtLkVmZmVjdC5BTExPVyxcbiAgICAgICAgYWN0aW9uczogWydjbG91ZHdhdGNoOkdldE1ldHJpY1N0YXRpc3RpY3MnXSxcbiAgICAgICAgcmVzb3VyY2VzOiBbJyonXSxcbiAgICAgIH0pXG4gICAgKTtcblxuICAgIGxvZ1BhcnNlckxhbWJkYS5hZGRUb1JvbGVQb2xpY3koXG4gICAgICBuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgIGVmZmVjdDogaWFtLkVmZmVjdC5BTExPVyxcbiAgICAgICAgYWN0aW9uczogW1xuICAgICAgICAgICdsb2dzOkNyZWF0ZUxvZ0dyb3VwJyxcbiAgICAgICAgICAnbG9nczpDcmVhdGVMb2dTdHJlYW0nLFxuICAgICAgICAgICdsb2dzOlB1dExvZ0V2ZW50cycsXG4gICAgICAgIF0sXG4gICAgICAgIHJlc291cmNlczogW1xuICAgICAgICAgIGBhcm46JHtjZGsuQXdzLlBBUlRJVElPTn06YXRoZW5hOiR7Y2RrLkF3cy5SRUdJT059OiR7Y2RrLkF3cy5BQ0NPVU5UX0lEfTpsb2ctZ3JvdXA6L2F3cy9sYW1iZGEvKkxvZ1BhcnNlcipgLFxuICAgICAgICBdLFxuICAgICAgfSlcbiAgICApO1xuXG4gICAgbG9nUGFyc2VyTGFtYmRhLmFkZFRvUm9sZVBvbGljeShcbiAgICAgIG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgZWZmZWN0OiBpYW0uRWZmZWN0LkFMTE9XLFxuICAgICAgICBhY3Rpb25zOiBbXG4gICAgICAgICAgJ3MzOkdldEJ1Y2tldExvY2F0aW9uJyxcbiAgICAgICAgICAnczM6R2V0T2JqZWN0JyxcbiAgICAgICAgICAnczM6TGlzdEJ1Y2tldCcsXG4gICAgICAgICAgJ3MzOkxpc3RCdWNrZXRNdWx0aXBhcnRVcGxvYWRzJyxcbiAgICAgICAgICAnczM6TGlzdE11bHRpcGFydFVwbG9hZFBhcnRzJyxcbiAgICAgICAgICAnczM6QWJvcnRNdWx0aXBhcnRVcGxvYWQnLFxuICAgICAgICAgICdzMzpDcmVhdGVCdWNrZXQnLFxuICAgICAgICAgICdzMzpQdXRPYmplY3QnLFxuICAgICAgICBdLFxuICAgICAgICByZXNvdXJjZXM6IFtcbiAgICAgICAgICBhY2Nlc3NMb2dCdWNrZXQuYnVja2V0QXJuICsgJy9hdGhlbmFfcmVzdWx0cy8qJyxcbiAgICAgICAgICBhY2Nlc3NMb2dCdWNrZXQuYnVja2V0QXJuICsgJy8qJyxcbiAgICAgICAgICBhY2Nlc3NMb2dCdWNrZXQuYnVja2V0QXJuLFxuICAgICAgICAgIHdhZkxvZ0J1Y2tldC5idWNrZXRBcm4gKyAnL2F0aGVuYV9yZXN1bHRzLyonLFxuICAgICAgICAgIHdhZkxvZ0J1Y2tldC5idWNrZXRBcm4gKyAnLyonLFxuICAgICAgICAgIHdhZkxvZ0J1Y2tldC5idWNrZXRBcm4sXG4gICAgICAgIF0sXG4gICAgICB9KVxuICAgICk7XG5cbiAgICAvLyBTM1xuICAgIGxvZ1BhcnNlckxhbWJkYS5hZGRUb1JvbGVQb2xpY3koXG4gICAgICBuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgIGVmZmVjdDogaWFtLkVmZmVjdC5BTExPVyxcbiAgICAgICAgYWN0aW9uczogW1xuICAgICAgICAgICdzMzpHZXRCdWNrZXRMb2NhdGlvbicsXG4gICAgICAgICAgJ3MzOkdldE9iamVjdCcsXG4gICAgICAgICAgJ3MzOkxpc3RCdWNrZXQnLFxuICAgICAgICAgICdzMzpMaXN0QnVja2V0TXVsdGlwYXJ0VXBsb2FkcycsXG4gICAgICAgICAgJ3MzOkxpc3RNdWx0aXBhcnRVcGxvYWRQYXJ0cycsXG4gICAgICAgICAgJ3MzOkFib3J0TXVsdGlwYXJ0VXBsb2FkJyxcbiAgICAgICAgICAnczM6Q3JlYXRlQnVja2V0JyxcbiAgICAgICAgICAnczM6UHV0T2JqZWN0JyxcbiAgICAgICAgXSxcbiAgICAgICAgcmVzb3VyY2VzOiBbXG4gICAgICAgICAgd2FmTG9nQnVja2V0LmJ1Y2tldEFybiArICcvYXRoZW5hX3Jlc3VsdHMvKicsXG4gICAgICAgICAgYWNjZXNzTG9nQnVja2V0LmJ1Y2tldEFybiArICcvYXRoZW5hX3Jlc3VsdHMvKicsXG4gICAgICAgICAgd2FmTG9nQnVja2V0LmJ1Y2tldEFybixcbiAgICAgICAgICBhY2Nlc3NMb2dCdWNrZXQuYnVja2V0QXJuLFxuICAgICAgICBdLFxuICAgICAgfSlcbiAgICApO1xuXG4gICAgbG9nUGFyc2VyTGFtYmRhLmFkZFRvUm9sZVBvbGljeShcbiAgICAgIG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgZWZmZWN0OiBpYW0uRWZmZWN0LkFMTE9XLFxuICAgICAgICBhY3Rpb25zOiBbJ3MzOkdldE9iamVjdCddLFxuICAgICAgICByZXNvdXJjZXM6IFtcbiAgICAgICAgICB3YWZMb2dCdWNrZXQuYnVja2V0QXJuICsgJy8qJyxcbiAgICAgICAgICBhY2Nlc3NMb2dCdWNrZXQuYnVja2V0QXJuICsgJy8qJyxcbiAgICAgICAgXSxcbiAgICAgIH0pXG4gICAgKTtcblxuICAgIGxvZ1BhcnNlckxhbWJkYS5hZGRUb1JvbGVQb2xpY3koXG4gICAgICBuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgIGVmZmVjdDogaWFtLkVmZmVjdC5BTExPVyxcbiAgICAgICAgYWN0aW9uczogWydzMzpQdXRPYmplY3QnXSxcbiAgICAgICAgcmVzb3VyY2VzOiBbXG4gICAgICAgICAgd2FmTG9nQnVja2V0LmJ1Y2tldEFybiArICcvJyArIHN0YWNrLnN0YWNrTmFtZSArICctd2FmX2xvZ19vdXQuanNvbicsXG4gICAgICAgICAgd2FmTG9nQnVja2V0LmJ1Y2tldEFybiArICcvJyArIHN0YWNrLnN0YWNrTmFtZSArICctd2FmX2xvZ19jb25mLmpzb24nLFxuICAgICAgICAgIGFjY2Vzc0xvZ0J1Y2tldC5idWNrZXRBcm4gK1xuICAgICAgICAgICAgJy8nICtcbiAgICAgICAgICAgIHN0YWNrLnN0YWNrTmFtZSArXG4gICAgICAgICAgICAnLXdhZl9sb2dfb3V0Lmpzb24nLFxuICAgICAgICAgIGFjY2Vzc0xvZ0J1Y2tldC5idWNrZXRBcm4gK1xuICAgICAgICAgICAgJy8nICtcbiAgICAgICAgICAgIHN0YWNrLnN0YWNrTmFtZSArXG4gICAgICAgICAgICAnLXdhZl9sb2dfY29uZi5qc29uJyxcbiAgICAgICAgXSxcbiAgICAgIH0pXG4gICAgKTtcblxuICAgIG1vdmVMb2dUb1BhcnRpdGlvbkxhbWJkYS5hZGRUb1JvbGVQb2xpY3koXG4gICAgICBuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgIGVmZmVjdDogaWFtLkVmZmVjdC5BTExPVyxcbiAgICAgICAgYWN0aW9uczogW1xuICAgICAgICAgICdsb2dzOkNyZWF0ZUxvZ0dyb3VwJyxcbiAgICAgICAgICAnbG9nczpDcmVhdGVMb2dTdHJlYW0nLFxuICAgICAgICAgICdsb2dzOlB1dExvZ0V2ZW50cycsXG4gICAgICAgIF0sXG4gICAgICAgIHJlc291cmNlczogW1xuICAgICAgICAgIGBhcm46JHtjZGsuQXdzLlBBUlRJVElPTn06YXRoZW5hOiR7Y2RrLkF3cy5SRUdJT059OiR7Y2RrLkF3cy5BQ0NPVU5UX0lEfTpsb2ctZ3JvdXA6L2F3cy9sYW1iZGEvKk1vdmVTM0xvZ3NGb3JQYXJ0aXRpb24qYCxcbiAgICAgICAgXSxcbiAgICAgIH0pXG4gICAgKTtcblxuICAgIG1vdmVMb2dUb1BhcnRpdGlvbkxhbWJkYS5hZGRUb1JvbGVQb2xpY3koXG4gICAgICBuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgIGVmZmVjdDogaWFtLkVmZmVjdC5BTExPVyxcbiAgICAgICAgYWN0aW9uczogWydzMzpHZXRPYmplY3QnLCAnczM6RGVsZXRlT2JqZWN0JywgJ3MzOlB1dE9iamVjdCddLFxuICAgICAgICByZXNvdXJjZXM6IFthY2Nlc3NMb2dCdWNrZXQuYnVja2V0QXJuICsgJy8qJ10sXG4gICAgICB9KVxuICAgICk7XG5cbiAgICBjdXN0b21SZXNvdXJjZUxhbWJkYS5hZGRUb1JvbGVQb2xpY3koXG4gICAgICBuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgIGVmZmVjdDogaWFtLkVmZmVjdC5BTExPVyxcbiAgICAgICAgYWN0aW9uczogWydjbG91ZGZvcm1hdGlvbjpEZXNjcmliZVN0YWNrcyddLFxuICAgICAgICByZXNvdXJjZXM6IFtcbiAgICAgICAgICBgYXJuOiR7Y2RrLkF3cy5QQVJUSVRJT059OmNsb3VkZm9ybWF0aW9uOiR7Y2RrLkF3cy5SRUdJT059OiR7Y2RrLkF3cy5BQ0NPVU5UX0lEfTpzdGFjay8ke2Nkay5TdGFjay5uYW1lfS8qYCxcbiAgICAgICAgXSxcbiAgICAgIH0pXG4gICAgKTtcblxuICAgIGN1c3RvbVJlc291cmNlTGFtYmRhLmFkZFRvUm9sZVBvbGljeShcbiAgICAgIG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgZWZmZWN0OiBpYW0uRWZmZWN0LkFMTE9XLFxuICAgICAgICBhY3Rpb25zOiBbJ2xhbWJkYTpJbnZva2VGdW5jdGlvbiddLFxuICAgICAgICByZXNvdXJjZXM6IFtcbiAgICAgICAgICBgYXJuOiR7Y2RrLkF3cy5QQVJUSVRJT059OmxhbWJkYToke2Nkay5Bd3MuUkVHSU9OfToke2Nkay5Bd3MuQUNDT1VOVF9JRH06ZnVuY3Rpb246JHtzdGFjay5zdGFja05hbWV9KmAsXG4gICAgICAgIF0sXG4gICAgICB9KVxuICAgICk7XG5cbiAgICBjdXN0b21SZXNvdXJjZUxhbWJkYS5hZGRUb1JvbGVQb2xpY3koXG4gICAgICBuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgIGVmZmVjdDogaWFtLkVmZmVjdC5BTExPVyxcbiAgICAgICAgYWN0aW9uczogW1xuICAgICAgICAgICd3YWZ2MjpHZXRJUFNldCcsXG4gICAgICAgICAgJ3dhZnYyOkRlbGV0ZUlQU2V0JyxcbiAgICAgICAgICAnd2FmdjI6R2V0V2ViQUNMJyxcbiAgICAgICAgICAnd2FmdjI6VXBkYXRlV2ViQUNMJyxcbiAgICAgICAgICAnd2FmdjI6RGVsZXRlTG9nZ2luZ0NvbmZpZ3VyYXRpb24nLFxuICAgICAgICAgICd3YWZ2MjpQdXRMb2dnaW5nQ29uZmlndXJhdGlvbicsXG4gICAgICAgIF0sXG4gICAgICAgIHJlc291cmNlczogW3dhZndlYi5hdHRyQXJuXSxcbiAgICAgIH0pXG4gICAgKTtcblxuICAgIGN1c3RvbVJlc291cmNlTGFtYmRhLmFkZFRvUm9sZVBvbGljeShcbiAgICAgIG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgZWZmZWN0OiBpYW0uRWZmZWN0LkFMTE9XLFxuICAgICAgICBhY3Rpb25zOiBbJ3dhZnYyOkdldElQU2V0JywgJ3dhZnYyOkRlbGV0ZUlQU2V0J10sXG4gICAgICAgIHJlc291cmNlczogW1xuICAgICAgICAgIGBhcm46JHtjZGsuQXdzLlBBUlRJVElPTn06bG9nczoke2Nkay5Bd3MuUkVHSU9OfToke2Nkay5Bd3MuQUNDT1VOVF9JRH06cmVnaW9uYWwvaXBzZXQvJHtzdGFjay5zdGFja05hbWV9KmAsXG4gICAgICAgICAgYGFybjoke2Nkay5Bd3MuUEFSVElUSU9OfTpsb2dzOiR7Y2RrLkF3cy5SRUdJT059OiR7Y2RrLkF3cy5BQ0NPVU5UX0lEfTpnbG9iYWwvaXBzZXQvJHtzdGFjay5zdGFja05hbWV9KmAsXG4gICAgICAgIF0sXG4gICAgICB9KVxuICAgICk7XG5cbiAgICBjdXN0b21SZXNvdXJjZUxhbWJkYS5hZGRUb1JvbGVQb2xpY3koXG4gICAgICBuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgIGVmZmVjdDogaWFtLkVmZmVjdC5BTExPVyxcbiAgICAgICAgYWN0aW9uczogW1xuICAgICAgICAgICdsb2dzOkNyZWF0ZUxvZ0dyb3VwJyxcbiAgICAgICAgICAnbG9nczpDcmVhdGVMb2dTdHJlYW0nLFxuICAgICAgICAgICdsb2dzOlB1dExvZ0V2ZW50cycsXG4gICAgICAgIF0sXG4gICAgICAgIHJlc291cmNlczogW1xuICAgICAgICAgIGBhcm46JHtjZGsuQXdzLlBBUlRJVElPTn06bG9nczoke2Nkay5Bd3MuUkVHSU9OfToke2Nkay5Bd3MuQUNDT1VOVF9JRH06bG9nLWdyb3VwOi9hd3MvbGFtYmRhLypDdXN0b21SZXNvdXJjZSpgLFxuICAgICAgICBdLFxuICAgICAgfSlcbiAgICApO1xuXG4gICAgY3VzdG9tUmVzb3VyY2VMYW1iZGEuYWRkVG9Sb2xlUG9saWN5KFxuICAgICAgbmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgICBlZmZlY3Q6IGlhbS5FZmZlY3QuQUxMT1csXG4gICAgICAgIGFjdGlvbnM6IFtcbiAgICAgICAgICAnczM6R2V0QnVja2V0TG9jYXRpb24nLFxuICAgICAgICAgICdzMzpMaXN0QnVja2V0JyxcbiAgICAgICAgICAnczM6Q3JlYXRlQnVja2V0JyxcbiAgICAgICAgICAnczM6R2V0QnVja2V0Tm90aWZpY2F0aW9uJyxcbiAgICAgICAgICAnczM6UHV0QnVja2V0Tm90aWZpY2F0aW9uJyxcbiAgICAgICAgICAnczM6UHV0RW5jcnlwdGlvbkNvbmZpZ3VyYXRpb24nLFxuICAgICAgICAgICdzMzpQdXRCdWNrZXRQdWJsaWNBY2Nlc3NCbG9jaycsXG4gICAgICAgICAgJ3MzOkdldE9iamVjdConLFxuICAgICAgICAgICdzMzpQdXRPYmplY3QqJyxcbiAgICAgICAgICAnczM6RGVsZXRlT2JqZWN0KicsXG4gICAgICAgIF0sXG4gICAgICAgIHJlc291cmNlczogW1xuICAgICAgICAgIGFjY2Vzc0xvZ0J1Y2tldC5idWNrZXRBcm4sXG4gICAgICAgICAgYCR7YWNjZXNzTG9nQnVja2V0LmJ1Y2tldEFybn0vKmAsXG4gICAgICAgIF0sXG4gICAgICB9KVxuICAgICk7XG5cbiAgICBjdXN0b21SZXNvdXJjZUxhbWJkYS5hZGRUb1JvbGVQb2xpY3koXG4gICAgICBuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgIGVmZmVjdDogaWFtLkVmZmVjdC5BTExPVyxcbiAgICAgICAgYWN0aW9uczogW1xuICAgICAgICAgICdzMzpDcmVhdGVCdWNrZXQnLFxuICAgICAgICAgICdzMzpHZXRCdWNrZXROb3RpZmljYXRpb24nLFxuICAgICAgICAgICdzMzpQdXRCdWNrZXROb3RpZmljYXRpb24nLFxuICAgICAgICAgICdzMzpHZXRPYmplY3QqJyxcbiAgICAgICAgICAnczM6UHV0T2JqZWN0KicsXG4gICAgICAgICAgJ3MzOkRlbGV0ZU9iamVjdConLFxuICAgICAgICBdLFxuICAgICAgICByZXNvdXJjZXM6IFt3YWZMb2dCdWNrZXQuYnVja2V0QXJuLCBgJHt3YWZMb2dCdWNrZXQuYnVja2V0QXJufS8qYF0sXG4gICAgICB9KVxuICAgICk7XG5cbiAgICBjdXN0b21SZXNvdXJjZUxhbWJkYS5hZGRUb1JvbGVQb2xpY3koXG4gICAgICBuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgIGVmZmVjdDogaWFtLkVmZmVjdC5BTExPVyxcbiAgICAgICAgYWN0aW9uczogWydpYW06Q3JlYXRlU2VydmljZUxpbmtlZFJvbGUnXSxcbiAgICAgICAgcmVzb3VyY2VzOiBbXG4gICAgICAgICAgJ2Fybjphd3M6aWFtOjoqOnJvbGUvYXdzLXNlcnZpY2Utcm9sZS93YWZ2Mi5hbWF6b25hd3MuY29tL0FXU1NlcnZpY2VSb2xlRm9yV0FGVjJMb2dnaW5nJyxcbiAgICAgICAgXSxcbiAgICAgIH0pXG4gICAgKTtcblxuICAgIHdhZkxvZ0J1Y2tldC5hZGRFdmVudE5vdGlmaWNhdGlvbihcbiAgICAgIHMzLkV2ZW50VHlwZS5PQkpFQ1RfQ1JFQVRFRCxcbiAgICAgIG5ldyBzM24uTGFtYmRhRGVzdGluYXRpb24obG9nUGFyc2VyTGFtYmRhKSxcbiAgICAgIHtcbiAgICAgICAgcHJlZml4OiAnYXRoZW5hX3Jlc3VsdHMvJyxcbiAgICAgICAgc3VmZml4OiAnY3N2JyxcbiAgICAgIH1cbiAgICApO1xuXG4gICAgYWNjZXNzTG9nQnVja2V0LmFkZEV2ZW50Tm90aWZpY2F0aW9uKFxuICAgICAgczMuRXZlbnRUeXBlLk9CSkVDVF9DUkVBVEVELFxuICAgICAgbmV3IHMzbi5MYW1iZGFEZXN0aW5hdGlvbihtb3ZlTG9nVG9QYXJ0aXRpb25MYW1iZGEpLFxuICAgICAge1xuICAgICAgICBwcmVmaXg6ICdBV1NMb2dzLycsXG4gICAgICAgIHN1ZmZpeDogJ2d6JyxcbiAgICAgIH1cbiAgICApO1xuXG4gICAgYWNjZXNzTG9nQnVja2V0LmFkZEV2ZW50Tm90aWZpY2F0aW9uKFxuICAgICAgczMuRXZlbnRUeXBlLk9CSkVDVF9DUkVBVEVELFxuICAgICAgbmV3IHMzbi5MYW1iZGFEZXN0aW5hdGlvbihsb2dQYXJzZXJMYW1iZGEpLFxuICAgICAge1xuICAgICAgICBwcmVmaXg6ICdhdGhlbmFfcmVzdWx0cy8nLFxuICAgICAgICBzdWZmaXg6ICdjc3YnLFxuICAgICAgfVxuICAgICk7XG5cbiAgICBjb25zdCBjdXN0b21SZXNvdXJjZVByb3ZpZGVyID0gbmV3IGNyLlByb3ZpZGVyKFxuICAgICAgdGhpcyxcbiAgICAgICdjdXN0b21SZXNvdXJjZVByb3ZpZGVyJyxcbiAgICAgIHtcbiAgICAgICAgb25FdmVudEhhbmRsZXI6IGN1c3RvbVJlc291cmNlTGFtYmRhLFxuICAgICAgfVxuICAgICk7XG5cbiAgICBuZXcgY2RrLkN1c3RvbVJlc291cmNlKHRoaXMsICdDb25maWd1cmVBV1NXQUZMb2dzJywge1xuICAgICAgc2VydmljZVRva2VuOiBjdXN0b21SZXNvdXJjZVByb3ZpZGVyLnNlcnZpY2VUb2tlbixcbiAgICAgIHJlc291cmNlVHlwZTogJ0N1c3RvbTo6Q29uZmlndXJlQVdTV0FGTG9ncycsXG4gICAgICBwcm9wZXJ0aWVzOiB7XG4gICAgICAgIFdBRldlYkFDTEFybjogd2Fmd2ViLmF0dHJBcm4sXG4gICAgICAgIERlbGl2ZXJ5U3RyZWFtQXJuOiBmaXJlaG9zZVN0cmVhbS5hdHRyQXJuLFxuICAgICAgfSxcbiAgICB9KTtcblxuICAgIG5ldyBjZGsuQ3VzdG9tUmVzb3VyY2UodGhpcywgJ0NvbmZpZ3VyZUFwcEFjY2Vzc0xvZ0J1Y2tldCcsIHtcbiAgICAgIHNlcnZpY2VUb2tlbjogY3VzdG9tUmVzb3VyY2VQcm92aWRlci5zZXJ2aWNlVG9rZW4sXG4gICAgICByZXNvdXJjZVR5cGU6ICdDdXN0b206OkNvbmZpZ3VyZUFwcEFjY2Vzc0xvZ0J1Y2tldCcsXG4gICAgICBwcm9wZXJ0aWVzOiB7XG4gICAgICAgIFJlZ2lvbjogY2RrLkF3cy5SRUdJT04sXG4gICAgICAgIEFwcEFjY2Vzc0xvZ0J1Y2tldDogYWNjZXNzTG9nQnVja2V0LmJ1Y2tldE5hbWUsXG4gICAgICAgIExvZ1BhcnNlcjogbG9nUGFyc2VyTGFtYmRhLmZ1bmN0aW9uQXJuLFxuICAgICAgICBTY2FubmVyc1Byb2Jlc0F0aGVuYUxvZ1BhcnNlcjogJ3llcycsXG4gICAgICAgIE1vdmVTM0xvZ3NGb3JQYXJ0aXRpb246IG1vdmVMb2dUb1BhcnRpdGlvbkxhbWJkYS5mdW5jdGlvbk5hbWUsXG4gICAgICB9LFxuICAgIH0pO1xuXG4gICAgbmV3IGNkay5DdXN0b21SZXNvdXJjZSh0aGlzLCAnQ29uZmlndXJlV2FmTG9nQnVja2V0Jywge1xuICAgICAgc2VydmljZVRva2VuOiBjdXN0b21SZXNvdXJjZVByb3ZpZGVyLnNlcnZpY2VUb2tlbixcbiAgICAgIHJlc291cmNlVHlwZTogJ0N1c3RvbTo6Q29uZmlndXJlV2FmTG9nQnVja2V0JyxcbiAgICAgIHByb3BlcnRpZXM6IHtcbiAgICAgICAgUmVnaW9uOiBjZGsuQXdzLlJFR0lPTixcbiAgICAgICAgV2FmTG9nQnVja2V0OiB3YWZMb2dCdWNrZXQuYnVja2V0TmFtZSxcbiAgICAgICAgTG9nUGFyc2VyOiBsb2dQYXJzZXJMYW1iZGEuZnVuY3Rpb25Bcm4sXG4gICAgICAgIEh0dHBGbG9vZEF0aGVuYUxvZ1BhcnNlcjogJ3llcycsXG4gICAgICB9LFxuICAgIH0pO1xuXG4gICAgbmV3IGNkay5DdXN0b21SZXNvdXJjZSh0aGlzLCAnQ29uZmlndXJlV2ViQWNsJywge1xuICAgICAgc2VydmljZVRva2VuOiBjdXN0b21SZXNvdXJjZVByb3ZpZGVyLnNlcnZpY2VUb2tlbixcbiAgICAgIHJlc291cmNlVHlwZTogJ0N1c3RvbTo6Q29uZmlndXJlV2ViQWNsJyxcbiAgICAgIHByb3BlcnRpZXM6IHtcbiAgICAgICAgV0FGV2hpdGVsaXN0U2V0SVBWNDogd2hpdGVsaXN0SXBTZXRWNC5hdHRySWQsXG4gICAgICAgIFdBRldoaXRlbGlzdFNldElQVjROYW1lOiB3aGl0ZWxpc3RJcFNldFY0Lm5hbWUsXG4gICAgICB9LFxuICAgIH0pO1xuXG4gICAgbmV3IGNkay5DdXN0b21SZXNvdXJjZSh0aGlzLCAnR2VuZXJhdGVBcHBMb2dQYXJzZXJDb25mRmlsZScsIHtcbiAgICAgIHNlcnZpY2VUb2tlbjogY3VzdG9tUmVzb3VyY2VQcm92aWRlci5zZXJ2aWNlVG9rZW4sXG4gICAgICByZXNvdXJjZVR5cGU6ICdDdXN0b206OkdlbmVyYXRlQXBwTG9nUGFyc2VyQ29uZkZpbGUnLFxuICAgICAgcHJvcGVydGllczoge1xuICAgICAgICBTdGFja05hbWU6IHN0YWNrLnN0YWNrTmFtZSxcbiAgICAgICAgRXJyb3JUaHJlc2hvbGQ6IGVycm9yVGhyZXNob2xkLFxuICAgICAgICBXQUZCbG9ja1BlcmlvZDogYmxvY2tQZXJpb2QsXG4gICAgICAgIEFwcEFjY2Vzc0xvZ0J1Y2tldDogYWNjZXNzTG9nQnVja2V0LmJ1Y2tldE5hbWUsXG4gICAgICB9LFxuICAgIH0pO1xuXG4gICAgbmV3IGNkay5DdXN0b21SZXNvdXJjZSh0aGlzLCAnR2VuZXJhdGVXYWZMb2dQYXJzZXJDb25mRmlsZScsIHtcbiAgICAgIHNlcnZpY2VUb2tlbjogY3VzdG9tUmVzb3VyY2VQcm92aWRlci5zZXJ2aWNlVG9rZW4sXG4gICAgICByZXNvdXJjZVR5cGU6ICdDdXN0b206OkdlbmVyYXRlV2FmTG9nUGFyc2VyQ29uZkZpbGUnLFxuICAgICAgcHJvcGVydGllczoge1xuICAgICAgICBTdGFja05hbWU6IHN0YWNrLnN0YWNrTmFtZSxcbiAgICAgICAgUmVxdWVzdFRocmVzaG9sZDogcmVxdWVzdFRocmVzaG9sZCxcbiAgICAgICAgV0FGQmxvY2tQZXJpb2Q6IGJsb2NrUGVyaW9kLFxuICAgICAgICBXYWZBY2Nlc3NMb2dCdWNrZXQ6IHdhZkxvZ0J1Y2tldC5idWNrZXROYW1lLFxuICAgICAgfSxcbiAgICB9KTtcblxuICAgIG5ldyBjZGsuQ3VzdG9tUmVzb3VyY2UodGhpcywgJ0FkZEF0aGVuYVBhcnRpdGlvbnMnLCB7XG4gICAgICBzZXJ2aWNlVG9rZW46IGN1c3RvbVJlc291cmNlUHJvdmlkZXIuc2VydmljZVRva2VuLFxuICAgICAgcmVzb3VyY2VUeXBlOiAnQ3VzdG9tOjpBZGRBdGhlbmFQYXJ0aXRpb25zJyxcbiAgICAgIHByb3BlcnRpZXM6IHtcbiAgICAgICAgUmVzb3VyY2VUeXBlOiAnQ3VzdG9tOjpBZGRBdGhlbmFQYXJ0aXRpb25zJyxcbiAgICAgICAgQWRkQXRoZW5hUGFydGl0aW9uc0xhbWJkYTogYWRkQXRoZW5hUGFydGl0aW9uc0xhbWJkYS5mdW5jdGlvbk5hbWUsXG4gICAgICAgIEdsdWVBY2Nlc3NMb2dzRGF0YWJhc2U6IGdsdWVBY2Nlc3NMb2dzRGF0YWJhc2UuZGF0YWJhc2VOYW1lLFxuICAgICAgICBBcHBBY2Nlc3NMb2dCdWNrZXQ6IGFjY2Vzc0xvZ0J1Y2tldC5idWNrZXROYW1lLFxuICAgICAgICBHbHVlQXBwQWNjZXNzTG9nc1RhYmxlOiAnYXBwX2FjY2Vzc19sb2dzJyxcbiAgICAgICAgR2x1ZVdhZkFjY2Vzc0xvZ3NUYWJsZTogZ2x1ZVdhZkFjY2Vzc0xvZ3NUYWJsZS50YWJsZU5hbWUsXG4gICAgICAgIFdhZkxvZ0J1Y2tldDogd2FmTG9nQnVja2V0LmJ1Y2tldE5hbWUsXG4gICAgICAgIEF0aGVuYVdvcmtHcm91cDogYWRkUGFydGl0aW9uQXRoZW5hUXVlcnlXb3JrR3JvdXAubmFtZSxcbiAgICAgIH0sXG4gICAgfSk7XG5cbiAgICBuZXcgY2RrLkNmbk91dHB1dCh0aGlzLCAnQXBwQWNjZXNzTG9nQnVja2V0TmFtZScsIHtcbiAgICAgIHZhbHVlOiBhY2Nlc3NMb2dCdWNrZXQuYnVja2V0TmFtZSxcbiAgICB9KTtcblxuICAgIG5ldyBjZGsuQ2ZuT3V0cHV0KHRoaXMsICdXYWZMb2dCdWNrZXROYW1lJywge1xuICAgICAgdmFsdWU6IHdhZkxvZ0J1Y2tldC5idWNrZXROYW1lLFxuICAgIH0pO1xuXG4gICAgbmV3IGNkay5DZm5PdXRwdXQodGhpcywgJ1dBRldlYkFDTE5hbWUnLCB7IHZhbHVlOiB3YWZ3ZWIuYXR0cklkIH0pO1xuXG4gICAgbmV3IGNkay5DZm5PdXRwdXQodGhpcywgJ1dBRldlYkFDTEFybicsIHsgdmFsdWU6IHdhZndlYi5hdHRyQXJuIH0pO1xuICB9XG5cbiAgdmFsaWRhdGVSZXNvdXJjZU5hbWluZ1ByZWZpeChyZXNvdXJjZU5hbWluZ1ByZWZpeDogc3RyaW5nKTogYm9vbGVhbiB7XG4gICAgcmV0dXJuIC9eW2EtekEtWl0rW2EtekEtWjAtOV9dKyQvLnRlc3QocmVzb3VyY2VOYW1pbmdQcmVmaXgpO1xuICB9XG59XG4iXX0=