# -*- coding: utf-8 -*-
from setuptools import setup

package_dir = \
{'': 'src'}

packages = \
['tracers', 'tracers.function']

package_data = \
{'': ['*']}

setup_kwargs = {
    'name': 'tracers',
    'version': '20.5.23480',
    'description': 'Decorator-like performance tracing tool',
    'long_description': "# Table of contents\n\n- [Table of contents](#table-of-contents)\n- [Project description](#project-description)\n  * [Key Features](#key-features)\n- [Quick Introduction](#quick-introduction)\n- [Installation](#installation)\n- [Examples](#examples)\n\n# Project description\n\n**Tracers** is an Open-Source **APM** (Application monitoring) project\nthat offers you zero overhead wrappers for profiling your code execution flow.\n\n```\n🛈  Finished transaction: 332eeaef449841c48d5daf0d2a551fe8, 0.81 seconds\n\n  #    Timestamp      %     Total    Nested Call Chain\n\n     1     0.00s 100.0%     0.81s    ✓ async function_a()\n     2     0.00s  12.4%     0.10s    ¦   ✓ async asyncio.tasks.sleep(delay, result=None, *, loop=None)\n     3     0.10s  87.5%     0.71s    ¦   ✓ async function_b()\n     4     0.10s  12.5%     0.10s    ¦   ¦   ✓ async asyncio.tasks.sleep(delay, result=None, *, loop=None)\n     5     0.20s  25.0%     0.20s    ¦   ¦   ✓ async function_c()\n     6     0.20s  12.5%     0.10s    ¦   ¦   ¦   ✓ async asyncio.tasks.sleep(delay, result=None, *, loop=None)\n     7     0.30s  12.5%     0.10s    ¦   ¦   ¦   ✓ async function_d()\n     8     0.30s  12.5%     0.10s    ¦   ¦   ¦   ¦   ✓ async asyncio.tasks.sleep(delay, result=None, *, loop=None)\n     9     0.41s  12.5%     0.10s    ¦   ¦   ✓ async asyncio.tasks.sleep(delay, result=None, *, loop=None)\n    10     0.51s  12.5%     0.10s    ¦   ¦   ✓ async function_d()\n    11     0.51s  12.5%     0.10s    ¦   ¦   ¦   ✓ async asyncio.tasks.sleep(delay, result=None, *, loop=None)\n    12     0.61s  12.4%     0.10s    ¦   ¦   ✓ async asyncio.tasks.sleep(delay, result=None, *, loop=None)\n    13     0.71s  12.5%     0.10s    ¦   ¦   ✓ async function_e()\n    14     0.71s  12.5%     0.10s    ¦   ¦   ¦   ✓ async asyncio.tasks.sleep(delay, result=None, *, loop=None)\n```\n\n## Key Features\n\n- Handles **any callable** object, which includes **your own code**,\n  **third party libraries**, and even the low-level **Python standard library**\n- Handles [**async**](https://docs.python.org/3/library/asyncio.html) code\n  **out-of-the box**, no config required\n- Exposes a **high-level API**:\n  - `@trace` decorator (which internally handles async/sync cases)\n- It's **Thread-safe**, **Async-safe**, **Process-safe** and **Context-safe**.\n  - Accurate results in any scenario\n  - No worries about leaking, bleeding corrupting, or locking stuff into other\n    code\n- Introduces **zero overhead** in production!\n  - The `@trace` decorator accepts a `do_trace` parameter\n    that you can dynamically set to `True` of `False` to differentiate\n    testing environments from production environments.\n- It's **easy to deploy**\n  - Has no external dependencies\n- Profiles without using dirty introspection stuff\n  - The main code is just 50 lines long, pretty high level, go and read it :)\n\n# Quick Introduction\n\nLet's start with a very basic example:\n\n```py\nimport time\n\n\ndef example():\n    time.sleep(2.0)\n    your_business_logic('Sat Oct 11')\n\n\ndef your_business_logic(date: str):\n    time.sleep(1.0)\n\n\nexample()\n```\n\nTracing its flow and gathering profiling information is a matter of\ndecorating your functions:\n\n```diff\n--- a/examples/without_tracers.py\n+++ b/examples/with_tracers.py\n@@ -1,15 +1,18 @@\n import time\n from dateutil.parser import parse\n+from tracers.function import trace\n\n\n+@trace\n def example():\n     time.sleep(2.0)\n     your_business_logic('Sat Oct 11')\n\n\n+@trace\n def your_business_logic(date: str):\n     parse(date)\n     time.sleep(1.0)\n\n\n example()\n```\n\nIf you run it, all the functions you decorated will be traced\nand you'll have metrics of the execution flow:\n\n```\n🛈  Finished transaction: ce72c9dbe3d64e4cb43714fb87738ac4, 3.00 seconds\n\n  #    Timestamp      %     Total    Nested Call Chain\n\n     1     0.00s 100.0%     3.00s    ✓ example()\n     2     2.00s  33.4%     1.00s    ¦   ✓ your_business_logic(date: str)\n```\n\nFrom the output you can conclude:\n- executing function *example* took a total of *3.0* seconds to complete\n- function *example* represents *100%* of your code time\n- function *example* called function: *your_business_logic*\n- function *your_business_logic* took *1.0* seconds out of the *3.0* seconds\n  the function *example* needed to complete\n- function *your_business_logic* represents *33.4%* of your execution time\n- There is *66.6%* of execution time\n  that we've not instrumented... yet!\n\nTracing code is not limited to your own code.\nYou can trace any **callable object** including **third party packages**,\n**Python's standard library**, and almost anything.\n\nThe level of detail is up to you!\n\n\n```py\n--- a/examples/with_tracers.py\n+++ b/examples/with_detailed_tracers.py\n@@ -1,18 +1,18 @@\n import time\n from dateutil.parser import parse\n from tracers.function import trace\n\n\n @trace\n def example():\n-    time.sleep(2.0)\n+    trace(time.sleep)(2.0)\n     your_business_logic('Sat Oct 11')\n\n\n @trace\n def your_business_logic(date: str):\n-    parse(date)\n-    time.sleep(1.0)\n+    trace(parse)(date)\n+    trace(time.sleep)(1.0)\n\n\n example()\n```\n\n```\n🛈  Finished transaction: 10b3878b12e647c1b326a9c55f954537, 3.00 seconds\n\n  #    Timestamp      %     Total    Nested Call Chain\n\n     1     0.00s 100.0%     3.00s    ✓ example()\n     2     0.00s  66.6%     2.00s    ¦   ✓ time.sleep(...)\n     3     2.00s  33.4%     1.00s    ¦   ✓ your_business_logic(date: str)\n     4     2.00s   0.0%     0.00s    ¦   ¦   ✓ dateutil.parser._parser.parse(timestr, parserinfo=None, **kwargs)\n     5     2.00s  33.3%     1.00s    ¦   ¦   ✓ time.sleep(...)\n```\n\n# Installation\n\nWe are hosted on **PyPi**: https://pypi.org/project/tracers\n\nJust run: `pip install tracers`\nor use the package manager you like the most\n\n# Examples\n\nCheck them out in the\n[examples](https://github.com/kamadorueda/tracers/tree/master/examples)\nfolder\n",
    'author': 'Kevin Amado',
    'author_email': 'kamadorueda@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/kamadorueda/tracers',
    'package_dir': package_dir,
    'packages': packages,
    'package_data': package_data,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
