"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const client_secrets_manager_1 = require("@aws-sdk/client-secrets-manager");
const core_1 = require("@octokit/core");
const aws_secret_helper_1 = require("../aws-secret-helper");
const github_helper_1 = require("../github-helper");
const github_secret_encryptor_1 = require("../github-secret-encryptor");
const github_secret_name_validator_1 = require("../github-secret-name-validator");
const onEvent = async (event) => {
    console.log(`Event: ${JSON.stringify(event)}`);
    github_secret_name_validator_1.validateSecretName(event.ResourceProperties.repositorySecretName);
    const smClient = new client_secrets_manager_1.SecretsManager({ region: event.ResourceProperties.awsRegion });
    const githubTokenSecret = await smClient.getSecretValue({ SecretId: event.ResourceProperties.githubTokenSecret });
    const octokit = new core_1.Octokit({ auth: githubTokenSecret.SecretString });
    const requestType = event.RequestType;
    switch (requestType) {
        case 'Create':
            return onCreate(event, octokit, smClient);
        case 'Update':
            return onUpdate(event, octokit, smClient);
        case 'Delete':
            return onDelete(event, octokit);
        default:
            throw new Error(`Unexpected request type: '${requestType}'`);
    }
};
const onCreate = async (event, octokit, smClient) => {
    const props = event.ResourceProperties;
    console.log('Create new ActionSecret with props ' + JSON.stringify(props));
    await createOrUpdateRepoSecret(event, octokit, smClient);
    const PhysicalResourceId = await buildPhysicalResourceId(event, octokit);
    return { PhysicalResourceId };
};
const onUpdate = async (event, octokit, smClient) => {
    const props = event.ResourceProperties;
    console.log(`Update ActionSecret ${props.repositorySecretName} with props ${JSON.stringify(props)}`);
    await createOrUpdateRepoSecret(event, octokit, smClient);
    const PhysicalResourceId = await buildPhysicalResourceId(event, octokit);
    return { PhysicalResourceId };
};
const onDelete = async (event, octokit) => {
    console.log('Delete ActionSecret ' + event.ResourceProperties.repositorySecretName);
    await deleteRepoSecret(event, octokit);
    const PhysicalResourceId = await buildPhysicalResourceId(event, octokit);
    return { PhysicalResourceId };
};
const createOrUpdateRepoSecret = async (event, octokit, smClient) => {
    const { repositoryOwner, repositoryName: repo, repositorySecretName: secret_name, sourceSecretArn: secretId, sourceSecretJsonField, } = event.ResourceProperties;
    console.log(`Encrypt value of secret with id: ${secretId}`);
    const secretString = await aws_secret_helper_1.getSecretString(secretId, smClient, sourceSecretJsonField);
    const owner = await github_helper_1.getOwner(octokit, repositoryOwner);
    const { data } = await octokit.request('GET /repos/{owner}/{repo}/actions/secrets/public-key', { owner, repo });
    const encryptedSecret = await github_secret_encryptor_1.encryptValue(secretString, data.key);
    console.log('Encrypted secret, attempting to create/update github secret');
    const secretResponse = await octokit.request('PUT /repos/{owner}/{repo}/actions/secrets/{secret_name}', {
        owner,
        repo,
        secret_name,
        encrypted_value: encryptedSecret,
        key_id: data.key_id,
    });
    console.log(JSON.stringify(secretResponse));
    return secretResponse;
};
const deleteRepoSecret = async (event, octokit) => {
    const { repositoryOwner, repositoryName: repo, repositorySecretName: secret_name } = event.ResourceProperties;
    const owner = await github_helper_1.getOwner(octokit, repositoryOwner);
    const deleteSecretResponse = await octokit.request('DELETE /repos/{owner}/{repo}/actions/secrets/{secret_name}', {
        owner,
        repo,
        secret_name,
    });
    console.log(`Delete: ${JSON.stringify(deleteSecretResponse)}`);
    return deleteSecretResponse;
};
const buildPhysicalResourceId = async (event, octokit) => {
    const { repositorySecretName: secret, repositoryOwner, repositoryName: repo } = event.ResourceProperties;
    const owner = await github_helper_1.getOwner(octokit, repositoryOwner);
    return [secret, owner, repo].join('/');
};
exports.handler = onEvent;
//# sourceMappingURL=data:application/json;base64,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