from enum import Enum
from typing import TypeAlias, TypeGuard

import attrs


class TriggerEdge(Enum):
    RISING = "rising"
    FALLING = "falling"
    BOTH = "both"

    def __repr__(self):
        return f"{self.__class__.__name__}.{self.name}"


@attrs.define
class SoftwareTrigger:
    """Represents a trigger that is generated by a software command."""

    pass


@attrs.define
class ExternalTriggerStart:
    """A trigger generated by an external source once at the start of the sequence."""

    edge: TriggerEdge = TriggerEdge.RISING


@attrs.define
class ExternalClock:
    """A trigger generated by an external source for every time step."""

    edge: TriggerEdge = TriggerEdge.RISING


@attrs.define
class ExternalClockOnChange:
    """A trigger generated by an external source every time a value changes."""

    edge: TriggerEdge = TriggerEdge.RISING


Trigger: TypeAlias = (
    SoftwareTrigger | ExternalTriggerStart | ExternalClock | ExternalClockOnChange
)


def is_trigger(value) -> TypeGuard[Trigger]:
    return isinstance(
        value,
        (SoftwareTrigger, ExternalTriggerStart, ExternalClock, ExternalClockOnChange),
    )
