import abc
import concurrent.futures
import glob
import gzip
import multiprocessing
import os
import queue
import re
import signal
import threading
import time
import uuid

from functools import cached_property
from collections import UserDict, UserList

from searchkit.log import log

RESULTS_QUEUES = {}
RESULTS_QUEUE_TIMEOUT = 60
MAX_QUEUE_RETRIES = 10


class FileSearchException(Exception):
    def __init__(self, msg):
        self.msg = msg


class SearchDefBase(object):

    def __init__(self, constraints=None):
        """
        @param constraints: an optional list of constraints to apply to
                            results.
        """
        self.id
        self.constraints = {c.id: c for c in constraints or {}}

    @cached_property
    def id(self):
        """ A unique identifier for this search definition. """
        return str(uuid.uuid4())


class SearchDef(SearchDefBase):

    def __init__(self, pattern, tag=None, hint=None,
                 store_result_contents=True, **kwargs):
        """
        Simple search definition.

        @param pattern: pattern or list of patterns to search for
        @param tag: optional user-friendly identifier for this search term.
                    This is useful for retrieving results.
        @param hint: optional pre-search term. If provided, this is expected to
                     match in order for the main search to be executed.
        @param store_result_contents: by default the content of a search result
                                      is saved but if it is not needed this
                                      can be set to False. This effectively
                                      makes the result True/False.
        """
        if type(pattern) != list:
            self.patterns = [re.compile(pattern)]
        else:
            self.patterns = []
            for _pattern in pattern:
                self.patterns.append(re.compile(_pattern))

        self.store_result_contents = store_result_contents
        self.tag = tag
        self.hint = hint
        if hint:
            self.hint = re.compile(hint)

        # do this last
        super().__init__(**kwargs)

    def run(self, line):
        """ Execute search patterns against line and return first match. """
        if self.hint:
            ret = self.hint.search(line)
            if not ret:
                return None

        ret = None
        for pattern in self.patterns:
            ret = pattern.match(line)
            if ret:
                break

        return ret


class SequenceSearchDef(SearchDefBase):

    def __init__(self, start, tag, end=None, body=None, **kwargs):
        """
        Sequence search definition.

        A sequence must match a start and end with optional body in between.
        If no end is defined, the sequence ends with the start of the next or
        EOF.

        NOTE: sequences must not overlap. This is therefore not suitable for
        finding sequences generated by parallel/concurrent tasks.

        @param start: SearchDef object for matching start
        @param tag: tag used to identify this sequence definition
        @param end: optional SearchDef object for matching end
        @param body: optional SearchDef object for matching body
        """
        self.s_start = start
        self.s_end = end
        self.s_body = body
        self.tag = tag
        self._mark = None
        # Each section matched gets its own id. Since each file is processed
        # using a separate process and memory is not shared, these values must
        # be unique to avoid collisions when results are aggregated.
        self._section_id = None
        # do this last
        super().__init__(**kwargs)

    @property
    def start_tag(self):
        """ Tag used to identify start of section. """
        return "{}-start".format(self.tag)

    @property
    def end_tag(self):
        """ Tag used to identify end of section. """
        return "{}-end".format(self.tag)

    @property
    def body_tag(self):
        """ Tag used to identify body of section. """
        return "{}-body".format(self.tag)

    @property
    def section_id(self):
        """ ID of current section. A new id should be set after each
        completed section. """
        return self._section_id

    @property
    def started(self):
        """ Indicate a section sequence has been started. """
        return self._mark == 1

    def start(self):
        """ Indicate that a sequence start has been detected. """
        self._section_id = str(uuid.uuid4())
        self._mark = 1

    def reset(self):
        """ Used to restart a section. This is used e.g. if the start
        expression matches midway through a sequence (and before the end).
        """
        self._mark = 0

    def stop(self):
        """ Indicate that a sequence is complete. """
        self._mark = 0
        self._section_id = str(uuid.uuid4())


class SequenceSearchResults(UserDict):

    def __init__(self):
        self.data = {}

    def add(self, result):
        id = result.sequence_obj_id
        if id in self.data:
            self.data[id].append(result)
        else:
            self.data[id] = [result]

    def remove(self, id):
        if id in self.data:
            del self.data[id]


class SearchResultPart(object):

    def __init__(self, index, value):
        self.index = index
        self.value = value


class SearchResult(UserList):

    def __init__(self, linenumber, source_id, result, tag=None,
                 section_id=None, sequence_obj_id=None, store_contents=True):
        """
        @param linenumber: line number that produced a match.
        @param source_id: data source id - resolves to a path in the
                          SearchCatalog.
        @param result: python.re.match object.
        @param tag: SearchDef object tag.
        @param section_id: SequenceSearchDef object section id.
        @param sequence_obj_id: SequenceSearchDef object unique id.
        """
        self.data = []
        self.tag = tag
        self.source_id = source_id
        self.linenumber = linenumber
        self.sequence_obj_id = sequence_obj_id
        self.section_id = section_id

        if not store_contents:
            log.debug("store_contents is False - skipping save")
            return

        num_groups = len(result.groups())
        # NOTE: this does not include group(0)
        if num_groups:
            # To reduce memory footprint, don't store group(0) i.e. the whole
            # line, if there are actual groups in the result.
            for i in range(1, num_groups + 1):
                self._add(i, result.group(i))
        else:
            log.debug("saving full search result which can lead to high "
                      "memory usage")
            self._add(0, result.group(0))

    @cached_property
    def id(self):
        """ Unique Result ID """
        id_string = "{}-{}-{}".format(uuid.uuid4(), self.source_id,
                                      self.linenumber)
        if self.sequence_obj_id:
            id_string = "{}-{}-{}".format(id_string, self.sequence_obj_id,
                                          self.section_id)
        return id_string

    def _add(self, index, value):
        self.data.append(SearchResultPart(index, value))

    def get(self, index):
        """Retrieve result part value by index."""
        for group in self.data:
            if group.index == index:
                return group.value

    def __iter__(self):
        """ Only return part values when iterating over this object. """
        for part in self.data:
            yield part.value

    def __repr__(self):
        r_list = ["{}='{}'".format(rp.index, rp.value) for rp in self.data]
        return "ln:{} {}".format(self.linenumber, ", ".join(r_list))


class SearchResultsCollection(UserDict):

    def __init__(self, search_catalog):
        self.search_catalog = search_catalog
        self.reset()

    @property
    def data(self):
        results = {}
        for path, ids in self._results_by_path.items():
            results[path] = [self._results_by_id[id] for id in ids]

        return results

    def reset(self):
        self._iter_idx = 0
        self._results_by_path = {}
        self._results_by_id = {}

    @property
    def files(self):
        return list(self._results_by_path.keys())

    def add(self, result):
        # resolve
        path = self.search_catalog.source_id_to_path(result.source_id)
        self._results_by_id[result.id] = result
        if path not in self._results_by_path:
            self._results_by_path[path] = [result.id]
        else:
            self._results_by_path[path].append(result.id)

    def find_by_path(self, path):
        """ Return results for a given path. """
        results = self._results_by_path.get(path, [])
        return [self._results_by_id[id] for id in results]

    def find_by_tag(self, tag, path=None):
        """ Return results matched by tag.

        @param tag: tag used to identify search results.
        @param path: optional path used to filter results to only include those
                     matched from a given path.
        @param sequence_obj_id: optionally provide sequence object id to only
                                match sequence results.
        """
        if path:
            paths = [path]
        else:
            paths = list(self._results_by_path.keys())

        results = []
        for path in paths:
            for result in self.find_by_path(path):
                if result.tag != tag:
                    continue

                results.append(result)

        return results

    def _get_all_sequence_results(self, path=None):
        """ Return a list of ids for all sequence match results.

        @param path: optionally filter results for a given path.
        """
        if path:
            paths = [path]
        else:
            paths = list(self._results_by_path.keys())

        sequences = []
        for path in paths:
            for result in self.find_by_path(path):
                if result.sequence_obj_id is not None:
                    sequences.append(result.id)

        return sequences

    def find_sequence_by_tag(self, tag, path=None):
        """ Find results for the sequence search(es) identified from tag.

        Returns a list of "sections" i.e. complete sequences matched using
        associated SequenceSearchDef objects. Each section is a list of
        SearchResult objects representing start/body/end for that section.

        @param tag: tag used to identify sequence results.
        @param path: optionally filter results for a given path.
        """
        sections = {}
        for seq_obj in self.search_catalog.resolve_from_tag(tag):
            sections.update(self.find_sequence_sections(seq_obj, path))

        return sections

    def find_sequence_sections(self, sequence_obj, path=None):
        """ Find results for the given sequence search.

        Returns a list of "sections" i.e. complete sequences matched using
        the associated SequenceSearchDef object. Each section is a list of
        SearchResult objects representing start/body/end for that section.

        @param sequence_obj: SequenceSearch object
        @param path: optionally filter results for a given path.
        """
        _results = {}
        for r in self._get_all_sequence_results(path=path):
            result = self._results_by_id[r]
            s_id = result.sequence_obj_id
            if s_id != sequence_obj.id:
                continue

            section_id = result.section_id
            if section_id not in _results:
                _results[section_id] = []

            _results[section_id].append(result)

        return _results

    def __len__(self):
        """ Returns total number of search results. """
        _count = 0
        for f in self.files:
            _count += len(self.find_by_path(f))

        return _count


class LogrotateLogSort(object):

    def __call__(self, fname):
        """
        Sort contents of a directory by passing the function as the key to a
        list sort. Directory is expected to contain logfiles with extensions
        used by logrotate e.g. .log, .log.1, .log.2.gz etc.
        """
        filters = [r"\S+\.log$",
                   r"\S+\.log\.(\d+)$",
                   r"\S+\.log\.(\d+)\.gz?$"]
        for filter in filters:
            ret = re.compile(filter).match(fname)
            if ret:
                break

        # files that don't follow logrotate naming format go to the end.
        if not ret:
            # put at the end
            return 100000

        if len(ret.groups()) == 0:
            return 0

        return int(ret.group(1))


class SearchCatalog(object):

    def __init__(self, max_logrotate_depth=7):
        self.max_logrotate_depth = max_logrotate_depth
        self._user_paths = {}
        self._source_ids = {}
        self._search_tags = {}
        self._simple_searches = {}
        self._sequence_searches = {}

    def register(self, search, user_path):
        """
        Register a search against a path.

        The same search can be registered against more than one path.

        @param search: object implemented from SearchDefBase.
        @param user_path: directory or file path.
        """
        if search.tag is not None:
            if search.tag in self._search_tags:
                if search.id not in self._search_tags[search.tag]:
                    log.debug("one or more search tagged '%s' has already "
                              "been registered against path '%s'",
                              search.tag, user_path)
                    self._search_tags[search.tag].append(search.id)
            else:
                self._search_tags[search.tag] = [search.id]

        if isinstance(search, SequenceSearchDef):
            self._sequence_searches[search.id] = search
        else:
            self._simple_searches[search.id] = search

        if user_path in self._user_paths:
            self._user_paths[user_path].append(search.id)
        else:
            self._user_paths[user_path] = [search.id]

    def resolve_from_id(self, id):
        """ Resolve search definition from unique id. """
        if id in self._simple_searches:
            return self._simple_searches[id]

        return self._sequence_searches[id]

    def resolve_from_tag(self, tag):
        """ Resolve search definition from tag.

        Returns a list of resolved searches.
        """
        searches = []
        for id in self._search_tags[tag]:
            searches.append(self.resolve_from_id(id))

        return searches

    def _filtered_dir(self, contents, max_logrotate_depth=7):
        """ Filter contents of a directory. Directories are ignored and if any
        files look like logrotated log files they are sorted and only
        max_logrotate_depth are kept.
        """
        logrotated = {}
        new_contents = []
        for path in contents:
            if not os.path.isfile(path):
                continue

            ret = re.compile(r"(\S+)\.log\S*").match(path)
            if not ret:
                new_contents.append(path)
                continue

            fnamepfix = ret.group(1)
            if path.endswith('.log'):
                new_contents.append(fnamepfix + '.log')
            else:
                if fnamepfix not in logrotated:
                    logrotated[fnamepfix] = [path]
                else:
                    logrotated[fnamepfix].append(path)

        limit = max_logrotate_depth
        for logrotated in logrotated.values():
            capped = sorted(logrotated,
                            key=LogrotateLogSort())[:limit]
            new_contents += capped

        return new_contents

    def _expand_path(self, path):
        if os.path.isfile(path):
            return [path]
        elif os.path.isdir(path):
            return self._filtered_dir(os.listdir(path),
                                      self.max_logrotate_depth)

        return self._filtered_dir(glob.glob(path), self.max_logrotate_depth)

    def source_id_to_path(self, s_id):
        try:
            return self._source_ids[s_id]
        except KeyError:
            log.exception("ALL PATHS:")
            log.error('\n'.join(list(self._source_ids.keys())))

    def _get_source_id(self, path):
        for id, _path in self._source_ids.items():
            if _path == path:
                return id

        s_id = str(uuid.uuid4())
        while s_id in self._source_ids:
            log.error("source id %s already exists - trying again", s_id)
            s_id = str(uuid.uuid4())

        log.debug("path=%s source_id=%s", path, s_id)
        self._source_ids[s_id] = path
        return s_id

    def __iter__(self):
        for user_path, searches in self._user_paths.items():
            for path in self._expand_path(user_path):
                yield {'user_path': user_path,
                       'path': path,
                       'source_id': self._get_source_id(path),
                       'searches': [self.resolve_from_id(id)
                                    for id in searches]}


class SearchTask(object):

    def __init__(self, info, lock, constraints_manager, results_queue_id):
        self.info = info
        self.lock = lock
        self.stats = SearchTaskStats()
        self.constraints_manager = constraints_manager
        self.results_queue_id = results_queue_id

    @cached_property
    def id(self):
        return str(uuid.uuid4())

    @cached_property
    def search_defs_conditional(self):
        return [s_term for s_term in self.info['searches']
                if s_term.constraints]

    @cached_property
    def search_defs(self):
        all = {s_term: True for s_term in self.info['searches']}
        for s_term in all:
            if s_term in self.search_defs_conditional:
                all[s_term] = False

        return all

    def put_result(self, result):
        self.stats['results'] += 1
        max_tries = MAX_QUEUE_RETRIES
        _queue = RESULTS_QUEUES[self.results_queue_id]
        while max_tries > 0:
            try:
                if max_tries == MAX_QUEUE_RETRIES:
                    _queue.put_nowait(result)
                else:
                    _queue.put(result, timeout=RESULTS_QUEUE_TIMEOUT)

                break
            except queue.Full:
                if max_tries == MAX_QUEUE_RETRIES:
                    msg = ("search task queue for '%s' is full - switching "
                           "to using blocking put with timeout")
                    log.info(msg, self.info['path'])
                else:
                    msg = ("search task queue for '%s' is full even after "
                           "waiting %ss - trying again")
                    log.warning(msg, self.info['path'], RESULTS_QUEUE_TIMEOUT)

                max_tries -= 1

        if max_tries == 0:
            log.error("exceeded max number of retries (%s) to put results "
                      "data on the queue", MAX_QUEUE_RETRIES)

    def _simple_search(self, s_term, line, ln):
        """ Perform a simple search on line.

        @param s_term: SearchDef object
        @param line: current line (string)
        @param ln: current line number
        """
        ret = s_term.run(line)
        if not ret:
            return

        self.put_result(SearchResult(
                            ln, self.info['source_id'], ret, s_term.tag,
                            store_contents=s_term.store_result_contents))

    def _sequence_search(self, s_term, line, ln, sequence_results):
        """ Perform a sequence search on line.

        @param s_term: SequenceSearchDef object
        @param line: current line (string)
        @param ln: current line number
        @param sequence_results: SequenceSearchResults object
        """
        ret = s_term.s_start.run(line)
        # if the ending is defined and we match a start while
        # already in a section, we start again.
        if s_term.s_end and s_term.started:
            if ret:
                # reset and start again
                sequence_results.remove(s_term.id)
                s_term.reset()
            else:
                ret = s_term.s_end.run(line)

        if ret:
            tag = s_term.tag
            if not s_term.started:
                tag = s_term.start_tag
                s_term.start()
                section_id = s_term.section_id
            else:
                tag = s_term.end_tag
                section_id = s_term.section_id
                s_term.stop()
                # if no end is defined then we don't bother storing
                # the result, just complete the section and start
                # the next.
                if s_term.s_end is None:
                    tag = s_term.start_tag
                    s_term.start()
                    section_id = s_term.section_id

            sequence_results.add(SearchResult(ln, self.info['source_id'], ret,
                                              tag, section_id=section_id,
                                              sequence_obj_id=s_term.id))
        elif s_term.started and s_term.s_body:
            ret = s_term.s_body.run(line)
            if ret:
                sequence_results.add(SearchResult(ln, self.info['source_id'],
                                                  ret, s_term.body_tag,
                                                  section_id=s_term.section_id,
                                                  sequence_obj_id=s_term.id))

    def _process_sequence_results(self, sequence_results, current_ln):
        """
        Perform post processing to sequence search results.

        @param sequence_results: SequenceSearchResults object
        @param current_ln: number of the last line to be read from file
        """
        # If a sequence ending definition is provided and we reached EOF
        # while a sequence is started, complete the sequence if s_end
        # matches an empty string. If s_end is not defined we just go ahead
        # and complete the section.
        filter_section_id = {}
        for s_term in self.search_defs:
            if type(s_term) == SequenceSearchDef:
                if s_term.started:
                    if s_term.s_end is None:
                        s_term.stop()
                    else:
                        ret = s_term.s_end.run("")
                        if ret:
                            section_id = s_term.section_id
                            s_term.stop()
                            tag = s_term.end_tag
                            r = SearchResult(current_ln + 1,
                                             self.info['source_id'], ret,
                                             tag,
                                             section_id=section_id,
                                             sequence_obj_id=s_term.id)
                            sequence_results.add(r)
                        else:
                            if s_term.id not in filter_section_id:
                                filter_section_id[s_term.id] = []

                            filter_section_id[s_term.id].append(
                                s_term.section_id)

        if len(sequence_results) < 1:
            log.debug("no sequence results to process")
            return

        # Now add sequence results to main results list, excluding any
        # incomplete sections.
        for s_results in sequence_results.values():
            for r in s_results:
                if filter_section_id:
                    if r.sequence_obj_id is None:
                        continue

                    seq_id = r.sequence_obj_id
                    if seq_id in filter_section_id:
                        if r.section_id in filter_section_id[seq_id]:
                            continue

                self.put_result(r)

    def _run_search(self, fd):
        """
        @param fd: open file descriptor
        """
        self.stats.reset()
        sequence_results = SequenceSearchResults()
        offset = self.constraints_manager.apply_global(self.info['user_path'],
                                                       fd)
        log.debug("starting search of %s (offset=%s, pos=%s)", fd.name, offset,
                  fd.tell())
        runnable = {s.id: _runnable
                    for s, _runnable in self.search_defs.items()}
        ln = 0
        # NOTE: line numbers start at 1 hence offset + 1
        for ln, line in enumerate(fd, start=offset + 1):
            self.stats['lines_searched'] += 1
            if type(line) == bytes:
                line = line.decode("utf-8")

            for s_term in self.search_defs:
                if not runnable[s_term.id]:
                    if not self.constraints_manager.apply_single(s_term, line):
                        continue

                    # enable from here on in
                    runnable[s_term.id] = True

                if type(s_term) == SequenceSearchDef:
                    self._sequence_search(s_term, line, ln, sequence_results)
                else:
                    self._simple_search(s_term, line, ln)

        self._process_sequence_results(sequence_results, ln)
        log.debug("completed search of %s lines", self.stats['lines_searched'])
        if self.search_defs_conditional:
            msg = "constraints stats {}:".format(fd.name)
            for sd in self.search_defs_conditional:
                if sd.constraints:
                    for c in sd.constraints.values():
                        msg += "\n  id={}: {}".format(c.id, c.stats())

            log.debug(msg)

        return self.stats

    def failed(self, exc):
        """ This should be called if the task failed to execute. """
        log.error("search task failed for path=%s with exception %s",
                  self.info['path'], exc)

    def execute(self):
        path = self.info['path']
        if os.path.getsize(path) == 0:
            log.debug("filesearcher: zero-length file %s - skipping search",
                      path)
            return

        try:
            with gzip.open(path, 'rb') as fd:
                try:
                    # test if file is gzip
                    fd.read(1)
                    fd.seek(0)
                    return self._run_search(fd)
                except OSError:
                    pass

            with open(path, 'rb') as fd:
                return self._run_search(fd)
        except UnicodeDecodeError:
            log.exception("")
            # ignore the file if it can't be decoded
            log.debug("caught UnicodeDecodeError for path %s - skipping",
                      path)
        except EOFError as e:
            log.exception("")
            msg = ("an exception occurred while searching {} - {}".
                   format(path, e))
            raise FileSearchException(msg) from e
        except Exception as e:
            log.exception("")
            msg = ("an unexpected exception occurred while searching {} - {}".
                   format(path, e))
            raise FileSearchException(msg) from e

        return SearchTaskStats()


class SearchTaskStats(UserDict):

    def __init__(self):
        self.reset()

    def reset(self):
        self.data = {'lines_searched': 0,
                     'results': 0}

    def combine(self, stats):
        if not stats:
            return

        for key, val in stats.items():
            self.data[key] += val

    def __repr__(self):
        return str(self.data)


class SearcherBase(abc.ABC):

    @abc.abstractproperty
    def files(self):
        """ Returns a list of files we will be searching. """

    @abc.abstractproperty
    def num_parallel_tasks(self):
        """
        Returns an integer representing the maximum number of tasks we can
        run in parallel. This will typically be bound by the number of
        cpu threads available.
        """

    @abc.abstractmethod
    def add(self, searchdef):
        """
        Add a search criterea.

        @param searchdef: SearchDef object
        """

    @abc.abstractmethod
    def run(self):
        """
        Execute all searches.
        """


class SearchConstraintsManager(object):

    def __init__(self, search_catalog):
        self.search_catalog = search_catalog
        self.global_constraints = []
        self.global_restrictions = set()

    def apply_global(self, user_path, fd):
        """ Apply any global constraints to the entire file. """
        offset = 0
        if not self.global_constraints:
            log.debug("no global constraint to apply to %s", fd.name)
            return offset

        if user_path in self.global_restrictions:
            log.debug("skipping global constraint for %s", fd.name)
            return offset

        for c in self.global_constraints:
            log.debug("applying task global constraint %s to %s", c.id,
                      fd.name)
            _offset = c.apply_to_file(fd)
            if _offset is not None:
                return _offset

        return offset

    def apply_single(self, searchdef, line):
        """
        Apply any constraints for this searchdef to the give line.
        """
        if not searchdef.constraints:
            return True

        for c in searchdef.constraints.values():
            if c.apply_to_line(line):
                continue

            return False

        return True


class FileSearcher(SearcherBase):

    def __init__(self, max_parallel_tasks=8, max_logrotate_depth=7,
                 constraint=None):
        """
        @param max_parallel_tasks: max number of search tasks that can run in
                                   parallel.
        @param max_logrotate_depth: used by SearchCatalog to filter logfiles
                                    based on their name if it matches a
                                    logrotate format and want to constrain how
                                    much history we search.
        @param constraint: constraint to be used with this
                                   searcher that applies to all files searched.
        """
        self.max_parallel_tasks = max_parallel_tasks
        self._stats = SearchTaskStats()
        self.catalog = SearchCatalog(max_logrotate_depth)
        self.constraints_manager = SearchConstraintsManager(self.catalog)
        if constraint:
            self.constraints_manager.global_constraints.append(constraint)

    @property
    def files(self):
        return [e['path'] for e in self.catalog]

    def resolve_source_id(self, source_id):
        return self.catalog.source_id_to_path(source_id)

    def add(self, searchdef, path, allow_global_constraints=True):
        """
        Add a search definition.

        @param searchdef: a SearchDef or SequenceSearchDef object.
        @param path: path we want to search. this can be a file, dir or glob.
        @param allow_global_constraints: boolean determining whether we want
                                         any global constraints available to be
                                         applied to this path.
        """
        if not allow_global_constraints:
            self.constraints_manager.global_restrictions.add(path)

        self.catalog.register(searchdef, path)

    @property
    def num_parallel_tasks(self):
        if self.max_parallel_tasks == 0:
            cpus = 1  # i.e. no parallelism
        else:
            cpus = min(self.max_parallel_tasks, os.cpu_count())

        return min(len(self.files) or 1, cpus)

    @property
    def stats(self):
        """
        Provide stats for the last search run.

        @return: SearchTaskStats object
        """
        return self._stats

    def _get_results(self, results, queue_id, expected=None,
                     event=None):
        """
        Collect results from all search task processes.

        @param results: SearchResultsCollection object.
        @param queue_id: id of results queue used for this search session.
        @param expected: number of results we expect to receive. this is used
                         to do a final sweep once all search tasks are complete
                         to ensure all results have been collected.
        @param event: event object used to notify this thread to stop.
        """
        _queue = RESULTS_QUEUES[queue_id]
        while True:
            if not _queue.empty():
                results.add(_queue.get())
            elif expected and expected > len(results):
                try:
                    r = _queue.get(timeout=RESULTS_QUEUE_TIMEOUT)
                    results.add(r)
                except queue.Empty:
                    log.info("timeout waiting > %s secs to receive results - "
                             "expected=%s, actual=%s", RESULTS_QUEUE_TIMEOUT,
                             expected, len(results))
            elif event:
                if event.is_set():
                    log.debug("exiting results thread")
                    break

                # yield
                time.sleep(0)
            else:
                break

    def _start_results_thread(self, results, queue_id):
        log.debug("starting results queue consumer thread")
        event = threading.Event()
        event.clear()
        t = threading.Thread(target=self._get_results,
                             args=[results, queue_id],
                             kwargs={'event': event}, daemon=False)
        t.start()
        return t, event

    def _stop_results_thread(self, thread, event):
        log.debug("joining/stopping queue consumer thread")
        event.set()
        thread.join()

    def _run(self, queue_id):
        """ Run all searches.

        @return: SearchResultsCollection object
        """
        self.stats.reset()
        results = SearchResultsCollection(self.catalog)
        jobs = {}
        with multiprocessing.Manager() as m:
            # See https://github.com/PyCQA/pylint/issues/3313 on why we ignore
            # the following pylint error
            lock = m.Lock()  # pylint: disable=E1101
            results_thread = None
            try:
                # make child processes ignore sigint so it can be handled here
                orig_sigint = signal.signal(signal.SIGINT, signal.SIG_IGN)
                num_p_tasks = self.num_parallel_tasks
                with concurrent.futures.ProcessPoolExecutor(
                                        max_workers=num_p_tasks) as executor:
                    results_thread, event = self._start_results_thread(
                                                                   results,
                                                                   queue_id)
                    # re-establish sigint
                    signal.signal(signal.SIGINT, orig_sigint)
                    for info in self.catalog:
                        c_mgr = self.constraints_manager
                        task = SearchTask(info, lock,
                                          constraints_manager=c_mgr,
                                          results_queue_id=queue_id)
                        jobs[executor.submit(task.execute)] = task.id

                    log.debug("filesearcher: syncing %s jobs", len(jobs))
                    try:
                        for future in concurrent.futures.as_completed(jobs):
                            self.stats.combine(future.result())
                    except concurrent.futures.process.BrokenProcessPool as exc:
                        msg = ("one or more worker processes has died - "
                               "aborting search")
                        raise FileSearchException(msg) from exc

                    log.debug("joining queue (expected=%s, remaining=%s)",
                              self.stats['results'],
                              self.stats['results'] - len(results))
                    self._stop_results_thread(results_thread, event)
                    results_thread = None
                    self._get_results(results, queue_id,
                                      expected=self.stats['results'])
            finally:
                if results_thread is not None:
                    self._stop_results_thread(results_thread, event)

        log.debug("filesearcher: stats=%s", self.stats)
        log.debug("filesearcher: completed (results=%s)", len(results))
        return results

    def run(self):
        """ Run all searches.

        @return: SearchResultsCollection object
        """
        queue_id = "{}:{}".format(os.getpid(), uuid.uuid4())
        if queue_id in RESULTS_QUEUES:
            raise Exception("results queue already exists with id {}".
                            format(queue_id))

        RESULTS_QUEUES[queue_id] = multiprocessing.Queue()
        try:
            return self._run(queue_id)
        finally:
            del RESULTS_QUEUES[queue_id]
