"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("@aws-cdk/core");
const codedeploy_generated_1 = require("../codedeploy.generated");
const utils_1 = require("../utils");
/**
 * Minimum number of healthy hosts for a server deployment.
 */
class MinimumHealthyHosts {
    constructor(json) {
        this.json = json;
    }
    /**
     * The minimum healhty hosts threshold expressed as an absolute number.
     */
    static count(value) {
        return new MinimumHealthyHosts({
            type: 'HOST_COUNT',
            value,
        });
    }
    /**
     * The minmum healhty hosts threshold expressed as a percentage of the fleet.
     */
    static percentage(value) {
        return new MinimumHealthyHosts({
            type: 'FLEET_PERCENT',
            value,
        });
    }
    /**
     * @internal
     */
    get _json() {
        return this.json;
    }
}
exports.MinimumHealthyHosts = MinimumHealthyHosts;
/**
 * A custom Deployment Configuration for an EC2/on-premise Deployment Group.
 *
 * @resource AWS::CodeDeploy::DeploymentConfig
 */
class ServerDeploymentConfig extends cdk.Resource {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.deploymentConfigName,
        });
        const resource = new codedeploy_generated_1.CfnDeploymentConfig(this, 'Resource', {
            deploymentConfigName: this.physicalName,
            minimumHealthyHosts: props.minimumHealthyHosts._json,
        });
        this.deploymentConfigName = resource.ref;
        this.deploymentConfigArn = utils_1.arnForDeploymentConfig(this.deploymentConfigName);
    }
    /**
     * Import a custom Deployment Configuration for an EC2/on-premise Deployment Group defined either outside the CDK app,
     * or in a different region.
     *
     * @param scope the parent Construct for this new Construct
     * @param id the logical ID of this new Construct
     * @param serverDeploymentConfigName the properties of the referenced custom Deployment Configuration
     * @returns a Construct representing a reference to an existing custom Deployment Configuration
     */
    static fromServerDeploymentConfigName(scope, id, serverDeploymentConfigName) {
        ignore(scope);
        ignore(id);
        return deploymentConfig(serverDeploymentConfigName);
    }
}
exports.ServerDeploymentConfig = ServerDeploymentConfig;
ServerDeploymentConfig.ONE_AT_A_TIME = deploymentConfig('CodeDeployDefault.OneAtATime');
ServerDeploymentConfig.HALF_AT_A_TIME = deploymentConfig('CodeDeployDefault.HalfAtATime');
ServerDeploymentConfig.ALL_AT_ONCE = deploymentConfig('CodeDeployDefault.AllAtOnce');
function deploymentConfig(name) {
    return {
        deploymentConfigName: name,
        deploymentConfigArn: utils_1.arnForDeploymentConfig(name),
    };
}
function ignore(_x) { return; }
//# sourceMappingURL=data:application/json;base64,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